import Result from './result';
import Transaction from './transaction';
import { ConnectionHolder } from './internal/connection-holder';
import { Bookmark } from './internal/bookmark';
import { TxConfig } from './internal/tx-config';
import ConnectionProvider from './connection-provider';
import { Query, SessionMode } from './types';
import Connection from './connection';
import { NumberOrInteger } from './graph-types';
declare type ConnectionConsumer = (connection: Connection | void) => any | undefined;
declare type TransactionWork<T> = (tx: Transaction) => Promise<T> | T;
interface TransactionConfig {
    timeout?: NumberOrInteger;
    metadata?: object;
}
/**
 * A Session instance is used for handling the connection and
 * sending queries through the connection.
 * In a single session, multiple queries will be executed serially.
 * In order to execute parallel queries, multiple sessions are required.
 * @access public
 */
declare class Session {
    private _mode;
    private _database;
    private _reactive;
    private _fetchSize;
    private _readConnectionHolder;
    private _writeConnectionHolder;
    private _open;
    private _hasTx;
    private _lastBookmark;
    private _transactionExecutor;
    private _impersonatedUser?;
    private _onComplete;
    private _databaseNameResolved;
    /**
     * @constructor
     * @protected
     * @param {Object} args
     * @param {string} args.mode the default access mode for this session.
     * @param {ConnectionProvider} args.connectionProvider - The connection provider to acquire connections from.
     * @param {Bookmark} args.bookmark - The initial bookmark for this session.
     * @param {string} args.database the database name
     * @param {Object} args.config={} - This driver configuration.
     * @param {boolean} args.reactive - Whether this session should create reactive streams
     * @param {number} args.fetchSize - Defines how many records is pulled in each pulling batch
     * @param {string} args.impersonatedUser - The username which the user wants to impersonate for the duration of the session.
     */
    constructor({ mode, connectionProvider, bookmark, database, config, reactive, fetchSize, impersonatedUser }: {
        mode: SessionMode;
        connectionProvider: ConnectionProvider;
        bookmark?: Bookmark;
        database: string;
        config: any;
        reactive: boolean;
        fetchSize: number;
        impersonatedUser?: string;
    });
    /**
     * Run Cypher query
     * Could be called with a query object i.e.: `{text: "MATCH ...", parameters: {param: 1}}`
     * or with the query and parameters as separate arguments.
     *
     * @public
     * @param {mixed} query - Cypher query to execute
     * @param {Object} parameters - Map with parameters to use in query
     * @param {TransactionConfig} [transactionConfig] - Configuration for the new auto-commit transaction.
     * @return {Result} New Result.
     */
    run(query: Query, parameters?: any, transactionConfig?: TransactionConfig): Result;
    _run(query: Query, parameters: any, customRunner: ConnectionConsumer): Result;
    _acquireConnection(connectionConsumer: ConnectionConsumer): Promise<any>;
    /**
     * Begin a new transaction in this session. A session can have at most one transaction running at a time, if you
     * want to run multiple concurrent transactions, you should use multiple concurrent sessions.
     *
     * While a transaction is open the session cannot be used to run queries outside the transaction.
     *
     * @param {TransactionConfig} [transactionConfig] - Configuration for the new auto-commit transaction.
     * @returns {Transaction} New Transaction.
     */
    beginTransaction(transactionConfig?: TransactionConfig): Transaction;
    _beginTransaction(accessMode: SessionMode, txConfig: TxConfig): Transaction;
    /**
     * @private
     * @returns {void}
     */
    _assertSessionIsOpen(): void;
    /**
     * @private
     * @returns {void}
     */
    _transactionClosed(): void;
    /**
     * Return the bookmark received following the last completed {@link Transaction}.
     *
     * @return {string[]} A reference to a previous transaction.
     */
    lastBookmark(): string[];
    /**
     * Execute given unit of work in a {@link READ} transaction.
     *
     * Transaction will automatically be committed unless the given function throws or returns a rejected promise.
     * Some failures of the given function or the commit itself will be retried with exponential backoff with initial
     * delay of 1 second and maximum retry time of 30 seconds. Maximum retry time is configurable via driver config's
     * `maxTransactionRetryTime` property in milliseconds.
     *
     * @param {function(tx: Transaction): Promise} transactionWork - Callback that executes operations against
     * a given {@link Transaction}.
     * @param {TransactionConfig} [transactionConfig] - Configuration for all transactions started to execute the unit of work.
     * @return {Promise} Resolved promise as returned by the given function or rejected promise when given
     * function or commit fails.
     */
    readTransaction<T>(transactionWork: TransactionWork<T>, transactionConfig?: TransactionConfig): Promise<T>;
    /**
     * Execute given unit of work in a {@link WRITE} transaction.
     *
     * Transaction will automatically be committed unless the given function throws or returns a rejected promise.
     * Some failures of the given function or the commit itself will be retried with exponential backoff with initial
     * delay of 1 second and maximum retry time of 30 seconds. Maximum retry time is configurable via driver config's
     * `maxTransactionRetryTime` property in milliseconds.
     *
     * @param {function(tx: Transaction): Promise} transactionWork - Callback that executes operations against
     * a given {@link Transaction}.
     * @param {TransactionConfig} [transactionConfig] - Configuration for all transactions started to execute the unit of work.
     * @return {Promise} Resolved promise as returned by the given function or rejected promise when given
     * function or commit fails.
     */
    writeTransaction<T>(transactionWork: TransactionWork<T>, transactionConfig?: TransactionConfig): Promise<T>;
    _runTransaction<T>(accessMode: SessionMode, transactionConfig: TxConfig, transactionWork: TransactionWork<T>): Promise<T>;
    /**
     * Sets the resolved database name in the session context.
     * @private
     * @param {string|undefined} database The resolved database name
     * @returns {void}
     */
    _onDatabaseNameResolved(database?: string): void;
    /**
     * Update value of the last bookmark.
     * @private
     * @param {Bookmark} newBookmark - The new bookmark.
     * @returns {void}
     */
    _updateBookmark(newBookmark?: Bookmark): void;
    /**
     * Close this session.
     * @return {Promise}
     */
    close(): Promise<void>;
    _connectionHolderWithMode(mode: SessionMode): ConnectionHolder;
    /**
     * @private
     * @param {Object} meta Connection metadatada
     * @returns {void}
     */
    _onCompleteCallback(meta: {
        bookmark: string | string[];
    }): void;
    /**
     * @protected
     */
    static _validateSessionMode(rawMode?: SessionMode): SessionMode;
}
export default Session;
export type { TransactionConfig };
