from __future__ import annotations
from pathlib import Path
from typing import Dict, Any, List

from checkov.runner_filter import RunnerFilter
from checkov.sca_image.runner import Runner
from checkov.common.images.image_referencer import  Image


def mock_scan(self: Runner, image_id: str, dockerfile_path: str, runner_filter: RunnerFilter | None = None) -> Dict[str, Any]:
    return dict(results=[{'id': 'sha256:973f3910e3465433dbc712f147f2ce15c42be69ccd558a13a3ec74127b4bd801',
                          'distro': 'Ubuntu 22.04.1 LTS', 'distroRelease': 'jammy', 'collections': ['All'],
                          'packages': [{'type': 'os', 'name': 'pcre2', 'version': '10.39-3build1'},
                                       {'type': 'os', 'name': 'libidn2', 'version': '2.3.2-2build1',
                                        'licenses': ['GPL-3+']},
                                       {'type': 'os', 'name': 'perl', 'version': '5.34.0-3ubuntu1',
                                        'licenses': ['GPL-1+ or Artistic']},
                                       {'type': 'os', 'name': 'bzip2', 'version': '1.0.8-5build1',
                                        'licenses': ['BSD-variant']}],
                          'complianceDistribution': {'critical': 0, 'high': 0, 'medium': 0, 'low': 0, 'total': 0},
                          'complianceScanPassed': True,
                          'vulnerabilities': [
             {'id': 'CVE-2020-16156', 'status': 'needed', 'cvss': 7.8,
             'vector': 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H',
             'description': 'CPAN 2.28 allows Signature Verification Bypass.', 'severity': 'medium',
             'packageName': 'perl', 'packageVersion': '5.34.0-3ubuntu1',
             'link': 'https://people.canonical.com/~ubuntu-security/cve/2020/CVE-2020-16156',
             'riskFactors': ['Attack complexity: low', 'Medium severity'], 'impactedVersions': ['*'],
             'publishedDate': '2021-12-13T20:15:00+02:00', 'discoveredDate': '2022-08-11T14:03:31+03:00'},
             {'id': 'CVE-2022-1587', 'status': 'needed', 'cvss': 9.1,
                                                       'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H',
                                                       'description': 'An out-of-bounds read vulnerability was discovered in the PCRE2 library in the get_recurse_data_length() function of the pcre2_jit_compile.c file. This issue affects recursions in JIT-compiled regular expressions caused by duplicate data transfers.',
                                                       'severity': 'low', 'packageName': 'pcre2',
                                                       'packageVersion': '10.39-3build1',
                                                       'link': 'https://people.canonical.com/~ubuntu-security/cve/2022/CVE-2022-1587',
                                                       'riskFactors': ['Recent vulnerability', 'Attack complexity: low',
                                                                       'Attack vector: network'],
                                                       'impactedVersions': ['*'],
                                                       'publishedDate': '2022-05-17T00:15:00+03:00',
                                                       'discoveredDate': '2022-08-11T14:03:31+03:00'},
             {'id': 'CVE-2022-1586', 'status': 'needed', 'cvss': 9.1,
             'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H',
             'description': 'An out-of-bounds read vulnerability was discovered in the PCRE2 library in the compile_xclass_matchingpath() function of the pcre2_jit_compile.c file. This involves a unicode property matching issue in JIT-compiled regular expressions. The issue occurs because the character was not fully read in case-less matching within JIT.',
             'severity': 'low', 'packageName': 'pcre2', 'packageVersion': '10.39-3build1',
             'link': 'https://people.canonical.com/~ubuntu-security/cve/2022/CVE-2022-1586',
             'riskFactors': ['Attack vector: network', 'Recent vulnerability', 'Attack complexity: low'],
             'impactedVersions': ['*'], 'publishedDate': '2022-05-17T00:15:00+03:00',
             'discoveredDate': '2022-08-11T14:03:31+03:00'}
                ],
                          'vulnerabilityDistribution': {'critical': 0, 'high': 0, 'medium': 2, 'low': 6, 'total': 8},
                          'vulnerabilityScanPassed': True}])


def mock_get_images(self: Runner, file_path: str) -> list[Image]:
    return [Image(
                file_path='fake_path',
                name='fake_image',
                start_line=1,
                end_line=12
            )]


def mock_scan_empty(self: Runner, image_id: str, dockerfile_path: str, runner_filter: RunnerFilter | None = None) -> Dict[str, Any]:
    return {}


def mock_scan_image(self: Runner, dockerfile_path: str, image: Image, image_cached_result: Dict[str, Any], root_folder: str | Path | None) -> Dict[str, Any]:
    return {'dockerImageName': 'node:14.16', 'dockerFilePath': '/Users/lshindelman/development/examples/bce-7720/lshindelman-supplygoat2-7d53e1de969e/.github/workflows/main.yml', 'dockerFileContent': '/Users/lshindelman/development/examples/bce-7720/lshindelman-supplygoat2-7d53e1de969e/.github/workflows/main.yml', 'type': 'Image', 'check_type': 'sca_images', 'sourceId': 'cli_repo/lshindelman-supplygoat2-7d53e1de969e', 'branch': 'master', 'sourceType': 'cli', 'vulnerabilities': [{'cveId': 'CVE-2021-44906', 'status': 'fixed in 1.2.6', 'severity': 'critical', 'packageName': 'minimist', 'packageVersion': '1.2.5', 'link': 'https://github.com/advisories/GHSA-xvch-5gv4-984h', 'cvss': 9.8, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', 'description': 'Minimist <=1.2.5 is vulnerable to Prototype Pollution via file index.js, function setKey() (lines 69-95).', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Critical severity', 'Has fix', 'Recent vulnerability'], 'publishedDate': '2022-03-17T16:15:00Z'}, {'cveId': 'CVE-2019-9636', 'status': 'fixed in 3.7.3, 3.6.9, 3.5.7,...', 'severity': 'critical', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2019-9636', 'cvss': 9.8, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', 'description': 'Python 2.7.x through 2.7.16 and 3.x through 3.7.2 is affected by: Improper Handling of Unicode Encoding (with an incorrect netloc) during NFKC normalization. The impact is: Information disclosure (credentials, cookies, etc. that are cached against a given hostname). The components are: urllib.parse.urlsplit, urllib.parse.urlparse. The attack vector is: A specially crafted URL could be incorrectly parsed to locate cookies or authentication data and send that information to a different host than when parsed correctly. This is fixed in: v2.7.17, v2.7.17rc1, v2.7.18, v2.7.18rc1; v3.5.10, v3.5.10rc1, v3.5.7, v3.5.8, v3.5.8rc1, v3.5.8rc2, v3.5.9; v3.6.10, v3.6.10rc1, v3.6.11, v3.6.11rc1, v3.6.12, v3.6.9, v3.6.9rc1; v3.7.3, v3.7.3rc1, v3.7.4, v3.7.4rc1, v3.7.4rc2, v3.7.5, v3.7.5rc1, v3.7.6, v3.7.6rc1, v3.7.7, v3.7.7rc1, v3.7.8, v3.7.8rc1, v3.7.9.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Critical severity', 'Has fix'], 'publishedDate': '2019-03-08T21:29:00Z'}, {'cveId': 'CVE-2019-10160', 'status': 'fixed in 3.7.4, 3.6.9, 3.5.8,...', 'severity': 'critical', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2019-10160', 'cvss': 9.8, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', 'description': 'A security regression of CVE-2019-9636 was discovered in python since commit d537ab0ff9767ef024f26246899728f0116b1ec3 affecting versions 2.7, 3.5, 3.6, 3.7 and from v3.8.0a4 through v3.8.0b1, which still allows an attacker to exploit CVE-2019-9636 by abusing the user and password parts of a URL. When an application parses user-supplied URLs to store cookies, authentication credentials, or other kind of information, it is possible for an attacker to provide specially crafted URLs to make the application locate host-related information (e.g. cookies, authentication data) and send them to a different host than where it should, unlike if the URLs had been correctly parsed. The result of an attack may vary based on the application.', 'riskFactors': ['Attack vector: network', 'Critical severity', 'Has fix', 'Attack complexity: low'], 'publishedDate': '2019-06-07T18:29:00Z'}, {'cveId': 'CVE-2018-13347', 'status': 'fixed in 4.6.1', 'severity': 'critical', 'packageName': 'mercurial', 'packageVersion': '4.0', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2018-13347', 'cvss': 9.8, 'vector': 'CVSS:3.0/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', 'description': 'mpatch.c in Mercurial before 4.6.1 mishandles integer addition and subtraction, aka OVE-20180430-0002.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Critical severity', 'Has fix'], 'publishedDate': '2018-07-06T00:29:00Z'}, {'cveId': 'CVE-2018-1000802', 'status': 'fixed in 2.7.16', 'severity': 'critical', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2018-1000802', 'cvss': 9.8, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', 'description': "Python Software Foundation Python (CPython) version 2.7 contains a CWE-77: Improper Neutralization of Special Elements used in a Command (\\'Command Injection\\') vulnerability in shutil module (make_archive function) that can result in Denial of service, Information gain via injection of arbitrary files on the system or entire drive. This attack appear to be exploitable via Passage of unfiltered user input to the function. This vulnerability appears to have been fixed in after commit add531a1e55b0a739b0f42582f1c9747e5649ace.", 'riskFactors': ['Attack vector: network', 'Critical severity', 'DoS', 'Has fix', 'Attack complexity: low'], 'publishedDate': '2018-09-18T17:29:00Z'}, {'cveId': 'CVE-2017-17458', 'status': 'fixed in 4.4.1', 'severity': 'critical', 'packageName': 'mercurial', 'packageVersion': '4.0', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2017-17458', 'cvss': 9.8, 'vector': 'CVSS:3.0/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', 'description': 'In Mercurial before 4.4.1, it is possible that a specially malformed repository can cause Git subrepositories to run arbitrary code in the form of a .git/hooks/post-update script checked into the repository. Typical use of Mercurial prevents construction of such repositories, but they can be created programmatically.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Critical severity', 'Has fix'], 'publishedDate': '2017-12-07T18:29:00Z'}, {'cveId': 'CVE-2017-1000158', 'status': 'fixed in 3.5.5, 3.4.8, 2.7.15', 'severity': 'critical', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2017-1000158', 'cvss': 9.8, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', 'description': 'CPython (aka Python) up to 2.7.13 is vulnerable to an integer overflow in the PyString_DecodeEscape function in stringobject.c, resulting in heap-based buffer overflow (and possible arbitrary code execution)', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Critical severity', 'Has fix'], 'publishedDate': '2017-11-17T05:29:00Z'}, {'cveId': 'CVE-2017-1000116', 'status': 'fixed in 4.3', 'severity': 'critical', 'packageName': 'mercurial', 'packageVersion': '4.0', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2017-1000116', 'cvss': 9.8, 'vector': 'CVSS:3.0/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', 'description': 'Mercurial prior to 4.3 did not adequately sanitize hostnames passed to ssh, leading to possible shell-injection attacks.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Critical severity', 'Has fix'], 'publishedDate': '2017-10-05T01:29:00Z'}, {'cveId': 'CVE-2016-9063', 'status': 'fixed in 3.6.2, 3.5.4, 3.4.7,...', 'severity': 'critical', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2016-9063', 'cvss': 9.8, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', 'description': 'An integer overflow during the parsing of XML using the Expat library. This vulnerability affects Firefox < 50.', 'riskFactors': ['Critical severity', 'Has fix', 'Attack complexity: low', 'Attack vector: network'], 'publishedDate': '2018-06-11T21:29:00Z'}, {'cveId': 'CVE-2016-0718', 'status': 'fixed in 3.6.2, 3.5.4, 3.4.7,...', 'severity': 'critical', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2016-0718', 'cvss': 9.8, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', 'description': 'Expat allows context-dependent attackers to cause a denial of service (crash) or possibly execute arbitrary code via a malformed input document, which triggers a buffer overflow.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Critical severity', 'DoS', 'Has fix'], 'publishedDate': '2016-05-26T16:59:00Z'}, {'cveId': 'CVE-2019-9948', 'status': 'fixed in 3.7.4, 3.6.9, 3.5.8,...', 'severity': 'critical', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2019-9948', 'cvss': 9.1, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', 'description': "urllib in Python 2.x through 2.7.16 supports the local_file: scheme, which makes it easier for remote attackers to bypass protection mechanisms that blacklist file: URIs, as demonstrated by triggering a urllib.urlopen(\\'local_file:///etc/passwd\\') call.", 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Critical severity', 'Has fix'], 'publishedDate': '2019-03-23T18:29:00Z'}, {'cveId': 'CVE-2018-17983', 'status': 'fixed in 4.7.2', 'severity': 'critical', 'packageName': 'mercurial', 'packageVersion': '4.0', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2018-17983', 'cvss': 9.1, 'vector': 'CVSS:3.0/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H', 'description': 'cext/manifest.c in Mercurial before 4.7.2 has an out-of-bounds read during parsing of a malformed manifest entry.', 'riskFactors': ['Attack vector: network', 'Critical severity', 'Has fix', 'Attack complexity: low'], 'publishedDate': '2018-10-04T23:29:00Z'}, {'cveId': 'CVE-2018-1000132', 'status': 'fixed in 4.5.1', 'severity': 'critical', 'packageName': 'mercurial', 'packageVersion': '4.0', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2018-1000132', 'cvss': 9.1, 'vector': 'CVSS:3.0/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', 'description': 'Mercurial version 4.5 and earlier contains a Incorrect Access Control (CWE-285) vulnerability in Protocol server that can result in Unauthorized data access. This attack appear to be exploitable via network connectivity. This vulnerability appears to have been fixed in 4.5.1.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Critical severity', 'Has fix'], 'publishedDate': '2018-03-14T13:29:00Z'}, {'cveId': 'CVE-2021-3918', 'status': 'fixed in 0.4.0', 'severity': 'critical', 'packageName': 'json-schema', 'packageVersion': '0.2.3', 'link': 'https://github.com/advisories/GHSA-896r-f27r-55mw', 'cvss': 9, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', 'description': "json-schema is vulnerable to Improperly Controlled Modification of Object Prototype Attributes (\\'Prototype Pollution\\')", 'riskFactors': ['Recent vulnerability', 'Attack complexity: low', 'Attack vector: network', 'Critical severity', 'Has fix'], 'publishedDate': '2021-11-13T09:15:00Z'}, {'cveId': 'CVE-2017-9462', 'status': 'fixed in 4.1.3', 'severity': 'high', 'packageName': 'mercurial', 'packageVersion': '4.0', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2017-9462', 'cvss': 8.8, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', 'description': 'In Mercurial before 4.1.3, \\"hg serve --stdio\\" allows remote authenticated users to launch the Python debugger, and consequently execute arbitrary code, by using --debugger as a repository name.', 'riskFactors': ['High severity', 'Remote execution', 'Attack complexity: low', 'Attack vector: network', 'Has fix'], 'publishedDate': '2017-06-06T21:29:00Z'}, {'cveId': 'CVE-2021-37701', 'status': 'fixed in 6.1.7, 5.0.8, 4.4.16', 'severity': 'high', 'packageName': 'tar', 'packageVersion': '4.4.13', 'link': 'https://github.com/advisories/GHSA-9r2w-394v-53qc', 'cvss': 8.6, 'vector': 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:H', 'description': 'The npm package \\"tar\\" (aka node-tar) before versions 4.4.16, 5.0.8, and 6.1.7 has an arbitrary file creation/overwrite and arbitrary code execution vulnerability. node-tar aims to guarantee that any file whose location would be modified by a symbolic link is not extracted. This is, in part, achieved by ensuring that extracted directories are not symlinks. Additionally, in order to prevent unnecessary stat calls to determine whether a given path is a directory, paths are cached when directories are created. This logic was insufficient when extracting tar files that contained both a directory and a symlink with the same name as the directory, where the symlink and directory names in the archive entry used backslashes as a path separator on posix systems. The cache checking logic used both `\\\\` and `/` characters as path separators, however `\\\\` is a valid filename character on posix systems. By first creating a directory, and then replacing that directory with a symlink, it was thus possible to bypass node-tar symlink checks on directories, essentially allowing an untrusted tar file to symlink into an arbitrary location and subsequently extracting arbitrary files into that location, thus allowing arbitrary file creation and overwrite. Additionally, a similar confusion could arise on case-insensitive filesystems. If a tar archive contained a directory at `FOO`, followed by a symboli', 'riskFactors': ['Has fix', 'High severity', 'Recent vulnerability', 'Attack complexity: low'], 'publishedDate': '2021-08-31T17:15:00Z'}, {'cveId': 'CVE-2021-32804', 'status': 'fixed in 6.1.1, 5.0.6, 4.4.14,...', 'severity': 'high', 'packageName': 'tar', 'packageVersion': '4.4.13', 'link': 'https://github.com/advisories/GHSA-3jfq-g458-7qm9', 'cvss': 8.1, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:H', 'description': 'The npm package \\"tar\\" (aka node-tar) before versions 6.1.1, 5.0.6, 4.4.14, and 3.3.2 has a arbitrary File Creation/Overwrite vulnerability due to insufficient absolute path sanitization. node-tar aims to prevent extraction of absolute file paths by turning absolute paths into relative paths when the `preservePaths` flag is not set to `true`. This is achieved by stripping the absolute path root from any absolute file paths contained in a tar file. For example `/home/user/.bashrc` would turn into `home/user/.bashrc`. This logic was insufficient when file paths contained repeated path roots such as `////home/user/.bashrc`. `node-tar` would only strip a single path root from such paths. When given an absolute file path with repeating path roots, the resulting path (e.g. `///home/user/.bashrc`) would still resolve to an absolute path, thus allowing arbitrary file creation and overwrite. This issue was addressed in releases 3.2.2, 4.4.14, 5.0.6 and 6.1.1. Users may work around this vulnerability without upgrading by creating a custom `onentry` method which sanitizes the `entry.path` or a `filter` method which removes entries with absolute paths. See referenced GitHub Advisory for details. Be aware of CVE-2021-32803 which fixes a similar bug in later versions of tar.', 'riskFactors': ['Attack vector: network', 'Has fix', 'High severity', 'Recent vulnerability', 'Attack complexity: low'], 'publishedDate': '2021-08-03T19:15:00Z'}, {'cveId': 'CVE-2021-32803', 'status': 'fixed in 6.1.2, 5.0.7, 4.4.15,...', 'severity': 'high', 'packageName': 'tar', 'packageVersion': '4.4.13', 'link': 'https://github.com/advisories/GHSA-r628-mhmh-qjhw', 'cvss': 8.1, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:H', 'description': 'The npm package \\"tar\\" (aka node-tar) before versions 6.1.2, 5.0.7, 4.4.15, and 3.2.3 has an arbitrary File Creation/Overwrite vulnerability via insufficient symlink protection. `node-tar` aims to guarantee that any file whose location would be modified by a symbolic link is not extracted. This is, in part, achieved by ensuring that extracted directories are not symlinks. Additionally, in order to prevent unnecessary `stat` calls to determine whether a given path is a directory, paths are cached when directories are created. This logic was insufficient when extracting tar files that contained both a directory and a symlink with the same name as the directory. This order of operations resulted in the directory being created and added to the `node-tar` directory cache. When a directory is present in the directory cache, subsequent calls to mkdir for that directory are skipped. However, this is also where `node-tar` checks for symlinks occur. By first creating a directory, and then replacing that directory with a symlink, it was thus possible to bypass `node-tar` symlink checks on directories, essentially allowing an untrusted tar file to symlink into an arbitrary location and subsequently extracting arbitrary files into that location, thus allowing arbitrary file creation and overwrite. This issue was addressed in releases 3.2.3, 4.4.15, 5.0.7 and 6.1.2.', 'riskFactors': ['High severity', 'Recent vulnerability', 'Attack complexity: low', 'Attack vector: network', 'Has fix'], 'publishedDate': '2021-08-03T19:15:00Z'}, {'cveId': 'CVE-2016-4472', 'status': 'fixed in 3.6.2, 3.5.4, 3.4.7,...', 'severity': 'high', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2016-4472', 'cvss': 8.1, 'vector': 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', 'description': 'The overflow protection in Expat is removed by compilers with certain optimization settings, which allows remote attackers to cause a denial of service (crash) or possibly execute arbitrary code via crafted XML data.  NOTE: this vulnerability exists because of an incomplete fix for CVE-2015-1283 and CVE-2015-2716.', 'riskFactors': ['Attack vector: network', 'DoS', 'Has fix', 'High severity', 'Remote execution'], 'publishedDate': '2016-06-30T17:59:00Z'}, {'cveId': 'CVE-2017-20052', 'status': 'open', 'severity': 'high', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2017-20052', 'cvss': 7.8, 'vector': 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', 'description': 'A vulnerability classified as problematic was found in Python 2.7.13. This vulnerability affects unknown code of the component pgAdmin4. The manipulation leads to uncontrolled search path. The attack can be initiated remotely. The exploit has been disclosed to the public and may be used.', 'riskFactors': ['High severity', 'Attack complexity: low'], 'publishedDate': '2022-06-16T07:15:00Z'}, {'cveId': 'CVE-2017-20052', 'status': 'open', 'severity': 'high', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2017-20052', 'cvss': 7.8, 'vector': 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', 'description': 'A vulnerability classified as problematic was found in Python 2.7.13. This vulnerability affects unknown code of the component pgAdmin4. The manipulation leads to uncontrolled search path. The attack can be initiated remotely. The exploit has been disclosed to the public and may be used.', 'riskFactors': ['Attack complexity: low', 'High severity'], 'publishedDate': '2022-06-16T07:15:00Z'}, {'cveId': 'PRISMA-2022-0039', 'status': 'fixed in 3.0.5', 'severity': 'high', 'packageName': 'minimatch', 'packageVersion': '3.0.4', 'link': 'https://github.com/isaacs/minimatch/commit/a8763f4388e51956be62dc6025cec1126beeb5e6', 'cvss': 7.5, 'vector': None, 'description': "minimatch package versions before 3.0.5 are vulnerable to Regular Expression Denial of Service (ReDoS). It\\'s possible to cause a denial of service when calling function braceExpand (The regex /\\\\{.*\\\\}/ is vulnerable and can be exploited).", 'riskFactors': ['DoS', 'Has fix', 'High severity'], 'publishedDate': '2022-02-21T09:51:41Z'}, {'cveId': 'CVE-2022-0391', 'status': 'fixed in 3.9.5, 3.8.11, 3.7.11,...', 'severity': 'high', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2022-0391', 'cvss': 7.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', 'description': "A flaw was found in Python, specifically within the urllib.parse module. This module helps break Uniform Resource Locator (URL) strings into components. The issue involves how the urlparse method does not sanitize input and allows characters like \\'\\r\\' and \\'\\n\\' in the URL path. This flaw allows an attacker to input a crafted URL, leading to injection attacks. This flaw affects Python versions prior to 3.10.0b1, 3.9.5, 3.8.11, 3.7.11 and 3.6.14.", 'riskFactors': ['Attack vector: network', 'Has fix', 'High severity', 'Recent vulnerability', 'Attack complexity: low'], 'publishedDate': '2022-02-09T23:15:00Z'}, {'cveId': 'CVE-2021-3807', 'status': 'fixed in 4.1.1', 'severity': 'high', 'packageName': 'ansi-regex', 'packageVersion': '3.0.0', 'link': 'https://github.com/advisories/GHSA-93q8-gq69-wqmw', 'cvss': 7.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', 'description': 'ansi-regex is vulnerable to Inefficient Regular Expression Complexity', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Has fix', 'High severity', 'Recent vulnerability'], 'publishedDate': '2021-09-17T07:15:00Z'}, {'cveId': 'CVE-2021-3807', 'status': 'fixed in 4.1.1', 'severity': 'high', 'packageName': 'ansi-regex', 'packageVersion': '4.1.0', 'link': 'https://github.com/advisories/GHSA-93q8-gq69-wqmw', 'cvss': 7.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', 'description': 'ansi-regex is vulnerable to Inefficient Regular Expression Complexity', 'riskFactors': ['Attack vector: network', 'Has fix', 'High severity', 'Recent vulnerability', 'Attack complexity: low'], 'publishedDate': '2021-09-17T07:15:00Z'}, {'cveId': 'CVE-2021-23343', 'status': 'fixed in 1.0.7', 'severity': 'high', 'packageName': 'path-parse', 'packageVersion': '1.0.6', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2021-23343', 'cvss': 7.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', 'description': 'All versions of package path-parse are vulnerable to Regular Expression Denial of Service (ReDoS) via splitDeviceRe, splitTailRe, and splitPathRe regular expressions. ReDoS exhibits polynomial worst-case time complexity.', 'riskFactors': ['Attack vector: network', 'DoS', 'Has fix', 'High severity', 'Recent vulnerability', 'Attack complexity: low'], 'publishedDate': '2021-05-04T00:00:00Z'}, {'cveId': 'CVE-2019-9674', 'status': 'open', 'severity': 'high', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2019-9674', 'cvss': 7.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', 'description': 'Lib/zipfile.py in Python through 3.7.2 allows remote attackers to cause a denial of service (resource consumption) via a ZIP bomb.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'DoS', 'High severity'], 'publishedDate': '2020-02-04T15:15:00Z'}, {'cveId': 'CVE-2019-5010', 'status': 'fixed in 3.7.3, 3.6.9, 3.5.7,...', 'severity': 'high', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2019-5010', 'cvss': 7.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', 'description': 'An exploitable denial-of-service vulnerability exists in the X509 certificate parser of Python.org Python 2.7.11 / 3.6.6. A specially crafted X509 certificate can cause a NULL pointer dereference, resulting in a denial of service. An attacker can initiate or accept TLS connections using crafted certificates to trigger this vulnerability.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'DoS', 'Has fix', 'High severity'], 'publishedDate': '2019-10-31T21:15:00Z'}, {'cveId': 'CVE-2019-16056', 'status': 'open', 'severity': 'high', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2019-16056', 'cvss': 7.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', 'description': 'An issue was discovered in Python through 2.7.16, 3.x through 3.5.7, 3.6.x through 3.6.9, and 3.7.x through 3.7.4. The email module wrongly parses email addresses that contain multiple @ characters. An application that uses the email module and implements some kind of checks on the From/To headers of a message could be tricked into accepting an email address that should be denied. An attack may be the same as in CVE-2019-11340; however, this CVE applies to Python more generally.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'High severity'], 'publishedDate': '2019-09-06T18:15:00Z'}, {'cveId': 'CVE-2019-15903', 'status': 'fixed in 2.2.8', 'severity': 'high', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2019-15903', 'cvss': 7.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', 'description': 'In libexpat before 2.2.8, crafted XML input could fool the parser into changing from DTD parsing to document parsing too early; a consecutive call to XML_GetCurrentLineNumber (or XML_GetCurrentColumnNumber) then resulted in a heap-based buffer over-read.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Has fix', 'High severity'], 'publishedDate': '2019-09-04T06:15:00Z'}, {'cveId': 'CVE-2018-14647', 'status': 'open', 'severity': 'high', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2018-14647', 'cvss': 7.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', 'description': "Python\\'s elementtree C accelerator failed to initialise Expat\\'s hash salt during initialization. This could make it easy to conduct denial of service attacks against Expat by constructing an XML document that would cause pathological hash collisions in Expat\\'s internal data structures, consuming large amounts CPU and RAM. The vulnerability exists in Python versions 3.7.0, 3.6.0 through 3.6.6, 3.5.0 through 3.5.6, 3.4.0 through 3.4.9, 2.7.0 through 2.7.15.", 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'DoS', 'High severity'], 'publishedDate': '2018-09-25T00:29:00Z'}, {'cveId': 'CVE-2018-13348', 'status': 'fixed in 4.6.1', 'severity': 'high', 'packageName': 'mercurial', 'packageVersion': '4.0', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2018-13348', 'cvss': 7.5, 'vector': 'CVSS:3.0/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', 'description': 'The mpatch_decode function in mpatch.c in Mercurial before 4.6.1 mishandles certain situations where there should be at least 12 bytes remaining after the current position in the patch data, but actually are not, aka OVE-20180430-0001.', 'riskFactors': ['Attack vector: network', 'Has fix', 'High severity', 'Attack complexity: low'], 'publishedDate': '2018-07-06T00:29:00Z'}, {'cveId': 'CVE-2018-13346', 'status': 'fixed in 4.6.1', 'severity': 'high', 'packageName': 'mercurial', 'packageVersion': '4.0', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2018-13346', 'cvss': 7.5, 'vector': 'CVSS:3.0/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', 'description': 'The mpatch_apply function in mpatch.c in Mercurial before 4.6.1 incorrectly proceeds in cases where the fragment start is past the end of the original data, aka OVE-20180430-0004.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Has fix', 'High severity'], 'publishedDate': '2018-07-06T00:29:00Z'}, {'cveId': 'CVE-2018-1061', 'status': 'fixed in 2.7.15', 'severity': 'high', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2018-1061', 'cvss': 7.5, 'vector': 'CVSS:3.0/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', 'description': 'python before versions 2.7.15, 3.4.9, 3.5.6rc1, 3.6.5rc1 and 3.7.0 is vulnerable to catastrophic backtracking in the difflib.IS_LINE_JUNK method.  An attacker could use this flaw to cause denial of service.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'DoS', 'Has fix', 'High severity'], 'publishedDate': '2018-06-19T12:29:00Z'}, {'cveId': 'CVE-2018-1060', 'status': 'fixed in 3.6.5, 3.5.6, 2.7.15', 'severity': 'high', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2018-1060', 'cvss': 7.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', 'description': "python before versions 2.7.15, 3.4.9, 3.5.6rc1, 3.6.5rc1 and 3.7.0 is vulnerable to catastrophic backtracking in pop3lib\\'s apop() method. An attacker could use this flaw to cause denial of service.", 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'DoS', 'Has fix', 'High severity'], 'publishedDate': '2018-06-18T14:29:00Z'}, {'cveId': 'CVE-2017-9233', 'status': 'fixed in 3.6.2, 3.5.4, 3.4.7,...', 'severity': 'high', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2017-9233', 'cvss': 7.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', 'description': 'XML External Entity vulnerability in libexpat 2.2.0 and earlier (Expat XML Parser Library) allows attackers to put the parser in an infinite loop using a malformed external entity definition from an external DTD.', 'riskFactors': ['High severity', 'Attack complexity: low', 'Attack vector: network', 'Has fix'], 'publishedDate': '2017-07-25T20:29:00Z'}, {'cveId': 'CVE-2017-1000115', 'status': 'fixed in 4.3', 'severity': 'high', 'packageName': 'mercurial', 'packageVersion': '4.0', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2017-1000115', 'cvss': 7.5, 'vector': 'CVSS:3.0/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', 'description': 'Mercurial prior to version 4.3 is vulnerable to a missing symlink check that can malicious repositories to modify files outside the repository', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Has fix', 'High severity'], 'publishedDate': '2017-10-05T01:29:00Z'}, {'cveId': 'CVE-2021-37713', 'status': 'fixed in 6.1.9, 5.0.10, 4.4.18', 'severity': 'high', 'packageName': 'tar', 'packageVersion': '4.4.13', 'link': 'https://github.com/advisories/GHSA-5955-9wpr-37jh', 'cvss': 7, 'vector': 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:H', 'description': 'The npm package \\"tar\\" (aka node-tar) before versions 4.4.18, 5.0.10, and 6.1.9 has an arbitrary file creation/overwrite and arbitrary code execution vulnerability. node-tar aims to guarantee that any file whose location would be outside of the extraction target directory is not extracted. This is, in part, accomplished by sanitizing absolute paths of entries within the archive, skipping archive entries that contain `..` path portions, and resolving the sanitized paths against the extraction target directory. This logic was insufficient on Windows systems when extracting tar files that contained a path that was not an absolute path, but specified a drive letter different from the extraction target, such as `C:some\\\\path`. If the drive letter does not match the extraction target, for example `D:\\\\extraction\\\\dir`, then the result of `path.resolve(extractionDirectory, entryPath)` would resolve against the current working directory on the `C:` drive, rather than the extraction target directory. Additionally, a `..` portion of the path could occur immediately after the drive letter, such as `C:../foo`, and was not properly sanitized by the logic that checked for `..` within the normalized and split portions of the path. This only affects users of `node-tar` on Windows systems. These issues were addressed in releases 4.4.18, 5.0.10 and 6.1.9. The v3 branch of node-tar has been deprecate', 'riskFactors': ['Attack complexity: low', 'Has fix', 'High severity', 'Recent vulnerability'], 'publishedDate': '2021-08-31T17:15:00Z'}, {'cveId': 'CVE-2021-37712', 'status': 'fixed in 6.1.9, 5.0.10, 4.4.18', 'severity': 'high', 'packageName': 'tar', 'packageVersion': '4.4.13', 'link': 'https://github.com/advisories/GHSA-qq89-hq3f-393p', 'cvss': 7, 'vector': 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:H', 'description': 'The npm package \\"tar\\" (aka node-tar) before versions 4.4.18, 5.0.10, and 6.1.9 has an arbitrary file creation/overwrite and arbitrary code execution vulnerability. node-tar aims to guarantee that any file whose location would be modified by a symbolic link is not extracted. This is, in part, achieved by ensuring that extracted directories are not symlinks. Additionally, in order to prevent unnecessary stat calls to determine whether a given path is a directory, paths are cached when directories are created. This logic was insufficient when extracting tar files that contained both a directory and a symlink with names containing unicode values that normalized to the same value. Additionally, on Windows systems, long path portions would resolve to the same file system entities as their 8.3 \\"short path\\" counterparts. A specially crafted tar archive could thus include a directory with one form of the path, followed by a symbolic link with a different string that resolves to the same file system entity, followed by a file using the first form. By first creating a directory, and then replacing that directory with a symlink that had a different apparent name that resolved to the same entry in the filesystem, it was thus possible to bypass node-tar symlink checks on directories, essentially allowing an untrusted tar file to symlink into an arbitrary location and subsequently extracting ', 'riskFactors': ['Attack complexity: low', 'Has fix', 'High severity', 'Recent vulnerability'], 'publishedDate': '2021-08-31T17:15:00Z'}, {'cveId': 'CVE-2021-27290', 'status': 'fixed in 8.0.1, 7.1.1, 6.0.2', 'severity': 'high', 'packageName': 'ssri', 'packageVersion': '6.0.1', 'link': 'https://github.com/advisories/GHSA-vx3p-948g-6vhq', 'cvss': 7, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', 'description': 'ssri 5.2.2-8.0.0, fixed in 8.0.1, processes SRIs using a regular expression which is vulnerable to a denial of service. Malicious SRIs could take an extremely long time to process, leading to denial of service. This issue only affects consumers using the strict option.', 'riskFactors': ['Recent vulnerability', 'Attack complexity: low', 'Attack vector: network', 'DoS', 'Has fix', 'High severity'], 'publishedDate': '2021-03-12T22:15:00Z'}, {'cveId': 'CVE-2020-29599', 'status': 'fixed in 8:6.9.10.23+dfsg-2.1+deb10u1', 'severity': 'medium', 'packageName': 'imagemagick', 'packageVersion': '8:6.9.7.4+dfsg-11+deb9u12', 'link': 'https://security-tracker.debian.org/tracker/CVE-2020-29599', 'cvss': 7.8, 'vector': 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', 'description': 'ImageMagick before 6.9.11-40 and 7.x before 7.0.10-40 mishandles the -authenticate option, which allows setting a password for password-protected PDF files. The user-controlled password was not properly escaped/sanitized and it was therefore possible to inject additional shell commands via coders/pdf.c.', 'riskFactors': ['Attack complexity: low', 'Has fix', 'Medium severity'], 'publishedDate': '2020-12-07T20:15:00Z'}, {'cveId': 'CVE-2021-3733', 'status': 'fixed in 3.9.5, 3.8.10, 3.7.11,...', 'severity': 'medium', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2021-3733', 'cvss': 6.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', 'description': "There\\'s a flaw in urllib\\'s AbstractBasicAuthHandler class. An attacker who controls a malicious HTTP server that an HTTP client (such as web browser) connects to, could trigger a Regular Expression Denial of Service (ReDOS) during an authentication request with a specially crafted payload that is sent by the server to the client. The greatest threat that this flaw poses is to application availability.", 'riskFactors': ['Medium severity', 'Recent vulnerability', 'Attack complexity: low', 'Attack vector: network', 'DoS', 'Has fix'], 'publishedDate': '2022-03-10T17:42:00Z'}, {'cveId': 'CVE-2020-8492', 'status': 'open', 'severity': 'medium', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2020-8492', 'cvss': 6.5, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', 'description': 'Python 2.7 through 2.7.17, 3.5 through 3.5.9, 3.6 through 3.6.10, 3.7 through 3.7.6, and 3.8 through 3.8.1 allows an HTTP server to conduct Regular Expression Denial of Service (ReDoS) attacks against a client because of urllib.request.AbstractBasicAuthHandler catastrophic backtracking.', 'riskFactors': ['DoS', 'Medium severity', 'Attack complexity: low', 'Attack vector: network'], 'publishedDate': '2020-01-30T19:15:00Z'}, {'cveId': 'CVE-2019-9947', 'status': 'open', 'severity': 'medium', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2019-9947', 'cvss': 6.1, 'vector': 'CVSS:3.0/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', 'description': 'An issue was discovered in urllib2 in Python 2.x through 2.7.16 and urllib in Python 3.x through 3.7.3. CRLF injection is possible if the attacker controls a url parameter, as demonstrated by the first argument to urllib.request.urlopen with \\r\\n (specifically in the path component of a URL that lacks a ? character) followed by an HTTP header or a Redis command. This is similar to the CVE-2019-9740 query string issue. This is fixed in: v2.7.17, v2.7.17rc1, v2.7.18, v2.7.18rc1; v3.5.10, v3.5.10rc1, v3.5.8, v3.5.8rc1, v3.5.8rc2, v3.5.9; v3.6.10, v3.6.10rc1, v3.6.11, v3.6.11rc1, v3.6.12, v3.6.9, v3.6.9rc1; v3.7.4, v3.7.4rc1, v3.7.4rc2, v3.7.5, v3.7.5rc1, v3.7.6, v3.7.6rc1, v3.7.7, v3.7.7rc1, v3.7.8, v3.7.8rc1, v3.7.9.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Medium severity'], 'publishedDate': '2019-03-23T18:29:00Z'}, {'cveId': 'CVE-2019-9740', 'status': 'open', 'severity': 'medium', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2019-9740', 'cvss': 6.1, 'vector': 'CVSS:3.0/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', 'description': 'An issue was discovered in urllib2 in Python 2.x through 2.7.16 and urllib in Python 3.x through 3.7.3. CRLF injection is possible if the attacker controls a url parameter, as demonstrated by the first argument to urllib.request.urlopen with \\r\\n (specifically in the query string after a ? character) followed by an HTTP header or a Redis command. This is fixed in: v2.7.17, v2.7.17rc1, v2.7.18, v2.7.18rc1; v3.5.10, v3.5.10rc1, v3.5.8, v3.5.8rc1, v3.5.8rc2, v3.5.9; v3.6.10, v3.6.10rc1, v3.6.11, v3.6.11rc1, v3.6.12, v3.6.9, v3.6.9rc1; v3.7.4, v3.7.4rc1, v3.7.4rc2, v3.7.5, v3.7.5rc1, v3.7.6, v3.7.6rc1, v3.7.7, v3.7.7rc1, v3.7.8, v3.7.8rc1, v3.7.9.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Medium severity'], 'publishedDate': '2019-03-13T03:29:00Z'}, {'cveId': 'CVE-2019-18348', 'status': 'fixed in 3.8.3, 3.7.8, 3.6.11', 'severity': 'medium', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2019-18348', 'cvss': 6.1, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', 'description': 'An issue was discovered in urllib2 in Python 2.x through 2.7.17 and urllib in Python 3.x through 3.8.0. CRLF injection is possible if the attacker controls a url parameter, as demonstrated by the first argument to urllib.request.urlopen with \\r\\n (specifically in the host component of a URL) followed by an HTTP header. This is similar to the CVE-2019-9740 query string issue and the CVE-2019-9947 path string issue. (This is not exploitable when glibc has CVE-2016-10739 fixed.). This is fixed in: v2.7.18, v2.7.18rc1; v3.5.10, v3.5.10rc1; v3.6.11, v3.6.11rc1, v3.6.12; v3.7.8, v3.7.8rc1, v3.7.9; v3.8.3, v3.8.3rc1, v3.8.4, v3.8.4rc1, v3.8.5, v3.8.6, v3.8.6rc1.', 'riskFactors': ['Attack vector: network', 'Has fix', 'Medium severity', 'Attack complexity: low'], 'publishedDate': '2019-10-23T17:15:00Z'}, {'cveId': 'CVE-2019-16935', 'status': 'open', 'severity': 'medium', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2019-16935', 'cvss': 6.1, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', 'description': 'The documentation XML-RPC server in Python through 2.7.16, 3.x through 3.6.9, and 3.7.x through 3.7.4 has XSS via the server_title field. This occurs in Lib/DocXMLRPCServer.py in Python 2.x, and in Lib/xmlrpc/server.py in Python 3.x. If set_server_title is called with untrusted input, arbitrary JavaScript can be delivered to clients that visit the http URL for this server.', 'riskFactors': ['Attack vector: network', 'Medium severity', 'Attack complexity: low'], 'publishedDate': '2019-09-28T02:15:00Z'}, {'cveId': 'CVE-2021-23336', 'status': 'fixed in 3.9.2, 3.8.8, 3.7.10,...', 'severity': 'medium', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2021-23336', 'cvss': 5.9, 'vector': 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:L/A:H', 'description': 'The package python/cpython from 0 and before 3.6.13, from 3.7.0 and before 3.7.10, from 3.8.0 and before 3.8.8, from 3.9.0 and before 3.9.2 are vulnerable to Web Cache Poisoning via urllib.parse.parse_qsl and urllib.parse.parse_qs by using a vector called parameter cloaking. When the attacker can separate query parameters using a semicolon (;), they can cause a difference in the interpretation of the request between the proxy (running with default configuration) and the server. This can result in malicious requests being cached as completely safe ones, as the proxy would usually not see the semicolon as a separator, and therefore would not include it in a cache key of an unkeyed parameter.', 'riskFactors': ['Attack vector: network', 'Has fix', 'Medium severity', 'Recent vulnerability'], 'publishedDate': '2021-02-15T13:15:00Z'}, {'cveId': 'CVE-2019-3902', 'status': 'fixed in 4.9', 'severity': 'medium', 'packageName': 'mercurial', 'packageVersion': '4.0', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2019-3902', 'cvss': 5.9, 'vector': 'CVSS:3.0/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', 'description': "A flaw was found in Mercurial before 4.9. It was possible to use symlinks and subrepositories to defeat Mercurial\\'s path-checking logic and write files outside a repository.", 'riskFactors': ['Attack vector: network', 'Has fix', 'Medium severity'], 'publishedDate': '2019-04-22T16:29:00Z'}, {'cveId': 'CVE-2021-3426', 'status': 'fixed in 3.10.0a7, 3.9.3, 3.8.9', 'severity': 'medium', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2021-3426', 'cvss': 5.7, 'vector': 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', 'description': "There\\'s a flaw in Python 3\\'s pydoc. A local or adjacent attacker who discovers or is able to convince another local or adjacent user to start a pydoc server could access the server and use it to disclose sensitive information belonging to the other user that they would not normally be able to access. The highest risk of this flaw is to data confidentiality. This flaw affects Python versions before 3.8.9, Python versions before 3.9.3 and Python versions before 3.10.0a7.", 'riskFactors': ['Attack complexity: low', 'Has fix', 'Medium severity', 'Recent vulnerability'], 'publishedDate': '2021-05-20T13:15:00Z'}, {'cveId': 'CVE-2022-33987', 'status': 'fixed in 12.1.0', 'severity': 'medium', 'packageName': 'got', 'packageVersion': '6.7.1', 'link': 'https://github.com/advisories/GHSA-pfrx-2q88-qq97', 'cvss': 5.3, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', 'description': 'The got package before 12.1.0 (also fixed in 11.8.5) for Node.js allows a redirect to a UNIX socket.', 'riskFactors': ['Medium severity', 'Recent vulnerability', 'Attack complexity: low', 'Attack vector: network', 'Has fix'], 'publishedDate': '2022-06-18T21:15:00Z'}, {'cveId': 'CVE-2018-20852', 'status': 'fixed in 3.6.9, 3.5.7, 3.4.10', 'severity': 'medium', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2018-20852', 'cvss': 5.3, 'vector': 'CVSS:3.0/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', 'description': 'http.cookiejar.DefaultPolicy.domain_return_ok in Lib/http/cookiejar.py in Python before 3.7.3 does not correctly validate the domain: it can be tricked into sending existing cookies to the wrong server. An attacker may abuse this flaw by using a server with a hostname that has another valid hostname as a suffix (e.g., pythonicexample.com to steal cookies for example.com). When a program uses http.cookiejar.DefaultPolicy and tries to do an HTTP connection to an attacker-controlled server, existing cookies can be leaked to the attacker. This affects 2.x through 2.7.16, 3.x before 3.4.10, 3.5.x before 3.5.7, 3.6.x before 3.6.9, and 3.7.x before 3.7.3.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'Has fix', 'Medium severity'], 'publishedDate': '2019-07-13T21:15:00Z'}, {'cveId': 'CVE-2021-23362', 'status': 'fixed in 2.8.9, 3.0.8', 'severity': 'moderate', 'packageName': 'hosted-git-info', 'packageVersion': '2.8.8', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2021-23362', 'cvss': 4, 'vector': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', 'description': 'The package hosted-git-info before 3.0.8 are vulnerable to Regular Expression Denial of Service (ReDoS) via regular expression shortcutMatch in the fromUrl function in index.js. The affected regular expression exhibits polynomial worst-case time complexity.', 'riskFactors': ['Attack complexity: low', 'Attack vector: network', 'DoS', 'Has fix', 'Medium severity', 'Recent vulnerability'], 'publishedDate': '2021-03-23T17:15:00Z'}, {'cveId': 'CVE-2018-1000030', 'status': 'open', 'severity': 'low', 'packageName': 'python', 'packageVersion': '2.7.13', 'link': 'https://nvd.nist.gov/vuln/detail/CVE-2018-1000030', 'cvss': 3.6, 'vector': 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:L', 'description': "Python 2.7.14 is vulnerable to a Heap-Buffer-Overflow as well as a Heap-Use-After-Free. Python versions prior to 2.7.14 may also be vulnerable and it appears that Python 2.7.17 and prior may also be vulnerable however this has not been confirmed. The vulnerability lies when multiply threads are handling large amounts of data. In both cases there is essentially a race condition that occurs. For the Heap-Buffer-Overflow, Thread 2 is creating the size for a buffer, but Thread1 is already writing to the buffer without knowing how much to write. So when a large amount of data is being processed, it is very easy to cause memory corruption using a Heap-Buffer-Overflow. As for the Use-After-Free, Thread3->Malloc->Thread1->Free\\'s->Thread2-Re-uses-Free\\'d Memory. The PSRT has stated that this is not a security vulnerability due to the fact that the attacker must be able to run code, however in some situations, such as function as a service, this vulnerability can potentially be used by an attacker to violate a trust boundary, as such the DWF feels this issue deserves a CVE.", 'riskFactors': None, 'publishedDate': '2018-02-08T17:29:00Z'}, {'cveId': 'PRISMA-2021-0096', 'status': 'fixed in 6.1.4', 'severity': 'low', 'packageName': 'tar', 'packageVersion': '4.4.13', 'link': 'https://github.com/npm/node-tar/commit/06cbde5935aa7643f578f874de84a7da2a74fe3a', 'cvss': 3.5, 'vector': None, 'description': 'tar package versions before 6.1.4 are vulnerable to Regular Expression Denial of Service (ReDoS). When stripping the trailing slash from `files` arguments, we were using `f.replace(/\\\\/+$/, \\\'\\\')`, which can get exponentially slow when `f` contains many `/` characters. This is \\"\\"unlikely but theoretically possible\\"\\" because it requires that the user is passing untrusted input into the `tar.extract()` or `tar.list()` array of entries to parse/extract, which would be quite unusual.', 'riskFactors': ['DoS', 'Has fix'], 'publishedDate': '2021-08-30T08:44:48Z'}], 'packages': [{'type': 'os', 'name': 'iproute2', 'version': '4.9.0-1+deb9u1', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'base-files', 'version': '9.9+deb9u13', 'licenses': ['GPL']}, {'type': 'os', 'name': 'dh-python', 'version': '2.20170125', 'licenses': []}, {'type': 'os', 'name': 'gzip', 'version': '1.6-5', 'licenses': ['GPL']}, {'type': 'os', 'name': 'findutils', 'version': '4.6.0+git+20161106-2', 'licenses': ['GPL-3']}, {'type': 'os', 'name': 'diffutils', 'version': '1:3.5-3', 'licenses': ['GPL']}, {'type': 'os', 'name': 'debian-archive-keyring', 'version': '2017.5+deb9u1', 'licenses': ['GPL']}, {'type': 'os', 'name': 'grep', 'version': '2.27-2', 'licenses': ['GPL-3+']}, {'type': 'os', 'name': 'm4', 'version': '1.4.18-1', 'licenses': ['GPL']}, {'type': 'os', 'name': 'coreutils', 'version': '8.26-3', 'licenses': ['GPL-3']}, {'type': 'os', 'name': 'ucf', 'version': '3.0036', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'netbase', 'version': '5.4', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'wget', 'version': '1.18-5+deb9u3', 'licenses': ['GPL-3']}, {'type': 'os', 'name': 'sed', 'version': '4.4-1', 'licenses': ['GPL-3']}, {'type': 'os', 'name': 'debianutils', 'version': '4.8.1.1', 'licenses': ['GPL']}, {'type': 'os', 'name': 'autoconf', 'version': '2.69-10', 'licenses': ['GPL-3+']}, {'type': 'os', 'name': 'sensible-utils', 'version': '0.0.9+deb9u1', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'patch', 'version': '2.7.5-1+deb9u2', 'licenses': ['GPL']}, {'type': 'os', 'name': 'debconf', 'version': '1.5.61', 'licenses': ['BSD-2-clause']}, {'type': 'os', 'name': 'autotools-dev', 'version': '20161112.1', 'licenses': ['GPL']}, {'type': 'os', 'name': 'binutils', 'version': '2.28-5', 'licenses': ['GPL']}, {'type': 'os', 'name': 'adduser', 'version': '3.115', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'pkg-config', 'version': '0.29-4', 'licenses': ['GPL']}, {'type': 'os', 'name': 'libidn2-0', 'version': '0.16-1+deb9u1', 'licenses': ['GPL-3+']}, {'type': 'os', 'name': 'base-passwd', 'version': '3.5.43', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'tzdata', 'version': '2021a-0+deb9u1', 'licenses': []}, {'type': 'os', 'name': 'liberror-perl', 'version': '0.17024-1', 'licenses': ['Artistic or GPL-1+']}, {'type': 'os', 'name': 'unzip', 'version': '6.0-21+deb9u2', 'licenses': []}, {'type': 'os', 'name': 'init-system-helpers', 'version': '1.48', 'licenses': ['BSD-3-clause']}, {'type': 'os', 'name': 'libtasn1-6', 'version': '4.10-1.1+deb9u1', 'licenses': ['LGPL']}, {'type': 'os', 'name': 'hostname', 'version': '3.18', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'mawk', 'version': '1.3.3-17', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'libcap2', 'version': '1:2.25-1', 'licenses': ['BSD-3-clause or GPL-2']}, {'type': 'os', 'name': 'hicolor-icon-theme', 'version': '0.15-1', 'licenses': ['GPL']}, {'type': 'os', 'name': 'bash', 'version': '4.4-5', 'licenses': ['GPL-3']}, {'type': 'os', 'name': 'tar', 'version': '1.29b-1.1', 'licenses': ['GPL-3']}, {'type': 'os', 'name': 'shared-mime-info', 'version': '1.8-1+deb9u1', 'licenses': ['GPL']}, {'type': 'os', 'name': 'libgcrypt20', 'version': '1.7.6-2+deb9u3', 'licenses': ['LGPL']}, {'type': 'os', 'name': 'xorg-sgml-doctools', 'version': '1:1.11-1', 'licenses': []}, {'type': 'os', 'name': 'mime-support', 'version': '3.60', 'licenses': []}, {'type': 'os', 'name': 'dash', 'version': '0.5.8-2.4', 'licenses': ['GPL']}, {'type': 'os', 'name': 'gnome-icon-theme', 'version': '3.12.0-2', 'licenses': ['LGPL-3']}, {'type': 'os', 'name': 'ca-certificates', 'version': '20200601~deb9u2', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'sqlite3', 'version': '3.16.2-5+deb9u3', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'x11proto-core', 'version': '7.0.31-1', 'licenses': []}, {'type': 'os', 'name': 'libtool', 'version': '2.4.6-2', 'licenses': ['GPL']}, {'type': 'os', 'name': 'gcc-6', 'version': '6.3.0-18+deb9u1', 'licenses': ['GPL']}, {'type': 'os', 'name': 'imagemagick', 'version': '8:6.9.7.4+dfsg-11+deb9u12', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'apr-util', 'version': '1.5.4-3', 'licenses': ['Apache-2.0']}, {'type': 'os', 'name': 'configobj', 'version': '5.0.6-2', 'licenses': ['BSD-3-clause']}, {'type': 'os', 'name': 'gdk-pixbuf', 'version': '2.36.5-2+deb9u2', 'licenses': ['LGPL-2']}, {'type': 'os', 'name': 'libxpm', 'version': '1:3.5.12-1', 'licenses': []}, {'type': 'os', 'name': 'lsb', 'version': '9.20161125', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'x11proto-kb', 'version': '1.0.7-1', 'licenses': []}, {'type': 'os', 'name': 'fonts-dejavu', 'version': '2.37-1', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'util-linux', 'version': '2.29.2-1+deb9u1', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'procps', 'version': '2:3.3.12-3+deb9u1', 'licenses': ['LGPL-2.1+']}, {'type': 'os', 'name': 'dpkg', 'version': '1.18.25', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'mpclib3', 'version': '1.0.3-1', 'licenses': ['LGPL-2.1']}, {'type': 'os', 'name': 'libbsd', 'version': '0.8.3-1+deb9u1', 'licenses': ['BSD-3-clause']}, {'type': 'os', 'name': 'sysvinit', 'version': '2.88dsf-59.9', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'openssh', 'version': '1:7.4p1-10+deb9u7', 'licenses': ['BSD-3-clause']}, {'type': 'os', 'name': 'isl', 'version': '0.18-1', 'licenses': ['MIT']}, {'type': 'os', 'name': 'libpsl', 'version': '0.17.0-3', 'licenses': ['MIT']}, {'type': 'os', 'name': 'cairo', 'version': '1.14.8-1+deb9u1', 'licenses': ['LGPL-2.1']}, {'type': 'os', 'name': 'cdebconf', 'version': '0.227', 'licenses': []}, {'type': 'os', 'name': 'libxau', 'version': '1:1.0.8-1', 'licenses': []}, {'type': 'os', 'name': 'apr', 'version': '1.5.2-5', 'licenses': ['Apache-2.0']}, {'type': 'os', 'name': 'xz-utils', 'version': '5.2.2-1.2', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'libcap-ng', 'version': '0.7.7-3', 'licenses': ['LGPL-2.1']}, {'type': 'os', 'name': 'djvulibre', 'version': '3.5.27.1-7', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'subversion', 'version': '1.9.5-1+deb9u6', 'licenses': ['Apache-2.0']}, {'type': 'os', 'name': 'acl', 'version': '2.2.52-3', 'licenses': ['LGPL-2.1']}, {'type': 'os', 'name': 'x11proto-input', 'version': '2.3.2-1', 'licenses': []}, {'type': 'os', 'name': 'harfbuzz', 'version': '1.4.2-1', 'licenses': ['MIT']}, {'type': 'os', 'name': 'libsemanage', 'version': '2.6-2', 'licenses': ['LGPL']}, {'type': 'os', 'name': 'libxdamage', 'version': '1:1.1.4-2', 'licenses': []}, {'type': 'os', 'name': 'libksba', 'version': '1.3.5-2', 'licenses': ['GPL-3']}, {'type': 'os', 'name': 'mariadb-10.1', 'version': '10.1.48-0+deb9u2', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'gnutls28', 'version': '3.5.8-5+deb9u5', 'licenses': ['LGPL']}, {'type': 'os', 'name': 'postgresql-9.6', 'version': '9.6.20-0+deb9u1', 'licenses': ['GPL-1']}, {'type': 'os', 'name': 'libsm', 'version': '2:1.2.2-1', 'licenses': []}, {'type': 'os', 'name': 'libxext', 'version': '2:1.3.3-1', 'licenses': []}, {'type': 'os', 'name': 'libxml2', 'version': '2.9.4+dfsg1-2.2+deb9u4', 'licenses': []}, {'type': 'os', 'name': 'openssl1.0', 'version': '1.0.2u-1~deb9u4', 'licenses': []}, {'type': 'os', 'name': 'libxdmcp', 'version': '1:1.1.2-3', 'licenses': []}, {'type': 'os', 'name': 'libthai', 'version': '0.1.26-1', 'licenses': ['LGPL-2.1+']}, {'type': 'os', 'name': 'libunistring', 'version': '0.9.6+really0.9.3-0.1', 'licenses': ['LGPL-3+']}, {'type': 'os', 'name': 'python3-defaults', 'version': '3.5.3-1', 'licenses': []}, {'type': 'os', 'name': 'gtk+2.0', 'version': '2.24.31-2', 'licenses': ['LGPL-2']}, {'type': 'os', 'name': 'libexif', 'version': '0.6.21-2+deb9u5', 'licenses': ['LGPL-2.1+']}, {'type': 'os', 'name': 'libxfixes', 'version': '1:5.0.3-1', 'licenses': []}, {'type': 'os', 'name': 'systemd', 'version': '232-25+deb9u12', 'licenses': ['LGPL-2.1+']}, {'type': 'os', 'name': 'gcc-defaults', 'version': '1.168', 'licenses': ['GPL']}, {'type': 'os', 'name': 'gnupg2', 'version': '2.1.18-8~deb9u4', 'licenses': ['GPL-3+']}, {'type': 'os', 'name': 'curl', 'version': '7.52.1-5+deb9u13', 'licenses': ['BSD-3-Clause']}, {'type': 'os', 'name': 'libevent', 'version': '2.0.21-stable-3', 'licenses': []}, {'type': 'os', 'name': 'ncurses', 'version': '6.0+20161126-1+deb9u2', 'licenses': []}, {'type': 'os', 'name': 'libdatrie', 'version': '0.2.10-4', 'licenses': ['LGPL-2.1+']}, {'type': 'os', 'name': 'six', 'version': '1.10.0-3', 'licenses': []}, {'type': 'os', 'name': 'libxcursor', 'version': '1:1.1.14-1+deb9u2', 'licenses': []}, {'type': 'os', 'name': 'xtrans', 'version': '1.3.5-1', 'licenses': []}, {'type': 'os', 'name': 'e2fsprogs', 'version': '1.43.4-2+deb9u2', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'atk1.0', 'version': '2.22.0-1', 'licenses': ['LGPL-2']}, {'type': 'os', 'name': 'shadow', 'version': '1:4.4-4.1+deb9u1', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'gtk+3.0', 'version': '3.22.11-1', 'licenses': ['LGPL-2+']}, {'type': 'os', 'name': 'libxrandr', 'version': '2:1.5.1-1', 'licenses': []}, {'type': 'os', 'name': 'libyaml', 'version': '0.1.7-2', 'licenses': []}, {'type': 'os', 'name': 'liblqr', 'version': '0.4.2-2', 'licenses': ['LGPL-3']}, {'type': 'os', 'name': 'python-defaults', 'version': '2.7.13-2', 'licenses': []}, {'type': 'os', 'name': 'librsvg', 'version': '2.40.21-0+deb9u1', 'licenses': ['LGPL-2+']}, {'type': 'os', 'name': 'libxinerama', 'version': '2:1.1.3-1', 'licenses': []}, {'type': 'os', 'name': 'python3.5', 'version': '3.5.3-1+deb9u4', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'libxt', 'version': '1:1.1.5-1', 'licenses': []}, {'type': 'os', 'name': 'gobject-introspection', 'version': '1.50.0-1', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'mpfr4', 'version': '3.1.5-1', 'licenses': ['LGPL-3']}, {'type': 'os', 'name': 'mpdecimal', 'version': '2.4.2-1', 'licenses': ['BSD']}, {'type': 'os', 'name': 'libpthread-stubs', 'version': '0.3-4', 'licenses': []}, {'type': 'os', 'name': 'openssl', 'version': '1.1.0l-1~deb9u3', 'licenses': []}, {'type': 'os', 'name': 'libjpeg-turbo', 'version': '1:1.5.1-2+deb9u1', 'licenses': ['BSD-BY-LC-NE']}, {'type': 'os', 'name': 'expat', 'version': '2.2.0-2+deb9u3', 'licenses': []}, {'type': 'os', 'name': 'openldap', 'version': '2.4.44+dfsg-5+deb9u8', 'licenses': []}, {'type': 'os', 'name': 'libmnl', 'version': '1.0.4-2', 'licenses': ['LGPL-2.1']}, {'type': 'os', 'name': 'jquery', 'version': '3.1.1-2+deb9u2', 'licenses': ['GPL-2 or MIT']}, {'type': 'os', 'name': 'elfutils', 'version': '0.168-1', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'pcre3', 'version': '2:8.39-3', 'licenses': []}, {'type': 'os', 'name': 'attr', 'version': '1:2.4.47-2', 'licenses': ['LGPL-2.1']}, {'type': 'os', 'name': 'ilmbase', 'version': '2.2.0-12', 'licenses': []}, {'type': 'os', 'name': 'python2.7', 'version': '2.7.13-2+deb9u5', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'gts', 'version': '0.7.6+darcs121130-4', 'licenses': ['LGPL-2+']}, {'type': 'os', 'name': 'libmaxminddb', 'version': '1.2.0-1+deb9u1', 'licenses': ['Apache-2.0']}, {'type': 'os', 'name': 'libssh2', 'version': '1.7.0-1+deb9u1', 'licenses': ['BSD3']}, {'type': 'os', 'name': 'lz4', 'version': '0.0~r131-2', 'licenses': ['BSD-2-clause']}, {'type': 'os', 'name': 'keyutils', 'version': '1.5.9-9', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'libcroco', 'version': '0.6.11-3', 'licenses': ['LGPL']}, {'type': 'os', 'name': 'explorercanvas', 'version': '0.r3-4', 'licenses': ['Apache-2.0']}, {'type': 'os', 'name': 'git', 'version': '1:2.11.0-3+deb9u7', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'avahi', 'version': '0.6.32-2', 'licenses': ['LGPL-2.1']}, {'type': 'os', 'name': 'openjpeg2', 'version': '2.1.2-1.1+deb9u6', 'licenses': ['BSD-2']}, {'type': 'os', 'name': 'fftw3', 'version': '3.3.5-3', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'glib2.0', 'version': '2.50.3-2+deb9u2', 'licenses': ['LGPL']}, {'type': 'os', 'name': 'libx11', 'version': '2:1.6.4-3+deb9u3', 'licenses': []}, {'type': 'os', 'name': 'x11proto-render', 'version': '2:0.11.1-2', 'licenses': []}, {'type': 'os', 'name': 'x11proto-xext', 'version': '7.3.0-1', 'licenses': []}, {'type': 'os', 'name': 'db5.3', 'version': '5.3.28-12+deb9u1', 'licenses': []}, {'type': 'os', 'name': 'p11-kit', 'version': '0.23.3-2+deb9u1', 'licenses': ['BSD-3-Clause']}, {'type': 'os', 'name': 'libpng1.6', 'version': '1.6.28-1+deb9u1', 'licenses': ['GPL-2+ or BSD-like-with-advertising-clause']}, {'type': 'os', 'name': 'libxcb', 'version': '1.12-1', 'licenses': []}, {'type': 'os', 'name': 'libxslt', 'version': '1.1.29-2.1+deb9u2', 'licenses': []}, {'type': 'os', 'name': 'npth', 'version': '1.3-1', 'licenses': ['GPL-2+ or LGPL-3+']}, {'type': 'os', 'name': 'libsepol', 'version': '2.6-2', 'licenses': ['LGPL']}, {'type': 'os', 'name': 'automake-1.15', 'version': '1:1.15-6', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'gmp', 'version': '2:6.1.2+dfsg-1', 'licenses': ['LGPL-3']}, {'type': 'os', 'name': 'rtmpdump', 'version': '2.4+20151223.gitfa8646d.1-1', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'lcms2', 'version': '2.8-4+deb9u1', 'licenses': ['MIT']}, {'type': 'os', 'name': 'openexr', 'version': '2.2.0-11+deb9u2', 'licenses': []}, {'type': 'os', 'name': 'apt', 'version': '1.4.11', 'licenses': ['GPLv2+']}, {'type': 'os', 'name': 'iputils', 'version': '3:20161105-1', 'licenses': ['GPL']}, {'type': 'os', 'name': 'libice', 'version': '2:1.0.9-2', 'licenses': []}, {'type': 'os', 'name': 'xorg', 'version': '1:7.7+19', 'licenses': ['GPL']}, {'type': 'os', 'name': 'libselinux', 'version': '2.6-3', 'licenses': ['LGPL-2.1']}, {'type': 'os', 'name': 'freetype', 'version': '2.6.3-3.2+deb9u2', 'licenses': ['GPL-2+ or FTL']}, {'type': 'os', 'name': 'libwmf', 'version': '0.2.8.4-10.6', 'licenses': ['LGPL-2']}, {'type': 'os', 'name': 'dbus', 'version': '1.10.32-0+deb9u1', 'licenses': ['GPL-2+ or AFL-2.1']}, {'type': 'os', 'name': 'libffi', 'version': '3.2.1-6', 'licenses': ['GPL']}, {'type': 'os', 'name': 'serf', 'version': '1.3.9-3+deb9u1', 'licenses': ['Apache']}, {'type': 'os', 'name': 'audit', 'version': '1:2.6.7-2', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'pixman', 'version': '0.34.0-1', 'licenses': []}, {'type': 'os', 'name': 'krb5', 'version': '1.15-1+deb9u2', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'readline', 'version': '7.0-3', 'licenses': ['GPL-3']}, {'type': 'os', 'name': 'libgd2', 'version': '2.2.4-2+deb9u5', 'licenses': ['GPL-2+ with Autoconf exception']}, {'type': 'os', 'name': 'ustr', 'version': '1.0.4-6', 'licenses': ['LGPL-2+ or BSD-2-clause or MIT']}, {'type': 'os', 'name': 'libedit', 'version': '3.1-20160903-3', 'licenses': []}, {'type': 'os', 'name': 'cyrus-sasl2', 'version': '2.1.27~101-g0780600+dfsg-3+deb9u1', 'licenses': ['BSD-4-clause']}, {'type': 'os', 'name': 'pango1.0', 'version': '1.40.5-1', 'licenses': ['LGPL-2']}, {'type': 'os', 'name': 'pinentry', 'version': '1.0.0-2', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'libidn', 'version': '1.33-1+deb9u1', 'licenses': ['GPL-3+']}, {'type': 'os', 'name': 'tiff', 'version': '4.0.8-2+deb9u5', 'licenses': []}, {'type': 'os', 'name': 'fontconfig', 'version': '2.11.0-6.7', 'licenses': []}, {'type': 'os', 'name': 'libxrender', 'version': '1:0.9.10-1', 'licenses': []}, {'type': 'os', 'name': 'bzip2', 'version': '1.0.6-8.1', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'gdbm', 'version': '1.8.3-14', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'bzr', 'version': '2.7.0+bzr6619-7+deb9u1', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'icu', 'version': '57.1-6+deb9u4', 'licenses': []}, {'type': 'os', 'name': 'pam', 'version': '1.1.8-3.6', 'licenses': ['GPL']}, {'type': 'os', 'name': 'mercurial', 'version': '4.0-1+deb9u2', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'zlib', 'version': '1:1.2.8.dfsg-5', 'licenses': []}, {'type': 'os', 'name': 'libgpg-error', 'version': '1.26-2', 'licenses': ['GPL-2.1+']}, {'type': 'os', 'name': 'libxi', 'version': '2:1.7.9-1', 'licenses': []}, {'type': 'os', 'name': 'nghttp2', 'version': '1.18.1-1+deb9u1', 'licenses': ['MIT']}, {'type': 'os', 'name': 'cups', 'version': '2.2.1-8+deb9u6', 'licenses': ['GPL-2.0 with AOSDL exception']}, {'type': 'os', 'name': 'jbigkit', 'version': '2.1-3.1', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'perl', 'version': '5.24.1-3+deb9u7', 'licenses': ['GPL-1+ or Artistic']}, {'type': 'os', 'name': 'db-defaults', 'version': '5.3.1', 'licenses': ['GPL-3']}, {'type': 'os', 'name': 'libxcomposite', 'version': '1:0.4.4-2', 'licenses': []}, {'type': 'os', 'name': 'graphite2', 'version': '1.3.10-1', 'licenses': ['LGPL-2+']}, {'type': 'os', 'name': 'glibc', 'version': '2.24-11+deb9u4', 'licenses': ['LGPL-2.1']}, {'type': 'os', 'name': 'file', 'version': '1:5.30-1+deb9u3', 'licenses': ['BSD-2-Clause-alike']}, {'type': 'os', 'name': 'nettle', 'version': '3.3-1', 'licenses': ['LGPL-2.1+']}, {'type': 'os', 'name': 'libwebp', 'version': '0.5.2-1', 'licenses': ['Apache-2.0']}, {'type': 'os', 'name': 'graphviz', 'version': '2.38.0-17', 'licenses': ['MIT']}, {'type': 'os', 'name': 'libassuan', 'version': '2.4.3-2', 'licenses': ['LGPL-2.1+']}, {'type': 'os', 'name': 'mysql-defaults', 'version': '1.0.2', 'licenses': ['GPL-2+']}, {'type': 'os', 'name': 'libsigsegv', 'version': '2.10-5', 'licenses': ['GPL']}, {'type': 'os', 'name': 'lzo2', 'version': '2.08-1.2', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'linux', 'version': '4.9.258-1', 'licenses': ['GPL-2']}, {'type': 'os', 'name': 'make-dfsg', 'version': '4.1-9.1', 'licenses': ['GPL-3+']}, {'type': 'nodejs', 'name': 'readable-stream', 'version': '2.3.6', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'string_decoder', 'version': '1.1.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'buffer-from', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'ms', 'version': '2.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lodash._createcache', 'version': '3.1.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'read-installed', 'version': '4.0.3', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'semver', 'version': '5.7.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'ansi-styles', 'version': '3.2.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'spdx-exceptions', 'version': '2.1.0', 'licenses': ['CC-BY-3.0']}, {'type': 'nodejs', 'name': 'xtend', 'version': '4.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'set-blocking', 'version': '2.0.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'create-error-class', 'version': '3.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'define-properties', 'version': '1.1.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'minizlib', 'version': '1.3.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'mkdirp', 'version': '0.5.5', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'npm-init', 'version': '0.0.0', 'licenses': ['BSD']}, {'type': 'nodejs', 'name': 'rc', 'version': '1.2.8', 'licenses': ['(BSD-2-Clause OR MIT OR Apache-2.0)']}, {'type': 'nodejs', 'name': 'readable-stream', 'version': '1.1.14', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'cli-table3', 'version': '0.5.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'configstore', 'version': '3.1.5', 'licenses': ['BSD-2-Clause']}, {'type': 'nodejs', 'name': 'delayed-stream', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'extsprintf', 'version': '1.3.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'libnpmaccess', 'version': '3.0.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'qw', 'version': '1.0.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'sha', 'version': '3.0.0', 'licenses': ['(BSD-2-Clause OR MIT)']}, {'type': 'nodejs', 'name': 'strip-json-comments', 'version': '2.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'asn1', 'version': '0.2.4', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'libnpmorg', 'version': '1.0.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'meant', 'version': '1.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'sshpk', 'version': '1.14.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'string_decoder', 'version': '1.3.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'unpipe', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'wide-align', 'version': '1.1.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'is-installed-globally', 'version': '0.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'flush-write-stream', 'version': '1.0.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'from2', 'version': '2.3.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'p-limit', 'version': '2.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lru-cache', 'version': '5.1.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'npm-cache-filename', 'version': '1.0.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'package-json', 'version': '4.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'readable-stream', 'version': '3.6.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'dashdash', 'version': '1.14.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'slide', 'version': '1.1.6', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'dezalgo', 'version': '1.0.3', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'gentle-fs', 'version': '2.3.1', 'licenses': ['Artistic-2.0']}, {'type': 'nodejs', 'name': 'jsbn', 'version': '0.1.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lodash._root', 'version': '3.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'mississippi', 'version': '3.0.0', 'licenses': ['BSD-2-Clause']}, {'type': 'nodejs', 'name': 'normalize-package-data', 'version': '2.5.0', 'licenses': ['BSD-2-Clause']}, {'type': 'nodejs', 'name': 'opener', 'version': '1.5.2', 'licenses': ['(WTFPL OR MIT)']}, {'type': 'nodejs', 'name': 'detect-newline', 'version': '2.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'retry', 'version': '0.12.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'sorted-object', 'version': '2.0.1', 'licenses': ['(WTFPL OR MIT)']}, {'type': 'nodejs', 'name': 'uuid', 'version': '3.3.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'path-exists', 'version': '3.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'execa', 'version': '0.7.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lodash._baseuniq', 'version': '4.6.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'require-directory', 'version': '2.1.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'ansistyles', 'version': '0.1.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'https-proxy-agent', 'version': '2.2.4', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'is-ci', 'version': '1.2.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lockfile', 'version': '1.0.4', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'pseudomap', 'version': '1.0.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'has-flag', 'version': '3.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'spdx-license-ids', 'version': '3.0.5', 'licenses': ['CC0-1.0']}, {'type': 'nodejs', 'name': 'validate-npm-package-name', 'version': '3.0.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'yargs-parser', 'version': '15.0.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'path-is-absolute', 'version': '1.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lodash._bindcallback', 'version': '3.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lodash.uniq', 'version': '4.5.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'minimist', 'version': '1.2.5', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'npm-user-validate', 'version': '1.0.1', 'licenses': ['BSD-2-Clause']}, {'type': 'nodejs', 'name': 'prepend-http', 'version': '1.0.4', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'wrappy', 'version': '1.0.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'chalk', 'version': '2.4.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'emoji-regex', 'version': '7.0.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'iferr', 'version': '1.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'ignore-walk', 'version': '3.0.3', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'import-lazy', 'version': '2.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'ini', 'version': '1.3.8', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'number-is-nan', 'version': '1.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'string-width', 'version': '2.1.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'asynckit', 'version': '0.4.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'worker-farm', 'version': '1.7.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'concat-map', 'version': '0.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'escape-string-regexp', 'version': '1.0.5', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'mute-stream', 'version': '0.0.7', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'parallel-transform', 'version': '1.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'bcrypt-pbkdf', 'version': '1.0.2', 'licenses': ['BSD-3-Clause']}, {'type': 'nodejs', 'name': 'is-date-object', 'version': '1.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lodash.restparam', 'version': '3.6.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'resolve', 'version': '1.10.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'npm-audit-report', 'version': '1.3.3', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'npm-lifecycle', 'version': '3.1.5', 'licenses': ['Artistic-2.0']}, {'type': 'nodejs', 'name': 'shebang-regex', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'fs-minipass', 'version': '1.2.7', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'libnpm', 'version': '3.0.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'object-keys', 'version': '1.0.12', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'tar', 'version': '4.4.13', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'write-file-atomic', 'version': '2.4.3', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'is-cidr', 'version': '3.0.0', 'licenses': ['BSD-2-Clause']}, {'type': 'nodejs', 'name': 'byte-size', 'version': '5.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'humanize-ms', 'version': '1.2.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'ansi-regex', 'version': '2.1.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'spdx-expression-parse', 'version': '3.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'wcwidth', 'version': '1.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'yallist', 'version': '3.0.3', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'is-npm', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'tiny-relative-date', 'version': '1.3.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'got', 'version': '6.7.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'unzip-response', 'version': '2.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'p-finally', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'spdx-correct', 'version': '3.0.0', 'licenses': ['Apache-2.0']}, {'type': 'nodejs', 'name': 'through', 'version': '2.3.8', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'object-assign', 'version': '4.1.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'isexe', 'version': '2.0.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'npm-package-arg', 'version': '6.1.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'osenv', 'version': '0.1.5', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'safe-buffer', 'version': '5.1.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'is-redirect', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'chownr', 'version': '1.1.4', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'forever-agent', 'version': '0.6.1', 'licenses': ['Apache-2.0']}, {'type': 'nodejs', 'name': 'fs-vacuum', 'version': '1.2.10', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'is-stream', 'version': '1.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'os-homedir', 'version': '1.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'path-parse', 'version': '1.0.6', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'promzard', 'version': '0.3.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'boxen', 'version': '1.3.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'socks', 'version': '2.3.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'camelcase', 'version': '4.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'dot-prop', 'version': '4.2.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'move-concurrently', 'version': '1.0.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'typedarray', 'version': '0.0.6', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'punycode', 'version': '2.1.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'abbrev', 'version': '1.1.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'editor', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'gauge', 'version': '2.7.4', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'is-fullwidth-code-point', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'prr', 'version': '1.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'qs', 'version': '6.5.2', 'licenses': ['BSD-3-Clause']}, {'type': 'nodejs', 'name': 'strip-ansi', 'version': '4.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'yargs', 'version': '14.2.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'JSONStream', 'version': '1.3.5', 'licenses': ['(MIT OR Apache-2.0)']}, {'type': 'nodejs', 'name': 'figgy-pudding', 'version': '3.5.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'getpass', 'version': '0.1.7', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'nopt', 'version': '4.0.3', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'npm-normalize-package-bin', 'version': '1.0.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'retry', 'version': '0.10.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'require-main-filename', 'version': '2.0.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'ansi-regex', 'version': '3.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'es6-promise', 'version': '4.2.8', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'safe-buffer', 'version': '5.2.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'psl', 'version': '1.1.29', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'registry-auth-token', 'version': '3.4.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'strict-uri-encode', 'version': '2.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'timed-out', 'version': '4.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'tweetnacl', 'version': '0.14.5', 'licenses': ['Unlicense']}, {'type': 'nodejs', 'name': 'unique-filename', 'version': '1.1.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'aws-sign2', 'version': '0.7.0', 'licenses': ['Apache-2.0']}, {'type': 'nodejs', 'name': 'y18n', 'version': '4.0.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'util-deprecate', 'version': '1.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'libnpmpublish', 'version': '1.1.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'pacote', 'version': '9.5.12', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'path-is-inside', 'version': '1.0.2', 'licenses': ['(WTFPL OR MIT)']}, {'type': 'nodejs', 'name': 'rimraf', 'version': '2.7.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'ssri', 'version': '6.0.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'is-retry-allowed', 'version': '1.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'iferr', 'version': '0.1.5', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'core-util-is', 'version': '1.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'http-signature', 'version': '1.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'byline', 'version': '5.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'agentkeepalive', 'version': '3.5.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'libnpmconfig', 'version': '1.2.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'libnpmhook', 'version': '5.0.3', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'make-dir', 'version': '1.3.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'minimatch', 'version': '3.0.4', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'ms', 'version': '2.1.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'string_decoder', 'version': '0.10.31', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'yarn', 'version': '1.22.5', 'licenses': ['BSD-2-Clause']}, {'type': 'nodejs', 'name': 'umask', 'version': '1.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'caseless', 'version': '0.12.0', 'licenses': ['Apache-2.0']}, {'type': 'nodejs', 'name': 'detect-indent', 'version': '5.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'fast-deep-equal', 'version': '3.1.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'jsprim', 'version': '1.4.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'node-fetch-npm', 'version': '2.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'pify', 'version': '3.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'semver-diff', 'version': '2.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'brace-expansion', 'version': '1.1.11', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'p-limit', 'version': '2.3.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'through2', 'version': '2.0.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'genfun', 'version': '5.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'safer-buffer', 'version': '2.1.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'stream-each', 'version': '1.2.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'validate-npm-package-license', 'version': '3.0.4', 'licenses': ['Apache-2.0']}, {'type': 'nodejs', 'name': 'xdg-basedir', 'version': '3.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'deep-extend', 'version': '0.6.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'errno', 'version': '0.1.7', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'ajv', 'version': '6.12.6', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'infer-owner', 'version': '1.0.4', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'ci-info', 'version': '1.6.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lodash.union', 'version': '4.6.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lodash.without', 'version': '4.4.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'term-size', 'version': '1.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'cyclist', 'version': '0.2.2', 'licenses': []}, {'type': 'nodejs', 'name': 'columnify', 'version': '1.5.4', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'end-of-stream', 'version': '1.4.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'npm-packlist', 'version': '1.4.8', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'stream-iterate', 'version': '1.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'are-we-there-yet', 'version': '1.1.4', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'console-control-strings', 'version': '1.1.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'defaults', 'version': '1.0.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'hosted-git-info', 'version': '2.8.8', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'imurmurhash', 'version': '0.1.4', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'promise-retry', 'version': '1.1.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'protoduck', 'version': '5.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'read-package-json', 'version': '2.1.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'bin-links', 'version': '1.1.8', 'licenses': ['Artistic-2.0']}, {'type': 'nodejs', 'name': 'json-parse-better-errors', 'version': '1.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'pumpify', 'version': '1.5.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'supports-color', 'version': '5.4.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'encoding', 'version': '0.1.12', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'aproba', 'version': '1.2.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'decamelize', 'version': '1.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'has-unicode', 'version': '2.0.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'is-typedarray', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'read-cmd-shim', 'version': '1.0.5', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'sorted-union-stream', 'version': '2.1.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'url-parse-lax', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'cmd-shim', 'version': '3.0.3', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'extend', 'version': '3.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'wrap-ansi', 'version': '5.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'cliui', 'version': '5.0.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'lodash.clonedeep', 'version': '4.5.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'proto-list', 'version': '1.2.4', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'query-string', 'version': '6.8.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'es6-promisify', 'version': '5.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'fast-json-stable-stringify', 'version': '2.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'form-data', 'version': '2.3.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'jsonparse', 'version': '1.3.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'npm-pick-manifest', 'version': '3.0.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'stringify-package', 'version': '1.0.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'tough-cookie', 'version': '2.4.3', 'licenses': ['BSD-3-Clause']}, {'type': 'nodejs', 'name': 'which', 'version': '1.3.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'balanced-match', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'cidr-regex', 'version': '2.0.10', 'licenses': ['BSD-2-Clause']}, {'type': 'nodejs', 'name': 'decode-uri-component', 'version': '0.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'get-stream', 'version': '4.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'http-cache-semantics', 'version': '3.8.1', 'licenses': ['BSD-2-Clause']}, {'type': 'nodejs', 'name': 'inflight', 'version': '1.0.6', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'readdir-scoped-modules', 'version': '1.1.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'unique-string', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'asap', 'version': '2.0.6', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'graceful-fs', 'version': '4.2.4', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'iconv-lite', 'version': '0.4.23', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'unique-slug', 'version': '2.0.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'which-module', 'version': '2.0.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'err-code', 'version': '1.1.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'ansi-regex', 'version': '4.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'color-name', 'version': '1.1.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'combined-stream', 'version': '1.0.6', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'crypto-random-string', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'p-locate', 'version': '3.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'npm-bundled', 'version': '1.1.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'oauth-sign', 'version': '0.9.0', 'licenses': ['Apache-2.0']}, {'type': 'nodejs', 'name': 'ansi-align', 'version': '2.0.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'aproba', 'version': '2.0.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'function-bind', 'version': '1.1.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'has-symbols', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'registry-url', 'version': '3.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'strip-ansi', 'version': '3.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'agent-base', 'version': '4.3.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'widest-line', 'version': '2.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'split-on-first', 'version': '1.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'capture-stack-trace', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'config-chain', 'version': '1.1.12', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'err-code', 'version': '1.1.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'har-validator', 'version': '5.1.5', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'ip-regex', 'version': '2.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'qrcode-terminal', 'version': '0.12.0', 'licenses': ['Apache 2.0']}, {'type': 'nodejs', 'name': 'request', 'version': '2.88.0', 'licenses': ['Apache-2.0']}, {'type': 'nodejs', 'name': 'ansicolors', 'version': '0.3.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'fs.realpath', 'version': '1.0.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'p-try', 'version': '2.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lodash._cacheindexof', 'version': '3.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'safe-buffer', 'version': '5.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'tunnel-agent', 'version': '0.6.0', 'licenses': ['Apache-2.0']}, {'type': 'nodejs', 'name': 'archy', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'once', 'version': '1.4.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'npm-install-checks', 'version': '3.0.2', 'licenses': ['BSD-2-Clause']}, {'type': 'nodejs', 'name': 'colors', 'version': '1.3.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'es-to-primitive', 'version': '1.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'libcipm', 'version': '4.0.8', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'uid-number', 'version': '0.0.6', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'update-notifier', 'version': '2.5.0', 'licenses': ['BSD-2-Clause']}, {'type': 'nodejs', 'name': 'assert-plus', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'has', 'version': '1.0.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'http-proxy-agent', 'version': '2.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'inherits', 'version': '2.0.4', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'is-callable', 'version': '1.1.4', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'uri-js', 'version': '4.4.0', 'licenses': ['BSD-2-Clause']}, {'type': 'nodejs', 'name': 'call-limit', 'version': '1.1.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'concat-stream', 'version': '1.6.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'duplexify', 'version': '3.6.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'json-schema-traverse', 'version': '0.4.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'json-stringify-safe', 'version': '5.0.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'make-fetch-happen', 'version': '5.0.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'resolve-from', 'version': '4.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'string-width', 'version': '3.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'punycode', 'version': '1.4.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'is-symbol', 'version': '1.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'duplexer3', 'version': '0.1.4', 'licenses': ['BSD-3-Clause']}, {'type': 'nodejs', 'name': 'ecc-jsbn', 'version': '0.1.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'find-npm-prefix', 'version': '1.0.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'global-dirs', 'version': '0.1.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'npm-logical-tree', 'version': '1.2.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'performance-now', 'version': '2.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'builtins', 'version': '1.0.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'is-regex', 'version': '1.0.4', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'json-schema', 'version': '0.2.3', 'licenses': ['AFLv2.1']}, {'type': 'nodejs', 'name': 'latest-version', 'version': '3.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lock-verify', 'version': '2.1.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'read', 'version': '1.0.7', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'agent-base', 'version': '4.2.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'npm', 'version': '6.14.12', 'licenses': ['Artistic-2.0']}, {'type': 'nodejs', 'name': 'ip', 'version': '1.1.5', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'cross-spawn', 'version': '5.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'env-paths', 'version': '2.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'is-path-inside', 'version': '1.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lodash._createset', 'version': '4.0.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lodash._getnative', 'version': '3.9.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'object.getownpropertydescriptors', 'version': '2.0.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'pump', 'version': '2.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'bluebird', 'version': '3.5.5', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'isarray', 'version': '0.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'text-table', 'version': '0.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'util-extend', 'version': '1.0.3', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'run-queue', 'version': '1.0.3', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'fs-write-stream-atomic', 'version': '1.0.10', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'is-obj', 'version': '1.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'isarray', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lodash._baseindexof', 'version': '3.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'cli-columns', 'version': '3.1.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'har-schema', 'version': '2.0.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'find-up', 'version': '3.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'path-key', 'version': '2.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'shebang-command', 'version': '1.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'verror', 'version': '1.10.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'is-fullwidth-code-point', 'version': '2.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'debug', 'version': '3.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'debuglog', 'version': '1.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'delegates', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'lazy-property', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'locate-path', 'version': '3.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'mime-types', 'version': '2.1.19', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'npm-registry-fetch', 'version': '4.0.7', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'copy-concurrently', 'version': '1.0.5', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'npmlog', 'version': '4.1.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'signal-exit', 'version': '3.0.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'util-promisify', 'version': '2.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'npm-run-path', 'version': '2.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'glob', 'version': '7.1.6', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'init-package-json', 'version': '1.10.3', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'libnpmsearch', 'version': '2.0.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'libnpmteam', 'version': '1.0.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'smart-buffer', 'version': '4.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'get-caller-file', 'version': '2.0.5', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'color-convert', 'version': '1.9.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'yallist', 'version': '2.1.2', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'lowercase-keys', 'version': '1.0.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'stream-shift', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'npm-cli-docs', 'version': '0.1.0', 'licenses': ['Artistic-2.0']}, {'type': 'nodejs', 'name': 'lru-cache', 'version': '4.1.5', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'dotenv', 'version': '5.0.1', 'licenses': ['BSD-2-Clause']}, {'type': 'nodejs', 'name': 'es-abstract', 'version': '1.12.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'promise-inflight', 'version': '1.0.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'pump', 'version': '3.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'camelcase', 'version': '5.3.1', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'cli-boxes', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'ci-info', 'version': '2.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'minipass', 'version': '2.9.0', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'node-gyp', 'version': '5.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'npm-profile', 'version': '4.0.4', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'strip-eof', 'version': '1.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'cacache', 'version': '12.0.3', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'clone', 'version': '1.0.4', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'code-point-at', 'version': '1.1.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'string-width', 'version': '1.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'os-tmpdir', 'version': '1.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'aws4', 'version': '1.8.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'get-stream', 'version': '3.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'read-package-tree', 'version': '5.3.1', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'socks-proxy-agent', 'version': '4.0.2', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'from2', 'version': '1.3.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'strip-ansi', 'version': '5.2.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'libnpx', 'version': '10.2.4', 'licenses': ['ISC']}, {'type': 'nodejs', 'name': 'mime-db', 'version': '1.35.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'process-nextick-args', 'version': '2.0.0', 'licenses': ['MIT']}, {'type': 'nodejs', 'name': 'isstream', 'version': '0.1.2', 'licenses': ['MIT']}, {'type': 'python', 'name': 'configobj', 'version': '5.0.6', 'licenses': ['UNKNOWN']}, {'type': 'python', 'name': 'mercurial', 'version': '4.0', 'licenses': ['GNU GPLv2 or any later version']}, {'type': 'python', 'name': 'six', 'version': '1.10.0', 'licenses': ['MIT']}, {'type': 'python', 'name': 'python', 'version': '2.7.13', 'licenses': ['PSF license']}, {'type': 'python', 'name': 'wsgiref', 'version': '0.1.2', 'licenses': ['PSF or ZPL']}, {'type': 'python', 'name': 'argparse', 'version': '1.2.1', 'licenses': ['Python Software Foundation License']}, {'type': 'python', 'name': 'bzr', 'version': '2.8.0dev1', 'licenses': ['GNU GPL v2']}], 'relatedResourceId': 'jobs.container-test-job'}