// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
import { __asyncDelegator, __asyncGenerator, __asyncValues, __await, __rest } from "tslib";
import { isNamedKeyCredential, isSASCredential, isTokenCredential } from "@azure/core-auth";
import { COSMOS_SCOPE, STORAGE_SCOPE, TablesLoggingAllowedHeaderNames } from "./utils/constants.js";
import { injectSecondaryEndpointHeader, tablesSecondaryEndpointPolicy, } from "./secondaryEndpointPolicy.js";
import { parseXML, stringifyXML } from "@azure/core-xml";
import { GeneratedClient } from "./generated/generatedClient.js";
import { apiVersionPolicy } from "./utils/apiVersionPolicy.js";
import { getClientParamsFromConnectionString } from "./utils/connectionString.js";
import { handleTableAlreadyExists } from "./utils/errorHelpers.js";
import { isCredential } from "./utils/isCredential.js";
import { logger } from "./logger.js";
import { setTokenChallengeAuthenticationPolicy } from "./utils/challengeAuthenticationUtils.js";
import { tablesNamedKeyCredentialPolicy } from "./tablesNamedCredentialPolicy.js";
import { tablesSASTokenPolicy } from "./tablesSASTokenPolicy.js";
import { tracingClient } from "./utils/tracing.js";
import { isCosmosEndpoint } from "./utils/isCosmosEndpoint.js";
/**
 * A TableServiceClient represents a Client to the Azure Tables service allowing you
 * to perform operations on the tables and the entities.
 */
export class TableServiceClient {
    constructor(url, credentialOrOptions, options) {
        this.url = url;
        const isCosmos = isCosmosEndpoint(this.url);
        const credential = isCredential(credentialOrOptions) ? credentialOrOptions : undefined;
        const clientOptions = (!isCredential(credentialOrOptions) ? credentialOrOptions : options) || {};
        const internalPipelineOptions = Object.assign(Object.assign({}, clientOptions), { endpoint: clientOptions.endpoint || this.url, loggingOptions: {
                logger: logger.info,
                additionalAllowedHeaderNames: [...TablesLoggingAllowedHeaderNames],
            }, deserializationOptions: {
                parseXML,
            }, serializationOptions: {
                stringifyXML,
            } });
        const client = new GeneratedClient(this.url, internalPipelineOptions);
        client.pipeline.addPolicy(tablesSecondaryEndpointPolicy);
        if (isNamedKeyCredential(credential)) {
            client.pipeline.addPolicy(tablesNamedKeyCredentialPolicy(credential));
        }
        else if (isSASCredential(credential)) {
            client.pipeline.addPolicy(tablesSASTokenPolicy(credential));
        }
        if (isTokenCredential(credential)) {
            const scope = isCosmos ? COSMOS_SCOPE : STORAGE_SCOPE;
            setTokenChallengeAuthenticationPolicy(client.pipeline, credential, scope);
        }
        if (options === null || options === void 0 ? void 0 : options.version) {
            client.pipeline.addPolicy(apiVersionPolicy(options.version));
        }
        this.pipeline = client.pipeline;
        this.table = client.table;
        this.service = client.service;
    }
    /**
     * Retrieves statistics related to replication for the Table service. It is only available on the
     * secondary location endpoint when read-access geo-redundant replication is enabled for the account.
     * @param options - The options parameters.
     */
    async getStatistics(options = {}) {
        return tracingClient.withSpan("TableServiceClient.getStatistics", options, (updatedOptions) => this.service.getStatistics(injectSecondaryEndpointHeader(updatedOptions)));
    }
    /**
     * Gets the properties of an account's Table service, including properties for Analytics and CORS
     * (Cross-Origin Resource Sharing) rules.
     * @param options - The options parameters.
     */
    getProperties(options = {}) {
        return tracingClient.withSpan("TableServiceClient.getProperties", options, (updatedOptions) => this.service.getProperties(updatedOptions));
    }
    /**
     * Sets properties for an account's Table service endpoint, including properties for Analytics and CORS
     * (Cross-Origin Resource Sharing) rules.
     * @param properties - The Table Service properties.
     * @param options - The options parameters.
     */
    setProperties(properties, options = {}) {
        return tracingClient.withSpan("TableServiceClient.setProperties", options, (updatedOptions) => this.service.setProperties(properties, updatedOptions));
    }
    /**
     * Creates a new table under the given account.
     * @param name - The name of the table.
     * @param options - The options parameters.
     */
    createTable(name, options = {}) {
        return tracingClient.withSpan("TableServiceClient.createTable", options, async (updatedOptions) => {
            try {
                await this.table.create({ name }, updatedOptions);
            }
            catch (e) {
                handleTableAlreadyExists(e, Object.assign(Object.assign({}, updatedOptions), { logger, tableName: name }));
            }
        });
    }
    /**
     * Operation permanently deletes the specified table.
     * @param name - The name of the table.
     * @param options - The options parameters.
     */
    deleteTable(name, options = {}) {
        return tracingClient.withSpan("TableServiceClient.deleteTable", options, async (updatedOptions) => {
            try {
                await this.table.delete(name, updatedOptions);
            }
            catch (e) {
                if (e.statusCode === 404) {
                    logger.info("TableServiceClient.deleteTable: Table doesn't exist");
                }
                else {
                    throw e;
                }
            }
        });
    }
    /**
     * Queries tables under the given account.
     * @param options - The options parameters.
     */
    listTables(
    // eslint-disable-next-line @azure/azure-sdk/ts-naming-options
    options) {
        const iter = this.listTablesAll(options);
        return {
            next() {
                return iter.next();
            },
            [Symbol.asyncIterator]() {
                return this;
            },
            byPage: (settings) => {
                const pageOptions = Object.assign(Object.assign({}, options), { queryOptions: Object.assign(Object.assign({}, options === null || options === void 0 ? void 0 : options.queryOptions), { top: settings === null || settings === void 0 ? void 0 : settings.maxPageSize }) });
                if (settings === null || settings === void 0 ? void 0 : settings.continuationToken) {
                    pageOptions.continuationToken = settings.continuationToken;
                }
                return this.listTablesPage(pageOptions);
            },
        };
    }
    listTablesAll(options) {
        return __asyncGenerator(this, arguments, function* listTablesAll_1() {
            var _a, e_1, _b, _c;
            const firstPage = yield __await(this._listTables(options));
            const { continuationToken } = firstPage;
            yield __await(yield* __asyncDelegator(__asyncValues(firstPage)));
            if (continuationToken) {
                const optionsWithContinuation = Object.assign(Object.assign({}, options), { continuationToken });
                try {
                    for (var _d = true, _e = __asyncValues(this.listTablesPage(optionsWithContinuation)), _f; _f = yield __await(_e.next()), _a = _f.done, !_a; _d = true) {
                        _c = _f.value;
                        _d = false;
                        const page = _c;
                        yield __await(yield* __asyncDelegator(__asyncValues(page)));
                    }
                }
                catch (e_1_1) { e_1 = { error: e_1_1 }; }
                finally {
                    try {
                        if (!_d && !_a && (_b = _e.return)) yield __await(_b.call(_e));
                    }
                    finally { if (e_1) throw e_1.error; }
                }
            }
        });
    }
    listTablesPage() {
        return __asyncGenerator(this, arguments, function* listTablesPage_1(options = {}) {
            let result = yield __await(tracingClient.withSpan("TableServiceClient.listTablesPage", options, (updatedOptions) => this._listTables(updatedOptions)));
            yield yield __await(result);
            while (result.continuationToken) {
                const optionsWithContinuation = Object.assign(Object.assign({}, options), { continuationToken: result.continuationToken });
                result = yield __await(tracingClient.withSpan("TableServiceClient.listTablesPage", optionsWithContinuation, async (updatedOptions, span) => {
                    span.setAttribute("continuationToken", updatedOptions.continuationToken);
                    return this._listTables(updatedOptions);
                }));
                yield yield __await(result);
            }
        });
    }
    async _listTables(options = {}) {
        const { continuationToken: nextTableName } = options, listOptions = __rest(options, ["continuationToken"]);
        const { xMsContinuationNextTableName: continuationToken, value = [] } = await this.table.query(Object.assign(Object.assign({}, listOptions), { nextTableName }));
        return Object.assign([...value], { continuationToken });
    }
    /**
     *
     * Creates an instance of TableServiceClient from connection string.
     *
     * @param connectionString - Account connection string or a SAS connection string of an Azure storage account.
     *                           [ Note - Account connection string can only be used in NODE.JS runtime. ]
     *                           Account connection string example -
     *                           `DefaultEndpointsProtocol=https;AccountName=myaccount;AccountKey=accountKey;EndpointSuffix=core.windows.net`
     *                           SAS connection string example -
     *                           `BlobEndpoint=https://myaccount.table.core.windows.net/;QueueEndpoint=https://myaccount.queue.core.windows.net/;FileEndpoint=https://myaccount.file.core.windows.net/;TableEndpoint=https://myaccount.table.core.windows.net/;SharedAccessSignature=sasString`
     * @param options - Options to configure the HTTP pipeline.
     * @returns A new TableServiceClient from the given connection string.
     */
    static fromConnectionString(connectionString, 
    // eslint-disable-next-line @azure/azure-sdk/ts-naming-options
    options) {
        const { url, options: clientOptions, credential, } = getClientParamsFromConnectionString(connectionString, options);
        if (credential) {
            return new TableServiceClient(url, credential, clientOptions);
        }
        else {
            return new TableServiceClient(url, clientOptions);
        }
    }
}
//# sourceMappingURL=TableServiceClient.js.map