import type { AbortSignalLike } from "@azure/abort-controller";
import type { ModifiedAccessConditions } from "./models.js";
import type { CommonOptions } from "./StorageClient.js";
import type { BlobClient } from "./Clients.js";
import type { ContainerClient } from "./ContainerClient.js";
import type { WithResponse } from "./utils/utils.common.js";
/**
 * The details for a specific lease.
 */
export interface Lease {
    /**
     * The ETag contains a value that you can use to
     * perform operations conditionally. If the request version is 2011-08-18 or
     * newer, the ETag value will be in quotes.
     */
    etag?: string;
    /**
     * Returns the date and time the container was
     * last modified. Any operation that modifies the blob, including an update
     * of the blob's metadata or properties, changes the last-modified time of
     * the blob.
     */
    lastModified?: Date;
    /**
     * Uniquely identifies a container's lease
     */
    leaseId?: string;
    /**
     * Approximate time remaining in the lease
     * period, in seconds.
     */
    leaseTime?: number;
    /**
     * This header uniquely identifies the request
     * that was made and can be used for troubleshooting the request.
     */
    requestId?: string;
    /**
     * Indicates the version of the Blob service used
     * to execute the request. This header is returned for requests made against
     * version 2009-09-19 and above.
     */
    version?: string;
    /**
     * UTC date/time value generated by the service that
     * indicates the time at which the response was initiated
     */
    date?: Date;
    /**
     * Error code if any associated with the response that returned
     * the Lease information.
     */
    errorCode?: string;
}
/**
 * Contains the response data for operations that create, modify, or delete a lease.
 *
 * See {@link BlobLeaseClient}.
 */
export type LeaseOperationResponse = WithResponse<Lease, Lease>;
/**
 * Configures lease operations.
 */
export interface LeaseOperationOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Conditions to meet when changing the lease.
     */
    conditions?: ModifiedAccessConditions;
}
/**
 * A client that manages leases for a {@link ContainerClient} or a {@link BlobClient}.
 */
export declare class BlobLeaseClient {
    private _leaseId;
    private _url;
    private _containerOrBlobOperation;
    private _isContainer;
    /**
     * Gets the lease Id.
     *
     * @readonly
     */
    get leaseId(): string;
    /**
     * Gets the url.
     *
     * @readonly
     */
    get url(): string;
    /**
     * Creates an instance of BlobLeaseClient.
     * @param client - The client to make the lease operation requests.
     * @param leaseId - Initial proposed lease id.
     */
    constructor(client: ContainerClient | BlobClient, leaseId?: string);
    /**
     * Establishes and manages a lock on a container for delete operations, or on a blob
     * for write and delete operations.
     * The lock duration can be 15 to 60 seconds, or can be infinite.
     * @see https://learn.microsoft.com/rest/api/storageservices/lease-container
     * and
     * @see https://learn.microsoft.com/rest/api/storageservices/lease-blob
     *
     * @param duration - Must be between 15 to 60 seconds, or infinite (-1)
     * @param options - option to configure lease management operations.
     * @returns Response data for acquire lease operation.
     */
    acquireLease(duration: number, options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
    /**
     * To change the ID of the lease.
     * @see https://learn.microsoft.com/rest/api/storageservices/lease-container
     * and
     * @see https://learn.microsoft.com/rest/api/storageservices/lease-blob
     *
     * @param proposedLeaseId - the proposed new lease Id.
     * @param options - option to configure lease management operations.
     * @returns Response data for change lease operation.
     */
    changeLease(proposedLeaseId: string, options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
    /**
     * To free the lease if it is no longer needed so that another client may
     * immediately acquire a lease against the container or the blob.
     * @see https://learn.microsoft.com/rest/api/storageservices/lease-container
     * and
     * @see https://learn.microsoft.com/rest/api/storageservices/lease-blob
     *
     * @param options - option to configure lease management operations.
     * @returns Response data for release lease operation.
     */
    releaseLease(options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
    /**
     * To renew the lease.
     * @see https://learn.microsoft.com/rest/api/storageservices/lease-container
     * and
     * @see https://learn.microsoft.com/rest/api/storageservices/lease-blob
     *
     * @param options - Optional option to configure lease management operations.
     * @returns Response data for renew lease operation.
     */
    renewLease(options?: LeaseOperationOptions): Promise<Lease>;
    /**
     * To end the lease but ensure that another client cannot acquire a new lease
     * until the current lease period has expired.
     * @see https://learn.microsoft.com/rest/api/storageservices/lease-container
     * and
     * @see https://learn.microsoft.com/rest/api/storageservices/lease-blob
     *
     * @param breakPeriod - Break period
     * @param options - Optional options to configure lease management operations.
     * @returns Response data for break lease operation.
     */
    breakLease(breakPeriod: number, options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
}
//# sourceMappingURL=BlobLeaseClient.d.ts.map