"use strict";
// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.StorageFileAudience = void 0;
exports.toShareProtocols = toShareProtocols;
exports.toShareProtocolsString = toShareProtocolsString;
exports.validateFilePermissionOptions = validateFilePermissionOptions;
exports.validateAndSetDefaultsForFileAndDirectoryCreateCommonOptions = validateAndSetDefaultsForFileAndDirectoryCreateCommonOptions;
exports.validateAndSetDefaultsForFileAndDirectorySetPropertiesCommonOptions = validateAndSetDefaultsForFileAndDirectorySetPropertiesCommonOptions;
exports.fileAttributesToString = fileAttributesToString;
exports.fileCreationTimeToString = fileCreationTimeToString;
exports.fileLastWriteTimeToString = fileLastWriteTimeToString;
exports.fileChangeTimeToString = fileChangeTimeToString;
exports.getFileServiceAccountAudience = getFileServiceAccountAudience;
const FileSystemAttributes_js_1 = require("./FileSystemAttributes.js");
const utils_common_js_1 = require("./utils/utils.common.js");
const log_js_1 = require("./log.js");
/**
 * Convert protocols from joined string to ShareProtocols.
 *
 * @param protocolsString -
 */
function toShareProtocols(protocolsString) {
    if (protocolsString === undefined) {
        return undefined;
    }
    const protocolStrArray = protocolsString.split(";");
    const protocols = {};
    for (const protocol of protocolStrArray) {
        if (protocol === "SMB") {
            protocols.smbEnabled = true;
        }
        else if (protocol === "NFS") {
            protocols.nfsEnabled = true;
        }
    }
    return protocols;
}
/**
 * Convert ShareProtocols to joined string.
 *
 * @param protocols -
 */
function toShareProtocolsString(protocols = {}) {
    let protocolStr = undefined;
    if (protocols.smbEnabled === true) {
        protocolStr = "SMB";
    }
    if (protocols.nfsEnabled === true) {
        log_js_1.logger.info(`Using "NFS" in favor of "SMB" for the share protocol as currently they can't be supported at the same time.`);
        protocolStr = "NFS";
    }
    return protocolStr;
}
function validateFilePermissionOptions(filePermission, filePermissionKey) {
    if (filePermission && filePermissionKey) {
        throw new RangeError("Only one of filePermission or filePermissionKey can be specified.");
    }
}
function validateAndSetDefaultsForFileAndDirectoryCreateCommonOptions(options) {
    // Would better deep copy params set by user.
    if (!options) {
        options = {};
    }
    validateFilePermissionOptions(options.filePermission, options.filePermissionKey);
    return options;
}
function validateAndSetDefaultsForFileAndDirectorySetPropertiesCommonOptions(options) {
    // Would better deep copy params set by user.
    if (!options) {
        options = {};
    }
    validateFilePermissionOptions(options.filePermission, options.filePermissionKey);
    return options;
}
function fileAttributesToString(fileAttributes) {
    return fileAttributes instanceof FileSystemAttributes_js_1.FileSystemAttributes
        ? fileAttributes.toString()
        : fileAttributes;
}
function fileCreationTimeToString(time) {
    return time instanceof Date ? (0, utils_common_js_1.truncatedISO8061Date)(time) : time;
}
function fileLastWriteTimeToString(time) {
    return time instanceof Date ? (0, utils_common_js_1.truncatedISO8061Date)(time) : time;
}
function fileChangeTimeToString(time) {
    return time instanceof Date ? (0, utils_common_js_1.truncatedISO8061Date)(time) : time;
}
/**
 * Defines the known cloud audiences for Storage.
 */
var StorageFileAudience;
(function (StorageFileAudience) {
    /**
     * The OAuth scope to use to retrieve an AAD token for Azure Storage.
     */
    StorageFileAudience["StorageOAuthScopes"] = "https://storage.azure.com/.default";
})(StorageFileAudience || (exports.StorageFileAudience = StorageFileAudience = {}));
/**
 * To get the OAuth audience for a storage account for file service.
 */
function getFileServiceAccountAudience(storageAccountName) {
    return `https://${storageAccountName}.file.core.windows.net/.default`;
}
//# sourceMappingURL=models.js.map