var async = require('async');

var helpers = require('../../../helpers/azure/');

module.exports = {
    title: 'VM Windows AntiMalware Extension',
    category: 'Virtual Machines',
    domain: 'Compute',
    severity: 'Medium',
    description: 'Ensures that Virtual Machine windows instances have IaaS AntiMalware extension installed.',
    more_info: 'The VM Windows AntiMalware Extension provides real-time protection against viruses, spyware, and other malicious software for virtual machines running on the Windows operating system.',
    recommended_action: 'Modify Virtual Machine and install IaaS AntiMalware extension.',
    link: 'https://learn.microsoft.com/en-us/azure/virtual-machines/extensions/iaas-antimalware-windows',
    apis: ['virtualMachines:listAll', 'virtualMachineExtensions:list'],
    realtime_triggers: ['microsoftcompute:virtualmachines:write', 'microsoftcompute:virtualmachines:delete', 'microsoftcompute:virtualmachines:extensions:write', 'microsoftcompute:virtualmachines:extensions:delete'],

    run: function(cache, settings, callback) {
        var results = [];
        var source = {};
        var locations = helpers.locations(settings.govcloud);

        async.each(locations.virtualMachines, function(location, rcb){
            var virtualMachines = helpers.addSource(cache, source,
                ['virtualMachines', 'listAll', location]);

            if (!virtualMachines) return rcb();

            if (virtualMachines.err || !virtualMachines.data) {
                helpers.addResult(results, 3, 'Unable to query for Virtual Machines: ' + helpers.addError(virtualMachines), location);
                return rcb();
            }

            if (!virtualMachines.data.length) {
                helpers.addResult(results, 0, 'No Virtual Machines found', location);
                return rcb();
            }

            for (let virtualMachine of virtualMachines.data) {
                if (!virtualMachine.id || (virtualMachine.osDisk && virtualMachine.osDisk.osType &&
                virtualMachine.osDisk.osType.toLowerCase() != 'windows')) {
                    continue;
                }

                const virtualMachineExtensions = helpers.addSource(cache, source,
                    ['virtualMachineExtensions', 'list', location, virtualMachine.id]);

                if (!virtualMachineExtensions || virtualMachineExtensions.err || !virtualMachineExtensions.data) {
                    helpers.addResult(results, 3, 'Unable to query for VM Extensions: ' + helpers.addError(virtualMachineExtensions), location, virtualMachine.id);
                    continue;
                }
                
                if (!virtualMachineExtensions.data.length) {
                    helpers.addResult(results, 2, 'No VM Extensions found', location, virtualMachine.id);
                    continue;
                }

                let found = virtualMachineExtensions.data.find(vmExt => vmExt.name && vmExt.name.toLowerCase() === 'iaasantimalware');

                if (found) {
                    helpers.addResult(results, 0, 'Windows Virtual Machine has IaaS Antimalware extension installed', location, virtualMachine.id);
                } else {
                    helpers.addResult(results, 2, 'Windows Virtual Machine does not have IaaS Antimalware extension installed', location,
                        virtualMachine.id);
                }
            }

            rcb();
        }, function() {
            callback(null, results, source);
        });
    }
};
