import type { GetPropertiesResponse, GetStatisticsResponse, ServiceProperties, SetPropertiesOptions, SetPropertiesResponse } from "./generatedModels.js";
import type { OperationOptions } from "@azure/core-client";
import type { ListTableItemsOptions, TableItem, TableServiceClientOptions } from "./models.js";
import type { NamedKeyCredential, SASCredential, TokenCredential } from "@azure/core-auth";
import type { PagedAsyncIterableIterator } from "@azure/core-paging";
import type { Pipeline } from "@azure/core-rest-pipeline";
import type { TableItemResultPage } from "./models.js";
/**
 * A TableServiceClient represents a Client to the Azure Tables service allowing you
 * to perform operations on the tables and the entities.
 */
export declare class TableServiceClient {
    /**
     * Table Account URL
     */
    url: string;
    /**
     * Represents a pipeline for making a HTTP request to a URL.
     * Pipelines can have multiple policies to manage manipulating each request before and after it is made to the server.
     */
    pipeline: Pipeline;
    private table;
    private service;
    /**
     * Creates a new instance of the TableServiceClient class.
     *
     * @param url - The URL of the service account that is the target of the desired operation., such as "https://myaccount.table.core.windows.net".
     * @param credential - NamedKeyCredential | SASCredential used to authenticate requests. Only Supported for Node
     * @param options - Options to configure the HTTP pipeline.
     *
     * ### Example using an account name/key:
     *
     * ```ts snippet:ReadmeSampleCreateClient_NamedKeyCredential
     * import { AzureNamedKeyCredential, TableServiceClient } from "@azure/data-tables";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new AzureNamedKeyCredential(account, accountKey);
     * const serviceClient = new TableServiceClient(
     *   `https://${account}.table.core.windows.net`,
     *   credential,
     * );
     * ```
     */
    constructor(url: string, credential: NamedKeyCredential, options?: TableServiceClientOptions);
    /**
     * Creates a new instance of the TableServiceClient class.
     *
     * @param url - The URL of the service account that is the target of the desired operation., such as "https://myaccount.table.core.windows.net".
     * @param credential - SASCredential used to authenticate requests
     * @param options - Options to configure the HTTP pipeline.
     *
     * ### Example using a SAS Token.
     *
     * ```ts snippet:ReadmeSampleCreateClient_SASToken
     * import { TableServiceClient, AzureSASCredential } from "@azure/data-tables";
     *
     * const account = "<account name>";
     * const sas = "<service Shared Access Signature Token>";
     *
     * const serviceClientWithSAS = new TableServiceClient(
     *   `https://${account}.table.core.windows.net`,
     *   new AzureSASCredential(sas),
     * );
     * ```
     */
    constructor(url: string, credential: SASCredential, options?: TableServiceClientOptions);
    /**
     * Creates a new instance of the TableServiceClient class.
     *
     * @param url - The URL of the service account that is the target of the desired operation., such as "https://myaccount.table.core.windows.net".
     * @param credential - Azure Active Directory credential used to authenticate requests
     * @param options - Options to configure the HTTP pipeline.
     *
     * ### Example using an Azure Active Directory credential:
     *
     * ```ts snippet:ReadmeSampleCreateClient_TokenCredential
     * import { DefaultAzureCredential } from "@azure/identity";
     * import { TableServiceClient } from "@azure/data-tables";
     *
     * const credential = new DefaultAzureCredential();
     * const account = "<account name>";
     *
     * const clientWithAAD = new TableServiceClient(
     *   `https://${account}.table.core.windows.net`,
     *   credential,
     * );
     * ```
     */
    constructor(url: string, credential: TokenCredential, options?: TableServiceClientOptions);
    /**
     * Creates a new instance of the TableServiceClient class.
     *
     * @param url - The URL of the service account that is the target of the desired operation., such as
     *              "https://myaccount.table.core.windows.net". You can append a SAS,
     *              such as "https://myaccount.table.core.windows.net?sasString".
     * @param options - Options to configure the HTTP pipeline.
     * Example appending a SAS token:
     *
     * ```ts snippet:ReadmeSampleCreateClient_SASTokenURL
     * import { TableServiceClient } from "@azure/data-tables";
     *
     * const account = "<account name>";
     * const sasToken = "<SAS token>";
     * const tableName = "<tableName>";
     *
     * const serviceClientWithSASURL = new TableServiceClient(
     *   `https://${account}.table.core.windows.net?${sasToken}`,
     * );
     * ```
     */
    constructor(url: string, options?: TableServiceClientOptions);
    /**
     * Retrieves statistics related to replication for the Table service. It is only available on the
     * secondary location endpoint when read-access geo-redundant replication is enabled for the account.
     * @param options - The options parameters.
     */
    getStatistics(options?: OperationOptions): Promise<GetStatisticsResponse>;
    /**
     * Gets the properties of an account's Table service, including properties for Analytics and CORS
     * (Cross-Origin Resource Sharing) rules.
     * @param options - The options parameters.
     */
    getProperties(options?: OperationOptions): Promise<GetPropertiesResponse>;
    /**
     * Sets properties for an account's Table service endpoint, including properties for Analytics and CORS
     * (Cross-Origin Resource Sharing) rules.
     * @param properties - The Table Service properties.
     * @param options - The options parameters.
     */
    setProperties(properties: ServiceProperties, options?: SetPropertiesOptions): Promise<SetPropertiesResponse>;
    /**
     * Creates a new table under the given account.
     * @param name - The name of the table.
     * @param options - The options parameters.
     */
    createTable(name: string, options?: OperationOptions): Promise<void>;
    /**
     * Operation permanently deletes the specified table.
     * @param name - The name of the table.
     * @param options - The options parameters.
     */
    deleteTable(name: string, options?: OperationOptions): Promise<void>;
    /**
     * Queries tables under the given account.
     * @param options - The options parameters.
     */
    listTables(options?: ListTableItemsOptions): PagedAsyncIterableIterator<TableItem, TableItemResultPage>;
    private listTablesAll;
    private listTablesPage;
    private _listTables;
    /**
     *
     * Creates an instance of TableServiceClient from connection string.
     *
     * @param connectionString - Account connection string or a SAS connection string of an Azure storage account.
     *                           [ Note - Account connection string can only be used in NODE.JS runtime. ]
     *                           Account connection string example -
     *                           `DefaultEndpointsProtocol=https;AccountName=myaccount;AccountKey=accountKey;EndpointSuffix=core.windows.net`
     *                           SAS connection string example -
     *                           `BlobEndpoint=https://myaccount.table.core.windows.net/;QueueEndpoint=https://myaccount.queue.core.windows.net/;FileEndpoint=https://myaccount.file.core.windows.net/;TableEndpoint=https://myaccount.table.core.windows.net/;SharedAccessSignature=sasString`
     * @param options - Options to configure the HTTP pipeline.
     * @returns A new TableServiceClient from the given connection string.
     */
    static fromConnectionString(connectionString: string, options?: TableServiceClientOptions): TableServiceClient;
}
//# sourceMappingURL=TableServiceClient.d.ts.map