from unittest import mock

from prowler.providers.aws.services.backup.backup_service import BackupVault

AWS_REGION = "eu-west-1"
AWS_ACCOUNT_NUMBER = "0123456789012"


class Test_backup_vaults_encrypted:
    def test_no_backup_vaults(self):
        backup_client = mock.MagicMock
        backup_client.backup_vaults = []
        with (
            mock.patch(
                "prowler.providers.aws.services.backup.backup_service.Backup",
                new=backup_client,
            ),
            mock.patch(
                "prowler.providers.aws.services.backup.backup_vaults_encrypted.backup_vaults_encrypted.backup_client",
                new=backup_client,
            ),
        ):
            # Test Check
            from prowler.providers.aws.services.backup.backup_vaults_encrypted.backup_vaults_encrypted import (
                backup_vaults_encrypted,
            )

            check = backup_vaults_encrypted()
            result = check.execute()

            assert len(result) == 0

    def test_one_backup_vault_unencrypted(self):
        backup_client = mock.MagicMock
        backup_vault_arn = f"arn:aws:backup:{AWS_REGION}:{AWS_ACCOUNT_NUMBER}:backup-vault:MyBackupVault"
        backup_client.backup_vaults = [
            BackupVault(
                arn=backup_vault_arn,
                name="MyBackupVault",
                region=AWS_REGION,
                encryption="",
                recovery_points=1,
                locked=True,
                min_retention_days=1,
                max_retention_days=2,
                tags=[],
            )
        ]

        with (
            mock.patch(
                "prowler.providers.aws.services.backup.backup_service.Backup",
                new=backup_client,
            ),
            mock.patch(
                "prowler.providers.aws.services.backup.backup_vaults_encrypted.backup_vaults_encrypted.backup_client",
                new=backup_client,
            ),
        ):
            # Test Check
            from prowler.providers.aws.services.backup.backup_vaults_encrypted.backup_vaults_encrypted import (
                backup_vaults_encrypted,
            )

            check = backup_vaults_encrypted()
            result = check.execute()

            assert len(result) == 1
            assert result[0].status == "FAIL"
            assert (
                result[0].status_extended
                == f"Backup Vault {result[0].resource_id} is not encrypted at rest."
            )
            assert result[0].resource_id == "MyBackupVault"
            assert result[0].resource_arn == backup_vault_arn
            assert result[0].region == AWS_REGION
            assert result[0].resource_tags == []

    def test_one_backup_vault_encrypted(self):
        backup_client = mock.MagicMock
        backup_vault_arn = f"arn:aws:backup:{AWS_REGION}:{AWS_ACCOUNT_NUMBER}:backup-vault:MyBackupVault"
        backup_client.backup_vaults = [
            BackupVault(
                arn=backup_vault_arn,
                name="MyBackupVault",
                region=AWS_REGION,
                encryption="test",
                recovery_points=1,
                locked=True,
                min_retention_days=1,
                max_retention_days=2,
                tags=[],
            )
        ]

        with (
            mock.patch(
                "prowler.providers.aws.services.backup.backup_service.Backup",
                new=backup_client,
            ),
            mock.patch(
                "prowler.providers.aws.services.backup.backup_vaults_encrypted.backup_vaults_encrypted.backup_client",
                new=backup_client,
            ),
        ):
            # Test Check
            from prowler.providers.aws.services.backup.backup_vaults_encrypted.backup_vaults_encrypted import (
                backup_vaults_encrypted,
            )

            check = backup_vaults_encrypted()
            result = check.execute()

            assert len(result) == 1
            assert result[0].status == "PASS"
            assert (
                result[0].status_extended
                == f"Backup Vault {result[0].resource_id} is encrypted at rest."
            )
            assert result[0].resource_id == "MyBackupVault"
            assert result[0].resource_arn == backup_vault_arn
            assert result[0].region == AWS_REGION
            assert result[0].resource_tags == []
