# Configuration Overview

Normally, [Presets](presets.md) are used to configure a scan. However, there may be cases where you want to change BBOT's global defaults so a certain option is always set, even if it's not specified in a preset.

BBOT has a YAML config at `~/.config/bbot.yml`. This is the first config that BBOT loads, so it's a good place to put default settings like `http_proxy`, `max_threads`, or `http_user_agent`. You can also put any module settings here, including **API keys**.

For a list of all possible config options, see:

- [Global Options](#global-config-options)
- [Module Options](#module-config-options)

For examples of common config changes, see [Tips and Tricks](tips_and_tricks.md).

## Configuration Files

BBOT loads its config from the following files, in this order (last one loaded == highest priority):

- `~/.config/bbot/bbot.yml`  <-- Global BBOT config
- presets (`-p`)             <-- Presets are good for scan-specific settings
- command line (`-c`)        <-- CLI overrides everything

`bbot.yml` will be automatically created for you when you first run BBOT.

## YAML Config vs Command Line

You can specify config options either via the command line or the config. For example, if you want to proxy your BBOT scan through a local proxy like [Burp Suite](https://portswigger.net/burp), you could either do:

```bash
# send BBOT traffic through an HTTP proxy
bbot -t evilcorp.com -c http_proxy=http://127.0.0.1:8080
```

Or, in `~/.config/bbot/config.yml`:

```yaml title="~/.bbot/config/bbot.yml"
http_proxy: http://127.0.0.1:8080
```

These two are equivalent.

Config options specified via the command-line take precedence over all others. You can give BBOT a custom config file with `-c myconf.yml`, or individual arguments like this: `-c modules.shodan_dns.api_key=deadbeef`. To display the full and current BBOT config, including any command-line arguments, use `bbot -c`.

Note that placing the following in `bbot.yml`:
```yaml title="~/.bbot/config/bbot.yml"
modules:
  shodan_dns:
    api_key: deadbeef
```
Is the same as:
```bash
bbot -c modules.shodan_dns.api_key=deadbeef
```

## Global Config Options

Below is a full list of the config options supported, along with their defaults.

<!-- BBOT DEFAULT CONFIG -->
```yaml title="defaults.yml"
### BASIC OPTIONS ###

# BBOT working directory
home: ~/.bbot
# How many scan results to keep before cleaning up the older ones
keep_scans: 20
# Interval for displaying status messages
status_frequency: 15
# Include the raw data of files (i.e. PDFs, web screenshots) as base64 in the event
file_blobs: false
# Include the raw data of directories (i.e. git repos) as tar.gz base64 in the event
folder_blobs: false

### SCOPE ###

scope:
  # strict scope means only exact DNS names are considered in-scope
  # subdomains are not included unless they are explicitly provided in the target list
  strict: false
  # Filter by scope distance which events are displayed in the output
  # 0 == show only in-scope events (affiliates are always shown)
  # 1 == show all events up to distance-1 (1 hop from target)
  report_distance: 0
  # How far out from the main scope to search
  # Do not change this setting unless you know what you're doing
  search_distance: 0

### DNS ###

dns:
  # Completely disable DNS resolution (careful if you have IP whitelists/blacklists, consider using minimal=true instead)
  disable: false
  # Speed up scan by not creating any new DNS events, and only resolving A and AAAA records
  minimal: false
  # How many instances of the dns module to run concurrently
  threads: 25
  # How many concurrent DNS resolvers to use when brute-forcing
  # (under the hood this is passed through directly to massdns -s)
  brute_threads: 1000
  # nameservers to use for DNS brute-forcing
  # default is updated weekly and contains ~10K high-quality public servers
  brute_nameservers: https://raw.githubusercontent.com/blacklanternsecurity/public-dns-servers/master/nameservers.txt
  # How far away from the main target to explore via DNS resolution (independent of scope.search_distance)
  # This is safe to change
  search_distance: 1
  # Limit how many DNS records can be followed in a row (stop malicious/runaway DNS records)
  runaway_limit: 5
  # DNS query timeout
  timeout: 5
  # How many times to retry DNS queries
  retries: 1
  # Completely disable BBOT's DNS wildcard detection
  wildcard_disable: False
  # Disable BBOT's DNS wildcard detection for select domains
  wildcard_ignore: []
  # How many sanity checks to make when verifying wildcard DNS
  # Increase this value if BBOT's wildcard detection isn't working
  wildcard_tests: 10
  # Skip DNS requests for a certain domain and rdtype after encountering this many timeouts or SERVFAILs
  # This helps prevent faulty DNS servers from hanging up the scan
  abort_threshold: 50
  # Don't show PTR records containing IP addresses
  filter_ptrs: true
  # Enable/disable debug messages for DNS queries
  debug: false
  # For performance reasons, always skip these DNS queries
  # Microsoft's DNS infrastructure is misconfigured so that certain queries to mail.protection.outlook.com always time out
  omit_queries:
    - SRV:mail.protection.outlook.com
    - CNAME:mail.protection.outlook.com
    - TXT:mail.protection.outlook.com

### WEB ###

web:
  # HTTP proxy
  http_proxy:
  # Web user-agent
  user_agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/119.0.0.0 Safari/537.36 Edg/119.0.2151.97
  # Set the maximum number of HTTP links that can be followed in a row (0 == no spidering allowed)
  spider_distance: 0
  # Set the maximum directory depth for the web spider
  spider_depth: 1
  # Set the maximum number of links that can be followed per page
  spider_links_per_page: 25
  # HTTP timeout (for Python requests; API calls, etc.)
  http_timeout: 10
  # HTTP timeout (for httpx)
  httpx_timeout: 5
  # Custom HTTP headers (e.g. cookies, etc.)
  # in the format { "Header-Key": "header_value" }
  # These are attached to all in-scope HTTP requests
  # Note that some modules (e.g. github) may end up sending these to out-of-scope resources
  http_headers: {}
  # How many times to retry API requests
  # Note that this is a separate mechanism on top of HTTP retries
  # which will retry API requests that don't return a successful status code
  api_retries: 2
  # HTTP retries - try again if the raw connection fails
  http_retries: 1
  # HTTP retries (for httpx)
  httpx_retries: 1
  # Default sleep interval when rate limited by 429 (and retry-after isn't provided)
  429_sleep_interval: 30
  # Maximum sleep interval when rate limited by 429 (and an excessive retry-after is provided)
  429_max_sleep_interval: 60
  # Enable/disable debug messages for web requests/responses
  debug: false
  # Maximum number of HTTP redirects to follow
  http_max_redirects: 5
  # Whether to verify SSL certificates
  ssl_verify: false

### ENGINE ###

engine:
  debug: false

# Tool dependencies
deps:
  ffuf:
    version: "2.1.0"
  # How to handle installation of module dependencies
  # Choices are:
  #  - abort_on_failure (default) - if a module dependency fails to install, abort the scan
  #  - retry_failed - try again to install failed dependencies
  #  - ignore_failed - run the scan regardless of what happens with dependency installation
  #  - disable - completely disable BBOT's dependency system (you are responsible for installing tools, pip packages, etc.)
  behavior: abort_on_failure

### ADVANCED OPTIONS ###

# Load BBOT modules from these custom paths
module_dirs: []

# maximum runtime in seconds for each module's handle_event() is 60 minutes
# when the timeout is reached, the offending handle_event() will be cancelled and the module will move on to the next event
module_handle_event_timeout: 3600
# handle_batch() default timeout is 2 hours
module_handle_batch_timeout: 7200

# Infer certain events from others, e.g. IPs from IP ranges, DNS_NAMEs from URLs, etc.
speculate: True
# Passively search event data for URLs, hostnames, emails, etc.
excavate: True
# Summarize activity at the end of a scan
aggregate: True
# DNS resolution, wildcard detection, etc.
dnsresolve: True
# Cloud provider tagging
cloudcheck: True

# Strip querystring from URLs by default
url_querystring_remove: True
# When query string is retained, by default collapse parameter values down to a single value per parameter
url_querystring_collapse: True

# Completely ignore URLs with these extensions
url_extension_blacklist:
  # images
  - png
  - jpg
  - bmp
  - ico
  - jpeg
  - gif
  - svg
  - webp
  # web/fonts
  - css
  - woff
  - woff2
  - ttf
  - eot
  - sass
  - scss
  # audio
  - mp3
  - m4a
  - wav
  - flac
  # video
  - mp4
  - mkv
  - avi
  - wmv
  - mov
  - flv
  - webm

# URLs with these extensions are not distributed to modules unless the module opts in via `accept_url_special = True`
# They are also excluded from output. If you want to see them in output, remove them from this list.
url_extension_special:
  - js

# These url extensions are almost always static, so we exclude them from modules that fuzz things
url_extension_static:
  - pdf
  - doc
  - docx
  - xls
  - xlsx
  - ppt
  - pptx
  - txt
  - csv
  - xml
  - yaml
  - ini
  - log
  - conf
  - cfg
  - env
  - md
  - rtf
  - tiff
  - bmp
  - jpg
  - jpeg
  - png
  - gif
  - svg
  - ico
  - mp3
  - wav
  - flac
  - mp4
  - mov
  - avi
  - mkv
  - webm
  - zip
  - tar
  - gz
  - bz2
  - 7z
  - rar

parameter_blacklist:
  - __VIEWSTATE
  - __EVENTARGUMENT
  - __EVENTVALIDATION
  - __EVENTTARGET
  - __EVENTARGUMENT
  - __VIEWSTATEGENERATOR
  - __SCROLLPOSITIONY
  - __SCROLLPOSITIONX
  - ASP.NET_SessionId
  - PHPSESSID
  - __cf_bm
  - f5_cspm

parameter_blacklist_prefixes:
  - TS01
  - BIGipServer
  - incap_
  - visid_incap_
  - AWSALB
  - utm_
  - ApplicationGatewayAffinity
  - JSESSIONID
  - ARRAffinity

# Don't output these types of events (they are still distributed to modules)
omit_event_types:
  - HTTP_RESPONSE
  - RAW_TEXT
  - URL_UNVERIFIED
  - DNS_NAME_UNRESOLVED
  - FILESYSTEM
  - WEB_PARAMETER
  - RAW_DNS_RECORD
  # - IP_ADDRESS

# Custom interactsh server settings
interactsh_server: null
interactsh_token: null
interactsh_disable: false

```
<!-- END BBOT DEFAULT CONFIG -->

## Module Config Options

Many modules accept their own configuration options. These options have the ability to change their behavior. For example, the `portscan` module accepts options for `ports`, `rate`, etc. Below is a list of all possible module config options.

### Universal Module Options

In addition to the stated options for each module, the following universal options are also accepted:

<!-- BBOT UNIVERSAL MODULE OPTIONS -->
**batch_size**: The number of events to process in a single batch (only applies to batch modules)
**module_threads**: How many event handlers to run in parallel
**module_timeout**: Max time in seconds to spend handling each event or batch of events

<!-- END BBOT UNIVERSAL MODULE OPTIONS -->

### Module Options

<!-- BBOT MODULE OPTIONS -->
| Config Option                                       | Type     | Description                                                                                                                                                                                                                                                                                                                    | Default                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
|-----------------------------------------------------|----------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| modules.baddns.custom_nameservers                   | list     | Force BadDNS to use a list of custom nameservers                                                                                                                                                                                                                                                                               | []                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.baddns.enabled_submodules                   | list     | A list of submodules to enable. Empty list (default) enables CNAME, TXT and MX Only                                                                                                                                                                                                                                            | []                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.baddns.only_high_confidence                 | bool     | Do not emit low-confidence or generic detections                                                                                                                                                                                                                                                                               | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.baddns_direct.custom_nameservers            | list     | Force BadDNS to use a list of custom nameservers                                                                                                                                                                                                                                                                               | []                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.baddns_zone.custom_nameservers              | list     | Force BadDNS to use a list of custom nameservers                                                                                                                                                                                                                                                                               | []                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.baddns_zone.only_high_confidence            | bool     | Do not emit low-confidence or generic detections                                                                                                                                                                                                                                                                               | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.badsecrets.custom_secrets                   | NoneType | Include custom secrets loaded from a local file                                                                                                                                                                                                                                                                                | None                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.bucket_amazon.permutations                  | bool     | Whether to try permutations                                                                                                                                                                                                                                                                                                    | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.bucket_digitalocean.permutations            | bool     | Whether to try permutations                                                                                                                                                                                                                                                                                                    | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.bucket_firebase.permutations                | bool     | Whether to try permutations                                                                                                                                                                                                                                                                                                    | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.bucket_google.permutations                  | bool     | Whether to try permutations                                                                                                                                                                                                                                                                                                    | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.bucket_microsoft.permutations               | bool     | Whether to try permutations                                                                                                                                                                                                                                                                                                    | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.dnsbrute.max_depth                          | int      | How many subdomains deep to brute force, i.e. 5.4.3.2.1.evilcorp.com                                                                                                                                                                                                                                                           | 5                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.dnsbrute.wordlist                           | str      | Subdomain wordlist URL                                                                                                                                                                                                                                                                                                         | https://raw.githubusercontent.com/danielmiessler/SecLists/master/Discovery/DNS/subdomains-top1million-5000.txt                                                                                                                                                                                                                                                                                                                                          |
| modules.dnsbrute_mutations.max_mutations            | int      | Maximum number of target-specific mutations to try per subdomain                                                                                                                                                                                                                                                               | 100                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.dnscommonsrv.max_depth                      | int      | The maximum subdomain depth to brute-force SRV records                                                                                                                                                                                                                                                                         | 2                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.ffuf.extensions                             | str      | Optionally include a list of extensions to extend the keyword with (comma separated)                                                                                                                                                                                                                                           |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.ffuf.ignore_case                            | bool     | Only put lowercase words into the wordlist                                                                                                                                                                                                                                                                                     | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.ffuf.lines                                  | int      | take only the first N lines from the wordlist when finding directories                                                                                                                                                                                                                                                         | 5000                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.ffuf.max_depth                              | int      | the maximum directory depth to attempt to solve                                                                                                                                                                                                                                                                                | 0                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.ffuf.rate                                   | int      | Rate of requests per second (default: 0)                                                                                                                                                                                                                                                                                       | 0                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.ffuf.wordlist                               | str      | Specify wordlist to use when finding directories                                                                                                                                                                                                                                                                               | https://raw.githubusercontent.com/danielmiessler/SecLists/master/Discovery/Web-Content/raft-small-directories.txt                                                                                                                                                                                                                                                                                                                                       |
| modules.ffuf_shortnames.extensions                  | str      | Optionally include a list of extensions to extend the keyword with (comma separated)                                                                                                                                                                                                                                           |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.ffuf_shortnames.find_common_prefixes        | bool     | Attempt to automatically detect common prefixes and make additional ffuf runs against them                                                                                                                                                                                                                                     | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.ffuf_shortnames.find_delimiters             | bool     | Attempt to detect common delimiters and make additional ffuf runs against them                                                                                                                                                                                                                                                 | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.ffuf_shortnames.find_subwords               | bool     | Attempt to detect subwords and make additional ffuf runs against them                                                                                                                                                                                                                                                          | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.ffuf_shortnames.ignore_redirects            | bool     | Explicitly ignore redirects (301,302)                                                                                                                                                                                                                                                                                          | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.ffuf_shortnames.max_depth                   | int      | the maximum directory depth to attempt to solve                                                                                                                                                                                                                                                                                | 1                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.ffuf_shortnames.max_predictions             | int      | The maximum number of predictions to generate per shortname prefix                                                                                                                                                                                                                                                             | 250                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.ffuf_shortnames.rate                        | int      | Rate of requests per second (default: 0)                                                                                                                                                                                                                                                                                       | 0                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.ffuf_shortnames.version                     | str      | ffuf version                                                                                                                                                                                                                                                                                                                   | 2.0.0                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.ffuf_shortnames.wordlist_extensions         | str      | Specify wordlist to use when making extension lists                                                                                                                                                                                                                                                                            |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.filedownload.extensions                     | list     | File extensions to download                                                                                                                                                                                                                                                                                                    | ['bak', 'bash', 'bashrc', 'cfg', 'conf', 'crt', 'csv', 'db', 'dll', 'doc', 'docx', 'exe', 'ica', 'indd', 'ini', 'jar', 'json', 'key', 'log', 'markdown', 'md', 'msi', 'odg', 'odp', 'ods', 'odt', 'pdf', 'pem', 'pps', 'ppsx', 'ppt', 'pptx', 'ps1', 'pub', 'raw', 'rdp', 'rsa', 'sh', 'sql', 'sqlite', 'swp', 'sxw', 'tar.gz', 'tgz', 'tar', 'txt', 'vbs', 'war', 'wpd', 'xls', 'xlsx', 'xml', 'yaml', 'yml', 'zip', 'lzma', 'rar', '7z', 'xz', 'bz2'] |
| modules.filedownload.max_filesize                   | str      | Cancel download if filesize is greater than this size                                                                                                                                                                                                                                                                          | 10MB                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.filedownload.output_folder                  | str      | Folder to download files to. If not specified, downloaded files will be deleted when the scan completes, to minimize disk usage.                                                                                                                                                                                               |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.fingerprintx.skip_common_web                | bool     | Skip common web ports such as 80, 443, 8080, 8443, etc.                                                                                                                                                                                                                                                                        | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.fingerprintx.version                        | str      | fingerprintx version                                                                                                                                                                                                                                                                                                           | 1.1.4                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.generic_ssrf.skip_dns_interaction           | bool     | Do not report DNS interactions (only HTTP interaction)                                                                                                                                                                                                                                                                         | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.gitlab_com.api_key                          | str      | GitLab access token (for gitlab.com/org only)                                                                                                                                                                                                                                                                                  |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.gitlab_onprem.api_key                       | str      | GitLab access token (for self-hosted instances only)                                                                                                                                                                                                                                                                           |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.gowitness.chrome_path                       | str      | Path to chrome executable                                                                                                                                                                                                                                                                                                      |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.gowitness.idle_timeout                      | int      | Skip the current gowitness batch if it stalls for longer than this many seconds                                                                                                                                                                                                                                                | 1800                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.gowitness.output_path                       | str      | Where to save screenshots                                                                                                                                                                                                                                                                                                      |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.gowitness.resolution_x                      | int      | Screenshot resolution x                                                                                                                                                                                                                                                                                                        | 1440                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.gowitness.resolution_y                      | int      | Screenshot resolution y                                                                                                                                                                                                                                                                                                        | 900                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.gowitness.social                            | bool     | Whether to screenshot social media webpages                                                                                                                                                                                                                                                                                    | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.gowitness.threads                           | int      | How many gowitness threads to spawn (default is number of CPUs x 2)                                                                                                                                                                                                                                                            | 0                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.gowitness.timeout                           | int      | Preflight check timeout                                                                                                                                                                                                                                                                                                        | 10                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.gowitness.version                           | str      | Gowitness version                                                                                                                                                                                                                                                                                                              | 3.0.5                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.graphql_introspection.graphql_endpoint_urls | list     | List of GraphQL endpoint to suffix to the target URL                                                                                                                                                                                                                                                                           | ['/', '/graphql', '/v1/graphql']                                                                                                                                                                                                                                                                                                                                                                                                                        |
| modules.graphql_introspection.output_folder         | str      | Folder to save the GraphQL schemas to                                                                                                                                                                                                                                                                                          |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.httpx.in_scope_only                         | bool     | Only visit web reparents that are in scope.                                                                                                                                                                                                                                                                                    | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.httpx.max_response_size                     | int      | Max response size in bytes                                                                                                                                                                                                                                                                                                     | 5242880                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| modules.httpx.probe_all_ips                         | bool     | Probe all the ips associated with same host                                                                                                                                                                                                                                                                                    | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.httpx.store_responses                       | bool     | Save raw HTTP responses to scan folder                                                                                                                                                                                                                                                                                         | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.httpx.threads                               | int      | Number of httpx threads to use                                                                                                                                                                                                                                                                                                 | 50                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.httpx.version                               | str      | httpx version                                                                                                                                                                                                                                                                                                                  | 1.2.5                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.iis_shortnames.detect_only                  | bool     | Only detect the vulnerability and do not run the shortname scanner                                                                                                                                                                                                                                                             | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.iis_shortnames.max_node_count               | int      | Limit how many nodes to attempt to resolve on any given recursion branch                                                                                                                                                                                                                                                       | 50                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.iis_shortnames.speculate_magic_urls         | bool     | Attempt to discover iis 'magic' special folders                                                                                                                                                                                                                                                                                | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.legba.concurrency                           | int      | Number of concurrent workers, gets overridden for SSH                                                                                                                                                                                                                                                                          | 3                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.legba.ftp_wordlist                          | str      | Wordlist URL for FTP combined username:password wordlist, newline separated                                                                                                                                                                                                                                                    | https://raw.githubusercontent.com/danielmiessler/SecLists/refs/heads/master/Passwords/Default-Credentials/ftp-betterdefaultpasslist.txt                                                                                                                                                                                                                                                                                                                 |
| modules.legba.mssql_wordlist                        | str      | Wordlist URL for MSSQL combined username:password wordlist, newline separated                                                                                                                                                                                                                                                  | https://raw.githubusercontent.com/danielmiessler/SecLists/refs/heads/master/Passwords/Default-Credentials/mssql-betterdefaultpasslist.txt                                                                                                                                                                                                                                                                                                               |
| modules.legba.mysql_wordlist                        | str      | Wordlist URL for MySQL combined username:password wordlist, newline separated                                                                                                                                                                                                                                                  | https://raw.githubusercontent.com/danielmiessler/SecLists/refs/heads/master/Passwords/Default-Credentials/mysql-betterdefaultpasslist.txt                                                                                                                                                                                                                                                                                                               |
| modules.legba.postgresql_wordlist                   | str      | Wordlist URL for PostgreSQL combined username:password wordlist, newline separated                                                                                                                                                                                                                                             | https://raw.githubusercontent.com/danielmiessler/SecLists/refs/heads/master/Passwords/Default-Credentials/postgres-betterdefaultpasslist.txt                                                                                                                                                                                                                                                                                                            |
| modules.legba.rate_limit                            | int      | Limit the number of requests per second, gets overridden for SSH                                                                                                                                                                                                                                                               | 3                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.legba.ssh_wordlist                          | str      | Wordlist URL for SSH combined username:password wordlist, newline separated                                                                                                                                                                                                                                                    | https://raw.githubusercontent.com/danielmiessler/SecLists/refs/heads/master/Passwords/Default-Credentials/ssh-betterdefaultpasslist.txt                                                                                                                                                                                                                                                                                                                 |
| modules.legba.telnet_wordlist                       | str      | Wordlist URL for TELNET combined username:password wordlist, newline separated                                                                                                                                                                                                                                                 | https://raw.githubusercontent.com/danielmiessler/SecLists/refs/heads/master/Passwords/Default-Credentials/telnet-betterdefaultpasslist.txt                                                                                                                                                                                                                                                                                                              |
| modules.legba.version                               | str      | legba version                                                                                                                                                                                                                                                                                                                  | 1.1.1                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.legba.vnc_wordlist                          | str      | Wordlist URL for VNC password wordlist, newline separated                                                                                                                                                                                                                                                                      | https://raw.githubusercontent.com/danielmiessler/SecLists/refs/heads/master/Passwords/Default-Credentials/vnc-betterdefaultpasslist.txt                                                                                                                                                                                                                                                                                                                 |
| modules.lightfuzz.avoid_wafs                        | bool     | Avoid running against confirmed WAFs, which are likely to block lightfuzz requests                                                                                                                                                                                                                                             | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.lightfuzz.disable_post                      | bool     | Disable processing of POST parameters, avoiding form submissions.                                                                                                                                                                                                                                                              | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.lightfuzz.enabled_submodules                | list     | A list of submodules to enable. Empty list enabled all modules.                                                                                                                                                                                                                                                                | ['sqli', 'cmdi', 'xss', 'path', 'ssti', 'crypto', 'serial', 'esi']                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.lightfuzz.force_common_headers              | bool     | Force emit commonly exploitable parameters that may be difficult to detect                                                                                                                                                                                                                                                     | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.lightfuzz.try_get_as_post                   | bool     | For each GETPARAM, also fuzz it as a POSTPARAM (in addition to normal GET fuzzing).                                                                                                                                                                                                                                            | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.lightfuzz.try_post_as_get                   | bool     | For each POSTPARAM, also fuzz it as a GETPARAM (in addition to normal POST fuzzing).                                                                                                                                                                                                                                           | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.medusa.snmp_versions                        | list     | List of SNMP versions to attempt against the SNMP server (default ['1', '2C'])                                                                                                                                                                                                                                                 | ['1', '2C']                                                                                                                                                                                                                                                                                                                                                                                                                                             |
| modules.medusa.snmp_wordlist                        | str      | Wordlist url for SNMP community strings, newline separated (default https://raw.githubusercontent.com/danielmiessler/SecLists/refs/heads/master/Discovery/SNMP/snmp.txt)                                                                                                                                                       | https://raw.githubusercontent.com/danielmiessler/SecLists/refs/heads/master/Discovery/SNMP/common-snmp-community-strings.txt                                                                                                                                                                                                                                                                                                                            |
| modules.medusa.threads                              | int      | Number of communities to be tested concurrently (default 5)                                                                                                                                                                                                                                                                    | 5                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.medusa.timeout_s                            | int      | Wait time for the SNMP response(s) once at the end of all attempts (default 5)                                                                                                                                                                                                                                                 | 5                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.medusa.wait_microseconds                    | int      | Wait time after every SNMP request in microseconds (default 200)                                                                                                                                                                                                                                                               | 200                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.ntlm.try_all                                | bool     | Try every NTLM endpoint                                                                                                                                                                                                                                                                                                        | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.nuclei.batch_size                           | int      | Number of targets to send to Nuclei per batch (default 200)                                                                                                                                                                                                                                                                    | 200                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.nuclei.budget                               | int      | Used in budget mode to set the number of allowed requests per host                                                                                                                                                                                                                                                             | 1                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.nuclei.concurrency                          | int      | maximum number of templates to be executed in parallel (default 25)                                                                                                                                                                                                                                                            | 25                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.nuclei.directory_only                       | bool     | Filter out 'file' URL event (default True)                                                                                                                                                                                                                                                                                     | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.nuclei.etags                                | str      | tags to exclude from the scan                                                                                                                                                                                                                                                                                                  |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.nuclei.mode                                 | str      | manual &#124; technology &#124; severe &#124; budget. Technology: Only activate based on technology events that match nuclei tags (nuclei -as mode). Manual (DEFAULT): Fully manual settings. Severe: Only critical and high severity templates without intrusive. Budget: Limit Nuclei to a specified number of HTTP requests | manual                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
| modules.nuclei.module_timeout                       | int      | Max time in seconds to spend handling each batch of events                                                                                                                                                                                                                                                                     | 21600                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.nuclei.ratelimit                            | int      | maximum number of requests to send per second (default 150)                                                                                                                                                                                                                                                                    | 150                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.nuclei.retries                              | int      | number of times to retry a failed request (default 0)                                                                                                                                                                                                                                                                          | 0                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.nuclei.severity                             | str      | Filter based on severity field available in the template.                                                                                                                                                                                                                                                                      |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.nuclei.silent                               | bool     | Don't display nuclei's banner or status messages                                                                                                                                                                                                                                                                               | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.nuclei.tags                                 | str      | execute a subset of templates that contain the provided tags                                                                                                                                                                                                                                                                   |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.nuclei.templates                            | str      | template or template directory paths to include in the scan                                                                                                                                                                                                                                                                    |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.nuclei.version                              | str      | nuclei version                                                                                                                                                                                                                                                                                                                 | 3.7.0                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.oauth.try_all                               | bool     | Check for OAUTH/IODC on every subdomain and URL.                                                                                                                                                                                                                                                                               | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.paramminer_cookies.recycle_words            | bool     | Attempt to use words found during the scan on all other endpoints                                                                                                                                                                                                                                                              | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.paramminer_cookies.skip_boring_words        | bool     | Remove commonly uninteresting words from the wordlist                                                                                                                                                                                                                                                                          | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.paramminer_cookies.wordlist                 | str      | Define the wordlist to be used to derive cookies                                                                                                                                                                                                                                                                               |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.paramminer_getparams.recycle_words          | bool     | Attempt to use words found during the scan on all other endpoints                                                                                                                                                                                                                                                              | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.paramminer_getparams.skip_boring_words      | bool     | Remove commonly uninteresting words from the wordlist                                                                                                                                                                                                                                                                          | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.paramminer_getparams.wordlist               | str      | Define the wordlist to be used to derive headers                                                                                                                                                                                                                                                                               |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.paramminer_headers.recycle_words            | bool     | Attempt to use words found during the scan on all other endpoints                                                                                                                                                                                                                                                              | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.paramminer_headers.skip_boring_words        | bool     | Remove commonly uninteresting words from the wordlist                                                                                                                                                                                                                                                                          | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.paramminer_headers.wordlist                 | str      | Define the wordlist to be used to derive headers                                                                                                                                                                                                                                                                               |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.portscan.adapter                            | str      | Manually specify a network interface, such as "eth0" or "tun0". If not specified, the first network interface found with a default gateway will be used.                                                                                                                                                                       |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.portscan.adapter_ip                         | str      | Send packets using this IP address. Not needed unless masscan's autodetection fails                                                                                                                                                                                                                                            |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.portscan.adapter_mac                        | str      | Send packets using this as the source MAC address. Not needed unless masscan's autodetection fails                                                                                                                                                                                                                             |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.portscan.module_timeout                     | int      | Max time in seconds to spend handling each batch of events                                                                                                                                                                                                                                                                     | 259200                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
| modules.portscan.ping_first                         | bool     | Only portscan hosts that reply to pings                                                                                                                                                                                                                                                                                        | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.portscan.ping_only                          | bool     | Ping sweep only, no portscan                                                                                                                                                                                                                                                                                                   | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.portscan.ports                              | str      | Ports to scan                                                                                                                                                                                                                                                                                                                  |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.portscan.rate                               | int      | Rate in packets per second                                                                                                                                                                                                                                                                                                     | 300                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.portscan.router_mac                         | str      | Send packets to this MAC address as the destination. Not needed unless masscan's autodetection fails                                                                                                                                                                                                                           |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.portscan.top_ports                          | int      | Top ports to scan (default 100) (to override, specify 'ports')                                                                                                                                                                                                                                                                 | 100                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.portscan.wait                               | int      | Seconds to wait for replies after scan is complete                                                                                                                                                                                                                                                                             | 5                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.retirejs.node_version                       | str      | Node.js version to install locally                                                                                                                                                                                                                                                                                             | 18.19.1                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| modules.retirejs.severity                           | str      | Minimum severity level to report (none, low, medium, high, critical)                                                                                                                                                                                                                                                           | medium                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
| modules.retirejs.version                            | str      | retire.js version                                                                                                                                                                                                                                                                                                              | 5.3.0                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.robots.include_allow                        | bool     | Include 'Allow' Entries                                                                                                                                                                                                                                                                                                        | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.robots.include_disallow                     | bool     | Include 'Disallow' Entries                                                                                                                                                                                                                                                                                                     | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.robots.include_sitemap                      | bool     | Include 'sitemap' entries                                                                                                                                                                                                                                                                                                      | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.securitytxt.emails                          | bool     | emit EMAIL_ADDRESS events                                                                                                                                                                                                                                                                                                      | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.securitytxt.urls                            | bool     | emit URL_UNVERIFIED events                                                                                                                                                                                                                                                                                                     | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.sslcert.skip_non_ssl                        | bool     | Don't try common non-SSL ports                                                                                                                                                                                                                                                                                                 | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.sslcert.timeout                             | float    | Socket connect timeout in seconds                                                                                                                                                                                                                                                                                              | 5.0                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.telerik.exploit_RAU_crypto                  | bool     | Attempt to confirm any RAU AXD detections are vulnerable                                                                                                                                                                                                                                                                       | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.telerik.include_subdirs                     | bool     | Include subdirectories in the scan (off by default)                                                                                                                                                                                                                                                                            | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.url_manipulation.allow_redirects            | bool     | Allowing redirects will sometimes create false positives. Disallowing will sometimes create false negatives. Allowed by default.                                                                                                                                                                                               | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.vhost.force_basehost                        | str      | Use a custom base host (e.g. evilcorp.com) instead of the default behavior of using the current URL                                                                                                                                                                                                                            |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.vhost.lines                                 | int      | take only the first N lines from the wordlist when finding directories                                                                                                                                                                                                                                                         | 5000                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.vhost.wordlist                              | str      | Wordlist containing subdomains                                                                                                                                                                                                                                                                                                 | https://raw.githubusercontent.com/danielmiessler/SecLists/master/Discovery/DNS/subdomains-top1million-5000.txt                                                                                                                                                                                                                                                                                                                                          |
| modules.wafw00f.generic_detect                      | bool     | When no specific WAF detections are made, try to perform a generic detect                                                                                                                                                                                                                                                      | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.wpscan.api_key                              | str      | WPScan API Key                                                                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.wpscan.connection_timeout                   | int      | The connection timeout in seconds (default 2)                                                                                                                                                                                                                                                                                  | 2                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.wpscan.disable_tls_checks                   | bool     | Disables the SSL/TLS certificate verification (Default True)                                                                                                                                                                                                                                                                   | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.wpscan.enumerate                            | str      | Enumeration Process see wpscan help documentation (default: vp,vt,cb,dbe)                                                                                                                                                                                                                                                      | vp,vt,cb,dbe                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| modules.wpscan.force                                | bool     | Do not check if the target is running WordPress or returns a 403                                                                                                                                                                                                                                                               | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.wpscan.request_timeout                      | int      | The request timeout in seconds (default 5)                                                                                                                                                                                                                                                                                     | 5                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.wpscan.threads                              | int      | How many wpscan threads to spawn (default is 5)                                                                                                                                                                                                                                                                                | 5                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.anubisdb.limit                              | int      | Limit the number of subdomains returned per query (increasing this may slow the scan due to garbage results from this API)                                                                                                                                                                                                     | 1000                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.apkpure.output_folder                       | str      | Folder to download APKs to. If not specified, downloaded APKs will be deleted when the scan completes, to minimize disk usage.                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.bevigil.api_key                             | str      | BeVigil OSINT API Key                                                                                                                                                                                                                                                                                                          |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.bevigil.urls                                | bool     | Emit URLs in addition to DNS_NAMEs                                                                                                                                                                                                                                                                                             | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.bucket_file_enum.file_limit                 | int      | Limit the number of files downloaded per bucket                                                                                                                                                                                                                                                                                | 50                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.bufferoverrun.api_key                       | str      | BufferOverrun API key                                                                                                                                                                                                                                                                                                          |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.bufferoverrun.commercial                    | bool     | Use commercial API                                                                                                                                                                                                                                                                                                             | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.builtwith.api_key                           | str      | Builtwith API key                                                                                                                                                                                                                                                                                                              |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.builtwith.redirects                         | bool     | Also look up inbound and outbound redirects                                                                                                                                                                                                                                                                                    | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.c99.api_key                                 | str      | c99.nl API key                                                                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.censys_dns.api_key                          | str      | Censys.io API Key in the format of 'key:secret'                                                                                                                                                                                                                                                                                |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.censys_dns.max_pages                        | int      | Maximum number of pages to fetch (100 results per page)                                                                                                                                                                                                                                                                        | 5                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.censys_ip.api_key                           | str      | Censys.io API Key in the format of 'key:secret'                                                                                                                                                                                                                                                                                |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.censys_ip.dns_names_limit                   | int      | Maximum number of DNS names to extract from dns.names (default 100)                                                                                                                                                                                                                                                            | 100                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.censys_ip.in_scope_only                     | bool     | Only query in-scope IPs. If False, will query up to distance 1.                                                                                                                                                                                                                                                                | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.chaos.api_key                               | str      | Chaos API key                                                                                                                                                                                                                                                                                                                  |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.credshed.credshed_url                       | str      | URL of credshed server                                                                                                                                                                                                                                                                                                         |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.credshed.password                           | str      | Credshed password                                                                                                                                                                                                                                                                                                              |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.credshed.username                           | str      | Credshed username                                                                                                                                                                                                                                                                                                              |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.dehashed.api_key                            | str      | DeHashed API Key                                                                                                                                                                                                                                                                                                               |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.dnsbimi.emit_raw_dns_records                | bool     | Emit RAW_DNS_RECORD events                                                                                                                                                                                                                                                                                                     | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.dnsbimi.emit_urls                           | bool     | Emit URL_UNVERIFIED events                                                                                                                                                                                                                                                                                                     | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.dnsbimi.selectors                           | str      | CSV list of BIMI selectors to check                                                                                                                                                                                                                                                                                            | default,email,mail,bimi                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| modules.dnscaa.dns_names                            | bool     | emit DNS_NAME events                                                                                                                                                                                                                                                                                                           | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.dnscaa.emails                               | bool     | emit EMAIL_ADDRESS events                                                                                                                                                                                                                                                                                                      | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.dnscaa.in_scope_only                        | bool     | Only check in-scope domains                                                                                                                                                                                                                                                                                                    | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.dnscaa.urls                                 | bool     | emit URL_UNVERIFIED events                                                                                                                                                                                                                                                                                                     | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.dnstlsrpt.emit_emails                       | bool     | Emit EMAIL_ADDRESS events                                                                                                                                                                                                                                                                                                      | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.dnstlsrpt.emit_raw_dns_records              | bool     | Emit RAW_DNS_RECORD events                                                                                                                                                                                                                                                                                                     | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.dnstlsrpt.emit_urls                         | bool     | Emit URL_UNVERIFIED events                                                                                                                                                                                                                                                                                                     | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.docker_pull.all_tags                        | bool     | Download all tags from each registry (Default False)                                                                                                                                                                                                                                                                           | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.docker_pull.output_folder                   | str      | Folder to download docker repositories to. If not specified, downloaded docker images will be deleted when the scan completes, to minimize disk usage.                                                                                                                                                                         |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.extractous.extensions                       | list     | File extensions to parse                                                                                                                                                                                                                                                                                                       | ['bak', 'bash', 'bashrc', 'conf', 'cfg', 'crt', 'csv', 'db', 'sqlite', 'doc', 'docx', 'ica', 'indd', 'ini', 'json', 'key', 'pub', 'log', 'markdown', 'md', 'odg', 'odp', 'ods', 'odt', 'pdf', 'pem', 'pps', 'ppsx', 'ppt', 'pptx', 'ps1', 'rdp', 'rsa', 'sh', 'sql', 'swp', 'sxw', 'txt', 'vbs', 'wpd', 'xls', 'xlsx', 'xml', 'yml', 'yaml']                                                                                                            |
| modules.fullhunt.api_key                            | str      | FullHunt API Key                                                                                                                                                                                                                                                                                                               |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.git_clone.api_key                           | str      | Github token                                                                                                                                                                                                                                                                                                                   |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.git_clone.output_folder                     | str      | Folder to clone repositories to. If not specified, cloned repositories will be deleted when the scan completes, to minimize disk usage.                                                                                                                                                                                        |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.gitdumper.fuzz_tags                         | bool     | Fuzz for common git tag names (v0.0.1, 0.0.2, etc.) up to the max_semanic_version                                                                                                                                                                                                                                              | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.gitdumper.max_semanic_version               | int      |` Maximum version number to fuzz for (default < v10.10.10)                                                                                                                                                                                                                                                                       `| 10                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.gitdumper.output_folder                     | str      | Folder to download repositories to. If not specified, downloaded repositories will be deleted when the scan completes, to minimize disk usage.                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.github_codesearch.api_key                   | str      | Github token                                                                                                                                                                                                                                                                                                                   |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.github_codesearch.limit                     | int      | Limit code search to this many results                                                                                                                                                                                                                                                                                         | 100                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.github_org.api_key                          | str      | Github token                                                                                                                                                                                                                                                                                                                   |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.github_org.include_member_repos             | bool     | Also enumerate organization members' repositories                                                                                                                                                                                                                                                                              | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.github_org.include_members                  | bool     | Enumerate organization members                                                                                                                                                                                                                                                                                                 | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.github_usersearch.api_key                   | str      | Github token                                                                                                                                                                                                                                                                                                                   |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.github_workflows.api_key                    | str      | Github token                                                                                                                                                                                                                                                                                                                   |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.github_workflows.num_logs                   | int      | For each workflow fetch the last N successful runs logs (max 100)                                                                                                                                                                                                                                                              | 1                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.github_workflows.output_folder              | str      | Folder to download workflow logs and artifacts to                                                                                                                                                                                                                                                                              |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.hunterio.api_key                            | str      | Hunter.IO API key                                                                                                                                                                                                                                                                                                              |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.ip2location.api_key                         | str      | IP2location.io API Key                                                                                                                                                                                                                                                                                                         |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.ip2location.lang                            | str      | Translation information(ISO639-1). The translation is only applicable for continent, country, region and city name.                                                                                                                                                                                                            |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.ipneighbor.num_bits                         | int      | Netmask size (in CIDR notation) to check. Default is 4 bits (16 hosts)                                                                                                                                                                                                                                                         | 4                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.ipstack.api_key                             | str      | IPStack GeoIP API Key                                                                                                                                                                                                                                                                                                          |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.jadx.threads                                | int      | Maximum jadx threads for extracting apk's, default: 4                                                                                                                                                                                                                                                                          | 4                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.leakix.api_key                              | str      | LeakIX API Key                                                                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.otx.api_key                                 | str      | OTX API key                                                                                                                                                                                                                                                                                                                    |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.passivetotal.api_key                        | str      | PassiveTotal API Key in the format of 'username:api_key'                                                                                                                                                                                                                                                                       |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.pgp.search_urls                             | list     | PGP key servers to search                                                                                                                                                                                                                                                                                                      |` ['https://keyserver.ubuntu.com/pks/lookup?fingerprint=on&op=vindex&search=<query>', 'http://the.earth.li:11371/pks/lookup?fingerprint=on&op=vindex&search=<query>', 'https://pgpkeys.eu/pks/lookup?search=<query>&op=index', 'https://pgp.mit.edu/pks/lookup?search=<query>&op=index']                                                                                                                                                                  `|
| modules.portfilter.allowed_cdn_ports                | str      | Comma-separated list of ports that are allowed to be scanned for CDNs                                                                                                                                                                                                                                                          | 80,443                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
| modules.portfilter.cdn_tags                         | str      | Comma-separated list of tags to skip, e.g. 'cdn,cloud'                                                                                                                                                                                                                                                                         | cdn-                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.postman.api_key                             | str      | Postman API Key                                                                                                                                                                                                                                                                                                                |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.postman_download.api_key                    | str      | Postman API Key                                                                                                                                                                                                                                                                                                                |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.postman_download.output_folder              | str      | Folder to download postman workspaces to. If not specified, downloaded workspaces will be deleted when the scan completes, to minimize disk usage.                                                                                                                                                                             |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.securitytrails.api_key                      | str      | SecurityTrails API key                                                                                                                                                                                                                                                                                                         |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.shodan_dns.api_key                          | str      | Shodan API key                                                                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.shodan_idb.retries                          | NoneType | How many times to retry API requests (e.g. after a 429 error). Overrides the global web.api_retries setting.                                                                                                                                                                                                                   | None                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.subdomainradar.api_key                      | str      | SubDomainRadar.io API key                                                                                                                                                                                                                                                                                                      |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.subdomainradar.group                        | str      | The enumeration group to use. Choose from fast, medium, deep                                                                                                                                                                                                                                                                   | fast                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.subdomainradar.timeout                      | int      | Timeout in seconds                                                                                                                                                                                                                                                                                                             | 120                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.trickest.api_key                            | str      | Trickest API key                                                                                                                                                                                                                                                                                                               |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.trufflehog.concurrency                      | int      | Number of concurrent workers                                                                                                                                                                                                                                                                                                   | 8                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| modules.trufflehog.config                           | str      | File path or URL to YAML trufflehog config                                                                                                                                                                                                                                                                                     |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.trufflehog.deleted_forks                    | bool     | Scan for deleted github forks. WARNING: This is SLOW. For a smaller repository, this process can take 20 minutes. For a larger repository, it could take hours.                                                                                                                                                                | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.trufflehog.only_verified                    | bool     | Only report credentials that have been verified                                                                                                                                                                                                                                                                                | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.trufflehog.version                          | str      | trufflehog version                                                                                                                                                                                                                                                                                                             | 3.90.8                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
| modules.urlscan.urls                                | bool     | Emit URLs in addition to DNS_NAMEs                                                                                                                                                                                                                                                                                             | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.virustotal.api_key                          | str      | VirusTotal API Key                                                                                                                                                                                                                                                                                                             |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.wayback.garbage_threshold                   | int      | Dedupe similar urls if they are in a group of this size or higher (lower values == less garbage data)                                                                                                                                                                                                                          | 10                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.wayback.urls                                | bool     | emit URLs in addition to DNS_NAMEs                                                                                                                                                                                                                                                                                             | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.asset_inventory.output_file                 | str      | Set a custom output file                                                                                                                                                                                                                                                                                                       |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.asset_inventory.recheck                     | bool     | When use_previous=True, don't retain past details like open ports or findings. Instead, allow them to be rediscovered by the new scan                                                                                                                                                                                          | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.asset_inventory.summary_netmask             | int      | Subnet mask to use when summarizing IP addresses at end of scan                                                                                                                                                                                                                                                                | 16                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.asset_inventory.use_previous                | bool     |` Emit previous asset inventory as new events (use in conjunction with -n <old_scan_name>)                                                                                                                                                                                                                                       `| False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.csv.output_file                             | str      | Output to CSV file                                                                                                                                                                                                                                                                                                             |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.discord.event_types                         | list     | Types of events to send                                                                                                                                                                                                                                                                                                        | ['VULNERABILITY', 'FINDING']                                                                                                                                                                                                                                                                                                                                                                                                                            |
| modules.discord.min_severity                        | str      | Only allow VULNERABILITY events of this severity or higher                                                                                                                                                                                                                                                                     | LOW                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.discord.retries                             | int      | Number of times to retry sending the message before skipping the event                                                                                                                                                                                                                                                         | 10                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.discord.webhook_url                         | str      | Discord webhook URL                                                                                                                                                                                                                                                                                                            |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.emails.output_file                          | str      | Output to file                                                                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.http.bearer                                 | str      | Authorization Bearer token                                                                                                                                                                                                                                                                                                     |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.http.method                                 | str      | HTTP method                                                                                                                                                                                                                                                                                                                    | POST                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.http.password                               | str      | Password (basic auth)                                                                                                                                                                                                                                                                                                          |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.http.siem_friendly                          | bool     | Format JSON in a SIEM-friendly way for ingestion into Elastic, Splunk, etc.                                                                                                                                                                                                                                                    | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.http.timeout                                | int      | HTTP timeout                                                                                                                                                                                                                                                                                                                   | 10                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.http.url                                    | str      | Web URL                                                                                                                                                                                                                                                                                                                        |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.http.username                               | str      | Username (basic auth)                                                                                                                                                                                                                                                                                                          |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.json.output_file                            | str      | Output to file                                                                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.json.siem_friendly                          | bool     | Output JSON in a SIEM-friendly format for ingestion into Elastic, Splunk, etc.                                                                                                                                                                                                                                                 | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.mysql.database                              | str      | The database name to connect to                                                                                                                                                                                                                                                                                                | bbot                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.mysql.host                                  | str      | The server running MySQL                                                                                                                                                                                                                                                                                                       | localhost                                                                                                                                                                                                                                                                                                                                                                                                                                               |
| modules.mysql.password                              | str      | The password to connect to MySQL                                                                                                                                                                                                                                                                                               | bbotislife                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| modules.mysql.port                                  | int      | The port to connect to MySQL                                                                                                                                                                                                                                                                                                   | 3306                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.mysql.username                              | str      | The username to connect to MySQL                                                                                                                                                                                                                                                                                               | root                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.neo4j.password                              | str      | Neo4j password                                                                                                                                                                                                                                                                                                                 | bbotislife                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| modules.neo4j.uri                                   | str      | Neo4j server + port                                                                                                                                                                                                                                                                                                            | bolt://localhost:7687                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.neo4j.username                              | str      | Neo4j username                                                                                                                                                                                                                                                                                                                 | neo4j                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.postgres.database                           | str      | The database name to connect to                                                                                                                                                                                                                                                                                                | bbot                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.postgres.host                               | str      | The server running Postgres                                                                                                                                                                                                                                                                                                    | localhost                                                                                                                                                                                                                                                                                                                                                                                                                                               |
| modules.postgres.password                           | str      | The password to connect to Postgres                                                                                                                                                                                                                                                                                            | bbotislife                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| modules.postgres.port                               | int      | The port to connect to Postgres                                                                                                                                                                                                                                                                                                | 5432                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.postgres.username                           | str      | The username to connect to Postgres                                                                                                                                                                                                                                                                                            | postgres                                                                                                                                                                                                                                                                                                                                                                                                                                                |
| modules.slack.event_types                           | list     | Types of events to send                                                                                                                                                                                                                                                                                                        | ['VULNERABILITY', 'FINDING']                                                                                                                                                                                                                                                                                                                                                                                                                            |
| modules.slack.min_severity                          | str      | Only allow VULNERABILITY events of this severity or higher                                                                                                                                                                                                                                                                     | LOW                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.slack.retries                               | int      | Number of times to retry sending the message before skipping the event                                                                                                                                                                                                                                                         | 10                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.slack.webhook_url                           | str      | Discord webhook URL                                                                                                                                                                                                                                                                                                            |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.splunk.hectoken                             | str      | HEC Token                                                                                                                                                                                                                                                                                                                      |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.splunk.index                                | str      | Index to send data to                                                                                                                                                                                                                                                                                                          |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.splunk.source                               | str      | Source path to be added to the metadata                                                                                                                                                                                                                                                                                        |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.splunk.timeout                              | int      | HTTP timeout                                                                                                                                                                                                                                                                                                                   | 10                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.splunk.url                                  | str      | Web URL                                                                                                                                                                                                                                                                                                                        |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.sqlite.database                             | str      | The path to the sqlite database file                                                                                                                                                                                                                                                                                           |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.stdout.accept_dupes                         | bool     | Whether to show duplicate events, default True                                                                                                                                                                                                                                                                                 | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.stdout.event_fields                         | list     | Which event fields to display                                                                                                                                                                                                                                                                                                  | []                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.stdout.event_types                          | list     | Which events to display, default all event types                                                                                                                                                                                                                                                                               | []                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.stdout.format                               | str      | Which text format to display, choices: text,json                                                                                                                                                                                                                                                                               | text                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.stdout.in_scope_only                        | bool     | Whether to only show in-scope events                                                                                                                                                                                                                                                                                           | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.subdomains.include_unresolved               | bool     | Include unresolved subdomains in output                                                                                                                                                                                                                                                                                        | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.subdomains.output_file                      | str      | Output to file                                                                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.teams.event_types                           | list     | Types of events to send                                                                                                                                                                                                                                                                                                        | ['VULNERABILITY', 'FINDING']                                                                                                                                                                                                                                                                                                                                                                                                                            |
| modules.teams.min_severity                          | str      | Only allow VULNERABILITY events of this severity or higher                                                                                                                                                                                                                                                                     | LOW                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| modules.teams.retries                               | int      | Number of times to retry sending the message before skipping the event                                                                                                                                                                                                                                                         | 10                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| modules.teams.webhook_url                           | str      | Teams webhook URL                                                                                                                                                                                                                                                                                                              |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.txt.output_file                             | str      | Output to file                                                                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.web_parameters.include_count                | bool     | Include the count of each parameter in the output                                                                                                                                                                                                                                                                              | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.web_parameters.output_file                  | str      | Output to file                                                                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.web_report.css_theme_file                   | str      | CSS theme URL for HTML output                                                                                                                                                                                                                                                                                                  | https://cdnjs.cloudflare.com/ajax/libs/github-markdown-css/5.1.0/github-markdown.min.css                                                                                                                                                                                                                                                                                                                                                                |
| modules.web_report.output_file                      | str      | Output to file                                                                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.websocket.ignore_ssl                        | bool     | Ignores all Websocket SSL related errors (like Self-Signed Certificates, etc.)                                                                                                                                                                                                                                                 | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.websocket.preserve_graph                    | bool     | Preserve full chains of events in the graph (prevents orphans)                                                                                                                                                                                                                                                                 | True                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.websocket.token                             | str      | Authorization Bearer token                                                                                                                                                                                                                                                                                                     |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.websocket.url                               | str      | Web URL                                                                                                                                                                                                                                                                                                                        |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.excavate.custom_yara_rules                  | str      | Include custom Yara rules                                                                                                                                                                                                                                                                                                      |                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| modules.excavate.speculate_params                   | bool     | Enable speculative parameter extraction from JSON and XML content                                                                                                                                                                                                                                                              | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.excavate.yara_max_match_data                | int      | Sets the maximum amount of text that can extracted from a YARA regex                                                                                                                                                                                                                                                           | 2000                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| modules.speculate.essential_only                    | bool     | Only enable essential speculate features (no extra discovery)                                                                                                                                                                                                                                                                  | False                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.speculate.max_hosts                         | int      | Max number of IP_RANGE hosts to convert into IP_ADDRESS events                                                                                                                                                                                                                                                                 | 65536                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| modules.speculate.ports                             | str      | The set of ports to speculate on                                                                                                                                                                                                                                                                                               | 80,443                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
<!-- END BBOT MODULE OPTIONS -->
