"use strict";
/**
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bookmark = void 0;
var util = __importStar(require("./util"));
var BOOKMARKS_KEY = 'bookmarks';
var Bookmark = /** @class */ (function () {
    /**
     * @constructor
     * @param {string|string[]} values single bookmark as string or multiple bookmarks as a string array.
     */
    function Bookmark(values) {
        this._values = asStringArray(values);
    }
    Bookmark.empty = function () {
        return EMPTY_BOOKMARK;
    };
    /**
     * Check if the given bookmark is meaningful and can be send to the database.
     * @return {boolean} returns `true` bookmark has a value, `false` otherwise.
     */
    Bookmark.prototype.isEmpty = function () {
        return this._values.length === 0;
    };
    /**
     * Get all bookmark values as an array.
     * @return {string[]} all values.
     */
    Bookmark.prototype.values = function () {
        return this._values;
    };
    /**
     * Get this bookmark as an object for begin transaction call.
     * @return {Object} the value of this bookmark as object.
     */
    Bookmark.prototype.asBeginTransactionParameters = function () {
        var _a;
        if (this.isEmpty()) {
            return {};
        }
        // Driver sends {bookmark: "max", bookmarks: ["one", "two", "max"]} instead of simple
        // {bookmarks: ["one", "two", "max"]} for backwards compatibility reasons. Old servers can only accept single
        // bookmark that is why driver has to parse and compare given list of bookmarks. This functionality will
        // eventually be removed.
        return _a = {},
            _a[BOOKMARKS_KEY] = this._values,
            _a;
    };
    return Bookmark;
}());
exports.Bookmark = Bookmark;
var EMPTY_BOOKMARK = new Bookmark(null);
/**
 * Converts given value to an array.
 * @param {string|string[]|Array} [value=undefined] argument to convert.
 * @return {string[]} value converted to an array.
 */
function asStringArray(value) {
    if (!value) {
        return [];
    }
    if (util.isString(value)) {
        return [value];
    }
    if (Array.isArray(value)) {
        var result = [];
        var flattenedValue = flattenArray(value);
        for (var i = 0; i < flattenedValue.length; i++) {
            var element = flattenedValue[i];
            // if it is undefined or null, ignore it
            if (element !== undefined && element !== null) {
                if (!util.isString(element)) {
                    throw new TypeError("Bookmark value should be a string, given: '" + element + "'");
                }
                result.push(element);
            }
        }
        return result;
    }
    throw new TypeError("Bookmark should either be a string or a string array, given: '" + value + "'");
}
/**
 * Recursively flattens an array so that the result becomes a single array
 * of values, which does not include any sub-arrays
 *
 * @param {Array} value
 */
function flattenArray(values) {
    return values.reduce(function (dest, value) {
        return Array.isArray(value)
            ? dest.concat(flattenArray(value))
            : dest.concat(value);
    }, []);
}
