'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = parseLinter;

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _jsYaml = require('js-yaml');

var _jsYaml2 = _interopRequireDefault(_jsYaml);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _requirePackageName = require('require-package-name');

var _requirePackageName2 = _interopRequireDefault(_requirePackageName);

var _utils = require('../utils');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function parse(content) {
  try {
    return JSON.parse(content);
  } catch (error) {
    // not JSON format
  }

  try {
    return _jsYaml2.default.safeLoad(content);
  } catch (error) {
    // not YAML format
  }

  try {
    return (0, _utils.evaluate)('module.exports = ' + content);
  } catch (error) {}
  // not valid JavaScript code


  // parse fail, return nothing
  return {};
}

function wrapToArray(obj) {
  if (!obj) {
    return [];
  } else if (_lodash2.default.isArray(obj)) {
    return obj;
  }

  return [obj];
}

function isLinterConfigAnAbsolutePath(specifier) {
  return _path2.default.isAbsolute(specifier);
}

function isLinterConfigARelativePath(specifier) {
  return _lodash2.default.startsWith(specifier, './') || _lodash2.default.startsWith(specifier, '../');
}

function isLinterConfigFromAPlugin(specifier) {
  return _lodash2.default.startsWith(specifier, 'plugin:');
}

function isLinterConfigFromAScopedModule(specifier) {
  return _lodash2.default.startsWith(specifier, '@');
}

function isLinterConfigFromAFullyQualifiedModuleName(specifier, prefix) {
  return _lodash2.default.startsWith(specifier, prefix);
}

function resolvePresetPackage(flavour, preset, rootDir) {
  // inspired from https://github.com/eslint/eslint/blob/5b4a94e26d0ef247fe222dacab5749805d9780dd/lib/config/config-file.js#L347
  if (isLinterConfigAnAbsolutePath(preset)) {
    return preset;
  }
  if (isLinterConfigARelativePath(preset)) {
    return _path2.default.resolve(rootDir, preset);
  }

  var _ref = isLinterConfigFromAPlugin(preset) ? { prefix: flavour + '-plugin-', specifier: preset.substring(preset.indexOf(':') + 1) } : { prefix: flavour + '-config-', specifier: preset },
      prefix = _ref.prefix,
      specifier = _ref.specifier;

  if (isLinterConfigFromAScopedModule(specifier)) {
    var scope = specifier.substring(0, specifier.indexOf('/'));
    var module = specifier.substring(specifier.indexOf('/') + 1);

    if (isLinterConfigFromAFullyQualifiedModuleName(module, prefix)) {
      return specifier;
    }
    return scope + '/' + prefix + module;
  }
  if (isLinterConfigFromAFullyQualifiedModuleName(specifier, prefix)) {
    return specifier;
  }
  return '' + prefix + specifier;
}

function loadConfig(preset, rootDir) {
  var presetPath = _path2.default.isAbsolute(preset) ? preset : _path2.default.resolve(rootDir, 'node_modules', preset);

  try {
    return require(presetPath); // eslint-disable-line global-require
  } catch (error) {
    return {}; // silently return nothing
  }
}

function checkConfig(flavour, config, rootDir) {
  var parser = wrapToArray(config.parser);
  var plugins = wrapToArray(config.plugins).map(function (plugin) {
    return flavour + '-plugin-' + plugin;
  });

  var presets = wrapToArray(config.extends).filter(function (preset) {
    return preset !== flavour + ':recommended';
  }).map(function (preset) {
    return resolvePresetPackage(flavour, preset, rootDir);
  });

  var presetPackages = presets.filter(function (preset) {
    return !_path2.default.isAbsolute(preset);
  }).map(_requirePackageName2.default);

  var presetDeps = (0, _lodash2.default)(presets).map(function (preset) {
    return loadConfig(preset, rootDir);
  }).map(function (presetConfig) {
    return checkConfig(flavour, presetConfig, rootDir);
  }).flatten().value();

  return _lodash2.default.union(parser, plugins, presetPackages, presetDeps);
}

function parseLinter(flavour, content, filename, deps, rootDir) {
  var basename = _path2.default.basename(filename);
  var filenameRegex = new RegExp('^\\.' + flavour + 'rc(\\.json|\\.js|\\.yml|\\.yaml)?$');
  if (filenameRegex.test(basename)) {
    var config = parse(content);
    return checkConfig(flavour, config, rootDir);
  }

  return [];
}
module.exports = exports['default'];