"use strict";
// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateAccountSasQueryParameters = generateAccountSasQueryParameters;
const accountSasPermissions_js_1 = require("./accountSasPermissions.js");
const sasIPRange_js_1 = require("./sasIPRange.js");
const sasQueryParameters_js_1 = require("./sasQueryParameters.js");
const accountSasResourceTypes_js_1 = require("./accountSasResourceTypes.js");
const accountSasServices_js_1 = require("./accountSasServices.js");
const constants_js_1 = require("../utils/constants.js");
const computeHMACSHA256_js_1 = require("../utils/computeHMACSHA256.js");
const truncateISO8061Date_js_1 = require("../utils/truncateISO8061Date.js");
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Generates a {@link SasQueryParameters} object which contains all SAS query parameters needed to make an actual
 * REST request.
 *
 * @see https://learn.microsoft.com/rest/api/storageservices/constructing-an-account-sas
 *
 * @param accountSasSignatureValues -
 * @param sharedKeyCredential -
 */
function generateAccountSasQueryParameters(accountSasSignatureValues, credential) {
    const version = accountSasSignatureValues.version
        ? accountSasSignatureValues.version
        : constants_js_1.SERVICE_VERSION;
    const parsedPermissions = (0, accountSasPermissions_js_1.accountSasPermissionsToString)(accountSasSignatureValues.permissions);
    const parsedServices = (0, accountSasServices_js_1.accountSasServicesToString)((0, accountSasServices_js_1.accountSasServicesFromString)(accountSasSignatureValues.services));
    // to and from string to guarantee the correct order of resoruce types is generated
    const parsedResourceTypes = (0, accountSasResourceTypes_js_1.accountSasResourceTypesToString)((0, accountSasResourceTypes_js_1.accountSasResourceTypesFromString)(accountSasSignatureValues.resourceTypes));
    const stringToSign = [
        credential.name,
        parsedPermissions,
        parsedServices,
        parsedResourceTypes,
        accountSasSignatureValues.startsOn
            ? (0, truncateISO8061Date_js_1.truncatedISO8061Date)(accountSasSignatureValues.startsOn, false)
            : "",
        (0, truncateISO8061Date_js_1.truncatedISO8061Date)(accountSasSignatureValues.expiresOn, false),
        accountSasSignatureValues.ipRange ? (0, sasIPRange_js_1.ipRangeToString)(accountSasSignatureValues.ipRange) : "",
        accountSasSignatureValues.protocol ? accountSasSignatureValues.protocol : "",
        version,
        "", // Account SAS requires an additional newline character
    ].join("\n");
    const signature = (0, computeHMACSHA256_js_1.computeHMACSHA256)(stringToSign, credential.key);
    return new sasQueryParameters_js_1.SasQueryParameters(version, signature, {
        permissions: parsedPermissions.toString(),
        services: parsedServices,
        resourceTypes: parsedResourceTypes,
        protocol: accountSasSignatureValues.protocol,
        startsOn: accountSasSignatureValues.startsOn,
        expiresOn: accountSasSignatureValues.expiresOn,
        ipRange: accountSasSignatureValues.ipRange,
    });
}
//# sourceMappingURL=accountSasSignatureValues.js.map