"use strict";
// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.getChangeSetBoundary = getChangeSetBoundary;
exports.getTransactionBoundary = getTransactionBoundary;
exports.getInitialTransactionBody = getInitialTransactionBody;
exports.getTransactionHttpRequestBody = getTransactionHttpRequestBody;
const constants_js_1 = require("./constants.js");
/**
 * Builds a transaction change set boundary to be added to the transaction request body
 * @param changesetId - Id of the transaction changeset
 */
function getChangeSetBoundary(changesetId) {
    return `changeset_${changesetId}`;
}
/**
 * Builds a transaction boundary to be added to the transaction request body
 * @param transactionId - Id of the transaction
 */
function getTransactionBoundary(transactionId) {
    return `batch_${transactionId}`;
}
/**
 * Returns an initial representation of the Transaction body.
 * @param transactionId - Id of the transaction
 * @param changesetId - Id of the transaction changeset
 */
function getInitialTransactionBody(transactionId, changesetId) {
    const transactionBoundary = `batch_${transactionId}`;
    return [
        `--${transactionBoundary}${constants_js_1.TRANSACTION_HTTP_LINE_ENDING}${constants_js_1.HeaderConstants.CONTENT_TYPE}: multipart/mixed; boundary=changeset_${changesetId}${constants_js_1.TRANSACTION_HTTP_LINE_ENDING}${constants_js_1.TRANSACTION_HTTP_LINE_ENDING}`,
    ];
}
/**
 * Build the Transaction http request body to send to the service.
 * @param bodyParts - Parts of the transaction body, containing information about the actions to be included in the transaction request
 * @param transactionId - Id of the transaction
 * @param changesetId - Id of the transaction changeset
 */
function getTransactionHttpRequestBody(bodyParts, transactionId, changesetId) {
    const transactionBoundary = getTransactionBoundary(transactionId);
    const changesetBoundary = getChangeSetBoundary(changesetId);
    const changesetEnding = `--${changesetBoundary}--`;
    const transactionEnding = `--${transactionBoundary}--`;
    const bodyContent = bodyParts.join(constants_js_1.TRANSACTION_HTTP_LINE_ENDING);
    return `${bodyContent}${constants_js_1.TRANSACTION_HTTP_LINE_ENDING}${changesetEnding}${constants_js_1.TRANSACTION_HTTP_LINE_ENDING}${transactionEnding}${constants_js_1.TRANSACTION_HTTP_LINE_ENDING}`;
}
//# sourceMappingURL=transactionHelpers.js.map