// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
import { accountSasPermissionsToString } from "./accountSasPermissions.js";
import { ipRangeToString } from "./sasIPRange.js";
import { SasQueryParameters } from "./sasQueryParameters.js";
import { accountSasResourceTypesFromString, accountSasResourceTypesToString, } from "./accountSasResourceTypes.js";
import { accountSasServicesFromString, accountSasServicesToString } from "./accountSasServices.js";
import { SERVICE_VERSION } from "../utils/constants.js";
import { computeHMACSHA256 } from "../utils/computeHMACSHA256.js";
import { truncatedISO8061Date } from "../utils/truncateISO8061Date.js";
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Generates a {@link SasQueryParameters} object which contains all SAS query parameters needed to make an actual
 * REST request.
 *
 * @see https://learn.microsoft.com/rest/api/storageservices/constructing-an-account-sas
 *
 * @param accountSasSignatureValues -
 * @param sharedKeyCredential -
 */
export function generateAccountSasQueryParameters(accountSasSignatureValues, credential) {
    const version = accountSasSignatureValues.version
        ? accountSasSignatureValues.version
        : SERVICE_VERSION;
    const parsedPermissions = accountSasPermissionsToString(accountSasSignatureValues.permissions);
    const parsedServices = accountSasServicesToString(accountSasServicesFromString(accountSasSignatureValues.services));
    // to and from string to guarantee the correct order of resoruce types is generated
    const parsedResourceTypes = accountSasResourceTypesToString(accountSasResourceTypesFromString(accountSasSignatureValues.resourceTypes));
    const stringToSign = [
        credential.name,
        parsedPermissions,
        parsedServices,
        parsedResourceTypes,
        accountSasSignatureValues.startsOn
            ? truncatedISO8061Date(accountSasSignatureValues.startsOn, false)
            : "",
        truncatedISO8061Date(accountSasSignatureValues.expiresOn, false),
        accountSasSignatureValues.ipRange ? ipRangeToString(accountSasSignatureValues.ipRange) : "",
        accountSasSignatureValues.protocol ? accountSasSignatureValues.protocol : "",
        version,
        "", // Account SAS requires an additional newline character
    ].join("\n");
    const signature = computeHMACSHA256(stringToSign, credential.key);
    return new SasQueryParameters(version, signature, {
        permissions: parsedPermissions.toString(),
        services: parsedServices,
        resourceTypes: parsedResourceTypes,
        protocol: accountSasSignatureValues.protocol,
        startsOn: accountSasSignatureValues.startsOn,
        expiresOn: accountSasSignatureValues.expiresOn,
        ipRange: accountSasSignatureValues.ipRange,
    });
}
//# sourceMappingURL=accountSasSignatureValues.js.map