import type { SasIPRange } from "./SasIPRange.js";
import { UserDelegationKey } from "@azure/storage-common";
/**
 * Protocols for generated SAS.
 */
export declare enum SASProtocol {
    /**
     * Protocol that allows HTTPS only
     */
    Https = "https",
    /**
     * Protocol that allows both HTTPS and HTTP
     */
    HttpsAndHttp = "https,http"
}
/**
 * Options to construct {@link SASQueryParameters}.
 */
export interface SASQueryParametersOptions {
    /**
     * Optional only when identifier is provided.
     * Please refer to {@link AccountSASPermissions}, {@link BlobSASPermissions}, or {@link ContainerSASPermissions} for
     * more details.
     */
    permissions?: string;
    /**
     * Optional. The storage services being accessed (only for Account SAS). Please refer to {@link AccountSASServices}
     * for more details.
     */
    services?: string;
    /**
     * Optional. The storage resource types being accessed (only for Account SAS). Please refer to
     * {@link AccountSASResourceTypes} for more details.
     */
    resourceTypes?: string;
    /**
     * Optional. The allowed HTTP protocol(s).
     */
    protocol?: SASProtocol;
    /**
     * Optional. The start time for this SAS token.
     */
    startsOn?: Date;
    /**
     * Optional only when identifier is provided. The expiry time for this SAS token.
     */
    expiresOn?: Date;
    /**
     * Optional. IP ranges allowed in this SAS.
     */
    ipRange?: SasIPRange;
    /**
     * Optional. The signed identifier (only for {@link BlobSASSignatureValues}).
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/establishing-a-stored-access-policy
     */
    identifier?: string;
    /**
     * Optional. Beginning in version 2025-07-05, this value specifies the Entra ID of the user would is authorized to
     * use the resulting SAS URL.  The resulting SAS URL must be used in conjunction with an Entra ID token that has been
     * issued to the user specified in this value.
     */
    delegatedUserObjectId?: string;
    /**
     * Optional. Encryption scope to use when sending requests authorized with this SAS URI.
     */
    encryptionScope?: string;
    /**
     * Optional. Specifies which resources are accessible via the SAS (only for {@link BlobSASSignatureValues}).
     * @see https://learn.microsoft.com/rest/api/storageservices/create-service-sas#specifying-the-signed-resource-blob-service-only
     */
    resource?: string;
    /**
     * Value for cache-control header in Blob/File Service SAS.
     */
    cacheControl?: string;
    /**
     * Value for content-disposition header in Blob/File Service SAS.
     */
    contentDisposition?: string;
    /**
     * Value for content-encoding header in Blob/File Service SAS.
     */
    contentEncoding?: string;
    /**
     * Value for content-length header in Blob/File Service SAS.
     */
    contentLanguage?: string;
    /**
     * Value for content-type header in Blob/File Service SAS.
     */
    contentType?: string;
    /**
     * User delegation key properties.
     */
    userDelegationKey?: UserDelegationKey;
    /**
     * Authorized AAD Object ID in GUID format. The AAD Object ID of a user authorized by the owner of the User Delegation Key
     * to perform the action granted by the SAS. The Azure Storage service will ensure that the owner of the user delegation key
     * has the required permissions before granting access but no additional permission check for the user specified in
     * this value will be performed. This cannot be used in conjuction with {@link signedUnauthorizedUserObjectId}.
     * This is only used for User Delegation SAS.
     */
    preauthorizedAgentObjectId?: string;
    /**
     * A GUID value that will be logged in the storage diagnostic logs and can be used to correlate SAS generation with storage resource access.
     * This is only used for User Delegation SAS.
     */
    correlationId?: string;
}
/**
 * Represents the components that make up an Azure Storage SAS' query parameters. This type is not constructed directly
 * by the user; it is only generated by the {@link AccountSASSignatureValues} and {@link BlobSASSignatureValues}
 * types. Once generated, it can be encoded into a {@link String} and appended to a URL directly (though caution should
 * be taken here in case there are existing query parameters, which might affect the appropriate means of appending
 * these query parameters).
 *
 * NOTE: Instances of this class are immutable.
 */
export declare class SASQueryParameters {
    /**
     * The storage API version.
     */
    readonly version: string;
    /**
     * Optional. The allowed HTTP protocol(s).
     */
    readonly protocol?: SASProtocol;
    /**
     * Optional. The start time for this SAS token.
     */
    readonly startsOn?: Date;
    /**
     * Optional only when identifier is provided. The expiry time for this SAS token.
     */
    readonly expiresOn?: Date;
    /**
     * Optional only when identifier is provided.
     * Please refer to {@link AccountSASPermissions}, {@link BlobSASPermissions}, or {@link ContainerSASPermissions} for
     * more details.
     */
    readonly permissions?: string;
    /**
     * Optional. The storage services being accessed (only for Account SAS). Please refer to {@link AccountSASServices}
     * for more details.
     */
    readonly services?: string;
    /**
     * Optional. The storage resource types being accessed (only for Account SAS). Please refer to
     * {@link AccountSASResourceTypes} for more details.
     */
    readonly resourceTypes?: string;
    /**
     * Optional. The signed identifier (only for {@link BlobSASSignatureValues}).
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/establishing-a-stored-access-policy
     */
    readonly identifier?: string;
    /**
     * Optional. Beginning in version 2025-07-05, this value specifies the Entra ID of the user would is authorized to
     * use the resulting SAS URL.  The resulting SAS URL must be used in conjunction with an Entra ID token that has been
     * issued to the user specified in this value.
     */
    readonly delegatedUserObjectId?: string;
    /**
     * Optional. Encryption scope to use when sending requests authorized with this SAS URI.
     */
    readonly encryptionScope?: string;
    /**
     * Optional. Specifies which resources are accessible via the SAS (only for {@link BlobSASSignatureValues}).
     * @see https://learn.microsoft.com/rest/api/storageservices/create-service-sas#specifying-the-signed-resource-blob-service-only
     */
    readonly resource?: string;
    /**
     * The signature for the SAS token.
     */
    readonly signature: string;
    /**
     * Value for cache-control header in Blob/File Service SAS.
     */
    readonly cacheControl?: string;
    /**
     * Value for content-disposition header in Blob/File Service SAS.
     */
    readonly contentDisposition?: string;
    /**
     * Value for content-encoding header in Blob/File Service SAS.
     */
    readonly contentEncoding?: string;
    /**
     * Value for content-length header in Blob/File Service SAS.
     */
    readonly contentLanguage?: string;
    /**
     * Value for content-type header in Blob/File Service SAS.
     */
    readonly contentType?: string;
    /**
     * Inner value of getter ipRange.
     */
    private readonly ipRangeInner?;
    /**
     * The Azure Active Directory object ID in GUID format.
     * Property of user delegation key.
     */
    private readonly signedOid?;
    /**
     * The Azure Active Directory tenant ID in GUID format.
     * Property of user delegation key.
     */
    private readonly signedTenantId?;
    /**
     * The date-time the key is active.
     * Property of user delegation key.
     */
    private readonly signedStartsOn?;
    /**
     * The date-time the key expires.
     * Property of user delegation key.
     */
    private readonly signedExpiresOn?;
    /**
     * Abbreviation of the Azure Storage service that accepts the user delegation key.
     * Property of user delegation key.
     */
    private readonly signedService?;
    /**
     * The service version that created the user delegation key.
     * Property of user delegation key.
     */
    private readonly signedVersion?;
    /**
     * Authorized AAD Object ID in GUID format. The AAD Object ID of a user authorized by the owner of the User Delegation Key
     * to perform the action granted by the SAS. The Azure Storage service will ensure that the owner of the user delegation key
     * has the required permissions before granting access but no additional permission check for the user specified in
     * this value will be performed. This is only used for User Delegation SAS.
     */
    readonly preauthorizedAgentObjectId?: string;
    /**
     * A GUID value that will be logged in the storage diagnostic logs and can be used to correlate SAS generation with storage resource access.
     * This is only used for User Delegation SAS.
     */
    readonly correlationId?: string;
    /**
     * Optional. IP range allowed for this SAS.
     *
     * @readonly
     */
    get ipRange(): SasIPRange | undefined;
    /**
     * Creates an instance of SASQueryParameters.
     *
     * @param version - Representing the storage version
     * @param signature - Representing the signature for the SAS token
     * @param permissions - Representing the storage permissions
     * @param services - Representing the storage services being accessed (only for Account SAS)
     * @param resourceTypes - Representing the storage resource types being accessed (only for Account SAS)
     * @param protocol - Representing the allowed HTTP protocol(s)
     * @param startsOn - Representing the start time for this SAS token
     * @param expiresOn - Representing the expiry time for this SAS token
     * @param ipRange - Representing the range of valid IP addresses for this SAS token
     * @param identifier - Representing the signed identifier (only for Service SAS)
     * @param resource - Representing the storage container or blob (only for Service SAS)
     * @param cacheControl - Representing the cache-control header (only for Blob/File Service SAS)
     * @param contentDisposition - Representing the content-disposition header (only for Blob/File Service SAS)
     * @param contentEncoding - Representing the content-encoding header (only for Blob/File Service SAS)
     * @param contentLanguage - Representing the content-language header (only for Blob/File Service SAS)
     * @param contentType - Representing the content-type header (only for Blob/File Service SAS)
     * @param userDelegationKey - Representing the user delegation key properties
     * @param preauthorizedAgentObjectId - Representing the authorized AAD Object ID (only for User Delegation SAS)
     * @param correlationId - Representing the correlation ID (only for User Delegation SAS)
     * @param encryptionScope -
     * @param delegatedUserObjectId -
     */
    constructor(version: string, signature: string, permissions?: string, services?: string, resourceTypes?: string, protocol?: SASProtocol, startsOn?: Date, expiresOn?: Date, ipRange?: SasIPRange, identifier?: string, resource?: string, cacheControl?: string, contentDisposition?: string, contentEncoding?: string, contentLanguage?: string, contentType?: string, userDelegationKey?: UserDelegationKey, preauthorizedAgentObjectId?: string, correlationId?: string, encryptionScope?: string, delegatedUserObjectId?: string);
    /**
     * Creates an instance of SASQueryParameters.
     *
     * @param version - Representing the storage version
     * @param signature - Representing the signature for the SAS token
     * @param options - Optional. Options to construct the SASQueryParameters.
     */
    constructor(version: string, signature: string, options?: SASQueryParametersOptions);
    /**
     * Encodes all SAS query parameters into a string that can be appended to a URL.
     *
     */
    toString(): string;
    /**
     * A private helper method used to filter and append query key/value pairs into an array.
     *
     * @param queries -
     * @param key -
     * @param value -
     */
    private tryAppendQueryParameter;
}
//# sourceMappingURL=SASQueryParameters.d.ts.map