// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
import { StorageContextClient } from "./StorageContextClient.js";
import { getCoreClientOptions, getCredentialFromPipeline } from "./Pipeline.js";
import { escapeURLPath, getURLScheme, iEqual, getAccountNameFromUrl, } from "./utils/utils.common.js";
/**
 * A StorageClient represents a based URL class for {@link BlobServiceClient}, {@link ContainerClient}
 * and etc.
 */
export class StorageClient {
    /**
     * Encoded URL string value.
     */
    url;
    accountName;
    /**
     * Request policy pipeline.
     *
     * @internal
     */
    pipeline;
    /**
     * Such as AnonymousCredential, StorageSharedKeyCredential or any credential from the `@azure/identity` package to authenticate requests to the service. You can also provide an object that implements the TokenCredential interface. If not specified, AnonymousCredential is used.
     */
    credential;
    /**
     * StorageClient is a reference to protocol layer operations entry, which is
     * generated by AutoRest generator.
     */
    storageClientContext;
    /**
     */
    isHttps;
    /**
     * Creates an instance of StorageClient.
     * @param url - url to resource
     * @param pipeline - request policy pipeline.
     */
    constructor(url, pipeline) {
        // URL should be encoded and only once, protocol layer shouldn't encode URL again
        this.url = escapeURLPath(url);
        this.accountName = getAccountNameFromUrl(url);
        this.pipeline = pipeline;
        this.storageClientContext = new StorageContextClient(this.url, getCoreClientOptions(pipeline));
        this.isHttps = iEqual(getURLScheme(this.url) || "", "https");
        this.credential = getCredentialFromPipeline(pipeline);
        // Override protocol layer's default content-type
        const storageClientContext = this.storageClientContext;
        storageClientContext.requestContentType = undefined;
    }
}
//# sourceMappingURL=StorageClient.js.map