import type { TokenCredential } from "@azure/core-auth";
import type { RequestBodyType as HttpRequestBody, TransferProgressEvent } from "@azure/core-rest-pipeline";
import type { AbortSignalLike } from "@azure/abort-controller";
import type { CopyFileSmbInfo, DeleteSnapshotsOptionType, DirectoryCreateResponse, DirectoryDeleteResponse, DirectoryGetPropertiesResponse, DirectoryItem, DirectoryListFilesAndDirectoriesSegmentResponse, DirectoryListHandlesResponse, DirectorySetMetadataResponse, DirectorySetPropertiesResponse, DirectoryForceCloseHandlesHeaders, FileAbortCopyResponse, FileCreateResponse, FileDeleteResponse, FileDownloadResponseModel, FileForceCloseHandlesHeaders, FileGetPropertiesResponse, FileGetRangeListHeaders, FileGetRangeListDiffResponse, FileItem, FileListHandlesResponse, FilePermissionFormat, FileSetHTTPHeadersResponse, FileSetMetadataResponse, FileStartCopyResponse, FileUploadRangeFromURLResponse, FileUploadRangeResponse, HandleItem, LeaseAccessConditions, ModeCopyMode, OwnerCopyMode, RangeModel, ShareCreatePermissionResponse, ShareCreateResponse, ShareCreateSnapshotResponse, ShareDeleteResponse, ShareGetAccessPolicyHeaders, ShareGetPermissionResponse, ShareGetPropertiesResponseModel, ShareGetStatisticsResponseModel, ShareSetAccessPolicyResponse, ShareSetMetadataResponse, ShareSetQuotaResponse, SignedIdentifierModel, SourceModifiedAccessConditions, ShareAccessTier, ShareSetPropertiesResponse, ShareRootSquash, FileRenameResponse, DirectoryRenameResponse, FileLastWrittenMode, SharePermission, FileCreateHardLinkResponse, FileCreateSymbolicLinkResponse, FileGetSymbolicLinkResponse } from "./generatedModels.js";
import type { Pipeline } from "./Pipeline.js";
import type { WithResponse } from "./utils/utils.common.js";
import { Credential, UserDelegationKey } from "@azure/storage-common";
import type { CommonOptions } from "./StorageClient.js";
import { StorageClient } from "./StorageClient.js";
import type { PagedAsyncIterableIterator } from "@azure/core-paging";
import type { Range } from "./Range.js";
import type { CloseHandlesInfo, FileAndDirectoryCreateCommonOptions, FileAndDirectorySetPropertiesCommonOptions, FileHttpHeaders, Metadata, ShareProtocols, HttpAuthorization, ShareClientOptions, ShareClientConfig, FilePosixProperties, TimeNowType } from "./models.js";
import type { ShareSASPermissions } from "./ShareSASPermissions.js";
import type { SASProtocol } from "./SASQueryParameters.js";
import type { SasIPRange } from "./SasIPRange.js";
import type { FileSASPermissions } from "./FileSASPermissions.js";
import { Readable } from "node:stream";
export { ShareClientOptions, ShareClientConfig } from "./models.js";
/**
 * Options to configure the {@link ShareClient.create} operation.
 */
export interface ShareCreateOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * A name-value pair to associate with a file storage object.
     */
    metadata?: {
        [propertyName: string]: string;
    };
    /**
     * Specifies the maximum size of the share, in
     * gigabytes.
     */
    quota?: number;
    /**
     * Specifies the access tier of the share. Possible values include: 'TransactionOptimized',
     * 'Hot', 'Cool'
     */
    accessTier?: ShareAccessTier;
    /**
     * Supported in version 2020-02-10 and above. Specifies the enabled protocols on the share. If not specified, the default is SMB.
     */
    protocols?: ShareProtocols;
    /**
     * Root squash to set on the share.  Only valid for NFS shares. Possible values include:
     * 'NoRootSquash', 'RootSquash', 'AllSquash'.
     */
    rootSquash?: ShareRootSquash;
    /**
     * Specifies whether the snapshot virtual directory should be accessible at the root of share mount point when NFS is enabled.
     * If not specified, the default is true.
     */
    enableSnapshotVirtualDirectoryAccess?: boolean;
    /**
     * Optional. Boolean. Default if not specified is false. This property enables paid bursting.
     */
    paidBurstingEnabled?: boolean;
    /**
     * Optional. Integer. Default if not specified is the maximum throughput the file share can support. Current maximum for a file share is 10,340  MiB/sec.
     */
    paidBurstingMaxBandwidthMibps?: number;
    /**
     * Optional. Integer. Default if not specified is the maximum IOPS the file share can support. Current maximum for a file share is 102,400 IOPS.
     */
    paidBurstingMaxIops?: number;
    /**
     * Optional. Supported in version 2025-01-05 and later. Only allowed for provisioned v2 file shares.
     * Specifies the provisioned number of input/output operations per second (IOPS) of the share. If this is not specified, the provisioned IOPS is set to value calculated based on recommendation formula.
     */
    shareProvisionedIops?: number;
    /** Optional. Supported in version 2025-01-05 and later. Only allowed for provisioned v2 file shares. Specifies the provisioned bandwidth of the share, in mebibytes per second (MiBps). If this is not specified, the provisioned bandwidth is set to value calculated based on recommendation formula. */
    shareProvisionedBandwidthMibps?: number;
}
/**
 * Options to configure the {@link ShareClient.delete} operation.
 */
export interface ShareDeleteMethodOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Specifies the option
     * include to delete the base share and all of its snapshots. Possible values
     * include: 'include'
     */
    deleteSnapshots?: DeleteSnapshotsOptionType;
    /**
     * If specified, the operation only succeeds if the resource's lease is active and matches this ID.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure the {@link ShareClient.setMetadata} operation.
 */
export interface ShareSetMetadataOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * If specified, the operation only succeeds if the resource's lease is active and matches this ID.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure the {@link ShareClient.setAccessPolicy} operation.
 */
export interface ShareSetAccessPolicyOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * If specified, the operation only succeeds if the resource's lease is active and matches this ID.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure the {@link ShareClient.getAccessPolicy} operation.
 */
export interface ShareGetAccessPolicyOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * If specified, the operation only succeeds if the resource's lease is active and matches this ID.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure the {@link ShareClient.exists} operation.
 */
export interface ShareExistsOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * If specified, the operation only succeeds if the resource's lease is active and matches this ID.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure the {@link ShareClient.getProperties} operation.
 */
export interface ShareGetPropertiesOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * If specified, the operation only succeeds if the resource's lease is active and matches this ID.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure the {@link ShareClient.setQuota} operation.
 */
export interface ShareSetQuotaOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * If specified, the operation only succeeds if the resource's lease is active and matches this ID.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure the {@link ShareClient.setProperties} operation.
 */
export interface ShareSetPropertiesOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Specifies the access tier of the share. Possible values include: 'TransactionOptimized',
     * 'Hot', 'Cool'.
     */
    accessTier?: ShareAccessTier;
    /**
     * Specifies the maximum size of the share, in gigabytes.
     */
    quotaInGB?: number;
    /**
     * Root squash to set on the share.  Only valid for NFS shares. Possible values include:
     * 'NoRootSquash', 'RootSquash', 'AllSquash'.
     */
    rootSquash?: ShareRootSquash;
    /**
     * If specified, the operation only succeeds if the resource's lease is active and matches this ID.
     */
    leaseAccessConditions?: LeaseAccessConditions;
    /**
     * Specifies whether the snapshot virtual directory should be accessible at the root of share mount point when NFS is enabled.
     * If not specified, the default is true.
     */
    enableSnapshotVirtualDirectoryAccess?: boolean;
    /**
     * Optional. Boolean. Default if not specified is false. This property enables paid bursting.
     */
    paidBurstingEnabled?: boolean;
    /**
     * Optional. Integer. Default if not specified is the maximum throughput the file share can support. Current maximum for a file share is 10,340  MiB/sec.
     */
    paidBurstingMaxBandwidthMibps?: number;
    /**
     * Optional. Integer. Default if not specified is the maximum IOPS the file share can support. Current maximum for a file share is 102,400 IOPS.
     */
    paidBurstingMaxIops?: number;
    /**
     * Optional. Supported in version 2025-01-05 and later. Only allowed for provisioned v2 file shares.
     * Specifies the provisioned number of input/output operations per second (IOPS) of the share. If this is not specified, the provisioned IOPS is set to value calculated based on recommendation formula.
     */
    shareProvisionedIops?: number;
    /** Optional. Supported in version 2025-01-05 and later. Only allowed for provisioned v2 file shares. Specifies the provisioned bandwidth of the share, in mebibytes per second (MiBps).
     * If this is not specified, the provisioned bandwidth is set to value calculated based on recommendation formula.
     */
    shareProvisionedBandwidthMibps?: number;
}
/**
 * Options to configure the {@link ShareClient.getStatistics} operation.
 */
export interface ShareGetStatisticsOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * If specified, the operation only succeeds if the resource's lease is active and matches this ID.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Signed Identifier
 */
export interface SignedIdentifier {
    /**
     * a unique id
     */
    id: string;
    /**
     * Access Policy
     */
    accessPolicy: {
        /**
         * the date-time the policy is active.
         */
        startsOn: Date;
        /**
         * the date-time the policy expires.
         */
        expiresOn: Date;
        /**
         * the permissions for the acl policy
         * @see https://learn.microsoft.com/rest/api/storageservices/set-share-acl
         */
        permissions: string;
    };
}
export declare type ShareGetAccessPolicyResponse = WithResponse<{
    signedIdentifiers: SignedIdentifier[];
} & ShareGetAccessPolicyHeaders, ShareGetAccessPolicyHeaders, SignedIdentifierModel[]>;
/**
 * Options to configure the {@link ShareClient.createSnapshot} operation.
 */
export interface ShareCreateSnapshotOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * A name-value pair to associate with a file storage object.
     */
    metadata?: {
        [propertyName: string]: string;
    };
}
/**
 * Options to configure the {@link ShareClient.createPermission} operation.
 */
export interface ShareCreatePermissionOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure the {@link ShareClient.getPermission} operation.
 */
export interface ShareGetPermissionOptions extends CommonOptions {
    /**
     * Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned.
     * Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format.
     * If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission
     */
    filePermissionFormat?: FilePermissionFormat;
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Response data for the {@link ShareClient.getStatistics} Operation.
 */
export type ShareGetStatisticsResponse = ShareGetStatisticsResponseModel & {
    /**
     * @deprecated shareUsage is going to be deprecated. Please use ShareUsageBytes instead.
     *
     * The approximate size of the data stored on the share, rounded up to the nearest gigabyte. Note
     * that this value may not include all recently created or recently resized files.
     */
    shareUsage: number;
};
/**
 * Contains response data for the {@link ShareClient.createIfNotExists} operation.
 */
export interface ShareCreateIfNotExistsResponse extends ShareCreateResponse {
    /**
     * Indicate whether the share is successfully created. Is false when the share is not changed as it already exists.
     */
    succeeded: boolean;
}
/**
 * Contains response data for the {@link ShareClient.deleteIfExists} operation.
 */
export interface ShareDeleteIfExistsResponse extends ShareDeleteResponse {
    /**
     * Indicate whether the share is successfully deleted. Is false if the share does not exist in the first place.
     */
    succeeded: boolean;
}
/**
 * Contains response data for the {@link ShareClient.getProperties} operation.
 */
export type ShareGetPropertiesResponse = ShareGetPropertiesResponseModel & {
    /**
     * The protocols that have been enabled on the share.
     */
    protocols?: ShareProtocols;
};
/**
 * Common options of the {@link ShareGenerateSasUrlOptions} and {@link FileGenerateSasUrlOptions}.
 */
export interface CommonGenerateSasUrlOptions {
    /**
     * The version of the service this SAS will target. If not specified, it will default to the version targeted by the
     * library.
     */
    version?: string;
    /**
     * Optional. SAS protocols, HTTPS only or HTTPSandHTTP
     */
    protocol?: SASProtocol;
    /**
     * Optional. When the SAS will take effect.
     */
    startsOn?: Date;
    /**
     * Optional only when identifier is provided. The time after which the SAS will no longer work.
     */
    expiresOn?: Date;
    /**
     * Optional. IP ranges allowed in this SAS.
     */
    ipRange?: SasIPRange;
    /**
     * Optional. The name of the access policy on the share this SAS references if any.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/establishing-a-stored-access-policy
     */
    identifier?: string;
    /**
     * Optional. The cache-control header for the SAS.
     */
    cacheControl?: string;
    /**
     * Optional. The content-disposition header for the SAS.
     */
    contentDisposition?: string;
    /**
     * Optional. The content-encoding header for the SAS.
     */
    contentEncoding?: string;
    /**
     * Optional. The content-language header for the SAS.
     */
    contentLanguage?: string;
    /**
     * Optional. The content-type header for the SAS.
     */
    contentType?: string;
}
/**
 * Options to configure {@link ShareClient.generateSasUrl} operation.
 */
export interface ShareGenerateSasUrlOptions extends CommonGenerateSasUrlOptions {
    /**
     * Optional only when identifier is provided. Specifies the list of permissions to be associated with the SAS.
     */
    permissions?: ShareSASPermissions;
}
/**
 * A ShareClient represents a URL to the Azure Storage share allowing you to manipulate its directories and files.
 */
export declare class ShareClient extends StorageClient {
    /**
     * Share operation context provided by protocol layer.
     */
    private context;
    private _name;
    private shareClientConfig?;
    /**
     * The name of the share
     */
    get name(): string;
    /**
     * @param connectionString - Account connection string or a SAS connection string of an Azure storage account.
     *                                  [ Note - Account connection string can only be used in NODE.JS runtime. ]
     *                                  Account connection string example -
     *                                  `DefaultEndpointsProtocol=https;AccountName=myaccount;AccountKey=accountKey;EndpointSuffix=core.windows.net`
     *                                  SAS connection string example -
     *                                  `BlobEndpoint=https://myaccount.blob.core.windows.net/;QueueEndpoint=https://myaccount.queue.core.windows.net/;FileEndpoint=https://myaccount.file.core.windows.net/;TableEndpoint=https://myaccount.table.core.windows.net/;SharedAccessSignature=sasString`
     * @param name - Share name.
     * @param options - Optional. Options to configure the HTTP pipeline.
     */
    constructor(connectionString: string, name: string, options?: ShareClientOptions);
    /**
     * Creates an instance of ShareClient.
     *
     * @param url - A URL string pointing to Azure Storage file share, such as
     *                     "https://myaccount.file.core.windows.net/share". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.file.core.windows.net/share?sasString".
     * @param credential - Such as AnonymousCredential or StorageSharedKeyCredential.
     *                                  If not specified, AnonymousCredential is used.
     * @param options - Optional. Options to configure the HTTP pipeline.
     */
    constructor(url: string, credential?: Credential | TokenCredential, options?: ShareClientOptions);
    /**
     * Creates an instance of ShareClient.
     *
     * @param url - A URL string pointing to Azure Storage file share, such as
     *                     "https://myaccount.file.core.windows.net/share". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.file.core.windows.net/share?sasString".
     * @param pipeline - Call newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     */
    constructor(url: string, pipeline: Pipeline, options?: ShareClientConfig);
    /**
     * Creates a new ShareClient object identical to the source but with the specified snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base share.
     *
     * @param snapshot - The snapshot timestamp.
     * @returns A new ShareClient object identical to the source but with the specified snapshot timestamp
     */
    withSnapshot(snapshot: string): ShareClient;
    /**
     * Creates a new share under the specified account. If the share with
     * the same name already exists, the operation fails.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-share
     *
     * @param options - Options to Share Create operation.
     * @returns Response data for the Share Create operation.
     */
    create(options?: ShareCreateOptions): Promise<ShareCreateResponse>;
    /**
     * Creates a new share under the specified account. If the share with
     * the same name already exists, it is not changed.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-share
     *
     * @param options -
     */
    createIfNotExists(options?: ShareCreateOptions): Promise<ShareCreateIfNotExistsResponse>;
    /**
     * Creates a {@link ShareDirectoryClient} object.
     *
     * @param directoryName - A directory name
     * @returns The ShareDirectoryClient object for the given directory name.
     */
    getDirectoryClient(directoryName: string): ShareDirectoryClient;
    /**
     * Gets the directory client for the root directory of this share.
     * Note that the root directory always exists and cannot be deleted.
     *
     * @readonly A new ShareDirectoryClient object for the root directory.
     */
    get rootDirectoryClient(): ShareDirectoryClient;
    /**
     * Creates a new subdirectory under this share.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-directory
     *
     * @param directoryName -
     * @param options - Options to Directory Create operation.
     * @returns Directory creation response data and the corresponding directory client.
     */
    createDirectory(directoryName: string, options?: DirectoryCreateOptions): Promise<{
        directoryClient: ShareDirectoryClient;
        directoryCreateResponse: DirectoryCreateResponse;
    }>;
    /**
     * Removes the specified empty sub directory under this share.
     * Note that the directory must be empty before it can be deleted.
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-directory
     *
     * @param directoryName -
     * @param options - Options to Directory Delete operation.
     * @returns Directory deletion response data.
     */
    deleteDirectory(directoryName: string, options?: DirectoryDeleteOptions): Promise<DirectoryDeleteResponse>;
    /**
     * Creates a new file or replaces a file under the root directory of this share.
     * Note it only initializes the file with no content.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-file
     *
     * @param fileName -
     * @param size - Specifies the maximum size in bytes for the file, up to 4 TB.
     * @param options - Options to File Create operation.
     * @returns File creation response data and the corresponding file client.
     */
    createFile(fileName: string, size: number, options?: FileCreateOptions): Promise<{
        fileClient: ShareFileClient;
        fileCreateResponse: FileCreateResponse;
    }>;
    /**
     * Removes a file under the root directory of this share from the storage account.
     * When a file is successfully deleted, it is immediately removed from the storage
     * account's index and is no longer accessible to clients. The file's data is later
     * removed from the service during garbage collection.
     *
     * Delete File will fail with status code 409 (Conflict) and error code `SharingViolation`
     * if the file is open on an SMB client.
     *
     * Delete File is not supported on a share snapshot, which is a read-only copy of
     * a share. An attempt to perform this operation on a share snapshot will fail with 400
     * (`InvalidQueryParameterValue`)
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-file2
     *
     * @param directoryName -
     * @param fileName -
     * @param options - Options to File Delete operation.
     * @returns Promise<FileDeleteResponse> File Delete response data.
     */
    deleteFile(fileName: string, options?: FileDeleteOptions): Promise<FileDeleteResponse>;
    /**
     * Returns true if the Azrue share resource represented by this client exists; false otherwise.
     *
     * NOTE: use this function with care since an existing share might be deleted by other clients or
     * applications. Vice versa new shares might be added by other clients or applications after this
     * function completes.
     *
     * @param options - options to Exists operation.
     */
    exists(options?: ShareExistsOptions): Promise<boolean>;
    /**
     * Returns all user-defined metadata and system properties for the specified
     * share.
     * @see https://learn.microsoft.com/rest/api/storageservices/get-share-properties
     *
     * WARNING: The `metadata` object returned in the response will have its keys in lowercase, even if
     * they originally contained uppercase characters. This differs from the metadata keys returned by
     * the `listShares` method of {@link ShareServiceClient} using the `includeMetadata` option, which
     * will retain their original casing.
     *
     * @returns Response data for the Share Get Properties operation.
     */
    getProperties(options?: ShareGetPropertiesOptions): Promise<ShareGetPropertiesResponse>;
    /**
     * Marks the specified share for deletion. The share and any directories or files
     * contained within it are later deleted during garbage collection.
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-share
     *
     * @param options - Options to Share Delete operation.
     * @returns Response data for the Share Delete operation.
     */
    delete(options?: ShareDeleteMethodOptions): Promise<ShareDeleteResponse>;
    /**
     * Marks the specified share for deletion if it exists. The share and any directories or files
     * contained within it are later deleted during garbage collection.
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-share
     *
     * @param options -
     */
    deleteIfExists(options?: ShareDeleteMethodOptions): Promise<ShareDeleteIfExistsResponse>;
    /**
     * Sets one or more user-defined name-value pairs for the specified share.
     *
     * If no option provided, or no metadata defined in the option parameter, the share
     * metadata will be removed.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-share-metadata
     *
     * @param metadata - If no metadata provided, all existing directory metadata will be removed.
     * @param option - Options to Share Set Metadata operation.
     * @returns Response data for the Share Set Metadata operation.
     */
    setMetadata(metadata?: Metadata, options?: ShareSetMetadataOptions): Promise<ShareSetMetadataResponse>;
    /**
     * Gets the permissions for the specified share. The permissions indicate
     * whether share data may be accessed publicly.
     *
     * WARNING: JavaScript Date will potential lost precision when parsing start and expiry string.
     * For example, new Date("2018-12-31T03:44:23.8827891Z").toISOString() will get "2018-12-31T03:44:23.882Z".
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/get-share-acl
     *
     * @param option - Options to Share Get Access Policy operation.
     * @returns Response data for the Share Get Access Policy operation.
     */
    getAccessPolicy(options?: ShareGetAccessPolicyOptions): Promise<ShareGetAccessPolicyResponse>;
    /**
     * Sets the permissions for the specified share. The permissions indicate
     * whether directories or files in a share may be accessed publicly.
     *
     * When you set permissions for a share, the existing permissions are replaced.
     * If no shareAcl provided, the existing share ACL will be
     * removed.
     *
     * When you establish a stored access policy on a share, it may take up to 30 seconds to take effect.
     * During this interval, a shared access signature that is associated with the stored access policy will
     * fail with status code 403 (Forbidden), until the access policy becomes active.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-share-acl
     *
     * @param shareAcl - Array of signed identifiers, each having a unique Id and details of access policy.
     * @param option - Options to Share Set Access Policy operation.
     * @returns Response data for the Share Set Access Policy operation.
     */
    setAccessPolicy(shareAcl?: SignedIdentifier[], options?: ShareSetAccessPolicyOptions): Promise<ShareSetAccessPolicyResponse>;
    /**
     * Creates a read-only snapshot of a share.
     *
     * @param options - Options to Share Create Snapshot operation.
     * @returns Response data for the Share Create Snapshot operation.
     */
    createSnapshot(options?: ShareCreateSnapshotOptions): Promise<ShareCreateSnapshotResponse>;
    /**
     * Sets quota for the specified share.
     *
     * @deprecated Use {@link ShareClient.setProperties} instead.
     *
     * @param quotaInGB - Specifies the maximum size of the share in gigabytes
     * @param option - Options to Share Set Quota operation.
     * @returns Response data for the Share Get Quota operation.
     */
    setQuota(quotaInGB: number, options?: ShareSetQuotaOptions): Promise<ShareSetQuotaResponse>;
    /**
     * Sets properties of the share.
     *
     * @param option - Options to Share Set Properties operation.
     * @returns Response data for the Share Set Properties operation.
     */
    setProperties(options?: ShareSetPropertiesOptions): Promise<ShareSetPropertiesResponse>;
    /**
     * Retrieves statistics related to the share.
     *
     * @param option - Options to Share Get Statistics operation.
     * @returns Response data for the Share Get Statistics operation.
     */
    getStatistics(options?: ShareGetStatisticsOptions): Promise<ShareGetStatisticsResponse>;
    /**
     * Creates a file permission (a security descriptor) at the share level.
     * The created security descriptor can be used for the files/directories in the share.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-permission
     *
     * @param options - Options to Share Create Permission operation.
     * @param filePermission - File permission described in the SDDL
     */
    createPermission(filePermission: string | SharePermission, options?: ShareCreatePermissionOptions): Promise<ShareCreatePermissionResponse>;
    /**
     * Gets the Security Descriptor Definition Language (SDDL) for a given file permission key
     * which indicates a security descriptor.
     * @see https://learn.microsoft.com/rest/api/storageservices/get-permission
     *
     * @param options - Options to Share Create Permission operation.
     * @param filePermissionKey - File permission key which indicates the security descriptor of the permission.
     */
    getPermission(filePermissionKey: string, options?: ShareGetPermissionOptions): Promise<ShareGetPermissionResponse>;
    /**
     * Get a {@link ShareLeaseClient} that manages leases on the file.
     *
     * @param proposeLeaseId - Initial proposed lease Id.
     * @returns A new ShareLeaseClient object for managing leases on the file.
     */
    getShareLeaseClient(proposeLeaseId?: string): ShareLeaseClient;
    /**
     * Only available for ShareClient constructed with a shared key credential.
     *
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasUrl(options: ShareGenerateSasUrlOptions): string;
    /**
     * Only available for ShareClient constructed with a shared key credential.
     *
     * Generates string to sign for a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasStringToSign(options: ShareGenerateSasUrlOptions): string;
    /**
     *
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the user delegation key credential input.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @param userDelegationKey - user delegation key used to sign the SAS URI
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateUserDelegationSasUrl(options: ShareGenerateSasUrlOptions, userDelegationKey: UserDelegationKey): string;
    /**
     *
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the user delegation key credential input.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @param userDelegationKey - user delegation key used to sign the SAS URI
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateUserDelegationStringToSign(options: ShareGenerateSasUrlOptions, userDelegationKey: UserDelegationKey): string;
}
/**
 * Options to configure {@link ShareDirectoryClient.create} operation.
 */
export interface DirectoryCreateOptions extends FileAndDirectoryCreateCommonOptions, CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * A collection of key-value string pair to associate with the file storage object.
     */
    metadata?: Metadata;
}
export interface DirectoryProperties extends FileAndDirectorySetPropertiesCommonOptions, CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure {@link ShareDirectoryClient.listFilesAndDirectories} operation.
 */
export interface DirectoryListFilesAndDirectoriesOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Filters the results to return only entries whose
     * name begins with the specified prefix.
     */
    prefix?: string;
    includeTimestamps?: boolean;
    includeEtag?: boolean;
    includeAttributes?: boolean;
    includePermissionKey?: boolean;
    /**
     * Optional. Specified that extended info should be included in the returned {@link FileItem} or {@link DirectoryItem}.
     * If true, the Content-Length property will be up-to-date, FileId will be returned in response.
     */
    includeExtendedInfo?: boolean;
}
/**
 * Options to configure the {@link ShareDirectoryClient.delete} operation.
 */
export interface DirectoryDeleteOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure the {@link ShareDirectoryClient.exists} operation.
 */
export interface DirectoryExistsOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure the {@link ShareDirectoryClient.getProperties} operation.
 */
export interface DirectoryGetPropertiesOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure the {@link ShareDirectoryClient.setMetadata} operation.
 */
export interface DirectorySetMetadataOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure Directory - List Handles Segment operations.
 *
 * See:
 * - {@link ShareDirectoryClient.listHandlesSegment}
 * - {@link ShareDirectoryClient.iterateHandleSegments}
 * - {@link ShareDirectoryClient.listHandleItems}
 *
 */
export interface DirectoryListHandlesSegmentOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Specifies the maximum number of entries to return. If the request does not specify maxResults,
     * or specifies a value greater than 5,000, the server will return up to 5,000 items.
     */
    maxResults?: number;
    /**
     * Specifies operation should apply to the directory specified in the URI, its files, its
     * subdirectories and their files.
     */
    recursive?: boolean;
}
/**
 * Options to configure the {@link ShareDirectoryClient.listHandles} operation.
 */
export interface DirectoryListHandlesOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Specifies whether operation should apply to the directory specified in the URI, its files, its
     * subdirectories and their files.
     */
    recursive?: boolean;
}
/**
 * Options to configure Directory - Force Close Handles Segment operations.
 *
 * See:
 * - {@link ShareDirectoryClient.forceCloseHandlesSegment}
 * - {@link ShareDirectoryClient.forceCloseAllHandles}
 */
export interface DirectoryForceCloseHandlesSegmentOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Specifies operation should apply to the directory specified in the URI, its files, its
     * subdirectories and their files.
     */
    recursive?: boolean;
}
/**
 * Additional response header values for close handles request.
 */
export interface DirectoryCloseHandlesHeaders {
    /**
     * This header uniquely identifies the request that was made and can be used for troubleshooting
     * the request.
     */
    requestId?: string;
    /**
     * Indicates the version of the File service used to execute the request.
     */
    version?: string;
    /**
     * A UTC date/time value generated by the service that indicates the time at which the response
     * was initiated.
     */
    date?: Date;
    /**
     * A string describing next handle to be closed. It is returned when more handles need to be
     * closed to complete the request.
     */
    marker?: string;
}
/**
 * Response type for {@link ShareDirectoryClient.forceCloseHandle}.
 */
export type DirectoryForceCloseHandlesResponse = WithResponse<CloseHandlesInfo & DirectoryCloseHandlesHeaders, DirectoryForceCloseHandlesHeaders>;
/**
 * Options to configure {@link ShareDirectoryClient.forceCloseHandle}.
 */
export interface DirectoryForceCloseHandlesOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Contains response data for the {@link DirectoryClient.createIfNotExists} operation.
 */
export interface DirectoryCreateIfNotExistsResponse extends DirectoryCreateResponse {
    /**
     * Indicate whether the directory is successfully created. Is false when the directory is not changed as it already exists.
     */
    succeeded: boolean;
}
/**
 * Contains response data for the {@link DirectoryClient.deleteIfExists} operation.
 */
export interface DirectoryDeleteIfExistsResponse extends DirectoryDeleteResponse {
    /**
     * Indicate whether the directory is successfully deleted. Is false if the directory does not exist in the first place.
     */
    succeeded: boolean;
}
/**
 * A ShareDirectoryClient represents a URL to the Azure Storage directory allowing you to manipulate its files and directories.
 */
export declare class ShareDirectoryClient extends StorageClient {
    /**
     * context provided by protocol layer.
     */
    private context;
    private _shareName;
    private _path;
    private _name;
    private shareClientConfig?;
    /**
     * The share name corresponding to this directory client
     */
    get shareName(): string;
    /**
     * The full path of the directory
     */
    get path(): string;
    /**
     * The name of the directory
     */
    get name(): string;
    /**
     * Creates an instance of DirectoryClient.
     *
     * @param url - A URL string pointing to Azure Storage file directory, such as
     *                     "https://myaccount.file.core.windows.net/myshare/mydirectory". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.file.core.windows.net/myshare/mydirectory?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a directory.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a directory name includes %, directory name must be encoded in the URL.
     *                     Such as a directory named "mydir%", the URL should be "https://myaccount.file.core.windows.net/myshare/mydir%25".
     * @param credential - Such as AnonymousCredential or StorageSharedKeyCredential.
     *                                  If not specified, AnonymousCredential is used.
     * @param options - Optional. Options to configure the HTTP pipeline.
     */
    constructor(url: string, credential?: Credential | TokenCredential, options?: ShareClientOptions);
    /**
     * Creates an instance of DirectoryClient.
     *
     * @param url - A URL string pointing to Azure Storage file directory, such as
     *                     "https://myaccount.file.core.windows.net/myshare/mydirectory". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.file.core.windows.net/myshare/mydirectory?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a directory.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a directory name includes %, directory name must be encoded in the URL.
     *                     Such as a directory named "mydir%", the URL should be "https://myaccount.file.core.windows.net/myshare/mydir%25".
     * @param pipeline - Call newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     */
    constructor(url: string, pipeline: Pipeline, options?: ShareClientConfig);
    /**
     * Creates a new directory under the specified share or parent directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-directory
     *
     * @param options - Options to Directory Create operation.
     * @returns Response data for the Directory  operation.
     */
    create(options?: DirectoryCreateOptions): Promise<DirectoryCreateResponse>;
    /**
     * Creates a new directory under the specified share or parent directory if it does not already exists.
     * If the directory already exists, it is not modified.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-directory
     *
     * @param options -
     */
    createIfNotExists(options?: DirectoryCreateOptions): Promise<DirectoryCreateIfNotExistsResponse>;
    /**
     * Sets properties on the directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-directory-properties
     *
     * @param DirectoryProperties - Directory properties. If no values are provided,
     *                                            existing values will be preserved.
     */
    setProperties(properties?: DirectoryProperties): Promise<DirectorySetPropertiesResponse>;
    /**
     * Creates a ShareDirectoryClient object for a sub directory.
     *
     * @param subDirectoryName - A subdirectory name
     * @returns The ShareDirectoryClient object for the given subdirectory name.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleGetDirectoryClient
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const shareClient = serviceClient.getShareClient(shareName);
     * const directoryClient = shareClient.getDirectoryClient(directoryName);
     * await directoryClient.create();
     * ```
     */
    getDirectoryClient(subDirectoryName: string): ShareDirectoryClient;
    /**
     * Creates a new subdirectory under this directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-directory
     *
     * @param directoryName -
     * @param options - Options to Directory Create operation.
     * @returns Directory create response data and the corresponding DirectoryClient instance.
     */
    createSubdirectory(directoryName: string, options?: DirectoryCreateOptions): Promise<{
        directoryClient: ShareDirectoryClient;
        directoryCreateResponse: DirectoryCreateResponse;
    }>;
    /**
     * Removes the specified empty sub directory under this directory.
     * Note that the directory must be empty before it can be deleted.
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-directory
     *
     * @param directoryName -
     * @param options - Options to Directory Delete operation.
     * @returns Directory deletion response data.
     */
    deleteSubdirectory(directoryName: string, options?: DirectoryDeleteOptions): Promise<DirectoryDeleteResponse>;
    /**
     * Creates a new file or replaces a file under this directory. Note it only initializes the file with no content.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-file
     *
     * @param fileName -
     * @param size - Specifies the maximum size in bytes for the file, up to 4 TB.
     * @param options - Options to File Create operation.
     * @returns File creation response data and the corresponding file client.
     */
    createFile(fileName: string, size: number, options?: FileCreateOptions): Promise<{
        fileClient: ShareFileClient;
        fileCreateResponse: FileCreateResponse;
    }>;
    /**
     * Removes the specified file under this directory from the storage account.
     * When a file is successfully deleted, it is immediately removed from the storage
     * account's index and is no longer accessible to clients. The file's data is later
     * removed from the service during garbage collection.
     *
     * Delete File will fail with status code 409 (Conflict) and error code SharingViolation
     * if the file is open on an SMB client.
     *
     * Delete File is not supported on a share snapshot, which is a read-only copy of
     * a share. An attempt to perform this operation on a share snapshot will fail with 400 (InvalidQueryParameterValue)
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-file2
     *
     * @param fileName - Name of the file to delete
     * @param options - Options to File Delete operation.
     * @returns File deletion response data.
     */
    deleteFile(fileName: string, options?: FileDeleteOptions): Promise<FileDeleteResponse>;
    /**
     * Creates a {@link ShareFileClient} object.
     *
     * @param fileName - A file name.
     * @returns A new ShareFileClient object for the given file name.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleCreateFileAndUpload
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * const content = "Hello World!";
     * const fileName = `newdirectory${+new Date()}`;
     * const fileClient = directoryClient.getFileClient(fileName);
     * await fileClient.create(content.length);
     * console.log(`Create file ${fileName} successfully`);
     *
     * // Upload file range
     * await fileClient.uploadRange(content, 0, content.length);
     * console.log(`Upload file range "${content}" to ${fileName} successfully`);
     * ```
     */
    getFileClient(fileName: string): ShareFileClient;
    /**
     * Returns true if the specified directory exists; false otherwise.
     *
     * NOTE: use this function with care since an existing directory might be deleted by other clients or
     * applications. Vice versa new directories might be added by other clients or applications after this
     * function completes.
     *
     * @param options - options to Exists operation.
     */
    exists(options?: DirectoryExistsOptions): Promise<boolean>;
    /**
     * Returns all system properties for the specified directory, and can also be used to check the
     * existence of a directory. The data returned does not include the files in the directory or any
     * subdirectories.
     * @see https://learn.microsoft.com/rest/api/storageservices/get-directory-properties
     *
     * @param options - Options to Directory Get Properties operation.
     * @returns Response data for the Directory Get Properties operation.
     */
    getProperties(options?: DirectoryGetPropertiesOptions): Promise<DirectoryGetPropertiesResponse>;
    /**
     * Removes the specified empty directory. Note that the directory must be empty before it can be
     * deleted.
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-directory
     *
     * @param options - Options to Directory Delete operation.
     * @returns Response data for the Directory Delete operation.
     */
    delete(options?: DirectoryDeleteOptions): Promise<DirectoryDeleteResponse>;
    /**
     * Removes the specified empty directory if it exists. Note that the directory must be empty before it can be
     * deleted.
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-directory
     *
     * @param options -
     */
    deleteIfExists(options?: DirectoryDeleteOptions): Promise<DirectoryDeleteIfExistsResponse>;
    /**
     * Updates user defined metadata for the specified directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-directory-metadata
     *
     * @param metadata - If no metadata provided, all existing directory metadata will be removed
     * @param options - Options to Directory Set Metadata operation.
     * @returns Response data for the Directory Set Metadata operation.
     */
    setMetadata(metadata?: Metadata, options?: DirectorySetMetadataOptions): Promise<DirectorySetMetadataResponse>;
    /**
     * Returns an AsyncIterableIterator for {@link DirectoryListFilesAndDirectoriesSegmentResponse} objects
     *
     * @param marker - A string value that identifies the portion of
     *                          the list of files and directories to be returned with the next listing operation. The
     *                          operation returns the ContinuationToken value within the response body if the
     *                          listing operation did not return all files and directories remaining to be listed
     *                          with the current page. The ContinuationToken value can be used as the value for
     *                          the marker parameter in a subsequent call to request the next page of list
     *                          items. The marker value is opaque to the client.
     * @param options - Options to list files and directories operation.
     */
    private iterateFilesAndDirectoriesSegments;
    /**
     * Returns an AsyncIterableIterator for file and directory items
     *
     * @param options - Options to list files and directories operation.
     */
    private listFilesAndDirectoriesItems;
    /**
     * Returns an async iterable iterator to list all the files and directories
     * under the specified account.
     *
     * .byPage() returns an async iterable iterator to list the files and directories in pages.
     *
     * Example using `for await` syntax:
     *
     * ```ts snippet:ReadmeSampleListFilesAndDirectories
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * let i = 1;
     * for await (const item of directoryClient.listFilesAndDirectories()) {
     *   if (item.kind === "directory") {
     *     console.log(`${i} - directory\t: ${item.name}`);
     *   } else {
     *     console.log(`${i} - file\t: ${item.name}`);
     *   }
     *   i++;
     * }
     * ```
     *
     * Example using `iter.next()`:
     *
     * ```ts snippet:ReadmeSampleListFilesAndDirectories_Iterator
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * let i = 1;
     * const iter = directoryClient.listFilesAndDirectories();
     * let { value, done } = await iter.next();
     * while (!done) {
     *   if (value.kind === "directory") {
     *     console.log(`${i} - directory\t: ${value.name}`);
     *   } else {
     *     console.log(`${i} - file\t: ${value.name}`);
     *   }
     *   ({ value, done } = await iter.next());
     *   i++;
     * }
     * ```
     *
     * Example using `byPage()`:
     *
     * ```ts snippet:ReadmeSampleListFilesAndDirectories_ByPage
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * let i = 1;
     * for await (const response of directoryClient
     *   .listFilesAndDirectories()
     *   .byPage({ maxPageSize: 20 })) {
     *   console.log(`Page ${i++}:`);
     *   for (const item of response.segment.directoryItems) {
     *     console.log(`\tdirectory: ${item.name}`);
     *   }
     *   for (const item of response.segment.fileItems) {
     *     console.log(`\tfile: ${item.name}`);
     *   }
     * }
     * ```
     *
     * Example using paging with a marker:
     *
     * ```ts snippet:ReadmeSampleListFilesAndDirectories_Continuation
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * let iterator = directoryClient.listFilesAndDirectories().byPage({ maxPageSize: 2 });
     * let response = (await iterator.next()).value;
     *
     * for await (const item of response.segment.directoryItems) {
     *   console.log(`\tdirectory: ${item.name}`);
     * }
     *
     * for await (const item of response.segment.fileItems) {
     *   console.log(`\tfile: ${item.name}`);
     * }
     *
     * // Gets next marker
     * let marker = response.continuationToken;
     *
     * // Passing next marker as continuationToken
     * iterator = directoryClient
     *   .listFilesAndDirectories()
     *   .byPage({ continuationToken: marker, maxPageSize: 10 });
     * response = (await iterator.next()).value;
     *
     * for await (const item of response.segment.directoryItems) {
     *   console.log(`\tdirectory: ${item.name}`);
     * }
     *
     * for await (const item of response.segment.fileItems) {
     *   console.log(`\tfile: ${item.name}`);
     * }
     * ```
     *
     * @param options - Options to list files and directories operation.
     * @returns An asyncIterableIterator that supports paging.
     */
    listFilesAndDirectories(options?: DirectoryListFilesAndDirectoriesOptions): PagedAsyncIterableIterator<({
        kind: "file";
    } & FileItem) | ({
        kind: "directory";
    } & DirectoryItem), DirectoryListFilesAndDirectoriesSegmentResponse>;
    /**
     * Returns a list of files or directories under the specified share or directory. It lists the
     * contents only for a single level of the directory hierarchy.
     * @see https://learn.microsoft.com/rest/api/storageservices/list-directories-and-files
     *
     * @param marker - A string value that identifies the portion of the list to be returned with the next list operation.
     * @param options - Options to Directory List Files and Directories Segment operation.
     * @returns Response data for the Directory List Files and Directories operation.
     */
    private listFilesAndDirectoriesSegment;
    /**
     * Returns an AsyncIterableIterator for {@link DirectoryListHandlesResponse}
     *
     * @param marker - A string value that identifies the portion of the list to be
     *                          returned with the next list handles operation. The operation returns a
     *                          marker value within the response body if the list returned was not complete.
     *                          The marker value may then be used in a subsequent call to request the next
     *                          set of list items.
     * @param options - Options to list handles operation.
     */
    private iterateHandleSegments;
    /**
     * Returns an AsyncIterableIterator for handles
     *
     * @param options - Options to list handles operation.
     */
    private listHandleItems;
    /**
     * Returns an async iterable iterator to list all the handles.
     * under the specified account.
     *
     * .byPage() returns an async iterable iterator to list the handles in pages.
     *
     * Example using `for await` syntax:
     *
     * ```ts snippet:ReadmeSampleListHandles
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * for await (const handle of directoryClient.listHandles()) {
     *   console.log(`Handle: ${handle.handleId}`);
     * }
     * ```
     *
     * Example using `iter.next()`:
     *
     * ```ts snippet:ReadmeSampleListHandles_Iterator
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * const handleIter = directoryClient.listHandles();
     * let { value, done } = await handleIter.next();
     * while (!done) {
     *   console.log(`Handle: ${value.handleId}`);
     *   ({ value, done } = await handleIter.next());
     * }
     * ```
     *
     * Example using `byPage()`:
     *
     * ```ts snippet:ReadmeSampleListHandles_ByPage
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * let i = 1;
     * for await (const response of directoryClient.listHandles().byPage({ maxPageSize: 20 })) {
     *   console.log(`Page ${i++}:`);
     *   for (const handle of response.handleList || []) {
     *     console.log(`\thandle: ${handle.handleId}`);
     *   }
     * }
     * ```
     *
     * Example using paging with a marker:
     *
     * ```ts snippet:ReadmeSampleListHandles_Continuation
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * let iterator = directoryClient.listHandles().byPage({ maxPageSize: 2 });
     * let response = (await iterator.next()).value;
     *
     * for await (const handle of response.handleList || []) {
     *   console.log(`\thandle: ${handle.handleId}`);
     * }
     *
     * // Gets next marker
     * let marker = response.continuationToken;
     *
     * // Passing next marker as continuationToken
     * iterator = directoryClient.listHandles().byPage({ continuationToken: marker, maxPageSize: 10 });
     * response = (await iterator.next()).value;
     *
     * for await (const handle of response.handleList || []) {
     *   console.log(`\thandle: ${handle.handleId}`);
     * }
     * ```
     *
     * @param options - Options to list handles operation.
     *
     * An asyncIterableIterator that supports paging.
     */
    listHandles(options?: DirectoryListHandlesOptions): PagedAsyncIterableIterator<HandleItem, DirectoryListHandlesResponse>;
    /**
     * Lists handles for a directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/list-handles
     *
     * @param marker - Optional. A string value that identifies the portion of the list to be
     *                          returned with the next list handles operation. The operation returns a
     *                          marker value within the response body if the list returned was not complete.
     *                          The marker value may then be used in a subsequent call to request the next
     *                          set of list items.
     * @param options -
     */
    private listHandlesSegment;
    /**
     * Force close all handles for a directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/force-close-handles
     *
     * @param marker - Optional. A string value that identifies the position of handles that will
     *                          be closed with the next force close handles operation.
     *                          The operation returns a marker value within the response
     *                          body if there are more handles to close. The marker value
     *                          may then be used in a subsequent call to close the next set of handles.
     * @param options -
     */
    private forceCloseHandlesSegment;
    /**
     * Force close all handles for a directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/force-close-handles
     *
     * @param options -
     */
    forceCloseAllHandles(options?: DirectoryForceCloseHandlesSegmentOptions): Promise<CloseHandlesInfo>;
    /**
     * Force close a specific handle for a directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/force-close-handles
     *
     * @param aborter - Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param handleId - Specific handle ID, cannot be asterisk "*".
     *                          Use forceCloseHandlesSegment() to close all handles.
     * @param options -
     */
    forceCloseHandle(handleId: string, options?: DirectoryForceCloseHandlesOptions): Promise<DirectoryForceCloseHandlesResponse>;
    /**
     * Renames a directory.
     * This API only supports renaming a directory in the same share.
     *
     * @param destinationPath - Specifies the destination path to rename to. The path will be encoded to put into a URL to specify the destination.
     * @param options - Options for the renaming operation.
     * @returns Response data for the file renaming operation.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleRenameDirectory
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const destinationPath = "<destination path>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * await directoryClient.rename(destinationPath);
     * ```
     */
    rename(destinationPath: string, options?: DirectoryRenameOptions): Promise<{
        destinationDirectoryClient: ShareDirectoryClient;
        directoryRenameResponse: DirectoryRenameResponse;
    }>;
}
/**
 * Options to configure the {@link ShareFileClient.create} operation.
 */
export interface FileCreateOptions extends FileAndDirectoryCreateCommonOptions, CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * File HTTP headers like Content-Type.
     */
    fileHttpHeaders?: FileHttpHeaders;
    /**
     * A collection of key-value string pair to associate with the file storage object.
     */
    metadata?: Metadata;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
export interface FileProperties extends FileAndDirectorySetPropertiesCommonOptions, CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * File HTTP headers like Content-Type.
     */
    fileHttpHeaders?: FileHttpHeaders;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
export interface SetPropertiesResponse extends FileSetHTTPHeadersResponse {
}
/**
 * Options to configure the {@link ShareFileClient.delete} operation.
 */
export interface FileDeleteOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure File - Download operations.
 *
 * See:
 * - {@link ShareFileClient.download}
 * - {@link ShareFileClient.downloadToFile}
 */
export interface FileDownloadOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Optional. ONLY AVAILABLE IN NODE.JS.
     *
     * How many retries will perform when original body download stream unexpected ends.
     * Above kind of ends will not trigger retry policy defined in a pipeline,
     * because they doesn't emit network errors.
     *
     * With this option, every additional retry means an additional ShareFileClient.download() request will be made
     * from the broken point, until the requested range has been successfully downloaded or maxRetryRequests is reached.
     *
     * Default value is 5, please set a larger value when loading large files in poor network.
     */
    maxRetryRequests?: number;
    /**
     * When this header is set to true and
     * specified together with the Range header, the service returns the MD5 hash
     * for the range, as long as the range is less than or equal to 4 MB in size.
     */
    rangeGetContentMD5?: boolean;
    /**
     * Download progress updating event handler.
     */
    onProgress?: (progress: TransferProgressEvent) => void;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure the {@link ShareFileClient.uploadRange} operation.
 */
export interface FileUploadRangeOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * An MD5 hash of the content. This hash is
     * used to verify the integrity of the data during transport. When the
     * Content-MD5 header is specified, the File service compares the hash of the
     * content that has arrived with the header value that was sent. If the two
     * hashes do not match, the operation will fail with error code 400 (Bad
     * Request).
     */
    contentMD5?: Uint8Array;
    /**
     * Progress updating event handler.
     */
    onProgress?: (progress: TransferProgressEvent) => void;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
    /**
     * The last write time for the file.
     * A value of Preserve may be passed to keep an existing value unchanged.
     * A value of Now may be used to indicate the time of the request.
     * By default, the value will be set as Now.
     */
    fileLastWrittenMode?: FileLastWrittenMode;
}
/**
 * Options to configure the {@link ShareFileClient.uploadRangeFromURL} operation.
 */
export interface FileUploadRangeFromURLOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * The timeout parameter is expressed in seconds. For more information, see <a
     * href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN">Setting
     * Timeouts for File Service Operations.</a>
     */
    timeoutInSeconds?: number;
    /**
     * Specify the crc64 calculated for the range of bytes that must be read from the copy source.
     */
    sourceContentCrc64?: Uint8Array;
    /**
     * Additional parameters for the operation
     */
    sourceConditions?: SourceModifiedAccessConditions;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
    /**
     * Only Bearer type is supported. Credentials should be a valid OAuth access token to copy source.
     */
    sourceAuthorization?: HttpAuthorization;
    /**
     * The last write time for the file.
     * A value of preserve may be passed to keep an existing value unchanged.
     * A value of now may be used to indicate the time of the request.
     * By default, the value will be set as now.
     */
    fileLastWrittenMode?: FileLastWrittenMode;
}
/**
 * The option is defined as parity to REST definition.
 * While it's not ready to be used now, considering Crc64 of source content is
 * not accessible.
 */
/**
 * Options to configure the {@link ShareFileClient.getRangeList} operation.
 */
export interface FileGetRangeListOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Optional. Specifies the range of bytes over which to list ranges, inclusively.
     */
    range?: Range;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
    /**
     * This header is allowed only when prevShareSnapshot parameter is set.
     * Determines whether the changed ranges for a file that has been renamed or moved between the target snapshot (or the live file) and the previous snapshot should be listed.
     * If the value is true, the valid changed ranges for the file will be returned. If the value is false, the operation will result in a failure with 409 (Conflict) response.
     * The default value is false.
     */
    includeRenames?: boolean;
}
/**
 * Options to configure the {@link ShareFileClient.exists} operation.
 */
export interface FileExistsOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure the {@link ShareFileClient.getProperties} operation.
 */
export interface FileGetPropertiesOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Contains response data for the {@link ShareFileClient.getRangeList} operation.
 */
export type FileGetRangeListResponse = WithResponse<FileGetRangeListHeaders & {
    /**
     * Range list for an Azure file.
     */
    rangeList: RangeModel[];
}, FileGetRangeListHeaders, RangeModel[]>;
/**
 * Options to configure the {@link ShareFileClient.startCopyFromURL} operation.
 */
export interface FileStartCopyOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * A collection of key-value string pair to associate with the file storage object.
     */
    metadata?: Metadata;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
    /**
     * If specified the permission (security descriptor) shall be set for the directory/file. This
     * header can be used if Permission size is lesser than or equal to 8KB, else x-ms-file-permission-key header shall be
     * used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and
     * dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be
     * specified.
     */
    filePermission?: string;
    /**
     * Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned.
     * Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format.
     * If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission
     */
    filePermissionFormat?: FilePermissionFormat;
    /**
     * Key of the permission to be set for the directory/file. Note: Only one of the
     * x-ms-file-permission or x-ms-file-permission-key should be specified.
     */
    filePermissionKey?: string;
    /**
     * SMB info.
     */
    copyFileSmbInfo?: CopyFileSmbInfo;
    /**
     * Optional properties to set on NFS files.
       Note that this property is only applicable to files created in NFS shares.
     */
    posixProperties?: FilePosixProperties;
    /**
     * Optional, only applicable to NFS Files.
     * Applicable only when the copy source is a File. Determines the copy behavior of the mode bits of the file.
     * source: The mode on the destination file is copied from the source file.
     * override: The mode on the destination file is determined via the x-ms-mode header.
     */
    fileModeCopyMode?: ModeCopyMode;
    /**
     * Optional, only applicable to NFS Files. Determines the copy behavior of the owner user identifier (UID) and group identifier (GID) of the file.
     * source: The owner user identifier (UID) and group identifier (GID) on the destination file is copied from the source file.
     * override: The owner user identifier (UID) and group identifier (GID) on the destination file is determined via the x-ms-owner and x-ms-group  headers.
     */
    fileOwnerCopyMode?: OwnerCopyMode;
}
/**
 * Options to configure the {@link ShareFileClient.setMetadata} operation.
 */
export interface FileSetMetadataOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure the {@link ShareFileClient.setHttpHeaders} operation.
 */
export interface FileSetHttpHeadersOptions extends FileAndDirectorySetPropertiesCommonOptions, CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure the {@link ShareFileClient.abortCopyFromURL} operation.
 */
export interface FileAbortCopyFromURLOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure the {@link ShareFileClient.resize} operation.
 */
export interface FileResizeOptions extends FileAndDirectorySetPropertiesCommonOptions, CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Options to configure the {@link ShareFileClient.clearRange} operation.
 */
export interface FileClearRangeOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
    /**
     * The last write time for the file.
     * A value of preserve may be passed to keep an existing value unchanged.
     * A value of now may be used to indicate the time of the request.
     * By default, the value will be set as now.
     */
    fileLastWrittenMode?: FileLastWrittenMode;
}
/**
 * Options to configure File - List Handles Segment operations.
 *
 * See:
 * - {@link ShareFileClient.listHandlesSegment}
 * - {@link ShareFileClient.iterateHandleSegments}
 * - {@link ShareFileClient.listHandleItems}
 */
export interface FileListHandlesSegmentOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Specifies the maximum number of entries to return. If the request does not specify maxResults,
     * or specifies a value greater than 5,000, the server will return up to 5,000 items.
     */
    maxPageSize?: number;
}
export interface FileListHandlesOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure File - Force Close Handles operations.
 *
 * See:
 * - {@link ShareFileClient.forceCloseHandlesSegment}
 * - {@link ShareFileClient.forceCloseAllHandles}
 * - {@link ShareFileClient.forceCloseHandle}
 */
export interface FileForceCloseHandlesOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure File - Create Hard Link operations.
 *
 * See:
 * - {@link ShareFileClient.createHardLink}
 */
export interface FileCreateHardLinkOptions extends CommonOptions {
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure File - Create Symbolic Link operations.
 *
 * See:
 * - {@link ShareFileClient.createSymbolicLink}
 */
export interface FileCreateSymbolicLinkOptions extends CommonOptions {
    /**
     * Metadata of the Azure file.
     */
    metadata?: Metadata;
    /**
     * The Coordinated Universal Time (UTC) creation time property for the directory.
     * A value of now may be used to indicate the time of the request.
     * By default, the value will be set as now.
     */
    creationTime?: Date | TimeNowType;
    /**
     * The Coordinated Universal Time (UTC) last write property for the directory.
     * A value of now may be used to indicate the time of the request.
     * By default, the value will be set as now.
     */
    lastWriteTime?: Date | TimeNowType;
    /** Optional, NFS only.
     * The owner of the file or directory.
     * */
    owner?: string;
    /** Optional, NFS only.
     * The owning group of the file or directory.
     * */
    group?: string;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure File - Get Symbolic Link operations.
 *
 * See:
 * - {@link ShareFileClient.getSymbolicLink}
 */
export interface FileGetSymbolicLinkOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Additional response header values for close handles request.
 */
export interface FileCloseHandlesHeaders {
    /**
     * This header uniquely identifies the request that was made and can be used for troubleshooting
     * the request.
     */
    requestId?: string;
    /**
     * Indicates the version of the File service used to execute the request.
     */
    version?: string;
    /**
     * A UTC date/time value generated by the service that indicates the time at which the response
     * was initiated.
     */
    date?: Date;
    /**
     * A string describing next handle to be closed. It is returned when more handles need to be
     * closed to complete the request.
     */
    marker?: string;
}
/**
 * Response type for {@link ShareFileClient.forceCloseHandle}.
 */
export type FileForceCloseHandlesResponse = WithResponse<CloseHandlesInfo & FileCloseHandlesHeaders, FileForceCloseHandlesHeaders>;
/**
 * Option interface for ShareFileClient.uploadStream().
 */
export interface FileUploadStreamOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Azure File HTTP Headers.
     */
    fileHttpHeaders?: FileHttpHeaders;
    /**
     * Metadata of the Azure file.
     */
    metadata?: Metadata;
    /**
     * Progress updater.
     */
    onProgress?: (progress: TransferProgressEvent) => void;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Option interface for File - Upload operations
 *
 * See:
 * - {@link ShareFileClient.uploadFile}
 * - {@link ShareFileClient.uploadSeekableStream}
 */
export interface FileParallelUploadOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * RangeSize specifies the range size to use in each parallel upload,
     * the default (and maximum size) is FILE_RANGE_MAX_SIZE_BYTES.
     */
    rangeSize?: number;
    /**
     * Progress updater.
     */
    onProgress?: (progress: TransferProgressEvent) => void;
    /**
     * File HTTP Headers.
     */
    fileHttpHeaders?: FileHttpHeaders;
    /**
     * Metadata of an Azure file.
     */
    metadata?: Metadata;
    /**
     * Concurrency indicates the maximum number of ranges to upload in parallel.
     * If not provided, 5 concurrency will be used by default.
     */
    concurrency?: number;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Option interface for the {@link ShareFileClient.downloadToBuffer} operation.
 */
export interface FileDownloadToBufferOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * When downloading Azure files, download method will try to split large file into small ranges.
     * Every small range will be downloaded via a separate request.
     * This option defines size data every small request trying to download.
     * Must be greater than 0, will use the default value if undefined,
     */
    rangeSize?: number;
    /**
     * Optional. ONLY AVAILABLE IN NODE.JS.
     *
     * How many retries will perform when original range download stream unexpected ends.
     * Above kind of ends will not trigger retry policy defined in a pipeline,
     * because they doesn't emit network errors.
     *
     * With this option, every additional retry means an additional ShareFileClient.download() request will be made
     * from the broken point, until the requested range has been successfully downloaded or
     * maxRetryRequestsPerRange is reached.
     *
     * Default value is 5, please set a larger value when in poor network.
     */
    maxRetryRequestsPerRange?: number;
    /**
     * Progress updater.
     */
    onProgress?: (progress: TransferProgressEvent) => void;
    /**
     * Concurrency indicates the maximum number of ranges to download in parallel.
     * If not provided, 5 concurrency will be used by default.
     */
    concurrency?: number;
    /**
     * Lease access conditions.
     */
    leaseAccessConditions?: LeaseAccessConditions;
}
/**
 * Contains response data for the {@link ShareFileClient.deleteIfExists} operation.
 */
export interface FileDeleteIfExistsResponse extends FileDeleteResponse {
    /**
     * Indicate whether the file is successfully deleted. Is false if the file does not exist in the first place.
     */
    succeeded: boolean;
}
/**
 * Options to configure {@link ShareFileClient.generateSasUrl} operation.
 */
export interface FileGenerateSasUrlOptions extends CommonGenerateSasUrlOptions {
    /**
     * Optional only when identifier is provided. Specifies the list of permissions to be associated with the SAS.
     */
    permissions?: FileSASPermissions;
}
/**
 * Options to configure the {@link ShareFileClient.rename} operation.
 */
export interface FileRenameOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Lease access condition for source file. Required if the source file has an active infinite lease.
     */
    sourceLeaseAccessConditions?: LeaseAccessConditions;
    /**
     * Lease access condition for destination file. Required if the destination file has an active infinite lease.
     */
    destinationLeaseAccessConditions?: LeaseAccessConditions;
    /**
     * Optional.
     * Specifies the option to copy file security descriptor from source file or to set it using the value which is defined by the header value of x-ms-file-permission or x-ms-file-permission-key.
     */
    copyFileSmbInfo?: CopyFileSmbInfo;
    /**
     * A name-value pair to associate with a file storage object.
     */
    metadata?: Metadata;
    /**
     * Optional.
     * The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN">Setting Timeouts for File Service Operations.</a>
     */
    timeoutInSeconds?: number;
    /**
     * Optional.
     * If specified the permission (security descriptor) shall be set for the directory/file.
     */
    filePermission?: string;
    /**
     * Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned.
     * Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format.
     * If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission
     */
    filePermissionFormat?: FilePermissionFormat;
    /**
     * Optional.
     * Key of the permission to be set for the directory/file. Note: Only one of the filePermission or filePermissionKey should be specified.
     */
    filePermissionKey?: string;
    /**
     * Optional.
     * A boolean value for if the destination file already exists, whether this request will overwrite the file or not. If true, the rename will succeed and will overwrite the destination file. If not provided or if false and the destination file does exist, the request will not overwrite the destination file. If provided and the destination file doesn’t exist, the rename will succeed. Note: This value does not override the x-ms-file-copy-ignore-read-only header value.
     */
    replaceIfExists?: boolean;
    /**
     * Optional.
     * A boolean value that specifies whether the ReadOnly attribute on a preexisting destination file should be respected. If true, the rename will succeed, otherwise, a previous file at the destination with the ReadOnly attribute set will cause the rename to fail.
     */
    ignoreReadOnly?: boolean;
    /**
     * Optional.
     * Content type to set on the File.
     */
    contentType?: string;
}
/**
 * Options to configure the {@link ShareDirectoryClient.rename} operation.
 */
export interface DirectoryRenameOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Lease access condition for source file. Required if the source file has an active infinite lease.
     */
    sourceLeaseAccessConditions?: LeaseAccessConditions;
    /**
     * Lease access condition for destination file. Required if the destination file has an active infinite lease.
     */
    destinationLeaseAccessConditions?: LeaseAccessConditions;
    /**
     * Optional.
     * Specifies the option to copy file security descriptor from source file or to set it using the value which is defined by the header value of x-ms-file-permission or x-ms-file-permission-key.
     */
    copyFileSmbInfo?: CopyFileSmbInfo;
    /**
     * Optional.
     * The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN">Setting Timeouts for File Service Operations.</a>
     */
    timeoutInSeconds?: number;
    /**
     * Optional.
     * A name-value pair to associate with a file storage object.
     */
    metadata?: Metadata;
    /**
     * Optional.
     * If specified the permission (security descriptor) shall be set for the directory/file.
     */
    filePermission?: string;
    /**
     * Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned.
     * Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format.
     * If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission
     */
    filePermissionFormat?: FilePermissionFormat;
    /**
     * Optional.
     * Key of the permission to be set for the directory/file. Note: Only one of the filePermission or filePermissionKey should be specified.
     */
    filePermissionKey?: string;
    /**
     * Optional.
     * A boolean value for if the destination file already exists, whether this request will overwrite the file or not. If true, the rename will succeed and will overwrite the destination file. If not provided or if false and the destination file does exist, the request will not overwrite the destination file. If provided and the destination file doesn’t exist, the rename will succeed. Note: This value does not override the x-ms-file-copy-ignore-read-only header value.
     */
    replaceIfExists?: boolean;
    /**
     * Optional.
     * A boolean value that specifies whether the ReadOnly attribute on a preexisting destination file should be respected. If true, the rename will succeed, otherwise, a previous file at the destination with the ReadOnly attribute set will cause the rename to fail.
     */
    ignoreReadOnly?: boolean;
}
/**
 * A ShareFileClient represents a URL to an Azure Storage file.
 */
export declare class ShareFileClient extends StorageClient {
    /**
     * context provided by protocol layer.
     */
    private context;
    private _shareName;
    private _path;
    private _name;
    private shareClientConfig?;
    /**
     * The share name corresponding to this file client
     */
    get shareName(): string;
    /**
     * The full path of the file
     */
    get path(): string;
    /**
     * The name of the file
     */
    get name(): string;
    /**
     * Creates an instance of ShareFileClient.
     *
     * @param url - A URL string pointing to Azure Storage file, such as
     *                     "https://myaccount.file.core.windows.net/myshare/mydirectory/file". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.file.core.windows.net/myshare/mydirectory/file?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a file.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a file or directory name includes %, file or directory name must be encoded in the URL.
     *                     Such as a file named "myfile%", the URL should be "https://myaccount.file.core.windows.net/myshare/mydirectory/myfile%25".
     * @param credential - Such as , StorageSharedKeyCredential or TokenCredential,
     *                                  If not specified, AnonymousCredential is used.
     * @param options - Optional. Options to configure the HTTP pipeline.
     */
    constructor(url: string, credential?: Credential | TokenCredential, options?: ShareClientOptions);
    /**
     * Creates an instance of ShareFileClient.
     *
     * @param url - A URL string pointing to Azure Storage file, such as
     *                     "https://myaccount.file.core.windows.net/myshare/mydirectory/file". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.file.core.windows.net/myshare/mydirectory/file?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a file.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a file or directory name includes %, file or directory name must be encoded in the URL.
     *                     Such as a file named "myfile%", the URL should be "https://myaccount.file.core.windows.net/myshare/mydirectory/myfile%25".
     * @param pipeline - Call newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     */
    constructor(url: string, pipeline: Pipeline, options?: ShareClientConfig);
    /**
     * Creates a new ShareFileClient object identical to the source but with the specified share snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base ShareFileClient.
     *
     * @param shareSnapshot - The share snapshot timestamp.
     * @returns A new ShareFileClient object identical to the source but with the specified share snapshot timestamp.
     */
    withShareSnapshot(shareSnapshot: string): ShareFileClient;
    /**
     * Creates a new file or replaces a file. Note it only initializes the file with no content.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-file
     *
     * @param size - Specifies the maximum size in bytes for the file, up to 4 TB.
     * @param options - Options to File Create operation.
     * @returns Response data for the File Create  operation.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleCreateFileAndUpload
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * const content = "Hello World!";
     * const fileName = `newdirectory${+new Date()}`;
     * const fileClient = directoryClient.getFileClient(fileName);
     * await fileClient.create(content.length);
     * console.log(`Create file ${fileName} successfully`);
     *
     * // Upload file range
     * await fileClient.uploadRange(content, 0, content.length);
     * console.log(`Upload file range "${content}" to ${fileName} successfully`);
     * ```
     */
    create(size: number, options?: FileCreateOptions): Promise<FileCreateResponse>;
    /**
     * Reads or downloads a file from the system, including its metadata and properties.
     *
     * * In Node.js, data returns in a Readable stream `readableStreamBody`
     * * In browsers, data returns in a promise `contentAsBlob`
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/get-file
     *
     * @param offset - From which position of the file to download, greater than or equal to 0
     * @param count - How much data to be downloaded, greater than 0. Will download to the end when undefined
     * @param options - Options to File Download operation.
     * @returns Response data for the File Download operation.
     *
     * Example usage (Node.js):
     *
     * ```ts snippet:ReadmeSampleDownloadFileAndConvertToString_Node
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const fileName = "<file name>";
     * const fileClient = serviceClient
     *   .getShareClient(shareName)
     *   .rootDirectoryClient.getFileClient(fileName);
     *
     * // Get file content from position 0 to the end
     * // In Node.js, get downloaded data by accessing downloadFileResponse.readableStreamBody
     * const downloadFileResponse = await fileClient.download();
     * if (downloadFileResponse.readableStreamBody) {
     *   const buffer = await streamToBuffer(downloadFileResponse.readableStreamBody);
     *   console.log(`Downloaded file content: ${buffer.toString()}`);
     * }
     *
     * // [Node.js only] A helper method used to read a Node.js readable stream into a Buffer
     * async function streamToBuffer(readableStream: NodeJS.ReadableStream): Promise<Buffer> {
     *   return new Promise((resolve, reject) => {
     *     const chunks: Buffer[] = [];
     *     readableStream.on("data", (data) => {
     *       chunks.push(data instanceof Buffer ? data : Buffer.from(data));
     *     });
     *     readableStream.on("end", () => {
     *       resolve(Buffer.concat(chunks));
     *     });
     *     readableStream.on("error", reject);
     *   });
     * }
     * ```
     *
     * Example usage (browsers):
     *
     * ```ts snippet:ReadmeSampleDownloadFileAndConvertToString_Browser
     * import { ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account name>";
     * const sas = "<service Shared Access Signature Token>";
     *
     * const serviceClient = new ShareServiceClient(`https://${account}.file.core.windows.net?${sas}`);
     *
     * const shareName = "<share name>";
     * const fileName = "<file name>";
     * const fileClient = serviceClient
     *   .getShareClient(shareName)
     *   .rootDirectoryClient.getFileClient(fileName);
     *
     * // Get file content from position 0 to the end
     * // In browsers, get downloaded data by accessing downloadFileResponse.blobBody
     * const downloadFileResponse = await fileClient.download(0);
     * if (downloadFileResponse.blobBody) {
     *   console.log(`Downloaded file content: ${(await downloadFileResponse.blobBody).text()}`);
     * }
     * ```
     */
    download(offset?: number, count?: number, options?: FileDownloadOptions): Promise<FileDownloadResponseModel>;
    /**
     * Returns true if the specified file exists; false otherwise.
     *
     * NOTE: use this function with care since an existing file might be deleted by other clients or
     * applications. Vice versa new files might be added by other clients or applications after this
     * function completes.
     *
     * @param options - options to Exists operation.
     */
    exists(options?: FileExistsOptions): Promise<boolean>;
    /**
     * Returns all user-defined metadata, standard HTTP properties, and system properties
     * for the file. It does not return the content of the file.
     * @see https://learn.microsoft.com/rest/api/storageservices/get-file-properties
     *
     * @param options - Options to File Get Properties operation.
     * @returns Response data for the File Get Properties operation.
     */
    getProperties(options?: FileGetPropertiesOptions): Promise<FileGetPropertiesResponse>;
    /**
     * Sets properties on the file.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-file-properties
     *
     * @param properties - File properties. For file HTTP headers(e.g. Content-Type),
     *                                       if no values are provided, existing HTTP headers will be removed.
     *                                       For other file properties(e.g. fileAttributes), if no values are provided,
     *                                       existing values will be preserved.
     */
    setProperties(properties?: FileProperties): Promise<SetPropertiesResponse>;
    /**
     * Removes the file from the storage account.
     * When a file is successfully deleted, it is immediately removed from the storage
     * account's index and is no longer accessible to clients. The file's data is later
     * removed from the service during garbage collection.
     *
     * Delete File will fail with status code 409 (Conflict) and error code SharingViolation
     * if the file is open on an SMB client.
     *
     * Delete File is not supported on a share snapshot, which is a read-only copy of
     * a share. An attempt to perform this operation on a share snapshot will fail with 400 (InvalidQueryParameterValue)
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-file2
     *
     * @param options - Options to File Delete operation.
     * @returns Response data for the File Delete operation.
     */
    delete(options?: FileDeleteOptions): Promise<FileDeleteResponse>;
    /**
     * Removes the file from the storage account if it exists.
     * When a file is successfully deleted, it is immediately removed from the storage
     * account's index and is no longer accessible to clients. The file's data is later
     * removed from the service during garbage collection.
     *
     * Delete File will fail with status code 409 (Conflict) and error code SharingViolation
     * if the file is open on an SMB client.
     *
     * Delete File is not supported on a share snapshot, which is a read-only copy of
     * a share. An attempt to perform this operation on a share snapshot will fail with 400 (InvalidQueryParameterValue)
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-file2
     *
     * @param options -
     */
    deleteIfExists(options?: FileDeleteOptions): Promise<FileDeleteIfExistsResponse>;
    /**
     * Sets HTTP headers on the file.
     *
     * If no option provided, or no value provided for the file HTTP headers in the options,
     * these file HTTP headers without a value will be cleared.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-file-properties
     *
     * @param FileHttpHeaders - File HTTP headers like Content-Type.
     *                                             Provide undefined will remove existing HTTP headers.
     * @param options - Options to File Set HTTP Headers operation.
     * @returns Response data for the File Set HTTP Headers operation.
     */
    setHttpHeaders(fileHttpHeaders?: FileHttpHeaders, options?: FileSetHttpHeadersOptions): Promise<FileSetHTTPHeadersResponse>;
    /**
     * Resize file.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/set-file-properties
     *
     * @param length - Resizes a file to the specified size in bytes.
     *                        If the specified byte value is less than the current size of the file,
     *                        then all ranges above the specified byte value are cleared.
     * @param options - Options to File Resize operation.
     * @returns Response data for the File Set HTTP Headers operation.
     */
    resize(length: number, options?: FileResizeOptions): Promise<FileSetHTTPHeadersResponse>;
    /**
     * Updates user-defined metadata for the specified file.
     *
     * If no metadata defined in the option parameter, the file
     * metadata will be removed.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-file-metadata
     *
     * @param metadata - If no metadata provided, all existing directory metadata will be removed
     * @param options - Options to File Set Metadata operation.
     * @returns Response data for the File Set Metadata operation.
     */
    setMetadata(metadata?: Metadata, options?: FileSetMetadataOptions): Promise<FileSetMetadataResponse>;
    /**
     * Upload a range of bytes to a file. This operation can only be called on an existing file.
     * It won't change the size, properties or metadata of the file.
     * Both the start and count of the range must be specified. The range can be up to 4 MB in size.
     *
     * @param body - Blob, string, ArrayBuffer, ArrayBufferView or a function
     *                               which returns a new Readable stream whose offset is from data source beginning.
     * @param offset - Offset position of the destination Azure File to upload.
     * @param contentLength - Length of body in bytes. Use Buffer.byteLength() to calculate body length for a
     *                               string including non non-Base64/Hex-encoded characters.
     * @param options - Options to File Upload Range operation.
     * @returns Response data for the File Upload Range operation.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleCreateFileAndUpload
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * const content = "Hello World!";
     * const fileName = `newdirectory${+new Date()}`;
     * const fileClient = directoryClient.getFileClient(fileName);
     * await fileClient.create(content.length);
     * console.log(`Create file ${fileName} successfully`);
     *
     * // Upload file range
     * await fileClient.uploadRange(content, 0, content.length);
     * console.log(`Upload file range "${content}" to ${fileName} successfully`);
     * ```
     */
    uploadRange(body: HttpRequestBody, offset: number, contentLength: number, options?: FileUploadRangeOptions): Promise<FileUploadRangeResponse>;
    /**
     * Upload a range of bytes to a file where the contents are read from a another file's URL.
     * The range can be up to 4 MB in size.
     *
     * @param sourceURL - Specify a URL to the copy source, Shared Access Signature(SAS) maybe needed for authentication.
     * @param sourceOffset - The source offset to copy from. Pass 0 to copy from the beginning of source file.
     * @param destOffset - Offset of destination file.
     * @param count - Number of bytes to be uploaded from source file.
     * @param options - Options to configure File - Upload Range from URL operation.
     */
    uploadRangeFromURL(sourceURL: string, sourceOffset: number, destOffset: number, count: number, options?: FileUploadRangeFromURLOptions): Promise<FileUploadRangeFromURLResponse>;
    /**
     * Clears the specified range and
     * releases the space used in storage for that range.
     *
     * @param offset -
     * @param contentLength -
     * @param options - Options to File Clear Range operation.
     */
    clearRange(offset: number, contentLength: number, options?: FileClearRangeOptions): Promise<FileUploadRangeResponse>;
    /**
     * Returns the list of valid ranges for a file.
     *
     * @param options - Options to File Get range List operation.
     */
    getRangeList(options?: FileGetRangeListOptions): Promise<FileGetRangeListResponse>;
    /**
     * Returns the list of ranges that differ between a previous share snapshot and this file.
     *
     * @param prevShareSnapshot - The previous snapshot parameter is an opaque DateTime value that specifies the previous share snapshot to compare with.
     * @param options -
     */
    getRangeListDiff(prevShareSnapshot: string, options?: FileGetRangeListOptions): Promise<FileGetRangeListDiffResponse>;
    /**
     * Copies a blob or file to a destination file within the storage account.
     *
     * @param copySource - Specifies the URL of the source file or blob, up to 2 KB in length.
     * To copy a file to another file within the same storage account, you may use Shared Key to
     * authenticate the source file. If you are copying a file from another storage account, or if you
     * are copying a blob from the same storage account or another storage account, then you must
     * authenticate the source file or blob using a shared access signature. If the source is a public
     * blob, no authentication is required to perform the copy operation. A file in a share snapshot
     * can also be specified as a copy source.
     * @param options - Options to File Start Copy operation.
     */
    startCopyFromURL(copySource: string, options?: FileStartCopyOptions): Promise<FileStartCopyResponse>;
    /**
     * Aborts a pending Copy File operation, and leaves a destination file with zero length and full
     * metadata.
     * @see https://learn.microsoft.com/rest/api/storageservices/abort-copy-file
     *
     * @param copyId - Id of the Copy File operation to abort.
     * @param options - Options to File Abort Copy From URL operation.
     */
    abortCopyFromURL(copyId: string, options?: FileAbortCopyFromURLOptions): Promise<FileAbortCopyResponse>;
    /**
     * Creates a new Azure File or replaces an existing Azure File, and then uploads a Buffer(Node)/Blob/ArrayBuffer/ArrayBufferView to it.
     *
     * @param data - Buffer(Node), Blob, ArrayBuffer or ArrayBufferView
     * @param options -
     */
    uploadData(data: Buffer | Blob | ArrayBuffer | ArrayBufferView, options?: FileParallelUploadOptions): Promise<void>;
    /**
     * ONLY AVAILABLE IN BROWSERS.
     *
     * Uploads a browser Blob object to an Azure file. Requires a blobFactory as the data source,
     * which need to return a Blob object with the offset and size provided.
     *
     * @param blobFactory -
     * @param size -
     * @param options -
     */
    uploadSeekableBlob(blobFactory: (offset: number, size: number) => Blob, size: number, options?: FileParallelUploadOptions): Promise<void>;
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Creates a new Azure File or replaces an existing Azure File, and then uploads a local file to it.
     *
     * @param filePath - Full path of local file
     * @param fileClient - ShareFileClient
     * @param options -
     */
    uploadFile(filePath: string, options?: FileParallelUploadOptions): Promise<void>;
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Accepts a Node.js Readable stream factory, and uploads in blocks to an Azure File.
     * The Readable stream factory must returns a Node.js Readable stream starting from the offset defined. The offset
     * is the offset in the Azure file to be uploaded.
     *
     * @param streamFactory - Returns a Node.js Readable stream starting
     *                                                                  from the offset defined
     * @param size - Size of the Azure file
     * @param fileClient - ShareFileClient
     * @param options -
     */
    uploadResetableStream(streamFactory: (offset: number, count?: number) => NodeJS.ReadableStream, size: number, options?: FileParallelUploadOptions): Promise<void>;
    /**
     *
     * @param bodyFactory -
     * @param size - Size of the Azure file
     * @param fileClient - ShareFileClient
     * @param options -
     */
    private uploadSeekableInternal;
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Downloads an Azure file in parallel to a buffer.
     * Offset and count are optional, pass 0 for both to download the entire file.
     *
     * Warning: Buffers can only support files up to about one gigabyte on 32-bit systems or about two
     * gigabytes on 64-bit systems due to limitations of Node.js/V8. For files larger than this size,
     * consider {@link downloadToFile}.
     *
     * @param buffer - Buffer to be fill, must have length larger than count
     * @param offset - From which position of the Azure File to download
     * @param count - How much data to be downloaded. Will download to the end when passing undefined
     * @param options -
     */
    downloadToBuffer(buffer: Buffer, offset?: number, count?: number, options?: FileDownloadToBufferOptions): Promise<Buffer>;
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME
     *
     * Downloads an Azure file in parallel to a buffer.
     * Offset and count are optional, pass 0 for both to download the entire file
     *
     * Warning: Buffers can only support files up to about one gigabyte on 32-bit systems or about two
     * gigabytes on 64-bit systems due to limitations of Node.js/V8. For files larger than this size,
     * consider {@link downloadToFile}.
     *
     * @param offset - From which position of the Azure file to download
     * @param count - How much data to be downloaded. Will download to the end when passing undefined
     * @param options -
     */
    downloadToBuffer(offset?: number, count?: number, options?: FileDownloadToBufferOptions): Promise<Buffer>;
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Creates a new Azure File or replaces an existing Azure File, and then uploads a Node.js Readable stream into it.
     * This method will try to create an Azure File, then starts uploading chunk by chunk.
     * Size of chunk is defined by `bufferSize` parameter.
     * Please make sure potential size of stream doesn't exceed file size.
     *
     * PERFORMANCE IMPROVEMENT TIPS:
     * * Input stream highWaterMark is better to set a same value with bufferSize
     *   parameter, which will avoid Buffer.concat() operations.
     *
     * @param stream - Node.js Readable stream. Must be less or equal than file size.
     * @param size - Size of file to be created. Maximum size allowed is 4 TB.
     *                      If this value is larger than stream size, there will be empty bytes in file tail.
     * @param bufferSize - Size of every buffer allocated in bytes, also the chunk/range size during
     *                            the uploaded file. Size must be greater than 0 and lesser than or equal to 4 * 1024 * 1024 (4MB)
     * @param maxBuffers - Max buffers will allocate during uploading, positive correlation
     *                            with max uploading concurrency
     * @param options -
     */
    uploadStream(stream: Readable, size: number, bufferSize: number, maxBuffers: number, options?: FileUploadStreamOptions): Promise<void>;
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Downloads an Azure Blob to a local file.
     * Fails if the the given file path already exits.
     * Offset and count are optional, pass 0 and undefined respectively to download the entire blob.
     *
     * @param filePath -
     * @param offset - From which position of the block blob to download.
     * @param count - How much data to be downloaded. Will download to the end when passing undefined.
     * @param options - Options to Blob download options.
     * @returns The response data for blob download operation,
     *                                                 but with readableStreamBody set to undefined since its
     *                                                 content is already read and written into a local file
     *                                                 at the specified path.
     */
    downloadToFile(filePath: string, offset?: number, count?: number, options?: FileDownloadOptions): Promise<FileDownloadResponseModel>;
    /**
     * Lists handles for a file.
     * @see https://learn.microsoft.com/rest/api/storageservices/list-handles
     *
     * @param marker - Optional. A string value that identifies the portion of the list to be
     *                          returned with the next list handles operation. The operation returns a
     *                          marker value within the response body if the list returned was not complete.
     *                          The marker value may then be used in a subsequent call to request the next
     *                          set of list items.
     * @param options -
     */
    private listHandlesSegment;
    /**
     * Returns an AsyncIterableIterator for FileListHandlesResponse
     *
     * @param marker - A string value that identifies the portion of the list to be
     *                          returned with the next list handles operation. The operation returns a
     *                          marker value within the response body if the list returned was not complete.
     *                          The marker value may then be used in a subsequent call to request the next
     *                          set of list items.
     * @param options - Options to list handles operation.
     */
    private iterateHandleSegments;
    /**
     * Returns an AsyncIterableIterator for handles
     *
     * @param options - Options to list handles operation.
     */
    private listHandleItems;
    /**
     * Returns an async iterable iterator to list all the handles.
     * under the specified account.
     *
     * .byPage() returns an async iterable iterator to list the handles in pages.
     *
     * @param options - Options to list handles operation.
     *
     * An asyncIterableIterator that supports paging.
     */
    listHandles(options?: FileListHandlesOptions): PagedAsyncIterableIterator<HandleItem, FileListHandlesResponse>;
    /**
     * Force close all handles for a file.
     * @see https://learn.microsoft.com/rest/api/storageservices/force-close-handles
     *
     * @param marker - Optional. A string value that identifies the position of handles that will
     *                          be closed with the next force close handles operation.
     *                          The operation returns a marker value within the response
     *                          body if there are more handles to close. The marker value
     *                          may then be used in a subsequent call to close the next set of handles.
     * @param options - Options to force close handles operation.
     */
    private forceCloseHandlesSegment;
    /**
     * Force close all handles for a file.
     * @see https://learn.microsoft.com/rest/api/storageservices/force-close-handles
     *
     * @param options - Options to force close handles operation.
     */
    forceCloseAllHandles(options?: FileForceCloseHandlesOptions): Promise<CloseHandlesInfo>;
    /**
     * Force close a specific handle for a file.
     * @see https://learn.microsoft.com/rest/api/storageservices/force-close-handles
     *
     * @param handleId - Specific handle ID, cannot be asterisk "*".
     *                          Use forceCloseAllHandles() to close all handles.
     * @param FileForceCloseHandlesOptions - Options to force close handles operation.
     */
    forceCloseHandle(handleId: string, options?: FileForceCloseHandlesOptions): Promise<FileForceCloseHandlesResponse>;
    /**
     * NFS only.  Creates a hard link to the file file specified by path.
     * @param targetFile - Path of the file to create the hard link to, not including the share.
     *  For example: "targetDirectory/targetSubDirectory/.../targetFile"
     * @param options - Options to create hard link operation.
     */
    createHardLink(targetFile: string, options?: FileCreateHardLinkOptions): Promise<FileCreateHardLinkResponse>;
    /**
     * NFS only.  Creates a symbolic link.
     * @param linkText - The path to the original file, the symbolic link is pointing to.
     *                  The path is of type string which is not resolved and is stored as is. The path can be absolute path
     *                 or the relative path depending on the content stored in the symbolic link file.
     * @param options - Options to create hard link operation.
     */
    createSymbolicLink(linkText: string, options?: FileCreateSymbolicLinkOptions): Promise<FileCreateSymbolicLinkResponse>;
    /**
     * NFS only.  Gets content of a symbolic link.
     * @param options - Options to get symbolic link operation.
     */
    getSymbolicLink(options?: FileGetSymbolicLinkOptions): Promise<FileGetSymbolicLinkResponse>;
    /**
     * Get a {@link ShareLeaseClient} that manages leases on the file.
     *
     * @param proposeLeaseId - Initial proposed lease Id.
     * @returns A new ShareLeaseClient object for managing leases on the file.
     */
    getShareLeaseClient(proposeLeaseId?: string): ShareLeaseClient;
    /**
     * Only available for clients constructed with a shared key credential.
     *
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasUrl(options: FileGenerateSasUrlOptions): string;
    /**
     * Only available for clients constructed with a shared key credential.
     *
     * Generates string to sign for a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasStringToSign(options: FileGenerateSasUrlOptions): string;
    /**
     *
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the user delegation key credential input.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @param userDelegationKey - user delegation key used to sign the SAS URI
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateUserDelegationSasUrl(options: ShareGenerateSasUrlOptions, userDelegationKey: UserDelegationKey): string;
    /**
     *
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the user delegation key credential input.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @param userDelegationKey - user delegation key used to sign the SAS URI
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateUserDelegationStringToSign(options: ShareGenerateSasUrlOptions, userDelegationKey: UserDelegationKey): string;
    /**
     * Renames a file.
     * This API only supports renaming a file in the same share.
     *
     * @param destinationPath - Specifies the destination path to rename to. The path will be encoded to put into a URL to specify the destination.
     * @param options - Options for the renaming operation.
     * @returns Response data for the file renaming operation.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleRenameFile
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const fileName = "<file name>";
     * const destinationPath = "<destination path>";
     * const fileClient = serviceClient
     *   .getShareClient(shareName)
     *   .getDirectoryClient(directoryName)
     *   .getFileClient(fileName);
     *
     * await fileClient.rename(destinationPath);
     * ```
     */
    rename(destinationPath: string, options?: FileRenameOptions): Promise<{
        destinationFileClient: ShareFileClient;
        fileRenameResponse: FileRenameResponse;
    }>;
}
/**
 * The details of the response for a specific lease operation.
 */
export interface LeaseOperationResponseHeaders {
    /**
     * The ETag contains a value that you can use to perform operations conditionally. If the request
     * version is 2011-08-18 or newer, the ETag value will be in quotes.
     */
    etag?: string;
    /**
     * Returns the date and time the file was last modified. Any operation that modifies the file,
     * including an update of the file's metadata or properties, changes the last-modified time of
     * the file.
     */
    lastModified?: Date;
    /**
     * Approximate time remaining in the lease period, in seconds. Only availabe for {@link ShareLeaseClient.breakLease} for share lease.
     */
    leaseTimeInSeconds?: number;
    /**
     * Uniquely identifies a file's lease, won't be set when returned by releaseLease.
     */
    leaseId?: string;
    /**
     * This header uniquely identifies the request that was made and can be used for troubleshooting
     * the request.
     */
    requestId?: string;
    /**
     * Indicates the version of the Blob service used to execute the request. This header is returned
     * for requests made against version 2009-09-19 and above.
     */
    version?: string;
    /**
     * UTC date/time value generated by the service that indicates the time at which the response was
     * initiated
     */
    date?: Date;
    errorCode?: string;
}
/**
 * Contains the response data for operations that acquire, change, break or release a lease.
 *
 * See {@link ShareLeaseClient}.
 */
export type LeaseOperationResponse = WithResponse<LeaseOperationResponseHeaders, LeaseOperationResponseHeaders>;
/**
 * lease operations options.
 */
export interface LeaseOperationOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * A client that manages leases for a {@link ShareFileClient} or {@link ShareClient}.
 * @see https://learn.microsoft.com/rest/api/storageservices/lease-file
 * and
 * @see https://learn.microsoft.com/rest/api/storageservices/lease-share
 */
export declare class ShareLeaseClient {
    private _leaseId;
    private _url;
    private fileOrShare;
    private shareClientConfig?;
    /**
     * Gets the lease Id.
     *
     * @readonly
     */
    get leaseId(): string;
    /**
     * Gets the url.
     *
     * @readonly
     */
    get url(): string;
    /**
     * Creates an instance of ShareLeaseClient.
     * @param client - The client to make the lease operation requests.
     * @param leaseId - Initial proposed lease id.
     */
    constructor(client: ShareFileClient | ShareClient, leaseId?: string);
    /**
     * Establishes and manages a lock on a file, share or share snapshot for write and delete operations.
     *
     * @param duration - Specifies the duration of lease in seconds. For file, the only allowed value is -1 for a lease that never expires. For share, must be -1 or between 15 to 60.
     * @param options - Options for the lease management operation.
     * @returns Response data for acquire lease operation.
     */
    acquireLease(duration?: number, options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
    /**
     * To change the ID of an existing lease.
     *
     * @param proposedLeaseId - the proposed new lease Id.
     * @param options - Options for the lease management operation.
     * @returns Response data for change lease operation.
     */
    changeLease(proposedLeaseId: string, options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
    /**
     * To free the lease if it is no longer needed so that another client may
     * immediately acquire a lease.
     *
     * @param options - Options for the lease management operation.
     * @returns Response data for release lease operation.
     */
    releaseLease(options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
    /**
     * To force end the lease.
     *
     * @param options - Options for the lease management operation.
     * @returns Response data for break lease operation.
     */
    breakLease(options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
    /**
     * To renew the lease. Only available for lease on share or share snapshot.
     * Note that the lease may be renewed even if it has expired as long as the share has not been leased again since the expiration of that lease.
     * When you renew a lease, the lease duration clock resets.
     *
     * @param options - Options for the lease management operation.
     * @returns Response data for renew lease operation.
     */
    renewLease(options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
}
//# sourceMappingURL=Clients.d.ts.map