"use strict";
// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateFileSASQueryParameters = generateFileSASQueryParameters;
exports.generateFileSASQueryParametersInternal = generateFileSASQueryParametersInternal;
exports.generateFileSASQueryParametersDefault = generateFileSASQueryParametersDefault;
exports.generateFileSASQueryParametersUDK20250705 = generateFileSASQueryParametersUDK20250705;
const storage_common_1 = require("@azure/storage-common");
const FileSASPermissions_js_1 = require("./FileSASPermissions.js");
const SasIPRange_js_1 = require("./SasIPRange.js");
const SASQueryParameters_js_1 = require("./SASQueryParameters.js");
const ShareSASPermissions_js_1 = require("./ShareSASPermissions.js");
const constants_js_1 = require("./utils/constants.js");
const utils_common_js_1 = require("./utils/utils.common.js");
const storage_common_2 = require("@azure/storage-common");
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Creates an instance of SASQueryParameters.
 *
 * Only accepts required settings needed to create a SAS. For optional settings please
 * set corresponding properties directly, such as permissions, startsOn and identifier.
 *
 * WARNING: When identifier is not provided, permissions and expiresOn are required.
 * You MUST assign value to identifier or expiresOn & permissions manually if you initial with
 * this constructor.
 *
 * @param fileSASSignatureValues -
 * @param sharedKeyCredential -
 */
function generateFileSASQueryParameters(fileSASSignatureValues, sharedKeyCredentialOrUserDelegationKey, accountName) {
    return generateFileSASQueryParametersInternal(fileSASSignatureValues, sharedKeyCredentialOrUserDelegationKey, accountName).sasQueryParameters;
}
function generateFileSASQueryParametersInternal(fileSASSignatureValues, sharedKeyCredentialOrUserDelegationKey, accountName) {
    const version = fileSASSignatureValues.version ? fileSASSignatureValues.version : constants_js_1.SERVICE_VERSION;
    const sharedKeyCredential = sharedKeyCredentialOrUserDelegationKey instanceof storage_common_1.StorageSharedKeyCredential
        ? sharedKeyCredentialOrUserDelegationKey
        : undefined;
    let userDelegationKeyCredential;
    if (sharedKeyCredential === undefined && accountName !== undefined) {
        userDelegationKeyCredential = new storage_common_2.UserDelegationKeyCredential(accountName, sharedKeyCredentialOrUserDelegationKey);
    }
    if (sharedKeyCredential !== undefined) {
        return generateFileSASQueryParametersDefault(fileSASSignatureValues, sharedKeyCredential);
    }
    else {
        if (version >= "2025-07-05") {
            return generateFileSASQueryParametersUDK20250705(fileSASSignatureValues, userDelegationKeyCredential, accountName);
        }
        else {
            throw new RangeError("'version' must be >= '2025-07-05' when generating user delegation SAS using user delegation key.");
        }
    }
}
function generateFileSASQueryParametersDefault(fileSASSignatureValues, sharedKeyCredential) {
    if (!fileSASSignatureValues.identifier &&
        !(fileSASSignatureValues.permissions && fileSASSignatureValues.expiresOn)) {
        throw new RangeError("Must provide 'permissions' and 'expiresOn' for File SAS generation when 'identifier' is not provided.");
    }
    const version = fileSASSignatureValues.version ? fileSASSignatureValues.version : constants_js_1.SERVICE_VERSION;
    let resource = "s";
    if (fileSASSignatureValues.filePath) {
        resource = "f";
    }
    let verifiedPermissions;
    // Calling parse and toString guarantees the proper ordering and throws on invalid characters.
    if (fileSASSignatureValues.permissions) {
        if (fileSASSignatureValues.filePath) {
            verifiedPermissions = FileSASPermissions_js_1.FileSASPermissions.parse(fileSASSignatureValues.permissions.toString()).toString();
        }
        else {
            verifiedPermissions = ShareSASPermissions_js_1.ShareSASPermissions.parse(fileSASSignatureValues.permissions.toString()).toString();
        }
    }
    // Signature is generated on the un-url-encoded values.
    const stringToSign = [
        verifiedPermissions,
        fileSASSignatureValues.startsOn
            ? (0, utils_common_js_1.truncatedISO8061Date)(fileSASSignatureValues.startsOn, false)
            : "",
        fileSASSignatureValues.expiresOn
            ? (0, utils_common_js_1.truncatedISO8061Date)(fileSASSignatureValues.expiresOn, false)
            : "",
        getCanonicalName(sharedKeyCredential.accountName, fileSASSignatureValues.shareName, fileSASSignatureValues.filePath),
        fileSASSignatureValues.identifier,
        fileSASSignatureValues.ipRange ? (0, SasIPRange_js_1.ipRangeToString)(fileSASSignatureValues.ipRange) : "",
        fileSASSignatureValues.protocol,
        version,
        fileSASSignatureValues.cacheControl,
        fileSASSignatureValues.contentDisposition,
        fileSASSignatureValues.contentEncoding,
        fileSASSignatureValues.contentLanguage,
        fileSASSignatureValues.contentType,
    ].join("\n");
    const signature = sharedKeyCredential.computeHMACSHA256(stringToSign);
    return {
        sasQueryParameters: new SASQueryParameters_js_1.SASQueryParameters(version, signature, verifiedPermissions, undefined, undefined, fileSASSignatureValues.protocol, fileSASSignatureValues.startsOn, fileSASSignatureValues.expiresOn, fileSASSignatureValues.ipRange, fileSASSignatureValues.identifier, resource, fileSASSignatureValues.cacheControl, fileSASSignatureValues.contentDisposition, fileSASSignatureValues.contentEncoding, fileSASSignatureValues.contentLanguage, fileSASSignatureValues.contentType),
        stringToSign: stringToSign,
    };
}
function generateFileSASQueryParametersUDK20250705(fileSASSignatureValues, userDelegationKeyCredential, accountName) {
    if (!(fileSASSignatureValues.permissions && fileSASSignatureValues.expiresOn)) {
        throw new RangeError("Must provide 'permissions' and 'expiresOn' for File SAS generation when generating user delegation SAS.");
    }
    const version = fileSASSignatureValues.version ? fileSASSignatureValues.version : constants_js_1.SERVICE_VERSION;
    let resource = "s";
    if (fileSASSignatureValues.filePath) {
        resource = "f";
    }
    let verifiedPermissions;
    // Calling parse and toString guarantees the proper ordering and throws on invalid characters.
    if (fileSASSignatureValues.permissions) {
        if (fileSASSignatureValues.filePath) {
            verifiedPermissions = FileSASPermissions_js_1.FileSASPermissions.parse(fileSASSignatureValues.permissions.toString()).toString();
        }
        else {
            verifiedPermissions = ShareSASPermissions_js_1.ShareSASPermissions.parse(fileSASSignatureValues.permissions.toString()).toString();
        }
    }
    // Signature is generated on the un-url-encoded values.
    const stringToSign = [
        verifiedPermissions,
        fileSASSignatureValues.startsOn
            ? (0, utils_common_js_1.truncatedISO8061Date)(fileSASSignatureValues.startsOn, false)
            : "",
        fileSASSignatureValues.expiresOn
            ? (0, utils_common_js_1.truncatedISO8061Date)(fileSASSignatureValues.expiresOn, false)
            : "",
        getCanonicalName(accountName, fileSASSignatureValues.shareName, fileSASSignatureValues.filePath),
        userDelegationKeyCredential.userDelegationKey.signedObjectId,
        userDelegationKeyCredential.userDelegationKey.signedTenantId,
        userDelegationKeyCredential.userDelegationKey.signedStartsOn
            ? (0, utils_common_js_1.truncatedISO8061Date)(userDelegationKeyCredential.userDelegationKey.signedStartsOn, false)
            : "",
        userDelegationKeyCredential.userDelegationKey.signedExpiresOn
            ? (0, utils_common_js_1.truncatedISO8061Date)(userDelegationKeyCredential.userDelegationKey.signedExpiresOn, false)
            : "",
        userDelegationKeyCredential.userDelegationKey.signedService,
        userDelegationKeyCredential.userDelegationKey.signedVersion,
        undefined, // shared key delegation signed tenant id.
        fileSASSignatureValues.delegatedUserObjectId,
        fileSASSignatureValues.ipRange ? (0, SasIPRange_js_1.ipRangeToString)(fileSASSignatureValues.ipRange) : "",
        fileSASSignatureValues.protocol,
        version,
        fileSASSignatureValues.cacheControl,
        fileSASSignatureValues.contentDisposition,
        fileSASSignatureValues.contentEncoding,
        fileSASSignatureValues.contentLanguage,
        fileSASSignatureValues.contentType,
    ].join("\n");
    const signature = userDelegationKeyCredential.computeHMACSHA256(stringToSign);
    return {
        sasQueryParameters: new SASQueryParameters_js_1.SASQueryParameters(version, signature, verifiedPermissions, undefined, undefined, fileSASSignatureValues.protocol, fileSASSignatureValues.startsOn, fileSASSignatureValues.expiresOn, fileSASSignatureValues.ipRange, fileSASSignatureValues.identifier, resource, fileSASSignatureValues.cacheControl, fileSASSignatureValues.contentDisposition, fileSASSignatureValues.contentEncoding, fileSASSignatureValues.contentLanguage, fileSASSignatureValues.contentType, userDelegationKeyCredential.userDelegationKey, fileSASSignatureValues.delegatedUserObjectId),
        stringToSign: stringToSign,
    };
}
function getCanonicalName(accountName, shareName, filePath) {
    // Share: "/file/account/sharename"
    // File:  "/file/account/sharename/filename"
    // File:  "/file/account/sharename/directoryname/filename"
    const elements = [`/file/${accountName}/${shareName}`];
    if (filePath) {
        elements.push(`/${filePath}`);
    }
    return elements.join("");
}
//# sourceMappingURL=FileSASSignatureValues.js.map