"use strict";
// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.ShareServiceClient = void 0;
const Pipeline_js_1 = require("./Pipeline.js");
const StorageClient_js_1 = require("./StorageClient.js");
const ShareClientInternal_js_1 = require("./ShareClientInternal.js");
const Clients_js_1 = require("./Clients.js");
const utils_common_js_1 = require("./utils/utils.common.js");
const storage_common_1 = require("@azure/storage-common");
const storage_common_2 = require("@azure/storage-common");
const storage_common_3 = require("@azure/storage-common");
const core_util_1 = require("@azure/core-util");
const tracing_js_1 = require("./utils/tracing.js");
const models_js_1 = require("./models.js");
const AccountSASPermissions_js_1 = require("./AccountSASPermissions.js");
const AccountSASSignatureValues_js_1 = require("./AccountSASSignatureValues.js");
const AccountSASServices_js_1 = require("./AccountSASServices.js");
const utils_common_js_2 = require("./utils/utils.common.js");
const core_auth_1 = require("@azure/core-auth");
/**
 * A ShareServiceClient represents a URL to the Azure Storage File service allowing you
 * to manipulate file shares.
 */
class ShareServiceClient extends StorageClient_js_1.StorageClient {
    /**
     * serviceContext provided by protocol layer.
     */
    serviceContext;
    shareClientConfig;
    /**
     *
     * Creates an instance of ShareServiceClient from connection string.
     *
     * @param connectionString - Account connection string or a SAS connection string of an Azure storage account.
     *                                  [ Note - Account connection string can only be used in NODE.JS runtime. ]
     *                                  Account connection string example -
     *                                  `DefaultEndpointsProtocol=https;AccountName=myaccount;AccountKey=accountKey;EndpointSuffix=core.windows.net`
     *                                  SAS connection string example -
     *                                  `BlobEndpoint=https://myaccount.blob.core.windows.net/;QueueEndpoint=https://myaccount.queue.core.windows.net/;FileEndpoint=https://myaccount.file.core.windows.net/;TableEndpoint=https://myaccount.table.core.windows.net/;SharedAccessSignature=sasString`
     * @param options - Options to configure the HTTP pipeline.
     * @returns A new ShareServiceClient from the given connection string.
     */
    static fromConnectionString(connectionString, 
    // Legacy, no way to fix the eslint error without breaking. Disable the rule for this line.
    /* eslint-disable-next-line @azure/azure-sdk/ts-naming-options */
    options) {
        const extractedCreds = (0, utils_common_js_1.extractConnectionStringParts)(connectionString);
        if (extractedCreds.kind === "AccountConnString") {
            if (core_util_1.isNodeLike) {
                const sharedKeyCredential = new storage_common_2.StorageSharedKeyCredential(extractedCreds.accountName, extractedCreds.accountKey);
                const pipeline = (0, Pipeline_js_1.newPipeline)(sharedKeyCredential, options);
                return new ShareServiceClient(extractedCreds.url, pipeline, options);
            }
            else {
                throw new Error("Account connection string is only supported in Node.js environment");
            }
        }
        else if (extractedCreds.kind === "SASConnString") {
            const pipeline = (0, Pipeline_js_1.newPipeline)(new storage_common_3.AnonymousCredential(), options);
            return new ShareServiceClient(extractedCreds.url + "?" + extractedCreds.accountSas, pipeline, options);
        }
        else {
            throw new Error("Connection string must be either an Account connection string or a SAS connection string");
        }
    }
    constructor(url, credentialOrPipeline, 
    // Legacy, no way to fix the eslint error without breaking. Disable the rule for this line.
    /* eslint-disable-next-line @azure/azure-sdk/ts-naming-options */
    options) {
        let pipeline;
        if ((0, Pipeline_js_1.isPipelineLike)(credentialOrPipeline)) {
            pipeline = credentialOrPipeline;
        }
        else if (credentialOrPipeline instanceof storage_common_1.Credential ||
            (0, core_auth_1.isTokenCredential)(credentialOrPipeline)) {
            pipeline = (0, Pipeline_js_1.newPipeline)(credentialOrPipeline, options);
        }
        else {
            // The second parameter is undefined. Use anonymous credential.
            pipeline = (0, Pipeline_js_1.newPipeline)(new storage_common_3.AnonymousCredential(), options);
        }
        super(url, pipeline);
        this.shareClientConfig = options;
        this.serviceContext = this.storageClientContext.service;
    }
    /**
     * Creates a ShareClient object.
     *
     * @param shareName - Name of a share.
     * @returns The ShareClient object for the given share name.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleCreateShareClient
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const shareClient = serviceClient.getShareClient(shareName);
     * await shareClient.create();
     * ```
     */
    getShareClient(shareName) {
        return new Clients_js_1.ShareClient((0, utils_common_js_1.appendToURLPath)(this.url, shareName), this.pipeline, this.shareClientConfig);
    }
    /**
     * Creates a Share.
     *
     * @param shareName -
     * @param options -
     * @returns Share creation response and the corresponding share client.
     */
    async createShare(shareName, 
    // eslint-disable-next-line @azure/azure-sdk/ts-naming-options
    options = {}) {
        return tracing_js_1.tracingClient.withSpan("ShareServiceClient-createShare", options, async (updatedOptions) => {
            const shareClient = this.getShareClient(shareName);
            const shareCreateResponse = await shareClient.create(updatedOptions);
            return {
                shareCreateResponse,
                shareClient,
            };
        });
    }
    /**
     * Deletes a Share.
     *
     * @param shareName -
     * @param options -
     * @returns Share deletion response and the corresponding share client.
     */
    async deleteShare(shareName, 
    // eslint-disable-next-line @azure/azure-sdk/ts-naming-options
    options = {}) {
        return tracing_js_1.tracingClient.withSpan("ShareServiceClient-deleteShare", options, async (updatedOptions) => {
            const shareClient = this.getShareClient(shareName);
            return shareClient.delete(updatedOptions);
        });
    }
    /**
     * Gets the properties of a storage account’s file service, including properties
     * for Storage Analytics and CORS (Cross-Origin Resource Sharing) rules.
     * @see https://learn.microsoft.com/rest/api/storageservices/get-file-service-properties
     *
     * @param options - Options to Get Properties operation.
     * @returns Response data for the Get Properties operation.
     */
    async getProperties(options = {}) {
        return tracing_js_1.tracingClient.withSpan("ShareServiceClient-getProperties", options, async (updatedOptions) => {
            return (0, utils_common_js_1.assertResponse)(await this.serviceContext.getProperties({
                ...updatedOptions,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * Sets properties for a storage account’s file service endpoint, including properties
     * for Storage Analytics, CORS (Cross-Origin Resource Sharing) rules and soft delete settings.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-file-service-properties
     *
     * @param properties -
     * @param options - Options to Set Properties operation.
     * @returns Response data for the Set Properties operation.
     */
    async setProperties(properties, options = {}) {
        return tracing_js_1.tracingClient.withSpan("ShareServiceClient-setProperties", options, async (updatedOptions) => {
            return (0, utils_common_js_1.assertResponse)(await this.serviceContext.setProperties(properties, {
                ...updatedOptions,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * Returns an AsyncIterableIterator for {@link ServiceListSharesSegmentResponse} objects
     *
     * @param marker - A string value that identifies the portion of
     *                          the list of shares to be returned with the next listing operation. The
     *                          operation returns the ContinuationToken value within the response body if the
     *                          listing operation did not return all shares remaining to be listed
     *                          with the current page. The ContinuationToken value can be used as the value for
     *                          the marker parameter in a subsequent call to request the next page of list
     *                          items. The marker value is opaque to the client.
     * @param options - Options to list shares operation.
     */
    async *listSegments(marker, options = {}) {
        if (options.prefix === "") {
            options.prefix = undefined;
        }
        let listSharesSegmentResponse;
        do {
            listSharesSegmentResponse = await this.listSharesSegment(marker, options);
            marker = listSharesSegmentResponse.continuationToken;
            yield await listSharesSegmentResponse;
        } while (marker);
    }
    /**
     * Returns an AsyncIterableIterator for share items
     *
     * @param options - Options to list shares operation.
     */
    async *listItems(options = {}) {
        if (options.prefix === "") {
            options.prefix = undefined;
        }
        let marker;
        for await (const segment of this.listSegments(marker, options)) {
            if (segment.shareItems) {
                yield* segment.shareItems;
            }
        }
    }
    /**
     * Returns an async iterable iterator to list all the shares
     * under the specified account.
     *
     * .byPage() returns an async iterable iterator to list the shares in pages.
     *
     * Example using `for await` syntax:
     *
     * ```ts snippet:ReadmeSampleListShares
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * let i = 1;
     * for await (const share of serviceClient.listShares()) {
     *   console.log(`Share${i++}: ${share.name}`);
     * }
     * ```
     *
     * Example using `iter.next()`:
     *
     * ```ts snippet:ReadmeSampleListShares_Iterator
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareIter = serviceClient.listShares();
     * let i = 1;
     * let { value, done } = await shareIter.next();
     * while (!done) {
     *   console.log(`Share ${i++}: ${value.name}`);
     *   ({ value, done } = await shareIter.next());
     * }
     * ```
     *
     * Example using `byPage()`:
     *
     * ```ts snippet:ReadmeSampleListShares_ByPage
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * let i = 1;
     * for await (const response of serviceClient.listShares().byPage({ maxPageSize: 20 })) {
     *   console.log(`Page ${i++}:`);
     *   for (const share of response.shareItems || []) {
     *     console.log(`\tShare: ${share.name}`);
     *   }
     * }
     * ```
     *
     * Example using paging with a marker:
     *
     * ```ts snippet:ReadmeSampleListShares_Continuation
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * let iterator = serviceClient.listShares().byPage({ maxPageSize: 2 });
     * let response = (await iterator.next()).value;
     *
     * for await (const share of response.shareItems || []) {
     *   console.log(`\tShare: ${share.name}`);
     * }
     *
     * // Gets next marker
     * let marker = response.continuationToken;
     *
     * // Passing next marker as continuationToken
     * iterator = serviceClient.listShares().byPage({ continuationToken: marker, maxPageSize: 10 });
     * response = (await iterator.next()).value;
     *
     * for await (const share of response.shareItems || []) {
     *   console.log(`\tShare: ${share.name}`);
     * }
     * ```
     *
     * @param options - Options to list shares operation.
     *
     * An asyncIterableIterator that supports paging.
     */
    listShares(options = {}) {
        if (options.prefix === "") {
            options.prefix = undefined;
        }
        const include = [];
        if (options.includeMetadata) {
            include.push("metadata");
        }
        if (options.includeSnapshots) {
            include.push("snapshots");
        }
        if (options.includeDeleted) {
            include.push("deleted");
        }
        const updatedOptions = {
            ...options,
            ...(include.length > 0 ? { include: include } : {}),
        };
        // AsyncIterableIterator to iterate over queues
        const iter = this.listItems(updatedOptions);
        return {
            /**
             * The next method, part of the iteration protocol
             */
            next() {
                return iter.next();
            },
            /**
             * The connection to the async iterator, part of the iteration protocol
             */
            [Symbol.asyncIterator]() {
                return this;
            },
            /**
             * Return an AsyncIterableIterator that works a page at a time
             */
            byPage: (settings = {}) => {
                return this.listSegments((0, utils_common_js_1.removeEmptyString)(settings.continuationToken), {
                    maxResults: settings.maxPageSize,
                    ...updatedOptions,
                });
            },
        };
    }
    /**
     * Gets the properties of a storage account's File service, including properties for Storage
     * Analytics metrics and CORS (Cross-Origin Resource Sharing) rules.
     *
     * @param marker - A string value that identifies the portion of
     *                          the list to be returned with the next list operation. The operation
     *                          returns a marker value within the response body if the list returned was
     *                          not complete. The marker value may then be used in a subsequent call to
     *                          request the next set of list items. The marker value is opaque to the
     *                          client.
     * @param options - Options to List Shares Segment operation.
     * @returns Response data for the List Shares Segment operation.
     */
    async listSharesSegment(marker, options = {}) {
        if (options.prefix === "") {
            options.prefix = undefined;
        }
        return tracing_js_1.tracingClient.withSpan("ShareServiceClient-listSharesSegment", options, async (updatedOptions) => {
            const res = (0, utils_common_js_1.assertResponse)(await this.serviceContext.listSharesSegment({
                ...updatedOptions,
                ...this.shareClientConfig,
                marker,
            }));
            // parse protocols
            if (res.shareItems) {
                for (let i = 0; i < res.shareItems.length; i++) {
                    const protocolsStr = res.shareItems[i].properties.enabledProtocols;
                    res.shareItems[i].properties.protocols = (0, models_js_1.toShareProtocols)(protocolsStr);
                }
            }
            return res;
        });
    }
    /**
     * Restores a previously deleted share.
     * This API is only functional if Share Soft Delete is enabled
     * for the storage account associated with the share.
     *
     * @param deletedShareName - The name of the previously deleted share.
     * @param deletedShareVersion - The version of the previously deleted share.
     * @param options - Options to Share undelete operation.
     * @returns Restored share.
     */
    async undeleteShare(deletedShareName, deletedShareVersion, options = {}) {
        return tracing_js_1.tracingClient.withSpan("ShareServiceClient-undeleteShare", options, async (updatedOptions) => {
            const shareClient = this.getShareClient(deletedShareName);
            await new ShareClientInternal_js_1.ShareClientInternal(shareClient.url, this.pipeline).restore({
                ...updatedOptions,
                ...this.shareClientConfig,
                deletedShareName: deletedShareName,
                deletedShareVersion: deletedShareVersion,
            });
            return shareClient;
        });
    }
    /**
     * Only available for ShareServiceClient constructed with a shared key credential.
     *
     * Generates an account Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/create-account-sas
     *
     * @param expiresOn - Optional. The time at which the shared access signature becomes invalid. Default to an hour later if not specified.
     * @param permissions - Specifies the list of permissions to be associated with the SAS.
     * @param resourceTypes - Specifies the resource types associated with the shared access signature.
     * @param options - Optional parameters.
     * @returns An account SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateAccountSasUrl(expiresOn, permissions = AccountSASPermissions_js_1.AccountSASPermissions.parse("r"), resourceTypes = "sco", options = {}) {
        if (!(this.credential instanceof storage_common_2.StorageSharedKeyCredential)) {
            throw RangeError("Can only generate the account SAS when the client is initialized with a shared key credential");
        }
        if (expiresOn === undefined) {
            const now = new Date();
            expiresOn = new Date(now.getTime() + 3600 * 1000);
        }
        const sas = (0, AccountSASSignatureValues_js_1.generateAccountSASQueryParameters)({
            permissions,
            expiresOn,
            resourceTypes,
            services: AccountSASServices_js_1.AccountSASServices.parse("f").toString(),
            ...options,
        }, this.credential).toString();
        return (0, utils_common_js_2.appendToURLQuery)(this.url, sas);
    }
    /**
     * Only available for ShareServiceClient constructed with a shared key credential.
     *
     * Generates string to sign for an account Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/create-account-sas
     *
     * @param expiresOn - Optional. The time at which the shared access signature becomes invalid. Default to an hour later if not specified.
     * @param permissions - Specifies the list of permissions to be associated with the SAS.
     * @param resourceTypes - Specifies the resource types associated with the shared access signature.
     * @param options - Optional parameters.
     * @returns An account SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasStringToSign(expiresOn, permissions = AccountSASPermissions_js_1.AccountSASPermissions.parse("r"), resourceTypes = "sco", 
    // eslint-disable-next-line @azure/azure-sdk/ts-naming-options
    options = {}) {
        if (!(this.credential instanceof storage_common_2.StorageSharedKeyCredential)) {
            throw RangeError("Can only generate the account SAS when the client is initialized with a shared key credential");
        }
        if (expiresOn === undefined) {
            const now = new Date();
            expiresOn = new Date(now.getTime() + 3600 * 1000);
        }
        return (0, AccountSASSignatureValues_js_1.generateAccountSASQueryParametersInternal)({
            permissions,
            expiresOn,
            resourceTypes,
            services: AccountSASServices_js_1.AccountSASServices.parse("f").toString(),
            ...options,
        }, this.credential).stringToSign;
    }
    /**
     * ONLY AVAILABLE WHEN USING BEARER TOKEN AUTHENTICATION (TokenCredential).
     *
     * Retrieves a user delegation key for the File service. This is only a valid operation when using
     * bearer token authentication.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/get-user-delegation-key
     *
     * @param startsOn -      The start time for the user delegation SAS. Must be within 7 days of the current time
     * @param expiresOn -     The end time for the user delegation SAS. Must be within 7 days of the current time
     */
    async getUserDelegationKey(startsOn, expiresOn, options = {}) {
        return tracing_js_1.tracingClient.withSpan("ShareServiceClient-getUserDelegationKey", options, async (updatedOptions) => {
            const response = (0, utils_common_js_1.assertResponse)(await this.serviceContext.getUserDelegationKey({
                start: (0, utils_common_js_1.truncatedISO8061Date)(startsOn, false),
                expiry: (0, utils_common_js_1.truncatedISO8061Date)(expiresOn, false),
            }, {
                abortSignal: options.abortSignal,
                tracingOptions: updatedOptions.tracingOptions,
            }));
            const userDelegationKey = {
                signedObjectId: response.signedObjectId,
                signedTenantId: response.signedTenantId,
                signedStartsOn: new Date(response.signedStartsOn),
                signedExpiresOn: new Date(response.signedExpiresOn),
                signedService: response.signedService,
                signedVersion: response.signedVersion,
                value: response.value,
            };
            const res = {
                _response: response._response,
                requestId: response.requestId,
                clientRequestId: response.clientRequestId,
                version: response.version,
                date: response.date,
                ...userDelegationKey,
            };
            return res;
        });
    }
}
exports.ShareServiceClient = ShareServiceClient;
//# sourceMappingURL=ShareServiceClient.js.map