// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
import { FileSystemAttributes } from "./FileSystemAttributes.js";
import { truncatedISO8061Date } from "./utils/utils.common.js";
import { logger } from "./log.js";
/**
 * Convert protocols from joined string to ShareProtocols.
 *
 * @param protocolsString -
 */
export function toShareProtocols(protocolsString) {
    if (protocolsString === undefined) {
        return undefined;
    }
    const protocolStrArray = protocolsString.split(";");
    const protocols = {};
    for (const protocol of protocolStrArray) {
        if (protocol === "SMB") {
            protocols.smbEnabled = true;
        }
        else if (protocol === "NFS") {
            protocols.nfsEnabled = true;
        }
    }
    return protocols;
}
/**
 * Convert ShareProtocols to joined string.
 *
 * @param protocols -
 */
export function toShareProtocolsString(protocols = {}) {
    let protocolStr = undefined;
    if (protocols.smbEnabled === true) {
        protocolStr = "SMB";
    }
    if (protocols.nfsEnabled === true) {
        logger.info(`Using "NFS" in favor of "SMB" for the share protocol as currently they can't be supported at the same time.`);
        protocolStr = "NFS";
    }
    return protocolStr;
}
export function validateFilePermissionOptions(filePermission, filePermissionKey) {
    if (filePermission && filePermissionKey) {
        throw new RangeError("Only one of filePermission or filePermissionKey can be specified.");
    }
}
export function validateAndSetDefaultsForFileAndDirectoryCreateCommonOptions(options) {
    // Would better deep copy params set by user.
    if (!options) {
        options = {};
    }
    validateFilePermissionOptions(options.filePermission, options.filePermissionKey);
    return options;
}
export function validateAndSetDefaultsForFileAndDirectorySetPropertiesCommonOptions(options) {
    // Would better deep copy params set by user.
    if (!options) {
        options = {};
    }
    validateFilePermissionOptions(options.filePermission, options.filePermissionKey);
    return options;
}
export function fileAttributesToString(fileAttributes) {
    return fileAttributes instanceof FileSystemAttributes
        ? fileAttributes.toString()
        : fileAttributes;
}
export function fileCreationTimeToString(time) {
    return time instanceof Date ? truncatedISO8061Date(time) : time;
}
export function fileLastWriteTimeToString(time) {
    return time instanceof Date ? truncatedISO8061Date(time) : time;
}
export function fileChangeTimeToString(time) {
    return time instanceof Date ? truncatedISO8061Date(time) : time;
}
/**
 * Defines the known cloud audiences for Storage.
 */
export var StorageFileAudience;
(function (StorageFileAudience) {
    /**
     * The OAuth scope to use to retrieve an AAD token for Azure Storage.
     */
    StorageFileAudience["StorageOAuthScopes"] = "https://storage.azure.com/.default";
})(StorageFileAudience || (StorageFileAudience = {}));
/**
 * To get the OAuth audience for a storage account for file service.
 */
export function getFileServiceAccountAudience(storageAccountName) {
    return `https://${storageAccountName}.file.core.windows.net/.default`;
}
//# sourceMappingURL=models.js.map