import { UserDelegationKey } from "@azure/storage-common";
import type { SasIPRange } from "./SasIPRange.js";
/**
 * Protocols for generated SAS.
 */
export declare enum SASProtocol {
    /**
     * Protocol that allows HTTPS only
     */
    Https = "https",
    /**
     * Protocol that allows both HTTPS and HTTP
     */
    HttpsAndHttp = "https,http"
}
/**
 * Represents the components that make up an Azure Storage SAS' query parameters. This type is not constructed directly
 * by the user; it is only generated by the {@link AccountSASSignatureValues} and {@link QueueSASSignatureValues}
 * types. Once generated, it can be encoded into a {@link String} and appended to a URL directly (though caution should
 * be taken here in case there are existing query parameters, which might affect the appropriate means of appending
 * these query parameters).
 *
 * NOTE: Instances of this class are immutable.
 */
export declare class SASQueryParameters {
    /**
     * The storage API version.
     */
    readonly version: string;
    /**
     * Optional. The allowed HTTP protocol(s).
     */
    readonly protocol?: SASProtocol;
    /**
     * Optional. The start time for this SAS token.
     */
    readonly startsOn?: Date;
    /**
     * Optional only when identifier is provided. The expiry time for this SAS token.
     */
    readonly expiresOn?: Date;
    /**
     * Optional only when identifier is provided.
     * Please refer to {@link AccountSASPermissions}, {@link QueueSASPermissions} for more details.
     */
    readonly permissions?: string;
    /**
     * Optional. The storage services being accessed (only for Account SAS). Please refer to {@link AccountSASServices}
     * for more details.
     */
    readonly services?: string;
    /**
     * Optional. The storage resource types being accessed (only for Account SAS). Please refer to
     * {@link AccountSASResourceTypes} for more details.
     */
    readonly resourceTypes?: string;
    /**
     * Optional. The signed identifier (only for {@link QueueSASSignatureValues}).
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/establishing-a-stored-access-policy
     */
    readonly identifier?: string;
    /**
     * Optional. The storage queue (only for {@link QueueSASSignatureValues}).
     */
    readonly resource?: string;
    /**
     * The signature for the SAS token.
     */
    readonly signature: string;
    /**
     * Inner value of getter ipRange.
     */
    private readonly ipRangeInner?;
    /**
     * The Azure Active Directory object ID in GUID format.
     * Property of user delegation key.
     */
    private readonly signedOid?;
    /**
     * The Azure Active Directory tenant ID in GUID format.
     * Property of user delegation key.
     */
    private readonly signedTenantId?;
    /**
     * The date-time the key is active.
     * Property of user delegation key.
     */
    private readonly signedStartsOn?;
    /**
     * The date-time the key expires.
     * Property of user delegation key.
     */
    private readonly signedExpiresOn?;
    /**
     * Abbreviation of the Azure Storage service that accepts the user delegation key.
     * Property of user delegation key.
     */
    private readonly signedService?;
    /**
     * The service version that created the user delegation key.
     * Property of user delegation key.
     */
    private readonly signedVersion?;
    /**
     * Optional. Beginning in version 2025-07-05, this value specifies the Entra ID of the user who is authorized to
     * use the resulting SAS URL.  The resulting SAS URL must be used in conjunction with an Entra ID token that has been
     * issued to the user specified in this value.
     */
    readonly delegatedUserObjectId?: string;
    /**
     * Optional. IP range allowed for this SAS.
     *
     * @readonly
     */
    get ipRange(): SasIPRange | undefined;
    /**
     * Creates an instance of SASQueryParameters.
     *
     * @param version - Representing the storage version
     * @param signature - Representing the signature for the SAS token
     * @param permissions - Representing the storage permissions
     * @param services - Representing the storage services being accessed (only for Account SAS)
     * @param resourceTypes - Representing the storage resource types being accessed (only for Account SAS)
     * @param protocol - Representing the allowed HTTP protocol(s)
     * @param startsOn - Representing the start time for this SAS token
     * @param expiresOn - Representing the expiry time for this SAS token
     * @param ipRange - Representing the range of valid IP addresses for this SAS token
     * @param identifier - Representing the signed identifier (only for Service SAS)
     * @param resource - Representing the storage queue (only for Service SAS)
     */
    constructor(version: string, signature: string, permissions?: string, services?: string, resourceTypes?: string, protocol?: SASProtocol, startsOn?: Date, expiresOn?: Date, ipRange?: SasIPRange, identifier?: string, resource?: string, userDelegationKey?: UserDelegationKey, delegatedUserObjectId?: string);
    /**
     * Encodes all SAS query parameters into a string that can be appended to a URL.
     *
     */
    toString(): string;
    /**
     * A private helper method used to filter and append query key/value pairs into an array.
     *
     * @param queries -
     * @param key -
     * @param value -
     */
    private tryAppendQueryParameter;
}
//# sourceMappingURL=SASQueryParameters.d.ts.map