#!/usr/bin/env python

__description__ = 'This is essentially a wrapper for xml.etree.ElementTree'
__author__ = 'Didier Stevens'
__version__ = '0.0.10'
__date__ = '2025/03/05'

"""

Source code put in public domain by Didier Stevens, no Copyright
https://DidierStevens.com
Use at your own risk

History:
  2017/11/03: start
  2017/12/16: refactoring
  2017/12/16: 0.0.2 added elementtext and attributes command
  2017/12/31: added option -u
  2018/04/01: 0.0.3 added support for xmlns with single quote
  2018/06/29: 0.0.4 ProcessFile for Linux/OSX
  2020/01/12: added pretty print
  2020/04/16: 0.0.5 added celltext; option --encoding
  2020/04/16: 0.0.6 updated man page
  2020/06/07: 0.0.7 added option -j
  2024/07/11: 0.0.8 added officeprotection
  2024/07/11: 0.0.9 added option -j support to command pretty
  2025/03/05: 0.0.10 bugfix xambroz

Todo:
"""

import optparse
import glob
import collections
import time
import sys
import textwrap
import xml.etree.ElementTree
import re
import xml.dom.minidom
import json
import binascii

def PrintManual():
    manual = r'''
Manual:

xmldump.py can be used to extract information from XML files, it is essentially a wrapper for xml.etree.ElementTree.

This Python script was developed with Python 2.7 and tested with Python 2.7 and 3.7.

It reads one or more files or stdin to parse XML files. If no file arguments are provided to this tool, it will read data from standard input (stdin). This way, this tool can be used in a piped chain of commands.

The first argument to the tool is a command, which can be:
 text
 wordtext
 elementtext
 attributes
 pretty
 celltext
 officeprotection

Command text will extract all text from the elements in the XML file.
Example:
zipdump.py -s 4 -d test.docx | xmldump.py text

This is a test document.Second line.Third linehttps://DidierStevens.comLast line

Command wordtext will extract all text from <w:p> elements in the XML file and print each on a separate line.
Example:
zipdump.py -s 4 -d test.docx | xmldump.py wordtext

This is a test document.
Second line.
Third line
https://DidierStevens.com
Last line

Command elementtext will extract all elements with their text from the XML file.
Example:
zipdump.py -s 4 -d test.docx | xmldump.py elementtext

w:document: This is a test document.Second line.Third linehttps://DidierStevens.comLast line
w:body: This is a test document.Second line.Third linehttps://DidierStevens.comLast line
w:p: This is a test document.
w:r: This is a test document.
w:t: This is a test document.
w:p: Second line.
w:r: Second line.
w:t: Second line.
w:p: Third line
w:r: Third line
w:t: Third line
w:bookmarkStart:
w:bookmarkEnd:
w:p: https://DidierStevens.com
w:hyperlink: https://DidierStevens.com
w:r: https://DidierStevens.com
w:rPr:
w:rStyle:
w:t: https://DidierStevens.com
w:p: Last line
w:r: Last line
w:t: Last line
w:sectPr:
w:pgSz:
w:pgMar:
w:cols:
w:docGrid:

By default, the namespace URI (xmlns) is suppressed. Use option -u to include it.
Example:
zipdump.py -s 4 -d test.docx | xmldump.py -u elementtext

{http://schemas.openxmlformats.org/wordprocessingml/2006/main}document: This is a test document.Second line.Third linehttps://DidierStevens.comLast line
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}body: This is a test document.Second line.Third linehttps://DidierStevens.comLast line
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}p: This is a test document.
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}r: This is a test document.
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}t: This is a test document.
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}p: Second line.
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}r: Second line.
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}t: Second line.
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}p: Third line
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}r: Third line
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}t: Third line
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}bookmarkStart:
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}bookmarkEnd:
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}p: https://DidierStevens.com
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}hyperlink: https://DidierStevens.com
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}r: https://DidierStevens.com
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}rPr:
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}rStyle:
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}t: https://DidierStevens.com
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}p: Last line
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}r: Last line
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}t: Last line
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}sectPr:
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}pgSz:
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}pgMar:
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}cols:
{http://schemas.openxmlformats.org/wordprocessingml/2006/main}docGrid:

Command attributes will extract all attributes from the elements in the XML file.
Example:
zipdump.py -s 4 -d test.docx | xmldump.py attributes

w:document
  mc:Ignorable: w14 w15 w16se w16cid wp14
w:body
w:p
  w:rsidRDefault: 006E3AD4
  w:rsidR: 00F41D2A
w:r
w:t
w:p
  w:rsidRDefault: 006E3AD4
  w:rsidR: 006E3AD4
w:r
w:t
w:p
  w:rsidRDefault: 006E3AD4
  w:rsidR: 006E3AD4
w:r
w:t
w:bookmarkStart
  w:id: 0
  w:name: _GoBack
w:bookmarkEnd
  w:id: 0
w:p
  w:rsidRDefault: 006D313B
  w:rsidR: 006E3AD4
w:hyperlink
  w:history: 1
  r:id: rId4
w:r
  w:rsidRPr: 0074025F
  w:rsidR: 006E3AD4
w:rPr
w:rStyle
  w:val: Hyperlink
w:t
w:p
  w:rsidRDefault: 006E3AD4
  w:rsidR: 006E3AD4
w:r
w:t
w:sectPr
  w:rsidR: 006E3AD4
w:pgSz
  w:h: 15840
  w:w: 12240
w:pgMar
  w:left: 1440
  w:header: 720
  w:top: 1440
  w:right: 1440
  w:bottom: 1440
  w:footer: 720
  w:gutter: 0
w:cols
  w:space: 720
w:docGrid
  w:linePitch: 360

Command pretty will just perform a pretty print of the XML file.

Command celltext will produce a CSV file of all the cells in an Excel sheet.

Example:
zipdump.py -s 5 -d spreadsheet.xlsm | xmldump.py celltext

Reference,Formula,Value
A1,EXEC("note"&"pad"),33
A2,HALT(),1

Option -j can be used with celltext to pass JSON output from zipdump to xmldump, and have shared strings references resolved.
Option -j takes the id number of the XML file to convert to CSV.

Command officeprotection will extract all protection hashes from the elements in the XML file.
Example:
zipdump.py -s 4 -d test.xlsx | xmldump.py officeprotection

It can also be used with option -j to process all XML files inside an Office document:
Example:
zipdump.py --jsonoutput test.xlsx | xmldump.py -j all officeprotection


By default, output is printed to the console (stdout). It can be directed to a file using option -o.
'''
    for line in manual.split('\n'):
        print(textwrap.fill(line))

DEFAULT_SEPARATOR = ','
QUOTE = '"'

def File2Strings(filename):
    try:
        f = open(filename, 'r')
    except:
        return None
    try:
        return map(lambda line:line.rstrip('\n'), f.readlines())
    except:
        return None
    finally:
        f.close()

def ProcessAt(argument):
    if argument.startswith('@'):
        strings = File2Strings(argument[1:])
        if strings == None:
            raise Exception('Error reading %s' % argument)
        else:
            return strings
    else:
        return [argument]

# CIC: Call If Callable
def CIC(expression):
    if callable(expression):
        return expression()
    else:
        return expression

# IFF: IF Function
def IFF(expression, valueTrue, valueFalse):
    if expression:
        return CIC(valueTrue)
    else:
        return CIC(valueFalse)

def ToString(value):
    if isinstance(value, str):
        return value
    else:
        return str(value)

def Quote(value, separator, quote):
    value = ToString(value)
    if len(value) > 1 and value[0] == quote and value[-1] == quote:
        return value
    if separator in value or value == '':
        return quote + value + quote
    else:
        return value

def MakeCSVLine(row, separator, quote):
    return separator.join([Quote(value, separator, quote) for value in row])

class cOutput():
    def __init__(self, filename=None):
        self.filename = filename
        self.separatingline = False
        if self.filename and self.filename != '':
            self.f = open(self.filename, 'w')
        else:
            self.f = None

    def Line(self, line):
        if self.f:
            self.f.write(line + '\n')
        else:
            try:
                print(line)
            except UnicodeEncodeError:
                encoding = sys.stdout.encoding
                print(line.encode(encoding, errors='backslashreplace').decode(encoding))
#            sys.stdout.flush()

    def Close(self):
        if self.f:
            self.f.close()
            self.f = None

    def StartSeparatingLine(self):
        self.separatingline = False

    def SeparatingLine(self, line):
        if self.separatingline:
            self.Line(line)
        else:
            self.separatingline = True

def ExpandFilenameArguments(filenames):
    return list(collections.OrderedDict.fromkeys(sum(map(glob.glob, sum(map(ProcessAt, filenames), [])), [])))

class cOutputResult():
    def __init__(self, options):
        if options.output:
            self.oOutput = cOutput(options.output)
        else:
            self.oOutput = cOutput()
        self.options = options

    def Line(self, line):
        self.oOutput.Line(line)

    def Close(self):
        self.oOutput.Close()

    def StartSeparatingLine(self):
        self.oOutput.StartSeparatingLine()

    def SeparatingLine(self, line):
        self.oOutput.SeparatingLine(line)

def ProcessFile(fIn, fullread):
    if fullread:
        yield fIn.read()
    else:
        for line in fIn:
            yield line.strip('\n\r')

def CheckJSON(stringJSON):
    try:
        object = json.loads(stringJSON)
    except:
        print('Error parsing JSON')
        print(sys.exc_info()[1])
        return None
    if not isinstance(object, dict):
        print('Error JSON is not a dictionary')
        return None
    if not 'version' in object:
        print('Error JSON dictionary has no version')
        return None
    if object['version'] != 2:
        print('Error JSON dictionary has wrong version')
        return None
    if not 'id' in object:
        print('Error JSON dictionary has no id')
        return None
    if object['id'] != 'didierstevens.com':
        print('Error JSON dictionary has wrong id')
        return None
    if not 'type' in object:
        print('Error JSON dictionary has no type')
        return None
    if object['type'] != 'content':
        print('Error JSON dictionary has wrong type')
        return None
    if not 'fields' in object:
        print('Error JSON dictionary has no fields')
        return None
    if not 'name' in object['fields']:
        print('Error JSON dictionary has no name field')
        return None
    if not 'content' in object['fields']:
        print('Error JSON dictionary has no content field')
        return None
    if not 'items' in object:
        print('Error JSON dictionary has no items')
        return None
    for item in object['items']:
        item['content'] = binascii.a2b_base64(item['content'])
    return object['items']

def XMLGetText(element):
    if sys.version_info[0] > 2:
        encoding = 'unicode'
    else:
        encoding = 'utf8'
    return xml.etree.ElementTree.tostring(element, encoding=encoding, method='text')

def TransformTag(tag, dXMLNS, includeURI):
    if includeURI:
        return tag
    elif tag.startswith('{'):
        uri, separator, remainder = tag[1:].partition('}')
        if uri in dXMLNS:
            if dXMLNS[uri] == '':
                return remainder
            else:
                return dXMLNS[uri] + ':' + remainder
        else:
            return tag
    else:
        return tag

def AnalyzeXMLNS(data):
    dXMLNS = {}
    for match in re.findall('xmlns(:([^=]+))?="([^"]+)"', data):
        dXMLNS[match[2]] = match[1]
    for match in re.findall("xmlns(:([^=]+))?='([^']+)'", data):
        dXMLNS[match[2]] = match[1]
    root = xml.etree.ElementTree.fromstring(data)
    return root, dXMLNS

def ExtractText(data, oOutput, options):
    root, dXMLNS = AnalyzeXMLNS(data)
    oOutput.Line(XMLGetText(root))

def ExtractWordText(data, oOutput, options):
    root, dXMLNS = AnalyzeXMLNS(data)
    for element in root.iter('{http://schemas.openxmlformats.org/wordprocessingml/2006/main}p'):
        oOutput.Line(XMLGetText(element))

def ExtractElementText(data, oOutput, options):
    root, dXMLNS = AnalyzeXMLNS(data)
    for element in root.iter():
        oOutput.Line('%s: %s' % (TransformTag(element.tag, dXMLNS, options.includeuri), XMLGetText(element)))

def ExtractElementAttributes(data, oOutput, options):
    root, dXMLNS = AnalyzeXMLNS(data)
    for element in root.iter():
        oOutput.Line('%s' % (TransformTag(element.tag, dXMLNS, options.includeuri)))
        for key, value in element.items():
            oOutput.Line('  %s: %s' % (TransformTag(key, dXMLNS, options.includeuri), value))

def PrettyPrint(data, oOutput, options):
    if options.jsoninput != '':
        items = CheckJSON(data)
        if items == None:
            return
        parsingerrors = []
        for item in items:
            try:
                root = xml.dom.minidom.parseString(item['content'].decode())
            except:
                parsingerrors.append(item)
                continue
            oOutput.Line(root.toprettyxml())
        if parsingerrors != []:
            oOutput.Line('\nXML error parsing file(s):')
            for item in parsingerrors:
                oOutput.Line(' %s' % item['name'])
    else:
        oOutput.Line(xml.dom.minidom.parseString(data).toprettyxml())

def ExtractCellText(data, oOutput, options):
    dSharedStrings = {}
    if options.jsoninput != '':
        items = CheckJSON(data)
        if items == None:
            return
        for item in items:
            if item['name'].endswith('/sharedStrings.xml'):
                root, dXMLNS = AnalyzeXMLNS(item['content'].decode())
                counter = 0
                for element in root.iter('{http://schemas.openxmlformats.org/spreadsheetml/2006/main}si'):
                    for child in element:
                        dSharedStrings[counter] = child.text
                    counter += 1
            elif item['id'] == int(options.jsoninput):
                data = item['content'].decode()

    oOutput.Line(MakeCSVLine(['Reference', 'Formula', 'Value'], DEFAULT_SEPARATOR, QUOTE))
    root, dXMLNS = AnalyzeXMLNS(data)
    for element in root.iter('{http://schemas.openxmlformats.org/spreadsheetml/2006/main}c'):
        if 'r' in element.attrib:
            reference = element.attrib['r']
        else:
            reference = ''
        cellType = element.attrib.get('t', '')
        formula = ''
        value = ''
        for child in element:
            if child.tag == '{http://schemas.openxmlformats.org/spreadsheetml/2006/main}f':
                formula = XMLGetText(child)
            if child.tag == '{http://schemas.openxmlformats.org/spreadsheetml/2006/main}v':
                value = XMLGetText(child)
                if cellType == 's':
                    formula = dSharedStrings.get(int(value), '')
        oOutput.Line(MakeCSVLine([reference, formula, value], DEFAULT_SEPARATOR, QUOTE))

dVerifier = {
    0x8000: 'ccccca049',
    0x8001: 'ccccaban',
    0x8002: 'ccccccab49',
    0x8003: 'ccccccccb2',
    0x8004: 'cccccccaaaI',
    0x8005: 'cccccccaaaY',
    0x8006: 'cccccccaaai',
    0x8007: 'cccccccaaay',
    0x8008: 'ccccccccbbbM',
    0x8009: 'ccccccccbbbE',
    0x800a: 'ccccccccbbhX',
    0x800b: 'ccccccccbbbU',
    0x800c: 'ccccccccbbbm',
    0x800d: 'ccccccccbbbe',
    0x800e: 'ccccccccbbhx',
    0x800f: 'ccccccccbbbu',
    0x8010: 'qqqqqqqqbabl',
    0x8011: 'kkkkbbbo',
    0x8012: 'qqqqqqqqbahy',
    0x8013: 'hhhhaaan',
    0x8014: 'qqqqqqqqbabL',
    0x8015: 'mmmmbaan',
    0x8016: 'qqqqqqqqbahY',
    0x8017: 'nnnnabbo',
    0x8018: 'qqqqqqqabaH',
    0x8019: 'ggggbban',
    0x801a: 'qqqqqqqabah',
    0x801b: 'ddddaabo',
    0x801c: 'qqqqqa158',
    0x801d: 'aaaababo',
    0x801e: 'qqqqqqaa48',
    0x801f: 'bbbbaban',
    0x8020: 'sssssb058',
    0x8021: 'ccccbabo',
    0x8022: 'pppppa158',
    0x8023: 'ppppbbbn',
    0x8024: 'uuuuub158',
    0x8025: 'eeeebban',
    0x8026: 'vvvvva058',
    0x8027: 'ffffaabo',
    0x8028: 'PPPPPPPPaaiZ',
    0x8029: 'oooobaan',
    0x802a: 'PPPPPPPPaaaN',
    0x802b: 'llllabbo',
    0x802c: 'yyyyyb148',
    0x802d: 'iiiibbbo',
    0x802e: 'zzzzza048',
    0x802f: 'jjjjaaan',
    0x8030: 'kkkkkb059',
    0x8031: 'kkkkaaan',
    0x8032: 'hhhhha159',
    0x8033: 'hhhhbbbo',
    0x8034: 'mmmmmb159',
    0x8035: 'mmmmabbo',
    0x8036: 'nnnnna059',
    0x8037: 'nnnnbaan',
    0x8038: 'gggggb049',
    0x8039: 'ggggaabo',
    0x803a: 'ddddda149',
    0x803b: 'ddddbban',
    0x803c: 'aaaaab149',
    0x803d: 'aaaaaban',
    0x803e: 'bbbbba049',
    0x803f: 'bbbbbabo',
    0x8040: 'cccccb149',
    0x8041: 'ccccaabo',
    0x8042: 'CCCCCCbb18',
    0x8043: 'ppppabbn',
    0x8044: 'eeeeeb049',
    0x8045: 'eeeeaban',
    0x8046: 'fffffa149',
    0x8047: 'ffffbabo',
    0x8048: 'ooooob159',
    0x8049: 'ooooaaan',
    0x804a: 'llllla059',
    0x804b: 'llllbbbo',
    0x804c: 'iiiiib059',
    0x804d: 'iiiiabbo',
    0x804e: 'jjjjja159',
    0x804f: 'jjjjbaan',
    0x8050: 'XXXXXXaa09',
    0x8051: 'kkkkbaan',
    0x8052: 'xxxxxa048',
    0x8053: 'hhhhabbo',
    0x8054: '=====b199',
    0x8055: 'mmmmbbbo',
    0x8056: '>>>>>a099',
    0x8057: 'nnnnaaan',
    0x8058: 'wwwwwb158',
    0x8059: 'ggggbabo',
    0x805a: 'ttttta058',
    0x805b: 'ddddaban',
    0x805c: 'qqqqqb058',
    0x805d: 'aaaabban',
    0x805e: 'rrrrra158',
    0x805f: 'bbbbaabo',
    0x8060: 'ppppppaa48',
    0x8061: 'ccccbban',
    0x8062: 'pppppb058',
    0x8063: 'ppppbaao',
    0x8064: 'uuuuua058',
    0x8065: 'eeeebabo',
    0x8066: 'uuuuuuba48',
    0x8067: 'ffffaban',
    0x8068: '00000000aaaZ',
    0x8069: 'oooobbbo',
    0x806a: '00000000aaaJ',
    0x806b: 'llllaaan',
    0x806c: 'yyyyya048',
    0x806d: 'iiiibaan',
    0x806e: 'yyyyyyba58',
    0x806f: 'jjjjabbo',
    0x8070: 'hhhhhhaa49',
    0x8071: 'kkkkabbo',
    0x8072: 'hhhhhb059',
    0x8073: 'hhhhbaan',
    0x8074: 'mmmmma059',
    0x8075: 'mmmmaaan',
    0x8076: 'mmmmmmba49',
    0x8077: 'nnnnbbbo',
    0x8078: 'ddddddaa59',
    0x8079: 'ggggaban',
    0x807a: 'dddddb049',
    0x807b: 'ddddbabo',
    0x807c: 'aaaaaa049',
    0x807d: 'aaaaaabo',
    0x807e: 'aaaaaaba59',
    0x807f: 'bbbbbban',
    0x8080: 'ccccca149',
    0x8081: 'ccccabbo',
    0x8082: 'ccccccba59',
    0x8083: 'ppppaabn',
    0x8084: 'eeeeea049',
    0x8085: 'eeeeaaan',
    0x8086: 'eeeeeebb49',
    0x8087: 'ffffbbbo',
    0x8088: 'llllllab59',
    0x8089: 'ooooaban',
    0x808a: 'lllllb059',
    0x808b: 'llllbabo',
    0x808c: 'iiiiia059',
    0x808d: 'iiiiaabo',
    0x808e: 'iiiiiibb59',
    0x808f: 'jjjjbban',
    0x8090: 'xxxxxxab48',
    0x8091: 'kkkkbban',
    0x8092: 'xxxxxb048',
    0x8093: 'hhhhaabo',
    0x8094: '8888888abao',
    0x8095: 'mmmmbabo',
    0x8096: '8888888abaO',
    0x8097: 'nnnnaban',
    0x8098: 'ttttttab58',
    0x8099: 'ggggbbbo',
    0x809a: 'tttttb058',
    0x809b: 'ddddaaan',
    0x809c: 'qqqqqa058',
    0x809d: 'aaaabaan',
    0x809e: 'qqqqqqbb58',
    0x809f: 'bbbbabbo',
    0x80a0: 'sssssb158',
    0x80a1: 'ccccbaan',
    0x80a2: 'pppppa058',
    0x80a3: 'ppppbbao',
    0x80a4: 'uuuuub058',
    0x80a5: 'eeeebbbo',
    0x80a6: 'vvvvva158',
    0x80a7: 'ffffaaan',
    0x80a8: '?????b099',
    0x80a9: 'oooobabo',
    0x80aa: '<<<<<a199',
    0x80ab: 'llllaban',
    0x80ac: 'yyyyyb048',
    0x80ad: 'iiiibban',
    0x80ae: 'zzzzza148',
    0x80af: 'jjjjaabo',
    0x80b0: 'kkkkkb159',
    0x80b1: 'kkkkaabo',
    0x80b2: 'hhhhha059',
    0x80b3: 'hhhhbban',
    0x80b4: 'mmmmmb059',
    0x80b5: 'mmmmaban',
    0x80b6: 'nnnnna159',
    0x80b7: 'nnnnbabo',
    0x80b8: 'gggggb149',
    0x80b9: 'ggggaaan',
    0x80ba: 'ddddda049',
    0x80bb: 'ddddbbbo',
    0x80bc: 'aaaaab049',
    0x80bd: 'aaaaabbo',
    0x80be: 'bbbbba149',
    0x80bf: 'bbbbbaan',
    0x80c0: 'cccccb049',
    0x80c1: 'ccccaaan',
    0x80c2: 'CCCCCCaa08',
    0x80c3: 'ppppabao',
    0x80c4: 'eeeeeb149',
    0x80c5: 'eeeeabbo',
    0x80c6: 'fffffa049',
    0x80c7: 'ffffbaan',
    0x80c8: 'ooooob059',
    0x80c9: 'ooooaabo',
    0x80ca: 'llllla159',
    0x80cb: 'llllbban',
    0x80cc: 'iiiiib159',
    0x80cd: 'iiiiaban',
    0x80ce: 'jjjjja059',
    0x80cf: 'jjjjbabo',
    0x80d0: 'XXXXXXbb19',
    0x80d1: 'kkkkbabo',
    0x80d2: 'xxxxxa148',
    0x80d3: 'hhhhaban',
    0x80d4: 'XXXXXXXabak',
    0x80d5: 'mmmmbban',
    0x80d6: 'XXXXXXXabaK',
    0x80d7: 'nnnnaabo',
    0x80d8: 'wwwwwb058',
    0x80d9: 'ggggbaan',
    0x80da: 'ttttta158',
    0x80db: 'ddddabbo',
    0x80dc: 'qqqqqb158',
    0x80dd: 'aaaabbbo',
    0x80de: 'rrrrra058',
    0x80df: 'bbbbaaan',
    0x80e0: 'ppppppbb58',
    0x80e1: 'ccccbbbo',
    0x80e2: 'pppppb158',
    0x80e3: 'ppppbabn',
    0x80e4: 'pppppppabah',
    0x80e5: 'eeeebaan',
    0x80e6: 'pppppppabaH',
    0x80e7: 'ffffabbo',
    0x80e8: 'yyyyyyybbbI',
    0x80e9: 'oooobban',
    0x80ea: 'yyyyyyybbbi',
    0x80eb: 'llllaabo',
    0x80ec: 'yyyyya148',
    0x80ed: 'iiiibabo',
    0x80ee: 'yyyyyyab48',
    0x80ef: 'jjjjaban',
    0x80f0: 'hhhhhhbb59',
    0x80f1: 'kkkkaban',
    0x80f2: 'hhhhhb159',
    0x80f3: 'hhhhbabo',
    0x80f4: 'hhhhhhhabai',
    0x80f5: 'hhhhhhhabay',
    0x80f6: 'hhhhhhhabaI',
    0x80f7: 'hhhhhhhabaY',
    0x80f8: 'aaaaaaabbbH',
    0x80f9: 'aaaaaaabbbX',
    0x80fa: 'aaaaaaabbbh',
    0x80fb: 'aaaaaaabbbx',
    0x80fc: 'aaaaaa149',
    0x80fd: 'aaaaaaan',
    0x80fe: 'aaaaaaab49',
    0x80ff: 'bbbbbbbo',
    0x8100: 'ccccca059',
    0x8101: 'ccccabao',
    0x8102: 'ccccccaa59',
    0x8103: 'eeeeeeebaay',
    0x8104: 'cccccccbbbH',
    0x8105: 'cccccccbbbX',
    0x8106: 'cccccccbbbh',
    0x8107: 'cccccccbbbx',
    0x8108: 'iiiiiiibabH',
    0x8109: 'iiiiiiibabX',
    0x810a: 'iiiiiiibabh',
    0x810b: 'iiiiiiibabx',
    0x810c: 'iiiiia149',
    0x810d: 'iiiiaaao',
    0x810e: 'iiiiiiab59',
    0x810f: 'jjjjbbbn',
    0x8110: 'xxxxxxbb48',
    0x8111: 'kkkkbbbn',
    0x8112: 'xxxxxb158',
    0x8113: 'hhhhaaao',
    0x8114: 'xxxxxxxaaah',
    0x8115: 'mmmmbaao',
    0x8116: 'xxxxxxxaaaH',
    0x8117: 'nnnnabbn',
    0x8118: 'qqqqqqqbabI',
    0x8119: 'ggggbbao',
    0x811a: 'qqqqqqqbabi',
    0x811b: 'ddddaabn',
    0x811c: 'qqqqqa148',
    0x811d: 'aaaababn',
    0x811e: 'qqqqqqab58',
    0x811f: 'bbbbabao',
    0x8120: 'sssssb048',
    0x8121: 'ccccbabn',
    0x8122: 'pppppa148',
    0x8123: 'ppppbbbo',
    0x8124: 'uuuuub148',
    0x8125: 'eeeebbao',
    0x8126: 'vvvvva048',
    0x8127: 'ffffaabn',
    0x8128: 'YYYYYYYbabJ',
    0x8129: 'oooobaao',
    0x812a: 'YYYYYYYbabj',
    0x812b: 'llllabbn',
    0x812c: 'yyyyyb158',
    0x812d: 'iiiibbbn',
    0x812e: 'zzzzza058',
    0x812f: 'jjjjaaao',
    0x8130: 'kkkkkb049',
    0x8131: 'kkkkaaao',
    0x8132: 'hhhhha149',
    0x8133: 'hhhhbbbn',
    0x8134: 'mmmmmb149',
    0x8135: 'mmmmabbn',
    0x8136: 'nnnnna049',
    0x8137: 'nnnnbaao',
    0x8138: 'gggggb059',
    0x8139: 'ggggaabn',
    0x813a: 'ddddda159',
    0x813b: 'ddddbbao',
    0x813c: 'aaaaab159',
    0x813d: 'aaaaabao',
    0x813e: 'bbbbba059',
    0x813f: 'bbbbbabn',
    0x8140: 'cccccb159',
    0x8141: 'ccccaabn',
    0x8142: 'CCCCCCba08',
    0x8143: 'ppppabbo',
    0x8144: 'eeeeeb059',
    0x8145: 'eeeeabao',
    0x8146: 'fffffa159',
    0x8147: 'ffffbabn',
    0x8148: 'ooooob149',
    0x8149: 'ooooaaao',
    0x814a: 'llllla049',
    0x814b: 'llllbbbn',
    0x814c: 'iiiiib049',
    0x814d: 'iiiiabbn',
    0x814e: 'jjjjja149',
    0x814f: 'jjjjbaao',
    0x8150: 'XXXXXXab19',
    0x8151: 'kkkkbaao',
    0x8152: 'xxxxxa058',
    0x8153: 'hhhhabbn',
    0x8154: '=====b189',
    0x8155: 'mmmmbbbn',
    0x8156: '>>>>>a089',
    0x8157: 'nnnnaaao',
    0x8158: 'wwwwwb148',
    0x8159: 'ggggbabn',
    0x815a: 'ttttta048',
    0x815b: 'ddddabao',
    0x815c: 'qqqqqb048',
    0x815d: 'aaaabbao',
    0x815e: 'rrrrra148',
    0x815f: 'bbbbaabn',
    0x8160: 'ppppppab58',
    0x8161: 'ccccbbao',
    0x8162: 'pppppb048',
    0x8163: 'ppppbaan',
    0x8164: 'uuuuua048',
    0x8165: 'eeeebabn',
    0x8166: 'uuuuuubb58',
    0x8167: 'ffffabao',
    0x8168: '9999999babN',
    0x8169: 'oooobbbn',
    0x816a: '9999999babn',
    0x816b: 'llllaaao',
    0x816c: 'yyyyya058',
    0x816d: 'iiiibaao',
    0x816e: 'yyyyyybb48',
    0x816f: 'jjjjabbn',
    0x8170: 'hhhhhhab59',
    0x8171: 'kkkkabbn',
    0x8172: 'hhhhhb049',
    0x8173: 'hhhhbaao',
    0x8174: 'mmmmma049',
    0x8175: 'mmmmaaao',
    0x8176: 'mmmmmmbb59',
    0x8177: 'nnnnbbbn',
    0x8178: 'ddddddab49',
    0x8179: 'ggggabao',
    0x817a: 'dddddb059',
    0x817b: 'ddddbabn',
    0x817c: 'aaaaaa059',
    0x817d: 'aaaaaabn',
    0x817e: 'aaaaaabb49',
    0x817f: 'bbbbbbao',
    0x8180: 'aaaaaaaaaaabaaZ',
    0x8181: 'aaaaaaaaaaabacZ',
    0x8182: 'aaaaaaaaaaabaaX',
    0x8183: 'aaaaaaaaaaabaaY',
    0x8184: 'aaaaaaaaaaabaiZ',
    0x8185: 'aaaaaaaaaaabakZ',
    0x8186: 'aaaaaaaaaaabaiX',
    0x8187: 'aaaaaaaaaaabaiY',
    0x8188: 'aaaaaaaaaaabaaR',
    0x8189: 'aaaaaaaaaaabaaS',
    0x818a: 'aaaaaaaaaaabaaP',
    0x818b: 'aaaaaaaaaaabaaQ',
    0x818c: 'aaaaaaaaaaabaaV',
    0x818d: 'aaaaaaaaaaabaaW',
    0x818e: 'aaaaaaaaaaabaaT',
    0x818f: 'aaaaaaaaaaabaaU',
    0x8190: 'aaaaaaaaaaabaaJ',
    0x8191: 'aaaaaaaaaaabaaK',
    0x8192: 'aaaaaaaaaaabaaH',
    0x8193: 'aaaaaaaaaaabaaI',
    0x8194: 'aaaaaaaaaaabaaN',
    0x8195: 'aaaaaaaaaaabaaO',
    0x8196: 'aaaaaaaaaaabaaL',
    0x8197: 'aaaaaaaaaaabaaM',
    0x8198: 'aaaaaaaaaaabaaB',
    0x8199: 'aaaaaaaaaaabaaC',
    0x819a: 'aaaaaaaaaaabacA',
    0x819b: 'aaaaaaaaaaabaaA',
    0x819c: 'aaaaaaaaaaabaaF',
    0x819d: 'aaaabaao',
    0x819e: 'aaaaaaaaaaabaaD',
    0x819f: 'aaaaaaaaaaabaaE',
    0x81a0: 'aaaaaaaaaaabaaz',
    0x81a1: 'aaaaaaaaaaabacz',
    0x81a2: 'aaaaaaaaaaabaax',
    0x81a3: 'aaaaaaaaaaabaay',
    0x81a4: 'aaaaaaaaaaabaiz',
    0x81a5: 'aaaaaaaaaaabakz',
    0x81a6: 'aaaaaaaaaaabaix',
    0x81a7: 'aaaaaaaaaaabaiy',
    0x81a8: 'aaaaaaaaaaabaar',
    0x81a9: 'aaaaaaaaaaabaas',
    0x81aa: 'aaaaaaaaaaabaap',
    0x81ab: 'aaaaaaaaaaabaaq',
    0x81ac: 'aaaaaaaaaaabaav',
    0x81ad: 'aaaaaaaaaaabaaw',
    0x81ae: 'aaaaaaaaaaabaat',
    0x81af: 'aaaaaaaaaaabaau',
    0x81b0: 'aaaaaaaaaaabaaj',
    0x81b1: 'aaaaaaaaaaabaak',
    0x81b2: 'aaaaaaaaaaabaah',
    0x81b3: 'aaaaaaaaaaabaai',
    0x81b4: 'aaaaaaaaaaabaan',
    0x81b5: 'aaaaaaaaaaabaao',
    0x81b6: 'aaaaaaaaaaabaal',
    0x81b7: 'aaaaaaaaaaabaam',
    0x81b8: 'aaaaaaaaaaabaab',
    0x81b9: 'aaaaaaaaaaabaac',
    0x81ba: 'aaaaaaaaaaabaca',
    0x81bb: 'aaaaaaaaaaabaaa',
    0x81bc: 'aaaaab059',
    0x81bd: 'aaaaabbn',
    0x81be: 'aaaaaaaaaaabaad',
    0x81bf: 'aaaaaaaaaaabaae',
    0x81c0: 'aaaaaaaaaaabaW',
    0x81c1: 'aaaaaaaaaaabaU',
    0x81c2: 'aaaaaaaaaaabaS',
    0x81c3: 'aaaaaaaaaaabaQ',
    0x81c4: 'aaaaaaaaaaabkZ',
    0x81c5: 'aaaaaaaaaaabiY',
    0x81c6: 'aaaaaaaaaaabcZ',
    0x81c7: 'aaaaaaaaaaabaY',
    0x81c8: 'aaaaaaaaaaabaG',
    0x81c9: 'aaaaaaaaaaabaE',
    0x81ca: 'aaaaaaaaaaabaC',
    0x81cb: 'aaaaaaaaaaabaA',
    0x81cc: 'aaaaaaaaaaabaO',
    0x81cd: 'aaaaaaaaaaabaM',
    0x81ce: 'aaaaaaaaaaabaK',
    0x81cf: 'aaaaaaaaaaabaI',
    0x81d0: 'aaaaaaaaaaabaw',
    0x81d1: 'aaaaaaaaaaabau',
    0x81d2: 'aaaaaaaaaaabas',
    0x81d3: 'aaaaaaaaaaabaq',
    0x81d4: 'aaaaaaaaaaabkz',
    0x81d5: 'aaaaaaaaaaabiy',
    0x81d6: 'aaaaaaaaaaabcz',
    0x81d7: 'aaaaaaaaaaabay',
    0x81d8: 'aaaaaaaaaaabag',
    0x81d9: 'aaaaaaaaaaabae',
    0x81da: 'aaaaaaaaaaabac',
    0x81db: 'aaaaaaaaaaabaa',
    0x81dc: 'aaaaaaaaaaabao',
    0x81dd: 'aaaabbbn',
    0x81de: 'aaaaaaaaaaabak',
    0x81df: 'aaaaaaaaaaabai',
    0x81e0: 'aaaaaaaaaaabC',
    0x81e1: 'aaaaaaaaaaabG',
    0x81e2: 'aaaaaaaaaaabK',
    0x81e3: 'aaaaaaaaaaabO',
    0x81e4: 'aaaaaaaaaaabS',
    0x81e5: 'aaaaaaaaaaabW',
    0x81e6: 'aaaaaaaaaaadX',
    0x81e7: 'aaaaaaaaaaahZ',
    0x81e8: 'aaaaaaaaaaabc',
    0x81e9: 'aaaaaaaaaaabg',
    0x81ea: 'aaaaaaaaaaabk',
    0x81eb: 'aaaaaaaaaaabo',
    0x81ec: 'aaaaaaaaaaabs',
    0x81ed: 'aaaaaaaaaaabw',
    0x81ee: 'aaaaaaaaaaadx',
    0x81ef: 'aaaaaaaaaaahz',
    0x81f0: 'aaaaaaaaaaal',
    0x81f1: 'aaaaaaaaaaad',
    0x81f2: 'aaaaaaaaaaix',
    0x81f3: 'aaaaaaaaaaat',
    0x81f4: 'aaaaaaaaaaaL',
    0x81f5: 'aaaaaaaaaaaD',
    0x81f6: 'aaaaaaaaaaiX',
    0x81f7: 'aaaaaaaaaaaT',
    0x81f8: 'aaaaaaaaaaI',
    0x81f9: 'aaaaaaaaaaY',
    0x81fa: 'aaaaaaaaaai',
    0x81fb: 'aaaaaaaaaay',
    0x81fc: 'aaaaaa159',
    0x81fd: 'aaaaaaao',
    0x81fe: 'aaaaaaaa59',
    0x81ff: 'aaaaaaaaa3',
    0x8200: 'ccccca048',
    0x8201: 'ccccabal',
    0x8202: 'ccccccab59',
    0x8203: 'cccccccca3',
    0x8204: 'cccccccabbH',
    0x8205: 'cccccccabbX',
    0x8206: 'cccccccabbh',
    0x8207: 'cccccccabbx',
    0x8208: 'ccccccccaaaL',
    0x8209: 'ccccccccaaaD',
    0x820a: 'ccccccccaaiX',
    0x820b: 'ccccccccaaaT',
    0x820c: 'ccccccccaaal',
    0x820d: 'ccccccccaaad',
    0x820e: 'ccccccccaaix',
    0x820f: 'ccccccccaaat',
    0x8210: 'cccccccccbbar',
    0x8211: 'cccccccccbbav',
    0x8212: 'cccccccccbbaz',
    0x8213: 'cccccccccbbiz',
    0x8214: 'cccccccccbbab',
    0x8215: 'cccccccccbbaf',
    0x8216: 'cccccccccbbaj',
    0x8217: 'cccccccccbban',
    0x8218: 'cccccccccbbaR',
    0x8219: 'cccccccccbbaV',
    0x821a: 'cccccccccbbaZ',
    0x821b: 'cccccccccbbiZ',
    0x821c: 'cccccccccbbaB',
    0x821d: 'aaaababm',
    0x821e: 'cccccccccbbaJ',
    0x821f: 'bbbbabal',
    0x8220: 'sssssb059',
    0x8221: 'ccccbabm',
    0x8222: 'pppppa159',
    0x8223: 'ppppbbbl',
    0x8224: 'uuuuub159',
    0x8225: 'eeeebbal',
    0x8226: 'vvvvva059',
    0x8227: 'ffffaabm',
    0x8228: 'AAAAAAAAAbaaa',
    0x8229: 'oooobaal',
    0x822a: 'AAAAAAAAAbaai',
    0x822b: 'llllabbm',
    0x822c: 'yyyyyb149',
    0x822d: 'iiiibbbm',
    0x822e: 'zzzzza049',
    0x822f: 'jjjjaaal',
    0x8230: 'kkkkkb058',
    0x8231: 'kkkkaaal',
    0x8232: 'hhhhha158',
    0x8233: 'hhhhbbbm',
    0x8234: 'mmmmmb158',
    0x8235: 'mmmmabbm',
    0x8236: 'nnnnna058',
    0x8237: 'nnnnbaal',
    0x8238: 'gggggb048',
    0x8239: 'ggggaabm',
    0x823a: 'ddddda148',
    0x823b: 'ddddbbal',
    0x823c: 'aaaaab148',
    0x823d: 'aaaaabal',
    0x823e: 'bbbbba048',
    0x823f: 'bbbbbabm',
    0x8240: 'cccccb148',
    0x8241: 'ccccaabm',
    0x8242: 'CCCCCCbb08',
    0x8243: 'ppppabbl',
    0x8244: 'eeeeeb048',
    0x8245: 'eeeeabal',
    0x8246: 'fffffa148',
    0x8247: 'ffffbabm',
    0x8248: 'ooooob158',
    0x8249: 'ooooaaal',
    0x824a: 'llllla058',
    0x824b: 'llllbbbm',
    0x824c: 'iiiiib058',
    0x824d: 'iiiiabbm',
    0x824e: 'jjjjja158',
    0x824f: 'jjjjbaal',
    0x8250: 'XXXXXXaa19',
    0x8251: 'kkkkbaal',
    0x8252: 'xxxxxa049',
    0x8253: 'hhhhabbm',
    0x8254: '=====b198',
    0x8255: 'mmmmbbbm',
    0x8256: '>>>>>a098',
    0x8257: 'nnnnaaal',
    0x8258: 'wwwwwb159',
    0x8259: 'ggggbabm',
    0x825a: 'ttttta059',
    0x825b: 'ddddabal',
    0x825c: 'qqqqqb059',
    0x825d: 'aaaabbal',
    0x825e: 'rrrrra159',
    0x825f: 'bbbbaabm',
    0x8260: 'ppppppaa58',
    0x8261: 'ccccbbal',
    0x8262: 'pppppb059',
    0x8263: 'ppppbaam',
    0x8264: 'uuuuua059',
    0x8265: 'eeeebabm',
    0x8266: 'uuuuuuba58',
    0x8267: 'ffffabal',
    0x8268: '00000000bbdX',
    0x8269: 'oooobbbm',
    0x826a: '00000000bbbK',
    0x826b: 'llllaaal',
    0x826c: 'yyyyya049',
    0x826d: 'iiiibaal',
    0x826e: 'yyyyyyba48',
    0x826f: 'jjjjabbm',
    0x8270: 'hhhhhhaa59',
    0x8271: 'kkkkabbm',
    0x8272: 'hhhhhb058',
    0x8273: 'hhhhbaal',
    0x8274: 'mmmmma058',
    0x8275: 'mmmmaaal',
    0x8276: 'mmmmmmba59',
    0x8277: 'nnnnbbbm',
    0x8278: 'ddddddaa49',
    0x8279: 'ggggabal',
    0x827a: 'dddddb048',
    0x827b: 'ddddbabm',
    0x827c: 'aaaaaa048',
    0x827d: 'aaaaaabm',
    0x827e: 'aaaaaaba49',
    0x827f: 'bbbbbbal',
    0x8280: 'ccccca148',
    0x8281: 'ccccabbm',
    0x8282: 'ccccccba49',
    0x8283: 'ppppaabl',
    0x8284: 'eeeeea048',
    0x8285: 'eeeeaaal',
    0x8286: 'eeeeeebb59',
    0x8287: 'ffffbbbm',
    0x8288: 'llllllab49',
    0x8289: 'ooooabal',
    0x828a: 'lllllb058',
    0x828b: 'llllbabm',
    0x828c: 'iiiiia058',
    0x828d: 'iiiiaabm',
    0x828e: 'iiiiiibb49',
    0x828f: 'jjjjbbal',
    0x8290: 'xxxxxxab58',
    0x8291: 'kkkkbbal',
    0x8292: 'xxxxxb049',
    0x8293: 'hhhhaabm',
    0x8294: '8888888aabn',
    0x8295: 'mmmmbabm',
    0x8296: '8888888aabN',
    0x8297: 'nnnnabal',
    0x8298: 'ttttttab48',
    0x8299: 'ggggbbbm',
    0x829a: 'tttttb059',
    0x829b: 'ddddaaal',
    0x829c: 'qqqqqa059',
    0x829d: 'aaaabaal',
    0x829e: 'qqqqqqbb48',
    0x829f: 'bbbbabbm',
    0x82a0: 'sssssb159',
    0x82a1: 'ccccbaal',
    0x82a2: 'pppppa059',
    0x82a3: 'ppppbbam',
    0x82a4: 'uuuuub059',
    0x82a5: 'eeeebbbm',
    0x82a6: 'vvvvva159',
    0x82a7: 'ffffaaal',
    0x82a8: '?????b098',
    0x82a9: 'oooobabm',
    0x82aa: '<<<<<a198',
    0x82ab: 'llllabal',
    0x82ac: 'yyyyyb049',
    0x82ad: 'iiiibbal',
    0x82ae: 'zzzzza149',
    0x82af: 'jjjjaabm',
    0x82b0: 'kkkkkb158',
    0x82b1: 'kkkkaabm',
    0x82b2: 'hhhhha058',
    0x82b3: 'hhhhbbal',
    0x82b4: 'mmmmmb058',
    0x82b5: 'mmmmabal',
    0x82b6: 'nnnnna158',
    0x82b7: 'nnnnbabm',
    0x82b8: 'gggggb148',
    0x82b9: 'ggggaaal',
    0x82ba: 'ddddda048',
    0x82bb: 'ddddbbbm',
    0x82bc: 'aaaaab048',
    0x82bd: 'aaaaabbm',
    0x82be: 'bbbbba148',
    0x82bf: 'bbbbbaal',
    0x82c0: 'cccccb048',
    0x82c1: 'ccccaaal',
    0x82c2: 'CCCCCCaa18',
    0x82c3: 'ppppabam',
    0x82c4: 'eeeeeb148',
    0x82c5: 'eeeeabbm',
    0x82c6: 'fffffa048',
    0x82c7: 'ffffbaal',
    0x82c8: 'ooooob058',
    0x82c9: 'ooooaabm',
    0x82ca: 'llllla158',
    0x82cb: 'llllbbal',
    0x82cc: 'iiiiib158',
    0x82cd: 'iiiiabal',
    0x82ce: 'jjjjja058',
    0x82cf: 'jjjjbabm',
    0x82d0: 'XXXXXXbb09',
    0x82d1: 'kkkkbabm',
    0x82d2: 'xxxxxa149',
    0x82d3: 'hhhhabal',
    0x82d4: 'XXXXXXXaabj',
    0x82d5: 'mmmmbbal',
    0x82d6: 'XXXXXXXaabJ',
    0x82d7: 'nnnnaabm',
    0x82d8: 'wwwwwb059',
    0x82d9: 'ggggbaal',
    0x82da: 'ttttta159',
    0x82db: 'ddddabbm',
    0x82dc: 'qqqqqb159',
    0x82dd: 'aaaabbbm',
    0x82de: 'rrrrra059',
    0x82df: 'bbbbaaal',
    0x82e0: 'ppppppbb48',
    0x82e1: 'ccccbbbm',
    0x82e2: 'pppppb159',
    0x82e3: 'ppppbabl',
    0x82e4: 'pppppppaabi',
    0x82e5: 'eeeebaal',
    0x82e6: 'pppppppaabI',
    0x82e7: 'ffffabbm',
    0x82e8: 'yyyyyyybaaH',
    0x82e9: 'oooobbal',
    0x82ea: 'yyyyyyybaah',
    0x82eb: 'llllaabm',
    0x82ec: 'yyyyya149',
    0x82ed: 'iiiibabm',
    0x82ee: 'yyyyyyab58',
    0x82ef: 'jjjjabal',
    0x82f0: 'hhhhhhbb49',
    0x82f1: 'kkkkabal',
    0x82f2: 'hhhhhb158',
    0x82f3: 'hhhhbabm',
    0x82f4: 'hhhhhhhaabh',
    0x82f5: 'hhhhhhhaabx',
    0x82f6: 'hhhhhhhaabH',
    0x82f7: 'hhhhhhhaabX',
    0x82f8: 'aaaaaaabaaI',
    0x82f9: 'aaaaaaabaaY',
    0x82fa: 'aaaaaaabaai',
    0x82fb: 'aaaaaaabaay',
    0x82fc: 'aaaaaa148',
    0x82fd: 'aaaaaaal',
    0x82fe: 'aaaaaaab59',
    0x82ff: 'bbbbbbbm',
    0x8300: 'ccccca058',
    0x8301: 'ccccabam',
    0x8302: 'ccccccaa49',
    0x8303: 'eeeeeeebbbx',
    0x8304: 'cccccccbaaI',
    0x8305: 'cccccccbaaY',
    0x8306: 'cccccccbaai',
    0x8307: 'cccccccbaay',
    0x8308: 'iiiiiiibbaI',
    0x8309: 'iiiiiiibbaY',
    0x830a: 'iiiiiiibbai',
    0x830b: 'iiiiiiibbay',
    0x830c: 'iiiiia148',
    0x830d: 'iiiiaaam',
    0x830e: 'iiiiiiab49',
    0x830f: 'jjjjbbbl',
    0x8310: 'xxxxxxbb58',
    0x8311: 'kkkkbbbl',
    0x8312: 'xxxxxb159',
    0x8313: 'hhhhaaam',
    0x8314: 'xxxxxxxabbi',
    0x8315: 'mmmmbaam',
    0x8316: 'xxxxxxxabbI',
    0x8317: 'nnnnabbl',
    0x8318: 'qqqqqqqbbaH',
    0x8319: 'ggggbbam',
    0x831a: 'qqqqqqqbbah',
    0x831b: 'ddddaabl',
    0x831c: 'qqqqqa149',
    0x831d: 'aaaababl',
    0x831e: 'qqqqqqab48',
    0x831f: 'bbbbabam',
    0x8320: 'sssssb049',
    0x8321: 'ccccbabl',
    0x8322: 'pppppa149',
    0x8323: 'ppppbbbm',
    0x8324: 'uuuuub149',
    0x8325: 'eeeebbam',
    0x8326: 'vvvvva049',
    0x8327: 'ffffaabl',
    0x8328: 'YYYYYYYbbaK',
    0x8329: 'oooobaam',
    0x832a: 'YYYYYYYbbak',
    0x832b: 'llllabbl',
    0x832c: 'yyyyyb159',
    0x832d: 'iiiibbbl',
    0x832e: 'zzzzza059',
    0x832f: 'jjjjaaam',
    0x8330: 'kkkkkb048',
    0x8331: 'kkkkaaam',
    0x8332: 'hhhhha148',
    0x8333: 'hhhhbbbl',
    0x8334: 'mmmmmb148',
    0x8335: 'mmmmabbl',
    0x8336: 'nnnnna048',
    0x8337: 'nnnnbaam',
    0x8338: 'gggggb058',
    0x8339: 'ggggaabl',
    0x833a: 'ddddda158',
    0x833b: 'ddddbbam',
    0x833c: 'aaaaab158',
    0x833d: 'aaaaabam',
    0x833e: 'bbbbba058',
    0x833f: 'bbbbbabl',
    0x8340: 'cccccb158',
    0x8341: 'ccccaabl',
    0x8342: 'CCCCCCba18',
    0x8343: 'ppppabbm',
    0x8344: 'eeeeeb058',
    0x8345: 'eeeeabam',
    0x8346: 'fffffa158',
    0x8347: 'ffffbabl',
    0x8348: 'ooooob148',
    0x8349: 'ooooaaam',
    0x834a: 'llllla048',
    0x834b: 'llllbbbl',
    0x834c: 'iiiiib048',
    0x834d: 'iiiiabbl',
    0x834e: 'jjjjja148',
    0x834f: 'jjjjbaam',
    0x8350: 'XXXXXXab09',
    0x8351: 'kkkkbaam',
    0x8352: 'xxxxxa059',
    0x8353: 'hhhhabbl',
    0x8354: '=====b188',
    0x8355: 'mmmmbbbl',
    0x8356: '>>>>>a088',
    0x8357: 'nnnnaaam',
    0x8358: 'wwwwwb149',
    0x8359: 'ggggbabl',
    0x835a: 'ttttta049',
    0x835b: 'ddddabam',
    0x835c: 'qqqqqb049',
    0x835d: 'aaaabbam',
    0x835e: 'rrrrra149',
    0x835f: 'bbbbaabl',
    0x8360: 'ppppppab48',
    0x8361: 'ccccbbam',
    0x8362: 'pppppb049',
    0x8363: 'ppppbaal',
    0x8364: 'uuuuua049',
    0x8365: 'eeeebabl',
    0x8366: 'uuuuuubb48',
    0x8367: 'ffffabam',
    0x8368: '9999999bbaO',
    0x8369: 'oooobbbl',
    0x836a: '9999999bbao',
    0x836b: 'llllaaam',
    0x836c: 'yyyyya059',
    0x836d: 'iiiibaam',
    0x836e: 'yyyyyybb58',
    0x836f: 'jjjjabbl',
    0x8370: 'hhhhhhab49',
    0x8371: 'kkkkabbl',
    0x8372: 'hhhhhb048',
    0x8373: 'hhhhbaam',
    0x8374: 'mmmmma048',
    0x8375: 'mmmmaaam',
    0x8376: 'mmmmmmbb49',
    0x8377: 'nnnnbbbl',
    0x8378: 'ddddddab59',
    0x8379: 'ggggabam',
    0x837a: 'dddddb058',
    0x837b: 'ddddbabl',
    0x837c: 'aaaaaa058',
    0x837d: 'aaaaaabl',
    0x837e: 'aaaaaabb59',
    0x837f: 'bbbbbbam',
    0x8380: 'ccccca158',
    0x8381: 'ccccabbl',
    0x8382: 'ccccccbb59',
    0x8383: 'ppppaabm',
    0x8384: 'eeeeea058',
    0x8385: 'eeeeaaam',
    0x8386: 'eeeeeeba49',
    0x8387: 'ffffbbbl',
    0x8388: 'llllllaa59',
    0x8389: 'ooooabam',
    0x838a: 'lllllb048',
    0x838b: 'llllbabl',
    0x838c: 'iiiiia048',
    0x838d: 'iiiiaabl',
    0x838e: 'iiiiiiba59',
    0x838f: 'jjjjbbam',
    0x8390: 'xxxxxxaa48',
    0x8391: 'kkkkbbam',
    0x8392: 'xxxxxb059',
    0x8393: 'hhhhaabl',
    0x8394: '11111111bbbK',
    0x8395: 'mmmmbabl',
    0x8396: '11111111bbdX',
    0x8397: 'nnnnabam',
    0x8398: 'ttttttaa58',
    0x8399: 'ggggbbbl',
    0x839a: 'tttttb049',
    0x839b: 'ddddaaam',
    0x839c: 'qqqqqa049',
    0x839d: 'aaaabaam',
    0x839e: 'qqqqqqba58',
    0x839f: 'bbbbabbl',
    0x83a0: 'sssssb149',
    0x83a1: 'ccccbaam',
    0x83a2: 'pppppa049',
    0x83a3: 'ppppbbal',
    0x83a4: 'uuuuub049',
    0x83a5: 'eeeebbbl',
    0x83a6: 'vvvvva149',
    0x83a7: 'ffffaaam',
    0x83a8: '?????b088',
    0x83a9: 'oooobabl',
    0x83aa: '<<<<<a188',
    0x83ab: 'llllabam',
    0x83ac: 'yyyyyb059',
    0x83ad: 'iiiibbam',
    0x83ae: 'zzzzza159',
    0x83af: 'jjjjaabl',
    0x83b0: 'kkkkkb148',
    0x83b1: 'kkkkaabl',
    0x83b2: 'hhhhha048',
    0x83b3: 'hhhhbbam',
    0x83b4: 'mmmmmb048',
    0x83b5: 'mmmmabam',
    0x83b6: 'nnnnna148',
    0x83b7: 'nnnnbabl',
    0x83b8: 'gggggb158',
    0x83b9: 'ggggaaam',
    0x83ba: 'ddddda058',
    0x83bb: 'ddddbbbl',
    0x83bc: 'aaaaab058',
    0x83bd: 'aaaaabbl',
    0x83be: 'bbbbba158',
    0x83bf: 'bbbbbaam',
    0x83c0: 'cccccb058',
    0x83c1: 'ccccaaam',
    0x83c2: 'CCCCCCab08',
    0x83c3: 'ppppabal',
    0x83c4: 'eeeeeb158',
    0x83c5: 'eeeeabbl',
    0x83c6: 'fffffa058',
    0x83c7: 'ffffbaam',
    0x83c8: 'ooooob048',
    0x83c9: 'ooooaabl',
    0x83ca: 'llllla148',
    0x83cb: 'llllbbam',
    0x83cc: 'iiiiib148',
    0x83cd: 'iiiiabam',
    0x83ce: 'jjjjja048',
    0x83cf: 'jjjjbabl',
    0x83d0: 'QQQQQQQQbbbo',
    0x83d1: 'kkkkbabl',
    0x83d2: 'xxxxxa159',
    0x83d3: 'hhhhabam',
    0x83d4: 'QQQQQQQQbbbO',
    0x83d5: 'mmmmbbam',
    0x83d6: 'QQQQQQQQbbhZ',
    0x83d7: 'nnnnaabl',
    0x83d8: 'wwwwwb049',
    0x83d9: 'ggggbaam',
    0x83da: 'ttttta149',
    0x83db: 'ddddabbl',
    0x83dc: 'qqqqqb149',
    0x83dd: 'aaaabbbl',
    0x83de: 'rrrrra049',
    0x83df: 'bbbbaaam',
    0x83e0: 'ppppppba58',
    0x83e1: 'ccccbbbl',
    0x83e2: 'pppppb149',
    0x83e3: 'ppppbabm',
    0x83e4: 'pppppppbbah',
    0x83e5: 'eeeebaam',
    0x83e6: 'pppppppbbaH',
    0x83e7: 'ffffabbl',
    0x83e8: 'ppppppppabiY',
    0x83e9: 'oooobbam',
    0x83ea: 'ppppppppabaM',
    0x83eb: 'llllaabl',
    0x83ec: 'ppppppppabiy',
    0x83ed: 'iiiibabl',
    0x83ee: 'ppppppppabam',
    0x83ef: 'jjjjabam',
    0x83f0: 'aaaaaaaabbbm',
    0x83f1: 'aaaaaaaabbbe',
    0x83f2: 'aaaaaaaabbhx',
    0x83f3: 'aaaaaaaabbbu',
    0x83f4: 'aaaaaaaabbbM',
    0x83f5: 'aaaaaaaabbbE',
    0x83f6: 'aaaaaaaabbhX',
    0x83f7: 'aaaaaaaabbbU',
    0x83f8: 'aaaaaaaabbH',
    0x83f9: 'aaaaaaaabbX',
    0x83fa: 'aaaaaaaabbh',
    0x83fb: 'aaaaaaaabbx',
    0x83fc: 'aaaaaa158',
    0x83fd: 'aaaaaaam',
    0x83fe: 'aaaaaaaa49',
    0x83ff: 'aaaaaaaab2',
    0x8400: 'ccccca009',
    0x8401: 'ccccabaj',
    0x8402: 'ccccccab48',
    0x8403: 'ccccccccb3',
    0x8404: 'cccccccaabH',
    0x8405: 'cccccccaabX',
    0x8406: 'cccccccaabh',
    0x8407: 'cccccccaabx',
    0x8408: 'ccccccccbaaL',
    0x8409: 'ccccccccbaaD',
    0x840a: 'ccccccccbaiX',
    0x840b: 'ccccccccbaaT',
    0x840c: 'ccccccccbaal',
    0x840d: 'ccccccccbaad',
    0x840e: 'ccccccccbaix',
    0x840f: 'ccccccccbaat',
    0x8410: 'qqqqqqqqbbam',
    0x8411: 'kkkkbbbk',
    0x8412: 'qqqqqqqqbbiy',
    0x8413: 'hhhhaaaj',
    0x8414: 'qqqqqqqqbbaM',
    0x8415: 'mmmmbaaj',
    0x8416: 'qqqqqqqqbbiY',
    0x8417: 'nnnnabbk',
    0x8418: 'qqqqqqqabbI',
    0x8419: 'ggggbbaj',
    0x841a: 'qqqqqqqabbi',
    0x841b: 'ddddaabk',
    0x841c: 'qqqqqa118',
    0x841d: 'aaaababk',
    0x841e: 'qqqqqqaa49',
    0x841f: 'bbbbabaj',
    0x8420: 'sssssb018',
    0x8421: 'ccccbabk',
    0x8422: 'pppppa118',
    0x8423: 'ppppbbbj',
    0x8424: 'uuuuub118',
    0x8425: 'eeeebbaj',
    0x8426: 'vvvvva018',
    0x8427: 'ffffaabk',
    0x8428: 'PPPPPPPPabhZ',
    0x8429: 'oooobaaj',
    0x842a: 'PPPPPPPPabbO',
    0x842b: 'llllabbk',
    0x842c: 'yyyyyb108',
    0x842d: 'iiiibbbk',
    0x842e: 'zzzzza008',
    0x842f: 'jjjjaaaj',
    0x8430: 'kkkkkb019',
    0x8431: 'kkkkaaaj',
    0x8432: 'hhhhha119',
    0x8433: 'hhhhbbbk',
    0x8434: 'mmmmmb119',
    0x8435: 'mmmmabbk',
    0x8436: 'nnnnna019',
    0x8437: 'nnnnbaaj',
    0x8438: 'gggggb009',
    0x8439: 'ggggaabk',
    0x843a: 'ddddda109',
    0x843b: 'ddddbbaj',
    0x843c: 'aaaaab109',
    0x843d: 'aaaaabaj',
    0x843e: 'bbbbba009',
    0x843f: 'bbbbbabk',
    0x8440: 'cccccb109',
    0x8441: 'ccccaabk',
    0x8442: 'CCCCCCbb19',
    0x8443: 'ppppabbj',
    0x8444: 'eeeeeb009',
    0x8445: 'eeeeabaj',
    0x8446: 'fffffa109',
    0x8447: 'ffffbabk',
    0x8448: 'ooooob119',
    0x8449: 'ooooaaaj',
    0x844a: 'llllla019',
    0x844b: 'llllbbbk',
    0x844c: 'iiiiib019',
    0x844d: 'iiiiabbk',
    0x844e: 'jjjjja119',
    0x844f: 'jjjjbaaj',
    0x8450: 'XXXXXXaa08',
    0x8451: 'kkkkbaaj',
    0x8452: 'xxxxxa008',
    0x8453: 'hhhhabbk',
    0x8454: '=====b999',
    0x8455: 'mmmmbbbk',
    0x8456: '>>>>>a899',
    0x8457: 'nnnnaaaj',
    0x8458: 'wwwwwb118',
    0x8459: 'ggggbabk',
    0x845a: 'ttttta018',
    0x845b: 'ddddabaj',
    0x845c: 'qqqqqb018',
    0x845d: 'aaaabbaj',
    0x845e: 'rrrrra118',
    0x845f: 'bbbbaabk',
    0x8460: 'ppppppaa49',
    0x8461: 'ccccbbaj',
    0x8462: 'pppppb018',
    0x8463: 'ppppbaak',
    0x8464: 'uuuuua018',
    0x8465: 'eeeebabk',
    0x8466: 'uuuuuuba49',
    0x8467: 'ffffabaj',
    0x8468: '00000000abdX',
    0x8469: 'oooobbbk',
    0x846a: '00000000abbK',
    0x846b: 'llllaaaj',
    0x846c: 'yyyyya008',
    0x846d: 'iiiibaaj',
    0x846e: 'yyyyyyba59',
    0x846f: 'jjjjabbk',
    0x8470: 'hhhhhhaa48',
    0x8471: 'kkkkabbk',
    0x8472: 'hhhhhb019',
    0x8473: 'hhhhbaaj',
    0x8474: 'mmmmma019',
    0x8475: 'mmmmaaaj',
    0x8476: 'mmmmmmba48',
    0x8477: 'nnnnbbbk',
    0x8478: 'ddddddaa58',
    0x8479: 'ggggabaj',
    0x847a: 'dddddb009',
    0x847b: 'ddddbabk',
    0x847c: 'aaaaaa009',
    0x847d: 'aaaaaabk',
    0x847e: 'aaaaaaba58',
    0x847f: 'bbbbbbaj',
    0x8480: 'ccccca109',
    0x8481: 'ccccabbk',
    0x8482: 'ccccccba58',
    0x8483: 'ppppaabj',
    0x8484: 'eeeeea009',
    0x8485: 'eeeeaaaj',
    0x8486: 'eeeeeebb48',
    0x8487: 'ffffbbbk',
    0x8488: 'llllllab58',
    0x8489: 'ooooabaj',
    0x848a: 'lllllb019',
    0x848b: 'llllbabk',
    0x848c: 'iiiiia019',
    0x848d: 'iiiiaabk',
    0x848e: 'iiiiiibb58',
    0x848f: 'jjjjbbaj',
    0x8490: 'xxxxxxab49',
    0x8491: 'kkkkbbaj',
    0x8492: 'xxxxxb008',
    0x8493: 'hhhhaabk',
    0x8494: '8888888abbn',
    0x8495: 'mmmmbabk',
    0x8496: '8888888abbN',
    0x8497: 'nnnnabaj',
    0x8498: 'ttttttab59',
    0x8499: 'ggggbbbk',
    0x849a: 'tttttb018',
    0x849b: 'ddddaaaj',
    0x849c: 'qqqqqa018',
    0x849d: 'aaaabaaj',
    0x849e: 'qqqqqqbb59',
    0x849f: 'bbbbabbk',
    0x84a0: 'sssssb118',
    0x84a1: 'ccccbaaj',
    0x84a2: 'pppppa018',
    0x84a3: 'ppppbbak',
    0x84a4: 'uuuuub018',
    0x84a5: 'eeeebbbk',
    0x84a6: 'vvvvva118',
    0x84a7: 'ffffaaaj',
    0x84a8: '?????b899',
    0x84a9: 'oooobabk',
    0x84aa: '<<<<<a999',
    0x84ab: 'llllabaj',
    0x84ac: 'yyyyyb008',
    0x84ad: 'iiiibbaj',
    0x84ae: 'zzzzza108',
    0x84af: 'jjjjaabk',
    0x84b0: 'kkkkkb119',
    0x84b1: 'kkkkaabk',
    0x84b2: 'hhhhha019',
    0x84b3: 'hhhhbbaj',
    0x84b4: 'mmmmmb019',
    0x84b5: 'mmmmabaj',
    0x84b6: 'nnnnna119',
    0x84b7: 'nnnnbabk',
    0x84b8: 'gggggb109',
    0x84b9: 'ggggaaaj',
    0x84ba: 'ddddda009',
    0x84bb: 'ddddbbbk',
    0x84bc: 'aaaaab009',
    0x84bd: 'aaaaabbk',
    0x84be: 'bbbbba109',
    0x84bf: 'bbbbbaaj',
    0x84c0: 'cccccb009',
    0x84c1: 'ccccaaaj',
    0x84c2: 'CCCCCCaa09',
    0x84c3: 'ppppabak',
    0x84c4: 'eeeeeb109',
    0x84c5: 'eeeeabbk',
    0x84c6: 'fffffa009',
    0x84c7: 'ffffbaaj',
    0x84c8: 'ooooob019',
    0x84c9: 'ooooaabk',
    0x84ca: 'llllla119',
    0x84cb: 'llllbbaj',
    0x84cc: 'iiiiib119',
    0x84cd: 'iiiiabaj',
    0x84ce: 'jjjjja019',
    0x84cf: 'jjjjbabk',
    0x84d0: 'XXXXXXbb18',
    0x84d1: 'kkkkbabk',
    0x84d2: 'xxxxxa108',
    0x84d3: 'hhhhabaj',
    0x84d4: 'XXXXXXXabbj',
    0x84d5: 'mmmmbbaj',
    0x84d6: 'XXXXXXXabbJ',
    0x84d7: 'nnnnaabk',
    0x84d8: 'wwwwwb018',
    0x84d9: 'ggggbaaj',
    0x84da: 'ttttta118',
    0x84db: 'ddddabbk',
    0x84dc: 'qqqqqb118',
    0x84dd: 'aaaabbbk',
    0x84de: 'rrrrra018',
    0x84df: 'bbbbaaaj',
    0x84e0: 'ppppppbb59',
    0x84e1: 'ccccbbbk',
    0x84e2: 'pppppb118',
    0x84e3: 'ppppbabj',
    0x84e4: 'pppppppabbi',
    0x84e5: 'eeeebaaj',
    0x84e6: 'pppppppabbI',
    0x84e7: 'ffffabbk',
    0x84e8: 'yyyyyyybbaH',
    0x84e9: 'oooobbaj',
    0x84ea: 'yyyyyyybbah',
    0x84eb: 'llllaabk',
    0x84ec: 'yyyyya108',
    0x84ed: 'iiiibabk',
    0x84ee: 'yyyyyyab49',
    0x84ef: 'jjjjabaj',
    0x84f0: 'hhhhhhbb58',
    0x84f1: 'kkkkabaj',
    0x84f2: 'hhhhhb119',
    0x84f3: 'hhhhbabk',
    0x84f4: 'hhhhhhhabbh',
    0x84f5: 'hhhhhhhabbx',
    0x84f6: 'hhhhhhhabbH',
    0x84f7: 'hhhhhhhabbX',
    0x84f8: 'aaaaaaabbaI',
    0x84f9: 'aaaaaaabbaY',
    0x84fa: 'aaaaaaabbai',
    0x84fb: 'aaaaaaabbay',
    0x84fc: 'aaaaaa109',
    0x84fd: 'aaaaaaaj',
    0x84fe: 'aaaaaaab48',
    0x84ff: 'bbbbbbbk',
    0x8500: 'ccccca019',
    0x8501: 'ccccabak',
    0x8502: 'ccccccaa58',
    0x8503: 'eeeeeeebabx',
    0x8504: 'cccccccbbaI',
    0x8505: 'cccccccbbaY',
    0x8506: 'cccccccbbai',
    0x8507: 'cccccccbbay',
    0x8508: 'iiiiiiibaaI',
    0x8509: 'iiiiiiibaaY',
    0x850a: 'iiiiiiibaai',
    0x850b: 'iiiiiiibaay',
    0x850c: 'iiiiia109',
    0x850d: 'iiiiaaak',
    0x850e: 'iiiiiiab58',
    0x850f: 'jjjjbbbj',
    0x8510: 'xxxxxxbb49',
    0x8511: 'kkkkbbbj',
    0x8512: 'xxxxxb118',
    0x8513: 'hhhhaaak',
    0x8514: 'xxxxxxxaabi',
    0x8515: 'mmmmbaak',
    0x8516: 'xxxxxxxaabI',
    0x8517: 'nnnnabbj',
    0x8518: 'qqqqqqqbaaH',
    0x8519: 'ggggbbak',
    0x851a: 'qqqqqqqbaah',
    0x851b: 'ddddaabj',
    0x851c: 'qqqqqa108',
    0x851d: 'aaaababj',
    0x851e: 'qqqqqqab59',
    0x851f: 'bbbbabak',
    0x8520: 'sssssb008',
    0x8521: 'ccccbabj',
    0x8522: 'pppppa108',
    0x8523: 'ppppbbbk',
    0x8524: 'uuuuub108',
    0x8525: 'eeeebbak',
    0x8526: 'vvvvva008',
    0x8527: 'ffffaabj',
    0x8528: 'YYYYYYYbaaK',
    0x8529: 'oooobaak',
    0x852a: 'YYYYYYYbaak',
    0x852b: 'llllabbj',
    0x852c: 'yyyyyb118',
    0x852d: 'iiiibbbj',
    0x852e: 'zzzzza018',
    0x852f: 'jjjjaaak',
    0x8530: 'kkkkkb009',
    0x8531: 'kkkkaaak',
    0x8532: 'hhhhha109',
    0x8533: 'hhhhbbbj',
    0x8534: 'mmmmmb109',
    0x8535: 'mmmmabbj',
    0x8536: 'nnnnna009',
    0x8537: 'nnnnbaak',
    0x8538: 'gggggb019',
    0x8539: 'ggggaabj',
    0x853a: 'ddddda119',
    0x853b: 'ddddbbak',
    0x853c: 'aaaaab119',
    0x853d: 'aaaaabak',
    0x853e: 'bbbbba019',
    0x853f: 'bbbbbabj',
    0x8540: 'cccccb119',
    0x8541: 'ccccaabj',
    0x8542: 'CCCCCCba09',
    0x8543: 'ppppabbk',
    0x8544: 'eeeeeb019',
    0x8545: 'eeeeabak',
    0x8546: 'fffffa119',
    0x8547: 'ffffbabj',
    0x8548: 'ooooob109',
    0x8549: 'ooooaaak',
    0x854a: 'llllla009',
    0x854b: 'llllbbbj',
    0x854c: 'iiiiib009',
    0x854d: 'iiiiabbj',
    0x854e: 'jjjjja109',
    0x854f: 'jjjjbaak',
    0x8550: 'XXXXXXab18',
    0x8551: 'kkkkbaak',
    0x8552: 'xxxxxa018',
    0x8553: 'hhhhabbj',
    0x8554: '=====b989',
    0x8555: 'mmmmbbbj',
    0x8556: '>>>>>a889',
    0x8557: 'nnnnaaak',
    0x8558: 'wwwwwb108',
    0x8559: 'ggggbabj',
    0x855a: 'ttttta008',
    0x855b: 'ddddabak',
    0x855c: 'qqqqqb008',
    0x855d: 'aaaabbak',
    0x855e: 'rrrrra108',
    0x855f: 'bbbbaabj',
    0x8560: 'ppppppab59',
    0x8561: 'ccccbbak',
    0x8562: 'pppppb008',
    0x8563: 'ppppbaaj',
    0x8564: 'uuuuua008',
    0x8565: 'eeeebabj',
    0x8566: 'uuuuuubb59',
    0x8567: 'ffffabak',
    0x8568: '9999999baaO',
    0x8569: 'oooobbbj',
    0x856a: '9999999baao',
    0x856b: 'llllaaak',
    0x856c: 'yyyyya018',
    0x856d: 'iiiibaak',
    0x856e: 'yyyyyybb49',
    0x856f: 'jjjjabbj',
    0x8570: 'hhhhhhab58',
    0x8571: 'kkkkabbj',
    0x8572: 'hhhhhb009',
    0x8573: 'hhhhbaak',
    0x8574: 'mmmmma009',
    0x8575: 'mmmmaaak',
    0x8576: 'mmmmmmbb58',
    0x8577: 'nnnnbbbj',
    0x8578: 'ddddddab48',
    0x8579: 'ggggabak',
    0x857a: 'dddddb019',
    0x857b: 'ddddbabj',
    0x857c: 'aaaaaa019',
    0x857d: 'aaaaaabj',
    0x857e: 'aaaaaabb48',
    0x857f: 'bbbbbbak',
    0x8580: 'ccccca119',
    0x8581: 'ccccabbj',
    0x8582: 'ccccccbb48',
    0x8583: 'ppppaabk',
    0x8584: 'eeeeea019',
    0x8585: 'eeeeaaak',
    0x8586: 'eeeeeeba58',
    0x8587: 'ffffbbbj',
    0x8588: 'llllllaa48',
    0x8589: 'ooooabak',
    0x858a: 'lllllb009',
    0x858b: 'llllbabj',
    0x858c: 'iiiiia009',
    0x858d: 'iiiiaabj',
    0x858e: 'iiiiiiba48',
    0x858f: 'jjjjbbak',
    0x8590: 'xxxxxxaa59',
    0x8591: 'kkkkbbak',
    0x8592: 'xxxxxb018',
    0x8593: 'hhhhaabj',
    0x8594: '11111111abbK',
    0x8595: 'mmmmbabj',
    0x8596: '11111111abdX',
    0x8597: 'nnnnabak',
    0x8598: 'ttttttaa49',
    0x8599: 'ggggbbbj',
    0x859a: 'tttttb008',
    0x859b: 'ddddaaak',
    0x859c: 'qqqqqa008',
    0x859d: 'aaaabaak',
    0x859e: 'qqqqqqba49',
    0x859f: 'bbbbabbj',
    0x85a0: 'sssssb108',
    0x85a1: 'ccccbaak',
    0x85a2: 'pppppa008',
    0x85a3: 'ppppbbaj',
    0x85a4: 'uuuuub008',
    0x85a5: 'eeeebbbj',
    0x85a6: 'vvvvva108',
    0x85a7: 'ffffaaak',
    0x85a8: '?????b889',
    0x85a9: 'oooobabj',
    0x85aa: '<<<<<a989',
    0x85ab: 'llllabak',
    0x85ac: 'yyyyyb018',
    0x85ad: 'iiiibbak',
    0x85ae: 'zzzzza118',
    0x85af: 'jjjjaabj',
    0x85b0: 'kkkkkb109',
    0x85b1: 'kkkkaabj',
    0x85b2: 'hhhhha009',
    0x85b3: 'hhhhbbak',
    0x85b4: 'mmmmmb009',
    0x85b5: 'mmmmabak',
    0x85b6: 'nnnnna109',
    0x85b7: 'nnnnbabj',
    0x85b8: 'gggggb119',
    0x85b9: 'ggggaaak',
    0x85ba: 'ddddda019',
    0x85bb: 'ddddbbbj',
    0x85bc: 'aaaaab019',
    0x85bd: 'aaaaabbj',
    0x85be: 'bbbbba119',
    0x85bf: 'bbbbbaak',
    0x85c0: 'cccccb019',
    0x85c1: 'ccccaaak',
    0x85c2: 'CCCCCCab19',
    0x85c3: 'ppppabaj',
    0x85c4: 'eeeeeb119',
    0x85c5: 'eeeeabbj',
    0x85c6: 'fffffa019',
    0x85c7: 'ffffbaak',
    0x85c8: 'ooooob009',
    0x85c9: 'ooooaabj',
    0x85ca: 'llllla109',
    0x85cb: 'llllbbak',
    0x85cc: 'iiiiib109',
    0x85cd: 'iiiiabak',
    0x85ce: 'jjjjja009',
    0x85cf: 'jjjjbabj',
    0x85d0: 'CCCCCCCCCbaaq',
    0x85d1: 'kkkkbabj',
    0x85d2: 'xxxxxa118',
    0x85d3: 'hhhhabak',
    0x85d4: 'CCCCCCCCCbaaa',
    0x85d5: 'mmmmbbak',
    0x85d6: 'CCCCCCCCCbaai',
    0x85d7: 'nnnnaabj',
    0x85d8: 'wwwwwb008',
    0x85d9: 'ggggbaak',
    0x85da: 'ttttta108',
    0x85db: 'ddddabbj',
    0x85dc: 'qqqqqb108',
    0x85dd: 'aaaabbbj',
    0x85de: 'rrrrra008',
    0x85df: 'bbbbaaak',
    0x85e0: 'aaaaaaaaabbaB',
    0x85e1: 'aaaaaaaaabbaF',
    0x85e2: 'aaaaaaaaabbaJ',
    0x85e3: 'aaaaaaaaabbaN',
    0x85e4: 'aaaaaaaaabbaR',
    0x85e5: 'aaaaaaaaabbaV',
    0x85e6: 'aaaaaaaaabbaZ',
    0x85e7: 'aaaaaaaaabbiZ',
    0x85e8: 'aaaaaaaaabbab',
    0x85e9: 'aaaaaaaaabbaf',
    0x85ea: 'aaaaaaaaabbaj',
    0x85eb: 'aaaaaaaaabban',
    0x85ec: 'aaaaaaaaabbar',
    0x85ed: 'aaaaaaaaabbav',
    0x85ee: 'aaaaaaaaabbaz',
    0x85ef: 'aaaaaaaaabbiz',
    0x85f0: 'aaaaaaaaabbm',
    0x85f1: 'aaaaaaaaabbe',
    0x85f2: 'aaaaaaaaabhx',
    0x85f3: 'aaaaaaaaabbu',
    0x85f4: 'aaaaaaaaabbM',
    0x85f5: 'aaaaaaaaabbE',
    0x85f6: 'aaaaaaaaabhX',
    0x85f7: 'aaaaaaaaabbU',
    0x85f8: 'aaaaaaaaabH',
    0x85f9: 'aaaaaaaaabX',
    0x85fa: 'aaaaaaaaabh',
    0x85fb: 'aaaaaaaaabx',
    0x85fc: 'aaaaaa119',
    0x85fd: 'aaaaaaak',
    0x85fe: 'aaaaaaaa58',
    0x85ff: 'aaaaaaaaa2',
    0x8600: 'ccccca008',
    0x8601: 'ccccabah',
    0x8602: 'ccccccab58',
    0x8603: 'cccccccca2',
    0x8604: 'cccccccabaI',
    0x8605: 'cccccccabaY',
    0x8606: 'cccccccabai',
    0x8607: 'cccccccabay',
    0x8608: 'ccccccccabbM',
    0x8609: 'ccccccccabbE',
    0x860a: 'ccccccccabhX',
    0x860b: 'ccccccccabbU',
    0x860c: 'ccccccccabbm',
    0x860d: 'ccccccccabbe',
    0x860e: 'ccccccccabhx',
    0x860f: 'ccccccccabbu',
    0x8610: 'cccccccccaabs',
    0x8611: 'cccccccccaabw',
    0x8612: 'cccccccccaadx',
    0x8613: 'cccccccccaahz',
    0x8614: 'cccccccccaabc',
    0x8615: 'cccccccccaabg',
    0x8616: 'cccccccccaabk',
    0x8617: 'cccccccccaabo',
    0x8618: 'cccccccccaabS',
    0x8619: 'cccccccccaabW',
    0x861a: 'cccccccccaadX',
    0x861b: 'cccccccccaahZ',
    0x861c: 'cccccccccaabC',
    0x861d: 'aaaababi',
    0x861e: 'cccccccccaabK',
    0x861f: 'bbbbabah',
    0x8620: 'ccccccccccbabn',
    0x8621: 'ccccbabi',
    0x8622: 'ccccccccccbabj',
    0x8623: 'ccccccccccbabh',
    0x8624: 'ccccccccccbabf',
    0x8625: 'ccccccccccbabd',
    0x8626: 'ccccccccccbabb',
    0x8627: 'ccccccccccbadc',
    0x8628: 'ccccccccccbajz',
    0x8629: 'ccccccccccbahy',
    0x862a: 'ccccccccccbabz',
    0x862b: 'ccccccccccbabx',
    0x862c: 'ccccccccccbabv',
    0x862d: 'ccccccccccbabt',
    0x862e: 'ccccccccccbabr',
    0x862f: 'ccccccccccbabp',
    0x8630: 'ccccccccccbabN',
    0x8631: 'ccccccccccbabL',
    0x8632: 'ccccccccccbabJ',
    0x8633: 'ccccccccccbabH',
    0x8634: 'ccccccccccbabF',
    0x8635: 'ccccccccccbabD',
    0x8636: 'ccccccccccbabB',
    0x8637: 'ccccccccccbadC',
    0x8638: 'ccccccccccbajZ',
    0x8639: 'ccccccccccbahY',
    0x863a: 'ccccccccccbabZ',
    0x863b: 'ccccccccccbabX',
    0x863c: 'aaaaab108',
    0x863d: 'aaaaabah',
    0x863e: 'bbbbba008',
    0x863f: 'bbbbbabi',
    0x8640: 'cccccb108',
    0x8641: 'ccccaabi',
    0x8642: 'CCCCCCbb09',
    0x8643: 'ppppabbh',
    0x8644: 'eeeeeb008',
    0x8645: 'eeeeabah',
    0x8646: 'fffffa108',
    0x8647: 'ffffbabi',
    0x8648: 'ooooob118',
    0x8649: 'ooooaaah',
    0x864a: 'llllla018',
    0x864b: 'llllbbbi',
    0x864c: 'iiiiib018',
    0x864d: 'iiiiabbi',
    0x864e: 'jjjjja118',
    0x864f: 'jjjjbaah',
    0x8650: 'XXXXXXaa18',
    0x8651: 'kkkkbaah',
    0x8652: 'xxxxxa009',
    0x8653: 'hhhhabbi',
    0x8654: '0000000000abaE',
    0x8655: 'mmmmbbbi',
    0x8656: '0000000000abaA',
    0x8657: 'nnnnaaah',
    0x8658: 'wwwwwb119',
    0x8659: 'ggggbabi',
    0x865a: 'ttttta019',
    0x865b: 'ddddabah',
    0x865c: 'qqqqqb019',
    0x865d: 'aaaabbah',
    0x865e: 'rrrrra119',
    0x865f: 'bbbbaabi',
    0x8660: 'ppppppaa59',
    0x8661: 'ccccbbah',
    0x8662: 'pppppb019',
    0x8663: 'ppppbaai',
    0x8664: 'uuuuua019',
    0x8665: 'eeeebabi',
    0x8666: 'uuuuuuba59',
    0x8667: 'ffffabah',
    0x8668: '00000000baaZ',
    0x8669: 'oooobbbi',
    0x866a: '00000000baaJ',
    0x866b: 'llllaaah',
    0x866c: 'yyyyya009',
    0x866d: 'iiiibaah',
    0x866e: 'yyyyyyba49',
    0x866f: 'jjjjabbi',
    0x8670: 'hhhhhhaa58',
    0x8671: 'kkkkabbi',
    0x8672: 'hhhhhb018',
    0x8673: 'hhhhbaah',
    0x8674: 'mmmmma018',
    0x8675: 'mmmmaaah',
    0x8676: 'mmmmmmba58',
    0x8677: 'nnnnbbbi',
    0x8678: 'ddddddaa48',
    0x8679: 'ggggabah',
    0x867a: 'dddddb008',
    0x867b: 'ddddbabi',
    0x867c: 'aaaaaa008',
    0x867d: 'aaaaaabi',
    0x867e: 'aaaaaaba48',
    0x867f: 'bbbbbbah',
    0x8680: 'ccccca108',
    0x8681: 'ccccabbi',
    0x8682: 'ccccccba48',
    0x8683: 'ppppaabh',
    0x8684: 'eeeeea008',
    0x8685: 'eeeeaaah',
    0x8686: 'eeeeeebb58',
    0x8687: 'ffffbbbi',
    0x8688: 'llllllab48',
    0x8689: 'ooooabah',
    0x868a: 'lllllb018',
    0x868b: 'llllbabi',
    0x868c: 'iiiiia018',
    0x868d: 'iiiiaabi',
    0x868e: 'iiiiiibb48',
    0x868f: 'jjjjbbah',
    0x8690: 'xxxxxxab59',
    0x8691: 'kkkkbbah',
    0x8692: 'xxxxxb009',
    0x8693: 'hhhhaabi',
    0x8694: '8888888aaao',
    0x8695: 'mmmmbabi',
    0x8696: '8888888aaaO',
    0x8697: 'nnnnabah',
    0x8698: 'ttttttab49',
    0x8699: 'ggggbbbi',
    0x869a: 'tttttb019',
    0x869b: 'ddddaaah',
    0x869c: 'qqqqqa019',
    0x869d: 'aaaabaah',
    0x869e: 'qqqqqqbb49',
    0x869f: 'bbbbabbi',
    0x86a0: 'sssssb119',
    0x86a1: 'ccccbaah',
    0x86a2: 'pppppa019',
    0x86a3: 'ppppbbai',
    0x86a4: 'uuuuub019',
    0x86a5: 'eeeebbbi',
    0x86a6: 'vvvvva119',
    0x86a7: 'ffffaaah',
    0x86a8: '?????b898',
    0x86a9: 'oooobabi',
    0x86aa: '<<<<<a998',
    0x86ab: 'llllabah',
    0x86ac: 'yyyyyb009',
    0x86ad: 'iiiibbah',
    0x86ae: 'zzzzza109',
    0x86af: 'jjjjaabi',
    0x86b0: 'kkkkkb118',
    0x86b1: 'kkkkaabi',
    0x86b2: 'hhhhha018',
    0x86b3: 'hhhhbbah',
    0x86b4: 'mmmmmb018',
    0x86b5: 'mmmmabah',
    0x86b6: 'nnnnna118',
    0x86b7: 'nnnnbabi',
    0x86b8: 'gggggb108',
    0x86b9: 'ggggaaah',
    0x86ba: 'ddddda008',
    0x86bb: 'ddddbbbi',
    0x86bc: 'aaaaab008',
    0x86bd: 'aaaaabbi',
    0x86be: 'bbbbba108',
    0x86bf: 'bbbbbaah',
    0x86c0: 'cccccb008',
    0x86c1: 'ccccaaah',
    0x86c2: 'CCCCCCaa19',
    0x86c3: 'ppppabai',
    0x86c4: 'eeeeeb108',
    0x86c5: 'eeeeabbi',
    0x86c6: 'fffffa008',
    0x86c7: 'ffffbaah',
    0x86c8: 'ooooob018',
    0x86c9: 'ooooaabi',
    0x86ca: 'llllla118',
    0x86cb: 'llllbbah',
    0x86cc: 'iiiiib118',
    0x86cd: 'iiiiabah',
    0x86ce: 'jjjjja018',
    0x86cf: 'jjjjbabi',
    0x86d0: 'XXXXXXbb08',
    0x86d1: 'kkkkbabi',
    0x86d2: 'xxxxxa109',
    0x86d3: 'hhhhabah',
    0x86d4: 'XXXXXXXaaak',
    0x86d5: 'mmmmbbah',
    0x86d6: 'XXXXXXXaaaK',
    0x86d7: 'nnnnaabi',
    0x86d8: 'wwwwwb019',
    0x86d9: 'ggggbaah',
    0x86da: 'ttttta119',
    0x86db: 'ddddabbi',
    0x86dc: 'qqqqqb119',
    0x86dd: 'aaaabbbi',
    0x86de: 'rrrrra019',
    0x86df: 'bbbbaaah',
    0x86e0: 'ppppppbb49',
    0x86e1: 'ccccbbbi',
    0x86e2: 'pppppb119',
    0x86e3: 'ppppbabh',
    0x86e4: 'pppppppaaah',
    0x86e5: 'eeeebaah',
    0x86e6: 'pppppppaaaH',
    0x86e7: 'ffffabbi',
    0x86e8: 'yyyyyyybabI',
    0x86e9: 'oooobbah',
    0x86ea: 'yyyyyyybabi',
    0x86eb: 'llllaabi',
    0x86ec: 'yyyyya109',
    0x86ed: 'iiiibabi',
    0x86ee: 'yyyyyyab59',
    0x86ef: 'jjjjabah',
    0x86f0: 'hhhhhhbb48',
    0x86f1: 'kkkkabah',
    0x86f2: 'hhhhhb118',
    0x86f3: 'hhhhbabi',
    0x86f4: 'hhhhhhhaaai',
    0x86f5: 'hhhhhhhaaay',
    0x86f6: 'hhhhhhhaaaI',
    0x86f7: 'hhhhhhhaaaY',
    0x86f8: 'aaaaaaababH',
    0x86f9: 'aaaaaaababX',
    0x86fa: 'aaaaaaababh',
    0x86fb: 'aaaaaaababx',
    0x86fc: 'aaaaaa108',
    0x86fd: 'aaaaaaah',
    0x86fe: 'aaaaaaab58',
    0x86ff: 'bbbbbbbi',
    0x8700: 'ccccca018',
    0x8701: 'ccccabai',
    0x8702: 'ccccccaa48',
    0x8703: 'eeeeeeebbay',
    0x8704: 'cccccccbabH',
    0x8705: 'cccccccbabX',
    0x8706: 'cccccccbabh',
    0x8707: 'cccccccbabx',
    0x8708: 'iiiiiiibbbH',
    0x8709: 'iiiiiiibbbX',
    0x870a: 'iiiiiiibbbh',
    0x870b: 'iiiiiiibbbx',
    0x870c: 'iiiiia108',
    0x870d: 'iiiiaaai',
    0x870e: 'iiiiiiab48',
    0x870f: 'jjjjbbbh',
    0x8710: 'xxxxxxbb59',
    0x8711: 'kkkkbbbh',
    0x8712: 'xxxxxb119',
    0x8713: 'hhhhaaai',
    0x8714: 'xxxxxxxabah',
    0x8715: 'mmmmbaai',
    0x8716: 'xxxxxxxabaH',
    0x8717: 'nnnnabbh',
    0x8718: 'qqqqqqqbbbI',
    0x8719: 'ggggbbai',
    0x871a: 'qqqqqqqbbbi',
    0x871b: 'ddddaabh',
    0x871c: 'qqqqqa109',
    0x871d: 'aaaababh',
    0x871e: 'qqqqqqab49',
    0x871f: 'bbbbabai',
    0x8720: 'sssssb009',
    0x8721: 'ccccbabh',
    0x8722: 'pppppa109',
    0x8723: 'ppppbbbi',
    0x8724: 'uuuuub109',
    0x8725: 'eeeebbai',
    0x8726: 'vvvvva009',
    0x8727: 'ffffaabh',
    0x8728: 'YYYYYYYbbbJ',
    0x8729: 'oooobaai',
    0x872a: 'YYYYYYYbbbj',
    0x872b: 'llllabbh',
    0x872c: 'yyyyyb119',
    0x872d: 'iiiibbbh',
    0x872e: 'zzzzza019',
    0x872f: 'jjjjaaai',
    0x8730: 'kkkkkb008',
    0x8731: 'kkkkaaai',
    0x8732: 'hhhhha108',
    0x8733: 'hhhhbbbh',
    0x8734: 'mmmmmb108',
    0x8735: 'mmmmabbh',
    0x8736: 'nnnnna008',
    0x8737: 'nnnnbaai',
    0x8738: 'gggggb018',
    0x8739: 'ggggaabh',
    0x873a: 'ddddda118',
    0x873b: 'ddddbbai',
    0x873c: 'aaaaab118',
    0x873d: 'aaaaabai',
    0x873e: 'bbbbba018',
    0x873f: 'bbbbbabh',
    0x8740: 'cccccb118',
    0x8741: 'ccccaabh',
    0x8742: 'CCCCCCba19',
    0x8743: 'ppppabbi',
    0x8744: 'eeeeeb018',
    0x8745: 'eeeeabai',
    0x8746: 'fffffa118',
    0x8747: 'ffffbabh',
    0x8748: 'ooooob108',
    0x8749: 'ooooaaai',
    0x874a: 'llllla008',
    0x874b: 'llllbbbh',
    0x874c: 'iiiiib008',
    0x874d: 'iiiiabbh',
    0x874e: 'jjjjja108',
    0x874f: 'jjjjbaai',
    0x8750: 'XXXXXXab08',
    0x8751: 'kkkkbaai',
    0x8752: 'xxxxxa019',
    0x8753: 'hhhhabbh',
    0x8754: '=====b988',
    0x8755: 'mmmmbbbh',
    0x8756: '>>>>>a888',
    0x8757: 'nnnnaaai',
    0x8758: 'wwwwwb109',
    0x8759: 'ggggbabh',
    0x875a: 'ttttta009',
    0x875b: 'ddddabai',
    0x875c: 'qqqqqb009',
    0x875d: 'aaaabbai',
    0x875e: 'rrrrra109',
    0x875f: 'bbbbaabh',
    0x8760: 'ppppppab49',
    0x8761: 'ccccbbai',
    0x8762: 'pppppb009',
    0x8763: 'ppppbaah',
    0x8764: 'uuuuua009',
    0x8765: 'eeeebabh',
    0x8766: 'uuuuuubb49',
    0x8767: 'ffffabai',
    0x8768: '9999999bbbN',
    0x8769: 'oooobbbh',
    0x876a: '9999999bbbn',
    0x876b: 'llllaaai',
    0x876c: 'yyyyya019',
    0x876d: 'iiiibaai',
    0x876e: 'yyyyyybb59',
    0x876f: 'jjjjabbh',
    0x8770: 'hhhhhhab48',
    0x8771: 'kkkkabbh',
    0x8772: 'hhhhhb008',
    0x8773: 'hhhhbaai',
    0x8774: 'mmmmma008',
    0x8775: 'mmmmaaai',
    0x8776: 'mmmmmmbb48',
    0x8777: 'nnnnbbbh',
    0x8778: 'ddddddab58',
    0x8779: 'ggggabai',
    0x877a: 'dddddb018',
    0x877b: 'ddddbabh',
    0x877c: 'aaaaaa018',
    0x877d: 'aaaaaabh',
    0x877e: 'aaaaaabb58',
    0x877f: 'bbbbbbai',
    0x8780: 'ccccca118',
    0x8781: 'ccccabbh',
    0x8782: 'ccccccbb58',
    0x8783: 'ppppaabi',
    0x8784: 'eeeeea018',
    0x8785: 'eeeeaaai',
    0x8786: 'eeeeeeba48',
    0x8787: 'ffffbbbh',
    0x8788: 'llllllaa58',
    0x8789: 'ooooabai',
    0x878a: 'lllllb008',
    0x878b: 'llllbabh',
    0x878c: 'iiiiia008',
    0x878d: 'iiiiaabh',
    0x878e: 'iiiiiiba58',
    0x878f: 'jjjjbbai',
    0x8790: 'xxxxxxaa49',
    0x8791: 'kkkkbbai',
    0x8792: 'xxxxxb019',
    0x8793: 'hhhhaabh',
    0x8794: '11111111baaJ',
    0x8795: 'mmmmbabh',
    0x8796: '11111111baaZ',
    0x8797: 'nnnnabai',
    0x8798: 'ttttttaa59',
    0x8799: 'ggggbbbh',
    0x879a: 'tttttb009',
    0x879b: 'ddddaaai',
    0x879c: 'qqqqqa009',
    0x879d: 'aaaabaai',
    0x879e: 'qqqqqqba59',
    0x879f: 'bbbbabbh',
    0x87a0: 'sssssb109',
    0x87a1: 'ccccbaai',
    0x87a2: 'pppppa009',
    0x87a3: 'ppppbbah',
    0x87a4: 'uuuuub009',
    0x87a5: 'eeeebbbh',
    0x87a6: 'vvvvva109',
    0x87a7: 'ffffaaai',
    0x87a8: '?????b888',
    0x87a9: 'oooobabh',
    0x87aa: '<<<<<a988',
    0x87ab: 'llllabai',
    0x87ac: 'yyyyyb019',
    0x87ad: 'iiiibbai',
    0x87ae: 'zzzzza119',
    0x87af: 'jjjjaabh',
    0x87b0: 'kkkkkb108',
    0x87b1: 'kkkkaabh',
    0x87b2: 'hhhhha008',
    0x87b3: 'hhhhbbai',
    0x87b4: 'mmmmmb008',
    0x87b5: 'mmmmabai',
    0x87b6: 'nnnnna108',
    0x87b7: 'nnnnbabh',
    0x87b8: 'gggggb118',
    0x87b9: 'ggggaaai',
    0x87ba: 'ddddda018',
    0x87bb: 'ddddbbbh',
    0x87bc: 'aaaaab018',
    0x87bd: 'aaaaabbh',
    0x87be: 'bbbbba118',
    0x87bf: 'bbbbbaai',
    0x87c0: 'cccccb018',
    0x87c1: 'ccccaaai',
    0x87c2: 'CCCCCCab09',
    0x87c3: 'ppppabah',
    0x87c4: 'eeeeeb118',
    0x87c5: 'eeeeabbh',
    0x87c6: 'fffffa018',
    0x87c7: 'ffffbaai',
    0x87c8: 'ooooob008',
    0x87c9: 'ooooaabh',
    0x87ca: 'llllla108',
    0x87cb: 'llllbbai',
    0x87cc: 'iiiiib108',
    0x87cd: 'iiiiabai',
    0x87ce: 'jjjjja008',
    0x87cf: 'jjjjbabh',
    0x87d0: 'QQQQQQQQbaan',
    0x87d1: 'kkkkbabh',
    0x87d2: 'xxxxxa119',
    0x87d3: 'hhhhabai',
    0x87d4: 'QQQQQQQQbaaN',
    0x87d5: 'mmmmbbai',
    0x87d6: 'QQQQQQQQbaiZ',
    0x87d7: 'nnnnaabh',
    0x87d8: 'wwwwwb009',
    0x87d9: 'ggggbaai',
    0x87da: 'ttttta109',
    0x87db: 'ddddabbh',
    0x87dc: 'qqqqqb109',
    0x87dd: 'aaaabbbh',
    0x87de: 'rrrrra009',
    0x87df: 'bbbbaaai',
    0x87e0: 'ppppppba59',
    0x87e1: 'ccccbbbh',
    0x87e2: 'pppppb109',
    0x87e3: 'ppppbabi',
    0x87e4: 'pppppppbbbi',
    0x87e5: 'eeeebaai',
    0x87e6: 'pppppppbbbI',
    0x87e7: 'ffffabbh',
    0x87e8: 'ppppppppaahY',
    0x87e9: 'oooobbai',
    0x87ea: 'ppppppppaabL',
    0x87eb: 'llllaabh',
    0x87ec: 'ppppppppaahy',
    0x87ed: 'iiiibabh',
    0x87ee: 'ppppppppaabl',
    0x87ef: 'jjjjabai',
    0x87f0: 'aaaaaaaabaal',
    0x87f1: 'aaaaaaaabaad',
    0x87f2: 'aaaaaaaabaix',
    0x87f3: 'aaaaaaaabaat',
    0x87f4: 'aaaaaaaabaaL',
    0x87f5: 'aaaaaaaabaaD',
    0x87f6: 'aaaaaaaabaiX',
    0x87f7: 'aaaaaaaabaaT',
    0x87f8: 'aaaaaaaabaI',
    0x87f9: 'aaaaaaaabaY',
    0x87fa: 'aaaaaaaabai',
    0x87fb: 'aaaaaaaabay',
    0x87fc: 'aaaaaa118',
    0x87fd: 'aaaaaaai',
    0x87fe: 'aaaaaaaa48',
    0x87ff: 'aaaaaaaab3',
    0x8800: 'ccccca889',
    0x8801: 'ccccabaf',
    0x8802: 'ccccccab09',
    0x8803: 'ccccccccb0',
    0x8804: 'cccccccaaaH',
    0x8805: 'cccccccaaaX',
    0x8806: 'cccccccaaah',
    0x8807: 'cccccccaaax',
    0x8808: 'ccccccccbbaL',
    0x8809: 'ccccccccbbaD',
    0x880a: 'ccccccccbbiX',
    0x880b: 'ccccccccbbaT',
    0x880c: 'ccccccccbbal',
    0x880d: 'ccccccccbbad',
    0x880e: 'ccccccccbbix',
    0x880f: 'ccccccccbbat',
    0x8810: 'qqqqqqqqbaam',
    0x8811: 'kkkkbbbg',
    0x8812: 'qqqqqqqqbaiy',
    0x8813: 'hhhhaaaf',
    0x8814: 'qqqqqqqqbaaM',
    0x8815: 'mmmmbaaf',
    0x8816: 'qqqqqqqqbaiY',
    0x8817: 'nnnnabbg',
    0x8818: 'qqqqqqqabaI',
    0x8819: 'ggggbbaf',
    0x881a: 'qqqqqqqabai',
    0x881b: 'ddddaabg',
    0x881c: 'qqqqqa998',
    0x881d: 'aaaababg',
    0x881e: 'qqqqqqaa08',
    0x881f: 'bbbbabaf',
    0x8820: 'sssssb898',
    0x8821: 'ccccbabg',
    0x8822: 'pppppa998',
    0x8823: 'ppppbbbf',
    0x8824: 'uuuuub998',
    0x8825: 'eeeebbaf',
    0x8826: 'vvvvva898',
    0x8827: 'ffffaabg',
    0x8828: 'PPPPPPPPaahZ',
    0x8829: 'oooobaaf',
    0x882a: 'PPPPPPPPaabO',
    0x882b: 'llllabbg',
    0x882c: 'yyyyyb988',
    0x882d: 'iiiibbbg',
    0x882e: 'zzzzza888',
    0x882f: 'jjjjaaaf',
    0x8830: 'kkkkkb899',
    0x8831: 'kkkkaaaf',
    0x8832: 'hhhhha999',
    0x8833: 'hhhhbbbg',
    0x8834: 'mmmmmb999',
    0x8835: 'mmmmabbg',
    0x8836: 'nnnnna899',
    0x8837: 'nnnnbaaf',
    0x8838: 'gggggb889',
    0x8839: 'ggggaabg',
    0x883a: 'ddddda989',
    0x883b: 'ddddbbaf',
    0x883c: 'aaaaab989',
    0x883d: 'aaaaabaf',
    0x883e: 'bbbbba889',
    0x883f: 'bbbbbabg',
    0x8840: 'cccccb989',
    0x8841: 'ccccaabg',
    0x8842: 'CCCCCCbb58',
    0x8843: 'ppppabbf',
    0x8844: 'eeeeeb889',
    0x8845: 'eeeeabaf',
    0x8846: 'fffffa989',
    0x8847: 'ffffbabg',
    0x8848: 'ooooob999',
    0x8849: 'ooooaaaf',
    0x884a: 'llllla899',
    0x884b: 'llllbbbg',
    0x884c: 'iiiiib899',
    0x884d: 'iiiiabbg',
    0x884e: 'jjjjja999',
    0x884f: 'jjjjbaaf',
    0x8850: 'XXXXXXaa49',
    0x8851: 'kkkkbaaf',
    0x8852: 'xxxxxa888',
    0x8853: 'hhhhabbg',
    0x8854: '=====b119',
    0x8855: 'mmmmbbbg',
    0x8856: '>>>>>a019',
    0x8857: 'nnnnaaaf',
    0x8858: 'wwwwwb998',
    0x8859: 'ggggbabg',
    0x885a: 'ttttta898',
    0x885b: 'ddddabaf',
    0x885c: 'qqqqqb898',
    0x885d: 'aaaabbaf',
    0x885e: 'rrrrra998',
    0x885f: 'bbbbaabg',
    0x8860: 'ppppppaa08',
    0x8861: 'ccccbbaf',
    0x8862: 'pppppb898',
    0x8863: 'ppppbaag',
    0x8864: 'uuuuua898',
    0x8865: 'eeeebabg',
    0x8866: 'uuuuuuba08',
    0x8867: 'ffffabaf',
    0x8868: '00000000aadX',
    0x8869: 'oooobbbg',
    0x886a: '00000000aabK',
    0x886b: 'llllaaaf',
    0x886c: 'yyyyya888',
    0x886d: 'iiiibaaf',
    0x886e: 'yyyyyyba18',
    0x886f: 'jjjjabbg',
    0x8870: 'hhhhhhaa09',
    0x8871: 'kkkkabbg',
    0x8872: 'hhhhhb899',
    0x8873: 'hhhhbaaf',
    0x8874: 'mmmmma899',
    0x8875: 'mmmmaaaf',
    0x8876: 'mmmmmmba09',
    0x8877: 'nnnnbbbg',
    0x8878: 'ddddddaa19',
    0x8879: 'ggggabaf',
    0x887a: 'dddddb889',
    0x887b: 'ddddbabg',
    0x887c: 'aaaaaa889',
    0x887d: 'aaaaaabg',
    0x887e: 'aaaaaaba19',
    0x887f: 'bbbbbbaf',
    0x8880: 'ccccca989',
    0x8881: 'ccccabbg',
    0x8882: 'ccccccba19',
    0x8883: 'ppppaabf',
    0x8884: 'eeeeea889',
    0x8885: 'eeeeaaaf',
    0x8886: 'eeeeeebb09',
    0x8887: 'ffffbbbg',
    0x8888: 'llllllab19',
    0x8889: 'ooooabaf',
    0x888a: 'lllllb899',
    0x888b: 'llllbabg',
    0x888c: 'iiiiia899',
    0x888d: 'iiiiaabg',
    0x888e: 'iiiiiibb19',
    0x888f: 'jjjjbbaf',
    0x8890: 'xxxxxxab08',
    0x8891: 'kkkkbbaf',
    0x8892: 'xxxxxb888',
    0x8893: 'hhhhaabg',
    0x8894: '8888888aban',
    0x8895: 'mmmmbabg',
    0x8896: '8888888abaN',
    0x8897: 'nnnnabaf',
    0x8898: 'ttttttab18',
    0x8899: 'ggggbbbg',
    0x889a: 'tttttb898',
    0x889b: 'ddddaaaf',
    0x889c: 'qqqqqa898',
    0x889d: 'aaaabaaf',
    0x889e: 'qqqqqqbb18',
    0x889f: 'bbbbabbg',
    0x88a0: 'sssssb998',
    0x88a1: 'ccccbaaf',
    0x88a2: 'pppppa898',
    0x88a3: 'ppppbbag',
    0x88a4: 'uuuuub898',
    0x88a5: 'eeeebbbg',
    0x88a6: 'vvvvva998',
    0x88a7: 'ffffaaaf',
    0x88a8: '?????b019',
    0x88a9: 'oooobabg',
    0x88aa: '<<<<<a119',
    0x88ab: 'llllabaf',
    0x88ac: 'yyyyyb888',
    0x88ad: 'iiiibbaf',
    0x88ae: 'zzzzza988',
    0x88af: 'jjjjaabg',
    0x88b0: 'kkkkkb999',
    0x88b1: 'kkkkaabg',
    0x88b2: 'hhhhha899',
    0x88b3: 'hhhhbbaf',
    0x88b4: 'mmmmmb899',
    0x88b5: 'mmmmabaf',
    0x88b6: 'nnnnna999',
    0x88b7: 'nnnnbabg',
    0x88b8: 'gggggb989',
    0x88b9: 'ggggaaaf',
    0x88ba: 'ddddda889',
    0x88bb: 'ddddbbbg',
    0x88bc: 'aaaaab889',
    0x88bd: 'aaaaabbg',
    0x88be: 'bbbbba989',
    0x88bf: 'bbbbbaaf',
    0x88c0: 'cccccb889',
    0x88c1: 'ccccaaaf',
    0x88c2: 'CCCCCCaa48',
    0x88c3: 'ppppabag',
    0x88c4: 'eeeeeb989',
    0x88c5: 'eeeeabbg',
    0x88c6: 'fffffa889',
    0x88c7: 'ffffbaaf',
    0x88c8: 'ooooob899',
    0x88c9: 'ooooaabg',
    0x88ca: 'llllla999',
    0x88cb: 'llllbbaf',
    0x88cc: 'iiiiib999',
    0x88cd: 'iiiiabaf',
    0x88ce: 'jjjjja899',
    0x88cf: 'jjjjbabg',
    0x88d0: 'XXXXXXbb59',
    0x88d1: 'kkkkbabg',
    0x88d2: 'xxxxxa988',
    0x88d3: 'hhhhabaf',
    0x88d4: 'XXXXXXXabaj',
    0x88d5: 'mmmmbbaf',
    0x88d6: 'XXXXXXXabaJ',
    0x88d7: 'nnnnaabg',
    0x88d8: 'wwwwwb898',
    0x88d9: 'ggggbaaf',
    0x88da: 'ttttta998',
    0x88db: 'ddddabbg',
    0x88dc: 'qqqqqb998',
    0x88dd: 'aaaabbbg',
    0x88de: 'rrrrra898',
    0x88df: 'bbbbaaaf',
    0x88e0: 'ppppppbb18',
    0x88e1: 'ccccbbbg',
    0x88e2: 'pppppb998',
    0x88e3: 'ppppbabf',
    0x88e4: 'pppppppabai',
    0x88e5: 'eeeebaaf',
    0x88e6: 'pppppppabaI',
    0x88e7: 'ffffabbg',
    0x88e8: 'yyyyyyybbbH',
    0x88e9: 'oooobbaf',
    0x88ea: 'yyyyyyybbbh',
    0x88eb: 'llllaabg',
    0x88ec: 'yyyyya988',
    0x88ed: 'iiiibabg',
    0x88ee: 'yyyyyyab08',
    0x88ef: 'jjjjabaf',
    0x88f0: 'hhhhhhbb19',
    0x88f1: 'hhhhhhhab1',
    0x88f2: 'hhhhhb999',
    0x88f3: 'hhhhbabg',
    0x88f4: 'hhhhhhhabah',
    0x88f5: 'hhhhhhhabax',
    0x88f6: 'hhhhhhhabaH',
    0x88f7: 'hhhhhhhabaX',
    0x88f8: 'aaaaaaabbbI',
    0x88f9: 'aaaaaaabbbY',
    0x88fa: 'aaaaaaabbbi',
    0x88fb: 'aaaaaaabbby',
    0x88fc: 'aaaaaa989',
    0x88fd: 'aaaaaaaf',
    0x88fe: 'aaaaaaab09',
    0x88ff: 'bbbbbbbg',
    0x8900: 'ccccca899',
    0x8901: 'ccccabag',
    0x8902: 'ccccccaa19',
    0x8903: 'eeeeeeebaax',
    0x8904: 'cccccccbbbI',
    0x8905: 'cccccccbbbY',
    0x8906: 'cccccccbbbi',
    0x8907: 'cccccccbbby',
    0x8908: 'iiiiiiibabI',
    0x8909: 'iiiiiiibabY',
    0x890a: 'iiiiiiibabi',
    0x890b: 'iiiiiiibaby',
    0x890c: 'iiiiia989',
    0x890d: 'iiiiaaag',
    0x890e: 'iiiiiiab19',
    0x890f: 'jjjjbbbf',
    0x8910: 'xxxxxxbb08',
    0x8911: 'kkkkbbbf',
    0x8912: 'xxxxxb998',
    0x8913: 'hhhhaaag',
    0x8914: 'xxxxxxxaaai',
    0x8915: 'mmmmbaag',
    0x8916: 'xxxxxxxaaaI',
    0x8917: 'nnnnabbf',
    0x8918: 'qqqqqqqbabH',
    0x8919: 'ggggbbag',
    0x891a: 'qqqqqqqbabh',
    0x891b: 'ddddaabf',
    0x891c: 'qqqqqa988',
    0x891d: 'aaaababf',
    0x891e: 'qqqqqqab18',
    0x891f: 'bbbbabag',
    0x8920: 'sssssb888',
    0x8921: 'ccccbabf',
    0x8922: 'pppppa988',
    0x8923: 'ppppbbbg',
    0x8924: 'uuuuub988',
    0x8925: 'eeeebbag',
    0x8926: 'vvvvva888',
    0x8927: 'ffffaabf',
    0x8928: 'YYYYYYYbabK',
    0x8929: 'oooobaag',
    0x892a: 'YYYYYYYbabk',
    0x892b: 'llllabbf',
    0x892c: 'yyyyyb998',
    0x892d: 'iiiibbbf',
    0x892e: 'zzzzza898',
    0x892f: 'jjjjaaag',
    0x8930: 'kkkkkb889',
    0x8931: 'kkkkaaag',
    0x8932: 'hhhhha989',
    0x8933: 'hhhhbbbf',
    0x8934: 'mmmmmb989',
    0x8935: 'mmmmabbf',
    0x8936: 'nnnnna889',
    0x8937: 'nnnnbaag',
    0x8938: 'gggggb899',
    0x8939: 'ggggaabf',
    0x893a: 'ddddda999',
    0x893b: 'ddddbbag',
    0x893c: 'aaaaab999',
    0x893d: 'aaaaabag',
    0x893e: 'bbbbba899',
    0x893f: 'bbbbbabf',
    0x8940: 'cccccb999',
    0x8941: 'ccccaabf',
    0x8942: 'CCCCCCba48',
    0x8943: 'ppppabbg',
    0x8944: 'eeeeeb899',
    0x8945: 'eeeeabag',
    0x8946: 'fffffa999',
    0x8947: 'ffffbabf',
    0x8948: 'ooooob989',
    0x8949: 'ooooaaag',
    0x894a: 'llllla889',
    0x894b: 'llllbbbf',
    0x894c: 'iiiiib889',
    0x894d: 'iiiiabbf',
    0x894e: 'jjjjja989',
    0x894f: 'jjjjbaag',
    0x8950: 'XXXXXXab59',
    0x8951: 'kkkkbaag',
    0x8952: 'xxxxxa898',
    0x8953: 'hhhhabbf',
    0x8954: '=====b109',
    0x8955: 'mmmmbbbf',
    0x8956: '>>>>>a009',
    0x8957: 'nnnnaaag',
    0x8958: 'wwwwwb988',
    0x8959: 'ggggbabf',
    0x895a: 'ttttta888',
    0x895b: 'ddddabag',
    0x895c: 'qqqqqb888',
    0x895d: 'aaaabbag',
    0x895e: 'rrrrra988',
    0x895f: 'bbbbaabf',
    0x8960: 'ppppppab18',
    0x8961: 'ccccbbag',
    0x8962: 'pppppb888',
    0x8963: 'ppppbaaf',
    0x8964: 'uuuuua888',
    0x8965: 'eeeebabf',
    0x8966: 'uuuuuubb18',
    0x8967: 'ffffabag',
    0x8968: '9999999babO',
    0x8969: 'oooobbbf',
    0x896a: '9999999babo',
    0x896b: 'llllaaag',
    0x896c: 'yyyyya898',
    0x896d: 'iiiibaag',
    0x896e: 'yyyyyybb08',
    0x896f: 'jjjjabbf',
    0x8970: 'hhhhhhab19',
    0x8971: 'kkkkabbf',
    0x8972: 'hhhhhb889',
    0x8973: 'hhhhbaag',
    0x8974: 'mmmmma889',
    0x8975: 'mmmmaaag',
    0x8976: 'mmmmmmbb19',
    0x8977: 'nnnnbbbf',
    0x8978: 'ddddddab09',
    0x8979: 'ggggabag',
    0x897a: 'dddddb899',
    0x897b: 'ddddbabf',
    0x897c: 'aaaaaa899',
    0x897d: 'aaaaaabf',
    0x897e: 'aaaaaabb09',
    0x897f: 'bbbbbbag',
    0x8980: 'ccccca999',
    0x8981: 'ccccabbf',
    0x8982: 'ccccccbb09',
    0x8983: 'ppppaabg',
    0x8984: 'eeeeea899',
    0x8985: 'eeeeaaag',
    0x8986: 'eeeeeeba19',
    0x8987: 'ffffbbbf',
    0x8988: 'llllllaa09',
    0x8989: 'ooooabag',
    0x898a: 'lllllb889',
    0x898b: 'llllbabf',
    0x898c: 'iiiiia889',
    0x898d: 'iiiiaabf',
    0x898e: 'iiiiiiba09',
    0x898f: 'jjjjbbag',
    0x8990: 'xxxxxxaa18',
    0x8991: 'kkkkbbag',
    0x8992: 'xxxxxb898',
    0x8993: 'hhhhaabf',
    0x8994: '11111111aabK',
    0x8995: 'mmmmbabf',
    0x8996: '11111111aadX',
    0x8997: 'nnnnabag',
    0x8998: 'ttttttaa08',
    0x8999: 'ggggbbbf',
    0x899a: 'tttttb888',
    0x899b: 'ddddaaag',
    0x899c: 'qqqqqa888',
    0x899d: 'aaaabaag',
    0x899e: 'qqqqqqba08',
    0x899f: 'bbbbabbf',
    0x89a0: 'sssssb988',
    0x89a1: 'ccccbaag',
    0x89a2: 'pppppa888',
    0x89a3: 'ppppbbaf',
    0x89a4: 'uuuuub888',
    0x89a5: 'eeeebbbf',
    0x89a6: 'vvvvva988',
    0x89a7: 'ffffaaag',
    0x89a8: '1111111111badC',
    0x89a9: 'oooobabf',
    0x89aa: '1111111111babD',
    0x89ab: 'llllabag',
    0x89ac: 'yyyyyb898',
    0x89ad: 'iiiibbag',
    0x89ae: 'zzzzza998',
    0x89af: 'jjjjaabf',
    0x89b0: 'kkkkkb989',
    0x89b1: 'kkkkaabf',
    0x89b2: 'hhhhha889',
    0x89b3: 'hhhhbbag',
    0x89b4: 'mmmmmb889',
    0x89b5: 'mmmmabag',
    0x89b6: 'nnnnna989',
    0x89b7: 'nnnnbabf',
    0x89b8: 'gggggb999',
    0x89b9: 'ggggaaag',
    0x89ba: 'ddddda899',
    0x89bb: 'ddddbbbf',
    0x89bc: 'aaaaab899',
    0x89bd: 'aaaaabbf',
    0x89be: 'bbbbba999',
    0x89bf: 'bbbbbaag',
    0x89c0: 'aaaaaaaaaababV',
    0x89c1: 'aaaaaaaaaababT',
    0x89c2: 'aaaaaaaaaababR',
    0x89c3: 'aaaaaaaaaababP',
    0x89c4: 'aaaaaaaaaabajZ',
    0x89c5: 'aaaaaaaaaabahY',
    0x89c6: 'aaaaaaaaaababZ',
    0x89c7: 'aaaaaaaaaababX',
    0x89c8: 'aaaaaaaaaababF',
    0x89c9: 'aaaaaaaaaababD',
    0x89ca: 'aaaaaaaaaababB',
    0x89cb: 'aaaaaaaaaabadC',
    0x89cc: 'aaaaaaaaaababN',
    0x89cd: 'aaaaaaaaaababL',
    0x89ce: 'aaaaaaaaaababJ',
    0x89cf: 'aaaaaaaaaababH',
    0x89d0: 'aaaaaaaaaababv',
    0x89d1: 'aaaaaaaaaababt',
    0x89d2: 'aaaaaaaaaababr',
    0x89d3: 'aaaaaaaaaababp',
    0x89d4: 'aaaaaaaaaabajz',
    0x89d5: 'aaaaaaaaaabahy',
    0x89d6: 'aaaaaaaaaababz',
    0x89d7: 'aaaaaaaaaababx',
    0x89d8: 'aaaaaaaaaababf',
    0x89d9: 'aaaaaaaaaababd',
    0x89da: 'aaaaaaaaaababb',
    0x89db: 'aaaaaaaaaabadc',
    0x89dc: 'aaaaaaaaaababn',
    0x89dd: 'aaaabbbf',
    0x89de: 'aaaaaaaaaababj',
    0x89df: 'aaaaaaaaaababh',
    0x89e0: 'aaaaaaaaaabaB',
    0x89e1: 'aaaaaaaaaabaF',
    0x89e2: 'aaaaaaaaaabaJ',
    0x89e3: 'aaaaaaaaaabaN',
    0x89e4: 'aaaaaaaaaabaR',
    0x89e5: 'aaaaaaaaaabaV',
    0x89e6: 'aaaaaaaaaabaZ',
    0x89e7: 'aaaaaaaaaabiZ',
    0x89e8: 'aaaaaaaaaabab',
    0x89e9: 'aaaaaaaaaabaf',
    0x89ea: 'aaaaaaaaaabaj',
    0x89eb: 'aaaaaaaaaaban',
    0x89ec: 'aaaaaaaaaabar',
    0x89ed: 'aaaaaaaaaabav',
    0x89ee: 'aaaaaaaaaabaz',
    0x89ef: 'aaaaaaaaaabiz',
    0x89f0: 'aaaaaaaaaabm',
    0x89f1: 'aaaaaaaaaabe',
    0x89f2: 'aaaaaaaaaahx',
    0x89f3: 'aaaaaaaaaabu',
    0x89f4: 'aaaaaaaaaabM',
    0x89f5: 'aaaaaaaaaabE',
    0x89f6: 'aaaaaaaaaahX',
    0x89f7: 'aaaaaaaaaabU',
    0x89f8: 'aaaaaaaaaaH',
    0x89f9: 'aaaaaaaaaaX',
    0x89fa: 'aaaaaaaaaah',
    0x89fb: 'aaaaaaaaaax',
    0x89fc: 'aaaaaa999',
    0x89fd: 'aaaaaaag',
    0x89fe: 'aaaaaaaa19',
    0x89ff: 'aaaaaaaaa1',
    0x8a00: 'ccccca888',
    0x8a01: 'ccccabad',
    0x8a02: 'ccccccab19',
    0x8a03: 'cccccccca1',
    0x8a04: 'cccccccabbI',
    0x8a05: 'cccccccabbY',
    0x8a06: 'cccccccabbi',
    0x8a07: 'cccccccabby',
    0x8a08: 'ccccccccaabM',
    0x8a09: 'ccccccccaabE',
    0x8a0a: 'ccccccccaahX',
    0x8a0b: 'ccccccccaabU',
    0x8a0c: 'ccccccccaabm',
    0x8a0d: 'ccccccccaabe',
    0x8a0e: 'ccccccccaahx',
    0x8a0f: 'ccccccccaabu',
    0x8a10: 'cccccccccbabs',
    0x8a11: 'cccccccccbabw',
    0x8a12: 'cccccccccbadx',
    0x8a13: 'cccccccccbahz',
    0x8a14: 'cccccccccbabc',
    0x8a15: 'cccccccccbabg',
    0x8a16: 'cccccccccbabk',
    0x8a17: 'cccccccccbabo',
    0x8a18: 'cccccccccbabS',
    0x8a19: 'cccccccccbabW',
    0x8a1a: 'cccccccccbadX',
    0x8a1b: 'cccccccccbahZ',
    0x8a1c: 'cccccccccbabC',
    0x8a1d: 'aaaababe',
    0x8a1e: 'cccccccccbabK',
    0x8a1f: 'bbbbabad',
    0x8a20: 'sssssb899',
    0x8a21: 'ccccbabe',
    0x8a22: 'pppppa999',
    0x8a23: 'ppppbbbd',
    0x8a24: 'uuuuub999',
    0x8a25: 'eeeebbad',
    0x8a26: 'vvvvva899',
    0x8a27: 'ffffaabe',
    0x8a28: 'AAAAAAAAAbbdc',
    0x8a29: 'oooobaad',
    0x8a2a: 'AAAAAAAAAbbbh',
    0x8a2b: 'llllabbe',
    0x8a2c: 'yyyyyb989',
    0x8a2d: 'iiiibbbe',
    0x8a2e: 'zzzzza889',
    0x8a2f: 'jjjjaaad',
    0x8a30: 'kkkkkb898',
    0x8a31: 'kkkkaaad',
    0x8a32: 'hhhhha998',
    0x8a33: 'hhhhbbbe',
    0x8a34: 'mmmmmb998',
    0x8a35: 'mmmmabbe',
    0x8a36: 'nnnnna898',
    0x8a37: 'nnnnbaad',
    0x8a38: 'gggggb888',
    0x8a39: 'ggggaabe',
    0x8a3a: 'ddddda988',
    0x8a3b: 'ddddbbad',
    0x8a3c: 'aaaaab988',
    0x8a3d: 'aaaaabad',
    0x8a3e: 'bbbbba888',
    0x8a3f: 'bbbbbabe',
    0x8a40: 'cccccb988',
    0x8a41: 'ccccaabe',
    0x8a42: 'CCCCCCbb48',
    0x8a43: 'ppppabbd',
    0x8a44: 'eeeeeb888',
    0x8a45: 'eeeeabad',
    0x8a46: 'fffffa988',
    0x8a47: 'ffffbabe',
    0x8a48: 'ooooob998',
    0x8a49: 'ooooaaad',
    0x8a4a: 'llllla898',
    0x8a4b: 'llllbbbe',
    0x8a4c: 'iiiiib898',
    0x8a4d: 'iiiiabbe',
    0x8a4e: 'jjjjja998',
    0x8a4f: 'jjjjbaad',
    0x8a50: 'XXXXXXaa59',
    0x8a51: 'kkkkbaad',
    0x8a52: 'xxxxxa889',
    0x8a53: 'hhhhabbe',
    0x8a54: '=====b118',
    0x8a55: 'mmmmbbbe',
    0x8a56: '>>>>>a018',
    0x8a57: 'nnnnaaad',
    0x8a58: 'wwwwwb999',
    0x8a59: 'ggggbabe',
    0x8a5a: 'ttttta899',
    0x8a5b: 'ddddabad',
    0x8a5c: 'qqqqqb899',
    0x8a5d: 'aaaabbad',
    0x8a5e: 'rrrrra999',
    0x8a5f: 'bbbbaabe',
    0x8a60: 'ppppppaa18',
    0x8a61: 'ccccbbad',
    0x8a62: 'pppppb899',
    0x8a63: 'ppppbaae',
    0x8a64: 'uuuuua899',
    0x8a65: 'eeeebabe',
    0x8a66: 'uuuuuuba18',
    0x8a67: 'ffffabad',
    0x8a68: '00000000bbaZ',
    0x8a69: 'oooobbbe',
    0x8a6a: '00000000bbaJ',
    0x8a6b: 'llllaaad',
    0x8a6c: 'yyyyya889',
    0x8a6d: 'iiiibaad',
    0x8a6e: 'yyyyyyba08',
    0x8a6f: 'jjjjabbe',
    0x8a70: 'hhhhhhaa19',
    0x8a71: 'kkkkabbe',
    0x8a72: 'hhhhhb898',
    0x8a73: 'hhhhbaad',
    0x8a74: 'mmmmma898',
    0x8a75: 'mmmmaaad',
    0x8a76: 'mmmmmmba19',
    0x8a77: 'nnnnbbbe',
    0x8a78: 'ddddddaa09',
    0x8a79: 'ggggabad',
    0x8a7a: 'dddddb888',
    0x8a7b: 'ddddbabe',
    0x8a7c: 'aaaaaa888',
    0x8a7d: 'aaaaaabe',
    0x8a7e: 'aaaaaaba09',
    0x8a7f: 'bbbbbbad',
    0x8a80: 'ccccca988',
    0x8a81: 'ccccabbe',
    0x8a82: 'ccccccba09',
    0x8a83: 'ppppaabd',
    0x8a84: 'eeeeea888',
    0x8a85: 'eeeeaaad',
    0x8a86: 'eeeeeebb19',
    0x8a87: 'ffffbbbe',
    0x8a88: 'llllllab09',
    0x8a89: 'ooooabad',
    0x8a8a: 'lllllb898',
    0x8a8b: 'llllbabe',
    0x8a8c: 'iiiiia898',
    0x8a8d: 'iiiiaabe',
    0x8a8e: 'iiiiiibb09',
    0x8a8f: 'jjjjbbad',
    0x8a90: 'xxxxxxab18',
    0x8a91: 'kkkkbbad',
    0x8a92: 'xxxxxb889',
    0x8a93: 'hhhhaabe',
    0x8a94: '8888888aabo',
    0x8a95: 'mmmmbabe',
    0x8a96: '8888888aabO',
    0x8a97: 'nnnnabad',
    0x8a98: 'ttttttab08',
    0x8a99: 'ggggbbbe',
    0x8a9a: 'tttttb899',
    0x8a9b: 'ddddaaad',
    0x8a9c: 'qqqqqa899',
    0x8a9d: 'aaaabaad',
    0x8a9e: 'qqqqqqbb08',
    0x8a9f: 'bbbbabbe',
    0x8aa0: 'sssssb999',
    0x8aa1: 'ccccbaad',
    0x8aa2: 'pppppa899',
    0x8aa3: 'ppppbbae',
    0x8aa4: 'uuuuub899',
    0x8aa5: 'eeeebbbe',
    0x8aa6: 'vvvvva999',
    0x8aa7: 'ffffaaad',
    0x8aa8: '?????b018',
    0x8aa9: 'oooobabe',
    0x8aaa: '<<<<<a118',
    0x8aab: 'llllabad',
    0x8aac: 'yyyyyb889',
    0x8aad: 'iiiibbad',
    0x8aae: 'zzzzza989',
    0x8aaf: 'jjjjaabe',
    0x8ab0: 'kkkkkb998',
    0x8ab1: 'kkkkaabe',
    0x8ab2: 'hhhhha898',
    0x8ab3: 'hhhhbbad',
    0x8ab4: 'mmmmmb898',
    0x8ab5: 'mmmmabad',
    0x8ab6: 'nnnnna998',
    0x8ab7: 'nnnnbabe',
    0x8ab8: 'gggggb988',
    0x8ab9: 'ggggaaad',
    0x8aba: 'ddddda888',
    0x8abb: 'ddddbbbe',
    0x8abc: 'aaaaab888',
    0x8abd: 'aaaaabbe',
    0x8abe: 'bbbbba988',
    0x8abf: 'bbbbbaad',
    0x8ac0: 'cccccb888',
    0x8ac1: 'ccccaaad',
    0x8ac2: 'CCCCCCaa58',
    0x8ac3: 'ppppabae',
    0x8ac4: 'eeeeeb988',
    0x8ac5: 'eeeeabbe',
    0x8ac6: 'fffffa888',
    0x8ac7: 'ffffbaad',
    0x8ac8: 'ooooob898',
    0x8ac9: 'ooooaabe',
    0x8aca: 'llllla998',
    0x8acb: 'llllbbad',
    0x8acc: 'iiiiib998',
    0x8acd: 'iiiiabad',
    0x8ace: 'jjjjja898',
    0x8acf: 'jjjjbabe',
    0x8ad0: 'XXXXXXbb49',
    0x8ad1: 'kkkkbabe',
    0x8ad2: 'xxxxxa989',
    0x8ad3: 'hhhhabad',
    0x8ad4: 'XXXXXXXaabk',
    0x8ad5: 'mmmmbbad',
    0x8ad6: 'XXXXXXXaabK',
    0x8ad7: 'nnnnaabe',
    0x8ad8: 'wwwwwb899',
    0x8ad9: 'ggggbaad',
    0x8ada: 'ttttta999',
    0x8adb: 'ddddabbe',
    0x8adc: 'qqqqqb999',
    0x8add: 'aaaabbbe',
    0x8ade: 'rrrrra899',
    0x8adf: 'bbbbaaad',
    0x8ae0: 'ppppppbb08',
    0x8ae1: 'ccccbbbe',
    0x8ae2: 'pppppb999',
    0x8ae3: 'ppppbabd',
    0x8ae4: 'pppppppaabh',
    0x8ae5: 'eeeebaad',
    0x8ae6: 'pppppppaabH',
    0x8ae7: 'ffffabbe',
    0x8ae8: 'yyyyyyybaaI',
    0x8ae9: 'oooobbad',
    0x8aea: 'yyyyyyybaai',
    0x8aeb: 'llllaabe',
    0x8aec: 'yyyyya989',
    0x8aed: 'iiiibabe',
    0x8aee: 'yyyyyyab18',
    0x8aef: 'jjjjabad',
    0x8af0: 'hhhhhhbb09',
    0x8af1: 'kkkkabad',
    0x8af2: 'hhhhhb998',
    0x8af3: 'hhhhbabe',
    0x8af4: 'hhhhhhhaabi',
    0x8af5: 'hhhhhhhaaby',
    0x8af6: 'hhhhhhhaabI',
    0x8af7: 'hhhhhhhaabY',
    0x8af8: 'aaaaaaabaaH',
    0x8af9: 'aaaaaaabaaX',
    0x8afa: 'aaaaaaabaah',
    0x8afb: 'aaaaaaabaax',
    0x8afc: 'aaaaaa988',
    0x8afd: 'aaaaaaad',
    0x8afe: 'aaaaaaab19',
    0x8aff: 'bbbbbbbe',
    0x8b00: 'ccccca898',
    0x8b01: 'ccccabae',
    0x8b02: 'ccccccaa09',
    0x8b03: 'eeeeeeebbby',
    0x8b04: 'cccccccbaaH',
    0x8b05: 'cccccccbaaX',
    0x8b06: 'cccccccbaah',
    0x8b07: 'cccccccbaax',
    0x8b08: 'iiiiiiibbaH',
    0x8b09: 'iiiiiiibbaX',
    0x8b0a: 'iiiiiiibbah',
    0x8b0b: 'iiiiiiibbax',
    0x8b0c: 'iiiiia988',
    0x8b0d: 'iiiiaaae',
    0x8b0e: 'iiiiiiab09',
    0x8b0f: 'jjjjbbbd',
    0x8b10: 'xxxxxxbb18',
    0x8b11: 'kkkkbbbd',
    0x8b12: 'xxxxxb999',
    0x8b13: 'hhhhaaae',
    0x8b14: 'xxxxxxxabbh',
    0x8b15: 'mmmmbaae',
    0x8b16: 'xxxxxxxabbH',
    0x8b17: 'nnnnabbd',
    0x8b18: 'qqqqqqqbbaI',
    0x8b19: 'ggggbbae',
    0x8b1a: 'qqqqqqqbbai',
    0x8b1b: 'ddddaabd',
    0x8b1c: 'qqqqqa989',
    0x8b1d: 'aaaababd',
    0x8b1e: 'qqqqqqab08',
    0x8b1f: 'bbbbabae',
    0x8b20: 'sssssb889',
    0x8b21: 'ccccbabd',
    0x8b22: 'pppppa989',
    0x8b23: 'ppppbbbe',
    0x8b24: 'uuuuub989',
    0x8b25: 'eeeebbae',
    0x8b26: 'vvvvva889',
    0x8b27: 'ffffaabd',
    0x8b28: 'YYYYYYYbbaJ',
    0x8b29: 'oooobaae',
    0x8b2a: 'YYYYYYYbbaj',
    0x8b2b: 'llllabbd',
    0x8b2c: 'yyyyyb999',
    0x8b2d: 'iiiibbbd',
    0x8b2e: 'zzzzza899',
    0x8b2f: 'jjjjaaae',
    0x8b30: 'kkkkkb888',
    0x8b31: 'kkkkaaae',
    0x8b32: 'hhhhha988',
    0x8b33: 'hhhhbbbd',
    0x8b34: 'mmmmmb988',
    0x8b35: 'mmmmabbd',
    0x8b36: 'nnnnna888',
    0x8b37: 'nnnnbaae',
    0x8b38: 'gggggb898',
    0x8b39: 'ggggaabd',
    0x8b3a: 'ddddda998',
    0x8b3b: 'ddddbbae',
    0x8b3c: 'aaaaab998',
    0x8b3d: 'aaaaabae',
    0x8b3e: 'bbbbba898',
    0x8b3f: 'bbbbbabd',
    0x8b40: 'cccccb998',
    0x8b41: 'ccccaabd',
    0x8b42: 'CCCCCCba58',
    0x8b43: 'ppppabbe',
    0x8b44: 'eeeeeb898',
    0x8b45: 'eeeeabae',
    0x8b46: 'fffffa998',
    0x8b47: 'ffffbabd',
    0x8b48: 'ooooob988',
    0x8b49: 'ooooaaae',
    0x8b4a: 'llllla888',
    0x8b4b: 'llllbbbd',
    0x8b4c: 'iiiiib888',
    0x8b4d: 'iiiiabbd',
    0x8b4e: 'jjjjja988',
    0x8b4f: 'jjjjbaae',
    0x8b50: 'XXXXXXab49',
    0x8b51: 'kkkkbaae',
    0x8b52: 'xxxxxa899',
    0x8b53: 'hhhhabbd',
    0x8b54: '=====b108',
    0x8b55: 'mmmmbbbd',
    0x8b56: '>>>>>a008',
    0x8b57: 'nnnnaaae',
    0x8b58: 'wwwwwb989',
    0x8b59: 'ggggbabd',
    0x8b5a: 'ttttta889',
    0x8b5b: 'ddddabae',
    0x8b5c: 'qqqqqb889',
    0x8b5d: 'aaaabbae',
    0x8b5e: 'rrrrra989',
    0x8b5f: 'bbbbaabd',
    0x8b60: 'ppppppab08',
    0x8b61: 'ccccbbae',
    0x8b62: 'pppppb889',
    0x8b63: 'ppppbaad',
    0x8b64: 'uuuuua889',
    0x8b65: 'eeeebabd',
    0x8b66: 'uuuuuubb08',
    0x8b67: 'ffffabae',
    0x8b68: '9999999bbaN',
    0x8b69: 'oooobbbd',
    0x8b6a: '9999999bban',
    0x8b6b: 'llllaaae',
    0x8b6c: 'yyyyya899',
    0x8b6d: 'iiiibaae',
    0x8b6e: 'yyyyyybb18',
    0x8b6f: 'jjjjabbd',
    0x8b70: 'hhhhhhab09',
    0x8b71: 'kkkkabbd',
    0x8b72: 'hhhhhb888',
    0x8b73: 'hhhhbaae',
    0x8b74: 'mmmmma888',
    0x8b75: 'mmmmaaae',
    0x8b76: 'mmmmmmbb09',
    0x8b77: 'nnnnbbbd',
    0x8b78: 'ddddddab19',
    0x8b79: 'ggggabae',
    0x8b7a: 'dddddb898',
    0x8b7b: 'ddddbabd',
    0x8b7c: 'aaaaaa898',
    0x8b7d: 'aaaaaabd',
    0x8b7e: 'aaaaaabb19',
    0x8b7f: 'bbbbbbae',
    0x8b80: 'ccccca998',
    0x8b81: 'ccccabbd',
    0x8b82: 'ccccccbb19',
    0x8b83: 'ppppaabe',
    0x8b84: 'eeeeea898',
    0x8b85: 'eeeeaaae',
    0x8b86: 'eeeeeeba09',
    0x8b87: 'ffffbbbd',
    0x8b88: 'llllllaa19',
    0x8b89: 'ooooabae',
    0x8b8a: 'lllllb888',
    0x8b8b: 'llllbabd',
    0x8b8c: 'iiiiia888',
    0x8b8d: 'iiiiaabd',
    0x8b8e: 'iiiiiiba19',
    0x8b8f: 'jjjjbbae',
    0x8b90: 'xxxxxxaa08',
    0x8b91: 'kkkkbbae',
    0x8b92: 'xxxxxb899',
    0x8b93: 'hhhhaabd',
    0x8b94: '11111111bbaJ',
    0x8b95: 'mmmmbabd',
    0x8b96: '11111111bbaZ',
    0x8b97: 'nnnnabae',
    0x8b98: 'ttttttaa18',
    0x8b99: 'ggggbbbd',
    0x8b9a: 'tttttb889',
    0x8b9b: 'ddddaaae',
    0x8b9c: 'qqqqqa889',
    0x8b9d: 'aaaabaae',
    0x8b9e: 'qqqqqqba18',
    0x8b9f: 'bbbbabbd',
    0x8ba0: 'sssssb989',
    0x8ba1: 'ccccbaae',
    0x8ba2: 'pppppa889',
    0x8ba3: 'ppppbbad',
    0x8ba4: 'uuuuub889',
    0x8ba5: 'eeeebbbd',
    0x8ba6: 'vvvvva989',
    0x8ba7: 'ffffaaae',
    0x8ba8: '?????b008',
    0x8ba9: 'oooobabd',
    0x8baa: '<<<<<a108',
    0x8bab: 'llllabae',
    0x8bac: 'yyyyyb899',
    0x8bad: 'iiiibbae',
    0x8bae: 'zzzzza999',
    0x8baf: 'jjjjaabd',
    0x8bb0: 'kkkkkb988',
    0x8bb1: 'kkkkaabd',
    0x8bb2: 'hhhhha888',
    0x8bb3: 'hhhhbbae',
    0x8bb4: 'mmmmmb888',
    0x8bb5: 'mmmmabae',
    0x8bb6: 'nnnnna988',
    0x8bb7: 'nnnnbabd',
    0x8bb8: 'gggggb998',
    0x8bb9: 'ggggaaae',
    0x8bba: 'ddddda898',
    0x8bbb: 'ddddbbbd',
    0x8bbc: 'aaaaab898',
    0x8bbd: 'aaaaabbd',
    0x8bbe: 'bbbbba998',
    0x8bbf: 'bbbbbaae',
    0x8bc0: 'cccccb898',
    0x8bc1: 'ccccaaae',
    0x8bc2: 'CCCCCCab48',
    0x8bc3: 'ppppabad',
    0x8bc4: 'eeeeeb998',
    0x8bc5: 'eeeeabbd',
    0x8bc6: 'fffffa898',
    0x8bc7: 'ffffbaae',
    0x8bc8: 'ooooob888',
    0x8bc9: 'ooooaabd',
    0x8bca: 'llllla988',
    0x8bcb: 'llllbbae',
    0x8bcc: 'iiiiib988',
    0x8bcd: 'iiiiabae',
    0x8bce: 'jjjjja888',
    0x8bcf: 'jjjjbabd',
    0x8bd0: 'QQQQQQQQbban',
    0x8bd1: 'kkkkbabd',
    0x8bd2: 'xxxxxa999',
    0x8bd3: 'hhhhabae',
    0x8bd4: 'QQQQQQQQbbaN',
    0x8bd5: 'mmmmbbae',
    0x8bd6: 'QQQQQQQQbbiZ',
    0x8bd7: 'nnnnaabd',
    0x8bd8: 'wwwwwb889',
    0x8bd9: 'ggggbaae',
    0x8bda: 'ttttta989',
    0x8bdb: 'ddddabbd',
    0x8bdc: 'qqqqqb989',
    0x8bdd: 'aaaabbbd',
    0x8bde: 'rrrrra889',
    0x8bdf: 'bbbbaaae',
    0x8be0: 'ppppppba18',
    0x8be1: 'ccccbbbd',
    0x8be2: 'pppppb989',
    0x8be3: 'ppppbabe',
    0x8be4: 'pppppppbbai',
    0x8be5: 'eeeebaae',
    0x8be6: 'pppppppbbaI',
    0x8be7: 'ffffabbd',
    0x8be8: 'ppppppppabhY',
    0x8be9: 'oooobbae',
    0x8bea: 'ppppppppabbL',
    0x8beb: 'llllaabd',
    0x8bec: 'ppppppppabhy',
    0x8bed: 'iiiibabd',
    0x8bee: 'ppppppppabbl',
    0x8bef: 'jjjjabae',
    0x8bf0: 'aaaaaaaabbal',
    0x8bf1: 'aaaaaaaabbad',
    0x8bf2: 'aaaaaaaabbix',
    0x8bf3: 'aaaaaaaabbat',
    0x8bf4: 'aaaaaaaabbaL',
    0x8bf5: 'aaaaaaaabbaD',
    0x8bf6: 'aaaaaaaabbiX',
    0x8bf7: 'aaaaaaaabbaT',
    0x8bf8: 'aaaaaaaabbI',
    0x8bf9: 'aaaaaaaabbY',
    0x8bfa: 'aaaaaaaabbi',
    0x8bfb: 'aaaaaaaabby',
    0x8bfc: 'aaaaaa998',
    0x8bfd: 'aaaaaaae',
    0x8bfe: 'aaaaaaaa09',
    0x8bff: 'aaaaaaaab0',
    0x8c00: 'ccccca089',
    0x8c01: 'ccccabab',
    0x8c02: 'ccccccab08',
    0x8c03: 'ccccccccb1',
    0x8c04: 'cccccccaabI',
    0x8c05: 'cccccccaabY',
    0x8c06: 'cccccccaabi',
    0x8c07: 'cccccccaaby',
    0x8c08: 'ccccccccbabM',
    0x8c09: 'ccccccccbabE',
    0x8c0a: 'ccccccccbahX',
    0x8c0b: 'ccccccccbabU',
    0x8c0c: 'ccccccccbabm',
    0x8c0d: 'ccccccccbabe',
    0x8c0e: 'ccccccccbahx',
    0x8c0f: 'ccccccccbabu',
    0x8c10: 'qqqqqqqqbbbl',
    0x8c11: 'kkkkbbbc',
    0x8c12: 'qqqqqqqqbbhy',
    0x8c13: 'hhhhaaab',
    0x8c14: 'qqqqqqqqbbbL',
    0x8c15: 'mmmmbaab',
    0x8c16: 'qqqqqqqqbbhY',
    0x8c17: 'nnnnabbc',
    0x8c18: 'qqqqqqqabbH',
    0x8c19: 'ggggbbab',
    0x8c1a: 'qqqqqqqabbh',
    0x8c1b: 'ddddaabc',
    0x8c1c: 'qqqqqa198',
    0x8c1d: 'aaaababc',
    0x8c1e: 'qqqqqqaa09',
    0x8c1f: 'bbbbabab',
    0x8c20: 'sssssb098',
    0x8c21: 'ccccbabc',
    0x8c22: 'pppppa198',
    0x8c23: 'ppppbbbb',
    0x8c24: 'uuuuub198',
    0x8c25: 'eeeebbab',
    0x8c26: 'vvvvva098',
    0x8c27: 'ffffaabc',
    0x8c28: 'PPPPPPPPabiZ',
    0x8c29: 'oooobaab',
    0x8c2a: 'PPPPPPPPabaN',
    0x8c2b: 'llllabbc',
    0x8c2c: 'yyyyyb188',
    0x8c2d: 'iiiibbbc',
    0x8c2e: 'zzzzza088',
    0x8c2f: 'jjjjaaab',
    0x8c30: 'kkkkkb099',
    0x8c31: 'kkkkaaab',
    0x8c32: 'hhhhha199',
    0x8c33: 'hhhhbbbc',
    0x8c34: 'mmmmmb199',
    0x8c35: 'mmmmabbc',
    0x8c36: 'nnnnna099',
    0x8c37: 'nnnnbaab',
    0x8c38: 'gggggb089',
    0x8c39: 'ggggaabc',
    0x8c3a: 'ddddda189',
    0x8c3b: 'ddddbbab',
    0x8c3c: 'aaaaab189',
    0x8c3d: 'aaaaabab',
    0x8c3e: 'bbbbba089',
    0x8c3f: 'bbbbbabc',
    0x8c40: 'cccccb189',
    0x8c41: 'ccccaabc',
    0x8c42: 'CCCCCCbb59',
    0x8c43: 'ppppabbb',
    0x8c44: 'eeeeeb089',
    0x8c45: 'eeeeabab',
    0x8c46: 'fffffa189',
    0x8c47: 'ffffbabc',
    0x8c48: 'ooooob199',
    0x8c49: 'ooooaaab',
    0x8c4a: 'llllla099',
    0x8c4b: 'llllbbbc',
    0x8c4c: 'iiiiib099',
    0x8c4d: 'iiiiabbc',
    0x8c4e: 'jjjjja199',
    0x8c4f: 'jjjjbaab',
    0x8c50: 'XXXXXXaa48',
    0x8c51: 'kkkkbaab',
    0x8c52: 'xxxxxa088',
    0x8c53: 'hhhhabbc',
    0x8c54: '=====b159',
    0x8c55: 'mmmmbbbc',
    0x8c56: '>>>>>a059',
    0x8c57: 'nnnnaaab',
    0x8c58: 'wwwwwb198',
    0x8c59: 'ggggbabc',
    0x8c5a: 'ttttta098',
    0x8c5b: 'ddddabab',
    0x8c5c: 'qqqqqb098',
    0x8c5d: 'aaaabbab',
    0x8c5e: 'rrrrra198',
    0x8c5f: 'bbbbaabc',
    0x8c60: 'ppppppaa09',
    0x8c61: 'ccccbbab',
    0x8c62: 'pppppb098',
    0x8c63: 'ppppbaac',
    0x8c64: 'uuuuua098',
    0x8c65: 'eeeebabc',
    0x8c66: 'uuuuuuba09',
    0x8c67: 'ffffabab',
    0x8c68: '00000000abaZ',
    0x8c69: 'oooobbbc',
    0x8c6a: '00000000abaJ',
    0x8c6b: 'llllaaab',
    0x8c6c: 'yyyyya088',
    0x8c6d: 'iiiibaab',
    0x8c6e: 'yyyyyyba19',
    0x8c6f: 'jjjjabbc',
    0x8c70: 'hhhhhhaa08',
    0x8c71: 'kkkkabbc',
    0x8c72: 'hhhhhb099',
    0x8c73: 'hhhhbaab',
    0x8c74: 'mmmmma099',
    0x8c75: 'mmmmaaab',
    0x8c76: 'mmmmmmba08',
    0x8c77: 'nnnnbbbc',
    0x8c78: 'ddddddaa18',
    0x8c79: 'ggggabab',
    0x8c7a: 'dddddb089',
    0x8c7b: 'ddddbabc',
    0x8c7c: 'aaaaaa089',
    0x8c7d: 'aaaaaabc',
    0x8c7e: 'aaaaaaba18',
    0x8c7f: 'bbbbbbab',
    0x8c80: 'ccccca189',
    0x8c81: 'ccccabbc',
    0x8c82: 'ccccccba18',
    0x8c83: 'ppppaabb',
    0x8c84: 'eeeeea089',
    0x8c85: 'eeeeaaab',
    0x8c86: 'eeeeeebb08',
    0x8c87: 'ffffbbbc',
    0x8c88: 'llllllab18',
    0x8c89: 'ooooabab',
    0x8c8a: 'lllllb099',
    0x8c8b: 'llllbabc',
    0x8c8c: 'iiiiia099',
    0x8c8d: 'iiiiaabc',
    0x8c8e: 'iiiiiibb18',
    0x8c8f: 'jjjjbbab',
    0x8c90: 'xxxxxxab09',
    0x8c91: 'kkkkbbab',
    0x8c92: 'xxxxxb088',
    0x8c93: 'hhhhaabc',
    0x8c94: '8888888abbo',
    0x8c95: 'mmmmbabc',
    0x8c96: '8888888abbO',
    0x8c97: 'nnnnabab',
    0x8c98: 'ttttttab19',
    0x8c99: 'ggggbbbc',
    0x8c9a: 'tttttb098',
    0x8c9b: 'ddddaaab',
    0x8c9c: 'qqqqqa098',
    0x8c9d: 'aaaabaab',
    0x8c9e: 'qqqqqqbb19',
    0x8c9f: 'bbbbabbc',
    0x8ca0: 'sssssb198',
    0x8ca1: 'ccccbaab',
    0x8ca2: 'pppppa098',
    0x8ca3: 'ppppbbac',
    0x8ca4: 'uuuuub098',
    0x8ca5: 'eeeebbbc',
    0x8ca6: 'vvvvva198',
    0x8ca7: 'ffffaaab',
    0x8ca8: '?????b059',
    0x8ca9: 'oooobabc',
    0x8caa: '<<<<<a159',
    0x8cab: 'llllabab',
    0x8cac: 'yyyyyb088',
    0x8cad: 'iiiibbab',
    0x8cae: 'zzzzza188',
    0x8caf: 'jjjjaabc',
    0x8cb0: 'kkkkkb199',
    0x8cb1: 'kkkkaabc',
    0x8cb2: 'hhhhha099',
    0x8cb3: 'hhhhbbab',
    0x8cb4: 'mmmmmb099',
    0x8cb5: 'mmmmabab',
    0x8cb6: 'nnnnna199',
    0x8cb7: 'nnnnbabc',
    0x8cb8: 'gggggb189',
    0x8cb9: 'ggggaaab',
    0x8cba: 'ddddda089',
    0x8cbb: 'ddddbbbc',
    0x8cbc: 'aaaaab089',
    0x8cbd: 'aaaaabbc',
    0x8cbe: 'bbbbba189',
    0x8cbf: 'bbbbbaab',
    0x8cc0: 'cccccb089',
    0x8cc1: 'ccccaaab',
    0x8cc2: 'CCCCCCaa49',
    0x8cc3: 'ppppabac',
    0x8cc4: 'eeeeeb189',
    0x8cc5: 'eeeeabbc',
    0x8cc6: 'fffffa089',
    0x8cc7: 'ffffbaab',
    0x8cc8: 'ooooob099',
    0x8cc9: 'ooooaabc',
    0x8cca: 'llllla199',
    0x8ccb: 'llllbbab',
    0x8ccc: 'iiiiib199',
    0x8ccd: 'iiiiabab',
    0x8cce: 'jjjjja099',
    0x8ccf: 'jjjjbabc',
    0x8cd0: 'XXXXXXbb58',
    0x8cd1: 'kkkkbabc',
    0x8cd2: 'xxxxxa188',
    0x8cd3: 'hhhhabab',
    0x8cd4: 'XXXXXXXabbk',
    0x8cd5: 'mmmmbbab',
    0x8cd6: 'XXXXXXXabbK',
    0x8cd7: 'nnnnaabc',
    0x8cd8: 'wwwwwb098',
    0x8cd9: 'ggggbaab',
    0x8cda: 'ttttta198',
    0x8cdb: 'ddddabbc',
    0x8cdc: 'qqqqqb198',
    0x8cdd: 'aaaabbbc',
    0x8cde: 'rrrrra098',
    0x8cdf: 'bbbbaaab',
    0x8ce0: 'ppppppbb19',
    0x8ce1: 'ccccbbbc',
    0x8ce2: 'pppppb198',
    0x8ce3: 'ppppbabb',
    0x8ce4: 'pppppppabbh',
    0x8ce5: 'eeeebaab',
    0x8ce6: 'pppppppabbH',
    0x8ce7: 'ffffabbc',
    0x8ce8: 'yyyyyyybbaI',
    0x8ce9: 'oooobbab',
    0x8cea: 'yyyyyyybbai',
    0x8ceb: 'llllaabc',
    0x8cec: 'yyyyya188',
    0x8ced: 'iiiibabc',
    0x8cee: 'yyyyyyab09',
    0x8cef: 'jjjjabab',
    0x8cf0: 'hhhhhhbb18',
    0x8cf1: 'kkkkabab',
    0x8cf2: 'hhhhhb199',
    0x8cf3: 'hhhhbabc',
    0x8cf4: 'hhhhhhhabbi',
    0x8cf5: 'hhhhhhhabby',
    0x8cf6: 'hhhhhhhabbI',
    0x8cf7: 'hhhhhhhabbY',
    0x8cf8: 'aaaaaaabbaH',
    0x8cf9: 'aaaaaaabbaX',
    0x8cfa: 'aaaaaaabbah',
    0x8cfb: 'aaaaaaabbax',
    0x8cfc: 'aaaaaa189',
    0x8cfd: 'aaaaaaab',
    0x8cfe: 'aaaaaaab08',
    0x8cff: 'bbbbbbbc',
    0x8d00: 'ccccca099',
    0x8d01: 'ccccabac',
    0x8d02: 'ccccccaa18',
    0x8d03: 'eeeeeeebaby',
    0x8d04: 'cccccccbbaH',
    0x8d05: 'cccccccbbaX',
    0x8d06: 'cccccccbbah',
    0x8d07: 'cccccccbbax',
    0x8d08: 'iiiiiiibaaH',
    0x8d09: 'iiiiiiibaaX',
    0x8d0a: 'iiiiiiibaah',
    0x8d0b: 'iiiiiiibaax',
    0x8d0c: 'iiiiia189',
    0x8d0d: 'iiiiaaac',
    0x8d0e: 'iiiiiiab18',
    0x8d0f: 'jjjjbbbb',
    0x8d10: 'xxxxxxbb09',
    0x8d11: 'kkkkbbbb',
    0x8d12: 'xxxxxb198',
    0x8d13: 'hhhhaaac',
    0x8d14: 'xxxxxxxaabh',
    0x8d15: 'mmmmbaac',
    0x8d16: 'xxxxxxxaabH',
    0x8d17: 'nnnnabbb',
    0x8d18: 'qqqqqqqbaaI',
    0x8d19: 'ggggbbac',
    0x8d1a: 'qqqqqqqbaai',
    0x8d1b: 'ddddaabb',
    0x8d1c: 'qqqqqa188',
    0x8d1d: 'aaaababb',
    0x8d1e: 'qqqqqqab19',
    0x8d1f: 'bbbbabac',
    0x8d20: 'sssssb088',
    0x8d21: 'ccccbabb',
    0x8d22: 'pppppa188',
    0x8d23: 'ppppbbbc',
    0x8d24: 'uuuuub188',
    0x8d25: 'eeeebbac',
    0x8d26: 'vvvvva088',
    0x8d27: 'ffffaabb',
    0x8d28: 'YYYYYYYbaaJ',
    0x8d29: 'oooobaac',
    0x8d2a: 'YYYYYYYbaaj',
    0x8d2b: 'llllabbb',
    0x8d2c: 'yyyyyb198',
    0x8d2d: 'iiiibbbb',
    0x8d2e: 'zzzzza098',
    0x8d2f: 'jjjjaaac',
    0x8d30: 'kkkkkb089',
    0x8d31: 'kkkkaaac',
    0x8d32: 'hhhhha189',
    0x8d33: 'hhhhbbbb',
    0x8d34: 'mmmmmb189',
    0x8d35: 'mmmmabbb',
    0x8d36: 'nnnnna089',
    0x8d37: 'nnnnbaac',
    0x8d38: 'gggggb099',
    0x8d39: 'ggggaabb',
    0x8d3a: 'ddddda199',
    0x8d3b: 'ddddbbac',
    0x8d3c: 'aaaaab199',
    0x8d3d: 'aaaaabac',
    0x8d3e: 'bbbbba099',
    0x8d3f: 'bbbbbabb',
    0x8d40: 'cccccb199',
    0x8d41: 'ccccaabb',
    0x8d42: 'CCCCCCba49',
    0x8d43: 'ppppabbc',
    0x8d44: 'eeeeeb099',
    0x8d45: 'eeeeabac',
    0x8d46: 'fffffa199',
    0x8d47: 'ffffbabb',
    0x8d48: 'ooooob189',
    0x8d49: 'ooooaaac',
    0x8d4a: 'llllla089',
    0x8d4b: 'llllbbbb',
    0x8d4c: 'iiiiib089',
    0x8d4d: 'iiiiabbb',
    0x8d4e: 'jjjjja189',
    0x8d4f: 'jjjjbaac',
    0x8d50: 'XXXXXXab58',
    0x8d51: 'kkkkbaac',
    0x8d52: 'xxxxxa098',
    0x8d53: 'hhhhabbb',
    0x8d54: '=====b149',
    0x8d55: 'mmmmbbbb',
    0x8d56: '>>>>>a049',
    0x8d57: 'nnnnaaac',
    0x8d58: 'wwwwwb188',
    0x8d59: 'ggggbabb',
    0x8d5a: 'ttttta088',
    0x8d5b: 'ddddabac',
    0x8d5c: 'qqqqqb088',
    0x8d5d: 'aaaabbac',
    0x8d5e: 'rrrrra188',
    0x8d5f: 'bbbbaabb',
    0x8d60: 'ppppppab19',
    0x8d61: 'ccccbbac',
    0x8d62: 'pppppb088',
    0x8d63: 'ppppbaab',
    0x8d64: 'uuuuua088',
    0x8d65: 'eeeebabb',
    0x8d66: 'uuuuuubb19',
    0x8d67: 'ffffabac',
    0x8d68: '9999999baaN',
    0x8d69: 'oooobbbb',
    0x8d6a: '9999999baan',
    0x8d6b: 'llllaaac',
    0x8d6c: 'yyyyya098',
    0x8d6d: 'iiiibaac',
    0x8d6e: 'yyyyyybb09',
    0x8d6f: 'jjjjabbb',
    0x8d70: 'hhhhhhab18',
    0x8d71: 'kkkkabbb',
    0x8d72: 'hhhhhb089',
    0x8d73: 'hhhhbaac',
    0x8d74: 'mmmmma089',
    0x8d75: 'mmmmaaac',
    0x8d76: 'mmmmmmbb18',
    0x8d77: 'nnnnbbbb',
    0x8d78: 'ddddddab08',
    0x8d79: 'ggggabac',
    0x8d7a: 'dddddb099',
    0x8d7b: 'ddddbabb',
    0x8d7c: 'aaaaaa099',
    0x8d7d: 'aaaaaabb',
    0x8d7e: 'aaaaaabb08',
    0x8d7f: 'bbbbbbac',
    0x8d80: 'ccccca199',
    0x8d81: 'ccccabbb',
    0x8d82: 'ccccccbb08',
    0x8d83: 'ppppaabc',
    0x8d84: 'eeeeea099',
    0x8d85: 'eeeeaaac',
    0x8d86: 'eeeeeeba18',
    0x8d87: 'ffffbbbb',
    0x8d88: 'llllllaa08',
    0x8d89: 'ooooabac',
    0x8d8a: 'lllllb089',
    0x8d8b: 'llllbabb',
    0x8d8c: 'iiiiia089',
    0x8d8d: 'iiiiaabb',
    0x8d8e: 'iiiiiiba08',
    0x8d8f: 'jjjjbbac',
    0x8d90: 'xxxxxxaa19',
    0x8d91: 'kkkkbbac',
    0x8d92: 'xxxxxb098',
    0x8d93: 'hhhhaabb',
    0x8d94: '11111111abaJ',
    0x8d95: 'mmmmbabb',
    0x8d96: '11111111abaZ',
    0x8d97: 'nnnnabac',
    0x8d98: 'ttttttaa09',
    0x8d99: 'ggggbbbb',
    0x8d9a: 'tttttb088',
    0x8d9b: 'ddddaaac',
    0x8d9c: 'qqqqqa088',
    0x8d9d: 'aaaabaac',
    0x8d9e: 'qqqqqqba09',
    0x8d9f: 'bbbbabbb',
    0x8da0: 'sssssb188',
    0x8da1: 'ccccbaac',
    0x8da2: 'pppppa088',
    0x8da3: 'ppppbbab',
    0x8da4: 'uuuuub088',
    0x8da5: 'eeeebbbb',
    0x8da6: 'vvvvva188',
    0x8da7: 'ffffaaac',
    0x8da8: '?????b049',
    0x8da9: 'oooobabb',
    0x8daa: '<<<<<a149',
    0x8dab: 'llllabac',
    0x8dac: 'yyyyyb098',
    0x8dad: 'iiiibbac',
    0x8dae: 'zzzzza198',
    0x8daf: 'jjjjaabb',
    0x8db0: 'kkkkkb189',
    0x8db1: 'kkkkaabb',
    0x8db2: 'hhhhha089',
    0x8db3: 'hhhhbbac',
    0x8db4: 'mmmmmb089',
    0x8db5: 'mmmmabac',
    0x8db6: 'nnnnna189',
    0x8db7: 'nnnnbabb',
    0x8db8: 'gggggb199',
    0x8db9: 'ggggaaac',
    0x8dba: 'ddddda099',
    0x8dbb: 'ddddbbbb',
    0x8dbc: 'aaaaab099',
    0x8dbd: 'aaaaabbb',
    0x8dbe: 'bbbbba199',
    0x8dbf: 'bbbbbaac',
    0x8dc0: 'cccccb099',
    0x8dc1: 'ccccaaac',
    0x8dc2: 'CCCCCCab59',
    0x8dc3: 'ppppabab',
    0x8dc4: 'eeeeeb199',
    0x8dc5: 'eeeeabbb',
    0x8dc6: 'fffffa099',
    0x8dc7: 'ffffbaac',
    0x8dc8: 'ooooob089',
    0x8dc9: 'ooooaabb',
    0x8dca: 'llllla189',
    0x8dcb: 'llllbbac',
    0x8dcc: 'iiiiib189',
    0x8dcd: 'iiiiabac',
    0x8dce: 'jjjjja089',
    0x8dcf: 'jjjjbabb',
    0x8dd0: 'CCCCCCCCCbbbp',
    0x8dd1: 'kkkkbabb',
    0x8dd2: 'xxxxxa198',
    0x8dd3: 'hhhhabac',
    0x8dd4: 'CCCCCCCCCbbdc',
    0x8dd5: 'mmmmbbac',
    0x8dd6: 'CCCCCCCCCbbbh',
    0x8dd7: 'nnnnaabb',
    0x8dd8: 'wwwwwb088',
    0x8dd9: 'ggggbaac',
    0x8dda: 'ttttta188',
    0x8ddb: 'ddddabbb',
    0x8ddc: 'qqqqqb188',
    0x8ddd: 'aaaabbbb',
    0x8dde: 'rrrrra088',
    0x8ddf: 'bbbbaaac',
    0x8de0: 'aaaaaaaaababC',
    0x8de1: 'aaaaaaaaababG',
    0x8de2: 'aaaaaaaaababK',
    0x8de3: 'aaaaaaaaababO',
    0x8de4: 'aaaaaaaaababS',
    0x8de5: 'aaaaaaaaababW',
    0x8de6: 'aaaaaaaaabadX',
    0x8de7: 'aaaaaaaaabahZ',
    0x8de8: 'aaaaaaaaababc',
    0x8de9: 'aaaaaaaaababg',
    0x8dea: 'aaaaaaaaababk',
    0x8deb: 'aaaaaaaaababo',
    0x8dec: 'aaaaaaaaababs',
    0x8ded: 'aaaaaaaaababw',
    0x8dee: 'aaaaaaaaabadx',
    0x8def: 'aaaaaaaaabahz',
    0x8df0: 'aaaaaaaaabal',
    0x8df1: 'aaaaaaaaabad',
    0x8df2: 'aaaaaaaaabix',
    0x8df3: 'aaaaaaaaabat',
    0x8df4: 'aaaaaaaaabaL',
    0x8df5: 'aaaaaaaaabaD',
    0x8df6: 'aaaaaaaaabiX',
    0x8df7: 'aaaaaaaaabaT',
    0x8df8: 'aaaaaaaaabI',
    0x8df9: 'aaaaaaaaabY',
    0x8dfa: 'aaaaaaaaabi',
    0x8dfb: 'aaaaaaaaaby',
    0x8dfc: 'aaaaaa199',
    0x8dfd: 'aaaaaaac',
    0x8dfe: 'aaaaaaaa18',
    0x8dff: 'aaaaaaaaab9',
    0x8e00: 'ccccca088',
    0x8e01: 'ccccabca',
    0x8e02: 'ccccccab18',
    0x8e03: 'cccccccab1',
    0x8e04: 'cccccccabaH',
    0x8e05: 'cccccccabaX',
    0x8e06: 'cccccccabah',
    0x8e07: 'cccccccabax',
    0x8e08: 'ccccccccabaL',
    0x8e09: 'ccccccccabaD',
    0x8e0a: 'ccccccccabiX',
    0x8e0b: 'ccccccccabaT',
    0x8e0c: 'ccccccccabal',
    0x8e0d: 'ccccccccabad',
    0x8e0e: 'ccccccccabix',
    0x8e0f: 'ccccccccabat',
    0x8e10: 'cccccccccabar',
    0x8e11: 'cccccccccabav',
    0x8e12: 'cccccccccabaz',
    0x8e13: 'cccccccccabiz',
    0x8e14: 'cccccccccabab',
    0x8e15: 'cccccccccabaf',
    0x8e16: 'cccccccccabaj',
    0x8e17: 'cccccccccaban',
    0x8e18: 'cccccccccabaR',
    0x8e19: 'cccccccccabaV',
    0x8e1a: 'cccccccccabaZ',
    0x8e1b: 'cccccccccabiZ',
    0x8e1c: 'cccccccccabaB',
    0x8e1d: 'aaaababa',
    0x8e1e: 'cccccccccabaJ',
    0x8e1f: 'bbbbabca',
    0x8e20: 'ccccccccccabao',
    0x8e21: 'ccccbaba',
    0x8e22: 'ccccccccccabak',
    0x8e23: 'ccccccccccabai',
    0x8e24: 'ccccccccccabag',
    0x8e25: 'ccccccccccabae',
    0x8e26: 'ccccccccccabac',
    0x8e27: 'ccccccccccabaa',
    0x8e28: 'ccccccccccabkz',
    0x8e29: 'ccccccccccabiy',
    0x8e2a: 'ccccccccccabcz',
    0x8e2b: 'ccccccccccabay',
    0x8e2c: 'ccccccccccabaw',
    0x8e2d: 'ccccccccccabau',
    0x8e2e: 'ccccccccccabas',
    0x8e2f: 'ccccccccccabaq',
    0x8e30: 'ccccccccccabaO',
    0x8e31: 'ccccccccccabaM',
    0x8e32: 'ccccccccccabaK',
    0x8e33: 'ccccccccccabaI',
    0x8e34: 'ccccccccccabaG',
    0x8e35: 'ccccccccccabaE',
    0x8e36: 'ccccccccccabaC',
    0x8e37: 'ccccccccccabaA',
    0x8e38: 'ccccccccccabkZ',
    0x8e39: 'ccccccccccabiY',
    0x8e3a: 'ccccccccccabcZ',
    0x8e3b: 'ccccccccccabaY',
    0x8e3c: 'aaaaab188',
    0x8e3d: 'aaaaabca',
    0x8e3e: 'bbbbba088',
    0x8e3f: 'bbbbbaba',
    0x8e40: 'cccccb188',
    0x8e41: 'ccccaaba',
    0x8e42: 'cccccccccccabbe',
    0x8e43: 'cccccccccccabbd',
    0x8e44: 'cccccccccccabbc',
    0x8e45: 'cccccccccccabbb',
    0x8e46: 'cccccccccccabba',
    0x8e47: 'cccccccccccabdc',
    0x8e48: 'cccccccccccabbo',
    0x8e49: 'cccccccccccabbn',
    0x8e4a: 'cccccccccccabbm',
    0x8e4b: 'cccccccccccabbl',
    0x8e4c: 'cccccccccccabbk',
    0x8e4d: 'cccccccccccabbj',
    0x8e4e: 'cccccccccccabbi',
    0x8e4f: 'cccccccccccabbh',
    0x8e50: 'cccccccccccabbw',
    0x8e51: 'cccccccccccabbv',
    0x8e52: 'cccccccccccabbu',
    0x8e53: 'cccccccccccabbt',
    0x8e54: 'cccccccccccabbs',
    0x8e55: 'cccccccccccabbr',
    0x8e56: 'cccccccccccabbq',
    0x8e57: 'cccccccccccabbp',
    0x8e58: 'cccccccccccabhz',
    0x8e59: 'cccccccccccabjz',
    0x8e5a: 'cccccccccccabhx',
    0x8e5b: 'cccccccccccabhy',
    0x8e5c: 'cccccccccccabdx',
    0x8e5d: 'aaaabbca',
    0x8e5e: 'cccccccccccabby',
    0x8e5f: 'bbbbaaba',
    0x8e60: 'cccccccccccabbG',
    0x8e61: 'ccccbbca',
    0x8e62: 'cccccccccccabbE',
    0x8e63: 'cccccccccccabbD',
    0x8e64: 'cccccccccccabbC',
    0x8e65: 'cccccccccccabbB',
    0x8e66: 'cccccccccccabbA',
    0x8e67: 'cccccccccccabdC',
    0x8e68: 'cccccccccccabbO',
    0x8e69: 'cccccccccccabbN',
    0x8e6a: 'cccccccccccabbM',
    0x8e6b: 'cccccccccccabbL',
    0x8e6c: 'cccccccccccabbK',
    0x8e6d: 'cccccccccccabbJ',
    0x8e6e: 'cccccccccccabbI',
    0x8e6f: 'cccccccccccabbH',
    0x8e70: 'cccccccccccabbW',
    0x8e71: 'cccccccccccabbV',
    0x8e72: 'cccccccccccabbU',
    0x8e73: 'cccccccccccabbT',
    0x8e74: 'cccccccccccabbS',
    0x8e75: 'cccccccccccabbR',
    0x8e76: 'cccccccccccabbQ',
    0x8e77: 'cccccccccccabbP',
    0x8e78: 'cccccccccccabhZ',
    0x8e79: 'cccccccccccabjZ',
    0x8e7a: 'cccccccccccabhX',
    0x8e7b: 'cccccccccccabhY',
    0x8e7c: 'aaaaaa088',
    0x8e7d: 'aaaaaaba',
    0x8e7e: 'aaaaaaba08',
    0x8e7f: 'bbbbbbca',
    0x8e80: 'ccccca188',
    0x8e81: 'ccccabba',
    0x8e82: 'ccccccba08',
    0x8e83: 'ppppaadc',
    0x8e84: 'eeeeea088',
    0x8e85: 'eeeeaaca',
    0x8e86: 'eeeeeebb18',
    0x8e87: 'ffffbbba',
    0x8e88: 'llllllab08',
    0x8e89: 'ooooabca',
    0x8e8a: 'lllllb098',
    0x8e8b: 'llllbaba',
    0x8e8c: 'iiiiia098',
    0x8e8d: 'iiiiaaba',
    0x8e8e: 'iiiiiibb08',
    0x8e8f: 'jjjjbbca',
    0x8e90: 'xxxxxxab19',
    0x8e91: 'kkkkbbca',
    0x8e92: 'xxxxxb089',
    0x8e93: 'hhhhaaba',
    0x8e94: '8888888aaan',
    0x8e95: 'mmmmbaba',
    0x8e96: '8888888aaaN',
    0x8e97: 'nnnnabca',
    0x8e98: 'ttttttab09',
    0x8e99: 'ggggbbba',
    0x8e9a: 'tttttb099',
    0x8e9b: 'ddddaaca',
    0x8e9c: 'qqqqqa099',
    0x8e9d: 'aaaabaca',
    0x8e9e: 'qqqqqqbb09',
    0x8e9f: 'bbbbabba',
    0x8ea0: 'sssssb199',
    0x8ea1: 'ccccbaca',
    0x8ea2: 'pppppa099',
    0x8ea3: 'ppppbbaa',
    0x8ea4: 'uuuuub099',
    0x8ea5: 'eeeebbba',
    0x8ea6: 'vvvvva199',
    0x8ea7: 'ffffaaca',
    0x8ea8: '?????b058',
    0x8ea9: 'oooobaba',
    0x8eaa: '<<<<<a158',
    0x8eab: 'llllabca',
    0x8eac: 'yyyyyb089',
    0x8ead: 'iiiibbca',
    0x8eae: 'zzzzza189',
    0x8eaf: 'jjjjaaba',
    0x8eb0: 'kkkkkb198',
    0x8eb1: 'kkkkaaba',
    0x8eb2: 'hhhhha098',
    0x8eb3: 'hhhhbbca',
    0x8eb4: 'mmmmmb098',
    0x8eb5: 'mmmmabca',
    0x8eb6: 'nnnnna198',
    0x8eb7: 'nnnnbaba',
    0x8eb8: 'gggggb188',
    0x8eb9: 'ggggaaca',
    0x8eba: 'ddddda088',
    0x8ebb: 'ddddbbba',
    0x8ebc: 'aaaaab088',
    0x8ebd: 'aaaaabba',
    0x8ebe: 'bbbbba188',
    0x8ebf: 'bbbbbaca',
    0x8ec0: 'cccccb088',
    0x8ec1: 'ccccaaca',
    0x8ec2: 'CCCCCCaa59',
    0x8ec3: 'ppppabaa',
    0x8ec4: 'eeeeeb188',
    0x8ec5: 'eeeeabba',
    0x8ec6: 'fffffa088',
    0x8ec7: 'ffffbaca',
    0x8ec8: 'ooooob098',
    0x8ec9: 'ooooaaba',
    0x8eca: 'llllla198',
    0x8ecb: 'llllbbca',
    0x8ecc: 'iiiiib198',
    0x8ecd: 'iiiiabca',
    0x8ece: 'jjjjja098',
    0x8ecf: 'jjjjbaba',
    0x8ed0: 'XXXXXXbb48',
    0x8ed1: 'kkkkbaba',
    0x8ed2: 'xxxxxa189',
    0x8ed3: 'hhhhabca',
    0x8ed4: 'XXXXXXXaaaj',
    0x8ed5: 'mmmmbbca',
    0x8ed6: 'XXXXXXXaaaJ',
    0x8ed7: 'nnnnaaba',
    0x8ed8: 'wwwwwb099',
    0x8ed9: 'ggggbaca',
    0x8eda: 'ttttta199',
    0x8edb: 'ddddabba',
    0x8edc: 'qqqqqb199',
    0x8edd: 'aaaabbba',
    0x8ede: 'rrrrra099',
    0x8edf: 'bbbbaaca',
    0x8ee0: 'ppppppbb09',
    0x8ee1: 'ccccbbba',
    0x8ee2: 'pppppb199',
    0x8ee3: 'ppppbadc',
    0x8ee4: 'pppppppaaai',
    0x8ee5: 'eeeebaca',
    0x8ee6: 'pppppppaaaI',
    0x8ee7: 'ffffabba',
    0x8ee8: 'yyyyyyybabH',
    0x8ee9: 'oooobbca',
    0x8eea: 'yyyyyyybabh',
    0x8eeb: 'llllaaba',
    0x8eec: 'yyyyya189',
    0x8eed: 'iiiibaba',
    0x8eee: 'yyyyyyab19',
    0x8eef: 'jjjjabca',
    0x8ef0: 'hhhhhhbb08',
    0x8ef1: 'kkkkabca',
    0x8ef2: 'hhhhhb198',
    0x8ef3: 'hhhhbaba',
    0x8ef4: 'hhhhhhhaaah',
    0x8ef5: 'hhhhhhhaaax',
    0x8ef6: 'hhhhhhhaaaH',
    0x8ef7: 'hhhhhhhaaaX',
    0x8ef8: 'aaaaaaababI',
    0x8ef9: 'aaaaaaababY',
    0x8efa: 'aaaaaaababi',
    0x8efb: 'aaaaaaababy',
    0x8efc: 'aaaaaa188',
    0x8efd: 'aaaaaaca',
    0x8efe: 'aaaaaaab18',
    0x8eff: 'bbbbbbba',
    0x8f00: 'ccccca098',
    0x8f01: 'ccccabaa',
    0x8f02: 'ccccccaa08',
    0x8f03: 'eeeeeeebbax',
    0x8f04: 'cccccccbabI',
    0x8f05: 'cccccccbabY',
    0x8f06: 'cccccccbabi',
    0x8f07: 'cccccccbaby',
    0x8f08: 'iiiiiiibbbI',
    0x8f09: 'iiiiiiibbbY',
    0x8f0a: 'iiiiiiibbbi',
    0x8f0b: 'iiiiiiibbby',
    0x8f0c: 'iiiiia188',
    0x8f0d: 'iiiiaaaa',
    0x8f0e: 'iiiiiiab08',
    0x8f0f: 'jjjjbbdc',
    0x8f10: 'xxxxxxbb19',
    0x8f11: 'kkkkbbdc',
    0x8f12: 'xxxxxb199',
    0x8f13: 'hhhhaaaa',
    0x8f14: 'xxxxxxxabai',
    0x8f15: 'mmmmbaaa',
    0x8f16: 'xxxxxxxabaI',
    0x8f17: 'nnnnabdc',
    0x8f18: 'qqqqqqqbbbH',
    0x8f19: 'ggggbbaa',
    0x8f1a: 'qqqqqqqbbbh',
    0x8f1b: 'ddddaadc',
    0x8f1c: 'qqqqqa189',
    0x8f1d: 'aaaabadc',
    0x8f1e: 'qqqqqqab09',
    0x8f1f: 'bbbbabaa',
    0x8f20: 'sssssb089',
    0x8f21: 'ccccbadc',
    0x8f22: 'pppppa189',
    0x8f23: 'ppppbbba',
    0x8f24: 'uuuuub189',
    0x8f25: 'eeeebbaa',
    0x8f26: 'vvvvva089',
    0x8f27: 'ffffaadc',
    0x8f28: 'YYYYYYYbbbK',
    0x8f29: 'oooobaaa',
    0x8f2a: 'YYYYYYYbbbk',
    0x8f2b: 'llllabdc',
    0x8f2c: 'yyyyyb199',
    0x8f2d: 'iiiibbdc',
    0x8f2e: 'zzzzza099',
    0x8f2f: 'jjjjaaaa',
    0x8f30: 'kkkkkb088',
    0x8f31: 'kkkkaaaa',
    0x8f32: 'hhhhha188',
    0x8f33: 'hhhhbbdc',
    0x8f34: 'mmmmmb188',
    0x8f35: 'mmmmabdc',
    0x8f36: 'nnnnna088',
    0x8f37: 'nnnnbaaa',
    0x8f38: 'gggggb098',
    0x8f39: 'ggggaadc',
    0x8f3a: 'ddddda198',
    0x8f3b: 'ddddbbaa',
    0x8f3c: 'aaaaab198',
    0x8f3d: 'aaaaabaa',
    0x8f3e: 'bbbbba098',
    0x8f3f: 'bbbbbadc',
    0x8f40: 'cccccb198',
    0x8f41: 'ccccaadc',
    0x8f42: 'CCCCCCba59',
    0x8f43: 'ppppabba',
    0x8f44: 'eeeeeb098',
    0x8f45: 'eeeeabaa',
    0x8f46: 'fffffa198',
    0x8f47: 'ffffbadc',
    0x8f48: 'ooooob188',
    0x8f49: 'ooooaaaa',
    0x8f4a: 'llllla088',
    0x8f4b: 'llllbbdc',
    0x8f4c: 'iiiiib088',
    0x8f4d: 'iiiiabdc',
    0x8f4e: 'jjjjja188',
    0x8f4f: 'jjjjbaaa',
    0x8f50: 'XXXXXXab48',
    0x8f51: 'kkkkbaaa',
    0x8f52: 'xxxxxa099',
    0x8f53: 'hhhhabdc',
    0x8f54: '=====b148',
    0x8f55: 'mmmmbbdc',
    0x8f56: '>>>>>a048',
    0x8f57: 'nnnnaaaa',
    0x8f58: 'wwwwwb189',
    0x8f59: 'ggggbadc',
    0x8f5a: 'ttttta089',
    0x8f5b: 'ddddabaa',
    0x8f5c: 'qqqqqb089',
    0x8f5d: 'aaaabbaa',
    0x8f5e: 'rrrrra189',
    0x8f5f: 'bbbbaadc',
    0x8f60: 'ppppppab09',
    0x8f61: 'ccccbbaa',
    0x8f62: 'pppppb089',
    0x8f63: 'ppppbaca',
    0x8f64: 'uuuuua089',
    0x8f65: 'eeeebadc',
    0x8f66: 'uuuuuubb09',
    0x8f67: 'ffffabaa',
    0x8f68: '9999999bbbO',
    0x8f69: 'oooobbdc',
    0x8f6a: '9999999bbbo',
    0x8f6b: 'llllaaaa',
    0x8f6c: 'yyyyya099',
    0x8f6d: 'iiiibaaa',
    0x8f6e: 'yyyyyybb19',
    0x8f6f: 'jjjjabdc',
    0x8f70: 'hhhhhhab08',
    0x8f71: 'kkkkabdc',
    0x8f72: 'hhhhhb088',
    0x8f73: 'hhhhbaaa',
    0x8f74: 'mmmmma088',
    0x8f75: 'mmmmaaaa',
    0x8f76: 'mmmmmmbb08',
    0x8f77: 'nnnnbbdc',
    0x8f78: 'ddddddab18',
    0x8f79: 'ggggabaa',
    0x8f7a: 'dddddb098',
    0x8f7b: 'ddddbadc',
    0x8f7c: 'aaaaaa098',
    0x8f7d: 'aaaaaadc',
    0x8f7e: 'aaaaaabb18',
    0x8f7f: 'bbbbbbaa',
    0x8f80: 'ccccca198',
    0x8f81: 'ccccabdc',
    0x8f82: 'ccccccbb18',
    0x8f83: 'ppppaaba',
    0x8f84: 'eeeeea098',
    0x8f85: 'eeeeaaaa',
    0x8f86: 'eeeeeeba08',
    0x8f87: 'ffffbbdc',
    0x8f88: 'llllllaa18',
    0x8f89: 'ooooabaa',
    0x8f8a: 'lllllb088',
    0x8f8b: 'llllbadc',
    0x8f8c: 'iiiiia088',
    0x8f8d: 'iiiiaadc',
    0x8f8e: 'iiiiiiba18',
    0x8f8f: 'jjjjbbaa',
    0x8f90: 'xxxxxxaa09',
    0x8f91: 'kkkkbbaa',
    0x8f92: 'xxxxxb099',
    0x8f93: 'hhhhaadc',
    0x8f94: '11111111babK',
    0x8f95: 'mmmmbadc',
    0x8f96: '11111111badX',
    0x8f97: 'nnnnabaa',
    0x8f98: 'ttttttaa19',
    0x8f99: 'ggggbbdc',
    0x8f9a: 'tttttb089',
    0x8f9b: 'ddddaaaa',
    0x8f9c: 'qqqqqa089',
    0x8f9d: 'aaaabaaa',
    0x8f9e: 'qqqqqqba19',
    0x8f9f: 'bbbbabdc',
    0x8fa0: 'sssssb189',
    0x8fa1: 'ccccbaaa',
    0x8fa2: 'pppppa089',
    0x8fa3: 'ppppbbca',
    0x8fa4: 'uuuuub089',
    0x8fa5: 'eeeebbdc',
    0x8fa6: 'vvvvva189',
    0x8fa7: 'ffffaaaa',
    0x8fa8: '?????b048',
    0x8fa9: 'oooobadc',
    0x8faa: '<<<<<a148',
    0x8fab: 'llllabaa',
    0x8fac: 'yyyyyb099',
    0x8fad: 'iiiibbaa',
    0x8fae: 'zzzzza199',
    0x8faf: 'jjjjaadc',
    0x8fb0: 'kkkkkb188',
    0x8fb1: 'kkkkaadc',
    0x8fb2: 'hhhhha088',
    0x8fb3: 'hhhhbbaa',
    0x8fb4: 'mmmmmb088',
    0x8fb5: 'mmmmabaa',
    0x8fb6: 'nnnnna188',
    0x8fb7: 'nnnnbadc',
    0x8fb8: 'gggggb198',
    0x8fb9: 'ggggaaaa',
    0x8fba: 'ddddda098',
    0x8fbb: 'ddddbbdc',
    0x8fbc: 'aaaaab098',
    0x8fbd: 'aaaaabdc',
    0x8fbe: 'bbbbba198',
    0x8fbf: 'bbbbbaaa',
    0x8fc0: 'cccccb098',
    0x8fc1: 'ccccaaaa',
    0x8fc2: 'CCCCCCab49',
    0x8fc3: 'ppppabca',
    0x8fc4: 'eeeeeb198',
    0x8fc5: 'eeeeabdc',
    0x8fc6: 'fffffa098',
    0x8fc7: 'ffffbaaa',
    0x8fc8: 'ooooob088',
    0x8fc9: 'ooooaadc',
    0x8fca: 'llllla188',
    0x8fcb: 'llllbbaa',
    0x8fcc: 'iiiiib188',
    0x8fcd: 'iiiiabaa',
    0x8fce: 'jjjjja088',
    0x8fcf: 'jjjjbadc',
    0x8fd0: 'QQQQQQQQbabo',
    0x8fd1: 'kkkkbadc',
    0x8fd2: 'xxxxxa199',
    0x8fd3: 'hhhhabaa',
    0x8fd4: 'QQQQQQQQbabO',
    0x8fd5: 'mmmmbbaa',
    0x8fd6: 'QQQQQQQQbahZ',
    0x8fd7: 'nnnnaadc',
    0x8fd8: 'wwwwwb089',
    0x8fd9: 'ggggbaaa',
    0x8fda: 'ttttta189',
    0x8fdb: 'ddddabdc',
    0x8fdc: 'qqqqqb189',
    0x8fdd: 'aaaabbdc',
    0x8fde: 'rrrrra089',
    0x8fdf: 'bbbbaaaa',
    0x8fe0: 'ppppppba19',
    0x8fe1: 'ccccbbdc',
    0x8fe2: 'pppppb189',
    0x8fe3: 'ppppbaba',
    0x8fe4: 'pppppppbbbh',
    0x8fe5: 'eeeebaaa',
    0x8fe6: 'pppppppbbbH',
    0x8fe7: 'ffffabdc',
    0x8fe8: 'ppppppppaaiY',
    0x8fe9: 'oooobbaa',
    0x8fea: 'ppppppppaaaM',
    0x8feb: 'llllaadc',
    0x8fec: 'ppppppppaaiy',
    0x8fed: 'iiiibadc',
    0x8fee: 'ppppppppaaam',
    0x8fef: 'jjjjabaa',
    0x8ff0: 'aaaaaaaababm',
    0x8ff1: 'aaaaaaaababe',
    0x8ff2: 'aaaaaaaabahx',
    0x8ff3: 'aaaaaaaababu',
    0x8ff4: 'aaaaaaaababM',
    0x8ff5: 'aaaaaaaababE',
    0x8ff6: 'aaaaaaaabahX',
    0x8ff7: 'aaaaaaaababU',
    0x8ff8: 'aaaaaaaabaH',
    0x8ff9: 'aaaaaaaabaX',
    0x8ffa: 'aaaaaaaabah',
    0x8ffb: 'aaaaaaaabax',
    0x8ffc: 'aaaaaa198',
    0x8ffd: 'aaaaaaaa',
    0x8ffe: 'aaaaaaaa08',
    0x8fff: 'aaaaaaaab1',
    0x9000: 'ccccca003',
    0x9001: 'ccccabiz',
    0x9002: 'ccccccah99',
    0x9003: 'cccccccaa7',
    0x9004: 'cccccccaaaK',
    0x9005: 'cccccccaacZ',
    0x9006: 'cccccccaaak',
    0x9007: 'cccccccaacz',
    0x9008: 'ccccccccbbbL',
    0x9009: 'ccccccccbbbD',
    0x900a: 'ccccccccbbhY',
    0x900b: 'ccccccccbbbT',
    0x900c: 'ccccccccbbbl',
    0x900d: 'ccccccccbbbd',
    0x900e: 'ccccccccbbhy',
    0x900f: 'ccccccccbbbt',
    0x9010: 'qqqqqqqqbabm',
    0x9011: 'kkkkbbhz',
    0x9012: 'qqqqqqqqbahx',
    0x9013: 'hhhhaaiz',
    0x9014: 'qqqqqqqqbabM',
    0x9015: 'mmmmbaiz',
    0x9016: 'qqqqqqqqbahX',
    0x9017: 'nnnnabhz',
    0x9018: 'qqqqqqqabaJ',
    0x9019: 'ggggbbiz',
    0x901a: 'qqqqqqqabaj',
    0x901b: 'ddddaahz',
    0x901c: 'qqqqqa112',
    0x901d: 'aaaabahz',
    0x901e: 'qqqqqqai88',
    0x901f: 'bbbbabiz',
    0x9020: 'sssssb012',
    0x9021: 'ccccbahz',
    0x9022: 'pppppa112',
    0x9023: 'ppppbbjz',
    0x9024: 'uuuuub112',
    0x9025: 'eeeebbiz',
    0x9026: 'vvvvva012',
    0x9027: 'ffffaahz',
    0x9028: 'PPPPPPPPaakZ',
    0x9029: 'oooobaiz',
    0x902a: 'PPPPPPPPaaaO',
    0x902b: 'llllabhz',
    0x902c: 'yyyyyb102',
    0x902d: 'iiiibbhz',
    0x902e: 'zzzzza002',
    0x902f: 'jjjjaaiz',
    0x9030: 'kkkkkb013',
    0x9031: 'kkkkaaiz',
    0x9032: 'hhhhha113',
    0x9033: 'hhhhbbhz',
    0x9034: 'mmmmmb113',
    0x9035: 'mmmmabhz',
    0x9036: 'nnnnna013',
    0x9037: 'nnnnbaiz',
    0x9038: 'gggggb003',
    0x9039: 'ggggaahz',
    0x903a: 'ddddda103',
    0x903b: 'ddddbbiz',
    0x903c: 'aaaaab103',
    0x903d: 'aaaaabiz',
    0x903e: 'bbbbba003',
    0x903f: 'bbbbbahz',
    0x9040: 'cccccb103',
    0x9041: 'ccccaahz',
    0x9042: 'CCCCCCbb98',
    0x9043: 'ppppabjz',
    0x9044: 'eeeeeb003',
    0x9045: 'eeeeabiz',
    0x9046: 'fffffa103',
    0x9047: 'ffffbahz',
    0x9048: 'ooooob113',
    0x9049: 'ooooaaiz',
    0x904a: 'llllla013',
    0x904b: 'llllbbhz',
    0x904c: 'iiiiib013',
    0x904d: 'iiiiabhz',
    0x904e: 'jjjjja113',
    0x904f: 'jjjjbaiz',
    0x9050: 'XXXXXXaa89',
    0x9051: 'kkkkbaiz',
    0x9052: 'xxxxxa002',
    0x9053: 'hhhhabhz',
    0x9054: '=====b115',
    0x9055: 'mmmmbbhz',
    0x9056: '>>>>>a015',
    0x9057: 'nnnnaaiz',
    0x9058: 'wwwwwb112',
    0x9059: 'ggggbahz',
    0x905a: 'ttttta012',
    0x905b: 'ddddabiz',
    0x905c: 'qqqqqb012',
    0x905d: 'aaaabbiz',
    0x905e: 'rrrrra112',
    0x905f: 'bbbbaahz',
    0x9060: 'ppppppai88',
    0x9061: 'ccccbbiz',
    0x9062: 'pppppb012',
    0x9063: 'ppppbakz',
    0x9064: 'uuuuua012',
    0x9065: 'eeeebahz',
    0x9066: 'uuuuuubi88',
    0x9067: 'ffffabiz',
    0x9068: '00000000aacZ',
    0x9069: 'oooobbhz',
    0x906a: '00000000aaaK',
    0x906b: 'llllaaiz',
    0x906c: 'yyyyya002',
    0x906d: 'iiiibaiz',
    0x906e: 'yyyyyybi98',
    0x906f: 'jjjjabhz',
    0x9070: 'hhhhhhai89',
    0x9071: 'kkkkabhz',
    0x9072: 'hhhhhb013',
    0x9073: 'hhhhbaiz',
    0x9074: 'mmmmma013',
    0x9075: 'mmmmaaiz',
    0x9076: 'mmmmmmbi89',
    0x9077: 'nnnnbbhz',
    0x9078: 'ddddddai99',
    0x9079: 'ggggabiz',
    0x907a: 'dddddb003',
    0x907b: 'ddddbahz',
    0x907c: 'aaaaaa003',
    0x907d: 'aaaaaahz',
    0x907e: 'aaaaaabi99',
    0x907f: 'bbbbbbiz',
    0x9080: 'ccccca103',
    0x9081: 'ccccabhz',
    0x9082: 'ccccccbi99',
    0x9083: 'ppppaajz',
    0x9084: 'eeeeea003',
    0x9085: 'eeeeaaiz',
    0x9086: 'eeeeeebh99',
    0x9087: 'ffffbbhz',
    0x9088: 'llllllah89',
    0x9089: 'ooooabiz',
    0x908a: 'lllllb013',
    0x908b: 'llllbahz',
    0x908c: 'iiiiia013',
    0x908d: 'iiiiaahz',
    0x908e: 'iiiiiibh89',
    0x908f: 'jjjjbbiz',
    0x9090: 'xxxxxxah98',
    0x9091: 'kkkkbbiz',
    0x9092: 'xxxxxb002',
    0x9093: 'hhhhaahz',
    0x9094: '8888888abam',
    0x9095: 'mmmmbahz',
    0x9096: '8888888abaM',
    0x9097: 'nnnnabiz',
    0x9098: 'ttttttah88',
    0x9099: 'ggggbbhz',
    0x909a: 'tttttb012',
    0x909b: 'ddddaaiz',
    0x909c: 'qqqqqa012',
    0x909d: 'aaaabaiz',
    0x909e: 'qqqqqqbh88',
    0x909f: 'bbbbabhz',
    0x90a0: 'sssssb112',
    0x90a1: 'ccccbaiz',
    0x90a2: 'pppppa012',
    0x90a3: 'ppppbbkz',
    0x90a4: 'uuuuub012',
    0x90a5: 'eeeebbhz',
    0x90a6: 'vvvvva112',
    0x90a7: 'ffffaaiz',
    0x90a8: '?????b015',
    0x90a9: 'oooobahz',
    0x90aa: '<<<<<a115',
    0x90ab: 'llllabiz',
    0x90ac: 'yyyyyb002',
    0x90ad: 'iiiibbiz',
    0x90ae: 'zzzzza102',
    0x90af: 'jjjjaahz',
    0x90b0: 'kkkkkb113',
    0x90b1: 'kkkkaahz',
    0x90b2: 'hhhhha013',
    0x90b3: 'hhhhbbiz',
    0x90b4: 'mmmmmb013',
    0x90b5: 'mmmmabiz',
    0x90b6: 'nnnnna113',
    0x90b7: 'nnnnbahz',
    0x90b8: 'gggggb103',
    0x90b9: 'ggggaaiz',
    0x90ba: 'ddddda003',
    0x90bb: 'ddddbbhz',
    0x90bc: 'aaaaab003',
    0x90bd: 'aaaaabhz',
    0x90be: 'bbbbba103',
    0x90bf: 'bbbbbaiz',
    0x90c0: 'cccccb003',
    0x90c1: 'ccccaaiz',
    0x90c2: 'CCCCCCaa88',
    0x90c3: 'ppppabkz',
    0x90c4: 'eeeeeb103',
    0x90c5: 'eeeeabhz',
    0x90c6: 'fffffa003',
    0x90c7: 'ffffbaiz',
    0x90c8: 'ooooob013',
    0x90c9: 'ooooaahz',
    0x90ca: 'llllla113',
    0x90cb: 'llllbbiz',
    0x90cc: 'iiiiib113',
    0x90cd: 'iiiiabiz',
    0x90ce: 'jjjjja013',
    0x90cf: 'jjjjbahz',
    0x90d0: 'XXXXXXbb99',
    0x90d1: 'kkkkbahz',
    0x90d2: 'xxxxxa102',
    0x90d3: 'hhhhabiz',
    0x90d4: 'XXXXXXXabai',
    0x90d5: 'mmmmbbiz',
    0x90d6: 'XXXXXXXabaI',
    0x90d7: 'nnnnaahz',
    0x90d8: 'wwwwwb012',
    0x90d9: 'ggggbaiz',
    0x90da: 'ttttta112',
    0x90db: 'ddddabhz',
    0x90dc: 'qqqqqb112',
    0x90dd: 'aaaabbhz',
    0x90de: 'rrrrra012',
    0x90df: 'bbbbaaiz',
    0x90e0: 'ppppppbh88',
    0x90e1: 'ccccbbhz',
    0x90e2: 'pppppb112',
    0x90e3: 'ppppbajz',
    0x90e4: 'pppppppabaj',
    0x90e5: 'eeeebaiz',
    0x90e6: 'pppppppabaJ',
    0x90e7: 'ffffabhz',
    0x90e8: 'yyyyyyybbbK',
    0x90e9: 'oooobbiz',
    0x90ea: 'yyyyyyybbbk',
    0x90eb: 'llllaahz',
    0x90ec: 'yyyyya102',
    0x90ed: 'iiiibahz',
    0x90ee: 'yyyyyyah98',
    0x90ef: 'jjjjabiz',
    0x90f0: 'hhhhhhbh89',
    0x90f1: 'kkkkabiz',
    0x90f2: 'hhhhhb113',
    0x90f3: 'hhhhbahz',
    0x90f4: 'hhhhhhhabak',
    0x90f5: 'hhhhhhhabcz',
    0x90f6: 'hhhhhhhabaK',
    0x90f7: 'hhhhhhhabcZ',
    0x90f8: 'aaaaaaabbbJ',
    0x90f9: 'aaaaaaabbbZ',
    0x90fa: 'aaaaaaabbbj',
    0x90fb: 'aaaaaaabbbz',
    0x90fc: 'aaaaaa103',
    0x90fd: 'aaaaaaiz',
    0x90fe: 'aaaaaaah99',
    0x90ff: 'aaaaaaabb6',
    0x9100: 'ccccca013',
    0x9101: 'ccccabkz',
    0x9102: 'ccccccai99',
    0x9103: 'cccccccbb6',
    0x9104: 'cccccccbbbJ',
    0x9105: 'cccccccbbbZ',
    0x9106: 'cccccccbbbj',
    0x9107: 'cccccccbbbz',
    0x9108: 'iiiiiiibabJ',
    0x9109: 'iiiiiiibabZ',
    0x910a: 'iiiiiiibabj',
    0x910b: 'iiiiiiibabz',
    0x910c: 'iiiiia103',
    0x910d: 'iiiiaakz',
    0x910e: 'iiiiiiah89',
    0x910f: 'jjjjbbjz',
    0x9110: 'xxxxxxbh98',
    0x9111: 'kkkkbbjz',
    0x9112: 'xxxxxb112',
    0x9113: 'hhhhaakz',
    0x9114: 'xxxxxxxaaaj',
    0x9115: 'mmmmbakz',
    0x9116: 'xxxxxxxaaaJ',
    0x9117: 'nnnnabjz',
    0x9118: 'qqqqqqqbabK',
    0x9119: 'ggggbbkz',
    0x911a: 'qqqqqqqbabk',
    0x911b: 'ddddaajz',
    0x911c: 'qqqqqa102',
    0x911d: 'aaaabajz',
    0x911e: 'qqqqqqah88',
    0x911f: 'bbbbabkz',
    0x9120: 'sssssb002',
    0x9121: 'ccccbajz',
    0x9122: 'pppppa102',
    0x9123: 'ppppbbhz',
    0x9124: 'uuuuub102',
    0x9125: 'eeeebbkz',
    0x9126: 'vvvvva002',
    0x9127: 'ffffaajz',
    0x9128: 'YYYYYYYbabH',
    0x9129: 'oooobakz',
    0x912a: 'YYYYYYYbabh',
    0x912b: 'llllabjz',
    0x912c: 'yyyyyb112',
    0x912d: 'iiiibbjz',
    0x912e: 'zzzzza012',
    0x912f: 'jjjjaakz',
    0x9130: 'kkkkkb003',
    0x9131: 'kkkkaakz',
    0x9132: 'hhhhha103',
    0x9133: 'hhhhbbjz',
    0x9134: 'mmmmmb103',
    0x9135: 'mmmmabjz',
    0x9136: 'nnnnna003',
    0x9137: 'nnnnbakz',
    0x9138: 'gggggb013',
    0x9139: 'ggggaajz',
    0x913a: 'ddddda113',
    0x913b: 'ddddbbkz',
    0x913c: 'aaaaab113',
    0x913d: 'aaaaabkz',
    0x913e: 'bbbbba013',
    0x913f: 'bbbbbajz',
    0x9140: 'cccccb113',
    0x9141: 'ccccaajz',
    0x9142: 'CCCCCCba88',
    0x9143: 'ppppabhz',
    0x9144: 'eeeeeb013',
    0x9145: 'eeeeabkz',
    0x9146: 'fffffa113',
    0x9147: 'ffffbajz',
    0x9148: 'ooooob103',
    0x9149: 'ooooaakz',
    0x914a: 'llllla003',
    0x914b: 'llllbbjz',
    0x914c: 'iiiiib003',
    0x914d: 'iiiiabjz',
    0x914e: 'jjjjja103',
    0x914f: 'jjjjbakz',
    0x9150: 'XXXXXXab99',
    0x9151: 'kkkkbakz',
    0x9152: 'xxxxxa012',
    0x9153: 'hhhhabjz',
    0x9154: '=====b105',
    0x9155: 'mmmmbbjz',
    0x9156: '>>>>>a005',
    0x9157: 'nnnnaakz',
    0x9158: 'wwwwwb102',
    0x9159: 'ggggbajz',
    0x915a: 'ttttta002',
    0x915b: 'ddddabkz',
    0x915c: 'qqqqqb002',
    0x915d: 'aaaabbkz',
    0x915e: 'rrrrra102',
    0x915f: 'bbbbaajz',
    0x9160: 'ppppppah88',
    0x9161: 'ccccbbkz',
    0x9162: 'pppppb002',
    0x9163: 'ppppbaiz',
    0x9164: 'uuuuua002',
    0x9165: 'eeeebajz',
    0x9166: 'uuuuuubh88',
    0x9167: 'ffffabkz',
    0x9168: '9999999babL',
    0x9169: 'oooobbjz',
    0x916a: '9999999babl',
    0x916b: 'llllaakz',
    0x916c: 'yyyyya012',
    0x916d: 'iiiibakz',
    0x916e: 'yyyyyybh98',
    0x916f: 'jjjjabjz',
    0x9170: 'hhhhhhah89',
    0x9171: 'kkkkabjz',
    0x9172: 'hhhhhb003',
    0x9173: 'hhhhbakz',
    0x9174: 'mmmmma003',
    0x9175: 'mmmmaakz',
    0x9176: 'mmmmmmbh89',
    0x9177: 'nnnnbbjz',
    0x9178: 'ddddddah99',
    0x9179: 'ggggabkz',
    0x917a: 'dddddb013',
    0x917b: 'ddddbajz',
    0x917c: 'aaaaaa013',
    0x917d: 'aaaaaajz',
    0x917e: 'aaaaaabh99',
    0x917f: 'bbbbbbkz',
    0x9180: 'aaaaaaaaaaaabdX',
    0x9181: 'aaaaaaaaaaaabbZ',
    0x9182: 'aaaaaaaaaaaabbY',
    0x9183: 'aaaaaaaaaaaabbX',
    0x9184: 'aaaaaaaaaaaabhZ',
    0x9185: 'aaaaaaaaaaaabjZ',
    0x9186: 'aaaaaaaaaaaabhX',
    0x9187: 'aaaaaaaaaaaabhY',
    0x9188: 'aaaaaaaaaaaabbS',
    0x9189: 'aaaaaaaaaaaabbR',
    0x918a: 'aaaaaaaaaaaabbQ',
    0x918b: 'aaaaaaaaaaaabbP',
    0x918c: 'aaaaaaaaaaaabbW',
    0x918d: 'aaaaaaaaaaaabbV',
    0x918e: 'aaaaaaaaaaaabbU',
    0x918f: 'aaaaaaaaaaaabbT',
    0x9190: 'aaaaaaaaaaaabbK',
    0x9191: 'aaaaaaaaaaaabbJ',
    0x9192: 'aaaaaaaaaaaabbI',
    0x9193: 'aaaaaaaaaaaabbH',
    0x9194: 'aaaaaaaaaaaabbO',
    0x9195: 'aaaaaaaaaaaabbN',
    0x9196: 'aaaaaaaaaaaabbM',
    0x9197: 'aaaaaaaaaaaabbL',
    0x9198: 'aaaaaaaaaaaabbC',
    0x9199: 'aaaaaaaaaaaabbB',
    0x919a: 'aaaaaaaaaaaabbA',
    0x919b: 'aaaaaaaaaaaabdC',
    0x919c: 'aaaaaaaaaaaabbG',
    0x919d: 'aaaabakz',
    0x919e: 'aaaaaaaaaaaabbE',
    0x919f: 'aaaaaaaaaaaabbD',
    0x91a0: 'aaaaaaaaaaaabdx',
    0x91a1: 'aaaaaaaaaaaabbz',
    0x91a2: 'aaaaaaaaaaaabby',
    0x91a3: 'aaaaaaaaaaaabbx',
    0x91a4: 'aaaaaaaaaaaabhz',
    0x91a5: 'aaaaaaaaaaaabjz',
    0x91a6: 'aaaaaaaaaaaabhx',
    0x91a7: 'aaaaaaaaaaaabhy',
    0x91a8: 'aaaaaaaaaaaabbs',
    0x91a9: 'aaaaaaaaaaaabbr',
    0x91aa: 'aaaaaaaaaaaabbq',
    0x91ab: 'aaaaaaaaaaaabbp',
    0x91ac: 'aaaaaaaaaaaabbw',
    0x91ad: 'aaaaaaaaaaaabbv',
    0x91ae: 'aaaaaaaaaaaabbu',
    0x91af: 'aaaaaaaaaaaabbt',
    0x91b0: 'aaaaaaaaaaaabbk',
    0x91b1: 'aaaaaaaaaaaabbj',
    0x91b2: 'aaaaaaaaaaaabbi',
    0x91b3: 'aaaaaaaaaaaabbh',
    0x91b4: 'aaaaaaaaaaaabbo',
    0x91b5: 'aaaaaaaaaaaabbn',
    0x91b6: 'aaaaaaaaaaaabbm',
    0x91b7: 'aaaaaaaaaaaabbl',
    0x91b8: 'aaaaaaaaaaaabbc',
    0x91b9: 'aaaaaaaaaaaabbb',
    0x91ba: 'aaaaaaaaaaaabba',
    0x91bb: 'aaaaaaaaaaaabdc',
    0x91bc: 'aaaaab013',
    0x91bd: 'aaaaabjz',
    0x91be: 'aaaaaaaaaaaabbe',
    0x91bf: 'aaaaaaaaaaaabbd',
    0x91c0: 'aaaaaaaaaaaabV',
    0x91c1: 'aaaaaaaaaaaabT',
    0x91c2: 'aaaaaaaaaaaabR',
    0x91c3: 'aaaaaaaaaaaabP',
    0x91c4: 'aaaaaaaaaaaajZ',
    0x91c5: 'aaaaaaaaaaaahY',
    0x91c6: 'aaaaaaaaaaaabZ',
    0x91c7: 'aaaaaaaaaaaabX',
    0x91c8: 'aaaaaaaaaaaabF',
    0x91c9: 'aaaaaaaaaaaabD',
    0x91ca: 'aaaaaaaaaaaabB',
    0x91cb: 'aaaaaaaaaaaadC',
    0x91cc: 'aaaaaaaaaaaabN',
    0x91cd: 'aaaaaaaaaaaabL',
    0x91ce: 'aaaaaaaaaaaabJ',
    0x91cf: 'aaaaaaaaaaaabH',
    0x91d0: 'aaaaaaaaaaaabv',
    0x91d1: 'aaaaaaaaaaaabt',
    0x91d2: 'aaaaaaaaaaaabr',
    0x91d3: 'aaaaaaaaaaaabp',
    0x91d4: 'aaaaaaaaaaaajz',
    0x91d5: 'aaaaaaaaaaaahy',
    0x91d6: 'aaaaaaaaaaaabz',
    0x91d7: 'aaaaaaaaaaaabx',
    0x91d8: 'aaaaaaaaaaaabf',
    0x91d9: 'aaaaaaaaaaaabd',
    0x91da: 'aaaaaaaaaaaabb',
    0x91db: 'aaaaaaaaaaaadc',
    0x91dc: 'aaaaaaaaaaaabn',
    0x91dd: 'aaaabbjz',
    0x91de: 'aaaaaaaaaaaabj',
    0x91df: 'aaaaaaaaaaaabh',
    0x91e0: 'aaaaaaaaaaaaB',
    0x91e1: 'aaaaaaaaaaaaF',
    0x91e2: 'aaaaaaaaaaaaJ',
    0x91e3: 'aaaaaaaaaaaaN',
    0x91e4: 'aaaaaaaaaaaaR',
    0x91e5: 'aaaaaaaaaaaaV',
    0x91e6: 'aaaaaaaaaaaaZ',
    0x91e7: 'aaaaaaaaaaaiZ',
    0x91e8: 'aaaaaaaaaaaab',
    0x91e9: 'aaaaaaaaaaaaf',
    0x91ea: 'aaaaaaaaaaaaj',
    0x91eb: 'aaaaaaaaaaaan',
    0x91ec: 'aaaaaaaaaaaar',
    0x91ed: 'aaaaaaaaaaaav',
    0x91ee: 'aaaaaaaaaaaaz',
    0x91ef: 'aaaaaaaaaaaiz',
    0x91f0: 'aaaaaaaaaaam',
    0x91f1: 'aaaaaaaaaaae',
    0x91f2: 'aaaaaaaaaaiy',
    0x91f3: 'aaaaaaaaaaau',
    0x91f4: 'aaaaaaaaaaaM',
    0x91f5: 'aaaaaaaaaaaE',
    0x91f6: 'aaaaaaaaaaiY',
    0x91f7: 'aaaaaaaaaaaU',
    0x91f8: 'aaaaaaaaaaK',
    0x91f9: 'aaaaaaaaacZ',
    0x91fa: 'aaaaaaaaaak',
    0x91fb: 'aaaaaaaaacz',
    0x91fc: 'aaaaaa113',
    0x91fd: 'aaaaaakz',
    0x91fe: 'aaaaaaai99',
    0x91ff: 'aaaaaaaaa7',
    0x9200: 'ccccca002',
    0x9201: 'ccccabix',
    0x9202: 'ccccccah89',
    0x9203: 'cccccccab6',
    0x9204: 'cccccccabbJ',
    0x9205: 'cccccccabbZ',
    0x9206: 'cccccccabbj',
    0x9207: 'cccccccabbz',
    0x9208: 'ccccccccaaaM',
    0x9209: 'ccccccccaaaE',
    0x920a: 'ccccccccaaiY',
    0x920b: 'ccccccccaaaU',
    0x920c: 'ccccccccaaam',
    0x920d: 'ccccccccaaae',
    0x920e: 'ccccccccaaiy',
    0x920f: 'ccccccccaaau',
    0x9210: 'cccccccccbbbs',
    0x9211: 'cccccccccbbbw',
    0x9212: 'cccccccccbbdx',
    0x9213: 'cccccccccbbhz',
    0x9214: 'cccccccccbbbc',
    0x9215: 'cccccccccbbbg',
    0x9216: 'cccccccccbbbk',
    0x9217: 'cccccccccbbbo',
    0x9218: 'cccccccccbbbS',
    0x9219: 'cccccccccbbbW',
    0x921a: 'cccccccccbbdX',
    0x921b: 'cccccccccbbhZ',
    0x921c: 'cccccccccbbbC',
    0x921d: 'aaaabahx',
    0x921e: 'cccccccccbbbK',
    0x921f: 'bbbbabix',
    0x9220: 'sssssb013',
    0x9221: 'ccccbahx',
    0x9222: 'pppppa113',
    0x9223: 'ppppbbhy',
    0x9224: 'uuuuub113',
    0x9225: 'eeeebbix',
    0x9226: 'vvvvva013',
    0x9227: 'ffffaahx',
    0x9228: 'AAAAAAAAAbadc',
    0x9229: 'oooobaix',
    0x922a: 'AAAAAAAAAbabh',
    0x922b: 'llllabhx',
    0x922c: 'yyyyyb103',
    0x922d: 'iiiibbhx',
    0x922e: 'zzzzza003',
    0x922f: 'jjjjaaix',
    0x9230: 'kkkkkb012',
    0x9231: 'kkkkaaix',
    0x9232: 'hhhhha112',
    0x9233: 'hhhhbbhx',
    0x9234: 'mmmmmb112',
    0x9235: 'mmmmabhx',
    0x9236: 'nnnnna012',
    0x9237: 'nnnnbaix',
    0x9238: 'gggggb002',
    0x9239: 'ggggaahx',
    0x923a: 'ddddda102',
    0x923b: 'ddddbbix',
    0x923c: 'aaaaab102',
    0x923d: 'aaaaabix',
    0x923e: 'bbbbba002',
    0x923f: 'bbbbbahx',
    0x9240: 'cccccb102',
    0x9241: 'ccccaahx',
    0x9242: 'CCCCCCbb88',
    0x9243: 'ppppabhy',
    0x9244: 'eeeeeb002',
    0x9245: 'eeeeabix',
    0x9246: 'fffffa102',
    0x9247: 'ffffbahx',
    0x9248: 'ooooob112',
    0x9249: 'ooooaaix',
    0x924a: 'llllla012',
    0x924b: 'llllbbhx',
    0x924c: 'iiiiib012',
    0x924d: 'iiiiabhx',
    0x924e: 'jjjjja112',
    0x924f: 'jjjjbaix',
    0x9250: 'XXXXXXaa99',
    0x9251: 'kkkkbaix',
    0x9252: 'xxxxxa003',
    0x9253: 'hhhhabhx',
    0x9254: '=====b114',
    0x9255: 'mmmmbbhx',
    0x9256: '>>>>>a014',
    0x9257: 'nnnnaaix',
    0x9258: 'wwwwwb113',
    0x9259: 'ggggbahx',
    0x925a: 'ttttta013',
    0x925b: 'ddddabix',
    0x925c: 'qqqqqb013',
    0x925d: 'aaaabbix',
    0x925e: 'rrrrra113',
    0x925f: 'bbbbaahx',
    0x9260: 'ppppppai98',
    0x9261: 'ccccbbix',
    0x9262: 'pppppb013',
    0x9263: 'ppppbaiy',
    0x9264: 'uuuuua013',
    0x9265: 'eeeebahx',
    0x9266: 'uuuuuubi98',
    0x9267: 'ffffabix',
    0x9268: '00000000bbbZ',
    0x9269: 'oooobbhx',
    0x926a: '00000000bbbJ',
    0x926b: 'llllaaix',
    0x926c: 'yyyyya003',
    0x926d: 'iiiibaix',
    0x926e: 'yyyyyybi88',
    0x926f: 'jjjjabhx',
    0x9270: 'hhhhhhai99',
    0x9271: 'kkkkabhx',
    0x9272: 'hhhhhb012',
    0x9273: 'hhhhbaix',
    0x9274: 'mmmmma012',
    0x9275: 'mmmmaaix',
    0x9276: 'mmmmmmbi99',
    0x9277: 'nnnnbbhx',
    0x9278: 'ddddddai89',
    0x9279: 'ggggabix',
    0x927a: 'dddddb002',
    0x927b: 'ddddbahx',
    0x927c: 'aaaaaa002',
    0x927d: 'aaaaaahx',
    0x927e: 'aaaaaabi89',
    0x927f: 'bbbbbbix',
    0x9280: 'ccccca102',
    0x9281: 'ccccabhx',
    0x9282: 'ccccccbi89',
    0x9283: 'ppppaahy',
    0x9284: 'eeeeea002',
    0x9285: 'eeeeaaix',
    0x9286: 'eeeeeebh89',
    0x9287: 'ffffbbhx',
    0x9288: 'llllllah99',
    0x9289: 'ooooabix',
    0x928a: 'lllllb012',
    0x928b: 'llllbahx',
    0x928c: 'iiiiia012',
    0x928d: 'iiiiaahx',
    0x928e: 'iiiiiibh99',
    0x928f: 'jjjjbbix',
    0x9290: 'xxxxxxah88',
    0x9291: 'kkkkbbix',
    0x9292: 'xxxxxb003',
    0x9293: 'hhhhaahx',
    0x9294: '8888888aabl',
    0x9295: 'mmmmbahx',
    0x9296: '8888888aabL',
    0x9297: 'nnnnabix',
    0x9298: 'ttttttah98',
    0x9299: 'ggggbbhx',
    0x929a: 'tttttb013',
    0x929b: 'ddddaaix',
    0x929c: 'qqqqqa013',
    0x929d: 'aaaabaix',
    0x929e: 'qqqqqqbh98',
    0x929f: 'bbbbabhx',
    0x92a0: 'sssssb113',
    0x92a1: 'ccccbaix',
    0x92a2: 'pppppa013',
    0x92a3: 'ppppbbiy',
    0x92a4: 'uuuuub013',
    0x92a5: 'eeeebbhx',
    0x92a6: 'vvvvva113',
    0x92a7: 'ffffaaix',
    0x92a8: '?????b014',
    0x92a9: 'oooobahx',
    0x92aa: '<<<<<a114',
    0x92ab: 'llllabix',
    0x92ac: 'yyyyyb003',
    0x92ad: 'iiiibbix',
    0x92ae: 'zzzzza103',
    0x92af: 'jjjjaahx',
    0x92b0: 'kkkkkb112',
    0x92b1: 'kkkkaahx',
    0x92b2: 'hhhhha012',
    0x92b3: 'hhhhbbix',
    0x92b4: 'mmmmmb012',
    0x92b5: 'mmmmabix',
    0x92b6: 'nnnnna112',
    0x92b7: 'nnnnbahx',
    0x92b8: 'gggggb102',
    0x92b9: 'ggggaaix',
    0x92ba: 'ddddda002',
    0x92bb: 'ddddbbhx',
    0x92bc: 'aaaaab002',
    0x92bd: 'aaaaabhx',
    0x92be: 'bbbbba102',
    0x92bf: 'bbbbbaix',
    0x92c0: 'cccccb002',
    0x92c1: 'ccccaaix',
    0x92c2: 'CCCCCCaa98',
    0x92c3: 'ppppabiy',
    0x92c4: 'eeeeeb102',
    0x92c5: 'eeeeabhx',
    0x92c6: 'fffffa002',
    0x92c7: 'ffffbaix',
    0x92c8: 'ooooob012',
    0x92c9: 'ooooaahx',
    0x92ca: 'llllla112',
    0x92cb: 'llllbbix',
    0x92cc: 'iiiiib112',
    0x92cd: 'iiiiabix',
    0x92ce: 'jjjjja012',
    0x92cf: 'jjjjbahx',
    0x92d0: 'XXXXXXbb89',
    0x92d1: 'kkkkbahx',
    0x92d2: 'xxxxxa103',
    0x92d3: 'hhhhabix',
    0x92d4: 'XXXXXXXaabh',
    0x92d5: 'mmmmbbix',
    0x92d6: 'XXXXXXXaabH',
    0x92d7: 'nnnnaahx',
    0x92d8: 'wwwwwb013',
    0x92d9: 'ggggbaix',
    0x92da: 'ttttta113',
    0x92db: 'ddddabhx',
    0x92dc: 'qqqqqb113',
    0x92dd: 'aaaabbhx',
    0x92de: 'rrrrra013',
    0x92df: 'bbbbaaix',
    0x92e0: 'ppppppbh98',
    0x92e1: 'ccccbbhx',
    0x92e2: 'pppppb113',
    0x92e3: 'ppppbahy',
    0x92e4: 'pppppppaabk',
    0x92e5: 'eeeebaix',
    0x92e6: 'pppppppaabK',
    0x92e7: 'ffffabhx',
    0x92e8: 'yyyyyyybaaJ',
    0x92e9: 'oooobbix',
    0x92ea: 'yyyyyyybaaj',
    0x92eb: 'llllaahx',
    0x92ec: 'yyyyya103',
    0x92ed: 'iiiibahx',
    0x92ee: 'yyyyyyah88',
    0x92ef: 'jjjjabix',
    0x92f0: 'hhhhhhbh99',
    0x92f1: 'kkkkabix',
    0x92f2: 'hhhhhb112',
    0x92f3: 'hhhhbahx',
    0x92f4: 'hhhhhhhaabj',
    0x92f5: 'hhhhhhhaabz',
    0x92f6: 'hhhhhhhaabJ',
    0x92f7: 'hhhhhhhaabZ',
    0x92f8: 'aaaaaaabaaK',
    0x92f9: 'aaaaaaabacZ',
    0x92fa: 'aaaaaaabaak',
    0x92fb: 'aaaaaaabacz',
    0x92fc: 'aaaaaa102',
    0x92fd: 'aaaaaaix',
    0x92fe: 'aaaaaaah89',
    0x92ff: 'aaaaaaaba7',
    0x9300: 'ccccca012',
    0x9301: 'ccccabiy',
    0x9302: 'ccccccai89',
    0x9303: 'cccccccba7',
    0x9304: 'cccccccbaaK',
    0x9305: 'cccccccbacZ',
    0x9306: 'cccccccbaak',
    0x9307: 'cccccccbacz',
    0x9308: 'iiiiiiibbaK',
    0x9309: 'iiiiiiibbcZ',
    0x930a: 'iiiiiiibbak',
    0x930b: 'iiiiiiibbcz',
    0x930c: 'iiiiia102',
    0x930d: 'iiiiaaiy',
    0x930e: 'iiiiiiah99',
    0x930f: 'iiiiiiibb7',
    0x9310: 'xxxxxxbh88',
    0x9311: 'kkkkbbhy',
    0x9312: 'xxxxxb113',
    0x9313: 'hhhhaaiy',
    0x9314: 'xxxxxxxabbk',
    0x9315: 'mmmmbaiy',
    0x9316: 'xxxxxxxabbK',
    0x9317: 'nnnnabhy',
    0x9318: 'qqqqqqqbbaJ',
    0x9319: 'ggggbbiy',
    0x931a: 'qqqqqqqbbaj',
    0x931b: 'ddddaahy',
    0x931c: 'qqqqqa103',
    0x931d: 'aaaabahy',
    0x931e: 'qqqqqqah98',
    0x931f: 'bbbbabiy',
    0x9320: 'sssssb003',
    0x9321: 'ccccbahy',
    0x9322: 'pppppa103',
    0x9323: 'ppppbbhx',
    0x9324: 'uuuuub103',
    0x9325: 'eeeebbiy',
    0x9326: 'vvvvva003',
    0x9327: 'ffffaahy',
    0x9328: 'YYYYYYYbbaI',
    0x9329: 'oooobaiy',
    0x932a: 'YYYYYYYbbai',
    0x932b: 'llllabhy',
    0x932c: 'yyyyyb113',
    0x932d: 'iiiibbhy',
    0x932e: 'zzzzza013',
    0x932f: 'jjjjaaiy',
    0x9330: 'kkkkkb002',
    0x9331: 'kkkkaaiy',
    0x9332: 'hhhhha102',
    0x9333: 'hhhhbbhy',
    0x9334: 'mmmmmb102',
    0x9335: 'mmmmabhy',
    0x9336: 'nnnnna002',
    0x9337: 'nnnnbaiy',
    0x9338: 'gggggb012',
    0x9339: 'ggggaahy',
    0x933a: 'ddddda112',
    0x933b: 'ddddbbiy',
    0x933c: 'aaaaab112',
    0x933d: 'aaaaabiy',
    0x933e: 'bbbbba012',
    0x933f: 'bbbbbahy',
    0x9340: 'cccccb112',
    0x9341: 'ccccaahy',
    0x9342: 'CCCCCCba98',
    0x9343: 'ppppabhx',
    0x9344: 'eeeeeb012',
    0x9345: 'eeeeabiy',
    0x9346: 'fffffa112',
    0x9347: 'ffffbahy',
    0x9348: 'ooooob102',
    0x9349: 'ooooaaiy',
    0x934a: 'llllla002',
    0x934b: 'llllbbhy',
    0x934c: 'iiiiib002',
    0x934d: 'iiiiabhy',
    0x934e: 'jjjjja102',
    0x934f: 'jjjjbaiy',
    0x9350: 'XXXXXXab89',
    0x9351: 'kkkkbaiy',
    0x9352: 'xxxxxa013',
    0x9353: 'hhhhabhy',
    0x9354: '=====b104',
    0x9355: 'mmmmbbhy',
    0x9356: '>>>>>a004',
    0x9357: 'nnnnaaiy',
    0x9358: 'wwwwwb103',
    0x9359: 'ggggbahy',
    0x935a: 'ttttta003',
    0x935b: 'ddddabiy',
    0x935c: 'qqqqqb003',
    0x935d: 'aaaabbiy',
    0x935e: 'rrrrra103',
    0x935f: 'bbbbaahy',
    0x9360: 'ppppppah98',
    0x9361: 'ccccbbiy',
    0x9362: 'pppppb003',
    0x9363: 'ppppbaix',
    0x9364: 'uuuuua003',
    0x9365: 'eeeebahy',
    0x9366: 'uuuuuubh98',
    0x9367: 'ffffabiy',
    0x9368: '9999999bbaM',
    0x9369: 'oooobbhy',
    0x936a: '9999999bbam',
    0x936b: 'llllaaiy',
    0x936c: 'yyyyya013',
    0x936d: 'iiiibaiy',
    0x936e: 'yyyyyybh88',
    0x936f: 'jjjjabhy',
    0x9370: 'hhhhhhah99',
    0x9371: 'kkkkabhy',
    0x9372: 'hhhhhb002',
    0x9373: 'hhhhbaiy',
    0x9374: 'mmmmma002',
    0x9375: 'mmmmaaiy',
    0x9376: 'mmmmmmbh99',
    0x9377: 'nnnnbbhy',
    0x9378: 'ddddddah89',
    0x9379: 'ggggabiy',
    0x937a: 'dddddb012',
    0x937b: 'ddddbahy',
    0x937c: 'aaaaaa012',
    0x937d: 'aaaaaahy',
    0x937e: 'aaaaaabh89',
    0x937f: 'bbbbbbiy',
    0x9380: 'ccccca112',
    0x9381: 'ccccabhy',
    0x9382: 'ccccccbh89',
    0x9383: 'ppppaahx',
    0x9384: 'eeeeea012',
    0x9385: 'eeeeaaiy',
    0x9386: 'eeeeeebi89',
    0x9387: 'ffffbbhy',
    0x9388: 'llllllai99',
    0x9389: 'ooooabiy',
    0x938a: 'lllllb002',
    0x938b: 'llllbahy',
    0x938c: 'iiiiia002',
    0x938d: 'iiiiaahy',
    0x938e: 'iiiiiibi99',
    0x938f: 'jjjjbbiy',
    0x9390: 'xxxxxxai88',
    0x9391: 'kkkkbbiy',
    0x9392: 'xxxxxb013',
    0x9393: 'hhhhaahy',
    0x9394: '11111111bbbJ',
    0x9395: 'mmmmbahy',
    0x9396: '11111111bbbZ',
    0x9397: 'nnnnabiy',
    0x9398: 'ttttttai98',
    0x9399: 'ggggbbhy',
    0x939a: 'tttttb003',
    0x939b: 'ddddaaiy',
    0x939c: 'qqqqqa003',
    0x939d: 'aaaabaiy',
    0x939e: 'qqqqqqbi98',
    0x939f: 'bbbbabhy',
    0x93a0: 'sssssb103',
    0x93a1: 'ccccbaiy',
    0x93a2: 'pppppa003',
    0x93a3: 'ppppbbix',
    0x93a4: 'uuuuub003',
    0x93a5: 'eeeebbhy',
    0x93a6: 'vvvvva103',
    0x93a7: 'ffffaaiy',
    0x93a8: '?????b004',
    0x93a9: 'oooobahy',
    0x93aa: '<<<<<a104',
    0x93ab: 'llllabiy',
    0x93ac: 'yyyyyb013',
    0x93ad: 'iiiibbiy',
    0x93ae: 'zzzzza113',
    0x93af: 'jjjjaahy',
    0x93b0: 'kkkkkb102',
    0x93b1: 'kkkkaahy',
    0x93b2: 'hhhhha002',
    0x93b3: 'hhhhbbiy',
    0x93b4: 'mmmmmb002',
    0x93b5: 'mmmmabiy',
    0x93b6: 'nnnnna102',
    0x93b7: 'nnnnbahy',
    0x93b8: 'gggggb112',
    0x93b9: 'ggggaaiy',
    0x93ba: 'ddddda012',
    0x93bb: 'ddddbbhy',
    0x93bc: 'aaaaab012',
    0x93bd: 'aaaaabhy',
    0x93be: 'bbbbba112',
    0x93bf: 'bbbbbaiy',
    0x93c0: 'cccccb012',
    0x93c1: 'ccccaaiy',
    0x93c2: 'CCCCCCab88',
    0x93c3: 'ppppabix',
    0x93c4: 'eeeeeb112',
    0x93c5: 'eeeeabhy',
    0x93c6: 'fffffa012',
    0x93c7: 'ffffbaiy',
    0x93c8: 'ooooob002',
    0x93c9: 'ooooaahy',
    0x93ca: 'llllla102',
    0x93cb: 'llllbbiy',
    0x93cc: 'iiiiib102',
    0x93cd: 'iiiiabiy',
    0x93ce: 'jjjjja002',
    0x93cf: 'jjjjbahy',
    0x93d0: 'QQQQQQQQbbbn',
    0x93d1: 'kkkkbahy',
    0x93d2: 'xxxxxa113',
    0x93d3: 'hhhhabiy',
    0x93d4: 'QQQQQQQQbbbN',
    0x93d5: 'mmmmbbiy',
    0x93d6: 'QQQQQQQQbbjZ',
    0x93d7: 'nnnnaahy',
    0x93d8: 'wwwwwb003',
    0x93d9: 'ggggbaiy',
    0x93da: 'ttttta103',
    0x93db: 'ddddabhy',
    0x93dc: 'qqqqqb103',
    0x93dd: 'aaaabbhy',
    0x93de: 'rrrrra003',
    0x93df: 'bbbbaaiy',
    0x93e0: 'ppppppbi98',
    0x93e1: 'ccccbbhy',
    0x93e2: 'pppppb103',
    0x93e3: 'ppppbahx',
    0x93e4: 'pppppppbbaj',
    0x93e5: 'eeeebaiy',
    0x93e6: 'pppppppbbaJ',
    0x93e7: 'ffffabhy',
    0x93e8: 'ppppppppabiX',
    0x93e9: 'oooobbiy',
    0x93ea: 'ppppppppabaL',
    0x93eb: 'llllaahy',
    0x93ec: 'ppppppppabix',
    0x93ed: 'iiiibahy',
    0x93ee: 'ppppppppabal',
    0x93ef: 'jjjjabiy',
    0x93f0: 'aaaaaaaabbbl',
    0x93f1: 'aaaaaaaabbbd',
    0x93f2: 'aaaaaaaabbhy',
    0x93f3: 'aaaaaaaabbbt',
    0x93f4: 'aaaaaaaabbbL',
    0x93f5: 'aaaaaaaabbbD',
    0x93f6: 'aaaaaaaabbhY',
    0x93f7: 'aaaaaaaabbbT',
    0x93f8: 'aaaaaaaabbJ',
    0x93f9: 'aaaaaaaabbZ',
    0x93fa: 'aaaaaaaabbj',
    0x93fb: 'aaaaaaaabbz',
    0x93fc: 'aaaaaa112',
    0x93fd: 'aaaaaaiy',
    0x93fe: 'aaaaaaai89',
    0x93ff: 'aaaaaaaab6',
    0x9400: 'ccccca001',
    0x9401: 'ccccabaz',
    0x9402: 'ccccccah98',
    0x9403: 'ccccccccb7',
    0x9404: 'cccccccaabJ',
    0x9405: 'cccccccaabZ',
    0x9406: 'cccccccaabj',
    0x9407: 'cccccccaabz',
    0x9408: 'ccccccccbaaM',
    0x9409: 'ccccccccbaaE',
    0x940a: 'ccccccccbaiY',
    0x940b: 'ccccccccbaaU',
    0x940c: 'ccccccccbaam',
    0x940d: 'ccccccccbaae',
    0x940e: 'ccccccccbaiy',
    0x940f: 'ccccccccbaau',
    0x9410: 'qqqqqqqqbbal',
    0x9411: 'kkkkbbdx',
    0x9412: 'qqqqqqqqbbix',
    0x9413: 'hhhhaaaz',
    0x9414: 'qqqqqqqqbbaL',
    0x9415: 'mmmmbaaz',
    0x9416: 'qqqqqqqqbbiX',
    0x9417: 'nnnnabdx',
    0x9418: 'qqqqqqqabbK',
    0x9419: 'ggggbbaz',
    0x941a: 'qqqqqqqabbk',
    0x941b: 'ddddaadx',
    0x941c: 'qqqqqa110',
    0x941d: 'aaaabadx',
    0x941e: 'qqqqqqai89',
    0x941f: 'bbbbabaz',
    0x9420: 'sssssb010',
    0x9421: 'ccccbadx',
    0x9422: 'pppppa110',
    0x9423: 'ppppbbbz',
    0x9424: 'uuuuub110',
    0x9425: 'eeeebbaz',
    0x9426: 'vvvvva010',
    0x9427: 'ffffaadx',
    0x9428: 'PPPPPPPPabjZ',
    0x9429: 'oooobaaz',
    0x942a: 'PPPPPPPPabbN',
    0x942b: 'llllabdx',
    0x942c: 'yyyyyb100',
    0x942d: 'iiiibbdx',
    0x942e: 'zzzzza000',
    0x942f: 'jjjjaaaz',
    0x9430: 'kkkkkb011',
    0x9431: 'kkkkaaaz',
    0x9432: 'hhhhha111',
    0x9433: 'hhhhbbdx',
    0x9434: 'mmmmmb111',
    0x9435: 'mmmmabdx',
    0x9436: 'nnnnna011',
    0x9437: 'nnnnbaaz',
    0x9438: 'gggggb001',
    0x9439: 'ggggaadx',
    0x943a: 'ddddda101',
    0x943b: 'ddddbbaz',
    0x943c: 'aaaaab101',
    0x943d: 'aaaaabaz',
    0x943e: 'bbbbba001',
    0x943f: 'bbbbbadx',
    0x9440: 'cccccb101',
    0x9441: 'ccccaadx',
    0x9442: 'CCCCCCbb99',
    0x9443: 'ppppabbz',
    0x9444: 'eeeeeb001',
    0x9445: 'eeeeabaz',
    0x9446: 'fffffa101',
    0x9447: 'ffffbadx',
    0x9448: 'ooooob111',
    0x9449: 'ooooaaaz',
    0x944a: 'llllla011',
    0x944b: 'llllbbdx',
    0x944c: 'iiiiib011',
    0x944d: 'iiiiabdx',
    0x944e: 'jjjjja111',
    0x944f: 'jjjjbaaz',
    0x9450: 'XXXXXXaa88',
    0x9451: 'kkkkbaaz',
    0x9452: 'xxxxxa000',
    0x9453: 'hhhhabdx',
    0x9454: '=====b117',
    0x9455: 'mmmmbbdx',
    0x9456: '>>>>>a017',
    0x9457: 'nnnnaaaz',
    0x9458: 'wwwwwb110',
    0x9459: 'ggggbadx',
    0x945a: 'ttttta010',
    0x945b: 'ddddabaz',
    0x945c: 'qqqqqb010',
    0x945d: 'aaaabbaz',
    0x945e: 'rrrrra110',
    0x945f: 'bbbbaadx',
    0x9460: 'ppppppai89',
    0x9461: 'ccccbbaz',
    0x9462: 'pppppb010',
    0x9463: 'ppppbacz',
    0x9464: 'uuuuua010',
    0x9465: 'eeeebadx',
    0x9466: 'uuuuuubi89',
    0x9467: 'ffffabaz',
    0x9468: '00000000abbZ',
    0x9469: 'oooobbdx',
    0x946a: '00000000abbJ',
    0x946b: 'llllaaaz',
    0x946c: 'yyyyya000',
    0x946d: 'iiiibaaz',
    0x946e: 'yyyyyybi99',
    0x946f: 'jjjjabdx',
    0x9470: 'hhhhhhai88',
    0x9471: 'kkkkabdx',
    0x9472: 'hhhhhb011',
    0x9473: 'hhhhbaaz',
    0x9474: 'mmmmma011',
    0x9475: 'mmmmaaaz',
    0x9476: 'mmmmmmbi88',
    0x9477: 'nnnnbbdx',
    0x9478: 'ddddddai98',
    0x9479: 'ggggabaz',
    0x947a: 'dddddb001',
    0x947b: 'ddddbadx',
    0x947c: 'aaaaaa001',
    0x947d: 'aaaaaadx',
    0x947e: 'aaaaaabi98',
    0x947f: 'bbbbbbaz',
    0x9480: 'ccccca101',
    0x9481: 'ccccabdx',
    0x9482: 'ccccccbi98',
    0x9483: 'ppppaabz',
    0x9484: 'eeeeea001',
    0x9485: 'eeeeaaaz',
    0x9486: 'eeeeeebh98',
    0x9487: 'ffffbbdx',
    0x9488: 'llllllah88',
    0x9489: 'ooooabaz',
    0x948a: 'lllllb011',
    0x948b: 'llllbadx',
    0x948c: 'iiiiia011',
    0x948d: 'iiiiaadx',
    0x948e: 'iiiiiibh88',
    0x948f: 'jjjjbbaz',
    0x9490: 'xxxxxxah99',
    0x9491: 'kkkkbbaz',
    0x9492: 'xxxxxb000',
    0x9493: 'hhhhaadx',
    0x9494: '8888888abbl',
    0x9495: 'mmmmbadx',
    0x9496: '8888888abbL',
    0x9497: 'nnnnabaz',
    0x9498: 'ttttttah89',
    0x9499: 'ggggbbdx',
    0x949a: 'tttttb010',
    0x949b: 'ddddaaaz',
    0x949c: 'qqqqqa010',
    0x949d: 'aaaabaaz',
    0x949e: 'qqqqqqbh89',
    0x949f: 'bbbbabdx',
    0x94a0: 'sssssb110',
    0x94a1: 'ccccbaaz',
    0x94a2: 'pppppa010',
    0x94a3: 'ppppbbcz',
    0x94a4: 'uuuuub010',
    0x94a5: 'eeeebbdx',
    0x94a6: 'vvvvva110',
    0x94a7: 'ffffaaaz',
    0x94a8: '?????b017',
    0x94a9: 'oooobadx',
    0x94aa: '<<<<<a117',
    0x94ab: 'llllabaz',
    0x94ac: 'yyyyyb000',
    0x94ad: 'iiiibbaz',
    0x94ae: 'zzzzza100',
    0x94af: 'jjjjaadx',
    0x94b0: 'kkkkkb111',
    0x94b1: 'kkkkaadx',
    0x94b2: 'hhhhha011',
    0x94b3: 'hhhhbbaz',
    0x94b4: 'mmmmmb011',
    0x94b5: 'mmmmabaz',
    0x94b6: 'nnnnna111',
    0x94b7: 'nnnnbadx',
    0x94b8: 'gggggb101',
    0x94b9: 'ggggaaaz',
    0x94ba: 'ddddda001',
    0x94bb: 'ddddbbdx',
    0x94bc: 'aaaaab001',
    0x94bd: 'aaaaabdx',
    0x94be: 'bbbbba101',
    0x94bf: 'bbbbbaaz',
    0x94c0: 'cccccb001',
    0x94c1: 'ccccaaaz',
    0x94c2: 'CCCCCCaa89',
    0x94c3: 'ppppabcz',
    0x94c4: 'eeeeeb101',
    0x94c5: 'eeeeabdx',
    0x94c6: 'fffffa001',
    0x94c7: 'ffffbaaz',
    0x94c8: 'ooooob011',
    0x94c9: 'ooooaadx',
    0x94ca: 'llllla111',
    0x94cb: 'llllbbaz',
    0x94cc: 'iiiiib111',
    0x94cd: 'iiiiabaz',
    0x94ce: 'jjjjja011',
    0x94cf: 'jjjjbadx',
    0x94d0: 'XXXXXXbb98',
    0x94d1: 'kkkkbadx',
    0x94d2: 'xxxxxa100',
    0x94d3: 'hhhhabaz',
    0x94d4: 'XXXXXXXabbh',
    0x94d5: 'mmmmbbaz',
    0x94d6: 'XXXXXXXabbH',
    0x94d7: 'nnnnaadx',
    0x94d8: 'wwwwwb010',
    0x94d9: 'ggggbaaz',
    0x94da: 'ttttta110',
    0x94db: 'ddddabdx',
    0x94dc: 'qqqqqb110',
    0x94dd: 'aaaabbdx',
    0x94de: 'rrrrra010',
    0x94df: 'bbbbaaaz',
    0x94e0: 'ppppppbh89',
    0x94e1: 'ccccbbdx',
    0x94e2: 'pppppb110',
    0x94e3: 'ppppbabz',
    0x94e4: 'pppppppabbk',
    0x94e5: 'eeeebaaz',
    0x94e6: 'pppppppabbK',
    0x94e7: 'ffffabdx',
    0x94e8: 'yyyyyyybbaJ',
    0x94e9: 'oooobbaz',
    0x94ea: 'yyyyyyybbaj',
    0x94eb: 'llllaadx',
    0x94ec: 'yyyyya100',
    0x94ed: 'iiiibadx',
    0x94ee: 'yyyyyyah99',
    0x94ef: 'jjjjabaz',
    0x94f0: 'hhhhhhbh88',
    0x94f1: 'hhhhhhhab6',
    0x94f2: 'hhhhhb111',
    0x94f3: 'hhhhbadx',
    0x94f4: 'hhhhhhhabbj',
    0x94f5: 'hhhhhhhabbz',
    0x94f6: 'hhhhhhhabbJ',
    0x94f7: 'hhhhhhhabbZ',
    0x94f8: 'aaaaaaabbaK',
    0x94f9: 'aaaaaaabbcZ',
    0x94fa: 'aaaaaaabbak',
    0x94fb: 'aaaaaaabbcz',
    0x94fc: 'aaaaaa101',
    0x94fd: 'aaaaaaaz',
    0x94fe: 'aaaaaaah98',
    0x94ff: 'aaaaaaabb7',
    0x9500: 'ccccca011',
    0x9501: 'ccccabcz',
    0x9502: 'ccccccai98',
    0x9503: 'cccccccbb7',
    0x9504: 'cccccccbbaK',
    0x9505: 'cccccccbbcZ',
    0x9506: 'cccccccbbak',
    0x9507: 'cccccccbbcz',
    0x9508: 'iiiiiiibaaK',
    0x9509: 'iiiiiiibacZ',
    0x950a: 'iiiiiiibaak',
    0x950b: 'iiiiiiibacz',
    0x950c: 'iiiiia101',
    0x950d: 'iiiiaacz',
    0x950e: 'iiiiiiah88',
    0x950f: 'iiiiiiiba7',
    0x9510: 'xxxxxxbh99',
    0x9511: 'kkkkbbbz',
    0x9512: 'xxxxxb110',
    0x9513: 'hhhhaacz',
    0x9514: 'xxxxxxxaabk',
    0x9515: 'mmmmbacz',
    0x9516: 'xxxxxxxaabK',
    0x9517: 'nnnnabbz',
    0x9518: 'qqqqqqqbaaJ',
    0x9519: 'ggggbbcz',
    0x951a: 'qqqqqqqbaaj',
    0x951b: 'ddddaabz',
    0x951c: 'qqqqqa100',
    0x951d: 'aaaababz',
    0x951e: 'qqqqqqah89',
    0x951f: 'bbbbabcz',
    0x9520: 'sssssb000',
    0x9521: 'ccccbabz',
    0x9522: 'pppppa100',
    0x9523: 'ppppbbdx',
    0x9524: 'uuuuub100',
    0x9525: 'eeeebbcz',
    0x9526: 'vvvvva000',
    0x9527: 'ffffaabz',
    0x9528: 'YYYYYYYbaaI',
    0x9529: 'oooobacz',
    0x952a: 'YYYYYYYbaai',
    0x952b: 'llllabbz',
    0x952c: 'yyyyyb110',
    0x952d: 'iiiibbbz',
    0x952e: 'zzzzza010',
    0x952f: 'jjjjaacz',
    0x9530: 'kkkkkb001',
    0x9531: 'kkkkaacz',
    0x9532: 'hhhhha101',
    0x9533: 'hhhhbbbz',
    0x9534: 'mmmmmb101',
    0x9535: 'mmmmabbz',
    0x9536: 'nnnnna001',
    0x9537: 'nnnnbacz',
    0x9538: 'gggggb011',
    0x9539: 'ggggaabz',
    0x953a: 'ddddda111',
    0x953b: 'ddddbbcz',
    0x953c: 'aaaaab111',
    0x953d: 'aaaaabcz',
    0x953e: 'bbbbba011',
    0x953f: 'bbbbbabz',
    0x9540: 'cccccb111',
    0x9541: 'ccccaabz',
    0x9542: 'CCCCCCba89',
    0x9543: 'ppppabdx',
    0x9544: 'eeeeeb011',
    0x9545: 'eeeeabcz',
    0x9546: 'fffffa111',
    0x9547: 'ffffbabz',
    0x9548: 'ooooob101',
    0x9549: 'ooooaacz',
    0x954a: 'llllla001',
    0x954b: 'llllbbbz',
    0x954c: 'iiiiib001',
    0x954d: 'iiiiabbz',
    0x954e: 'jjjjja101',
    0x954f: 'jjjjbacz',
    0x9550: 'XXXXXXab98',
    0x9551: 'kkkkbacz',
    0x9552: 'xxxxxa010',
    0x9553: 'hhhhabbz',
    0x9554: '=====b107',
    0x9555: 'mmmmbbbz',
    0x9556: '>>>>>a007',
    0x9557: 'nnnnaacz',
    0x9558: 'wwwwwb100',
    0x9559: 'ggggbabz',
    0x955a: 'ttttta000',
    0x955b: 'ddddabcz',
    0x955c: 'qqqqqb000',
    0x955d: 'aaaabbcz',
    0x955e: 'rrrrra100',
    0x955f: 'bbbbaabz',
    0x9560: 'ppppppah89',
    0x9561: 'ccccbbcz',
    0x9562: 'pppppb000',
    0x9563: 'ppppbaaz',
    0x9564: 'uuuuua000',
    0x9565: 'eeeebabz',
    0x9566: 'uuuuuubh89',
    0x9567: 'ffffabcz',
    0x9568: '9999999baaM',
    0x9569: 'oooobbbz',
    0x956a: '9999999baam',
    0x956b: 'llllaacz',
    0x956c: 'yyyyya010',
    0x956d: 'iiiibacz',
    0x956e: 'yyyyyybh99',
    0x956f: 'jjjjabbz',
    0x9570: 'hhhhhhah88',
    0x9571: 'kkkkabbz',
    0x9572: 'hhhhhb001',
    0x9573: 'hhhhbacz',
    0x9574: 'mmmmma001',
    0x9575: 'mmmmaacz',
    0x9576: 'mmmmmmbh88',
    0x9577: 'nnnnbbbz',
    0x9578: 'ddddddah98',
    0x9579: 'ggggabcz',
    0x957a: 'dddddb011',
    0x957b: 'ddddbabz',
    0x957c: 'aaaaaa011',
    0x957d: 'aaaaaabz',
    0x957e: 'aaaaaabh98',
    0x957f: 'bbbbbbcz',
    0x9580: 'ccccca111',
    0x9581: 'ccccabbz',
    0x9582: 'ccccccbh98',
    0x9583: 'ppppaadx',
    0x9584: 'eeeeea011',
    0x9585: 'eeeeaacz',
    0x9586: 'eeeeeebi98',
    0x9587: 'ffffbbbz',
    0x9588: 'llllllai88',
    0x9589: 'ooooabcz',
    0x958a: 'lllllb001',
    0x958b: 'llllbabz',
    0x958c: 'iiiiia001',
    0x958d: 'iiiiaabz',
    0x958e: 'iiiiiibi88',
    0x958f: 'jjjjbbcz',
    0x9590: 'xxxxxxai99',
    0x9591: 'kkkkbbcz',
    0x9592: 'xxxxxb010',
    0x9593: 'hhhhaabz',
    0x9594: '11111111abbJ',
    0x9595: 'mmmmbabz',
    0x9596: '11111111abbZ',
    0x9597: 'nnnnabcz',
    0x9598: 'ttttttai89',
    0x9599: 'ggggbbbz',
    0x959a: 'tttttb000',
    0x959b: 'ddddaacz',
    0x959c: 'qqqqqa000',
    0x959d: 'aaaabacz',
    0x959e: 'qqqqqqbi89',
    0x959f: 'bbbbabbz',
    0x95a0: 'sssssb100',
    0x95a1: 'ccccbacz',
    0x95a2: 'pppppa000',
    0x95a3: 'ppppbbaz',
    0x95a4: 'uuuuub000',
    0x95a5: 'eeeebbbz',
    0x95a6: 'vvvvva100',
    0x95a7: 'ffffaacz',
    0x95a8: '?????b007',
    0x95a9: 'oooobabz',
    0x95aa: '<<<<<a107',
    0x95ab: 'llllabcz',
    0x95ac: 'yyyyyb010',
    0x95ad: 'iiiibbcz',
    0x95ae: 'zzzzza110',
    0x95af: 'jjjjaabz',
    0x95b0: 'kkkkkb101',
    0x95b1: 'kkkkaabz',
    0x95b2: 'hhhhha001',
    0x95b3: 'hhhhbbcz',
    0x95b4: 'mmmmmb001',
    0x95b5: 'mmmmabcz',
    0x95b6: 'nnnnna101',
    0x95b7: 'nnnnbabz',
    0x95b8: 'gggggb111',
    0x95b9: 'ggggaacz',
    0x95ba: 'ddddda011',
    0x95bb: 'ddddbbbz',
    0x95bc: 'aaaaab011',
    0x95bd: 'aaaaabbz',
    0x95be: 'bbbbba111',
    0x95bf: 'bbbbbacz',
    0x95c0: 'cccccb011',
    0x95c1: 'ccccaacz',
    0x95c2: 'CCCCCCab99',
    0x95c3: 'ppppabaz',
    0x95c4: 'eeeeeb111',
    0x95c5: 'eeeeabbz',
    0x95c6: 'fffffa011',
    0x95c7: 'ffffbacz',
    0x95c8: 'ooooob001',
    0x95c9: 'ooooaabz',
    0x95ca: 'llllla101',
    0x95cb: 'llllbbcz',
    0x95cc: 'iiiiib101',
    0x95cd: 'iiiiabcz',
    0x95ce: 'jjjjja001',
    0x95cf: 'jjjjbabz',
    0x95d0: 'CCCCCCCCCbabp',
    0x95d1: 'kkkkbabz',
    0x95d2: 'xxxxxa110',
    0x95d3: 'hhhhabcz',
    0x95d4: 'CCCCCCCCCbadc',
    0x95d5: 'mmmmbbcz',
    0x95d6: 'CCCCCCCCCbabh',
    0x95d7: 'nnnnaabz',
    0x95d8: 'wwwwwb000',
    0x95d9: 'ggggbacz',
    0x95da: 'ttttta100',
    0x95db: 'ddddabbz',
    0x95dc: 'qqqqqb100',
    0x95dd: 'aaaabbbz',
    0x95de: 'rrrrra000',
    0x95df: 'bbbbaacz',
    0x95e0: 'aaaaaaaaabbbC',
    0x95e1: 'aaaaaaaaabbbG',
    0x95e2: 'aaaaaaaaabbbK',
    0x95e3: 'aaaaaaaaabbbO',
    0x95e4: 'aaaaaaaaabbbS',
    0x95e5: 'aaaaaaaaabbbW',
    0x95e6: 'aaaaaaaaabbdX',
    0x95e7: 'aaaaaaaaabbhZ',
    0x95e8: 'aaaaaaaaabbbc',
    0x95e9: 'aaaaaaaaabbbg',
    0x95ea: 'aaaaaaaaabbbk',
    0x95eb: 'aaaaaaaaabbbo',
    0x95ec: 'aaaaaaaaabbbs',
    0x95ed: 'aaaaaaaaabbbw',
    0x95ee: 'aaaaaaaaabbdx',
    0x95ef: 'aaaaaaaaabbhz',
    0x95f0: 'aaaaaaaaabbl',
    0x95f1: 'aaaaaaaaabbd',
    0x95f2: 'aaaaaaaaabhy',
    0x95f3: 'aaaaaaaaabbt',
    0x95f4: 'aaaaaaaaabbL',
    0x95f5: 'aaaaaaaaabbD',
    0x95f6: 'aaaaaaaaabhY',
    0x95f7: 'aaaaaaaaabbT',
    0x95f8: 'aaaaaaaaabJ',
    0x95f9: 'aaaaaaaaabZ',
    0x95fa: 'aaaaaaaaabj',
    0x95fb: 'aaaaaaaaabz',
    0x95fc: 'aaaaaa111',
    0x95fd: 'aaaaaacz',
    0x95fe: 'aaaaaaai98',
    0x95ff: 'aaaaaaaaa6',
    0x9600: 'ccccca000',
    0x9601: 'ccccabax',
    0x9602: 'ccccccah88',
    0x9603: 'cccccccca6',
    0x9604: 'cccccccabaK',
    0x9605: 'cccccccabcZ',
    0x9606: 'cccccccabak',
    0x9607: 'cccccccabcz',
    0x9608: 'ccccccccabbL',
    0x9609: 'ccccccccabbD',
    0x960a: 'ccccccccabhY',
    0x960b: 'ccccccccabbT',
    0x960c: 'ccccccccabbl',
    0x960d: 'ccccccccabbd',
    0x960e: 'ccccccccabhy',
    0x960f: 'ccccccccabbt',
    0x9610: 'cccccccccaaar',
    0x9611: 'cccccccccaaav',
    0x9612: 'cccccccccaaaz',
    0x9613: 'cccccccccaaiz',
    0x9614: 'cccccccccaaab',
    0x9615: 'cccccccccaaaf',
    0x9616: 'cccccccccaaaj',
    0x9617: 'cccccccccaaan',
    0x9618: 'cccccccccaaaR',
    0x9619: 'cccccccccaaaV',
    0x961a: 'cccccccccaaaZ',
    0x961b: 'cccccccccaaiZ',
    0x961c: 'cccccccccaaaB',
    0x961d: 'aaaababy',
    0x961e: 'cccccccccaaaJ',
    0x961f: 'bbbbabax',
    0x9620: 'ccccccccccbbao',
    0x9621: 'ccccbaby',
    0x9622: 'ccccccccccbbak',
    0x9623: 'ccccccccccbbai',
    0x9624: 'ccccccccccbbag',
    0x9625: 'ccccccccccbbae',
    0x9626: 'ccccccccccbbac',
    0x9627: 'ccccccccccbbaa',
    0x9628: 'ccccccccccbbkz',
    0x9629: 'ccccccccccbbiy',
    0x962a: 'ccccccccccbbcz',
    0x962b: 'ccccccccccbbay',
    0x962c: 'ccccccccccbbaw',
    0x962d: 'ccccccccccbbau',
    0x962e: 'ccccccccccbbas',
    0x962f: 'ccccccccccbbaq',
    0x9630: 'ccccccccccbbaO',
    0x9631: 'ccccccccccbbaM',
    0x9632: 'ccccccccccbbaK',
    0x9633: 'ccccccccccbbaI',
    0x9634: 'ccccccccccbbaG',
    0x9635: 'ccccccccccbbaE',
    0x9636: 'ccccccccccbbaC',
    0x9637: 'ccccccccccbbaA',
    0x9638: 'ccccccccccbbkZ',
    0x9639: 'ccccccccccbbiY',
    0x963a: 'ccccccccccbbcZ',
    0x963b: 'ccccccccccbbaY',
    0x963c: 'aaaaab100',
    0x963d: 'aaaaabax',
    0x963e: 'bbbbba000',
    0x963f: 'bbbbbaby',
    0x9640: 'cccccb100',
    0x9641: 'ccccaaby',
    0x9642: 'CCCCCCbb89',
    0x9643: 'ppppabbx',
    0x9644: 'eeeeeb000',
    0x9645: 'eeeeabax',
    0x9646: 'fffffa100',
    0x9647: 'ffffbaby',
    0x9648: 'ooooob110',
    0x9649: 'ooooaaax',
    0x964a: 'llllla010',
    0x964b: 'llllbbby',
    0x964c: 'iiiiib010',
    0x964d: 'iiiiabby',
    0x964e: 'jjjjja110',
    0x964f: 'jjjjbaax',
    0x9650: 'XXXXXXaa98',
    0x9651: 'kkkkbaax',
    0x9652: 'xxxxxa001',
    0x9653: 'hhhhabby',
    0x9654: '0000000000aabD',
    0x9655: 'mmmmbbby',
    0x9656: '0000000000aadC',
    0x9657: 'nnnnaaax',
    0x9658: 'wwwwwb111',
    0x9659: 'ggggbaby',
    0x965a: 'ttttta011',
    0x965b: 'ddddabax',
    0x965c: 'qqqqqb011',
    0x965d: 'aaaabbax',
    0x965e: 'rrrrra111',
    0x965f: 'bbbbaaby',
    0x9660: 'ppppppai99',
    0x9661: 'ccccbbax',
    0x9662: 'pppppb011',
    0x9663: 'ppppbaay',
    0x9664: 'uuuuua011',
    0x9665: 'eeeebaby',
    0x9666: 'uuuuuubi99',
    0x9667: 'ffffabax',
    0x9668: '00000000bacZ',
    0x9669: 'oooobbby',
    0x966a: '00000000baaK',
    0x966b: 'llllaaax',
    0x966c: 'yyyyya001',
    0x966d: 'iiiibaax',
    0x966e: 'yyyyyybi89',
    0x966f: 'jjjjabby',
    0x9670: 'hhhhhhai98',
    0x9671: 'kkkkabby',
    0x9672: 'hhhhhb010',
    0x9673: 'hhhhbaax',
    0x9674: 'mmmmma010',
    0x9675: 'mmmmaaax',
    0x9676: 'mmmmmmbi98',
    0x9677: 'nnnnbbby',
    0x9678: 'ddddddai88',
    0x9679: 'ggggabax',
    0x967a: 'dddddb000',
    0x967b: 'ddddbaby',
    0x967c: 'aaaaaa000',
    0x967d: 'aaaaaaby',
    0x967e: 'aaaaaabi88',
    0x967f: 'bbbbbbax',
    0x9680: 'ccccca100',
    0x9681: 'ccccabby',
    0x9682: 'ccccccbi88',
    0x9683: 'ppppaabx',
    0x9684: 'eeeeea000',
    0x9685: 'eeeeaaax',
    0x9686: 'eeeeeebh88',
    0x9687: 'ffffbbby',
    0x9688: 'llllllah98',
    0x9689: 'ooooabax',
    0x968a: 'lllllb010',
    0x968b: 'llllbaby',
    0x968c: 'iiiiia010',
    0x968d: 'iiiiaaby',
    0x968e: 'iiiiiibh98',
    0x968f: 'jjjjbbax',
    0x9690: 'xxxxxxah89',
    0x9691: 'kkkkbbax',
    0x9692: 'xxxxxb001',
    0x9693: 'hhhhaaby',
    0x9694: '8888888aaam',
    0x9695: 'mmmmbaby',
    0x9696: '8888888aaaM',
    0x9697: 'nnnnabax',
    0x9698: 'ttttttah99',
    0x9699: 'ggggbbby',
    0x969a: 'tttttb011',
    0x969b: 'ddddaaax',
    0x969c: 'qqqqqa011',
    0x969d: 'aaaabaax',
    0x969e: 'qqqqqqbh99',
    0x969f: 'bbbbabby',
    0x96a0: 'sssssb111',
    0x96a1: 'ccccbaax',
    0x96a2: 'pppppa011',
    0x96a3: 'ppppbbay',
    0x96a4: 'uuuuub011',
    0x96a5: 'eeeebbby',
    0x96a6: 'vvvvva111',
    0x96a7: 'ffffaaax',
    0x96a8: '?????b016',
    0x96a9: 'oooobaby',
    0x96aa: '<<<<<a116',
    0x96ab: 'llllabax',
    0x96ac: 'yyyyyb001',
    0x96ad: 'iiiibbax',
    0x96ae: 'zzzzza101',
    0x96af: 'jjjjaaby',
    0x96b0: 'kkkkkb110',
    0x96b1: 'kkkkaaby',
    0x96b2: 'hhhhha010',
    0x96b3: 'hhhhbbax',
    0x96b4: 'mmmmmb010',
    0x96b5: 'mmmmabax',
    0x96b6: 'nnnnna110',
    0x96b7: 'nnnnbaby',
    0x96b8: 'gggggb100',
    0x96b9: 'ggggaaax',
    0x96ba: 'ddddda000',
    0x96bb: 'ddddbbby',
    0x96bc: 'aaaaab000',
    0x96bd: 'aaaaabby',
    0x96be: 'bbbbba100',
    0x96bf: 'bbbbbaax',
    0x96c0: 'cccccb000',
    0x96c1: 'ccccaaax',
    0x96c2: 'CCCCCCaa99',
    0x96c3: 'ppppabay',
    0x96c4: 'eeeeeb100',
    0x96c5: 'eeeeabby',
    0x96c6: 'fffffa000',
    0x96c7: 'ffffbaax',
    0x96c8: 'ooooob010',
    0x96c9: 'ooooaaby',
    0x96ca: 'llllla110',
    0x96cb: 'llllbbax',
    0x96cc: 'iiiiib110',
    0x96cd: 'iiiiabax',
    0x96ce: 'jjjjja010',
    0x96cf: 'jjjjbaby',
    0x96d0: 'XXXXXXbb88',
    0x96d1: 'kkkkbaby',
    0x96d2: 'xxxxxa101',
    0x96d3: 'hhhhabax',
    0x96d4: 'XXXXXXXaaai',
    0x96d5: 'mmmmbbax',
    0x96d6: 'XXXXXXXaaaI',
    0x96d7: 'nnnnaaby',
    0x96d8: 'wwwwwb011',
    0x96d9: 'ggggbaax',
    0x96da: 'ttttta111',
    0x96db: 'ddddabby',
    0x96dc: 'qqqqqb111',
    0x96dd: 'aaaabbby',
    0x96de: 'rrrrra011',
    0x96df: 'bbbbaaax',
    0x96e0: 'ppppppbh99',
    0x96e1: 'ccccbbby',
    0x96e2: 'pppppb111',
    0x96e3: 'ppppbabx',
    0x96e4: 'pppppppaaaj',
    0x96e5: 'eeeebaax',
    0x96e6: 'pppppppaaaJ',
    0x96e7: 'ffffabby',
    0x96e8: 'yyyyyyybabK',
    0x96e9: 'oooobbax',
    0x96ea: 'yyyyyyybabk',
    0x96eb: 'llllaaby',
    0x96ec: 'yyyyya101',
    0x96ed: 'iiiibaby',
    0x96ee: 'yyyyyyah89',
    0x96ef: 'jjjjabax',
    0x96f0: 'hhhhhhbh98',
    0x96f1: 'hhhhhhhaa7',
    0x96f2: 'hhhhhb110',
    0x96f3: 'hhhhbaby',
    0x96f4: 'hhhhhhhaaak',
    0x96f5: 'hhhhhhhaacz',
    0x96f6: 'hhhhhhhaaaK',
    0x96f7: 'hhhhhhhaacZ',
    0x96f8: 'aaaaaaababJ',
    0x96f9: 'aaaaaaababZ',
    0x96fa: 'aaaaaaababj',
    0x96fb: 'aaaaaaababz',
    0x96fc: 'aaaaaa100',
    0x96fd: 'aaaaaaax',
    0x96fe: 'aaaaaaah88',
    0x96ff: 'bbbbbbby',
    0x9700: 'ccccca010',
    0x9701: 'ccccabay',
    0x9702: 'ccccccai88',
    0x9703: 'eeeeeeebbcz',
    0x9704: 'cccccccbabJ',
    0x9705: 'cccccccbabZ',
    0x9706: 'cccccccbabj',
    0x9707: 'cccccccbabz',
    0x9708: 'iiiiiiibbbJ',
    0x9709: 'iiiiiiibbbZ',
    0x970a: 'iiiiiiibbbj',
    0x970b: 'iiiiiiibbbz',
    0x970c: 'iiiiia100',
    0x970d: 'iiiiaaay',
    0x970e: 'iiiiiiah98',
    0x970f: 'iiiiiiibb6',
    0x9710: 'xxxxxxbh89',
    0x9711: 'kkkkbbbx',
    0x9712: 'xxxxxb111',
    0x9713: 'hhhhaaay',
    0x9714: 'xxxxxxxabaj',
    0x9715: 'mmmmbaay',
    0x9716: 'xxxxxxxabaJ',
    0x9717: 'nnnnabbx',
    0x9718: 'qqqqqqqbbbK',
    0x9719: 'ggggbbay',
    0x971a: 'qqqqqqqbbbk',
    0x971b: 'ddddaabx',
    0x971c: 'qqqqqa101',
    0x971d: 'aaaababx',
    0x971e: 'qqqqqqah99',
    0x971f: 'bbbbabay',
    0x9720: 'sssssb001',
    0x9721: 'ccccbabx',
    0x9722: 'pppppa101',
    0x9723: 'ppppbbby',
    0x9724: 'uuuuub101',
    0x9725: 'eeeebbay',
    0x9726: 'vvvvva001',
    0x9727: 'ffffaabx',
    0x9728: 'YYYYYYYbbbH',
    0x9729: 'oooobaay',
    0x972a: 'YYYYYYYbbbh',
    0x972b: 'llllabbx',
    0x972c: 'yyyyyb111',
    0x972d: 'iiiibbbx',
    0x972e: 'zzzzza011',
    0x972f: 'jjjjaaay',
    0x9730: 'kkkkkb000',
    0x9731: 'kkkkaaay',
    0x9732: 'hhhhha100',
    0x9733: 'hhhhbbbx',
    0x9734: 'mmmmmb100',
    0x9735: 'mmmmabbx',
    0x9736: 'nnnnna000',
    0x9737: 'nnnnbaay',
    0x9738: 'gggggb010',
    0x9739: 'ggggaabx',
    0x973a: 'ddddda110',
    0x973b: 'ddddbbay',
    0x973c: 'aaaaab110',
    0x973d: 'aaaaabay',
    0x973e: 'bbbbba010',
    0x973f: 'bbbbbabx',
    0x9740: 'cccccb110',
    0x9741: 'ccccaabx',
    0x9742: 'CCCCCCba99',
    0x9743: 'ppppabby',
    0x9744: 'eeeeeb010',
    0x9745: 'eeeeabay',
    0x9746: 'fffffa110',
    0x9747: 'ffffbabx',
    0x9748: 'ooooob100',
    0x9749: 'ooooaaay',
    0x974a: 'llllla000',
    0x974b: 'llllbbbx',
    0x974c: 'iiiiib000',
    0x974d: 'iiiiabbx',
    0x974e: 'jjjjja100',
    0x974f: 'jjjjbaay',
    0x9750: 'XXXXXXab88',
    0x9751: 'kkkkbaay',
    0x9752: 'xxxxxa011',
    0x9753: 'hhhhabbx',
    0x9754: '=====b106',
    0x9755: 'mmmmbbbx',
    0x9756: '>>>>>a006',
    0x9757: 'nnnnaaay',
    0x9758: 'wwwwwb101',
    0x9759: 'ggggbabx',
    0x975a: 'ttttta001',
    0x975b: 'ddddabay',
    0x975c: 'qqqqqb001',
    0x975d: 'aaaabbay',
    0x975e: 'rrrrra101',
    0x975f: 'bbbbaabx',
    0x9760: 'ppppppah99',
    0x9761: 'ccccbbay',
    0x9762: 'pppppb001',
    0x9763: 'ppppbaax',
    0x9764: 'uuuuua001',
    0x9765: 'eeeebabx',
    0x9766: 'uuuuuubh99',
    0x9767: 'ffffabay',
    0x9768: '9999999bbbL',
    0x9769: 'oooobbbx',
    0x976a: '9999999bbbl',
    0x976b: 'llllaaay',
    0x976c: 'yyyyya011',
    0x976d: 'iiiibaay',
    0x976e: 'yyyyyybh89',
    0x976f: 'jjjjabbx',
    0x9770: 'hhhhhhah98',
    0x9771: 'kkkkabbx',
    0x9772: 'hhhhhb000',
    0x9773: 'hhhhbaay',
    0x9774: 'mmmmma000',
    0x9775: 'mmmmaaay',
    0x9776: 'mmmmmmbh98',
    0x9777: 'nnnnbbbx',
    0x9778: 'ddddddah88',
    0x9779: 'ggggabay',
    0x977a: 'dddddb010',
    0x977b: 'ddddbabx',
    0x977c: 'aaaaaa010',
    0x977d: 'aaaaaabx',
    0x977e: 'aaaaaabh88',
    0x977f: 'bbbbbbay',
    0x9780: 'ccccca110',
    0x9781: 'ccccabbx',
    0x9782: 'ccccccbh88',
    0x9783: 'ppppaaby',
    0x9784: 'eeeeea010',
    0x9785: 'eeeeaaay',
    0x9786: 'eeeeeebi88',
    0x9787: 'ffffbbbx',
    0x9788: 'llllllai98',
    0x9789: 'ooooabay',
    0x978a: 'lllllb000',
    0x978b: 'llllbabx',
    0x978c: 'iiiiia000',
    0x978d: 'iiiiaabx',
    0x978e: 'iiiiiibi98',
    0x978f: 'jjjjbbay',
    0x9790: 'xxxxxxai89',
    0x9791: 'kkkkbbay',
    0x9792: 'xxxxxb011',
    0x9793: 'hhhhaabx',
    0x9794: '11111111baaK',
    0x9795: 'mmmmbabx',
    0x9796: '11111111bacZ',
    0x9797: 'nnnnabay',
    0x9798: 'ttttttai99',
    0x9799: 'ggggbbbx',
    0x979a: 'tttttb001',
    0x979b: 'ddddaaay',
    0x979c: 'qqqqqa001',
    0x979d: 'aaaabaay',
    0x979e: 'qqqqqqbi99',
    0x979f: 'bbbbabbx',
    0x97a0: 'sssssb101',
    0x97a1: 'ccccbaay',
    0x97a2: 'pppppa001',
    0x97a3: 'ppppbbax',
    0x97a4: 'uuuuub001',
    0x97a5: 'eeeebbbx',
    0x97a6: 'vvvvva101',
    0x97a7: 'ffffaaay',
    0x97a8: '?????b006',
    0x97a9: 'oooobabx',
    0x97aa: '<<<<<a106',
    0x97ab: 'llllabay',
    0x97ac: 'yyyyyb011',
    0x97ad: 'iiiibbay',
    0x97ae: 'zzzzza111',
    0x97af: 'jjjjaabx',
    0x97b0: 'kkkkkb100',
    0x97b1: 'kkkkaabx',
    0x97b2: 'hhhhha000',
    0x97b3: 'hhhhbbay',
    0x97b4: 'mmmmmb000',
    0x97b5: 'mmmmabay',
    0x97b6: 'nnnnna100',
    0x97b7: 'nnnnbabx',
    0x97b8: 'gggggb110',
    0x97b9: 'ggggaaay',
    0x97ba: 'ddddda010',
    0x97bb: 'ddddbbbx',
    0x97bc: 'aaaaab010',
    0x97bd: 'aaaaabbx',
    0x97be: 'bbbbba110',
    0x97bf: 'bbbbbaay',
    0x97c0: 'cccccb010',
    0x97c1: 'ccccaaay',
    0x97c2: 'CCCCCCab89',
    0x97c3: 'ppppabax',
    0x97c4: 'eeeeeb110',
    0x97c5: 'eeeeabbx',
    0x97c6: 'fffffa010',
    0x97c7: 'ffffbaay',
    0x97c8: 'ooooob000',
    0x97c9: 'ooooaabx',
    0x97ca: 'llllla100',
    0x97cb: 'llllbbay',
    0x97cc: 'iiiiib100',
    0x97cd: 'iiiiabay',
    0x97ce: 'jjjjja000',
    0x97cf: 'jjjjbabx',
    0x97d0: 'QQQQQQQQbaao',
    0x97d1: 'kkkkbabx',
    0x97d2: 'xxxxxa111',
    0x97d3: 'hhhhabay',
    0x97d4: 'QQQQQQQQbaaO',
    0x97d5: 'mmmmbbay',
    0x97d6: 'QQQQQQQQbakZ',
    0x97d7: 'nnnnaabx',
    0x97d8: 'wwwwwb001',
    0x97d9: 'ggggbaay',
    0x97da: 'ttttta101',
    0x97db: 'ddddabbx',
    0x97dc: 'qqqqqb101',
    0x97dd: 'aaaabbbx',
    0x97de: 'rrrrra001',
    0x97df: 'bbbbaaay',
    0x97e0: 'ppppppbi99',
    0x97e1: 'ccccbbbx',
    0x97e2: 'pppppb101',
    0x97e3: 'ppppbaby',
    0x97e4: 'pppppppbbbk',
    0x97e5: 'eeeebaay',
    0x97e6: 'pppppppbbbK',
    0x97e7: 'ffffabbx',
    0x97e8: 'ppppppppaahX',
    0x97e9: 'oooobbay',
    0x97ea: 'ppppppppaabM',
    0x97eb: 'llllaabx',
    0x97ec: 'ppppppppaahx',
    0x97ed: 'iiiibabx',
    0x97ee: 'ppppppppaabm',
    0x97ef: 'jjjjabay',
    0x97f0: 'aaaaaaaabaam',
    0x97f1: 'aaaaaaaabaae',
    0x97f2: 'aaaaaaaabaiy',
    0x97f3: 'aaaaaaaabaau',
    0x97f4: 'aaaaaaaabaaM',
    0x97f5: 'aaaaaaaabaaE',
    0x97f6: 'aaaaaaaabaiY',
    0x97f7: 'aaaaaaaabaaU',
    0x97f8: 'aaaaaaaabaK',
    0x97f9: 'aaaaaaaabcZ',
    0x97fa: 'aaaaaaaabak',
    0x97fb: 'aaaaaaaabcz',
    0x97fc: 'aaaaaa110',
    0x97fd: 'aaaaaaay',
    0x97fe: 'aaaaaaai88',
    0x97ff: 'aaaaaaaab7',
    0x9800: 'ccccca007',
    0x9801: 'ccccabav',
    0x9802: 'ccccccab89',
    0x9803: 'ccccccccb4',
    0x9804: 'cccccccaaaJ',
    0x9805: 'cccccccaaaZ',
    0x9806: 'cccccccaaaj',
    0x9807: 'cccccccaaaz',
    0x9808: 'ccccccccbbaM',
    0x9809: 'ccccccccbbaE',
    0x980a: 'ccccccccbbiY',
    0x980b: 'ccccccccbbaU',
    0x980c: 'ccccccccbbam',
    0x980d: 'ccccccccbbae',
    0x980e: 'ccccccccbbiy',
    0x980f: 'ccccccccbbau',
    0x9810: 'qqqqqqqqbaal',
    0x9811: 'kkkkbbbw',
    0x9812: 'qqqqqqqqbaix',
    0x9813: 'hhhhaaav',
    0x9814: 'qqqqqqqqbaaL',
    0x9815: 'mmmmbaav',
    0x9816: 'qqqqqqqqbaiX',
    0x9817: 'nnnnabbw',
    0x9818: 'qqqqqqqabaK',
    0x9819: 'ggggbbav',
    0x981a: 'qqqqqqqabak',
    0x981b: 'ddddaabw',
    0x981c: 'qqqqqa116',
    0x981d: 'aaaababw',
    0x981e: 'qqqqqqaa88',
    0x981f: 'bbbbabav',
    0x9820: 'sssssb016',
    0x9821: 'ccccbabw',
    0x9822: 'pppppa116',
    0x9823: 'ppppbbbv',
    0x9824: 'uuuuub116',
    0x9825: 'eeeebbav',
    0x9826: 'vvvvva016',
    0x9827: 'ffffaabw',
    0x9828: 'PPPPPPPPaajZ',
    0x9829: 'oooobaav',
    0x982a: 'PPPPPPPPaabN',
    0x982b: 'llllabbw',
    0x982c: 'yyyyyb106',
    0x982d: 'iiiibbbw',
    0x982e: 'zzzzza006',
    0x982f: 'jjjjaaav',
    0x9830: 'kkkkkb017',
    0x9831: 'kkkkaaav',
    0x9832: 'hhhhha117',
    0x9833: 'hhhhbbbw',
    0x9834: 'mmmmmb117',
    0x9835: 'mmmmabbw',
    0x9836: 'nnnnna017',
    0x9837: 'nnnnbaav',
    0x9838: 'gggggb007',
    0x9839: 'ggggaabw',
    0x983a: 'ddddda107',
    0x983b: 'ddddbbav',
    0x983c: 'aaaaab107',
    0x983d: 'aaaaabav',
    0x983e: 'bbbbba007',
    0x983f: 'bbbbbabw',
    0x9840: 'cccccb107',
    0x9841: 'ccccaabw',
    0x9842: 'CCCCCCbh88',
    0x9843: 'ppppabbv',
    0x9844: 'eeeeeb007',
    0x9845: 'eeeeabav',
    0x9846: 'fffffa107',
    0x9847: 'ffffbabw',
    0x9848: 'ooooob117',
    0x9849: 'ooooaaav',
    0x984a: 'llllla017',
    0x984b: 'llllbbbw',
    0x984c: 'iiiiib017',
    0x984d: 'iiiiabbw',
    0x984e: 'jjjjja117',
    0x984f: 'jjjjbaav',
    0x9850: 'XXXXXXai89',
    0x9851: 'kkkkbaav',
    0x9852: 'xxxxxa006',
    0x9853: 'hhhhabbw',
    0x9854: '=====b111',
    0x9855: 'mmmmbbbw',
    0x9856: '>>>>>a011',
    0x9857: 'nnnnaaav',
    0x9858: 'wwwwwb116',
    0x9859: 'ggggbabw',
    0x985a: 'ttttta016',
    0x985b: 'ddddabav',
    0x985c: 'qqqqqb016',
    0x985d: 'aaaabbav',
    0x985e: 'rrrrra116',
    0x985f: 'bbbbaabw',
    0x9860: 'ppppppaa88',
    0x9861: 'ccccbbav',
    0x9862: 'pppppb016',
    0x9863: 'ppppbaaw',
    0x9864: 'uuuuua016',
    0x9865: 'eeeebabw',
    0x9866: 'uuuuuuba88',
    0x9867: 'ffffabav',
    0x9868: '00000000aabZ',
    0x9869: 'oooobbbw',
    0x986a: '00000000aabJ',
    0x986b: 'llllaaav',
    0x986c: 'yyyyya006',
    0x986d: 'iiiibaav',
    0x986e: 'yyyyyyba98',
    0x986f: 'jjjjabbw',
    0x9870: 'hhhhhhaa89',
    0x9871: 'kkkkabbw',
    0x9872: 'hhhhhb017',
    0x9873: 'hhhhbaav',
    0x9874: 'mmmmma017',
    0x9875: 'mmmmaaav',
    0x9876: 'mmmmmmba89',
    0x9877: 'nnnnbbbw',
    0x9878: 'ddddddaa99',
    0x9879: 'ggggabav',
    0x987a: 'dddddb007',
    0x987b: 'ddddbabw',
    0x987c: 'aaaaaa007',
    0x987d: 'aaaaaabw',
    0x987e: 'aaaaaaba99',
    0x987f: 'bbbbbbav',
    0x9880: 'ccccca107',
    0x9881: 'ccccabbw',
    0x9882: 'ccccccba99',
    0x9883: 'ppppaabv',
    0x9884: 'eeeeea007',
    0x9885: 'eeeeaaav',
    0x9886: 'eeeeeebb89',
    0x9887: 'ffffbbbw',
    0x9888: 'llllllab99',
    0x9889: 'ooooabav',
    0x988a: 'lllllb017',
    0x988b: 'llllbabw',
    0x988c: 'iiiiia017',
    0x988d: 'iiiiaabw',
    0x988e: 'iiiiiibb99',
    0x988f: 'jjjjbbav',
    0x9890: 'xxxxxxab88',
    0x9891: 'kkkkbbav',
    0x9892: 'xxxxxb006',
    0x9893: 'hhhhaabw',
    0x9894: '8888888abal',
    0x9895: 'mmmmbabw',
    0x9896: '8888888abaL',
    0x9897: 'nnnnabav',
    0x9898: 'ttttttab98',
    0x9899: 'ggggbbbw',
    0x989a: 'tttttb016',
    0x989b: 'ddddaaav',
    0x989c: 'qqqqqa016',
    0x989d: 'aaaabaav',
    0x989e: 'qqqqqqbb98',
    0x989f: 'bbbbabbw',
    0x98a0: 'sssssb116',
    0x98a1: 'ccccbaav',
    0x98a2: 'pppppa016',
    0x98a3: 'ppppbbaw',
    0x98a4: 'uuuuub016',
    0x98a5: 'eeeebbbw',
    0x98a6: 'vvvvva116',
    0x98a7: 'ffffaaav',
    0x98a8: '?????b011',
    0x98a9: 'oooobabw',
    0x98aa: '<<<<<a111',
    0x98ab: 'llllabav',
    0x98ac: 'yyyyyb006',
    0x98ad: 'iiiibbav',
    0x98ae: 'zzzzza106',
    0x98af: 'jjjjaabw',
    0x98b0: 'kkkkkb117',
    0x98b1: 'kkkkaabw',
    0x98b2: 'hhhhha017',
    0x98b3: 'hhhhbbav',
    0x98b4: 'mmmmmb017',
    0x98b5: 'mmmmabav',
    0x98b6: 'nnnnna117',
    0x98b7: 'nnnnbabw',
    0x98b8: 'gggggb107',
    0x98b9: 'ggggaaav',
    0x98ba: 'ddddda007',
    0x98bb: 'ddddbbbw',
    0x98bc: 'aaaaab007',
    0x98bd: 'aaaaabbw',
    0x98be: 'bbbbba107',
    0x98bf: 'bbbbbaav',
    0x98c0: 'cccccb007',
    0x98c1: 'ccccaaav',
    0x98c2: 'CCCCCCai88',
    0x98c3: 'ppppabaw',
    0x98c4: 'eeeeeb107',
    0x98c5: 'eeeeabbw',
    0x98c6: 'fffffa007',
    0x98c7: 'ffffbaav',
    0x98c8: 'ooooob017',
    0x98c9: 'ooooaabw',
    0x98ca: 'llllla117',
    0x98cb: 'llllbbav',
    0x98cc: 'iiiiib117',
    0x98cd: 'iiiiabav',
    0x98ce: 'jjjjja017',
    0x98cf: 'jjjjbabw',
    0x98d0: 'XXXXXXbh89',
    0x98d1: 'kkkkbabw',
    0x98d2: 'xxxxxa106',
    0x98d3: 'hhhhabav',
    0x98d4: 'XXXXXXXabah',
    0x98d5: 'mmmmbbav',
    0x98d6: 'XXXXXXXabaH',
    0x98d7: 'nnnnaabw',
    0x98d8: 'wwwwwb016',
    0x98d9: 'ggggbaav',
    0x98da: 'ttttta116',
    0x98db: 'ddddabbw',
    0x98dc: 'qqqqqb116',
    0x98dd: 'aaaabbbw',
    0x98de: 'rrrrra016',
    0x98df: 'bbbbaaav',
    0x98e0: 'ppppppbb98',
    0x98e1: 'ccccbbbw',
    0x98e2: 'pppppb116',
    0x98e3: 'ppppbabv',
    0x98e4: 'pppppppabak',
    0x98e5: 'eeeebaav',
    0x98e6: 'pppppppabaK',
    0x98e7: 'ffffabbw',
    0x98e8: 'yyyyyyybbbJ',
    0x98e9: 'oooobbav',
    0x98ea: 'yyyyyyybbbj',
    0x98eb: 'llllaabw',
    0x98ec: 'yyyyya106',
    0x98ed: 'iiiibabw',
    0x98ee: 'yyyyyyab88',
    0x98ef: 'jjjjabav',
    0x98f0: 'hhhhhhbb99',
    0x98f1: 'kkkkabav',
    0x98f2: 'hhhhhb117',
    0x98f3: 'hhhhbabw',
    0x98f4: 'hhhhhhhabaj',
    0x98f5: 'hhhhhhhabaz',
    0x98f6: 'hhhhhhhabaJ',
    0x98f7: 'hhhhhhhabaZ',
    0x98f8: 'aaaaaaabbbK',
    0x98f9: 'aaaaaaabbdX',
    0x98fa: 'aaaaaaabbbk',
    0x98fb: 'aaaaaaabbdx',
    0x98fc: 'aaaaaa107',
    0x98fd: 'aaaaaaav',
    0x98fe: 'aaaaaaab89',
    0x98ff: 'bbbbbbbw',
    0x9900: 'ccccca017',
    0x9901: 'ccccabaw',
    0x9902: 'ccccccaa99',
    0x9903: 'eeeeeeebaaz',
    0x9904: 'cccccccbbbK',
    0x9905: 'cccccccbbdX',
    0x9906: 'cccccccbbbk',
    0x9907: 'cccccccbbdx',
    0x9908: 'iiiiiiibabK',
    0x9909: 'iiiiiiibadX',
    0x990a: 'iiiiiiibabk',
    0x990b: 'iiiiiiibadx',
    0x990c: 'iiiiia107',
    0x990d: 'iiiiaaaw',
    0x990e: 'iiiiiiab99',
    0x990f: 'jjjjbbbv',
    0x9910: 'xxxxxxbb88',
    0x9911: 'kkkkbbbv',
    0x9912: 'xxxxxb116',
    0x9913: 'hhhhaaaw',
    0x9914: 'xxxxxxxaaak',
    0x9915: 'mmmmbaaw',
    0x9916: 'xxxxxxxaaaK',
    0x9917: 'nnnnabbv',
    0x9918: 'qqqqqqqbabJ',
    0x9919: 'ggggbbaw',
    0x991a: 'qqqqqqqbabj',
    0x991b: 'ddddaabv',
    0x991c: 'qqqqqa106',
    0x991d: 'aaaababv',
    0x991e: 'qqqqqqab98',
    0x991f: 'bbbbabaw',
    0x9920: 'sssssb006',
    0x9921: 'ccccbabv',
    0x9922: 'pppppa106',
    0x9923: 'ppppbbbw',
    0x9924: 'uuuuub106',
    0x9925: 'eeeebbaw',
    0x9926: 'vvvvva006',
    0x9927: 'ffffaabv',
    0x9928: 'YYYYYYYbabI',
    0x9929: 'oooobaaw',
    0x992a: 'YYYYYYYbabi',
    0x992b: 'llllabbv',
    0x992c: 'yyyyyb116',
    0x992d: 'iiiibbbv',
    0x992e: 'zzzzza016',
    0x992f: 'jjjjaaaw',
    0x9930: 'kkkkkb007',
    0x9931: 'kkkkaaaw',
    0x9932: 'hhhhha107',
    0x9933: 'hhhhbbbv',
    0x9934: 'mmmmmb107',
    0x9935: 'mmmmabbv',
    0x9936: 'nnnnna007',
    0x9937: 'nnnnbaaw',
    0x9938: 'gggggb017',
    0x9939: 'ggggaabv',
    0x993a: 'ddddda117',
    0x993b: 'ddddbbaw',
    0x993c: 'aaaaab117',
    0x993d: 'aaaaabaw',
    0x993e: 'bbbbba017',
    0x993f: 'bbbbbabv',
    0x9940: 'cccccb117',
    0x9941: 'ccccaabv',
    0x9942: 'CCCCCCbi88',
    0x9943: 'ppppabbw',
    0x9944: 'eeeeeb017',
    0x9945: 'eeeeabaw',
    0x9946: 'fffffa117',
    0x9947: 'ffffbabv',
    0x9948: 'ooooob107',
    0x9949: 'ooooaaaw',
    0x994a: 'llllla007',
    0x994b: 'llllbbbv',
    0x994c: 'iiiiib007',
    0x994d: 'iiiiabbv',
    0x994e: 'jjjjja107',
    0x994f: 'jjjjbaaw',
    0x9950: 'XXXXXXah89',
    0x9951: 'kkkkbaaw',
    0x9952: 'xxxxxa016',
    0x9953: 'hhhhabbv',
    0x9954: '=====b101',
    0x9955: 'mmmmbbbv',
    0x9956: '>>>>>a001',
    0x9957: 'nnnnaaaw',
    0x9958: 'wwwwwb106',
    0x9959: 'ggggbabv',
    0x995a: 'ttttta006',
    0x995b: 'ddddabaw',
    0x995c: 'qqqqqb006',
    0x995d: 'aaaabbaw',
    0x995e: 'rrrrra106',
    0x995f: 'bbbbaabv',
    0x9960: 'ppppppab98',
    0x9961: 'ccccbbaw',
    0x9962: 'pppppb006',
    0x9963: 'ppppbaav',
    0x9964: 'uuuuua006',
    0x9965: 'eeeebabv',
    0x9966: 'uuuuuubb98',
    0x9967: 'ffffabaw',
    0x9968: '9999999babM',
    0x9969: 'oooobbbv',
    0x996a: '9999999babm',
    0x996b: 'llllaaaw',
    0x996c: 'yyyyya016',
    0x996d: 'iiiibaaw',
    0x996e: 'yyyyyybb88',
    0x996f: 'jjjjabbv',
    0x9970: 'hhhhhhab99',
    0x9971: 'kkkkabbv',
    0x9972: 'hhhhhb007',
    0x9973: 'hhhhbaaw',
    0x9974: 'mmmmma007',
    0x9975: 'mmmmaaaw',
    0x9976: 'mmmmmmbb99',
    0x9977: 'nnnnbbbv',
    0x9978: 'ddddddab89',
    0x9979: 'ggggabaw',
    0x997a: 'dddddb017',
    0x997b: 'ddddbabv',
    0x997c: 'aaaaaa017',
    0x997d: 'aaaaaabv',
    0x997e: 'aaaaaabb89',
    0x997f: 'bbbbbbaw',
    0x9980: 'ccccca117',
    0x9981: 'ccccabbv',
    0x9982: 'ccccccbb89',
    0x9983: 'ppppaabw',
    0x9984: 'eeeeea017',
    0x9985: 'eeeeaaaw',
    0x9986: 'eeeeeeba99',
    0x9987: 'ffffbbbv',
    0x9988: 'llllllaa89',
    0x9989: 'ooooabaw',
    0x998a: 'lllllb007',
    0x998b: 'llllbabv',
    0x998c: 'iiiiia007',
    0x998d: 'iiiiaabv',
    0x998e: 'iiiiiiba89',
    0x998f: 'jjjjbbaw',
    0x9990: 'xxxxxxaa98',
    0x9991: 'kkkkbbaw',
    0x9992: 'xxxxxb016',
    0x9993: 'hhhhaabv',
    0x9994: '11111111aabJ',
    0x9995: 'mmmmbabv',
    0x9996: '11111111aabZ',
    0x9997: 'nnnnabaw',
    0x9998: 'ttttttaa88',
    0x9999: 'ggggbbbv',
    0x999a: 'tttttb006',
    0x999b: 'ddddaaaw',
    0x999c: 'qqqqqa006',
    0x999d: 'aaaabaaw',
    0x999e: 'qqqqqqba88',
    0x999f: 'bbbbabbv',
    0x99a0: 'sssssb106',
    0x99a1: 'ccccbaaw',
    0x99a2: 'pppppa006',
    0x99a3: 'ppppbbav',
    0x99a4: 'uuuuub006',
    0x99a5: 'eeeebbbv',
    0x99a6: 'vvvvva106',
    0x99a7: 'ffffaaaw',
    0x99a8: '1111111111bbaA',
    0x99a9: 'oooobabv',
    0x99aa: '1111111111bbaE',
    0x99ab: 'llllabaw',
    0x99ac: 'yyyyyb016',
    0x99ad: 'iiiibbaw',
    0x99ae: 'zzzzza116',
    0x99af: 'jjjjaabv',
    0x99b0: 'kkkkkb107',
    0x99b1: 'kkkkaabv',
    0x99b2: 'hhhhha007',
    0x99b3: 'hhhhbbaw',
    0x99b4: 'mmmmmb007',
    0x99b5: 'mmmmabaw',
    0x99b6: 'nnnnna107',
    0x99b7: 'nnnnbabv',
    0x99b8: 'gggggb117',
    0x99b9: 'ggggaaaw',
    0x99ba: 'ddddda017',
    0x99bb: 'ddddbbbv',
    0x99bc: 'aaaaab017',
    0x99bd: 'aaaaabbv',
    0x99be: 'bbbbba117',
    0x99bf: 'bbbbbaaw',
    0x99c0: 'aaaaaaaaaabbaW',
    0x99c1: 'aaaaaaaaaabbaU',
    0x99c2: 'aaaaaaaaaabbaS',
    0x99c3: 'aaaaaaaaaabbaQ',
    0x99c4: 'aaaaaaaaaabbkZ',
    0x99c5: 'aaaaaaaaaabbiY',
    0x99c6: 'aaaaaaaaaabbcZ',
    0x99c7: 'aaaaaaaaaabbaY',
    0x99c8: 'aaaaaaaaaabbaG',
    0x99c9: 'aaaaaaaaaabbaE',
    0x99ca: 'aaaaaaaaaabbaC',
    0x99cb: 'aaaaaaaaaabbaA',
    0x99cc: 'aaaaaaaaaabbaO',
    0x99cd: 'aaaaaaaaaabbaM',
    0x99ce: 'aaaaaaaaaabbaK',
    0x99cf: 'aaaaaaaaaabbaI',
    0x99d0: 'aaaaaaaaaabbaw',
    0x99d1: 'aaaaaaaaaabbau',
    0x99d2: 'aaaaaaaaaabbas',
    0x99d3: 'aaaaaaaaaabbaq',
    0x99d4: 'aaaaaaaaaabbkz',
    0x99d5: 'aaaaaaaaaabbiy',
    0x99d6: 'aaaaaaaaaabbcz',
    0x99d7: 'aaaaaaaaaabbay',
    0x99d8: 'aaaaaaaaaabbag',
    0x99d9: 'aaaaaaaaaabbae',
    0x99da: 'aaaaaaaaaabbac',
    0x99db: 'aaaaaaaaaabbaa',
    0x99dc: 'aaaaaaaaaabbao',
    0x99dd: 'aaaabbbv',
    0x99de: 'aaaaaaaaaabbak',
    0x99df: 'aaaaaaaaaabbai',
    0x99e0: 'aaaaaaaaaabbC',
    0x99e1: 'aaaaaaaaaabbG',
    0x99e2: 'aaaaaaaaaabbK',
    0x99e3: 'aaaaaaaaaabbO',
    0x99e4: 'aaaaaaaaaabbS',
    0x99e5: 'aaaaaaaaaabbW',
    0x99e6: 'aaaaaaaaaabdX',
    0x99e7: 'aaaaaaaaaabhZ',
    0x99e8: 'aaaaaaaaaabbc',
    0x99e9: 'aaaaaaaaaabbg',
    0x99ea: 'aaaaaaaaaabbk',
    0x99eb: 'aaaaaaaaaabbo',
    0x99ec: 'aaaaaaaaaabbs',
    0x99ed: 'aaaaaaaaaabbw',
    0x99ee: 'aaaaaaaaaabdx',
    0x99ef: 'aaaaaaaaaabhz',
    0x99f0: 'aaaaaaaaaabl',
    0x99f1: 'aaaaaaaaaabd',
    0x99f2: 'aaaaaaaaaahy',
    0x99f3: 'aaaaaaaaaabt',
    0x99f4: 'aaaaaaaaaabL',
    0x99f5: 'aaaaaaaaaabD',
    0x99f6: 'aaaaaaaaaahY',
    0x99f7: 'aaaaaaaaaabT',
    0x99f8: 'aaaaaaaaaaJ',
    0x99f9: 'aaaaaaaaaaZ',
    0x99fa: 'aaaaaaaaaaj',
    0x99fb: 'aaaaaaaaaaz',
    0x99fc: 'aaaaaa117',
    0x99fd: 'aaaaaaaw',
    0x99fe: 'aaaaaaaa99',
    0x99ff: 'aaaaaaaaa5',
    0x9a00: 'ccccca006',
    0x9a01: 'ccccabat',
    0x9a02: 'ccccccab99',
    0x9a03: 'cccccccca5',
    0x9a04: 'cccccccabbK',
    0x9a05: 'cccccccabdX',
    0x9a06: 'cccccccabbk',
    0x9a07: 'cccccccabdx',
    0x9a08: 'ccccccccaabL',
    0x9a09: 'ccccccccaabD',
    0x9a0a: 'ccccccccaahY',
    0x9a0b: 'ccccccccaabT',
    0x9a0c: 'ccccccccaabl',
    0x9a0d: 'ccccccccaabd',
    0x9a0e: 'ccccccccaahy',
    0x9a0f: 'ccccccccaabt',
    0x9a10: 'cccccccccbaar',
    0x9a11: 'cccccccccbaav',
    0x9a12: 'cccccccccbaaz',
    0x9a13: 'cccccccccbaiz',
    0x9a14: 'cccccccccbaab',
    0x9a15: 'cccccccccbaaf',
    0x9a16: 'cccccccccbaaj',
    0x9a17: 'cccccccccbaan',
    0x9a18: 'cccccccccbaaR',
    0x9a19: 'cccccccccbaaV',
    0x9a1a: 'cccccccccbaaZ',
    0x9a1b: 'cccccccccbaiZ',
    0x9a1c: 'cccccccccbaaB',
    0x9a1d: 'aaaababu',
    0x9a1e: 'cccccccccbaaJ',
    0x9a1f: 'bbbbabat',
    0x9a20: 'sssssb017',
    0x9a21: 'ccccbabu',
    0x9a22: 'pppppa117',
    0x9a23: 'ppppbbbt',
    0x9a24: 'uuuuub117',
    0x9a25: 'eeeebbat',
    0x9a26: 'vvvvva017',
    0x9a27: 'ffffaabu',
    0x9a28: 'AAAAAAAAAbbaa',
    0x9a29: 'oooobaat',
    0x9a2a: 'AAAAAAAAAbbai',
    0x9a2b: 'llllabbu',
    0x9a2c: 'yyyyyb107',
    0x9a2d: 'iiiibbbu',
    0x9a2e: 'zzzzza007',
    0x9a2f: 'jjjjaaat',
    0x9a30: 'kkkkkb016',
    0x9a31: 'kkkkaaat',
    0x9a32: 'hhhhha116',
    0x9a33: 'hhhhbbbu',
    0x9a34: 'mmmmmb116',
    0x9a35: 'mmmmabbu',
    0x9a36: 'nnnnna016',
    0x9a37: 'nnnnbaat',
    0x9a38: 'gggggb006',
    0x9a39: 'ggggaabu',
    0x9a3a: 'ddddda106',
    0x9a3b: 'ddddbbat',
    0x9a3c: 'aaaaab106',
    0x9a3d: 'aaaaabat',
    0x9a3e: 'bbbbba006',
    0x9a3f: 'bbbbbabu',
    0x9a40: 'cccccb106',
    0x9a41: 'ccccaabu',
    0x9a42: 'CCCCCCbh98',
    0x9a43: 'ppppabbt',
    0x9a44: 'eeeeeb006',
    0x9a45: 'eeeeabat',
    0x9a46: 'fffffa106',
    0x9a47: 'ffffbabu',
    0x9a48: 'ooooob116',
    0x9a49: 'ooooaaat',
    0x9a4a: 'llllla016',
    0x9a4b: 'llllbbbu',
    0x9a4c: 'iiiiib016',
    0x9a4d: 'iiiiabbu',
    0x9a4e: 'jjjjja116',
    0x9a4f: 'jjjjbaat',
    0x9a50: 'XXXXXXai99',
    0x9a51: 'kkkkbaat',
    0x9a52: 'xxxxxa007',
    0x9a53: 'hhhhabbu',
    0x9a54: '=====b110',
    0x9a55: 'mmmmbbbu',
    0x9a56: '>>>>>a010',
    0x9a57: 'nnnnaaat',
    0x9a58: 'wwwwwb117',
    0x9a59: 'ggggbabu',
    0x9a5a: 'ttttta017',
    0x9a5b: 'ddddabat',
    0x9a5c: 'qqqqqb017',
    0x9a5d: 'aaaabbat',
    0x9a5e: 'rrrrra117',
    0x9a5f: 'bbbbaabu',
    0x9a60: 'ppppppaa98',
    0x9a61: 'ccccbbat',
    0x9a62: 'pppppb017',
    0x9a63: 'ppppbaau',
    0x9a64: 'uuuuua017',
    0x9a65: 'eeeebabu',
    0x9a66: 'uuuuuuba98',
    0x9a67: 'ffffabat',
    0x9a68: '00000000bbcZ',
    0x9a69: 'oooobbbu',
    0x9a6a: '00000000bbaK',
    0x9a6b: 'llllaaat',
    0x9a6c: 'yyyyya007',
    0x9a6d: 'iiiibaat',
    0x9a6e: 'yyyyyyba88',
    0x9a6f: 'jjjjabbu',
    0x9a70: 'hhhhhhaa99',
    0x9a71: 'kkkkabbu',
    0x9a72: 'hhhhhb016',
    0x9a73: 'hhhhbaat',
    0x9a74: 'mmmmma016',
    0x9a75: 'mmmmaaat',
    0x9a76: 'mmmmmmba99',
    0x9a77: 'nnnnbbbu',
    0x9a78: 'ddddddaa89',
    0x9a79: 'ggggabat',
    0x9a7a: 'dddddb006',
    0x9a7b: 'ddddbabu',
    0x9a7c: 'aaaaaa006',
    0x9a7d: 'aaaaaabu',
    0x9a7e: 'aaaaaaba89',
    0x9a7f: 'bbbbbbat',
    0x9a80: 'ccccca106',
    0x9a81: 'ccccabbu',
    0x9a82: 'ccccccba89',
    0x9a83: 'ppppaabt',
    0x9a84: 'eeeeea006',
    0x9a85: 'eeeeaaat',
    0x9a86: 'eeeeeebb99',
    0x9a87: 'ffffbbbu',
    0x9a88: 'llllllab89',
    0x9a89: 'ooooabat',
    0x9a8a: 'lllllb016',
    0x9a8b: 'llllbabu',
    0x9a8c: 'iiiiia016',
    0x9a8d: 'iiiiaabu',
    0x9a8e: 'iiiiiibb89',
    0x9a8f: 'jjjjbbat',
    0x9a90: 'xxxxxxab98',
    0x9a91: 'kkkkbbat',
    0x9a92: 'xxxxxb007',
    0x9a93: 'hhhhaabu',
    0x9a94: '8888888aabm',
    0x9a95: 'mmmmbabu',
    0x9a96: '8888888aabM',
    0x9a97: 'nnnnabat',
    0x9a98: 'ttttttab88',
    0x9a99: 'ggggbbbu',
    0x9a9a: 'tttttb017',
    0x9a9b: 'ddddaaat',
    0x9a9c: 'qqqqqa017',
    0x9a9d: 'aaaabaat',
    0x9a9e: 'qqqqqqbb88',
    0x9a9f: 'bbbbabbu',
    0x9aa0: 'sssssb117',
    0x9aa1: 'ccccbaat',
    0x9aa2: 'pppppa017',
    0x9aa3: 'ppppbbau',
    0x9aa4: 'uuuuub017',
    0x9aa5: 'eeeebbbu',
    0x9aa6: 'vvvvva117',
    0x9aa7: 'ffffaaat',
    0x9aa8: '?????b010',
    0x9aa9: 'oooobabu',
    0x9aaa: '<<<<<a110',
    0x9aab: 'llllabat',
    0x9aac: 'yyyyyb007',
    0x9aad: 'iiiibbat',
    0x9aae: 'zzzzza107',
    0x9aaf: 'jjjjaabu',
    0x9ab0: 'kkkkkb116',
    0x9ab1: 'kkkkaabu',
    0x9ab2: 'hhhhha016',
    0x9ab3: 'hhhhbbat',
    0x9ab4: 'mmmmmb016',
    0x9ab5: 'mmmmabat',
    0x9ab6: 'nnnnna116',
    0x9ab7: 'nnnnbabu',
    0x9ab8: 'gggggb106',
    0x9ab9: 'ggggaaat',
    0x9aba: 'ddddda006',
    0x9abb: 'ddddbbbu',
    0x9abc: 'aaaaab006',
    0x9abd: 'aaaaabbu',
    0x9abe: 'bbbbba106',
    0x9abf: 'bbbbbaat',
    0x9ac0: 'cccccb006',
    0x9ac1: 'ccccaaat',
    0x9ac2: 'CCCCCCai98',
    0x9ac3: 'ppppabau',
    0x9ac4: 'eeeeeb106',
    0x9ac5: 'eeeeabbu',
    0x9ac6: 'fffffa006',
    0x9ac7: 'ffffbaat',
    0x9ac8: 'ooooob016',
    0x9ac9: 'ooooaabu',
    0x9aca: 'llllla116',
    0x9acb: 'llllbbat',
    0x9acc: 'iiiiib116',
    0x9acd: 'iiiiabat',
    0x9ace: 'jjjjja016',
    0x9acf: 'jjjjbabu',
    0x9ad0: 'XXXXXXbh99',
    0x9ad1: 'kkkkbabu',
    0x9ad2: 'xxxxxa107',
    0x9ad3: 'hhhhabat',
    0x9ad4: 'XXXXXXXaabi',
    0x9ad5: 'mmmmbbat',
    0x9ad6: 'XXXXXXXaabI',
    0x9ad7: 'nnnnaabu',
    0x9ad8: 'wwwwwb017',
    0x9ad9: 'ggggbaat',
    0x9ada: 'ttttta117',
    0x9adb: 'ddddabbu',
    0x9adc: 'qqqqqb117',
    0x9add: 'aaaabbbu',
    0x9ade: 'rrrrra017',
    0x9adf: 'bbbbaaat',
    0x9ae0: 'ppppppbb88',
    0x9ae1: 'ccccbbbu',
    0x9ae2: 'pppppb117',
    0x9ae3: 'ppppbabt',
    0x9ae4: 'pppppppaabj',
    0x9ae5: 'eeeebaat',
    0x9ae6: 'pppppppaabJ',
    0x9ae7: 'ffffabbu',
    0x9ae8: 'yyyyyyybaaK',
    0x9ae9: 'oooobbat',
    0x9aea: 'yyyyyyybaak',
    0x9aeb: 'llllaabu',
    0x9aec: 'yyyyya107',
    0x9aed: 'iiiibabu',
    0x9aee: 'yyyyyyab98',
    0x9aef: 'jjjjabat',
    0x9af0: 'hhhhhhbb89',
    0x9af1: 'kkkkabat',
    0x9af2: 'hhhhhb116',
    0x9af3: 'hhhhbabu',
    0x9af4: 'hhhhhhhaabk',
    0x9af5: 'hhhhhhhaadx',
    0x9af6: 'hhhhhhhaabK',
    0x9af7: 'hhhhhhhaadX',
    0x9af8: 'aaaaaaabaaJ',
    0x9af9: 'aaaaaaabaaZ',
    0x9afa: 'aaaaaaabaaj',
    0x9afb: 'aaaaaaabaaz',
    0x9afc: 'aaaaaa106',
    0x9afd: 'aaaaaaat',
    0x9afe: 'aaaaaaab99',
    0x9aff: 'bbbbbbbu',
    0x9b00: 'ccccca016',
    0x9b01: 'ccccabau',
    0x9b02: 'ccccccaa89',
    0x9b03: 'eeeeeeebbdx',
    0x9b04: 'cccccccbaaJ',
    0x9b05: 'cccccccbaaZ',
    0x9b06: 'cccccccbaaj',
    0x9b07: 'cccccccbaaz',
    0x9b08: 'iiiiiiibbaJ',
    0x9b09: 'iiiiiiibbaZ',
    0x9b0a: 'iiiiiiibbaj',
    0x9b0b: 'iiiiiiibbaz',
    0x9b0c: 'iiiiia106',
    0x9b0d: 'iiiiaaau',
    0x9b0e: 'iiiiiiab89',
    0x9b0f: 'jjjjbbbt',
    0x9b10: 'xxxxxxbb98',
    0x9b11: 'kkkkbbbt',
    0x9b12: 'xxxxxb117',
    0x9b13: 'hhhhaaau',
    0x9b14: 'xxxxxxxabbj',
    0x9b15: 'mmmmbaau',
    0x9b16: 'xxxxxxxabbJ',
    0x9b17: 'nnnnabbt',
    0x9b18: 'qqqqqqqbbaK',
    0x9b19: 'ggggbbau',
    0x9b1a: 'qqqqqqqbbak',
    0x9b1b: 'ddddaabt',
    0x9b1c: 'qqqqqa107',
    0x9b1d: 'aaaababt',
    0x9b1e: 'qqqqqqab88',
    0x9b1f: 'bbbbabau',
    0x9b20: 'sssssb007',
    0x9b21: 'ccccbabt',
    0x9b22: 'pppppa107',
    0x9b23: 'ppppbbbu',
    0x9b24: 'uuuuub107',
    0x9b25: 'eeeebbau',
    0x9b26: 'vvvvva007',
    0x9b27: 'ffffaabt',
    0x9b28: 'YYYYYYYbbaH',
    0x9b29: 'oooobaau',
    0x9b2a: 'YYYYYYYbbah',
    0x9b2b: 'llllabbt',
    0x9b2c: 'yyyyyb117',
    0x9b2d: 'iiiibbbt',
    0x9b2e: 'zzzzza017',
    0x9b2f: 'jjjjaaau',
    0x9b30: 'kkkkkb006',
    0x9b31: 'kkkkaaau',
    0x9b32: 'hhhhha106',
    0x9b33: 'hhhhbbbt',
    0x9b34: 'mmmmmb106',
    0x9b35: 'mmmmabbt',
    0x9b36: 'nnnnna006',
    0x9b37: 'nnnnbaau',
    0x9b38: 'gggggb016',
    0x9b39: 'ggggaabt',
    0x9b3a: 'ddddda116',
    0x9b3b: 'ddddbbau',
    0x9b3c: 'aaaaab116',
    0x9b3d: 'aaaaabau',
    0x9b3e: 'bbbbba016',
    0x9b3f: 'bbbbbabt',
    0x9b40: 'cccccb116',
    0x9b41: 'ccccaabt',
    0x9b42: 'CCCCCCbi98',
    0x9b43: 'ppppabbu',
    0x9b44: 'eeeeeb016',
    0x9b45: 'eeeeabau',
    0x9b46: 'fffffa116',
    0x9b47: 'ffffbabt',
    0x9b48: 'ooooob106',
    0x9b49: 'ooooaaau',
    0x9b4a: 'llllla006',
    0x9b4b: 'llllbbbt',
    0x9b4c: 'iiiiib006',
    0x9b4d: 'iiiiabbt',
    0x9b4e: 'jjjjja106',
    0x9b4f: 'jjjjbaau',
    0x9b50: 'XXXXXXah99',
    0x9b51: 'kkkkbaau',
    0x9b52: 'xxxxxa017',
    0x9b53: 'hhhhabbt',
    0x9b54: '=====b100',
    0x9b55: 'mmmmbbbt',
    0x9b56: '>>>>>a000',
    0x9b57: 'nnnnaaau',
    0x9b58: 'wwwwwb107',
    0x9b59: 'ggggbabt',
    0x9b5a: 'ttttta007',
    0x9b5b: 'ddddabau',
    0x9b5c: 'qqqqqb007',
    0x9b5d: 'aaaabbau',
    0x9b5e: 'rrrrra107',
    0x9b5f: 'bbbbaabt',
    0x9b60: 'ppppppab88',
    0x9b61: 'ccccbbau',
    0x9b62: 'pppppb007',
    0x9b63: 'ppppbaat',
    0x9b64: 'uuuuua007',
    0x9b65: 'eeeebabt',
    0x9b66: 'uuuuuubb88',
    0x9b67: 'ffffabau',
    0x9b68: '9999999bbaL',
    0x9b69: 'oooobbbt',
    0x9b6a: '9999999bbal',
    0x9b6b: 'llllaaau',
    0x9b6c: 'yyyyya017',
    0x9b6d: 'iiiibaau',
    0x9b6e: 'yyyyyybb98',
    0x9b6f: 'jjjjabbt',
    0x9b70: 'hhhhhhab89',
    0x9b71: 'kkkkabbt',
    0x9b72: 'hhhhhb006',
    0x9b73: 'hhhhbaau',
    0x9b74: 'mmmmma006',
    0x9b75: 'mmmmaaau',
    0x9b76: 'mmmmmmbb89',
    0x9b77: 'nnnnbbbt',
    0x9b78: 'ddddddab99',
    0x9b79: 'ggggabau',
    0x9b7a: 'dddddb016',
    0x9b7b: 'ddddbabt',
    0x9b7c: 'aaaaaa016',
    0x9b7d: 'aaaaaabt',
    0x9b7e: 'aaaaaabb99',
    0x9b7f: 'bbbbbbau',
    0x9b80: 'ccccca116',
    0x9b81: 'ccccabbt',
    0x9b82: 'ccccccbb99',
    0x9b83: 'ppppaabu',
    0x9b84: 'eeeeea016',
    0x9b85: 'eeeeaaau',
    0x9b86: 'eeeeeeba89',
    0x9b87: 'ffffbbbt',
    0x9b88: 'llllllaa99',
    0x9b89: 'ooooabau',
    0x9b8a: 'lllllb006',
    0x9b8b: 'llllbabt',
    0x9b8c: 'iiiiia006',
    0x9b8d: 'iiiiaabt',
    0x9b8e: 'iiiiiiba99',
    0x9b8f: 'jjjjbbau',
    0x9b90: 'xxxxxxaa88',
    0x9b91: 'kkkkbbau',
    0x9b92: 'xxxxxb017',
    0x9b93: 'hhhhaabt',
    0x9b94: '11111111bbaK',
    0x9b95: 'mmmmbabt',
    0x9b96: '11111111bbcZ',
    0x9b97: 'nnnnabau',
    0x9b98: 'ttttttaa98',
    0x9b99: 'ggggbbbt',
    0x9b9a: 'tttttb007',
    0x9b9b: 'ddddaaau',
    0x9b9c: 'qqqqqa007',
    0x9b9d: 'aaaabaau',
    0x9b9e: 'qqqqqqba98',
    0x9b9f: 'bbbbabbt',
    0x9ba0: 'sssssb107',
    0x9ba1: 'ccccbaau',
    0x9ba2: 'pppppa007',
    0x9ba3: 'ppppbbat',
    0x9ba4: 'uuuuub007',
    0x9ba5: 'eeeebbbt',
    0x9ba6: 'vvvvva107',
    0x9ba7: 'ffffaaau',
    0x9ba8: '?????b000',
    0x9ba9: 'oooobabt',
    0x9baa: '<<<<<a100',
    0x9bab: 'llllabau',
    0x9bac: 'yyyyyb017',
    0x9bad: 'iiiibbau',
    0x9bae: 'zzzzza117',
    0x9baf: 'jjjjaabt',
    0x9bb0: 'kkkkkb106',
    0x9bb1: 'kkkkaabt',
    0x9bb2: 'hhhhha006',
    0x9bb3: 'hhhhbbau',
    0x9bb4: 'mmmmmb006',
    0x9bb5: 'mmmmabau',
    0x9bb6: 'nnnnna106',
    0x9bb7: 'nnnnbabt',
    0x9bb8: 'gggggb116',
    0x9bb9: 'ggggaaau',
    0x9bba: 'ddddda016',
    0x9bbb: 'ddddbbbt',
    0x9bbc: 'aaaaab016',
    0x9bbd: 'aaaaabbt',
    0x9bbe: 'bbbbba116',
    0x9bbf: 'bbbbbaau',
    0x9bc0: 'cccccb016',
    0x9bc1: 'ccccaaau',
    0x9bc2: 'CCCCCCah98',
    0x9bc3: 'ppppabat',
    0x9bc4: 'eeeeeb116',
    0x9bc5: 'eeeeabbt',
    0x9bc6: 'fffffa016',
    0x9bc7: 'ffffbaau',
    0x9bc8: 'ooooob006',
    0x9bc9: 'ooooaabt',
    0x9bca: 'llllla106',
    0x9bcb: 'llllbbau',
    0x9bcc: 'iiiiib106',
    0x9bcd: 'iiiiabau',
    0x9bce: 'jjjjja006',
    0x9bcf: 'jjjjbabt',
    0x9bd0: 'QQQQQQQQbbao',
    0x9bd1: 'kkkkbabt',
    0x9bd2: 'xxxxxa117',
    0x9bd3: 'hhhhabau',
    0x9bd4: 'QQQQQQQQbbaO',
    0x9bd5: 'mmmmbbau',
    0x9bd6: 'QQQQQQQQbbkZ',
    0x9bd7: 'nnnnaabt',
    0x9bd8: 'wwwwwb007',
    0x9bd9: 'ggggbaau',
    0x9bda: 'ttttta107',
    0x9bdb: 'ddddabbt',
    0x9bdc: 'qqqqqb107',
    0x9bdd: 'aaaabbbt',
    0x9bde: 'rrrrra007',
    0x9bdf: 'bbbbaaau',
    0x9be0: 'ppppppba98',
    0x9be1: 'ccccbbbt',
    0x9be2: 'pppppb107',
    0x9be3: 'ppppbabu',
    0x9be4: 'pppppppbbak',
    0x9be5: 'eeeebaau',
    0x9be6: 'pppppppbbaK',
    0x9be7: 'ffffabbt',
    0x9be8: 'ppppppppabhX',
    0x9be9: 'oooobbau',
    0x9bea: 'ppppppppabbM',
    0x9beb: 'llllaabt',
    0x9bec: 'ppppppppabhx',
    0x9bed: 'iiiibabt',
    0x9bee: 'ppppppppabbm',
    0x9bef: 'jjjjabau',
    0x9bf0: 'aaaaaaaabbam',
    0x9bf1: 'aaaaaaaabbae',
    0x9bf2: 'aaaaaaaabbiy',
    0x9bf3: 'aaaaaaaabbau',
    0x9bf4: 'aaaaaaaabbaM',
    0x9bf5: 'aaaaaaaabbaE',
    0x9bf6: 'aaaaaaaabbiY',
    0x9bf7: 'aaaaaaaabbaU',
    0x9bf8: 'aaaaaaaabbK',
    0x9bf9: 'aaaaaaaabdX',
    0x9bfa: 'aaaaaaaabbk',
    0x9bfb: 'aaaaaaaabdx',
    0x9bfc: 'aaaaaa116',
    0x9bfd: 'aaaaaaau',
    0x9bfe: 'aaaaaaaa89',
    0x9bff: 'aaaaaaaab4',
    0x9c00: 'ccccca005',
    0x9c01: 'ccccabar',
    0x9c02: 'ccccccab88',
    0x9c03: 'ccccccccb5',
    0x9c04: 'cccccccaabK',
    0x9c05: 'cccccccaadX',
    0x9c06: 'cccccccaabk',
    0x9c07: 'cccccccaadx',
    0x9c08: 'ccccccccbabL',
    0x9c09: 'ccccccccbabD',
    0x9c0a: 'ccccccccbahY',
    0x9c0b: 'ccccccccbabT',
    0x9c0c: 'ccccccccbabl',
    0x9c0d: 'ccccccccbabd',
    0x9c0e: 'ccccccccbahy',
    0x9c0f: 'ccccccccbabt',
    0x9c10: 'qqqqqqqqbbbm',
    0x9c11: 'kkkkbbbs',
    0x9c12: 'qqqqqqqqbbhx',
    0x9c13: 'hhhhaaar',
    0x9c14: 'qqqqqqqqbbbM',
    0x9c15: 'mmmmbaar',
    0x9c16: 'qqqqqqqqbbhX',
    0x9c17: 'nnnnabbs',
    0x9c18: 'qqqqqqqabbJ',
    0x9c19: 'ggggbbar',
    0x9c1a: 'qqqqqqqabbj',
    0x9c1b: 'ddddaabs',
    0x9c1c: 'qqqqqa114',
    0x9c1d: 'aaaababs',
    0x9c1e: 'qqqqqqaa89',
    0x9c1f: 'bbbbabar',
    0x9c20: 'sssssb014',
    0x9c21: 'ccccbabs',
    0x9c22: 'pppppa114',
    0x9c23: 'ppppbbbr',
    0x9c24: 'uuuuub114',
    0x9c25: 'eeeebbar',
    0x9c26: 'vvvvva014',
    0x9c27: 'ffffaabs',
    0x9c28: 'PPPPPPPPabkZ',
    0x9c29: 'oooobaar',
    0x9c2a: 'PPPPPPPPabaO',
    0x9c2b: 'llllabbs',
    0x9c2c: 'yyyyyb104',
    0x9c2d: 'iiiibbbs',
    0x9c2e: 'zzzzza004',
    0x9c2f: 'jjjjaaar',
    0x9c30: 'kkkkkb015',
    0x9c31: 'kkkkaaar',
    0x9c32: 'hhhhha115',
    0x9c33: 'hhhhbbbs',
    0x9c34: 'mmmmmb115',
    0x9c35: 'mmmmabbs',
    0x9c36: 'nnnnna015',
    0x9c37: 'nnnnbaar',
    0x9c38: 'gggggb005',
    0x9c39: 'ggggaabs',
    0x9c3a: 'ddddda105',
    0x9c3b: 'ddddbbar',
    0x9c3c: 'aaaaab105',
    0x9c3d: 'aaaaabar',
    0x9c3e: 'bbbbba005',
    0x9c3f: 'bbbbbabs',
    0x9c40: 'cccccb105',
    0x9c41: 'ccccaabs',
    0x9c42: 'CCCCCCbh89',
    0x9c43: 'ppppabbr',
    0x9c44: 'eeeeeb005',
    0x9c45: 'eeeeabar',
    0x9c46: 'fffffa105',
    0x9c47: 'ffffbabs',
    0x9c48: 'ooooob115',
    0x9c49: 'ooooaaar',
    0x9c4a: 'llllla015',
    0x9c4b: 'llllbbbs',
    0x9c4c: 'iiiiib015',
    0x9c4d: 'iiiiabbs',
    0x9c4e: 'jjjjja115',
    0x9c4f: 'jjjjbaar',
    0x9c50: 'XXXXXXai88',
    0x9c51: 'kkkkbaar',
    0x9c52: 'xxxxxa004',
    0x9c53: 'hhhhabbs',
    0x9c54: '=====b113',
    0x9c55: 'mmmmbbbs',
    0x9c56: '>>>>>a013',
    0x9c57: 'nnnnaaar',
    0x9c58: 'wwwwwb114',
    0x9c59: 'ggggbabs',
    0x9c5a: 'ttttta014',
    0x9c5b: 'ddddabar',
    0x9c5c: 'qqqqqb014',
    0x9c5d: 'aaaabbar',
    0x9c5e: 'rrrrra114',
    0x9c5f: 'bbbbaabs',
    0x9c60: 'ppppppaa89',
    0x9c61: 'ccccbbar',
    0x9c62: 'pppppb014',
    0x9c63: 'ppppbaas',
    0x9c64: 'uuuuua014',
    0x9c65: 'eeeebabs',
    0x9c66: 'uuuuuuba89',
    0x9c67: 'ffffabar',
    0x9c68: '00000000abcZ',
    0x9c69: 'oooobbbs',
    0x9c6a: '00000000abaK',
    0x9c6b: 'llllaaar',
    0x9c6c: 'yyyyya004',
    0x9c6d: 'iiiibaar',
    0x9c6e: 'yyyyyyba99',
    0x9c6f: 'jjjjabbs',
    0x9c70: 'hhhhhhaa88',
    0x9c71: 'kkkkabbs',
    0x9c72: 'hhhhhb015',
    0x9c73: 'hhhhbaar',
    0x9c74: 'mmmmma015',
    0x9c75: 'mmmmaaar',
    0x9c76: 'mmmmmmba88',
    0x9c77: 'nnnnbbbs',
    0x9c78: 'ddddddaa98',
    0x9c79: 'ggggabar',
    0x9c7a: 'dddddb005',
    0x9c7b: 'ddddbabs',
    0x9c7c: 'aaaaaa005',
    0x9c7d: 'aaaaaabs',
    0x9c7e: 'aaaaaaba98',
    0x9c7f: 'bbbbbbar',
    0x9c80: 'ccccca105',
    0x9c81: 'ccccabbs',
    0x9c82: 'ccccccba98',
    0x9c83: 'ppppaabr',
    0x9c84: 'eeeeea005',
    0x9c85: 'eeeeaaar',
    0x9c86: 'eeeeeebb88',
    0x9c87: 'ffffbbbs',
    0x9c88: 'llllllab98',
    0x9c89: 'ooooabar',
    0x9c8a: 'lllllb015',
    0x9c8b: 'llllbabs',
    0x9c8c: 'iiiiia015',
    0x9c8d: 'iiiiaabs',
    0x9c8e: 'iiiiiibb98',
    0x9c8f: 'jjjjbbar',
    0x9c90: 'xxxxxxab89',
    0x9c91: 'kkkkbbar',
    0x9c92: 'xxxxxb004',
    0x9c93: 'hhhhaabs',
    0x9c94: '8888888abbm',
    0x9c95: 'mmmmbabs',
    0x9c96: '8888888abbM',
    0x9c97: 'nnnnabar',
    0x9c98: 'ttttttab99',
    0x9c99: 'ggggbbbs',
    0x9c9a: 'tttttb014',
    0x9c9b: 'ddddaaar',
    0x9c9c: 'qqqqqa014',
    0x9c9d: 'aaaabaar',
    0x9c9e: 'qqqqqqbb99',
    0x9c9f: 'bbbbabbs',
    0x9ca0: 'sssssb114',
    0x9ca1: 'ccccbaar',
    0x9ca2: 'pppppa014',
    0x9ca3: 'ppppbbas',
    0x9ca4: 'uuuuub014',
    0x9ca5: 'eeeebbbs',
    0x9ca6: 'vvvvva114',
    0x9ca7: 'ffffaaar',
    0x9ca8: '?????b013',
    0x9ca9: 'oooobabs',
    0x9caa: '<<<<<a113',
    0x9cab: 'llllabar',
    0x9cac: 'yyyyyb004',
    0x9cad: 'iiiibbar',
    0x9cae: 'zzzzza104',
    0x9caf: 'jjjjaabs',
    0x9cb0: 'kkkkkb115',
    0x9cb1: 'kkkkaabs',
    0x9cb2: 'hhhhha015',
    0x9cb3: 'hhhhbbar',
    0x9cb4: 'mmmmmb015',
    0x9cb5: 'mmmmabar',
    0x9cb6: 'nnnnna115',
    0x9cb7: 'nnnnbabs',
    0x9cb8: 'gggggb105',
    0x9cb9: 'ggggaaar',
    0x9cba: 'ddddda005',
    0x9cbb: 'ddddbbbs',
    0x9cbc: 'aaaaab005',
    0x9cbd: 'aaaaabbs',
    0x9cbe: 'bbbbba105',
    0x9cbf: 'bbbbbaar',
    0x9cc0: 'cccccb005',
    0x9cc1: 'ccccaaar',
    0x9cc2: 'CCCCCCai89',
    0x9cc3: 'ppppabas',
    0x9cc4: 'eeeeeb105',
    0x9cc5: 'eeeeabbs',
    0x9cc6: 'fffffa005',
    0x9cc7: 'ffffbaar',
    0x9cc8: 'ooooob015',
    0x9cc9: 'ooooaabs',
    0x9cca: 'llllla115',
    0x9ccb: 'llllbbar',
    0x9ccc: 'iiiiib115',
    0x9ccd: 'iiiiabar',
    0x9cce: 'jjjjja015',
    0x9ccf: 'jjjjbabs',
    0x9cd0: 'XXXXXXbh88',
    0x9cd1: 'kkkkbabs',
    0x9cd2: 'xxxxxa104',
    0x9cd3: 'hhhhabar',
    0x9cd4: 'XXXXXXXabbi',
    0x9cd5: 'mmmmbbar',
    0x9cd6: 'XXXXXXXabbI',
    0x9cd7: 'nnnnaabs',
    0x9cd8: 'wwwwwb014',
    0x9cd9: 'ggggbaar',
    0x9cda: 'ttttta114',
    0x9cdb: 'ddddabbs',
    0x9cdc: 'qqqqqb114',
    0x9cdd: 'aaaabbbs',
    0x9cde: 'rrrrra014',
    0x9cdf: 'bbbbaaar',
    0x9ce0: 'ppppppbb99',
    0x9ce1: 'ccccbbbs',
    0x9ce2: 'pppppb114',
    0x9ce3: 'ppppbabr',
    0x9ce4: 'pppppppabbj',
    0x9ce5: 'eeeebaar',
    0x9ce6: 'pppppppabbJ',
    0x9ce7: 'ffffabbs',
    0x9ce8: 'yyyyyyybbaK',
    0x9ce9: 'oooobbar',
    0x9cea: 'yyyyyyybbak',
    0x9ceb: 'llllaabs',
    0x9cec: 'yyyyya104',
    0x9ced: 'iiiibabs',
    0x9cee: 'yyyyyyab89',
    0x9cef: 'jjjjabar',
    0x9cf0: 'hhhhhhbb98',
    0x9cf1: 'kkkkabar',
    0x9cf2: 'hhhhhb115',
    0x9cf3: 'hhhhbabs',
    0x9cf4: 'hhhhhhhabbk',
    0x9cf5: 'hhhhhhhabdx',
    0x9cf6: 'hhhhhhhabbK',
    0x9cf7: 'hhhhhhhabdX',
    0x9cf8: 'aaaaaaabbaJ',
    0x9cf9: 'aaaaaaabbaZ',
    0x9cfa: 'aaaaaaabbaj',
    0x9cfb: 'aaaaaaabbaz',
    0x9cfc: 'aaaaaa105',
    0x9cfd: 'aaaaaaar',
    0x9cfe: 'aaaaaaab88',
    0x9cff: 'bbbbbbbs',
    0x9d00: 'ccccca015',
    0x9d01: 'ccccabas',
    0x9d02: 'ccccccaa98',
    0x9d03: 'eeeeeeebadx',
    0x9d04: 'cccccccbbaJ',
    0x9d05: 'cccccccbbaZ',
    0x9d06: 'cccccccbbaj',
    0x9d07: 'cccccccbbaz',
    0x9d08: 'iiiiiiibaaJ',
    0x9d09: 'iiiiiiibaaZ',
    0x9d0a: 'iiiiiiibaaj',
    0x9d0b: 'iiiiiiibaaz',
    0x9d0c: 'iiiiia105',
    0x9d0d: 'iiiiaaas',
    0x9d0e: 'iiiiiiab98',
    0x9d0f: 'jjjjbbbr',
    0x9d10: 'xxxxxxbb89',
    0x9d11: 'kkkkbbbr',
    0x9d12: 'xxxxxb114',
    0x9d13: 'hhhhaaas',
    0x9d14: 'xxxxxxxaabj',
    0x9d15: 'mmmmbaas',
    0x9d16: 'xxxxxxxaabJ',
    0x9d17: 'nnnnabbr',
    0x9d18: 'qqqqqqqbaaK',
    0x9d19: 'ggggbbas',
    0x9d1a: 'qqqqqqqbaak',
    0x9d1b: 'ddddaabr',
    0x9d1c: 'qqqqqa104',
    0x9d1d: 'aaaababr',
    0x9d1e: 'qqqqqqab99',
    0x9d1f: 'bbbbabas',
    0x9d20: 'sssssb004',
    0x9d21: 'ccccbabr',
    0x9d22: 'pppppa104',
    0x9d23: 'ppppbbbs',
    0x9d24: 'uuuuub104',
    0x9d25: 'eeeebbas',
    0x9d26: 'vvvvva004',
    0x9d27: 'ffffaabr',
    0x9d28: 'YYYYYYYbaaH',
    0x9d29: 'oooobaas',
    0x9d2a: 'YYYYYYYbaah',
    0x9d2b: 'llllabbr',
    0x9d2c: 'yyyyyb114',
    0x9d2d: 'iiiibbbr',
    0x9d2e: 'zzzzza014',
    0x9d2f: 'jjjjaaas',
    0x9d30: 'kkkkkb005',
    0x9d31: 'kkkkaaas',
    0x9d32: 'hhhhha105',
    0x9d33: 'hhhhbbbr',
    0x9d34: 'mmmmmb105',
    0x9d35: 'mmmmabbr',
    0x9d36: 'nnnnna005',
    0x9d37: 'nnnnbaas',
    0x9d38: 'gggggb015',
    0x9d39: 'ggggaabr',
    0x9d3a: 'ddddda115',
    0x9d3b: 'ddddbbas',
    0x9d3c: 'aaaaab115',
    0x9d3d: 'aaaaabas',
    0x9d3e: 'bbbbba015',
    0x9d3f: 'bbbbbabr',
    0x9d40: 'cccccb115',
    0x9d41: 'ccccaabr',
    0x9d42: 'CCCCCCbi89',
    0x9d43: 'ppppabbs',
    0x9d44: 'eeeeeb015',
    0x9d45: 'eeeeabas',
    0x9d46: 'fffffa115',
    0x9d47: 'ffffbabr',
    0x9d48: 'ooooob105',
    0x9d49: 'ooooaaas',
    0x9d4a: 'llllla005',
    0x9d4b: 'llllbbbr',
    0x9d4c: 'iiiiib005',
    0x9d4d: 'iiiiabbr',
    0x9d4e: 'jjjjja105',
    0x9d4f: 'jjjjbaas',
    0x9d50: 'XXXXXXah88',
    0x9d51: 'kkkkbaas',
    0x9d52: 'xxxxxa014',
    0x9d53: 'hhhhabbr',
    0x9d54: '=====b103',
    0x9d55: 'mmmmbbbr',
    0x9d56: '>>>>>a003',
    0x9d57: 'nnnnaaas',
    0x9d58: 'wwwwwb104',
    0x9d59: 'ggggbabr',
    0x9d5a: 'ttttta004',
    0x9d5b: 'ddddabas',
    0x9d5c: 'qqqqqb004',
    0x9d5d: 'aaaabbas',
    0x9d5e: 'rrrrra104',
    0x9d5f: 'bbbbaabr',
    0x9d60: 'ppppppab99',
    0x9d61: 'ccccbbas',
    0x9d62: 'pppppb004',
    0x9d63: 'ppppbaar',
    0x9d64: 'uuuuua004',
    0x9d65: 'eeeebabr',
    0x9d66: 'uuuuuubb99',
    0x9d67: 'ffffabas',
    0x9d68: '9999999baaL',
    0x9d69: 'oooobbbr',
    0x9d6a: '9999999baal',
    0x9d6b: 'llllaaas',
    0x9d6c: 'yyyyya014',
    0x9d6d: 'iiiibaas',
    0x9d6e: 'yyyyyybb89',
    0x9d6f: 'jjjjabbr',
    0x9d70: 'hhhhhhab98',
    0x9d71: 'kkkkabbr',
    0x9d72: 'hhhhhb005',
    0x9d73: 'hhhhbaas',
    0x9d74: 'mmmmma005',
    0x9d75: 'mmmmaaas',
    0x9d76: 'mmmmmmbb98',
    0x9d77: 'nnnnbbbr',
    0x9d78: 'ddddddab88',
    0x9d79: 'ggggabas',
    0x9d7a: 'dddddb015',
    0x9d7b: 'ddddbabr',
    0x9d7c: 'aaaaaa015',
    0x9d7d: 'aaaaaabr',
    0x9d7e: 'aaaaaabb88',
    0x9d7f: 'bbbbbbas',
    0x9d80: 'ccccca115',
    0x9d81: 'ccccabbr',
    0x9d82: 'ccccccbb88',
    0x9d83: 'ppppaabs',
    0x9d84: 'eeeeea015',
    0x9d85: 'eeeeaaas',
    0x9d86: 'eeeeeeba98',
    0x9d87: 'ffffbbbr',
    0x9d88: 'llllllaa88',
    0x9d89: 'ooooabas',
    0x9d8a: 'lllllb005',
    0x9d8b: 'llllbabr',
    0x9d8c: 'iiiiia005',
    0x9d8d: 'iiiiaabr',
    0x9d8e: 'iiiiiiba88',
    0x9d8f: 'jjjjbbas',
    0x9d90: 'xxxxxxaa99',
    0x9d91: 'kkkkbbas',
    0x9d92: 'xxxxxb014',
    0x9d93: 'hhhhaabr',
    0x9d94: '11111111abaK',
    0x9d95: 'mmmmbabr',
    0x9d96: '11111111abcZ',
    0x9d97: 'nnnnabas',
    0x9d98: 'ttttttaa89',
    0x9d99: 'ggggbbbr',
    0x9d9a: 'tttttb004',
    0x9d9b: 'ddddaaas',
    0x9d9c: 'qqqqqa004',
    0x9d9d: 'aaaabaas',
    0x9d9e: 'qqqqqqba89',
    0x9d9f: 'bbbbabbr',
    0x9da0: 'sssssb104',
    0x9da1: 'ccccbaas',
    0x9da2: 'pppppa004',
    0x9da3: 'ppppbbar',
    0x9da4: 'uuuuub004',
    0x9da5: 'eeeebbbr',
    0x9da6: 'vvvvva104',
    0x9da7: 'ffffaaas',
    0x9da8: '?????b003',
    0x9da9: 'oooobabr',
    0x9daa: '<<<<<a103',
    0x9dab: 'llllabas',
    0x9dac: 'yyyyyb014',
    0x9dad: 'iiiibbas',
    0x9dae: 'zzzzza114',
    0x9daf: 'jjjjaabr',
    0x9db0: 'kkkkkb105',
    0x9db1: 'kkkkaabr',
    0x9db2: 'hhhhha005',
    0x9db3: 'hhhhbbas',
    0x9db4: 'mmmmmb005',
    0x9db5: 'mmmmabas',
    0x9db6: 'nnnnna105',
    0x9db7: 'nnnnbabr',
    0x9db8: 'gggggb115',
    0x9db9: 'ggggaaas',
    0x9dba: 'ddddda015',
    0x9dbb: 'ddddbbbr',
    0x9dbc: 'aaaaab015',
    0x9dbd: 'aaaaabbr',
    0x9dbe: 'bbbbba115',
    0x9dbf: 'bbbbbaas',
    0x9dc0: 'cccccb015',
    0x9dc1: 'ccccaaas',
    0x9dc2: 'CCCCCCah89',
    0x9dc3: 'ppppabar',
    0x9dc4: 'eeeeeb115',
    0x9dc5: 'eeeeabbr',
    0x9dc6: 'fffffa015',
    0x9dc7: 'ffffbaas',
    0x9dc8: 'ooooob005',
    0x9dc9: 'ooooaabr',
    0x9dca: 'llllla105',
    0x9dcb: 'llllbbas',
    0x9dcc: 'iiiiib105',
    0x9dcd: 'iiiiabas',
    0x9dce: 'jjjjja005',
    0x9dcf: 'jjjjbabr',
    0x9dd0: 'CCCCCCCCCbbaq',
    0x9dd1: 'kkkkbabr',
    0x9dd2: 'xxxxxa114',
    0x9dd3: 'hhhhabas',
    0x9dd4: 'CCCCCCCCCbbaa',
    0x9dd5: 'mmmmbbas',
    0x9dd6: 'CCCCCCCCCbbai',
    0x9dd7: 'nnnnaabr',
    0x9dd8: 'wwwwwb004',
    0x9dd9: 'ggggbaas',
    0x9dda: 'ttttta104',
    0x9ddb: 'ddddabbr',
    0x9ddc: 'qqqqqb104',
    0x9ddd: 'aaaabbbr',
    0x9dde: 'rrrrra004',
    0x9ddf: 'bbbbaaas',
    0x9de0: 'aaaaaaaaabaaB',
    0x9de1: 'aaaaaaaaabaaF',
    0x9de2: 'aaaaaaaaabaaJ',
    0x9de3: 'aaaaaaaaabaaN',
    0x9de4: 'aaaaaaaaabaaR',
    0x9de5: 'aaaaaaaaabaaV',
    0x9de6: 'aaaaaaaaabaaZ',
    0x9de7: 'aaaaaaaaabaiZ',
    0x9de8: 'aaaaaaaaabaab',
    0x9de9: 'aaaaaaaaabaaf',
    0x9dea: 'aaaaaaaaabaaj',
    0x9deb: 'aaaaaaaaabaan',
    0x9dec: 'aaaaaaaaabaar',
    0x9ded: 'aaaaaaaaabaav',
    0x9dee: 'aaaaaaaaabaaz',
    0x9def: 'aaaaaaaaabaiz',
    0x9df0: 'aaaaaaaaabam',
    0x9df1: 'aaaaaaaaabae',
    0x9df2: 'aaaaaaaaabiy',
    0x9df3: 'aaaaaaaaabau',
    0x9df4: 'aaaaaaaaabaM',
    0x9df5: 'aaaaaaaaabaE',
    0x9df6: 'aaaaaaaaabiY',
    0x9df7: 'aaaaaaaaabaU',
    0x9df8: 'aaaaaaaaabK',
    0x9df9: 'aaaaaaaaadX',
    0x9dfa: 'aaaaaaaaabk',
    0x9dfb: 'aaaaaaaaadx',
    0x9dfc: 'aaaaaa115',
    0x9dfd: 'aaaaaaas',
    0x9dfe: 'aaaaaaaa98',
    0x9dff: 'aaaaaaaaa4',
    0x9e00: 'ccccca004',
    0x9e01: 'ccccabap',
    0x9e02: 'ccccccab98',
    0x9e03: 'cccccccca4',
    0x9e04: 'cccccccabaJ',
    0x9e05: 'cccccccabaZ',
    0x9e06: 'cccccccabaj',
    0x9e07: 'cccccccabaz',
    0x9e08: 'ccccccccabaM',
    0x9e09: 'ccccccccabaE',
    0x9e0a: 'ccccccccabiY',
    0x9e0b: 'ccccccccabaU',
    0x9e0c: 'ccccccccabam',
    0x9e0d: 'ccccccccabae',
    0x9e0e: 'ccccccccabiy',
    0x9e0f: 'ccccccccabau',
    0x9e10: 'cccccccccabbs',
    0x9e11: 'cccccccccabbw',
    0x9e12: 'cccccccccabdx',
    0x9e13: 'cccccccccabhz',
    0x9e14: 'cccccccccabbc',
    0x9e15: 'cccccccccabbg',
    0x9e16: 'cccccccccabbk',
    0x9e17: 'cccccccccabbo',
    0x9e18: 'cccccccccabbS',
    0x9e19: 'cccccccccabbW',
    0x9e1a: 'cccccccccabdX',
    0x9e1b: 'cccccccccabhZ',
    0x9e1c: 'cccccccccabbC',
    0x9e1d: 'aaaababq',
    0x9e1e: 'cccccccccabbK',
    0x9e1f: 'bbbbabap',
    0x9e20: 'ccccccccccaabn',
    0x9e21: 'ccccbabq',
    0x9e22: 'ccccccccccaabj',
    0x9e23: 'ccccccccccaabh',
    0x9e24: 'ccccccccccaabf',
    0x9e25: 'ccccccccccaabd',
    0x9e26: 'ccccccccccaabb',
    0x9e27: 'ccccccccccaadc',
    0x9e28: 'ccccccccccaajz',
    0x9e29: 'ccccccccccaahy',
    0x9e2a: 'ccccccccccaabz',
    0x9e2b: 'ccccccccccaabx',
    0x9e2c: 'ccccccccccaabv',
    0x9e2d: 'ccccccccccaabt',
    0x9e2e: 'ccccccccccaabr',
    0x9e2f: 'ccccccccccaabp',
    0x9e30: 'ccccccccccaabN',
    0x9e31: 'ccccccccccaabL',
    0x9e32: 'ccccccccccaabJ',
    0x9e33: 'ccccccccccaabH',
    0x9e34: 'ccccccccccaabF',
    0x9e35: 'ccccccccccaabD',
    0x9e36: 'ccccccccccaabB',
    0x9e37: 'ccccccccccaadC',
    0x9e38: 'ccccccccccaajZ',
    0x9e39: 'ccccccccccaahY',
    0x9e3a: 'ccccccccccaabZ',
    0x9e3b: 'ccccccccccaabX',
    0x9e3c: 'aaaaab104',
    0x9e3d: 'aaaaabap',
    0x9e3e: 'bbbbba004',
    0x9e3f: 'bbbbbabq',
    0x9e40: 'cccccb104',
    0x9e41: 'ccccaabq',
    0x9e42: 'cccccccccccbaad',
    0x9e43: 'cccccccccccbaae',
    0x9e44: 'cccccccccccbaab',
    0x9e45: 'cccccccccccbaac',
    0x9e46: 'cccccccccccbaca',
    0x9e47: 'cccccccccccbaaa',
    0x9e48: 'cccccccccccbaan',
    0x9e49: 'cccccccccccbaao',
    0x9e4a: 'cccccccccccbaal',
    0x9e4b: 'cccccccccccbaam',
    0x9e4c: 'cccccccccccbaaj',
    0x9e4d: 'cccccccccccbaak',
    0x9e4e: 'cccccccccccbaah',
    0x9e4f: 'cccccccccccbaai',
    0x9e50: 'cccccccccccbaav',
    0x9e51: 'cccccccccccbaaw',
    0x9e52: 'cccccccccccbaat',
    0x9e53: 'cccccccccccbaau',
    0x9e54: 'cccccccccccbaar',
    0x9e55: 'cccccccccccbaas',
    0x9e56: 'cccccccccccbaap',
    0x9e57: 'cccccccccccbaaq',
    0x9e58: 'cccccccccccbaiz',
    0x9e59: 'cccccccccccbakz',
    0x9e5a: 'cccccccccccbaix',
    0x9e5b: 'cccccccccccbaiy',
    0x9e5c: 'cccccccccccbaaz',
    0x9e5d: 'aaaabbap',
    0x9e5e: 'cccccccccccbaax',
    0x9e5f: 'bbbbaabq',
    0x9e60: 'cccccccccccbaaF',
    0x9e61: 'ccccbbap',
    0x9e62: 'cccccccccccbaaD',
    0x9e63: 'cccccccccccbaaE',
    0x9e64: 'cccccccccccbaaB',
    0x9e65: 'cccccccccccbaaC',
    0x9e66: 'cccccccccccbacA',
    0x9e67: 'cccccccccccbaaA',
    0x9e68: 'cccccccccccbaaN',
    0x9e69: 'cccccccccccbaaO',
    0x9e6a: 'cccccccccccbaaL',
    0x9e6b: 'cccccccccccbaaM',
    0x9e6c: 'cccccccccccbaaJ',
    0x9e6d: 'cccccccccccbaaK',
    0x9e6e: 'cccccccccccbaaH',
    0x9e6f: 'cccccccccccbaaI',
    0x9e70: 'cccccccccccbaaV',
    0x9e71: 'cccccccccccbaaW',
    0x9e72: 'cccccccccccbaaT',
    0x9e73: 'cccccccccccbaaU',
    0x9e74: 'cccccccccccbaaR',
    0x9e75: 'cccccccccccbaaS',
    0x9e76: 'cccccccccccbaaP',
    0x9e77: 'cccccccccccbaaQ',
    0x9e78: 'cccccccccccbaiZ',
    0x9e79: 'cccccccccccbakZ',
    0x9e7a: 'cccccccccccbaiX',
    0x9e7b: 'cccccccccccbaiY',
    0x9e7c: 'aaaaaa004',
    0x9e7d: 'aaaaaabq',
    0x9e7e: 'aaaaaaba88',
    0x9e7f: 'bbbbbbap',
    0x9e80: 'ccccca104',
    0x9e81: 'ccccabbq',
    0x9e82: 'ccccccba88',
    0x9e83: 'ppppaabp',
    0x9e84: 'eeeeea004',
    0x9e85: 'eeeeaaap',
    0x9e86: 'eeeeeebb98',
    0x9e87: 'ffffbbbq',
    0x9e88: 'llllllab88',
    0x9e89: 'ooooabap',
    0x9e8a: 'lllllb014',
    0x9e8b: 'llllbabq',
    0x9e8c: 'iiiiia014',
    0x9e8d: 'iiiiaabq',
    0x9e8e: 'iiiiiibb88',
    0x9e8f: 'jjjjbbap',
    0x9e90: 'xxxxxxab99',
    0x9e91: 'kkkkbbap',
    0x9e92: 'xxxxxb005',
    0x9e93: 'hhhhaabq',
    0x9e94: '8888888aaal',
    0x9e95: 'mmmmbabq',
    0x9e96: '8888888aaaL',
    0x9e97: 'nnnnabap',
    0x9e98: 'ttttttab89',
    0x9e99: 'ggggbbbq',
    0x9e9a: 'tttttb015',
    0x9e9b: 'ddddaaap',
    0x9e9c: 'qqqqqa015',
    0x9e9d: 'aaaabaap',
    0x9e9e: 'qqqqqqbb89',
    0x9e9f: 'bbbbabbq',
    0x9ea0: 'sssssb115',
    0x9ea1: 'ccccbaap',
    0x9ea2: 'pppppa015',
    0x9ea3: 'ppppbbaq',
    0x9ea4: 'uuuuub015',
    0x9ea5: 'eeeebbbq',
    0x9ea6: 'vvvvva115',
    0x9ea7: 'ffffaaap',
    0x9ea8: '?????b012',
    0x9ea9: 'oooobabq',
    0x9eaa: '<<<<<a112',
    0x9eab: 'llllabap',
    0x9eac: 'yyyyyb005',
    0x9ead: 'iiiibbap',
    0x9eae: 'zzzzza105',
    0x9eaf: 'jjjjaabq',
    0x9eb0: 'kkkkkb114',
    0x9eb1: 'kkkkaabq',
    0x9eb2: 'hhhhha014',
    0x9eb3: 'hhhhbbap',
    0x9eb4: 'mmmmmb014',
    0x9eb5: 'mmmmabap',
    0x9eb6: 'nnnnna114',
    0x9eb7: 'nnnnbabq',
    0x9eb8: 'gggggb104',
    0x9eb9: 'ggggaaap',
    0x9eba: 'ddddda004',
    0x9ebb: 'ddddbbbq',
    0x9ebc: 'aaaaab004',
    0x9ebd: 'aaaaabbq',
    0x9ebe: 'bbbbba104',
    0x9ebf: 'bbbbbaap',
    0x9ec0: 'cccccb004',
    0x9ec1: 'ccccaaap',
    0x9ec2: 'CCCCCCai99',
    0x9ec3: 'ppppabaq',
    0x9ec4: 'eeeeeb104',
    0x9ec5: 'eeeeabbq',
    0x9ec6: 'fffffa004',
    0x9ec7: 'ffffbaap',
    0x9ec8: 'ooooob014',
    0x9ec9: 'ooooaabq',
    0x9eca: 'llllla114',
    0x9ecb: 'llllbbap',
    0x9ecc: 'iiiiib114',
    0x9ecd: 'iiiiabap',
    0x9ece: 'jjjjja014',
    0x9ecf: 'jjjjbabq',
    0x9ed0: 'XXXXXXbh98',
    0x9ed1: 'kkkkbabq',
    0x9ed2: 'xxxxxa105',
    0x9ed3: 'hhhhabap',
    0x9ed4: 'XXXXXXXaaah',
    0x9ed5: 'mmmmbbap',
    0x9ed6: 'XXXXXXXaaaH',
    0x9ed7: 'nnnnaabq',
    0x9ed8: 'wwwwwb015',
    0x9ed9: 'ggggbaap',
    0x9eda: 'ttttta115',
    0x9edb: 'ddddabbq',
    0x9edc: 'qqqqqb115',
    0x9edd: 'aaaabbbq',
    0x9ede: 'rrrrra015',
    0x9edf: 'bbbbaaap',
    0x9ee0: 'ppppppbb89',
    0x9ee1: 'ccccbbbq',
    0x9ee2: 'pppppb115',
    0x9ee3: 'ppppbabp',
    0x9ee4: 'pppppppaaak',
    0x9ee5: 'eeeebaap',
    0x9ee6: 'pppppppaaaK',
    0x9ee7: 'ffffabbq',
    0x9ee8: 'yyyyyyybabJ',
    0x9ee9: 'oooobbap',
    0x9eea: 'yyyyyyybabj',
    0x9eeb: 'llllaabq',
    0x9eec: 'yyyyya105',
    0x9eed: 'iiiibabq',
    0x9eee: 'yyyyyyab99',
    0x9eef: 'jjjjabap',
    0x9ef0: 'hhhhhhbb88',
    0x9ef1: 'kkkkabap',
    0x9ef2: 'hhhhhb114',
    0x9ef3: 'hhhhbabq',
    0x9ef4: 'hhhhhhhaaaj',
    0x9ef5: 'hhhhhhhaaaz',
    0x9ef6: 'hhhhhhhaaaJ',
    0x9ef7: 'hhhhhhhaaaZ',
    0x9ef8: 'aaaaaaababK',
    0x9ef9: 'aaaaaaabadX',
    0x9efa: 'aaaaaaababk',
    0x9efb: 'aaaaaaabadx',
    0x9efc: 'aaaaaa104',
    0x9efd: 'aaaaaaap',
    0x9efe: 'aaaaaaab98',
    0x9eff: 'bbbbbbbq',
    0x9f00: 'ccccca014',
    0x9f01: 'ccccabaq',
    0x9f02: 'ccccccaa88',
    0x9f03: 'eeeeeeebbaz',
    0x9f04: 'cccccccbabK',
    0x9f05: 'cccccccbadX',
    0x9f06: 'cccccccbabk',
    0x9f07: 'cccccccbadx',
    0x9f08: 'iiiiiiibbbK',
    0x9f09: 'iiiiiiibbdX',
    0x9f0a: 'iiiiiiibbbk',
    0x9f0b: 'iiiiiiibbdx',
    0x9f0c: 'iiiiia104',
    0x9f0d: 'iiiiaaaq',
    0x9f0e: 'iiiiiiab88',
    0x9f0f: 'jjjjbbbp',
    0x9f10: 'xxxxxxbb99',
    0x9f11: 'kkkkbbbp',
    0x9f12: 'xxxxxb115',
    0x9f13: 'hhhhaaaq',
    0x9f14: 'xxxxxxxabak',
    0x9f15: 'mmmmbaaq',
    0x9f16: 'xxxxxxxabaK',
    0x9f17: 'nnnnabbp',
    0x9f18: 'qqqqqqqbbbJ',
    0x9f19: 'ggggbbaq',
    0x9f1a: 'qqqqqqqbbbj',
    0x9f1b: 'ddddaabp',
    0x9f1c: 'qqqqqa105',
    0x9f1d: 'aaaababp',
    0x9f1e: 'qqqqqqab89',
    0x9f1f: 'bbbbabaq',
    0x9f20: 'sssssb005',
    0x9f21: 'ccccbabp',
    0x9f22: 'pppppa105',
    0x9f23: 'ppppbbbq',
    0x9f24: 'uuuuub105',
    0x9f25: 'eeeebbaq',
    0x9f26: 'vvvvva005',
    0x9f27: 'ffffaabp',
    0x9f28: 'YYYYYYYbbbI',
    0x9f29: 'oooobaaq',
    0x9f2a: 'YYYYYYYbbbi',
    0x9f2b: 'llllabbp',
    0x9f2c: 'yyyyyb115',
    0x9f2d: 'iiiibbbp',
    0x9f2e: 'zzzzza015',
    0x9f2f: 'jjjjaaaq',
    0x9f30: 'kkkkkb004',
    0x9f31: 'kkkkaaaq',
    0x9f32: 'hhhhha104',
    0x9f33: 'hhhhbbbp',
    0x9f34: 'mmmmmb104',
    0x9f35: 'mmmmabbp',
    0x9f36: 'nnnnna004',
    0x9f37: 'nnnnbaaq',
    0x9f38: 'gggggb014',
    0x9f39: 'ggggaabp',
    0x9f3a: 'ddddda114',
    0x9f3b: 'ddddbbaq',
    0x9f3c: 'aaaaab114',
    0x9f3d: 'aaaaabaq',
    0x9f3e: 'bbbbba014',
    0x9f3f: 'bbbbbabp',
    0x9f40: 'cccccb114',
    0x9f41: 'ccccaabp',
    0x9f42: 'CCCCCCbi99',
    0x9f43: 'ppppabbq',
    0x9f44: 'eeeeeb014',
    0x9f45: 'eeeeabaq',
    0x9f46: 'fffffa114',
    0x9f47: 'ffffbabp',
    0x9f48: 'ooooob104',
    0x9f49: 'ooooaaaq',
    0x9f4a: 'llllla004',
    0x9f4b: 'llllbbbp',
    0x9f4c: 'iiiiib004',
    0x9f4d: 'iiiiabbp',
    0x9f4e: 'jjjjja104',
    0x9f4f: 'jjjjbaaq',
    0x9f50: 'XXXXXXah98',
    0x9f51: 'kkkkbaaq',
    0x9f52: 'xxxxxa015',
    0x9f53: 'hhhhabbp',
    0x9f54: '=====b102',
    0x9f55: 'mmmmbbbp',
    0x9f56: '>>>>>a002',
    0x9f57: 'nnnnaaaq',
    0x9f58: 'wwwwwb105',
    0x9f59: 'ggggbabp',
    0x9f5a: 'ttttta005',
    0x9f5b: 'ddddabaq',
    0x9f5c: 'qqqqqb005',
    0x9f5d: 'aaaabbaq',
    0x9f5e: 'rrrrra105',
    0x9f5f: 'bbbbaabp',
    0x9f60: 'ppppppab89',
    0x9f61: 'ccccbbaq',
    0x9f62: 'pppppb005',
    0x9f63: 'ppppbaap',
    0x9f64: 'uuuuua005',
    0x9f65: 'eeeebabp',
    0x9f66: 'uuuuuubb89',
    0x9f67: 'ffffabaq',
    0x9f68: '9999999bbbM',
    0x9f69: 'oooobbbp',
    0x9f6a: '9999999bbbm',
    0x9f6b: 'llllaaaq',
    0x9f6c: 'yyyyya015',
    0x9f6d: 'iiiibaaq',
    0x9f6e: 'yyyyyybb99',
    0x9f6f: 'jjjjabbp',
    0x9f70: 'hhhhhhab88',
    0x9f71: 'kkkkabbp',
    0x9f72: 'hhhhhb004',
    0x9f73: 'hhhhbaaq',
    0x9f74: 'mmmmma004',
    0x9f75: 'mmmmaaaq',
    0x9f76: 'mmmmmmbb88',
    0x9f77: 'nnnnbbbp',
    0x9f78: 'ddddddab98',
    0x9f79: 'ggggabaq',
    0x9f7a: 'dddddb014',
    0x9f7b: 'ddddbabp',
    0x9f7c: 'aaaaaa014',
    0x9f7d: 'aaaaaabp',
    0x9f7e: 'aaaaaabb98',
    0x9f7f: 'bbbbbbaq',
    0x9f80: 'ccccca114',
    0x9f81: 'ccccabbp',
    0x9f82: 'ccccccbb98',
    0x9f83: 'ppppaabq',
    0x9f84: 'eeeeea014',
    0x9f85: 'eeeeaaaq',
    0x9f86: 'eeeeeeba88',
    0x9f87: 'ffffbbbp',
    0x9f88: 'llllllaa98',
    0x9f89: 'ooooabaq',
    0x9f8a: 'lllllb004',
    0x9f8b: 'llllbabp',
    0x9f8c: 'iiiiia004',
    0x9f8d: 'iiiiaabp',
    0x9f8e: 'iiiiiiba98',
    0x9f8f: 'jjjjbbaq',
    0x9f90: 'xxxxxxaa89',
    0x9f91: 'kkkkbbaq',
    0x9f92: 'xxxxxb015',
    0x9f93: 'hhhhaabp',
    0x9f94: '11111111babJ',
    0x9f95: 'mmmmbabp',
    0x9f96: '11111111babZ',
    0x9f97: 'nnnnabaq',
    0x9f98: 'ttttttaa99',
    0x9f99: 'ggggbbbp',
    0x9f9a: 'tttttb005',
    0x9f9b: 'ddddaaaq',
    0x9f9c: 'qqqqqa005',
    0x9f9d: 'aaaabaaq',
    0x9f9e: 'qqqqqqba99',
    0x9f9f: 'bbbbabbp',
    0x9fa0: 'sssssb105',
    0x9fa1: 'ccccbaaq',
    0x9fa2: 'pppppa005',
    0x9fa3: 'ppppbbap',
    0x9fa4: 'uuuuub005',
    0x9fa5: 'eeeebbbp',
    0x9fa6: 'vvvvva105',
    0x9fa7: 'ffffaaaq',
    0x9fa8: '?????b002',
    0x9fa9: 'oooobabp',
    0x9faa: '<<<<<a102',
    0x9fab: 'llllabaq',
    0x9fac: 'yyyyyb015',
    0x9fad: 'iiiibbaq',
    0x9fae: 'zzzzza115',
    0x9faf: 'jjjjaabp',
    0x9fb0: 'kkkkkb104',
    0x9fb1: 'kkkkaabp',
    0x9fb2: 'hhhhha004',
    0x9fb3: 'hhhhbbaq',
    0x9fb4: 'mmmmmb004',
    0x9fb5: 'mmmmabaq',
    0x9fb6: 'nnnnna104',
    0x9fb7: 'nnnnbabp',
    0x9fb8: 'gggggb114',
    0x9fb9: 'ggggaaaq',
    0x9fba: 'ddddda014',
    0x9fbb: 'ddddbbbp',
    0x9fbc: 'aaaaab014',
    0x9fbd: 'aaaaabbp',
    0x9fbe: 'bbbbba114',
    0x9fbf: 'bbbbbaaq',
    0x9fc0: 'cccccb014',
    0x9fc1: 'ccccaaaq',
    0x9fc2: 'CCCCCCah99',
    0x9fc3: 'ppppabap',
    0x9fc4: 'eeeeeb114',
    0x9fc5: 'eeeeabbp',
    0x9fc6: 'fffffa014',
    0x9fc7: 'ffffbaaq',
    0x9fc8: 'ooooob004',
    0x9fc9: 'ooooaabp',
    0x9fca: 'llllla104',
    0x9fcb: 'llllbbaq',
    0x9fcc: 'iiiiib104',
    0x9fcd: 'iiiiabaq',
    0x9fce: 'jjjjja004',
    0x9fcf: 'jjjjbabp',
    0x9fd0: 'QQQQQQQQbabn',
    0x9fd1: 'kkkkbabp',
    0x9fd2: 'xxxxxa115',
    0x9fd3: 'hhhhabaq',
    0x9fd4: 'QQQQQQQQbabN',
    0x9fd5: 'mmmmbbaq',
    0x9fd6: 'QQQQQQQQbajZ',
    0x9fd7: 'nnnnaabp',
    0x9fd8: 'wwwwwb005',
    0x9fd9: 'ggggbaaq',
    0x9fda: 'ttttta105',
    0x9fdb: 'ddddabbp',
    0x9fdc: 'qqqqqb105',
    0x9fdd: 'aaaabbbp',
    0x9fde: 'rrrrra005',
    0x9fdf: 'bbbbaaaq',
    0x9fe0: 'ppppppba99',
    0x9fe1: 'ccccbbbp',
    0x9fe2: 'pppppb105',
    0x9fe3: 'ppppbabq',
    0x9fe4: 'pppppppbbbj',
    0x9fe5: 'eeeebaaq',
    0x9fe6: 'pppppppbbbJ',
    0x9fe7: 'ffffabbp',
    0x9fe8: 'ppppppppaaiX',
    0x9fe9: 'oooobbaq',
    0x9fea: 'ppppppppaaaL',
    0x9feb: 'llllaabp',
    0x9fec: 'ppppppppaaix',
    0x9fed: 'iiiibabp',
    0x9fee: 'ppppppppaaal',
    0x9fef: 'jjjjabaq',
    0x9ff0: 'aaaaaaaababl',
    0x9ff1: 'aaaaaaaababd',
    0x9ff2: 'aaaaaaaabahy',
    0x9ff3: 'aaaaaaaababt',
    0x9ff4: 'aaaaaaaababL',
    0x9ff5: 'aaaaaaaababD',
    0x9ff6: 'aaaaaaaabahY',
    0x9ff7: 'aaaaaaaababT',
    0x9ff8: 'aaaaaaaabaJ',
    0x9ff9: 'aaaaaaaabaZ',
    0x9ffa: 'aaaaaaaabaj',
    0x9ffb: 'aaaaaaaabaz',
    0x9ffc: 'aaaaaa114',
    0x9ffd: 'aaaaaaaq',
    0x9ffe: 'aaaaaaaa88',
    0x9fff: 'aaaaaaaab5',
    0xa000: 'cccccah89',
    0xa001: 'ccccabaN',
    0xa002: 'ccccccab03',
    0xa003: 'ccccccccd9',
    0xa004: 'cccccccaaaM',
    0xa005: 'cccccccaaiY',
    0xa006: 'cccccccaaam',
    0xa007: 'cccccccaaiy',
    0xa008: 'ccccccccbbbO',
    0xa009: 'ccccccccbbbG',
    0xa00a: 'ccccccccbbhZ',
    0xa00b: 'ccccccccbbbW',
    0xa00c: 'ccccccccbbbo',
    0xa00d: 'ccccccccbbbg',
    0xa00e: 'ccccccccbbhz',
    0xa00f: 'ccccccccbbbw',
    0xa010: 'qqqqqqqqbabn',
    0xa011: 'kkkkbbbO',
    0xa012: 'qqqqqqqqbajz',
    0xa013: 'hhhhaaaN',
    0xa014: 'qqqqqqqqbabN',
    0xa015: 'mmmmbaaN',
    0xa016: 'qqqqqqqqbajZ',
    0xa017: 'nnnnabbO',
    0xa018: 'qqqqqqqabaL',
    0xa019: 'ggggbbaN',
    0xa01a: 'qqqqqqqabal',
    0xa01b: 'ddddaabO',
    0xa01c: 'qqqqqai98',
    0xa01d: 'aaaababO',
    0xa01e: 'qqqqqqaa02',
    0xa01f: 'bbbbabaN',
    0xa020: 'sssssbh98',
    0xa021: 'ccccbabO',
    0xa022: 'pppppai98',
    0xa023: 'ppppbbbN',
    0xa024: 'uuuuubi98',
    0xa025: 'eeeebbaN',
    0xa026: 'vvvvvah98',
    0xa027: 'ffffaabO',
    0xa028: 'PPPPPPPPaaiX',
    0xa029: 'oooobaaN',
    0xa02a: 'PPPPPPPPaaaL',
    0xa02b: 'llllabbO',
    0xa02c: 'yyyyybi88',
    0xa02d: 'iiiibbbO',
    0xa02e: 'zzzzzah88',
    0xa02f: 'jjjjaaaN',
    0xa030: 'kkkkkbh99',
    0xa031: 'kkkkaaaN',
    0xa032: 'hhhhhai99',
    0xa033: 'hhhhbbbO',
    0xa034: 'mmmmmbi99',
    0xa035: 'mmmmabbO',
    0xa036: 'nnnnnah99',
    0xa037: 'nnnnbaaN',
    0xa038: 'gggggbh89',
    0xa039: 'ggggaabO',
    0xa03a: 'dddddai89',
    0xa03b: 'ddddbbaN',
    0xa03c: 'aaaaabi89',
    0xa03d: 'aaaaabaN',
    0xa03e: 'bbbbbah89',
    0xa03f: 'bbbbbabO',
    0xa040: 'cccccbi89',
    0xa041: 'ccccaabO',
    0xa042: 'CCCCCCbb10',
    0xa043: 'ppppabbN',
    0xa044: 'eeeeebh89',
    0xa045: 'eeeeabaN',
    0xa046: 'fffffai89',
    0xa047: 'ffffbabO',
    0xa048: 'ooooobi99',
    0xa049: 'ooooaaaN',
    0xa04a: 'lllllah99',
    0xa04b: 'llllbbbO',
    0xa04c: 'iiiiibh99',
    0xa04d: 'iiiiabbO',
    0xa04e: 'jjjjjai99',
    0xa04f: 'jjjjbaaN',
    0xa050: 'XXXXXXaa01',
    0xa051: 'kkkkbaaN',
    0xa052: 'xxxxxah88',
    0xa053: 'hhhhabbO',
    0xa054: '=====ba19',
    0xa055: 'mmmmbbbO',
    0xa056: '>>>>>ab09',
    0xa057: 'nnnnaaaN',
    0xa058: 'wwwwwbi98',
    0xa059: 'ggggbabO',
    0xa05a: 'tttttah98',
    0xa05b: 'ddddabaN',
    0xa05c: 'qqqqqbh98',
    0xa05d: 'aaaabbaN',
    0xa05e: 'rrrrrai98',
    0xa05f: 'bbbbaabO',
    0xa060: 'ppppppaa02',
    0xa061: 'ccccbbaN',
    0xa062: 'pppppbh98',
    0xa063: 'ppppbaaO',
    0xa064: 'uuuuuah98',
    0xa065: 'eeeebabO',
    0xa066: 'uuuuuuba02',
    0xa067: 'ffffabaN',
    0xa068: '00000000aaaX',
    0xa069: 'oooobbbO',
    0xa06a: '00000000aaaH',
    0xa06b: 'llllaaaN',
    0xa06c: 'yyyyyah88',
    0xa06d: 'iiiibaaN',
    0xa06e: 'yyyyyyba12',
    0xa06f: 'jjjjabbO',
    0xa070: 'hhhhhhaa03',
    0xa071: 'kkkkabbO',
    0xa072: 'hhhhhbh99',
    0xa073: 'hhhhbaaN',
    0xa074: 'mmmmmah99',
    0xa075: 'mmmmaaaN',
    0xa076: 'mmmmmmba03',
    0xa077: 'nnnnbbbO',
    0xa078: 'ddddddaa13',
    0xa079: 'ggggabaN',
    0xa07a: 'dddddbh89',
    0xa07b: 'ddddbabO',
    0xa07c: 'aaaaaah89',
    0xa07d: 'aaaaaabO',
    0xa07e: 'aaaaaaba13',
    0xa07f: 'bbbbbbaN',
    0xa080: 'cccccai89',
    0xa081: 'ccccabbO',
    0xa082: 'ccccccba13',
    0xa083: 'ppppaabN',
    0xa084: 'eeeeeah89',
    0xa085: 'eeeeaaaN',
    0xa086: 'eeeeeebb03',
    0xa087: 'ffffbbbO',
    0xa088: 'llllllab13',
    0xa089: 'ooooabaN',
    0xa08a: 'lllllbh99',
    0xa08b: 'llllbabO',
    0xa08c: 'iiiiiah99',
    0xa08d: 'iiiiaabO',
    0xa08e: 'iiiiiibb13',
    0xa08f: 'jjjjbbaN',
    0xa090: 'xxxxxxab02',
    0xa091: 'kkkkbbaN',
    0xa092: 'xxxxxbh88',
    0xa093: 'hhhhaabO',
    0xa094: '8888888abak',
    0xa095: 'mmmmbabO',
    0xa096: '8888888abaK',
    0xa097: 'nnnnabaN',
    0xa098: 'ttttttab12',
    0xa099: 'ggggbbbO',
    0xa09a: 'tttttbh98',
    0xa09b: 'ddddaaaN',
    0xa09c: 'qqqqqah98',
    0xa09d: 'aaaabaaN',
    0xa09e: 'qqqqqqbb12',
    0xa09f: 'bbbbabbO',
    0xa0a0: 'sssssbi98',
    0xa0a1: 'ccccbaaN',
    0xa0a2: 'pppppah98',
    0xa0a3: 'ppppbbaO',
    0xa0a4: 'uuuuubh98',
    0xa0a5: 'eeeebbbO',
    0xa0a6: 'vvvvvai98',
    0xa0a7: 'ffffaaaN',
    0xa0a8: '?????bb09',
    0xa0a9: 'oooobabO',
    0xa0aa: '<<<<<aa19',
    0xa0ab: 'llllabaN',
    0xa0ac: 'yyyyybh88',
    0xa0ad: 'iiiibbaN',
    0xa0ae: 'zzzzzai88',
    0xa0af: 'jjjjaabO',
    0xa0b0: 'kkkkkbi99',
    0xa0b1: 'kkkkaabO',
    0xa0b2: 'hhhhhah99',
    0xa0b3: 'hhhhbbaN',
    0xa0b4: 'mmmmmbh99',
    0xa0b5: 'mmmmabaN',
    0xa0b6: 'nnnnnai99',
    0xa0b7: 'nnnnbabO',
    0xa0b8: 'gggggbi89',
    0xa0b9: 'ggggaaaN',
    0xa0ba: 'dddddah89',
    0xa0bb: 'ddddbbbO',
    0xa0bc: 'aaaaabh89',
    0xa0bd: 'aaaaabbO',
    0xa0be: 'bbbbbai89',
    0xa0bf: 'bbbbbaaN',
    0xa0c0: 'cccccbh89',
    0xa0c1: 'ccccaaaN',
    0xa0c2: 'CCCCCCaa00',
    0xa0c3: 'ppppabaO',
    0xa0c4: 'eeeeebi89',
    0xa0c5: 'eeeeabbO',
    0xa0c6: 'fffffah89',
    0xa0c7: 'ffffbaaN',
    0xa0c8: 'ooooobh99',
    0xa0c9: 'ooooaabO',
    0xa0ca: 'lllllai99',
    0xa0cb: 'llllbbaN',
    0xa0cc: 'iiiiibi99',
    0xa0cd: 'iiiiabaN',
    0xa0ce: 'jjjjjah99',
    0xa0cf: 'jjjjbabO',
    0xa0d0: 'XXXXXXbb11',
    0xa0d1: 'kkkkbabO',
    0xa0d2: 'xxxxxai88',
    0xa0d3: 'hhhhabaN',
    0xa0d4: 'XXXXXXXabao',
    0xa0d5: 'mmmmbbaN',
    0xa0d6: 'XXXXXXXabaO',
    0xa0d7: 'nnnnaabO',
    0xa0d8: 'wwwwwbh98',
    0xa0d9: 'ggggbaaN',
    0xa0da: 'tttttai98',
    0xa0db: 'ddddabbO',
    0xa0dc: 'qqqqqbi98',
    0xa0dd: 'aaaabbbO',
    0xa0de: 'rrrrrah98',
    0xa0df: 'bbbbaaaN',
    0xa0e0: 'ppppppbb12',
    0xa0e1: 'ccccbbbO',
    0xa0e2: 'pppppbi98',
    0xa0e3: 'ppppbabN',
    0xa0e4: 'pppppppabal',
    0xa0e5: 'eeeebaaN',
    0xa0e6: 'pppppppabaL',
    0xa0e7: 'ffffabbO',
    0xa0e8: 'yyyyyyybbbM',
    0xa0e9: 'oooobbaN',
    0xa0ea: 'yyyyyyybbbm',
    0xa0eb: 'llllaabO',
    0xa0ec: 'yyyyyai88',
    0xa0ed: 'iiiibabO',
    0xa0ee: 'yyyyyyab02',
    0xa0ef: 'jjjjabaN',
    0xa0f0: 'hhhhhhbb13',
    0xa0f1: 'kkkkabaN',
    0xa0f2: 'hhhhhbi99',
    0xa0f3: 'hhhhbabO',
    0xa0f4: 'hhhhhhhabam',
    0xa0f5: 'hhhhhhhabiy',
    0xa0f6: 'hhhhhhhabaM',
    0xa0f7: 'hhhhhhhabiY',
    0xa0f8: 'aaaaaaabbbL',
    0xa0f9: 'aaaaaaabbhY',
    0xa0fa: 'aaaaaaabbbl',
    0xa0fb: 'aaaaaaabbhy',
    0xa0fc: 'aaaaaai89',
    0xa0fd: 'aaaaaaaN',
    0xa0fe: 'aaaaaaab03',
    0xa0ff: 'bbbbbbbO',
    0xa100: 'cccccah99',
    0xa101: 'ccccabaO',
    0xa102: 'ccccccaa13',
    0xa103: 'eeeeeeebaiy',
    0xa104: 'cccccccbbbL',
    0xa105: 'cccccccbbhY',
    0xa106: 'cccccccbbbl',
    0xa107: 'cccccccbbhy',
    0xa108: 'iiiiiiibabL',
    0xa109: 'iiiiiiibahY',
    0xa10a: 'iiiiiiibabl',
    0xa10b: 'iiiiiiibahy',
    0xa10c: 'iiiiiai89',
    0xa10d: 'iiiiaaaO',
    0xa10e: 'iiiiiiab13',
    0xa10f: 'jjjjbbbN',
    0xa110: 'xxxxxxbb02',
    0xa111: 'kkkkbbbN',
    0xa112: 'xxxxxbi98',
    0xa113: 'hhhhaaaO',
    0xa114: 'xxxxxxxaaal',
    0xa115: 'mmmmbaaO',
    0xa116: 'xxxxxxxaaaL',
    0xa117: 'nnnnabbN',
    0xa118: 'qqqqqqqbabM',
    0xa119: 'ggggbbaO',
    0xa11a: 'qqqqqqqbabm',
    0xa11b: 'ddddaabN',
    0xa11c: 'qqqqqai88',
    0xa11d: 'aaaababN',
    0xa11e: 'qqqqqqab12',
    0xa11f: 'bbbbabaO',
    0xa120: 'sssssbh88',
    0xa121: 'ccccbabN',
    0xa122: 'pppppai88',
    0xa123: 'ppppbbbO',
    0xa124: 'uuuuubi88',
    0xa125: 'eeeebbaO',
    0xa126: 'vvvvvah88',
    0xa127: 'ffffaabN',
    0xa128: 'YYYYYYYbabN',
    0xa129: 'oooobaaO',
    0xa12a: 'YYYYYYYbabn',
    0xa12b: 'llllabbN',
    0xa12c: 'yyyyybi98',
    0xa12d: 'iiiibbbN',
    0xa12e: 'zzzzzah98',
    0xa12f: 'jjjjaaaO',
    0xa130: 'kkkkkbh89',
    0xa131: 'kkkkaaaO',
    0xa132: 'hhhhhai89',
    0xa133: 'hhhhbbbN',
    0xa134: 'mmmmmbi89',
    0xa135: 'mmmmabbN',
    0xa136: 'nnnnnah89',
    0xa137: 'nnnnbaaO',
    0xa138: 'gggggbh99',
    0xa139: 'ggggaabN',
    0xa13a: 'dddddai99',
    0xa13b: 'ddddbbaO',
    0xa13c: 'aaaaabi99',
    0xa13d: 'aaaaabaO',
    0xa13e: 'bbbbbah99',
    0xa13f: 'bbbbbabN',
    0xa140: 'cccccbi99',
    0xa141: 'ccccaabN',
    0xa142: 'CCCCCCba00',
    0xa143: 'ppppabbO',
    0xa144: 'eeeeebh99',
    0xa145: 'eeeeabaO',
    0xa146: 'fffffai99',
    0xa147: 'ffffbabN',
    0xa148: 'ooooobi89',
    0xa149: 'ooooaaaO',
    0xa14a: 'lllllah89',
    0xa14b: 'llllbbbN',
    0xa14c: 'iiiiibh89',
    0xa14d: 'iiiiabbN',
    0xa14e: 'jjjjjai89',
    0xa14f: 'jjjjbaaO',
    0xa150: 'XXXXXXab11',
    0xa151: 'kkkkbaaO',
    0xa152: 'xxxxxah98',
    0xa153: 'hhhhabbN',
    0xa154: '=====ba09',
    0xa155: 'mmmmbbbN',
    0xa156: '>>>>>ab19',
    0xa157: 'nnnnaaaO',
    0xa158: 'wwwwwbi88',
    0xa159: 'ggggbabN',
    0xa15a: 'tttttah88',
    0xa15b: 'ddddabaO',
    0xa15c: 'qqqqqbh88',
    0xa15d: 'aaaabbaO',
    0xa15e: 'rrrrrai88',
    0xa15f: 'bbbbaabN',
    0xa160: 'ppppppab12',
    0xa161: 'ccccbbaO',
    0xa162: 'pppppbh88',
    0xa163: 'ppppbaaN',
    0xa164: 'uuuuuah88',
    0xa165: 'eeeebabN',
    0xa166: 'uuuuuubb12',
    0xa167: 'ffffabaO',
    0xa168: '9999999babJ',
    0xa169: 'oooobbbN',
    0xa16a: '9999999babj',
    0xa16b: 'llllaaaO',
    0xa16c: 'yyyyyah98',
    0xa16d: 'iiiibaaO',
    0xa16e: 'yyyyyybb02',
    0xa16f: 'jjjjabbN',
    0xa170: 'hhhhhhab13',
    0xa171: 'kkkkabbN',
    0xa172: 'hhhhhbh89',
    0xa173: 'hhhhbaaO',
    0xa174: 'mmmmmah89',
    0xa175: 'mmmmaaaO',
    0xa176: 'mmmmmmbb13',
    0xa177: 'nnnnbbbN',
    0xa178: 'ddddddab03',
    0xa179: 'ggggabaO',
    0xa17a: 'dddddbh99',
    0xa17b: 'ddddbabN',
    0xa17c: 'aaaaaah99',
    0xa17d: 'aaaaaabN',
    0xa17e: 'aaaaaabb03',
    0xa17f: 'bbbbbbaO',
    0xa180: 'aaaaaaaaaaabbdX',
    0xa181: 'aaaaaaaaaaabbbZ',
    0xa182: 'aaaaaaaaaaabbbY',
    0xa183: 'aaaaaaaaaaabbbX',
    0xa184: 'aaaaaaaaaaabbhZ',
    0xa185: 'aaaaaaaaaaabbjZ',
    0xa186: 'aaaaaaaaaaabbhX',
    0xa187: 'aaaaaaaaaaabbhY',
    0xa188: 'aaaaaaaaaaabbbS',
    0xa189: 'aaaaaaaaaaabbbR',
    0xa18a: 'aaaaaaaaaaabbbQ',
    0xa18b: 'aaaaaaaaaaabbbP',
    0xa18c: 'aaaaaaaaaaabbbW',
    0xa18d: 'aaaaaaaaaaabbbV',
    0xa18e: 'aaaaaaaaaaabbbU',
    0xa18f: 'aaaaaaaaaaabbbT',
    0xa190: 'aaaaaaaaaaabbbK',
    0xa191: 'aaaaaaaaaaabbbJ',
    0xa192: 'aaaaaaaaaaabbbI',
    0xa193: 'aaaaaaaaaaabbbH',
    0xa194: 'aaaaaaaaaaabbbO',
    0xa195: 'aaaaaaaaaaabbbN',
    0xa196: 'aaaaaaaaaaabbbM',
    0xa197: 'aaaaaaaaaaabbbL',
    0xa198: 'aaaaaaaaaaabbbC',
    0xa199: 'aaaaaaaaaaabbbB',
    0xa19a: 'aaaaaaaaaaabbbA',
    0xa19b: 'aaaaaaaaaaabbdC',
    0xa19c: 'aaaaaaaaaaabbbG',
    0xa19d: 'aaaabaaO',
    0xa19e: 'aaaaaaaaaaabbbE',
    0xa19f: 'aaaaaaaaaaabbbD',
    0xa1a0: 'aaaaaaaaaaabbdx',
    0xa1a1: 'aaaaaaaaaaabbbz',
    0xa1a2: 'aaaaaaaaaaabbby',
    0xa1a3: 'aaaaaaaaaaabbbx',
    0xa1a4: 'aaaaaaaaaaabbhz',
    0xa1a5: 'aaaaaaaaaaabbjz',
    0xa1a6: 'aaaaaaaaaaabbhx',
    0xa1a7: 'aaaaaaaaaaabbhy',
    0xa1a8: 'aaaaaaaaaaabbbs',
    0xa1a9: 'aaaaaaaaaaabbbr',
    0xa1aa: 'aaaaaaaaaaabbbq',
    0xa1ab: 'aaaaaaaaaaabbbp',
    0xa1ac: 'aaaaaaaaaaabbbw',
    0xa1ad: 'aaaaaaaaaaabbbv',
    0xa1ae: 'aaaaaaaaaaabbbu',
    0xa1af: 'aaaaaaaaaaabbbt',
    0xa1b0: 'aaaaaaaaaaabbbk',
    0xa1b1: 'aaaaaaaaaaabbbj',
    0xa1b2: 'aaaaaaaaaaabbbi',
    0xa1b3: 'aaaaaaaaaaabbbh',
    0xa1b4: 'aaaaaaaaaaabbbo',
    0xa1b5: 'aaaaaaaaaaabbbn',
    0xa1b6: 'aaaaaaaaaaabbbm',
    0xa1b7: 'aaaaaaaaaaabbbl',
    0xa1b8: 'aaaaaaaaaaabbbc',
    0xa1b9: 'aaaaaaaaaaabbbb',
    0xa1ba: 'aaaaaaaaaaabbba',
    0xa1bb: 'aaaaaaaaaaabbdc',
    0xa1bc: 'aaaaabh99',
    0xa1bd: 'aaaaabbN',
    0xa1be: 'aaaaaaaaaaabbbe',
    0xa1bf: 'aaaaaaaaaaabbbd',
    0xa1c0: 'aaaaaaaaaaabbV',
    0xa1c1: 'aaaaaaaaaaabbT',
    0xa1c2: 'aaaaaaaaaaabbR',
    0xa1c3: 'aaaaaaaaaaabbP',
    0xa1c4: 'aaaaaaaaaaabjZ',
    0xa1c5: 'aaaaaaaaaaabhY',
    0xa1c6: 'aaaaaaaaaaabbZ',
    0xa1c7: 'aaaaaaaaaaabbX',
    0xa1c8: 'aaaaaaaaaaabbF',
    0xa1c9: 'aaaaaaaaaaabbD',
    0xa1ca: 'aaaaaaaaaaabbB',
    0xa1cb: 'aaaaaaaaaaabdC',
    0xa1cc: 'aaaaaaaaaaabbN',
    0xa1cd: 'aaaaaaaaaaabbL',
    0xa1ce: 'aaaaaaaaaaabbJ',
    0xa1cf: 'aaaaaaaaaaabbH',
    0xa1d0: 'aaaaaaaaaaabbv',
    0xa1d1: 'aaaaaaaaaaabbt',
    0xa1d2: 'aaaaaaaaaaabbr',
    0xa1d3: 'aaaaaaaaaaabbp',
    0xa1d4: 'aaaaaaaaaaabjz',
    0xa1d5: 'aaaaaaaaaaabhy',
    0xa1d6: 'aaaaaaaaaaabbz',
    0xa1d7: 'aaaaaaaaaaabbx',
    0xa1d8: 'aaaaaaaaaaabbf',
    0xa1d9: 'aaaaaaaaaaabbd',
    0xa1da: 'aaaaaaaaaaabbb',
    0xa1db: 'aaaaaaaaaaabdc',
    0xa1dc: 'aaaaaaaaaaabbn',
    0xa1dd: 'aaaabbbN',
    0xa1de: 'aaaaaaaaaaabbj',
    0xa1df: 'aaaaaaaaaaabbh',
    0xa1e0: 'aaaaaaaaaaabB',
    0xa1e1: 'aaaaaaaaaaabF',
    0xa1e2: 'aaaaaaaaaaabJ',
    0xa1e3: 'aaaaaaaaaaabN',
    0xa1e4: 'aaaaaaaaaaabR',
    0xa1e5: 'aaaaaaaaaaabV',
    0xa1e6: 'aaaaaaaaaaabZ',
    0xa1e7: 'aaaaaaaaaaajZ',
    0xa1e8: 'aaaaaaaaaaabb',
    0xa1e9: 'aaaaaaaaaaabf',
    0xa1ea: 'aaaaaaaaaaabj',
    0xa1eb: 'aaaaaaaaaaabn',
    0xa1ec: 'aaaaaaaaaaabr',
    0xa1ed: 'aaaaaaaaaaabv',
    0xa1ee: 'aaaaaaaaaaabz',
    0xa1ef: 'aaaaaaaaaaajz',
    0xa1f0: 'aaaaaaaaaaan',
    0xa1f1: 'aaaaaaaaaaaf',
    0xa1f2: 'aaaaaaaaaaiz',
    0xa1f3: 'aaaaaaaaaaav',
    0xa1f4: 'aaaaaaaaaaaN',
    0xa1f5: 'aaaaaaaaaaaF',
    0xa1f6: 'aaaaaaaaaaiZ',
    0xa1f7: 'aaaaaaaaaaaV',
    0xa1f8: 'aaaaaaaaaaM',
    0xa1f9: 'aaaaaaaaaiY',
    0xa1fa: 'aaaaaaaaaam',
    0xa1fb: 'aaaaaaaaaiy',
    0xa1fc: 'aaaaaai99',
    0xa1fd: 'aaaaaaaO',
    0xa1fe: 'aaaaaaaa13',
    0xa1ff: 'aaaaaaaae9',
    0xa200: 'cccccah88',
    0xa201: 'ccccabaL',
    0xa202: 'ccccccab13',
    0xa203: 'cccccccce9',
    0xa204: 'cccccccabbL',
    0xa205: 'cccccccabhY',
    0xa206: 'cccccccabbl',
    0xa207: 'cccccccabhy',
    0xa208: 'ccccccccaaaN',
    0xa209: 'ccccccccaaaF',
    0xa20a: 'ccccccccaaiZ',
    0xa20b: 'ccccccccaaaV',
    0xa20c: 'ccccccccaaan',
    0xa20d: 'ccccccccaaaf',
    0xa20e: 'ccccccccaaiz',
    0xa20f: 'ccccccccaaav',
    0xa210: 'cccccccccbbas',
    0xa211: 'cccccccccbbaw',
    0xa212: 'cccccccccbbcz',
    0xa213: 'cccccccccbbkz',
    0xa214: 'cccccccccbbac',
    0xa215: 'cccccccccbbag',
    0xa216: 'cccccccccbbak',
    0xa217: 'cccccccccbbao',
    0xa218: 'cccccccccbbaS',
    0xa219: 'cccccccccbbaW',
    0xa21a: 'cccccccccbbcZ',
    0xa21b: 'cccccccccbbkZ',
    0xa21c: 'cccccccccbbaC',
    0xa21d: 'aaaababM',
    0xa21e: 'cccccccccbbaK',
    0xa21f: 'bbbbabaL',
    0xa220: 'sssssbh99',
    0xa221: 'ccccbabM',
    0xa222: 'pppppai99',
    0xa223: 'ppppbbbL',
    0xa224: 'uuuuubi99',
    0xa225: 'eeeebbaL',
    0xa226: 'vvvvvah99',
    0xa227: 'ffffaabM',
    0xa228: 'AAAAAAAAAbaca',
    0xa229: 'oooobaaL',
    0xa22a: 'AAAAAAAAAbaah',
    0xa22b: 'llllabbM',
    0xa22c: 'yyyyybi89',
    0xa22d: 'iiiibbbM',
    0xa22e: 'zzzzzah89',
    0xa22f: 'jjjjaaaL',
    0xa230: 'kkkkkbh98',
    0xa231: 'kkkkaaaL',
    0xa232: 'hhhhhai98',
    0xa233: 'hhhhbbbM',
    0xa234: 'mmmmmbi98',
    0xa235: 'mmmmabbM',
    0xa236: 'nnnnnah98',
    0xa237: 'nnnnbaaL',
    0xa238: 'gggggbh88',
    0xa239: 'ggggaabM',
    0xa23a: 'dddddai88',
    0xa23b: 'ddddbbaL',
    0xa23c: 'aaaaabi88',
    0xa23d: 'aaaaabaL',
    0xa23e: 'bbbbbah88',
    0xa23f: 'bbbbbabM',
    0xa240: 'cccccbi88',
    0xa241: 'ccccaabM',
    0xa242: 'CCCCCCbb00',
    0xa243: 'ppppabbL',
    0xa244: 'eeeeebh88',
    0xa245: 'eeeeabaL',
    0xa246: 'fffffai88',
    0xa247: 'ffffbabM',
    0xa248: 'ooooobi98',
    0xa249: 'ooooaaaL',
    0xa24a: 'lllllah98',
    0xa24b: 'llllbbbM',
    0xa24c: 'iiiiibh98',
    0xa24d: 'iiiiabbM',
    0xa24e: 'jjjjjai98',
    0xa24f: 'jjjjbaaL',
    0xa250: 'XXXXXXaa11',
    0xa251: 'kkkkbaaL',
    0xa252: 'xxxxxah89',
    0xa253: 'hhhhabbM',
    0xa254: '=====ba18',
    0xa255: 'mmmmbbbM',
    0xa256: '>>>>>ab08',
    0xa257: 'nnnnaaaL',
    0xa258: 'wwwwwbi99',
    0xa259: 'ggggbabM',
    0xa25a: 'tttttah99',
    0xa25b: 'ddddabaL',
    0xa25c: 'qqqqqbh99',
    0xa25d: 'aaaabbaL',
    0xa25e: 'rrrrrai99',
    0xa25f: 'bbbbaabM',
    0xa260: 'ppppppaa12',
    0xa261: 'ccccbbaL',
    0xa262: 'pppppbh99',
    0xa263: 'ppppbaaM',
    0xa264: 'uuuuuah99',
    0xa265: 'eeeebabM',
    0xa266: 'uuuuuuba12',
    0xa267: 'ffffabaL',
    0xa268: '00000000bbbY',
    0xa269: 'oooobbbM',
    0xa26a: '00000000bbbI',
    0xa26b: 'llllaaaL',
    0xa26c: 'yyyyyah89',
    0xa26d: 'iiiibaaL',
    0xa26e: 'yyyyyyba02',
    0xa26f: 'jjjjabbM',
    0xa270: 'hhhhhhaa13',
    0xa271: 'kkkkabbM',
    0xa272: 'hhhhhbh98',
    0xa273: 'hhhhbaaL',
    0xa274: 'mmmmmah98',
    0xa275: 'mmmmaaaL',
    0xa276: 'mmmmmmba13',
    0xa277: 'nnnnbbbM',
    0xa278: 'ddddddaa03',
    0xa279: 'ggggabaL',
    0xa27a: 'dddddbh88',
    0xa27b: 'ddddbabM',
    0xa27c: 'aaaaaah88',
    0xa27d: 'aaaaaabM',
    0xa27e: 'aaaaaaba03',
    0xa27f: 'bbbbbbaL',
    0xa280: 'cccccai88',
    0xa281: 'ccccabbM',
    0xa282: 'ccccccba03',
    0xa283: 'ppppaabL',
    0xa284: 'eeeeeah88',
    0xa285: 'eeeeaaaL',
    0xa286: 'eeeeeebb13',
    0xa287: 'ffffbbbM',
    0xa288: 'llllllab03',
    0xa289: 'ooooabaL',
    0xa28a: 'lllllbh98',
    0xa28b: 'llllbabM',
    0xa28c: 'iiiiiah98',
    0xa28d: 'iiiiaabM',
    0xa28e: 'iiiiiibb03',
    0xa28f: 'jjjjbbaL',
    0xa290: 'xxxxxxab12',
    0xa291: 'kkkkbbaL',
    0xa292: 'xxxxxbh89',
    0xa293: 'hhhhaabM',
    0xa294: '8888888aabj',
    0xa295: 'mmmmbabM',
    0xa296: '8888888aabJ',
    0xa297: 'nnnnabaL',
    0xa298: 'ttttttab02',
    0xa299: 'ggggbbbM',
    0xa29a: 'tttttbh99',
    0xa29b: 'ddddaaaL',
    0xa29c: 'qqqqqah99',
    0xa29d: 'aaaabaaL',
    0xa29e: 'qqqqqqbb02',
    0xa29f: 'bbbbabbM',
    0xa2a0: 'sssssbi99',
    0xa2a1: 'ccccbaaL',
    0xa2a2: 'pppppah99',
    0xa2a3: 'ppppbbaM',
    0xa2a4: 'uuuuubh99',
    0xa2a5: 'eeeebbbM',
    0xa2a6: 'vvvvvai99',
    0xa2a7: 'ffffaaaL',
    0xa2a8: '?????bb08',
    0xa2a9: 'oooobabM',
    0xa2aa: '<<<<<aa18',
    0xa2ab: 'llllabaL',
    0xa2ac: 'yyyyybh89',
    0xa2ad: 'iiiibbaL',
    0xa2ae: 'zzzzzai89',
    0xa2af: 'jjjjaabM',
    0xa2b0: 'kkkkkbi98',
    0xa2b1: 'kkkkaabM',
    0xa2b2: 'hhhhhah98',
    0xa2b3: 'hhhhbbaL',
    0xa2b4: 'mmmmmbh98',
    0xa2b5: 'mmmmabaL',
    0xa2b6: 'nnnnnai98',
    0xa2b7: 'nnnnbabM',
    0xa2b8: 'gggggbi88',
    0xa2b9: 'ggggaaaL',
    0xa2ba: 'dddddah88',
    0xa2bb: 'ddddbbbM',
    0xa2bc: 'aaaaabh88',
    0xa2bd: 'aaaaabbM',
    0xa2be: 'bbbbbai88',
    0xa2bf: 'bbbbbaaL',
    0xa2c0: 'cccccbh88',
    0xa2c1: 'ccccaaaL',
    0xa2c2: 'CCCCCCaa10',
    0xa2c3: 'ppppabaM',
    0xa2c4: 'eeeeebi88',
    0xa2c5: 'eeeeabbM',
    0xa2c6: 'fffffah88',
    0xa2c7: 'ffffbaaL',
    0xa2c8: 'ooooobh98',
    0xa2c9: 'ooooaabM',
    0xa2ca: 'lllllai98',
    0xa2cb: 'llllbbaL',
    0xa2cc: 'iiiiibi98',
    0xa2cd: 'iiiiabaL',
    0xa2ce: 'jjjjjah98',
    0xa2cf: 'jjjjbabM',
    0xa2d0: 'XXXXXXbb01',
    0xa2d1: 'kkkkbabM',
    0xa2d2: 'xxxxxai89',
    0xa2d3: 'hhhhabaL',
    0xa2d4: 'XXXXXXXaabn',
    0xa2d5: 'mmmmbbaL',
    0xa2d6: 'XXXXXXXaabN',
    0xa2d7: 'nnnnaabM',
    0xa2d8: 'wwwwwbh99',
    0xa2d9: 'ggggbaaL',
    0xa2da: 'tttttai99',
    0xa2db: 'ddddabbM',
    0xa2dc: 'qqqqqbi99',
    0xa2dd: 'aaaabbbM',
    0xa2de: 'rrrrrah99',
    0xa2df: 'bbbbaaaL',
    0xa2e0: 'ppppppbb02',
    0xa2e1: 'ccccbbbM',
    0xa2e2: 'pppppbi99',
    0xa2e3: 'ppppbabL',
    0xa2e4: 'pppppppaabm',
    0xa2e5: 'eeeebaaL',
    0xa2e6: 'pppppppaabM',
    0xa2e7: 'ffffabbM',
    0xa2e8: 'yyyyyyybaaL',
    0xa2e9: 'oooobbaL',
    0xa2ea: 'yyyyyyybaal',
    0xa2eb: 'llllaabM',
    0xa2ec: 'yyyyyai89',
    0xa2ed: 'iiiibabM',
    0xa2ee: 'yyyyyyab12',
    0xa2ef: 'jjjjabaL',
    0xa2f0: 'hhhhhhbb03',
    0xa2f1: 'kkkkabaL',
    0xa2f2: 'hhhhhbi98',
    0xa2f3: 'hhhhbabM',
    0xa2f4: 'hhhhhhhaabl',
    0xa2f5: 'hhhhhhhaahy',
    0xa2f6: 'hhhhhhhaabL',
    0xa2f7: 'hhhhhhhaahY',
    0xa2f8: 'aaaaaaabaaM',
    0xa2f9: 'aaaaaaabaiY',
    0xa2fa: 'aaaaaaabaam',
    0xa2fb: 'aaaaaaabaiy',
    0xa2fc: 'aaaaaai88',
    0xa2fd: 'aaaaaaaL',
    0xa2fe: 'aaaaaaab13',
    0xa2ff: 'bbbbbbbM',
    0xa300: 'cccccah98',
    0xa301: 'ccccabaM',
    0xa302: 'ccccccaa03',
    0xa303: 'eeeeeeebbhy',
    0xa304: 'cccccccbaaM',
    0xa305: 'cccccccbaiY',
    0xa306: 'cccccccbaam',
    0xa307: 'cccccccbaiy',
    0xa308: 'iiiiiiibbaM',
    0xa309: 'iiiiiiibbiY',
    0xa30a: 'iiiiiiibbam',
    0xa30b: 'iiiiiiibbiy',
    0xa30c: 'iiiiiai88',
    0xa30d: 'iiiiaaaM',
    0xa30e: 'iiiiiiab03',
    0xa30f: 'jjjjbbbL',
    0xa310: 'xxxxxxbb12',
    0xa311: 'kkkkbbbL',
    0xa312: 'xxxxxbi99',
    0xa313: 'hhhhaaaM',
    0xa314: 'xxxxxxxabbm',
    0xa315: 'mmmmbaaM',
    0xa316: 'xxxxxxxabbM',
    0xa317: 'nnnnabbL',
    0xa318: 'qqqqqqqbbaL',
    0xa319: 'ggggbbaM',
    0xa31a: 'qqqqqqqbbal',
    0xa31b: 'ddddaabL',
    0xa31c: 'qqqqqai89',
    0xa31d: 'aaaababL',
    0xa31e: 'qqqqqqab02',
    0xa31f: 'bbbbabaM',
    0xa320: 'sssssbh89',
    0xa321: 'ccccbabL',
    0xa322: 'pppppai89',
    0xa323: 'ppppbbbM',
    0xa324: 'uuuuubi89',
    0xa325: 'eeeebbaM',
    0xa326: 'vvvvvah89',
    0xa327: 'ffffaabL',
    0xa328: 'YYYYYYYbbaO',
    0xa329: 'oooobaaM',
    0xa32a: 'YYYYYYYbbao',
    0xa32b: 'llllabbL',
    0xa32c: 'yyyyybi99',
    0xa32d: 'iiiibbbL',
    0xa32e: 'zzzzzah99',
    0xa32f: 'jjjjaaaM',
    0xa330: 'kkkkkbh88',
    0xa331: 'kkkkaaaM',
    0xa332: 'hhhhhai88',
    0xa333: 'hhhhbbbL',
    0xa334: 'mmmmmbi88',
    0xa335: 'mmmmabbL',
    0xa336: 'nnnnnah88',
    0xa337: 'nnnnbaaM',
    0xa338: 'gggggbh98',
    0xa339: 'ggggaabL',
    0xa33a: 'dddddai98',
    0xa33b: 'ddddbbaM',
    0xa33c: 'aaaaabi98',
    0xa33d: 'aaaaabaM',
    0xa33e: 'bbbbbah98',
    0xa33f: 'bbbbbabL',
    0xa340: 'cccccbi98',
    0xa341: 'ccccaabL',
    0xa342: 'CCCCCCba10',
    0xa343: 'ppppabbM',
    0xa344: 'eeeeebh98',
    0xa345: 'eeeeabaM',
    0xa346: 'fffffai98',
    0xa347: 'ffffbabL',
    0xa348: 'ooooobi88',
    0xa349: 'ooooaaaM',
    0xa34a: 'lllllah88',
    0xa34b: 'llllbbbL',
    0xa34c: 'iiiiibh88',
    0xa34d: 'iiiiabbL',
    0xa34e: 'jjjjjai88',
    0xa34f: 'jjjjbaaM',
    0xa350: 'XXXXXXab01',
    0xa351: 'kkkkbaaM',
    0xa352: 'xxxxxah99',
    0xa353: 'hhhhabbL',
    0xa354: '=====ba08',
    0xa355: 'mmmmbbbL',
    0xa356: '>>>>>ab18',
    0xa357: 'nnnnaaaM',
    0xa358: 'wwwwwbi89',
    0xa359: 'ggggbabL',
    0xa35a: 'tttttah89',
    0xa35b: 'ddddabaM',
    0xa35c: 'qqqqqbh89',
    0xa35d: 'aaaabbaM',
    0xa35e: 'rrrrrai89',
    0xa35f: 'bbbbaabL',
    0xa360: 'ppppppab02',
    0xa361: 'ccccbbaM',
    0xa362: 'pppppbh89',
    0xa363: 'ppppbaaL',
    0xa364: 'uuuuuah89',
    0xa365: 'eeeebabL',
    0xa366: 'uuuuuubb02',
    0xa367: 'ffffabaM',
    0xa368: '9999999bbaK',
    0xa369: 'oooobbbL',
    0xa36a: '9999999bbak',
    0xa36b: 'llllaaaM',
    0xa36c: 'yyyyyah99',
    0xa36d: 'iiiibaaM',
    0xa36e: 'yyyyyybb12',
    0xa36f: 'jjjjabbL',
    0xa370: 'hhhhhhab03',
    0xa371: 'kkkkabbL',
    0xa372: 'hhhhhbh88',
    0xa373: 'hhhhbaaM',
    0xa374: 'mmmmmah88',
    0xa375: 'mmmmaaaM',
    0xa376: 'mmmmmmbb03',
    0xa377: 'nnnnbbbL',
    0xa378: 'ddddddab13',
    0xa379: 'ggggabaM',
    0xa37a: 'dddddbh98',
    0xa37b: 'ddddbabL',
    0xa37c: 'aaaaaah98',
    0xa37d: 'aaaaaabL',
    0xa37e: 'aaaaaabb13',
    0xa37f: 'bbbbbbaM',
    0xa380: 'cccccai98',
    0xa381: 'ccccabbL',
    0xa382: 'ccccccbb13',
    0xa383: 'ppppaabM',
    0xa384: 'eeeeeah98',
    0xa385: 'eeeeaaaM',
    0xa386: 'eeeeeeba03',
    0xa387: 'ffffbbbL',
    0xa388: 'llllllaa13',
    0xa389: 'ooooabaM',
    0xa38a: 'lllllbh88',
    0xa38b: 'llllbabL',
    0xa38c: 'iiiiiah88',
    0xa38d: 'iiiiaabL',
    0xa38e: 'iiiiiiba13',
    0xa38f: 'jjjjbbaM',
    0xa390: 'xxxxxxaa02',
    0xa391: 'kkkkbbaM',
    0xa392: 'xxxxxbh99',
    0xa393: 'hhhhaabL',
    0xa394: '11111111bbbI',
    0xa395: 'mmmmbabL',
    0xa396: '11111111bbbY',
    0xa397: 'nnnnabaM',
    0xa398: 'ttttttaa12',
    0xa399: 'ggggbbbL',
    0xa39a: 'tttttbh89',
    0xa39b: 'ddddaaaM',
    0xa39c: 'qqqqqah89',
    0xa39d: 'aaaabaaM',
    0xa39e: 'qqqqqqba12',
    0xa39f: 'bbbbabbL',
    0xa3a0: 'sssssbi89',
    0xa3a1: 'ccccbaaM',
    0xa3a2: 'pppppah89',
    0xa3a3: 'ppppbbaL',
    0xa3a4: 'uuuuubh89',
    0xa3a5: 'eeeebbbL',
    0xa3a6: 'vvvvvai89',
    0xa3a7: 'ffffaaaM',
    0xa3a8: '?????bb18',
    0xa3a9: 'oooobabL',
    0xa3aa: '<<<<<aa08',
    0xa3ab: 'llllabaM',
    0xa3ac: 'yyyyybh99',
    0xa3ad: 'iiiibbaM',
    0xa3ae: 'zzzzzai99',
    0xa3af: 'jjjjaabL',
    0xa3b0: 'kkkkkbi88',
    0xa3b1: 'kkkkaabL',
    0xa3b2: 'hhhhhah88',
    0xa3b3: 'hhhhbbaM',
    0xa3b4: 'mmmmmbh88',
    0xa3b5: 'mmmmabaM',
    0xa3b6: 'nnnnnai88',
    0xa3b7: 'nnnnbabL',
    0xa3b8: 'gggggbi98',
    0xa3b9: 'ggggaaaM',
    0xa3ba: 'dddddah98',
    0xa3bb: 'ddddbbbL',
    0xa3bc: 'aaaaabh98',
    0xa3bd: 'aaaaabbL',
    0xa3be: 'bbbbbai98',
    0xa3bf: 'bbbbbaaM',
    0xa3c0: 'cccccbh98',
    0xa3c1: 'ccccaaaM',
    0xa3c2: 'CCCCCCab00',
    0xa3c3: 'ppppabaL',
    0xa3c4: 'eeeeebi98',
    0xa3c5: 'eeeeabbL',
    0xa3c6: 'fffffah98',
    0xa3c7: 'ffffbaaM',
    0xa3c8: 'ooooobh88',
    0xa3c9: 'ooooaabL',
    0xa3ca: 'lllllai88',
    0xa3cb: 'llllbbaM',
    0xa3cc: 'iiiiibi88',
    0xa3cd: 'iiiiabaM',
    0xa3ce: 'jjjjjah88',
    0xa3cf: 'jjjjbabL',
    0xa3d0: 'QQQQQQQQbbbm',
    0xa3d1: 'kkkkbabL',
    0xa3d2: 'xxxxxai99',
    0xa3d3: 'hhhhabaM',
    0xa3d4: 'QQQQQQQQbbbM',
    0xa3d5: 'mmmmbbaM',
    0xa3d6: 'QQQQQQQQbbhX',
    0xa3d7: 'nnnnaabL',
    0xa3d8: 'wwwwwbh89',
    0xa3d9: 'ggggbaaM',
    0xa3da: 'tttttai89',
    0xa3db: 'ddddabbL',
    0xa3dc: 'qqqqqbi89',
    0xa3dd: 'aaaabbbL',
    0xa3de: 'rrrrrah89',
    0xa3df: 'bbbbaaaM',
    0xa3e0: 'ppppppba12',
    0xa3e1: 'ccccbbbL',
    0xa3e2: 'pppppbi89',
    0xa3e3: 'ppppbabM',
    0xa3e4: 'pppppppbbal',
    0xa3e5: 'eeeebaaM',
    0xa3e6: 'pppppppbbaL',
    0xa3e7: 'ffffabbL',
    0xa3e8: 'ppppppppabkZ',
    0xa3e9: 'oooobbaM',
    0xa3ea: 'ppppppppabaO',
    0xa3eb: 'llllaabL',
    0xa3ec: 'ppppppppabkz',
    0xa3ed: 'iiiibabL',
    0xa3ee: 'ppppppppabao',
    0xa3ef: 'jjjjabaM',
    0xa3f0: 'aaaaaaaabbbo',
    0xa3f1: 'aaaaaaaabbbg',
    0xa3f2: 'aaaaaaaabbhz',
    0xa3f3: 'aaaaaaaabbbw',
    0xa3f4: 'aaaaaaaabbbO',
    0xa3f5: 'aaaaaaaabbbG',
    0xa3f6: 'aaaaaaaabbhZ',
    0xa3f7: 'aaaaaaaabbbW',
    0xa3f8: 'aaaaaaaabbL',
    0xa3f9: 'aaaaaaaabhY',
    0xa3fa: 'aaaaaaaabbl',
    0xa3fb: 'aaaaaaaabhy',
    0xa3fc: 'aaaaaai98',
    0xa3fd: 'aaaaaaaM',
    0xa3fe: 'aaaaaaaa03',
    0xa3ff: 'aaaaaaaad9',
    0xa400: 'cccccab99',
    0xa401: 'ccccabaJ',
    0xa402: 'ccccccab02',
    0xa403: 'ccccccccd8',
    0xa404: 'cccccccaabL',
    0xa405: 'cccccccaahY',
    0xa406: 'cccccccaabl',
    0xa407: 'cccccccaahy',
    0xa408: 'ccccccccbaaN',
    0xa409: 'ccccccccbaaF',
    0xa40a: 'ccccccccbaiZ',
    0xa40b: 'ccccccccbaaV',
    0xa40c: 'ccccccccbaan',
    0xa40d: 'ccccccccbaaf',
    0xa40e: 'ccccccccbaiz',
    0xa40f: 'ccccccccbaav',
    0xa410: 'qqqqqqqqbbao',
    0xa411: 'kkkkbbbK',
    0xa412: 'qqqqqqqqbbkz',
    0xa413: 'hhhhaaaJ',
    0xa414: 'qqqqqqqqbbaO',
    0xa415: 'mmmmbaaJ',
    0xa416: 'qqqqqqqqbbkZ',
    0xa417: 'nnnnabbK',
    0xa418: 'qqqqqqqabbM',
    0xa419: 'ggggbbaJ',
    0xa41a: 'qqqqqqqabbm',
    0xa41b: 'ddddaabK',
    0xa41c: 'qqqqqaa98',
    0xa41d: 'aaaababK',
    0xa41e: 'qqqqqqaa03',
    0xa41f: 'bbbbabaJ',
    0xa420: 'sssssbb88',
    0xa421: 'ccccbabK',
    0xa422: 'pppppaa98',
    0xa423: 'ppppbbbJ',
    0xa424: 'uuuuuba98',
    0xa425: 'eeeebbaJ',
    0xa426: 'vvvvvab88',
    0xa427: 'ffffaabK',
    0xa428: 'PPPPPPPPabhX',
    0xa429: 'oooobaaJ',
    0xa42a: 'PPPPPPPPabbM',
    0xa42b: 'llllabbK',
    0xa42c: 'yyyyyba88',
    0xa42d: 'iiiibbbK',
    0xa42e: 'zzzzzab98',
    0xa42f: 'jjjjaaaJ',
    0xa430: 'kkkkkbb89',
    0xa431: 'kkkkaaaJ',
    0xa432: 'hhhhhaa99',
    0xa433: 'hhhhbbbK',
    0xa434: 'mmmmmba99',
    0xa435: 'mmmmabbK',
    0xa436: 'nnnnnab89',
    0xa437: 'nnnnbaaJ',
    0xa438: 'gggggbb99',
    0xa439: 'ggggaabK',
    0xa43a: 'dddddaa89',
    0xa43b: 'ddddbbaJ',
    0xa43c: 'aaaaaba89',
    0xa43d: 'aaaaabaJ',
    0xa43e: 'bbbbbab99',
    0xa43f: 'bbbbbabK',
    0xa440: 'cccccba89',
    0xa441: 'ccccaabK',
    0xa442: 'CCCCCCbb11',
    0xa443: 'ppppabbJ',
    0xa444: 'eeeeebb99',
    0xa445: 'eeeeabaJ',
    0xa446: 'fffffaa89',
    0xa447: 'ffffbabK',
    0xa448: 'oooooba99',
    0xa449: 'ooooaaaJ',
    0xa44a: 'lllllab89',
    0xa44b: 'llllbbbK',
    0xa44c: 'iiiiibb89',
    0xa44d: 'iiiiabbK',
    0xa44e: 'jjjjjaa99',
    0xa44f: 'jjjjbaaJ',
    0xa450: 'XXXXXXaa00',
    0xa451: 'kkkkbaaJ',
    0xa452: 'xxxxxab98',
    0xa453: 'hhhhabbK',
    0xa454: '=====ba59',
    0xa455: 'mmmmbbbK',
    0xa456: '>>>>>ab49',
    0xa457: 'nnnnaaaJ',
    0xa458: 'wwwwwba98',
    0xa459: 'ggggbabK',
    0xa45a: 'tttttab88',
    0xa45b: 'ddddabaJ',
    0xa45c: 'qqqqqbb88',
    0xa45d: 'aaaabbaJ',
    0xa45e: 'rrrrraa98',
    0xa45f: 'bbbbaabK',
    0xa460: 'ppppppaa03',
    0xa461: 'ccccbbaJ',
    0xa462: 'pppppbb88',
    0xa463: 'ppppbaaK',
    0xa464: 'uuuuuab88',
    0xa465: 'eeeebabK',
    0xa466: 'uuuuuuba03',
    0xa467: 'ffffabaJ',
    0xa468: '00000000abbY',
    0xa469: 'oooobbbK',
    0xa46a: '00000000abbI',
    0xa46b: 'llllaaaJ',
    0xa46c: 'yyyyyab98',
    0xa46d: 'iiiibaaJ',
    0xa46e: 'yyyyyyba13',
    0xa46f: 'jjjjabbK',
    0xa470: 'hhhhhhaa02',
    0xa471: 'kkkkabbK',
    0xa472: 'hhhhhbb89',
    0xa473: 'hhhhbaaJ',
    0xa474: 'mmmmmab89',
    0xa475: 'mmmmaaaJ',
    0xa476: 'mmmmmmba02',
    0xa477: 'nnnnbbbK',
    0xa478: 'ddddddaa12',
    0xa479: 'ggggabaJ',
    0xa47a: 'dddddbb99',
    0xa47b: 'ddddbabK',
    0xa47c: 'aaaaaab99',
    0xa47d: 'aaaaaabK',
    0xa47e: 'aaaaaaba12',
    0xa47f: 'bbbbbbaJ',
    0xa480: 'cccccaa89',
    0xa481: 'ccccabbK',
    0xa482: 'ccccccba12',
    0xa483: 'ppppaabJ',
    0xa484: 'eeeeeab99',
    0xa485: 'eeeeaaaJ',
    0xa486: 'eeeeeebb02',
    0xa487: 'ffffbbbK',
    0xa488: 'llllllab12',
    0xa489: 'ooooabaJ',
    0xa48a: 'lllllbb89',
    0xa48b: 'llllbabK',
    0xa48c: 'iiiiiab89',
    0xa48d: 'iiiiaabK',
    0xa48e: 'iiiiiibb12',
    0xa48f: 'jjjjbbaJ',
    0xa490: 'xxxxxxab03',
    0xa491: 'kkkkbbaJ',
    0xa492: 'xxxxxbb98',
    0xa493: 'hhhhaabK',
    0xa494: '8888888abbj',
    0xa495: 'mmmmbabK',
    0xa496: '8888888abbJ',
    0xa497: 'nnnnabaJ',
    0xa498: 'ttttttab13',
    0xa499: 'ggggbbbK',
    0xa49a: 'tttttbb88',
    0xa49b: 'ddddaaaJ',
    0xa49c: 'qqqqqab88',
    0xa49d: 'aaaabaaJ',
    0xa49e: 'qqqqqqbb13',
    0xa49f: 'bbbbabbK',
    0xa4a0: 'sssssba98',
    0xa4a1: 'ccccbaaJ',
    0xa4a2: 'pppppab88',
    0xa4a3: 'ppppbbaK',
    0xa4a4: 'uuuuubb88',
    0xa4a5: 'eeeebbbK',
    0xa4a6: 'vvvvvaa98',
    0xa4a7: 'ffffaaaJ',
    0xa4a8: '?????bb49',
    0xa4a9: 'oooobabK',
    0xa4aa: '<<<<<aa59',
    0xa4ab: 'llllabaJ',
    0xa4ac: 'yyyyybb98',
    0xa4ad: 'iiiibbaJ',
    0xa4ae: 'zzzzzaa88',
    0xa4af: 'jjjjaabK',
    0xa4b0: 'kkkkkba99',
    0xa4b1: 'kkkkaabK',
    0xa4b2: 'hhhhhab89',
    0xa4b3: 'hhhhbbaJ',
    0xa4b4: 'mmmmmbb89',
    0xa4b5: 'mmmmabaJ',
    0xa4b6: 'nnnnnaa99',
    0xa4b7: 'nnnnbabK',
    0xa4b8: 'gggggba89',
    0xa4b9: 'ggggaaaJ',
    0xa4ba: 'dddddab99',
    0xa4bb: 'ddddbbbK',
    0xa4bc: 'aaaaabb99',
    0xa4bd: 'aaaaabbK',
    0xa4be: 'bbbbbaa89',
    0xa4bf: 'bbbbbaaJ',
    0xa4c0: 'cccccbb99',
    0xa4c1: 'ccccaaaJ',
    0xa4c2: 'CCCCCCaa01',
    0xa4c3: 'ppppabaK',
    0xa4c4: 'eeeeeba89',
    0xa4c5: 'eeeeabbK',
    0xa4c6: 'fffffab99',
    0xa4c7: 'ffffbaaJ',
    0xa4c8: 'ooooobb89',
    0xa4c9: 'ooooaabK',
    0xa4ca: 'lllllaa99',
    0xa4cb: 'llllbbaJ',
    0xa4cc: 'iiiiiba99',
    0xa4cd: 'iiiiabaJ',
    0xa4ce: 'jjjjjab89',
    0xa4cf: 'jjjjbabK',
    0xa4d0: 'XXXXXXbb10',
    0xa4d1: 'kkkkbabK',
    0xa4d2: 'xxxxxaa88',
    0xa4d3: 'hhhhabaJ',
    0xa4d4: 'XXXXXXXabbn',
    0xa4d5: 'mmmmbbaJ',
    0xa4d6: 'XXXXXXXabbN',
    0xa4d7: 'nnnnaabK',
    0xa4d8: 'wwwwwbb88',
    0xa4d9: 'ggggbaaJ',
    0xa4da: 'tttttaa98',
    0xa4db: 'ddddabbK',
    0xa4dc: 'qqqqqba98',
    0xa4dd: 'aaaabbbK',
    0xa4de: 'rrrrrab88',
    0xa4df: 'bbbbaaaJ',
    0xa4e0: 'ppppppbb13',
    0xa4e1: 'ccccbbbK',
    0xa4e2: 'pppppba98',
    0xa4e3: 'ppppbabJ',
    0xa4e4: 'pppppppabbm',
    0xa4e5: 'eeeebaaJ',
    0xa4e6: 'pppppppabbM',
    0xa4e7: 'ffffabbK',
    0xa4e8: 'yyyyyyybbaL',
    0xa4e9: 'oooobbaJ',
    0xa4ea: 'yyyyyyybbal',
    0xa4eb: 'llllaabK',
    0xa4ec: 'yyyyyaa88',
    0xa4ed: 'iiiibabK',
    0xa4ee: 'yyyyyyab03',
    0xa4ef: 'jjjjabaJ',
    0xa4f0: 'hhhhhhbb12',
    0xa4f1: 'kkkkabaJ',
    0xa4f2: 'hhhhhba99',
    0xa4f3: 'hhhhbabK',
    0xa4f4: 'hhhhhhhabbl',
    0xa4f5: 'hhhhhhhabhy',
    0xa4f6: 'hhhhhhhabbL',
    0xa4f7: 'hhhhhhhabhY',
    0xa4f8: 'aaaaaaabbaM',
    0xa4f9: 'aaaaaaabbiY',
    0xa4fa: 'aaaaaaabbam',
    0xa4fb: 'aaaaaaabbiy',
    0xa4fc: 'aaaaaaa89',
    0xa4fd: 'aaaaaaaJ',
    0xa4fe: 'aaaaaaab02',
    0xa4ff: 'bbbbbbbK',
    0xa500: 'cccccab89',
    0xa501: 'ccccabaK',
    0xa502: 'ccccccaa12',
    0xa503: 'eeeeeeebahy',
    0xa504: 'cccccccbbaM',
    0xa505: 'cccccccbbiY',
    0xa506: 'cccccccbbam',
    0xa507: 'cccccccbbiy',
    0xa508: 'iiiiiiibaaM',
    0xa509: 'iiiiiiibaiY',
    0xa50a: 'iiiiiiibaam',
    0xa50b: 'iiiiiiibaiy',
    0xa50c: 'iiiiiaa89',
    0xa50d: 'iiiiaaaK',
    0xa50e: 'iiiiiiab12',
    0xa50f: 'jjjjbbbJ',
    0xa510: 'xxxxxxbb03',
    0xa511: 'kkkkbbbJ',
    0xa512: 'xxxxxba98',
    0xa513: 'hhhhaaaK',
    0xa514: 'xxxxxxxaabm',
    0xa515: 'mmmmbaaK',
    0xa516: 'xxxxxxxaabM',
    0xa517: 'nnnnabbJ',
    0xa518: 'qqqqqqqbaaL',
    0xa519: 'ggggbbaK',
    0xa51a: 'qqqqqqqbaal',
    0xa51b: 'ddddaabJ',
    0xa51c: 'qqqqqaa88',
    0xa51d: 'aaaababJ',
    0xa51e: 'qqqqqqab13',
    0xa51f: 'bbbbabaK',
    0xa520: 'sssssbb98',
    0xa521: 'ccccbabJ',
    0xa522: 'pppppaa88',
    0xa523: 'ppppbbbK',
    0xa524: 'uuuuuba88',
    0xa525: 'eeeebbaK',
    0xa526: 'vvvvvab98',
    0xa527: 'ffffaabJ',
    0xa528: 'YYYYYYYbaaO',
    0xa529: 'oooobaaK',
    0xa52a: 'YYYYYYYbaao',
    0xa52b: 'llllabbJ',
    0xa52c: 'yyyyyba98',
    0xa52d: 'iiiibbbJ',
    0xa52e: 'zzzzzab88',
    0xa52f: 'jjjjaaaK',
    0xa530: 'kkkkkbb99',
    0xa531: 'kkkkaaaK',
    0xa532: 'hhhhhaa89',
    0xa533: 'hhhhbbbJ',
    0xa534: 'mmmmmba89',
    0xa535: 'mmmmabbJ',
    0xa536: 'nnnnnab99',
    0xa537: 'nnnnbaaK',
    0xa538: 'gggggbb89',
    0xa539: 'ggggaabJ',
    0xa53a: 'dddddaa99',
    0xa53b: 'ddddbbaK',
    0xa53c: 'aaaaaba99',
    0xa53d: 'aaaaabaK',
    0xa53e: 'bbbbbab89',
    0xa53f: 'bbbbbabJ',
    0xa540: 'cccccba99',
    0xa541: 'ccccaabJ',
    0xa542: 'CCCCCCba01',
    0xa543: 'ppppabbK',
    0xa544: 'eeeeebb89',
    0xa545: 'eeeeabaK',
    0xa546: 'fffffaa99',
    0xa547: 'ffffbabJ',
    0xa548: 'oooooba89',
    0xa549: 'ooooaaaK',
    0xa54a: 'lllllab99',
    0xa54b: 'llllbbbJ',
    0xa54c: 'iiiiibb99',
    0xa54d: 'iiiiabbJ',
    0xa54e: 'jjjjjaa89',
    0xa54f: 'jjjjbaaK',
    0xa550: 'XXXXXXab10',
    0xa551: 'kkkkbaaK',
    0xa552: 'xxxxxab88',
    0xa553: 'hhhhabbJ',
    0xa554: '=====ba49',
    0xa555: 'mmmmbbbJ',
    0xa556: '>>>>>ab59',
    0xa557: 'nnnnaaaK',
    0xa558: 'wwwwwba88',
    0xa559: 'ggggbabJ',
    0xa55a: 'tttttab98',
    0xa55b: 'ddddabaK',
    0xa55c: 'qqqqqbb98',
    0xa55d: 'aaaabbaK',
    0xa55e: 'rrrrraa88',
    0xa55f: 'bbbbaabJ',
    0xa560: 'ppppppab13',
    0xa561: 'ccccbbaK',
    0xa562: 'pppppbb98',
    0xa563: 'ppppbaaJ',
    0xa564: 'uuuuuab98',
    0xa565: 'eeeebabJ',
    0xa566: 'uuuuuubb13',
    0xa567: 'ffffabaK',
    0xa568: '9999999baaK',
    0xa569: 'oooobbbJ',
    0xa56a: '9999999baak',
    0xa56b: 'llllaaaK',
    0xa56c: 'yyyyyab88',
    0xa56d: 'iiiibaaK',
    0xa56e: 'yyyyyybb03',
    0xa56f: 'jjjjabbJ',
    0xa570: 'hhhhhhab12',
    0xa571: 'kkkkabbJ',
    0xa572: 'hhhhhbb99',
    0xa573: 'hhhhbaaK',
    0xa574: 'mmmmmab99',
    0xa575: 'mmmmaaaK',
    0xa576: 'mmmmmmbb12',
    0xa577: 'nnnnbbbJ',
    0xa578: 'ddddddab02',
    0xa579: 'ggggabaK',
    0xa57a: 'dddddbb89',
    0xa57b: 'ddddbabJ',
    0xa57c: 'aaaaaab89',
    0xa57d: 'aaaaaabJ',
    0xa57e: 'aaaaaabb02',
    0xa57f: 'bbbbbbaK',
    0xa580: 'cccccaa99',
    0xa581: 'ccccabbJ',
    0xa582: 'ccccccbb02',
    0xa583: 'ppppaabK',
    0xa584: 'eeeeeab89',
    0xa585: 'eeeeaaaK',
    0xa586: 'eeeeeeba12',
    0xa587: 'ffffbbbJ',
    0xa588: 'llllllaa02',
    0xa589: 'ooooabaK',
    0xa58a: 'lllllbb99',
    0xa58b: 'llllbabJ',
    0xa58c: 'iiiiiab99',
    0xa58d: 'iiiiaabJ',
    0xa58e: 'iiiiiiba02',
    0xa58f: 'jjjjbbaK',
    0xa590: 'xxxxxxaa13',
    0xa591: 'kkkkbbaK',
    0xa592: 'xxxxxbb88',
    0xa593: 'hhhhaabJ',
    0xa594: '11111111abbI',
    0xa595: 'mmmmbabJ',
    0xa596: '11111111abbY',
    0xa597: 'nnnnabaK',
    0xa598: 'ttttttaa03',
    0xa599: 'ggggbbbJ',
    0xa59a: 'tttttbb98',
    0xa59b: 'ddddaaaK',
    0xa59c: 'qqqqqab98',
    0xa59d: 'aaaabaaK',
    0xa59e: 'qqqqqqba03',
    0xa59f: 'bbbbabbJ',
    0xa5a0: 'sssssba88',
    0xa5a1: 'ccccbaaK',
    0xa5a2: 'pppppab98',
    0xa5a3: 'ppppbbaJ',
    0xa5a4: 'uuuuubb98',
    0xa5a5: 'eeeebbbJ',
    0xa5a6: 'vvvvvaa88',
    0xa5a7: 'ffffaaaK',
    0xa5a8: '?????bb59',
    0xa5a9: 'oooobabJ',
    0xa5aa: '<<<<<aa49',
    0xa5ab: 'llllabaK',
    0xa5ac: 'yyyyybb88',
    0xa5ad: 'iiiibbaK',
    0xa5ae: 'zzzzzaa98',
    0xa5af: 'jjjjaabJ',
    0xa5b0: 'kkkkkba89',
    0xa5b1: 'kkkkaabJ',
    0xa5b2: 'hhhhhab99',
    0xa5b3: 'hhhhbbaK',
    0xa5b4: 'mmmmmbb99',
    0xa5b5: 'mmmmabaK',
    0xa5b6: 'nnnnnaa89',
    0xa5b7: 'nnnnbabJ',
    0xa5b8: 'gggggba99',
    0xa5b9: 'ggggaaaK',
    0xa5ba: 'dddddab89',
    0xa5bb: 'ddddbbbJ',
    0xa5bc: 'aaaaabb89',
    0xa5bd: 'aaaaabbJ',
    0xa5be: 'bbbbbaa99',
    0xa5bf: 'bbbbbaaK',
    0xa5c0: 'cccccbb89',
    0xa5c1: 'ccccaaaK',
    0xa5c2: 'CCCCCCab11',
    0xa5c3: 'ppppabaJ',
    0xa5c4: 'eeeeeba99',
    0xa5c5: 'eeeeabbJ',
    0xa5c6: 'fffffab89',
    0xa5c7: 'ffffbaaK',
    0xa5c8: 'ooooobb99',
    0xa5c9: 'ooooaabJ',
    0xa5ca: 'lllllaa89',
    0xa5cb: 'llllbbaK',
    0xa5cc: 'iiiiiba89',
    0xa5cd: 'iiiiabaK',
    0xa5ce: 'jjjjjab99',
    0xa5cf: 'jjjjbabJ',
    0xa5d0: 'CCCCCCCCCbaap',
    0xa5d1: 'kkkkbabJ',
    0xa5d2: 'xxxxxaa98',
    0xa5d3: 'hhhhabaK',
    0xa5d4: 'CCCCCCCCCbaca',
    0xa5d5: 'mmmmbbaK',
    0xa5d6: 'CCCCCCCCCbaah',
    0xa5d7: 'nnnnaabJ',
    0xa5d8: 'wwwwwbb98',
    0xa5d9: 'ggggbaaK',
    0xa5da: 'tttttaa88',
    0xa5db: 'ddddabbJ',
    0xa5dc: 'qqqqqba88',
    0xa5dd: 'aaaabbbJ',
    0xa5de: 'rrrrrab98',
    0xa5df: 'bbbbaaaK',
    0xa5e0: 'aaaaaaaaabbaC',
    0xa5e1: 'aaaaaaaaabbaG',
    0xa5e2: 'aaaaaaaaabbaK',
    0xa5e3: 'aaaaaaaaabbaO',
    0xa5e4: 'aaaaaaaaabbaS',
    0xa5e5: 'aaaaaaaaabbaW',
    0xa5e6: 'aaaaaaaaabbcZ',
    0xa5e7: 'aaaaaaaaabbkZ',
    0xa5e8: 'aaaaaaaaabbac',
    0xa5e9: 'aaaaaaaaabbag',
    0xa5ea: 'aaaaaaaaabbak',
    0xa5eb: 'aaaaaaaaabbao',
    0xa5ec: 'aaaaaaaaabbas',
    0xa5ed: 'aaaaaaaaabbaw',
    0xa5ee: 'aaaaaaaaabbcz',
    0xa5ef: 'aaaaaaaaabbkz',
    0xa5f0: 'aaaaaaaaabbo',
    0xa5f1: 'aaaaaaaaabbg',
    0xa5f2: 'aaaaaaaaabhz',
    0xa5f3: 'aaaaaaaaabbw',
    0xa5f4: 'aaaaaaaaabbO',
    0xa5f5: 'aaaaaaaaabbG',
    0xa5f6: 'aaaaaaaaabhZ',
    0xa5f7: 'aaaaaaaaabbW',
    0xa5f8: 'aaaaaaaaabL',
    0xa5f9: 'aaaaaaaaahY',
    0xa5fa: 'aaaaaaaaabl',
    0xa5fb: 'aaaaaaaaahy',
    0xa5fc: 'aaaaaaa99',
    0xa5fd: 'aaaaaaaK',
    0xa5fe: 'aaaaaaaa12',
    0xa5ff: 'aaaaaaaae8',
    0xa600: 'cccccab98',
    0xa601: 'ccccabaH',
    0xa602: 'ccccccab12',
    0xa603: 'cccccccce8',
    0xa604: 'cccccccabaM',
    0xa605: 'cccccccabiY',
    0xa606: 'cccccccabam',
    0xa607: 'cccccccabiy',
    0xa608: 'ccccccccabbO',
    0xa609: 'ccccccccabbG',
    0xa60a: 'ccccccccabhZ',
    0xa60b: 'ccccccccabbW',
    0xa60c: 'ccccccccabbo',
    0xa60d: 'ccccccccabbg',
    0xa60e: 'ccccccccabhz',
    0xa60f: 'ccccccccabbw',
    0xa610: 'cccccccccaabr',
    0xa611: 'cccccccccaabv',
    0xa612: 'cccccccccaabz',
    0xa613: 'cccccccccaajz',
    0xa614: 'cccccccccaabb',
    0xa615: 'cccccccccaabf',
    0xa616: 'cccccccccaabj',
    0xa617: 'cccccccccaabn',
    0xa618: 'cccccccccaabR',
    0xa619: 'cccccccccaabV',
    0xa61a: 'cccccccccaabZ',
    0xa61b: 'cccccccccaajZ',
    0xa61c: 'cccccccccaabB',
    0xa61d: 'aaaababI',
    0xa61e: 'cccccccccaabJ',
    0xa61f: 'bbbbabaH',
    0xa620: 'ccccccccccbaao',
    0xa621: 'ccccbabI',
    0xa622: 'ccccccccccbaak',
    0xa623: 'ccccccccccbaai',
    0xa624: 'ccccccccccbaag',
    0xa625: 'ccccccccccbaae',
    0xa626: 'ccccccccccbaac',
    0xa627: 'ccccccccccbaaa',
    0xa628: 'ccccccccccbakz',
    0xa629: 'ccccccccccbaiy',
    0xa62a: 'ccccccccccbacz',
    0xa62b: 'ccccccccccbaay',
    0xa62c: 'ccccccccccbaaw',
    0xa62d: 'ccccccccccbaau',
    0xa62e: 'ccccccccccbaas',
    0xa62f: 'ccccccccccbaaq',
    0xa630: 'ccccccccccbaaO',
    0xa631: 'ccccccccccbaaM',
    0xa632: 'ccccccccccbaaK',
    0xa633: 'ccccccccccbaaI',
    0xa634: 'ccccccccccbaaG',
    0xa635: 'ccccccccccbaaE',
    0xa636: 'ccccccccccbaaC',
    0xa637: 'ccccccccccbaaA',
    0xa638: 'ccccccccccbakZ',
    0xa639: 'ccccccccccbaiY',
    0xa63a: 'ccccccccccbacZ',
    0xa63b: 'ccccccccccbaaY',
    0xa63c: 'aaaaaba88',
    0xa63d: 'aaaaabaH',
    0xa63e: 'bbbbbab98',
    0xa63f: 'bbbbbabI',
    0xa640: 'cccccba88',
    0xa641: 'ccccaabI',
    0xa642: 'CCCCCCbb01',
    0xa643: 'ppppabbH',
    0xa644: 'eeeeebb98',
    0xa645: 'eeeeabaH',
    0xa646: 'fffffaa88',
    0xa647: 'ffffbabI',
    0xa648: 'oooooba98',
    0xa649: 'ooooaaaH',
    0xa64a: 'lllllab88',
    0xa64b: 'llllbbbI',
    0xa64c: 'iiiiibb88',
    0xa64d: 'iiiiabbI',
    0xa64e: 'jjjjjaa98',
    0xa64f: 'jjjjbaaH',
    0xa650: 'XXXXXXaa10',
    0xa651: 'kkkkbaaH',
    0xa652: 'xxxxxab99',
    0xa653: 'hhhhabbI',
    0xa654: '0000000000abbD',
    0xa655: 'mmmmbbbI',
    0xa656: '0000000000abdC',
    0xa657: 'nnnnaaaH',
    0xa658: 'wwwwwba99',
    0xa659: 'ggggbabI',
    0xa65a: 'tttttab89',
    0xa65b: 'ddddabaH',
    0xa65c: 'qqqqqbb89',
    0xa65d: 'aaaabbaH',
    0xa65e: 'rrrrraa99',
    0xa65f: 'bbbbaabI',
    0xa660: 'ppppppaa13',
    0xa661: 'ccccbbaH',
    0xa662: 'pppppbb89',
    0xa663: 'ppppbaaI',
    0xa664: 'uuuuuab89',
    0xa665: 'eeeebabI',
    0xa666: 'uuuuuuba13',
    0xa667: 'ffffabaH',
    0xa668: '00000000baaX',
    0xa669: 'oooobbbI',
    0xa66a: '00000000baaH',
    0xa66b: 'llllaaaH',
    0xa66c: 'yyyyyab99',
    0xa66d: 'iiiibaaH',
    0xa66e: 'yyyyyyba03',
    0xa66f: 'jjjjabbI',
    0xa670: 'hhhhhhaa12',
    0xa671: 'kkkkabbI',
    0xa672: 'hhhhhbb88',
    0xa673: 'hhhhbaaH',
    0xa674: 'mmmmmab88',
    0xa675: 'mmmmaaaH',
    0xa676: 'mmmmmmba12',
    0xa677: 'nnnnbbbI',
    0xa678: 'ddddddaa02',
    0xa679: 'ggggabaH',
    0xa67a: 'dddddbb98',
    0xa67b: 'ddddbabI',
    0xa67c: 'aaaaaab98',
    0xa67d: 'aaaaaabI',
    0xa67e: 'aaaaaaba02',
    0xa67f: 'bbbbbbaH',
    0xa680: 'cccccaa88',
    0xa681: 'ccccabbI',
    0xa682: 'ccccccba02',
    0xa683: 'ppppaabH',
    0xa684: 'eeeeeab98',
    0xa685: 'eeeeaaaH',
    0xa686: 'eeeeeebb12',
    0xa687: 'ffffbbbI',
    0xa688: 'llllllab02',
    0xa689: 'ooooabaH',
    0xa68a: 'lllllbb88',
    0xa68b: 'llllbabI',
    0xa68c: 'iiiiiab88',
    0xa68d: 'iiiiaabI',
    0xa68e: 'iiiiiibb02',
    0xa68f: 'jjjjbbaH',
    0xa690: 'xxxxxxab13',
    0xa691: 'kkkkbbaH',
    0xa692: 'xxxxxbb99',
    0xa693: 'hhhhaabI',
    0xa694: '8888888aaak',
    0xa695: 'mmmmbabI',
    0xa696: '8888888aaaK',
    0xa697: 'nnnnabaH',
    0xa698: 'ttttttab03',
    0xa699: 'ggggbbbI',
    0xa69a: 'tttttbb89',
    0xa69b: 'ddddaaaH',
    0xa69c: 'qqqqqab89',
    0xa69d: 'aaaabaaH',
    0xa69e: 'qqqqqqbb03',
    0xa69f: 'bbbbabbI',
    0xa6a0: 'sssssba99',
    0xa6a1: 'ccccbaaH',
    0xa6a2: 'pppppab89',
    0xa6a3: 'ppppbbaI',
    0xa6a4: 'uuuuubb89',
    0xa6a5: 'eeeebbbI',
    0xa6a6: 'vvvvvaa99',
    0xa6a7: 'ffffaaaH',
    0xa6a8: '?????bb48',
    0xa6a9: 'oooobabI',
    0xa6aa: '<<<<<aa58',
    0xa6ab: 'llllabaH',
    0xa6ac: 'yyyyybb99',
    0xa6ad: 'iiiibbaH',
    0xa6ae: 'zzzzzaa89',
    0xa6af: 'jjjjaabI',
    0xa6b0: 'kkkkkba98',
    0xa6b1: 'kkkkaabI',
    0xa6b2: 'hhhhhab88',
    0xa6b3: 'hhhhbbaH',
    0xa6b4: 'mmmmmbb88',
    0xa6b5: 'mmmmabaH',
    0xa6b6: 'nnnnnaa98',
    0xa6b7: 'nnnnbabI',
    0xa6b8: 'gggggba88',
    0xa6b9: 'ggggaaaH',
    0xa6ba: 'dddddab98',
    0xa6bb: 'ddddbbbI',
    0xa6bc: 'aaaaabb98',
    0xa6bd: 'aaaaabbI',
    0xa6be: 'bbbbbaa88',
    0xa6bf: 'bbbbbaaH',
    0xa6c0: 'cccccbb98',
    0xa6c1: 'ccccaaaH',
    0xa6c2: 'CCCCCCaa11',
    0xa6c3: 'ppppabaI',
    0xa6c4: 'eeeeeba88',
    0xa6c5: 'eeeeabbI',
    0xa6c6: 'fffffab98',
    0xa6c7: 'ffffbaaH',
    0xa6c8: 'ooooobb88',
    0xa6c9: 'ooooaabI',
    0xa6ca: 'lllllaa98',
    0xa6cb: 'llllbbaH',
    0xa6cc: 'iiiiiba98',
    0xa6cd: 'iiiiabaH',
    0xa6ce: 'jjjjjab88',
    0xa6cf: 'jjjjbabI',
    0xa6d0: 'XXXXXXbb00',
    0xa6d1: 'kkkkbabI',
    0xa6d2: 'xxxxxaa89',
    0xa6d3: 'hhhhabaH',
    0xa6d4: 'XXXXXXXaaao',
    0xa6d5: 'mmmmbbaH',
    0xa6d6: 'XXXXXXXaaaO',
    0xa6d7: 'nnnnaabI',
    0xa6d8: 'wwwwwbb89',
    0xa6d9: 'ggggbaaH',
    0xa6da: 'tttttaa99',
    0xa6db: 'ddddabbI',
    0xa6dc: 'qqqqqba99',
    0xa6dd: 'aaaabbbI',
    0xa6de: 'rrrrrab89',
    0xa6df: 'bbbbaaaH',
    0xa6e0: 'ppppppbb03',
    0xa6e1: 'ccccbbbI',
    0xa6e2: 'pppppba99',
    0xa6e3: 'ppppbabH',
    0xa6e4: 'pppppppaaal',
    0xa6e5: 'eeeebaaH',
    0xa6e6: 'pppppppaaaL',
    0xa6e7: 'ffffabbI',
    0xa6e8: 'yyyyyyybabM',
    0xa6e9: 'oooobbaH',
    0xa6ea: 'yyyyyyybabm',
    0xa6eb: 'llllaabI',
    0xa6ec: 'yyyyyaa89',
    0xa6ed: 'iiiibabI',
    0xa6ee: 'yyyyyyab13',
    0xa6ef: 'jjjjabaH',
    0xa6f0: 'hhhhhhbb02',
    0xa6f1: 'kkkkabaH',
    0xa6f2: 'hhhhhba98',
    0xa6f3: 'hhhhbabI',
    0xa6f4: 'hhhhhhhaaam',
    0xa6f5: 'hhhhhhhaaiy',
    0xa6f6: 'hhhhhhhaaaM',
    0xa6f7: 'hhhhhhhaaiY',
    0xa6f8: 'aaaaaaababL',
    0xa6f9: 'aaaaaaabahY',
    0xa6fa: 'aaaaaaababl',
    0xa6fb: 'aaaaaaabahy',
    0xa6fc: 'aaaaaaa88',
    0xa6fd: 'aaaaaaaH',
    0xa6fe: 'aaaaaaab12',
    0xa6ff: 'bbbbbbbI',
    0xa700: 'cccccab88',
    0xa701: 'ccccabaI',
    0xa702: 'ccccccaa02',
    0xa703: 'eeeeeeebbiy',
    0xa704: 'cccccccbabL',
    0xa705: 'cccccccbahY',
    0xa706: 'cccccccbabl',
    0xa707: 'cccccccbahy',
    0xa708: 'iiiiiiibbbL',
    0xa709: 'iiiiiiibbhY',
    0xa70a: 'iiiiiiibbbl',
    0xa70b: 'iiiiiiibbhy',
    0xa70c: 'iiiiiaa88',
    0xa70d: 'iiiiaaaI',
    0xa70e: 'iiiiiiab02',
    0xa70f: 'jjjjbbbH',
    0xa710: 'xxxxxxbb13',
    0xa711: 'kkkkbbbH',
    0xa712: 'xxxxxba99',
    0xa713: 'hhhhaaaI',
    0xa714: 'xxxxxxxabal',
    0xa715: 'mmmmbaaI',
    0xa716: 'xxxxxxxabaL',
    0xa717: 'nnnnabbH',
    0xa718: 'qqqqqqqbbbM',
    0xa719: 'ggggbbaI',
    0xa71a: 'qqqqqqqbbbm',
    0xa71b: 'ddddaabH',
    0xa71c: 'qqqqqaa89',
    0xa71d: 'aaaababH',
    0xa71e: 'qqqqqqab03',
    0xa71f: 'bbbbabaI',
    0xa720: 'sssssbb99',
    0xa721: 'ccccbabH',
    0xa722: 'pppppaa89',
    0xa723: 'ppppbbbI',
    0xa724: 'uuuuuba89',
    0xa725: 'eeeebbaI',
    0xa726: 'vvvvvab99',
    0xa727: 'ffffaabH',
    0xa728: 'YYYYYYYbbbN',
    0xa729: 'oooobaaI',
    0xa72a: 'YYYYYYYbbbn',
    0xa72b: 'llllabbH',
    0xa72c: 'yyyyyba99',
    0xa72d: 'iiiibbbH',
    0xa72e: 'zzzzzab89',
    0xa72f: 'jjjjaaaI',
    0xa730: 'kkkkkbb98',
    0xa731: 'kkkkaaaI',
    0xa732: 'hhhhhaa88',
    0xa733: 'hhhhbbbH',
    0xa734: 'mmmmmba88',
    0xa735: 'mmmmabbH',
    0xa736: 'nnnnnab98',
    0xa737: 'nnnnbaaI',
    0xa738: 'gggggbb88',
    0xa739: 'ggggaabH',
    0xa73a: 'dddddaa98',
    0xa73b: 'ddddbbaI',
    0xa73c: 'aaaaaba98',
    0xa73d: 'aaaaabaI',
    0xa73e: 'bbbbbab88',
    0xa73f: 'bbbbbabH',
    0xa740: 'cccccba98',
    0xa741: 'ccccaabH',
    0xa742: 'CCCCCCba11',
    0xa743: 'ppppabbI',
    0xa744: 'eeeeebb88',
    0xa745: 'eeeeabaI',
    0xa746: 'fffffaa98',
    0xa747: 'ffffbabH',
    0xa748: 'oooooba88',
    0xa749: 'ooooaaaI',
    0xa74a: 'lllllab98',
    0xa74b: 'llllbbbH',
    0xa74c: 'iiiiibb98',
    0xa74d: 'iiiiabbH',
    0xa74e: 'jjjjjaa88',
    0xa74f: 'jjjjbaaI',
    0xa750: 'XXXXXXab00',
    0xa751: 'kkkkbaaI',
    0xa752: 'xxxxxab89',
    0xa753: 'hhhhabbH',
    0xa754: '=====ba48',
    0xa755: 'mmmmbbbH',
    0xa756: '>>>>>ab58',
    0xa757: 'nnnnaaaI',
    0xa758: 'wwwwwba89',
    0xa759: 'ggggbabH',
    0xa75a: 'tttttab99',
    0xa75b: 'ddddabaI',
    0xa75c: 'qqqqqbb99',
    0xa75d: 'aaaabbaI',
    0xa75e: 'rrrrraa89',
    0xa75f: 'bbbbaabH',
    0xa760: 'ppppppab03',
    0xa761: 'ccccbbaI',
    0xa762: 'pppppbb99',
    0xa763: 'ppppbaaH',
    0xa764: 'uuuuuab99',
    0xa765: 'eeeebabH',
    0xa766: 'uuuuuubb03',
    0xa767: 'ffffabaI',
    0xa768: '9999999bbbJ',
    0xa769: 'oooobbbH',
    0xa76a: '9999999bbbj',
    0xa76b: 'llllaaaI',
    0xa76c: 'yyyyyab89',
    0xa76d: 'iiiibaaI',
    0xa76e: 'yyyyyybb13',
    0xa76f: 'jjjjabbH',
    0xa770: 'hhhhhhab02',
    0xa771: 'kkkkabbH',
    0xa772: 'hhhhhbb98',
    0xa773: 'hhhhbaaI',
    0xa774: 'mmmmmab98',
    0xa775: 'mmmmaaaI',
    0xa776: 'mmmmmmbb02',
    0xa777: 'nnnnbbbH',
    0xa778: 'ddddddab12',
    0xa779: 'ggggabaI',
    0xa77a: 'dddddbb88',
    0xa77b: 'ddddbabH',
    0xa77c: 'aaaaaab88',
    0xa77d: 'aaaaaabH',
    0xa77e: 'aaaaaabb12',
    0xa77f: 'bbbbbbaI',
    0xa780: 'cccccaa98',
    0xa781: 'ccccabbH',
    0xa782: 'ccccccbb12',
    0xa783: 'ppppaabI',
    0xa784: 'eeeeeab88',
    0xa785: 'eeeeaaaI',
    0xa786: 'eeeeeeba02',
    0xa787: 'ffffbbbH',
    0xa788: 'llllllaa12',
    0xa789: 'ooooabaI',
    0xa78a: 'lllllbb98',
    0xa78b: 'llllbabH',
    0xa78c: 'iiiiiab98',
    0xa78d: 'iiiiaabH',
    0xa78e: 'iiiiiiba12',
    0xa78f: 'jjjjbbaI',
    0xa790: 'xxxxxxaa03',
    0xa791: 'kkkkbbaI',
    0xa792: 'xxxxxbb89',
    0xa793: 'hhhhaabH',
    0xa794: '11111111baaH',
    0xa795: 'mmmmbabH',
    0xa796: '11111111baaX',
    0xa797: 'nnnnabaI',
    0xa798: 'ttttttaa13',
    0xa799: 'ggggbbbH',
    0xa79a: 'tttttbb99',
    0xa79b: 'ddddaaaI',
    0xa79c: 'qqqqqab99',
    0xa79d: 'aaaabaaI',
    0xa79e: 'qqqqqqba13',
    0xa79f: 'bbbbabbH',
    0xa7a0: 'sssssba89',
    0xa7a1: 'ccccbaaI',
    0xa7a2: 'pppppab99',
    0xa7a3: 'ppppbbaH',
    0xa7a4: 'uuuuubb99',
    0xa7a5: 'eeeebbbH',
    0xa7a6: 'vvvvvaa89',
    0xa7a7: 'ffffaaaI',
    0xa7a8: '?????bb58',
    0xa7a9: 'oooobabH',
    0xa7aa: '<<<<<aa48',
    0xa7ab: 'llllabaI',
    0xa7ac: 'yyyyybb89',
    0xa7ad: 'iiiibbaI',
    0xa7ae: 'zzzzzaa99',
    0xa7af: 'jjjjaabH',
    0xa7b0: 'kkkkkba88',
    0xa7b1: 'kkkkaabH',
    0xa7b2: 'hhhhhab98',
    0xa7b3: 'hhhhbbaI',
    0xa7b4: 'mmmmmbb98',
    0xa7b5: 'mmmmabaI',
    0xa7b6: 'nnnnnaa88',
    0xa7b7: 'nnnnbabH',
    0xa7b8: 'gggggba98',
    0xa7b9: 'ggggaaaI',
    0xa7ba: 'dddddab88',
    0xa7bb: 'ddddbbbH',
    0xa7bc: 'aaaaabb88',
    0xa7bd: 'aaaaabbH',
    0xa7be: 'bbbbbaa98',
    0xa7bf: 'bbbbbaaI',
    0xa7c0: 'cccccbb88',
    0xa7c1: 'ccccaaaI',
    0xa7c2: 'CCCCCCab01',
    0xa7c3: 'ppppabaH',
    0xa7c4: 'eeeeeba98',
    0xa7c5: 'eeeeabbH',
    0xa7c6: 'fffffab88',
    0xa7c7: 'ffffbaaI',
    0xa7c8: 'ooooobb98',
    0xa7c9: 'ooooaabH',
    0xa7ca: 'lllllaa88',
    0xa7cb: 'llllbbaI',
    0xa7cc: 'iiiiiba88',
    0xa7cd: 'iiiiabaI',
    0xa7ce: 'jjjjjab98',
    0xa7cf: 'jjjjbabH',
    0xa7d0: 'QQQQQQQQbaal',
    0xa7d1: 'kkkkbabH',
    0xa7d2: 'xxxxxaa99',
    0xa7d3: 'hhhhabaI',
    0xa7d4: 'QQQQQQQQbaaL',
    0xa7d5: 'mmmmbbaI',
    0xa7d6: 'QQQQQQQQbaiX',
    0xa7d7: 'nnnnaabH',
    0xa7d8: 'wwwwwbb99',
    0xa7d9: 'ggggbaaI',
    0xa7da: 'tttttaa89',
    0xa7db: 'ddddabbH',
    0xa7dc: 'qqqqqba89',
    0xa7dd: 'aaaabbbH',
    0xa7de: 'rrrrrab99',
    0xa7df: 'bbbbaaaI',
    0xa7e0: 'ppppppba13',
    0xa7e1: 'ccccbbbH',
    0xa7e2: 'pppppba89',
    0xa7e3: 'ppppbabI',
    0xa7e4: 'pppppppbbbm',
    0xa7e5: 'eeeebaaI',
    0xa7e6: 'pppppppbbbM',
    0xa7e7: 'ffffabbH',
    0xa7e8: 'ppppppppaajZ',
    0xa7e9: 'oooobbaI',
    0xa7ea: 'ppppppppaabN',
    0xa7eb: 'llllaabH',
    0xa7ec: 'ppppppppaajz',
    0xa7ed: 'iiiibabH',
    0xa7ee: 'ppppppppaabn',
    0xa7ef: 'jjjjabaI',
    0xa7f0: 'aaaaaaaabaan',
    0xa7f1: 'aaaaaaaabaaf',
    0xa7f2: 'aaaaaaaabaiz',
    0xa7f3: 'aaaaaaaabaav',
    0xa7f4: 'aaaaaaaabaaN',
    0xa7f5: 'aaaaaaaabaaF',
    0xa7f6: 'aaaaaaaabaiZ',
    0xa7f7: 'aaaaaaaabaaV',
    0xa7f8: 'aaaaaaaabaM',
    0xa7f9: 'aaaaaaaabiY',
    0xa7fa: 'aaaaaaaabam',
    0xa7fb: 'aaaaaaaabiy',
    0xa7fc: 'aaaaaaa98',
    0xa7fd: 'aaaaaaaI',
    0xa7fe: 'aaaaaaaa02',
    0xa7ff: 'aaaaaaaad8',
    0xa800: 'cccccab59',
    0xa801: 'ccccabaF',
    0xa802: 'ccccccab01',
    0xa803: 'ccccccccb8',
    0xa804: 'cccccccaaaL',
    0xa805: 'cccccccaaiX',
    0xa806: 'cccccccaaal',
    0xa807: 'cccccccaaix',
    0xa808: 'ccccccccbbaN',
    0xa809: 'ccccccccbbaF',
    0xa80a: 'ccccccccbbiZ',
    0xa80b: 'ccccccccbbaV',
    0xa80c: 'ccccccccbban',
    0xa80d: 'ccccccccbbaf',
    0xa80e: 'ccccccccbbiz',
    0xa80f: 'ccccccccbbav',
    0xa810: 'qqqqqqqqbaao',
    0xa811: 'kkkkbbbG',
    0xa812: 'qqqqqqqqbakz',
    0xa813: 'hhhhaaaF',
    0xa814: 'qqqqqqqqbaaO',
    0xa815: 'mmmmbaaF',
    0xa816: 'qqqqqqqqbakZ',
    0xa817: 'nnnnabbG',
    0xa818: 'qqqqqqqabaM',
    0xa819: 'ggggbbaF',
    0xa81a: 'qqqqqqqabam',
    0xa81b: 'ddddaabG',
    0xa81c: 'qqqqqaa58',
    0xa81d: 'aaaababG',
    0xa81e: 'qqqqqqaa00',
    0xa81f: 'bbbbabaF',
    0xa820: 'sssssbb48',
    0xa821: 'ccccbabG',
    0xa822: 'pppppaa58',
    0xa823: 'ppppbbbF',
    0xa824: 'uuuuuba58',
    0xa825: 'eeeebbaF',
    0xa826: 'vvvvvab48',
    0xa827: 'ffffaabG',
    0xa828: 'PPPPPPPPaahX',
    0xa829: 'oooobaaF',
    0xa82a: 'PPPPPPPPaabM',
    0xa82b: 'llllabbG',
    0xa82c: 'yyyyyba48',
    0xa82d: 'iiiibbbG',
    0xa82e: 'zzzzzab58',
    0xa82f: 'jjjjaaaF',
    0xa830: 'kkkkkbb49',
    0xa831: 'kkkkaaaF',
    0xa832: 'hhhhhaa59',
    0xa833: 'hhhhbbbG',
    0xa834: 'mmmmmba59',
    0xa835: 'mmmmabbG',
    0xa836: 'nnnnnab49',
    0xa837: 'nnnnbaaF',
    0xa838: 'gggggbb59',
    0xa839: 'ggggaabG',
    0xa83a: 'dddddaa49',
    0xa83b: 'ddddbbaF',
    0xa83c: 'aaaaaba49',
    0xa83d: 'aaaaabaF',
    0xa83e: 'bbbbbab59',
    0xa83f: 'bbbbbabG',
    0xa840: 'cccccba49',
    0xa841: 'ccccaabG',
    0xa842: 'CCCCCCbb12',
    0xa843: 'ppppabbF',
    0xa844: 'eeeeebb59',
    0xa845: 'eeeeabaF',
    0xa846: 'fffffaa49',
    0xa847: 'ffffbabG',
    0xa848: 'oooooba59',
    0xa849: 'ooooaaaF',
    0xa84a: 'lllllab49',
    0xa84b: 'llllbbbG',
    0xa84c: 'iiiiibb49',
    0xa84d: 'iiiiabbG',
    0xa84e: 'jjjjjaa59',
    0xa84f: 'jjjjbaaF',
    0xa850: 'XXXXXXaa03',
    0xa851: 'kkkkbaaF',
    0xa852: 'xxxxxab58',
    0xa853: 'hhhhabbG',
    0xa854: '=====ba99',
    0xa855: 'mmmmbbbG',
    0xa856: '>>>>>ab89',
    0xa857: 'nnnnaaaF',
    0xa858: 'wwwwwba58',
    0xa859: 'ggggbabG',
    0xa85a: 'tttttab48',
    0xa85b: 'ddddabaF',
    0xa85c: 'qqqqqbb48',
    0xa85d: 'aaaabbaF',
    0xa85e: 'rrrrraa58',
    0xa85f: 'bbbbaabG',
    0xa860: 'ppppppaa00',
    0xa861: 'ccccbbaF',
    0xa862: 'pppppbb48',
    0xa863: 'ppppbaaG',
    0xa864: 'uuuuuab48',
    0xa865: 'eeeebabG',
    0xa866: 'uuuuuuba00',
    0xa867: 'ffffabaF',
    0xa868: '00000000aabY',
    0xa869: 'oooobbbG',
    0xa86a: '00000000aabI',
    0xa86b: 'llllaaaF',
    0xa86c: 'yyyyyab58',
    0xa86d: 'iiiibaaF',
    0xa86e: 'yyyyyyba10',
    0xa86f: 'jjjjabbG',
    0xa870: 'hhhhhhaa01',
    0xa871: 'kkkkabbG',
    0xa872: 'hhhhhbb49',
    0xa873: 'hhhhbaaF',
    0xa874: 'mmmmmab49',
    0xa875: 'mmmmaaaF',
    0xa876: 'mmmmmmba01',
    0xa877: 'nnnnbbbG',
    0xa878: 'ddddddaa11',
    0xa879: 'ggggabaF',
    0xa87a: 'dddddbb59',
    0xa87b: 'ddddbabG',
    0xa87c: 'aaaaaab59',
    0xa87d: 'aaaaaabG',
    0xa87e: 'aaaaaaba11',
    0xa87f: 'bbbbbbaF',
    0xa880: 'cccccaa49',
    0xa881: 'ccccabbG',
    0xa882: 'ccccccba11',
    0xa883: 'ppppaabF',
    0xa884: 'eeeeeab59',
    0xa885: 'eeeeaaaF',
    0xa886: 'eeeeeebb01',
    0xa887: 'ffffbbbG',
    0xa888: 'llllllab11',
    0xa889: 'ooooabaF',
    0xa88a: 'lllllbb49',
    0xa88b: 'llllbabG',
    0xa88c: 'iiiiiab49',
    0xa88d: 'iiiiaabG',
    0xa88e: 'iiiiiibb11',
    0xa88f: 'jjjjbbaF',
    0xa890: 'xxxxxxab00',
    0xa891: 'kkkkbbaF',
    0xa892: 'xxxxxbb58',
    0xa893: 'hhhhaabG',
    0xa894: '8888888abaj',
    0xa895: 'mmmmbabG',
    0xa896: '8888888abaJ',
    0xa897: 'nnnnabaF',
    0xa898: 'ttttttab10',
    0xa899: 'ggggbbbG',
    0xa89a: 'tttttbb48',
    0xa89b: 'ddddaaaF',
    0xa89c: 'qqqqqab48',
    0xa89d: 'aaaabaaF',
    0xa89e: 'qqqqqqbb10',
    0xa89f: 'bbbbabbG',
    0xa8a0: 'sssssba58',
    0xa8a1: 'ccccbaaF',
    0xa8a2: 'pppppab48',
    0xa8a3: 'ppppbbaG',
    0xa8a4: 'uuuuubb48',
    0xa8a5: 'eeeebbbG',
    0xa8a6: 'vvvvvaa58',
    0xa8a7: 'ffffaaaF',
    0xa8a8: '?????bb89',
    0xa8a9: 'oooobabG',
    0xa8aa: '<<<<<aa99',
    0xa8ab: 'llllabaF',
    0xa8ac: 'yyyyybb58',
    0xa8ad: 'iiiibbaF',
    0xa8ae: 'zzzzzaa48',
    0xa8af: 'jjjjaabG',
    0xa8b0: 'kkkkkba59',
    0xa8b1: 'kkkkaabG',
    0xa8b2: 'hhhhhab49',
    0xa8b3: 'hhhhbbaF',
    0xa8b4: 'mmmmmbb49',
    0xa8b5: 'mmmmabaF',
    0xa8b6: 'nnnnnaa59',
    0xa8b7: 'nnnnbabG',
    0xa8b8: 'gggggba49',
    0xa8b9: 'ggggaaaF',
    0xa8ba: 'dddddab59',
    0xa8bb: 'ddddbbbG',
    0xa8bc: 'aaaaabb59',
    0xa8bd: 'aaaaabbG',
    0xa8be: 'bbbbbaa49',
    0xa8bf: 'bbbbbaaF',
    0xa8c0: 'cccccbb59',
    0xa8c1: 'ccccaaaF',
    0xa8c2: 'CCCCCCaa02',
    0xa8c3: 'ppppabaG',
    0xa8c4: 'eeeeeba49',
    0xa8c5: 'eeeeabbG',
    0xa8c6: 'fffffab59',
    0xa8c7: 'ffffbaaF',
    0xa8c8: 'ooooobb49',
    0xa8c9: 'ooooaabG',
    0xa8ca: 'lllllaa59',
    0xa8cb: 'llllbbaF',
    0xa8cc: 'iiiiiba59',
    0xa8cd: 'iiiiabaF',
    0xa8ce: 'jjjjjab49',
    0xa8cf: 'jjjjbabG',
    0xa8d0: 'XXXXXXbb13',
    0xa8d1: 'kkkkbabG',
    0xa8d2: 'xxxxxaa48',
    0xa8d3: 'hhhhabaF',
    0xa8d4: 'XXXXXXXaban',
    0xa8d5: 'mmmmbbaF',
    0xa8d6: 'XXXXXXXabaN',
    0xa8d7: 'nnnnaabG',
    0xa8d8: 'wwwwwbb48',
    0xa8d9: 'ggggbaaF',
    0xa8da: 'tttttaa58',
    0xa8db: 'ddddabbG',
    0xa8dc: 'qqqqqba58',
    0xa8dd: 'aaaabbbG',
    0xa8de: 'rrrrrab48',
    0xa8df: 'bbbbaaaF',
    0xa8e0: 'ppppppbb10',
    0xa8e1: 'ccccbbbG',
    0xa8e2: 'pppppba58',
    0xa8e3: 'ppppbabF',
    0xa8e4: 'pppppppabam',
    0xa8e5: 'eeeebaaF',
    0xa8e6: 'pppppppabaM',
    0xa8e7: 'ffffabbG',
    0xa8e8: 'yyyyyyybbbL',
    0xa8e9: 'oooobbaF',
    0xa8ea: 'yyyyyyybbbl',
    0xa8eb: 'llllaabG',
    0xa8ec: 'yyyyyaa48',
    0xa8ed: 'iiiibabG',
    0xa8ee: 'yyyyyyab00',
    0xa8ef: 'jjjjabaF',
    0xa8f0: 'hhhhhhbb11',
    0xa8f1: 'hhhhhhhab9',
    0xa8f2: 'hhhhhba59',
    0xa8f3: 'hhhhbabG',
    0xa8f4: 'hhhhhhhabal',
    0xa8f5: 'hhhhhhhabix',
    0xa8f6: 'hhhhhhhabaL',
    0xa8f7: 'hhhhhhhabiX',
    0xa8f8: 'aaaaaaabbbM',
    0xa8f9: 'aaaaaaabbhX',
    0xa8fa: 'aaaaaaabbbm',
    0xa8fb: 'aaaaaaabbhx',
    0xa8fc: 'aaaaaaa49',
    0xa8fd: 'aaaaaaaF',
    0xa8fe: 'aaaaaaab01',
    0xa8ff: 'bbbbbbbG',
    0xa900: 'cccccab49',
    0xa901: 'ccccabaG',
    0xa902: 'ccccccaa11',
    0xa903: 'eeeeeeebaix',
    0xa904: 'cccccccbbbM',
    0xa905: 'cccccccbbhX',
    0xa906: 'cccccccbbbm',
    0xa907: 'cccccccbbhx',
    0xa908: 'iiiiiiibabM',
    0xa909: 'iiiiiiibahX',
    0xa90a: 'iiiiiiibabm',
    0xa90b: 'iiiiiiibahx',
    0xa90c: 'iiiiiaa49',
    0xa90d: 'iiiiaaaG',
    0xa90e: 'iiiiiiab11',
    0xa90f: 'jjjjbbbF',
    0xa910: 'xxxxxxbb00',
    0xa911: 'kkkkbbbF',
    0xa912: 'xxxxxba58',
    0xa913: 'hhhhaaaG',
    0xa914: 'xxxxxxxaaam',
    0xa915: 'mmmmbaaG',
    0xa916: 'xxxxxxxaaaM',
    0xa917: 'nnnnabbF',
    0xa918: 'qqqqqqqbabL',
    0xa919: 'ggggbbaG',
    0xa91a: 'qqqqqqqbabl',
    0xa91b: 'ddddaabF',
    0xa91c: 'qqqqqaa48',
    0xa91d: 'aaaababF',
    0xa91e: 'qqqqqqab10',
    0xa91f: 'bbbbabaG',
    0xa920: 'sssssbb58',
    0xa921: 'ccccbabF',
    0xa922: 'pppppaa48',
    0xa923: 'ppppbbbG',
    0xa924: 'uuuuuba48',
    0xa925: 'eeeebbaG',
    0xa926: 'vvvvvab58',
    0xa927: 'ffffaabF',
    0xa928: 'YYYYYYYbabO',
    0xa929: 'oooobaaG',
    0xa92a: 'YYYYYYYbabo',
    0xa92b: 'llllabbF',
    0xa92c: 'yyyyyba58',
    0xa92d: 'iiiibbbF',
    0xa92e: 'zzzzzab48',
    0xa92f: 'jjjjaaaG',
    0xa930: 'kkkkkbb59',
    0xa931: 'kkkkaaaG',
    0xa932: 'hhhhhaa49',
    0xa933: 'hhhhbbbF',
    0xa934: 'mmmmmba49',
    0xa935: 'mmmmabbF',
    0xa936: 'nnnnnab59',
    0xa937: 'nnnnbaaG',
    0xa938: 'gggggbb49',
    0xa939: 'ggggaabF',
    0xa93a: 'dddddaa59',
    0xa93b: 'ddddbbaG',
    0xa93c: 'aaaaaba59',
    0xa93d: 'aaaaabaG',
    0xa93e: 'bbbbbab49',
    0xa93f: 'bbbbbabF',
    0xa940: 'cccccba59',
    0xa941: 'ccccaabF',
    0xa942: 'CCCCCCba02',
    0xa943: 'ppppabbG',
    0xa944: 'eeeeebb49',
    0xa945: 'eeeeabaG',
    0xa946: 'fffffaa59',
    0xa947: 'ffffbabF',
    0xa948: 'oooooba49',
    0xa949: 'ooooaaaG',
    0xa94a: 'lllllab59',
    0xa94b: 'llllbbbF',
    0xa94c: 'iiiiibb59',
    0xa94d: 'iiiiabbF',
    0xa94e: 'jjjjjaa49',
    0xa94f: 'jjjjbaaG',
    0xa950: 'XXXXXXab13',
    0xa951: 'kkkkbaaG',
    0xa952: 'xxxxxab48',
    0xa953: 'hhhhabbF',
    0xa954: '=====ba89',
    0xa955: 'mmmmbbbF',
    0xa956: '>>>>>ab99',
    0xa957: 'nnnnaaaG',
    0xa958: 'wwwwwba48',
    0xa959: 'ggggbabF',
    0xa95a: 'tttttab58',
    0xa95b: 'ddddabaG',
    0xa95c: 'qqqqqbb58',
    0xa95d: 'aaaabbaG',
    0xa95e: 'rrrrraa48',
    0xa95f: 'bbbbaabF',
    0xa960: 'ppppppab10',
    0xa961: 'ccccbbaG',
    0xa962: 'pppppbb58',
    0xa963: 'ppppbaaF',
    0xa964: 'uuuuuab58',
    0xa965: 'eeeebabF',
    0xa966: 'uuuuuubb10',
    0xa967: 'ffffabaG',
    0xa968: '9999999babK',
    0xa969: 'oooobbbF',
    0xa96a: '9999999babk',
    0xa96b: 'llllaaaG',
    0xa96c: 'yyyyyab48',
    0xa96d: 'iiiibaaG',
    0xa96e: 'yyyyyybb00',
    0xa96f: 'jjjjabbF',
    0xa970: 'hhhhhhab11',
    0xa971: 'kkkkabbF',
    0xa972: 'hhhhhbb59',
    0xa973: 'hhhhbaaG',
    0xa974: 'mmmmmab59',
    0xa975: 'mmmmaaaG',
    0xa976: 'mmmmmmbb11',
    0xa977: 'nnnnbbbF',
    0xa978: 'ddddddab01',
    0xa979: 'ggggabaG',
    0xa97a: 'dddddbb49',
    0xa97b: 'ddddbabF',
    0xa97c: 'aaaaaab49',
    0xa97d: 'aaaaaabF',
    0xa97e: 'aaaaaabb01',
    0xa97f: 'bbbbbbaG',
    0xa980: 'cccccaa59',
    0xa981: 'ccccabbF',
    0xa982: 'ccccccbb01',
    0xa983: 'ppppaabG',
    0xa984: 'eeeeeab49',
    0xa985: 'eeeeaaaG',
    0xa986: 'eeeeeeba11',
    0xa987: 'ffffbbbF',
    0xa988: 'llllllaa01',
    0xa989: 'ooooabaG',
    0xa98a: 'lllllbb59',
    0xa98b: 'llllbabF',
    0xa98c: 'iiiiiab59',
    0xa98d: 'iiiiaabF',
    0xa98e: 'iiiiiiba01',
    0xa98f: 'jjjjbbaG',
    0xa990: 'xxxxxxaa10',
    0xa991: 'kkkkbbaG',
    0xa992: 'xxxxxbb48',
    0xa993: 'hhhhaabF',
    0xa994: '11111111aabI',
    0xa995: 'mmmmbabF',
    0xa996: '11111111aabY',
    0xa997: 'nnnnabaG',
    0xa998: 'ttttttaa00',
    0xa999: 'ggggbbbF',
    0xa99a: 'tttttbb58',
    0xa99b: 'ddddaaaG',
    0xa99c: 'qqqqqab58',
    0xa99d: 'aaaabaaG',
    0xa99e: 'qqqqqqba00',
    0xa99f: 'bbbbabbF',
    0xa9a0: 'sssssba48',
    0xa9a1: 'ccccbaaG',
    0xa9a2: 'pppppab58',
    0xa9a3: 'ppppbbaF',
    0xa9a4: 'uuuuubb58',
    0xa9a5: 'eeeebbbF',
    0xa9a6: 'vvvvvaa48',
    0xa9a7: 'ffffaaaG',
    0xa9a8: '1111111111baaA',
    0xa9a9: 'oooobabF',
    0xa9aa: '1111111111baaE',
    0xa9ab: 'llllabaG',
    0xa9ac: 'yyyyybb48',
    0xa9ad: 'iiiibbaG',
    0xa9ae: 'zzzzzaa58',
    0xa9af: 'jjjjaabF',
    0xa9b0: 'kkkkkba49',
    0xa9b1: 'kkkkaabF',
    0xa9b2: 'hhhhhab59',
    0xa9b3: 'hhhhbbaG',
    0xa9b4: 'mmmmmbb59',
    0xa9b5: 'mmmmabaG',
    0xa9b6: 'nnnnnaa49',
    0xa9b7: 'nnnnbabF',
    0xa9b8: 'gggggba59',
    0xa9b9: 'ggggaaaG',
    0xa9ba: 'dddddab49',
    0xa9bb: 'ddddbbbF',
    0xa9bc: 'aaaaabb49',
    0xa9bd: 'aaaaabbF',
    0xa9be: 'bbbbbaa59',
    0xa9bf: 'bbbbbaaG',
    0xa9c0: 'aaaaaaaaaabaaW',
    0xa9c1: 'aaaaaaaaaabaaU',
    0xa9c2: 'aaaaaaaaaabaaS',
    0xa9c3: 'aaaaaaaaaabaaQ',
    0xa9c4: 'aaaaaaaaaabakZ',
    0xa9c5: 'aaaaaaaaaabaiY',
    0xa9c6: 'aaaaaaaaaabacZ',
    0xa9c7: 'aaaaaaaaaabaaY',
    0xa9c8: 'aaaaaaaaaabaaG',
    0xa9c9: 'aaaaaaaaaabaaE',
    0xa9ca: 'aaaaaaaaaabaaC',
    0xa9cb: 'aaaaaaaaaabaaA',
    0xa9cc: 'aaaaaaaaaabaaO',
    0xa9cd: 'aaaaaaaaaabaaM',
    0xa9ce: 'aaaaaaaaaabaaK',
    0xa9cf: 'aaaaaaaaaabaaI',
    0xa9d0: 'aaaaaaaaaabaaw',
    0xa9d1: 'aaaaaaaaaabaau',
    0xa9d2: 'aaaaaaaaaabaas',
    0xa9d3: 'aaaaaaaaaabaaq',
    0xa9d4: 'aaaaaaaaaabakz',
    0xa9d5: 'aaaaaaaaaabaiy',
    0xa9d6: 'aaaaaaaaaabacz',
    0xa9d7: 'aaaaaaaaaabaay',
    0xa9d8: 'aaaaaaaaaabaag',
    0xa9d9: 'aaaaaaaaaabaae',
    0xa9da: 'aaaaaaaaaabaac',
    0xa9db: 'aaaaaaaaaabaaa',
    0xa9dc: 'aaaaaaaaaabaao',
    0xa9dd: 'aaaabbbF',
    0xa9de: 'aaaaaaaaaabaak',
    0xa9df: 'aaaaaaaaaabaai',
    0xa9e0: 'aaaaaaaaaabaC',
    0xa9e1: 'aaaaaaaaaabaG',
    0xa9e2: 'aaaaaaaaaabaK',
    0xa9e3: 'aaaaaaaaaabaO',
    0xa9e4: 'aaaaaaaaaabaS',
    0xa9e5: 'aaaaaaaaaabaW',
    0xa9e6: 'aaaaaaaaaabcZ',
    0xa9e7: 'aaaaaaaaaabkZ',
    0xa9e8: 'aaaaaaaaaabac',
    0xa9e9: 'aaaaaaaaaabag',
    0xa9ea: 'aaaaaaaaaabak',
    0xa9eb: 'aaaaaaaaaabao',
    0xa9ec: 'aaaaaaaaaabas',
    0xa9ed: 'aaaaaaaaaabaw',
    0xa9ee: 'aaaaaaaaaabcz',
    0xa9ef: 'aaaaaaaaaabkz',
    0xa9f0: 'aaaaaaaaaabo',
    0xa9f1: 'aaaaaaaaaabg',
    0xa9f2: 'aaaaaaaaaahz',
    0xa9f3: 'aaaaaaaaaabw',
    0xa9f4: 'aaaaaaaaaabO',
    0xa9f5: 'aaaaaaaaaabG',
    0xa9f6: 'aaaaaaaaaahZ',
    0xa9f7: 'aaaaaaaaaabW',
    0xa9f8: 'aaaaaaaaaaL',
    0xa9f9: 'aaaaaaaaaiX',
    0xa9fa: 'aaaaaaaaaal',
    0xa9fb: 'aaaaaaaaaix',
    0xa9fc: 'aaaaaaa59',
    0xa9fd: 'aaaaaaaG',
    0xa9fe: 'aaaaaaaa11',
    0xa9ff: 'aaaaaaaaa9',
    0xaa00: 'cccccab58',
    0xaa01: 'ccccabaD',
    0xaa02: 'ccccccab11',
    0xaa03: 'cccccccca9',
    0xaa04: 'cccccccabbM',
    0xaa05: 'cccccccabhX',
    0xaa06: 'cccccccabbm',
    0xaa07: 'cccccccabhx',
    0xaa08: 'ccccccccaabO',
    0xaa09: 'ccccccccaabG',
    0xaa0a: 'ccccccccaahZ',
    0xaa0b: 'ccccccccaabW',
    0xaa0c: 'ccccccccaabo',
    0xaa0d: 'ccccccccaabg',
    0xaa0e: 'ccccccccaahz',
    0xaa0f: 'ccccccccaabw',
    0xaa10: 'cccccccccbabr',
    0xaa11: 'cccccccccbabv',
    0xaa12: 'cccccccccbabz',
    0xaa13: 'cccccccccbajz',
    0xaa14: 'cccccccccbabb',
    0xaa15: 'cccccccccbabf',
    0xaa16: 'cccccccccbabj',
    0xaa17: 'cccccccccbabn',
    0xaa18: 'cccccccccbabR',
    0xaa19: 'cccccccccbabV',
    0xaa1a: 'cccccccccbabZ',
    0xaa1b: 'cccccccccbajZ',
    0xaa1c: 'cccccccccbabB',
    0xaa1d: 'aaaababE',
    0xaa1e: 'cccccccccbabJ',
    0xaa1f: 'bbbbabaD',
    0xaa20: 'sssssbb49',
    0xaa21: 'ccccbabE',
    0xaa22: 'pppppaa59',
    0xaa23: 'ppppbbbD',
    0xaa24: 'uuuuuba59',
    0xaa25: 'eeeebbaD',
    0xaa26: 'vvvvvab49',
    0xaa27: 'ffffaabE',
    0xaa28: 'AAAAAAAAAbbba',
    0xaa29: 'oooobaaD',
    0xaa2a: 'AAAAAAAAAbbbi',
    0xaa2b: 'llllabbE',
    0xaa2c: 'yyyyyba49',
    0xaa2d: 'iiiibbbE',
    0xaa2e: 'zzzzzab59',
    0xaa2f: 'jjjjaaaD',
    0xaa30: 'kkkkkbb48',
    0xaa31: 'kkkkaaaD',
    0xaa32: 'hhhhhaa58',
    0xaa33: 'hhhhbbbE',
    0xaa34: 'mmmmmba58',
    0xaa35: 'mmmmabbE',
    0xaa36: 'nnnnnab48',
    0xaa37: 'nnnnbaaD',
    0xaa38: 'gggggbb58',
    0xaa39: 'ggggaabE',
    0xaa3a: 'dddddaa48',
    0xaa3b: 'ddddbbaD',
    0xaa3c: 'aaaaaba48',
    0xaa3d: 'aaaaabaD',
    0xaa3e: 'bbbbbab58',
    0xaa3f: 'bbbbbabE',
    0xaa40: 'cccccba48',
    0xaa41: 'ccccaabE',
    0xaa42: 'CCCCCCbb02',
    0xaa43: 'ppppabbD',
    0xaa44: 'eeeeebb58',
    0xaa45: 'eeeeabaD',
    0xaa46: 'fffffaa48',
    0xaa47: 'ffffbabE',
    0xaa48: 'oooooba58',
    0xaa49: 'ooooaaaD',
    0xaa4a: 'lllllab48',
    0xaa4b: 'llllbbbE',
    0xaa4c: 'iiiiibb48',
    0xaa4d: 'iiiiabbE',
    0xaa4e: 'jjjjjaa58',
    0xaa4f: 'jjjjbaaD',
    0xaa50: 'XXXXXXaa13',
    0xaa51: 'kkkkbaaD',
    0xaa52: 'xxxxxab59',
    0xaa53: 'hhhhabbE',
    0xaa54: '=====ba98',
    0xaa55: 'mmmmbbbE',
    0xaa56: '>>>>>ab88',
    0xaa57: 'nnnnaaaD',
    0xaa58: 'wwwwwba59',
    0xaa59: 'ggggbabE',
    0xaa5a: 'tttttab49',
    0xaa5b: 'ddddabaD',
    0xaa5c: 'qqqqqbb49',
    0xaa5d: 'aaaabbaD',
    0xaa5e: 'rrrrraa59',
    0xaa5f: 'bbbbaabE',
    0xaa60: 'ppppppaa10',
    0xaa61: 'ccccbbaD',
    0xaa62: 'pppppbb49',
    0xaa63: 'ppppbaaE',
    0xaa64: 'uuuuuab49',
    0xaa65: 'eeeebabE',
    0xaa66: 'uuuuuuba10',
    0xaa67: 'ffffabaD',
    0xaa68: '00000000bbaX',
    0xaa69: 'oooobbbE',
    0xaa6a: '00000000bbaH',
    0xaa6b: 'llllaaaD',
    0xaa6c: 'yyyyyab59',
    0xaa6d: 'iiiibaaD',
    0xaa6e: 'yyyyyyba00',
    0xaa6f: 'jjjjabbE',
    0xaa70: 'hhhhhhaa11',
    0xaa71: 'kkkkabbE',
    0xaa72: 'hhhhhbb48',
    0xaa73: 'hhhhbaaD',
    0xaa74: 'mmmmmab48',
    0xaa75: 'mmmmaaaD',
    0xaa76: 'mmmmmmba11',
    0xaa77: 'nnnnbbbE',
    0xaa78: 'ddddddaa01',
    0xaa79: 'ggggabaD',
    0xaa7a: 'dddddbb58',
    0xaa7b: 'ddddbabE',
    0xaa7c: 'aaaaaab58',
    0xaa7d: 'aaaaaabE',
    0xaa7e: 'aaaaaaba01',
    0xaa7f: 'bbbbbbaD',
    0xaa80: 'cccccaa48',
    0xaa81: 'ccccabbE',
    0xaa82: 'ccccccba01',
    0xaa83: 'ppppaabD',
    0xaa84: 'eeeeeab58',
    0xaa85: 'eeeeaaaD',
    0xaa86: 'eeeeeebb11',
    0xaa87: 'ffffbbbE',
    0xaa88: 'llllllab01',
    0xaa89: 'ooooabaD',
    0xaa8a: 'lllllbb48',
    0xaa8b: 'llllbabE',
    0xaa8c: 'iiiiiab48',
    0xaa8d: 'iiiiaabE',
    0xaa8e: 'iiiiiibb01',
    0xaa8f: 'jjjjbbaD',
    0xaa90: 'xxxxxxab10',
    0xaa91: 'kkkkbbaD',
    0xaa92: 'xxxxxbb59',
    0xaa93: 'hhhhaabE',
    0xaa94: '8888888aabk',
    0xaa95: 'mmmmbabE',
    0xaa96: '8888888aabK',
    0xaa97: 'nnnnabaD',
    0xaa98: 'ttttttab00',
    0xaa99: 'ggggbbbE',
    0xaa9a: 'tttttbb49',
    0xaa9b: 'ddddaaaD',
    0xaa9c: 'qqqqqab49',
    0xaa9d: 'aaaabaaD',
    0xaa9e: 'qqqqqqbb00',
    0xaa9f: 'bbbbabbE',
    0xaaa0: 'sssssba59',
    0xaaa1: 'ccccbaaD',
    0xaaa2: 'pppppab49',
    0xaaa3: 'ppppbbaE',
    0xaaa4: 'uuuuubb49',
    0xaaa5: 'eeeebbbE',
    0xaaa6: 'vvvvvaa59',
    0xaaa7: 'ffffaaaD',
    0xaaa8: '?????bb88',
    0xaaa9: 'oooobabE',
    0xaaaa: '<<<<<aa98',
    0xaaab: 'llllabaD',
    0xaaac: 'yyyyybb59',
    0xaaad: 'iiiibbaD',
    0xaaae: 'zzzzzaa49',
    0xaaaf: 'jjjjaabE',
    0xaab0: 'kkkkkba58',
    0xaab1: 'kkkkaabE',
    0xaab2: 'hhhhhab48',
    0xaab3: 'hhhhbbaD',
    0xaab4: 'mmmmmbb48',
    0xaab5: 'mmmmabaD',
    0xaab6: 'nnnnnaa58',
    0xaab7: 'nnnnbabE',
    0xaab8: 'gggggba48',
    0xaab9: 'ggggaaaD',
    0xaaba: 'dddddab58',
    0xaabb: 'ddddbbbE',
    0xaabc: 'aaaaabb58',
    0xaabd: 'aaaaabbE',
    0xaabe: 'bbbbbaa48',
    0xaabf: 'bbbbbaaD',
    0xaac0: 'cccccbb58',
    0xaac1: 'ccccaaaD',
    0xaac2: 'CCCCCCaa12',
    0xaac3: 'ppppabaE',
    0xaac4: 'eeeeeba48',
    0xaac5: 'eeeeabbE',
    0xaac6: 'fffffab58',
    0xaac7: 'ffffbaaD',
    0xaac8: 'ooooobb48',
    0xaac9: 'ooooaabE',
    0xaaca: 'lllllaa58',
    0xaacb: 'llllbbaD',
    0xaacc: 'iiiiiba58',
    0xaacd: 'iiiiabaD',
    0xaace: 'jjjjjab48',
    0xaacf: 'jjjjbabE',
    0xaad0: 'XXXXXXbb03',
    0xaad1: 'kkkkbabE',
    0xaad2: 'xxxxxaa49',
    0xaad3: 'hhhhabaD',
    0xaad4: 'XXXXXXXaabo',
    0xaad5: 'mmmmbbaD',
    0xaad6: 'XXXXXXXaabO',
    0xaad7: 'nnnnaabE',
    0xaad8: 'wwwwwbb49',
    0xaad9: 'ggggbaaD',
    0xaada: 'tttttaa59',
    0xaadb: 'ddddabbE',
    0xaadc: 'qqqqqba59',
    0xaadd: 'aaaabbbE',
    0xaade: 'rrrrrab49',
    0xaadf: 'bbbbaaaD',
    0xaae0: 'ppppppbb00',
    0xaae1: 'ccccbbbE',
    0xaae2: 'pppppba59',
    0xaae3: 'ppppbabD',
    0xaae4: 'pppppppaabl',
    0xaae5: 'eeeebaaD',
    0xaae6: 'pppppppaabL',
    0xaae7: 'ffffabbE',
    0xaae8: 'yyyyyyybaaM',
    0xaae9: 'oooobbaD',
    0xaaea: 'yyyyyyybaam',
    0xaaeb: 'llllaabE',
    0xaaec: 'yyyyyaa49',
    0xaaed: 'iiiibabE',
    0xaaee: 'yyyyyyab10',
    0xaaef: 'jjjjabaD',
    0xaaf0: 'hhhhhhbb01',
    0xaaf1: 'kkkkabaD',
    0xaaf2: 'hhhhhba58',
    0xaaf3: 'hhhhbabE',
    0xaaf4: 'hhhhhhhaabm',
    0xaaf5: 'hhhhhhhaahx',
    0xaaf6: 'hhhhhhhaabM',
    0xaaf7: 'hhhhhhhaahX',
    0xaaf8: 'aaaaaaabaaL',
    0xaaf9: 'aaaaaaabaiX',
    0xaafa: 'aaaaaaabaal',
    0xaafb: 'aaaaaaabaix',
    0xaafc: 'aaaaaaa48',
    0xaafd: 'aaaaaaaD',
    0xaafe: 'aaaaaaab11',
    0xaaff: 'bbbbbbbE',
    0xab00: 'cccccab48',
    0xab01: 'ccccabaE',
    0xab02: 'ccccccaa01',
    0xab03: 'eeeeeeebbhx',
    0xab04: 'cccccccbaaL',
    0xab05: 'cccccccbaiX',
    0xab06: 'cccccccbaal',
    0xab07: 'cccccccbaix',
    0xab08: 'iiiiiiibbaL',
    0xab09: 'iiiiiiibbiX',
    0xab0a: 'iiiiiiibbal',
    0xab0b: 'iiiiiiibbix',
    0xab0c: 'iiiiiaa48',
    0xab0d: 'iiiiaaaE',
    0xab0e: 'iiiiiiab01',
    0xab0f: 'jjjjbbbD',
    0xab10: 'xxxxxxbb10',
    0xab11: 'kkkkbbbD',
    0xab12: 'xxxxxba59',
    0xab13: 'hhhhaaaE',
    0xab14: 'xxxxxxxabbl',
    0xab15: 'mmmmbaaE',
    0xab16: 'xxxxxxxabbL',
    0xab17: 'nnnnabbD',
    0xab18: 'qqqqqqqbbaM',
    0xab19: 'ggggbbaE',
    0xab1a: 'qqqqqqqbbam',
    0xab1b: 'ddddaabD',
    0xab1c: 'qqqqqaa49',
    0xab1d: 'aaaababD',
    0xab1e: 'qqqqqqab00',
    0xab1f: 'bbbbabaE',
    0xab20: 'sssssbb59',
    0xab21: 'ccccbabD',
    0xab22: 'pppppaa49',
    0xab23: 'ppppbbbE',
    0xab24: 'uuuuuba49',
    0xab25: 'eeeebbaE',
    0xab26: 'vvvvvab59',
    0xab27: 'ffffaabD',
    0xab28: 'YYYYYYYbbaN',
    0xab29: 'oooobaaE',
    0xab2a: 'YYYYYYYbban',
    0xab2b: 'llllabbD',
    0xab2c: 'yyyyyba59',
    0xab2d: 'iiiibbbD',
    0xab2e: 'zzzzzab49',
    0xab2f: 'jjjjaaaE',
    0xab30: 'kkkkkbb58',
    0xab31: 'kkkkaaaE',
    0xab32: 'hhhhhaa48',
    0xab33: 'hhhhbbbD',
    0xab34: 'mmmmmba48',
    0xab35: 'mmmmabbD',
    0xab36: 'nnnnnab58',
    0xab37: 'nnnnbaaE',
    0xab38: 'gggggbb48',
    0xab39: 'ggggaabD',
    0xab3a: 'dddddaa58',
    0xab3b: 'ddddbbaE',
    0xab3c: 'aaaaaba58',
    0xab3d: 'aaaaabaE',
    0xab3e: 'bbbbbab48',
    0xab3f: 'bbbbbabD',
    0xab40: 'cccccba58',
    0xab41: 'ccccaabD',
    0xab42: 'CCCCCCba12',
    0xab43: 'ppppabbE',
    0xab44: 'eeeeebb48',
    0xab45: 'eeeeabaE',
    0xab46: 'fffffaa58',
    0xab47: 'ffffbabD',
    0xab48: 'oooooba48',
    0xab49: 'ooooaaaE',
    0xab4a: 'lllllab58',
    0xab4b: 'llllbbbD',
    0xab4c: 'iiiiibb58',
    0xab4d: 'iiiiabbD',
    0xab4e: 'jjjjjaa48',
    0xab4f: 'jjjjbaaE',
    0xab50: 'XXXXXXab03',
    0xab51: 'kkkkbaaE',
    0xab52: 'xxxxxab49',
    0xab53: 'hhhhabbD',
    0xab54: '=====ba88',
    0xab55: 'mmmmbbbD',
    0xab56: '>>>>>ab98',
    0xab57: 'nnnnaaaE',
    0xab58: 'wwwwwba49',
    0xab59: 'ggggbabD',
    0xab5a: 'tttttab59',
    0xab5b: 'ddddabaE',
    0xab5c: 'qqqqqbb59',
    0xab5d: 'aaaabbaE',
    0xab5e: 'rrrrraa49',
    0xab5f: 'bbbbaabD',
    0xab60: 'ppppppab00',
    0xab61: 'ccccbbaE',
    0xab62: 'pppppbb59',
    0xab63: 'ppppbaaD',
    0xab64: 'uuuuuab59',
    0xab65: 'eeeebabD',
    0xab66: 'uuuuuubb00',
    0xab67: 'ffffabaE',
    0xab68: '9999999bbaJ',
    0xab69: 'oooobbbD',
    0xab6a: '9999999bbaj',
    0xab6b: 'llllaaaE',
    0xab6c: 'yyyyyab49',
    0xab6d: 'iiiibaaE',
    0xab6e: 'yyyyyybb10',
    0xab6f: 'jjjjabbD',
    0xab70: 'hhhhhhab01',
    0xab71: 'kkkkabbD',
    0xab72: 'hhhhhbb58',
    0xab73: 'hhhhbaaE',
    0xab74: 'mmmmmab58',
    0xab75: 'mmmmaaaE',
    0xab76: 'mmmmmmbb01',
    0xab77: 'nnnnbbbD',
    0xab78: 'ddddddab11',
    0xab79: 'ggggabaE',
    0xab7a: 'dddddbb48',
    0xab7b: 'ddddbabD',
    0xab7c: 'aaaaaab48',
    0xab7d: 'aaaaaabD',
    0xab7e: 'aaaaaabb11',
    0xab7f: 'bbbbbbaE',
    0xab80: 'cccccaa58',
    0xab81: 'ccccabbD',
    0xab82: 'ccccccbb11',
    0xab83: 'ppppaabE',
    0xab84: 'eeeeeab48',
    0xab85: 'eeeeaaaE',
    0xab86: 'eeeeeeba01',
    0xab87: 'ffffbbbD',
    0xab88: 'llllllaa11',
    0xab89: 'ooooabaE',
    0xab8a: 'lllllbb58',
    0xab8b: 'llllbabD',
    0xab8c: 'iiiiiab58',
    0xab8d: 'iiiiaabD',
    0xab8e: 'iiiiiiba11',
    0xab8f: 'jjjjbbaE',
    0xab90: 'xxxxxxaa00',
    0xab91: 'kkkkbbaE',
    0xab92: 'xxxxxbb49',
    0xab93: 'hhhhaabD',
    0xab94: '11111111bbaH',
    0xab95: 'mmmmbabD',
    0xab96: '11111111bbaX',
    0xab97: 'nnnnabaE',
    0xab98: 'ttttttaa10',
    0xab99: 'ggggbbbD',
    0xab9a: 'tttttbb59',
    0xab9b: 'ddddaaaE',
    0xab9c: 'qqqqqab59',
    0xab9d: 'aaaabaaE',
    0xab9e: 'qqqqqqba10',
    0xab9f: 'bbbbabbD',
    0xaba0: 'sssssba49',
    0xaba1: 'ccccbaaE',
    0xaba2: 'pppppab59',
    0xaba3: 'ppppbbaD',
    0xaba4: 'uuuuubb59',
    0xaba5: 'eeeebbbD',
    0xaba6: 'vvvvvaa49',
    0xaba7: 'ffffaaaE',
    0xaba8: '?????bb98',
    0xaba9: 'oooobabD',
    0xabaa: '<<<<<aa88',
    0xabab: 'llllabaE',
    0xabac: 'yyyyybb49',
    0xabad: 'iiiibbaE',
    0xabae: 'zzzzzaa59',
    0xabaf: 'jjjjaabD',
    0xabb0: 'kkkkkba48',
    0xabb1: 'kkkkaabD',
    0xabb2: 'hhhhhab58',
    0xabb3: 'hhhhbbaE',
    0xabb4: 'mmmmmbb58',
    0xabb5: 'mmmmabaE',
    0xabb6: 'nnnnnaa48',
    0xabb7: 'nnnnbabD',
    0xabb8: 'gggggba58',
    0xabb9: 'ggggaaaE',
    0xabba: 'dddddab48',
    0xabbb: 'ddddbbbD',
    0xabbc: 'aaaaabb48',
    0xabbd: 'aaaaabbD',
    0xabbe: 'bbbbbaa58',
    0xabbf: 'bbbbbaaE',
    0xabc0: 'cccccbb48',
    0xabc1: 'ccccaaaE',
    0xabc2: 'CCCCCCab02',
    0xabc3: 'ppppabaD',
    0xabc4: 'eeeeeba58',
    0xabc5: 'eeeeabbD',
    0xabc6: 'fffffab48',
    0xabc7: 'ffffbaaE',
    0xabc8: 'ooooobb58',
    0xabc9: 'ooooaabD',
    0xabca: 'lllllaa48',
    0xabcb: 'llllbbaE',
    0xabcc: 'iiiiiba48',
    0xabcd: 'iiiiabaE',
    0xabce: 'jjjjjab58',
    0xabcf: 'jjjjbabD',
    0xabd0: 'QQQQQQQQbbal',
    0xabd1: 'kkkkbabD',
    0xabd2: 'xxxxxaa59',
    0xabd3: 'hhhhabaE',
    0xabd4: 'QQQQQQQQbbaL',
    0xabd5: 'mmmmbbaE',
    0xabd6: 'QQQQQQQQbbiX',
    0xabd7: 'nnnnaabD',
    0xabd8: 'wwwwwbb59',
    0xabd9: 'ggggbaaE',
    0xabda: 'tttttaa49',
    0xabdb: 'ddddabbD',
    0xabdc: 'qqqqqba49',
    0xabdd: 'aaaabbbD',
    0xabde: 'rrrrrab59',
    0xabdf: 'bbbbaaaE',
    0xabe0: 'ppppppba10',
    0xabe1: 'ccccbbbD',
    0xabe2: 'pppppba49',
    0xabe3: 'ppppbabE',
    0xabe4: 'pppppppbbam',
    0xabe5: 'eeeebaaE',
    0xabe6: 'pppppppbbaM',
    0xabe7: 'ffffabbD',
    0xabe8: 'ppppppppabjZ',
    0xabe9: 'oooobbaE',
    0xabea: 'ppppppppabbN',
    0xabeb: 'llllaabD',
    0xabec: 'ppppppppabjz',
    0xabed: 'iiiibabD',
    0xabee: 'ppppppppabbn',
    0xabef: 'jjjjabaE',
    0xabf0: 'aaaaaaaabban',
    0xabf1: 'aaaaaaaabbaf',
    0xabf2: 'aaaaaaaabbiz',
    0xabf3: 'aaaaaaaabbav',
    0xabf4: 'aaaaaaaabbaN',
    0xabf5: 'aaaaaaaabbaF',
    0xabf6: 'aaaaaaaabbiZ',
    0xabf7: 'aaaaaaaabbaV',
    0xabf8: 'aaaaaaaabbM',
    0xabf9: 'aaaaaaaabhX',
    0xabfa: 'aaaaaaaabbm',
    0xabfb: 'aaaaaaaabhx',
    0xabfc: 'aaaaaaa58',
    0xabfd: 'aaaaaaaE',
    0xabfe: 'aaaaaaaa01',
    0xabff: 'aaaaaaaab8',
    0xac00: 'cccccab19',
    0xac01: 'ccccabaB',
    0xac02: 'ccccccab00',
    0xac03: 'ccccccccb9',
    0xac04: 'cccccccaabM',
    0xac05: 'cccccccaahX',
    0xac06: 'cccccccaabm',
    0xac07: 'cccccccaahx',
    0xac08: 'ccccccccbabO',
    0xac09: 'ccccccccbabG',
    0xac0a: 'ccccccccbahZ',
    0xac0b: 'ccccccccbabW',
    0xac0c: 'ccccccccbabo',
    0xac0d: 'ccccccccbabg',
    0xac0e: 'ccccccccbahz',
    0xac0f: 'ccccccccbabw',
    0xac10: 'qqqqqqqqbbbn',
    0xac11: 'kkkkbbbC',
    0xac12: 'qqqqqqqqbbjz',
    0xac13: 'hhhhaaaB',
    0xac14: 'qqqqqqqqbbbN',
    0xac15: 'mmmmbaaB',
    0xac16: 'qqqqqqqqbbjZ',
    0xac17: 'nnnnabbC',
    0xac18: 'qqqqqqqabbL',
    0xac19: 'ggggbbaB',
    0xac1a: 'qqqqqqqabbl',
    0xac1b: 'ddddaabC',
    0xac1c: 'qqqqqaa18',
    0xac1d: 'aaaababC',
    0xac1e: 'qqqqqqaa01',
    0xac1f: 'bbbbabaB',
    0xac20: 'sssssbb08',
    0xac21: 'ccccbabC',
    0xac22: 'pppppaa18',
    0xac23: 'ppppbbbB',
    0xac24: 'uuuuuba18',
    0xac25: 'eeeebbaB',
    0xac26: 'vvvvvab08',
    0xac27: 'ffffaabC',
    0xac28: 'PPPPPPPPabiX',
    0xac29: 'oooobaaB',
    0xac2a: 'PPPPPPPPabaL',
    0xac2b: 'llllabbC',
    0xac2c: 'yyyyyba08',
    0xac2d: 'iiiibbbC',
    0xac2e: 'zzzzzab18',
    0xac2f: 'jjjjaaaB',
    0xac30: 'kkkkkbb09',
    0xac31: 'kkkkaaaB',
    0xac32: 'hhhhhaa19',
    0xac33: 'hhhhbbbC',
    0xac34: 'mmmmmba19',
    0xac35: 'mmmmabbC',
    0xac36: 'nnnnnab09',
    0xac37: 'nnnnbaaB',
    0xac38: 'gggggbb19',
    0xac39: 'ggggaabC',
    0xac3a: 'dddddaa09',
    0xac3b: 'ddddbbaB',
    0xac3c: 'aaaaaba09',
    0xac3d: 'aaaaabaB',
    0xac3e: 'bbbbbab19',
    0xac3f: 'bbbbbabC',
    0xac40: 'cccccba09',
    0xac41: 'ccccaabC',
    0xac42: 'CCCCCCbb13',
    0xac43: 'ppppabbB',
    0xac44: 'eeeeebb19',
    0xac45: 'eeeeabaB',
    0xac46: 'fffffaa09',
    0xac47: 'ffffbabC',
    0xac48: 'oooooba19',
    0xac49: 'ooooaaaB',
    0xac4a: 'lllllab09',
    0xac4b: 'llllbbbC',
    0xac4c: 'iiiiibb09',
    0xac4d: 'iiiiabbC',
    0xac4e: 'jjjjjaa19',
    0xac4f: 'jjjjbaaB',
    0xac50: 'XXXXXXaa02',
    0xac51: 'kkkkbaaB',
    0xac52: 'xxxxxab18',
    0xac53: 'hhhhabbC',
    0xac54: '=====bi99',
    0xac55: 'mmmmbbbC',
    0xac56: '>>>>>ah99',
    0xac57: 'nnnnaaaB',
    0xac58: 'wwwwwba18',
    0xac59: 'ggggbabC',
    0xac5a: 'tttttab08',
    0xac5b: 'ddddabaB',
    0xac5c: 'qqqqqbb08',
    0xac5d: 'aaaabbaB',
    0xac5e: 'rrrrraa18',
    0xac5f: 'bbbbaabC',
    0xac60: 'ppppppaa01',
    0xac61: 'ccccbbaB',
    0xac62: 'pppppbb08',
    0xac63: 'ppppbaaC',
    0xac64: 'uuuuuab08',
    0xac65: 'eeeebabC',
    0xac66: 'uuuuuuba01',
    0xac67: 'ffffabaB',
    0xac68: '00000000abaX',
    0xac69: 'oooobbbC',
    0xac6a: '00000000abaH',
    0xac6b: 'llllaaaB',
    0xac6c: 'yyyyyab18',
    0xac6d: 'iiiibaaB',
    0xac6e: 'yyyyyyba11',
    0xac6f: 'jjjjabbC',
    0xac70: 'hhhhhhaa00',
    0xac71: 'kkkkabbC',
    0xac72: 'hhhhhbb09',
    0xac73: 'hhhhbaaB',
    0xac74: 'mmmmmab09',
    0xac75: 'mmmmaaaB',
    0xac76: 'mmmmmmba00',
    0xac77: 'nnnnbbbC',
    0xac78: 'ddddddaa10',
    0xac79: 'ggggabaB',
    0xac7a: 'dddddbb19',
    0xac7b: 'ddddbabC',
    0xac7c: 'aaaaaab19',
    0xac7d: 'aaaaaabC',
    0xac7e: 'aaaaaaba10',
    0xac7f: 'bbbbbbaB',
    0xac80: 'cccccaa09',
    0xac81: 'ccccabbC',
    0xac82: 'ccccccba10',
    0xac83: 'ppppaabB',
    0xac84: 'eeeeeab19',
    0xac85: 'eeeeaaaB',
    0xac86: 'eeeeeebb00',
    0xac87: 'ffffbbbC',
    0xac88: 'llllllab10',
    0xac89: 'ooooabaB',
    0xac8a: 'lllllbb09',
    0xac8b: 'llllbabC',
    0xac8c: 'iiiiiab09',
    0xac8d: 'iiiiaabC',
    0xac8e: 'iiiiiibb10',
    0xac8f: 'jjjjbbaB',
    0xac90: 'xxxxxxab01',
    0xac91: 'kkkkbbaB',
    0xac92: 'xxxxxbb18',
    0xac93: 'hhhhaabC',
    0xac94: '8888888abbk',
    0xac95: 'mmmmbabC',
    0xac96: '8888888abbK',
    0xac97: 'nnnnabaB',
    0xac98: 'ttttttab11',
    0xac99: 'ggggbbbC',
    0xac9a: 'tttttbb08',
    0xac9b: 'ddddaaaB',
    0xac9c: 'qqqqqab08',
    0xac9d: 'aaaabaaB',
    0xac9e: 'qqqqqqbb11',
    0xac9f: 'bbbbabbC',
    0xaca0: 'sssssba18',
    0xaca1: 'ccccbaaB',
    0xaca2: 'pppppab08',
    0xaca3: 'ppppbbaC',
    0xaca4: 'uuuuubb08',
    0xaca5: 'eeeebbbC',
    0xaca6: 'vvvvvaa18',
    0xaca7: 'ffffaaaB',
    0xaca8: '?????bh99',
    0xaca9: 'oooobabC',
    0xacaa: '<<<<<ai99',
    0xacab: 'llllabaB',
    0xacac: 'yyyyybb18',
    0xacad: 'iiiibbaB',
    0xacae: 'zzzzzaa08',
    0xacaf: 'jjjjaabC',
    0xacb0: 'kkkkkba19',
    0xacb1: 'kkkkaabC',
    0xacb2: 'hhhhhab09',
    0xacb3: 'hhhhbbaB',
    0xacb4: 'mmmmmbb09',
    0xacb5: 'mmmmabaB',
    0xacb6: 'nnnnnaa19',
    0xacb7: 'nnnnbabC',
    0xacb8: 'gggggba09',
    0xacb9: 'ggggaaaB',
    0xacba: 'dddddab19',
    0xacbb: 'ddddbbbC',
    0xacbc: 'aaaaabb19',
    0xacbd: 'aaaaabbC',
    0xacbe: 'bbbbbaa09',
    0xacbf: 'bbbbbaaB',
    0xacc0: 'cccccbb19',
    0xacc1: 'ccccaaaB',
    0xacc2: 'CCCCCCaa03',
    0xacc3: 'ppppabaC',
    0xacc4: 'eeeeeba09',
    0xacc5: 'eeeeabbC',
    0xacc6: 'fffffab19',
    0xacc7: 'ffffbaaB',
    0xacc8: 'ooooobb09',
    0xacc9: 'ooooaabC',
    0xacca: 'lllllaa19',
    0xaccb: 'llllbbaB',
    0xaccc: 'iiiiiba19',
    0xaccd: 'iiiiabaB',
    0xacce: 'jjjjjab09',
    0xaccf: 'jjjjbabC',
    0xacd0: 'XXXXXXbb12',
    0xacd1: 'kkkkbabC',
    0xacd2: 'xxxxxaa08',
    0xacd3: 'hhhhabaB',
    0xacd4: 'XXXXXXXabbo',
    0xacd5: 'mmmmbbaB',
    0xacd6: 'XXXXXXXabbO',
    0xacd7: 'nnnnaabC',
    0xacd8: 'wwwwwbb08',
    0xacd9: 'ggggbaaB',
    0xacda: 'tttttaa18',
    0xacdb: 'ddddabbC',
    0xacdc: 'qqqqqba18',
    0xacdd: 'aaaabbbC',
    0xacde: 'rrrrrab08',
    0xacdf: 'bbbbaaaB',
    0xace0: 'ppppppbb11',
    0xace1: 'ccccbbbC',
    0xace2: 'pppppba18',
    0xace3: 'ppppbabB',
    0xace4: 'pppppppabbl',
    0xace5: 'eeeebaaB',
    0xace6: 'pppppppabbL',
    0xace7: 'ffffabbC',
    0xace8: 'yyyyyyybbaM',
    0xace9: 'oooobbaB',
    0xacea: 'yyyyyyybbam',
    0xaceb: 'llllaabC',
    0xacec: 'yyyyyaa08',
    0xaced: 'iiiibabC',
    0xacee: 'yyyyyyab01',
    0xacef: 'jjjjabaB',
    0xacf0: 'hhhhhhbb10',
    0xacf1: 'kkkkabaB',
    0xacf2: 'hhhhhba19',
    0xacf3: 'hhhhbabC',
    0xacf4: 'hhhhhhhabbm',
    0xacf5: 'hhhhhhhabhx',
    0xacf6: 'hhhhhhhabbM',
    0xacf7: 'hhhhhhhabhX',
    0xacf8: 'aaaaaaabbaL',
    0xacf9: 'aaaaaaabbiX',
    0xacfa: 'aaaaaaabbal',
    0xacfb: 'aaaaaaabbix',
    0xacfc: 'aaaaaaa09',
    0xacfd: 'aaaaaaaB',
    0xacfe: 'aaaaaaab00',
    0xacff: 'bbbbbbbC',
    0xad00: 'cccccab09',
    0xad01: 'ccccabaC',
    0xad02: 'ccccccaa10',
    0xad03: 'eeeeeeebahx',
    0xad04: 'cccccccbbaL',
    0xad05: 'cccccccbbiX',
    0xad06: 'cccccccbbal',
    0xad07: 'cccccccbbix',
    0xad08: 'iiiiiiibaaL',
    0xad09: 'iiiiiiibaiX',
    0xad0a: 'iiiiiiibaal',
    0xad0b: 'iiiiiiibaix',
    0xad0c: 'iiiiiaa09',
    0xad0d: 'iiiiaaaC',
    0xad0e: 'iiiiiiab10',
    0xad0f: 'jjjjbbbB',
    0xad10: 'xxxxxxbb01',
    0xad11: 'kkkkbbbB',
    0xad12: 'xxxxxba18',
    0xad13: 'hhhhaaaC',
    0xad14: 'xxxxxxxaabl',
    0xad15: 'mmmmbaaC',
    0xad16: 'xxxxxxxaabL',
    0xad17: 'nnnnabbB',
    0xad18: 'qqqqqqqbaaM',
    0xad19: 'ggggbbaC',
    0xad1a: 'qqqqqqqbaam',
    0xad1b: 'ddddaabB',
    0xad1c: 'qqqqqaa08',
    0xad1d: 'aaaababB',
    0xad1e: 'qqqqqqab11',
    0xad1f: 'bbbbabaC',
    0xad20: 'sssssbb18',
    0xad21: 'ccccbabB',
    0xad22: 'pppppaa08',
    0xad23: 'ppppbbbC',
    0xad24: 'uuuuuba08',
    0xad25: 'eeeebbaC',
    0xad26: 'vvvvvab18',
    0xad27: 'ffffaabB',
    0xad28: 'YYYYYYYbaaN',
    0xad29: 'oooobaaC',
    0xad2a: 'YYYYYYYbaan',
    0xad2b: 'llllabbB',
    0xad2c: 'yyyyyba18',
    0xad2d: 'iiiibbbB',
    0xad2e: 'zzzzzab08',
    0xad2f: 'jjjjaaaC',
    0xad30: 'kkkkkbb19',
    0xad31: 'kkkkaaaC',
    0xad32: 'hhhhhaa09',
    0xad33: 'hhhhbbbB',
    0xad34: 'mmmmmba09',
    0xad35: 'mmmmabbB',
    0xad36: 'nnnnnab19',
    0xad37: 'nnnnbaaC',
    0xad38: 'gggggbb09',
    0xad39: 'ggggaabB',
    0xad3a: 'dddddaa19',
    0xad3b: 'ddddbbaC',
    0xad3c: 'aaaaaba19',
    0xad3d: 'aaaaabaC',
    0xad3e: 'bbbbbab09',
    0xad3f: 'bbbbbabB',
    0xad40: 'cccccba19',
    0xad41: 'ccccaabB',
    0xad42: 'CCCCCCba03',
    0xad43: 'ppppabbC',
    0xad44: 'eeeeebb09',
    0xad45: 'eeeeabaC',
    0xad46: 'fffffaa19',
    0xad47: 'ffffbabB',
    0xad48: 'oooooba09',
    0xad49: 'ooooaaaC',
    0xad4a: 'lllllab19',
    0xad4b: 'llllbbbB',
    0xad4c: 'iiiiibb19',
    0xad4d: 'iiiiabbB',
    0xad4e: 'jjjjjaa09',
    0xad4f: 'jjjjbaaC',
    0xad50: 'XXXXXXab12',
    0xad51: 'kkkkbaaC',
    0xad52: 'xxxxxab08',
    0xad53: 'hhhhabbB',
    0xad54: '=====bi89',
    0xad55: 'mmmmbbbB',
    0xad56: '>>>>>ah89',
    0xad57: 'nnnnaaaC',
    0xad58: 'wwwwwba08',
    0xad59: 'ggggbabB',
    0xad5a: 'tttttab18',
    0xad5b: 'ddddabaC',
    0xad5c: 'qqqqqbb18',
    0xad5d: 'aaaabbaC',
    0xad5e: 'rrrrraa08',
    0xad5f: 'bbbbaabB',
    0xad60: 'ppppppab11',
    0xad61: 'ccccbbaC',
    0xad62: 'pppppbb18',
    0xad63: 'ppppbaaB',
    0xad64: 'uuuuuab18',
    0xad65: 'eeeebabB',
    0xad66: 'uuuuuubb11',
    0xad67: 'ffffabaC',
    0xad68: '9999999baaJ',
    0xad69: 'oooobbbB',
    0xad6a: '9999999baaj',
    0xad6b: 'llllaaaC',
    0xad6c: 'yyyyyab08',
    0xad6d: 'iiiibaaC',
    0xad6e: 'yyyyyybb01',
    0xad6f: 'jjjjabbB',
    0xad70: 'hhhhhhab10',
    0xad71: 'kkkkabbB',
    0xad72: 'hhhhhbb19',
    0xad73: 'hhhhbaaC',
    0xad74: 'mmmmmab19',
    0xad75: 'mmmmaaaC',
    0xad76: 'mmmmmmbb10',
    0xad77: 'nnnnbbbB',
    0xad78: 'ddddddab00',
    0xad79: 'ggggabaC',
    0xad7a: 'dddddbb09',
    0xad7b: 'ddddbabB',
    0xad7c: 'aaaaaab09',
    0xad7d: 'aaaaaabB',
    0xad7e: 'aaaaaabb00',
    0xad7f: 'bbbbbbaC',
    0xad80: 'cccccaa19',
    0xad81: 'ccccabbB',
    0xad82: 'ccccccbb00',
    0xad83: 'ppppaabC',
    0xad84: 'eeeeeab09',
    0xad85: 'eeeeaaaC',
    0xad86: 'eeeeeeba10',
    0xad87: 'ffffbbbB',
    0xad88: 'llllllaa00',
    0xad89: 'ooooabaC',
    0xad8a: 'lllllbb19',
    0xad8b: 'llllbabB',
    0xad8c: 'iiiiiab19',
    0xad8d: 'iiiiaabB',
    0xad8e: 'iiiiiiba00',
    0xad8f: 'jjjjbbaC',
    0xad90: 'xxxxxxaa11',
    0xad91: 'kkkkbbaC',
    0xad92: 'xxxxxbb08',
    0xad93: 'hhhhaabB',
    0xad94: '11111111abaH',
    0xad95: 'mmmmbabB',
    0xad96: '11111111abaX',
    0xad97: 'nnnnabaC',
    0xad98: 'ttttttaa01',
    0xad99: 'ggggbbbB',
    0xad9a: 'tttttbb18',
    0xad9b: 'ddddaaaC',
    0xad9c: 'qqqqqab18',
    0xad9d: 'aaaabaaC',
    0xad9e: 'qqqqqqba01',
    0xad9f: 'bbbbabbB',
    0xada0: 'sssssba08',
    0xada1: 'ccccbaaC',
    0xada2: 'pppppab18',
    0xada3: 'ppppbbaB',
    0xada4: 'uuuuubb18',
    0xada5: 'eeeebbbB',
    0xada6: 'vvvvvaa08',
    0xada7: 'ffffaaaC',
    0xada8: '?????bh89',
    0xada9: 'oooobabB',
    0xadaa: '<<<<<ai89',
    0xadab: 'llllabaC',
    0xadac: 'yyyyybb08',
    0xadad: 'iiiibbaC',
    0xadae: 'zzzzzaa18',
    0xadaf: 'jjjjaabB',
    0xadb0: 'kkkkkba09',
    0xadb1: 'kkkkaabB',
    0xadb2: 'hhhhhab19',
    0xadb3: 'hhhhbbaC',
    0xadb4: 'mmmmmbb19',
    0xadb5: 'mmmmabaC',
    0xadb6: 'nnnnnaa09',
    0xadb7: 'nnnnbabB',
    0xadb8: 'gggggba19',
    0xadb9: 'ggggaaaC',
    0xadba: 'dddddab09',
    0xadbb: 'ddddbbbB',
    0xadbc: 'aaaaabb09',
    0xadbd: 'aaaaabbB',
    0xadbe: 'bbbbbaa19',
    0xadbf: 'bbbbbaaC',
    0xadc0: 'cccccbb09',
    0xadc1: 'ccccaaaC',
    0xadc2: 'CCCCCCab13',
    0xadc3: 'ppppabaB',
    0xadc4: 'eeeeeba19',
    0xadc5: 'eeeeabbB',
    0xadc6: 'fffffab09',
    0xadc7: 'ffffbaaC',
    0xadc8: 'ooooobb19',
    0xadc9: 'ooooaabB',
    0xadca: 'lllllaa09',
    0xadcb: 'llllbbaC',
    0xadcc: 'iiiiiba09',
    0xadcd: 'iiiiabaC',
    0xadce: 'jjjjjab19',
    0xadcf: 'jjjjbabB',
    0xadd0: 'CCCCCCCCCbbbq',
    0xadd1: 'kkkkbabB',
    0xadd2: 'xxxxxaa18',
    0xadd3: 'hhhhabaC',
    0xadd4: 'CCCCCCCCCbbba',
    0xadd5: 'mmmmbbaC',
    0xadd6: 'CCCCCCCCCbbbi',
    0xadd7: 'nnnnaabB',
    0xadd8: 'wwwwwbb18',
    0xadd9: 'ggggbaaC',
    0xadda: 'tttttaa08',
    0xaddb: 'ddddabbB',
    0xaddc: 'qqqqqba08',
    0xaddd: 'aaaabbbB',
    0xadde: 'rrrrrab18',
    0xaddf: 'bbbbaaaC',
    0xade0: 'aaaaaaaaababB',
    0xade1: 'aaaaaaaaababF',
    0xade2: 'aaaaaaaaababJ',
    0xade3: 'aaaaaaaaababN',
    0xade4: 'aaaaaaaaababR',
    0xade5: 'aaaaaaaaababV',
    0xade6: 'aaaaaaaaababZ',
    0xade7: 'aaaaaaaaabajZ',
    0xade8: 'aaaaaaaaababb',
    0xade9: 'aaaaaaaaababf',
    0xadea: 'aaaaaaaaababj',
    0xadeb: 'aaaaaaaaababn',
    0xadec: 'aaaaaaaaababr',
    0xaded: 'aaaaaaaaababv',
    0xadee: 'aaaaaaaaababz',
    0xadef: 'aaaaaaaaabajz',
    0xadf0: 'aaaaaaaaaban',
    0xadf1: 'aaaaaaaaabaf',
    0xadf2: 'aaaaaaaaabiz',
    0xadf3: 'aaaaaaaaabav',
    0xadf4: 'aaaaaaaaabaN',
    0xadf5: 'aaaaaaaaabaF',
    0xadf6: 'aaaaaaaaabiZ',
    0xadf7: 'aaaaaaaaabaV',
    0xadf8: 'aaaaaaaaabM',
    0xadf9: 'aaaaaaaaahX',
    0xadfa: 'aaaaaaaaabm',
    0xadfb: 'aaaaaaaaahx',
    0xadfc: 'aaaaaaa19',
    0xadfd: 'aaaaaaaC',
    0xadfe: 'aaaaaaaa10',
    0xadff: 'aaaaaaaaa8',
    0xae00: 'cccccab18',
    0xae01: 'ccccabcA',
    0xae02: 'ccccccab10',
    0xae03: 'cccccccab9',
    0xae04: 'cccccccabaL',
    0xae05: 'cccccccabiX',
    0xae06: 'cccccccabal',
    0xae07: 'cccccccabix',
    0xae08: 'ccccccccabaN',
    0xae09: 'ccccccccabaF',
    0xae0a: 'ccccccccabiZ',
    0xae0b: 'ccccccccabaV',
    0xae0c: 'ccccccccaban',
    0xae0d: 'ccccccccabaf',
    0xae0e: 'ccccccccabiz',
    0xae0f: 'ccccccccabav',
    0xae10: 'cccccccccabas',
    0xae11: 'cccccccccabaw',
    0xae12: 'cccccccccabcz',
    0xae13: 'cccccccccabkz',
    0xae14: 'cccccccccabac',
    0xae15: 'cccccccccabag',
    0xae16: 'cccccccccabak',
    0xae17: 'cccccccccabao',
    0xae18: 'cccccccccabaS',
    0xae19: 'cccccccccabaW',
    0xae1a: 'cccccccccabcZ',
    0xae1b: 'cccccccccabkZ',
    0xae1c: 'cccccccccabaC',
    0xae1d: 'aaaababA',
    0xae1e: 'cccccccccabaK',
    0xae1f: 'bbbbabcA',
    0xae20: 'ccccccccccabbn',
    0xae21: 'ccccbabA',
    0xae22: 'ccccccccccabbj',
    0xae23: 'ccccccccccabbh',
    0xae24: 'ccccccccccabbf',
    0xae25: 'ccccccccccabbd',
    0xae26: 'ccccccccccabbb',
    0xae27: 'ccccccccccabdc',
    0xae28: 'ccccccccccabjz',
    0xae29: 'ccccccccccabhy',
    0xae2a: 'ccccccccccabbz',
    0xae2b: 'ccccccccccabbx',
    0xae2c: 'ccccccccccabbv',
    0xae2d: 'ccccccccccabbt',
    0xae2e: 'ccccccccccabbr',
    0xae2f: 'ccccccccccabbp',
    0xae30: 'ccccccccccabbN',
    0xae31: 'ccccccccccabbL',
    0xae32: 'ccccccccccabbJ',
    0xae33: 'ccccccccccabbH',
    0xae34: 'ccccccccccabbF',
    0xae35: 'ccccccccccabbD',
    0xae36: 'ccccccccccabbB',
    0xae37: 'ccccccccccabdC',
    0xae38: 'ccccccccccabjZ',
    0xae39: 'ccccccccccabhY',
    0xae3a: 'ccccccccccabbZ',
    0xae3b: 'ccccccccccabbX',
    0xae3c: 'aaaaaba08',
    0xae3d: 'aaaaabcA',
    0xae3e: 'bbbbbab18',
    0xae3f: 'bbbbbabA',
    0xae40: 'cccccba08',
    0xae41: 'ccccaabA',
    0xae42: 'cccccccccccaaad',
    0xae43: 'cccccccccccaaae',
    0xae44: 'cccccccccccaaab',
    0xae45: 'cccccccccccaaac',
    0xae46: 'cccccccccccaaca',
    0xae47: 'cccccccccccaaaa',
    0xae48: 'cccccccccccaaan',
    0xae49: 'cccccccccccaaao',
    0xae4a: 'cccccccccccaaal',
    0xae4b: 'cccccccccccaaam',
    0xae4c: 'cccccccccccaaaj',
    0xae4d: 'cccccccccccaaak',
    0xae4e: 'cccccccccccaaah',
    0xae4f: 'cccccccccccaaai',
    0xae50: 'cccccccccccaaav',
    0xae51: 'cccccccccccaaaw',
    0xae52: 'cccccccccccaaat',
    0xae53: 'cccccccccccaaau',
    0xae54: 'cccccccccccaaar',
    0xae55: 'cccccccccccaaas',
    0xae56: 'cccccccccccaaap',
    0xae57: 'cccccccccccaaaq',
    0xae58: 'cccccccccccaaiz',
    0xae59: 'cccccccccccaakz',
    0xae5a: 'cccccccccccaaix',
    0xae5b: 'cccccccccccaaiy',
    0xae5c: 'cccccccccccaaaz',
    0xae5d: 'aaaabbcA',
    0xae5e: 'cccccccccccaaax',
    0xae5f: 'bbbbaabA',
    0xae60: 'cccccccccccaaaF',
    0xae61: 'ccccbbcA',
    0xae62: 'cccccccccccaaaD',
    0xae63: 'cccccccccccaaaE',
    0xae64: 'cccccccccccaaaB',
    0xae65: 'cccccccccccaaaC',
    0xae66: 'cccccccccccaacA',
    0xae67: 'cccccccccccaaaA',
    0xae68: 'cccccccccccaaaN',
    0xae69: 'cccccccccccaaaO',
    0xae6a: 'cccccccccccaaaL',
    0xae6b: 'cccccccccccaaaM',
    0xae6c: 'cccccccccccaaaJ',
    0xae6d: 'cccccccccccaaaK',
    0xae6e: 'cccccccccccaaaH',
    0xae6f: 'cccccccccccaaaI',
    0xae70: 'cccccccccccaaaV',
    0xae71: 'cccccccccccaaaW',
    0xae72: 'cccccccccccaaaT',
    0xae73: 'cccccccccccaaaU',
    0xae74: 'cccccccccccaaaR',
    0xae75: 'cccccccccccaaaS',
    0xae76: 'cccccccccccaaaP',
    0xae77: 'cccccccccccaaaQ',
    0xae78: 'cccccccccccaaiZ',
    0xae79: 'cccccccccccaakZ',
    0xae7a: 'cccccccccccaaiX',
    0xae7b: 'cccccccccccaaiY',
    0xae7c: 'aaaaaab18',
    0xae7d: 'aaaaaabA',
    0xae7e: 'aaaaaaba00',
    0xae7f: 'bbbbbbcA',
    0xae80: 'cccccaa08',
    0xae81: 'ccccabbA',
    0xae82: 'ccccccba00',
    0xae83: 'ppppaadC',
    0xae84: 'eeeeeab18',
    0xae85: 'eeeeaacA',
    0xae86: 'eeeeeebb10',
    0xae87: 'ffffbbbA',
    0xae88: 'llllllab00',
    0xae89: 'ooooabcA',
    0xae8a: 'lllllbb08',
    0xae8b: 'llllbabA',
    0xae8c: 'iiiiiab08',
    0xae8d: 'iiiiaabA',
    0xae8e: 'iiiiiibb00',
    0xae8f: 'jjjjbbcA',
    0xae90: 'xxxxxxab11',
    0xae91: 'kkkkbbcA',
    0xae92: 'xxxxxbb19',
    0xae93: 'hhhhaabA',
    0xae94: '8888888aaaj',
    0xae95: 'mmmmbabA',
    0xae96: '8888888aaaJ',
    0xae97: 'nnnnabcA',
    0xae98: 'ttttttab01',
    0xae99: 'ggggbbbA',
    0xae9a: 'tttttbb09',
    0xae9b: 'ddddaacA',
    0xae9c: 'qqqqqab09',
    0xae9d: 'aaaabacA',
    0xae9e: 'qqqqqqbb01',
    0xae9f: 'bbbbabbA',
    0xaea0: 'sssssba19',
    0xaea1: 'ccccbacA',
    0xaea2: 'pppppab09',
    0xaea3: 'ppppbbaA',
    0xaea4: 'uuuuubb09',
    0xaea5: 'eeeebbbA',
    0xaea6: 'vvvvvaa19',
    0xaea7: 'ffffaacA',
    0xaea8: '?????bh98',
    0xaea9: 'oooobabA',
    0xaeaa: '<<<<<ai98',
    0xaeab: 'llllabcA',
    0xaeac: 'yyyyybb19',
    0xaead: 'iiiibbcA',
    0xaeae: 'zzzzzaa09',
    0xaeaf: 'jjjjaabA',
    0xaeb0: 'kkkkkba18',
    0xaeb1: 'kkkkaabA',
    0xaeb2: 'hhhhhab08',
    0xaeb3: 'hhhhbbcA',
    0xaeb4: 'mmmmmbb08',
    0xaeb5: 'mmmmabcA',
    0xaeb6: 'nnnnnaa18',
    0xaeb7: 'nnnnbabA',
    0xaeb8: 'gggggba08',
    0xaeb9: 'ggggaacA',
    0xaeba: 'dddddab18',
    0xaebb: 'ddddbbbA',
    0xaebc: 'aaaaabb18',
    0xaebd: 'aaaaabbA',
    0xaebe: 'bbbbbaa08',
    0xaebf: 'bbbbbacA',
    0xaec0: 'cccccbb18',
    0xaec1: 'ccccaacA',
    0xaec2: 'CCCCCCaa13',
    0xaec3: 'ppppabaA',
    0xaec4: 'eeeeeba08',
    0xaec5: 'eeeeabbA',
    0xaec6: 'fffffab18',
    0xaec7: 'ffffbacA',
    0xaec8: 'ooooobb08',
    0xaec9: 'ooooaabA',
    0xaeca: 'lllllaa18',
    0xaecb: 'llllbbcA',
    0xaecc: 'iiiiiba18',
    0xaecd: 'iiiiabcA',
    0xaece: 'jjjjjab08',
    0xaecf: 'jjjjbabA',
    0xaed0: 'XXXXXXbb02',
    0xaed1: 'kkkkbabA',
    0xaed2: 'xxxxxaa09',
    0xaed3: 'hhhhabcA',
    0xaed4: 'XXXXXXXaaan',
    0xaed5: 'mmmmbbcA',
    0xaed6: 'XXXXXXXaaaN',
    0xaed7: 'nnnnaabA',
    0xaed8: 'wwwwwbb09',
    0xaed9: 'ggggbacA',
    0xaeda: 'tttttaa19',
    0xaedb: 'ddddabbA',
    0xaedc: 'qqqqqba19',
    0xaedd: 'aaaabbbA',
    0xaede: 'rrrrrab09',
    0xaedf: 'bbbbaacA',
    0xaee0: 'ppppppbb01',
    0xaee1: 'ccccbbbA',
    0xaee2: 'pppppba19',
    0xaee3: 'ppppbadC',
    0xaee4: 'pppppppaaam',
    0xaee5: 'eeeebacA',
    0xaee6: 'pppppppaaaM',
    0xaee7: 'ffffabbA',
    0xaee8: 'yyyyyyybabL',
    0xaee9: 'oooobbcA',
    0xaeea: 'yyyyyyybabl',
    0xaeeb: 'llllaabA',
    0xaeec: 'yyyyyaa09',
    0xaeed: 'iiiibabA',
    0xaeee: 'yyyyyyab11',
    0xaeef: 'jjjjabcA',
    0xaef0: 'hhhhhhbb00',
    0xaef1: 'kkkkabcA',
    0xaef2: 'hhhhhba18',
    0xaef3: 'hhhhbabA',
    0xaef4: 'hhhhhhhaaal',
    0xaef5: 'hhhhhhhaaix',
    0xaef6: 'hhhhhhhaaaL',
    0xaef7: 'hhhhhhhaaiX',
    0xaef8: 'aaaaaaababM',
    0xaef9: 'aaaaaaabahX',
    0xaefa: 'aaaaaaababm',
    0xaefb: 'aaaaaaabahx',
    0xaefc: 'aaaaaaa08',
    0xaefd: 'aaaaaacA',
    0xaefe: 'aaaaaaab10',
    0xaeff: 'bbbbbbbA',
    0xaf00: 'cccccab08',
    0xaf01: 'ccccabaA',
    0xaf02: 'ccccccaa00',
    0xaf03: 'eeeeeeebbix',
    0xaf04: 'cccccccbabM',
    0xaf05: 'cccccccbahX',
    0xaf06: 'cccccccbabm',
    0xaf07: 'cccccccbahx',
    0xaf08: 'iiiiiiibbbM',
    0xaf09: 'iiiiiiibbhX',
    0xaf0a: 'iiiiiiibbbm',
    0xaf0b: 'iiiiiiibbhx',
    0xaf0c: 'iiiiiaa08',
    0xaf0d: 'iiiiaaaA',
    0xaf0e: 'iiiiiiab00',
    0xaf0f: 'jjjjbbdC',
    0xaf10: 'xxxxxxbb11',
    0xaf11: 'kkkkbbdC',
    0xaf12: 'xxxxxba19',
    0xaf13: 'hhhhaaaA',
    0xaf14: 'xxxxxxxabam',
    0xaf15: 'mmmmbaaA',
    0xaf16: 'xxxxxxxabaM',
    0xaf17: 'nnnnabdC',
    0xaf18: 'qqqqqqqbbbL',
    0xaf19: 'ggggbbaA',
    0xaf1a: 'qqqqqqqbbbl',
    0xaf1b: 'ddddaadC',
    0xaf1c: 'qqqqqaa09',
    0xaf1d: 'aaaabadC',
    0xaf1e: 'qqqqqqab01',
    0xaf1f: 'bbbbabaA',
    0xaf20: 'sssssbb19',
    0xaf21: 'ccccbadC',
    0xaf22: 'pppppaa09',
    0xaf23: 'ppppbbbA',
    0xaf24: 'uuuuuba09',
    0xaf25: 'eeeebbaA',
    0xaf26: 'vvvvvab19',
    0xaf27: 'ffffaadC',
    0xaf28: 'YYYYYYYbbbO',
    0xaf29: 'oooobaaA',
    0xaf2a: 'YYYYYYYbbbo',
    0xaf2b: 'llllabdC',
    0xaf2c: 'yyyyyba19',
    0xaf2d: 'iiiibbdC',
    0xaf2e: 'zzzzzab09',
    0xaf2f: 'jjjjaaaA',
    0xaf30: 'kkkkkbb18',
    0xaf31: 'kkkkaaaA',
    0xaf32: 'hhhhhaa08',
    0xaf33: 'hhhhbbdC',
    0xaf34: 'mmmmmba08',
    0xaf35: 'mmmmabdC',
    0xaf36: 'nnnnnab18',
    0xaf37: 'nnnnbaaA',
    0xaf38: 'gggggbb08',
    0xaf39: 'ggggaadC',
    0xaf3a: 'dddddaa18',
    0xaf3b: 'ddddbbaA',
    0xaf3c: 'aaaaaba18',
    0xaf3d: 'aaaaabaA',
    0xaf3e: 'bbbbbab08',
    0xaf3f: 'bbbbbadC',
    0xaf40: 'cccccba18',
    0xaf41: 'ccccaadC',
    0xaf42: 'CCCCCCba13',
    0xaf43: 'ppppabbA',
    0xaf44: 'eeeeebb08',
    0xaf45: 'eeeeabaA',
    0xaf46: 'fffffaa18',
    0xaf47: 'ffffbadC',
    0xaf48: 'oooooba08',
    0xaf49: 'ooooaaaA',
    0xaf4a: 'lllllab18',
    0xaf4b: 'llllbbdC',
    0xaf4c: 'iiiiibb18',
    0xaf4d: 'iiiiabdC',
    0xaf4e: 'jjjjjaa08',
    0xaf4f: 'jjjjbaaA',
    0xaf50: 'XXXXXXab02',
    0xaf51: 'kkkkbaaA',
    0xaf52: 'xxxxxab09',
    0xaf53: 'hhhhabdC',
    0xaf54: '=====bi88',
    0xaf55: 'mmmmbbdC',
    0xaf56: '>>>>>ah88',
    0xaf57: 'nnnnaaaA',
    0xaf58: 'wwwwwba09',
    0xaf59: 'ggggbadC',
    0xaf5a: 'tttttab19',
    0xaf5b: 'ddddabaA',
    0xaf5c: 'qqqqqbb19',
    0xaf5d: 'aaaabbaA',
    0xaf5e: 'rrrrraa09',
    0xaf5f: 'bbbbaadC',
    0xaf60: 'ppppppab01',
    0xaf61: 'ccccbbaA',
    0xaf62: 'pppppbb19',
    0xaf63: 'ppppbacA',
    0xaf64: 'uuuuuab19',
    0xaf65: 'eeeebadC',
    0xaf66: 'uuuuuubb01',
    0xaf67: 'ffffabaA',
    0xaf68: '9999999bbbK',
    0xaf69: 'oooobbdC',
    0xaf6a: '9999999bbbk',
    0xaf6b: 'llllaaaA',
    0xaf6c: 'yyyyyab09',
    0xaf6d: 'iiiibaaA',
    0xaf6e: 'yyyyyybb11',
    0xaf6f: 'jjjjabdC',
    0xaf70: 'hhhhhhab00',
    0xaf71: 'kkkkabdC',
    0xaf72: 'hhhhhbb18',
    0xaf73: 'hhhhbaaA',
    0xaf74: 'mmmmmab18',
    0xaf75: 'mmmmaaaA',
    0xaf76: 'mmmmmmbb00',
    0xaf77: 'nnnnbbdC',
    0xaf78: 'ddddddab10',
    0xaf79: 'ggggabaA',
    0xaf7a: 'dddddbb08',
    0xaf7b: 'ddddbadC',
    0xaf7c: 'aaaaaab08',
    0xaf7d: 'aaaaaadC',
    0xaf7e: 'aaaaaabb10',
    0xaf7f: 'bbbbbbaA',
    0xaf80: 'cccccaa18',
    0xaf81: 'ccccabdC',
    0xaf82: 'ccccccbb10',
    0xaf83: 'ppppaabA',
    0xaf84: 'eeeeeab08',
    0xaf85: 'eeeeaaaA',
    0xaf86: 'eeeeeeba00',
    0xaf87: 'ffffbbdC',
    0xaf88: 'llllllaa10',
    0xaf89: 'ooooabaA',
    0xaf8a: 'lllllbb18',
    0xaf8b: 'llllbadC',
    0xaf8c: 'iiiiiab18',
    0xaf8d: 'iiiiaadC',
    0xaf8e: 'iiiiiiba10',
    0xaf8f: 'jjjjbbaA',
    0xaf90: 'xxxxxxaa01',
    0xaf91: 'kkkkbbaA',
    0xaf92: 'xxxxxbb09',
    0xaf93: 'hhhhaadC',
    0xaf94: '11111111babI',
    0xaf95: 'mmmmbadC',
    0xaf96: '11111111babY',
    0xaf97: 'nnnnabaA',
    0xaf98: 'ttttttaa11',
    0xaf99: 'ggggbbdC',
    0xaf9a: 'tttttbb19',
    0xaf9b: 'ddddaaaA',
    0xaf9c: 'qqqqqab19',
    0xaf9d: 'aaaabaaA',
    0xaf9e: 'qqqqqqba11',
    0xaf9f: 'bbbbabdC',
    0xafa0: 'sssssba09',
    0xafa1: 'ccccbaaA',
    0xafa2: 'pppppab19',
    0xafa3: 'ppppbbcA',
    0xafa4: 'uuuuubb19',
    0xafa5: 'eeeebbdC',
    0xafa6: 'vvvvvaa09',
    0xafa7: 'ffffaaaA',
    0xafa8: '?????bh88',
    0xafa9: 'oooobadC',
    0xafaa: '<<<<<ai88',
    0xafab: 'llllabaA',
    0xafac: 'yyyyybb09',
    0xafad: 'iiiibbaA',
    0xafae: 'zzzzzaa19',
    0xafaf: 'jjjjaadC',
    0xafb0: 'kkkkkba08',
    0xafb1: 'kkkkaadC',
    0xafb2: 'hhhhhab18',
    0xafb3: 'hhhhbbaA',
    0xafb4: 'mmmmmbb18',
    0xafb5: 'mmmmabaA',
    0xafb6: 'nnnnnaa08',
    0xafb7: 'nnnnbadC',
    0xafb8: 'gggggba18',
    0xafb9: 'ggggaaaA',
    0xafba: 'dddddab08',
    0xafbb: 'ddddbbdC',
    0xafbc: 'aaaaabb08',
    0xafbd: 'aaaaabdC',
    0xafbe: 'bbbbbaa18',
    0xafbf: 'bbbbbaaA',
    0xafc0: 'cccccbb08',
    0xafc1: 'ccccaaaA',
    0xafc2: 'CCCCCCab03',
    0xafc3: 'ppppabcA',
    0xafc4: 'eeeeeba18',
    0xafc5: 'eeeeabdC',
    0xafc6: 'fffffab08',
    0xafc7: 'ffffbaaA',
    0xafc8: 'ooooobb18',
    0xafc9: 'ooooaadC',
    0xafca: 'lllllaa08',
    0xafcb: 'llllbbaA',
    0xafcc: 'iiiiiba08',
    0xafcd: 'iiiiabaA',
    0xafce: 'jjjjjab18',
    0xafcf: 'jjjjbadC',
    0xafd0: 'QQQQQQQQbabm',
    0xafd1: 'kkkkbadC',
    0xafd2: 'xxxxxaa19',
    0xafd3: 'hhhhabaA',
    0xafd4: 'QQQQQQQQbabM',
    0xafd5: 'mmmmbbaA',
    0xafd6: 'QQQQQQQQbahX',
    0xafd7: 'nnnnaadC',
    0xafd8: 'wwwwwbb19',
    0xafd9: 'ggggbaaA',
    0xafda: 'tttttaa09',
    0xafdb: 'ddddabdC',
    0xafdc: 'qqqqqba09',
    0xafdd: 'aaaabbdC',
    0xafde: 'rrrrrab19',
    0xafdf: 'bbbbaaaA',
    0xafe0: 'ppppppba11',
    0xafe1: 'ccccbbdC',
    0xafe2: 'pppppba09',
    0xafe3: 'ppppbabA',
    0xafe4: 'pppppppbbbl',
    0xafe5: 'eeeebaaA',
    0xafe6: 'pppppppbbbL',
    0xafe7: 'ffffabdC',
    0xafe8: 'ppppppppaakZ',
    0xafe9: 'oooobbaA',
    0xafea: 'ppppppppaaaO',
    0xafeb: 'llllaadC',
    0xafec: 'ppppppppaakz',
    0xafed: 'iiiibadC',
    0xafee: 'ppppppppaaao',
    0xafef: 'jjjjabaA',
    0xaff0: 'aaaaaaaababo',
    0xaff1: 'aaaaaaaababg',
    0xaff2: 'aaaaaaaabahz',
    0xaff3: 'aaaaaaaababw',
    0xaff4: 'aaaaaaaababO',
    0xaff5: 'aaaaaaaababG',
    0xaff6: 'aaaaaaaabahZ',
    0xaff7: 'aaaaaaaababW',
    0xaff8: 'aaaaaaaabaL',
    0xaff9: 'aaaaaaaabiX',
    0xaffa: 'aaaaaaaabal',
    0xaffb: 'aaaaaaaabix',
    0xaffc: 'aaaaaaa18',
    0xaffd: 'aaaaaaaA',
    0xaffe: 'aaaaaaaa00',
    0xafff: 'aaaaaaaab9',
    0xb000: 'cccccab17',
    0xb001: 'ccccabiZ',
    0xb002: 'ccccccab07',
    0xb003: 'cccccccam9',
    0xb004: 'cccccccaaaO',
    0xb005: 'cccccccaakZ',
    0xb006: 'cccccccaaao',
    0xb007: 'cccccccaakz',
    0xb008: 'ccccccccbbbN',
    0xb009: 'ccccccccbbbF',
    0xb00a: 'ccccccccbbjZ',
    0xb00b: 'ccccccccbbbV',
    0xb00c: 'ccccccccbbbn',
    0xb00d: 'ccccccccbbbf',
    0xb00e: 'ccccccccbbjz',
    0xb00f: 'ccccccccbbbv',
    0xb010: 'qqqqqqqqbabo',
    0xb011: 'kkkkbbhZ',
    0xb012: 'qqqqqqqqbahz',
    0xb013: 'hhhhaaiZ',
    0xb014: 'qqqqqqqqbabO',
    0xb015: 'mmmmbaiZ',
    0xb016: 'qqqqqqqqbahZ',
    0xb017: 'nnnnabhZ',
    0xb018: 'qqqqqqqabaN',
    0xb019: 'ggggbbiZ',
    0xb01a: 'qqqqqqqaban',
    0xb01b: 'ddddaahZ',
    0xb01c: 'qqqqqaa16',
    0xb01d: 'aaaabahZ',
    0xb01e: 'qqqqqqaa06',
    0xb01f: 'bbbbabiZ',
    0xb020: 'sssssbb06',
    0xb021: 'ccccbahZ',
    0xb022: 'pppppaa16',
    0xb023: 'ppppbbjZ',
    0xb024: 'uuuuuba16',
    0xb025: 'eeeebbiZ',
    0xb026: 'vvvvvab06',
    0xb027: 'ffffaahZ',
    0xb028: 'PPPPPPPPaaiY',
    0xb029: 'oooobaiZ',
    0xb02a: 'PPPPPPPPaaaM',
    0xb02b: 'llllabhZ',
    0xb02c: 'yyyyyba06',
    0xb02d: 'iiiibbhZ',
    0xb02e: 'zzzzzab16',
    0xb02f: 'jjjjaaiZ',
    0xb030: 'kkkkkbb07',
    0xb031: 'kkkkaaiZ',
    0xb032: 'hhhhhaa17',
    0xb033: 'hhhhbbhZ',
    0xb034: 'mmmmmba17',
    0xb035: 'mmmmabhZ',
    0xb036: 'nnnnnab07',
    0xb037: 'nnnnbaiZ',
    0xb038: 'gggggbb17',
    0xb039: 'ggggaahZ',
    0xb03a: 'dddddaa07',
    0xb03b: 'ddddbbiZ',
    0xb03c: 'aaaaaba07',
    0xb03d: 'aaaaabiZ',
    0xb03e: 'bbbbbab17',
    0xb03f: 'bbbbbahZ',
    0xb040: 'cccccba07',
    0xb041: 'ccccaahZ',
    0xb042: 'CCCCCCbb14',
    0xb043: 'ppppabjZ',
    0xb044: 'eeeeebb17',
    0xb045: 'eeeeabiZ',
    0xb046: 'fffffaa07',
    0xb047: 'ffffbahZ',
    0xb048: 'oooooba17',
    0xb049: 'ooooaaiZ',
    0xb04a: 'lllllab07',
    0xb04b: 'llllbbhZ',
    0xb04c: 'iiiiibb07',
    0xb04d: 'iiiiabhZ',
    0xb04e: 'jjjjjaa17',
    0xb04f: 'jjjjbaiZ',
    0xb050: 'XXXXXXaa05',
    0xb051: 'kkkkbaiZ',
    0xb052: 'xxxxxab16',
    0xb053: 'hhhhabhZ',
    0xb054: '=====ba11',
    0xb055: 'mmmmbbhZ',
    0xb056: '>>>>>ab01',
    0xb057: 'nnnnaaiZ',
    0xb058: 'wwwwwba16',
    0xb059: 'ggggbahZ',
    0xb05a: 'tttttab06',
    0xb05b: 'ddddabiZ',
    0xb05c: 'qqqqqbb06',
    0xb05d: 'aaaabbiZ',
    0xb05e: 'rrrrraa16',
    0xb05f: 'bbbbaahZ',
    0xb060: 'ppppppaa06',
    0xb061: 'ccccbbiZ',
    0xb062: 'pppppbb06',
    0xb063: 'ppppbakZ',
    0xb064: 'uuuuuab06',
    0xb065: 'eeeebahZ',
    0xb066: 'uuuuuuba06',
    0xb067: 'ffffabiZ',
    0xb068: '00000000aaaY',
    0xb069: 'oooobbhZ',
    0xb06a: '00000000aaaI',
    0xb06b: 'llllaaiZ',
    0xb06c: 'yyyyyab16',
    0xb06d: 'iiiibaiZ',
    0xb06e: 'yyyyyyba16',
    0xb06f: 'jjjjabhZ',
    0xb070: 'hhhhhhaa07',
    0xb071: 'kkkkabhZ',
    0xb072: 'hhhhhbb07',
    0xb073: 'hhhhbaiZ',
    0xb074: 'mmmmmab07',
    0xb075: 'mmmmaaiZ',
    0xb076: 'mmmmmmba07',
    0xb077: 'nnnnbbhZ',
    0xb078: 'ddddddaa17',
    0xb079: 'ggggabiZ',
    0xb07a: 'dddddbb17',
    0xb07b: 'ddddbahZ',
    0xb07c: 'aaaaaab17',
    0xb07d: 'aaaaaahZ',
    0xb07e: 'aaaaaaba17',
    0xb07f: 'bbbbbbiZ',
    0xb080: 'cccccaa07',
    0xb081: 'ccccabhZ',
    0xb082: 'ccccccba17',
    0xb083: 'ppppaajZ',
    0xb084: 'eeeeeab17',
    0xb085: 'eeeeaaiZ',
    0xb086: 'eeeeeebb07',
    0xb087: 'ffffbbhZ',
    0xb088: 'llllllab17',
    0xb089: 'ooooabiZ',
    0xb08a: 'lllllbb07',
    0xb08b: 'llllbahZ',
    0xb08c: 'iiiiiab07',
    0xb08d: 'iiiiaahZ',
    0xb08e: 'iiiiiibb17',
    0xb08f: 'jjjjbbiZ',
    0xb090: 'xxxxxxab06',
    0xb091: 'kkkkbbiZ',
    0xb092: 'xxxxxbb16',
    0xb093: 'hhhhaahZ',
    0xb094: '8888888abai',
    0xb095: 'mmmmbahZ',
    0xb096: '8888888abaI',
    0xb097: 'nnnnabiZ',
    0xb098: 'ttttttab16',
    0xb099: 'ggggbbhZ',
    0xb09a: 'tttttbb06',
    0xb09b: 'ddddaaiZ',
    0xb09c: 'qqqqqab06',
    0xb09d: 'aaaabaiZ',
    0xb09e: 'qqqqqqbb16',
    0xb09f: 'bbbbabhZ',
    0xb0a0: 'sssssba16',
    0xb0a1: 'ccccbaiZ',
    0xb0a2: 'pppppab06',
    0xb0a3: 'ppppbbkZ',
    0xb0a4: 'uuuuubb06',
    0xb0a5: 'eeeebbhZ',
    0xb0a6: 'vvvvvaa16',
    0xb0a7: 'ffffaaiZ',
    0xb0a8: '?????bb01',
    0xb0a9: 'oooobahZ',
    0xb0aa: '<<<<<aa11',
    0xb0ab: 'llllabiZ',
    0xb0ac: 'yyyyybb16',
    0xb0ad: 'iiiibbiZ',
    0xb0ae: 'zzzzzaa06',
    0xb0af: 'jjjjaahZ',
    0xb0b0: 'kkkkkba17',
    0xb0b1: 'kkkkaahZ',
    0xb0b2: 'hhhhhab07',
    0xb0b3: 'hhhhbbiZ',
    0xb0b4: 'mmmmmbb07',
    0xb0b5: 'mmmmabiZ',
    0xb0b6: 'nnnnnaa17',
    0xb0b7: 'nnnnbahZ',
    0xb0b8: 'gggggba07',
    0xb0b9: 'ggggaaiZ',
    0xb0ba: 'dddddab17',
    0xb0bb: 'ddddbbhZ',
    0xb0bc: 'aaaaabb17',
    0xb0bd: 'aaaaabhZ',
    0xb0be: 'bbbbbaa07',
    0xb0bf: 'bbbbbaiZ',
    0xb0c0: 'cccccbb17',
    0xb0c1: 'ccccaaiZ',
    0xb0c2: 'CCCCCCaa04',
    0xb0c3: 'ppppabkZ',
    0xb0c4: 'eeeeeba07',
    0xb0c5: 'eeeeabhZ',
    0xb0c6: 'fffffab17',
    0xb0c7: 'ffffbaiZ',
    0xb0c8: 'ooooobb07',
    0xb0c9: 'ooooaahZ',
    0xb0ca: 'lllllaa17',
    0xb0cb: 'llllbbiZ',
    0xb0cc: 'iiiiiba17',
    0xb0cd: 'iiiiabiZ',
    0xb0ce: 'jjjjjab07',
    0xb0cf: 'jjjjbahZ',
    0xb0d0: 'XXXXXXbb15',
    0xb0d1: 'kkkkbahZ',
    0xb0d2: 'xxxxxaa06',
    0xb0d3: 'hhhhabiZ',
    0xb0d4: 'XXXXXXXabam',
    0xb0d5: 'mmmmbbiZ',
    0xb0d6: 'XXXXXXXabaM',
    0xb0d7: 'nnnnaahZ',
    0xb0d8: 'wwwwwbb06',
    0xb0d9: 'ggggbaiZ',
    0xb0da: 'tttttaa16',
    0xb0db: 'ddddabhZ',
    0xb0dc: 'qqqqqba16',
    0xb0dd: 'aaaabbhZ',
    0xb0de: 'rrrrrab06',
    0xb0df: 'bbbbaaiZ',
    0xb0e0: 'ppppppbb16',
    0xb0e1: 'ccccbbhZ',
    0xb0e2: 'pppppba16',
    0xb0e3: 'ppppbajZ',
    0xb0e4: 'pppppppaban',
    0xb0e5: 'eeeebaiZ',
    0xb0e6: 'pppppppabaN',
    0xb0e7: 'ffffabhZ',
    0xb0e8: 'yyyyyyybbbO',
    0xb0e9: 'oooobbiZ',
    0xb0ea: 'yyyyyyybbbo',
    0xb0eb: 'llllaahZ',
    0xb0ec: 'yyyyyaa06',
    0xb0ed: 'iiiibahZ',
    0xb0ee: 'yyyyyyab06',
    0xb0ef: 'jjjjabiZ',
    0xb0f0: 'hhhhhhbb17',
    0xb0f1: 'kkkkabiZ',
    0xb0f2: 'hhhhhba17',
    0xb0f3: 'hhhhbahZ',
    0xb0f4: 'hhhhhhhabao',
    0xb0f5: 'hhhhhhhabkz',
    0xb0f6: 'hhhhhhhabaO',
    0xb0f7: 'hhhhhhhabkZ',
    0xb0f8: 'aaaaaaabbbN',
    0xb0f9: 'aaaaaaabbjZ',
    0xb0fa: 'aaaaaaabbbn',
    0xb0fb: 'aaaaaaabbjz',
    0xb0fc: 'aaaaaaa07',
    0xb0fd: 'aaaaaaiZ',
    0xb0fe: 'aaaaaaab07',
    0xb0ff: 'aaaaaaabl9',
    0xb100: 'cccccab07',
    0xb101: 'ccccabkZ',
    0xb102: 'ccccccaa17',
    0xb103: 'cccccccbl9',
    0xb104: 'cccccccbbbN',
    0xb105: 'cccccccbbjZ',
    0xb106: 'cccccccbbbn',
    0xb107: 'cccccccbbjz',
    0xb108: 'iiiiiiibabN',
    0xb109: 'iiiiiiibajZ',
    0xb10a: 'iiiiiiibabn',
    0xb10b: 'iiiiiiibajz',
    0xb10c: 'iiiiiaa07',
    0xb10d: 'iiiiaakZ',
    0xb10e: 'iiiiiiab17',
    0xb10f: 'jjjjbbjZ',
    0xb110: 'xxxxxxbb06',
    0xb111: 'kkkkbbjZ',
    0xb112: 'xxxxxba16',
    0xb113: 'hhhhaakZ',
    0xb114: 'xxxxxxxaaan',
    0xb115: 'mmmmbakZ',
    0xb116: 'xxxxxxxaaaN',
    0xb117: 'nnnnabjZ',
    0xb118: 'qqqqqqqbabO',
    0xb119: 'ggggbbkZ',
    0xb11a: 'qqqqqqqbabo',
    0xb11b: 'ddddaajZ',
    0xb11c: 'qqqqqaa06',
    0xb11d: 'aaaabajZ',
    0xb11e: 'qqqqqqab16',
    0xb11f: 'bbbbabkZ',
    0xb120: 'sssssbb16',
    0xb121: 'ccccbajZ',
    0xb122: 'pppppaa06',
    0xb123: 'ppppbbhZ',
    0xb124: 'uuuuuba06',
    0xb125: 'eeeebbkZ',
    0xb126: 'vvvvvab16',
    0xb127: 'ffffaajZ',
    0xb128: 'YYYYYYYbabL',
    0xb129: 'oooobakZ',
    0xb12a: 'YYYYYYYbabl',
    0xb12b: 'llllabjZ',
    0xb12c: 'yyyyyba16',
    0xb12d: 'iiiibbjZ',
    0xb12e: 'zzzzzab06',
    0xb12f: 'jjjjaakZ',
    0xb130: 'kkkkkbb17',
    0xb131: 'kkkkaakZ',
    0xb132: 'hhhhhaa07',
    0xb133: 'hhhhbbjZ',
    0xb134: 'mmmmmba07',
    0xb135: 'mmmmabjZ',
    0xb136: 'nnnnnab17',
    0xb137: 'nnnnbakZ',
    0xb138: 'gggggbb07',
    0xb139: 'ggggaajZ',
    0xb13a: 'dddddaa17',
    0xb13b: 'ddddbbkZ',
    0xb13c: 'aaaaaba17',
    0xb13d: 'aaaaabkZ',
    0xb13e: 'bbbbbab07',
    0xb13f: 'bbbbbajZ',
    0xb140: 'cccccba17',
    0xb141: 'ccccaajZ',
    0xb142: 'CCCCCCba04',
    0xb143: 'ppppabhZ',
    0xb144: 'eeeeebb07',
    0xb145: 'eeeeabkZ',
    0xb146: 'fffffaa17',
    0xb147: 'ffffbajZ',
    0xb148: 'oooooba07',
    0xb149: 'ooooaakZ',
    0xb14a: 'lllllab17',
    0xb14b: 'llllbbjZ',
    0xb14c: 'iiiiibb17',
    0xb14d: 'iiiiabjZ',
    0xb14e: 'jjjjjaa07',
    0xb14f: 'jjjjbakZ',
    0xb150: 'XXXXXXab15',
    0xb151: 'kkkkbakZ',
    0xb152: 'xxxxxab06',
    0xb153: 'hhhhabjZ',
    0xb154: '=====ba01',
    0xb155: 'mmmmbbjZ',
    0xb156: '>>>>>ab11',
    0xb157: 'nnnnaakZ',
    0xb158: 'wwwwwba06',
    0xb159: 'ggggbajZ',
    0xb15a: 'tttttab16',
    0xb15b: 'ddddabkZ',
    0xb15c: 'qqqqqbb16',
    0xb15d: 'aaaabbkZ',
    0xb15e: 'rrrrraa06',
    0xb15f: 'bbbbaajZ',
    0xb160: 'ppppppab16',
    0xb161: 'ccccbbkZ',
    0xb162: 'pppppbb16',
    0xb163: 'ppppbaiZ',
    0xb164: 'uuuuuab16',
    0xb165: 'eeeebajZ',
    0xb166: 'uuuuuubb16',
    0xb167: 'ffffabkZ',
    0xb168: '9999999babH',
    0xb169: 'oooobbjZ',
    0xb16a: '9999999babh',
    0xb16b: 'llllaakZ',
    0xb16c: 'yyyyyab06',
    0xb16d: 'iiiibakZ',
    0xb16e: 'yyyyyybb06',
    0xb16f: 'jjjjabjZ',
    0xb170: 'hhhhhhab17',
    0xb171: 'kkkkabjZ',
    0xb172: 'hhhhhbb17',
    0xb173: 'hhhhbakZ',
    0xb174: 'mmmmmab17',
    0xb175: 'mmmmaakZ',
    0xb176: 'mmmmmmbb17',
    0xb177: 'nnnnbbjZ',
    0xb178: 'ddddddab07',
    0xb179: 'ggggabkZ',
    0xb17a: 'dddddbb07',
    0xb17b: 'ddddbajZ',
    0xb17c: 'aaaaaab07',
    0xb17d: 'aaaaaajZ',
    0xb17e: 'aaaaaabb07',
    0xb17f: 'bbbbbbkZ',
    0xb180: 'aaaaaaaaaaaaaaZ',
    0xb181: 'aaaaaaaaaaaaacZ',
    0xb182: 'aaaaaaaaaaaaaaX',
    0xb183: 'aaaaaaaaaaaaaaY',
    0xb184: 'aaaaaaaaaaaaaiZ',
    0xb185: 'aaaaaaaaaaaaakZ',
    0xb186: 'aaaaaaaaaaaaaiX',
    0xb187: 'aaaaaaaaaaaaaiY',
    0xb188: 'aaaaaaaaaaaaaaR',
    0xb189: 'aaaaaaaaaaaaaaS',
    0xb18a: 'aaaaaaaaaaaaaaP',
    0xb18b: 'aaaaaaaaaaaaaaQ',
    0xb18c: 'aaaaaaaaaaaaaaV',
    0xb18d: 'aaaaaaaaaaaaaaW',
    0xb18e: 'aaaaaaaaaaaaaaT',
    0xb18f: 'aaaaaaaaaaaaaaU',
    0xb190: 'aaaaaaaaaaaaaaJ',
    0xb191: 'aaaaaaaaaaaaaaK',
    0xb192: 'aaaaaaaaaaaaaaH',
    0xb193: 'aaaaaaaaaaaaaaI',
    0xb194: 'aaaaaaaaaaaaaaN',
    0xb195: 'aaaaaaaaaaaaaaO',
    0xb196: 'aaaaaaaaaaaaaaL',
    0xb197: 'aaaaaaaaaaaaaaM',
    0xb198: 'aaaaaaaaaaaaaaB',
    0xb199: 'aaaaaaaaaaaaaaC',
    0xb19a: 'aaaaaaaaaaaaacA',
    0xb19b: 'aaaaaaaaaaaaaaA',
    0xb19c: 'aaaaaaaaaaaaaaF',
    0xb19d: 'aaaabakZ',
    0xb19e: 'aaaaaaaaaaaaaaD',
    0xb19f: 'aaaaaaaaaaaaaaE',
    0xb1a0: 'aaaaaaaaaaaaaaz',
    0xb1a1: 'aaaaaaaaaaaaacz',
    0xb1a2: 'aaaaaaaaaaaaaax',
    0xb1a3: 'aaaaaaaaaaaaaay',
    0xb1a4: 'aaaaaaaaaaaaaiz',
    0xb1a5: 'aaaaaaaaaaaaakz',
    0xb1a6: 'aaaaaaaaaaaaaix',
    0xb1a7: 'aaaaaaaaaaaaaiy',
    0xb1a8: 'aaaaaaaaaaaaaar',
    0xb1a9: 'aaaaaaaaaaaaaas',
    0xb1aa: 'aaaaaaaaaaaaaap',
    0xb1ab: 'aaaaaaaaaaaaaaq',
    0xb1ac: 'aaaaaaaaaaaaaav',
    0xb1ad: 'aaaaaaaaaaaaaaw',
    0xb1ae: 'aaaaaaaaaaaaaat',
    0xb1af: 'aaaaaaaaaaaaaau',
    0xb1b0: 'aaaaaaaaaaaaaaj',
    0xb1b1: 'aaaaaaaaaaaaaak',
    0xb1b2: 'aaaaaaaaaaaaaah',
    0xb1b3: 'aaaaaaaaaaaaaai',
    0xb1b4: 'aaaaaaaaaaaaaan',
    0xb1b5: 'aaaaaaaaaaaaaao',
    0xb1b6: 'aaaaaaaaaaaaaal',
    0xb1b7: 'aaaaaaaaaaaaaam',
    0xb1b8: 'aaaaaaaaaaaaaab',
    0xb1b9: 'aaaaaaaaaaaaaac',
    0xb1ba: 'aaaaaaaaaaaaaca',
    0xb1bb: 'aaaaaaaaaaaaaaa',
    0xb1bc: 'aaaaabb07',
    0xb1bd: 'aaaaabjZ',
    0xb1be: 'aaaaaaaaaaaaaad',
    0xb1bf: 'aaaaaaaaaaaaaae',
    0xb1c0: 'aaaaaaaaaaaaaW',
    0xb1c1: 'aaaaaaaaaaaaaU',
    0xb1c2: 'aaaaaaaaaaaaaS',
    0xb1c3: 'aaaaaaaaaaaaaQ',
    0xb1c4: 'aaaaaaaaaaaakZ',
    0xb1c5: 'aaaaaaaaaaaaiY',
    0xb1c6: 'aaaaaaaaaaaacZ',
    0xb1c7: 'aaaaaaaaaaaaaY',
    0xb1c8: 'aaaaaaaaaaaaaG',
    0xb1c9: 'aaaaaaaaaaaaaE',
    0xb1ca: 'aaaaaaaaaaaaaC',
    0xb1cb: 'aaaaaaaaaaaaaA',
    0xb1cc: 'aaaaaaaaaaaaaO',
    0xb1cd: 'aaaaaaaaaaaaaM',
    0xb1ce: 'aaaaaaaaaaaaaK',
    0xb1cf: 'aaaaaaaaaaaaaI',
    0xb1d0: 'aaaaaaaaaaaaaw',
    0xb1d1: 'aaaaaaaaaaaaau',
    0xb1d2: 'aaaaaaaaaaaaas',
    0xb1d3: 'aaaaaaaaaaaaaq',
    0xb1d4: 'aaaaaaaaaaaakz',
    0xb1d5: 'aaaaaaaaaaaaiy',
    0xb1d6: 'aaaaaaaaaaaacz',
    0xb1d7: 'aaaaaaaaaaaaay',
    0xb1d8: 'aaaaaaaaaaaaag',
    0xb1d9: 'aaaaaaaaaaaaae',
    0xb1da: 'aaaaaaaaaaaaac',
    0xb1db: 'aaaaaaaaaaaaaa',
    0xb1dc: 'aaaaaaaaaaaaao',
    0xb1dd: 'aaaabbjZ',
    0xb1de: 'aaaaaaaaaaaaak',
    0xb1df: 'aaaaaaaaaaaaai',
    0xb1e0: 'aaaaaaaaaaaaC',
    0xb1e1: 'aaaaaaaaaaaaG',
    0xb1e2: 'aaaaaaaaaaaaK',
    0xb1e3: 'aaaaaaaaaaaaO',
    0xb1e4: 'aaaaaaaaaaaaS',
    0xb1e5: 'aaaaaaaaaaaaW',
    0xb1e6: 'aaaaaaaaaaacZ',
    0xb1e7: 'aaaaaaaaaaakZ',
    0xb1e8: 'aaaaaaaaaaaac',
    0xb1e9: 'aaaaaaaaaaaag',
    0xb1ea: 'aaaaaaaaaaaak',
    0xb1eb: 'aaaaaaaaaaaao',
    0xb1ec: 'aaaaaaaaaaaas',
    0xb1ed: 'aaaaaaaaaaaaw',
    0xb1ee: 'aaaaaaaaaaacz',
    0xb1ef: 'aaaaaaaaaaakz',
    0xb1f0: 'aaaaaaaaaaao',
    0xb1f1: 'aaaaaaaaaaag',
    0xb1f2: 'aaaaaaaaaakz',
    0xb1f3: 'aaaaaaaaaaaw',
    0xb1f4: 'aaaaaaaaaaaO',
    0xb1f5: 'aaaaaaaaaaaG',
    0xb1f6: 'aaaaaaaaaakZ',
    0xb1f7: 'aaaaaaaaaaaW',
    0xb1f8: 'aaaaaaaaaaO',
    0xb1f9: 'aaaaaaaaakZ',
    0xb1fa: 'aaaaaaaaaao',
    0xb1fb: 'aaaaaaaaakz',
    0xb1fc: 'aaaaaaa17',
    0xb1fd: 'aaaaaakZ',
    0xb1fe: 'aaaaaaaa17',
    0xb1ff: 'aaaaaaaam9',
    0xb200: 'cccccab16',
    0xb201: 'ccccabiX',
    0xb202: 'ccccccab17',
    0xb203: 'cccccccal9',
    0xb204: 'cccccccabbN',
    0xb205: 'cccccccabjZ',
    0xb206: 'cccccccabbn',
    0xb207: 'cccccccabjz',
    0xb208: 'ccccccccaaaO',
    0xb209: 'ccccccccaaaG',
    0xb20a: 'ccccccccaakZ',
    0xb20b: 'ccccccccaaaW',
    0xb20c: 'ccccccccaaao',
    0xb20d: 'ccccccccaaag',
    0xb20e: 'ccccccccaakz',
    0xb20f: 'ccccccccaaaw',
    0xb210: 'cccccccccbbbr',
    0xb211: 'cccccccccbbbv',
    0xb212: 'cccccccccbbbz',
    0xb213: 'cccccccccbbjz',
    0xb214: 'cccccccccbbbb',
    0xb215: 'cccccccccbbbf',
    0xb216: 'cccccccccbbbj',
    0xb217: 'cccccccccbbbn',
    0xb218: 'cccccccccbbbR',
    0xb219: 'cccccccccbbbV',
    0xb21a: 'cccccccccbbbZ',
    0xb21b: 'cccccccccbbjZ',
    0xb21c: 'cccccccccbbbB',
    0xb21d: 'aaaabahX',
    0xb21e: 'cccccccccbbbJ',
    0xb21f: 'bbbbabiX',
    0xb220: 'sssssbb07',
    0xb221: 'ccccbahX',
    0xb222: 'pppppaa17',
    0xb223: 'ppppbbhY',
    0xb224: 'uuuuuba17',
    0xb225: 'eeeebbiX',
    0xb226: 'vvvvvab07',
    0xb227: 'ffffaahX',
    0xb228: 'AAAAAAAAAbaba',
    0xb229: 'oooobaiX',
    0xb22a: 'AAAAAAAAAbabi',
    0xb22b: 'llllabhX',
    0xb22c: 'yyyyyba07',
    0xb22d: 'iiiibbhX',
    0xb22e: 'zzzzzab17',
    0xb22f: 'jjjjaaiX',
    0xb230: 'kkkkkbb06',
    0xb231: 'kkkkaaiX',
    0xb232: 'hhhhhaa16',
    0xb233: 'hhhhbbhX',
    0xb234: 'mmmmmba16',
    0xb235: 'mmmmabhX',
    0xb236: 'nnnnnab06',
    0xb237: 'nnnnbaiX',
    0xb238: 'gggggbb16',
    0xb239: 'ggggaahX',
    0xb23a: 'dddddaa06',
    0xb23b: 'ddddbbiX',
    0xb23c: 'aaaaaba06',
    0xb23d: 'aaaaabiX',
    0xb23e: 'bbbbbab16',
    0xb23f: 'bbbbbahX',
    0xb240: 'cccccba06',
    0xb241: 'ccccaahX',
    0xb242: 'CCCCCCbb04',
    0xb243: 'ppppabhY',
    0xb244: 'eeeeebb16',
    0xb245: 'eeeeabiX',
    0xb246: 'fffffaa06',
    0xb247: 'ffffbahX',
    0xb248: 'oooooba16',
    0xb249: 'ooooaaiX',
    0xb24a: 'lllllab06',
    0xb24b: 'llllbbhX',
    0xb24c: 'iiiiibb06',
    0xb24d: 'iiiiabhX',
    0xb24e: 'jjjjjaa16',
    0xb24f: 'jjjjbaiX',
    0xb250: 'XXXXXXaa15',
    0xb251: 'kkkkbaiX',
    0xb252: 'xxxxxab17',
    0xb253: 'hhhhabhX',
    0xb254: '=====ba10',
    0xb255: 'mmmmbbhX',
    0xb256: '>>>>>ab00',
    0xb257: 'nnnnaaiX',
    0xb258: 'wwwwwba17',
    0xb259: 'ggggbahX',
    0xb25a: 'tttttab07',
    0xb25b: 'ddddabiX',
    0xb25c: 'qqqqqbb07',
    0xb25d: 'aaaabbiX',
    0xb25e: 'rrrrraa17',
    0xb25f: 'bbbbaahX',
    0xb260: 'ppppppaa16',
    0xb261: 'ccccbbiX',
    0xb262: 'pppppbb07',
    0xb263: 'ppppbaiY',
    0xb264: 'uuuuuab07',
    0xb265: 'eeeebahX',
    0xb266: 'uuuuuuba16',
    0xb267: 'ffffabiX',
    0xb268: '00000000bbbX',
    0xb269: 'oooobbhX',
    0xb26a: '00000000bbbH',
    0xb26b: 'llllaaiX',
    0xb26c: 'yyyyyab17',
    0xb26d: 'iiiibaiX',
    0xb26e: 'yyyyyyba06',
    0xb26f: 'jjjjabhX',
    0xb270: 'hhhhhhaa17',
    0xb271: 'kkkkabhX',
    0xb272: 'hhhhhbb06',
    0xb273: 'hhhhbaiX',
    0xb274: 'mmmmmab06',
    0xb275: 'mmmmaaiX',
    0xb276: 'mmmmmmba17',
    0xb277: 'nnnnbbhX',
    0xb278: 'ddddddaa07',
    0xb279: 'ggggabiX',
    0xb27a: 'dddddbb16',
    0xb27b: 'ddddbahX',
    0xb27c: 'aaaaaab16',
    0xb27d: 'aaaaaahX',
    0xb27e: 'aaaaaaba07',
    0xb27f: 'bbbbbbiX',
    0xb280: 'cccccaa06',
    0xb281: 'ccccabhX',
    0xb282: 'ccccccba07',
    0xb283: 'ppppaahY',
    0xb284: 'eeeeeab16',
    0xb285: 'eeeeaaiX',
    0xb286: 'eeeeeebb17',
    0xb287: 'ffffbbhX',
    0xb288: 'llllllab07',
    0xb289: 'ooooabiX',
    0xb28a: 'lllllbb06',
    0xb28b: 'llllbahX',
    0xb28c: 'iiiiiab06',
    0xb28d: 'iiiiaahX',
    0xb28e: 'iiiiiibb07',
    0xb28f: 'jjjjbbiX',
    0xb290: 'xxxxxxab16',
    0xb291: 'kkkkbbiX',
    0xb292: 'xxxxxbb17',
    0xb293: 'hhhhaahX',
    0xb294: '8888888aabh',
    0xb295: 'mmmmbahX',
    0xb296: '8888888aabH',
    0xb297: 'nnnnabiX',
    0xb298: 'ttttttab06',
    0xb299: 'ggggbbhX',
    0xb29a: 'tttttbb07',
    0xb29b: 'ddddaaiX',
    0xb29c: 'qqqqqab07',
    0xb29d: 'aaaabaiX',
    0xb29e: 'qqqqqqbb06',
    0xb29f: 'bbbbabhX',
    0xb2a0: 'sssssba17',
    0xb2a1: 'ccccbaiX',
    0xb2a2: 'pppppab07',
    0xb2a3: 'ppppbbiY',
    0xb2a4: 'uuuuubb07',
    0xb2a5: 'eeeebbhX',
    0xb2a6: 'vvvvvaa17',
    0xb2a7: 'ffffaaiX',
    0xb2a8: '?????bb00',
    0xb2a9: 'oooobahX',
    0xb2aa: '<<<<<aa10',
    0xb2ab: 'llllabiX',
    0xb2ac: 'yyyyybb17',
    0xb2ad: 'iiiibbiX',
    0xb2ae: 'zzzzzaa07',
    0xb2af: 'jjjjaahX',
    0xb2b0: 'kkkkkba16',
    0xb2b1: 'kkkkaahX',
    0xb2b2: 'hhhhhab06',
    0xb2b3: 'hhhhbbiX',
    0xb2b4: 'mmmmmbb06',
    0xb2b5: 'mmmmabiX',
    0xb2b6: 'nnnnnaa16',
    0xb2b7: 'nnnnbahX',
    0xb2b8: 'gggggba06',
    0xb2b9: 'ggggaaiX',
    0xb2ba: 'dddddab16',
    0xb2bb: 'ddddbbhX',
    0xb2bc: 'aaaaabb16',
    0xb2bd: 'aaaaabhX',
    0xb2be: 'bbbbbaa06',
    0xb2bf: 'bbbbbaiX',
    0xb2c0: 'cccccbb16',
    0xb2c1: 'ccccaaiX',
    0xb2c2: 'CCCCCCaa14',
    0xb2c3: 'ppppabiY',
    0xb2c4: 'eeeeeba06',
    0xb2c5: 'eeeeabhX',
    0xb2c6: 'fffffab16',
    0xb2c7: 'ffffbaiX',
    0xb2c8: 'ooooobb06',
    0xb2c9: 'ooooaahX',
    0xb2ca: 'lllllaa16',
    0xb2cb: 'llllbbiX',
    0xb2cc: 'iiiiiba16',
    0xb2cd: 'iiiiabiX',
    0xb2ce: 'jjjjjab06',
    0xb2cf: 'jjjjbahX',
    0xb2d0: 'XXXXXXbb05',
    0xb2d1: 'kkkkbahX',
    0xb2d2: 'xxxxxaa07',
    0xb2d3: 'hhhhabiX',
    0xb2d4: 'XXXXXXXaabl',
    0xb2d5: 'mmmmbbiX',
    0xb2d6: 'XXXXXXXaabL',
    0xb2d7: 'nnnnaahX',
    0xb2d8: 'wwwwwbb07',
    0xb2d9: 'ggggbaiX',
    0xb2da: 'tttttaa17',
    0xb2db: 'ddddabhX',
    0xb2dc: 'qqqqqba17',
    0xb2dd: 'aaaabbhX',
    0xb2de: 'rrrrrab07',
    0xb2df: 'bbbbaaiX',
    0xb2e0: 'ppppppbb06',
    0xb2e1: 'ccccbbhX',
    0xb2e2: 'pppppba17',
    0xb2e3: 'ppppbahY',
    0xb2e4: 'pppppppaabo',
    0xb2e5: 'eeeebaiX',
    0xb2e6: 'pppppppaabO',
    0xb2e7: 'ffffabhX',
    0xb2e8: 'yyyyyyybaaN',
    0xb2e9: 'oooobbiX',
    0xb2ea: 'yyyyyyybaan',
    0xb2eb: 'llllaahX',
    0xb2ec: 'yyyyyaa07',
    0xb2ed: 'iiiibahX',
    0xb2ee: 'yyyyyyab16',
    0xb2ef: 'jjjjabiX',
    0xb2f0: 'hhhhhhbb07',
    0xb2f1: 'kkkkabiX',
    0xb2f2: 'hhhhhba16',
    0xb2f3: 'hhhhbahX',
    0xb2f4: 'hhhhhhhaabn',
    0xb2f5: 'hhhhhhhaajz',
    0xb2f6: 'hhhhhhhaabN',
    0xb2f7: 'hhhhhhhaajZ',
    0xb2f8: 'aaaaaaabaaO',
    0xb2f9: 'aaaaaaabakZ',
    0xb2fa: 'aaaaaaabaao',
    0xb2fb: 'aaaaaaabakz',
    0xb2fc: 'aaaaaaa06',
    0xb2fd: 'aaaaaaiX',
    0xb2fe: 'aaaaaaab17',
    0xb2ff: 'aaaaaaabm9',
    0xb300: 'cccccab06',
    0xb301: 'ccccabiY',
    0xb302: 'ccccccaa07',
    0xb303: 'cccccccbm9',
    0xb304: 'cccccccbaaO',
    0xb305: 'cccccccbakZ',
    0xb306: 'cccccccbaao',
    0xb307: 'cccccccbakz',
    0xb308: 'iiiiiiibbaO',
    0xb309: 'iiiiiiibbkZ',
    0xb30a: 'iiiiiiibbao',
    0xb30b: 'iiiiiiibbkz',
    0xb30c: 'iiiiiaa06',
    0xb30d: 'iiiiaaiY',
    0xb30e: 'iiiiiiab07',
    0xb30f: 'iiiiiiibl8',
    0xb310: 'xxxxxxbb16',
    0xb311: 'kkkkbbhY',
    0xb312: 'xxxxxba17',
    0xb313: 'hhhhaaiY',
    0xb314: 'xxxxxxxabbo',
    0xb315: 'mmmmbaiY',
    0xb316: 'xxxxxxxabbO',
    0xb317: 'nnnnabhY',
    0xb318: 'qqqqqqqbbaN',
    0xb319: 'ggggbbiY',
    0xb31a: 'qqqqqqqbban',
    0xb31b: 'ddddaahY',
    0xb31c: 'qqqqqaa07',
    0xb31d: 'aaaabahY',
    0xb31e: 'qqqqqqab06',
    0xb31f: 'bbbbabiY',
    0xb320: 'sssssbb17',
    0xb321: 'ccccbahY',
    0xb322: 'pppppaa07',
    0xb323: 'ppppbbhX',
    0xb324: 'uuuuuba07',
    0xb325: 'eeeebbiY',
    0xb326: 'vvvvvab17',
    0xb327: 'ffffaahY',
    0xb328: 'YYYYYYYbbaM',
    0xb329: 'oooobaiY',
    0xb32a: 'YYYYYYYbbam',
    0xb32b: 'llllabhY',
    0xb32c: 'yyyyyba17',
    0xb32d: 'iiiibbhY',
    0xb32e: 'zzzzzab07',
    0xb32f: 'jjjjaaiY',
    0xb330: 'kkkkkbb16',
    0xb331: 'kkkkaaiY',
    0xb332: 'hhhhhaa06',
    0xb333: 'hhhhbbhY',
    0xb334: 'mmmmmba06',
    0xb335: 'mmmmabhY',
    0xb336: 'nnnnnab16',
    0xb337: 'nnnnbaiY',
    0xb338: 'gggggbb06',
    0xb339: 'ggggaahY',
    0xb33a: 'dddddaa16',
    0xb33b: 'ddddbbiY',
    0xb33c: 'aaaaaba16',
    0xb33d: 'aaaaabiY',
    0xb33e: 'bbbbbab06',
    0xb33f: 'bbbbbahY',
    0xb340: 'cccccba16',
    0xb341: 'ccccaahY',
    0xb342: 'CCCCCCba14',
    0xb343: 'ppppabhX',
    0xb344: 'eeeeebb06',
    0xb345: 'eeeeabiY',
    0xb346: 'fffffaa16',
    0xb347: 'ffffbahY',
    0xb348: 'oooooba06',
    0xb349: 'ooooaaiY',
    0xb34a: 'lllllab16',
    0xb34b: 'llllbbhY',
    0xb34c: 'iiiiibb16',
    0xb34d: 'iiiiabhY',
    0xb34e: 'jjjjjaa06',
    0xb34f: 'jjjjbaiY',
    0xb350: 'XXXXXXab05',
    0xb351: 'kkkkbaiY',
    0xb352: 'xxxxxab07',
    0xb353: 'hhhhabhY',
    0xb354: '=====ba00',
    0xb355: 'mmmmbbhY',
    0xb356: '>>>>>ab10',
    0xb357: 'nnnnaaiY',
    0xb358: 'wwwwwba07',
    0xb359: 'ggggbahY',
    0xb35a: 'tttttab17',
    0xb35b: 'ddddabiY',
    0xb35c: 'qqqqqbb17',
    0xb35d: 'aaaabbiY',
    0xb35e: 'rrrrraa07',
    0xb35f: 'bbbbaahY',
    0xb360: 'ppppppab06',
    0xb361: 'ccccbbiY',
    0xb362: 'pppppbb17',
    0xb363: 'ppppbaiX',
    0xb364: 'uuuuuab17',
    0xb365: 'eeeebahY',
    0xb366: 'uuuuuubb06',
    0xb367: 'ffffabiY',
    0xb368: '9999999bbaI',
    0xb369: 'oooobbhY',
    0xb36a: '9999999bbai',
    0xb36b: 'llllaaiY',
    0xb36c: 'yyyyyab07',
    0xb36d: 'iiiibaiY',
    0xb36e: 'yyyyyybb16',
    0xb36f: 'jjjjabhY',
    0xb370: 'hhhhhhab07',
    0xb371: 'kkkkabhY',
    0xb372: 'hhhhhbb16',
    0xb373: 'hhhhbaiY',
    0xb374: 'mmmmmab16',
    0xb375: 'mmmmaaiY',
    0xb376: 'mmmmmmbb07',
    0xb377: 'nnnnbbhY',
    0xb378: 'ddddddab17',
    0xb379: 'ggggabiY',
    0xb37a: 'dddddbb06',
    0xb37b: 'ddddbahY',
    0xb37c: 'aaaaaab06',
    0xb37d: 'aaaaaahY',
    0xb37e: 'aaaaaabb17',
    0xb37f: 'bbbbbbiY',
    0xb380: 'cccccaa16',
    0xb381: 'ccccabhY',
    0xb382: 'ccccccbb17',
    0xb383: 'ppppaahX',
    0xb384: 'eeeeeab06',
    0xb385: 'eeeeaaiY',
    0xb386: 'eeeeeeba07',
    0xb387: 'ffffbbhY',
    0xb388: 'llllllaa17',
    0xb389: 'ooooabiY',
    0xb38a: 'lllllbb16',
    0xb38b: 'llllbahY',
    0xb38c: 'iiiiiab16',
    0xb38d: 'iiiiaahY',
    0xb38e: 'iiiiiiba17',
    0xb38f: 'jjjjbbiY',
    0xb390: 'xxxxxxaa06',
    0xb391: 'kkkkbbiY',
    0xb392: 'xxxxxbb07',
    0xb393: 'hhhhaahY',
    0xb394: '11111111bbbH',
    0xb395: 'mmmmbahY',
    0xb396: '11111111bbbX',
    0xb397: 'nnnnabiY',
    0xb398: 'ttttttaa16',
    0xb399: 'ggggbbhY',
    0xb39a: 'tttttbb17',
    0xb39b: 'ddddaaiY',
    0xb39c: 'qqqqqab17',
    0xb39d: 'aaaabaiY',
    0xb39e: 'qqqqqqba16',
    0xb39f: 'bbbbabhY',
    0xb3a0: 'sssssba07',
    0xb3a1: 'ccccbaiY',
    0xb3a2: 'pppppab17',
    0xb3a3: 'ppppbbiX',
    0xb3a4: 'uuuuubb17',
    0xb3a5: 'eeeebbhY',
    0xb3a6: 'vvvvvaa07',
    0xb3a7: 'ffffaaiY',
    0xb3a8: '?????bb10',
    0xb3a9: 'oooobahY',
    0xb3aa: '<<<<<aa00',
    0xb3ab: 'llllabiY',
    0xb3ac: 'yyyyybb07',
    0xb3ad: 'iiiibbiY',
    0xb3ae: 'zzzzzaa17',
    0xb3af: 'jjjjaahY',
    0xb3b0: 'kkkkkba06',
    0xb3b1: 'kkkkaahY',
    0xb3b2: 'hhhhhab16',
    0xb3b3: 'hhhhbbiY',
    0xb3b4: 'mmmmmbb16',
    0xb3b5: 'mmmmabiY',
    0xb3b6: 'nnnnnaa06',
    0xb3b7: 'nnnnbahY',
    0xb3b8: 'gggggba16',
    0xb3b9: 'ggggaaiY',
    0xb3ba: 'dddddab06',
    0xb3bb: 'ddddbbhY',
    0xb3bc: 'aaaaabb06',
    0xb3bd: 'aaaaabhY',
    0xb3be: 'bbbbbaa16',
    0xb3bf: 'bbbbbaiY',
    0xb3c0: 'cccccbb06',
    0xb3c1: 'ccccaaiY',
    0xb3c2: 'CCCCCCab04',
    0xb3c3: 'ppppabiX',
    0xb3c4: 'eeeeeba16',
    0xb3c5: 'eeeeabhY',
    0xb3c6: 'fffffab06',
    0xb3c7: 'ffffbaiY',
    0xb3c8: 'ooooobb16',
    0xb3c9: 'ooooaahY',
    0xb3ca: 'lllllaa06',
    0xb3cb: 'llllbbiY',
    0xb3cc: 'iiiiiba06',
    0xb3cd: 'iiiiabiY',
    0xb3ce: 'jjjjjab16',
    0xb3cf: 'jjjjbahY',
    0xb3d0: 'QQQQQQQQbbbl',
    0xb3d1: 'kkkkbahY',
    0xb3d2: 'xxxxxaa17',
    0xb3d3: 'hhhhabiY',
    0xb3d4: 'QQQQQQQQbbbL',
    0xb3d5: 'mmmmbbiY',
    0xb3d6: 'QQQQQQQQbbhY',
    0xb3d7: 'nnnnaahY',
    0xb3d8: 'wwwwwbb17',
    0xb3d9: 'ggggbaiY',
    0xb3da: 'tttttaa07',
    0xb3db: 'ddddabhY',
    0xb3dc: 'qqqqqba07',
    0xb3dd: 'aaaabbhY',
    0xb3de: 'rrrrrab17',
    0xb3df: 'bbbbaaiY',
    0xb3e0: 'ppppppba16',
    0xb3e1: 'ccccbbhY',
    0xb3e2: 'pppppba07',
    0xb3e3: 'ppppbahX',
    0xb3e4: 'pppppppbban',
    0xb3e5: 'eeeebaiY',
    0xb3e6: 'pppppppbbaN',
    0xb3e7: 'ffffabhY',
    0xb3e8: 'ppppppppabiZ',
    0xb3e9: 'oooobbiY',
    0xb3ea: 'ppppppppabaN',
    0xb3eb: 'llllaahY',
    0xb3ec: 'ppppppppabiz',
    0xb3ed: 'iiiibahY',
    0xb3ee: 'ppppppppaban',
    0xb3ef: 'jjjjabiY',
    0xb3f0: 'aaaaaaaabbbn',
    0xb3f1: 'aaaaaaaabbbf',
    0xb3f2: 'aaaaaaaabbjz',
    0xb3f3: 'aaaaaaaabbbv',
    0xb3f4: 'aaaaaaaabbbN',
    0xb3f5: 'aaaaaaaabbbF',
    0xb3f6: 'aaaaaaaabbjZ',
    0xb3f7: 'aaaaaaaabbbV',
    0xb3f8: 'aaaaaaaabbN',
    0xb3f9: 'aaaaaaaabjZ',
    0xb3fa: 'aaaaaaaabbn',
    0xb3fb: 'aaaaaaaabjz',
    0xb3fc: 'aaaaaaa16',
    0xb3fd: 'aaaaaaiY',
    0xb3fe: 'aaaaaaaa07',
    0xb3ff: 'aaaaaaaal9',
    0xb400: 'cccccab15',
    0xb401: 'ccccabaZ',
    0xb402: 'ccccccab06',
    0xb403: 'ccccccccl8',
    0xb404: 'cccccccaabN',
    0xb405: 'cccccccaajZ',
    0xb406: 'cccccccaabn',
    0xb407: 'cccccccaajz',
    0xb408: 'ccccccccbaaO',
    0xb409: 'ccccccccbaaG',
    0xb40a: 'ccccccccbakZ',
    0xb40b: 'ccccccccbaaW',
    0xb40c: 'ccccccccbaao',
    0xb40d: 'ccccccccbaag',
    0xb40e: 'ccccccccbakz',
    0xb40f: 'ccccccccbaaw',
    0xb410: 'qqqqqqqqbban',
    0xb411: 'kkkkbbdX',
    0xb412: 'qqqqqqqqbbiz',
    0xb413: 'hhhhaaaZ',
    0xb414: 'qqqqqqqqbbaN',
    0xb415: 'mmmmbaaZ',
    0xb416: 'qqqqqqqqbbiZ',
    0xb417: 'nnnnabdX',
    0xb418: 'qqqqqqqabbO',
    0xb419: 'ggggbbaZ',
    0xb41a: 'qqqqqqqabbo',
    0xb41b: 'ddddaadX',
    0xb41c: 'qqqqqaa14',
    0xb41d: 'aaaabadX',
    0xb41e: 'qqqqqqaa07',
    0xb41f: 'bbbbabaZ',
    0xb420: 'sssssbb04',
    0xb421: 'ccccbadX',
    0xb422: 'pppppaa14',
    0xb423: 'ppppbbbZ',
    0xb424: 'uuuuuba14',
    0xb425: 'eeeebbaZ',
    0xb426: 'vvvvvab04',
    0xb427: 'ffffaadX',
    0xb428: 'PPPPPPPPabhY',
    0xb429: 'oooobaaZ',
    0xb42a: 'PPPPPPPPabbL',
    0xb42b: 'llllabdX',
    0xb42c: 'yyyyyba04',
    0xb42d: 'iiiibbdX',
    0xb42e: 'zzzzzab14',
    0xb42f: 'jjjjaaaZ',
    0xb430: 'kkkkkbb05',
    0xb431: 'kkkkaaaZ',
    0xb432: 'hhhhhaa15',
    0xb433: 'hhhhbbdX',
    0xb434: 'mmmmmba15',
    0xb435: 'mmmmabdX',
    0xb436: 'nnnnnab05',
    0xb437: 'nnnnbaaZ',
    0xb438: 'gggggbb15',
    0xb439: 'ggggaadX',
    0xb43a: 'dddddaa05',
    0xb43b: 'ddddbbaZ',
    0xb43c: 'aaaaaba05',
    0xb43d: 'aaaaabaZ',
    0xb43e: 'bbbbbab15',
    0xb43f: 'bbbbbadX',
    0xb440: 'cccccba05',
    0xb441: 'ccccaadX',
    0xb442: 'CCCCCCbb15',
    0xb443: 'ppppabbZ',
    0xb444: 'eeeeebb15',
    0xb445: 'eeeeabaZ',
    0xb446: 'fffffaa05',
    0xb447: 'ffffbadX',
    0xb448: 'oooooba15',
    0xb449: 'ooooaaaZ',
    0xb44a: 'lllllab05',
    0xb44b: 'llllbbdX',
    0xb44c: 'iiiiibb05',
    0xb44d: 'iiiiabdX',
    0xb44e: 'jjjjjaa15',
    0xb44f: 'jjjjbaaZ',
    0xb450: 'XXXXXXaa04',
    0xb451: 'kkkkbaaZ',
    0xb452: 'xxxxxab14',
    0xb453: 'hhhhabdX',
    0xb454: '=====ba13',
    0xb455: 'mmmmbbdX',
    0xb456: '>>>>>ab03',
    0xb457: 'nnnnaaaZ',
    0xb458: 'wwwwwba14',
    0xb459: 'ggggbadX',
    0xb45a: 'tttttab04',
    0xb45b: 'ddddabaZ',
    0xb45c: 'qqqqqbb04',
    0xb45d: 'aaaabbaZ',
    0xb45e: 'rrrrraa14',
    0xb45f: 'bbbbaadX',
    0xb460: 'ppppppaa07',
    0xb461: 'ccccbbaZ',
    0xb462: 'pppppbb04',
    0xb463: 'ppppbacZ',
    0xb464: 'uuuuuab04',
    0xb465: 'eeeebadX',
    0xb466: 'uuuuuuba07',
    0xb467: 'ffffabaZ',
    0xb468: '00000000abbX',
    0xb469: 'oooobbdX',
    0xb46a: '00000000abbH',
    0xb46b: 'llllaaaZ',
    0xb46c: 'yyyyyab14',
    0xb46d: 'iiiibaaZ',
    0xb46e: 'yyyyyyba17',
    0xb46f: 'jjjjabdX',
    0xb470: 'hhhhhhaa06',
    0xb471: 'kkkkabdX',
    0xb472: 'hhhhhbb05',
    0xb473: 'hhhhbaaZ',
    0xb474: 'mmmmmab05',
    0xb475: 'mmmmaaaZ',
    0xb476: 'mmmmmmba06',
    0xb477: 'nnnnbbdX',
    0xb478: 'ddddddaa16',
    0xb479: 'ggggabaZ',
    0xb47a: 'dddddbb15',
    0xb47b: 'ddddbadX',
    0xb47c: 'aaaaaab15',
    0xb47d: 'aaaaaadX',
    0xb47e: 'aaaaaaba16',
    0xb47f: 'bbbbbbaZ',
    0xb480: 'cccccaa05',
    0xb481: 'ccccabdX',
    0xb482: 'ccccccba16',
    0xb483: 'ppppaabZ',
    0xb484: 'eeeeeab15',
    0xb485: 'eeeeaaaZ',
    0xb486: 'eeeeeebb06',
    0xb487: 'ffffbbdX',
    0xb488: 'llllllab16',
    0xb489: 'ooooabaZ',
    0xb48a: 'lllllbb05',
    0xb48b: 'llllbadX',
    0xb48c: 'iiiiiab05',
    0xb48d: 'iiiiaadX',
    0xb48e: 'iiiiiibb16',
    0xb48f: 'jjjjbbaZ',
    0xb490: 'xxxxxxab07',
    0xb491: 'kkkkbbaZ',
    0xb492: 'xxxxxbb14',
    0xb493: 'hhhhaadX',
    0xb494: '8888888abbh',
    0xb495: 'mmmmbadX',
    0xb496: '8888888abbH',
    0xb497: 'nnnnabaZ',
    0xb498: 'ttttttab17',
    0xb499: 'ggggbbdX',
    0xb49a: 'tttttbb04',
    0xb49b: 'ddddaaaZ',
    0xb49c: 'qqqqqab04',
    0xb49d: 'aaaabaaZ',
    0xb49e: 'qqqqqqbb17',
    0xb49f: 'bbbbabdX',
    0xb4a0: 'sssssba14',
    0xb4a1: 'ccccbaaZ',
    0xb4a2: 'pppppab04',
    0xb4a3: 'ppppbbcZ',
    0xb4a4: 'uuuuubb04',
    0xb4a5: 'eeeebbdX',
    0xb4a6: 'vvvvvaa14',
    0xb4a7: 'ffffaaaZ',
    0xb4a8: '?????bb03',
    0xb4a9: 'oooobadX',
    0xb4aa: '<<<<<aa13',
    0xb4ab: 'llllabaZ',
    0xb4ac: 'yyyyybb14',
    0xb4ad: 'iiiibbaZ',
    0xb4ae: 'zzzzzaa04',
    0xb4af: 'jjjjaadX',
    0xb4b0: 'kkkkkba15',
    0xb4b1: 'kkkkaadX',
    0xb4b2: 'hhhhhab05',
    0xb4b3: 'hhhhbbaZ',
    0xb4b4: 'mmmmmbb05',
    0xb4b5: 'mmmmabaZ',
    0xb4b6: 'nnnnnaa15',
    0xb4b7: 'nnnnbadX',
    0xb4b8: 'gggggba05',
    0xb4b9: 'ggggaaaZ',
    0xb4ba: 'dddddab15',
    0xb4bb: 'ddddbbdX',
    0xb4bc: 'aaaaabb15',
    0xb4bd: 'aaaaabdX',
    0xb4be: 'bbbbbaa05',
    0xb4bf: 'bbbbbaaZ',
    0xb4c0: 'cccccbb15',
    0xb4c1: 'ccccaaaZ',
    0xb4c2: 'CCCCCCaa05',
    0xb4c3: 'ppppabcZ',
    0xb4c4: 'eeeeeba05',
    0xb4c5: 'eeeeabdX',
    0xb4c6: 'fffffab15',
    0xb4c7: 'ffffbaaZ',
    0xb4c8: 'ooooobb05',
    0xb4c9: 'ooooaadX',
    0xb4ca: 'lllllaa15',
    0xb4cb: 'llllbbaZ',
    0xb4cc: 'iiiiiba15',
    0xb4cd: 'iiiiabaZ',
    0xb4ce: 'jjjjjab05',
    0xb4cf: 'jjjjbadX',
    0xb4d0: 'XXXXXXbb14',
    0xb4d1: 'kkkkbadX',
    0xb4d2: 'xxxxxaa04',
    0xb4d3: 'hhhhabaZ',
    0xb4d4: 'XXXXXXXabbl',
    0xb4d5: 'mmmmbbaZ',
    0xb4d6: 'XXXXXXXabbL',
    0xb4d7: 'nnnnaadX',
    0xb4d8: 'wwwwwbb04',
    0xb4d9: 'ggggbaaZ',
    0xb4da: 'tttttaa14',
    0xb4db: 'ddddabdX',
    0xb4dc: 'qqqqqba14',
    0xb4dd: 'aaaabbdX',
    0xb4de: 'rrrrrab04',
    0xb4df: 'bbbbaaaZ',
    0xb4e0: 'ppppppbb17',
    0xb4e1: 'ccccbbdX',
    0xb4e2: 'pppppba14',
    0xb4e3: 'ppppbabZ',
    0xb4e4: 'pppppppabbo',
    0xb4e5: 'eeeebaaZ',
    0xb4e6: 'pppppppabbO',
    0xb4e7: 'ffffabdX',
    0xb4e8: 'yyyyyyybbaN',
    0xb4e9: 'oooobbaZ',
    0xb4ea: 'yyyyyyybban',
    0xb4eb: 'llllaadX',
    0xb4ec: 'yyyyyaa04',
    0xb4ed: 'iiiibadX',
    0xb4ee: 'yyyyyyab07',
    0xb4ef: 'jjjjabaZ',
    0xb4f0: 'hhhhhhbb16',
    0xb4f1: 'hhhhhhhal9',
    0xb4f2: 'hhhhhba15',
    0xb4f3: 'hhhhbadX',
    0xb4f4: 'hhhhhhhabbn',
    0xb4f5: 'hhhhhhhabjz',
    0xb4f6: 'hhhhhhhabbN',
    0xb4f7: 'hhhhhhhabjZ',
    0xb4f8: 'aaaaaaabbaO',
    0xb4f9: 'aaaaaaabbkZ',
    0xb4fa: 'aaaaaaabbao',
    0xb4fb: 'aaaaaaabbkz',
    0xb4fc: 'aaaaaaa05',
    0xb4fd: 'aaaaaaaZ',
    0xb4fe: 'aaaaaaab06',
    0xb4ff: 'aaaaaaabl8',
    0xb500: 'cccccab05',
    0xb501: 'ccccabcZ',
    0xb502: 'ccccccaa16',
    0xb503: 'cccccccbl8',
    0xb504: 'cccccccbbaO',
    0xb505: 'cccccccbbkZ',
    0xb506: 'cccccccbbao',
    0xb507: 'cccccccbbkz',
    0xb508: 'iiiiiiibaaO',
    0xb509: 'iiiiiiibakZ',
    0xb50a: 'iiiiiiibaao',
    0xb50b: 'iiiiiiibakz',
    0xb50c: 'iiiiiaa05',
    0xb50d: 'iiiiaacZ',
    0xb50e: 'iiiiiiab16',
    0xb50f: 'iiiiiiibm9',
    0xb510: 'xxxxxxbb07',
    0xb511: 'kkkkbbbZ',
    0xb512: 'xxxxxba14',
    0xb513: 'hhhhaacZ',
    0xb514: 'xxxxxxxaabo',
    0xb515: 'mmmmbacZ',
    0xb516: 'xxxxxxxaabO',
    0xb517: 'nnnnabbZ',
    0xb518: 'qqqqqqqbaaN',
    0xb519: 'ggggbbcZ',
    0xb51a: 'qqqqqqqbaan',
    0xb51b: 'ddddaabZ',
    0xb51c: 'qqqqqaa04',
    0xb51d: 'aaaababZ',
    0xb51e: 'qqqqqqab17',
    0xb51f: 'bbbbabcZ',
    0xb520: 'sssssbb14',
    0xb521: 'ccccbabZ',
    0xb522: 'pppppaa04',
    0xb523: 'ppppbbdX',
    0xb524: 'uuuuuba04',
    0xb525: 'eeeebbcZ',
    0xb526: 'vvvvvab14',
    0xb527: 'ffffaabZ',
    0xb528: 'YYYYYYYbaaM',
    0xb529: 'oooobacZ',
    0xb52a: 'YYYYYYYbaam',
    0xb52b: 'llllabbZ',
    0xb52c: 'yyyyyba14',
    0xb52d: 'iiiibbbZ',
    0xb52e: 'zzzzzab04',
    0xb52f: 'jjjjaacZ',
    0xb530: 'kkkkkbb15',
    0xb531: 'kkkkaacZ',
    0xb532: 'hhhhhaa05',
    0xb533: 'hhhhbbbZ',
    0xb534: 'mmmmmba05',
    0xb535: 'mmmmabbZ',
    0xb536: 'nnnnnab15',
    0xb537: 'nnnnbacZ',
    0xb538: 'gggggbb05',
    0xb539: 'ggggaabZ',
    0xb53a: 'dddddaa15',
    0xb53b: 'ddddbbcZ',
    0xb53c: 'aaaaaba15',
    0xb53d: 'aaaaabcZ',
    0xb53e: 'bbbbbab05',
    0xb53f: 'bbbbbabZ',
    0xb540: 'cccccba15',
    0xb541: 'ccccaabZ',
    0xb542: 'CCCCCCba05',
    0xb543: 'ppppabdX',
    0xb544: 'eeeeebb05',
    0xb545: 'eeeeabcZ',
    0xb546: 'fffffaa15',
    0xb547: 'ffffbabZ',
    0xb548: 'oooooba05',
    0xb549: 'ooooaacZ',
    0xb54a: 'lllllab15',
    0xb54b: 'llllbbbZ',
    0xb54c: 'iiiiibb15',
    0xb54d: 'iiiiabbZ',
    0xb54e: 'jjjjjaa05',
    0xb54f: 'jjjjbacZ',
    0xb550: 'XXXXXXab14',
    0xb551: 'kkkkbacZ',
    0xb552: 'xxxxxab04',
    0xb553: 'hhhhabbZ',
    0xb554: '=====ba03',
    0xb555: 'mmmmbbbZ',
    0xb556: '>>>>>ab13',
    0xb557: 'nnnnaacZ',
    0xb558: 'wwwwwba04',
    0xb559: 'ggggbabZ',
    0xb55a: 'tttttab14',
    0xb55b: 'ddddabcZ',
    0xb55c: 'qqqqqbb14',
    0xb55d: 'aaaabbcZ',
    0xb55e: 'rrrrraa04',
    0xb55f: 'bbbbaabZ',
    0xb560: 'ppppppab17',
    0xb561: 'ccccbbcZ',
    0xb562: 'pppppbb14',
    0xb563: 'ppppbaaZ',
    0xb564: 'uuuuuab14',
    0xb565: 'eeeebabZ',
    0xb566: 'uuuuuubb17',
    0xb567: 'ffffabcZ',
    0xb568: '9999999baaI',
    0xb569: 'oooobbbZ',
    0xb56a: '9999999baai',
    0xb56b: 'llllaacZ',
    0xb56c: 'yyyyyab04',
    0xb56d: 'iiiibacZ',
    0xb56e: 'yyyyyybb07',
    0xb56f: 'jjjjabbZ',
    0xb570: 'hhhhhhab16',
    0xb571: 'kkkkabbZ',
    0xb572: 'hhhhhbb15',
    0xb573: 'hhhhbacZ',
    0xb574: 'mmmmmab15',
    0xb575: 'mmmmaacZ',
    0xb576: 'mmmmmmbb16',
    0xb577: 'nnnnbbbZ',
    0xb578: 'ddddddab06',
    0xb579: 'ggggabcZ',
    0xb57a: 'dddddbb05',
    0xb57b: 'ddddbabZ',
    0xb57c: 'aaaaaab05',
    0xb57d: 'aaaaaabZ',
    0xb57e: 'aaaaaabb06',
    0xb57f: 'bbbbbbcZ',
    0xb580: 'cccccaa15',
    0xb581: 'ccccabbZ',
    0xb582: 'ccccccbb06',
    0xb583: 'ppppaadX',
    0xb584: 'eeeeeab05',
    0xb585: 'eeeeaacZ',
    0xb586: 'eeeeeeba16',
    0xb587: 'ffffbbbZ',
    0xb588: 'llllllaa06',
    0xb589: 'ooooabcZ',
    0xb58a: 'lllllbb15',
    0xb58b: 'llllbabZ',
    0xb58c: 'iiiiiab15',
    0xb58d: 'iiiiaabZ',
    0xb58e: 'iiiiiiba06',
    0xb58f: 'jjjjbbcZ',
    0xb590: 'xxxxxxaa17',
    0xb591: 'kkkkbbcZ',
    0xb592: 'xxxxxbb04',
    0xb593: 'hhhhaabZ',
    0xb594: '11111111abbH',
    0xb595: 'mmmmbabZ',
    0xb596: '11111111abbX',
    0xb597: 'nnnnabcZ',
    0xb598: 'ttttttaa07',
    0xb599: 'ggggbbbZ',
    0xb59a: 'tttttbb14',
    0xb59b: 'ddddaacZ',
    0xb59c: 'qqqqqab14',
    0xb59d: 'aaaabacZ',
    0xb59e: 'qqqqqqba07',
    0xb59f: 'bbbbabbZ',
    0xb5a0: 'sssssba04',
    0xb5a1: 'ccccbacZ',
    0xb5a2: 'pppppab14',
    0xb5a3: 'ppppbbaZ',
    0xb5a4: 'uuuuubb14',
    0xb5a5: 'eeeebbbZ',
    0xb5a6: 'vvvvvaa04',
    0xb5a7: 'ffffaacZ',
    0xb5a8: '?????bb13',
    0xb5a9: 'oooobabZ',
    0xb5aa: '<<<<<aa03',
    0xb5ab: 'llllabcZ',
    0xb5ac: 'yyyyybb04',
    0xb5ad: 'iiiibbcZ',
    0xb5ae: 'zzzzzaa14',
    0xb5af: 'jjjjaabZ',
    0xb5b0: 'kkkkkba05',
    0xb5b1: 'kkkkaabZ',
    0xb5b2: 'hhhhhab15',
    0xb5b3: 'hhhhbbcZ',
    0xb5b4: 'mmmmmbb15',
    0xb5b5: 'mmmmabcZ',
    0xb5b6: 'nnnnnaa05',
    0xb5b7: 'nnnnbabZ',
    0xb5b8: 'gggggba15',
    0xb5b9: 'ggggaacZ',
    0xb5ba: 'dddddab05',
    0xb5bb: 'ddddbbbZ',
    0xb5bc: 'aaaaabb05',
    0xb5bd: 'aaaaabbZ',
    0xb5be: 'bbbbbaa15',
    0xb5bf: 'bbbbbacZ',
    0xb5c0: 'cccccbb05',
    0xb5c1: 'ccccaacZ',
    0xb5c2: 'CCCCCCab15',
    0xb5c3: 'ppppabaZ',
    0xb5c4: 'eeeeeba15',
    0xb5c5: 'eeeeabbZ',
    0xb5c6: 'fffffab05',
    0xb5c7: 'ffffbacZ',
    0xb5c8: 'ooooobb15',
    0xb5c9: 'ooooaabZ',
    0xb5ca: 'lllllaa05',
    0xb5cb: 'llllbbcZ',
    0xb5cc: 'iiiiiba05',
    0xb5cd: 'iiiiabcZ',
    0xb5ce: 'jjjjjab15',
    0xb5cf: 'jjjjbabZ',
    0xb5d0: 'CCCCCCCCCbabq',
    0xb5d1: 'kkkkbabZ',
    0xb5d2: 'xxxxxaa14',
    0xb5d3: 'hhhhabcZ',
    0xb5d4: 'CCCCCCCCCbaba',
    0xb5d5: 'mmmmbbcZ',
    0xb5d6: 'CCCCCCCCCbabi',
    0xb5d7: 'nnnnaabZ',
    0xb5d8: 'wwwwwbb14',
    0xb5d9: 'ggggbacZ',
    0xb5da: 'tttttaa04',
    0xb5db: 'ddddabbZ',
    0xb5dc: 'qqqqqba04',
    0xb5dd: 'aaaabbbZ',
    0xb5de: 'rrrrrab14',
    0xb5df: 'bbbbaacZ',
    0xb5e0: 'aaaaaaaaabbbB',
    0xb5e1: 'aaaaaaaaabbbF',
    0xb5e2: 'aaaaaaaaabbbJ',
    0xb5e3: 'aaaaaaaaabbbN',
    0xb5e4: 'aaaaaaaaabbbR',
    0xb5e5: 'aaaaaaaaabbbV',
    0xb5e6: 'aaaaaaaaabbbZ',
    0xb5e7: 'aaaaaaaaabbjZ',
    0xb5e8: 'aaaaaaaaabbbb',
    0xb5e9: 'aaaaaaaaabbbf',
    0xb5ea: 'aaaaaaaaabbbj',
    0xb5eb: 'aaaaaaaaabbbn',
    0xb5ec: 'aaaaaaaaabbbr',
    0xb5ed: 'aaaaaaaaabbbv',
    0xb5ee: 'aaaaaaaaabbbz',
    0xb5ef: 'aaaaaaaaabbjz',
    0xb5f0: 'aaaaaaaaabbn',
    0xb5f1: 'aaaaaaaaabbf',
    0xb5f2: 'aaaaaaaaabjz',
    0xb5f3: 'aaaaaaaaabbv',
    0xb5f4: 'aaaaaaaaabbN',
    0xb5f5: 'aaaaaaaaabbF',
    0xb5f6: 'aaaaaaaaabjZ',
    0xb5f7: 'aaaaaaaaabbV',
    0xb5f8: 'aaaaaaaaabN',
    0xb5f9: 'aaaaaaaaajZ',
    0xb5fa: 'aaaaaaaaabn',
    0xb5fb: 'aaaaaaaaajz',
    0xb5fc: 'aaaaaaa15',
    0xb5fd: 'aaaaaacZ',
    0xb5fe: 'aaaaaaaa16',
    0xb5ff: 'aaaaaaaam8',
    0xb600: 'cccccab14',
    0xb601: 'ccccabaX',
    0xb602: 'ccccccab16',
    0xb603: 'ccccccccm8',
    0xb604: 'cccccccabaO',
    0xb605: 'cccccccabkZ',
    0xb606: 'cccccccabao',
    0xb607: 'cccccccabkz',
    0xb608: 'ccccccccabbN',
    0xb609: 'ccccccccabbF',
    0xb60a: 'ccccccccabjZ',
    0xb60b: 'ccccccccabbV',
    0xb60c: 'ccccccccabbn',
    0xb60d: 'ccccccccabbf',
    0xb60e: 'ccccccccabjz',
    0xb60f: 'ccccccccabbv',
    0xb610: 'cccccccccaaas',
    0xb611: 'cccccccccaaaw',
    0xb612: 'cccccccccaacz',
    0xb613: 'cccccccccaakz',
    0xb614: 'cccccccccaaac',
    0xb615: 'cccccccccaaag',
    0xb616: 'cccccccccaaak',
    0xb617: 'cccccccccaaao',
    0xb618: 'cccccccccaaaS',
    0xb619: 'cccccccccaaaW',
    0xb61a: 'cccccccccaacZ',
    0xb61b: 'cccccccccaakZ',
    0xb61c: 'cccccccccaaaC',
    0xb61d: 'aaaababY',
    0xb61e: 'cccccccccaaaK',
    0xb61f: 'bbbbabaX',
    0xb620: 'ccccccccccbbbn',
    0xb621: 'ccccbabY',
    0xb622: 'ccccccccccbbbj',
    0xb623: 'ccccccccccbbbh',
    0xb624: 'ccccccccccbbbf',
    0xb625: 'ccccccccccbbbd',
    0xb626: 'ccccccccccbbbb',
    0xb627: 'ccccccccccbbdc',
    0xb628: 'ccccccccccbbjz',
    0xb629: 'ccccccccccbbhy',
    0xb62a: 'ccccccccccbbbz',
    0xb62b: 'ccccccccccbbbx',
    0xb62c: 'ccccccccccbbbv',
    0xb62d: 'ccccccccccbbbt',
    0xb62e: 'ccccccccccbbbr',
    0xb62f: 'ccccccccccbbbp',
    0xb630: 'ccccccccccbbbN',
    0xb631: 'ccccccccccbbbL',
    0xb632: 'ccccccccccbbbJ',
    0xb633: 'ccccccccccbbbH',
    0xb634: 'ccccccccccbbbF',
    0xb635: 'ccccccccccbbbD',
    0xb636: 'ccccccccccbbbB',
    0xb637: 'ccccccccccbbdC',
    0xb638: 'ccccccccccbbjZ',
    0xb639: 'ccccccccccbbhY',
    0xb63a: 'ccccccccccbbbZ',
    0xb63b: 'ccccccccccbbbX',
    0xb63c: 'aaaaaba04',
    0xb63d: 'aaaaabaX',
    0xb63e: 'bbbbbab14',
    0xb63f: 'bbbbbabY',
    0xb640: 'cccccba04',
    0xb641: 'ccccaabY',
    0xb642: 'CCCCCCbb05',
    0xb643: 'ppppabbX',
    0xb644: 'eeeeebb14',
    0xb645: 'eeeeabaX',
    0xb646: 'fffffaa04',
    0xb647: 'ffffbabY',
    0xb648: 'oooooba14',
    0xb649: 'ooooaaaX',
    0xb64a: 'lllllab04',
    0xb64b: 'llllbbbY',
    0xb64c: 'iiiiibb04',
    0xb64d: 'iiiiabbY',
    0xb64e: 'jjjjjaa14',
    0xb64f: 'jjjjbaaX',
    0xb650: 'XXXXXXaa14',
    0xb651: 'kkkkbaaX',
    0xb652: 'xxxxxab15',
    0xb653: 'hhhhabbY',
    0xb654: '0000000000aaaE',
    0xb655: 'mmmmbbbY',
    0xb656: '0000000000aaaA',
    0xb657: 'nnnnaaaX',
    0xb658: 'wwwwwba15',
    0xb659: 'ggggbabY',
    0xb65a: 'tttttab05',
    0xb65b: 'ddddabaX',
    0xb65c: 'qqqqqbb05',
    0xb65d: 'aaaabbaX',
    0xb65e: 'rrrrraa15',
    0xb65f: 'bbbbaabY',
    0xb660: 'ppppppaa17',
    0xb661: 'ccccbbaX',
    0xb662: 'pppppbb05',
    0xb663: 'ppppbaaY',
    0xb664: 'uuuuuab05',
    0xb665: 'eeeebabY',
    0xb666: 'uuuuuuba17',
    0xb667: 'ffffabaX',
    0xb668: '00000000baaY',
    0xb669: 'oooobbbY',
    0xb66a: '00000000baaI',
    0xb66b: 'llllaaaX',
    0xb66c: 'yyyyyab15',
    0xb66d: 'iiiibaaX',
    0xb66e: 'yyyyyyba07',
    0xb66f: 'jjjjabbY',
    0xb670: 'hhhhhhaa16',
    0xb671: 'kkkkabbY',
    0xb672: 'hhhhhbb04',
    0xb673: 'hhhhbaaX',
    0xb674: 'mmmmmab04',
    0xb675: 'mmmmaaaX',
    0xb676: 'mmmmmmba16',
    0xb677: 'nnnnbbbY',
    0xb678: 'ddddddaa06',
    0xb679: 'ggggabaX',
    0xb67a: 'dddddbb14',
    0xb67b: 'ddddbabY',
    0xb67c: 'aaaaaab14',
    0xb67d: 'aaaaaabY',
    0xb67e: 'aaaaaaba06',
    0xb67f: 'bbbbbbaX',
    0xb680: 'cccccaa04',
    0xb681: 'ccccabbY',
    0xb682: 'ccccccba06',
    0xb683: 'ppppaabX',
    0xb684: 'eeeeeab14',
    0xb685: 'eeeeaaaX',
    0xb686: 'eeeeeebb16',
    0xb687: 'ffffbbbY',
    0xb688: 'llllllab06',
    0xb689: 'ooooabaX',
    0xb68a: 'lllllbb04',
    0xb68b: 'llllbabY',
    0xb68c: 'iiiiiab04',
    0xb68d: 'iiiiaabY',
    0xb68e: 'iiiiiibb06',
    0xb68f: 'jjjjbbaX',
    0xb690: 'xxxxxxab17',
    0xb691: 'kkkkbbaX',
    0xb692: 'xxxxxbb15',
    0xb693: 'hhhhaabY',
    0xb694: '8888888aaai',
    0xb695: 'mmmmbabY',
    0xb696: '8888888aaaI',
    0xb697: 'nnnnabaX',
    0xb698: 'ttttttab07',
    0xb699: 'ggggbbbY',
    0xb69a: 'tttttbb05',
    0xb69b: 'ddddaaaX',
    0xb69c: 'qqqqqab05',
    0xb69d: 'aaaabaaX',
    0xb69e: 'qqqqqqbb07',
    0xb69f: 'bbbbabbY',
    0xb6a0: 'sssssba15',
    0xb6a1: 'ccccbaaX',
    0xb6a2: 'pppppab05',
    0xb6a3: 'ppppbbaY',
    0xb6a4: 'uuuuubb05',
    0xb6a5: 'eeeebbbY',
    0xb6a6: 'vvvvvaa15',
    0xb6a7: 'ffffaaaX',
    0xb6a8: '?????bb02',
    0xb6a9: 'oooobabY',
    0xb6aa: '<<<<<aa12',
    0xb6ab: 'llllabaX',
    0xb6ac: 'yyyyybb15',
    0xb6ad: 'iiiibbaX',
    0xb6ae: 'zzzzzaa05',
    0xb6af: 'jjjjaabY',
    0xb6b0: 'kkkkkba14',
    0xb6b1: 'kkkkaabY',
    0xb6b2: 'hhhhhab04',
    0xb6b3: 'hhhhbbaX',
    0xb6b4: 'mmmmmbb04',
    0xb6b5: 'mmmmabaX',
    0xb6b6: 'nnnnnaa14',
    0xb6b7: 'nnnnbabY',
    0xb6b8: 'gggggba04',
    0xb6b9: 'ggggaaaX',
    0xb6ba: 'dddddab14',
    0xb6bb: 'ddddbbbY',
    0xb6bc: 'aaaaabb14',
    0xb6bd: 'aaaaabbY',
    0xb6be: 'bbbbbaa04',
    0xb6bf: 'bbbbbaaX',
    0xb6c0: 'cccccbb14',
    0xb6c1: 'ccccaaaX',
    0xb6c2: 'CCCCCCaa15',
    0xb6c3: 'ppppabaY',
    0xb6c4: 'eeeeeba04',
    0xb6c5: 'eeeeabbY',
    0xb6c6: 'fffffab14',
    0xb6c7: 'ffffbaaX',
    0xb6c8: 'ooooobb04',
    0xb6c9: 'ooooaabY',
    0xb6ca: 'lllllaa14',
    0xb6cb: 'llllbbaX',
    0xb6cc: 'iiiiiba14',
    0xb6cd: 'iiiiabaX',
    0xb6ce: 'jjjjjab04',
    0xb6cf: 'jjjjbabY',
    0xb6d0: 'XXXXXXbb04',
    0xb6d1: 'kkkkbabY',
    0xb6d2: 'xxxxxaa05',
    0xb6d3: 'hhhhabaX',
    0xb6d4: 'XXXXXXXaaam',
    0xb6d5: 'mmmmbbaX',
    0xb6d6: 'XXXXXXXaaaM',
    0xb6d7: 'nnnnaabY',
    0xb6d8: 'wwwwwbb05',
    0xb6d9: 'ggggbaaX',
    0xb6da: 'tttttaa15',
    0xb6db: 'ddddabbY',
    0xb6dc: 'qqqqqba15',
    0xb6dd: 'aaaabbbY',
    0xb6de: 'rrrrrab05',
    0xb6df: 'bbbbaaaX',
    0xb6e0: 'ppppppbb07',
    0xb6e1: 'ccccbbbY',
    0xb6e2: 'pppppba15',
    0xb6e3: 'ppppbabX',
    0xb6e4: 'pppppppaaan',
    0xb6e5: 'eeeebaaX',
    0xb6e6: 'pppppppaaaN',
    0xb6e7: 'ffffabbY',
    0xb6e8: 'yyyyyyybabO',
    0xb6e9: 'oooobbaX',
    0xb6ea: 'yyyyyyybabo',
    0xb6eb: 'llllaabY',
    0xb6ec: 'yyyyyaa05',
    0xb6ed: 'iiiibabY',
    0xb6ee: 'yyyyyyab17',
    0xb6ef: 'jjjjabaX',
    0xb6f0: 'hhhhhhbb06',
    0xb6f1: 'hhhhhhham9',
    0xb6f2: 'hhhhhba14',
    0xb6f3: 'hhhhbabY',
    0xb6f4: 'hhhhhhhaaao',
    0xb6f5: 'hhhhhhhaakz',
    0xb6f6: 'hhhhhhhaaaO',
    0xb6f7: 'hhhhhhhaakZ',
    0xb6f8: 'aaaaaaababN',
    0xb6f9: 'aaaaaaabajZ',
    0xb6fa: 'aaaaaaababn',
    0xb6fb: 'aaaaaaabajz',
    0xb6fc: 'aaaaaaa04',
    0xb6fd: 'aaaaaaaX',
    0xb6fe: 'aaaaaaab16',
    0xb6ff: 'bbbbbbbY',
    0xb700: 'cccccab04',
    0xb701: 'ccccabaY',
    0xb702: 'ccccccaa06',
    0xb703: 'eeeeeeebbkz',
    0xb704: 'cccccccbabN',
    0xb705: 'cccccccbajZ',
    0xb706: 'cccccccbabn',
    0xb707: 'cccccccbajz',
    0xb708: 'iiiiiiibbbN',
    0xb709: 'iiiiiiibbjZ',
    0xb70a: 'iiiiiiibbbn',
    0xb70b: 'iiiiiiibbjz',
    0xb70c: 'iiiiiaa04',
    0xb70d: 'iiiiaaaY',
    0xb70e: 'iiiiiiab06',
    0xb70f: 'iiiiiiibl9',
    0xb710: 'xxxxxxbb17',
    0xb711: 'kkkkbbbX',
    0xb712: 'xxxxxba15',
    0xb713: 'hhhhaaaY',
    0xb714: 'xxxxxxxaban',
    0xb715: 'mmmmbaaY',
    0xb716: 'xxxxxxxabaN',
    0xb717: 'nnnnabbX',
    0xb718: 'qqqqqqqbbbO',
    0xb719: 'ggggbbaY',
    0xb71a: 'qqqqqqqbbbo',
    0xb71b: 'ddddaabX',
    0xb71c: 'qqqqqaa05',
    0xb71d: 'aaaababX',
    0xb71e: 'qqqqqqab07',
    0xb71f: 'bbbbabaY',
    0xb720: 'sssssbb15',
    0xb721: 'ccccbabX',
    0xb722: 'pppppaa05',
    0xb723: 'ppppbbbY',
    0xb724: 'uuuuuba05',
    0xb725: 'eeeebbaY',
    0xb726: 'vvvvvab15',
    0xb727: 'ffffaabX',
    0xb728: 'YYYYYYYbbbL',
    0xb729: 'oooobaaY',
    0xb72a: 'YYYYYYYbbbl',
    0xb72b: 'llllabbX',
    0xb72c: 'yyyyyba15',
    0xb72d: 'iiiibbbX',
    0xb72e: 'zzzzzab05',
    0xb72f: 'jjjjaaaY',
    0xb730: 'kkkkkbb14',
    0xb731: 'kkkkaaaY',
    0xb732: 'hhhhhaa04',
    0xb733: 'hhhhbbbX',
    0xb734: 'mmmmmba04',
    0xb735: 'mmmmabbX',
    0xb736: 'nnnnnab14',
    0xb737: 'nnnnbaaY',
    0xb738: 'gggggbb04',
    0xb739: 'ggggaabX',
    0xb73a: 'dddddaa14',
    0xb73b: 'ddddbbaY',
    0xb73c: 'aaaaaba14',
    0xb73d: 'aaaaabaY',
    0xb73e: 'bbbbbab04',
    0xb73f: 'bbbbbabX',
    0xb740: 'cccccba14',
    0xb741: 'ccccaabX',
    0xb742: 'CCCCCCba15',
    0xb743: 'ppppabbY',
    0xb744: 'eeeeebb04',
    0xb745: 'eeeeabaY',
    0xb746: 'fffffaa14',
    0xb747: 'ffffbabX',
    0xb748: 'oooooba04',
    0xb749: 'ooooaaaY',
    0xb74a: 'lllllab14',
    0xb74b: 'llllbbbX',
    0xb74c: 'iiiiibb14',
    0xb74d: 'iiiiabbX',
    0xb74e: 'jjjjjaa04',
    0xb74f: 'jjjjbaaY',
    0xb750: 'XXXXXXab04',
    0xb751: 'kkkkbaaY',
    0xb752: 'xxxxxab05',
    0xb753: 'hhhhabbX',
    0xb754: '=====ba02',
    0xb755: 'mmmmbbbX',
    0xb756: '>>>>>ab12',
    0xb757: 'nnnnaaaY',
    0xb758: 'wwwwwba05',
    0xb759: 'ggggbabX',
    0xb75a: 'tttttab15',
    0xb75b: 'ddddabaY',
    0xb75c: 'qqqqqbb15',
    0xb75d: 'aaaabbaY',
    0xb75e: 'rrrrraa05',
    0xb75f: 'bbbbaabX',
    0xb760: 'ppppppab07',
    0xb761: 'ccccbbaY',
    0xb762: 'pppppbb15',
    0xb763: 'ppppbaaX',
    0xb764: 'uuuuuab15',
    0xb765: 'eeeebabX',
    0xb766: 'uuuuuubb07',
    0xb767: 'ffffabaY',
    0xb768: '9999999bbbH',
    0xb769: 'oooobbbX',
    0xb76a: '9999999bbbh',
    0xb76b: 'llllaaaY',
    0xb76c: 'yyyyyab05',
    0xb76d: 'iiiibaaY',
    0xb76e: 'yyyyyybb17',
    0xb76f: 'jjjjabbX',
    0xb770: 'hhhhhhab06',
    0xb771: 'kkkkabbX',
    0xb772: 'hhhhhbb14',
    0xb773: 'hhhhbaaY',
    0xb774: 'mmmmmab14',
    0xb775: 'mmmmaaaY',
    0xb776: 'mmmmmmbb06',
    0xb777: 'nnnnbbbX',
    0xb778: 'ddddddab16',
    0xb779: 'ggggabaY',
    0xb77a: 'dddddbb04',
    0xb77b: 'ddddbabX',
    0xb77c: 'aaaaaab04',
    0xb77d: 'aaaaaabX',
    0xb77e: 'aaaaaabb16',
    0xb77f: 'bbbbbbaY',
    0xb780: 'cccccaa14',
    0xb781: 'ccccabbX',
    0xb782: 'ccccccbb16',
    0xb783: 'ppppaabY',
    0xb784: 'eeeeeab04',
    0xb785: 'eeeeaaaY',
    0xb786: 'eeeeeeba06',
    0xb787: 'ffffbbbX',
    0xb788: 'llllllaa16',
    0xb789: 'ooooabaY',
    0xb78a: 'lllllbb14',
    0xb78b: 'llllbabX',
    0xb78c: 'iiiiiab14',
    0xb78d: 'iiiiaabX',
    0xb78e: 'iiiiiiba16',
    0xb78f: 'jjjjbbaY',
    0xb790: 'xxxxxxaa07',
    0xb791: 'kkkkbbaY',
    0xb792: 'xxxxxbb05',
    0xb793: 'hhhhaabX',
    0xb794: '11111111baaI',
    0xb795: 'mmmmbabX',
    0xb796: '11111111baaY',
    0xb797: 'nnnnabaY',
    0xb798: 'ttttttaa17',
    0xb799: 'ggggbbbX',
    0xb79a: 'tttttbb15',
    0xb79b: 'ddddaaaY',
    0xb79c: 'qqqqqab15',
    0xb79d: 'aaaabaaY',
    0xb79e: 'qqqqqqba17',
    0xb79f: 'bbbbabbX',
    0xb7a0: 'sssssba05',
    0xb7a1: 'ccccbaaY',
    0xb7a2: 'pppppab15',
    0xb7a3: 'ppppbbaX',
    0xb7a4: 'uuuuubb15',
    0xb7a5: 'eeeebbbX',
    0xb7a6: 'vvvvvaa05',
    0xb7a7: 'ffffaaaY',
    0xb7a8: '?????bb12',
    0xb7a9: 'oooobabX',
    0xb7aa: '<<<<<aa02',
    0xb7ab: 'llllabaY',
    0xb7ac: 'yyyyybb05',
    0xb7ad: 'iiiibbaY',
    0xb7ae: 'zzzzzaa15',
    0xb7af: 'jjjjaabX',
    0xb7b0: 'kkkkkba04',
    0xb7b1: 'kkkkaabX',
    0xb7b2: 'hhhhhab14',
    0xb7b3: 'hhhhbbaY',
    0xb7b4: 'mmmmmbb14',
    0xb7b5: 'mmmmabaY',
    0xb7b6: 'nnnnnaa04',
    0xb7b7: 'nnnnbabX',
    0xb7b8: 'gggggba14',
    0xb7b9: 'ggggaaaY',
    0xb7ba: 'dddddab04',
    0xb7bb: 'ddddbbbX',
    0xb7bc: 'aaaaabb04',
    0xb7bd: 'aaaaabbX',
    0xb7be: 'bbbbbaa14',
    0xb7bf: 'bbbbbaaY',
    0xb7c0: 'cccccbb04',
    0xb7c1: 'ccccaaaY',
    0xb7c2: 'CCCCCCab05',
    0xb7c3: 'ppppabaX',
    0xb7c4: 'eeeeeba14',
    0xb7c5: 'eeeeabbX',
    0xb7c6: 'fffffab04',
    0xb7c7: 'ffffbaaY',
    0xb7c8: 'ooooobb14',
    0xb7c9: 'ooooaabX',
    0xb7ca: 'lllllaa04',
    0xb7cb: 'llllbbaY',
    0xb7cc: 'iiiiiba04',
    0xb7cd: 'iiiiabaY',
    0xb7ce: 'jjjjjab14',
    0xb7cf: 'jjjjbabX',
    0xb7d0: 'QQQQQQQQbaam',
    0xb7d1: 'kkkkbabX',
    0xb7d2: 'xxxxxaa15',
    0xb7d3: 'hhhhabaY',
    0xb7d4: 'QQQQQQQQbaaM',
    0xb7d5: 'mmmmbbaY',
    0xb7d6: 'QQQQQQQQbaiY',
    0xb7d7: 'nnnnaabX',
    0xb7d8: 'wwwwwbb15',
    0xb7d9: 'ggggbaaY',
    0xb7da: 'tttttaa05',
    0xb7db: 'ddddabbX',
    0xb7dc: 'qqqqqba05',
    0xb7dd: 'aaaabbbX',
    0xb7de: 'rrrrrab15',
    0xb7df: 'bbbbaaaY',
    0xb7e0: 'ppppppba17',
    0xb7e1: 'ccccbbbX',
    0xb7e2: 'pppppba05',
    0xb7e3: 'ppppbabY',
    0xb7e4: 'pppppppbbbo',
    0xb7e5: 'eeeebaaY',
    0xb7e6: 'pppppppbbbO',
    0xb7e7: 'ffffabbX',
    0xb7e8: 'ppppppppaahZ',
    0xb7e9: 'oooobbaY',
    0xb7ea: 'ppppppppaabO',
    0xb7eb: 'llllaabX',
    0xb7ec: 'ppppppppaahz',
    0xb7ed: 'iiiibabX',
    0xb7ee: 'ppppppppaabo',
    0xb7ef: 'jjjjabaY',
    0xb7f0: 'aaaaaaaabaao',
    0xb7f1: 'aaaaaaaabaag',
    0xb7f2: 'aaaaaaaabakz',
    0xb7f3: 'aaaaaaaabaaw',
    0xb7f4: 'aaaaaaaabaaO',
    0xb7f5: 'aaaaaaaabaaG',
    0xb7f6: 'aaaaaaaabakZ',
    0xb7f7: 'aaaaaaaabaaW',
    0xb7f8: 'aaaaaaaabaO',
    0xb7f9: 'aaaaaaaabkZ',
    0xb7fa: 'aaaaaaaabao',
    0xb7fb: 'aaaaaaaabkz',
    0xb7fc: 'aaaaaaa14',
    0xb7fd: 'aaaaaaaY',
    0xb7fe: 'aaaaaaaa06',
    0xb7ff: 'aaaaaaaal8',
    0xb800: 'cccccab13',
    0xb801: 'ccccabaV',
    0xb802: 'ccccccab05',
    0xb803: 'cccccccch9',
    0xb804: 'cccccccaaaN',
    0xb805: 'cccccccaaiZ',
    0xb806: 'cccccccaaan',
    0xb807: 'cccccccaaiz',
    0xb808: 'ccccccccbbaO',
    0xb809: 'ccccccccbbaG',
    0xb80a: 'ccccccccbbkZ',
    0xb80b: 'ccccccccbbaW',
    0xb80c: 'ccccccccbbao',
    0xb80d: 'ccccccccbbag',
    0xb80e: 'ccccccccbbkz',
    0xb80f: 'ccccccccbbaw',
    0xb810: 'qqqqqqqqbaan',
    0xb811: 'kkkkbbbW',
    0xb812: 'qqqqqqqqbaiz',
    0xb813: 'hhhhaaaV',
    0xb814: 'qqqqqqqqbaaN',
    0xb815: 'mmmmbaaV',
    0xb816: 'qqqqqqqqbaiZ',
    0xb817: 'nnnnabbW',
    0xb818: 'qqqqqqqabaO',
    0xb819: 'ggggbbaV',
    0xb81a: 'qqqqqqqabao',
    0xb81b: 'ddddaabW',
    0xb81c: 'qqqqqaa12',
    0xb81d: 'aaaababW',
    0xb81e: 'qqqqqqaa04',
    0xb81f: 'bbbbabaV',
    0xb820: 'sssssbb02',
    0xb821: 'ccccbabW',
    0xb822: 'pppppaa12',
    0xb823: 'ppppbbbV',
    0xb824: 'uuuuuba12',
    0xb825: 'eeeebbaV',
    0xb826: 'vvvvvab02',
    0xb827: 'ffffaabW',
    0xb828: 'PPPPPPPPaahY',
    0xb829: 'oooobaaV',
    0xb82a: 'PPPPPPPPaabL',
    0xb82b: 'llllabbW',
    0xb82c: 'yyyyyba02',
    0xb82d: 'iiiibbbW',
    0xb82e: 'zzzzzab12',
    0xb82f: 'jjjjaaaV',
    0xb830: 'kkkkkbb03',
    0xb831: 'kkkkaaaV',
    0xb832: 'hhhhhaa13',
    0xb833: 'hhhhbbbW',
    0xb834: 'mmmmmba13',
    0xb835: 'mmmmabbW',
    0xb836: 'nnnnnab03',
    0xb837: 'nnnnbaaV',
    0xb838: 'gggggbb13',
    0xb839: 'ggggaabW',
    0xb83a: 'dddddaa03',
    0xb83b: 'ddddbbaV',
    0xb83c: 'aaaaaba03',
    0xb83d: 'aaaaabaV',
    0xb83e: 'bbbbbab13',
    0xb83f: 'bbbbbabW',
    0xb840: 'cccccba03',
    0xb841: 'ccccaabW',
    0xb842: 'CCCCCCbb16',
    0xb843: 'ppppabbV',
    0xb844: 'eeeeebb13',
    0xb845: 'eeeeabaV',
    0xb846: 'fffffaa03',
    0xb847: 'ffffbabW',
    0xb848: 'oooooba13',
    0xb849: 'ooooaaaV',
    0xb84a: 'lllllab03',
    0xb84b: 'llllbbbW',
    0xb84c: 'iiiiibb03',
    0xb84d: 'iiiiabbW',
    0xb84e: 'jjjjjaa13',
    0xb84f: 'jjjjbaaV',
    0xb850: 'XXXXXXaa07',
    0xb851: 'kkkkbaaV',
    0xb852: 'xxxxxab12',
    0xb853: 'hhhhabbW',
    0xb854: '=====ba15',
    0xb855: 'mmmmbbbW',
    0xb856: '>>>>>ab05',
    0xb857: 'nnnnaaaV',
    0xb858: 'wwwwwba12',
    0xb859: 'ggggbabW',
    0xb85a: 'tttttab02',
    0xb85b: 'ddddabaV',
    0xb85c: 'qqqqqbb02',
    0xb85d: 'aaaabbaV',
    0xb85e: 'rrrrraa12',
    0xb85f: 'bbbbaabW',
    0xb860: 'ppppppaa04',
    0xb861: 'ccccbbaV',
    0xb862: 'pppppbb02',
    0xb863: 'ppppbaaW',
    0xb864: 'uuuuuab02',
    0xb865: 'eeeebabW',
    0xb866: 'uuuuuuba04',
    0xb867: 'ffffabaV',
    0xb868: '00000000aabX',
    0xb869: 'oooobbbW',
    0xb86a: '00000000aabH',
    0xb86b: 'llllaaaV',
    0xb86c: 'yyyyyab12',
    0xb86d: 'iiiibaaV',
    0xb86e: 'yyyyyyba14',
    0xb86f: 'jjjjabbW',
    0xb870: 'hhhhhhaa05',
    0xb871: 'kkkkabbW',
    0xb872: 'hhhhhbb03',
    0xb873: 'hhhhbaaV',
    0xb874: 'mmmmmab03',
    0xb875: 'mmmmaaaV',
    0xb876: 'mmmmmmba05',
    0xb877: 'nnnnbbbW',
    0xb878: 'ddddddaa15',
    0xb879: 'ggggabaV',
    0xb87a: 'dddddbb13',
    0xb87b: 'ddddbabW',
    0xb87c: 'aaaaaab13',
    0xb87d: 'aaaaaabW',
    0xb87e: 'aaaaaaba15',
    0xb87f: 'bbbbbbaV',
    0xb880: 'cccccaa03',
    0xb881: 'ccccabbW',
    0xb882: 'ccccccba15',
    0xb883: 'ppppaabV',
    0xb884: 'eeeeeab13',
    0xb885: 'eeeeaaaV',
    0xb886: 'eeeeeebb05',
    0xb887: 'ffffbbbW',
    0xb888: 'llllllab15',
    0xb889: 'ooooabaV',
    0xb88a: 'lllllbb03',
    0xb88b: 'llllbabW',
    0xb88c: 'iiiiiab03',
    0xb88d: 'iiiiaabW',
    0xb88e: 'iiiiiibb15',
    0xb88f: 'jjjjbbaV',
    0xb890: 'xxxxxxab04',
    0xb891: 'kkkkbbaV',
    0xb892: 'xxxxxbb12',
    0xb893: 'hhhhaabW',
    0xb894: '8888888abah',
    0xb895: 'mmmmbabW',
    0xb896: '8888888abaH',
    0xb897: 'nnnnabaV',
    0xb898: 'ttttttab14',
    0xb899: 'ggggbbbW',
    0xb89a: 'tttttbb02',
    0xb89b: 'ddddaaaV',
    0xb89c: 'qqqqqab02',
    0xb89d: 'aaaabaaV',
    0xb89e: 'qqqqqqbb14',
    0xb89f: 'bbbbabbW',
    0xb8a0: 'sssssba12',
    0xb8a1: 'ccccbaaV',
    0xb8a2: 'pppppab02',
    0xb8a3: 'ppppbbaW',
    0xb8a4: 'uuuuubb02',
    0xb8a5: 'eeeebbbW',
    0xb8a6: 'vvvvvaa12',
    0xb8a7: 'ffffaaaV',
    0xb8a8: '?????bb05',
    0xb8a9: 'oooobabW',
    0xb8aa: '<<<<<aa15',
    0xb8ab: 'llllabaV',
    0xb8ac: 'yyyyybb12',
    0xb8ad: 'iiiibbaV',
    0xb8ae: 'zzzzzaa02',
    0xb8af: 'jjjjaabW',
    0xb8b0: 'kkkkkba13',
    0xb8b1: 'kkkkaabW',
    0xb8b2: 'hhhhhab03',
    0xb8b3: 'hhhhbbaV',
    0xb8b4: 'mmmmmbb03',
    0xb8b5: 'mmmmabaV',
    0xb8b6: 'nnnnnaa13',
    0xb8b7: 'nnnnbabW',
    0xb8b8: 'gggggba03',
    0xb8b9: 'ggggaaaV',
    0xb8ba: 'dddddab13',
    0xb8bb: 'ddddbbbW',
    0xb8bc: 'aaaaabb13',
    0xb8bd: 'aaaaabbW',
    0xb8be: 'bbbbbaa03',
    0xb8bf: 'bbbbbaaV',
    0xb8c0: 'cccccbb13',
    0xb8c1: 'ccccaaaV',
    0xb8c2: 'CCCCCCaa06',
    0xb8c3: 'ppppabaW',
    0xb8c4: 'eeeeeba03',
    0xb8c5: 'eeeeabbW',
    0xb8c6: 'fffffab13',
    0xb8c7: 'ffffbaaV',
    0xb8c8: 'ooooobb03',
    0xb8c9: 'ooooaabW',
    0xb8ca: 'lllllaa13',
    0xb8cb: 'llllbbaV',
    0xb8cc: 'iiiiiba13',
    0xb8cd: 'iiiiabaV',
    0xb8ce: 'jjjjjab03',
    0xb8cf: 'jjjjbabW',
    0xb8d0: 'XXXXXXbb17',
    0xb8d1: 'kkkkbabW',
    0xb8d2: 'xxxxxaa02',
    0xb8d3: 'hhhhabaV',
    0xb8d4: 'XXXXXXXabal',
    0xb8d5: 'mmmmbbaV',
    0xb8d6: 'XXXXXXXabaL',
    0xb8d7: 'nnnnaabW',
    0xb8d8: 'wwwwwbb02',
    0xb8d9: 'ggggbaaV',
    0xb8da: 'tttttaa12',
    0xb8db: 'ddddabbW',
    0xb8dc: 'qqqqqba12',
    0xb8dd: 'aaaabbbW',
    0xb8de: 'rrrrrab02',
    0xb8df: 'bbbbaaaV',
    0xb8e0: 'ppppppbb14',
    0xb8e1: 'ccccbbbW',
    0xb8e2: 'pppppba12',
    0xb8e3: 'ppppbabV',
    0xb8e4: 'pppppppabao',
    0xb8e5: 'eeeebaaV',
    0xb8e6: 'pppppppabaO',
    0xb8e7: 'ffffabbW',
    0xb8e8: 'yyyyyyybbbN',
    0xb8e9: 'oooobbaV',
    0xb8ea: 'yyyyyyybbbn',
    0xb8eb: 'llllaabW',
    0xb8ec: 'yyyyyaa02',
    0xb8ed: 'iiiibabW',
    0xb8ee: 'yyyyyyab04',
    0xb8ef: 'jjjjabaV',
    0xb8f0: 'hhhhhhbb15',
    0xb8f1: 'kkkkabaV',
    0xb8f2: 'hhhhhba13',
    0xb8f3: 'hhhhbabW',
    0xb8f4: 'hhhhhhhaban',
    0xb8f5: 'hhhhhhhabiz',
    0xb8f6: 'hhhhhhhabaN',
    0xb8f7: 'hhhhhhhabiZ',
    0xb8f8: 'aaaaaaabbbO',
    0xb8f9: 'aaaaaaabbhZ',
    0xb8fa: 'aaaaaaabbbo',
    0xb8fb: 'aaaaaaabbhz',
    0xb8fc: 'aaaaaaa03',
    0xb8fd: 'aaaaaaaV',
    0xb8fe: 'aaaaaaab05',
    0xb8ff: 'bbbbbbbW',
    0xb900: 'cccccab03',
    0xb901: 'ccccabaW',
    0xb902: 'ccccccaa15',
    0xb903: 'eeeeeeebaiz',
    0xb904: 'cccccccbbbO',
    0xb905: 'cccccccbbhZ',
    0xb906: 'cccccccbbbo',
    0xb907: 'cccccccbbhz',
    0xb908: 'iiiiiiibabO',
    0xb909: 'iiiiiiibahZ',
    0xb90a: 'iiiiiiibabo',
    0xb90b: 'iiiiiiibahz',
    0xb90c: 'iiiiiaa03',
    0xb90d: 'iiiiaaaW',
    0xb90e: 'iiiiiiab15',
    0xb90f: 'jjjjbbbV',
    0xb910: 'xxxxxxbb04',
    0xb911: 'kkkkbbbV',
    0xb912: 'xxxxxba12',
    0xb913: 'hhhhaaaW',
    0xb914: 'xxxxxxxaaao',
    0xb915: 'mmmmbaaW',
    0xb916: 'xxxxxxxaaaO',
    0xb917: 'nnnnabbV',
    0xb918: 'qqqqqqqbabN',
    0xb919: 'ggggbbaW',
    0xb91a: 'qqqqqqqbabn',
    0xb91b: 'ddddaabV',
    0xb91c: 'qqqqqaa02',
    0xb91d: 'aaaababV',
    0xb91e: 'qqqqqqab14',
    0xb91f: 'bbbbabaW',
    0xb920: 'sssssbb12',
    0xb921: 'ccccbabV',
    0xb922: 'pppppaa02',
    0xb923: 'ppppbbbW',
    0xb924: 'uuuuuba02',
    0xb925: 'eeeebbaW',
    0xb926: 'vvvvvab12',
    0xb927: 'ffffaabV',
    0xb928: 'YYYYYYYbabM',
    0xb929: 'oooobaaW',
    0xb92a: 'YYYYYYYbabm',
    0xb92b: 'llllabbV',
    0xb92c: 'yyyyyba12',
    0xb92d: 'iiiibbbV',
    0xb92e: 'zzzzzab02',
    0xb92f: 'jjjjaaaW',
    0xb930: 'kkkkkbb13',
    0xb931: 'kkkkaaaW',
    0xb932: 'hhhhhaa03',
    0xb933: 'hhhhbbbV',
    0xb934: 'mmmmmba03',
    0xb935: 'mmmmabbV',
    0xb936: 'nnnnnab13',
    0xb937: 'nnnnbaaW',
    0xb938: 'gggggbb03',
    0xb939: 'ggggaabV',
    0xb93a: 'dddddaa13',
    0xb93b: 'ddddbbaW',
    0xb93c: 'aaaaaba13',
    0xb93d: 'aaaaabaW',
    0xb93e: 'bbbbbab03',
    0xb93f: 'bbbbbabV',
    0xb940: 'cccccba13',
    0xb941: 'ccccaabV',
    0xb942: 'CCCCCCba06',
    0xb943: 'ppppabbW',
    0xb944: 'eeeeebb03',
    0xb945: 'eeeeabaW',
    0xb946: 'fffffaa13',
    0xb947: 'ffffbabV',
    0xb948: 'oooooba03',
    0xb949: 'ooooaaaW',
    0xb94a: 'lllllab13',
    0xb94b: 'llllbbbV',
    0xb94c: 'iiiiibb13',
    0xb94d: 'iiiiabbV',
    0xb94e: 'jjjjjaa03',
    0xb94f: 'jjjjbaaW',
    0xb950: 'XXXXXXab17',
    0xb951: 'kkkkbaaW',
    0xb952: 'xxxxxab02',
    0xb953: 'hhhhabbV',
    0xb954: '=====ba05',
    0xb955: 'mmmmbbbV',
    0xb956: '>>>>>ab15',
    0xb957: 'nnnnaaaW',
    0xb958: 'wwwwwba02',
    0xb959: 'ggggbabV',
    0xb95a: 'tttttab12',
    0xb95b: 'ddddabaW',
    0xb95c: 'qqqqqbb12',
    0xb95d: 'aaaabbaW',
    0xb95e: 'rrrrraa02',
    0xb95f: 'bbbbaabV',
    0xb960: 'ppppppab14',
    0xb961: 'ccccbbaW',
    0xb962: 'pppppbb12',
    0xb963: 'ppppbaaV',
    0xb964: 'uuuuuab12',
    0xb965: 'eeeebabV',
    0xb966: 'uuuuuubb14',
    0xb967: 'ffffabaW',
    0xb968: '9999999babI',
    0xb969: 'oooobbbV',
    0xb96a: '9999999babi',
    0xb96b: 'llllaaaW',
    0xb96c: 'yyyyyab02',
    0xb96d: 'iiiibaaW',
    0xb96e: 'yyyyyybb04',
    0xb96f: 'jjjjabbV',
    0xb970: 'hhhhhhab15',
    0xb971: 'kkkkabbV',
    0xb972: 'hhhhhbb13',
    0xb973: 'hhhhbaaW',
    0xb974: 'mmmmmab13',
    0xb975: 'mmmmaaaW',
    0xb976: 'mmmmmmbb15',
    0xb977: 'nnnnbbbV',
    0xb978: 'ddddddab05',
    0xb979: 'ggggabaW',
    0xb97a: 'dddddbb03',
    0xb97b: 'ddddbabV',
    0xb97c: 'aaaaaab03',
    0xb97d: 'aaaaaabV',
    0xb97e: 'aaaaaabb05',
    0xb97f: 'bbbbbbaW',
    0xb980: 'cccccaa13',
    0xb981: 'ccccabbV',
    0xb982: 'ccccccbb05',
    0xb983: 'ppppaabW',
    0xb984: 'eeeeeab03',
    0xb985: 'eeeeaaaW',
    0xb986: 'eeeeeeba15',
    0xb987: 'ffffbbbV',
    0xb988: 'llllllaa05',
    0xb989: 'ooooabaW',
    0xb98a: 'lllllbb13',
    0xb98b: 'llllbabV',
    0xb98c: 'iiiiiab13',
    0xb98d: 'iiiiaabV',
    0xb98e: 'iiiiiiba05',
    0xb98f: 'jjjjbbaW',
    0xb990: 'xxxxxxaa14',
    0xb991: 'kkkkbbaW',
    0xb992: 'xxxxxbb02',
    0xb993: 'hhhhaabV',
    0xb994: '11111111aabH',
    0xb995: 'mmmmbabV',
    0xb996: '11111111aabX',
    0xb997: 'nnnnabaW',
    0xb998: 'ttttttaa04',
    0xb999: 'ggggbbbV',
    0xb99a: 'tttttbb12',
    0xb99b: 'ddddaaaW',
    0xb99c: 'qqqqqab12',
    0xb99d: 'aaaabaaW',
    0xb99e: 'qqqqqqba04',
    0xb99f: 'bbbbabbV',
    0xb9a0: 'sssssba02',
    0xb9a1: 'ccccbaaW',
    0xb9a2: 'pppppab12',
    0xb9a3: 'ppppbbaV',
    0xb9a4: 'uuuuubb12',
    0xb9a5: 'eeeebbbV',
    0xb9a6: 'vvvvvaa02',
    0xb9a7: 'ffffaaaW',
    0xb9a8: '1111111111bbdC',
    0xb9a9: 'oooobabV',
    0xb9aa: '1111111111bbbD',
    0xb9ab: 'llllabaW',
    0xb9ac: 'yyyyybb02',
    0xb9ad: 'iiiibbaW',
    0xb9ae: 'zzzzzaa12',
    0xb9af: 'jjjjaabV',
    0xb9b0: 'kkkkkba03',
    0xb9b1: 'kkkkaabV',
    0xb9b2: 'hhhhhab13',
    0xb9b3: 'hhhhbbaW',
    0xb9b4: 'mmmmmbb13',
    0xb9b5: 'mmmmabaW',
    0xb9b6: 'nnnnnaa03',
    0xb9b7: 'nnnnbabV',
    0xb9b8: 'gggggba13',
    0xb9b9: 'ggggaaaW',
    0xb9ba: 'dddddab03',
    0xb9bb: 'ddddbbbV',
    0xb9bc: 'aaaaabb03',
    0xb9bd: 'aaaaabbV',
    0xb9be: 'bbbbbaa13',
    0xb9bf: 'bbbbbaaW',
    0xb9c0: 'aaaaaaaaaabbbV',
    0xb9c1: 'aaaaaaaaaabbbT',
    0xb9c2: 'aaaaaaaaaabbbR',
    0xb9c3: 'aaaaaaaaaabbbP',
    0xb9c4: 'aaaaaaaaaabbjZ',
    0xb9c5: 'aaaaaaaaaabbhY',
    0xb9c6: 'aaaaaaaaaabbbZ',
    0xb9c7: 'aaaaaaaaaabbbX',
    0xb9c8: 'aaaaaaaaaabbbF',
    0xb9c9: 'aaaaaaaaaabbbD',
    0xb9ca: 'aaaaaaaaaabbbB',
    0xb9cb: 'aaaaaaaaaabbdC',
    0xb9cc: 'aaaaaaaaaabbbN',
    0xb9cd: 'aaaaaaaaaabbbL',
    0xb9ce: 'aaaaaaaaaabbbJ',
    0xb9cf: 'aaaaaaaaaabbbH',
    0xb9d0: 'aaaaaaaaaabbbv',
    0xb9d1: 'aaaaaaaaaabbbt',
    0xb9d2: 'aaaaaaaaaabbbr',
    0xb9d3: 'aaaaaaaaaabbbp',
    0xb9d4: 'aaaaaaaaaabbjz',
    0xb9d5: 'aaaaaaaaaabbhy',
    0xb9d6: 'aaaaaaaaaabbbz',
    0xb9d7: 'aaaaaaaaaabbbx',
    0xb9d8: 'aaaaaaaaaabbbf',
    0xb9d9: 'aaaaaaaaaabbbd',
    0xb9da: 'aaaaaaaaaabbbb',
    0xb9db: 'aaaaaaaaaabbdc',
    0xb9dc: 'aaaaaaaaaabbbn',
    0xb9dd: 'aaaabbbV',
    0xb9de: 'aaaaaaaaaabbbj',
    0xb9df: 'aaaaaaaaaabbbh',
    0xb9e0: 'aaaaaaaaaabbB',
    0xb9e1: 'aaaaaaaaaabbF',
    0xb9e2: 'aaaaaaaaaabbJ',
    0xb9e3: 'aaaaaaaaaabbN',
    0xb9e4: 'aaaaaaaaaabbR',
    0xb9e5: 'aaaaaaaaaabbV',
    0xb9e6: 'aaaaaaaaaabbZ',
    0xb9e7: 'aaaaaaaaaabjZ',
    0xb9e8: 'aaaaaaaaaabbb',
    0xb9e9: 'aaaaaaaaaabbf',
    0xb9ea: 'aaaaaaaaaabbj',
    0xb9eb: 'aaaaaaaaaabbn',
    0xb9ec: 'aaaaaaaaaabbr',
    0xb9ed: 'aaaaaaaaaabbv',
    0xb9ee: 'aaaaaaaaaabbz',
    0xb9ef: 'aaaaaaaaaabjz',
    0xb9f0: 'aaaaaaaaaabn',
    0xb9f1: 'aaaaaaaaaabf',
    0xb9f2: 'aaaaaaaaaajz',
    0xb9f3: 'aaaaaaaaaabv',
    0xb9f4: 'aaaaaaaaaabN',
    0xb9f5: 'aaaaaaaaaabF',
    0xb9f6: 'aaaaaaaaaajZ',
    0xb9f7: 'aaaaaaaaaabV',
    0xb9f8: 'aaaaaaaaaaN',
    0xb9f9: 'aaaaaaaaaiZ',
    0xb9fa: 'aaaaaaaaaan',
    0xb9fb: 'aaaaaaaaaiz',
    0xb9fc: 'aaaaaaa13',
    0xb9fd: 'aaaaaaaW',
    0xb9fe: 'aaaaaaaa15',
    0xb9ff: 'aaaaaaaai9',
    0xba00: 'cccccab12',
    0xba01: 'ccccabaT',
    0xba02: 'ccccccab15',
    0xba03: 'cccccccci9',
    0xba04: 'cccccccabbO',
    0xba05: 'cccccccabhZ',
    0xba06: 'cccccccabbo',
    0xba07: 'cccccccabhz',
    0xba08: 'ccccccccaabN',
    0xba09: 'ccccccccaabF',
    0xba0a: 'ccccccccaajZ',
    0xba0b: 'ccccccccaabV',
    0xba0c: 'ccccccccaabn',
    0xba0d: 'ccccccccaabf',
    0xba0e: 'ccccccccaajz',
    0xba0f: 'ccccccccaabv',
    0xba10: 'cccccccccbaas',
    0xba11: 'cccccccccbaaw',
    0xba12: 'cccccccccbacz',
    0xba13: 'cccccccccbakz',
    0xba14: 'cccccccccbaac',
    0xba15: 'cccccccccbaag',
    0xba16: 'cccccccccbaak',
    0xba17: 'cccccccccbaao',
    0xba18: 'cccccccccbaaS',
    0xba19: 'cccccccccbaaW',
    0xba1a: 'cccccccccbacZ',
    0xba1b: 'cccccccccbakZ',
    0xba1c: 'cccccccccbaaC',
    0xba1d: 'aaaababU',
    0xba1e: 'cccccccccbaaK',
    0xba1f: 'bbbbabaT',
    0xba20: 'sssssbb03',
    0xba21: 'ccccbabU',
    0xba22: 'pppppaa13',
    0xba23: 'ppppbbbT',
    0xba24: 'uuuuuba13',
    0xba25: 'eeeebbaT',
    0xba26: 'vvvvvab03',
    0xba27: 'ffffaabU',
    0xba28: 'AAAAAAAAAbbca',
    0xba29: 'oooobaaT',
    0xba2a: 'AAAAAAAAAbbah',
    0xba2b: 'llllabbU',
    0xba2c: 'yyyyyba03',
    0xba2d: 'iiiibbbU',
    0xba2e: 'zzzzzab13',
    0xba2f: 'jjjjaaaT',
    0xba30: 'kkkkkbb02',
    0xba31: 'kkkkaaaT',
    0xba32: 'hhhhhaa12',
    0xba33: 'hhhhbbbU',
    0xba34: 'mmmmmba12',
    0xba35: 'mmmmabbU',
    0xba36: 'nnnnnab02',
    0xba37: 'nnnnbaaT',
    0xba38: 'gggggbb12',
    0xba39: 'ggggaabU',
    0xba3a: 'dddddaa02',
    0xba3b: 'ddddbbaT',
    0xba3c: 'aaaaaba02',
    0xba3d: 'aaaaabaT',
    0xba3e: 'bbbbbab12',
    0xba3f: 'bbbbbabU',
    0xba40: 'cccccba02',
    0xba41: 'ccccaabU',
    0xba42: 'CCCCCCbb06',
    0xba43: 'ppppabbT',
    0xba44: 'eeeeebb12',
    0xba45: 'eeeeabaT',
    0xba46: 'fffffaa02',
    0xba47: 'ffffbabU',
    0xba48: 'oooooba12',
    0xba49: 'ooooaaaT',
    0xba4a: 'lllllab02',
    0xba4b: 'llllbbbU',
    0xba4c: 'iiiiibb02',
    0xba4d: 'iiiiabbU',
    0xba4e: 'jjjjjaa12',
    0xba4f: 'jjjjbaaT',
    0xba50: 'XXXXXXaa17',
    0xba51: 'kkkkbaaT',
    0xba52: 'xxxxxab13',
    0xba53: 'hhhhabbU',
    0xba54: '=====ba14',
    0xba55: 'mmmmbbbU',
    0xba56: '>>>>>ab04',
    0xba57: 'nnnnaaaT',
    0xba58: 'wwwwwba13',
    0xba59: 'ggggbabU',
    0xba5a: 'tttttab03',
    0xba5b: 'ddddabaT',
    0xba5c: 'qqqqqbb03',
    0xba5d: 'aaaabbaT',
    0xba5e: 'rrrrraa13',
    0xba5f: 'bbbbaabU',
    0xba60: 'ppppppaa14',
    0xba61: 'ccccbbaT',
    0xba62: 'pppppbb03',
    0xba63: 'ppppbaaU',
    0xba64: 'uuuuuab03',
    0xba65: 'eeeebabU',
    0xba66: 'uuuuuuba14',
    0xba67: 'ffffabaT',
    0xba68: '00000000bbaY',
    0xba69: 'oooobbbU',
    0xba6a: '00000000bbaI',
    0xba6b: 'llllaaaT',
    0xba6c: 'yyyyyab13',
    0xba6d: 'iiiibaaT',
    0xba6e: 'yyyyyyba04',
    0xba6f: 'jjjjabbU',
    0xba70: 'hhhhhhaa15',
    0xba71: 'kkkkabbU',
    0xba72: 'hhhhhbb02',
    0xba73: 'hhhhbaaT',
    0xba74: 'mmmmmab02',
    0xba75: 'mmmmaaaT',
    0xba76: 'mmmmmmba15',
    0xba77: 'nnnnbbbU',
    0xba78: 'ddddddaa05',
    0xba79: 'ggggabaT',
    0xba7a: 'dddddbb12',
    0xba7b: 'ddddbabU',
    0xba7c: 'aaaaaab12',
    0xba7d: 'aaaaaabU',
    0xba7e: 'aaaaaaba05',
    0xba7f: 'bbbbbbaT',
    0xba80: 'cccccaa02',
    0xba81: 'ccccabbU',
    0xba82: 'ccccccba05',
    0xba83: 'ppppaabT',
    0xba84: 'eeeeeab12',
    0xba85: 'eeeeaaaT',
    0xba86: 'eeeeeebb15',
    0xba87: 'ffffbbbU',
    0xba88: 'llllllab05',
    0xba89: 'ooooabaT',
    0xba8a: 'lllllbb02',
    0xba8b: 'llllbabU',
    0xba8c: 'iiiiiab02',
    0xba8d: 'iiiiaabU',
    0xba8e: 'iiiiiibb05',
    0xba8f: 'jjjjbbaT',
    0xba90: 'xxxxxxab14',
    0xba91: 'kkkkbbaT',
    0xba92: 'xxxxxbb13',
    0xba93: 'hhhhaabU',
    0xba94: '8888888aabi',
    0xba95: 'mmmmbabU',
    0xba96: '8888888aabI',
    0xba97: 'nnnnabaT',
    0xba98: 'ttttttab04',
    0xba99: 'ggggbbbU',
    0xba9a: 'tttttbb03',
    0xba9b: 'ddddaaaT',
    0xba9c: 'qqqqqab03',
    0xba9d: 'aaaabaaT',
    0xba9e: 'qqqqqqbb04',
    0xba9f: 'bbbbabbU',
    0xbaa0: 'sssssba13',
    0xbaa1: 'ccccbaaT',
    0xbaa2: 'pppppab03',
    0xbaa3: 'ppppbbaU',
    0xbaa4: 'uuuuubb03',
    0xbaa5: 'eeeebbbU',
    0xbaa6: 'vvvvvaa13',
    0xbaa7: 'ffffaaaT',
    0xbaa8: '?????bb04',
    0xbaa9: 'oooobabU',
    0xbaaa: '<<<<<aa14',
    0xbaab: 'llllabaT',
    0xbaac: 'yyyyybb13',
    0xbaad: 'iiiibbaT',
    0xbaae: 'zzzzzaa03',
    0xbaaf: 'jjjjaabU',
    0xbab0: 'kkkkkba12',
    0xbab1: 'kkkkaabU',
    0xbab2: 'hhhhhab02',
    0xbab3: 'hhhhbbaT',
    0xbab4: 'mmmmmbb02',
    0xbab5: 'mmmmabaT',
    0xbab6: 'nnnnnaa12',
    0xbab7: 'nnnnbabU',
    0xbab8: 'gggggba02',
    0xbab9: 'ggggaaaT',
    0xbaba: 'dddddab12',
    0xbabb: 'ddddbbbU',
    0xbabc: 'aaaaabb12',
    0xbabd: 'aaaaabbU',
    0xbabe: 'bbbbbaa02',
    0xbabf: 'bbbbbaaT',
    0xbac0: 'cccccbb12',
    0xbac1: 'ccccaaaT',
    0xbac2: 'CCCCCCaa16',
    0xbac3: 'ppppabaU',
    0xbac4: 'eeeeeba02',
    0xbac5: 'eeeeabbU',
    0xbac6: 'fffffab12',
    0xbac7: 'ffffbaaT',
    0xbac8: 'ooooobb02',
    0xbac9: 'ooooaabU',
    0xbaca: 'lllllaa12',
    0xbacb: 'llllbbaT',
    0xbacc: 'iiiiiba12',
    0xbacd: 'iiiiabaT',
    0xbace: 'jjjjjab02',
    0xbacf: 'jjjjbabU',
    0xbad0: 'XXXXXXbb07',
    0xbad1: 'kkkkbabU',
    0xbad2: 'xxxxxaa03',
    0xbad3: 'hhhhabaT',
    0xbad4: 'XXXXXXXaabm',
    0xbad5: 'mmmmbbaT',
    0xbad6: 'XXXXXXXaabM',
    0xbad7: 'nnnnaabU',
    0xbad8: 'wwwwwbb03',
    0xbad9: 'ggggbaaT',
    0xbada: 'tttttaa13',
    0xbadb: 'ddddabbU',
    0xbadc: 'qqqqqba13',
    0xbadd: 'aaaabbbU',
    0xbade: 'rrrrrab03',
    0xbadf: 'bbbbaaaT',
    0xbae0: 'ppppppbb04',
    0xbae1: 'ccccbbbU',
    0xbae2: 'pppppba13',
    0xbae3: 'ppppbabT',
    0xbae4: 'pppppppaabn',
    0xbae5: 'eeeebaaT',
    0xbae6: 'pppppppaabN',
    0xbae7: 'ffffabbU',
    0xbae8: 'yyyyyyybaaO',
    0xbae9: 'oooobbaT',
    0xbaea: 'yyyyyyybaao',
    0xbaeb: 'llllaabU',
    0xbaec: 'yyyyyaa03',
    0xbaed: 'iiiibabU',
    0xbaee: 'yyyyyyab14',
    0xbaef: 'jjjjabaT',
    0xbaf0: 'hhhhhhbb05',
    0xbaf1: 'kkkkabaT',
    0xbaf2: 'hhhhhba12',
    0xbaf3: 'hhhhbabU',
    0xbaf4: 'hhhhhhhaabo',
    0xbaf5: 'hhhhhhhaahz',
    0xbaf6: 'hhhhhhhaabO',
    0xbaf7: 'hhhhhhhaahZ',
    0xbaf8: 'aaaaaaabaaN',
    0xbaf9: 'aaaaaaabaiZ',
    0xbafa: 'aaaaaaabaan',
    0xbafb: 'aaaaaaabaiz',
    0xbafc: 'aaaaaaa02',
    0xbafd: 'aaaaaaaT',
    0xbafe: 'aaaaaaab15',
    0xbaff: 'bbbbbbbU',
    0xbb00: 'cccccab02',
    0xbb01: 'ccccabaU',
    0xbb02: 'ccccccaa05',
    0xbb03: 'eeeeeeebbhz',
    0xbb04: 'cccccccbaaN',
    0xbb05: 'cccccccbaiZ',
    0xbb06: 'cccccccbaan',
    0xbb07: 'cccccccbaiz',
    0xbb08: 'iiiiiiibbaN',
    0xbb09: 'iiiiiiibbiZ',
    0xbb0a: 'iiiiiiibban',
    0xbb0b: 'iiiiiiibbiz',
    0xbb0c: 'iiiiiaa02',
    0xbb0d: 'iiiiaaaU',
    0xbb0e: 'iiiiiiab05',
    0xbb0f: 'jjjjbbbT',
    0xbb10: 'xxxxxxbb14',
    0xbb11: 'kkkkbbbT',
    0xbb12: 'xxxxxba13',
    0xbb13: 'hhhhaaaU',
    0xbb14: 'xxxxxxxabbn',
    0xbb15: 'mmmmbaaU',
    0xbb16: 'xxxxxxxabbN',
    0xbb17: 'nnnnabbT',
    0xbb18: 'qqqqqqqbbaO',
    0xbb19: 'ggggbbaU',
    0xbb1a: 'qqqqqqqbbao',
    0xbb1b: 'ddddaabT',
    0xbb1c: 'qqqqqaa03',
    0xbb1d: 'aaaababT',
    0xbb1e: 'qqqqqqab04',
    0xbb1f: 'bbbbabaU',
    0xbb20: 'sssssbb13',
    0xbb21: 'ccccbabT',
    0xbb22: 'pppppaa03',
    0xbb23: 'ppppbbbU',
    0xbb24: 'uuuuuba03',
    0xbb25: 'eeeebbaU',
    0xbb26: 'vvvvvab13',
    0xbb27: 'ffffaabT',
    0xbb28: 'YYYYYYYbbaL',
    0xbb29: 'oooobaaU',
    0xbb2a: 'YYYYYYYbbal',
    0xbb2b: 'llllabbT',
    0xbb2c: 'yyyyyba13',
    0xbb2d: 'iiiibbbT',
    0xbb2e: 'zzzzzab03',
    0xbb2f: 'jjjjaaaU',
    0xbb30: 'kkkkkbb12',
    0xbb31: 'kkkkaaaU',
    0xbb32: 'hhhhhaa02',
    0xbb33: 'hhhhbbbT',
    0xbb34: 'mmmmmba02',
    0xbb35: 'mmmmabbT',
    0xbb36: 'nnnnnab12',
    0xbb37: 'nnnnbaaU',
    0xbb38: 'gggggbb02',
    0xbb39: 'ggggaabT',
    0xbb3a: 'dddddaa12',
    0xbb3b: 'ddddbbaU',
    0xbb3c: 'aaaaaba12',
    0xbb3d: 'aaaaabaU',
    0xbb3e: 'bbbbbab02',
    0xbb3f: 'bbbbbabT',
    0xbb40: 'cccccba12',
    0xbb41: 'ccccaabT',
    0xbb42: 'CCCCCCba16',
    0xbb43: 'ppppabbU',
    0xbb44: 'eeeeebb02',
    0xbb45: 'eeeeabaU',
    0xbb46: 'fffffaa12',
    0xbb47: 'ffffbabT',
    0xbb48: 'oooooba02',
    0xbb49: 'ooooaaaU',
    0xbb4a: 'lllllab12',
    0xbb4b: 'llllbbbT',
    0xbb4c: 'iiiiibb12',
    0xbb4d: 'iiiiabbT',
    0xbb4e: 'jjjjjaa02',
    0xbb4f: 'jjjjbaaU',
    0xbb50: 'XXXXXXab07',
    0xbb51: 'kkkkbaaU',
    0xbb52: 'xxxxxab03',
    0xbb53: 'hhhhabbT',
    0xbb54: '=====ba04',
    0xbb55: 'mmmmbbbT',
    0xbb56: '>>>>>ab14',
    0xbb57: 'nnnnaaaU',
    0xbb58: 'wwwwwba03',
    0xbb59: 'ggggbabT',
    0xbb5a: 'tttttab13',
    0xbb5b: 'ddddabaU',
    0xbb5c: 'qqqqqbb13',
    0xbb5d: 'aaaabbaU',
    0xbb5e: 'rrrrraa03',
    0xbb5f: 'bbbbaabT',
    0xbb60: 'ppppppab04',
    0xbb61: 'ccccbbaU',
    0xbb62: 'pppppbb13',
    0xbb63: 'ppppbaaT',
    0xbb64: 'uuuuuab13',
    0xbb65: 'eeeebabT',
    0xbb66: 'uuuuuubb04',
    0xbb67: 'ffffabaU',
    0xbb68: '9999999bbaH',
    0xbb69: 'oooobbbT',
    0xbb6a: '9999999bbah',
    0xbb6b: 'llllaaaU',
    0xbb6c: 'yyyyyab03',
    0xbb6d: 'iiiibaaU',
    0xbb6e: 'yyyyyybb14',
    0xbb6f: 'jjjjabbT',
    0xbb70: 'hhhhhhab05',
    0xbb71: 'kkkkabbT',
    0xbb72: 'hhhhhbb12',
    0xbb73: 'hhhhbaaU',
    0xbb74: 'mmmmmab12',
    0xbb75: 'mmmmaaaU',
    0xbb76: 'mmmmmmbb05',
    0xbb77: 'nnnnbbbT',
    0xbb78: 'ddddddab15',
    0xbb79: 'ggggabaU',
    0xbb7a: 'dddddbb02',
    0xbb7b: 'ddddbabT',
    0xbb7c: 'aaaaaab02',
    0xbb7d: 'aaaaaabT',
    0xbb7e: 'aaaaaabb15',
    0xbb7f: 'bbbbbbaU',
    0xbb80: 'cccccaa12',
    0xbb81: 'ccccabbT',
    0xbb82: 'ccccccbb15',
    0xbb83: 'ppppaabU',
    0xbb84: 'eeeeeab02',
    0xbb85: 'eeeeaaaU',
    0xbb86: 'eeeeeeba05',
    0xbb87: 'ffffbbbT',
    0xbb88: 'llllllaa15',
    0xbb89: 'ooooabaU',
    0xbb8a: 'lllllbb12',
    0xbb8b: 'llllbabT',
    0xbb8c: 'iiiiiab12',
    0xbb8d: 'iiiiaabT',
    0xbb8e: 'iiiiiiba15',
    0xbb8f: 'jjjjbbaU',
    0xbb90: 'xxxxxxaa04',
    0xbb91: 'kkkkbbaU',
    0xbb92: 'xxxxxbb03',
    0xbb93: 'hhhhaabT',
    0xbb94: '11111111bbaI',
    0xbb95: 'mmmmbabT',
    0xbb96: '11111111bbaY',
    0xbb97: 'nnnnabaU',
    0xbb98: 'ttttttaa14',
    0xbb99: 'ggggbbbT',
    0xbb9a: 'tttttbb13',
    0xbb9b: 'ddddaaaU',
    0xbb9c: 'qqqqqab13',
    0xbb9d: 'aaaabaaU',
    0xbb9e: 'qqqqqqba14',
    0xbb9f: 'bbbbabbT',
    0xbba0: 'sssssba03',
    0xbba1: 'ccccbaaU',
    0xbba2: 'pppppab13',
    0xbba3: 'ppppbbaT',
    0xbba4: 'uuuuubb13',
    0xbba5: 'eeeebbbT',
    0xbba6: 'vvvvvaa03',
    0xbba7: 'ffffaaaU',
    0xbba8: '?????bb14',
    0xbba9: 'oooobabT',
    0xbbaa: '<<<<<aa04',
    0xbbab: 'llllabaU',
    0xbbac: 'yyyyybb03',
    0xbbad: 'iiiibbaU',
    0xbbae: 'zzzzzaa13',
    0xbbaf: 'jjjjaabT',
    0xbbb0: 'kkkkkba02',
    0xbbb1: 'kkkkaabT',
    0xbbb2: 'hhhhhab12',
    0xbbb3: 'hhhhbbaU',
    0xbbb4: 'mmmmmbb12',
    0xbbb5: 'mmmmabaU',
    0xbbb6: 'nnnnnaa02',
    0xbbb7: 'nnnnbabT',
    0xbbb8: 'gggggba12',
    0xbbb9: 'ggggaaaU',
    0xbbba: 'dddddab02',
    0xbbbb: 'ddddbbbT',
    0xbbbc: 'aaaaabb02',
    0xbbbd: 'aaaaabbT',
    0xbbbe: 'bbbbbaa12',
    0xbbbf: 'bbbbbaaU',
    0xbbc0: 'cccccbb02',
    0xbbc1: 'ccccaaaU',
    0xbbc2: 'CCCCCCab06',
    0xbbc3: 'ppppabaT',
    0xbbc4: 'eeeeeba12',
    0xbbc5: 'eeeeabbT',
    0xbbc6: 'fffffab02',
    0xbbc7: 'ffffbaaU',
    0xbbc8: 'ooooobb12',
    0xbbc9: 'ooooaabT',
    0xbbca: 'lllllaa02',
    0xbbcb: 'llllbbaU',
    0xbbcc: 'iiiiiba02',
    0xbbcd: 'iiiiabaU',
    0xbbce: 'jjjjjab12',
    0xbbcf: 'jjjjbabT',
    0xbbd0: 'QQQQQQQQbbam',
    0xbbd1: 'kkkkbabT',
    0xbbd2: 'xxxxxaa13',
    0xbbd3: 'hhhhabaU',
    0xbbd4: 'QQQQQQQQbbaM',
    0xbbd5: 'mmmmbbaU',
    0xbbd6: 'QQQQQQQQbbiY',
    0xbbd7: 'nnnnaabT',
    0xbbd8: 'wwwwwbb13',
    0xbbd9: 'ggggbaaU',
    0xbbda: 'tttttaa03',
    0xbbdb: 'ddddabbT',
    0xbbdc: 'qqqqqba03',
    0xbbdd: 'aaaabbbT',
    0xbbde: 'rrrrrab13',
    0xbbdf: 'bbbbaaaU',
    0xbbe0: 'ppppppba14',
    0xbbe1: 'ccccbbbT',
    0xbbe2: 'pppppba03',
    0xbbe3: 'ppppbabU',
    0xbbe4: 'pppppppbbao',
    0xbbe5: 'eeeebaaU',
    0xbbe6: 'pppppppbbaO',
    0xbbe7: 'ffffabbT',
    0xbbe8: 'ppppppppabhZ',
    0xbbe9: 'oooobbaU',
    0xbbea: 'ppppppppabbO',
    0xbbeb: 'llllaabT',
    0xbbec: 'ppppppppabhz',
    0xbbed: 'iiiibabT',
    0xbbee: 'ppppppppabbo',
    0xbbef: 'jjjjabaU',
    0xbbf0: 'aaaaaaaabbao',
    0xbbf1: 'aaaaaaaabbag',
    0xbbf2: 'aaaaaaaabbkz',
    0xbbf3: 'aaaaaaaabbaw',
    0xbbf4: 'aaaaaaaabbaO',
    0xbbf5: 'aaaaaaaabbaG',
    0xbbf6: 'aaaaaaaabbkZ',
    0xbbf7: 'aaaaaaaabbaW',
    0xbbf8: 'aaaaaaaabbO',
    0xbbf9: 'aaaaaaaabhZ',
    0xbbfa: 'aaaaaaaabbo',
    0xbbfb: 'aaaaaaaabhz',
    0xbbfc: 'aaaaaaa12',
    0xbbfd: 'aaaaaaaU',
    0xbbfe: 'aaaaaaaa05',
    0xbbff: 'aaaaaaaah9',
    0xbc00: 'cccccab11',
    0xbc01: 'ccccabaR',
    0xbc02: 'ccccccab04',
    0xbc03: 'cccccccch8',
    0xbc04: 'cccccccaabO',
    0xbc05: 'cccccccaahZ',
    0xbc06: 'cccccccaabo',
    0xbc07: 'cccccccaahz',
    0xbc08: 'ccccccccbabN',
    0xbc09: 'ccccccccbabF',
    0xbc0a: 'ccccccccbajZ',
    0xbc0b: 'ccccccccbabV',
    0xbc0c: 'ccccccccbabn',
    0xbc0d: 'ccccccccbabf',
    0xbc0e: 'ccccccccbajz',
    0xbc0f: 'ccccccccbabv',
    0xbc10: 'qqqqqqqqbbbo',
    0xbc11: 'kkkkbbbS',
    0xbc12: 'qqqqqqqqbbhz',
    0xbc13: 'hhhhaaaR',
    0xbc14: 'qqqqqqqqbbbO',
    0xbc15: 'mmmmbaaR',
    0xbc16: 'qqqqqqqqbbhZ',
    0xbc17: 'nnnnabbS',
    0xbc18: 'qqqqqqqabbN',
    0xbc19: 'ggggbbaR',
    0xbc1a: 'qqqqqqqabbn',
    0xbc1b: 'ddddaabS',
    0xbc1c: 'qqqqqaa10',
    0xbc1d: 'aaaababS',
    0xbc1e: 'qqqqqqaa05',
    0xbc1f: 'bbbbabaR',
    0xbc20: 'sssssbb00',
    0xbc21: 'ccccbabS',
    0xbc22: 'pppppaa10',
    0xbc23: 'ppppbbbR',
    0xbc24: 'uuuuuba10',
    0xbc25: 'eeeebbaR',
    0xbc26: 'vvvvvab00',
    0xbc27: 'ffffaabS',
    0xbc28: 'PPPPPPPPabiY',
    0xbc29: 'oooobaaR',
    0xbc2a: 'PPPPPPPPabaM',
    0xbc2b: 'llllabbS',
    0xbc2c: 'yyyyyba00',
    0xbc2d: 'iiiibbbS',
    0xbc2e: 'zzzzzab10',
    0xbc2f: 'jjjjaaaR',
    0xbc30: 'kkkkkbb01',
    0xbc31: 'kkkkaaaR',
    0xbc32: 'hhhhhaa11',
    0xbc33: 'hhhhbbbS',
    0xbc34: 'mmmmmba11',
    0xbc35: 'mmmmabbS',
    0xbc36: 'nnnnnab01',
    0xbc37: 'nnnnbaaR',
    0xbc38: 'gggggbb11',
    0xbc39: 'ggggaabS',
    0xbc3a: 'dddddaa01',
    0xbc3b: 'ddddbbaR',
    0xbc3c: 'aaaaaba01',
    0xbc3d: 'aaaaabaR',
    0xbc3e: 'bbbbbab11',
    0xbc3f: 'bbbbbabS',
    0xbc40: 'cccccba01',
    0xbc41: 'ccccaabS',
    0xbc42: 'CCCCCCbb17',
    0xbc43: 'ppppabbR',
    0xbc44: 'eeeeebb11',
    0xbc45: 'eeeeabaR',
    0xbc46: 'fffffaa01',
    0xbc47: 'ffffbabS',
    0xbc48: 'oooooba11',
    0xbc49: 'ooooaaaR',
    0xbc4a: 'lllllab01',
    0xbc4b: 'llllbbbS',
    0xbc4c: 'iiiiibb01',
    0xbc4d: 'iiiiabbS',
    0xbc4e: 'jjjjjaa11',
    0xbc4f: 'jjjjbaaR',
    0xbc50: 'XXXXXXaa06',
    0xbc51: 'kkkkbaaR',
    0xbc52: 'xxxxxab10',
    0xbc53: 'hhhhabbS',
    0xbc54: '=====ba17',
    0xbc55: 'mmmmbbbS',
    0xbc56: '>>>>>ab07',
    0xbc57: 'nnnnaaaR',
    0xbc58: 'wwwwwba10',
    0xbc59: 'ggggbabS',
    0xbc5a: 'tttttab00',
    0xbc5b: 'ddddabaR',
    0xbc5c: 'qqqqqbb00',
    0xbc5d: 'aaaabbaR',
    0xbc5e: 'rrrrraa10',
    0xbc5f: 'bbbbaabS',
    0xbc60: 'ppppppaa05',
    0xbc61: 'ccccbbaR',
    0xbc62: 'pppppbb00',
    0xbc63: 'ppppbaaS',
    0xbc64: 'uuuuuab00',
    0xbc65: 'eeeebabS',
    0xbc66: 'uuuuuuba05',
    0xbc67: 'ffffabaR',
    0xbc68: '00000000abaY',
    0xbc69: 'oooobbbS',
    0xbc6a: '00000000abaI',
    0xbc6b: 'llllaaaR',
    0xbc6c: 'yyyyyab10',
    0xbc6d: 'iiiibaaR',
    0xbc6e: 'yyyyyyba15',
    0xbc6f: 'jjjjabbS',
    0xbc70: 'hhhhhhaa04',
    0xbc71: 'kkkkabbS',
    0xbc72: 'hhhhhbb01',
    0xbc73: 'hhhhbaaR',
    0xbc74: 'mmmmmab01',
    0xbc75: 'mmmmaaaR',
    0xbc76: 'mmmmmmba04',
    0xbc77: 'nnnnbbbS',
    0xbc78: 'ddddddaa14',
    0xbc79: 'ggggabaR',
    0xbc7a: 'dddddbb11',
    0xbc7b: 'ddddbabS',
    0xbc7c: 'aaaaaab11',
    0xbc7d: 'aaaaaabS',
    0xbc7e: 'aaaaaaba14',
    0xbc7f: 'bbbbbbaR',
    0xbc80: 'cccccaa01',
    0xbc81: 'ccccabbS',
    0xbc82: 'ccccccba14',
    0xbc83: 'ppppaabR',
    0xbc84: 'eeeeeab11',
    0xbc85: 'eeeeaaaR',
    0xbc86: 'eeeeeebb04',
    0xbc87: 'ffffbbbS',
    0xbc88: 'llllllab14',
    0xbc89: 'ooooabaR',
    0xbc8a: 'lllllbb01',
    0xbc8b: 'llllbabS',
    0xbc8c: 'iiiiiab01',
    0xbc8d: 'iiiiaabS',
    0xbc8e: 'iiiiiibb14',
    0xbc8f: 'jjjjbbaR',
    0xbc90: 'xxxxxxab05',
    0xbc91: 'kkkkbbaR',
    0xbc92: 'xxxxxbb10',
    0xbc93: 'hhhhaabS',
    0xbc94: '8888888abbi',
    0xbc95: 'mmmmbabS',
    0xbc96: '8888888abbI',
    0xbc97: 'nnnnabaR',
    0xbc98: 'ttttttab15',
    0xbc99: 'ggggbbbS',
    0xbc9a: 'tttttbb00',
    0xbc9b: 'ddddaaaR',
    0xbc9c: 'qqqqqab00',
    0xbc9d: 'aaaabaaR',
    0xbc9e: 'qqqqqqbb15',
    0xbc9f: 'bbbbabbS',
    0xbca0: 'sssssba10',
    0xbca1: 'ccccbaaR',
    0xbca2: 'pppppab00',
    0xbca3: 'ppppbbaS',
    0xbca4: 'uuuuubb00',
    0xbca5: 'eeeebbbS',
    0xbca6: 'vvvvvaa10',
    0xbca7: 'ffffaaaR',
    0xbca8: '?????bb07',
    0xbca9: 'oooobabS',
    0xbcaa: '<<<<<aa17',
    0xbcab: 'llllabaR',
    0xbcac: 'yyyyybb10',
    0xbcad: 'iiiibbaR',
    0xbcae: 'zzzzzaa00',
    0xbcaf: 'jjjjaabS',
    0xbcb0: 'kkkkkba11',
    0xbcb1: 'kkkkaabS',
    0xbcb2: 'hhhhhab01',
    0xbcb3: 'hhhhbbaR',
    0xbcb4: 'mmmmmbb01',
    0xbcb5: 'mmmmabaR',
    0xbcb6: 'nnnnnaa11',
    0xbcb7: 'nnnnbabS',
    0xbcb8: 'gggggba01',
    0xbcb9: 'ggggaaaR',
    0xbcba: 'dddddab11',
    0xbcbb: 'ddddbbbS',
    0xbcbc: 'aaaaabb11',
    0xbcbd: 'aaaaabbS',
    0xbcbe: 'bbbbbaa01',
    0xbcbf: 'bbbbbaaR',
    0xbcc0: 'cccccbb11',
    0xbcc1: 'ccccaaaR',
    0xbcc2: 'CCCCCCaa07',
    0xbcc3: 'ppppabaS',
    0xbcc4: 'eeeeeba01',
    0xbcc5: 'eeeeabbS',
    0xbcc6: 'fffffab11',
    0xbcc7: 'ffffbaaR',
    0xbcc8: 'ooooobb01',
    0xbcc9: 'ooooaabS',
    0xbcca: 'lllllaa11',
    0xbccb: 'llllbbaR',
    0xbccc: 'iiiiiba11',
    0xbccd: 'iiiiabaR',
    0xbcce: 'jjjjjab01',
    0xbccf: 'jjjjbabS',
    0xbcd0: 'XXXXXXbb16',
    0xbcd1: 'kkkkbabS',
    0xbcd2: 'xxxxxaa00',
    0xbcd3: 'hhhhabaR',
    0xbcd4: 'XXXXXXXabbm',
    0xbcd5: 'mmmmbbaR',
    0xbcd6: 'XXXXXXXabbM',
    0xbcd7: 'nnnnaabS',
    0xbcd8: 'wwwwwbb00',
    0xbcd9: 'ggggbaaR',
    0xbcda: 'tttttaa10',
    0xbcdb: 'ddddabbS',
    0xbcdc: 'qqqqqba10',
    0xbcdd: 'aaaabbbS',
    0xbcde: 'rrrrrab00',
    0xbcdf: 'bbbbaaaR',
    0xbce0: 'ppppppbb15',
    0xbce1: 'ccccbbbS',
    0xbce2: 'pppppba10',
    0xbce3: 'ppppbabR',
    0xbce4: 'pppppppabbn',
    0xbce5: 'eeeebaaR',
    0xbce6: 'pppppppabbN',
    0xbce7: 'ffffabbS',
    0xbce8: 'yyyyyyybbaO',
    0xbce9: 'oooobbaR',
    0xbcea: 'yyyyyyybbao',
    0xbceb: 'llllaabS',
    0xbcec: 'yyyyyaa00',
    0xbced: 'iiiibabS',
    0xbcee: 'yyyyyyab05',
    0xbcef: 'jjjjabaR',
    0xbcf0: 'hhhhhhbb14',
    0xbcf1: 'kkkkabaR',
    0xbcf2: 'hhhhhba11',
    0xbcf3: 'hhhhbabS',
    0xbcf4: 'hhhhhhhabbo',
    0xbcf5: 'hhhhhhhabhz',
    0xbcf6: 'hhhhhhhabbO',
    0xbcf7: 'hhhhhhhabhZ',
    0xbcf8: 'aaaaaaabbaN',
    0xbcf9: 'aaaaaaabbiZ',
    0xbcfa: 'aaaaaaabban',
    0xbcfb: 'aaaaaaabbiz',
    0xbcfc: 'aaaaaaa01',
    0xbcfd: 'aaaaaaaR',
    0xbcfe: 'aaaaaaab04',
    0xbcff: 'bbbbbbbS',
    0xbd00: 'cccccab01',
    0xbd01: 'ccccabaS',
    0xbd02: 'ccccccaa14',
    0xbd03: 'eeeeeeebahz',
    0xbd04: 'cccccccbbaN',
    0xbd05: 'cccccccbbiZ',
    0xbd06: 'cccccccbban',
    0xbd07: 'cccccccbbiz',
    0xbd08: 'iiiiiiibaaN',
    0xbd09: 'iiiiiiibaiZ',
    0xbd0a: 'iiiiiiibaan',
    0xbd0b: 'iiiiiiibaiz',
    0xbd0c: 'iiiiiaa01',
    0xbd0d: 'iiiiaaaS',
    0xbd0e: 'iiiiiiab14',
    0xbd0f: 'jjjjbbbR',
    0xbd10: 'xxxxxxbb05',
    0xbd11: 'kkkkbbbR',
    0xbd12: 'xxxxxba10',
    0xbd13: 'hhhhaaaS',
    0xbd14: 'xxxxxxxaabn',
    0xbd15: 'mmmmbaaS',
    0xbd16: 'xxxxxxxaabN',
    0xbd17: 'nnnnabbR',
    0xbd18: 'qqqqqqqbaaO',
    0xbd19: 'ggggbbaS',
    0xbd1a: 'qqqqqqqbaao',
    0xbd1b: 'ddddaabR',
    0xbd1c: 'qqqqqaa00',
    0xbd1d: 'aaaababR',
    0xbd1e: 'qqqqqqab15',
    0xbd1f: 'bbbbabaS',
    0xbd20: 'sssssbb10',
    0xbd21: 'ccccbabR',
    0xbd22: 'pppppaa00',
    0xbd23: 'ppppbbbS',
    0xbd24: 'uuuuuba00',
    0xbd25: 'eeeebbaS',
    0xbd26: 'vvvvvab10',
    0xbd27: 'ffffaabR',
    0xbd28: 'YYYYYYYbaaL',
    0xbd29: 'oooobaaS',
    0xbd2a: 'YYYYYYYbaal',
    0xbd2b: 'llllabbR',
    0xbd2c: 'yyyyyba10',
    0xbd2d: 'iiiibbbR',
    0xbd2e: 'zzzzzab00',
    0xbd2f: 'jjjjaaaS',
    0xbd30: 'kkkkkbb11',
    0xbd31: 'kkkkaaaS',
    0xbd32: 'hhhhhaa01',
    0xbd33: 'hhhhbbbR',
    0xbd34: 'mmmmmba01',
    0xbd35: 'mmmmabbR',
    0xbd36: 'nnnnnab11',
    0xbd37: 'nnnnbaaS',
    0xbd38: 'gggggbb01',
    0xbd39: 'ggggaabR',
    0xbd3a: 'dddddaa11',
    0xbd3b: 'ddddbbaS',
    0xbd3c: 'aaaaaba11',
    0xbd3d: 'aaaaabaS',
    0xbd3e: 'bbbbbab01',
    0xbd3f: 'bbbbbabR',
    0xbd40: 'cccccba11',
    0xbd41: 'ccccaabR',
    0xbd42: 'CCCCCCba07',
    0xbd43: 'ppppabbS',
    0xbd44: 'eeeeebb01',
    0xbd45: 'eeeeabaS',
    0xbd46: 'fffffaa11',
    0xbd47: 'ffffbabR',
    0xbd48: 'oooooba01',
    0xbd49: 'ooooaaaS',
    0xbd4a: 'lllllab11',
    0xbd4b: 'llllbbbR',
    0xbd4c: 'iiiiibb11',
    0xbd4d: 'iiiiabbR',
    0xbd4e: 'jjjjjaa01',
    0xbd4f: 'jjjjbaaS',
    0xbd50: 'XXXXXXab16',
    0xbd51: 'kkkkbaaS',
    0xbd52: 'xxxxxab00',
    0xbd53: 'hhhhabbR',
    0xbd54: '=====ba07',
    0xbd55: 'mmmmbbbR',
    0xbd56: '>>>>>ab17',
    0xbd57: 'nnnnaaaS',
    0xbd58: 'wwwwwba00',
    0xbd59: 'ggggbabR',
    0xbd5a: 'tttttab10',
    0xbd5b: 'ddddabaS',
    0xbd5c: 'qqqqqbb10',
    0xbd5d: 'aaaabbaS',
    0xbd5e: 'rrrrraa00',
    0xbd5f: 'bbbbaabR',
    0xbd60: 'ppppppab15',
    0xbd61: 'ccccbbaS',
    0xbd62: 'pppppbb10',
    0xbd63: 'ppppbaaR',
    0xbd64: 'uuuuuab10',
    0xbd65: 'eeeebabR',
    0xbd66: 'uuuuuubb15',
    0xbd67: 'ffffabaS',
    0xbd68: '9999999baaH',
    0xbd69: 'oooobbbR',
    0xbd6a: '9999999baah',
    0xbd6b: 'llllaaaS',
    0xbd6c: 'yyyyyab00',
    0xbd6d: 'iiiibaaS',
    0xbd6e: 'yyyyyybb05',
    0xbd6f: 'jjjjabbR',
    0xbd70: 'hhhhhhab14',
    0xbd71: 'kkkkabbR',
    0xbd72: 'hhhhhbb11',
    0xbd73: 'hhhhbaaS',
    0xbd74: 'mmmmmab11',
    0xbd75: 'mmmmaaaS',
    0xbd76: 'mmmmmmbb14',
    0xbd77: 'nnnnbbbR',
    0xbd78: 'ddddddab04',
    0xbd79: 'ggggabaS',
    0xbd7a: 'dddddbb01',
    0xbd7b: 'ddddbabR',
    0xbd7c: 'aaaaaab01',
    0xbd7d: 'aaaaaabR',
    0xbd7e: 'aaaaaabb04',
    0xbd7f: 'bbbbbbaS',
    0xbd80: 'cccccaa11',
    0xbd81: 'ccccabbR',
    0xbd82: 'ccccccbb04',
    0xbd83: 'ppppaabS',
    0xbd84: 'eeeeeab01',
    0xbd85: 'eeeeaaaS',
    0xbd86: 'eeeeeeba14',
    0xbd87: 'ffffbbbR',
    0xbd88: 'llllllaa04',
    0xbd89: 'ooooabaS',
    0xbd8a: 'lllllbb11',
    0xbd8b: 'llllbabR',
    0xbd8c: 'iiiiiab11',
    0xbd8d: 'iiiiaabR',
    0xbd8e: 'iiiiiiba04',
    0xbd8f: 'jjjjbbaS',
    0xbd90: 'xxxxxxaa15',
    0xbd91: 'kkkkbbaS',
    0xbd92: 'xxxxxbb00',
    0xbd93: 'hhhhaabR',
    0xbd94: '11111111abaI',
    0xbd95: 'mmmmbabR',
    0xbd96: '11111111abaY',
    0xbd97: 'nnnnabaS',
    0xbd98: 'ttttttaa05',
    0xbd99: 'ggggbbbR',
    0xbd9a: 'tttttbb10',
    0xbd9b: 'ddddaaaS',
    0xbd9c: 'qqqqqab10',
    0xbd9d: 'aaaabaaS',
    0xbd9e: 'qqqqqqba05',
    0xbd9f: 'bbbbabbR',
    0xbda0: 'sssssba00',
    0xbda1: 'ccccbaaS',
    0xbda2: 'pppppab10',
    0xbda3: 'ppppbbaR',
    0xbda4: 'uuuuubb10',
    0xbda5: 'eeeebbbR',
    0xbda6: 'vvvvvaa00',
    0xbda7: 'ffffaaaS',
    0xbda8: '?????bb17',
    0xbda9: 'oooobabR',
    0xbdaa: '<<<<<aa07',
    0xbdab: 'llllabaS',
    0xbdac: 'yyyyybb00',
    0xbdad: 'iiiibbaS',
    0xbdae: 'zzzzzaa10',
    0xbdaf: 'jjjjaabR',
    0xbdb0: 'kkkkkba01',
    0xbdb1: 'kkkkaabR',
    0xbdb2: 'hhhhhab11',
    0xbdb3: 'hhhhbbaS',
    0xbdb4: 'mmmmmbb11',
    0xbdb5: 'mmmmabaS',
    0xbdb6: 'nnnnnaa01',
    0xbdb7: 'nnnnbabR',
    0xbdb8: 'gggggba11',
    0xbdb9: 'ggggaaaS',
    0xbdba: 'dddddab01',
    0xbdbb: 'ddddbbbR',
    0xbdbc: 'aaaaabb01',
    0xbdbd: 'aaaaabbR',
    0xbdbe: 'bbbbbaa11',
    0xbdbf: 'bbbbbaaS',
    0xbdc0: 'cccccbb01',
    0xbdc1: 'ccccaaaS',
    0xbdc2: 'CCCCCCab17',
    0xbdc3: 'ppppabaR',
    0xbdc4: 'eeeeeba11',
    0xbdc5: 'eeeeabbR',
    0xbdc6: 'fffffab01',
    0xbdc7: 'ffffbaaS',
    0xbdc8: 'ooooobb11',
    0xbdc9: 'ooooaabR',
    0xbdca: 'lllllaa01',
    0xbdcb: 'llllbbaS',
    0xbdcc: 'iiiiiba01',
    0xbdcd: 'iiiiabaS',
    0xbdce: 'jjjjjab11',
    0xbdcf: 'jjjjbabR',
    0xbdd0: 'CCCCCCCCCbbap',
    0xbdd1: 'kkkkbabR',
    0xbdd2: 'xxxxxaa10',
    0xbdd3: 'hhhhabaS',
    0xbdd4: 'CCCCCCCCCbbca',
    0xbdd5: 'mmmmbbaS',
    0xbdd6: 'CCCCCCCCCbbah',
    0xbdd7: 'nnnnaabR',
    0xbdd8: 'wwwwwbb10',
    0xbdd9: 'ggggbaaS',
    0xbdda: 'tttttaa00',
    0xbddb: 'ddddabbR',
    0xbddc: 'qqqqqba00',
    0xbddd: 'aaaabbbR',
    0xbdde: 'rrrrrab10',
    0xbddf: 'bbbbaaaS',
    0xbde0: 'aaaaaaaaabaaC',
    0xbde1: 'aaaaaaaaabaaG',
    0xbde2: 'aaaaaaaaabaaK',
    0xbde3: 'aaaaaaaaabaaO',
    0xbde4: 'aaaaaaaaabaaS',
    0xbde5: 'aaaaaaaaabaaW',
    0xbde6: 'aaaaaaaaabacZ',
    0xbde7: 'aaaaaaaaabakZ',
    0xbde8: 'aaaaaaaaabaac',
    0xbde9: 'aaaaaaaaabaag',
    0xbdea: 'aaaaaaaaabaak',
    0xbdeb: 'aaaaaaaaabaao',
    0xbdec: 'aaaaaaaaabaas',
    0xbded: 'aaaaaaaaabaaw',
    0xbdee: 'aaaaaaaaabacz',
    0xbdef: 'aaaaaaaaabakz',
    0xbdf0: 'aaaaaaaaabao',
    0xbdf1: 'aaaaaaaaabag',
    0xbdf2: 'aaaaaaaaabkz',
    0xbdf3: 'aaaaaaaaabaw',
    0xbdf4: 'aaaaaaaaabaO',
    0xbdf5: 'aaaaaaaaabaG',
    0xbdf6: 'aaaaaaaaabkZ',
    0xbdf7: 'aaaaaaaaabaW',
    0xbdf8: 'aaaaaaaaabO',
    0xbdf9: 'aaaaaaaaahZ',
    0xbdfa: 'aaaaaaaaabo',
    0xbdfb: 'aaaaaaaaahz',
    0xbdfc: 'aaaaaaa11',
    0xbdfd: 'aaaaaaaS',
    0xbdfe: 'aaaaaaaa14',
    0xbdff: 'aaaaaaaai8',
    0xbe00: 'cccccab10',
    0xbe01: 'ccccabaP',
    0xbe02: 'ccccccab14',
    0xbe03: 'cccccccci8',
    0xbe04: 'cccccccabaN',
    0xbe05: 'cccccccabiZ',
    0xbe06: 'cccccccaban',
    0xbe07: 'cccccccabiz',
    0xbe08: 'ccccccccabaO',
    0xbe09: 'ccccccccabaG',
    0xbe0a: 'ccccccccabkZ',
    0xbe0b: 'ccccccccabaW',
    0xbe0c: 'ccccccccabao',
    0xbe0d: 'ccccccccabag',
    0xbe0e: 'ccccccccabkz',
    0xbe0f: 'ccccccccabaw',
    0xbe10: 'cccccccccabbr',
    0xbe11: 'cccccccccabbv',
    0xbe12: 'cccccccccabbz',
    0xbe13: 'cccccccccabjz',
    0xbe14: 'cccccccccabbb',
    0xbe15: 'cccccccccabbf',
    0xbe16: 'cccccccccabbj',
    0xbe17: 'cccccccccabbn',
    0xbe18: 'cccccccccabbR',
    0xbe19: 'cccccccccabbV',
    0xbe1a: 'cccccccccabbZ',
    0xbe1b: 'cccccccccabjZ',
    0xbe1c: 'cccccccccabbB',
    0xbe1d: 'aaaababQ',
    0xbe1e: 'cccccccccabbJ',
    0xbe1f: 'bbbbabaP',
    0xbe20: 'ccccccccccaaao',
    0xbe21: 'ccccbabQ',
    0xbe22: 'ccccccccccaaak',
    0xbe23: 'ccccccccccaaai',
    0xbe24: 'ccccccccccaaag',
    0xbe25: 'ccccccccccaaae',
    0xbe26: 'ccccccccccaaac',
    0xbe27: 'ccccccccccaaaa',
    0xbe28: 'ccccccccccaakz',
    0xbe29: 'ccccccccccaaiy',
    0xbe2a: 'ccccccccccaacz',
    0xbe2b: 'ccccccccccaaay',
    0xbe2c: 'ccccccccccaaaw',
    0xbe2d: 'ccccccccccaaau',
    0xbe2e: 'ccccccccccaaas',
    0xbe2f: 'ccccccccccaaaq',
    0xbe30: 'ccccccccccaaaO',
    0xbe31: 'ccccccccccaaaM',
    0xbe32: 'ccccccccccaaaK',
    0xbe33: 'ccccccccccaaaI',
    0xbe34: 'ccccccccccaaaG',
    0xbe35: 'ccccccccccaaaE',
    0xbe36: 'ccccccccccaaaC',
    0xbe37: 'ccccccccccaaaA',
    0xbe38: 'ccccccccccaakZ',
    0xbe39: 'ccccccccccaaiY',
    0xbe3a: 'ccccccccccaacZ',
    0xbe3b: 'ccccccccccaaaY',
    0xbe3c: 'aaaaaba00',
    0xbe3d: 'aaaaabaP',
    0xbe3e: 'bbbbbab10',
    0xbe3f: 'bbbbbabQ',
    0xbe40: 'cccccba00',
    0xbe41: 'ccccaabQ',
    0xbe42: 'cccccccccccbbbe',
    0xbe43: 'cccccccccccbbbd',
    0xbe44: 'cccccccccccbbbc',
    0xbe45: 'cccccccccccbbbb',
    0xbe46: 'cccccccccccbbba',
    0xbe47: 'cccccccccccbbdc',
    0xbe48: 'cccccccccccbbbo',
    0xbe49: 'cccccccccccbbbn',
    0xbe4a: 'cccccccccccbbbm',
    0xbe4b: 'cccccccccccbbbl',
    0xbe4c: 'cccccccccccbbbk',
    0xbe4d: 'cccccccccccbbbj',
    0xbe4e: 'cccccccccccbbbi',
    0xbe4f: 'cccccccccccbbbh',
    0xbe50: 'cccccccccccbbbw',
    0xbe51: 'cccccccccccbbbv',
    0xbe52: 'cccccccccccbbbu',
    0xbe53: 'cccccccccccbbbt',
    0xbe54: 'cccccccccccbbbs',
    0xbe55: 'cccccccccccbbbr',
    0xbe56: 'cccccccccccbbbq',
    0xbe57: 'cccccccccccbbbp',
    0xbe58: 'cccccccccccbbhz',
    0xbe59: 'cccccccccccbbjz',
    0xbe5a: 'cccccccccccbbhx',
    0xbe5b: 'cccccccccccbbhy',
    0xbe5c: 'cccccccccccbbdx',
    0xbe5d: 'aaaabbaP',
    0xbe5e: 'cccccccccccbbby',
    0xbe5f: 'bbbbaabQ',
    0xbe60: 'cccccccccccbbbG',
    0xbe61: 'ccccbbaP',
    0xbe62: 'cccccccccccbbbE',
    0xbe63: 'cccccccccccbbbD',
    0xbe64: 'cccccccccccbbbC',
    0xbe65: 'cccccccccccbbbB',
    0xbe66: 'cccccccccccbbbA',
    0xbe67: 'cccccccccccbbdC',
    0xbe68: 'cccccccccccbbbO',
    0xbe69: 'cccccccccccbbbN',
    0xbe6a: 'cccccccccccbbbM',
    0xbe6b: 'cccccccccccbbbL',
    0xbe6c: 'cccccccccccbbbK',
    0xbe6d: 'cccccccccccbbbJ',
    0xbe6e: 'cccccccccccbbbI',
    0xbe6f: 'cccccccccccbbbH',
    0xbe70: 'cccccccccccbbbW',
    0xbe71: 'cccccccccccbbbV',
    0xbe72: 'cccccccccccbbbU',
    0xbe73: 'cccccccccccbbbT',
    0xbe74: 'cccccccccccbbbS',
    0xbe75: 'cccccccccccbbbR',
    0xbe76: 'cccccccccccbbbQ',
    0xbe77: 'cccccccccccbbbP',
    0xbe78: 'cccccccccccbbhZ',
    0xbe79: 'cccccccccccbbjZ',
    0xbe7a: 'cccccccccccbbhX',
    0xbe7b: 'cccccccccccbbhY',
    0xbe7c: 'aaaaaab10',
    0xbe7d: 'aaaaaabQ',
    0xbe7e: 'aaaaaaba04',
    0xbe7f: 'bbbbbbaP',
    0xbe80: 'cccccaa00',
    0xbe81: 'ccccabbQ',
    0xbe82: 'ccccccba04',
    0xbe83: 'ppppaabP',
    0xbe84: 'eeeeeab10',
    0xbe85: 'eeeeaaaP',
    0xbe86: 'eeeeeebb14',
    0xbe87: 'ffffbbbQ',
    0xbe88: 'llllllab04',
    0xbe89: 'ooooabaP',
    0xbe8a: 'lllllbb00',
    0xbe8b: 'llllbabQ',
    0xbe8c: 'iiiiiab00',
    0xbe8d: 'iiiiaabQ',
    0xbe8e: 'iiiiiibb04',
    0xbe8f: 'jjjjbbaP',
    0xbe90: 'xxxxxxab15',
    0xbe91: 'kkkkbbaP',
    0xbe92: 'xxxxxbb11',
    0xbe93: 'hhhhaabQ',
    0xbe94: '8888888aaah',
    0xbe95: 'mmmmbabQ',
    0xbe96: '8888888aaaH',
    0xbe97: 'nnnnabaP',
    0xbe98: 'ttttttab05',
    0xbe99: 'ggggbbbQ',
    0xbe9a: 'tttttbb01',
    0xbe9b: 'ddddaaaP',
    0xbe9c: 'qqqqqab01',
    0xbe9d: 'aaaabaaP',
    0xbe9e: 'qqqqqqbb05',
    0xbe9f: 'bbbbabbQ',
    0xbea0: 'sssssba11',
    0xbea1: 'ccccbaaP',
    0xbea2: 'pppppab01',
    0xbea3: 'ppppbbaQ',
    0xbea4: 'uuuuubb01',
    0xbea5: 'eeeebbbQ',
    0xbea6: 'vvvvvaa11',
    0xbea7: 'ffffaaaP',
    0xbea8: '?????bb06',
    0xbea9: 'oooobabQ',
    0xbeaa: '<<<<<aa16',
    0xbeab: 'llllabaP',
    0xbeac: 'yyyyybb11',
    0xbead: 'iiiibbaP',
    0xbeae: 'zzzzzaa01',
    0xbeaf: 'jjjjaabQ',
    0xbeb0: 'kkkkkba10',
    0xbeb1: 'kkkkaabQ',
    0xbeb2: 'hhhhhab00',
    0xbeb3: 'hhhhbbaP',
    0xbeb4: 'mmmmmbb00',
    0xbeb5: 'mmmmabaP',
    0xbeb6: 'nnnnnaa10',
    0xbeb7: 'nnnnbabQ',
    0xbeb8: 'gggggba00',
    0xbeb9: 'ggggaaaP',
    0xbeba: 'dddddab10',
    0xbebb: 'ddddbbbQ',
    0xbebc: 'aaaaabb10',
    0xbebd: 'aaaaabbQ',
    0xbebe: 'bbbbbaa00',
    0xbebf: 'bbbbbaaP',
    0xbec0: 'cccccbb10',
    0xbec1: 'ccccaaaP',
    0xbec2: 'CCCCCCaa17',
    0xbec3: 'ppppabaQ',
    0xbec4: 'eeeeeba00',
    0xbec5: 'eeeeabbQ',
    0xbec6: 'fffffab10',
    0xbec7: 'ffffbaaP',
    0xbec8: 'ooooobb00',
    0xbec9: 'ooooaabQ',
    0xbeca: 'lllllaa10',
    0xbecb: 'llllbbaP',
    0xbecc: 'iiiiiba10',
    0xbecd: 'iiiiabaP',
    0xbece: 'jjjjjab00',
    0xbecf: 'jjjjbabQ',
    0xbed0: 'XXXXXXbb06',
    0xbed1: 'kkkkbabQ',
    0xbed2: 'xxxxxaa01',
    0xbed3: 'hhhhabaP',
    0xbed4: 'XXXXXXXaaal',
    0xbed5: 'mmmmbbaP',
    0xbed6: 'XXXXXXXaaaL',
    0xbed7: 'nnnnaabQ',
    0xbed8: 'wwwwwbb01',
    0xbed9: 'ggggbaaP',
    0xbeda: 'tttttaa11',
    0xbedb: 'ddddabbQ',
    0xbedc: 'qqqqqba11',
    0xbedd: 'aaaabbbQ',
    0xbede: 'rrrrrab01',
    0xbedf: 'bbbbaaaP',
    0xbee0: 'ppppppbb05',
    0xbee1: 'ccccbbbQ',
    0xbee2: 'pppppba11',
    0xbee3: 'ppppbabP',
    0xbee4: 'pppppppaaao',
    0xbee5: 'eeeebaaP',
    0xbee6: 'pppppppaaaO',
    0xbee7: 'ffffabbQ',
    0xbee8: 'yyyyyyybabN',
    0xbee9: 'oooobbaP',
    0xbeea: 'yyyyyyybabn',
    0xbeeb: 'llllaabQ',
    0xbeec: 'yyyyyaa01',
    0xbeed: 'iiiibabQ',
    0xbeee: 'yyyyyyab15',
    0xbeef: 'jjjjabaP',
    0xbef0: 'hhhhhhbb04',
    0xbef1: 'kkkkabaP',
    0xbef2: 'hhhhhba10',
    0xbef3: 'hhhhbabQ',
    0xbef4: 'hhhhhhhaaan',
    0xbef5: 'hhhhhhhaaiz',
    0xbef6: 'hhhhhhhaaaN',
    0xbef7: 'hhhhhhhaaiZ',
    0xbef8: 'aaaaaaababO',
    0xbef9: 'aaaaaaabahZ',
    0xbefa: 'aaaaaaababo',
    0xbefb: 'aaaaaaabahz',
    0xbefc: 'aaaaaaa00',
    0xbefd: 'aaaaaaaP',
    0xbefe: 'aaaaaaab14',
    0xbeff: 'bbbbbbbQ',
    0xbf00: 'cccccab00',
    0xbf01: 'ccccabaQ',
    0xbf02: 'ccccccaa04',
    0xbf03: 'eeeeeeebbiz',
    0xbf04: 'cccccccbabO',
    0xbf05: 'cccccccbahZ',
    0xbf06: 'cccccccbabo',
    0xbf07: 'cccccccbahz',
    0xbf08: 'iiiiiiibbbO',
    0xbf09: 'iiiiiiibbhZ',
    0xbf0a: 'iiiiiiibbbo',
    0xbf0b: 'iiiiiiibbhz',
    0xbf0c: 'iiiiiaa00',
    0xbf0d: 'iiiiaaaQ',
    0xbf0e: 'iiiiiiab04',
    0xbf0f: 'jjjjbbbP',
    0xbf10: 'xxxxxxbb15',
    0xbf11: 'kkkkbbbP',
    0xbf12: 'xxxxxba11',
    0xbf13: 'hhhhaaaQ',
    0xbf14: 'xxxxxxxabao',
    0xbf15: 'mmmmbaaQ',
    0xbf16: 'xxxxxxxabaO',
    0xbf17: 'nnnnabbP',
    0xbf18: 'qqqqqqqbbbN',
    0xbf19: 'ggggbbaQ',
    0xbf1a: 'qqqqqqqbbbn',
    0xbf1b: 'ddddaabP',
    0xbf1c: 'qqqqqaa01',
    0xbf1d: 'aaaababP',
    0xbf1e: 'qqqqqqab05',
    0xbf1f: 'bbbbabaQ',
    0xbf20: 'sssssbb11',
    0xbf21: 'ccccbabP',
    0xbf22: 'pppppaa01',
    0xbf23: 'ppppbbbQ',
    0xbf24: 'uuuuuba01',
    0xbf25: 'eeeebbaQ',
    0xbf26: 'vvvvvab11',
    0xbf27: 'ffffaabP',
    0xbf28: 'YYYYYYYbbbM',
    0xbf29: 'oooobaaQ',
    0xbf2a: 'YYYYYYYbbbm',
    0xbf2b: 'llllabbP',
    0xbf2c: 'yyyyyba11',
    0xbf2d: 'iiiibbbP',
    0xbf2e: 'zzzzzab01',
    0xbf2f: 'jjjjaaaQ',
    0xbf30: 'kkkkkbb10',
    0xbf31: 'kkkkaaaQ',
    0xbf32: 'hhhhhaa00',
    0xbf33: 'hhhhbbbP',
    0xbf34: 'mmmmmba00',
    0xbf35: 'mmmmabbP',
    0xbf36: 'nnnnnab10',
    0xbf37: 'nnnnbaaQ',
    0xbf38: 'gggggbb00',
    0xbf39: 'ggggaabP',
    0xbf3a: 'dddddaa10',
    0xbf3b: 'ddddbbaQ',
    0xbf3c: 'aaaaaba10',
    0xbf3d: 'aaaaabaQ',
    0xbf3e: 'bbbbbab00',
    0xbf3f: 'bbbbbabP',
    0xbf40: 'cccccba10',
    0xbf41: 'ccccaabP',
    0xbf42: 'CCCCCCba17',
    0xbf43: 'ppppabbQ',
    0xbf44: 'eeeeebb00',
    0xbf45: 'eeeeabaQ',
    0xbf46: 'fffffaa10',
    0xbf47: 'ffffbabP',
    0xbf48: 'oooooba00',
    0xbf49: 'ooooaaaQ',
    0xbf4a: 'lllllab10',
    0xbf4b: 'llllbbbP',
    0xbf4c: 'iiiiibb10',
    0xbf4d: 'iiiiabbP',
    0xbf4e: 'jjjjjaa00',
    0xbf4f: 'jjjjbaaQ',
    0xbf50: 'XXXXXXab06',
    0xbf51: 'kkkkbaaQ',
    0xbf52: 'xxxxxab01',
    0xbf53: 'hhhhabbP',
    0xbf54: '=====ba06',
    0xbf55: 'mmmmbbbP',
    0xbf56: '>>>>>ab16',
    0xbf57: 'nnnnaaaQ',
    0xbf58: 'wwwwwba01',
    0xbf59: 'ggggbabP',
    0xbf5a: 'tttttab11',
    0xbf5b: 'ddddabaQ',
    0xbf5c: 'qqqqqbb11',
    0xbf5d: 'aaaabbaQ',
    0xbf5e: 'rrrrraa01',
    0xbf5f: 'bbbbaabP',
    0xbf60: 'ppppppab05',
    0xbf61: 'ccccbbaQ',
    0xbf62: 'pppppbb11',
    0xbf63: 'ppppbaaP',
    0xbf64: 'uuuuuab11',
    0xbf65: 'eeeebabP',
    0xbf66: 'uuuuuubb05',
    0xbf67: 'ffffabaQ',
    0xbf68: '9999999bbbI',
    0xbf69: 'oooobbbP',
    0xbf6a: '9999999bbbi',
    0xbf6b: 'llllaaaQ',
    0xbf6c: 'yyyyyab01',
    0xbf6d: 'iiiibaaQ',
    0xbf6e: 'yyyyyybb15',
    0xbf6f: 'jjjjabbP',
    0xbf70: 'hhhhhhab04',
    0xbf71: 'kkkkabbP',
    0xbf72: 'hhhhhbb10',
    0xbf73: 'hhhhbaaQ',
    0xbf74: 'mmmmmab10',
    0xbf75: 'mmmmaaaQ',
    0xbf76: 'mmmmmmbb04',
    0xbf77: 'nnnnbbbP',
    0xbf78: 'ddddddab14',
    0xbf79: 'ggggabaQ',
    0xbf7a: 'dddddbb00',
    0xbf7b: 'ddddbabP',
    0xbf7c: 'aaaaaab00',
    0xbf7d: 'aaaaaabP',
    0xbf7e: 'aaaaaabb14',
    0xbf7f: 'bbbbbbaQ',
    0xbf80: 'cccccaa10',
    0xbf81: 'ccccabbP',
    0xbf82: 'ccccccbb14',
    0xbf83: 'ppppaabQ',
    0xbf84: 'eeeeeab00',
    0xbf85: 'eeeeaaaQ',
    0xbf86: 'eeeeeeba04',
    0xbf87: 'ffffbbbP',
    0xbf88: 'llllllaa14',
    0xbf89: 'ooooabaQ',
    0xbf8a: 'lllllbb10',
    0xbf8b: 'llllbabP',
    0xbf8c: 'iiiiiab10',
    0xbf8d: 'iiiiaabP',
    0xbf8e: 'iiiiiiba14',
    0xbf8f: 'jjjjbbaQ',
    0xbf90: 'xxxxxxaa05',
    0xbf91: 'kkkkbbaQ',
    0xbf92: 'xxxxxbb01',
    0xbf93: 'hhhhaabP',
    0xbf94: '11111111babH',
    0xbf95: 'mmmmbabP',
    0xbf96: '11111111babX',
    0xbf97: 'nnnnabaQ',
    0xbf98: 'ttttttaa15',
    0xbf99: 'ggggbbbP',
    0xbf9a: 'tttttbb11',
    0xbf9b: 'ddddaaaQ',
    0xbf9c: 'qqqqqab11',
    0xbf9d: 'aaaabaaQ',
    0xbf9e: 'qqqqqqba15',
    0xbf9f: 'bbbbabbP',
    0xbfa0: 'sssssba01',
    0xbfa1: 'ccccbaaQ',
    0xbfa2: 'pppppab11',
    0xbfa3: 'ppppbbaP',
    0xbfa4: 'uuuuubb11',
    0xbfa5: 'eeeebbbP',
    0xbfa6: 'vvvvvaa01',
    0xbfa7: 'ffffaaaQ',
    0xbfa8: '?????bb16',
    0xbfa9: 'oooobabP',
    0xbfaa: '<<<<<aa06',
    0xbfab: 'llllabaQ',
    0xbfac: 'yyyyybb01',
    0xbfad: 'iiiibbaQ',
    0xbfae: 'zzzzzaa11',
    0xbfaf: 'jjjjaabP',
    0xbfb0: 'kkkkkba00',
    0xbfb1: 'kkkkaabP',
    0xbfb2: 'hhhhhab10',
    0xbfb3: 'hhhhbbaQ',
    0xbfb4: 'mmmmmbb10',
    0xbfb5: 'mmmmabaQ',
    0xbfb6: 'nnnnnaa00',
    0xbfb7: 'nnnnbabP',
    0xbfb8: 'gggggba10',
    0xbfb9: 'ggggaaaQ',
    0xbfba: 'dddddab00',
    0xbfbb: 'ddddbbbP',
    0xbfbc: 'aaaaabb00',
    0xbfbd: 'aaaaabbP',
    0xbfbe: 'bbbbbaa10',
    0xbfbf: 'bbbbbaaQ',
    0xbfc0: 'cccccbb00',
    0xbfc1: 'ccccaaaQ',
    0xbfc2: 'CCCCCCab07',
    0xbfc3: 'ppppabaP',
    0xbfc4: 'eeeeeba10',
    0xbfc5: 'eeeeabbP',
    0xbfc6: 'fffffab00',
    0xbfc7: 'ffffbaaQ',
    0xbfc8: 'ooooobb10',
    0xbfc9: 'ooooaabP',
    0xbfca: 'lllllaa00',
    0xbfcb: 'llllbbaQ',
    0xbfcc: 'iiiiiba00',
    0xbfcd: 'iiiiabaQ',
    0xbfce: 'jjjjjab10',
    0xbfcf: 'jjjjbabP',
    0xbfd0: 'QQQQQQQQbabl',
    0xbfd1: 'kkkkbabP',
    0xbfd2: 'xxxxxaa11',
    0xbfd3: 'hhhhabaQ',
    0xbfd4: 'QQQQQQQQbabL',
    0xbfd5: 'mmmmbbaQ',
    0xbfd6: 'QQQQQQQQbahY',
    0xbfd7: 'nnnnaabP',
    0xbfd8: 'wwwwwbb11',
    0xbfd9: 'ggggbaaQ',
    0xbfda: 'tttttaa01',
    0xbfdb: 'ddddabbP',
    0xbfdc: 'qqqqqba01',
    0xbfdd: 'aaaabbbP',
    0xbfde: 'rrrrrab11',
    0xbfdf: 'bbbbaaaQ',
    0xbfe0: 'ppppppba15',
    0xbfe1: 'ccccbbbP',
    0xbfe2: 'pppppba01',
    0xbfe3: 'ppppbabQ',
    0xbfe4: 'pppppppbbbn',
    0xbfe5: 'eeeebaaQ',
    0xbfe6: 'pppppppbbbN',
    0xbfe7: 'ffffabbP',
    0xbfe8: 'ppppppppaaiZ',
    0xbfe9: 'oooobbaQ',
    0xbfea: 'ppppppppaaaN',
    0xbfeb: 'llllaabP',
    0xbfec: 'ppppppppaaiz',
    0xbfed: 'iiiibabP',
    0xbfee: 'ppppppppaaan',
    0xbfef: 'jjjjabaQ',
    0xbff0: 'aaaaaaaababn',
    0xbff1: 'aaaaaaaababf',
    0xbff2: 'aaaaaaaabajz',
    0xbff3: 'aaaaaaaababv',
    0xbff4: 'aaaaaaaababN',
    0xbff5: 'aaaaaaaababF',
    0xbff6: 'aaaaaaaabajZ',
    0xbff7: 'aaaaaaaababV',
    0xbff8: 'aaaaaaaabaN',
    0xbff9: 'aaaaaaaabiZ',
    0xbffa: 'aaaaaaaaban',
    0xbffb: 'aaaaaaaabiz',
    0xbffc: 'aaaaaaa10',
    0xbffd: 'aaaaaaaQ',
    0xbffe: 'aaaaaaaa04',
    0xbfff: 'aaaaaaaah8',
    0xc000: 'aaabQ',
    0xc001: 'bbba00',
    0xc002: 'bbbaP',
    0xc003: 'aaab10',
    0xc004: 'abbaP',
    0xc005: 'cccccccaaaQ',
    0xc006: 'baabQ',
    0xc007: 'ccba00',
    0xc008: 'aabaP',
    0xc009: 'bbab10',
    0xc00a: 'bbabQ',
    0xc00b: 'aaba00',
    0xc00c: 'ababQ',
    0xc00d: 'ccccccccbbba',
    0xc00e: 'babaP',
    0xc00f: 'ccab10',
    0xc010: 'aaaaP',
    0xc011: 'bbbb10',
    0xc012: 'bbbbQ',
    0xc013: 'aaaa00',
    0xc014: 'abbbQ',
    0xc015: 'ddba00',
    0xc016: 'baaaP',
    0xc017: 'ccbb10',
    0xc018: 'aabbQ',
    0xc019: 'bbaa00',
    0xc01a: 'bbaaP',
    0xc01b: 'aabb10',
    0xc01c: 'abaaP',
    0xc01d: 'ddab10',
    0xc01e: 'babbQ',
    0xc01f: 'ccaa00',
    0xc020: 'aaabP',
    0xc021: 'bbba10',
    0xc022: 'bbbaQ',
    0xc023: 'aaab00',
    0xc024: 'abbaQ',
    0xc025: 'ddbb00',
    0xc026: 'baabP',
    0xc027: 'ccba10',
    0xc028: 'aabaQ',
    0xc029: 'bbab00',
    0xc02a: 'bbabP',
    0xc02b: 'aaba10',
    0xc02c: 'ababP',
    0xc02d: 'ddaa10',
    0xc02e: 'babaQ',
    0xc02f: 'ccab00',
    0xc030: 'aaaaQ',
    0xc031: 'bbbb00',
    0xc032: 'aaaaah8',
    0xc033: 'aaaa10',
    0xc034: 'abbbP',
    0xc035: 'ddba10',
    0xc036: 'baaaQ',
    0xc037: 'ccbb00',
    0xc038: 'aabbP',
    0xc039: 'bbaa10',
    0xc03a: 'bbaaQ',
    0xc03b: 'aabb00',
    0xc03c: 'abaaQ',
    0xc03d: 'aaaaabM8',
    0xc03e: 'babbP',
    0xc03f: 'bbbbbaL8',
    0xc040: 'aaabS',
    0xc041: 'bbba01',
    0xc042: 'bbbaR',
    0xc043: 'aaab11',
    0xc044: 'abbaR',
    0xc045: 'ddbb11',
    0xc046: 'baabS',
    0xc047: 'ccba01',
    0xc048: 'aabaR',
    0xc049: 'bbab11',
    0xc04a: 'bbabS',
    0xc04b: 'aaba01',
    0xc04c: 'ababS',
    0xc04d: 'ddaa01',
    0xc04e: 'babaR',
    0xc04f: 'ccab11',
    0xc050: 'aaaaR',
    0xc051: 'bbbb11',
    0xc052: 'bbbbS',
    0xc053: 'aaaa01',
    0xc054: 'abbbS',
    0xc055: 'ddba01',
    0xc056: 'baaaR',
    0xc057: 'ccbb11',
    0xc058: 'aabbS',
    0xc059: 'bbaa01',
    0xc05a: 'bbaaR',
    0xc05b: 'aabb11',
    0xc05c: 'abaaR',
    0xc05d: 'ddab11',
    0xc05e: 'babbS',
    0xc05f: 'ccaa01',
    0xc060: 'aaabR',
    0xc061: 'bbba11',
    0xc062: 'bbbaS',
    0xc063: 'aaab01',
    0xc064: 'abbaS',
    0xc065: 'ddbb01',
    0xc066: 'baabR',
    0xc067: 'ccba11',
    0xc068: 'aabaS',
    0xc069: 'bbab01',
    0xc06a: 'bbabR',
    0xc06b: 'aaba11',
    0xc06c: 'ababR',
    0xc06d: 'ddaa11',
    0xc06e: 'babaS',
    0xc06f: 'ccab01',
    0xc070: 'aaaaS',
    0xc071: 'bbbb01',
    0xc072: 'aaaaai8',
    0xc073: 'aaaa11',
    0xc074: 'abbbR',
    0xc075: 'ddba11',
    0xc076: 'baaaS',
    0xc077: 'ccbb01',
    0xc078: 'aabbR',
    0xc079: 'bbaa11',
    0xc07a: 'bbaaS',
    0xc07b: 'aabb01',
    0xc07c: 'abaaS',
    0xc07d: 'aaaaaaL8',
    0xc07e: 'aaaaaab999',
    0xc07f: 'bbbbbbM8',
    0xc080: 'aaabU',
    0xc081: 'bbba02',
    0xc082: 'bbbaT',
    0xc083: 'aaab12',
    0xc084: 'abbaT',
    0xc085: 'ddbb12',
    0xc086: 'baabU',
    0xc087: 'ccba02',
    0xc088: 'aabaT',
    0xc089: 'bbab12',
    0xc08a: 'bbabU',
    0xc08b: 'aaba02',
    0xc08c: 'ababU',
    0xc08d: 'ddaa02',
    0xc08e: 'babaT',
    0xc08f: 'ccab12',
    0xc090: 'aaaaT',
    0xc091: 'bbbb12',
    0xc092: 'bbbbU',
    0xc093: 'aaaa02',
    0xc094: 'abbbU',
    0xc095: 'ddba02',
    0xc096: 'baaaT',
    0xc097: 'ccbb12',
    0xc098: 'aabbU',
    0xc099: 'bbaa02',
    0xc09a: 'bbaaT',
    0xc09b: 'aabb12',
    0xc09c: 'abaaT',
    0xc09d: 'ddab12',
    0xc09e: 'babbU',
    0xc09f: 'ccaa02',
    0xc0a0: 'aaabT',
    0xc0a1: 'bbba12',
    0xc0a2: 'bbbaU',
    0xc0a3: 'aaab02',
    0xc0a4: 'abbaU',
    0xc0a5: 'ddbb02',
    0xc0a6: 'baabT',
    0xc0a7: 'ccba12',
    0xc0a8: 'aabaU',
    0xc0a9: 'bbab02',
    0xc0aa: 'bbabT',
    0xc0ab: 'aaba12',
    0xc0ac: 'ababT',
    0xc0ad: 'ddaa12',
    0xc0ae: 'babaU',
    0xc0af: 'ccab02',
    0xc0b0: 'aaaaU',
    0xc0b1: 'bbbb02',
    0xc0b2: 'aaaaah9',
    0xc0b3: 'aaaa12',
    0xc0b4: 'abbbT',
    0xc0b5: 'ddba12',
    0xc0b6: 'baaaU',
    0xc0b7: 'ccbb02',
    0xc0b8: 'aabbT',
    0xc0b9: 'bbaa12',
    0xc0ba: 'bbaaU',
    0xc0bb: 'aabb02',
    0xc0bc: 'abaaU',
    0xc0bd: 'aaaaabL8',
    0xc0be: 'babbT',
    0xc0bf: 'bbbbbaM8',
    0xc0c0: 'aaabW',
    0xc0c1: 'bbba03',
    0xc0c2: 'bbbaV',
    0xc0c3: 'aaab13',
    0xc0c4: 'abbaV',
    0xc0c5: 'ddbb13',
    0xc0c6: 'baabW',
    0xc0c7: 'ccba03',
    0xc0c8: 'aabaV',
    0xc0c9: 'bbab13',
    0xc0ca: 'bbabW',
    0xc0cb: 'aaba03',
    0xc0cc: 'ababW',
    0xc0cd: 'ddaa03',
    0xc0ce: 'babaV',
    0xc0cf: 'ccab13',
    0xc0d0: 'aaaaV',
    0xc0d1: 'bbbb13',
    0xc0d2: 'bbbbW',
    0xc0d3: 'aaaa03',
    0xc0d4: 'abbbW',
    0xc0d5: 'ddba03',
    0xc0d6: 'baaaV',
    0xc0d7: 'ccbb13',
    0xc0d8: 'aabbW',
    0xc0d9: 'bbaa03',
    0xc0da: 'bbaaV',
    0xc0db: 'aabb13',
    0xc0dc: 'abaaV',
    0xc0dd: 'ddab13',
    0xc0de: 'babbW',
    0xc0df: 'ccaa03',
    0xc0e0: 'aaabV',
    0xc0e1: 'bbba13',
    0xc0e2: 'bbbaW',
    0xc0e3: 'aaab03',
    0xc0e4: 'abbaW',
    0xc0e5: 'ddbb03',
    0xc0e6: 'baabV',
    0xc0e7: 'ccba13',
    0xc0e8: 'aabaW',
    0xc0e9: 'bbab03',
    0xc0ea: 'bbabV',
    0xc0eb: 'aaba13',
    0xc0ec: 'ababV',
    0xc0ed: 'ddaa13',
    0xc0ee: 'babaW',
    0xc0ef: 'ccab03',
    0xc0f0: 'aaaaW',
    0xc0f1: 'bbbb03',
    0xc0f2: 'aaaaai9',
    0xc0f3: 'aaaa13',
    0xc0f4: 'abbbV',
    0xc0f5: 'ddba13',
    0xc0f6: 'baaaW',
    0xc0f7: 'ccbb03',
    0xc0f8: 'aabbV',
    0xc0f9: 'aaaaaaabbbP',
    0xc0fa: 'aaaaaaabbdc',
    0xc0fb: 'aabb03',
    0xc0fc: 'abaaW',
    0xc0fd: 'aaaaaaM8',
    0xc0fe: 'aaaaaaa899',
    0xc0ff: 'aaaaaaabB2',
    0xc100: 'aaabY',
    0xc101: 'bbba04',
    0xc102: 'bbbaX',
    0xc103: 'aaab14',
    0xc104: 'abbaX',
    0xc105: 'cccccccbbbP',
    0xc106: 'baabY',
    0xc107: 'ccba04',
    0xc108: 'aabaX',
    0xc109: 'bbab14',
    0xc10a: 'bbabY',
    0xc10b: 'aaba04',
    0xc10c: 'ababY',
    0xc10d: 'ddaa04',
    0xc10e: 'babaX',
    0xc10f: 'ccab14',
    0xc110: 'aaaaX',
    0xc111: 'bbbb14',
    0xc112: 'bbbbY',
    0xc113: 'aaaa04',
    0xc114: 'abbbY',
    0xc115: 'ddba04',
    0xc116: 'baaaX',
    0xc117: 'ccbb14',
    0xc118: 'aabbY',
    0xc119: 'bbaa04',
    0xc11a: 'bbaaX',
    0xc11b: 'aabb14',
    0xc11c: 'abaaX',
    0xc11d: 'ddab14',
    0xc11e: 'babbY',
    0xc11f: 'ccaa04',
    0xc120: 'aaabX',
    0xc121: 'bbba14',
    0xc122: 'bbbaY',
    0xc123: 'aaab04',
    0xc124: 'abbaY',
    0xc125: 'ddbb04',
    0xc126: 'baabX',
    0xc127: 'ccba14',
    0xc128: 'aabaY',
    0xc129: 'bbab04',
    0xc12a: 'bbabX',
    0xc12b: 'aaba14',
    0xc12c: 'ababX',
    0xc12d: 'ddaa14',
    0xc12e: 'babaY',
    0xc12f: 'ccab04',
    0xc130: 'aaaaY',
    0xc131: 'bbbb04',
    0xc132: 'aaaaal8',
    0xc133: 'aaaa14',
    0xc134: 'abbbX',
    0xc135: 'ddba14',
    0xc136: 'baaaY',
    0xc137: 'ccbb04',
    0xc138: 'aabbX',
    0xc139: 'bbaa14',
    0xc13a: 'bbaaY',
    0xc13b: 'aabb04',
    0xc13c: 'abaaY',
    0xc13d: 'aaaaabM9',
    0xc13e: 'babbX',
    0xc13f: 'bbbbbaL9',
    0xc140: 'aaadX',
    0xc141: 'bbba05',
    0xc142: 'bbbaZ',
    0xc143: 'aaab15',
    0xc144: 'abbaZ',
    0xc145: 'ddbb15',
    0xc146: 'baadX',
    0xc147: 'ccba05',
    0xc148: 'aabaZ',
    0xc149: 'bbab15',
    0xc14a: 'bbadX',
    0xc14b: 'aaba05',
    0xc14c: 'abadX',
    0xc14d: 'ddaa05',
    0xc14e: 'babaZ',
    0xc14f: 'ccab15',
    0xc150: 'aaaaZ',
    0xc151: 'bbbb15',
    0xc152: 'aaaabl8',
    0xc153: 'aaaa05',
    0xc154: 'abbdX',
    0xc155: 'ddba05',
    0xc156: 'baaaZ',
    0xc157: 'ccbb15',
    0xc158: 'aabdX',
    0xc159: 'bbaa05',
    0xc15a: 'bbaaZ',
    0xc15b: 'aabb15',
    0xc15c: 'abaaZ',
    0xc15d: 'ddab15',
    0xc15e: 'babdX',
    0xc15f: 'ccaa05',
    0xc160: 'aaabZ',
    0xc161: 'bbba15',
    0xc162: 'bbbcZ',
    0xc163: 'aaab05',
    0xc164: 'abbcZ',
    0xc165: 'ddbb05',
    0xc166: 'baabZ',
    0xc167: 'ccba15',
    0xc168: 'aabcZ',
    0xc169: 'bbab05',
    0xc16a: 'bbabZ',
    0xc16b: 'aaba15',
    0xc16c: 'ababZ',
    0xc16d: 'ddaa15',
    0xc16e: 'babcZ',
    0xc16f: 'ccab05',
    0xc170: 'aaacZ',
    0xc171: 'bbbb05',
    0xc172: 'aaaaam8',
    0xc173: 'aaaa15',
    0xc174: 'abbbZ',
    0xc175: 'ddba15',
    0xc176: 'baacZ',
    0xc177: 'ccbb05',
    0xc178: 'aabbZ',
    0xc179: 'bbaa15',
    0xc17a: 'bbacZ',
    0xc17b: 'aabb05',
    0xc17c: 'abacZ',
    0xc17d: 'aaaaaaL9',
    0xc17e: 'aaaaaab899',
    0xc17f: 'bbbbbbM9',
    0xc180: 'aaahX',
    0xc181: 'aaaaaaaaaaababZ',
    0xc182: 'aaaaaaaaaaababY',
    0xc183: 'aaab16',
    0xc184: 'abbiX',
    0xc185: 'aaaaaaaaaaabajZ',
    0xc186: 'aaaaaaaaaaabahX',
    0xc187: 'aaaaaaaaaaabahY',
    0xc188: 'aabiX',
    0xc189: 'aaaaaaaaaaababR',
    0xc18a: 'aaaaaaaaaaababQ',
    0xc18b: 'aaba06',
    0xc18c: 'abahX',
    0xc18d: 'aaaaaaaaaaababV',
    0xc18e: 'aaaaaaaaaaababU',
    0xc18f: 'aaaaaaaaaaababT',
    0xc190: 'aaaiX',
    0xc191: 'aaaaaaaaaaababJ',
    0xc192: 'aaaabm9',
    0xc193: 'aaaa06',
    0xc194: 'abbhX',
    0xc195: 'aaaaaaaaaaababN',
    0xc196: 'aaaaaaaaaaababM',
    0xc197: 'aaaaaaaaaaababL',
    0xc198: 'aabhX',
    0xc199: 'aaaaaaaaaaababB',
    0xc19a: 'aaaaaaaaaaababA',
    0xc19b: 'aabb16',
    0xc19c: 'abaiX',
    0xc19d: 'aaaaaaaaaaababF',
    0xc19e: 'aaaaaaaaaaababE',
    0xc19f: 'aaaaaaaaaaababD',
    0xc1a0: 'aaahY',
    0xc1a1: 'aaaaaaaaaaababz',
    0xc1a2: 'aaaaaaaaaaababy',
    0xc1a3: 'aaab06',
    0xc1a4: 'abbiY',
    0xc1a5: 'aaaaaaaaaaabajz',
    0xc1a6: 'aaaaaaaaaaabahx',
    0xc1a7: 'aaaaaaaaaaabahy',
    0xc1a8: 'aabiY',
    0xc1a9: 'aaaaaaaaaaababr',
    0xc1aa: 'aaaaaaaaaaababq',
    0xc1ab: 'aaba16',
    0xc1ac: 'abahY',
    0xc1ad: 'aaaaaaaaaaababv',
    0xc1ae: 'aaaaaaaaaaababu',
    0xc1af: 'aaaaaaaaaaababt',
    0xc1b0: 'aaaiY',
    0xc1b1: 'aaaaaaaaaaababj',
    0xc1b2: 'aaaaal9',
    0xc1b3: 'aaaa16',
    0xc1b4: 'abbhY',
    0xc1b5: 'aaaaaaaaaaababn',
    0xc1b6: 'aaaaaaaaaaababm',
    0xc1b7: 'aaaaaaaaaaababl',
    0xc1b8: 'aabhY',
    0xc1b9: 'aaaaaaaaaaababb',
    0xc1ba: 'aaaaaaaaaaababa',
    0xc1bb: 'aabb06',
    0xc1bc: 'abaiY',
    0xc1bd: 'aaaaabL9',
    0xc1be: 'aaaaaaaaaaababe',
    0xc1bf: 'aaaaaaaaaaababd',
    0xc1c0: 'aaahZ',
    0xc1c1: 'aaaaaaaaaaabaT',
    0xc1c2: 'aaaaaaaaaaabaR',
    0xc1c3: 'aaab17',
    0xc1c4: 'abbiZ',
    0xc1c5: 'aaaaaaaaaaabiX',
    0xc1c6: 'aaaaaaaaaaabaZ',
    0xc1c7: 'aaaaaaaaaaabaX',
    0xc1c8: 'aabiZ',
    0xc1c9: 'aaaaaaaaaaabaD',
    0xc1ca: 'aaaaaaaaaaabaB',
    0xc1cb: 'aaba07',
    0xc1cc: 'abahZ',
    0xc1cd: 'aaaaaaaaaaabaL',
    0xc1ce: 'aaaaaaaaaaabaJ',
    0xc1cf: 'aaaaaaaaaaabaH',
    0xc1d0: 'aaaiZ',
    0xc1d1: 'aaaaaaaaaaabat',
    0xc1d2: 'aaaabl9',
    0xc1d3: 'aaaa07',
    0xc1d4: 'abbhZ',
    0xc1d5: 'aaaaaaaaaaabix',
    0xc1d6: 'aaaaaaaaaaabaz',
    0xc1d7: 'aaaaaaaaaaabax',
    0xc1d8: 'aabhZ',
    0xc1d9: 'aaaaaaaaaaabad',
    0xc1da: 'aaaaaaaaaaabab',
    0xc1db: 'aabb17',
    0xc1dc: 'abaiZ',
    0xc1dd: 'aaaaaaaaaaabal',
    0xc1de: 'aaaaaaaaaaabaj',
    0xc1df: 'aaaaaaaaaaabah',
    0xc1e0: 'aaajZ',
    0xc1e1: 'aaaaaaaaaaabE',
    0xc1e2: 'aaaaaaaaaaabI',
    0xc1e3: 'aaab07',
    0xc1e4: 'abbkZ',
    0xc1e5: 'aaaaaaaaaaabU',
    0xc1e6: 'aaaaaaaaaaabY',
    0xc1e7: 'aaaaaaaaaaahX',
    0xc1e8: 'aabkZ',
    0xc1e9: 'aaaaaaaaaaabe',
    0xc1ea: 'aaaaaaaaaaabi',
    0xc1eb: 'aaba17',
    0xc1ec: 'abajZ',
    0xc1ed: 'aaaaaaaaaaabu',
    0xc1ee: 'aaaaaaaaaaaby',
    0xc1ef: 'aaaaaaaaaaahx',
    0xc1f0: 'aaakZ',
    0xc1f1: 'aaaaaaaaaaca',
    0xc1f2: 'aaaaam9',
    0xc1f3: 'aaaa17',
    0xc1f4: 'abbjZ',
    0xc1f5: 'aaaaaaaaaacA',
    0xc1f6: 'aaaaaaaaaaaX',
    0xc1f7: 'aaaaaaaaaaaP',
    0xc1f8: 'aabjZ',
    0xc1f9: 'aaaaaaaaaaQ',
    0xc1fa: 'aaaaaaaaaaa',
    0xc1fb: 'aabb07',
    0xc1fc: 'abakZ',
    0xc1fd: 'aaaaaaM9',
    0xc1fe: 'aaaaaaa999',
    0xc1ff: 'aaaaaaaaA3',
    0xc200: 'aaabA',
    0xc201: 'bbba08',
    0xc202: 'bbbcA',
    0xc203: 'aaab18',
    0xc204: 'abbcA',
    0xc205: 'cccccccabbP',
    0xc206: 'baabA',
    0xc207: 'ccba08',
    0xc208: 'aabcA',
    0xc209: 'bbab18',
    0xc20a: 'bbabA',
    0xc20b: 'aaba08',
    0xc20c: 'ababA',
    0xc20d: 'ccccccccaaca',
    0xc20e: 'babcA',
    0xc20f: 'ccab18',
    0xc210: 'aaacA',
    0xc211: 'bbbb18',
    0xc212: 'bbbbA',
    0xc213: 'aaaa08',
    0xc214: 'abbbA',
    0xc215: 'cccccccccbbad',
    0xc216: 'baacA',
    0xc217: 'ccbb18',
    0xc218: 'aabbA',
    0xc219: 'bbaa08',
    0xc21a: 'bbacA',
    0xc21b: 'aabb18',
    0xc21c: 'abacA',
    0xc21d: 'cccccccccbbaD',
    0xc21e: 'babbA',
    0xc21f: 'ccaa08',
    0xc220: 'aaadC',
    0xc221: 'bbba18',
    0xc222: 'bbbaA',
    0xc223: 'aaab08',
    0xc224: 'abbaA',
    0xc225: 'ddbb08',
    0xc226: 'baadC',
    0xc227: 'ccba18',
    0xc228: 'aabaA',
    0xc229: 'bbab08',
    0xc22a: 'bbadC',
    0xc22b: 'aaba18',
    0xc22c: 'abadC',
    0xc22d: 'ddaa18',
    0xc22e: 'babaA',
    0xc22f: 'ccab08',
    0xc230: 'aaaaA',
    0xc231: 'bbbb08',
    0xc232: 'aaaaab9',
    0xc233: 'aaaa18',
    0xc234: 'abbdC',
    0xc235: 'ddba18',
    0xc236: 'baaaA',
    0xc237: 'ccbb08',
    0xc238: 'aabdC',
    0xc239: 'bbaa18',
    0xc23a: 'bbaaA',
    0xc23b: 'aabb08',
    0xc23c: 'abaaA',
    0xc23d: 'aaaaabI8',
    0xc23e: 'babdC',
    0xc23f: 'bbbbbaH8',
    0xc240: 'aaabC',
    0xc241: 'bbba09',
    0xc242: 'bbbaB',
    0xc243: 'aaab19',
    0xc244: 'abbaB',
    0xc245: 'ddbb19',
    0xc246: 'baabC',
    0xc247: 'ccba09',
    0xc248: 'aabaB',
    0xc249: 'bbab19',
    0xc24a: 'bbabC',
    0xc24b: 'aaba09',
    0xc24c: 'ababC',
    0xc24d: 'ddaa09',
    0xc24e: 'babaB',
    0xc24f: 'ccab19',
    0xc250: 'aaaaB',
    0xc251: 'bbbb19',
    0xc252: 'bbbbC',
    0xc253: 'aaaa09',
    0xc254: 'abbbC',
    0xc255: 'ddba09',
    0xc256: 'baaaB',
    0xc257: 'ccbb19',
    0xc258: 'aabbC',
    0xc259: 'bbaa09',
    0xc25a: 'bbaaB',
    0xc25b: 'aabb19',
    0xc25c: 'abaaB',
    0xc25d: 'ddab19',
    0xc25e: 'babbC',
    0xc25f: 'ccaa09',
    0xc260: 'aaabB',
    0xc261: 'bbba19',
    0xc262: 'bbbaC',
    0xc263: 'aaab09',
    0xc264: 'abbaC',
    0xc265: 'ddbb09',
    0xc266: 'baabB',
    0xc267: 'ccba19',
    0xc268: 'aabaC',
    0xc269: 'bbab09',
    0xc26a: 'bbabB',
    0xc26b: 'aaba19',
    0xc26c: 'ababB',
    0xc26d: 'ddaa19',
    0xc26e: 'babaC',
    0xc26f: 'ccab09',
    0xc270: 'aaaaC',
    0xc271: 'bbbb09',
    0xc272: 'aaaaaa8',
    0xc273: 'aaaa19',
    0xc274: 'abbbB',
    0xc275: 'ddba19',
    0xc276: 'baaaC',
    0xc277: 'ccbb09',
    0xc278: 'aabbB',
    0xc279: 'bbaa19',
    0xc27a: 'bbaaC',
    0xc27b: 'aabb09',
    0xc27c: 'abaaC',
    0xc27d: 'aaaaaaH8',
    0xc27e: 'aaaaaab989',
    0xc27f: 'bbbbbbI8',
    0xc280: 'aaabE',
    0xc281: 'bbba48',
    0xc282: 'bbbaD',
    0xc283: 'aaab58',
    0xc284: 'abbaD',
    0xc285: 'ddbb58',
    0xc286: 'baabE',
    0xc287: 'ccba48',
    0xc288: 'aabaD',
    0xc289: 'bbab58',
    0xc28a: 'bbabE',
    0xc28b: 'aaba48',
    0xc28c: 'ababE',
    0xc28d: 'ddaa48',
    0xc28e: 'babaD',
    0xc28f: 'ccab58',
    0xc290: 'aaaaD',
    0xc291: 'bbbb58',
    0xc292: 'bbbbE',
    0xc293: 'aaaa48',
    0xc294: 'abbbE',
    0xc295: 'ddba48',
    0xc296: 'baaaD',
    0xc297: 'ccbb58',
    0xc298: 'aabbE',
    0xc299: 'bbaa48',
    0xc29a: 'bbaaD',
    0xc29b: 'aabb58',
    0xc29c: 'abaaD',
    0xc29d: 'ddab58',
    0xc29e: 'babbE',
    0xc29f: 'ccaa48',
    0xc2a0: 'aaabD',
    0xc2a1: 'bbba58',
    0xc2a2: 'bbbaE',
    0xc2a3: 'aaab48',
    0xc2a4: 'abbaE',
    0xc2a5: 'ddbb48',
    0xc2a6: 'baabD',
    0xc2a7: 'ccba58',
    0xc2a8: 'aabaE',
    0xc2a9: 'bbab48',
    0xc2aa: 'bbabD',
    0xc2ab: 'aaba58',
    0xc2ac: 'ababD',
    0xc2ad: 'ddaa58',
    0xc2ae: 'babaE',
    0xc2af: 'ccab48',
    0xc2b0: 'aaaaE',
    0xc2b1: 'bbbb48',
    0xc2b2: 'aaaaab8',
    0xc2b3: 'aaaa58',
    0xc2b4: 'abbbD',
    0xc2b5: 'ddba58',
    0xc2b6: 'baaaE',
    0xc2b7: 'ccbb48',
    0xc2b8: 'aabbD',
    0xc2b9: 'bbaa58',
    0xc2ba: 'bbaaE',
    0xc2bb: 'aabb48',
    0xc2bc: 'abaaE',
    0xc2bd: 'aaaaabH8',
    0xc2be: 'babbD',
    0xc2bf: 'bbbbbaI8',
    0xc2c0: 'aaabG',
    0xc2c1: 'bbba49',
    0xc2c2: 'bbbaF',
    0xc2c3: 'aaab59',
    0xc2c4: 'abbaF',
    0xc2c5: 'ddbb59',
    0xc2c6: 'baabG',
    0xc2c7: 'ccba49',
    0xc2c8: 'aabaF',
    0xc2c9: 'bbab59',
    0xc2ca: 'bbabG',
    0xc2cb: 'aaba49',
    0xc2cc: 'ababG',
    0xc2cd: 'ddaa49',
    0xc2ce: 'babaF',
    0xc2cf: 'ccab59',
    0xc2d0: 'aaaaF',
    0xc2d1: 'bbbb59',
    0xc2d2: 'bbbbG',
    0xc2d3: 'aaaa49',
    0xc2d4: 'abbbG',
    0xc2d5: 'ddba49',
    0xc2d6: 'baaaF',
    0xc2d7: 'ccbb59',
    0xc2d8: 'aabbG',
    0xc2d9: 'bbaa49',
    0xc2da: 'bbaaF',
    0xc2db: 'aabb59',
    0xc2dc: 'abaaF',
    0xc2dd: 'ddab59',
    0xc2de: 'babbG',
    0xc2df: 'ccaa49',
    0xc2e0: 'aaabF',
    0xc2e1: 'bbba59',
    0xc2e2: 'bbbaG',
    0xc2e3: 'aaab49',
    0xc2e4: 'abbaG',
    0xc2e5: 'ddbb49',
    0xc2e6: 'baabF',
    0xc2e7: 'ccba59',
    0xc2e8: 'aabaG',
    0xc2e9: 'bbab49',
    0xc2ea: 'bbabF',
    0xc2eb: 'aaba59',
    0xc2ec: 'ababF',
    0xc2ed: 'ddaa59',
    0xc2ee: 'babaG',
    0xc2ef: 'ccab49',
    0xc2f0: 'aaaaG',
    0xc2f1: 'bbbb49',
    0xc2f2: 'aaaaaa9',
    0xc2f3: 'aaaa59',
    0xc2f4: 'abbbF',
    0xc2f5: 'ddba59',
    0xc2f6: 'baaaG',
    0xc2f7: 'ccbb49',
    0xc2f8: 'aabbF',
    0xc2f9: 'aaaaaaabaaQ',
    0xc2fa: 'aaaaaaabaaa',
    0xc2fb: 'aabb49',
    0xc2fc: 'abaaG',
    0xc2fd: 'aaaaaaI8',
    0xc2fe: 'aaaaaaa889',
    0xc2ff: 'aaaaaaabA3',
    0xc300: 'aaabI',
    0xc301: 'bbba88',
    0xc302: 'bbbaH',
    0xc303: 'aaab98',
    0xc304: 'abbaH',
    0xc305: 'cccccccbaaQ',
    0xc306: 'baabI',
    0xc307: 'ccba88',
    0xc308: 'aabaH',
    0xc309: 'bbab98',
    0xc30a: 'bbabI',
    0xc30b: 'aaba88',
    0xc30c: 'ababI',
    0xc30d: 'ddaa88',
    0xc30e: 'babaH',
    0xc30f: 'ccab98',
    0xc310: 'aaaaH',
    0xc311: 'bbbb98',
    0xc312: 'bbbbI',
    0xc313: 'aaaa88',
    0xc314: 'abbbI',
    0xc315: 'ddba88',
    0xc316: 'baaaH',
    0xc317: 'ccbb98',
    0xc318: 'aabbI',
    0xc319: 'bbaa88',
    0xc31a: 'bbaaH',
    0xc31b: 'aabb98',
    0xc31c: 'abaaH',
    0xc31d: 'ddab98',
    0xc31e: 'babbI',
    0xc31f: 'ccaa88',
    0xc320: 'aaabH',
    0xc321: 'bbba98',
    0xc322: 'bbbaI',
    0xc323: 'aaab88',
    0xc324: 'abbaI',
    0xc325: 'ddbb88',
    0xc326: 'baabH',
    0xc327: 'ccba98',
    0xc328: 'aabaI',
    0xc329: 'bbab88',
    0xc32a: 'bbabH',
    0xc32b: 'aaba98',
    0xc32c: 'ababH',
    0xc32d: 'ddaa98',
    0xc32e: 'babaI',
    0xc32f: 'ccab88',
    0xc330: 'aaaaI',
    0xc331: 'bbbb88',
    0xc332: 'aaaaad8',
    0xc333: 'aaaa98',
    0xc334: 'abbbH',
    0xc335: 'ddba98',
    0xc336: 'baaaI',
    0xc337: 'ccbb88',
    0xc338: 'aabbH',
    0xc339: 'bbaa98',
    0xc33a: 'bbaaI',
    0xc33b: 'aabb88',
    0xc33c: 'abaaI',
    0xc33d: 'aaaaabI9',
    0xc33e: 'babbH',
    0xc33f: 'bbbbbaH9',
    0xc340: 'aaabK',
    0xc341: 'bbba89',
    0xc342: 'bbbaJ',
    0xc343: 'aaab99',
    0xc344: 'abbaJ',
    0xc345: 'ddbb99',
    0xc346: 'baabK',
    0xc347: 'ccba89',
    0xc348: 'aabaJ',
    0xc349: 'bbab99',
    0xc34a: 'bbabK',
    0xc34b: 'aaba89',
    0xc34c: 'ababK',
    0xc34d: 'ddaa89',
    0xc34e: 'babaJ',
    0xc34f: 'ccab99',
    0xc350: 'aaaaJ',
    0xc351: 'bbbb99',
    0xc352: 'bbbbK',
    0xc353: 'aaaa89',
    0xc354: 'abbbK',
    0xc355: 'ddba89',
    0xc356: 'baaaJ',
    0xc357: 'ccbb99',
    0xc358: 'aabbK',
    0xc359: 'bbaa89',
    0xc35a: 'bbaaJ',
    0xc35b: 'aabb99',
    0xc35c: 'abaaJ',
    0xc35d: 'ddab99',
    0xc35e: 'babbK',
    0xc35f: 'ccaa89',
    0xc360: 'aaabJ',
    0xc361: 'bbba99',
    0xc362: 'bbbaK',
    0xc363: 'aaab89',
    0xc364: 'abbaK',
    0xc365: 'ddbb89',
    0xc366: 'baabJ',
    0xc367: 'ccba99',
    0xc368: 'aabaK',
    0xc369: 'bbab89',
    0xc36a: 'bbabJ',
    0xc36b: 'aaba99',
    0xc36c: 'ababJ',
    0xc36d: 'ddaa99',
    0xc36e: 'babaK',
    0xc36f: 'ccab89',
    0xc370: 'aaaaK',
    0xc371: 'bbbb89',
    0xc372: 'aaaaae8',
    0xc373: 'aaaa99',
    0xc374: 'abbbJ',
    0xc375: 'ddba99',
    0xc376: 'baaaK',
    0xc377: 'ccbb89',
    0xc378: 'aabbJ',
    0xc379: 'bbaa99',
    0xc37a: 'bbaaK',
    0xc37b: 'aabb89',
    0xc37c: 'abaaK',
    0xc37d: 'aaaaaaH9',
    0xc37e: 'aaaaaab889',
    0xc37f: 'bbbbbbI9',
    0xc380: 'aaabM',
    0xc381: 'bbbi88',
    0xc382: 'bbbaL',
    0xc383: 'aaah88',
    0xc384: 'abbaL',
    0xc385: 'ddbh88',
    0xc386: 'baabM',
    0xc387: 'ccbi88',
    0xc388: 'aabaL',
    0xc389: 'bbah88',
    0xc38a: 'bbabM',
    0xc38b: 'aabi88',
    0xc38c: 'ababM',
    0xc38d: 'ddai88',
    0xc38e: 'babaL',
    0xc38f: 'ccah88',
    0xc390: 'aaaaL',
    0xc391: 'bbbh88',
    0xc392: 'bbbbM',
    0xc393: 'aaai88',
    0xc394: 'abbbM',
    0xc395: 'ddbi88',
    0xc396: 'baaaL',
    0xc397: 'ccbh88',
    0xc398: 'aabbM',
    0xc399: 'bbai88',
    0xc39a: 'bbaaL',
    0xc39b: 'aabh88',
    0xc39c: 'abaaL',
    0xc39d: 'aaaabaI9',
    0xc39e: 'babbM',
    0xc39f: 'bbbbabH9',
    0xc3a0: 'aaabL',
    0xc3a1: 'bbbi98',
    0xc3a2: 'bbbaM',
    0xc3a3: 'aaah98',
    0xc3a4: 'abbaM',
    0xc3a5: 'ddbh98',
    0xc3a6: 'baabL',
    0xc3a7: 'ccbi98',
    0xc3a8: 'aabaM',
    0xc3a9: 'bbah98',
    0xc3aa: 'bbabL',
    0xc3ab: 'aabi98',
    0xc3ac: 'ababL',
    0xc3ad: 'ddai98',
    0xc3ae: 'babaM',
    0xc3af: 'ccah98',
    0xc3b0: 'aaaaM',
    0xc3b1: 'bbbh98',
    0xc3b2: 'aaaaad9',
    0xc3b3: 'aaai98',
    0xc3b4: 'abbbL',
    0xc3b5: 'ddbi98',
    0xc3b6: 'baaaM',
    0xc3b7: 'ccbh98',
    0xc3b8: 'aabbL',
    0xc3b9: 'bbai98',
    0xc3ba: 'bbaaM',
    0xc3bb: 'aabh98',
    0xc3bc: 'abaaM',
    0xc3bd: 'aaaaabH9',
    0xc3be: 'babbL',
    0xc3bf: 'bbbbbaI9',
    0xc3c0: 'aaabO',
    0xc3c1: 'bbbi89',
    0xc3c2: 'bbbaN',
    0xc3c3: 'aaah89',
    0xc3c4: 'abbaN',
    0xc3c5: 'ddbh89',
    0xc3c6: 'baabO',
    0xc3c7: 'ccbi89',
    0xc3c8: 'aabaN',
    0xc3c9: 'bbah89',
    0xc3ca: 'bbabO',
    0xc3cb: 'aabi89',
    0xc3cc: 'ababO',
    0xc3cd: 'ddai89',
    0xc3ce: 'babaN',
    0xc3cf: 'ccah89',
    0xc3d0: 'aaaaN',
    0xc3d1: 'bbbh89',
    0xc3d2: 'bbbbO',
    0xc3d3: 'aaai89',
    0xc3d4: 'abbbO',
    0xc3d5: 'ddbi89',
    0xc3d6: 'baaaN',
    0xc3d7: 'ccbh89',
    0xc3d8: 'aabbO',
    0xc3d9: 'bbai89',
    0xc3da: 'bbaaN',
    0xc3db: 'aabh89',
    0xc3dc: 'abaaN',
    0xc3dd: 'aaaabbH9',
    0xc3de: 'babbO',
    0xc3df: 'bbbbaaI9',
    0xc3e0: 'aaabN',
    0xc3e1: 'bbbi99',
    0xc3e2: 'bbbaO',
    0xc3e3: 'aaah99',
    0xc3e4: 'abbaO',
    0xc3e5: 'ddbh99',
    0xc3e6: 'baabN',
    0xc3e7: 'ccbi99',
    0xc3e8: 'aabaO',
    0xc3e9: 'bbah99',
    0xc3ea: 'bbabN',
    0xc3eb: 'aabi99',
    0xc3ec: 'ababN',
    0xc3ed: 'ddai99',
    0xc3ee: 'babaO',
    0xc3ef: 'ccah99',
    0xc3f0: 'aaaaO',
    0xc3f1: 'aaaaaaaabbba',
    0xc3f2: 'aaaaae9',
    0xc3f3: 'aaai99',
    0xc3f4: 'abbbN',
    0xc3f5: 'aaaaaaaabbbA',
    0xc3f6: 'aaaaaaaabbbY',
    0xc3f7: 'aaaaaaaabbbQ',
    0xc3f8: 'aabbN',
    0xc3f9: 'aaaaaaaabbP',
    0xc3fa: 'aaaaaaaabdc',
    0xc3fb: 'aabh99',
    0xc3fc: 'abaaO',
    0xc3fd: 'aaaaaaI9',
    0xc3fe: 'aaaaaaa989',
    0xc3ff: 'aaaaaaaaB2',
    0xc400: 'aaabq',
    0xc401: 'bbb104',
    0xc402: 'bbbap',
    0xc403: 'aaa004',
    0xc404: 'abbap',
    0xc405: 'cccccccaabP',
    0xc406: 'baabq',
    0xc407: 'ccb104',
    0xc408: 'aabap',
    0xc409: 'bba004',
    0xc40a: 'bbabq',
    0xc40b: 'aab104',
    0xc40c: 'ababq',
    0xc40d: 'ccccccccbaca',
    0xc40e: 'babap',
    0xc40f: 'cca004',
    0xc410: 'aaaap',
    0xc411: 'bbb004',
    0xc412: 'bbbbq',
    0xc413: 'aaa104',
    0xc414: 'abbbq',
    0xc415: 'ddb104',
    0xc416: 'baaap',
    0xc417: 'ccb004',
    0xc418: 'aabbq',
    0xc419: 'bba104',
    0xc41a: 'bbaap',
    0xc41b: 'aab004',
    0xc41c: 'abaap',
    0xc41d: 'aaaabaD8',
    0xc41e: 'babbq',
    0xc41f: 'bbbbabE8',
    0xc420: 'aaabp',
    0xc421: 'bbb114',
    0xc422: 'bbbaq',
    0xc423: 'aaa014',
    0xc424: 'abbaq',
    0xc425: 'ddb014',
    0xc426: 'baabp',
    0xc427: 'ccb114',
    0xc428: 'aabaq',
    0xc429: 'bba014',
    0xc42a: 'bbabp',
    0xc42b: 'aab114',
    0xc42c: 'ababp',
    0xc42d: 'dda114',
    0xc42e: 'babaq',
    0xc42f: 'cca014',
    0xc430: 'aaaaq',
    0xc431: 'bbb014',
    0xc432: 'aaaaab5',
    0xc433: 'aaa114',
    0xc434: 'abbbp',
    0xc435: 'ddb114',
    0xc436: 'baaaq',
    0xc437: 'ccb014',
    0xc438: 'aabbp',
    0xc439: 'bba114',
    0xc43a: 'bbaaq',
    0xc43b: 'aab014',
    0xc43c: 'abaaq',
    0xc43d: 'aaaaabE8',
    0xc43e: 'babbp',
    0xc43f: 'bbbbbaD8',
    0xc440: 'aaabs',
    0xc441: 'bbb105',
    0xc442: 'bbbar',
    0xc443: 'aaa005',
    0xc444: 'abbar',
    0xc445: 'ddb005',
    0xc446: 'baabs',
    0xc447: 'ccb105',
    0xc448: 'aabar',
    0xc449: 'bba005',
    0xc44a: 'bbabs',
    0xc44b: 'aab105',
    0xc44c: 'ababs',
    0xc44d: 'dda105',
    0xc44e: 'babar',
    0xc44f: 'cca005',
    0xc450: 'aaaar',
    0xc451: 'bbb005',
    0xc452: 'bbbbs',
    0xc453: 'aaa105',
    0xc454: 'abbbs',
    0xc455: 'ddb105',
    0xc456: 'baaar',
    0xc457: 'ccb005',
    0xc458: 'aabbs',
    0xc459: 'bba105',
    0xc45a: 'bbaar',
    0xc45b: 'aab005',
    0xc45c: 'abaar',
    0xc45d: 'aaaabbE8',
    0xc45e: 'babbs',
    0xc45f: 'bbbbaaD8',
    0xc460: 'aaabr',
    0xc461: 'bbb115',
    0xc462: 'bbbas',
    0xc463: 'aaa015',
    0xc464: 'abbas',
    0xc465: 'ddb015',
    0xc466: 'baabr',
    0xc467: 'ccb115',
    0xc468: 'aabas',
    0xc469: 'bba015',
    0xc46a: 'bbabr',
    0xc46b: 'aab115',
    0xc46c: 'ababr',
    0xc46d: 'dda115',
    0xc46e: 'babas',
    0xc46f: 'cca015',
    0xc470: 'aaaas',
    0xc471: 'bbb015',
    0xc472: 'aaaaaa4',
    0xc473: 'aaa115',
    0xc474: 'abbbr',
    0xc475: 'ddb115',
    0xc476: 'baaas',
    0xc477: 'ccb015',
    0xc478: 'aabbr',
    0xc479: 'bba115',
    0xc47a: 'bbaas',
    0xc47b: 'aab015',
    0xc47c: 'abaas',
    0xc47d: 'aaaaaaD8',
    0xc47e: 'aaaaaab998',
    0xc47f: 'bbbbbbE8',
    0xc480: 'aaabu',
    0xc481: 'bbb106',
    0xc482: 'bbbat',
    0xc483: 'aaa006',
    0xc484: 'abbat',
    0xc485: 'ddb006',
    0xc486: 'baabu',
    0xc487: 'ccb106',
    0xc488: 'aabat',
    0xc489: 'bba006',
    0xc48a: 'bbabu',
    0xc48b: 'aab106',
    0xc48c: 'ababu',
    0xc48d: 'dda106',
    0xc48e: 'babat',
    0xc48f: 'cca006',
    0xc490: 'aaaat',
    0xc491: 'bbb006',
    0xc492: 'bbbbu',
    0xc493: 'aaa106',
    0xc494: 'abbbu',
    0xc495: 'ddb106',
    0xc496: 'baaat',
    0xc497: 'ccb006',
    0xc498: 'aabbu',
    0xc499: 'bba106',
    0xc49a: 'bbaat',
    0xc49b: 'aab006',
    0xc49c: 'abaat',
    0xc49d: 'aaaabaE8',
    0xc49e: 'babbu',
    0xc49f: 'bbbbabD8',
    0xc4a0: 'aaabt',
    0xc4a1: 'bbb116',
    0xc4a2: 'bbbau',
    0xc4a3: 'aaa016',
    0xc4a4: 'abbau',
    0xc4a5: 'ddb016',
    0xc4a6: 'baabt',
    0xc4a7: 'ccb116',
    0xc4a8: 'aabau',
    0xc4a9: 'bba016',
    0xc4aa: 'bbabt',
    0xc4ab: 'aab116',
    0xc4ac: 'ababt',
    0xc4ad: 'dda116',
    0xc4ae: 'babau',
    0xc4af: 'cca016',
    0xc4b0: 'aaaau',
    0xc4b1: 'bbb016',
    0xc4b2: 'aaaaab4',
    0xc4b3: 'aaa116',
    0xc4b4: 'abbbt',
    0xc4b5: 'ddb116',
    0xc4b6: 'baaau',
    0xc4b7: 'ccb016',
    0xc4b8: 'aabbt',
    0xc4b9: 'bba116',
    0xc4ba: 'bbaau',
    0xc4bb: 'aab016',
    0xc4bc: 'abaau',
    0xc4bd: 'aaaaabD8',
    0xc4be: 'babbt',
    0xc4bf: 'bbbbbaE8',
    0xc4c0: 'aaabw',
    0xc4c1: 'bbb107',
    0xc4c2: 'bbbav',
    0xc4c3: 'aaa007',
    0xc4c4: 'abbav',
    0xc4c5: 'ddb007',
    0xc4c6: 'baabw',
    0xc4c7: 'ccb107',
    0xc4c8: 'aabav',
    0xc4c9: 'bba007',
    0xc4ca: 'bbabw',
    0xc4cb: 'aab107',
    0xc4cc: 'ababw',
    0xc4cd: 'dda107',
    0xc4ce: 'babav',
    0xc4cf: 'cca007',
    0xc4d0: 'aaaav',
    0xc4d1: 'bbb007',
    0xc4d2: 'bbbbw',
    0xc4d3: 'aaa107',
    0xc4d4: 'abbbw',
    0xc4d5: 'ddb107',
    0xc4d6: 'baaav',
    0xc4d7: 'ccb007',
    0xc4d8: 'aabbw',
    0xc4d9: 'bba107',
    0xc4da: 'bbaav',
    0xc4db: 'aab007',
    0xc4dc: 'abaav',
    0xc4dd: 'aaaabbD8',
    0xc4de: 'babbw',
    0xc4df: 'bbbbaaE8',
    0xc4e0: 'aaabv',
    0xc4e1: 'bbb117',
    0xc4e2: 'bbbaw',
    0xc4e3: 'aaa017',
    0xc4e4: 'abbaw',
    0xc4e5: 'ddb017',
    0xc4e6: 'baabv',
    0xc4e7: 'ccb117',
    0xc4e8: 'aabaw',
    0xc4e9: 'bba017',
    0xc4ea: 'bbabv',
    0xc4eb: 'aab117',
    0xc4ec: 'ababv',
    0xc4ed: 'dda117',
    0xc4ee: 'babaw',
    0xc4ef: 'cca017',
    0xc4f0: 'aaaaw',
    0xc4f1: 'bbb017',
    0xc4f2: 'aaaaaa5',
    0xc4f3: 'aaa117',
    0xc4f4: 'abbbv',
    0xc4f5: 'ddb117',
    0xc4f6: 'baaaw',
    0xc4f7: 'ccb017',
    0xc4f8: 'aabbv',
    0xc4f9: 'aaaaaaabbaQ',
    0xc4fa: 'aaaaaaabbaa',
    0xc4fb: 'aab017',
    0xc4fc: 'abaaw',
    0xc4fd: 'aaaaaaE8',
    0xc4fe: 'aaaaaaa898',
    0xc4ff: 'aaaaaaabB3',
    0xc500: 'aaaby',
    0xc501: 'bbb100',
    0xc502: 'bbbax',
    0xc503: 'aaa000',
    0xc504: 'abbax',
    0xc505: 'cccccccbbaQ',
    0xc506: 'baaby',
    0xc507: 'ccb100',
    0xc508: 'aabax',
    0xc509: 'bba000',
    0xc50a: 'bbaby',
    0xc50b: 'aab100',
    0xc50c: 'ababy',
    0xc50d: 'dda100',
    0xc50e: 'babax',
    0xc50f: 'cca000',
    0xc510: 'aaaax',
    0xc511: 'bbb000',
    0xc512: 'bbbby',
    0xc513: 'aaa100',
    0xc514: 'abbby',
    0xc515: 'ddb100',
    0xc516: 'baaax',
    0xc517: 'ccb000',
    0xc518: 'aabby',
    0xc519: 'bba100',
    0xc51a: 'bbaax',
    0xc51b: 'aab000',
    0xc51c: 'abaax',
    0xc51d: 'aaaabaD9',
    0xc51e: 'babby',
    0xc51f: 'bbbbabE9',
    0xc520: 'aaabx',
    0xc521: 'bbb110',
    0xc522: 'bbbay',
    0xc523: 'aaa010',
    0xc524: 'abbay',
    0xc525: 'ddb010',
    0xc526: 'baabx',
    0xc527: 'ccb110',
    0xc528: 'aabay',
    0xc529: 'bba010',
    0xc52a: 'bbabx',
    0xc52b: 'aab110',
    0xc52c: 'ababx',
    0xc52d: 'dda110',
    0xc52e: 'babay',
    0xc52f: 'cca010',
    0xc530: 'aaaay',
    0xc531: 'bbb010',
    0xc532: 'aaaaab7',
    0xc533: 'aaa110',
    0xc534: 'abbbx',
    0xc535: 'ddb110',
    0xc536: 'baaay',
    0xc537: 'ccb010',
    0xc538: 'aabbx',
    0xc539: 'bba110',
    0xc53a: 'bbaay',
    0xc53b: 'aab010',
    0xc53c: 'abaay',
    0xc53d: 'aaaaabE9',
    0xc53e: 'babbx',
    0xc53f: 'bbbbbaD9',
    0xc540: 'aaadx',
    0xc541: 'bbb101',
    0xc542: 'bbbaz',
    0xc543: 'aaa001',
    0xc544: 'abbaz',
    0xc545: 'ddb001',
    0xc546: 'baadx',
    0xc547: 'ccb101',
    0xc548: 'aabaz',
    0xc549: 'bba001',
    0xc54a: 'bbadx',
    0xc54b: 'aab101',
    0xc54c: 'abadx',
    0xc54d: 'dda101',
    0xc54e: 'babaz',
    0xc54f: 'cca001',
    0xc550: 'aaaaz',
    0xc551: 'bbb001',
    0xc552: 'aaaabb7',
    0xc553: 'aaa101',
    0xc554: 'abbdx',
    0xc555: 'ddb101',
    0xc556: 'baaaz',
    0xc557: 'ccb001',
    0xc558: 'aabdx',
    0xc559: 'bba101',
    0xc55a: 'bbaaz',
    0xc55b: 'aab001',
    0xc55c: 'abaaz',
    0xc55d: 'aaaabbE9',
    0xc55e: 'babdx',
    0xc55f: 'bbbbaaD9',
    0xc560: 'aaabz',
    0xc561: 'bbb111',
    0xc562: 'bbbcz',
    0xc563: 'aaa011',
    0xc564: 'abbcz',
    0xc565: 'ddb011',
    0xc566: 'baabz',
    0xc567: 'ccb111',
    0xc568: 'aabcz',
    0xc569: 'bba011',
    0xc56a: 'bbabz',
    0xc56b: 'aab111',
    0xc56c: 'ababz',
    0xc56d: 'dda111',
    0xc56e: 'babcz',
    0xc56f: 'cca011',
    0xc570: 'aaacz',
    0xc571: 'bbb011',
    0xc572: 'aaaaaa6',
    0xc573: 'aaa111',
    0xc574: 'abbbz',
    0xc575: 'ddb111',
    0xc576: 'baacz',
    0xc577: 'ccb011',
    0xc578: 'aabbz',
    0xc579: 'bba111',
    0xc57a: 'bbacz',
    0xc57b: 'aab011',
    0xc57c: 'abacz',
    0xc57d: 'aaaaaaD9',
    0xc57e: 'aaaaaab898',
    0xc57f: 'bbbbbbE9',
    0xc580: 'aaahx',
    0xc581: 'bbb102',
    0xc582: 'bbbix',
    0xc583: 'aaa002',
    0xc584: 'abbix',
    0xc585: 'ddb002',
    0xc586: 'baahx',
    0xc587: 'ccb102',
    0xc588: 'aabix',
    0xc589: 'bba002',
    0xc58a: 'bbahx',
    0xc58b: 'aab102',
    0xc58c: 'abahx',
    0xc58d: 'dda102',
    0xc58e: 'babix',
    0xc58f: 'cca002',
    0xc590: 'aaaix',
    0xc591: 'bbb002',
    0xc592: 'aaaaba7',
    0xc593: 'aaa102',
    0xc594: 'abbhx',
    0xc595: 'ddb102',
    0xc596: 'baaix',
    0xc597: 'ccb002',
    0xc598: 'aabhx',
    0xc599: 'bba102',
    0xc59a: 'bbaix',
    0xc59b: 'aab002',
    0xc59c: 'abaix',
    0xc59d: 'aaaabaE9',
    0xc59e: 'babhx',
    0xc59f: 'bbbbabD9',
    0xc5a0: 'aaahy',
    0xc5a1: 'bbb112',
    0xc5a2: 'bbbiy',
    0xc5a3: 'aaa012',
    0xc5a4: 'abbiy',
    0xc5a5: 'ddb012',
    0xc5a6: 'baahy',
    0xc5a7: 'ccb112',
    0xc5a8: 'aabiy',
    0xc5a9: 'bba012',
    0xc5aa: 'bbahy',
    0xc5ab: 'aab112',
    0xc5ac: 'abahy',
    0xc5ad: 'dda112',
    0xc5ae: 'babiy',
    0xc5af: 'cca012',
    0xc5b0: 'aaaiy',
    0xc5b1: 'bbb012',
    0xc5b2: 'aaaaab6',
    0xc5b3: 'aaa112',
    0xc5b4: 'abbhy',
    0xc5b5: 'ddb112',
    0xc5b6: 'baaiy',
    0xc5b7: 'ccb012',
    0xc5b8: 'aabhy',
    0xc5b9: 'bba112',
    0xc5ba: 'bbaiy',
    0xc5bb: 'aab012',
    0xc5bc: 'abaiy',
    0xc5bd: 'aaaaabD9',
    0xc5be: 'babhy',
    0xc5bf: 'bbbbbaE9',
    0xc5c0: 'aaahz',
    0xc5c1: 'bbb103',
    0xc5c2: 'bbbiz',
    0xc5c3: 'aaa003',
    0xc5c4: 'abbiz',
    0xc5c5: 'ddb003',
    0xc5c6: 'baahz',
    0xc5c7: 'ccb103',
    0xc5c8: 'aabiz',
    0xc5c9: 'bba003',
    0xc5ca: 'bbahz',
    0xc5cb: 'aab103',
    0xc5cc: 'abahz',
    0xc5cd: 'dda103',
    0xc5ce: 'babiz',
    0xc5cf: 'cca003',
    0xc5d0: 'aaaiz',
    0xc5d1: 'bbb003',
    0xc5d2: 'aaaabb6',
    0xc5d3: 'aaa103',
    0xc5d4: 'abbhz',
    0xc5d5: 'ddb103',
    0xc5d6: 'baaiz',
    0xc5d7: 'ccb003',
    0xc5d8: 'aabhz',
    0xc5d9: 'bba103',
    0xc5da: 'bbaiz',
    0xc5db: 'aab003',
    0xc5dc: 'abaiz',
    0xc5dd: 'aaaabbD9',
    0xc5de: 'babhz',
    0xc5df: 'bbbbaaE9',
    0xc5e0: 'aaajz',
    0xc5e1: 'aaaaaaaaabbaD',
    0xc5e2: 'aaaaaaaaabbaH',
    0xc5e3: 'aaa013',
    0xc5e4: 'abbkz',
    0xc5e5: 'aaaaaaaaabbaT',
    0xc5e6: 'aaaaaaaaabbaX',
    0xc5e7: 'aaaaaaaaabbiX',
    0xc5e8: 'aabkz',
    0xc5e9: 'aaaaaaaaabbad',
    0xc5ea: 'aaaaaaaaabbah',
    0xc5eb: 'aab113',
    0xc5ec: 'abajz',
    0xc5ed: 'aaaaaaaaabbat',
    0xc5ee: 'aaaaaaaaabbax',
    0xc5ef: 'aaaaaaaaabbix',
    0xc5f0: 'aaakz',
    0xc5f1: 'aaaaaaaaabba',
    0xc5f2: 'aaaaaa7',
    0xc5f3: 'aaa113',
    0xc5f4: 'abbjz',
    0xc5f5: 'aaaaaaaaabbA',
    0xc5f6: 'aaaaaaaaabbY',
    0xc5f7: 'aaaaaaaaabbQ',
    0xc5f8: 'aabjz',
    0xc5f9: 'aaaaaaaaabP',
    0xc5fa: 'aaaaaaaaadc',
    0xc5fb: 'aab013',
    0xc5fc: 'abakz',
    0xc5fd: 'aaaaaaE9',
    0xc5fe: 'aaaaaaa998',
    0xc5ff: 'aaaaaaaaA2',
    0xc600: 'aaaba',
    0xc601: 'bbb188',
    0xc602: 'bbbca',
    0xc603: 'aaa088',
    0xc604: 'abbca',
    0xc605: 'cccccccabaQ',
    0xc606: 'baaba',
    0xc607: 'ccb188',
    0xc608: 'aabca',
    0xc609: 'bba088',
    0xc60a: 'bbaba',
    0xc60b: 'aab188',
    0xc60c: 'ababa',
    0xc60d: 'ccccccccabba',
    0xc60e: 'babca',
    0xc60f: 'cca088',
    0xc610: 'aaaca',
    0xc611: 'bbb088',
    0xc612: 'bbbba',
    0xc613: 'aaa188',
    0xc614: 'abbba',
    0xc615: 'cccccccccaabe',
    0xc616: 'baaca',
    0xc617: 'ccb088',
    0xc618: 'aabba',
    0xc619: 'bba188',
    0xc61a: 'bbaca',
    0xc61b: 'aab088',
    0xc61c: 'abaca',
    0xc61d: 'aaaabaB9',
    0xc61e: 'babba',
    0xc61f: 'bbbbabA8',
    0xc620: 'aaadc',
    0xc621: 'bbb198',
    0xc622: 'bbbaa',
    0xc623: 'aaa098',
    0xc624: 'abbaa',
    0xc625: 'ccccccccccbabe',
    0xc626: 'baadc',
    0xc627: 'ccb198',
    0xc628: 'aabaa',
    0xc629: 'bba098',
    0xc62a: 'bbadc',
    0xc62b: 'aab198',
    0xc62c: 'abadc',
    0xc62d: 'ccccccccccbabu',
    0xc62e: 'babaa',
    0xc62f: 'cca098',
    0xc630: 'aaaaa',
    0xc631: 'bbb098',
    0xc632: 'aaaaab1',
    0xc633: 'aaa198',
    0xc634: 'abbdc',
    0xc635: 'ccccccccccbabE',
    0xc636: 'baaaa',
    0xc637: 'ccb098',
    0xc638: 'aabdc',
    0xc639: 'bba198',
    0xc63a: 'bbaaa',
    0xc63b: 'aab098',
    0xc63c: 'abaaa',
    0xc63d: 'aaaaabA8',
    0xc63e: 'babdc',
    0xc63f: 'bbbbbaB9',
    0xc640: 'aaabc',
    0xc641: 'bbb189',
    0xc642: 'bbbab',
    0xc643: 'aaa089',
    0xc644: 'abbab',
    0xc645: 'ddb089',
    0xc646: 'baabc',
    0xc647: 'ccb189',
    0xc648: 'aabab',
    0xc649: 'bba089',
    0xc64a: 'bbabc',
    0xc64b: 'aab189',
    0xc64c: 'ababc',
    0xc64d: 'dda189',
    0xc64e: 'babab',
    0xc64f: 'cca089',
    0xc650: 'aaaab',
    0xc651: 'bbb089',
    0xc652: 'bbbbc',
    0xc653: 'aaa189',
    0xc654: 'abbbc',
    0xc655: 'ddb189',
    0xc656: 'baaab',
    0xc657: 'ccb089',
    0xc658: 'aabbc',
    0xc659: 'bba189',
    0xc65a: 'bbaab',
    0xc65b: 'aab089',
    0xc65c: 'abaab',
    0xc65d: 'aaaabbA8',
    0xc65e: 'babbc',
    0xc65f: 'bbbbaaB9',
    0xc660: 'aaabb',
    0xc661: 'bbb199',
    0xc662: 'bbbac',
    0xc663: 'aaa099',
    0xc664: 'abbac',
    0xc665: 'ddb099',
    0xc666: 'baabb',
    0xc667: 'ccb199',
    0xc668: 'aabac',
    0xc669: 'bba099',
    0xc66a: 'bbabb',
    0xc66b: 'aab199',
    0xc66c: 'ababb',
    0xc66d: 'dda199',
    0xc66e: 'babac',
    0xc66f: 'cca099',
    0xc670: 'aaaac',
    0xc671: 'bbb099',
    0xc672: 'aaaaaa0',
    0xc673: 'aaa199',
    0xc674: 'abbbb',
    0xc675: 'ddb199',
    0xc676: 'baaac',
    0xc677: 'ccb099',
    0xc678: 'aabbb',
    0xc679: 'bba199',
    0xc67a: 'bbaac',
    0xc67b: 'aab099',
    0xc67c: 'abaac',
    0xc67d: 'aaaaaaB9',
    0xc67e: 'aaaaaab988',
    0xc67f: 'bbbbbbA8',
    0xc680: 'aaabe',
    0xc681: 'bbb988',
    0xc682: 'bbbad',
    0xc683: 'aaa888',
    0xc684: 'abbad',
    0xc685: 'ddb888',
    0xc686: 'baabe',
    0xc687: 'ccb988',
    0xc688: 'aabad',
    0xc689: 'bba888',
    0xc68a: 'bbabe',
    0xc68b: 'aab988',
    0xc68c: 'ababe',
    0xc68d: 'dda988',
    0xc68e: 'babad',
    0xc68f: 'cca888',
    0xc690: 'aaaad',
    0xc691: 'bbb888',
    0xc692: 'bbbbe',
    0xc693: 'aaa988',
    0xc694: 'abbbe',
    0xc695: 'ddb988',
    0xc696: 'baaad',
    0xc697: 'ccb888',
    0xc698: 'aabbe',
    0xc699: 'bba988',
    0xc69a: 'bbaad',
    0xc69b: 'aab888',
    0xc69c: 'abaad',
    0xc69d: 'aaaabaA8',
    0xc69e: 'babbe',
    0xc69f: 'bbbbabB9',
    0xc6a0: 'aaabd',
    0xc6a1: 'bbb998',
    0xc6a2: 'bbbae',
    0xc6a3: 'aaa898',
    0xc6a4: 'abbae',
    0xc6a5: 'ddb898',
    0xc6a6: 'baabd',
    0xc6a7: 'ccb998',
    0xc6a8: 'aabae',
    0xc6a9: 'bba898',
    0xc6aa: 'bbabd',
    0xc6ab: 'aab998',
    0xc6ac: 'ababd',
    0xc6ad: 'dda998',
    0xc6ae: 'babae',
    0xc6af: 'cca898',
    0xc6b0: 'aaaae',
    0xc6b1: 'bbb898',
    0xc6b2: 'aaaaab0',
    0xc6b3: 'aaa998',
    0xc6b4: 'abbbd',
    0xc6b5: 'ddb998',
    0xc6b6: 'baaae',
    0xc6b7: 'ccb898',
    0xc6b8: 'aabbd',
    0xc6b9: 'bba998',
    0xc6ba: 'bbaae',
    0xc6bb: 'aab898',
    0xc6bc: 'abaae',
    0xc6bd: 'aaaaabB9',
    0xc6be: 'babbd',
    0xc6bf: 'bbbbbaA8',
    0xc6c0: 'aaabg',
    0xc6c1: 'bbb989',
    0xc6c2: 'bbbaf',
    0xc6c3: 'aaa889',
    0xc6c4: 'abbaf',
    0xc6c5: 'ddb889',
    0xc6c6: 'baabg',
    0xc6c7: 'ccb989',
    0xc6c8: 'aabaf',
    0xc6c9: 'bba889',
    0xc6ca: 'bbabg',
    0xc6cb: 'aab989',
    0xc6cc: 'ababg',
    0xc6cd: 'dda989',
    0xc6ce: 'babaf',
    0xc6cf: 'cca889',
    0xc6d0: 'aaaaf',
    0xc6d1: 'bbb889',
    0xc6d2: 'bbbbg',
    0xc6d3: 'aaa989',
    0xc6d4: 'abbbg',
    0xc6d5: 'ddb989',
    0xc6d6: 'baaaf',
    0xc6d7: 'ccb889',
    0xc6d8: 'aabbg',
    0xc6d9: 'bba989',
    0xc6da: 'bbaaf',
    0xc6db: 'aab889',
    0xc6dc: 'abaaf',
    0xc6dd: 'aaaabbB9',
    0xc6de: 'babbg',
    0xc6df: 'bbbbaaA8',
    0xc6e0: 'aaabf',
    0xc6e1: 'bbb999',
    0xc6e2: 'bbbag',
    0xc6e3: 'aaa899',
    0xc6e4: 'abbag',
    0xc6e5: 'ddb899',
    0xc6e6: 'baabf',
    0xc6e7: 'ccb999',
    0xc6e8: 'aabag',
    0xc6e9: 'bba899',
    0xc6ea: 'bbabf',
    0xc6eb: 'aab999',
    0xc6ec: 'ababf',
    0xc6ed: 'dda999',
    0xc6ee: 'babag',
    0xc6ef: 'cca899',
    0xc6f0: 'aaaag',
    0xc6f1: 'bbb899',
    0xc6f2: 'aaaaaa1',
    0xc6f3: 'aaa999',
    0xc6f4: 'abbbf',
    0xc6f5: 'ddb999',
    0xc6f6: 'baaag',
    0xc6f7: 'ccb899',
    0xc6f8: 'aabbf',
    0xc6f9: 'aaaaaaababP',
    0xc6fa: 'aaaaaaabadc',
    0xc6fb: 'aab899',
    0xc6fc: 'abaag',
    0xc6fd: 'aaaaaaA8',
    0xc6fe: 'aaaaaaa888',
    0xc6ff: 'aaaaaaabA2',
    0xc700: 'aaabi',
    0xc701: 'bbb108',
    0xc702: 'bbbah',
    0xc703: 'aaa008',
    0xc704: 'abbah',
    0xc705: 'cccccccbabP',
    0xc706: 'baabi',
    0xc707: 'ccb108',
    0xc708: 'aabah',
    0xc709: 'bba008',
    0xc70a: 'bbabi',
    0xc70b: 'aab108',
    0xc70c: 'ababi',
    0xc70d: 'dda108',
    0xc70e: 'babah',
    0xc70f: 'cca008',
    0xc710: 'aaaah',
    0xc711: 'bbb008',
    0xc712: 'bbbbi',
    0xc713: 'aaa108',
    0xc714: 'abbbi',
    0xc715: 'ddb108',
    0xc716: 'baaah',
    0xc717: 'ccb008',
    0xc718: 'aabbi',
    0xc719: 'bba108',
    0xc71a: 'bbaah',
    0xc71b: 'aab008',
    0xc71c: 'abaah',
    0xc71d: 'aaaabaB8',
    0xc71e: 'babbi',
    0xc71f: 'bbbbabA9',
    0xc720: 'aaabh',
    0xc721: 'bbb118',
    0xc722: 'bbbai',
    0xc723: 'aaa018',
    0xc724: 'abbai',
    0xc725: 'ddb018',
    0xc726: 'baabh',
    0xc727: 'ccb118',
    0xc728: 'aabai',
    0xc729: 'bba018',
    0xc72a: 'bbabh',
    0xc72b: 'aab118',
    0xc72c: 'ababh',
    0xc72d: 'dda118',
    0xc72e: 'babai',
    0xc72f: 'cca018',
    0xc730: 'aaaai',
    0xc731: 'bbb018',
    0xc732: 'aaaaab3',
    0xc733: 'aaa118',
    0xc734: 'abbbh',
    0xc735: 'ddb118',
    0xc736: 'baaai',
    0xc737: 'ccb018',
    0xc738: 'aabbh',
    0xc739: 'bba118',
    0xc73a: 'bbaai',
    0xc73b: 'aab018',
    0xc73c: 'abaai',
    0xc73d: 'aaaaabA9',
    0xc73e: 'babbh',
    0xc73f: 'bbbbbaB8',
    0xc740: 'aaabk',
    0xc741: 'bbb109',
    0xc742: 'bbbaj',
    0xc743: 'aaa009',
    0xc744: 'abbaj',
    0xc745: 'ddb009',
    0xc746: 'baabk',
    0xc747: 'ccb109',
    0xc748: 'aabaj',
    0xc749: 'bba009',
    0xc74a: 'bbabk',
    0xc74b: 'aab109',
    0xc74c: 'ababk',
    0xc74d: 'dda109',
    0xc74e: 'babaj',
    0xc74f: 'cca009',
    0xc750: 'aaaaj',
    0xc751: 'bbb009',
    0xc752: 'bbbbk',
    0xc753: 'aaa109',
    0xc754: 'abbbk',
    0xc755: 'ddb109',
    0xc756: 'baaaj',
    0xc757: 'ccb009',
    0xc758: 'aabbk',
    0xc759: 'bba109',
    0xc75a: 'bbaaj',
    0xc75b: 'aab009',
    0xc75c: 'abaaj',
    0xc75d: 'aaaabbA9',
    0xc75e: 'babbk',
    0xc75f: 'bbbbaaB8',
    0xc760: 'aaabj',
    0xc761: 'bbb119',
    0xc762: 'bbbak',
    0xc763: 'aaa019',
    0xc764: 'abbak',
    0xc765: 'ddb019',
    0xc766: 'baabj',
    0xc767: 'ccb119',
    0xc768: 'aabak',
    0xc769: 'bba019',
    0xc76a: 'bbabj',
    0xc76b: 'aab119',
    0xc76c: 'ababj',
    0xc76d: 'dda119',
    0xc76e: 'babak',
    0xc76f: 'cca019',
    0xc770: 'aaaak',
    0xc771: 'bbb019',
    0xc772: 'aaaaaa2',
    0xc773: 'aaa119',
    0xc774: 'abbbj',
    0xc775: 'ddb119',
    0xc776: 'baaak',
    0xc777: 'ccb019',
    0xc778: 'aabbj',
    0xc779: 'bba119',
    0xc77a: 'bbaak',
    0xc77b: 'aab019',
    0xc77c: 'abaak',
    0xc77d: 'aaaaaaB8',
    0xc77e: 'aaaaaab888',
    0xc77f: 'bbbbbbA9',
    0xc780: 'aaabm',
    0xc781: 'bbb148',
    0xc782: 'bbbal',
    0xc783: 'aaa048',
    0xc784: 'abbal',
    0xc785: 'ddb048',
    0xc786: 'baabm',
    0xc787: 'ccb148',
    0xc788: 'aabal',
    0xc789: 'bba048',
    0xc78a: 'bbabm',
    0xc78b: 'aab148',
    0xc78c: 'ababm',
    0xc78d: 'dda148',
    0xc78e: 'babal',
    0xc78f: 'cca048',
    0xc790: 'aaaal',
    0xc791: 'bbb048',
    0xc792: 'bbbbm',
    0xc793: 'aaa148',
    0xc794: 'abbbm',
    0xc795: 'ddb148',
    0xc796: 'baaal',
    0xc797: 'ccb048',
    0xc798: 'aabbm',
    0xc799: 'bba148',
    0xc79a: 'bbaal',
    0xc79b: 'aab048',
    0xc79c: 'abaal',
    0xc79d: 'aaaabaA9',
    0xc79e: 'babbm',
    0xc79f: 'bbbbabB8',
    0xc7a0: 'aaabl',
    0xc7a1: 'bbb158',
    0xc7a2: 'bbbam',
    0xc7a3: 'aaa058',
    0xc7a4: 'abbam',
    0xc7a5: 'ddb058',
    0xc7a6: 'baabl',
    0xc7a7: 'ccb158',
    0xc7a8: 'aabam',
    0xc7a9: 'bba058',
    0xc7aa: 'bbabl',
    0xc7ab: 'aab158',
    0xc7ac: 'ababl',
    0xc7ad: 'dda158',
    0xc7ae: 'babam',
    0xc7af: 'cca058',
    0xc7b0: 'aaaam',
    0xc7b1: 'bbb058',
    0xc7b2: 'aaaaab2',
    0xc7b3: 'aaa158',
    0xc7b4: 'abbbl',
    0xc7b5: 'ddb158',
    0xc7b6: 'baaam',
    0xc7b7: 'ccb058',
    0xc7b8: 'aabbl',
    0xc7b9: 'bba158',
    0xc7ba: 'bbaam',
    0xc7bb: 'aab058',
    0xc7bc: 'abaam',
    0xc7bd: 'aaaaabB8',
    0xc7be: 'babbl',
    0xc7bf: 'bbbbbaA9',
    0xc7c0: 'aaabo',
    0xc7c1: 'bbb149',
    0xc7c2: 'bbban',
    0xc7c3: 'aaa049',
    0xc7c4: 'abban',
    0xc7c5: 'ddb049',
    0xc7c6: 'baabo',
    0xc7c7: 'ccb149',
    0xc7c8: 'aaban',
    0xc7c9: 'bba049',
    0xc7ca: 'bbabo',
    0xc7cb: 'aab149',
    0xc7cc: 'ababo',
    0xc7cd: 'dda149',
    0xc7ce: 'baban',
    0xc7cf: 'cca049',
    0xc7d0: 'aaaan',
    0xc7d1: 'bbb049',
    0xc7d2: 'bbbbo',
    0xc7d3: 'aaa149',
    0xc7d4: 'abbbo',
    0xc7d5: 'ddb149',
    0xc7d6: 'baaan',
    0xc7d7: 'ccb049',
    0xc7d8: 'aabbo',
    0xc7d9: 'bba149',
    0xc7da: 'bbaan',
    0xc7db: 'aab049',
    0xc7dc: 'abaan',
    0xc7dd: 'aaaabbB8',
    0xc7de: 'babbo',
    0xc7df: 'bbbbaaA9',
    0xc7e0: 'aaabn',
    0xc7e1: 'bbb159',
    0xc7e2: 'bbbao',
    0xc7e3: 'aaa059',
    0xc7e4: 'abbao',
    0xc7e5: 'ddb059',
    0xc7e6: 'baabn',
    0xc7e7: 'ccb159',
    0xc7e8: 'aabao',
    0xc7e9: 'bba059',
    0xc7ea: 'bbabn',
    0xc7eb: 'aab159',
    0xc7ec: 'ababn',
    0xc7ed: 'dda159',
    0xc7ee: 'babao',
    0xc7ef: 'cca059',
    0xc7f0: 'aaaao',
    0xc7f1: 'aaaaaaaabaca',
    0xc7f2: 'aaaaaa3',
    0xc7f3: 'aaa159',
    0xc7f4: 'abbbn',
    0xc7f5: 'aaaaaaaabacA',
    0xc7f6: 'aaaaaaaabaaX',
    0xc7f7: 'aaaaaaaabaaP',
    0xc7f8: 'aabbn',
    0xc7f9: 'aaaaaaaabaQ',
    0xc7fa: 'aaaaaaaabaa',
    0xc7fb: 'aab059',
    0xc7fc: 'abaao',
    0xc7fd: 'aaaaaaA9',
    0xc7fe: 'aaaaaaa988',
    0xc7ff: 'aaaaaaaaB3',
    0xc800: 'aaa08',
    0xc801: 'aacA',
    0xc802: 'bbb18',
    0xc803: 'bbbA',
    0xc804: 'abb18',
    0xc805: 'abbA',
    0xc806: 'baa08',
    0xc807: 'bacA',
    0xc808: 'aab18',
    0xc809: 'aabA',
    0xc80a: 'bba08',
    0xc80b: 'bbcA',
    0xc80c: 'aba08',
    0xc80d: 'abcA',
    0xc80e: 'bab18',
    0xc80f: 'babA',
    0xc810: 'aaa18',
    0xc811: 'aaaA',
    0xc812: 'aaaabI8',
    0xc813: 'bbdC',
    0xc814: 'abb08',
    0xc815: 'abdC',
    0xc816: 'baa18',
    0xc817: 'baaA',
    0xc818: 'aab08',
    0xc819: 'aadC',
    0xc81a: 'bba18',
    0xc81b: 'bbaA',
    0xc81c: 'aba18',
    0xc81d: 'abaA',
    0xc81e: 'bab08',
    0xc81f: 'badC',
    0xc820: 'aaa09',
    0xc821: 'aaaB',
    0xc822: 'bbb19',
    0xc823: 'bbbC',
    0xc824: 'abb19',
    0xc825: 'abbC',
    0xc826: 'baa09',
    0xc827: 'baaB',
    0xc828: 'aab19',
    0xc829: 'aabC',
    0xc82a: 'bba09',
    0xc82b: 'bbaB',
    0xc82c: 'aba09',
    0xc82d: 'abaB',
    0xc82e: 'bab19',
    0xc82f: 'babC',
    0xc830: 'aaa19',
    0xc831: 'aaaC',
    0xc832: 'aaaaaH8',
    0xc833: 'bbbB',
    0xc834: 'abb09',
    0xc835: 'abbB',
    0xc836: 'baa19',
    0xc837: 'baaC',
    0xc838: 'aab09',
    0xc839: 'aabB',
    0xc83a: 'bba19',
    0xc83b: 'bbaC',
    0xc83c: 'aba19',
    0xc83d: 'abaC',
    0xc83e: 'bab09',
    0xc83f: 'babB',
    0xc840: 'aaa48',
    0xc841: 'aaaD',
    0xc842: 'bbb58',
    0xc843: 'bbbE',
    0xc844: 'abb58',
    0xc845: 'abbE',
    0xc846: 'baa48',
    0xc847: 'baaD',
    0xc848: 'aab58',
    0xc849: 'aabE',
    0xc84a: 'bba48',
    0xc84b: 'bbaD',
    0xc84c: 'aba48',
    0xc84d: 'abaD',
    0xc84e: 'bab58',
    0xc84f: 'babE',
    0xc850: 'aaa58',
    0xc851: 'aaaE',
    0xc852: 'aaaabH8',
    0xc853: 'bbbD',
    0xc854: 'abb48',
    0xc855: 'abbD',
    0xc856: 'baa58',
    0xc857: 'baaE',
    0xc858: 'aab48',
    0xc859: 'aabD',
    0xc85a: 'bba58',
    0xc85b: 'bbaE',
    0xc85c: 'aba58',
    0xc85d: 'abaE',
    0xc85e: 'bab48',
    0xc85f: 'babD',
    0xc860: 'aaa49',
    0xc861: 'aaaF',
    0xc862: 'bbb59',
    0xc863: 'bbbG',
    0xc864: 'abb59',
    0xc865: 'abbG',
    0xc866: 'baa49',
    0xc867: 'baaF',
    0xc868: 'aab59',
    0xc869: 'aabG',
    0xc86a: 'bba49',
    0xc86b: 'bbaF',
    0xc86c: 'aba49',
    0xc86d: 'abaF',
    0xc86e: 'bab59',
    0xc86f: 'babG',
    0xc870: 'aaa59',
    0xc871: 'aaaG',
    0xc872: 'aaaaaI8',
    0xc873: 'bbbF',
    0xc874: 'abb49',
    0xc875: 'abbF',
    0xc876: 'baa59',
    0xc877: 'baaG',
    0xc878: 'aab49',
    0xc879: 'aabF',
    0xc87a: 'bba59',
    0xc87b: 'bbaG',
    0xc87c: 'aba59',
    0xc87d: 'abaG',
    0xc87e: 'aaaaaab199',
    0xc87f: 'babF',
    0xc880: 'aaa88',
    0xc881: 'aaaH',
    0xc882: 'bbb98',
    0xc883: 'bbbI',
    0xc884: 'abb98',
    0xc885: 'abbI',
    0xc886: 'baa88',
    0xc887: 'baaH',
    0xc888: 'aab98',
    0xc889: 'aabI',
    0xc88a: 'bba88',
    0xc88b: 'bbaH',
    0xc88c: 'aba88',
    0xc88d: 'abaH',
    0xc88e: 'bab98',
    0xc88f: 'babI',
    0xc890: 'aaa98',
    0xc891: 'aaaI',
    0xc892: 'aaaabI9',
    0xc893: 'bbbH',
    0xc894: 'abb88',
    0xc895: 'abbH',
    0xc896: 'baa98',
    0xc897: 'baaI',
    0xc898: 'aab88',
    0xc899: 'aabH',
    0xc89a: 'bba98',
    0xc89b: 'bbaI',
    0xc89c: 'aba98',
    0xc89d: 'abaI',
    0xc89e: 'bab88',
    0xc89f: 'babH',
    0xc8a0: 'aaa89',
    0xc8a1: 'aaaJ',
    0xc8a2: 'bbb99',
    0xc8a3: 'bbbK',
    0xc8a4: 'abb99',
    0xc8a5: 'abbK',
    0xc8a6: 'baa89',
    0xc8a7: 'baaJ',
    0xc8a8: 'aab99',
    0xc8a9: 'aabK',
    0xc8aa: 'bba89',
    0xc8ab: 'bbaJ',
    0xc8ac: 'aba89',
    0xc8ad: 'abaJ',
    0xc8ae: 'bab99',
    0xc8af: 'babK',
    0xc8b0: 'aaa99',
    0xc8b1: 'aaaK',
    0xc8b2: 'aaaaaH9',
    0xc8b3: 'bbbJ',
    0xc8b4: 'abb89',
    0xc8b5: 'abbJ',
    0xc8b6: 'baa99',
    0xc8b7: 'baaK',
    0xc8b8: 'aab89',
    0xc8b9: 'aabJ',
    0xc8ba: 'bba99',
    0xc8bb: 'bbaK',
    0xc8bc: 'aba99',
    0xc8bd: 'abaK',
    0xc8be: 'bab89',
    0xc8bf: 'babJ',
    0xc8c0: 'aai88',
    0xc8c1: 'aaaL',
    0xc8c2: 'aaabaI9',
    0xc8c3: 'bbbM',
    0xc8c4: 'abh88',
    0xc8c5: 'abbM',
    0xc8c6: 'bai88',
    0xc8c7: 'baaL',
    0xc8c8: 'aah88',
    0xc8c9: 'aabM',
    0xc8ca: 'bbi88',
    0xc8cb: 'bbaL',
    0xc8cc: 'abi88',
    0xc8cd: 'abaL',
    0xc8ce: 'bah88',
    0xc8cf: 'babM',
    0xc8d0: 'aai98',
    0xc8d1: 'aaaM',
    0xc8d2: 'aaaabH9',
    0xc8d3: 'bbbL',
    0xc8d4: 'abh98',
    0xc8d5: 'abbL',
    0xc8d6: 'bai98',
    0xc8d7: 'baaM',
    0xc8d8: 'aah98',
    0xc8d9: 'aabL',
    0xc8da: 'bbi98',
    0xc8db: 'bbaM',
    0xc8dc: 'abi98',
    0xc8dd: 'abaM',
    0xc8de: 'bah98',
    0xc8df: 'babL',
    0xc8e0: 'aai89',
    0xc8e1: 'aaaN',
    0xc8e2: 'aaabbH9',
    0xc8e3: 'bbbO',
    0xc8e4: 'abh89',
    0xc8e5: 'abbO',
    0xc8e6: 'bai89',
    0xc8e7: 'baaN',
    0xc8e8: 'aah89',
    0xc8e9: 'aabO',
    0xc8ea: 'bbi89',
    0xc8eb: 'bbaN',
    0xc8ec: 'abi89',
    0xc8ed: 'abaN',
    0xc8ee: 'bah89',
    0xc8ef: 'babO',
    0xc8f0: 'aai99',
    0xc8f1: 'aaaO',
    0xc8f2: 'aaaaaI9',
    0xc8f3: 'bbbN',
    0xc8f4: 'abh99',
    0xc8f5: 'abbN',
    0xc8f6: 'bai99',
    0xc8f7: 'baaO',
    0xc8f8: 'aah99',
    0xc8f9: 'aabN',
    0xc8fa: 'aaaaaaabbba',
    0xc8fb: 'bbaO',
    0xc8fc: 'abi99',
    0xc8fd: 'abaO',
    0xc8fe: 'aaaaaaa099',
    0xc8ff: 'babN',
    0xc900: 'aaa00',
    0xc901: 'aaaP',
    0xc902: 'bbb10',
    0xc903: 'bbbQ',
    0xc904: 'abb10',
    0xc905: 'abbQ',
    0xc906: 'baa00',
    0xc907: 'baaP',
    0xc908: 'aab10',
    0xc909: 'aabQ',
    0xc90a: 'bba00',
    0xc90b: 'bbaP',
    0xc90c: 'aba00',
    0xc90d: 'abaP',
    0xc90e: 'bab10',
    0xc90f: 'babQ',
    0xc910: 'aaa10',
    0xc911: 'aaaQ',
    0xc912: 'aaaabM8',
    0xc913: 'bbbP',
    0xc914: 'abb00',
    0xc915: 'abbP',
    0xc916: 'baa10',
    0xc917: 'baaQ',
    0xc918: 'aab00',
    0xc919: 'aabP',
    0xc91a: 'bba10',
    0xc91b: 'bbaQ',
    0xc91c: 'aba10',
    0xc91d: 'abaQ',
    0xc91e: 'bab00',
    0xc91f: 'babP',
    0xc920: 'aaa01',
    0xc921: 'aaaR',
    0xc922: 'bbb11',
    0xc923: 'bbbS',
    0xc924: 'abb11',
    0xc925: 'abbS',
    0xc926: 'baa01',
    0xc927: 'baaR',
    0xc928: 'aab11',
    0xc929: 'aabS',
    0xc92a: 'bba01',
    0xc92b: 'bbaR',
    0xc92c: 'aba01',
    0xc92d: 'abaR',
    0xc92e: 'bab11',
    0xc92f: 'babS',
    0xc930: 'aaa11',
    0xc931: 'aaaS',
    0xc932: 'aaaaaL8',
    0xc933: 'bbbR',
    0xc934: 'abb01',
    0xc935: 'abbR',
    0xc936: 'baa11',
    0xc937: 'baaS',
    0xc938: 'aab01',
    0xc939: 'aabR',
    0xc93a: 'bba11',
    0xc93b: 'bbaS',
    0xc93c: 'aba11',
    0xc93d: 'abaS',
    0xc93e: 'bab01',
    0xc93f: 'babR',
    0xc940: 'aaa02',
    0xc941: 'aaaT',
    0xc942: 'bbb12',
    0xc943: 'bbbU',
    0xc944: 'abb12',
    0xc945: 'abbU',
    0xc946: 'baa02',
    0xc947: 'baaT',
    0xc948: 'aab12',
    0xc949: 'aabU',
    0xc94a: 'bba02',
    0xc94b: 'bbaT',
    0xc94c: 'aba02',
    0xc94d: 'abaT',
    0xc94e: 'bab12',
    0xc94f: 'babU',
    0xc950: 'aaa12',
    0xc951: 'aaaU',
    0xc952: 'aaaabL8',
    0xc953: 'bbbT',
    0xc954: 'abb02',
    0xc955: 'abbT',
    0xc956: 'baa12',
    0xc957: 'baaU',
    0xc958: 'aab02',
    0xc959: 'aabT',
    0xc95a: 'bba12',
    0xc95b: 'bbaU',
    0xc95c: 'aba12',
    0xc95d: 'abaU',
    0xc95e: 'bab02',
    0xc95f: 'babT',
    0xc960: 'aaa03',
    0xc961: 'aaaV',
    0xc962: 'bbb13',
    0xc963: 'bbbW',
    0xc964: 'abb13',
    0xc965: 'abbW',
    0xc966: 'baa03',
    0xc967: 'baaV',
    0xc968: 'aab13',
    0xc969: 'aabW',
    0xc96a: 'bba03',
    0xc96b: 'bbaV',
    0xc96c: 'aba03',
    0xc96d: 'abaV',
    0xc96e: 'bab13',
    0xc96f: 'babW',
    0xc970: 'aaa13',
    0xc971: 'aaaW',
    0xc972: 'aaaaaM8',
    0xc973: 'bbbV',
    0xc974: 'abb03',
    0xc975: 'abbV',
    0xc976: 'baa13',
    0xc977: 'baaW',
    0xc978: 'aab03',
    0xc979: 'aabV',
    0xc97a: 'bba13',
    0xc97b: 'bbaW',
    0xc97c: 'aba13',
    0xc97d: 'abaW',
    0xc97e: 'aaaaaab099',
    0xc97f: 'babV',
    0xc980: 'aaa04',
    0xc981: 'aaaX',
    0xc982: 'bbb14',
    0xc983: 'bbbY',
    0xc984: 'abb14',
    0xc985: 'abbY',
    0xc986: 'baa04',
    0xc987: 'baaX',
    0xc988: 'aab14',
    0xc989: 'aabY',
    0xc98a: 'bba04',
    0xc98b: 'bbaX',
    0xc98c: 'aba04',
    0xc98d: 'abaX',
    0xc98e: 'bab14',
    0xc98f: 'babY',
    0xc990: 'aaa14',
    0xc991: 'aaaY',
    0xc992: 'aaaabM9',
    0xc993: 'bbbX',
    0xc994: 'abb04',
    0xc995: 'abbX',
    0xc996: 'baa14',
    0xc997: 'baaY',
    0xc998: 'aab04',
    0xc999: 'aabX',
    0xc99a: 'bba14',
    0xc99b: 'bbaY',
    0xc99c: 'aba14',
    0xc99d: 'abaY',
    0xc99e: 'bab04',
    0xc99f: 'babX',
    0xc9a0: 'aaa05',
    0xc9a1: 'aaaZ',
    0xc9a2: 'bbb15',
    0xc9a3: 'bbdX',
    0xc9a4: 'abb15',
    0xc9a5: 'abdX',
    0xc9a6: 'baa05',
    0xc9a7: 'baaZ',
    0xc9a8: 'aab15',
    0xc9a9: 'aadX',
    0xc9aa: 'bba05',
    0xc9ab: 'bbaZ',
    0xc9ac: 'aba05',
    0xc9ad: 'abaZ',
    0xc9ae: 'bab15',
    0xc9af: 'badX',
    0xc9b0: 'aaa15',
    0xc9b1: 'aacZ',
    0xc9b2: 'aaaaaL9',
    0xc9b3: 'bbbZ',
    0xc9b4: 'abb05',
    0xc9b5: 'abbZ',
    0xc9b6: 'baa15',
    0xc9b7: 'bacZ',
    0xc9b8: 'aab05',
    0xc9b9: 'aabZ',
    0xc9ba: 'bba15',
    0xc9bb: 'bbcZ',
    0xc9bc: 'aba15',
    0xc9bd: 'abcZ',
    0xc9be: 'bab05',
    0xc9bf: 'babZ',
    0xc9c0: 'aaa06',
    0xc9c1: 'aaiX',
    0xc9c2: 'aaaaaaaaaababS',
    0xc9c3: 'bbhX',
    0xc9c4: 'abb16',
    0xc9c5: 'abhX',
    0xc9c6: 'aaaaaaaaaabadX',
    0xc9c7: 'baiX',
    0xc9c8: 'aab16',
    0xc9c9: 'aahX',
    0xc9ca: 'aaaaaaaaaababC',
    0xc9cb: 'bbiX',
    0xc9cc: 'aba06',
    0xc9cd: 'abiX',
    0xc9ce: 'aaaaaaaaaababK',
    0xc9cf: 'bahX',
    0xc9d0: 'aaa16',
    0xc9d1: 'aaiY',
    0xc9d2: 'aaaabL9',
    0xc9d3: 'bbhY',
    0xc9d4: 'abb06',
    0xc9d5: 'abhY',
    0xc9d6: 'aaaaaaaaaabadx',
    0xc9d7: 'baiY',
    0xc9d8: 'aab06',
    0xc9d9: 'aahY',
    0xc9da: 'aaaaaaaaaababc',
    0xc9db: 'bbiY',
    0xc9dc: 'aba16',
    0xc9dd: 'abiY',
    0xc9de: 'aaaaaaaaaababk',
    0xc9df: 'bahY',
    0xc9e0: 'aaa07',
    0xc9e1: 'aaiZ',
    0xc9e2: 'aaaaaaaaaabaH',
    0xc9e3: 'bbhZ',
    0xc9e4: 'abb17',
    0xc9e5: 'abhZ',
    0xc9e6: 'aaaaaaaaaabaX',
    0xc9e7: 'baiZ',
    0xc9e8: 'aab17',
    0xc9e9: 'aahZ',
    0xc9ea: 'aaaaaaaaaabah',
    0xc9eb: 'bbiZ',
    0xc9ec: 'aba07',
    0xc9ed: 'abiZ',
    0xc9ee: 'aaaaaaaaaabax',
    0xc9ef: 'bahZ',
    0xc9f0: 'aaa17',
    0xc9f1: 'aakZ',
    0xc9f2: 'aaaaaM9',
    0xc9f3: 'bbjZ',
    0xc9f4: 'abb07',
    0xc9f5: 'abjZ',
    0xc9f6: 'aaaaaaaaaabY',
    0xc9f7: 'bakZ',
    0xc9f8: 'aab07',
    0xc9f9: 'aajZ',
    0xc9fa: 'aaaaaaaaaca',
    0xc9fb: 'bbkZ',
    0xc9fc: 'aba17',
    0xc9fd: 'abkZ',
    0xc9fe: 'aaaaaaa199',
    0xc9ff: 'bajZ',
    0xca00: 'aa188',
    0xca01: 'aaca',
    0xca02: 'aaabaB9',
    0xca03: 'bbba',
    0xca04: 'ab088',
    0xca05: 'abba',
    0xca06: 'ba188',
    0xca07: 'baca',
    0xca08: 'aa088',
    0xca09: 'aaba',
    0xca0a: 'bb188',
    0xca0b: 'bbca',
    0xca0c: 'ab188',
    0xca0d: 'abca',
    0xca0e: 'ba088',
    0xca0f: 'baba',
    0xca10: 'aa198',
    0xca11: 'aaaa',
    0xca12: 'aaaabA8',
    0xca13: 'bbdc',
    0xca14: 'ab098',
    0xca15: 'abdc',
    0xca16: 'ba198',
    0xca17: 'baaa',
    0xca18: 'aa098',
    0xca19: 'aadc',
    0xca1a: 'bb198',
    0xca1b: 'bbaa',
    0xca1c: 'ab198',
    0xca1d: 'abaa',
    0xca1e: 'ba098',
    0xca1f: 'badc',
    0xca20: 'aa189',
    0xca21: 'aaab',
    0xca22: 'aaabbA8',
    0xca23: 'bbbc',
    0xca24: 'ab089',
    0xca25: 'abbc',
    0xca26: 'ba189',
    0xca27: 'baab',
    0xca28: 'aa089',
    0xca29: 'aabc',
    0xca2a: 'bb189',
    0xca2b: 'bbab',
    0xca2c: 'ab189',
    0xca2d: 'abab',
    0xca2e: 'ba089',
    0xca2f: 'babc',
    0xca30: 'aa199',
    0xca31: 'aaac',
    0xca32: 'aaaaaB9',
    0xca33: 'bbbb',
    0xca34: 'ab099',
    0xca35: 'abbb',
    0xca36: 'ba199',
    0xca37: 'baac',
    0xca38: 'aa099',
    0xca39: 'aabb',
    0xca3a: 'bb199',
    0xca3b: 'bbac',
    0xca3c: 'ab199',
    0xca3d: 'abac',
    0xca3e: 'ba099',
    0xca3f: 'babb',
    0xca40: 'aa988',
    0xca41: 'aaad',
    0xca42: 'aaabaA8',
    0xca43: 'bbbe',
    0xca44: 'ab888',
    0xca45: 'abbe',
    0xca46: 'ba988',
    0xca47: 'baad',
    0xca48: 'aa888',
    0xca49: 'aabe',
    0xca4a: 'bb988',
    0xca4b: 'bbad',
    0xca4c: 'ab988',
    0xca4d: 'abad',
    0xca4e: 'ba888',
    0xca4f: 'babe',
    0xca50: 'aa998',
    0xca51: 'aaae',
    0xca52: 'aaaabB9',
    0xca53: 'bbbd',
    0xca54: 'ab898',
    0xca55: 'abbd',
    0xca56: 'ba998',
    0xca57: 'baae',
    0xca58: 'aa898',
    0xca59: 'aabd',
    0xca5a: 'bb998',
    0xca5b: 'bbae',
    0xca5c: 'ab998',
    0xca5d: 'abae',
    0xca5e: 'ba898',
    0xca5f: 'babd',
    0xca60: 'aa989',
    0xca61: 'aaaf',
    0xca62: 'aaabbB9',
    0xca63: 'bbbg',
    0xca64: 'ab889',
    0xca65: 'abbg',
    0xca66: 'ba989',
    0xca67: 'baaf',
    0xca68: 'aa889',
    0xca69: 'aabg',
    0xca6a: 'bb989',
    0xca6b: 'bbaf',
    0xca6c: 'ab989',
    0xca6d: 'abaf',
    0xca6e: 'ba889',
    0xca6f: 'babg',
    0xca70: 'aa999',
    0xca71: 'aaag',
    0xca72: 'aaaaaA8',
    0xca73: 'bbbf',
    0xca74: 'ab899',
    0xca75: 'abbf',
    0xca76: 'ba999',
    0xca77: 'baag',
    0xca78: 'aa899',
    0xca79: 'aabf',
    0xca7a: 'bb999',
    0xca7b: 'bbag',
    0xca7c: 'ab999',
    0xca7d: 'abag',
    0xca7e: 'aaaaaab189',
    0xca7f: 'babf',
    0xca80: 'aa108',
    0xca81: 'aaah',
    0xca82: 'aaabaB8',
    0xca83: 'bbbi',
    0xca84: 'ab008',
    0xca85: 'abbi',
    0xca86: 'ba108',
    0xca87: 'baah',
    0xca88: 'aa008',
    0xca89: 'aabi',
    0xca8a: 'bb108',
    0xca8b: 'bbah',
    0xca8c: 'ab108',
    0xca8d: 'abah',
    0xca8e: 'ba008',
    0xca8f: 'babi',
    0xca90: 'aa118',
    0xca91: 'aaai',
    0xca92: 'aaaabA9',
    0xca93: 'bbbh',
    0xca94: 'ab018',
    0xca95: 'abbh',
    0xca96: 'ba118',
    0xca97: 'baai',
    0xca98: 'aa018',
    0xca99: 'aabh',
    0xca9a: 'bb118',
    0xca9b: 'bbai',
    0xca9c: 'ab118',
    0xca9d: 'abai',
    0xca9e: 'ba018',
    0xca9f: 'babh',
    0xcaa0: 'aa109',
    0xcaa1: 'aaaj',
    0xcaa2: 'aaabbA9',
    0xcaa3: 'bbbk',
    0xcaa4: 'ab009',
    0xcaa5: 'abbk',
    0xcaa6: 'ba109',
    0xcaa7: 'baaj',
    0xcaa8: 'aa009',
    0xcaa9: 'aabk',
    0xcaaa: 'bb109',
    0xcaab: 'bbaj',
    0xcaac: 'ab109',
    0xcaad: 'abaj',
    0xcaae: 'ba009',
    0xcaaf: 'babk',
    0xcab0: 'aa119',
    0xcab1: 'aaak',
    0xcab2: 'aaaaaB8',
    0xcab3: 'bbbj',
    0xcab4: 'ab019',
    0xcab5: 'abbj',
    0xcab6: 'ba119',
    0xcab7: 'baak',
    0xcab8: 'aa019',
    0xcab9: 'aabj',
    0xcaba: 'bb119',
    0xcabb: 'bbak',
    0xcabc: 'ab119',
    0xcabd: 'abak',
    0xcabe: 'ba019',
    0xcabf: 'babj',
    0xcac0: 'aa148',
    0xcac1: 'aaal',
    0xcac2: 'aaabaA9',
    0xcac3: 'bbbm',
    0xcac4: 'ab048',
    0xcac5: 'abbm',
    0xcac6: 'ba148',
    0xcac7: 'baal',
    0xcac8: 'aa048',
    0xcac9: 'aabm',
    0xcaca: 'bb148',
    0xcacb: 'bbal',
    0xcacc: 'ab148',
    0xcacd: 'abal',
    0xcace: 'ba048',
    0xcacf: 'babm',
    0xcad0: 'aa158',
    0xcad1: 'aaam',
    0xcad2: 'aaaabB8',
    0xcad3: 'bbbl',
    0xcad4: 'ab058',
    0xcad5: 'abbl',
    0xcad6: 'ba158',
    0xcad7: 'baam',
    0xcad8: 'aa058',
    0xcad9: 'aabl',
    0xcada: 'bb158',
    0xcadb: 'bbam',
    0xcadc: 'ab158',
    0xcadd: 'abam',
    0xcade: 'ba058',
    0xcadf: 'babl',
    0xcae0: 'aa149',
    0xcae1: 'aaan',
    0xcae2: 'aaabbB8',
    0xcae3: 'bbbo',
    0xcae4: 'ab049',
    0xcae5: 'abbo',
    0xcae6: 'ba149',
    0xcae7: 'baan',
    0xcae8: 'aa049',
    0xcae9: 'aabo',
    0xcaea: 'bb149',
    0xcaeb: 'bban',
    0xcaec: 'ab149',
    0xcaed: 'aban',
    0xcaee: 'ba049',
    0xcaef: 'babo',
    0xcaf0: 'aa159',
    0xcaf1: 'aaao',
    0xcaf2: 'aaaaaA9',
    0xcaf3: 'bbbn',
    0xcaf4: 'ab059',
    0xcaf5: 'abbn',
    0xcaf6: 'ba159',
    0xcaf7: 'baao',
    0xcaf8: 'aa059',
    0xcaf9: 'aabn',
    0xcafa: 'aaaaaaabaca',
    0xcafb: 'bbao',
    0xcafc: 'ab159',
    0xcafd: 'abao',
    0xcafe: 'aaaaaaa089',
    0xcaff: 'babn',
    0xcb00: 'aa104',
    0xcb01: 'aaap',
    0xcb02: 'aaabaD8',
    0xcb03: 'bbbq',
    0xcb04: 'ab004',
    0xcb05: 'abbq',
    0xcb06: 'ba104',
    0xcb07: 'baap',
    0xcb08: 'aa004',
    0xcb09: 'aabq',
    0xcb0a: 'bb104',
    0xcb0b: 'bbap',
    0xcb0c: 'ab104',
    0xcb0d: 'abap',
    0xcb0e: 'ba004',
    0xcb0f: 'babq',
    0xcb10: 'aa114',
    0xcb11: 'aaaq',
    0xcb12: 'aaaabE8',
    0xcb13: 'bbbp',
    0xcb14: 'ab014',
    0xcb15: 'abbp',
    0xcb16: 'ba114',
    0xcb17: 'baaq',
    0xcb18: 'aa014',
    0xcb19: 'aabp',
    0xcb1a: 'bb114',
    0xcb1b: 'bbaq',
    0xcb1c: 'ab114',
    0xcb1d: 'abaq',
    0xcb1e: 'ba014',
    0xcb1f: 'babp',
    0xcb20: 'aa105',
    0xcb21: 'aaar',
    0xcb22: 'aaabbE8',
    0xcb23: 'bbbs',
    0xcb24: 'ab005',
    0xcb25: 'abbs',
    0xcb26: 'ba105',
    0xcb27: 'baar',
    0xcb28: 'aa005',
    0xcb29: 'aabs',
    0xcb2a: 'bb105',
    0xcb2b: 'bbar',
    0xcb2c: 'ab105',
    0xcb2d: 'abar',
    0xcb2e: 'ba005',
    0xcb2f: 'babs',
    0xcb30: 'aa115',
    0xcb31: 'aaas',
    0xcb32: 'aaaaaD8',
    0xcb33: 'bbbr',
    0xcb34: 'ab015',
    0xcb35: 'abbr',
    0xcb36: 'ba115',
    0xcb37: 'baas',
    0xcb38: 'aa015',
    0xcb39: 'aabr',
    0xcb3a: 'bb115',
    0xcb3b: 'bbas',
    0xcb3c: 'ab115',
    0xcb3d: 'abas',
    0xcb3e: 'ba015',
    0xcb3f: 'babr',
    0xcb40: 'aa106',
    0xcb41: 'aaat',
    0xcb42: 'aaabaE8',
    0xcb43: 'bbbu',
    0xcb44: 'ab006',
    0xcb45: 'abbu',
    0xcb46: 'ba106',
    0xcb47: 'baat',
    0xcb48: 'aa006',
    0xcb49: 'aabu',
    0xcb4a: 'bb106',
    0xcb4b: 'bbat',
    0xcb4c: 'ab106',
    0xcb4d: 'abat',
    0xcb4e: 'ba006',
    0xcb4f: 'babu',
    0xcb50: 'aa116',
    0xcb51: 'aaau',
    0xcb52: 'aaaabD8',
    0xcb53: 'bbbt',
    0xcb54: 'ab016',
    0xcb55: 'abbt',
    0xcb56: 'ba116',
    0xcb57: 'baau',
    0xcb58: 'aa016',
    0xcb59: 'aabt',
    0xcb5a: 'bb116',
    0xcb5b: 'bbau',
    0xcb5c: 'ab116',
    0xcb5d: 'abau',
    0xcb5e: 'ba016',
    0xcb5f: 'babt',
    0xcb60: 'aa107',
    0xcb61: 'aaav',
    0xcb62: 'aaabbD8',
    0xcb63: 'bbbw',
    0xcb64: 'ab007',
    0xcb65: 'abbw',
    0xcb66: 'ba107',
    0xcb67: 'baav',
    0xcb68: 'aa007',
    0xcb69: 'aabw',
    0xcb6a: 'bb107',
    0xcb6b: 'bbav',
    0xcb6c: 'ab107',
    0xcb6d: 'abav',
    0xcb6e: 'ba007',
    0xcb6f: 'babw',
    0xcb70: 'aa117',
    0xcb71: 'aaaw',
    0xcb72: 'aaaaaE8',
    0xcb73: 'bbbv',
    0xcb74: 'ab017',
    0xcb75: 'abbv',
    0xcb76: 'ba117',
    0xcb77: 'baaw',
    0xcb78: 'aa017',
    0xcb79: 'aabv',
    0xcb7a: 'bb117',
    0xcb7b: 'bbaw',
    0xcb7c: 'ab117',
    0xcb7d: 'abaw',
    0xcb7e: 'aaaaaab089',
    0xcb7f: 'babv',
    0xcb80: 'aa100',
    0xcb81: 'aaax',
    0xcb82: 'aaabaD9',
    0xcb83: 'bbby',
    0xcb84: 'ab000',
    0xcb85: 'abby',
    0xcb86: 'ba100',
    0xcb87: 'baax',
    0xcb88: 'aa000',
    0xcb89: 'aaby',
    0xcb8a: 'bb100',
    0xcb8b: 'bbax',
    0xcb8c: 'ab100',
    0xcb8d: 'abax',
    0xcb8e: 'ba000',
    0xcb8f: 'baby',
    0xcb90: 'aa110',
    0xcb91: 'aaay',
    0xcb92: 'aaaabE9',
    0xcb93: 'bbbx',
    0xcb94: 'ab010',
    0xcb95: 'abbx',
    0xcb96: 'ba110',
    0xcb97: 'baay',
    0xcb98: 'aa010',
    0xcb99: 'aabx',
    0xcb9a: 'bb110',
    0xcb9b: 'bbay',
    0xcb9c: 'ab110',
    0xcb9d: 'abay',
    0xcb9e: 'ba010',
    0xcb9f: 'babx',
    0xcba0: 'aa101',
    0xcba1: 'aaaz',
    0xcba2: 'aaabbE9',
    0xcba3: 'bbdx',
    0xcba4: 'ab001',
    0xcba5: 'abdx',
    0xcba6: 'ba101',
    0xcba7: 'baaz',
    0xcba8: 'aa001',
    0xcba9: 'aadx',
    0xcbaa: 'bb101',
    0xcbab: 'bbaz',
    0xcbac: 'ab101',
    0xcbad: 'abaz',
    0xcbae: 'ba001',
    0xcbaf: 'badx',
    0xcbb0: 'aa111',
    0xcbb1: 'aacz',
    0xcbb2: 'aaaaaD9',
    0xcbb3: 'bbbz',
    0xcbb4: 'ab011',
    0xcbb5: 'abbz',
    0xcbb6: 'ba111',
    0xcbb7: 'bacz',
    0xcbb8: 'aa011',
    0xcbb9: 'aabz',
    0xcbba: 'bb111',
    0xcbbb: 'bbcz',
    0xcbbc: 'ab111',
    0xcbbd: 'abcz',
    0xcbbe: 'ba011',
    0xcbbf: 'babz',
    0xcbc0: 'aa102',
    0xcbc1: 'aaix',
    0xcbc2: 'aaabaE9',
    0xcbc3: 'bbhx',
    0xcbc4: 'ab002',
    0xcbc5: 'abhx',
    0xcbc6: 'ba102',
    0xcbc7: 'baix',
    0xcbc8: 'aa002',
    0xcbc9: 'aahx',
    0xcbca: 'bb102',
    0xcbcb: 'bbix',
    0xcbcc: 'ab102',
    0xcbcd: 'abix',
    0xcbce: 'ba002',
    0xcbcf: 'bahx',
    0xcbd0: 'aa112',
    0xcbd1: 'aaiy',
    0xcbd2: 'aaaabD9',
    0xcbd3: 'bbhy',
    0xcbd4: 'ab012',
    0xcbd5: 'abhy',
    0xcbd6: 'ba112',
    0xcbd7: 'baiy',
    0xcbd8: 'aa012',
    0xcbd9: 'aahy',
    0xcbda: 'bb112',
    0xcbdb: 'bbiy',
    0xcbdc: 'ab112',
    0xcbdd: 'abiy',
    0xcbde: 'ba012',
    0xcbdf: 'bahy',
    0xcbe0: 'aa103',
    0xcbe1: 'aaiz',
    0xcbe2: 'aaabbD9',
    0xcbe3: 'bbhz',
    0xcbe4: 'ab003',
    0xcbe5: 'abhz',
    0xcbe6: 'ba103',
    0xcbe7: 'baiz',
    0xcbe8: 'aa003',
    0xcbe9: 'aahz',
    0xcbea: 'bb103',
    0xcbeb: 'bbiz',
    0xcbec: 'ab103',
    0xcbed: 'abiz',
    0xcbee: 'ba003',
    0xcbef: 'bahz',
    0xcbf0: 'aa113',
    0xcbf1: 'aakz',
    0xcbf2: 'aaaaaE9',
    0xcbf3: 'bbjz',
    0xcbf4: 'ab013',
    0xcbf5: 'abjz',
    0xcbf6: 'aaaaaaaabbaX',
    0xcbf7: 'bakz',
    0xcbf8: 'aa013',
    0xcbf9: 'aajz',
    0xcbfa: 'aaaaaaaabba',
    0xcbfb: 'bbkz',
    0xcbfc: 'ab113',
    0xcbfd: 'abkz',
    0xcbfe: 'aaaaaaa189',
    0xcbff: 'bajz',
    0xcc00: 'baa',
    0xcc01: 'a198',
    0xcc02: 'adc',
    0xcc03: 'b098',
    0xcc04: 'bdc',
    0xcc05: 'a098',
    0xcc06: 'aaa',
    0xcc07: 'b198',
    0xcc08: 'bca',
    0xcc09: 'a188',
    0xcc0a: 'aba',
    0xcc0b: 'b088',
    0xcc0c: 'bba',
    0xcc0d: 'a088',
    0xcc0e: 'aca',
    0xcc0f: 'b188',
    0xcc10: 'bac',
    0xcc11: 'a199',
    0xcc12: 'abb',
    0xcc13: 'b099',
    0xcc14: 'bbb',
    0xcc15: 'a099',
    0xcc16: 'aac',
    0xcc17: 'b199',
    0xcc18: 'bab',
    0xcc19: 'a189',
    0xcc1a: 'abc',
    0xcc1b: 'b089',
    0xcc1c: 'bbc',
    0xcc1d: 'a089',
    0xcc1e: 'aab',
    0xcc1f: 'b189',
    0xcc20: 'bae',
    0xcc21: 'a998',
    0xcc22: 'abd',
    0xcc23: 'b898',
    0xcc24: 'bbd',
    0xcc25: 'a898',
    0xcc26: 'aae',
    0xcc27: 'b998',
    0xcc28: 'bad',
    0xcc29: 'a988',
    0xcc2a: 'abe',
    0xcc2b: 'b888',
    0xcc2c: 'bbe',
    0xcc2d: 'a888',
    0xcc2e: 'aad',
    0xcc2f: 'b988',
    0xcc30: 'bag',
    0xcc31: 'a999',
    0xcc32: 'abf',
    0xcc33: 'b899',
    0xcc34: 'bbf',
    0xcc35: 'a899',
    0xcc36: 'aag',
    0xcc37: 'b999',
    0xcc38: 'baf',
    0xcc39: 'a989',
    0xcc3a: 'abg',
    0xcc3b: 'b889',
    0xcc3c: 'bbg',
    0xcc3d: 'a889',
    0xcc3e: 'aaf',
    0xcc3f: 'b989',
    0xcc40: 'bai',
    0xcc41: 'a118',
    0xcc42: 'abh',
    0xcc43: 'b018',
    0xcc44: 'bbh',
    0xcc45: 'a018',
    0xcc46: 'aai',
    0xcc47: 'b118',
    0xcc48: 'bah',
    0xcc49: 'a108',
    0xcc4a: 'abi',
    0xcc4b: 'b008',
    0xcc4c: 'bbi',
    0xcc4d: 'a008',
    0xcc4e: 'aah',
    0xcc4f: 'b108',
    0xcc50: 'bak',
    0xcc51: 'a119',
    0xcc52: 'abj',
    0xcc53: 'b019',
    0xcc54: 'bbj',
    0xcc55: 'a019',
    0xcc56: 'aak',
    0xcc57: 'b119',
    0xcc58: 'baj',
    0xcc59: 'a109',
    0xcc5a: 'abk',
    0xcc5b: 'b009',
    0xcc5c: 'bbk',
    0xcc5d: 'a009',
    0xcc5e: 'aaj',
    0xcc5f: 'b109',
    0xcc60: 'bam',
    0xcc61: 'a158',
    0xcc62: 'abl',
    0xcc63: 'b058',
    0xcc64: 'bbl',
    0xcc65: 'a058',
    0xcc66: 'aam',
    0xcc67: 'b158',
    0xcc68: 'bal',
    0xcc69: 'a148',
    0xcc6a: 'abm',
    0xcc6b: 'b048',
    0xcc6c: 'bbm',
    0xcc6d: 'a048',
    0xcc6e: 'aal',
    0xcc6f: 'b148',
    0xcc70: 'bao',
    0xcc71: 'a159',
    0xcc72: 'abn',
    0xcc73: 'b059',
    0xcc74: 'bbn',
    0xcc75: 'a059',
    0xcc76: 'aao',
    0xcc77: 'b159',
    0xcc78: 'ban',
    0xcc79: 'a149',
    0xcc7a: 'abo',
    0xcc7b: 'b049',
    0xcc7c: 'bbo',
    0xcc7d: 'a049',
    0xcc7e: 'aan',
    0xcc7f: 'b149',
    0xcc80: 'baq',
    0xcc81: 'a114',
    0xcc82: 'abp',
    0xcc83: 'b014',
    0xcc84: 'bbp',
    0xcc85: 'a014',
    0xcc86: 'aaq',
    0xcc87: 'b114',
    0xcc88: 'bap',
    0xcc89: 'a104',
    0xcc8a: 'abq',
    0xcc8b: 'b004',
    0xcc8c: 'bbq',
    0xcc8d: 'a004',
    0xcc8e: 'aap',
    0xcc8f: 'b104',
    0xcc90: 'bas',
    0xcc91: 'a115',
    0xcc92: 'abr',
    0xcc93: 'b015',
    0xcc94: 'bbr',
    0xcc95: 'a015',
    0xcc96: 'aas',
    0xcc97: 'b115',
    0xcc98: 'bar',
    0xcc99: 'a105',
    0xcc9a: 'abs',
    0xcc9b: 'b005',
    0xcc9c: 'bbs',
    0xcc9d: 'a005',
    0xcc9e: 'aar',
    0xcc9f: 'b105',
    0xcca0: 'bau',
    0xcca1: 'a116',
    0xcca2: 'abt',
    0xcca3: 'b016',
    0xcca4: 'bbt',
    0xcca5: 'a016',
    0xcca6: 'aau',
    0xcca7: 'b116',
    0xcca8: 'bat',
    0xcca9: 'a106',
    0xccaa: 'abu',
    0xccab: 'b006',
    0xccac: 'bbu',
    0xccad: 'a006',
    0xccae: 'aat',
    0xccaf: 'b106',
    0xccb0: 'baw',
    0xccb1: 'a117',
    0xccb2: 'abv',
    0xccb3: 'b017',
    0xccb4: 'bbv',
    0xccb5: 'a017',
    0xccb6: 'aaw',
    0xccb7: 'b117',
    0xccb8: 'bav',
    0xccb9: 'a107',
    0xccba: 'abw',
    0xccbb: 'b007',
    0xccbc: 'bbw',
    0xccbd: 'a007',
    0xccbe: 'aav',
    0xccbf: 'b107',
    0xccc0: 'bay',
    0xccc1: 'a110',
    0xccc2: 'abx',
    0xccc3: 'b010',
    0xccc4: 'bbx',
    0xccc5: 'a010',
    0xccc6: 'aay',
    0xccc7: 'b110',
    0xccc8: 'bax',
    0xccc9: 'a100',
    0xccca: 'aby',
    0xcccb: 'b000',
    0xcccc: 'bby',
    0xcccd: 'a000',
    0xccce: 'aax',
    0xcccf: 'b100',
    0xccd0: 'bcz',
    0xccd1: 'a111',
    0xccd2: 'abz',
    0xccd3: 'b011',
    0xccd4: 'bbz',
    0xccd5: 'a011',
    0xccd6: 'acz',
    0xccd7: 'b111',
    0xccd8: 'baz',
    0xccd9: 'a101',
    0xccda: 'adx',
    0xccdb: 'b001',
    0xccdc: 'bdx',
    0xccdd: 'a001',
    0xccde: 'aaz',
    0xccdf: 'b101',
    0xcce0: 'biy',
    0xcce1: 'a112',
    0xcce2: 'ahy',
    0xcce3: 'b012',
    0xcce4: 'bhy',
    0xcce5: 'a012',
    0xcce6: 'aiy',
    0xcce7: 'b112',
    0xcce8: 'bix',
    0xcce9: 'a102',
    0xccea: 'ahx',
    0xcceb: 'b002',
    0xccec: 'bhx',
    0xcced: 'a002',
    0xccee: 'aix',
    0xccef: 'b102',
    0xccf0: 'bkz',
    0xccf1: 'a113',
    0xccf2: 'ajz',
    0xccf3: 'b013',
    0xccf4: 'bjz',
    0xccf5: 'a013',
    0xccf6: 'akz',
    0xccf7: 'b113',
    0xccf8: 'biz',
    0xccf9: 'a103',
    0xccfa: 'ahz',
    0xccfb: 'b003',
    0xccfc: 'bhz',
    0xccfd: 'a003',
    0xccfe: 'aiz',
    0xccff: 'b103',
    0xcd00: 'baA',
    0xcd01: 'aa18',
    0xcd02: 'adC',
    0xcd03: 'bb08',
    0xcd04: 'bdC',
    0xcd05: 'ab08',
    0xcd06: 'aaA',
    0xcd07: 'ba18',
    0xcd08: 'bcA',
    0xcd09: 'aa08',
    0xcd0a: 'abA',
    0xcd0b: 'bb18',
    0xcd0c: 'bbA',
    0xcd0d: 'ab18',
    0xcd0e: 'acA',
    0xcd0f: 'ba08',
    0xcd10: 'baC',
    0xcd11: 'aa19',
    0xcd12: 'abB',
    0xcd13: 'bb09',
    0xcd14: 'bbB',
    0xcd15: 'ab09',
    0xcd16: 'aaC',
    0xcd17: 'ba19',
    0xcd18: 'baB',
    0xcd19: 'aa09',
    0xcd1a: 'abC',
    0xcd1b: 'bb19',
    0xcd1c: 'bbC',
    0xcd1d: 'ab19',
    0xcd1e: 'aaB',
    0xcd1f: 'ba09',
    0xcd20: 'baE',
    0xcd21: 'aa58',
    0xcd22: 'abD',
    0xcd23: 'bb48',
    0xcd24: 'bbD',
    0xcd25: 'ab48',
    0xcd26: 'aaE',
    0xcd27: 'ba58',
    0xcd28: 'baD',
    0xcd29: 'aa48',
    0xcd2a: 'abE',
    0xcd2b: 'bb58',
    0xcd2c: 'bbE',
    0xcd2d: 'ab58',
    0xcd2e: 'aaD',
    0xcd2f: 'ba48',
    0xcd30: 'baG',
    0xcd31: 'aa59',
    0xcd32: 'abF',
    0xcd33: 'bb49',
    0xcd34: 'bbF',
    0xcd35: 'ab49',
    0xcd36: 'aaG',
    0xcd37: 'ba59',
    0xcd38: 'baF',
    0xcd39: 'aa49',
    0xcd3a: 'abG',
    0xcd3b: 'bb59',
    0xcd3c: 'bbG',
    0xcd3d: 'ab59',
    0xcd3e: 'aaF',
    0xcd3f: 'ba49',
    0xcd40: 'baI',
    0xcd41: 'aa98',
    0xcd42: 'abH',
    0xcd43: 'bb88',
    0xcd44: 'bbH',
    0xcd45: 'ab88',
    0xcd46: 'aaI',
    0xcd47: 'ba98',
    0xcd48: 'baH',
    0xcd49: 'aa88',
    0xcd4a: 'abI',
    0xcd4b: 'bb98',
    0xcd4c: 'bbI',
    0xcd4d: 'ab98',
    0xcd4e: 'aaH',
    0xcd4f: 'ba88',
    0xcd50: 'baK',
    0xcd51: 'aa99',
    0xcd52: 'abJ',
    0xcd53: 'bb89',
    0xcd54: 'bbJ',
    0xcd55: 'ab89',
    0xcd56: 'aaK',
    0xcd57: 'ba99',
    0xcd58: 'baJ',
    0xcd59: 'aa89',
    0xcd5a: 'abK',
    0xcd5b: 'bb99',
    0xcd5c: 'bbK',
    0xcd5d: 'ab99',
    0xcd5e: 'aaJ',
    0xcd5f: 'ba89',
    0xcd60: 'baM',
    0xcd61: 'ai98',
    0xcd62: 'abL',
    0xcd63: 'bh98',
    0xcd64: 'bbL',
    0xcd65: 'ah98',
    0xcd66: 'aaM',
    0xcd67: 'bi98',
    0xcd68: 'baL',
    0xcd69: 'ai88',
    0xcd6a: 'abM',
    0xcd6b: 'bh88',
    0xcd6c: 'bbM',
    0xcd6d: 'ah88',
    0xcd6e: 'aaL',
    0xcd6f: 'bi88',
    0xcd70: 'baO',
    0xcd71: 'ai99',
    0xcd72: 'abN',
    0xcd73: 'bh99',
    0xcd74: 'bbN',
    0xcd75: 'ah99',
    0xcd76: 'aaO',
    0xcd77: 'bi99',
    0xcd78: 'baN',
    0xcd79: 'ai89',
    0xcd7a: 'abO',
    0xcd7b: 'bh89',
    0xcd7c: 'bbO',
    0xcd7d: 'ah89',
    0xcd7e: 'aaN',
    0xcd7f: 'bi89',
    0xcd80: 'baQ',
    0xcd81: 'aa10',
    0xcd82: 'abP',
    0xcd83: 'bb00',
    0xcd84: 'bbP',
    0xcd85: 'ab00',
    0xcd86: 'aaQ',
    0xcd87: 'ba10',
    0xcd88: 'baP',
    0xcd89: 'aa00',
    0xcd8a: 'abQ',
    0xcd8b: 'bb10',
    0xcd8c: 'bbQ',
    0xcd8d: 'ab10',
    0xcd8e: 'aaP',
    0xcd8f: 'ba00',
    0xcd90: 'baS',
    0xcd91: 'aa11',
    0xcd92: 'abR',
    0xcd93: 'bb01',
    0xcd94: 'bbR',
    0xcd95: 'ab01',
    0xcd96: 'aaS',
    0xcd97: 'ba11',
    0xcd98: 'baR',
    0xcd99: 'aa01',
    0xcd9a: 'abS',
    0xcd9b: 'bb11',
    0xcd9c: 'bbS',
    0xcd9d: 'ab11',
    0xcd9e: 'aaR',
    0xcd9f: 'ba01',
    0xcda0: 'baU',
    0xcda1: 'aa12',
    0xcda2: 'abT',
    0xcda3: 'bb02',
    0xcda4: 'bbT',
    0xcda5: 'ab02',
    0xcda6: 'aaU',
    0xcda7: 'ba12',
    0xcda8: 'baT',
    0xcda9: 'aa02',
    0xcdaa: 'abU',
    0xcdab: 'bb12',
    0xcdac: 'bbU',
    0xcdad: 'ab12',
    0xcdae: 'aaT',
    0xcdaf: 'ba02',
    0xcdb0: 'baW',
    0xcdb1: 'aa13',
    0xcdb2: 'abV',
    0xcdb3: 'bb03',
    0xcdb4: 'bbV',
    0xcdb5: 'ab03',
    0xcdb6: 'aaW',
    0xcdb7: 'ba13',
    0xcdb8: 'baV',
    0xcdb9: 'aa03',
    0xcdba: 'abW',
    0xcdbb: 'bb13',
    0xcdbc: 'bbW',
    0xcdbd: 'ab13',
    0xcdbe: 'aaV',
    0xcdbf: 'ba03',
    0xcdc0: 'baY',
    0xcdc1: 'aa14',
    0xcdc2: 'abX',
    0xcdc3: 'bb04',
    0xcdc4: 'bbX',
    0xcdc5: 'ab04',
    0xcdc6: 'aaY',
    0xcdc7: 'ba14',
    0xcdc8: 'baX',
    0xcdc9: 'aa04',
    0xcdca: 'abY',
    0xcdcb: 'bb14',
    0xcdcc: 'bbY',
    0xcdcd: 'ab14',
    0xcdce: 'aaX',
    0xcdcf: 'ba04',
    0xcdd0: 'bcZ',
    0xcdd1: 'aa15',
    0xcdd2: 'abZ',
    0xcdd3: 'bb05',
    0xcdd4: 'bbZ',
    0xcdd5: 'ab05',
    0xcdd6: 'acZ',
    0xcdd7: 'ba15',
    0xcdd8: 'baZ',
    0xcdd9: 'aa05',
    0xcdda: 'adX',
    0xcddb: 'bb15',
    0xcddc: 'bdX',
    0xcddd: 'ab15',
    0xcdde: 'aaZ',
    0xcddf: 'ba05',
    0xcde0: 'biY',
    0xcde1: 'aa16',
    0xcde2: 'ahY',
    0xcde3: 'bb06',
    0xcde4: 'bhY',
    0xcde5: 'ab06',
    0xcde6: 'aiY',
    0xcde7: 'ba16',
    0xcde8: 'biX',
    0xcde9: 'aa06',
    0xcdea: 'ahX',
    0xcdeb: 'bb16',
    0xcdec: 'bhX',
    0xcded: 'ab16',
    0xcdee: 'aiX',
    0xcdef: 'ba06',
    0xcdf0: 'bkZ',
    0xcdf1: 'aa17',
    0xcdf2: 'ajZ',
    0xcdf3: 'bb07',
    0xcdf4: 'bjZ',
    0xcdf5: 'ab07',
    0xcdf6: 'akZ',
    0xcdf7: 'ba17',
    0xcdf8: 'biZ',
    0xcdf9: 'aa07',
    0xcdfa: 'ahZ',
    0xcdfb: 'bb17',
    0xcdfc: 'bhZ',
    0xcdfd: 'ab17',
    0xcdfe: 'aiZ',
    0xcdff: 'ba07',
    0xce00: 'bA1',
    0xce01: 'B3',
    0xce02: 'aB0',
    0xce03: 'A2',
    0xce04: 'bB0',
    0xce05: 'B2',
    0xce06: 'aA1',
    0xce07: 'A3',
    0xce08: 'bA0',
    0xce09: 'B1',
    0xce0a: 'aB1',
    0xce0b: 'A0',
    0xce0c: 'bB1',
    0xce0d: 'B0',
    0xce0e: 'aA0',
    0xce0f: 'A1',
    0xce10: 'bA3',
    0xce11: 'B7',
    0xce12: 'aB2',
    0xce13: 'A6',
    0xce14: 'bB2',
    0xce15: 'B6',
    0xce16: 'aA3',
    0xce17: 'A7',
    0xce18: 'bA2',
    0xce19: 'B5',
    0xce1a: 'aB3',
    0xce1b: 'A4',
    0xce1c: 'bB3',
    0xce1d: 'B4',
    0xce1e: 'aA2',
    0xce1f: 'A5',
    0xce20: '5',
    0xce21: 'D8',
    0xce22: '4',
    0xce23: 'E8',
    0xce24: '7',
    0xce25: 'D9',
    0xce26: '6',
    0xce27: 'E9',
    0xce28: '1',
    0xce29: 'B9',
    0xce2a: '0',
    0xce2b: 'A8',
    0xce2c: '3',
    0xce2d: 'B8',
    0xce2e: '2',
    0xce2f: 'A9',
    0xce30: 'bA7',
    0xce31: 'L8',
    0xce32: 'aB6',
    0xce33: 'M8',
    0xce34: 'bB6',
    0xce35: 'L9',
    0xce36: 'aA7',
    0xce37: 'M9',
    0xce38: '9',
    0xce39: 'H8',
    0xce3a: '8',
    0xce3b: 'I8',
    0xce3c: 'bB7',
    0xce3d: 'H9',
    0xce3e: 'aA6',
    0xce3f: 'I9',
    0xce40: 'bA9',
    0xce41: 'b3',
    0xce42: 'aB8',
    0xce43: 'a2',
    0xce44: 'bB8',
    0xce45: 'b2',
    0xce46: 'aA9',
    0xce47: 'a3',
    0xce48: 'bA8',
    0xce49: 'b1',
    0xce4a: 'aB9',
    0xce4b: 'a0',
    0xce4c: 'bB9',
    0xce4d: 'b0',
    0xce4e: 'aA8',
    0xce4f: 'a1',
    0xce50: 'bE9',
    0xce51: 'b7',
    0xce52: 'aD9',
    0xce53: 'a6',
    0xce54: 'bD9',
    0xce55: 'b6',
    0xce56: 'aE9',
    0xce57: 'a7',
    0xce58: 'bE8',
    0xce59: 'b5',
    0xce5a: 'aD8',
    0xce5b: 'a4',
    0xce5c: 'bD8',
    0xce5d: 'b4',
    0xce5e: 'aE8',
    0xce5f: 'a5',
    0xce60: 'bI9',
    0xce61: 'd8',
    0xce62: 'aH9',
    0xce63: 'e8',
    0xce64: 'bH9',
    0xce65: 'd9',
    0xce66: 'aI9',
    0xce67: 'e9',
    0xce68: 'bI8',
    0xce69: 'b9',
    0xce6a: 'aH8',
    0xce6b: 'a8',
    0xce6c: 'bH8',
    0xce6d: 'b8',
    0xce6e: 'aI8',
    0xce6f: 'a9',
    0xce70: 'bM9',
    0xce71: 'l8',
    0xce72: 'aL9',
    0xce73: 'm8',
    0xce74: 'bL9',
    0xce75: 'l9',
    0xce76: 'aM9',
    0xce77: 'm9',
    0xce78: 'bM8',
    0xce79: 'h8',
    0xce7a: 'aL8',
    0xce7b: 'i8',
    0xce7c: 'bL8',
    0xce7d: 'h9',
    0xce7e: 'aM8',
    0xce7f: 'i9',
    0xce80: 'e',
    0xce81: 'aaA8',
    0xce82: 'd',
    0xce83: 'bbB9',
    0xce84: 'g',
    0xce85: 'abB9',
    0xce86: 'f',
    0xce87: 'baA8',
    0xce88: 'a',
    0xce89: 'aaB9',
    0xce8a: 'ab1',
    0xce8b: 'bbA8',
    0xce8c: 'c',
    0xce8d: 'abA8',
    0xce8e: 'b',
    0xce8f: 'baB9',
    0xce90: 'm',
    0xce91: 'aaA9',
    0xce92: 'l',
    0xce93: 'bbB8',
    0xce94: 'o',
    0xce95: 'abB8',
    0xce96: 'n',
    0xce97: 'baA9',
    0xce98: 'i',
    0xce99: 'aaB8',
    0xce9a: 'h',
    0xce9b: 'bbA9',
    0xce9c: 'k',
    0xce9d: 'abA9',
    0xce9e: 'j',
    0xce9f: 'baB8',
    0xcea0: 'u',
    0xcea1: 'aaE8',
    0xcea2: 't',
    0xcea3: 'bbD8',
    0xcea4: 'w',
    0xcea5: 'abD8',
    0xcea6: 'v',
    0xcea7: 'baE8',
    0xcea8: 'q',
    0xcea9: 'aaD8',
    0xceaa: 'p',
    0xceab: 'bbE8',
    0xceac: 's',
    0xcead: 'abE8',
    0xceae: 'r',
    0xceaf: 'baD8',
    0xceb0: 'ba7',
    0xceb1: 'aaE9',
    0xceb2: 'ab6',
    0xceb3: 'bbD9',
    0xceb4: 'bb6',
    0xceb5: 'abD9',
    0xceb6: 'aa7',
    0xceb7: 'baE9',
    0xceb8: 'y',
    0xceb9: 'aaD9',
    0xceba: 'x',
    0xcebb: 'bbE9',
    0xcebc: 'bb7',
    0xcebd: 'abE9',
    0xcebe: 'z',
    0xcebf: 'baD9',
    0xcec0: 'E',
    0xcec1: '48',
    0xcec2: 'D',
    0xcec3: '58',
    0xcec4: 'G',
    0xcec5: '49',
    0xcec6: 'F',
    0xcec7: '59',
    0xcec8: 'A',
    0xcec9: '08',
    0xceca: 'ab9',
    0xcecb: '18',
    0xcecc: 'C',
    0xcecd: '09',
    0xcece: 'B',
    0xcecf: '19',
    0xced0: 'M',
    0xced1: 'aaI9',
    0xced2: 'L',
    0xced3: 'bbH9',
    0xced4: 'O',
    0xced5: 'abH9',
    0xced6: 'N',
    0xced7: 'baI9',
    0xced8: 'I',
    0xced9: '88',
    0xceda: 'H',
    0xcedb: '98',
    0xcedc: 'K',
    0xcedd: '89',
    0xcede: 'J',
    0xcedf: '99',
    0xcee0: 'U',
    0xcee1: '02',
    0xcee2: 'T',
    0xcee3: '12',
    0xcee4: 'W',
    0xcee5: '03',
    0xcee6: 'V',
    0xcee7: '13',
    0xcee8: 'Q',
    0xcee9: '00',
    0xceea: 'P',
    0xceeb: '10',
    0xceec: 'S',
    0xceed: '01',
    0xceee: 'R',
    0xceef: '11',
    0xcef0: 'bm9',
    0xcef1: '06',
    0xcef2: 'al9',
    0xcef3: '16',
    0xcef4: 'bl9',
    0xcef5: '07',
    0xcef6: 'am9',
    0xcef7: '17',
    0xcef8: 'Y',
    0xcef9: '04',
    0xcefa: 'X',
    0xcefb: '14',
    0xcefc: 'bl8',
    0xcefd: '05',
    0xcefe: 'Z',
    0xceff: '15',
    0xcf00: '089',
    0xcf01: 'bc',
    0xcf02: '189',
    0xcf03: 'ab',
    0xcf04: '099',
    0xcf05: 'bb',
    0xcf06: '199',
    0xcf07: 'ac',
    0xcf08: '088',
    0xcf09: 'ba',
    0xcf0a: '188',
    0xcf0b: 'ca',
    0xcf0c: '098',
    0xcf0d: 'dc',
    0xcf0e: '198',
    0xcf0f: 'aa',
    0xcf10: '889',
    0xcf11: 'bg',
    0xcf12: '989',
    0xcf13: 'af',
    0xcf14: '899',
    0xcf15: 'bf',
    0xcf16: '999',
    0xcf17: 'ag',
    0xcf18: '888',
    0xcf19: 'be',
    0xcf1a: '988',
    0xcf1b: 'ad',
    0xcf1c: '898',
    0xcf1d: 'bd',
    0xcf1e: '998',
    0xcf1f: 'ae',
    0xcf20: '009',
    0xcf21: 'bk',
    0xcf22: '109',
    0xcf23: 'aj',
    0xcf24: '019',
    0xcf25: 'bj',
    0xcf26: '119',
    0xcf27: 'ak',
    0xcf28: '008',
    0xcf29: 'bi',
    0xcf2a: '108',
    0xcf2b: 'ah',
    0xcf2c: '018',
    0xcf2d: 'bh',
    0xcf2e: '118',
    0xcf2f: 'ai',
    0xcf30: '049',
    0xcf31: 'bo',
    0xcf32: '149',
    0xcf33: 'an',
    0xcf34: '059',
    0xcf35: 'bn',
    0xcf36: '159',
    0xcf37: 'ao',
    0xcf38: '048',
    0xcf39: 'bm',
    0xcf3a: '148',
    0xcf3b: 'al',
    0xcf3c: '058',
    0xcf3d: 'bl',
    0xcf3e: '158',
    0xcf3f: 'am',
    0xcf40: '005',
    0xcf41: 'bs',
    0xcf42: '105',
    0xcf43: 'ar',
    0xcf44: '015',
    0xcf45: 'br',
    0xcf46: '115',
    0xcf47: 'as',
    0xcf48: '004',
    0xcf49: 'bq',
    0xcf4a: '104',
    0xcf4b: 'ap',
    0xcf4c: '014',
    0xcf4d: 'bp',
    0xcf4e: '114',
    0xcf4f: 'aq',
    0xcf50: '007',
    0xcf51: 'bw',
    0xcf52: '107',
    0xcf53: 'av',
    0xcf54: '017',
    0xcf55: 'bv',
    0xcf56: '117',
    0xcf57: 'aw',
    0xcf58: '006',
    0xcf59: 'bu',
    0xcf5a: '106',
    0xcf5b: 'at',
    0xcf5c: '016',
    0xcf5d: 'bt',
    0xcf5e: '116',
    0xcf5f: 'au',
    0xcf60: '001',
    0xcf61: 'dx',
    0xcf62: '101',
    0xcf63: 'az',
    0xcf64: '011',
    0xcf65: 'bz',
    0xcf66: '111',
    0xcf67: 'cz',
    0xcf68: '000',
    0xcf69: 'by',
    0xcf6a: '100',
    0xcf6b: 'ax',
    0xcf6c: '010',
    0xcf6d: 'bx',
    0xcf6e: '110',
    0xcf6f: 'ay',
    0xcf70: '003',
    0xcf71: 'hz',
    0xcf72: '103',
    0xcf73: 'iz',
    0xcf74: '013',
    0xcf75: 'jz',
    0xcf76: '113',
    0xcf77: 'kz',
    0xcf78: '002',
    0xcf79: 'hx',
    0xcf7a: '102',
    0xcf7b: 'ix',
    0xcf7c: '012',
    0xcf7d: 'hy',
    0xcf7e: '112',
    0xcf7f: 'iy',
    0xcf80: 'b19',
    0xcf81: 'bC',
    0xcf82: 'a09',
    0xcf83: 'aB',
    0xcf84: 'b09',
    0xcf85: 'bB',
    0xcf86: 'a19',
    0xcf87: 'aC',
    0xcf88: 'b18',
    0xcf89: 'bA',
    0xcf8a: 'a08',
    0xcf8b: 'cA',
    0xcf8c: 'b08',
    0xcf8d: 'dC',
    0xcf8e: 'a18',
    0xcf8f: 'aA',
    0xcf90: 'b59',
    0xcf91: 'bG',
    0xcf92: 'a49',
    0xcf93: 'aF',
    0xcf94: 'b49',
    0xcf95: 'bF',
    0xcf96: 'a59',
    0xcf97: 'aG',
    0xcf98: 'b58',
    0xcf99: 'bE',
    0xcf9a: 'a48',
    0xcf9b: 'aD',
    0xcf9c: 'b48',
    0xcf9d: 'bD',
    0xcf9e: 'a58',
    0xcf9f: 'aE',
    0xcfa0: 'b99',
    0xcfa1: 'bK',
    0xcfa2: 'a89',
    0xcfa3: 'aJ',
    0xcfa4: 'b89',
    0xcfa5: 'bJ',
    0xcfa6: 'a99',
    0xcfa7: 'aK',
    0xcfa8: 'b98',
    0xcfa9: 'bI',
    0xcfaa: 'a88',
    0xcfab: 'aH',
    0xcfac: 'b88',
    0xcfad: 'bH',
    0xcfae: 'a98',
    0xcfaf: 'aI',
    0xcfb0: 'h89',
    0xcfb1: 'bO',
    0xcfb2: 'i89',
    0xcfb3: 'aN',
    0xcfb4: 'h99',
    0xcfb5: 'bN',
    0xcfb6: 'i99',
    0xcfb7: 'aO',
    0xcfb8: 'h88',
    0xcfb9: 'bM',
    0xcfba: 'i88',
    0xcfbb: 'aL',
    0xcfbc: 'h98',
    0xcfbd: 'bL',
    0xcfbe: 'i98',
    0xcfbf: 'aM',
    0xcfc0: 'b11',
    0xcfc1: 'bS',
    0xcfc2: 'a01',
    0xcfc3: 'aR',
    0xcfc4: 'b01',
    0xcfc5: 'bR',
    0xcfc6: 'a11',
    0xcfc7: 'aS',
    0xcfc8: 'b10',
    0xcfc9: 'bQ',
    0xcfca: 'a00',
    0xcfcb: 'aP',
    0xcfcc: 'b00',
    0xcfcd: 'bP',
    0xcfce: 'a10',
    0xcfcf: 'aQ',
    0xcfd0: 'b13',
    0xcfd1: 'bW',
    0xcfd2: 'a03',
    0xcfd3: 'aV',
    0xcfd4: 'b03',
    0xcfd5: 'bV',
    0xcfd6: 'a13',
    0xcfd7: 'aW',
    0xcfd8: 'b12',
    0xcfd9: 'bU',
    0xcfda: 'a02',
    0xcfdb: 'aT',
    0xcfdc: 'b02',
    0xcfdd: 'bT',
    0xcfde: 'a12',
    0xcfdf: 'aU',
    0xcfe0: 'b15',
    0xcfe1: 'dX',
    0xcfe2: 'a05',
    0xcfe3: 'aZ',
    0xcfe4: 'b05',
    0xcfe5: 'bZ',
    0xcfe6: 'a15',
    0xcfe7: 'cZ',
    0xcfe8: 'b14',
    0xcfe9: 'bY',
    0xcfea: 'a04',
    0xcfeb: 'aX',
    0xcfec: 'b04',
    0xcfed: 'bX',
    0xcfee: 'a14',
    0xcfef: 'aY',
    0xcff0: 'b17',
    0xcff1: 'hZ',
    0xcff2: 'a07',
    0xcff3: 'iZ',
    0xcff4: 'b07',
    0xcff5: 'jZ',
    0xcff6: 'a17',
    0xcff7: 'kZ',
    0xcff8: 'b16',
    0xcff9: 'hX',
    0xcffa: 'a06',
    0xcffb: 'iX',
    0xcffc: 'b06',
    0xcffd: 'hY',
    0xcffe: 'a16',
    0xcfff: 'iY',
    0xd000: 'bbbab14',
    0xd001: 'bbbabY',
    0xd002: 'aaaba04',
    0xd003: 'aaabaX',
    0xd004: 'cccccccaaaC',
    0xd005: 'cccccccaaaS',
    0xd006: 'cccccccaaac',
    0xd007: 'ccbabY',
    0xd008: 'ccccccccbbbH',
    0xd009: 'bbabaX',
    0xd00a: 'ccccccccbbbX',
    0xd00b: 'aababY',
    0xd00c: 'ccccccccbbbh',
    0xd00d: 'ccccccccbbdc',
    0xd00e: 'cccab14',
    0xd00f: 'ccabaX',
    0xd010: 'bbbba04',
    0xd011: 'bbbbaX',
    0xd012: 'aaaab14',
    0xd013: 'aaaabY',
    0xd014: 'dddbb14',
    0xd015: 'ddbabY',
    0xd016: 'gggaa04',
    0xd017: 'ccbbaX',
    0xd018: 'fffaa04',
    0xd019: 'bbaabY',
    0xd01a: 'eeebb14',
    0xd01b: 'aabbaX',
    0xd01c: 'hhhbb04',
    0xd01d: 'ddabaX',
    0xd01e: 'cccba04',
    0xd01f: 'ccaabY',
    0xd020: 'bbbaa04',
    0xd021: 'bbbaaX',
    0xd022: 'aaabb14',
    0xd023: 'aaabbY',
    0xd024: 'dddab14',
    0xd025: 'ddbbbY',
    0xd026: 'gggba04',
    0xd027: 'ccbaaX',
    0xd028: 'fffba04',
    0xd029: 'bbabbY',
    0xd02a: 'eeeab14',
    0xd02b: 'aabaaX',
    0xd02c: 'hhhab04',
    0xd02d: 'ddaaaX',
    0xd02e: 'cccaa04',
    0xd02f: 'ccabbY',
    0xd030: 'bbbbb14',
    0xd031: 'bbbbbY',
    0xd032: 'aaaaa04',
    0xd033: 'aaaaaX',
    0xd034: 'dddba04',
    0xd035: 'ddbaaX',
    0xd036: 'gggab14',
    0xd037: 'ccbbbY',
    0xd038: 'fffab14',
    0xd039: 'bbaaaX',
    0xd03a: 'eeeba04',
    0xd03b: 'aabbbY',
    0xd03c: 'hhhba14',
    0xd03d: 'aaaaababn',
    0xd03e: 'cccbb14',
    0xd03f: 'bbbbbabao',
    0xd040: 'bbbab04',
    0xd041: 'bbbabX',
    0xd042: 'aaaba14',
    0xd043: 'aaabaY',
    0xd044: 'dddaa14',
    0xd045: 'ddbbaY',
    0xd046: 'gggbb04',
    0xd047: 'ccbabX',
    0xd048: 'fffbb04',
    0xd049: 'bbabaY',
    0xd04a: 'eeeaa14',
    0xd04b: 'aababX',
    0xd04c: 'hhhaa04',
    0xd04d: 'ddaabX',
    0xd04e: 'cccab04',
    0xd04f: 'ccabaY',
    0xd050: 'bbbba14',
    0xd051: 'bbbbaY',
    0xd052: 'aaaab04',
    0xd053: 'aaaabX',
    0xd054: 'dddbb04',
    0xd055: 'ddbabX',
    0xd056: 'gggaa14',
    0xd057: 'ccbbaY',
    0xd058: 'fffaa14',
    0xd059: 'bbaabX',
    0xd05a: 'eeebb04',
    0xd05b: 'aabbaY',
    0xd05c: 'hhhbb14',
    0xd05d: 'ddabaY',
    0xd05e: 'cccba14',
    0xd05f: 'ccaabX',
    0xd060: 'bbbaa14',
    0xd061: 'bbbaaY',
    0xd062: 'aaabb04',
    0xd063: 'aaabbX',
    0xd064: 'dddab04',
    0xd065: 'ddbbbX',
    0xd066: 'gggba14',
    0xd067: 'ccbaaY',
    0xd068: 'fffba14',
    0xd069: 'bbabbX',
    0xd06a: 'eeeab04',
    0xd06b: 'aabaaY',
    0xd06c: 'hhhab14',
    0xd06d: 'ddaaaY',
    0xd06e: 'cccaa14',
    0xd06f: 'ccabbX',
    0xd070: 'bbbbb04',
    0xd071: 'bbbbbX',
    0xd072: 'aaaaa14',
    0xd073: 'aaaaaY',
    0xd074: 'dddba14',
    0xd075: 'ddbaaY',
    0xd076: 'gggab04',
    0xd077: 'ccbbbX',
    0xd078: 'dddddda159',
    0xd079: 'bbaaaY',
    0xd07a: 'ddddddaM9',
    0xd07b: 'aabbbX',
    0xd07c: 'aaaaaabM9',
    0xd07d: 'aaaaaabao',
    0xd07e: 'aaaaaab159',
    0xd07f: 'bbbbbbabn',
    0xd080: 'bbbab15',
    0xd081: 'bbbadX',
    0xd082: 'aaaba05',
    0xd083: 'aaabaZ',
    0xd084: 'dddaa05',
    0xd085: 'ddbbaZ',
    0xd086: 'eeeeeeb059',
    0xd087: 'ccbadX',
    0xd088: 'fffbb15',
    0xd089: 'bbabaZ',
    0xd08a: 'eeeaa05',
    0xd08b: 'aabadX',
    0xd08c: 'hhhaa15',
    0xd08d: 'ddaadX',
    0xd08e: 'cccab15',
    0xd08f: 'ccabaZ',
    0xd090: 'bbbba05',
    0xd091: 'bbbbaZ',
    0xd092: 'aaaab15',
    0xd093: 'aaaadX',
    0xd094: 'dddbb15',
    0xd095: 'ddbadX',
    0xd096: 'gggaa05',
    0xd097: 'ccbbaZ',
    0xd098: 'fffaa05',
    0xd099: 'bbaadX',
    0xd09a: 'eeebb15',
    0xd09b: 'aabbaZ',
    0xd09c: 'hhhbb05',
    0xd09d: 'ddabaZ',
    0xd09e: 'cccba05',
    0xd09f: 'ccaadX',
    0xd0a0: 'bbbaa05',
    0xd0a1: 'bbbaaZ',
    0xd0a2: 'aaabb15',
    0xd0a3: 'aaabdX',
    0xd0a4: 'dddab15',
    0xd0a5: 'ddbbdX',
    0xd0a6: 'gggba05',
    0xd0a7: 'ccbaaZ',
    0xd0a8: 'fffba05',
    0xd0a9: 'bbabdX',
    0xd0aa: 'eeeab15',
    0xd0ab: 'aabaaZ',
    0xd0ac: 'hhhab05',
    0xd0ad: 'ddaaaZ',
    0xd0ae: 'cccaa05',
    0xd0af: 'ccabdX',
    0xd0b0: 'bbbbb15',
    0xd0b1: 'bbbbdX',
    0xd0b2: 'aaaaa05',
    0xd0b3: 'aaaaaZ',
    0xd0b4: 'dddba05',
    0xd0b5: 'ddbaaZ',
    0xd0b6: 'gggab15',
    0xd0b7: 'ccbbdX',
    0xd0b8: 'fffab15',
    0xd0b9: 'bbaaaZ',
    0xd0ba: 'eeeba05',
    0xd0bb: 'aabbdX',
    0xd0bc: 'hhhba15',
    0xd0bd: 'aaaaabl8',
    0xd0be: 'cccbb15',
    0xd0bf: 'bbbbbaabn',
    0xd0c0: 'bbbab05',
    0xd0c1: 'bbbabZ',
    0xd0c2: 'aaaba15',
    0xd0c3: 'aaabcZ',
    0xd0c4: 'dddaa15',
    0xd0c5: 'ddbbcZ',
    0xd0c6: 'gggbb05',
    0xd0c7: 'ccbabZ',
    0xd0c8: 'fffbb05',
    0xd0c9: 'bbabcZ',
    0xd0ca: 'eeeaa15',
    0xd0cb: 'aababZ',
    0xd0cc: 'hhhaa05',
    0xd0cd: 'ddaabZ',
    0xd0ce: 'cccab05',
    0xd0cf: 'ccabcZ',
    0xd0d0: 'bbbba15',
    0xd0d1: 'bbbbcZ',
    0xd0d2: 'aaaab05',
    0xd0d3: 'aaaabZ',
    0xd0d4: 'dddbb05',
    0xd0d5: 'ddbabZ',
    0xd0d6: 'gggaa15',
    0xd0d7: 'ccbbcZ',
    0xd0d8: 'fffaa15',
    0xd0d9: 'bbaabZ',
    0xd0da: 'eeebb05',
    0xd0db: 'aabbcZ',
    0xd0dc: 'hhhbb15',
    0xd0dd: 'ddabcZ',
    0xd0de: 'cccba15',
    0xd0df: 'ccaabZ',
    0xd0e0: 'bbbaa15',
    0xd0e1: 'bbbacZ',
    0xd0e2: 'aaabb05',
    0xd0e3: 'aaabbZ',
    0xd0e4: 'dddab05',
    0xd0e5: 'ddbbbZ',
    0xd0e6: 'gggba15',
    0xd0e7: 'ccbacZ',
    0xd0e8: 'fffba15',
    0xd0e9: 'bbabbZ',
    0xd0ea: 'eeeab05',
    0xd0eb: 'aabacZ',
    0xd0ec: 'hhhab15',
    0xd0ed: 'ddaacZ',
    0xd0ee: 'cccaa15',
    0xd0ef: 'ccabbZ',
    0xd0f0: 'bbbbb05',
    0xd0f1: 'bbbbbZ',
    0xd0f2: 'aaaaa15',
    0xd0f3: 'aaaacZ',
    0xd0f4: 'dddba15',
    0xd0f5: 'ddbacZ',
    0xd0f6: 'gggab05',
    0xd0f7: 'ccbbbZ',
    0xd0f8: 'aaaaaaabbbB',
    0xd0f9: 'aaaaaaabbbR',
    0xd0fa: 'aaaaaaabbbb',
    0xd0fb: 'aabbbZ',
    0xd0fc: 'aaaaaaaL9',
    0xd0fd: 'aaaaaaabn',
    0xd0fe: 'aaaaaaa059',
    0xd0ff: 'aaaaaaabB6',
    0xd100: 'bbbab16',
    0xd101: 'bbbahX',
    0xd102: 'aaaba06',
    0xd103: 'aaabiX',
    0xd104: 'cccccccbbbB',
    0xd105: 'cccccccbbbR',
    0xd106: 'cccccccbbbb',
    0xd107: 'ccbahX',
    0xd108: 'fffbb16',
    0xd109: 'bbabiX',
    0xd10a: 'eeeaa06',
    0xd10b: 'aabahX',
    0xd10c: 'hhhaa16',
    0xd10d: 'ddaahX',
    0xd10e: 'cccab16',
    0xd10f: 'ccabiX',
    0xd110: 'bbbba06',
    0xd111: 'bbbbiX',
    0xd112: 'aaaab16',
    0xd113: 'aaaahX',
    0xd114: 'dddbb16',
    0xd115: 'ddbahX',
    0xd116: 'gggaa06',
    0xd117: 'ccbbiX',
    0xd118: 'fffaa06',
    0xd119: 'bbaahX',
    0xd11a: 'eeebb16',
    0xd11b: 'aabbiX',
    0xd11c: 'hhhbb06',
    0xd11d: 'ddabiX',
    0xd11e: 'cccba06',
    0xd11f: 'ccaahX',
    0xd120: 'bbbaa06',
    0xd121: 'bbbaiX',
    0xd122: 'aaabb16',
    0xd123: 'aaabhX',
    0xd124: 'dddab16',
    0xd125: 'ddbbhX',
    0xd126: 'gggba06',
    0xd127: 'ccbaiX',
    0xd128: 'fffba06',
    0xd129: 'bbabhX',
    0xd12a: 'eeeab16',
    0xd12b: 'aabaiX',
    0xd12c: 'hhhab06',
    0xd12d: 'ddaaiX',
    0xd12e: 'cccaa06',
    0xd12f: 'ccabhX',
    0xd130: 'bbbbb16',
    0xd131: 'bbbbhX',
    0xd132: 'aaaaa06',
    0xd133: 'aaaaiX',
    0xd134: 'dddba06',
    0xd135: 'ddbaiX',
    0xd136: 'gggab16',
    0xd137: 'ccbbhX',
    0xd138: 'fffab16',
    0xd139: 'bbaaiX',
    0xd13a: 'eeeba06',
    0xd13b: 'aabbhX',
    0xd13c: 'hhhba16',
    0xd13d: 'aaaaabm9',
    0xd13e: 'cccbb16',
    0xd13f: 'bbbbbal9',
    0xd140: 'bbbab06',
    0xd141: 'bbbahY',
    0xd142: 'aaaba16',
    0xd143: 'aaabiY',
    0xd144: 'dddaa16',
    0xd145: 'ddbbiY',
    0xd146: 'gggbb06',
    0xd147: 'ccbahY',
    0xd148: 'fffbb06',
    0xd149: 'bbabiY',
    0xd14a: 'eeeaa16',
    0xd14b: 'aabahY',
    0xd14c: 'hhhaa06',
    0xd14d: 'ddaahY',
    0xd14e: 'cccab06',
    0xd14f: 'ccabiY',
    0xd150: 'bbbba16',
    0xd151: 'bbbbiY',
    0xd152: 'aaaab06',
    0xd153: 'aaaahY',
    0xd154: 'dddbb06',
    0xd155: 'ddbahY',
    0xd156: 'gggaa16',
    0xd157: 'ccbbiY',
    0xd158: 'fffaa16',
    0xd159: 'bbaahY',
    0xd15a: 'eeebb06',
    0xd15b: 'aabbiY',
    0xd15c: 'hhhbb16',
    0xd15d: 'ddabiY',
    0xd15e: 'cccba16',
    0xd15f: 'ccaahY',
    0xd160: 'bbbaa16',
    0xd161: 'bbbaiY',
    0xd162: 'aaabb06',
    0xd163: 'aaabhY',
    0xd164: 'dddab06',
    0xd165: 'ddbbhY',
    0xd166: 'gggba16',
    0xd167: 'ccbaiY',
    0xd168: 'fffba16',
    0xd169: 'bbabhY',
    0xd16a: 'eeeab06',
    0xd16b: 'aabaiY',
    0xd16c: 'hhhab16',
    0xd16d: 'ddaaiY',
    0xd16e: 'cccaa16',
    0xd16f: 'ccabhY',
    0xd170: 'bbbbb06',
    0xd171: 'bbbbhY',
    0xd172: 'aaaaa16',
    0xd173: 'aaaaiY',
    0xd174: 'dddba16',
    0xd175: 'ddbaiY',
    0xd176: 'gggab06',
    0xd177: 'ccbbhY',
    0xd178: 'dddddda059',
    0xd179: 'bbaaiY',
    0xd17a: 'ddddddaL9',
    0xd17b: 'aabbhY',
    0xd17c: 'aaaaaabL9',
    0xd17d: 'aaaaaal9',
    0xd17e: 'aaaaaab059',
    0xd17f: 'bbbbbbm9',
    0xd180: 'aaaaaaaaaaaabaZ',
    0xd181: 'aaaaaaaaaaaabcZ',
    0xd182: 'aaaba07',
    0xd183: 'aaabiZ',
    0xd184: 'aaaaaaaaaaaabiZ',
    0xd185: 'aaaaaaaaaaaabkZ',
    0xd186: 'aaaaaaaaaaaabiX',
    0xd187: 'aaaaaaaaaaaabiY',
    0xd188: 'aaaaaaaaaaaabaR',
    0xd189: 'aaaaaaaaaaaabaS',
    0xd18a: 'aaaaaaaaaaaabaP',
    0xd18b: 'aabahZ',
    0xd18c: 'aaaaaaaaaaaabaV',
    0xd18d: 'aaaaaaaaaaaabaW',
    0xd18e: 'aaaaaaaaaaaabaT',
    0xd18f: 'aaaaaaaaaaaabaU',
    0xd190: 'aaaaaaaaaaaabaJ',
    0xd191: 'aaaaaaaaaaaabaK',
    0xd192: 'aaaab17',
    0xd193: 'aaaahZ',
    0xd194: 'aaaaaaaaaaaabaN',
    0xd195: 'aaaaaaaaaaaabaO',
    0xd196: 'aaaaaaaaaaaabaL',
    0xd197: 'aaaaaaaaaaaabaM',
    0xd198: 'aaaaaaaaaaaabaB',
    0xd199: 'aaaaaaaaaaaabaC',
    0xd19a: 'aaaaaaaaaaaabcA',
    0xd19b: 'aabbiZ',
    0xd19c: 'aaaaaaaaaaaabaF',
    0xd19d: 'aaaaaaaaaaaabaG',
    0xd19e: 'aaaaaaaaaaaabaD',
    0xd19f: 'aaaaaaaaaaaabaE',
    0xd1a0: 'aaaaaaaaaaaabaz',
    0xd1a1: 'aaaaaaaaaaaabcz',
    0xd1a2: 'aaabb17',
    0xd1a3: 'aaabhZ',
    0xd1a4: 'aaaaaaaaaaaabiz',
    0xd1a5: 'aaaaaaaaaaaabkz',
    0xd1a6: 'aaaaaaaaaaaabix',
    0xd1a7: 'aaaaaaaaaaaabiy',
    0xd1a8: 'aaaaaaaaaaaabar',
    0xd1a9: 'aaaaaaaaaaaabas',
    0xd1aa: 'aaaaaaaaaaaabap',
    0xd1ab: 'aabaiZ',
    0xd1ac: 'aaaaaaaaaaaabav',
    0xd1ad: 'aaaaaaaaaaaabaw',
    0xd1ae: 'aaaaaaaaaaaabat',
    0xd1af: 'aaaaaaaaaaaabau',
    0xd1b0: 'aaaaaaaaaaaabaj',
    0xd1b1: 'aaaaaaaaaaaabak',
    0xd1b2: 'aaaaa07',
    0xd1b3: 'aaaaiZ',
    0xd1b4: 'aaaaaaaaaaaaban',
    0xd1b5: 'aaaaaaaaaaaabao',
    0xd1b6: 'aaaaaaaaaaaabal',
    0xd1b7: 'aaaaaaaaaaaabam',
    0xd1b8: 'aaaaaaaaaaaabab',
    0xd1b9: 'aaaaaaaaaaaabac',
    0xd1ba: 'aaaaaaaaaaaabca',
    0xd1bb: 'aabbhZ',
    0xd1bc: 'aaaaaaaaaaaabaf',
    0xd1bd: 'aaaaabl9',
    0xd1be: 'aaaaaaaaaaaabad',
    0xd1bf: 'aaaaaaaaaaaabae',
    0xd1c0: 'aaaaaaaaaaaabW',
    0xd1c1: 'aaaaaaaaaaaabU',
    0xd1c2: 'aaaba17',
    0xd1c3: 'aaabkZ',
    0xd1c4: 'aaaaaaaaaaaahZ',
    0xd1c5: 'aaaaaaaaaaaahX',
    0xd1c6: 'aaaaaaaaaaaadX',
    0xd1c7: 'aaaaaaaaaaaabY',
    0xd1c8: 'aaaaaaaaaaaabG',
    0xd1c9: 'aaaaaaaaaaaabE',
    0xd1ca: 'aaaaaaaaaaaabC',
    0xd1cb: 'aabajZ',
    0xd1cc: 'aaaaaaaaaaaabO',
    0xd1cd: 'aaaaaaaaaaaabM',
    0xd1ce: 'aaaaaaaaaaaabK',
    0xd1cf: 'aaaaaaaaaaaabI',
    0xd1d0: 'aaaaaaaaaaaabw',
    0xd1d1: 'aaaaaaaaaaaabu',
    0xd1d2: 'aaaab07',
    0xd1d3: 'aaaajZ',
    0xd1d4: 'aaaaaaaaaaaahz',
    0xd1d5: 'aaaaaaaaaaaahx',
    0xd1d6: 'aaaaaaaaaaaadx',
    0xd1d7: 'aaaaaaaaaaaaby',
    0xd1d8: 'aaaaaaaaaaaabg',
    0xd1d9: 'aaaaaaaaaaaabe',
    0xd1da: 'aaaaaaaaaaaabc',
    0xd1db: 'aabbkZ',
    0xd1dc: 'aaaaaaaaaaaabo',
    0xd1dd: 'aaaaaaaaaaaabm',
    0xd1de: 'aaaaaaaaaaaabk',
    0xd1df: 'aaaaaaaaaaaabi',
    0xd1e0: 'aaaaaaaaaaacA',
    0xd1e1: 'aaaaaaaaaaaaD',
    0xd1e2: 'aaabb07',
    0xd1e3: 'aaabjZ',
    0xd1e4: 'aaaaaaaaaaaaP',
    0xd1e5: 'aaaaaaaaaaaaT',
    0xd1e6: 'aaaaaaaaaaaaX',
    0xd1e7: 'aaaaaaaaaaaiX',
    0xd1e8: 'aaaaaaaaaaaca',
    0xd1e9: 'aaaaaaaaaaaad',
    0xd1ea: 'aaaaaaaaaaaah',
    0xd1eb: 'aabakZ',
    0xd1ec: 'aaaaaaaaaaaap',
    0xd1ed: 'aaaaaaaaaaaat',
    0xd1ee: 'aaaaaaaaaaaax',
    0xd1ef: 'aaaaaaaaaaaix',
    0xd1f0: 'aaaaaaaaaaai',
    0xd1f1: 'aaaaaaaaaaaa',
    0xd1f2: 'aaaaa17',
    0xd1f3: 'aaaakZ',
    0xd1f4: 'aaaaaaaaaaaI',
    0xd1f5: 'aaaaaaaaaaaA',
    0xd1f6: 'aaaaaaaaaaaY',
    0xd1f7: 'aaaaaaaaaaaQ',
    0xd1f8: 'aaaaaaaaaaC',
    0xd1f9: 'aaaaaaaaaaS',
    0xd1fa: 'aaaaaaaaaac',
    0xd1fb: 'aabbjZ',
    0xd1fc: 'aaaaaaaM9',
    0xd1fd: 'aaaaaam9',
    0xd1fe: 'aaaaaaa159',
    0xd1ff: 'aaaaaaaaA7',
    0xd200: 'bbbab10',
    0xd201: 'bbbabQ',
    0xd202: 'aaaba00',
    0xd203: 'aaabaP',
    0xd204: 'cccccccabbB',
    0xd205: 'cccccccabbR',
    0xd206: 'cccccccabbb',
    0xd207: 'ccbabQ',
    0xd208: 'ccccccccaaaI',
    0xd209: 'bbabaP',
    0xd20a: 'ccccccccaaaY',
    0xd20b: 'aababQ',
    0xd20c: 'ccccccccaaai',
    0xd20d: 'ccccccccaaaa',
    0xd20e: 'cccab10',
    0xd20f: 'ccabaP',
    0xd210: 'bbbba00',
    0xd211: 'bbbbaP',
    0xd212: 'aaaab10',
    0xd213: 'aaaabQ',
    0xd214: 'cccccccccbbba',
    0xd215: 'cccccccccbbbe',
    0xd216: 'cccccccccbbbi',
    0xd217: 'ccbbaP',
    0xd218: 'cccccccccbbbQ',
    0xd219: 'bbaabQ',
    0xd21a: 'cccccccccbbbY',
    0xd21b: 'aabbaP',
    0xd21c: 'cccccccccbbbA',
    0xd21d: 'cccccccccbbbE',
    0xd21e: 'cccba00',
    0xd21f: 'ccaabQ',
    0xd220: 'bbbaa00',
    0xd221: 'bbbaaP',
    0xd222: 'aaabb10',
    0xd223: 'aaabbQ',
    0xd224: 'dddab10',
    0xd225: 'ddbbbQ',
    0xd226: 'gggba00',
    0xd227: 'ccbaaP',
    0xd228: 'fffba00',
    0xd229: 'bbabbQ',
    0xd22a: 'eeeab10',
    0xd22b: 'aabaaP',
    0xd22c: 'hhhab00',
    0xd22d: 'ddaaaP',
    0xd22e: 'cccaa00',
    0xd22f: 'ccabbQ',
    0xd230: 'bbbbb10',
    0xd231: 'bbbbbQ',
    0xd232: 'aaaaa00',
    0xd233: 'aaaaaP',
    0xd234: 'dddba00',
    0xd235: 'ddbaaP',
    0xd236: 'gggab10',
    0xd237: 'ccbbbQ',
    0xd238: 'fffab10',
    0xd239: 'bbaaaP',
    0xd23a: 'eeeba00',
    0xd23b: 'aabbbQ',
    0xd23c: 'hhhba10',
    0xd23d: 'aaaaababo',
    0xd23e: 'cccbb10',
    0xd23f: 'bbbbbaban',
    0xd240: 'bbbab00',
    0xd241: 'bbbabP',
    0xd242: 'aaaba10',
    0xd243: 'aaabaQ',
    0xd244: 'dddaa10',
    0xd245: 'ddbbaQ',
    0xd246: 'gggbb00',
    0xd247: 'ccbabP',
    0xd248: 'fffbb00',
    0xd249: 'bbabaQ',
    0xd24a: 'eeeaa10',
    0xd24b: 'aababP',
    0xd24c: 'hhhaa00',
    0xd24d: 'ddaabP',
    0xd24e: 'cccab00',
    0xd24f: 'ccabaQ',
    0xd250: 'bbbba10',
    0xd251: 'bbbbaQ',
    0xd252: 'aaaab00',
    0xd253: 'aaaabP',
    0xd254: 'dddbb00',
    0xd255: 'ddbabP',
    0xd256: 'gggaa10',
    0xd257: 'ccbbaQ',
    0xd258: 'fffaa10',
    0xd259: 'bbaabP',
    0xd25a: 'eeebb00',
    0xd25b: 'aabbaQ',
    0xd25c: 'hhhbb10',
    0xd25d: 'ddabaQ',
    0xd25e: 'cccba10',
    0xd25f: 'ccaabP',
    0xd260: 'bbbaa10',
    0xd261: 'bbbaaQ',
    0xd262: 'aaabb00',
    0xd263: 'aaabbP',
    0xd264: 'dddab00',
    0xd265: 'ddbbbP',
    0xd266: 'gggba10',
    0xd267: 'ccbaaQ',
    0xd268: 'fffba10',
    0xd269: 'bbabbP',
    0xd26a: 'eeeab00',
    0xd26b: 'aabaaQ',
    0xd26c: 'hhhab10',
    0xd26d: 'ddaaaQ',
    0xd26e: 'cccaa10',
    0xd26f: 'ccabbP',
    0xd270: 'bbbbb00',
    0xd271: 'bbbbbP',
    0xd272: 'aaaaa10',
    0xd273: 'aaaaaQ',
    0xd274: 'dddba10',
    0xd275: 'ddbaaQ',
    0xd276: 'gggab00',
    0xd277: 'ccbbbP',
    0xd278: 'dddddda149',
    0xd279: 'bbaaaQ',
    0xd27a: 'ddddddaM8',
    0xd27b: 'aabbbP',
    0xd27c: 'aaaaaabM8',
    0xd27d: 'aaaaaaban',
    0xd27e: 'aaaaaab149',
    0xd27f: 'bbbbbbabo',
    0xd280: 'bbbab11',
    0xd281: 'bbbabS',
    0xd282: 'aaaba01',
    0xd283: 'aaabaR',
    0xd284: 'dddaa01',
    0xd285: 'ddbbaR',
    0xd286: 'eeeeeeb049',
    0xd287: 'ccbabS',
    0xd288: 'fffbb11',
    0xd289: 'bbabaR',
    0xd28a: 'eeeaa01',
    0xd28b: 'aababS',
    0xd28c: 'hhhaa11',
    0xd28d: 'ddaabS',
    0xd28e: 'cccab11',
    0xd28f: 'ccabaR',
    0xd290: 'bbbba01',
    0xd291: 'bbbbaR',
    0xd292: 'aaaab11',
    0xd293: 'aaaabS',
    0xd294: 'dddbb11',
    0xd295: 'ddbabS',
    0xd296: 'gggaa01',
    0xd297: 'ccbbaR',
    0xd298: 'fffaa01',
    0xd299: 'bbaabS',
    0xd29a: 'eeebb11',
    0xd29b: 'aabbaR',
    0xd29c: 'hhhbb01',
    0xd29d: 'ddabaR',
    0xd29e: 'cccba01',
    0xd29f: 'ccaabS',
    0xd2a0: 'bbbaa01',
    0xd2a1: 'bbbaaR',
    0xd2a2: 'aaabb11',
    0xd2a3: 'aaabbS',
    0xd2a4: 'dddab11',
    0xd2a5: 'ddbbbS',
    0xd2a6: 'gggba01',
    0xd2a7: 'ccbaaR',
    0xd2a8: 'fffba01',
    0xd2a9: 'bbabbS',
    0xd2aa: 'eeeab11',
    0xd2ab: 'aabaaR',
    0xd2ac: 'hhhab01',
    0xd2ad: 'ddaaaR',
    0xd2ae: 'cccaa01',
    0xd2af: 'ccabbS',
    0xd2b0: 'bbbbb11',
    0xd2b1: 'bbbbbS',
    0xd2b2: 'aaaaa01',
    0xd2b3: 'aaaaaR',
    0xd2b4: 'dddba01',
    0xd2b5: 'ddbaaR',
    0xd2b6: 'gggab11',
    0xd2b7: 'ccbbbS',
    0xd2b8: 'fffab11',
    0xd2b9: 'bbaaaR',
    0xd2ba: 'eeeba01',
    0xd2bb: 'aabbbS',
    0xd2bc: 'hhhba11',
    0xd2bd: 'aaaaabban',
    0xd2be: 'cccbb11',
    0xd2bf: 'bbbbbaabo',
    0xd2c0: 'bbbab01',
    0xd2c1: 'bbbabR',
    0xd2c2: 'aaaba11',
    0xd2c3: 'aaabaS',
    0xd2c4: 'dddaa11',
    0xd2c5: 'ddbbaS',
    0xd2c6: 'gggbb01',
    0xd2c7: 'ccbabR',
    0xd2c8: 'fffbb01',
    0xd2c9: 'bbabaS',
    0xd2ca: 'eeeaa11',
    0xd2cb: 'aababR',
    0xd2cc: 'hhhaa01',
    0xd2cd: 'ddaabR',
    0xd2ce: 'cccab01',
    0xd2cf: 'ccabaS',
    0xd2d0: 'bbbba11',
    0xd2d1: 'bbbbaS',
    0xd2d2: 'aaaab01',
    0xd2d3: 'aaaabR',
    0xd2d4: 'dddbb01',
    0xd2d5: 'ddbabR',
    0xd2d6: 'gggaa11',
    0xd2d7: 'ccbbaS',
    0xd2d8: 'fffaa11',
    0xd2d9: 'bbaabR',
    0xd2da: 'eeebb01',
    0xd2db: 'aabbaS',
    0xd2dc: 'hhhbb11',
    0xd2dd: 'ddabaS',
    0xd2de: 'cccba11',
    0xd2df: 'ccaabR',
    0xd2e0: 'bbbaa11',
    0xd2e1: 'bbbaaS',
    0xd2e2: 'aaabb01',
    0xd2e3: 'aaabbR',
    0xd2e4: 'dddab01',
    0xd2e5: 'ddbbbR',
    0xd2e6: 'gggba11',
    0xd2e7: 'ccbaaS',
    0xd2e8: 'fffba11',
    0xd2e9: 'bbabbR',
    0xd2ea: 'eeeab01',
    0xd2eb: 'aabaaS',
    0xd2ec: 'hhhab11',
    0xd2ed: 'ddaaaS',
    0xd2ee: 'cccaa11',
    0xd2ef: 'ccabbR',
    0xd2f0: 'bbbbb01',
    0xd2f1: 'bbbbbR',
    0xd2f2: 'aaaaa11',
    0xd2f3: 'aaaaaS',
    0xd2f4: 'dddba11',
    0xd2f5: 'ddbaaS',
    0xd2f6: 'gggab01',
    0xd2f7: 'ccbbbR',
    0xd2f8: 'aaaaaaabaaC',
    0xd2f9: 'aaaaaaabaaS',
    0xd2fa: 'aaaaaaabaac',
    0xd2fb: 'aabbbR',
    0xd2fc: 'aaaaaaaL8',
    0xd2fd: 'aaaaaaabo',
    0xd2fe: 'aaaaaaa049',
    0xd2ff: 'aaaaaaabA7',
    0xd300: 'bbbab12',
    0xd301: 'bbbabU',
    0xd302: 'aaaba02',
    0xd303: 'aaabaT',
    0xd304: 'cccccccbaaC',
    0xd305: 'cccccccbaaS',
    0xd306: 'cccccccbaac',
    0xd307: 'ccbabU',
    0xd308: 'fffbb12',
    0xd309: 'bbabaT',
    0xd30a: 'eeeaa02',
    0xd30b: 'aababU',
    0xd30c: 'hhhaa12',
    0xd30d: 'ddaabU',
    0xd30e: 'cccab12',
    0xd30f: 'ccabaT',
    0xd310: 'bbbba02',
    0xd311: 'bbbbaT',
    0xd312: 'aaaab12',
    0xd313: 'aaaabU',
    0xd314: 'dddbb12',
    0xd315: 'ddbabU',
    0xd316: 'gggaa02',
    0xd317: 'ccbbaT',
    0xd318: 'fffaa02',
    0xd319: 'bbaabU',
    0xd31a: 'eeebb12',
    0xd31b: 'aabbaT',
    0xd31c: 'hhhbb02',
    0xd31d: 'ddabaT',
    0xd31e: 'cccba02',
    0xd31f: 'ccaabU',
    0xd320: 'bbbaa02',
    0xd321: 'bbbaaT',
    0xd322: 'aaabb12',
    0xd323: 'aaabbU',
    0xd324: 'dddab12',
    0xd325: 'ddbbbU',
    0xd326: 'gggba02',
    0xd327: 'ccbaaT',
    0xd328: 'fffba02',
    0xd329: 'bbabbU',
    0xd32a: 'eeeab12',
    0xd32b: 'aabaaT',
    0xd32c: 'hhhab02',
    0xd32d: 'ddaaaT',
    0xd32e: 'cccaa02',
    0xd32f: 'ccabbU',
    0xd330: 'bbbbb12',
    0xd331: 'bbbbbU',
    0xd332: 'aaaaa02',
    0xd333: 'aaaaaT',
    0xd334: 'dddba02',
    0xd335: 'ddbaaT',
    0xd336: 'gggab12',
    0xd337: 'ccbbbU',
    0xd338: 'fffab12',
    0xd339: 'bbaaaT',
    0xd33a: 'eeeba02',
    0xd33b: 'aabbbU',
    0xd33c: 'hhhba12',
    0xd33d: 'aaaaabaan',
    0xd33e: 'cccbb12',
    0xd33f: 'bbbbbabbo',
    0xd340: 'bbbab02',
    0xd341: 'bbbabT',
    0xd342: 'aaaba12',
    0xd343: 'aaabaU',
    0xd344: 'dddaa12',
    0xd345: 'ddbbaU',
    0xd346: 'gggbb02',
    0xd347: 'ccbabT',
    0xd348: 'fffbb02',
    0xd349: 'bbabaU',
    0xd34a: 'eeeaa12',
    0xd34b: 'aababT',
    0xd34c: 'hhhaa02',
    0xd34d: 'ddaabT',
    0xd34e: 'cccab02',
    0xd34f: 'ccabaU',
    0xd350: 'bbbba12',
    0xd351: 'bbbbaU',
    0xd352: 'aaaab02',
    0xd353: 'aaaabT',
    0xd354: 'dddbb02',
    0xd355: 'ddbabT',
    0xd356: 'gggaa12',
    0xd357: 'ccbbaU',
    0xd358: 'fffaa12',
    0xd359: 'bbaabT',
    0xd35a: 'eeebb02',
    0xd35b: 'aabbaU',
    0xd35c: 'hhhbb12',
    0xd35d: 'ddabaU',
    0xd35e: 'cccba12',
    0xd35f: 'ccaabT',
    0xd360: 'bbbaa12',
    0xd361: 'bbbaaU',
    0xd362: 'aaabb02',
    0xd363: 'aaabbT',
    0xd364: 'dddab02',
    0xd365: 'ddbbbT',
    0xd366: 'gggba12',
    0xd367: 'ccbaaU',
    0xd368: 'fffba12',
    0xd369: 'bbabbT',
    0xd36a: 'eeeab02',
    0xd36b: 'aabaaU',
    0xd36c: 'hhhab12',
    0xd36d: 'ddaaaU',
    0xd36e: 'cccaa12',
    0xd36f: 'ccabbT',
    0xd370: 'bbbbb02',
    0xd371: 'bbbbbT',
    0xd372: 'aaaaa12',
    0xd373: 'aaaaaU',
    0xd374: 'dddba12',
    0xd375: 'ddbaaU',
    0xd376: 'gggab02',
    0xd377: 'ccbbbT',
    0xd378: 'dddddda049',
    0xd379: 'bbaaaU',
    0xd37a: 'ddddddaL8',
    0xd37b: 'aabbbT',
    0xd37c: 'aaaaaabL8',
    0xd37d: 'aaaaaabbo',
    0xd37e: 'aaaaaab049',
    0xd37f: 'bbbbbbaan',
    0xd380: 'bbbab13',
    0xd381: 'bbbabW',
    0xd382: 'aaaba03',
    0xd383: 'aaabaV',
    0xd384: 'dddaa03',
    0xd385: 'ddbbaV',
    0xd386: 'eeeeeeb149',
    0xd387: 'ccbabW',
    0xd388: 'fffbb13',
    0xd389: 'bbabaV',
    0xd38a: 'eeeaa03',
    0xd38b: 'aababW',
    0xd38c: 'hhhaa13',
    0xd38d: 'ddaabW',
    0xd38e: 'cccab13',
    0xd38f: 'ccabaV',
    0xd390: 'bbbba03',
    0xd391: 'bbbbaV',
    0xd392: 'aaaab13',
    0xd393: 'aaaabW',
    0xd394: 'dddbb13',
    0xd395: 'ddbabW',
    0xd396: 'gggaa03',
    0xd397: 'ccbbaV',
    0xd398: 'fffaa03',
    0xd399: 'bbaabW',
    0xd39a: 'eeebb13',
    0xd39b: 'aabbaV',
    0xd39c: 'hhhbb03',
    0xd39d: 'ddabaV',
    0xd39e: 'cccba03',
    0xd39f: 'ccaabW',
    0xd3a0: 'bbbaa03',
    0xd3a1: 'bbbaaV',
    0xd3a2: 'aaabb13',
    0xd3a3: 'aaabbW',
    0xd3a4: 'dddab13',
    0xd3a5: 'ddbbbW',
    0xd3a6: 'gggba03',
    0xd3a7: 'ccbaaV',
    0xd3a8: 'fffba03',
    0xd3a9: 'bbabbW',
    0xd3aa: 'eeeab13',
    0xd3ab: 'aabaaV',
    0xd3ac: 'hhhab03',
    0xd3ad: 'ddaaaV',
    0xd3ae: 'cccaa03',
    0xd3af: 'ccabbW',
    0xd3b0: 'bbbbb13',
    0xd3b1: 'bbbbbW',
    0xd3b2: 'aaaaa03',
    0xd3b3: 'aaaaaV',
    0xd3b4: 'dddba03',
    0xd3b5: 'ddbaaV',
    0xd3b6: 'gggab13',
    0xd3b7: 'ccbbbW',
    0xd3b8: 'fffab13',
    0xd3b9: 'bbaaaV',
    0xd3ba: 'eeeba03',
    0xd3bb: 'aabbbW',
    0xd3bc: 'hhhba13',
    0xd3bd: 'aaaaabbbo',
    0xd3be: 'cccbb13',
    0xd3bf: 'bbbbbaaan',
    0xd3c0: 'bbbab03',
    0xd3c1: 'bbbabV',
    0xd3c2: 'aaaba13',
    0xd3c3: 'aaabaW',
    0xd3c4: 'dddaa13',
    0xd3c5: 'ddbbaW',
    0xd3c6: 'gggbb03',
    0xd3c7: 'ccbabV',
    0xd3c8: 'fffbb03',
    0xd3c9: 'bbabaW',
    0xd3ca: 'eeeaa13',
    0xd3cb: 'aababV',
    0xd3cc: 'hhhaa03',
    0xd3cd: 'ddaabV',
    0xd3ce: 'cccab03',
    0xd3cf: 'ccabaW',
    0xd3d0: 'bbbba13',
    0xd3d1: 'bbbbaW',
    0xd3d2: 'aaaab03',
    0xd3d3: 'aaaabV',
    0xd3d4: 'dddbb03',
    0xd3d5: 'ddbabV',
    0xd3d6: 'gggaa13',
    0xd3d7: 'ccbbaW',
    0xd3d8: 'fffaa13',
    0xd3d9: 'bbaabV',
    0xd3da: 'eeebb03',
    0xd3db: 'aabbaW',
    0xd3dc: 'hhhbb13',
    0xd3dd: 'ddabaW',
    0xd3de: 'cccba13',
    0xd3df: 'ccaabV',
    0xd3e0: 'bbbaa13',
    0xd3e1: 'bbbaaW',
    0xd3e2: 'aaabb03',
    0xd3e3: 'aaabbV',
    0xd3e4: 'dddab03',
    0xd3e5: 'ddbbbV',
    0xd3e6: 'gggba13',
    0xd3e7: 'ccbaaW',
    0xd3e8: 'fffba13',
    0xd3e9: 'bbabbV',
    0xd3ea: 'eeeab03',
    0xd3eb: 'aabaaW',
    0xd3ec: 'hhhab13',
    0xd3ed: 'ddaaaW',
    0xd3ee: 'cccaa13',
    0xd3ef: 'ccabbV',
    0xd3f0: 'aaaaaaaabbbh',
    0xd3f1: 'aaaaaaaabbdc',
    0xd3f2: 'aaaaa13',
    0xd3f3: 'aaaaaW',
    0xd3f4: 'aaaaaaaabbbH',
    0xd3f5: 'aaaaaaaabbdC',
    0xd3f6: 'aaaaaaaabbbX',
    0xd3f7: 'aaaaaaaabbbP',
    0xd3f8: 'aaaaaaaabbB',
    0xd3f9: 'aaaaaaaabbR',
    0xd3fa: 'aaaaaaaabbb',
    0xd3fb: 'aabbbV',
    0xd3fc: 'aaaaaaaM8',
    0xd3fd: 'aaaaaaaan',
    0xd3fe: 'aaaaaaa149',
    0xd3ff: 'aaaaaaaaB6',
    0xd400: 'bbbab98',
    0xd401: 'bbbabI',
    0xd402: 'aaaba88',
    0xd403: 'aaabaH',
    0xd404: 'cccccccaabB',
    0xd405: 'cccccccaabR',
    0xd406: 'cccccccaabb',
    0xd407: 'ccbabI',
    0xd408: 'ccccccccbaaI',
    0xd409: 'bbabaH',
    0xd40a: 'ccccccccbaaY',
    0xd40b: 'aababI',
    0xd40c: 'ccccccccbaai',
    0xd40d: 'ccccccccbaaa',
    0xd40e: 'cccab98',
    0xd40f: 'ccabaH',
    0xd410: 'bbbba88',
    0xd411: 'bbbbaH',
    0xd412: 'aaaab98',
    0xd413: 'aaaabI',
    0xd414: 'dddbb98',
    0xd415: 'ddbabI',
    0xd416: 'gggaa88',
    0xd417: 'ccbbaH',
    0xd418: 'fffaa88',
    0xd419: 'bbaabI',
    0xd41a: 'eeebb98',
    0xd41b: 'aabbaH',
    0xd41c: 'hhhbb88',
    0xd41d: 'ddabaH',
    0xd41e: 'cccba88',
    0xd41f: 'ccaabI',
    0xd420: 'bbbaa88',
    0xd421: 'bbbaaH',
    0xd422: 'aaabb98',
    0xd423: 'aaabbI',
    0xd424: 'dddab98',
    0xd425: 'ddbbbI',
    0xd426: 'gggba88',
    0xd427: 'ccbaaH',
    0xd428: 'fffba88',
    0xd429: 'bbabbI',
    0xd42a: 'eeeab98',
    0xd42b: 'aabaaH',
    0xd42c: 'hhhab88',
    0xd42d: 'ddaaaH',
    0xd42e: 'cccaa88',
    0xd42f: 'ccabbI',
    0xd430: 'bbbbb98',
    0xd431: 'bbbbbI',
    0xd432: 'aaaaa88',
    0xd433: 'aaaaaH',
    0xd434: 'dddba88',
    0xd435: 'ddbaaH',
    0xd436: 'gggab98',
    0xd437: 'ccbbbI',
    0xd438: 'fffab98',
    0xd439: 'bbaaaH',
    0xd43a: 'eeeba88',
    0xd43b: 'aabbbI',
    0xd43c: 'hhhba98',
    0xd43d: 'aaaaababl',
    0xd43e: 'cccbb98',
    0xd43f: 'bbbbbabam',
    0xd440: 'bbbab88',
    0xd441: 'bbbabH',
    0xd442: 'aaaba98',
    0xd443: 'aaabaI',
    0xd444: 'dddaa98',
    0xd445: 'ddbbaI',
    0xd446: 'gggbb88',
    0xd447: 'ccbabH',
    0xd448: 'fffbb88',
    0xd449: 'bbabaI',
    0xd44a: 'eeeaa98',
    0xd44b: 'aababH',
    0xd44c: 'hhhaa88',
    0xd44d: 'ddaabH',
    0xd44e: 'cccab88',
    0xd44f: 'ccabaI',
    0xd450: 'bbbba98',
    0xd451: 'bbbbaI',
    0xd452: 'aaaab88',
    0xd453: 'aaaabH',
    0xd454: 'dddbb88',
    0xd455: 'ddbabH',
    0xd456: 'gggaa98',
    0xd457: 'ccbbaI',
    0xd458: 'fffaa98',
    0xd459: 'bbaabH',
    0xd45a: 'eeebb88',
    0xd45b: 'aabbaI',
    0xd45c: 'hhhbb98',
    0xd45d: 'ddabaI',
    0xd45e: 'cccba98',
    0xd45f: 'ccaabH',
    0xd460: 'bbbaa98',
    0xd461: 'bbbaaI',
    0xd462: 'aaabb88',
    0xd463: 'aaabbH',
    0xd464: 'dddab88',
    0xd465: 'ddbbbH',
    0xd466: 'gggba98',
    0xd467: 'ccbaaI',
    0xd468: 'fffba98',
    0xd469: 'bbabbH',
    0xd46a: 'eeeab88',
    0xd46b: 'aabaaI',
    0xd46c: 'hhhab98',
    0xd46d: 'ddaaaI',
    0xd46e: 'cccaa98',
    0xd46f: 'ccabbH',
    0xd470: 'bbbbb88',
    0xd471: 'bbbbbH',
    0xd472: 'aaaaa98',
    0xd473: 'aaaaaI',
    0xd474: 'dddba98',
    0xd475: 'ddbaaI',
    0xd476: 'gggab88',
    0xd477: 'ccbbbH',
    0xd478: 'dddddda158',
    0xd479: 'bbaaaI',
    0xd47a: 'ddddddaI9',
    0xd47b: 'aabbbH',
    0xd47c: 'aaaaaabI9',
    0xd47d: 'aaaaaabam',
    0xd47e: 'aaaaaab158',
    0xd47f: 'bbbbbbabl',
    0xd480: 'bbbab99',
    0xd481: 'bbbabK',
    0xd482: 'aaaba89',
    0xd483: 'aaabaJ',
    0xd484: 'dddaa89',
    0xd485: 'ddbbaJ',
    0xd486: 'eeeeeeb058',
    0xd487: 'ccbabK',
    0xd488: 'fffbb99',
    0xd489: 'bbabaJ',
    0xd48a: 'eeeaa89',
    0xd48b: 'aababK',
    0xd48c: 'hhhaa99',
    0xd48d: 'ddaabK',
    0xd48e: 'cccab99',
    0xd48f: 'ccabaJ',
    0xd490: 'bbbba89',
    0xd491: 'bbbbaJ',
    0xd492: 'aaaab99',
    0xd493: 'aaaabK',
    0xd494: 'dddbb99',
    0xd495: 'ddbabK',
    0xd496: 'gggaa89',
    0xd497: 'ccbbaJ',
    0xd498: 'fffaa89',
    0xd499: 'bbaabK',
    0xd49a: 'eeebb99',
    0xd49b: 'aabbaJ',
    0xd49c: 'hhhbb89',
    0xd49d: 'ddabaJ',
    0xd49e: 'cccba89',
    0xd49f: 'ccaabK',
    0xd4a0: 'bbbaa89',
    0xd4a1: 'bbbaaJ',
    0xd4a2: 'aaabb99',
    0xd4a3: 'aaabbK',
    0xd4a4: 'dddab99',
    0xd4a5: 'ddbbbK',
    0xd4a6: 'gggba89',
    0xd4a7: 'ccbaaJ',
    0xd4a8: 'fffba89',
    0xd4a9: 'bbabbK',
    0xd4aa: 'eeeab99',
    0xd4ab: 'aabaaJ',
    0xd4ac: 'hhhab89',
    0xd4ad: 'ddaaaJ',
    0xd4ae: 'cccaa89',
    0xd4af: 'ccabbK',
    0xd4b0: 'bbbbb99',
    0xd4b1: 'bbbbbK',
    0xd4b2: 'aaaaa89',
    0xd4b3: 'aaaaaJ',
    0xd4b4: 'dddba89',
    0xd4b5: 'ddbaaJ',
    0xd4b6: 'gggab99',
    0xd4b7: 'ccbbbK',
    0xd4b8: 'fffab99',
    0xd4b9: 'bbaaaJ',
    0xd4ba: 'eeeba89',
    0xd4bb: 'aabbbK',
    0xd4bc: 'hhhba99',
    0xd4bd: 'aaaaabbam',
    0xd4be: 'cccbb99',
    0xd4bf: 'bbbbbaabl',
    0xd4c0: 'bbbab89',
    0xd4c1: 'bbbabJ',
    0xd4c2: 'aaaba99',
    0xd4c3: 'aaabaK',
    0xd4c4: 'dddaa99',
    0xd4c5: 'ddbbaK',
    0xd4c6: 'gggbb89',
    0xd4c7: 'ccbabJ',
    0xd4c8: 'fffbb89',
    0xd4c9: 'bbabaK',
    0xd4ca: 'eeeaa99',
    0xd4cb: 'aababJ',
    0xd4cc: 'hhhaa89',
    0xd4cd: 'ddaabJ',
    0xd4ce: 'cccab89',
    0xd4cf: 'ccabaK',
    0xd4d0: 'bbbba99',
    0xd4d1: 'bbbbaK',
    0xd4d2: 'aaaab89',
    0xd4d3: 'aaaabJ',
    0xd4d4: 'dddbb89',
    0xd4d5: 'ddbabJ',
    0xd4d6: 'gggaa99',
    0xd4d7: 'ccbbaK',
    0xd4d8: 'fffaa99',
    0xd4d9: 'bbaabJ',
    0xd4da: 'eeebb89',
    0xd4db: 'aabbaK',
    0xd4dc: 'hhhbb99',
    0xd4dd: 'ddabaK',
    0xd4de: 'cccba99',
    0xd4df: 'ccaabJ',
    0xd4e0: 'bbbaa99',
    0xd4e1: 'bbbaaK',
    0xd4e2: 'aaabb89',
    0xd4e3: 'aaabbJ',
    0xd4e4: 'dddab89',
    0xd4e5: 'ddbbbJ',
    0xd4e6: 'gggba99',
    0xd4e7: 'ccbaaK',
    0xd4e8: 'fffba99',
    0xd4e9: 'bbabbJ',
    0xd4ea: 'eeeab89',
    0xd4eb: 'aabaaK',
    0xd4ec: 'hhhab99',
    0xd4ed: 'ddaaaK',
    0xd4ee: 'cccaa99',
    0xd4ef: 'ccabbJ',
    0xd4f0: 'bbbbb89',
    0xd4f1: 'bbbbbJ',
    0xd4f2: 'aaaaa99',
    0xd4f3: 'aaaaaK',
    0xd4f4: 'dddba99',
    0xd4f5: 'ddbaaK',
    0xd4f6: 'gggab89',
    0xd4f7: 'ccbbbJ',
    0xd4f8: 'aaaaaaabbaC',
    0xd4f9: 'aaaaaaabbaS',
    0xd4fa: 'aaaaaaabbac',
    0xd4fb: 'aabbbJ',
    0xd4fc: 'aaaaaaaH9',
    0xd4fd: 'aaaaaaabl',
    0xd4fe: 'aaaaaaa058',
    0xd4ff: 'aaaaaaabB7',
    0xd500: 'bbbah88',
    0xd501: 'bbbabM',
    0xd502: 'aaabi88',
    0xd503: 'aaabaL',
    0xd504: 'cccccccbbaC',
    0xd505: 'cccccccbbaS',
    0xd506: 'cccccccbbac',
    0xd507: 'ccbabM',
    0xd508: 'fffbh88',
    0xd509: 'bbabaL',
    0xd50a: 'eeeai88',
    0xd50b: 'aababM',
    0xd50c: 'hhhai98',
    0xd50d: 'ddaabM',
    0xd50e: 'cccah88',
    0xd50f: 'ccabaL',
    0xd510: 'bbbbi88',
    0xd511: 'bbbbaL',
    0xd512: 'aaaah88',
    0xd513: 'aaaabM',
    0xd514: 'dddbh88',
    0xd515: 'ddbabM',
    0xd516: 'gggai88',
    0xd517: 'ccbbaL',
    0xd518: 'fffai88',
    0xd519: 'bbaabM',
    0xd51a: 'eeebh88',
    0xd51b: 'aabbaL',
    0xd51c: 'hhhbh98',
    0xd51d: 'ddabaL',
    0xd51e: 'cccbi88',
    0xd51f: 'ccaabM',
    0xd520: 'bbbai88',
    0xd521: 'bbbaaL',
    0xd522: 'aaabh88',
    0xd523: 'aaabbM',
    0xd524: 'dddah88',
    0xd525: 'ddbbbM',
    0xd526: 'gggbi88',
    0xd527: 'ccbaaL',
    0xd528: 'fffbi88',
    0xd529: 'bbabbM',
    0xd52a: 'eeeah88',
    0xd52b: 'aabaaL',
    0xd52c: 'hhhah98',
    0xd52d: 'ddaaaL',
    0xd52e: 'cccai88',
    0xd52f: 'ccabbM',
    0xd530: 'bbbbh88',
    0xd531: 'bbbbbM',
    0xd532: 'aaaai88',
    0xd533: 'aaaaaL',
    0xd534: 'dddbi88',
    0xd535: 'ddbaaL',
    0xd536: 'gggah88',
    0xd537: 'ccbbbM',
    0xd538: 'fffah88',
    0xd539: 'bbaaaL',
    0xd53a: 'dddddaaI9',
    0xd53b: 'aabbbM',
    0xd53c: 'aaaaabaI9',
    0xd53d: 'aaaaabaam',
    0xd53e: 'bbbbbabH9',
    0xd53f: 'bbbbbabbl',
    0xd540: 'bbbah98',
    0xd541: 'bbbabL',
    0xd542: 'aaabi98',
    0xd543: 'aaabaM',
    0xd544: 'dddai98',
    0xd545: 'ddbbaM',
    0xd546: 'fffffaaI9',
    0xd547: 'ccbabL',
    0xd548: 'fffbh98',
    0xd549: 'bbabaM',
    0xd54a: 'eeeai98',
    0xd54b: 'aababL',
    0xd54c: 'hhhai88',
    0xd54d: 'ddaabL',
    0xd54e: 'cccah98',
    0xd54f: 'ccabaM',
    0xd550: 'bbbbi98',
    0xd551: 'bbbbaM',
    0xd552: 'aaaah98',
    0xd553: 'aaaabL',
    0xd554: 'dddbh98',
    0xd555: 'ddbabL',
    0xd556: 'gggai98',
    0xd557: 'ccbbaM',
    0xd558: 'fffai98',
    0xd559: 'bbaabL',
    0xd55a: 'eeebh98',
    0xd55b: 'aabbaM',
    0xd55c: 'hhhbh88',
    0xd55d: 'ddabaM',
    0xd55e: 'cccbi98',
    0xd55f: 'ccaabL',
    0xd560: 'bbbai98',
    0xd561: 'bbbaaM',
    0xd562: 'aaabh98',
    0xd563: 'aaabbL',
    0xd564: 'dddah98',
    0xd565: 'ddbbbL',
    0xd566: 'gggbi98',
    0xd567: 'ccbaaM',
    0xd568: 'fffbi98',
    0xd569: 'bbabbL',
    0xd56a: 'eeeah98',
    0xd56b: 'aabaaM',
    0xd56c: 'hhhah88',
    0xd56d: 'ddaaaM',
    0xd56e: 'cccai98',
    0xd56f: 'ccabbL',
    0xd570: 'bbbbh98',
    0xd571: 'bbbbbL',
    0xd572: 'aaaai98',
    0xd573: 'aaaaaM',
    0xd574: 'dddbi98',
    0xd575: 'ddbaaM',
    0xd576: 'gggah98',
    0xd577: 'ccbbbL',
    0xd578: 'dddddda058',
    0xd579: 'bbaaaM',
    0xd57a: 'dddddbbH9',
    0xd57b: 'aabbbL',
    0xd57c: 'aaaaaabH9',
    0xd57d: 'aaaaaabbl',
    0xd57e: 'aaaaaab058',
    0xd57f: 'bbbbbbaam',
    0xd580: 'bbbah89',
    0xd581: 'bbbabO',
    0xd582: 'aaabi89',
    0xd583: 'aaabaN',
    0xd584: 'dddai89',
    0xd585: 'ddbbaN',
    0xd586: 'eeeeeeb158',
    0xd587: 'ccbabO',
    0xd588: 'fffbh89',
    0xd589: 'bbabaN',
    0xd58a: 'eeeai89',
    0xd58b: 'aababO',
    0xd58c: 'hhhai99',
    0xd58d: 'ddaabO',
    0xd58e: 'cccah89',
    0xd58f: 'ccabaN',
    0xd590: 'bbbbi89',
    0xd591: 'bbbbaN',
    0xd592: 'aaaah89',
    0xd593: 'aaaabO',
    0xd594: 'dddbh89',
    0xd595: 'ddbabO',
    0xd596: 'gggai89',
    0xd597: 'ccbbaN',
    0xd598: 'fffai89',
    0xd599: 'bbaabO',
    0xd59a: 'eeebh89',
    0xd59b: 'aabbaN',
    0xd59c: 'hhhbh99',
    0xd59d: 'ddabaN',
    0xd59e: 'cccbi89',
    0xd59f: 'ccaabO',
    0xd5a0: 'bbbai89',
    0xd5a1: 'bbbaaN',
    0xd5a2: 'aaabh89',
    0xd5a3: 'aaabbO',
    0xd5a4: 'dddah89',
    0xd5a5: 'ddbbbO',
    0xd5a6: 'gggbi89',
    0xd5a7: 'ccbaaN',
    0xd5a8: 'fffbi89',
    0xd5a9: 'bbabbO',
    0xd5aa: 'eeeah89',
    0xd5ab: 'aabaaN',
    0xd5ac: 'hhhah99',
    0xd5ad: 'ddaaaN',
    0xd5ae: 'cccai89',
    0xd5af: 'ccabbO',
    0xd5b0: 'bbbbh89',
    0xd5b1: 'bbbbbO',
    0xd5b2: 'aaaai89',
    0xd5b3: 'aaaaaN',
    0xd5b4: 'dddbi89',
    0xd5b5: 'ddbaaN',
    0xd5b6: 'gggah89',
    0xd5b7: 'ccbbbO',
    0xd5b8: 'fffah89',
    0xd5b9: 'bbaaaN',
    0xd5ba: 'dddddabH9',
    0xd5bb: 'aabbbO',
    0xd5bc: 'aaaaabbH9',
    0xd5bd: 'aaaaabbbl',
    0xd5be: 'bbbbbaaI9',
    0xd5bf: 'bbbbbaaam',
    0xd5c0: 'bbbah99',
    0xd5c1: 'bbbabN',
    0xd5c2: 'aaabi99',
    0xd5c3: 'aaabaO',
    0xd5c4: 'dddai99',
    0xd5c5: 'ddbbaO',
    0xd5c6: 'fffffabH9',
    0xd5c7: 'ccbabN',
    0xd5c8: 'fffbh99',
    0xd5c9: 'bbabaO',
    0xd5ca: 'eeeai99',
    0xd5cb: 'aababN',
    0xd5cc: 'hhhai89',
    0xd5cd: 'ddaabN',
    0xd5ce: 'cccah99',
    0xd5cf: 'ccabaO',
    0xd5d0: 'bbbbi99',
    0xd5d1: 'bbbbaO',
    0xd5d2: 'aaaah99',
    0xd5d3: 'aaaabN',
    0xd5d4: 'dddbh99',
    0xd5d5: 'ddbabN',
    0xd5d6: 'gggai99',
    0xd5d7: 'ccbbaO',
    0xd5d8: 'fffai99',
    0xd5d9: 'bbaabN',
    0xd5da: 'eeebh99',
    0xd5db: 'aabbaO',
    0xd5dc: 'hhhbh89',
    0xd5dd: 'ddabaO',
    0xd5de: 'cccbi99',
    0xd5df: 'ccaabN',
    0xd5e0: 'aaaaaaaaabbbA',
    0xd5e1: 'aaaaaaaaabbbE',
    0xd5e2: 'aaabh99',
    0xd5e3: 'aaabbN',
    0xd5e4: 'aaaaaaaaabbbQ',
    0xd5e5: 'aaaaaaaaabbbU',
    0xd5e6: 'aaaaaaaaabbbY',
    0xd5e7: 'aaaaaaaaabbhX',
    0xd5e8: 'aaaaaaaaabbba',
    0xd5e9: 'aaaaaaaaabbbe',
    0xd5ea: 'aaaaaaaaabbbi',
    0xd5eb: 'aabaaO',
    0xd5ec: 'aaaaaaaaabbbq',
    0xd5ed: 'aaaaaaaaabbbu',
    0xd5ee: 'aaaaaaaaabbby',
    0xd5ef: 'aaaaaaaaabbhx',
    0xd5f0: 'aaaaaaaaabbh',
    0xd5f1: 'aaaaaaaaabdc',
    0xd5f2: 'aaaai99',
    0xd5f3: 'aaaaaO',
    0xd5f4: 'aaaaaaaaabbH',
    0xd5f5: 'aaaaaaaaabdC',
    0xd5f6: 'aaaaaaaaabbX',
    0xd5f7: 'aaaaaaaaabbP',
    0xd5f8: 'aaaaaaaaabB',
    0xd5f9: 'aaaaaaaaabR',
    0xd5fa: 'aaaaaaaaabb',
    0xd5fb: 'aabbbN',
    0xd5fc: 'aaaaaaaI9',
    0xd5fd: 'aaaaaaaam',
    0xd5fe: 'aaaaaaa158',
    0xd5ff: 'aaaaaaaaA6',
    0xd600: 'bbbab18',
    0xd601: 'bbbabA',
    0xd602: 'aaaba08',
    0xd603: 'aaabcA',
    0xd604: 'cccccccabaC',
    0xd605: 'cccccccabaS',
    0xd606: 'cccccccabac',
    0xd607: 'ccbabA',
    0xd608: 'ccccccccabbH',
    0xd609: 'bbabcA',
    0xd60a: 'ccccccccabbX',
    0xd60b: 'aababA',
    0xd60c: 'ccccccccabbh',
    0xd60d: 'ccccccccabdc',
    0xd60e: 'cccab18',
    0xd60f: 'ccabcA',
    0xd610: 'bbbba08',
    0xd611: 'bbbbcA',
    0xd612: 'aaaab18',
    0xd613: 'aaaabA',
    0xd614: 'cccccccccaaca',
    0xd615: 'cccccccccaaad',
    0xd616: 'cccccccccaaah',
    0xd617: 'ccbbcA',
    0xd618: 'cccccccccaaaP',
    0xd619: 'bbaabA',
    0xd61a: 'cccccccccaaaX',
    0xd61b: 'aabbcA',
    0xd61c: 'cccccccccaacA',
    0xd61d: 'cccccccccaaaD',
    0xd61e: 'cccba08',
    0xd61f: 'ccaabA',
    0xd620: 'bbbaa08',
    0xd621: 'bbbacA',
    0xd622: 'aaabb18',
    0xd623: 'aaabbA',
    0xd624: 'ccccccccccbbaf',
    0xd625: 'ccccccccccbbad',
    0xd626: 'ccccccccccbbab',
    0xd627: 'ccbacA',
    0xd628: 'ccccccccccbbiz',
    0xd629: 'bbabbA',
    0xd62a: 'ccccccccccbbaz',
    0xd62b: 'aabacA',
    0xd62c: 'ccccccccccbbav',
    0xd62d: 'ccccccccccbbat',
    0xd62e: 'cccaa08',
    0xd62f: 'ccabbA',
    0xd630: 'bbbbb18',
    0xd631: 'bbbbbA',
    0xd632: 'aaaaa08',
    0xd633: 'aaaacA',
    0xd634: 'ccccccccccbbaF',
    0xd635: 'ccccccccccbbaD',
    0xd636: 'ccccccccccbbaB',
    0xd637: 'ccbbbA',
    0xd638: 'ccccccccccbbiZ',
    0xd639: 'bbaacA',
    0xd63a: 'ccccccccccbbaZ',
    0xd63b: 'aabbbA',
    0xd63c: 'ccccccccccbbaV',
    0xd63d: 'aaaaababm',
    0xd63e: 'cccbb18',
    0xd63f: 'bbbbbab9',
    0xd640: 'bbbab08',
    0xd641: 'bbbadC',
    0xd642: 'aaaba18',
    0xd643: 'aaabaA',
    0xd644: 'dddaa18',
    0xd645: 'ddbbaA',
    0xd646: 'gggbb08',
    0xd647: 'ccbadC',
    0xd648: 'fffbb08',
    0xd649: 'bbabaA',
    0xd64a: 'eeeaa18',
    0xd64b: 'aabadC',
    0xd64c: 'hhhaa08',
    0xd64d: 'ddaadC',
    0xd64e: 'cccab08',
    0xd64f: 'ccabaA',
    0xd650: 'bbbba18',
    0xd651: 'bbbbaA',
    0xd652: 'aaaab08',
    0xd653: 'aaaadC',
    0xd654: 'dddbb08',
    0xd655: 'ddbadC',
    0xd656: 'gggaa18',
    0xd657: 'ccbbaA',
    0xd658: 'fffaa18',
    0xd659: 'bbaadC',
    0xd65a: 'eeebb08',
    0xd65b: 'aabbaA',
    0xd65c: 'hhhbb18',
    0xd65d: 'ddabaA',
    0xd65e: 'cccba18',
    0xd65f: 'ccaadC',
    0xd660: 'bbbaa18',
    0xd661: 'bbbaaA',
    0xd662: 'aaabb08',
    0xd663: 'aaabdC',
    0xd664: 'dddab08',
    0xd665: 'ddbbdC',
    0xd666: 'gggba18',
    0xd667: 'ccbaaA',
    0xd668: 'fffba18',
    0xd669: 'bbabdC',
    0xd66a: 'eeeab08',
    0xd66b: 'aabaaA',
    0xd66c: 'hhhab18',
    0xd66d: 'ddaaaA',
    0xd66e: 'cccaa18',
    0xd66f: 'ccabdC',
    0xd670: 'bbbbb08',
    0xd671: 'bbbbdC',
    0xd672: 'aaaaa18',
    0xd673: 'aaaaaA',
    0xd674: 'dddba18',
    0xd675: 'ddbaaA',
    0xd676: 'gggab08',
    0xd677: 'ccbbdC',
    0xd678: 'dddddda148',
    0xd679: 'bbaaaA',
    0xd67a: 'ddddddaI8',
    0xd67b: 'aabbdC',
    0xd67c: 'aaaaaabI8',
    0xd67d: 'aaaaaab9',
    0xd67e: 'aaaaaab148',
    0xd67f: 'bbbbbbabm',
    0xd680: 'bbbab19',
    0xd681: 'bbbabC',
    0xd682: 'aaaba09',
    0xd683: 'aaabaB',
    0xd684: 'dddaa09',
    0xd685: 'ddbbaB',
    0xd686: 'eeeeeeb048',
    0xd687: 'ccbabC',
    0xd688: 'fffbb19',
    0xd689: 'bbabaB',
    0xd68a: 'eeeaa09',
    0xd68b: 'aababC',
    0xd68c: 'hhhaa19',
    0xd68d: 'ddaabC',
    0xd68e: 'cccab19',
    0xd68f: 'ccabaB',
    0xd690: 'bbbba09',
    0xd691: 'bbbbaB',
    0xd692: 'aaaab19',
    0xd693: 'aaaabC',
    0xd694: 'dddbb19',
    0xd695: 'ddbabC',
    0xd696: 'gggaa09',
    0xd697: 'ccbbaB',
    0xd698: 'fffaa09',
    0xd699: 'bbaabC',
    0xd69a: 'eeebb19',
    0xd69b: 'aabbaB',
    0xd69c: 'hhhbb09',
    0xd69d: 'ddabaB',
    0xd69e: 'cccba09',
    0xd69f: 'ccaabC',
    0xd6a0: 'bbbaa09',
    0xd6a1: 'bbbaaB',
    0xd6a2: 'aaabb19',
    0xd6a3: 'aaabbC',
    0xd6a4: 'dddab19',
    0xd6a5: 'ddbbbC',
    0xd6a6: 'gggba09',
    0xd6a7: 'ccbaaB',
    0xd6a8: 'fffba09',
    0xd6a9: 'bbabbC',
    0xd6aa: 'eeeab19',
    0xd6ab: 'aabaaB',
    0xd6ac: 'hhhab09',
    0xd6ad: 'ddaaaB',
    0xd6ae: 'cccaa09',
    0xd6af: 'ccabbC',
    0xd6b0: 'bbbbb19',
    0xd6b1: 'bbbbbC',
    0xd6b2: 'aaaaa09',
    0xd6b3: 'aaaaaB',
    0xd6b4: 'dddba09',
    0xd6b5: 'ddbaaB',
    0xd6b6: 'gggab19',
    0xd6b7: 'ccbbbC',
    0xd6b8: 'fffab19',
    0xd6b9: 'bbaaaB',
    0xd6ba: 'eeeba09',
    0xd6bb: 'aabbbC',
    0xd6bc: 'hhhba19',
    0xd6bd: 'aaaaabbal',
    0xd6be: 'cccbb19',
    0xd6bf: 'bbbbbaabm',
    0xd6c0: 'bbbab09',
    0xd6c1: 'bbbabB',
    0xd6c2: 'aaaba19',
    0xd6c3: 'aaabaC',
    0xd6c4: 'dddaa19',
    0xd6c5: 'ddbbaC',
    0xd6c6: 'gggbb09',
    0xd6c7: 'ccbabB',
    0xd6c8: 'fffbb09',
    0xd6c9: 'bbabaC',
    0xd6ca: 'eeeaa19',
    0xd6cb: 'aababB',
    0xd6cc: 'hhhaa09',
    0xd6cd: 'ddaabB',
    0xd6ce: 'cccab09',
    0xd6cf: 'ccabaC',
    0xd6d0: 'bbbba19',
    0xd6d1: 'bbbbaC',
    0xd6d2: 'aaaab09',
    0xd6d3: 'aaaabB',
    0xd6d4: 'dddbb09',
    0xd6d5: 'ddbabB',
    0xd6d6: 'gggaa19',
    0xd6d7: 'ccbbaC',
    0xd6d8: 'fffaa19',
    0xd6d9: 'bbaabB',
    0xd6da: 'eeebb09',
    0xd6db: 'aabbaC',
    0xd6dc: 'hhhbb19',
    0xd6dd: 'ddabaC',
    0xd6de: 'cccba19',
    0xd6df: 'ccaabB',
    0xd6e0: 'bbbaa19',
    0xd6e1: 'bbbaaC',
    0xd6e2: 'aaabb09',
    0xd6e3: 'aaabbB',
    0xd6e4: 'dddab09',
    0xd6e5: 'ddbbbB',
    0xd6e6: 'gggba19',
    0xd6e7: 'ccbaaC',
    0xd6e8: 'fffba19',
    0xd6e9: 'bbabbB',
    0xd6ea: 'eeeab09',
    0xd6eb: 'aabaaC',
    0xd6ec: 'hhhab19',
    0xd6ed: 'ddaaaC',
    0xd6ee: 'cccaa19',
    0xd6ef: 'ccabbB',
    0xd6f0: 'bbbbb09',
    0xd6f1: 'bbbbbB',
    0xd6f2: 'aaaaa19',
    0xd6f3: 'aaaaaC',
    0xd6f4: 'dddba19',
    0xd6f5: 'ddbaaC',
    0xd6f6: 'gggab09',
    0xd6f7: 'ccbbbB',
    0xd6f8: 'aaaaaaababB',
    0xd6f9: 'aaaaaaababR',
    0xd6fa: 'aaaaaaababb',
    0xd6fb: 'aabbbB',
    0xd6fc: 'aaaaaaaH8',
    0xd6fd: 'aaaaaaabm',
    0xd6fe: 'aaaaaaa048',
    0xd6ff: 'bbbbbbbal',
    0xd700: 'bbbab58',
    0xd701: 'bbbabE',
    0xd702: 'aaaba48',
    0xd703: 'aaabaD',
    0xd704: 'cccccccbabB',
    0xd705: 'cccccccbabR',
    0xd706: 'cccccccbabb',
    0xd707: 'ccbabE',
    0xd708: 'fffbb58',
    0xd709: 'bbabaD',
    0xd70a: 'eeeaa48',
    0xd70b: 'aababE',
    0xd70c: 'hhhaa58',
    0xd70d: 'ddaabE',
    0xd70e: 'cccab58',
    0xd70f: 'ccabaD',
    0xd710: 'bbbba48',
    0xd711: 'bbbbaD',
    0xd712: 'aaaab58',
    0xd713: 'aaaabE',
    0xd714: 'dddbb58',
    0xd715: 'ddbabE',
    0xd716: 'gggaa48',
    0xd717: 'ccbbaD',
    0xd718: 'fffaa48',
    0xd719: 'bbaabE',
    0xd71a: 'eeebb58',
    0xd71b: 'aabbaD',
    0xd71c: 'hhhbb48',
    0xd71d: 'ddabaD',
    0xd71e: 'cccba48',
    0xd71f: 'ccaabE',
    0xd720: 'bbbaa48',
    0xd721: 'bbbaaD',
    0xd722: 'aaabb58',
    0xd723: 'aaabbE',
    0xd724: 'dddab58',
    0xd725: 'ddbbbE',
    0xd726: 'gggba48',
    0xd727: 'ccbaaD',
    0xd728: 'fffba48',
    0xd729: 'bbabbE',
    0xd72a: 'eeeab58',
    0xd72b: 'aabaaD',
    0xd72c: 'hhhab48',
    0xd72d: 'ddaaaD',
    0xd72e: 'cccaa48',
    0xd72f: 'ccabbE',
    0xd730: 'bbbbb58',
    0xd731: 'bbbbbE',
    0xd732: 'aaaaa48',
    0xd733: 'aaaaaD',
    0xd734: 'dddba48',
    0xd735: 'ddbaaD',
    0xd736: 'gggab58',
    0xd737: 'ccbbbE',
    0xd738: 'fffab58',
    0xd739: 'bbaaaD',
    0xd73a: 'eeeba48',
    0xd73b: 'aabbbE',
    0xd73c: 'hhhba58',
    0xd73d: 'aaaaabaal',
    0xd73e: 'cccbb58',
    0xd73f: 'bbbbbabbm',
    0xd740: 'bbbab48',
    0xd741: 'bbbabD',
    0xd742: 'aaaba58',
    0xd743: 'aaabaE',
    0xd744: 'dddaa58',
    0xd745: 'ddbbaE',
    0xd746: 'gggbb48',
    0xd747: 'ccbabD',
    0xd748: 'fffbb48',
    0xd749: 'bbabaE',
    0xd74a: 'eeeaa58',
    0xd74b: 'aababD',
    0xd74c: 'hhhaa48',
    0xd74d: 'ddaabD',
    0xd74e: 'cccab48',
    0xd74f: 'ccabaE',
    0xd750: 'bbbba58',
    0xd751: 'bbbbaE',
    0xd752: 'aaaab48',
    0xd753: 'aaaabD',
    0xd754: 'dddbb48',
    0xd755: 'ddbabD',
    0xd756: 'gggaa58',
    0xd757: 'ccbbaE',
    0xd758: 'fffaa58',
    0xd759: 'bbaabD',
    0xd75a: 'eeebb48',
    0xd75b: 'aabbaE',
    0xd75c: 'hhhbb58',
    0xd75d: 'ddabaE',
    0xd75e: 'cccba58',
    0xd75f: 'ccaabD',
    0xd760: 'bbbaa58',
    0xd761: 'bbbaaE',
    0xd762: 'aaabb48',
    0xd763: 'aaabbD',
    0xd764: 'dddab48',
    0xd765: 'ddbbbD',
    0xd766: 'gggba58',
    0xd767: 'ccbaaE',
    0xd768: 'fffba58',
    0xd769: 'bbabbD',
    0xd76a: 'eeeab48',
    0xd76b: 'aabaaE',
    0xd76c: 'hhhab58',
    0xd76d: 'ddaaaE',
    0xd76e: 'cccaa58',
    0xd76f: 'ccabbD',
    0xd770: 'bbbbb48',
    0xd771: 'bbbbbD',
    0xd772: 'aaaaa58',
    0xd773: 'aaaaaE',
    0xd774: 'dddba58',
    0xd775: 'ddbaaE',
    0xd776: 'gggab48',
    0xd777: 'ccbbbD',
    0xd778: 'dddddda048',
    0xd779: 'bbaaaE',
    0xd77a: 'ddddddaH8',
    0xd77b: 'aabbbD',
    0xd77c: 'aaaaaabH8',
    0xd77d: 'aaaaaabbm',
    0xd77e: 'aaaaaab048',
    0xd77f: 'bbbbbbaal',
    0xd780: 'bbbab59',
    0xd781: 'bbbabG',
    0xd782: 'aaaba49',
    0xd783: 'aaabaF',
    0xd784: 'dddaa49',
    0xd785: 'ddbbaF',
    0xd786: 'eeeeeeb148',
    0xd787: 'ccbabG',
    0xd788: 'fffbb59',
    0xd789: 'bbabaF',
    0xd78a: 'eeeaa49',
    0xd78b: 'aababG',
    0xd78c: 'hhhaa59',
    0xd78d: 'ddaabG',
    0xd78e: 'cccab59',
    0xd78f: 'ccabaF',
    0xd790: 'bbbba49',
    0xd791: 'bbbbaF',
    0xd792: 'aaaab59',
    0xd793: 'aaaabG',
    0xd794: 'dddbb59',
    0xd795: 'ddbabG',
    0xd796: 'gggaa49',
    0xd797: 'ccbbaF',
    0xd798: 'fffaa49',
    0xd799: 'bbaabG',
    0xd79a: 'eeebb59',
    0xd79b: 'aabbaF',
    0xd79c: 'hhhbb49',
    0xd79d: 'ddabaF',
    0xd79e: 'cccba49',
    0xd79f: 'ccaabG',
    0xd7a0: 'bbbaa49',
    0xd7a1: 'bbbaaF',
    0xd7a2: 'aaabb59',
    0xd7a3: 'aaabbG',
    0xd7a4: 'dddab59',
    0xd7a5: 'ddbbbG',
    0xd7a6: 'gggba49',
    0xd7a7: 'ccbaaF',
    0xd7a8: 'fffba49',
    0xd7a9: 'bbabbG',
    0xd7aa: 'eeeab59',
    0xd7ab: 'aabaaF',
    0xd7ac: 'hhhab49',
    0xd7ad: 'ddaaaF',
    0xd7ae: 'cccaa49',
    0xd7af: 'ccabbG',
    0xd7b0: 'bbbbb59',
    0xd7b1: 'bbbbbG',
    0xd7b2: 'aaaaa49',
    0xd7b3: 'aaaaaF',
    0xd7b4: 'dddba49',
    0xd7b5: 'ddbaaF',
    0xd7b6: 'gggab59',
    0xd7b7: 'ccbbbG',
    0xd7b8: 'fffab59',
    0xd7b9: 'bbaaaF',
    0xd7ba: 'eeeba49',
    0xd7bb: 'aabbbG',
    0xd7bc: 'hhhba59',
    0xd7bd: 'aaaaabbbm',
    0xd7be: 'cccbb59',
    0xd7bf: 'bbbbbaaal',
    0xd7c0: 'bbbab49',
    0xd7c1: 'bbbabF',
    0xd7c2: 'aaaba59',
    0xd7c3: 'aaabaG',
    0xd7c4: 'dddaa59',
    0xd7c5: 'ddbbaG',
    0xd7c6: 'gggbb49',
    0xd7c7: 'ccbabF',
    0xd7c8: 'fffbb49',
    0xd7c9: 'bbabaG',
    0xd7ca: 'eeeaa59',
    0xd7cb: 'aababF',
    0xd7cc: 'hhhaa49',
    0xd7cd: 'ddaabF',
    0xd7ce: 'cccab49',
    0xd7cf: 'ccabaG',
    0xd7d0: 'bbbba59',
    0xd7d1: 'bbbbaG',
    0xd7d2: 'aaaab49',
    0xd7d3: 'aaaabF',
    0xd7d4: 'dddbb49',
    0xd7d5: 'ddbabF',
    0xd7d6: 'gggaa59',
    0xd7d7: 'ccbbaG',
    0xd7d8: 'fffaa59',
    0xd7d9: 'bbaabF',
    0xd7da: 'eeebb49',
    0xd7db: 'aabbaG',
    0xd7dc: 'hhhbb59',
    0xd7dd: 'ddabaG',
    0xd7de: 'cccba59',
    0xd7df: 'ccaabF',
    0xd7e0: 'bbbaa59',
    0xd7e1: 'bbbaaG',
    0xd7e2: 'aaabb49',
    0xd7e3: 'aaabbF',
    0xd7e4: 'dddab49',
    0xd7e5: 'ddbbbF',
    0xd7e6: 'gggba59',
    0xd7e7: 'ccbaaG',
    0xd7e8: 'fffba59',
    0xd7e9: 'bbabbF',
    0xd7ea: 'eeeab49',
    0xd7eb: 'aabaaG',
    0xd7ec: 'hhhab59',
    0xd7ed: 'ddaaaG',
    0xd7ee: 'cccaa59',
    0xd7ef: 'ccabbF',
    0xd7f0: 'aaaaaaaabaai',
    0xd7f1: 'aaaaaaaabaaa',
    0xd7f2: 'aaaaa59',
    0xd7f3: 'aaaaaG',
    0xd7f4: 'aaaaaaaabaaI',
    0xd7f5: 'aaaaaaaabaaA',
    0xd7f6: 'aaaaaaaabaaY',
    0xd7f7: 'aaaaaaaabaaQ',
    0xd7f8: 'aaaaaaaabaC',
    0xd7f9: 'aaaaaaaabaS',
    0xd7fa: 'aaaaaaaabac',
    0xd7fb: 'aabbbF',
    0xd7fc: 'aaaaaaaI8',
    0xd7fd: 'aaaaaaaal',
    0xd7fe: 'aaaaaaa148',
    0xd7ff: 'aaaaaaaaB7',
    0xd800: 'bbba000',
    0xd801: 'bbbaby',
    0xd802: 'aaab100',
    0xd803: 'aaabax',
    0xd804: 'cccccccaaaB',
    0xd805: 'cccccccaaaR',
    0xd806: 'cccccccaaab',
    0xd807: 'ccbaby',
    0xd808: 'ccccccccbbaI',
    0xd809: 'bbabax',
    0xd80a: 'ccccccccbbaY',
    0xd80b: 'aababy',
    0xd80c: 'ccccccccbbai',
    0xd80d: 'ccccccccbbaa',
    0xd80e: 'ccca000',
    0xd80f: 'ccabax',
    0xd810: 'bbbb100',
    0xd811: 'bbbbax',
    0xd812: 'aaaa000',
    0xd813: 'aaaaby',
    0xd814: 'dddb000',
    0xd815: 'ddbaby',
    0xd816: 'ggga100',
    0xd817: 'ccbbax',
    0xd818: 'fffa100',
    0xd819: 'bbaaby',
    0xd81a: 'eeeb000',
    0xd81b: 'aabbax',
    0xd81c: 'hhhb010',
    0xd81d: 'ddabax',
    0xd81e: 'cccb100',
    0xd81f: 'ccaaby',
    0xd820: 'bbba100',
    0xd821: 'bbbaax',
    0xd822: 'aaab000',
    0xd823: 'aaabby',
    0xd824: 'ddda000',
    0xd825: 'ddbbby',
    0xd826: 'gggb100',
    0xd827: 'ccbaax',
    0xd828: 'fffb100',
    0xd829: 'bbabby',
    0xd82a: 'eeea000',
    0xd82b: 'aabaax',
    0xd82c: 'hhha010',
    0xd82d: 'ddaaax',
    0xd82e: 'ccca100',
    0xd82f: 'ccabby',
    0xd830: 'bbbb000',
    0xd831: 'bbbbby',
    0xd832: 'aaaa100',
    0xd833: 'aaaaax',
    0xd834: 'dddb100',
    0xd835: 'ddbaax',
    0xd836: 'ggga000',
    0xd837: 'ccbbby',
    0xd838: 'fffa000',
    0xd839: 'bbaaax',
    0xd83a: 'dddddaaD9',
    0xd83b: 'aabbby',
    0xd83c: 'aaaaabaD9',
    0xd83d: 'aaaaababj',
    0xd83e: 'bbbbbabE9',
    0xd83f: 'bbbbbabak',
    0xd840: 'bbba010',
    0xd841: 'bbbabx',
    0xd842: 'aaab110',
    0xd843: 'aaabay',
    0xd844: 'ddda110',
    0xd845: 'ddbbay',
    0xd846: 'fffffaaD9',
    0xd847: 'ccbabx',
    0xd848: 'fffb010',
    0xd849: 'bbabay',
    0xd84a: 'eeea110',
    0xd84b: 'aababx',
    0xd84c: 'hhha100',
    0xd84d: 'ddaabx',
    0xd84e: 'ccca010',
    0xd84f: 'ccabay',
    0xd850: 'bbbb110',
    0xd851: 'bbbbay',
    0xd852: 'aaaa010',
    0xd853: 'aaaabx',
    0xd854: 'dddb010',
    0xd855: 'ddbabx',
    0xd856: 'ggga110',
    0xd857: 'ccbbay',
    0xd858: 'fffa110',
    0xd859: 'bbaabx',
    0xd85a: 'eeeb010',
    0xd85b: 'aabbay',
    0xd85c: 'hhhb000',
    0xd85d: 'ddabay',
    0xd85e: 'cccb110',
    0xd85f: 'ccaabx',
    0xd860: 'bbba110',
    0xd861: 'bbbaay',
    0xd862: 'aaab010',
    0xd863: 'aaabbx',
    0xd864: 'ddda010',
    0xd865: 'ddbbbx',
    0xd866: 'gggb110',
    0xd867: 'ccbaay',
    0xd868: 'fffb110',
    0xd869: 'bbabbx',
    0xd86a: 'eeea010',
    0xd86b: 'aabaay',
    0xd86c: 'hhha000',
    0xd86d: 'ddaaay',
    0xd86e: 'ccca110',
    0xd86f: 'ccabbx',
    0xd870: 'bbbb010',
    0xd871: 'bbbbbx',
    0xd872: 'aaaa110',
    0xd873: 'aaaaay',
    0xd874: 'dddb110',
    0xd875: 'ddbaay',
    0xd876: 'ggga010',
    0xd877: 'ccbbbx',
    0xd878: 'dddddda119',
    0xd879: 'bbaaay',
    0xd87a: 'dddddbbE9',
    0xd87b: 'aabbbx',
    0xd87c: 'aaaaaabE9',
    0xd87d: 'aaaaaabak',
    0xd87e: 'aaaaaab119',
    0xd87f: 'bbbbbbabj',
    0xd880: 'bbba001',
    0xd881: 'bbbadx',
    0xd882: 'aaab101',
    0xd883: 'aaabaz',
    0xd884: 'ddda101',
    0xd885: 'ddbbaz',
    0xd886: 'eeeeeeb019',
    0xd887: 'ccbadx',
    0xd888: 'fffb001',
    0xd889: 'bbabaz',
    0xd88a: 'eeea101',
    0xd88b: 'aabadx',
    0xd88c: 'hhha111',
    0xd88d: 'ddaadx',
    0xd88e: 'ccca001',
    0xd88f: 'ccabaz',
    0xd890: 'bbbb101',
    0xd891: 'bbbbaz',
    0xd892: 'aaaa001',
    0xd893: 'aaaadx',
    0xd894: 'dddb001',
    0xd895: 'ddbadx',
    0xd896: 'ggga101',
    0xd897: 'ccbbaz',
    0xd898: 'fffa101',
    0xd899: 'bbaadx',
    0xd89a: 'eeeb001',
    0xd89b: 'aabbaz',
    0xd89c: 'hhhb011',
    0xd89d: 'ddabaz',
    0xd89e: 'cccb101',
    0xd89f: 'ccaadx',
    0xd8a0: 'bbba101',
    0xd8a1: 'bbbaaz',
    0xd8a2: 'aaab001',
    0xd8a3: 'aaabdx',
    0xd8a4: 'ddda001',
    0xd8a5: 'ddbbdx',
    0xd8a6: 'gggb101',
    0xd8a7: 'ccbaaz',
    0xd8a8: 'fffb101',
    0xd8a9: 'bbabdx',
    0xd8aa: 'eeea001',
    0xd8ab: 'aabaaz',
    0xd8ac: 'hhha011',
    0xd8ad: 'ddaaaz',
    0xd8ae: 'ccca101',
    0xd8af: 'ccabdx',
    0xd8b0: 'bbbb001',
    0xd8b1: 'bbbbdx',
    0xd8b2: 'aaaa101',
    0xd8b3: 'aaaaaz',
    0xd8b4: 'dddb101',
    0xd8b5: 'ddbaaz',
    0xd8b6: 'ggga001',
    0xd8b7: 'ccbbdx',
    0xd8b8: 'fffa001',
    0xd8b9: 'bbaaaz',
    0xd8ba: 'dddddabE9',
    0xd8bb: 'aabbdx',
    0xd8bc: 'aaaaabbE9',
    0xd8bd: 'aaaaabb7',
    0xd8be: 'bbbbbaaD9',
    0xd8bf: 'bbbbbaabj',
    0xd8c0: 'bbba011',
    0xd8c1: 'bbbabz',
    0xd8c2: 'aaab111',
    0xd8c3: 'aaabcz',
    0xd8c4: 'ddda111',
    0xd8c5: 'ddbbcz',
    0xd8c6: 'fffffabE9',
    0xd8c7: 'ccbabz',
    0xd8c8: 'fffb011',
    0xd8c9: 'bbabcz',
    0xd8ca: 'eeea111',
    0xd8cb: 'aababz',
    0xd8cc: 'hhha101',
    0xd8cd: 'ddaabz',
    0xd8ce: 'ccca011',
    0xd8cf: 'ccabcz',
    0xd8d0: 'bbbb111',
    0xd8d1: 'bbbbcz',
    0xd8d2: 'aaaa011',
    0xd8d3: 'aaaabz',
    0xd8d4: 'dddb011',
    0xd8d5: 'ddbabz',
    0xd8d6: 'ggga111',
    0xd8d7: 'ccbbcz',
    0xd8d8: 'fffa111',
    0xd8d9: 'bbaabz',
    0xd8da: 'eeeb011',
    0xd8db: 'aabbcz',
    0xd8dc: 'hhhb001',
    0xd8dd: 'ddabcz',
    0xd8de: 'cccb111',
    0xd8df: 'ccaabz',
    0xd8e0: 'bbba111',
    0xd8e1: 'bbbacz',
    0xd8e2: 'aaab011',
    0xd8e3: 'aaabbz',
    0xd8e4: 'ddda011',
    0xd8e5: 'ddbbbz',
    0xd8e6: 'gggb111',
    0xd8e7: 'ccbacz',
    0xd8e8: 'fffb111',
    0xd8e9: 'bbabbz',
    0xd8ea: 'eeea011',
    0xd8eb: 'aabacz',
    0xd8ec: 'hhha001',
    0xd8ed: 'ddaacz',
    0xd8ee: 'ccca111',
    0xd8ef: 'ccabbz',
    0xd8f0: 'bbbb011',
    0xd8f1: 'bbbbbz',
    0xd8f2: 'aaaa111',
    0xd8f3: 'aaaacz',
    0xd8f4: 'dddb111',
    0xd8f5: 'ddbacz',
    0xd8f6: 'ggga011',
    0xd8f7: 'ccbbbz',
    0xd8f8: 'aaaaaaabbbC',
    0xd8f9: 'aaaaaaabbbS',
    0xd8fa: 'aaaaaaabbbc',
    0xd8fb: 'aabbbz',
    0xd8fc: 'aaaaaaaD9',
    0xd8fd: 'aaaaaaabj',
    0xd8fe: 'aaaaaaa019',
    0xd8ff: 'bbbbbbb7',
    0xd900: 'bbba002',
    0xd901: 'bbbahx',
    0xd902: 'aaab102',
    0xd903: 'aaabix',
    0xd904: 'cccccccbbbC',
    0xd905: 'cccccccbbbS',
    0xd906: 'cccccccbbbc',
    0xd907: 'ccbahx',
    0xd908: 'fffb002',
    0xd909: 'bbabix',
    0xd90a: 'eeea102',
    0xd90b: 'aabahx',
    0xd90c: 'hhha112',
    0xd90d: 'ddaahx',
    0xd90e: 'ccca002',
    0xd90f: 'ccabix',
    0xd910: 'bbbb102',
    0xd911: 'bbbbix',
    0xd912: 'aaaa002',
    0xd913: 'aaaahx',
    0xd914: 'dddb002',
    0xd915: 'ddbahx',
    0xd916: 'ggga102',
    0xd917: 'ccbbix',
    0xd918: 'fffa102',
    0xd919: 'bbaahx',
    0xd91a: 'eeeb002',
    0xd91b: 'aabbix',
    0xd91c: 'hhhb012',
    0xd91d: 'ddabix',
    0xd91e: 'cccb102',
    0xd91f: 'ccaahx',
    0xd920: 'bbba102',
    0xd921: 'bbbaix',
    0xd922: 'aaab002',
    0xd923: 'aaabhx',
    0xd924: 'ddda002',
    0xd925: 'ddbbhx',
    0xd926: 'gggb102',
    0xd927: 'ccbaix',
    0xd928: 'fffb102',
    0xd929: 'bbabhx',
    0xd92a: 'eeea002',
    0xd92b: 'aabaix',
    0xd92c: 'hhha012',
    0xd92d: 'ddaaix',
    0xd92e: 'ccca102',
    0xd92f: 'ccabhx',
    0xd930: 'bbbb002',
    0xd931: 'bbbbhx',
    0xd932: 'aaaa102',
    0xd933: 'aaaaix',
    0xd934: 'dddb102',
    0xd935: 'ddbaix',
    0xd936: 'ggga002',
    0xd937: 'ccbbhx',
    0xd938: 'fffa002',
    0xd939: 'bbaaix',
    0xd93a: 'dddddaaE9',
    0xd93b: 'aabbhx',
    0xd93c: 'aaaaabaE9',
    0xd93d: 'aaaaaba7',
    0xd93e: 'bbbbbabD9',
    0xd93f: 'bbbbbab6',
    0xd940: 'bbba012',
    0xd941: 'bbbahy',
    0xd942: 'aaab112',
    0xd943: 'aaabiy',
    0xd944: 'ddda112',
    0xd945: 'ddbbiy',
    0xd946: 'fffffaaE9',
    0xd947: 'ccbahy',
    0xd948: 'fffb012',
    0xd949: 'bbabiy',
    0xd94a: 'eeea112',
    0xd94b: 'aabahy',
    0xd94c: 'hhha102',
    0xd94d: 'ddaahy',
    0xd94e: 'ccca012',
    0xd94f: 'ccabiy',
    0xd950: 'bbbb112',
    0xd951: 'bbbbiy',
    0xd952: 'aaaa012',
    0xd953: 'aaaahy',
    0xd954: 'dddb012',
    0xd955: 'ddbahy',
    0xd956: 'ggga112',
    0xd957: 'ccbbiy',
    0xd958: 'fffa112',
    0xd959: 'bbaahy',
    0xd95a: 'eeeb012',
    0xd95b: 'aabbiy',
    0xd95c: 'hhhb002',
    0xd95d: 'ddabiy',
    0xd95e: 'cccb112',
    0xd95f: 'ccaahy',
    0xd960: 'bbba112',
    0xd961: 'bbbaiy',
    0xd962: 'aaab012',
    0xd963: 'aaabhy',
    0xd964: 'ddda012',
    0xd965: 'ddbbhy',
    0xd966: 'gggb112',
    0xd967: 'ccbaiy',
    0xd968: 'fffb112',
    0xd969: 'bbabhy',
    0xd96a: 'eeea012',
    0xd96b: 'aabaiy',
    0xd96c: 'hhha002',
    0xd96d: 'ddaaiy',
    0xd96e: 'ccca112',
    0xd96f: 'ccabhy',
    0xd970: 'bbbb012',
    0xd971: 'bbbbhy',
    0xd972: 'aaaa112',
    0xd973: 'aaaaiy',
    0xd974: 'dddb112',
    0xd975: 'ddbaiy',
    0xd976: 'ggga012',
    0xd977: 'ccbbhy',
    0xd978: 'dddddda019',
    0xd979: 'bbaaiy',
    0xd97a: 'dddddbbD9',
    0xd97b: 'aabbhy',
    0xd97c: 'aaaaaabD9',
    0xd97d: 'aaaaaab6',
    0xd97e: 'aaaaaab019',
    0xd97f: 'bbbbbba7',
    0xd980: 'bbba003',
    0xd981: 'bbbahz',
    0xd982: 'aaab103',
    0xd983: 'aaabiz',
    0xd984: 'ddda103',
    0xd985: 'ddbbiz',
    0xd986: 'eeeeeeb119',
    0xd987: 'ccbahz',
    0xd988: 'fffb003',
    0xd989: 'bbabiz',
    0xd98a: 'eeea103',
    0xd98b: 'aabahz',
    0xd98c: 'hhha113',
    0xd98d: 'ddaahz',
    0xd98e: 'ccca003',
    0xd98f: 'ccabiz',
    0xd990: 'bbbb103',
    0xd991: 'bbbbiz',
    0xd992: 'aaaa003',
    0xd993: 'aaaahz',
    0xd994: 'dddb003',
    0xd995: 'ddbahz',
    0xd996: 'ggga103',
    0xd997: 'ccbbiz',
    0xd998: 'fffa103',
    0xd999: 'bbaahz',
    0xd99a: 'eeeb003',
    0xd99b: 'aabbiz',
    0xd99c: 'hhhb013',
    0xd99d: 'ddabiz',
    0xd99e: 'cccb103',
    0xd99f: 'ccaahz',
    0xd9a0: 'bbba103',
    0xd9a1: 'bbbaiz',
    0xd9a2: 'aaab003',
    0xd9a3: 'aaabhz',
    0xd9a4: 'ddda003',
    0xd9a5: 'ddbbhz',
    0xd9a6: 'gggb103',
    0xd9a7: 'ccbaiz',
    0xd9a8: 'fffb103',
    0xd9a9: 'bbabhz',
    0xd9aa: 'eeea003',
    0xd9ab: 'aabaiz',
    0xd9ac: 'hhha013',
    0xd9ad: 'ddaaiz',
    0xd9ae: 'ccca103',
    0xd9af: 'ccabhz',
    0xd9b0: 'bbbb003',
    0xd9b1: 'bbbbhz',
    0xd9b2: 'aaaa103',
    0xd9b3: 'aaaaiz',
    0xd9b4: 'dddb103',
    0xd9b5: 'ddbaiz',
    0xd9b6: 'ggga003',
    0xd9b7: 'ccbbhz',
    0xd9b8: 'fffa003',
    0xd9b9: 'bbaaiz',
    0xd9ba: 'dddddabD9',
    0xd9bb: 'aabbhz',
    0xd9bc: 'aaaaabbD9',
    0xd9bd: 'aaaaabb6',
    0xd9be: 'bbbbbaaE9',
    0xd9bf: 'bbbbbaa7',
    0xd9c0: 'aaaaaaaaaabbaV',
    0xd9c1: 'aaaaaaaaaabbaT',
    0xd9c2: 'aaab113',
    0xd9c3: 'aaabkz',
    0xd9c4: 'aaaaaaaaaabbiZ',
    0xd9c5: 'aaaaaaaaaabbiX',
    0xd9c6: 'aaaaaaaaaabbaZ',
    0xd9c7: 'aaaaaaaaaabbaX',
    0xd9c8: 'aaaaaaaaaabbaF',
    0xd9c9: 'aaaaaaaaaabbaD',
    0xd9ca: 'aaaaaaaaaabbaB',
    0xd9cb: 'aabajz',
    0xd9cc: 'aaaaaaaaaabbaN',
    0xd9cd: 'aaaaaaaaaabbaL',
    0xd9ce: 'aaaaaaaaaabbaJ',
    0xd9cf: 'aaaaaaaaaabbaH',
    0xd9d0: 'aaaaaaaaaabbav',
    0xd9d1: 'aaaaaaaaaabbat',
    0xd9d2: 'aaaa013',
    0xd9d3: 'aaaajz',
    0xd9d4: 'aaaaaaaaaabbiz',
    0xd9d5: 'aaaaaaaaaabbix',
    0xd9d6: 'aaaaaaaaaabbaz',
    0xd9d7: 'aaaaaaaaaabbax',
    0xd9d8: 'aaaaaaaaaabbaf',
    0xd9d9: 'aaaaaaaaaabbad',
    0xd9da: 'aaaaaaaaaabbab',
    0xd9db: 'aabbkz',
    0xd9dc: 'aaaaaaaaaabban',
    0xd9dd: 'aaaaaaaaaabbal',
    0xd9de: 'aaaaaaaaaabbaj',
    0xd9df: 'aaaaaaaaaabbah',
    0xd9e0: 'aaaaaaaaaabbA',
    0xd9e1: 'aaaaaaaaaabbE',
    0xd9e2: 'aaab013',
    0xd9e3: 'aaabjz',
    0xd9e4: 'aaaaaaaaaabbQ',
    0xd9e5: 'aaaaaaaaaabbU',
    0xd9e6: 'aaaaaaaaaabbY',
    0xd9e7: 'aaaaaaaaaabhX',
    0xd9e8: 'aaaaaaaaaabba',
    0xd9e9: 'aaaaaaaaaabbe',
    0xd9ea: 'aaaaaaaaaabbi',
    0xd9eb: 'aabakz',
    0xd9ec: 'aaaaaaaaaabbq',
    0xd9ed: 'aaaaaaaaaabbu',
    0xd9ee: 'aaaaaaaaaabby',
    0xd9ef: 'aaaaaaaaaabhx',
    0xd9f0: 'aaaaaaaaaabh',
    0xd9f1: 'aaaaaaaaaadc',
    0xd9f2: 'aaaa113',
    0xd9f3: 'aaaakz',
    0xd9f4: 'aaaaaaaaaabH',
    0xd9f5: 'aaaaaaaaaadC',
    0xd9f6: 'aaaaaaaaaabX',
    0xd9f7: 'aaaaaaaaaabP',
    0xd9f8: 'aaaaaaaaaaB',
    0xd9f9: 'aaaaaaaaaaR',
    0xd9fa: 'aaaaaaaaaab',
    0xd9fb: 'aabbjz',
    0xd9fc: 'aaaaaaaE9',
    0xd9fd: 'aaaaaaa7',
    0xd9fe: 'aaaaaaa119',
    0xd9ff: 'aaaaaaaaA5',
    0xda00: 'bbba004',
    0xda01: 'bbbabq',
    0xda02: 'aaab104',
    0xda03: 'aaabap',
    0xda04: 'cccccccabbC',
    0xda05: 'cccccccabbS',
    0xda06: 'cccccccabbc',
    0xda07: 'ccbabq',
    0xda08: 'ccccccccaabH',
    0xda09: 'bbabap',
    0xda0a: 'ccccccccaabX',
    0xda0b: 'aababq',
    0xda0c: 'ccccccccaabh',
    0xda0d: 'ccccccccaadc',
    0xda0e: 'ccca004',
    0xda0f: 'ccabap',
    0xda10: 'bbbb104',
    0xda11: 'bbbbap',
    0xda12: 'aaaa004',
    0xda13: 'aaaabq',
    0xda14: 'cccccccccbaca',
    0xda15: 'cccccccccbaad',
    0xda16: 'cccccccccbaah',
    0xda17: 'ccbbap',
    0xda18: 'cccccccccbaaP',
    0xda19: 'bbaabq',
    0xda1a: 'cccccccccbaaX',
    0xda1b: 'aabbap',
    0xda1c: 'cccccccccbacA',
    0xda1d: 'cccccccccbaaD',
    0xda1e: 'cccb104',
    0xda1f: 'ccaabq',
    0xda20: 'bbba104',
    0xda21: 'bbbaap',
    0xda22: 'aaab004',
    0xda23: 'aaabbq',
    0xda24: 'ddda004',
    0xda25: 'ddbbbq',
    0xda26: 'gggb104',
    0xda27: 'ccbaap',
    0xda28: 'fffb104',
    0xda29: 'bbabbq',
    0xda2a: 'eeea004',
    0xda2b: 'aabaap',
    0xda2c: 'hhha014',
    0xda2d: 'ddaaap',
    0xda2e: 'ccca104',
    0xda2f: 'ccabbq',
    0xda30: 'bbbb004',
    0xda31: 'bbbbbq',
    0xda32: 'aaaa104',
    0xda33: 'aaaaap',
    0xda34: 'dddb104',
    0xda35: 'ddbaap',
    0xda36: 'ggga004',
    0xda37: 'ccbbbq',
    0xda38: 'fffa004',
    0xda39: 'bbaaap',
    0xda3a: 'dddddaaD8',
    0xda3b: 'aabbbq',
    0xda3c: 'aaaaabaD8',
    0xda3d: 'aaaaababk',
    0xda3e: 'bbbbbabE8',
    0xda3f: 'bbbbbabaj',
    0xda40: 'bbba014',
    0xda41: 'bbbabp',
    0xda42: 'aaab114',
    0xda43: 'aaabaq',
    0xda44: 'ddda114',
    0xda45: 'ddbbaq',
    0xda46: 'fffffaaD8',
    0xda47: 'ccbabp',
    0xda48: 'fffb014',
    0xda49: 'bbabaq',
    0xda4a: 'eeea114',
    0xda4b: 'aababp',
    0xda4c: 'hhha104',
    0xda4d: 'ddaabp',
    0xda4e: 'ccca014',
    0xda4f: 'ccabaq',
    0xda50: 'bbbb114',
    0xda51: 'bbbbaq',
    0xda52: 'aaaa014',
    0xda53: 'aaaabp',
    0xda54: 'dddb014',
    0xda55: 'ddbabp',
    0xda56: 'ggga114',
    0xda57: 'ccbbaq',
    0xda58: 'fffa114',
    0xda59: 'bbaabp',
    0xda5a: 'eeeb014',
    0xda5b: 'aabbaq',
    0xda5c: 'hhhb004',
    0xda5d: 'ddabaq',
    0xda5e: 'cccb114',
    0xda5f: 'ccaabp',
    0xda60: 'bbba114',
    0xda61: 'bbbaaq',
    0xda62: 'aaab014',
    0xda63: 'aaabbp',
    0xda64: 'ddda014',
    0xda65: 'ddbbbp',
    0xda66: 'gggb114',
    0xda67: 'ccbaaq',
    0xda68: 'fffb114',
    0xda69: 'bbabbp',
    0xda6a: 'eeea014',
    0xda6b: 'aabaaq',
    0xda6c: 'hhha004',
    0xda6d: 'ddaaaq',
    0xda6e: 'ccca114',
    0xda6f: 'ccabbp',
    0xda70: 'bbbb014',
    0xda71: 'bbbbbp',
    0xda72: 'aaaa114',
    0xda73: 'aaaaaq',
    0xda74: 'dddb114',
    0xda75: 'ddbaaq',
    0xda76: 'ggga014',
    0xda77: 'ccbbbp',
    0xda78: 'dddddda109',
    0xda79: 'bbaaaq',
    0xda7a: 'dddddbbE8',
    0xda7b: 'aabbbp',
    0xda7c: 'aaaaaabE8',
    0xda7d: 'aaaaaabaj',
    0xda7e: 'aaaaaab109',
    0xda7f: 'bbbbbbabk',
    0xda80: 'bbba005',
    0xda81: 'bbbabs',
    0xda82: 'aaab105',
    0xda83: 'aaabar',
    0xda84: 'ddda105',
    0xda85: 'ddbbar',
    0xda86: 'eeeeeeb009',
    0xda87: 'ccbabs',
    0xda88: 'fffb005',
    0xda89: 'bbabar',
    0xda8a: 'eeea105',
    0xda8b: 'aababs',
    0xda8c: 'hhha115',
    0xda8d: 'ddaabs',
    0xda8e: 'ccca005',
    0xda8f: 'ccabar',
    0xda90: 'bbbb105',
    0xda91: 'bbbbar',
    0xda92: 'aaaa005',
    0xda93: 'aaaabs',
    0xda94: 'dddb005',
    0xda95: 'ddbabs',
    0xda96: 'ggga105',
    0xda97: 'ccbbar',
    0xda98: 'fffa105',
    0xda99: 'bbaabs',
    0xda9a: 'eeeb005',
    0xda9b: 'aabbar',
    0xda9c: 'hhhb015',
    0xda9d: 'ddabar',
    0xda9e: 'cccb105',
    0xda9f: 'ccaabs',
    0xdaa0: 'bbba105',
    0xdaa1: 'bbbaar',
    0xdaa2: 'aaab005',
    0xdaa3: 'aaabbs',
    0xdaa4: 'ddda005',
    0xdaa5: 'ddbbbs',
    0xdaa6: 'gggb105',
    0xdaa7: 'ccbaar',
    0xdaa8: 'fffb105',
    0xdaa9: 'bbabbs',
    0xdaaa: 'eeea005',
    0xdaab: 'aabaar',
    0xdaac: 'hhha015',
    0xdaad: 'ddaaar',
    0xdaae: 'ccca105',
    0xdaaf: 'ccabbs',
    0xdab0: 'bbbb005',
    0xdab1: 'bbbbbs',
    0xdab2: 'aaaa105',
    0xdab3: 'aaaaar',
    0xdab4: 'dddb105',
    0xdab5: 'ddbaar',
    0xdab6: 'ggga005',
    0xdab7: 'ccbbbs',
    0xdab8: 'fffa005',
    0xdab9: 'bbaaar',
    0xdaba: 'dddddabE8',
    0xdabb: 'aabbbs',
    0xdabc: 'aaaaabbE8',
    0xdabd: 'aaaaabbaj',
    0xdabe: 'bbbbbaaD8',
    0xdabf: 'bbbbbaabk',
    0xdac0: 'bbba015',
    0xdac1: 'bbbabr',
    0xdac2: 'aaab115',
    0xdac3: 'aaabas',
    0xdac4: 'ddda115',
    0xdac5: 'ddbbas',
    0xdac6: 'fffffabE8',
    0xdac7: 'ccbabr',
    0xdac8: 'fffb015',
    0xdac9: 'bbabas',
    0xdaca: 'eeea115',
    0xdacb: 'aababr',
    0xdacc: 'hhha105',
    0xdacd: 'ddaabr',
    0xdace: 'ccca015',
    0xdacf: 'ccabas',
    0xdad0: 'bbbb115',
    0xdad1: 'bbbbas',
    0xdad2: 'aaaa015',
    0xdad3: 'aaaabr',
    0xdad4: 'dddb015',
    0xdad5: 'ddbabr',
    0xdad6: 'ggga115',
    0xdad7: 'ccbbas',
    0xdad8: 'fffa115',
    0xdad9: 'bbaabr',
    0xdada: 'eeeb015',
    0xdadb: 'aabbas',
    0xdadc: 'hhhb005',
    0xdadd: 'ddabas',
    0xdade: 'cccb115',
    0xdadf: 'ccaabr',
    0xdae0: 'bbba115',
    0xdae1: 'bbbaas',
    0xdae2: 'aaab015',
    0xdae3: 'aaabbr',
    0xdae4: 'ddda015',
    0xdae5: 'ddbbbr',
    0xdae6: 'gggb115',
    0xdae7: 'ccbaas',
    0xdae8: 'fffb115',
    0xdae9: 'bbabbr',
    0xdaea: 'eeea015',
    0xdaeb: 'aabaas',
    0xdaec: 'hhha005',
    0xdaed: 'ddaaas',
    0xdaee: 'ccca115',
    0xdaef: 'ccabbr',
    0xdaf0: 'bbbb015',
    0xdaf1: 'bbbbbr',
    0xdaf2: 'aaaa115',
    0xdaf3: 'aaaaas',
    0xdaf4: 'dddb115',
    0xdaf5: 'ddbaas',
    0xdaf6: 'ggga015',
    0xdaf7: 'ccbbbr',
    0xdaf8: 'aaaaaaabaaB',
    0xdaf9: 'aaaaaaabaaR',
    0xdafa: 'aaaaaaabaab',
    0xdafb: 'aabbbr',
    0xdafc: 'aaaaaaaD8',
    0xdafd: 'aaaaaaabk',
    0xdafe: 'aaaaaaa009',
    0xdaff: 'bbbbbbbaj',
    0xdb00: 'bbba006',
    0xdb01: 'bbbabu',
    0xdb02: 'aaab106',
    0xdb03: 'aaabat',
    0xdb04: 'cccccccbaaB',
    0xdb05: 'cccccccbaaR',
    0xdb06: 'cccccccbaab',
    0xdb07: 'ccbabu',
    0xdb08: 'fffb006',
    0xdb09: 'bbabat',
    0xdb0a: 'eeea106',
    0xdb0b: 'aababu',
    0xdb0c: 'hhha116',
    0xdb0d: 'ddaabu',
    0xdb0e: 'ccca006',
    0xdb0f: 'ccabat',
    0xdb10: 'bbbb106',
    0xdb11: 'bbbbat',
    0xdb12: 'aaaa006',
    0xdb13: 'aaaabu',
    0xdb14: 'dddb006',
    0xdb15: 'ddbabu',
    0xdb16: 'ggga106',
    0xdb17: 'ccbbat',
    0xdb18: 'fffa106',
    0xdb19: 'bbaabu',
    0xdb1a: 'eeeb006',
    0xdb1b: 'aabbat',
    0xdb1c: 'hhhb016',
    0xdb1d: 'ddabat',
    0xdb1e: 'cccb106',
    0xdb1f: 'ccaabu',
    0xdb20: 'bbba106',
    0xdb21: 'bbbaat',
    0xdb22: 'aaab006',
    0xdb23: 'aaabbu',
    0xdb24: 'ddda006',
    0xdb25: 'ddbbbu',
    0xdb26: 'gggb106',
    0xdb27: 'ccbaat',
    0xdb28: 'fffb106',
    0xdb29: 'bbabbu',
    0xdb2a: 'eeea006',
    0xdb2b: 'aabaat',
    0xdb2c: 'hhha016',
    0xdb2d: 'ddaaat',
    0xdb2e: 'ccca106',
    0xdb2f: 'ccabbu',
    0xdb30: 'bbbb006',
    0xdb31: 'bbbbbu',
    0xdb32: 'aaaa106',
    0xdb33: 'aaaaat',
    0xdb34: 'dddb106',
    0xdb35: 'ddbaat',
    0xdb36: 'ggga006',
    0xdb37: 'ccbbbu',
    0xdb38: 'fffa006',
    0xdb39: 'bbaaat',
    0xdb3a: 'dddddaaE8',
    0xdb3b: 'aabbbu',
    0xdb3c: 'aaaaabaE8',
    0xdb3d: 'aaaaabaaj',
    0xdb3e: 'bbbbbabD8',
    0xdb3f: 'bbbbbabbk',
    0xdb40: 'bbba016',
    0xdb41: 'bbbabt',
    0xdb42: 'aaab116',
    0xdb43: 'aaabau',
    0xdb44: 'ddda116',
    0xdb45: 'ddbbau',
    0xdb46: 'fffffaaE8',
    0xdb47: 'ccbabt',
    0xdb48: 'fffb016',
    0xdb49: 'bbabau',
    0xdb4a: 'eeea116',
    0xdb4b: 'aababt',
    0xdb4c: 'hhha106',
    0xdb4d: 'ddaabt',
    0xdb4e: 'ccca016',
    0xdb4f: 'ccabau',
    0xdb50: 'bbbb116',
    0xdb51: 'bbbbau',
    0xdb52: 'aaaa016',
    0xdb53: 'aaaabt',
    0xdb54: 'dddb016',
    0xdb55: 'ddbabt',
    0xdb56: 'ggga116',
    0xdb57: 'ccbbau',
    0xdb58: 'fffa116',
    0xdb59: 'bbaabt',
    0xdb5a: 'eeeb016',
    0xdb5b: 'aabbau',
    0xdb5c: 'hhhb006',
    0xdb5d: 'ddabau',
    0xdb5e: 'cccb116',
    0xdb5f: 'ccaabt',
    0xdb60: 'bbba116',
    0xdb61: 'bbbaau',
    0xdb62: 'aaab016',
    0xdb63: 'aaabbt',
    0xdb64: 'ddda016',
    0xdb65: 'ddbbbt',
    0xdb66: 'gggb116',
    0xdb67: 'ccbaau',
    0xdb68: 'fffb116',
    0xdb69: 'bbabbt',
    0xdb6a: 'eeea016',
    0xdb6b: 'aabaau',
    0xdb6c: 'hhha006',
    0xdb6d: 'ddaaau',
    0xdb6e: 'ccca116',
    0xdb6f: 'ccabbt',
    0xdb70: 'bbbb016',
    0xdb71: 'bbbbbt',
    0xdb72: 'aaaa116',
    0xdb73: 'aaaaau',
    0xdb74: 'dddb116',
    0xdb75: 'ddbaau',
    0xdb76: 'ggga016',
    0xdb77: 'ccbbbt',
    0xdb78: 'dddddda009',
    0xdb79: 'bbaaau',
    0xdb7a: 'dddddbbD8',
    0xdb7b: 'aabbbt',
    0xdb7c: 'aaaaaabD8',
    0xdb7d: 'aaaaaabbk',
    0xdb7e: 'aaaaaab009',
    0xdb7f: 'bbbbbbaaj',
    0xdb80: 'bbba007',
    0xdb81: 'bbbabw',
    0xdb82: 'aaab107',
    0xdb83: 'aaabav',
    0xdb84: 'ddda107',
    0xdb85: 'ddbbav',
    0xdb86: 'eeeeeeb109',
    0xdb87: 'ccbabw',
    0xdb88: 'fffb007',
    0xdb89: 'bbabav',
    0xdb8a: 'eeea107',
    0xdb8b: 'aababw',
    0xdb8c: 'hhha117',
    0xdb8d: 'ddaabw',
    0xdb8e: 'ccca007',
    0xdb8f: 'ccabav',
    0xdb90: 'bbbb107',
    0xdb91: 'bbbbav',
    0xdb92: 'aaaa007',
    0xdb93: 'aaaabw',
    0xdb94: 'dddb007',
    0xdb95: 'ddbabw',
    0xdb96: 'ggga107',
    0xdb97: 'ccbbav',
    0xdb98: 'fffa107',
    0xdb99: 'bbaabw',
    0xdb9a: 'eeeb007',
    0xdb9b: 'aabbav',
    0xdb9c: 'hhhb017',
    0xdb9d: 'ddabav',
    0xdb9e: 'cccb107',
    0xdb9f: 'ccaabw',
    0xdba0: 'bbba107',
    0xdba1: 'bbbaav',
    0xdba2: 'aaab007',
    0xdba3: 'aaabbw',
    0xdba4: 'ddda007',
    0xdba5: 'ddbbbw',
    0xdba6: 'gggb107',
    0xdba7: 'ccbaav',
    0xdba8: 'fffb107',
    0xdba9: 'bbabbw',
    0xdbaa: 'eeea007',
    0xdbab: 'aabaav',
    0xdbac: 'hhha017',
    0xdbad: 'ddaaav',
    0xdbae: 'ccca107',
    0xdbaf: 'ccabbw',
    0xdbb0: 'bbbb007',
    0xdbb1: 'bbbbbw',
    0xdbb2: 'aaaa107',
    0xdbb3: 'aaaaav',
    0xdbb4: 'dddb107',
    0xdbb5: 'ddbaav',
    0xdbb6: 'ggga007',
    0xdbb7: 'ccbbbw',
    0xdbb8: 'fffa007',
    0xdbb9: 'bbaaav',
    0xdbba: 'dddddabD8',
    0xdbbb: 'aabbbw',
    0xdbbc: 'aaaaabbD8',
    0xdbbd: 'aaaaabbbk',
    0xdbbe: 'bbbbbaaE8',
    0xdbbf: 'bbbbbaaaj',
    0xdbc0: 'bbba017',
    0xdbc1: 'bbbabv',
    0xdbc2: 'aaab117',
    0xdbc3: 'aaabaw',
    0xdbc4: 'ddda117',
    0xdbc5: 'ddbbaw',
    0xdbc6: 'fffffabD8',
    0xdbc7: 'ccbabv',
    0xdbc8: 'fffb017',
    0xdbc9: 'bbabaw',
    0xdbca: 'eeea117',
    0xdbcb: 'aababv',
    0xdbcc: 'hhha107',
    0xdbcd: 'ddaabv',
    0xdbce: 'ccca017',
    0xdbcf: 'ccabaw',
    0xdbd0: 'bbbb117',
    0xdbd1: 'bbbbaw',
    0xdbd2: 'aaaa017',
    0xdbd3: 'aaaabv',
    0xdbd4: 'dddb017',
    0xdbd5: 'ddbabv',
    0xdbd6: 'ggga117',
    0xdbd7: 'ccbbaw',
    0xdbd8: 'fffa117',
    0xdbd9: 'bbaabv',
    0xdbda: 'eeeb017',
    0xdbdb: 'aabbaw',
    0xdbdc: 'hhhb007',
    0xdbdd: 'ddabaw',
    0xdbde: 'cccb117',
    0xdbdf: 'ccaabv',
    0xdbe0: 'bbba117',
    0xdbe1: 'bbbaaw',
    0xdbe2: 'aaab017',
    0xdbe3: 'aaabbv',
    0xdbe4: 'ddda017',
    0xdbe5: 'ddbbbv',
    0xdbe6: 'gggb117',
    0xdbe7: 'ccbaaw',
    0xdbe8: 'fffb117',
    0xdbe9: 'bbabbv',
    0xdbea: 'eeea017',
    0xdbeb: 'aabaaw',
    0xdbec: 'hhha007',
    0xdbed: 'ddaaaw',
    0xdbee: 'ccca117',
    0xdbef: 'ccabbv',
    0xdbf0: 'aaaaaaaabbai',
    0xdbf1: 'aaaaaaaabbaa',
    0xdbf2: 'aaaa117',
    0xdbf3: 'aaaaaw',
    0xdbf4: 'aaaaaaaabbaI',
    0xdbf5: 'aaaaaaaabbaA',
    0xdbf6: 'aaaaaaaabbaY',
    0xdbf7: 'aaaaaaaabbaQ',
    0xdbf8: 'aaaaaaaabbC',
    0xdbf9: 'aaaaaaaabbS',
    0xdbfa: 'aaaaaaaabbc',
    0xdbfb: 'aabbbv',
    0xdbfc: 'aaaaaaaE8',
    0xdbfd: 'aaaaaaaaj',
    0xdbfe: 'aaaaaaa109',
    0xdbff: 'aaaaaaaaB4',
    0xdc00: 'bbba008',
    0xdc01: 'bbbabi',
    0xdc02: 'aaab108',
    0xdc03: 'aaabah',
    0xdc04: 'cccccccaabC',
    0xdc05: 'cccccccaabS',
    0xdc06: 'cccccccaabc',
    0xdc07: 'ccbabi',
    0xdc08: 'ccccccccbabH',
    0xdc09: 'bbabah',
    0xdc0a: 'ccccccccbabX',
    0xdc0b: 'aababi',
    0xdc0c: 'ccccccccbabh',
    0xdc0d: 'ccccccccbadc',
    0xdc0e: 'ccca008',
    0xdc0f: 'ccabah',
    0xdc10: 'bbbb108',
    0xdc11: 'bbbbah',
    0xdc12: 'aaaa008',
    0xdc13: 'aaaabi',
    0xdc14: 'dddb008',
    0xdc15: 'ddbabi',
    0xdc16: 'ggga108',
    0xdc17: 'ccbbah',
    0xdc18: 'fffa108',
    0xdc19: 'bbaabi',
    0xdc1a: 'eeeb008',
    0xdc1b: 'aabbah',
    0xdc1c: 'hhhb018',
    0xdc1d: 'ddabah',
    0xdc1e: 'cccb108',
    0xdc1f: 'ccaabi',
    0xdc20: 'bbba108',
    0xdc21: 'bbbaah',
    0xdc22: 'aaab008',
    0xdc23: 'aaabbi',
    0xdc24: 'ddda008',
    0xdc25: 'ddbbbi',
    0xdc26: 'gggb108',
    0xdc27: 'ccbaah',
    0xdc28: 'fffb108',
    0xdc29: 'bbabbi',
    0xdc2a: 'eeea008',
    0xdc2b: 'aabaah',
    0xdc2c: 'hhha018',
    0xdc2d: 'ddaaah',
    0xdc2e: 'ccca108',
    0xdc2f: 'ccabbi',
    0xdc30: 'bbbb008',
    0xdc31: 'bbbbbi',
    0xdc32: 'aaaa108',
    0xdc33: 'aaaaah',
    0xdc34: 'dddb108',
    0xdc35: 'ddbaah',
    0xdc36: 'ggga008',
    0xdc37: 'ccbbbi',
    0xdc38: 'fffa008',
    0xdc39: 'bbaaah',
    0xdc3a: 'dddddaaB8',
    0xdc3b: 'aabbbi',
    0xdc3c: 'aaaaabaB8',
    0xdc3d: 'aaaaababh',
    0xdc3e: 'bbbbbabA9',
    0xdc3f: 'bbbbbabai',
    0xdc40: 'bbba018',
    0xdc41: 'bbbabh',
    0xdc42: 'aaab118',
    0xdc43: 'aaabai',
    0xdc44: 'ddda118',
    0xdc45: 'ddbbai',
    0xdc46: 'fffffaaB8',
    0xdc47: 'ccbabh',
    0xdc48: 'fffb018',
    0xdc49: 'bbabai',
    0xdc4a: 'eeea118',
    0xdc4b: 'aababh',
    0xdc4c: 'hhha108',
    0xdc4d: 'ddaabh',
    0xdc4e: 'ccca018',
    0xdc4f: 'ccabai',
    0xdc50: 'bbbb118',
    0xdc51: 'bbbbai',
    0xdc52: 'aaaa018',
    0xdc53: 'aaaabh',
    0xdc54: 'dddb018',
    0xdc55: 'ddbabh',
    0xdc56: 'ggga118',
    0xdc57: 'ccbbai',
    0xdc58: 'fffa118',
    0xdc59: 'bbaabh',
    0xdc5a: 'eeeb018',
    0xdc5b: 'aabbai',
    0xdc5c: 'hhhb008',
    0xdc5d: 'ddabai',
    0xdc5e: 'cccb118',
    0xdc5f: 'ccaabh',
    0xdc60: 'bbba118',
    0xdc61: 'bbbaai',
    0xdc62: 'aaab018',
    0xdc63: 'aaabbh',
    0xdc64: 'ddda018',
    0xdc65: 'ddbbbh',
    0xdc66: 'gggb118',
    0xdc67: 'ccbaai',
    0xdc68: 'fffb118',
    0xdc69: 'bbabbh',
    0xdc6a: 'eeea018',
    0xdc6b: 'aabaai',
    0xdc6c: 'hhha008',
    0xdc6d: 'ddaaai',
    0xdc6e: 'ccca118',
    0xdc6f: 'ccabbh',
    0xdc70: 'bbbb018',
    0xdc71: 'bbbbbh',
    0xdc72: 'aaaa118',
    0xdc73: 'aaaaai',
    0xdc74: 'dddb118',
    0xdc75: 'ddbaai',
    0xdc76: 'ggga018',
    0xdc77: 'ccbbbh',
    0xdc78: 'dddddda118',
    0xdc79: 'bbaaai',
    0xdc7a: 'dddddbbA9',
    0xdc7b: 'aabbbh',
    0xdc7c: 'aaaaaabA9',
    0xdc7d: 'aaaaaabai',
    0xdc7e: 'aaaaaab118',
    0xdc7f: 'bbbbbbabh',
    0xdc80: 'bbba009',
    0xdc81: 'bbbabk',
    0xdc82: 'aaab109',
    0xdc83: 'aaabaj',
    0xdc84: 'ddda109',
    0xdc85: 'ddbbaj',
    0xdc86: 'eeeeeeb018',
    0xdc87: 'ccbabk',
    0xdc88: 'fffb009',
    0xdc89: 'bbabaj',
    0xdc8a: 'eeea109',
    0xdc8b: 'aababk',
    0xdc8c: 'hhha119',
    0xdc8d: 'ddaabk',
    0xdc8e: 'ccca009',
    0xdc8f: 'ccabaj',
    0xdc90: 'bbbb109',
    0xdc91: 'bbbbaj',
    0xdc92: 'aaaa009',
    0xdc93: 'aaaabk',
    0xdc94: 'dddb009',
    0xdc95: 'ddbabk',
    0xdc96: 'ggga109',
    0xdc97: 'ccbbaj',
    0xdc98: 'fffa109',
    0xdc99: 'bbaabk',
    0xdc9a: 'eeeb009',
    0xdc9b: 'aabbaj',
    0xdc9c: 'hhhb019',
    0xdc9d: 'ddabaj',
    0xdc9e: 'cccb109',
    0xdc9f: 'ccaabk',
    0xdca0: 'bbba109',
    0xdca1: 'bbbaaj',
    0xdca2: 'aaab009',
    0xdca3: 'aaabbk',
    0xdca4: 'ddda009',
    0xdca5: 'ddbbbk',
    0xdca6: 'gggb109',
    0xdca7: 'ccbaaj',
    0xdca8: 'fffb109',
    0xdca9: 'bbabbk',
    0xdcaa: 'eeea009',
    0xdcab: 'aabaaj',
    0xdcac: 'hhha019',
    0xdcad: 'ddaaaj',
    0xdcae: 'ccca109',
    0xdcaf: 'ccabbk',
    0xdcb0: 'bbbb009',
    0xdcb1: 'bbbbbk',
    0xdcb2: 'aaaa109',
    0xdcb3: 'aaaaaj',
    0xdcb4: 'dddb109',
    0xdcb5: 'ddbaaj',
    0xdcb6: 'ggga009',
    0xdcb7: 'ccbbbk',
    0xdcb8: 'fffa009',
    0xdcb9: 'bbaaaj',
    0xdcba: 'dddddabA9',
    0xdcbb: 'aabbbk',
    0xdcbc: 'aaaaabbA9',
    0xdcbd: 'aaaaabbai',
    0xdcbe: 'bbbbbaaB8',
    0xdcbf: 'bbbbbaabh',
    0xdcc0: 'bbba019',
    0xdcc1: 'bbbabj',
    0xdcc2: 'aaab119',
    0xdcc3: 'aaabak',
    0xdcc4: 'ddda119',
    0xdcc5: 'ddbbak',
    0xdcc6: 'fffffabA9',
    0xdcc7: 'ccbabj',
    0xdcc8: 'fffb019',
    0xdcc9: 'bbabak',
    0xdcca: 'eeea119',
    0xdccb: 'aababj',
    0xdccc: 'hhha109',
    0xdccd: 'ddaabj',
    0xdcce: 'ccca019',
    0xdccf: 'ccabak',
    0xdcd0: 'bbbb119',
    0xdcd1: 'bbbbak',
    0xdcd2: 'aaaa019',
    0xdcd3: 'aaaabj',
    0xdcd4: 'dddb019',
    0xdcd5: 'ddbabj',
    0xdcd6: 'ggga119',
    0xdcd7: 'ccbbak',
    0xdcd8: 'fffa119',
    0xdcd9: 'bbaabj',
    0xdcda: 'eeeb019',
    0xdcdb: 'aabbak',
    0xdcdc: 'hhhb009',
    0xdcdd: 'ddabak',
    0xdcde: 'cccb119',
    0xdcdf: 'ccaabj',
    0xdce0: 'bbba119',
    0xdce1: 'bbbaak',
    0xdce2: 'aaab019',
    0xdce3: 'aaabbj',
    0xdce4: 'ddda019',
    0xdce5: 'ddbbbj',
    0xdce6: 'gggb119',
    0xdce7: 'ccbaak',
    0xdce8: 'fffb119',
    0xdce9: 'bbabbj',
    0xdcea: 'eeea019',
    0xdceb: 'aabaak',
    0xdcec: 'hhha009',
    0xdced: 'ddaaak',
    0xdcee: 'ccca119',
    0xdcef: 'ccabbj',
    0xdcf0: 'bbbb019',
    0xdcf1: 'bbbbbj',
    0xdcf2: 'aaaa119',
    0xdcf3: 'aaaaak',
    0xdcf4: 'dddb119',
    0xdcf5: 'ddbaak',
    0xdcf6: 'ggga019',
    0xdcf7: 'ccbbbj',
    0xdcf8: 'aaaaaaabbaB',
    0xdcf9: 'aaaaaaabbaR',
    0xdcfa: 'aaaaaaabbab',
    0xdcfb: 'aabbbj',
    0xdcfc: 'aaaaaaaB8',
    0xdcfd: 'aaaaaaabh',
    0xdcfe: 'aaaaaaa018',
    0xdcff: 'bbbbbbbai',
    0xdd00: 'bbba048',
    0xdd01: 'bbbabm',
    0xdd02: 'aaab148',
    0xdd03: 'aaabal',
    0xdd04: 'cccccccbbaB',
    0xdd05: 'cccccccbbaR',
    0xdd06: 'cccccccbbab',
    0xdd07: 'ccbabm',
    0xdd08: 'fffb048',
    0xdd09: 'bbabal',
    0xdd0a: 'eeea148',
    0xdd0b: 'aababm',
    0xdd0c: 'hhha158',
    0xdd0d: 'ddaabm',
    0xdd0e: 'ccca048',
    0xdd0f: 'ccabal',
    0xdd10: 'bbbb148',
    0xdd11: 'bbbbal',
    0xdd12: 'aaaa048',
    0xdd13: 'aaaabm',
    0xdd14: 'dddb048',
    0xdd15: 'ddbabm',
    0xdd16: 'ggga148',
    0xdd17: 'ccbbal',
    0xdd18: 'fffa148',
    0xdd19: 'bbaabm',
    0xdd1a: 'eeeb048',
    0xdd1b: 'aabbal',
    0xdd1c: 'hhhb058',
    0xdd1d: 'ddabal',
    0xdd1e: 'cccb148',
    0xdd1f: 'ccaabm',
    0xdd20: 'bbba148',
    0xdd21: 'bbbaal',
    0xdd22: 'aaab048',
    0xdd23: 'aaabbm',
    0xdd24: 'ddda048',
    0xdd25: 'ddbbbm',
    0xdd26: 'gggb148',
    0xdd27: 'ccbaal',
    0xdd28: 'fffb148',
    0xdd29: 'bbabbm',
    0xdd2a: 'eeea048',
    0xdd2b: 'aabaal',
    0xdd2c: 'hhha058',
    0xdd2d: 'ddaaal',
    0xdd2e: 'ccca148',
    0xdd2f: 'ccabbm',
    0xdd30: 'bbbb048',
    0xdd31: 'bbbbbm',
    0xdd32: 'aaaa148',
    0xdd33: 'aaaaal',
    0xdd34: 'dddb148',
    0xdd35: 'ddbaal',
    0xdd36: 'ggga048',
    0xdd37: 'ccbbbm',
    0xdd38: 'fffa048',
    0xdd39: 'bbaaal',
    0xdd3a: 'dddddaaA9',
    0xdd3b: 'aabbbm',
    0xdd3c: 'aaaaabaA9',
    0xdd3d: 'aaaaabaai',
    0xdd3e: 'bbbbbabB8',
    0xdd3f: 'bbbbbabbh',
    0xdd40: 'bbba058',
    0xdd41: 'bbbabl',
    0xdd42: 'aaab158',
    0xdd43: 'aaabam',
    0xdd44: 'ddda158',
    0xdd45: 'ddbbam',
    0xdd46: 'fffffaaA9',
    0xdd47: 'ccbabl',
    0xdd48: 'fffb058',
    0xdd49: 'bbabam',
    0xdd4a: 'eeea158',
    0xdd4b: 'aababl',
    0xdd4c: 'hhha148',
    0xdd4d: 'ddaabl',
    0xdd4e: 'ccca058',
    0xdd4f: 'ccabam',
    0xdd50: 'bbbb158',
    0xdd51: 'bbbbam',
    0xdd52: 'aaaa058',
    0xdd53: 'aaaabl',
    0xdd54: 'dddb058',
    0xdd55: 'ddbabl',
    0xdd56: 'ggga158',
    0xdd57: 'ccbbam',
    0xdd58: 'fffa158',
    0xdd59: 'bbaabl',
    0xdd5a: 'eeeb058',
    0xdd5b: 'aabbam',
    0xdd5c: 'hhhb048',
    0xdd5d: 'ddabam',
    0xdd5e: 'cccb158',
    0xdd5f: 'ccaabl',
    0xdd60: 'bbba158',
    0xdd61: 'bbbaam',
    0xdd62: 'aaab058',
    0xdd63: 'aaabbl',
    0xdd64: 'ddda058',
    0xdd65: 'ddbbbl',
    0xdd66: 'gggb158',
    0xdd67: 'ccbaam',
    0xdd68: 'fffb158',
    0xdd69: 'bbabbl',
    0xdd6a: 'eeea058',
    0xdd6b: 'aabaam',
    0xdd6c: 'hhha048',
    0xdd6d: 'ddaaam',
    0xdd6e: 'ccca158',
    0xdd6f: 'ccabbl',
    0xdd70: 'bbbb058',
    0xdd71: 'bbbbbl',
    0xdd72: 'aaaa158',
    0xdd73: 'aaaaam',
    0xdd74: 'dddb158',
    0xdd75: 'ddbaam',
    0xdd76: 'ggga058',
    0xdd77: 'ccbbbl',
    0xdd78: 'dddddda018',
    0xdd79: 'bbaaam',
    0xdd7a: 'dddddbbB8',
    0xdd7b: 'aabbbl',
    0xdd7c: 'aaaaaabB8',
    0xdd7d: 'aaaaaabbh',
    0xdd7e: 'aaaaaab018',
    0xdd7f: 'bbbbbbaai',
    0xdd80: 'bbba049',
    0xdd81: 'bbbabo',
    0xdd82: 'aaab149',
    0xdd83: 'aaaban',
    0xdd84: 'ddda149',
    0xdd85: 'ddbban',
    0xdd86: 'eeeeeeb118',
    0xdd87: 'ccbabo',
    0xdd88: 'fffb049',
    0xdd89: 'bbaban',
    0xdd8a: 'eeea149',
    0xdd8b: 'aababo',
    0xdd8c: 'hhha159',
    0xdd8d: 'ddaabo',
    0xdd8e: 'ccca049',
    0xdd8f: 'ccaban',
    0xdd90: 'bbbb149',
    0xdd91: 'bbbban',
    0xdd92: 'aaaa049',
    0xdd93: 'aaaabo',
    0xdd94: 'dddb049',
    0xdd95: 'ddbabo',
    0xdd96: 'ggga149',
    0xdd97: 'ccbban',
    0xdd98: 'fffa149',
    0xdd99: 'bbaabo',
    0xdd9a: 'eeeb049',
    0xdd9b: 'aabban',
    0xdd9c: 'hhhb059',
    0xdd9d: 'ddaban',
    0xdd9e: 'cccb149',
    0xdd9f: 'ccaabo',
    0xdda0: 'bbba149',
    0xdda1: 'bbbaan',
    0xdda2: 'aaab049',
    0xdda3: 'aaabbo',
    0xdda4: 'ddda049',
    0xdda5: 'ddbbbo',
    0xdda6: 'gggb149',
    0xdda7: 'ccbaan',
    0xdda8: 'fffb149',
    0xdda9: 'bbabbo',
    0xddaa: 'eeea049',
    0xddab: 'aabaan',
    0xddac: 'hhha059',
    0xddad: 'ddaaan',
    0xddae: 'ccca149',
    0xddaf: 'ccabbo',
    0xddb0: 'bbbb049',
    0xddb1: 'bbbbbo',
    0xddb2: 'aaaa149',
    0xddb3: 'aaaaan',
    0xddb4: 'dddb149',
    0xddb5: 'ddbaan',
    0xddb6: 'ggga049',
    0xddb7: 'ccbbbo',
    0xddb8: 'fffa049',
    0xddb9: 'bbaaan',
    0xddba: 'dddddabB8',
    0xddbb: 'aabbbo',
    0xddbc: 'aaaaabbB8',
    0xddbd: 'aaaaabbbh',
    0xddbe: 'bbbbbaaA9',
    0xddbf: 'bbbbbaaai',
    0xddc0: 'bbba059',
    0xddc1: 'bbbabn',
    0xddc2: 'aaab159',
    0xddc3: 'aaabao',
    0xddc4: 'ddda159',
    0xddc5: 'ddbbao',
    0xddc6: 'fffffabB8',
    0xddc7: 'ccbabn',
    0xddc8: 'fffb059',
    0xddc9: 'bbabao',
    0xddca: 'eeea159',
    0xddcb: 'aababn',
    0xddcc: 'hhha149',
    0xddcd: 'ddaabn',
    0xddce: 'ccca059',
    0xddcf: 'ccabao',
    0xddd0: 'bbbb159',
    0xddd1: 'bbbbao',
    0xddd2: 'aaaa059',
    0xddd3: 'aaaabn',
    0xddd4: 'dddb059',
    0xddd5: 'ddbabn',
    0xddd6: 'ggga159',
    0xddd7: 'ccbbao',
    0xddd8: 'fffa159',
    0xddd9: 'bbaabn',
    0xddda: 'eeeb059',
    0xdddb: 'aabbao',
    0xdddc: 'hhhb049',
    0xdddd: 'ddabao',
    0xddde: 'cccb159',
    0xdddf: 'ccaabn',
    0xdde0: 'aaaaaaaaabacA',
    0xdde1: 'aaaaaaaaabaaD',
    0xdde2: 'aaab059',
    0xdde3: 'aaabbn',
    0xdde4: 'aaaaaaaaabaaP',
    0xdde5: 'aaaaaaaaabaaT',
    0xdde6: 'aaaaaaaaabaaX',
    0xdde7: 'aaaaaaaaabaiX',
    0xdde8: 'aaaaaaaaabaca',
    0xdde9: 'aaaaaaaaabaad',
    0xddea: 'aaaaaaaaabaah',
    0xddeb: 'aabaao',
    0xddec: 'aaaaaaaaabaap',
    0xdded: 'aaaaaaaaabaat',
    0xddee: 'aaaaaaaaabaax',
    0xddef: 'aaaaaaaaabaix',
    0xddf0: 'aaaaaaaaabai',
    0xddf1: 'aaaaaaaaabaa',
    0xddf2: 'aaaa159',
    0xddf3: 'aaaaao',
    0xddf4: 'aaaaaaaaabaI',
    0xddf5: 'aaaaaaaaabaA',
    0xddf6: 'aaaaaaaaabaY',
    0xddf7: 'aaaaaaaaabaQ',
    0xddf8: 'aaaaaaaaabC',
    0xddf9: 'aaaaaaaaabS',
    0xddfa: 'aaaaaaaaabc',
    0xddfb: 'aabbbn',
    0xddfc: 'aaaaaaaA9',
    0xddfd: 'aaaaaaaai',
    0xddfe: 'aaaaaaa118',
    0xddff: 'aaaaaaaaA4',
    0xde00: 'bbba088',
    0xde01: 'bbbaba',
    0xde02: 'aaab188',
    0xde03: 'aaabca',
    0xde04: 'cccccccabaB',
    0xde05: 'cccccccabaR',
    0xde06: 'cccccccabab',
    0xde07: 'ccbaba',
    0xde08: 'ccccccccabaI',
    0xde09: 'bbabca',
    0xde0a: 'ccccccccabaY',
    0xde0b: 'aababa',
    0xde0c: 'ccccccccabai',
    0xde0d: 'ccccccccabaa',
    0xde0e: 'ccca088',
    0xde0f: 'ccabca',
    0xde10: 'bbbb188',
    0xde11: 'bbbbca',
    0xde12: 'aaaa088',
    0xde13: 'aaaaba',
    0xde14: 'cccccccccabba',
    0xde15: 'cccccccccabbe',
    0xde16: 'cccccccccabbi',
    0xde17: 'ccbbca',
    0xde18: 'cccccccccabbQ',
    0xde19: 'bbaaba',
    0xde1a: 'cccccccccabbY',
    0xde1b: 'aabbca',
    0xde1c: 'cccccccccabbA',
    0xde1d: 'cccccccccabbE',
    0xde1e: 'cccb188',
    0xde1f: 'ccaaba',
    0xde20: 'bbba188',
    0xde21: 'bbbaca',
    0xde22: 'aaab088',
    0xde23: 'aaabba',
    0xde24: 'ccccccccccaabg',
    0xde25: 'ccccccccccaabe',
    0xde26: 'ccccccccccaabc',
    0xde27: 'ccbaca',
    0xde28: 'ccccccccccaahz',
    0xde29: 'bbabba',
    0xde2a: 'ccccccccccaadx',
    0xde2b: 'aabaca',
    0xde2c: 'ccccccccccaabw',
    0xde2d: 'ccccccccccaabu',
    0xde2e: 'ccca188',
    0xde2f: 'ccabba',
    0xde30: 'bbbb088',
    0xde31: 'bbbbba',
    0xde32: 'aaaa188',
    0xde33: 'aaaaca',
    0xde34: 'ccccccccccaabG',
    0xde35: 'ccccccccccaabE',
    0xde36: 'ccccccccccaabC',
    0xde37: 'ccbbba',
    0xde38: 'ccccccccccaahZ',
    0xde39: 'bbaaca',
    0xde3a: 'ccccccccccaadX',
    0xde3b: 'aabbba',
    0xde3c: 'aaaaabaB9',
    0xde3d: 'aaaaababi',
    0xde3e: 'bbbbbabA8',
    0xde3f: 'bbbbbab1',
    0xde40: 'bbba098',
    0xde41: 'bbbadc',
    0xde42: 'aaab198',
    0xde43: 'aaabaa',
    0xde44: 'cccccccccccbabc',
    0xde45: 'cccccccccccbabb',
    0xde46: 'cccccccccccbaba',
    0xde47: 'ccbadc',
    0xde48: 'cccccccccccbabo',
    0xde49: 'bbabaa',
    0xde4a: 'cccccccccccbabm',
    0xde4b: 'aabadc',
    0xde4c: 'cccccccccccbabk',
    0xde4d: 'cccccccccccbabj',
    0xde4e: 'ccca098',
    0xde4f: 'ccabaa',
    0xde50: 'bbbb198',
    0xde51: 'bbbbaa',
    0xde52: 'aaaa098',
    0xde53: 'aaaadc',
    0xde54: 'cccccccccccbabs',
    0xde55: 'cccccccccccbabr',
    0xde56: 'cccccccccccbabq',
    0xde57: 'ccbbaa',
    0xde58: 'cccccccccccbahz',
    0xde59: 'bbaadc',
    0xde5a: 'cccccccccccbahx',
    0xde5b: 'aabbaa',
    0xde5c: 'cccccccccccbadx',
    0xde5d: 'cccccccccccbabz',
    0xde5e: 'cccb198',
    0xde5f: 'ccaadc',
    0xde60: 'bbba198',
    0xde61: 'bbbaaa',
    0xde62: 'aaab098',
    0xde63: 'aaabdc',
    0xde64: 'cccccccccccbabC',
    0xde65: 'cccccccccccbabB',
    0xde66: 'cccccccccccbabA',
    0xde67: 'ccbaaa',
    0xde68: 'cccccccccccbabO',
    0xde69: 'bbabdc',
    0xde6a: 'cccccccccccbabM',
    0xde6b: 'aabaaa',
    0xde6c: 'cccccccccccbabK',
    0xde6d: 'cccccccccccbabJ',
    0xde6e: 'ccca198',
    0xde6f: 'ccabdc',
    0xde70: 'bbbb098',
    0xde71: 'bbbbdc',
    0xde72: 'aaaa198',
    0xde73: 'aaaaaa',
    0xde74: 'cccccccccccbabS',
    0xde75: 'cccccccccccbabR',
    0xde76: 'cccccccccccbabQ',
    0xde77: 'ccbbdc',
    0xde78: 'cccccccccccbahZ',
    0xde79: 'bbaaaa',
    0xde7a: 'cccccccccccbahX',
    0xde7b: 'aabbdc',
    0xde7c: 'aaaaaabA8',
    0xde7d: 'aaaaaab1',
    0xde7e: 'aaaaaab108',
    0xde7f: 'bbbbbbabi',
    0xde80: 'bbba089',
    0xde81: 'bbbabc',
    0xde82: 'aaab189',
    0xde83: 'aaabab',
    0xde84: 'ddda189',
    0xde85: 'ddbbab',
    0xde86: 'eeeeeeb008',
    0xde87: 'ccbabc',
    0xde88: 'fffb089',
    0xde89: 'bbabab',
    0xde8a: 'eeea189',
    0xde8b: 'aababc',
    0xde8c: 'hhha199',
    0xde8d: 'ddaabc',
    0xde8e: 'ccca089',
    0xde8f: 'ccabab',
    0xde90: 'bbbb189',
    0xde91: 'bbbbab',
    0xde92: 'aaaa089',
    0xde93: 'aaaabc',
    0xde94: 'dddb089',
    0xde95: 'ddbabc',
    0xde96: 'ggga189',
    0xde97: 'ccbbab',
    0xde98: 'fffa189',
    0xde99: 'bbaabc',
    0xde9a: 'eeeb089',
    0xde9b: 'aabbab',
    0xde9c: 'hhhb099',
    0xde9d: 'ddabab',
    0xde9e: 'cccb189',
    0xde9f: 'ccaabc',
    0xdea0: 'bbba189',
    0xdea1: 'bbbaab',
    0xdea2: 'aaab089',
    0xdea3: 'aaabbc',
    0xdea4: 'ddda089',
    0xdea5: 'ddbbbc',
    0xdea6: 'gggb189',
    0xdea7: 'ccbaab',
    0xdea8: 'fffb189',
    0xdea9: 'bbabbc',
    0xdeaa: 'eeea089',
    0xdeab: 'aabaab',
    0xdeac: 'hhha099',
    0xdead: 'ddaaab',
    0xdeae: 'ccca189',
    0xdeaf: 'ccabbc',
    0xdeb0: 'bbbb089',
    0xdeb1: 'bbbbbc',
    0xdeb2: 'aaaa189',
    0xdeb3: 'aaaaab',
    0xdeb4: 'dddb189',
    0xdeb5: 'ddbaab',
    0xdeb6: 'ggga089',
    0xdeb7: 'ccbbbc',
    0xdeb8: 'fffa089',
    0xdeb9: 'bbaaab',
    0xdeba: 'dddddabA8',
    0xdebb: 'aabbbc',
    0xdebc: 'aaaaabbA8',
    0xdebd: 'aaaaabbah',
    0xdebe: 'bbbbbaaB9',
    0xdebf: 'bbbbbaabi',
    0xdec0: 'bbba099',
    0xdec1: 'bbbabb',
    0xdec2: 'aaab199',
    0xdec3: 'aaabac',
    0xdec4: 'ddda199',
    0xdec5: 'ddbbac',
    0xdec6: 'fffffabA8',
    0xdec7: 'ccbabb',
    0xdec8: 'fffb099',
    0xdec9: 'bbabac',
    0xdeca: 'eeea199',
    0xdecb: 'aababb',
    0xdecc: 'hhha189',
    0xdecd: 'ddaabb',
    0xdece: 'ccca099',
    0xdecf: 'ccabac',
    0xded0: 'bbbb199',
    0xded1: 'bbbbac',
    0xded2: 'aaaa099',
    0xded3: 'aaaabb',
    0xded4: 'dddb099',
    0xded5: 'ddbabb',
    0xded6: 'ggga199',
    0xded7: 'ccbbac',
    0xded8: 'fffa199',
    0xded9: 'bbaabb',
    0xdeda: 'eeeb099',
    0xdedb: 'aabbac',
    0xdedc: 'hhhb089',
    0xdedd: 'ddabac',
    0xdede: 'cccb199',
    0xdedf: 'ccaabb',
    0xdee0: 'bbba199',
    0xdee1: 'bbbaac',
    0xdee2: 'aaab099',
    0xdee3: 'aaabbb',
    0xdee4: 'ddda099',
    0xdee5: 'ddbbbb',
    0xdee6: 'gggb199',
    0xdee7: 'ccbaac',
    0xdee8: 'fffb199',
    0xdee9: 'bbabbb',
    0xdeea: 'eeea099',
    0xdeeb: 'aabaac',
    0xdeec: 'hhha089',
    0xdeed: 'ddaaac',
    0xdeee: 'ccca199',
    0xdeef: 'ccabbb',
    0xdef0: 'bbbb099',
    0xdef1: 'bbbbbb',
    0xdef2: 'aaaa199',
    0xdef3: 'aaaaac',
    0xdef4: 'dddb199',
    0xdef5: 'ddbaac',
    0xdef6: 'ggga099',
    0xdef7: 'ccbbbb',
    0xdef8: 'aaaaaaababC',
    0xdef9: 'aaaaaaababS',
    0xdefa: 'aaaaaaababc',
    0xdefb: 'aabbbb',
    0xdefc: 'aaaaaaaB9',
    0xdefd: 'aaaaaaabi',
    0xdefe: 'aaaaaaa008',
    0xdeff: 'bbbbbbbah',
    0xdf00: 'bbba888',
    0xdf01: 'bbbabe',
    0xdf02: 'aaab988',
    0xdf03: 'aaabad',
    0xdf04: 'cccccccbabC',
    0xdf05: 'cccccccbabS',
    0xdf06: 'cccccccbabc',
    0xdf07: 'ccbabe',
    0xdf08: 'fffb888',
    0xdf09: 'bbabad',
    0xdf0a: 'eeea988',
    0xdf0b: 'aababe',
    0xdf0c: 'hhha998',
    0xdf0d: 'ddaabe',
    0xdf0e: 'ccca888',
    0xdf0f: 'ccabad',
    0xdf10: 'bbbb988',
    0xdf11: 'bbbbad',
    0xdf12: 'aaaa888',
    0xdf13: 'aaaabe',
    0xdf14: 'dddb888',
    0xdf15: 'ddbabe',
    0xdf16: 'ggga988',
    0xdf17: 'ccbbad',
    0xdf18: 'fffa988',
    0xdf19: 'bbaabe',
    0xdf1a: 'eeeb888',
    0xdf1b: 'aabbad',
    0xdf1c: 'hhhb898',
    0xdf1d: 'ddabad',
    0xdf1e: 'cccb988',
    0xdf1f: 'ccaabe',
    0xdf20: 'bbba988',
    0xdf21: 'bbbaad',
    0xdf22: 'aaab888',
    0xdf23: 'aaabbe',
    0xdf24: 'ddda888',
    0xdf25: 'ddbbbe',
    0xdf26: 'gggb988',
    0xdf27: 'ccbaad',
    0xdf28: 'fffb988',
    0xdf29: 'bbabbe',
    0xdf2a: 'eeea888',
    0xdf2b: 'aabaad',
    0xdf2c: 'hhha898',
    0xdf2d: 'ddaaad',
    0xdf2e: 'ccca988',
    0xdf2f: 'ccabbe',
    0xdf30: 'bbbb888',
    0xdf31: 'bbbbbe',
    0xdf32: 'aaaa988',
    0xdf33: 'aaaaad',
    0xdf34: 'dddb988',
    0xdf35: 'ddbaad',
    0xdf36: 'ggga888',
    0xdf37: 'ccbbbe',
    0xdf38: 'fffa888',
    0xdf39: 'bbaaad',
    0xdf3a: 'dddddaaA8',
    0xdf3b: 'aabbbe',
    0xdf3c: 'aaaaabaA8',
    0xdf3d: 'aaaaabaah',
    0xdf3e: 'bbbbbabB9',
    0xdf3f: 'bbbbbabbi',
    0xdf40: 'bbba898',
    0xdf41: 'bbbabd',
    0xdf42: 'aaab998',
    0xdf43: 'aaabae',
    0xdf44: 'ddda998',
    0xdf45: 'ddbbae',
    0xdf46: 'fffffaaA8',
    0xdf47: 'ccbabd',
    0xdf48: 'fffb898',
    0xdf49: 'bbabae',
    0xdf4a: 'eeea998',
    0xdf4b: 'aababd',
    0xdf4c: 'hhha988',
    0xdf4d: 'ddaabd',
    0xdf4e: 'ccca898',
    0xdf4f: 'ccabae',
    0xdf50: 'bbbb998',
    0xdf51: 'bbbbae',
    0xdf52: 'aaaa898',
    0xdf53: 'aaaabd',
    0xdf54: 'dddb898',
    0xdf55: 'ddbabd',
    0xdf56: 'ggga998',
    0xdf57: 'ccbbae',
    0xdf58: 'fffa998',
    0xdf59: 'bbaabd',
    0xdf5a: 'eeeb898',
    0xdf5b: 'aabbae',
    0xdf5c: 'hhhb888',
    0xdf5d: 'ddabae',
    0xdf5e: 'cccb998',
    0xdf5f: 'ccaabd',
    0xdf60: 'bbba998',
    0xdf61: 'bbbaae',
    0xdf62: 'aaab898',
    0xdf63: 'aaabbd',
    0xdf64: 'ddda898',
    0xdf65: 'ddbbbd',
    0xdf66: 'gggb998',
    0xdf67: 'ccbaae',
    0xdf68: 'fffb998',
    0xdf69: 'bbabbd',
    0xdf6a: 'eeea898',
    0xdf6b: 'aabaae',
    0xdf6c: 'hhha888',
    0xdf6d: 'ddaaae',
    0xdf6e: 'ccca998',
    0xdf6f: 'ccabbd',
    0xdf70: 'bbbb898',
    0xdf71: 'bbbbbd',
    0xdf72: 'aaaa998',
    0xdf73: 'aaaaae',
    0xdf74: 'dddb998',
    0xdf75: 'ddbaae',
    0xdf76: 'ggga898',
    0xdf77: 'ccbbbd',
    0xdf78: 'dddddda008',
    0xdf79: 'bbaaae',
    0xdf7a: 'dddddbbB9',
    0xdf7b: 'aabbbd',
    0xdf7c: 'aaaaaabB9',
    0xdf7d: 'aaaaaabbi',
    0xdf7e: 'aaaaaab008',
    0xdf7f: 'bbbbbbaah',
    0xdf80: 'bbba889',
    0xdf81: 'bbbabg',
    0xdf82: 'aaab989',
    0xdf83: 'aaabaf',
    0xdf84: 'ddda989',
    0xdf85: 'ddbbaf',
    0xdf86: 'eeeeeeb108',
    0xdf87: 'ccbabg',
    0xdf88: 'fffb889',
    0xdf89: 'bbabaf',
    0xdf8a: 'eeea989',
    0xdf8b: 'aababg',
    0xdf8c: 'hhha999',
    0xdf8d: 'ddaabg',
    0xdf8e: 'ccca889',
    0xdf8f: 'ccabaf',
    0xdf90: 'bbbb989',
    0xdf91: 'bbbbaf',
    0xdf92: 'aaaa889',
    0xdf93: 'aaaabg',
    0xdf94: 'dddb889',
    0xdf95: 'ddbabg',
    0xdf96: 'ggga989',
    0xdf97: 'ccbbaf',
    0xdf98: 'fffa989',
    0xdf99: 'bbaabg',
    0xdf9a: 'eeeb889',
    0xdf9b: 'aabbaf',
    0xdf9c: 'hhhb899',
    0xdf9d: 'ddabaf',
    0xdf9e: 'cccb989',
    0xdf9f: 'ccaabg',
    0xdfa0: 'bbba989',
    0xdfa1: 'bbbaaf',
    0xdfa2: 'aaab889',
    0xdfa3: 'aaabbg',
    0xdfa4: 'ddda889',
    0xdfa5: 'ddbbbg',
    0xdfa6: 'gggb989',
    0xdfa7: 'ccbaaf',
    0xdfa8: 'fffb989',
    0xdfa9: 'bbabbg',
    0xdfaa: 'eeea889',
    0xdfab: 'aabaaf',
    0xdfac: 'hhha899',
    0xdfad: 'ddaaaf',
    0xdfae: 'ccca989',
    0xdfaf: 'ccabbg',
    0xdfb0: 'bbbb889',
    0xdfb1: 'bbbbbg',
    0xdfb2: 'aaaa989',
    0xdfb3: 'aaaaaf',
    0xdfb4: 'dddb989',
    0xdfb5: 'ddbaaf',
    0xdfb6: 'ggga889',
    0xdfb7: 'ccbbbg',
    0xdfb8: 'fffa889',
    0xdfb9: 'bbaaaf',
    0xdfba: 'dddddabB9',
    0xdfbb: 'aabbbg',
    0xdfbc: 'aaaaabbB9',
    0xdfbd: 'aaaaabbbi',
    0xdfbe: 'bbbbbaaA8',
    0xdfbf: 'bbbbbaaah',
    0xdfc0: 'bbba899',
    0xdfc1: 'bbbabf',
    0xdfc2: 'aaab999',
    0xdfc3: 'aaabag',
    0xdfc4: 'ddda999',
    0xdfc5: 'ddbbag',
    0xdfc6: 'fffffabB9',
    0xdfc7: 'ccbabf',
    0xdfc8: 'fffb899',
    0xdfc9: 'bbabag',
    0xdfca: 'eeea999',
    0xdfcb: 'aababf',
    0xdfcc: 'hhha989',
    0xdfcd: 'ddaabf',
    0xdfce: 'ccca899',
    0xdfcf: 'ccabag',
    0xdfd0: 'bbbb999',
    0xdfd1: 'bbbbag',
    0xdfd2: 'aaaa899',
    0xdfd3: 'aaaabf',
    0xdfd4: 'dddb899',
    0xdfd5: 'ddbabf',
    0xdfd6: 'ggga999',
    0xdfd7: 'ccbbag',
    0xdfd8: 'fffa999',
    0xdfd9: 'bbaabf',
    0xdfda: 'eeeb899',
    0xdfdb: 'aabbag',
    0xdfdc: 'hhhb889',
    0xdfdd: 'ddabag',
    0xdfde: 'cccb999',
    0xdfdf: 'ccaabf',
    0xdfe0: 'bbba999',
    0xdfe1: 'bbbaag',
    0xdfe2: 'aaab899',
    0xdfe3: 'aaabbf',
    0xdfe4: 'ddda899',
    0xdfe5: 'ddbbbf',
    0xdfe6: 'gggb999',
    0xdfe7: 'ccbaag',
    0xdfe8: 'fffb999',
    0xdfe9: 'bbabbf',
    0xdfea: 'eeea899',
    0xdfeb: 'aabaag',
    0xdfec: 'hhha889',
    0xdfed: 'ddaaag',
    0xdfee: 'ccca999',
    0xdfef: 'ccabbf',
    0xdff0: 'aaaaaaaababh',
    0xdff1: 'aaaaaaaabadc',
    0xdff2: 'aaaa999',
    0xdff3: 'aaaaag',
    0xdff4: 'aaaaaaaababH',
    0xdff5: 'aaaaaaaabadC',
    0xdff6: 'aaaaaaaababX',
    0xdff7: 'aaaaaaaababP',
    0xdff8: 'aaaaaaaabaB',
    0xdff9: 'aaaaaaaabaR',
    0xdffa: 'aaaaaaaabab',
    0xdffb: 'aabbbf',
    0xdffc: 'aaaaaaaA8',
    0xdffd: 'aaaaaaaah',
    0xdffe: 'aaaaaaa108',
    0xdfff: 'aaaaaaaaB5',
    0xe000: 'bbbabix',
    0xe001: 'cccca002',
    0xe002: 'aaabahx',
    0xe003: 'ccccccabD9',
    0xe004: 'cccccccaaaE',
    0xe005: 'cccccccaaaU',
    0xe006: 'cccccccaaae',
    0xe007: 'cccccccaaau',
    0xe008: 'ccccccccbbbK',
    0xe009: 'ccccccccbbbC',
    0xe00a: 'ccccccccbbdX',
    0xe00b: 'ccccccccbbbS',
    0xe00c: 'ccccccccbbbk',
    0xe00d: 'ccccccccbbbc',
    0xe00e: 'cccabix',
    0xe00f: 'ccccccccbbbs',
    0xe010: 'bbbbahx',
    0xe011: 'kkkkb012',
    0xe012: 'aaaabix',
    0xe013: 'hhhha112',
    0xe014: 'dddbbix',
    0xe015: 'mmmmb112',
    0xe016: 'gggaahx',
    0xe017: 'nnnna012',
    0xe018: 'fffaahx',
    0xe019: 'ggggb002',
    0xe01a: 'eeebbix',
    0xe01b: 'dddda102',
    0xe01c: 'hhhbbhx',
    0xe01d: 'aaaab102',
    0xe01e: 'cccbahx',
    0xe01f: 'bbbba002',
    0xe020: 'bbbaahx',
    0xe021: 'ccccb102',
    0xe022: 'aaabbix',
    0xe023: 'ppppb013',
    0xe024: 'dddabix',
    0xe025: 'eeeeb002',
    0xe026: 'gggbahx',
    0xe027: 'ffffa102',
    0xe028: 'fffbahx',
    0xe029: 'oooob112',
    0xe02a: 'eeeabix',
    0xe02b: 'lllla012',
    0xe02c: 'hhhabhx',
    0xe02d: 'iiiib012',
    0xe02e: 'cccaahx',
    0xe02f: 'jjjja112',
    0xe030: 'bbbbbix',
    0xe031: 'kkkka112',
    0xe032: 'aaaaahx',
    0xe033: 'hhhhb012',
    0xe034: 'dddbahx',
    0xe035: 'mmmma012',
    0xe036: 'gggabix',
    0xe037: 'nnnnb112',
    0xe038: 'fffabix',
    0xe039: 'gggga102',
    0xe03a: 'eeebahx',
    0xe03b: 'ddddb002',
    0xe03c: 'hhhbaix',
    0xe03d: 'aaaaa002',
    0xe03e: 'cccbbix',
    0xe03f: 'bbbbb102',
    0xe040: 'bbbabhx',
    0xe041: 'cccca102',
    0xe042: 'aaabaix',
    0xe043: 'ppppa013',
    0xe044: 'dddaaix',
    0xe045: 'eeeea002',
    0xe046: 'gggbbhx',
    0xe047: 'ffffb102',
    0xe048: 'fffbbhx',
    0xe049: 'ooooa112',
    0xe04a: 'eeeaaix',
    0xe04b: 'llllb012',
    0xe04c: 'hhhaahx',
    0xe04d: 'iiiia012',
    0xe04e: 'cccabhx',
    0xe04f: 'jjjjb112',
    0xe050: 'bbbbaix',
    0xe051: 'kkkkb112',
    0xe052: 'aaaabhx',
    0xe053: 'hhhha012',
    0xe054: 'dddbbhx',
    0xe055: 'mmmmb012',
    0xe056: 'gggaaix',
    0xe057: 'nnnna112',
    0xe058: 'fffaaix',
    0xe059: 'ggggb102',
    0xe05a: 'eeebbhx',
    0xe05b: 'dddda002',
    0xe05c: 'hhhbbix',
    0xe05d: 'aaaab002',
    0xe05e: 'cccbaix',
    0xe05f: 'bbbba102',
    0xe060: 'bbbaaix',
    0xe061: 'ccccb002',
    0xe062: 'aaabbhx',
    0xe063: 'ppppb113',
    0xe064: 'dddabhx',
    0xe065: 'eeeeb102',
    0xe066: 'gggbaix',
    0xe067: 'ffffa002',
    0xe068: 'fffbaix',
    0xe069: 'oooob012',
    0xe06a: 'eeeabhx',
    0xe06b: 'lllla112',
    0xe06c: 'hhhabix',
    0xe06d: 'iiiib112',
    0xe06e: 'cccaaix',
    0xe06f: 'jjjja012',
    0xe070: 'bbbbbhx',
    0xe071: 'hhhhhhaaD9',
    0xe072: 'aaaaaix',
    0xe073: 'hhhhb112',
    0xe074: 'dddbaix',
    0xe075: 'mmmma112',
    0xe076: 'gggabhx',
    0xe077: 'mmmmmmbaD9',
    0xe078: 'dddddda117',
    0xe079: 'ddddddaaE9',
    0xe07a: 'ddddddaa7',
    0xe07b: 'ddddb102',
    0xe07c: 'aaaaaaba7',
    0xe07d: 'aaaaa102',
    0xe07e: 'aaaaaab117',
    0xe07f: 'aaaaaabaE9',
    0xe080: 'bbbabiy',
    0xe081: 'cccca012',
    0xe082: 'aaabahy',
    0xe083: 'ccccccbaE9',
    0xe084: 'dddaahy',
    0xe085: 'eeeea112',
    0xe086: 'eeeeeeb017',
    0xe087: 'eeeeeebbD9',
    0xe088: 'fffbbiy',
    0xe089: 'llllllabE9',
    0xe08a: 'eeeaahy',
    0xe08b: 'llllb102',
    0xe08c: 'hhhaaiy',
    0xe08d: 'iiiia102',
    0xe08e: 'cccabiy',
    0xe08f: 'iiiiiibbE9',
    0xe090: 'bbbbahy',
    0xe091: 'kkkkb002',
    0xe092: 'aaaabiy',
    0xe093: 'hhhha102',
    0xe094: 'dddbbiy',
    0xe095: 'mmmmb102',
    0xe096: 'gggaahy',
    0xe097: 'nnnna002',
    0xe098: 'fffaahy',
    0xe099: 'ggggb012',
    0xe09a: 'eeebbiy',
    0xe09b: 'dddda112',
    0xe09c: 'hhhbbhy',
    0xe09d: 'aaaab112',
    0xe09e: 'cccbahy',
    0xe09f: 'bbbba012',
    0xe0a0: 'bbbaahy',
    0xe0a1: 'ccccb112',
    0xe0a2: 'aaabbiy',
    0xe0a3: 'ppppb003',
    0xe0a4: 'dddabiy',
    0xe0a5: 'eeeeb012',
    0xe0a6: 'gggbahy',
    0xe0a7: 'ffffa112',
    0xe0a8: 'fffbahy',
    0xe0a9: 'oooob102',
    0xe0aa: 'eeeabiy',
    0xe0ab: 'lllla002',
    0xe0ac: 'hhhabhy',
    0xe0ad: 'iiiib002',
    0xe0ae: 'cccaahy',
    0xe0af: 'jjjja102',
    0xe0b0: 'bbbbbiy',
    0xe0b1: 'kkkka102',
    0xe0b2: 'aaaaahy',
    0xe0b3: 'hhhhb002',
    0xe0b4: 'dddbahy',
    0xe0b5: 'mmmma002',
    0xe0b6: 'gggabiy',
    0xe0b7: 'nnnnb102',
    0xe0b8: 'fffabiy',
    0xe0b9: 'gggga112',
    0xe0ba: 'eeebahy',
    0xe0bb: 'ddddb012',
    0xe0bc: 'hhhbaiy',
    0xe0bd: 'aaaaa012',
    0xe0be: 'cccbbiy',
    0xe0bf: 'bbbbb112',
    0xe0c0: 'bbbabhy',
    0xe0c1: 'cccca112',
    0xe0c2: 'aaabaiy',
    0xe0c3: 'ppppa003',
    0xe0c4: 'dddaaiy',
    0xe0c5: 'eeeea012',
    0xe0c6: 'gggbbhy',
    0xe0c7: 'ffffb112',
    0xe0c8: 'fffbbhy',
    0xe0c9: 'ooooa102',
    0xe0ca: 'eeeaaiy',
    0xe0cb: 'llllb002',
    0xe0cc: 'hhhaahy',
    0xe0cd: 'iiiia002',
    0xe0ce: 'cccabhy',
    0xe0cf: 'jjjjb102',
    0xe0d0: 'bbbbaiy',
    0xe0d1: 'kkkkb102',
    0xe0d2: 'aaaabhy',
    0xe0d3: 'hhhha002',
    0xe0d4: 'dddbbhy',
    0xe0d5: 'mmmmb002',
    0xe0d6: 'gggaaiy',
    0xe0d7: 'nnnna102',
    0xe0d8: 'fffaaiy',
    0xe0d9: 'ggggb112',
    0xe0da: 'eeebbhy',
    0xe0db: 'dddda012',
    0xe0dc: 'hhhbbiy',
    0xe0dd: 'aaaab012',
    0xe0de: 'cccbaiy',
    0xe0df: 'bbbba112',
    0xe0e0: 'bbbaaiy',
    0xe0e1: 'ccccb012',
    0xe0e2: 'aaabbhy',
    0xe0e3: 'ppppb103',
    0xe0e4: 'dddabhy',
    0xe0e5: 'eeeeb112',
    0xe0e6: 'gggbaiy',
    0xe0e7: 'ffffa012',
    0xe0e8: 'fffbaiy',
    0xe0e9: 'oooob002',
    0xe0ea: 'eeeabhy',
    0xe0eb: 'lllla102',
    0xe0ec: 'hhhabiy',
    0xe0ed: 'iiiib102',
    0xe0ee: 'cccaaiy',
    0xe0ef: 'jjjja002',
    0xe0f0: 'bbbbbhy',
    0xe0f1: 'hhhhhhbbE9',
    0xe0f2: 'aaaaaiy',
    0xe0f3: 'hhhhb102',
    0xe0f4: 'dddbaiy',
    0xe0f5: 'hhhhhhhabau',
    0xe0f6: 'gggabhy',
    0xe0f7: 'hhhhhhhabaU',
    0xe0f8: 'aaaaaaabbbD',
    0xe0f9: 'aaaaaaabbbT',
    0xe0fa: 'aaaaaaabbbd',
    0xe0fb: 'aaaaaaabbbt',
    0xe0fc: 'aaaaaaab6',
    0xe0fd: 'aaaaa112',
    0xe0fe: 'aaaaaaa017',
    0xe0ff: 'aaaaaaabD9',
    0xe100: 'bbbabiz',
    0xe101: 'cccca003',
    0xe102: 'aaabahz',
    0xe103: 'ccccccaaE9',
    0xe104: 'cccccccbbbD',
    0xe105: 'cccccccbbbT',
    0xe106: 'cccccccbbbd',
    0xe107: 'cccccccbbbt',
    0xe108: 'fffbbiz',
    0xe109: 'iiiiiiibabT',
    0xe10a: 'eeeaahz',
    0xe10b: 'iiiiiiibabt',
    0xe10c: 'hhhaaiz',
    0xe10d: 'iiiia113',
    0xe10e: 'cccabiz',
    0xe10f: 'iiiiiiabE9',
    0xe110: 'bbbbahz',
    0xe111: 'kkkkb013',
    0xe112: 'aaaabiz',
    0xe113: 'hhhha113',
    0xe114: 'dddbbiz',
    0xe115: 'mmmmb113',
    0xe116: 'gggaahz',
    0xe117: 'nnnna013',
    0xe118: 'fffaahz',
    0xe119: 'ggggb003',
    0xe11a: 'eeebbiz',
    0xe11b: 'dddda103',
    0xe11c: 'hhhbbhz',
    0xe11d: 'aaaab103',
    0xe11e: 'cccbahz',
    0xe11f: 'bbbba003',
    0xe120: 'bbbaahz',
    0xe121: 'ccccb103',
    0xe122: 'aaabbiz',
    0xe123: 'ppppb012',
    0xe124: 'dddabiz',
    0xe125: 'eeeeb003',
    0xe126: 'gggbahz',
    0xe127: 'ffffa103',
    0xe128: 'fffbahz',
    0xe129: 'oooob113',
    0xe12a: 'eeeabiz',
    0xe12b: 'lllla013',
    0xe12c: 'hhhabhz',
    0xe12d: 'iiiib013',
    0xe12e: 'cccaahz',
    0xe12f: 'jjjja113',
    0xe130: 'bbbbbiz',
    0xe131: 'kkkka113',
    0xe132: 'aaaaahz',
    0xe133: 'hhhhb013',
    0xe134: 'dddbahz',
    0xe135: 'mmmma013',
    0xe136: 'gggabiz',
    0xe137: 'nnnnb113',
    0xe138: 'fffabiz',
    0xe139: 'gggga103',
    0xe13a: 'eeebahz',
    0xe13b: 'ddddb003',
    0xe13c: 'hhhbaiz',
    0xe13d: 'aaaaa003',
    0xe13e: 'cccbbiz',
    0xe13f: 'bbbbb103',
    0xe140: 'bbbabhz',
    0xe141: 'cccca103',
    0xe142: 'aaabaiz',
    0xe143: 'ppppa012',
    0xe144: 'dddaaiz',
    0xe145: 'eeeea003',
    0xe146: 'gggbbhz',
    0xe147: 'ffffb103',
    0xe148: 'fffbbhz',
    0xe149: 'ooooa113',
    0xe14a: 'eeeaaiz',
    0xe14b: 'llllb013',
    0xe14c: 'hhhaahz',
    0xe14d: 'iiiia013',
    0xe14e: 'cccabhz',
    0xe14f: 'jjjjb113',
    0xe150: 'bbbbaiz',
    0xe151: 'kkkkb113',
    0xe152: 'aaaabhz',
    0xe153: 'hhhha013',
    0xe154: 'dddbbhz',
    0xe155: 'mmmmb013',
    0xe156: 'gggaaiz',
    0xe157: 'nnnna113',
    0xe158: 'fffaaiz',
    0xe159: 'ggggb103',
    0xe15a: 'eeebbhz',
    0xe15b: 'dddda003',
    0xe15c: 'hhhbbiz',
    0xe15d: 'aaaab003',
    0xe15e: 'cccbaiz',
    0xe15f: 'bbbba103',
    0xe160: 'bbbaaiz',
    0xe161: 'ccccb003',
    0xe162: 'aaabbhz',
    0xe163: 'ppppb112',
    0xe164: 'dddabhz',
    0xe165: 'eeeeb103',
    0xe166: 'gggbaiz',
    0xe167: 'ffffa003',
    0xe168: 'fffbaiz',
    0xe169: 'oooob013',
    0xe16a: 'eeeabhz',
    0xe16b: 'lllla113',
    0xe16c: 'hhhabiz',
    0xe16d: 'iiiib113',
    0xe16e: 'cccaaiz',
    0xe16f: 'jjjja013',
    0xe170: 'bbbbbhz',
    0xe171: 'hhhhhhabE9',
    0xe172: 'aaaaaiz',
    0xe173: 'hhhhb113',
    0xe174: 'dddbaiz',
    0xe175: 'mmmma113',
    0xe176: 'gggabhz',
    0xe177: 'mmmmmmbbE9',
    0xe178: 'dddddda017',
    0xe179: 'ddddddabD9',
    0xe17a: 'ddddddab6',
    0xe17b: 'ddddb103',
    0xe17c: 'aaaaaabb6',
    0xe17d: 'aaaaa103',
    0xe17e: 'aaaaaab017',
    0xe17f: 'aaaaaabbD9',
    0xe180: 'aaaaaaaaaaabbaZ',
    0xe181: 'aaaaaaaaaaabbcZ',
    0xe182: 'aaabajz',
    0xe183: 'aaaaaaaaaaabbaY',
    0xe184: 'aaaaaaaaaaabbiZ',
    0xe185: 'aaaaaaaaaaabbkZ',
    0xe186: 'aaaaaaaaaaabbiX',
    0xe187: 'aaaaaaaaaaabbiY',
    0xe188: 'aaaaaaaaaaabbaR',
    0xe189: 'aaaaaaaaaaabbaS',
    0xe18a: 'aaaaaaaaaaabbaP',
    0xe18b: 'aaaaaaaaaaabbaQ',
    0xe18c: 'aaaaaaaaaaabbaV',
    0xe18d: 'aaaaaaaaaaabbaW',
    0xe18e: 'aaaaaaaaaaabbaT',
    0xe18f: 'aaaaaaaaaaabbaU',
    0xe190: 'aaaaaaaaaaabbaJ',
    0xe191: 'aaaaaaaaaaabbaK',
    0xe192: 'aaaabkz',
    0xe193: 'aaaaaaaaaaabbaI',
    0xe194: 'aaaaaaaaaaabbaN',
    0xe195: 'aaaaaaaaaaabbaO',
    0xe196: 'aaaaaaaaaaabbaL',
    0xe197: 'aaaaaaaaaaabbaM',
    0xe198: 'aaaaaaaaaaabbaB',
    0xe199: 'aaaaaaaaaaabbaC',
    0xe19a: 'aaaaaaaaaaabbcA',
    0xe19b: 'aaaaaaaaaaabbaA',
    0xe19c: 'aaaaaaaaaaabbaF',
    0xe19d: 'aaaab113',
    0xe19e: 'aaaaaaaaaaabbaD',
    0xe19f: 'aaaaaaaaaaabbaE',
    0xe1a0: 'aaaaaaaaaaabbaz',
    0xe1a1: 'aaaaaaaaaaabbcz',
    0xe1a2: 'aaabbkz',
    0xe1a3: 'aaaaaaaaaaabbay',
    0xe1a4: 'aaaaaaaaaaabbiz',
    0xe1a5: 'aaaaaaaaaaabbkz',
    0xe1a6: 'aaaaaaaaaaabbix',
    0xe1a7: 'aaaaaaaaaaabbiy',
    0xe1a8: 'aaaaaaaaaaabbar',
    0xe1a9: 'aaaaaaaaaaabbas',
    0xe1aa: 'aaaaaaaaaaabbap',
    0xe1ab: 'aaaaaaaaaaabbaq',
    0xe1ac: 'aaaaaaaaaaabbav',
    0xe1ad: 'aaaaaaaaaaabbaw',
    0xe1ae: 'aaaaaaaaaaabbat',
    0xe1af: 'aaaaaaaaaaabbau',
    0xe1b0: 'aaaaaaaaaaabbaj',
    0xe1b1: 'aaaaaaaaaaabbak',
    0xe1b2: 'aaaaajz',
    0xe1b3: 'aaaaaaaaaaabbai',
    0xe1b4: 'aaaaaaaaaaabban',
    0xe1b5: 'aaaaaaaaaaabbao',
    0xe1b6: 'aaaaaaaaaaabbal',
    0xe1b7: 'aaaaaaaaaaabbam',
    0xe1b8: 'aaaaaaaaaaabbab',
    0xe1b9: 'aaaaaaaaaaabbac',
    0xe1ba: 'aaaaaaaaaaabbca',
    0xe1bb: 'aaaaaaaaaaabbaa',
    0xe1bc: 'aaaaaaaaaaabbaf',
    0xe1bd: 'aaaaa013',
    0xe1be: 'aaaaaaaaaaabbad',
    0xe1bf: 'aaaaaaaaaaabbae',
    0xe1c0: 'aaaaaaaaaaabbW',
    0xe1c1: 'aaaaaaaaaaabbU',
    0xe1c2: 'aaabakz',
    0xe1c3: 'aaaaaaaaaaabbQ',
    0xe1c4: 'aaaaaaaaaaabhZ',
    0xe1c5: 'aaaaaaaaaaabhX',
    0xe1c6: 'aaaaaaaaaaabdX',
    0xe1c7: 'aaaaaaaaaaabbY',
    0xe1c8: 'aaaaaaaaaaabbG',
    0xe1c9: 'aaaaaaaaaaabbE',
    0xe1ca: 'aaaaaaaaaaabbC',
    0xe1cb: 'aaaaaaaaaaabbA',
    0xe1cc: 'aaaaaaaaaaabbO',
    0xe1cd: 'aaaaaaaaaaabbM',
    0xe1ce: 'aaaaaaaaaaabbK',
    0xe1cf: 'aaaaaaaaaaabbI',
    0xe1d0: 'aaaaaaaaaaabbw',
    0xe1d1: 'aaaaaaaaaaabbu',
    0xe1d2: 'aaaabjz',
    0xe1d3: 'aaaaaaaaaaabbq',
    0xe1d4: 'aaaaaaaaaaabhz',
    0xe1d5: 'aaaaaaaaaaabhx',
    0xe1d6: 'aaaaaaaaaaabdx',
    0xe1d7: 'aaaaaaaaaaabby',
    0xe1d8: 'aaaaaaaaaaabbg',
    0xe1d9: 'aaaaaaaaaaabbe',
    0xe1da: 'aaaaaaaaaaabbc',
    0xe1db: 'aaaaaaaaaaabba',
    0xe1dc: 'aaaaaaaaaaabbo',
    0xe1dd: 'aaaab013',
    0xe1de: 'aaaaaaaaaaabbk',
    0xe1df: 'aaaaaaaaaaabbi',
    0xe1e0: 'aaaaaaaaaaadC',
    0xe1e1: 'aaaaaaaaaaabD',
    0xe1e2: 'aaabbjz',
    0xe1e3: 'aaaaaaaaaaabL',
    0xe1e4: 'aaaaaaaaaaabP',
    0xe1e5: 'aaaaaaaaaaabT',
    0xe1e6: 'aaaaaaaaaaabX',
    0xe1e7: 'aaaaaaaaaaahY',
    0xe1e8: 'aaaaaaaaaaadc',
    0xe1e9: 'aaaaaaaaaaabd',
    0xe1ea: 'aaaaaaaaaaabh',
    0xe1eb: 'aaaaaaaaaaabl',
    0xe1ec: 'aaaaaaaaaaabp',
    0xe1ed: 'aaaaaaaaaaabt',
    0xe1ee: 'aaaaaaaaaaabx',
    0xe1ef: 'aaaaaaaaaaahy',
    0xe1f0: 'aaaaaaaaaaaj',
    0xe1f1: 'aaaaaaaaaaab',
    0xe1f2: 'aaaaakz',
    0xe1f3: 'aaaaaaaaaaar',
    0xe1f4: 'aaaaaaaaaaaJ',
    0xe1f5: 'aaaaaaaaaaaB',
    0xe1f6: 'aaaaaaaaaaaZ',
    0xe1f7: 'aaaaaaaaaaaR',
    0xe1f8: 'aaaaaaaaaaE',
    0xe1f9: 'aaaaaaaaaaU',
    0xe1fa: 'aaaaaaaaaae',
    0xe1fb: 'aaaaaaaaaau',
    0xe1fc: 'aaaaaaaa7',
    0xe1fd: 'aaaaa113',
    0xe1fe: 'aaaaaaa117',
    0xe1ff: 'aaaaaaaaE9',
    0xe200: 'bbbabax',
    0xe201: 'cccca000',
    0xe202: 'aaababy',
    0xe203: 'ccccccabE9',
    0xe204: 'cccccccabbD',
    0xe205: 'cccccccabbT',
    0xe206: 'cccccccabbd',
    0xe207: 'cccccccabbt',
    0xe208: 'ccccccccaaaJ',
    0xe209: 'ccccccccaaaB',
    0xe20a: 'ccccccccaaaZ',
    0xe20b: 'ccccccccaaaR',
    0xe20c: 'ccccccccaaaj',
    0xe20d: 'ccccccccaaab',
    0xe20e: 'cccabax',
    0xe20f: 'ccccccccaaar',
    0xe210: 'bbbbaby',
    0xe211: 'cccccccccbbau',
    0xe212: 'aaaabax',
    0xe213: 'cccccccccbbiy',
    0xe214: 'cccccccccbbaa',
    0xe215: 'cccccccccbbae',
    0xe216: 'cccccccccbbai',
    0xe217: 'cccccccccbbam',
    0xe218: 'cccccccccbbaQ',
    0xe219: 'cccccccccbbaU',
    0xe21a: 'cccccccccbbaY',
    0xe21b: 'cccccccccbbiY',
    0xe21c: 'cccccccccbbaA',
    0xe21d: 'aaaab100',
    0xe21e: 'cccbaby',
    0xe21f: 'bbbba000',
    0xe220: 'bbbaaby',
    0xe221: 'ccccb100',
    0xe222: 'aaabbax',
    0xe223: 'ppppb011',
    0xe224: 'dddabax',
    0xe225: 'eeeeb000',
    0xe226: 'gggbaby',
    0xe227: 'ffffa100',
    0xe228: 'fffbaby',
    0xe229: 'oooob110',
    0xe22a: 'eeeabax',
    0xe22b: 'lllla010',
    0xe22c: 'hhhabby',
    0xe22d: 'iiiib010',
    0xe22e: 'cccaaby',
    0xe22f: 'jjjja110',
    0xe230: 'bbbbbax',
    0xe231: 'kkkka110',
    0xe232: 'aaaaaby',
    0xe233: 'hhhhb010',
    0xe234: 'dddbaby',
    0xe235: 'mmmma010',
    0xe236: 'gggabax',
    0xe237: 'nnnnb110',
    0xe238: 'fffabax',
    0xe239: 'gggga100',
    0xe23a: 'eeebaby',
    0xe23b: 'ddddb000',
    0xe23c: 'hhhbaax',
    0xe23d: 'aaaaa000',
    0xe23e: 'cccbbax',
    0xe23f: 'bbbbb100',
    0xe240: 'bbbabby',
    0xe241: 'cccca100',
    0xe242: 'aaabaax',
    0xe243: 'ppppa011',
    0xe244: 'dddaaax',
    0xe245: 'eeeea000',
    0xe246: 'gggbbby',
    0xe247: 'ffffb100',
    0xe248: 'fffbbby',
    0xe249: 'ooooa110',
    0xe24a: 'eeeaaax',
    0xe24b: 'llllb010',
    0xe24c: 'hhhaaby',
    0xe24d: 'iiiia010',
    0xe24e: 'cccabby',
    0xe24f: 'jjjjb110',
    0xe250: 'bbbbaax',
    0xe251: 'kkkkb110',
    0xe252: 'aaaabby',
    0xe253: 'hhhha010',
    0xe254: 'dddbbby',
    0xe255: 'mmmmb010',
    0xe256: 'gggaaax',
    0xe257: 'nnnna110',
    0xe258: 'fffaaax',
    0xe259: 'ggggb100',
    0xe25a: 'eeebbby',
    0xe25b: 'dddda000',
    0xe25c: 'hhhbbax',
    0xe25d: 'aaaab000',
    0xe25e: 'cccbaax',
    0xe25f: 'bbbba100',
    0xe260: 'bbbaaax',
    0xe261: 'ccccb000',
    0xe262: 'aaabbby',
    0xe263: 'ppppb111',
    0xe264: 'dddabby',
    0xe265: 'eeeeb100',
    0xe266: 'gggbaax',
    0xe267: 'ffffa000',
    0xe268: 'fffbaax',
    0xe269: 'oooob010',
    0xe26a: 'eeeabby',
    0xe26b: 'lllla110',
    0xe26c: 'hhhabax',
    0xe26d: 'iiiib110',
    0xe26e: 'cccaaax',
    0xe26f: 'jjjja010',
    0xe270: 'bbbbbby',
    0xe271: 'hhhhhhaaE9',
    0xe272: 'aaaaaax',
    0xe273: 'hhhhb110',
    0xe274: 'dddbaax',
    0xe275: 'mmmma110',
    0xe276: 'gggabby',
    0xe277: 'mmmmmmbaE9',
    0xe278: 'dddddda107',
    0xe279: 'ddddddaaD9',
    0xe27a: 'ddddddaabJ',
    0xe27b: 'ddddb100',
    0xe27c: 'aaaaaababJ',
    0xe27d: 'aaaaa100',
    0xe27e: 'aaaaaab107',
    0xe27f: 'aaaaaabaD9',
    0xe280: 'bbbabay',
    0xe281: 'cccca010',
    0xe282: 'aaababx',
    0xe283: 'ccccccbaD9',
    0xe284: 'dddaabx',
    0xe285: 'eeeea110',
    0xe286: 'eeeeeeb007',
    0xe287: 'eeeeeebbE9',
    0xe288: 'fffbbay',
    0xe289: 'llllllabD9',
    0xe28a: 'eeeaabx',
    0xe28b: 'llllb100',
    0xe28c: 'hhhaaay',
    0xe28d: 'iiiia100',
    0xe28e: 'cccabay',
    0xe28f: 'iiiiiibbD9',
    0xe290: 'bbbbabx',
    0xe291: 'kkkkb000',
    0xe292: 'aaaabay',
    0xe293: 'hhhha100',
    0xe294: 'dddbbay',
    0xe295: 'mmmmb100',
    0xe296: 'gggaabx',
    0xe297: 'nnnna000',
    0xe298: 'fffaabx',
    0xe299: 'ggggb010',
    0xe29a: 'eeebbay',
    0xe29b: 'dddda110',
    0xe29c: 'hhhbbbx',
    0xe29d: 'aaaab110',
    0xe29e: 'cccbabx',
    0xe29f: 'bbbba010',
    0xe2a0: 'bbbaabx',
    0xe2a1: 'ccccb110',
    0xe2a2: 'aaabbay',
    0xe2a3: 'ppppb001',
    0xe2a4: 'dddabay',
    0xe2a5: 'eeeeb010',
    0xe2a6: 'gggbabx',
    0xe2a7: 'ffffa110',
    0xe2a8: 'fffbabx',
    0xe2a9: 'oooob100',
    0xe2aa: 'eeeabay',
    0xe2ab: 'lllla000',
    0xe2ac: 'hhhabbx',
    0xe2ad: 'iiiib000',
    0xe2ae: 'cccaabx',
    0xe2af: 'jjjja100',
    0xe2b0: 'bbbbbay',
    0xe2b1: 'kkkka100',
    0xe2b2: 'aaaaabx',
    0xe2b3: 'hhhhb000',
    0xe2b4: 'dddbabx',
    0xe2b5: 'mmmma000',
    0xe2b6: 'gggabay',
    0xe2b7: 'nnnnb100',
    0xe2b8: 'fffabay',
    0xe2b9: 'gggga110',
    0xe2ba: 'eeebabx',
    0xe2bb: 'ddddb010',
    0xe2bc: 'hhhbaay',
    0xe2bd: 'aaaaa010',
    0xe2be: 'cccbbay',
    0xe2bf: 'bbbbb110',
    0xe2c0: 'bbbabbx',
    0xe2c1: 'cccca110',
    0xe2c2: 'aaabaay',
    0xe2c3: 'ppppa001',
    0xe2c4: 'dddaaay',
    0xe2c5: 'eeeea010',
    0xe2c6: 'gggbbbx',
    0xe2c7: 'ffffb110',
    0xe2c8: 'fffbbbx',
    0xe2c9: 'ooooa100',
    0xe2ca: 'eeeaaay',
    0xe2cb: 'llllb000',
    0xe2cc: 'hhhaabx',
    0xe2cd: 'iiiia000',
    0xe2ce: 'cccabbx',
    0xe2cf: 'jjjjb100',
    0xe2d0: 'bbbbaay',
    0xe2d1: 'kkkkb100',
    0xe2d2: 'aaaabbx',
    0xe2d3: 'hhhha000',
    0xe2d4: 'dddbbbx',
    0xe2d5: 'mmmmb000',
    0xe2d6: 'gggaaay',
    0xe2d7: 'nnnna100',
    0xe2d8: 'fffaaay',
    0xe2d9: 'ggggb110',
    0xe2da: 'eeebbbx',
    0xe2db: 'dddda010',
    0xe2dc: 'hhhbbay',
    0xe2dd: 'aaaab010',
    0xe2de: 'cccbaay',
    0xe2df: 'bbbba110',
    0xe2e0: 'bbbaaay',
    0xe2e1: 'ccccb010',
    0xe2e2: 'aaabbbx',
    0xe2e3: 'ppppb101',
    0xe2e4: 'dddabbx',
    0xe2e5: 'eeeeb110',
    0xe2e6: 'gggbaay',
    0xe2e7: 'ffffa010',
    0xe2e8: 'fffbaay',
    0xe2e9: 'oooob000',
    0xe2ea: 'eeeabbx',
    0xe2eb: 'lllla100',
    0xe2ec: 'hhhabay',
    0xe2ed: 'iiiib100',
    0xe2ee: 'cccaaay',
    0xe2ef: 'jjjja000',
    0xe2f0: 'bbbbbbx',
    0xe2f1: 'hhhhhhbbD9',
    0xe2f2: 'aaaaaay',
    0xe2f3: 'hhhhb100',
    0xe2f4: 'dddbaay',
    0xe2f5: 'hhhhhhhaabt',
    0xe2f6: 'gggabbx',
    0xe2f7: 'hhhhhhhaabT',
    0xe2f8: 'aaaaaaabaaE',
    0xe2f9: 'aaaaaaabaaU',
    0xe2fa: 'aaaaaaabaae',
    0xe2fb: 'aaaaaaabaau',
    0xe2fc: 'aaaaaaabaK',
    0xe2fd: 'aaaaa110',
    0xe2fe: 'aaaaaaa007',
    0xe2ff: 'aaaaaaabE9',
    0xe300: 'bbbabaz',
    0xe301: 'cccca001',
    0xe302: 'aaabadx',
    0xe303: 'ccccccaaD9',
    0xe304: 'cccccccbaaE',
    0xe305: 'cccccccbaaU',
    0xe306: 'cccccccbaae',
    0xe307: 'cccccccbaau',
    0xe308: 'fffbbaz',
    0xe309: 'iiiiiiibbaU',
    0xe30a: 'eeeaadx',
    0xe30b: 'iiiiiiibbau',
    0xe30c: 'hhhaaaz',
    0xe30d: 'iiiia111',
    0xe30e: 'cccabaz',
    0xe30f: 'iiiiiiabD9',
    0xe310: 'bbbbadx',
    0xe311: 'kkkkb011',
    0xe312: 'aaaabaz',
    0xe313: 'hhhha111',
    0xe314: 'dddbbaz',
    0xe315: 'mmmmb111',
    0xe316: 'gggaadx',
    0xe317: 'nnnna011',
    0xe318: 'fffaadx',
    0xe319: 'ggggb001',
    0xe31a: 'eeebbaz',
    0xe31b: 'dddda101',
    0xe31c: 'hhhbbdx',
    0xe31d: 'aaaab101',
    0xe31e: 'cccbadx',
    0xe31f: 'bbbba001',
    0xe320: 'bbbaadx',
    0xe321: 'ccccb101',
    0xe322: 'aaabbaz',
    0xe323: 'ppppb010',
    0xe324: 'dddabaz',
    0xe325: 'eeeeb001',
    0xe326: 'gggbadx',
    0xe327: 'ffffa101',
    0xe328: 'fffbadx',
    0xe329: 'oooob111',
    0xe32a: 'eeeabaz',
    0xe32b: 'lllla011',
    0xe32c: 'hhhabdx',
    0xe32d: 'iiiib011',
    0xe32e: 'cccaadx',
    0xe32f: 'jjjja111',
    0xe330: 'bbbbbaz',
    0xe331: 'kkkka111',
    0xe332: 'aaaaadx',
    0xe333: 'hhhhb011',
    0xe334: 'dddbadx',
    0xe335: 'mmmma011',
    0xe336: 'gggabaz',
    0xe337: 'nnnnb111',
    0xe338: 'fffabaz',
    0xe339: 'gggga101',
    0xe33a: 'eeebadx',
    0xe33b: 'ddddb001',
    0xe33c: 'hhhbaaz',
    0xe33d: 'aaaaa001',
    0xe33e: 'cccbbaz',
    0xe33f: 'bbbbb101',
    0xe340: 'bbbabdx',
    0xe341: 'cccca101',
    0xe342: 'aaabaaz',
    0xe343: 'ppppa010',
    0xe344: 'dddaaaz',
    0xe345: 'eeeea001',
    0xe346: 'gggbbdx',
    0xe347: 'ffffb101',
    0xe348: 'fffbbdx',
    0xe349: 'ooooa111',
    0xe34a: 'eeeaaaz',
    0xe34b: 'llllb011',
    0xe34c: 'hhhaadx',
    0xe34d: 'iiiia011',
    0xe34e: 'cccabdx',
    0xe34f: 'jjjjb111',
    0xe350: 'bbbbaaz',
    0xe351: 'kkkkb111',
    0xe352: 'aaaabdx',
    0xe353: 'hhhha011',
    0xe354: 'dddbbdx',
    0xe355: 'mmmmb011',
    0xe356: 'gggaaaz',
    0xe357: 'nnnna111',
    0xe358: 'fffaaaz',
    0xe359: 'ggggb101',
    0xe35a: 'eeebbdx',
    0xe35b: 'dddda001',
    0xe35c: 'hhhbbaz',
    0xe35d: 'aaaab001',
    0xe35e: 'cccbaaz',
    0xe35f: 'bbbba101',
    0xe360: 'bbbaaaz',
    0xe361: 'ccccb001',
    0xe362: 'aaabbdx',
    0xe363: 'ppppb110',
    0xe364: 'dddabdx',
    0xe365: 'eeeeb101',
    0xe366: 'gggbaaz',
    0xe367: 'ffffa001',
    0xe368: 'fffbaaz',
    0xe369: 'oooob011',
    0xe36a: 'eeeabdx',
    0xe36b: 'lllla111',
    0xe36c: 'hhhabaz',
    0xe36d: 'iiiib111',
    0xe36e: 'cccaaaz',
    0xe36f: 'jjjja011',
    0xe370: 'bbbbbdx',
    0xe371: 'hhhhhhabD9',
    0xe372: 'aaaaaaz',
    0xe373: 'hhhhb111',
    0xe374: 'dddbaaz',
    0xe375: 'mmmma111',
    0xe376: 'gggabdx',
    0xe377: 'mmmmmmbbD9',
    0xe378: 'dddddda007',
    0xe379: 'ddddddabE9',
    0xe37a: 'ddddddabaK',
    0xe37b: 'ddddb101',
    0xe37c: 'aaaaaabb7',
    0xe37d: 'aaaaa101',
    0xe37e: 'aaaaaab007',
    0xe37f: 'aaaaaabbE9',
    0xe380: 'bbbabcz',
    0xe381: 'cccca011',
    0xe382: 'aaababz',
    0xe383: 'ccccccbbE9',
    0xe384: 'dddaabz',
    0xe385: 'eeeea111',
    0xe386: 'eeeeeeb107',
    0xe387: 'eeeeeebaD9',
    0xe388: 'fffbbcz',
    0xe389: 'llllllaaE9',
    0xe38a: 'eeeaabz',
    0xe38b: 'llllb101',
    0xe38c: 'hhhaacz',
    0xe38d: 'iiiia101',
    0xe38e: 'cccabcz',
    0xe38f: 'iiiiiibaE9',
    0xe390: 'bbbbabz',
    0xe391: 'kkkkb001',
    0xe392: 'aaaabcz',
    0xe393: 'hhhha101',
    0xe394: 'dddbbcz',
    0xe395: 'mmmmb101',
    0xe396: 'gggaabz',
    0xe397: 'nnnna001',
    0xe398: 'fffaabz',
    0xe399: 'ggggb011',
    0xe39a: 'eeebbcz',
    0xe39b: 'dddda111',
    0xe39c: 'hhhbbbz',
    0xe39d: 'aaaab111',
    0xe39e: 'cccbabz',
    0xe39f: 'bbbba011',
    0xe3a0: 'bbbaabz',
    0xe3a1: 'ccccb111',
    0xe3a2: 'aaabbcz',
    0xe3a3: 'ppppb000',
    0xe3a4: 'dddabcz',
    0xe3a5: 'eeeeb011',
    0xe3a6: 'gggbabz',
    0xe3a7: 'ffffa111',
    0xe3a8: 'fffbabz',
    0xe3a9: 'oooob101',
    0xe3aa: 'eeeabcz',
    0xe3ab: 'lllla001',
    0xe3ac: 'hhhabbz',
    0xe3ad: 'iiiib001',
    0xe3ae: 'cccaabz',
    0xe3af: 'jjjja101',
    0xe3b0: 'bbbbbcz',
    0xe3b1: 'kkkka101',
    0xe3b2: 'aaaaabz',
    0xe3b3: 'hhhhb001',
    0xe3b4: 'dddbabz',
    0xe3b5: 'mmmma001',
    0xe3b6: 'gggabcz',
    0xe3b7: 'nnnnb101',
    0xe3b8: 'fffabcz',
    0xe3b9: 'gggga111',
    0xe3ba: 'eeebabz',
    0xe3bb: 'ddddb011',
    0xe3bc: 'hhhbacz',
    0xe3bd: 'aaaaa011',
    0xe3be: 'cccbbcz',
    0xe3bf: 'bbbbb111',
    0xe3c0: 'bbbabbz',
    0xe3c1: 'cccca111',
    0xe3c2: 'aaabacz',
    0xe3c3: 'ppppa000',
    0xe3c4: 'dddaacz',
    0xe3c5: 'eeeea011',
    0xe3c6: 'gggbbbz',
    0xe3c7: 'ffffb111',
    0xe3c8: 'fffbbbz',
    0xe3c9: 'ooooa101',
    0xe3ca: 'eeeaacz',
    0xe3cb: 'llllb001',
    0xe3cc: 'hhhaabz',
    0xe3cd: 'iiiia001',
    0xe3ce: 'cccabbz',
    0xe3cf: 'jjjjb101',
    0xe3d0: 'bbbbacz',
    0xe3d1: 'kkkkb101',
    0xe3d2: 'aaaabbz',
    0xe3d3: 'hhhha001',
    0xe3d4: 'dddbbbz',
    0xe3d5: 'mmmmb001',
    0xe3d6: 'gggaacz',
    0xe3d7: 'nnnna101',
    0xe3d8: 'fffaacz',
    0xe3d9: 'ggggb111',
    0xe3da: 'eeebbbz',
    0xe3db: 'dddda011',
    0xe3dc: 'hhhbbcz',
    0xe3dd: 'aaaab011',
    0xe3de: 'cccbacz',
    0xe3df: 'bbbba111',
    0xe3e0: 'bbbaacz',
    0xe3e1: 'ccccb011',
    0xe3e2: 'aaabbbz',
    0xe3e3: 'ppppb100',
    0xe3e4: 'dddabbz',
    0xe3e5: 'eeeeb111',
    0xe3e6: 'gggbacz',
    0xe3e7: 'ffffa011',
    0xe3e8: 'fffbacz',
    0xe3e9: 'oooob001',
    0xe3ea: 'eeeabbz',
    0xe3eb: 'lllla101',
    0xe3ec: 'hhhabcz',
    0xe3ed: 'iiiib101',
    0xe3ee: 'cccaacz',
    0xe3ef: 'jjjja001',
    0xe3f0: 'aaaaaaaabbbk',
    0xe3f1: 'aaaaaaaabbbc',
    0xe3f2: 'aaaaacz',
    0xe3f3: 'aaaaaaaabbbs',
    0xe3f4: 'aaaaaaaabbbK',
    0xe3f5: 'aaaaaaaabbbC',
    0xe3f6: 'aaaaaaaabbdX',
    0xe3f7: 'aaaaaaaabbbS',
    0xe3f8: 'aaaaaaaabbD',
    0xe3f9: 'aaaaaaaabbT',
    0xe3fa: 'aaaaaaaabbd',
    0xe3fb: 'aaaaaaaabbt',
    0xe3fc: 'aaaaaaaabJ',
    0xe3fd: 'aaaaa111',
    0xe3fe: 'aaaaaaa107',
    0xe3ff: 'aaaaaaaaD9',
    0xe400: 'bbbabat',
    0xe401: 'cccca006',
    0xe402: 'aaababu',
    0xe403: 'ccccccabD8',
    0xe404: 'cccccccaabD',
    0xe405: 'cccccccaabT',
    0xe406: 'cccccccaabd',
    0xe407: 'cccccccaabt',
    0xe408: 'ccccccccbaaJ',
    0xe409: 'ccccccccbaaB',
    0xe40a: 'ccccccccbaaZ',
    0xe40b: 'ccccccccbaaR',
    0xe40c: 'ccccccccbaaj',
    0xe40d: 'ccccccccbaab',
    0xe40e: 'cccabat',
    0xe40f: 'ccccccccbaar',
    0xe410: 'bbbbabu',
    0xe411: 'kkkkb016',
    0xe412: 'aaaabat',
    0xe413: 'hhhha116',
    0xe414: 'dddbbat',
    0xe415: 'mmmmb116',
    0xe416: 'gggaabu',
    0xe417: 'nnnna016',
    0xe418: 'fffaabu',
    0xe419: 'ggggb006',
    0xe41a: 'eeebbat',
    0xe41b: 'dddda106',
    0xe41c: 'hhhbbbu',
    0xe41d: 'aaaab106',
    0xe41e: 'cccbabu',
    0xe41f: 'bbbba006',
    0xe420: 'bbbaabu',
    0xe421: 'ccccb106',
    0xe422: 'aaabbat',
    0xe423: 'ppppb017',
    0xe424: 'dddabat',
    0xe425: 'eeeeb006',
    0xe426: 'gggbabu',
    0xe427: 'ffffa106',
    0xe428: 'fffbabu',
    0xe429: 'oooob116',
    0xe42a: 'eeeabat',
    0xe42b: 'lllla016',
    0xe42c: 'hhhabbu',
    0xe42d: 'iiiib016',
    0xe42e: 'cccaabu',
    0xe42f: 'jjjja116',
    0xe430: 'bbbbbat',
    0xe431: 'kkkka116',
    0xe432: 'aaaaabu',
    0xe433: 'hhhhb016',
    0xe434: 'dddbabu',
    0xe435: 'mmmma016',
    0xe436: 'gggabat',
    0xe437: 'nnnnb116',
    0xe438: 'fffabat',
    0xe439: 'gggga106',
    0xe43a: 'eeebabu',
    0xe43b: 'ddddb006',
    0xe43c: 'hhhbaat',
    0xe43d: 'aaaaa006',
    0xe43e: 'cccbbat',
    0xe43f: 'bbbbb106',
    0xe440: 'bbbabbu',
    0xe441: 'cccca106',
    0xe442: 'aaabaat',
    0xe443: 'ppppa017',
    0xe444: 'dddaaat',
    0xe445: 'eeeea006',
    0xe446: 'gggbbbu',
    0xe447: 'ffffb106',
    0xe448: 'fffbbbu',
    0xe449: 'ooooa116',
    0xe44a: 'eeeaaat',
    0xe44b: 'llllb016',
    0xe44c: 'hhhaabu',
    0xe44d: 'iiiia016',
    0xe44e: 'cccabbu',
    0xe44f: 'jjjjb116',
    0xe450: 'bbbbaat',
    0xe451: 'kkkkb116',
    0xe452: 'aaaabbu',
    0xe453: 'hhhha016',
    0xe454: 'dddbbbu',
    0xe455: 'mmmmb016',
    0xe456: 'gggaaat',
    0xe457: 'nnnna116',
    0xe458: 'fffaaat',
    0xe459: 'ggggb106',
    0xe45a: 'eeebbbu',
    0xe45b: 'dddda006',
    0xe45c: 'hhhbbat',
    0xe45d: 'aaaab006',
    0xe45e: 'cccbaat',
    0xe45f: 'bbbba106',
    0xe460: 'bbbaaat',
    0xe461: 'ccccb006',
    0xe462: 'aaabbbu',
    0xe463: 'ppppb117',
    0xe464: 'dddabbu',
    0xe465: 'eeeeb106',
    0xe466: 'gggbaat',
    0xe467: 'ffffa006',
    0xe468: 'fffbaat',
    0xe469: 'oooob016',
    0xe46a: 'eeeabbu',
    0xe46b: 'lllla116',
    0xe46c: 'hhhabat',
    0xe46d: 'iiiib116',
    0xe46e: 'cccaaat',
    0xe46f: 'jjjja016',
    0xe470: 'bbbbbbu',
    0xe471: 'hhhhhhaaD8',
    0xe472: 'aaaaaat',
    0xe473: 'hhhhb116',
    0xe474: 'dddbaat',
    0xe475: 'mmmma116',
    0xe476: 'gggabbu',
    0xe477: 'mmmmmmbaD8',
    0xe478: 'dddddda116',
    0xe479: 'ddddddaaE8',
    0xe47a: 'ddddddaaaJ',
    0xe47b: 'ddddb106',
    0xe47c: 'aaaaaabaaJ',
    0xe47d: 'aaaaa106',
    0xe47e: 'aaaaaab116',
    0xe47f: 'aaaaaabaE8',
    0xe480: 'bbbabau',
    0xe481: 'cccca016',
    0xe482: 'aaababt',
    0xe483: 'ccccccbaE8',
    0xe484: 'dddaabt',
    0xe485: 'eeeea116',
    0xe486: 'eeeeeeb016',
    0xe487: 'eeeeeebbD8',
    0xe488: 'fffbbau',
    0xe489: 'llllllabE8',
    0xe48a: 'eeeaabt',
    0xe48b: 'llllb106',
    0xe48c: 'hhhaaau',
    0xe48d: 'iiiia106',
    0xe48e: 'cccabau',
    0xe48f: 'iiiiiibbE8',
    0xe490: 'bbbbabt',
    0xe491: 'kkkkb006',
    0xe492: 'aaaabau',
    0xe493: 'hhhha106',
    0xe494: 'dddbbau',
    0xe495: 'mmmmb106',
    0xe496: 'gggaabt',
    0xe497: 'nnnna006',
    0xe498: 'fffaabt',
    0xe499: 'ggggb016',
    0xe49a: 'eeebbau',
    0xe49b: 'dddda116',
    0xe49c: 'hhhbbbt',
    0xe49d: 'aaaab116',
    0xe49e: 'cccbabt',
    0xe49f: 'bbbba016',
    0xe4a0: 'bbbaabt',
    0xe4a1: 'ccccb116',
    0xe4a2: 'aaabbau',
    0xe4a3: 'ppppb007',
    0xe4a4: 'dddabau',
    0xe4a5: 'eeeeb016',
    0xe4a6: 'gggbabt',
    0xe4a7: 'ffffa116',
    0xe4a8: 'fffbabt',
    0xe4a9: 'oooob106',
    0xe4aa: 'eeeabau',
    0xe4ab: 'lllla006',
    0xe4ac: 'hhhabbt',
    0xe4ad: 'iiiib006',
    0xe4ae: 'cccaabt',
    0xe4af: 'jjjja106',
    0xe4b0: 'bbbbbau',
    0xe4b1: 'kkkka106',
    0xe4b2: 'aaaaabt',
    0xe4b3: 'hhhhb006',
    0xe4b4: 'dddbabt',
    0xe4b5: 'mmmma006',
    0xe4b6: 'gggabau',
    0xe4b7: 'nnnnb106',
    0xe4b8: 'fffabau',
    0xe4b9: 'gggga116',
    0xe4ba: 'eeebabt',
    0xe4bb: 'ddddb016',
    0xe4bc: 'hhhbaau',
    0xe4bd: 'aaaaa016',
    0xe4be: 'cccbbau',
    0xe4bf: 'bbbbb116',
    0xe4c0: 'bbbabbt',
    0xe4c1: 'cccca116',
    0xe4c2: 'aaabaau',
    0xe4c3: 'ppppa007',
    0xe4c4: 'dddaaau',
    0xe4c5: 'eeeea016',
    0xe4c6: 'gggbbbt',
    0xe4c7: 'ffffb116',
    0xe4c8: 'fffbbbt',
    0xe4c9: 'ooooa106',
    0xe4ca: 'eeeaaau',
    0xe4cb: 'llllb006',
    0xe4cc: 'hhhaabt',
    0xe4cd: 'iiiia006',
    0xe4ce: 'cccabbt',
    0xe4cf: 'jjjjb106',
    0xe4d0: 'bbbbaau',
    0xe4d1: 'kkkkb106',
    0xe4d2: 'aaaabbt',
    0xe4d3: 'hhhha006',
    0xe4d4: 'dddbbbt',
    0xe4d5: 'mmmmb006',
    0xe4d6: 'gggaaau',
    0xe4d7: 'nnnna106',
    0xe4d8: 'fffaaau',
    0xe4d9: 'ggggb116',
    0xe4da: 'eeebbbt',
    0xe4db: 'dddda016',
    0xe4dc: 'hhhbbau',
    0xe4dd: 'aaaab016',
    0xe4de: 'cccbaau',
    0xe4df: 'bbbba116',
    0xe4e0: 'bbbaaau',
    0xe4e1: 'ccccb016',
    0xe4e2: 'aaabbbt',
    0xe4e3: 'ppppb107',
    0xe4e4: 'dddabbt',
    0xe4e5: 'eeeeb116',
    0xe4e6: 'gggbaau',
    0xe4e7: 'ffffa016',
    0xe4e8: 'fffbaau',
    0xe4e9: 'oooob006',
    0xe4ea: 'eeeabbt',
    0xe4eb: 'lllla106',
    0xe4ec: 'hhhabau',
    0xe4ed: 'iiiib106',
    0xe4ee: 'cccaaau',
    0xe4ef: 'jjjja006',
    0xe4f0: 'bbbbbbt',
    0xe4f1: 'hhhhhhbbE8',
    0xe4f2: 'aaaaaau',
    0xe4f3: 'hhhhb106',
    0xe4f4: 'dddbaau',
    0xe4f5: 'hhhhhhhabbt',
    0xe4f6: 'gggabbt',
    0xe4f7: 'hhhhhhhabbT',
    0xe4f8: 'aaaaaaabbaE',
    0xe4f9: 'aaaaaaabbaU',
    0xe4fa: 'aaaaaaabbae',
    0xe4fb: 'aaaaaaabbau',
    0xe4fc: 'aaaaaaabbK',
    0xe4fd: 'aaaaa116',
    0xe4fe: 'aaaaaaa016',
    0xe4ff: 'aaaaaaabD8',
    0xe500: 'bbbabav',
    0xe501: 'cccca007',
    0xe502: 'aaababw',
    0xe503: 'ccccccaaE8',
    0xe504: 'cccccccbbaE',
    0xe505: 'cccccccbbaU',
    0xe506: 'cccccccbbae',
    0xe507: 'cccccccbbau',
    0xe508: 'fffbbav',
    0xe509: 'iiiiiiibaaU',
    0xe50a: 'eeeaabw',
    0xe50b: 'iiiiiiibaau',
    0xe50c: 'hhhaaav',
    0xe50d: 'iiiia117',
    0xe50e: 'cccabav',
    0xe50f: 'iiiiiiabE8',
    0xe510: 'bbbbabw',
    0xe511: 'kkkkb017',
    0xe512: 'aaaabav',
    0xe513: 'hhhha117',
    0xe514: 'dddbbav',
    0xe515: 'mmmmb117',
    0xe516: 'gggaabw',
    0xe517: 'nnnna017',
    0xe518: 'fffaabw',
    0xe519: 'ggggb007',
    0xe51a: 'eeebbav',
    0xe51b: 'dddda107',
    0xe51c: 'hhhbbbw',
    0xe51d: 'aaaab107',
    0xe51e: 'cccbabw',
    0xe51f: 'bbbba007',
    0xe520: 'bbbaabw',
    0xe521: 'ccccb107',
    0xe522: 'aaabbav',
    0xe523: 'ppppb016',
    0xe524: 'dddabav',
    0xe525: 'eeeeb007',
    0xe526: 'gggbabw',
    0xe527: 'ffffa107',
    0xe528: 'fffbabw',
    0xe529: 'oooob117',
    0xe52a: 'eeeabav',
    0xe52b: 'lllla017',
    0xe52c: 'hhhabbw',
    0xe52d: 'iiiib017',
    0xe52e: 'cccaabw',
    0xe52f: 'jjjja117',
    0xe530: 'bbbbbav',
    0xe531: 'kkkka117',
    0xe532: 'aaaaabw',
    0xe533: 'hhhhb017',
    0xe534: 'dddbabw',
    0xe535: 'mmmma017',
    0xe536: 'gggabav',
    0xe537: 'nnnnb117',
    0xe538: 'fffabav',
    0xe539: 'gggga107',
    0xe53a: 'eeebabw',
    0xe53b: 'ddddb007',
    0xe53c: 'hhhbaav',
    0xe53d: 'aaaaa007',
    0xe53e: 'cccbbav',
    0xe53f: 'bbbbb107',
    0xe540: 'bbbabbw',
    0xe541: 'cccca107',
    0xe542: 'aaabaav',
    0xe543: 'ppppa016',
    0xe544: 'dddaaav',
    0xe545: 'eeeea007',
    0xe546: 'gggbbbw',
    0xe547: 'ffffb107',
    0xe548: 'fffbbbw',
    0xe549: 'ooooa117',
    0xe54a: 'eeeaaav',
    0xe54b: 'llllb017',
    0xe54c: 'hhhaabw',
    0xe54d: 'iiiia017',
    0xe54e: 'cccabbw',
    0xe54f: 'jjjjb117',
    0xe550: 'bbbbaav',
    0xe551: 'kkkkb117',
    0xe552: 'aaaabbw',
    0xe553: 'hhhha017',
    0xe554: 'dddbbbw',
    0xe555: 'mmmmb017',
    0xe556: 'gggaaav',
    0xe557: 'nnnna117',
    0xe558: 'fffaaav',
    0xe559: 'ggggb107',
    0xe55a: 'eeebbbw',
    0xe55b: 'dddda007',
    0xe55c: 'hhhbbav',
    0xe55d: 'aaaab007',
    0xe55e: 'cccbaav',
    0xe55f: 'bbbba107',
    0xe560: 'bbbaaav',
    0xe561: 'ccccb007',
    0xe562: 'aaabbbw',
    0xe563: 'ppppb116',
    0xe564: 'dddabbw',
    0xe565: 'eeeeb107',
    0xe566: 'gggbaav',
    0xe567: 'ffffa007',
    0xe568: 'fffbaav',
    0xe569: 'oooob017',
    0xe56a: 'eeeabbw',
    0xe56b: 'lllla117',
    0xe56c: 'hhhabav',
    0xe56d: 'iiiib117',
    0xe56e: 'cccaaav',
    0xe56f: 'jjjja017',
    0xe570: 'bbbbbbw',
    0xe571: 'hhhhhhabE8',
    0xe572: 'aaaaaav',
    0xe573: 'hhhhb117',
    0xe574: 'dddbaav',
    0xe575: 'mmmma117',
    0xe576: 'gggabbw',
    0xe577: 'mmmmmmbbE8',
    0xe578: 'dddddda016',
    0xe579: 'ddddddabD8',
    0xe57a: 'ddddddabbK',
    0xe57b: 'ddddb107',
    0xe57c: 'aaaaaabbbK',
    0xe57d: 'aaaaa107',
    0xe57e: 'aaaaaab016',
    0xe57f: 'aaaaaabbD8',
    0xe580: 'bbbabaw',
    0xe581: 'cccca017',
    0xe582: 'aaababv',
    0xe583: 'ccccccbbD8',
    0xe584: 'dddaabv',
    0xe585: 'eeeea117',
    0xe586: 'eeeeeeb116',
    0xe587: 'eeeeeebaE8',
    0xe588: 'fffbbaw',
    0xe589: 'llllllaaD8',
    0xe58a: 'eeeaabv',
    0xe58b: 'llllb107',
    0xe58c: 'hhhaaaw',
    0xe58d: 'iiiia107',
    0xe58e: 'cccabaw',
    0xe58f: 'iiiiiibaD8',
    0xe590: 'bbbbabv',
    0xe591: 'kkkkb007',
    0xe592: 'aaaabaw',
    0xe593: 'hhhha107',
    0xe594: 'dddbbaw',
    0xe595: 'mmmmb107',
    0xe596: 'gggaabv',
    0xe597: 'nnnna007',
    0xe598: 'fffaabv',
    0xe599: 'ggggb017',
    0xe59a: 'eeebbaw',
    0xe59b: 'dddda117',
    0xe59c: 'hhhbbbv',
    0xe59d: 'aaaab117',
    0xe59e: 'cccbabv',
    0xe59f: 'bbbba017',
    0xe5a0: 'bbbaabv',
    0xe5a1: 'ccccb117',
    0xe5a2: 'aaabbaw',
    0xe5a3: 'ppppb006',
    0xe5a4: 'dddabaw',
    0xe5a5: 'eeeeb017',
    0xe5a6: 'gggbabv',
    0xe5a7: 'ffffa117',
    0xe5a8: 'fffbabv',
    0xe5a9: 'oooob107',
    0xe5aa: 'eeeabaw',
    0xe5ab: 'lllla007',
    0xe5ac: 'hhhabbv',
    0xe5ad: 'iiiib007',
    0xe5ae: 'cccaabv',
    0xe5af: 'jjjja107',
    0xe5b0: 'bbbbbaw',
    0xe5b1: 'kkkka107',
    0xe5b2: 'aaaaabv',
    0xe5b3: 'hhhhb007',
    0xe5b4: 'dddbabv',
    0xe5b5: 'mmmma007',
    0xe5b6: 'gggabaw',
    0xe5b7: 'nnnnb107',
    0xe5b8: 'fffabaw',
    0xe5b9: 'gggga117',
    0xe5ba: 'eeebabv',
    0xe5bb: 'ddddb017',
    0xe5bc: 'hhhbaaw',
    0xe5bd: 'aaaaa017',
    0xe5be: 'cccbbaw',
    0xe5bf: 'bbbbb117',
    0xe5c0: 'bbbabbv',
    0xe5c1: 'cccca117',
    0xe5c2: 'aaabaaw',
    0xe5c3: 'ppppa006',
    0xe5c4: 'dddaaaw',
    0xe5c5: 'eeeea017',
    0xe5c6: 'gggbbbv',
    0xe5c7: 'ffffb117',
    0xe5c8: 'fffbbbv',
    0xe5c9: 'ooooa107',
    0xe5ca: 'eeeaaaw',
    0xe5cb: 'llllb007',
    0xe5cc: 'hhhaabv',
    0xe5cd: 'iiiia007',
    0xe5ce: 'cccabbv',
    0xe5cf: 'jjjjb107',
    0xe5d0: 'bbbbaaw',
    0xe5d1: 'kkkkb107',
    0xe5d2: 'aaaabbv',
    0xe5d3: 'hhhha007',
    0xe5d4: 'dddbbbv',
    0xe5d5: 'mmmmb007',
    0xe5d6: 'gggaaaw',
    0xe5d7: 'nnnna107',
    0xe5d8: 'fffaaaw',
    0xe5d9: 'ggggb117',
    0xe5da: 'eeebbbv',
    0xe5db: 'dddda017',
    0xe5dc: 'hhhbbaw',
    0xe5dd: 'aaaab017',
    0xe5de: 'cccbaaw',
    0xe5df: 'bbbba117',
    0xe5e0: 'aaaaaaaaabbaA',
    0xe5e1: 'aaaaaaaaabbaE',
    0xe5e2: 'aaabbbv',
    0xe5e3: 'aaaaaaaaabbaM',
    0xe5e4: 'aaaaaaaaabbaQ',
    0xe5e5: 'aaaaaaaaabbaU',
    0xe5e6: 'aaaaaaaaabbaY',
    0xe5e7: 'aaaaaaaaabbiY',
    0xe5e8: 'aaaaaaaaabbaa',
    0xe5e9: 'aaaaaaaaabbae',
    0xe5ea: 'aaaaaaaaabbai',
    0xe5eb: 'aaaaaaaaabbam',
    0xe5ec: 'aaaaaaaaabbaq',
    0xe5ed: 'aaaaaaaaabbau',
    0xe5ee: 'aaaaaaaaabbay',
    0xe5ef: 'aaaaaaaaabbiy',
    0xe5f0: 'aaaaaaaaabbk',
    0xe5f1: 'aaaaaaaaabbc',
    0xe5f2: 'aaaaaaw',
    0xe5f3: 'aaaaaaaaabbs',
    0xe5f4: 'aaaaaaaaabbK',
    0xe5f5: 'aaaaaaaaabbC',
    0xe5f6: 'aaaaaaaaabdX',
    0xe5f7: 'aaaaaaaaabbS',
    0xe5f8: 'aaaaaaaaabD',
    0xe5f9: 'aaaaaaaaabT',
    0xe5fa: 'aaaaaaaaabd',
    0xe5fb: 'aaaaaaaaabt',
    0xe5fc: 'aaaaaaaaaJ',
    0xe5fd: 'aaaaa117',
    0xe5fe: 'aaaaaaa116',
    0xe5ff: 'aaaaaaaaE8',
    0xe600: 'bbbabap',
    0xe601: 'cccca004',
    0xe602: 'aaababq',
    0xe603: 'ccccccabE8',
    0xe604: 'cccccccabaE',
    0xe605: 'cccccccabaU',
    0xe606: 'cccccccabae',
    0xe607: 'cccccccabau',
    0xe608: 'ccccccccabbK',
    0xe609: 'ccccccccabbC',
    0xe60a: 'ccccccccabdX',
    0xe60b: 'ccccccccabbS',
    0xe60c: 'ccccccccabbk',
    0xe60d: 'ccccccccabbc',
    0xe60e: 'cccabap',
    0xe60f: 'ccccccccabbs',
    0xe610: 'bbbbabq',
    0xe611: 'cccccccccaabt',
    0xe612: 'aaaabap',
    0xe613: 'cccccccccaahy',
    0xe614: 'cccccccccaadc',
    0xe615: 'cccccccccaabd',
    0xe616: 'cccccccccaabh',
    0xe617: 'cccccccccaabl',
    0xe618: 'cccccccccaabP',
    0xe619: 'cccccccccaabT',
    0xe61a: 'cccccccccaabX',
    0xe61b: 'cccccccccaahY',
    0xe61c: 'cccccccccaadC',
    0xe61d: 'aaaab104',
    0xe61e: 'cccbabq',
    0xe61f: 'bbbba004',
    0xe620: 'bbbaabq',
    0xe621: 'ccccb104',
    0xe622: 'aaabbap',
    0xe623: 'ccccccccccbaah',
    0xe624: 'ccccccccccbaaf',
    0xe625: 'ccccccccccbaad',
    0xe626: 'ccccccccccbaab',
    0xe627: 'ccccccccccbaca',
    0xe628: 'ccccccccccbaiz',
    0xe629: 'ccccccccccbaix',
    0xe62a: 'ccccccccccbaaz',
    0xe62b: 'ccccccccccbaax',
    0xe62c: 'ccccccccccbaav',
    0xe62d: 'ccccccccccbaat',
    0xe62e: 'cccaabq',
    0xe62f: 'ccccccccccbaap',
    0xe630: 'bbbbbap',
    0xe631: 'ccccccccccbaaL',
    0xe632: 'aaaaabq',
    0xe633: 'ccccccccccbaaH',
    0xe634: 'ccccccccccbaaF',
    0xe635: 'ccccccccccbaaD',
    0xe636: 'ccccccccccbaaB',
    0xe637: 'ccccccccccbacA',
    0xe638: 'ccccccccccbaiZ',
    0xe639: 'ccccccccccbaiX',
    0xe63a: 'ccccccccccbaaZ',
    0xe63b: 'ccccccccccbaaX',
    0xe63c: 'ccccccccccbaaV',
    0xe63d: 'aaaaa004',
    0xe63e: 'cccbbap',
    0xe63f: 'bbbbb104',
    0xe640: 'bbbabbq',
    0xe641: 'cccca104',
    0xe642: 'aaabaap',
    0xe643: 'ppppa015',
    0xe644: 'dddaaap',
    0xe645: 'eeeea004',
    0xe646: 'gggbbbq',
    0xe647: 'ffffb104',
    0xe648: 'fffbbbq',
    0xe649: 'ooooa114',
    0xe64a: 'eeeaaap',
    0xe64b: 'llllb014',
    0xe64c: 'hhhaabq',
    0xe64d: 'iiiia014',
    0xe64e: 'cccabbq',
    0xe64f: 'jjjjb114',
    0xe650: 'bbbbaap',
    0xe651: 'kkkkb114',
    0xe652: 'aaaabbq',
    0xe653: 'hhhha014',
    0xe654: 'dddbbbq',
    0xe655: 'mmmmb014',
    0xe656: 'gggaaap',
    0xe657: 'nnnna114',
    0xe658: 'fffaaap',
    0xe659: 'ggggb104',
    0xe65a: 'eeebbbq',
    0xe65b: 'dddda004',
    0xe65c: 'hhhbbap',
    0xe65d: 'aaaab004',
    0xe65e: 'cccbaap',
    0xe65f: 'bbbba104',
    0xe660: 'bbbaaap',
    0xe661: 'ccccb004',
    0xe662: 'aaabbbq',
    0xe663: 'ppppb115',
    0xe664: 'dddabbq',
    0xe665: 'eeeeb104',
    0xe666: 'gggbaap',
    0xe667: 'ffffa004',
    0xe668: 'fffbaap',
    0xe669: 'oooob014',
    0xe66a: 'eeeabbq',
    0xe66b: 'lllla114',
    0xe66c: 'hhhabap',
    0xe66d: 'iiiib114',
    0xe66e: 'cccaaap',
    0xe66f: 'jjjja014',
    0xe670: 'bbbbbbq',
    0xe671: 'hhhhhhaaE8',
    0xe672: 'aaaaaap',
    0xe673: 'hhhhb114',
    0xe674: 'dddbaap',
    0xe675: 'mmmma114',
    0xe676: 'gggabbq',
    0xe677: 'mmmmmmbaE8',
    0xe678: 'dddddda106',
    0xe679: 'ddddddaaD8',
    0xe67a: 'ddddddaabK',
    0xe67b: 'ddddb104',
    0xe67c: 'aaaaaababK',
    0xe67d: 'aaaaa104',
    0xe67e: 'aaaaaab106',
    0xe67f: 'aaaaaabaD8',
    0xe680: 'bbbabaq',
    0xe681: 'cccca014',
    0xe682: 'aaababp',
    0xe683: 'ccccccbaD8',
    0xe684: 'dddaabp',
    0xe685: 'eeeea114',
    0xe686: 'eeeeeeb006',
    0xe687: 'eeeeeebbE8',
    0xe688: 'fffbbaq',
    0xe689: 'llllllabD8',
    0xe68a: 'eeeaabp',
    0xe68b: 'llllb104',
    0xe68c: 'hhhaaaq',
    0xe68d: 'iiiia104',
    0xe68e: 'cccabaq',
    0xe68f: 'iiiiiibbD8',
    0xe690: 'bbbbabp',
    0xe691: 'kkkkb004',
    0xe692: 'aaaabaq',
    0xe693: 'hhhha104',
    0xe694: 'dddbbaq',
    0xe695: 'mmmmb104',
    0xe696: 'gggaabp',
    0xe697: 'nnnna004',
    0xe698: 'fffaabp',
    0xe699: 'ggggb014',
    0xe69a: 'eeebbaq',
    0xe69b: 'dddda114',
    0xe69c: 'hhhbbbp',
    0xe69d: 'aaaab114',
    0xe69e: 'cccbabp',
    0xe69f: 'bbbba014',
    0xe6a0: 'bbbaabp',
    0xe6a1: 'ccccb114',
    0xe6a2: 'aaabbaq',
    0xe6a3: 'ppppb005',
    0xe6a4: 'dddabaq',
    0xe6a5: 'eeeeb014',
    0xe6a6: 'gggbabp',
    0xe6a7: 'ffffa114',
    0xe6a8: 'fffbabp',
    0xe6a9: 'oooob104',
    0xe6aa: 'eeeabaq',
    0xe6ab: 'lllla004',
    0xe6ac: 'hhhabbp',
    0xe6ad: 'iiiib004',
    0xe6ae: 'cccaabp',
    0xe6af: 'jjjja104',
    0xe6b0: 'bbbbbaq',
    0xe6b1: 'kkkka104',
    0xe6b2: 'aaaaabp',
    0xe6b3: 'hhhhb004',
    0xe6b4: 'dddbabp',
    0xe6b5: 'mmmma004',
    0xe6b6: 'gggabaq',
    0xe6b7: 'nnnnb104',
    0xe6b8: 'fffabaq',
    0xe6b9: 'gggga114',
    0xe6ba: 'eeebabp',
    0xe6bb: 'ddddb014',
    0xe6bc: 'hhhbaaq',
    0xe6bd: 'aaaaa014',
    0xe6be: 'cccbbaq',
    0xe6bf: 'bbbbb114',
    0xe6c0: 'bbbabbp',
    0xe6c1: 'cccca114',
    0xe6c2: 'aaabaaq',
    0xe6c3: 'ppppa005',
    0xe6c4: 'dddaaaq',
    0xe6c5: 'eeeea014',
    0xe6c6: 'gggbbbp',
    0xe6c7: 'ffffb114',
    0xe6c8: 'fffbbbp',
    0xe6c9: 'ooooa104',
    0xe6ca: 'eeeaaaq',
    0xe6cb: 'llllb004',
    0xe6cc: 'hhhaabp',
    0xe6cd: 'iiiia004',
    0xe6ce: 'cccabbp',
    0xe6cf: 'jjjjb104',
    0xe6d0: 'bbbbaaq',
    0xe6d1: 'kkkkb104',
    0xe6d2: 'aaaabbp',
    0xe6d3: 'hhhha004',
    0xe6d4: 'dddbbbp',
    0xe6d5: 'mmmmb004',
    0xe6d6: 'gggaaaq',
    0xe6d7: 'nnnna104',
    0xe6d8: 'fffaaaq',
    0xe6d9: 'ggggb114',
    0xe6da: 'eeebbbp',
    0xe6db: 'dddda014',
    0xe6dc: 'hhhbbaq',
    0xe6dd: 'aaaab014',
    0xe6de: 'cccbaaq',
    0xe6df: 'bbbba114',
    0xe6e0: 'bbbaaaq',
    0xe6e1: 'ccccb014',
    0xe6e2: 'aaabbbp',
    0xe6e3: 'ppppb105',
    0xe6e4: 'dddabbp',
    0xe6e5: 'eeeeb114',
    0xe6e6: 'gggbaaq',
    0xe6e7: 'ffffa014',
    0xe6e8: 'fffbaaq',
    0xe6e9: 'oooob004',
    0xe6ea: 'eeeabbp',
    0xe6eb: 'lllla104',
    0xe6ec: 'hhhabaq',
    0xe6ed: 'iiiib104',
    0xe6ee: 'cccaaaq',
    0xe6ef: 'jjjja004',
    0xe6f0: 'bbbbbbp',
    0xe6f1: 'hhhhhhbbD8',
    0xe6f2: 'aaaaaaq',
    0xe6f3: 'hhhhb104',
    0xe6f4: 'dddbaaq',
    0xe6f5: 'hhhhhhhaaau',
    0xe6f6: 'gggabbp',
    0xe6f7: 'hhhhhhhaaaU',
    0xe6f8: 'aaaaaaababD',
    0xe6f9: 'aaaaaaababT',
    0xe6fa: 'aaaaaaababd',
    0xe6fb: 'aaaaaaababt',
    0xe6fc: 'aaaaaaabaJ',
    0xe6fd: 'aaaaa114',
    0xe6fe: 'aaaaaaa006',
    0xe6ff: 'aaaaaaabE8',
    0xe700: 'bbbabar',
    0xe701: 'cccca005',
    0xe702: 'aaababs',
    0xe703: 'ccccccaaD8',
    0xe704: 'cccccccbabD',
    0xe705: 'cccccccbabT',
    0xe706: 'cccccccbabd',
    0xe707: 'cccccccbabt',
    0xe708: 'fffbbar',
    0xe709: 'iiiiiiibbbT',
    0xe70a: 'eeeaabs',
    0xe70b: 'iiiiiiibbbt',
    0xe70c: 'hhhaaar',
    0xe70d: 'iiiia115',
    0xe70e: 'cccabar',
    0xe70f: 'iiiiiiabD8',
    0xe710: 'bbbbabs',
    0xe711: 'kkkkb015',
    0xe712: 'aaaabar',
    0xe713: 'hhhha115',
    0xe714: 'dddbbar',
    0xe715: 'mmmmb115',
    0xe716: 'gggaabs',
    0xe717: 'nnnna015',
    0xe718: 'fffaabs',
    0xe719: 'ggggb005',
    0xe71a: 'eeebbar',
    0xe71b: 'dddda105',
    0xe71c: 'hhhbbbs',
    0xe71d: 'aaaab105',
    0xe71e: 'cccbabs',
    0xe71f: 'bbbba005',
    0xe720: 'bbbaabs',
    0xe721: 'ccccb105',
    0xe722: 'aaabbar',
    0xe723: 'ppppb014',
    0xe724: 'dddabar',
    0xe725: 'eeeeb005',
    0xe726: 'gggbabs',
    0xe727: 'ffffa105',
    0xe728: 'fffbabs',
    0xe729: 'oooob115',
    0xe72a: 'eeeabar',
    0xe72b: 'lllla015',
    0xe72c: 'hhhabbs',
    0xe72d: 'iiiib015',
    0xe72e: 'cccaabs',
    0xe72f: 'jjjja115',
    0xe730: 'bbbbbar',
    0xe731: 'kkkka115',
    0xe732: 'aaaaabs',
    0xe733: 'hhhhb015',
    0xe734: 'dddbabs',
    0xe735: 'mmmma015',
    0xe736: 'gggabar',
    0xe737: 'nnnnb115',
    0xe738: 'fffabar',
    0xe739: 'gggga105',
    0xe73a: 'eeebabs',
    0xe73b: 'ddddb005',
    0xe73c: 'hhhbaar',
    0xe73d: 'aaaaa005',
    0xe73e: 'cccbbar',
    0xe73f: 'bbbbb105',
    0xe740: 'bbbabbs',
    0xe741: 'cccca105',
    0xe742: 'aaabaar',
    0xe743: 'ppppa014',
    0xe744: 'dddaaar',
    0xe745: 'eeeea005',
    0xe746: 'gggbbbs',
    0xe747: 'ffffb105',
    0xe748: 'fffbbbs',
    0xe749: 'ooooa115',
    0xe74a: 'eeeaaar',
    0xe74b: 'llllb015',
    0xe74c: 'hhhaabs',
    0xe74d: 'iiiia015',
    0xe74e: 'cccabbs',
    0xe74f: 'jjjjb115',
    0xe750: 'bbbbaar',
    0xe751: 'kkkkb115',
    0xe752: 'aaaabbs',
    0xe753: 'hhhha015',
    0xe754: 'dddbbbs',
    0xe755: 'mmmmb015',
    0xe756: 'gggaaar',
    0xe757: 'nnnna115',
    0xe758: 'fffaaar',
    0xe759: 'ggggb105',
    0xe75a: 'eeebbbs',
    0xe75b: 'dddda005',
    0xe75c: 'hhhbbar',
    0xe75d: 'aaaab005',
    0xe75e: 'cccbaar',
    0xe75f: 'bbbba105',
    0xe760: 'bbbaaar',
    0xe761: 'ccccb005',
    0xe762: 'aaabbbs',
    0xe763: 'ppppb114',
    0xe764: 'dddabbs',
    0xe765: 'eeeeb105',
    0xe766: 'gggbaar',
    0xe767: 'ffffa005',
    0xe768: 'fffbaar',
    0xe769: 'oooob015',
    0xe76a: 'eeeabbs',
    0xe76b: 'lllla115',
    0xe76c: 'hhhabar',
    0xe76d: 'iiiib115',
    0xe76e: 'cccaaar',
    0xe76f: 'jjjja015',
    0xe770: 'bbbbbbs',
    0xe771: 'hhhhhhabD8',
    0xe772: 'aaaaaar',
    0xe773: 'hhhhb115',
    0xe774: 'dddbaar',
    0xe775: 'mmmma115',
    0xe776: 'gggabbs',
    0xe777: 'mmmmmmbbD8',
    0xe778: 'dddddda006',
    0xe779: 'ddddddabE8',
    0xe77a: 'ddddddabaJ',
    0xe77b: 'ddddb105',
    0xe77c: 'aaaaaabbaJ',
    0xe77d: 'aaaaa105',
    0xe77e: 'aaaaaab006',
    0xe77f: 'aaaaaabbE8',
    0xe780: 'bbbabas',
    0xe781: 'cccca015',
    0xe782: 'aaababr',
    0xe783: 'ccccccbbE8',
    0xe784: 'dddaabr',
    0xe785: 'eeeea115',
    0xe786: 'eeeeeeb106',
    0xe787: 'eeeeeebaD8',
    0xe788: 'fffbbas',
    0xe789: 'llllllaaE8',
    0xe78a: 'eeeaabr',
    0xe78b: 'llllb105',
    0xe78c: 'hhhaaas',
    0xe78d: 'iiiia105',
    0xe78e: 'cccabas',
    0xe78f: 'iiiiiibaE8',
    0xe790: 'bbbbabr',
    0xe791: 'kkkkb005',
    0xe792: 'aaaabas',
    0xe793: 'hhhha105',
    0xe794: 'dddbbas',
    0xe795: 'mmmmb105',
    0xe796: 'gggaabr',
    0xe797: 'nnnna005',
    0xe798: 'fffaabr',
    0xe799: 'ggggb015',
    0xe79a: 'eeebbas',
    0xe79b: 'dddda115',
    0xe79c: 'hhhbbbr',
    0xe79d: 'aaaab115',
    0xe79e: 'cccbabr',
    0xe79f: 'bbbba015',
    0xe7a0: 'bbbaabr',
    0xe7a1: 'ccccb115',
    0xe7a2: 'aaabbas',
    0xe7a3: 'ppppb004',
    0xe7a4: 'dddabas',
    0xe7a5: 'eeeeb015',
    0xe7a6: 'gggbabr',
    0xe7a7: 'ffffa115',
    0xe7a8: 'fffbabr',
    0xe7a9: 'oooob105',
    0xe7aa: 'eeeabas',
    0xe7ab: 'lllla005',
    0xe7ac: 'hhhabbr',
    0xe7ad: 'iiiib005',
    0xe7ae: 'cccaabr',
    0xe7af: 'jjjja105',
    0xe7b0: 'bbbbbas',
    0xe7b1: 'kkkka105',
    0xe7b2: 'aaaaabr',
    0xe7b3: 'hhhhb005',
    0xe7b4: 'dddbabr',
    0xe7b5: 'mmmma005',
    0xe7b6: 'gggabas',
    0xe7b7: 'nnnnb105',
    0xe7b8: 'fffabas',
    0xe7b9: 'gggga115',
    0xe7ba: 'eeebabr',
    0xe7bb: 'ddddb015',
    0xe7bc: 'hhhbaas',
    0xe7bd: 'aaaaa015',
    0xe7be: 'cccbbas',
    0xe7bf: 'bbbbb115',
    0xe7c0: 'bbbabbr',
    0xe7c1: 'cccca115',
    0xe7c2: 'aaabaas',
    0xe7c3: 'ppppa004',
    0xe7c4: 'dddaaas',
    0xe7c5: 'eeeea015',
    0xe7c6: 'gggbbbr',
    0xe7c7: 'ffffb115',
    0xe7c8: 'fffbbbr',
    0xe7c9: 'ooooa105',
    0xe7ca: 'eeeaaas',
    0xe7cb: 'llllb005',
    0xe7cc: 'hhhaabr',
    0xe7cd: 'iiiia005',
    0xe7ce: 'cccabbr',
    0xe7cf: 'jjjjb105',
    0xe7d0: 'bbbbaas',
    0xe7d1: 'kkkkb105',
    0xe7d2: 'aaaabbr',
    0xe7d3: 'hhhha005',
    0xe7d4: 'dddbbbr',
    0xe7d5: 'mmmmb005',
    0xe7d6: 'gggaaas',
    0xe7d7: 'nnnna105',
    0xe7d8: 'fffaaas',
    0xe7d9: 'ggggb115',
    0xe7da: 'eeebbbr',
    0xe7db: 'dddda015',
    0xe7dc: 'hhhbbas',
    0xe7dd: 'aaaab015',
    0xe7de: 'cccbaas',
    0xe7df: 'bbbba115',
    0xe7e0: 'bbbaaas',
    0xe7e1: 'ccccb015',
    0xe7e2: 'aaabbbr',
    0xe7e3: 'ppppb104',
    0xe7e4: 'dddabbr',
    0xe7e5: 'eeeeb115',
    0xe7e6: 'gggbaas',
    0xe7e7: 'ffffa015',
    0xe7e8: 'fffbaas',
    0xe7e9: 'oooob005',
    0xe7ea: 'eeeabbr',
    0xe7eb: 'lllla105',
    0xe7ec: 'hhhabas',
    0xe7ed: 'iiiib105',
    0xe7ee: 'cccaaas',
    0xe7ef: 'jjjja005',
    0xe7f0: 'aaaaaaaabaaj',
    0xe7f1: 'aaaaaaaabaab',
    0xe7f2: 'aaaaaas',
    0xe7f3: 'aaaaaaaabaar',
    0xe7f4: 'aaaaaaaabaaJ',
    0xe7f5: 'aaaaaaaabaaB',
    0xe7f6: 'aaaaaaaabaaZ',
    0xe7f7: 'aaaaaaaabaaR',
    0xe7f8: 'aaaaaaaabaE',
    0xe7f9: 'aaaaaaaabaU',
    0xe7fa: 'aaaaaaaabae',
    0xe7fb: 'aaaaaaaabau',
    0xe7fc: 'aaaaaaaabK',
    0xe7fd: 'aaaaa115',
    0xe7fe: 'aaaaaaa106',
    0xe7ff: 'aaaaaaaaD8',
    0xe800: 'bbbabal',
    0xe801: 'cccca048',
    0xe802: 'aaababm',
    0xe803: 'ccccccabB8',
    0xe804: 'cccccccaaaD',
    0xe805: 'cccccccaaaT',
    0xe806: 'cccccccaaad',
    0xe807: 'cccccccaaat',
    0xe808: 'ccccccccbbaJ',
    0xe809: 'ccccccccbbaB',
    0xe80a: 'ccccccccbbaZ',
    0xe80b: 'ccccccccbbaR',
    0xe80c: 'ccccccccbbaj',
    0xe80d: 'ccccccccbbab',
    0xe80e: 'cccabal',
    0xe80f: 'ccccccccbbar',
    0xe810: 'bbbbabm',
    0xe811: 'kkkkb058',
    0xe812: 'aaaabal',
    0xe813: 'hhhha158',
    0xe814: 'dddbbal',
    0xe815: 'mmmmb158',
    0xe816: 'gggaabm',
    0xe817: 'nnnna058',
    0xe818: 'fffaabm',
    0xe819: 'ggggb048',
    0xe81a: 'eeebbal',
    0xe81b: 'dddda148',
    0xe81c: 'hhhbbbm',
    0xe81d: 'aaaab148',
    0xe81e: 'cccbabm',
    0xe81f: 'bbbba048',
    0xe820: 'bbbaabm',
    0xe821: 'ccccb148',
    0xe822: 'aaabbal',
    0xe823: 'ppppb059',
    0xe824: 'dddabal',
    0xe825: 'eeeeb048',
    0xe826: 'gggbabm',
    0xe827: 'ffffa148',
    0xe828: 'fffbabm',
    0xe829: 'oooob158',
    0xe82a: 'eeeabal',
    0xe82b: 'lllla058',
    0xe82c: 'hhhabbm',
    0xe82d: 'iiiib058',
    0xe82e: 'cccaabm',
    0xe82f: 'jjjja158',
    0xe830: 'bbbbbal',
    0xe831: 'kkkka158',
    0xe832: 'aaaaabm',
    0xe833: 'hhhhb058',
    0xe834: 'dddbabm',
    0xe835: 'mmmma058',
    0xe836: 'gggabal',
    0xe837: 'nnnnb158',
    0xe838: 'fffabal',
    0xe839: 'gggga148',
    0xe83a: 'eeebabm',
    0xe83b: 'ddddb048',
    0xe83c: 'hhhbaal',
    0xe83d: 'aaaaa048',
    0xe83e: 'cccbbal',
    0xe83f: 'bbbbb148',
    0xe840: 'bbbabbm',
    0xe841: 'cccca148',
    0xe842: 'aaabaal',
    0xe843: 'ppppa059',
    0xe844: 'dddaaal',
    0xe845: 'eeeea048',
    0xe846: 'gggbbbm',
    0xe847: 'ffffb148',
    0xe848: 'fffbbbm',
    0xe849: 'ooooa158',
    0xe84a: 'eeeaaal',
    0xe84b: 'llllb058',
    0xe84c: 'hhhaabm',
    0xe84d: 'iiiia058',
    0xe84e: 'cccabbm',
    0xe84f: 'jjjjb158',
    0xe850: 'bbbbaal',
    0xe851: 'kkkkb158',
    0xe852: 'aaaabbm',
    0xe853: 'hhhha058',
    0xe854: 'dddbbbm',
    0xe855: 'mmmmb058',
    0xe856: 'gggaaal',
    0xe857: 'nnnna158',
    0xe858: 'fffaaal',
    0xe859: 'ggggb148',
    0xe85a: 'eeebbbm',
    0xe85b: 'dddda048',
    0xe85c: 'hhhbbal',
    0xe85d: 'aaaab048',
    0xe85e: 'cccbaal',
    0xe85f: 'bbbba148',
    0xe860: 'bbbaaal',
    0xe861: 'ccccb048',
    0xe862: 'aaabbbm',
    0xe863: 'ppppb159',
    0xe864: 'dddabbm',
    0xe865: 'eeeeb148',
    0xe866: 'gggbaal',
    0xe867: 'ffffa048',
    0xe868: 'fffbaal',
    0xe869: 'oooob058',
    0xe86a: 'eeeabbm',
    0xe86b: 'lllla158',
    0xe86c: 'hhhabal',
    0xe86d: 'iiiib158',
    0xe86e: 'cccaaal',
    0xe86f: 'jjjja058',
    0xe870: 'bbbbbbm',
    0xe871: 'hhhhhhaaB8',
    0xe872: 'aaaaaal',
    0xe873: 'hhhhb158',
    0xe874: 'dddbaal',
    0xe875: 'mmmma158',
    0xe876: 'gggabbm',
    0xe877: 'mmmmmmbaB8',
    0xe878: 'dddddda115',
    0xe879: 'ddddddaaA9',
    0xe87a: 'ddddddaaaI',
    0xe87b: 'ddddb148',
    0xe87c: 'aaaaaabaaI',
    0xe87d: 'aaaaa148',
    0xe87e: 'aaaaaab115',
    0xe87f: 'aaaaaabaA9',
    0xe880: 'bbbabam',
    0xe881: 'cccca058',
    0xe882: 'aaababl',
    0xe883: 'ccccccbaA9',
    0xe884: 'dddaabl',
    0xe885: 'eeeea158',
    0xe886: 'eeeeeeb015',
    0xe887: 'eeeeeebbB8',
    0xe888: 'fffbbam',
    0xe889: 'llllllabA9',
    0xe88a: 'eeeaabl',
    0xe88b: 'llllb148',
    0xe88c: 'hhhaaam',
    0xe88d: 'iiiia148',
    0xe88e: 'cccabam',
    0xe88f: 'iiiiiibbA9',
    0xe890: 'bbbbabl',
    0xe891: 'kkkkb048',
    0xe892: 'aaaabam',
    0xe893: 'hhhha148',
    0xe894: 'dddbbam',
    0xe895: 'mmmmb148',
    0xe896: 'gggaabl',
    0xe897: 'nnnna048',
    0xe898: 'fffaabl',
    0xe899: 'ggggb058',
    0xe89a: 'eeebbam',
    0xe89b: 'dddda158',
    0xe89c: 'hhhbbbl',
    0xe89d: 'aaaab158',
    0xe89e: 'cccbabl',
    0xe89f: 'bbbba058',
    0xe8a0: 'bbbaabl',
    0xe8a1: 'ccccb158',
    0xe8a2: 'aaabbam',
    0xe8a3: 'ppppb049',
    0xe8a4: 'dddabam',
    0xe8a5: 'eeeeb058',
    0xe8a6: 'gggbabl',
    0xe8a7: 'ffffa158',
    0xe8a8: 'fffbabl',
    0xe8a9: 'oooob148',
    0xe8aa: 'eeeabam',
    0xe8ab: 'lllla048',
    0xe8ac: 'hhhabbl',
    0xe8ad: 'iiiib048',
    0xe8ae: 'cccaabl',
    0xe8af: 'jjjja148',
    0xe8b0: 'bbbbbam',
    0xe8b1: 'kkkka148',
    0xe8b2: 'aaaaabl',
    0xe8b3: 'hhhhb048',
    0xe8b4: 'dddbabl',
    0xe8b5: 'mmmma048',
    0xe8b6: 'gggabam',
    0xe8b7: 'nnnnb148',
    0xe8b8: 'fffabam',
    0xe8b9: 'gggga158',
    0xe8ba: 'eeebabl',
    0xe8bb: 'ddddb058',
    0xe8bc: 'hhhbaam',
    0xe8bd: 'aaaaa058',
    0xe8be: 'cccbbam',
    0xe8bf: 'bbbbb158',
    0xe8c0: 'bbbabbl',
    0xe8c1: 'cccca158',
    0xe8c2: 'aaabaam',
    0xe8c3: 'ppppa049',
    0xe8c4: 'dddaaam',
    0xe8c5: 'eeeea058',
    0xe8c6: 'gggbbbl',
    0xe8c7: 'ffffb158',
    0xe8c8: 'fffbbbl',
    0xe8c9: 'ooooa148',
    0xe8ca: 'eeeaaam',
    0xe8cb: 'llllb048',
    0xe8cc: 'hhhaabl',
    0xe8cd: 'iiiia048',
    0xe8ce: 'cccabbl',
    0xe8cf: 'jjjjb148',
    0xe8d0: 'bbbbaam',
    0xe8d1: 'kkkkb148',
    0xe8d2: 'aaaabbl',
    0xe8d3: 'hhhha048',
    0xe8d4: 'dddbbbl',
    0xe8d5: 'mmmmb048',
    0xe8d6: 'gggaaam',
    0xe8d7: 'nnnna148',
    0xe8d8: 'fffaaam',
    0xe8d9: 'ggggb158',
    0xe8da: 'eeebbbl',
    0xe8db: 'dddda058',
    0xe8dc: 'hhhbbam',
    0xe8dd: 'aaaab058',
    0xe8de: 'cccbaam',
    0xe8df: 'bbbba158',
    0xe8e0: 'bbbaaam',
    0xe8e1: 'ccccb058',
    0xe8e2: 'aaabbbl',
    0xe8e3: 'ppppb149',
    0xe8e4: 'dddabbl',
    0xe8e5: 'eeeeb158',
    0xe8e6: 'gggbaam',
    0xe8e7: 'ffffa058',
    0xe8e8: 'fffbaam',
    0xe8e9: 'oooob048',
    0xe8ea: 'eeeabbl',
    0xe8eb: 'lllla148',
    0xe8ec: 'hhhabam',
    0xe8ed: 'iiiib148',
    0xe8ee: 'cccaaam',
    0xe8ef: 'jjjja048',
    0xe8f0: 'bbbbbbl',
    0xe8f1: 'hhhhhhbbA9',
    0xe8f2: 'aaaaaam',
    0xe8f3: 'hhhhb148',
    0xe8f4: 'dddbaam',
    0xe8f5: 'hhhhhhhabat',
    0xe8f6: 'gggabbl',
    0xe8f7: 'hhhhhhhabaT',
    0xe8f8: 'aaaaaaabbbE',
    0xe8f9: 'aaaaaaabbbU',
    0xe8fa: 'aaaaaaabbbe',
    0xe8fb: 'aaaaaaabbbu',
    0xe8fc: 'aaaaaaabbH',
    0xe8fd: 'aaaaa158',
    0xe8fe: 'aaaaaaa015',
    0xe8ff: 'aaaaaaabB8',
    0xe900: 'bbbaban',
    0xe901: 'cccca049',
    0xe902: 'aaababo',
    0xe903: 'ccccccaaA9',
    0xe904: 'cccccccbbbE',
    0xe905: 'cccccccbbbU',
    0xe906: 'cccccccbbbe',
    0xe907: 'cccccccbbbu',
    0xe908: 'fffbban',
    0xe909: 'iiiiiiibabU',
    0xe90a: 'eeeaabo',
    0xe90b: 'iiiiiiibabu',
    0xe90c: 'hhhaaan',
    0xe90d: 'iiiia159',
    0xe90e: 'cccaban',
    0xe90f: 'iiiiiiabA9',
    0xe910: 'bbbbabo',
    0xe911: 'kkkkb059',
    0xe912: 'aaaaban',
    0xe913: 'hhhha159',
    0xe914: 'dddbban',
    0xe915: 'mmmmb159',
    0xe916: 'gggaabo',
    0xe917: 'nnnna059',
    0xe918: 'fffaabo',
    0xe919: 'ggggb049',
    0xe91a: 'eeebban',
    0xe91b: 'dddda149',
    0xe91c: 'hhhbbbo',
    0xe91d: 'aaaab149',
    0xe91e: 'cccbabo',
    0xe91f: 'bbbba049',
    0xe920: 'bbbaabo',
    0xe921: 'ccccb149',
    0xe922: 'aaabban',
    0xe923: 'ppppb058',
    0xe924: 'dddaban',
    0xe925: 'eeeeb049',
    0xe926: 'gggbabo',
    0xe927: 'ffffa149',
    0xe928: 'fffbabo',
    0xe929: 'oooob159',
    0xe92a: 'eeeaban',
    0xe92b: 'lllla059',
    0xe92c: 'hhhabbo',
    0xe92d: 'iiiib059',
    0xe92e: 'cccaabo',
    0xe92f: 'jjjja159',
    0xe930: 'bbbbban',
    0xe931: 'kkkka159',
    0xe932: 'aaaaabo',
    0xe933: 'hhhhb059',
    0xe934: 'dddbabo',
    0xe935: 'mmmma059',
    0xe936: 'gggaban',
    0xe937: 'nnnnb159',
    0xe938: 'fffaban',
    0xe939: 'gggga149',
    0xe93a: 'eeebabo',
    0xe93b: 'ddddb049',
    0xe93c: 'hhhbaan',
    0xe93d: 'aaaaa049',
    0xe93e: 'cccbban',
    0xe93f: 'bbbbb149',
    0xe940: 'bbbabbo',
    0xe941: 'cccca149',
    0xe942: 'aaabaan',
    0xe943: 'ppppa058',
    0xe944: 'dddaaan',
    0xe945: 'eeeea049',
    0xe946: 'gggbbbo',
    0xe947: 'ffffb149',
    0xe948: 'fffbbbo',
    0xe949: 'ooooa159',
    0xe94a: 'eeeaaan',
    0xe94b: 'llllb059',
    0xe94c: 'hhhaabo',
    0xe94d: 'iiiia059',
    0xe94e: 'cccabbo',
    0xe94f: 'jjjjb159',
    0xe950: 'bbbbaan',
    0xe951: 'kkkkb159',
    0xe952: 'aaaabbo',
    0xe953: 'hhhha059',
    0xe954: 'dddbbbo',
    0xe955: 'mmmmb059',
    0xe956: 'gggaaan',
    0xe957: 'nnnna159',
    0xe958: 'fffaaan',
    0xe959: 'ggggb149',
    0xe95a: 'eeebbbo',
    0xe95b: 'dddda049',
    0xe95c: 'hhhbban',
    0xe95d: 'aaaab049',
    0xe95e: 'cccbaan',
    0xe95f: 'bbbba149',
    0xe960: 'bbbaaan',
    0xe961: 'ccccb049',
    0xe962: 'aaabbbo',
    0xe963: 'ppppb158',
    0xe964: 'dddabbo',
    0xe965: 'eeeeb149',
    0xe966: 'gggbaan',
    0xe967: 'ffffa049',
    0xe968: 'fffbaan',
    0xe969: 'oooob059',
    0xe96a: 'eeeabbo',
    0xe96b: 'lllla159',
    0xe96c: 'hhhaban',
    0xe96d: 'iiiib159',
    0xe96e: 'cccaaan',
    0xe96f: 'jjjja059',
    0xe970: 'bbbbbbo',
    0xe971: 'hhhhhhabA9',
    0xe972: 'aaaaaan',
    0xe973: 'hhhhb159',
    0xe974: 'dddbaan',
    0xe975: 'mmmma159',
    0xe976: 'gggabbo',
    0xe977: 'mmmmmmbbA9',
    0xe978: 'dddddda015',
    0xe979: 'ddddddabB8',
    0xe97a: 'ddddddabbH',
    0xe97b: 'ddddb149',
    0xe97c: 'aaaaaabbbH',
    0xe97d: 'aaaaa149',
    0xe97e: 'aaaaaab015',
    0xe97f: 'aaaaaabbB8',
    0xe980: 'bbbabao',
    0xe981: 'cccca059',
    0xe982: 'aaababn',
    0xe983: 'ccccccbbB8',
    0xe984: 'dddaabn',
    0xe985: 'eeeea159',
    0xe986: 'eeeeeeb115',
    0xe987: 'eeeeeebaA9',
    0xe988: 'fffbbao',
    0xe989: 'llllllaaB8',
    0xe98a: 'eeeaabn',
    0xe98b: 'llllb149',
    0xe98c: 'hhhaaao',
    0xe98d: 'iiiia149',
    0xe98e: 'cccabao',
    0xe98f: 'iiiiiibaB8',
    0xe990: 'bbbbabn',
    0xe991: 'kkkkb049',
    0xe992: 'aaaabao',
    0xe993: 'hhhha149',
    0xe994: 'dddbbao',
    0xe995: 'mmmmb149',
    0xe996: 'gggaabn',
    0xe997: 'nnnna049',
    0xe998: 'fffaabn',
    0xe999: 'ggggb059',
    0xe99a: 'eeebbao',
    0xe99b: 'dddda159',
    0xe99c: 'hhhbbbn',
    0xe99d: 'aaaab159',
    0xe99e: 'cccbabn',
    0xe99f: 'bbbba059',
    0xe9a0: 'bbbaabn',
    0xe9a1: 'ccccb159',
    0xe9a2: 'aaabbao',
    0xe9a3: 'ppppb048',
    0xe9a4: 'dddabao',
    0xe9a5: 'eeeeb059',
    0xe9a6: 'gggbabn',
    0xe9a7: 'ffffa159',
    0xe9a8: 'fffbabn',
    0xe9a9: 'oooob149',
    0xe9aa: 'eeeabao',
    0xe9ab: 'lllla049',
    0xe9ac: 'hhhabbn',
    0xe9ad: 'iiiib049',
    0xe9ae: 'cccaabn',
    0xe9af: 'jjjja149',
    0xe9b0: 'bbbbbao',
    0xe9b1: 'kkkka149',
    0xe9b2: 'aaaaabn',
    0xe9b3: 'hhhhb049',
    0xe9b4: 'dddbabn',
    0xe9b5: 'mmmma049',
    0xe9b6: 'gggabao',
    0xe9b7: 'nnnnb149',
    0xe9b8: 'fffabao',
    0xe9b9: 'gggga159',
    0xe9ba: 'eeebabn',
    0xe9bb: 'ddddb059',
    0xe9bc: 'hhhbaao',
    0xe9bd: 'aaaaa059',
    0xe9be: 'cccbbao',
    0xe9bf: 'bbbbb159',
    0xe9c0: 'aaaaaaaaaabaaV',
    0xe9c1: 'aaaaaaaaaabaaT',
    0xe9c2: 'aaabaao',
    0xe9c3: 'aaaaaaaaaabaaP',
    0xe9c4: 'aaaaaaaaaabaiZ',
    0xe9c5: 'aaaaaaaaaabaiX',
    0xe9c6: 'aaaaaaaaaabaaZ',
    0xe9c7: 'aaaaaaaaaabaaX',
    0xe9c8: 'aaaaaaaaaabaaF',
    0xe9c9: 'aaaaaaaaaabaaD',
    0xe9ca: 'aaaaaaaaaabaaB',
    0xe9cb: 'aaaaaaaaaabacA',
    0xe9cc: 'aaaaaaaaaabaaN',
    0xe9cd: 'aaaaaaaaaabaaL',
    0xe9ce: 'aaaaaaaaaabaaJ',
    0xe9cf: 'aaaaaaaaaabaaH',
    0xe9d0: 'aaaaaaaaaabaav',
    0xe9d1: 'aaaaaaaaaabaat',
    0xe9d2: 'aaaabbn',
    0xe9d3: 'aaaaaaaaaabaap',
    0xe9d4: 'aaaaaaaaaabaiz',
    0xe9d5: 'aaaaaaaaaabaix',
    0xe9d6: 'aaaaaaaaaabaaz',
    0xe9d7: 'aaaaaaaaaabaax',
    0xe9d8: 'aaaaaaaaaabaaf',
    0xe9d9: 'aaaaaaaaaabaad',
    0xe9da: 'aaaaaaaaaabaab',
    0xe9db: 'aaaaaaaaaabaca',
    0xe9dc: 'aaaaaaaaaabaan',
    0xe9dd: 'aaaab059',
    0xe9de: 'aaaaaaaaaabaaj',
    0xe9df: 'aaaaaaaaaabaah',
    0xe9e0: 'aaaaaaaaaabaA',
    0xe9e1: 'aaaaaaaaaabaE',
    0xe9e2: 'aaabbbn',
    0xe9e3: 'aaaaaaaaaabaM',
    0xe9e4: 'aaaaaaaaaabaQ',
    0xe9e5: 'aaaaaaaaaabaU',
    0xe9e6: 'aaaaaaaaaabaY',
    0xe9e7: 'aaaaaaaaaabiY',
    0xe9e8: 'aaaaaaaaaabaa',
    0xe9e9: 'aaaaaaaaaabae',
    0xe9ea: 'aaaaaaaaaabai',
    0xe9eb: 'aaaaaaaaaabam',
    0xe9ec: 'aaaaaaaaaabaq',
    0xe9ed: 'aaaaaaaaaabau',
    0xe9ee: 'aaaaaaaaaabay',
    0xe9ef: 'aaaaaaaaaabiy',
    0xe9f0: 'aaaaaaaaaabk',
    0xe9f1: 'aaaaaaaaaabc',
    0xe9f2: 'aaaaaao',
    0xe9f3: 'aaaaaaaaaabs',
    0xe9f4: 'aaaaaaaaaabK',
    0xe9f5: 'aaaaaaaaaabC',
    0xe9f6: 'aaaaaaaaaadX',
    0xe9f7: 'aaaaaaaaaabS',
    0xe9f8: 'aaaaaaaaaaD',
    0xe9f9: 'aaaaaaaaaaT',
    0xe9fa: 'aaaaaaaaaad',
    0xe9fb: 'aaaaaaaaaat',
    0xe9fc: 'aaaaaaaaaI',
    0xe9fd: 'aaaaa159',
    0xe9fe: 'aaaaaaa115',
    0xe9ff: 'aaaaaaaaA9',
    0xea00: 'bbbabah',
    0xea01: 'cccca008',
    0xea02: 'aaababi',
    0xea03: 'ccccccabA9',
    0xea04: 'cccccccabbE',
    0xea05: 'cccccccabbU',
    0xea06: 'cccccccabbe',
    0xea07: 'cccccccabbu',
    0xea08: 'ccccccccaabK',
    0xea09: 'ccccccccaabC',
    0xea0a: 'ccccccccaadX',
    0xea0b: 'ccccccccaabS',
    0xea0c: 'ccccccccaabk',
    0xea0d: 'ccccccccaabc',
    0xea0e: 'cccabah',
    0xea0f: 'ccccccccaabs',
    0xea10: 'bbbbabi',
    0xea11: 'cccccccccbabt',
    0xea12: 'aaaabah',
    0xea13: 'cccccccccbahy',
    0xea14: 'cccccccccbadc',
    0xea15: 'cccccccccbabd',
    0xea16: 'cccccccccbabh',
    0xea17: 'cccccccccbabl',
    0xea18: 'cccccccccbabP',
    0xea19: 'cccccccccbabT',
    0xea1a: 'cccccccccbabX',
    0xea1b: 'cccccccccbahY',
    0xea1c: 'cccccccccbadC',
    0xea1d: 'aaaab108',
    0xea1e: 'cccbabi',
    0xea1f: 'bbbba008',
    0xea20: 'bbbaabi',
    0xea21: 'ccccb108',
    0xea22: 'aaabbah',
    0xea23: 'ppppb019',
    0xea24: 'dddabah',
    0xea25: 'eeeeb008',
    0xea26: 'gggbabi',
    0xea27: 'ffffa108',
    0xea28: 'fffbabi',
    0xea29: 'oooob118',
    0xea2a: 'eeeabah',
    0xea2b: 'lllla018',
    0xea2c: 'hhhabbi',
    0xea2d: 'iiiib018',
    0xea2e: 'cccaabi',
    0xea2f: 'jjjja118',
    0xea30: 'bbbbbah',
    0xea31: 'kkkka118',
    0xea32: 'aaaaabi',
    0xea33: 'hhhhb018',
    0xea34: 'dddbabi',
    0xea35: 'mmmma018',
    0xea36: 'gggabah',
    0xea37: 'nnnnb118',
    0xea38: 'fffabah',
    0xea39: 'gggga108',
    0xea3a: 'eeebabi',
    0xea3b: 'ddddb008',
    0xea3c: 'hhhbaah',
    0xea3d: 'aaaaa008',
    0xea3e: 'cccbbah',
    0xea3f: 'bbbbb108',
    0xea40: 'bbbabbi',
    0xea41: 'cccca108',
    0xea42: 'aaabaah',
    0xea43: 'ppppa019',
    0xea44: 'dddaaah',
    0xea45: 'eeeea008',
    0xea46: 'gggbbbi',
    0xea47: 'ffffb108',
    0xea48: 'fffbbbi',
    0xea49: 'ooooa118',
    0xea4a: 'eeeaaah',
    0xea4b: 'llllb018',
    0xea4c: 'hhhaabi',
    0xea4d: 'iiiia018',
    0xea4e: 'cccabbi',
    0xea4f: 'jjjjb118',
    0xea50: 'bbbbaah',
    0xea51: 'kkkkb118',
    0xea52: 'aaaabbi',
    0xea53: 'hhhha018',
    0xea54: 'dddbbbi',
    0xea55: 'mmmmb018',
    0xea56: 'gggaaah',
    0xea57: 'nnnna118',
    0xea58: 'fffaaah',
    0xea59: 'ggggb108',
    0xea5a: 'eeebbbi',
    0xea5b: 'dddda008',
    0xea5c: 'hhhbbah',
    0xea5d: 'aaaab008',
    0xea5e: 'cccbaah',
    0xea5f: 'bbbba108',
    0xea60: 'bbbaaah',
    0xea61: 'ccccb008',
    0xea62: 'aaabbbi',
    0xea63: 'ppppb119',
    0xea64: 'dddabbi',
    0xea65: 'eeeeb108',
    0xea66: 'gggbaah',
    0xea67: 'ffffa008',
    0xea68: 'fffbaah',
    0xea69: 'oooob018',
    0xea6a: 'eeeabbi',
    0xea6b: 'lllla118',
    0xea6c: 'hhhabah',
    0xea6d: 'iiiib118',
    0xea6e: 'cccaaah',
    0xea6f: 'jjjja018',
    0xea70: 'bbbbbbi',
    0xea71: 'hhhhhhaaA9',
    0xea72: 'aaaaaah',
    0xea73: 'hhhhb118',
    0xea74: 'dddbaah',
    0xea75: 'mmmma118',
    0xea76: 'gggabbi',
    0xea77: 'mmmmmmbaA9',
    0xea78: 'dddddda105',
    0xea79: 'ddddddaaB8',
    0xea7a: 'ddddddaabH',
    0xea7b: 'ddddb108',
    0xea7c: 'aaaaaababH',
    0xea7d: 'aaaaa108',
    0xea7e: 'aaaaaab105',
    0xea7f: 'aaaaaabaB8',
    0xea80: 'bbbabai',
    0xea81: 'cccca018',
    0xea82: 'aaababh',
    0xea83: 'ccccccbaB8',
    0xea84: 'dddaabh',
    0xea85: 'eeeea118',
    0xea86: 'eeeeeeb005',
    0xea87: 'eeeeeebbA9',
    0xea88: 'fffbbai',
    0xea89: 'llllllabB8',
    0xea8a: 'eeeaabh',
    0xea8b: 'llllb108',
    0xea8c: 'hhhaaai',
    0xea8d: 'iiiia108',
    0xea8e: 'cccabai',
    0xea8f: 'iiiiiibbB8',
    0xea90: 'bbbbabh',
    0xea91: 'kkkkb008',
    0xea92: 'aaaabai',
    0xea93: 'hhhha108',
    0xea94: 'dddbbai',
    0xea95: 'mmmmb108',
    0xea96: 'gggaabh',
    0xea97: 'nnnna008',
    0xea98: 'fffaabh',
    0xea99: 'ggggb018',
    0xea9a: 'eeebbai',
    0xea9b: 'dddda118',
    0xea9c: 'hhhbbbh',
    0xea9d: 'aaaab118',
    0xea9e: 'cccbabh',
    0xea9f: 'bbbba018',
    0xeaa0: 'bbbaabh',
    0xeaa1: 'ccccb118',
    0xeaa2: 'aaabbai',
    0xeaa3: 'ppppb009',
    0xeaa4: 'dddabai',
    0xeaa5: 'eeeeb018',
    0xeaa6: 'gggbabh',
    0xeaa7: 'ffffa118',
    0xeaa8: 'fffbabh',
    0xeaa9: 'oooob108',
    0xeaaa: 'eeeabai',
    0xeaab: 'lllla008',
    0xeaac: 'hhhabbh',
    0xeaad: 'iiiib008',
    0xeaae: 'cccaabh',
    0xeaaf: 'jjjja108',
    0xeab0: 'bbbbbai',
    0xeab1: 'kkkka108',
    0xeab2: 'aaaaabh',
    0xeab3: 'hhhhb008',
    0xeab4: 'dddbabh',
    0xeab5: 'mmmma008',
    0xeab6: 'gggabai',
    0xeab7: 'nnnnb108',
    0xeab8: 'fffabai',
    0xeab9: 'gggga118',
    0xeaba: 'eeebabh',
    0xeabb: 'ddddb018',
    0xeabc: 'hhhbaai',
    0xeabd: 'aaaaa018',
    0xeabe: 'cccbbai',
    0xeabf: 'bbbbb118',
    0xeac0: 'bbbabbh',
    0xeac1: 'cccca118',
    0xeac2: 'aaabaai',
    0xeac3: 'ppppa009',
    0xeac4: 'dddaaai',
    0xeac5: 'eeeea018',
    0xeac6: 'gggbbbh',
    0xeac7: 'ffffb118',
    0xeac8: 'fffbbbh',
    0xeac9: 'ooooa108',
    0xeaca: 'eeeaaai',
    0xeacb: 'llllb008',
    0xeacc: 'hhhaabh',
    0xeacd: 'iiiia008',
    0xeace: 'cccabbh',
    0xeacf: 'jjjjb108',
    0xead0: 'bbbbaai',
    0xead1: 'kkkkb108',
    0xead2: 'aaaabbh',
    0xead3: 'hhhha008',
    0xead4: 'dddbbbh',
    0xead5: 'mmmmb008',
    0xead6: 'gggaaai',
    0xead7: 'nnnna108',
    0xead8: 'fffaaai',
    0xead9: 'ggggb118',
    0xeada: 'eeebbbh',
    0xeadb: 'dddda018',
    0xeadc: 'hhhbbai',
    0xeadd: 'aaaab018',
    0xeade: 'cccbaai',
    0xeadf: 'bbbba118',
    0xeae0: 'bbbaaai',
    0xeae1: 'ccccb018',
    0xeae2: 'aaabbbh',
    0xeae3: 'ppppb109',
    0xeae4: 'dddabbh',
    0xeae5: 'eeeeb118',
    0xeae6: 'gggbaai',
    0xeae7: 'ffffa018',
    0xeae8: 'fffbaai',
    0xeae9: 'oooob008',
    0xeaea: 'eeeabbh',
    0xeaeb: 'lllla108',
    0xeaec: 'hhhabai',
    0xeaed: 'iiiib108',
    0xeaee: 'cccaaai',
    0xeaef: 'jjjja008',
    0xeaf0: 'bbbbbbh',
    0xeaf1: 'hhhhhhbbB8',
    0xeaf2: 'aaaaaai',
    0xeaf3: 'hhhhb108',
    0xeaf4: 'dddbaai',
    0xeaf5: 'hhhhhhhaabu',
    0xeaf6: 'gggabbh',
    0xeaf7: 'hhhhhhhaabU',
    0xeaf8: 'aaaaaaabaaD',
    0xeaf9: 'aaaaaaabaaT',
    0xeafa: 'aaaaaaabaad',
    0xeafb: 'aaaaaaabaat',
    0xeafc: 'aaaaaaabaI',
    0xeafd: 'aaaaa118',
    0xeafe: 'aaaaaaa005',
    0xeaff: 'aaaaaaabA9',
    0xeb00: 'bbbabaj',
    0xeb01: 'cccca009',
    0xeb02: 'aaababk',
    0xeb03: 'ccccccaaB8',
    0xeb04: 'cccccccbaaD',
    0xeb05: 'cccccccbaaT',
    0xeb06: 'cccccccbaad',
    0xeb07: 'cccccccbaat',
    0xeb08: 'fffbbaj',
    0xeb09: 'iiiiiiibbaT',
    0xeb0a: 'eeeaabk',
    0xeb0b: 'iiiiiiibbat',
    0xeb0c: 'hhhaaaj',
    0xeb0d: 'iiiia119',
    0xeb0e: 'cccabaj',
    0xeb0f: 'iiiiiiabB8',
    0xeb10: 'bbbbabk',
    0xeb11: 'kkkkb019',
    0xeb12: 'aaaabaj',
    0xeb13: 'hhhha119',
    0xeb14: 'dddbbaj',
    0xeb15: 'mmmmb119',
    0xeb16: 'gggaabk',
    0xeb17: 'nnnna019',
    0xeb18: 'fffaabk',
    0xeb19: 'ggggb009',
    0xeb1a: 'eeebbaj',
    0xeb1b: 'dddda109',
    0xeb1c: 'hhhbbbk',
    0xeb1d: 'aaaab109',
    0xeb1e: 'cccbabk',
    0xeb1f: 'bbbba009',
    0xeb20: 'bbbaabk',
    0xeb21: 'ccccb109',
    0xeb22: 'aaabbaj',
    0xeb23: 'ppppb018',
    0xeb24: 'dddabaj',
    0xeb25: 'eeeeb009',
    0xeb26: 'gggbabk',
    0xeb27: 'ffffa109',
    0xeb28: 'fffbabk',
    0xeb29: 'oooob119',
    0xeb2a: 'eeeabaj',
    0xeb2b: 'lllla019',
    0xeb2c: 'hhhabbk',
    0xeb2d: 'iiiib019',
    0xeb2e: 'cccaabk',
    0xeb2f: 'jjjja119',
    0xeb30: 'bbbbbaj',
    0xeb31: 'kkkka119',
    0xeb32: 'aaaaabk',
    0xeb33: 'hhhhb019',
    0xeb34: 'dddbabk',
    0xeb35: 'mmmma019',
    0xeb36: 'gggabaj',
    0xeb37: 'nnnnb119',
    0xeb38: 'fffabaj',
    0xeb39: 'gggga109',
    0xeb3a: 'eeebabk',
    0xeb3b: 'ddddb009',
    0xeb3c: 'hhhbaaj',
    0xeb3d: 'aaaaa009',
    0xeb3e: 'cccbbaj',
    0xeb3f: 'bbbbb109',
    0xeb40: 'bbbabbk',
    0xeb41: 'cccca109',
    0xeb42: 'aaabaaj',
    0xeb43: 'ppppa018',
    0xeb44: 'dddaaaj',
    0xeb45: 'eeeea009',
    0xeb46: 'gggbbbk',
    0xeb47: 'ffffb109',
    0xeb48: 'fffbbbk',
    0xeb49: 'ooooa119',
    0xeb4a: 'eeeaaaj',
    0xeb4b: 'llllb019',
    0xeb4c: 'hhhaabk',
    0xeb4d: 'iiiia019',
    0xeb4e: 'cccabbk',
    0xeb4f: 'jjjjb119',
    0xeb50: 'bbbbaaj',
    0xeb51: 'kkkkb119',
    0xeb52: 'aaaabbk',
    0xeb53: 'hhhha019',
    0xeb54: 'dddbbbk',
    0xeb55: 'mmmmb019',
    0xeb56: 'gggaaaj',
    0xeb57: 'nnnna119',
    0xeb58: 'fffaaaj',
    0xeb59: 'ggggb109',
    0xeb5a: 'eeebbbk',
    0xeb5b: 'dddda009',
    0xeb5c: 'hhhbbaj',
    0xeb5d: 'aaaab009',
    0xeb5e: 'cccbaaj',
    0xeb5f: 'bbbba109',
    0xeb60: 'bbbaaaj',
    0xeb61: 'ccccb009',
    0xeb62: 'aaabbbk',
    0xeb63: 'ppppb118',
    0xeb64: 'dddabbk',
    0xeb65: 'eeeeb109',
    0xeb66: 'gggbaaj',
    0xeb67: 'ffffa009',
    0xeb68: 'fffbaaj',
    0xeb69: 'oooob019',
    0xeb6a: 'eeeabbk',
    0xeb6b: 'lllla119',
    0xeb6c: 'hhhabaj',
    0xeb6d: 'iiiib119',
    0xeb6e: 'cccaaaj',
    0xeb6f: 'jjjja019',
    0xeb70: 'bbbbbbk',
    0xeb71: 'hhhhhhabB8',
    0xeb72: 'aaaaaaj',
    0xeb73: 'hhhhb119',
    0xeb74: 'dddbaaj',
    0xeb75: 'mmmma119',
    0xeb76: 'gggabbk',
    0xeb77: 'mmmmmmbbB8',
    0xeb78: 'dddddda005',
    0xeb79: 'ddddddabA9',
    0xeb7a: 'ddddddabaI',
    0xeb7b: 'ddddb109',
    0xeb7c: 'aaaaaabbaI',
    0xeb7d: 'aaaaa109',
    0xeb7e: 'aaaaaab005',
    0xeb7f: 'aaaaaabbA9',
    0xeb80: 'bbbabak',
    0xeb81: 'cccca019',
    0xeb82: 'aaababj',
    0xeb83: 'ccccccbbA9',
    0xeb84: 'dddaabj',
    0xeb85: 'eeeea119',
    0xeb86: 'eeeeeeb105',
    0xeb87: 'eeeeeebaB8',
    0xeb88: 'fffbbak',
    0xeb89: 'llllllaaA9',
    0xeb8a: 'eeeaabj',
    0xeb8b: 'llllb109',
    0xeb8c: 'hhhaaak',
    0xeb8d: 'iiiia109',
    0xeb8e: 'cccabak',
    0xeb8f: 'iiiiiibaA9',
    0xeb90: 'bbbbabj',
    0xeb91: 'kkkkb009',
    0xeb92: 'aaaabak',
    0xeb93: 'hhhha109',
    0xeb94: 'dddbbak',
    0xeb95: 'mmmmb109',
    0xeb96: 'gggaabj',
    0xeb97: 'nnnna009',
    0xeb98: 'fffaabj',
    0xeb99: 'ggggb019',
    0xeb9a: 'eeebbak',
    0xeb9b: 'dddda119',
    0xeb9c: 'hhhbbbj',
    0xeb9d: 'aaaab119',
    0xeb9e: 'cccbabj',
    0xeb9f: 'bbbba019',
    0xeba0: 'bbbaabj',
    0xeba1: 'ccccb119',
    0xeba2: 'aaabbak',
    0xeba3: 'ppppb008',
    0xeba4: 'dddabak',
    0xeba5: 'eeeeb019',
    0xeba6: 'gggbabj',
    0xeba7: 'ffffa119',
    0xeba8: 'fffbabj',
    0xeba9: 'oooob109',
    0xebaa: 'eeeabak',
    0xebab: 'lllla009',
    0xebac: 'hhhabbj',
    0xebad: 'iiiib009',
    0xebae: 'cccaabj',
    0xebaf: 'jjjja109',
    0xebb0: 'bbbbbak',
    0xebb1: 'kkkka109',
    0xebb2: 'aaaaabj',
    0xebb3: 'hhhhb009',
    0xebb4: 'dddbabj',
    0xebb5: 'mmmma009',
    0xebb6: 'gggabak',
    0xebb7: 'nnnnb109',
    0xebb8: 'fffabak',
    0xebb9: 'gggga119',
    0xebba: 'eeebabj',
    0xebbb: 'ddddb019',
    0xebbc: 'hhhbaak',
    0xebbd: 'aaaaa019',
    0xebbe: 'cccbbak',
    0xebbf: 'bbbbb119',
    0xebc0: 'bbbabbj',
    0xebc1: 'cccca119',
    0xebc2: 'aaabaak',
    0xebc3: 'ppppa008',
    0xebc4: 'dddaaak',
    0xebc5: 'eeeea019',
    0xebc6: 'gggbbbj',
    0xebc7: 'ffffb119',
    0xebc8: 'fffbbbj',
    0xebc9: 'ooooa109',
    0xebca: 'eeeaaak',
    0xebcb: 'llllb009',
    0xebcc: 'hhhaabj',
    0xebcd: 'iiiia009',
    0xebce: 'cccabbj',
    0xebcf: 'jjjjb109',
    0xebd0: 'bbbbaak',
    0xebd1: 'kkkkb109',
    0xebd2: 'aaaabbj',
    0xebd3: 'hhhha009',
    0xebd4: 'dddbbbj',
    0xebd5: 'mmmmb009',
    0xebd6: 'gggaaak',
    0xebd7: 'nnnna109',
    0xebd8: 'fffaaak',
    0xebd9: 'ggggb119',
    0xebda: 'eeebbbj',
    0xebdb: 'dddda019',
    0xebdc: 'hhhbbak',
    0xebdd: 'aaaab019',
    0xebde: 'cccbaak',
    0xebdf: 'bbbba119',
    0xebe0: 'bbbaaak',
    0xebe1: 'ccccb019',
    0xebe2: 'aaabbbj',
    0xebe3: 'ppppb108',
    0xebe4: 'dddabbj',
    0xebe5: 'eeeeb119',
    0xebe6: 'gggbaak',
    0xebe7: 'ffffa019',
    0xebe8: 'fffbaak',
    0xebe9: 'oooob009',
    0xebea: 'eeeabbj',
    0xebeb: 'lllla109',
    0xebec: 'hhhabak',
    0xebed: 'iiiib109',
    0xebee: 'cccaaak',
    0xebef: 'jjjja009',
    0xebf0: 'aaaaaaaabbaj',
    0xebf1: 'aaaaaaaabbab',
    0xebf2: 'aaaaaak',
    0xebf3: 'aaaaaaaabbar',
    0xebf4: 'aaaaaaaabbaJ',
    0xebf5: 'aaaaaaaabbaB',
    0xebf6: 'aaaaaaaabbaZ',
    0xebf7: 'aaaaaaaabbaR',
    0xebf8: 'aaaaaaaabbE',
    0xebf9: 'aaaaaaaabbU',
    0xebfa: 'aaaaaaaabbe',
    0xebfb: 'aaaaaaaabbu',
    0xebfc: 'aaaaaaaabH',
    0xebfd: 'aaaaa119',
    0xebfe: 'aaaaaaa105',
    0xebff: 'aaaaaaaaB8',
    0xec00: 'bbbabad',
    0xec01: 'cccca888',
    0xec02: 'aaababe',
    0xec03: 'ccccccabB9',
    0xec04: 'cccccccaabE',
    0xec05: 'cccccccaabU',
    0xec06: 'cccccccaabe',
    0xec07: 'cccccccaabu',
    0xec08: 'ccccccccbabK',
    0xec09: 'ccccccccbabC',
    0xec0a: 'ccccccccbadX',
    0xec0b: 'ccccccccbabS',
    0xec0c: 'ccccccccbabk',
    0xec0d: 'ccccccccbabc',
    0xec0e: 'cccabad',
    0xec0f: 'ccccccccbabs',
    0xec10: 'bbbbabe',
    0xec11: 'kkkkb898',
    0xec12: 'aaaabad',
    0xec13: 'hhhha998',
    0xec14: 'dddbbad',
    0xec15: 'mmmmb998',
    0xec16: 'gggaabe',
    0xec17: 'nnnna898',
    0xec18: 'fffaabe',
    0xec19: 'ggggb888',
    0xec1a: 'eeebbad',
    0xec1b: 'dddda988',
    0xec1c: 'hhhbbbe',
    0xec1d: 'aaaab988',
    0xec1e: 'cccbabe',
    0xec1f: 'bbbba888',
    0xec20: 'bbbaabe',
    0xec21: 'ccccb988',
    0xec22: 'aaabbad',
    0xec23: 'ppppb899',
    0xec24: 'dddabad',
    0xec25: 'eeeeb888',
    0xec26: 'gggbabe',
    0xec27: 'ffffa988',
    0xec28: 'fffbabe',
    0xec29: 'oooob998',
    0xec2a: 'eeeabad',
    0xec2b: 'lllla898',
    0xec2c: 'hhhabbe',
    0xec2d: 'iiiib898',
    0xec2e: 'cccaabe',
    0xec2f: 'jjjja998',
    0xec30: 'bbbbbad',
    0xec31: 'kkkka998',
    0xec32: 'aaaaabe',
    0xec33: 'hhhhb898',
    0xec34: 'dddbabe',
    0xec35: 'mmmma898',
    0xec36: 'gggabad',
    0xec37: 'nnnnb998',
    0xec38: 'fffabad',
    0xec39: 'gggga988',
    0xec3a: 'eeebabe',
    0xec3b: 'ddddb888',
    0xec3c: 'hhhbaad',
    0xec3d: 'aaaaa888',
    0xec3e: 'cccbbad',
    0xec3f: 'bbbbb988',
    0xec40: 'bbbabbe',
    0xec41: 'cccca988',
    0xec42: 'aaabaad',
    0xec43: 'ppppa899',
    0xec44: 'dddaaad',
    0xec45: 'eeeea888',
    0xec46: 'gggbbbe',
    0xec47: 'ffffb988',
    0xec48: 'fffbbbe',
    0xec49: 'ooooa998',
    0xec4a: 'eeeaaad',
    0xec4b: 'llllb898',
    0xec4c: 'hhhaabe',
    0xec4d: 'iiiia898',
    0xec4e: 'cccabbe',
    0xec4f: 'jjjjb998',
    0xec50: 'bbbbaad',
    0xec51: 'kkkkb998',
    0xec52: 'aaaabbe',
    0xec53: 'hhhha898',
    0xec54: 'dddbbbe',
    0xec55: 'mmmmb898',
    0xec56: 'gggaaad',
    0xec57: 'nnnna998',
    0xec58: 'fffaaad',
    0xec59: 'ggggb988',
    0xec5a: 'eeebbbe',
    0xec5b: 'dddda888',
    0xec5c: 'hhhbbad',
    0xec5d: 'aaaab888',
    0xec5e: 'cccbaad',
    0xec5f: 'bbbba988',
    0xec60: 'bbbaaad',
    0xec61: 'ccccb888',
    0xec62: 'aaabbbe',
    0xec63: 'ppppb999',
    0xec64: 'dddabbe',
    0xec65: 'eeeeb988',
    0xec66: 'gggbaad',
    0xec67: 'ffffa888',
    0xec68: 'fffbaad',
    0xec69: 'oooob898',
    0xec6a: 'eeeabbe',
    0xec6b: 'lllla998',
    0xec6c: 'hhhabad',
    0xec6d: 'iiiib998',
    0xec6e: 'cccaaad',
    0xec6f: 'jjjja898',
    0xec70: 'bbbbbbe',
    0xec71: 'hhhhhhaaB9',
    0xec72: 'aaaaaad',
    0xec73: 'hhhhb998',
    0xec74: 'dddbaad',
    0xec75: 'mmmma998',
    0xec76: 'gggabbe',
    0xec77: 'mmmmmmbaB9',
    0xec78: 'dddddda114',
    0xec79: 'ddddddaaA8',
    0xec7a: 'ddddddaaaH',
    0xec7b: 'ddddb988',
    0xec7c: 'aaaaaabaaH',
    0xec7d: 'aaaaa988',
    0xec7e: 'aaaaaab114',
    0xec7f: 'aaaaaabaA8',
    0xec80: 'bbbabae',
    0xec81: 'cccca898',
    0xec82: 'aaababd',
    0xec83: 'ccccccbaA8',
    0xec84: 'dddaabd',
    0xec85: 'eeeea998',
    0xec86: 'eeeeeeb014',
    0xec87: 'eeeeeebbB9',
    0xec88: 'fffbbae',
    0xec89: 'llllllabA8',
    0xec8a: 'eeeaabd',
    0xec8b: 'llllb988',
    0xec8c: 'hhhaaae',
    0xec8d: 'iiiia988',
    0xec8e: 'cccabae',
    0xec8f: 'iiiiiibbA8',
    0xec90: 'bbbbabd',
    0xec91: 'kkkkb888',
    0xec92: 'aaaabae',
    0xec93: 'hhhha988',
    0xec94: 'dddbbae',
    0xec95: 'mmmmb988',
    0xec96: 'gggaabd',
    0xec97: 'nnnna888',
    0xec98: 'fffaabd',
    0xec99: 'ggggb898',
    0xec9a: 'eeebbae',
    0xec9b: 'dddda998',
    0xec9c: 'hhhbbbd',
    0xec9d: 'aaaab998',
    0xec9e: 'cccbabd',
    0xec9f: 'bbbba898',
    0xeca0: 'bbbaabd',
    0xeca1: 'ccccb998',
    0xeca2: 'aaabbae',
    0xeca3: 'ppppb889',
    0xeca4: 'dddabae',
    0xeca5: 'eeeeb898',
    0xeca6: 'gggbabd',
    0xeca7: 'ffffa998',
    0xeca8: 'fffbabd',
    0xeca9: 'oooob988',
    0xecaa: 'eeeabae',
    0xecab: 'lllla888',
    0xecac: 'hhhabbd',
    0xecad: 'iiiib888',
    0xecae: 'cccaabd',
    0xecaf: 'jjjja988',
    0xecb0: 'bbbbbae',
    0xecb1: 'kkkka988',
    0xecb2: 'aaaaabd',
    0xecb3: 'hhhhb888',
    0xecb4: 'dddbabd',
    0xecb5: 'mmmma888',
    0xecb6: 'gggabae',
    0xecb7: 'nnnnb988',
    0xecb8: 'fffabae',
    0xecb9: 'gggga998',
    0xecba: 'eeebabd',
    0xecbb: 'ddddb898',
    0xecbc: 'hhhbaae',
    0xecbd: 'aaaaa898',
    0xecbe: 'cccbbae',
    0xecbf: 'bbbbb998',
    0xecc0: 'bbbabbd',
    0xecc1: 'cccca998',
    0xecc2: 'aaabaae',
    0xecc3: 'ppppa889',
    0xecc4: 'dddaaae',
    0xecc5: 'eeeea898',
    0xecc6: 'gggbbbd',
    0xecc7: 'ffffb998',
    0xecc8: 'fffbbbd',
    0xecc9: 'ooooa988',
    0xecca: 'eeeaaae',
    0xeccb: 'llllb888',
    0xeccc: 'hhhaabd',
    0xeccd: 'iiiia888',
    0xecce: 'cccabbd',
    0xeccf: 'jjjjb988',
    0xecd0: 'bbbbaae',
    0xecd1: 'kkkkb988',
    0xecd2: 'aaaabbd',
    0xecd3: 'hhhha888',
    0xecd4: 'dddbbbd',
    0xecd5: 'mmmmb888',
    0xecd6: 'gggaaae',
    0xecd7: 'nnnna988',
    0xecd8: 'fffaaae',
    0xecd9: 'ggggb998',
    0xecda: 'eeebbbd',
    0xecdb: 'dddda898',
    0xecdc: 'hhhbbae',
    0xecdd: 'aaaab898',
    0xecde: 'cccbaae',
    0xecdf: 'bbbba998',
    0xece0: 'bbbaaae',
    0xece1: 'ccccb898',
    0xece2: 'aaabbbd',
    0xece3: 'ppppb989',
    0xece4: 'dddabbd',
    0xece5: 'eeeeb998',
    0xece6: 'gggbaae',
    0xece7: 'ffffa898',
    0xece8: 'fffbaae',
    0xece9: 'oooob888',
    0xecea: 'eeeabbd',
    0xeceb: 'lllla988',
    0xecec: 'hhhabae',
    0xeced: 'iiiib988',
    0xecee: 'cccaaae',
    0xecef: 'jjjja888',
    0xecf0: 'bbbbbbd',
    0xecf1: 'hhhhhhbbA8',
    0xecf2: 'aaaaaae',
    0xecf3: 'hhhhb988',
    0xecf4: 'dddbaae',
    0xecf5: 'hhhhhhhabbu',
    0xecf6: 'gggabbd',
    0xecf7: 'hhhhhhhabbU',
    0xecf8: 'aaaaaaabbaD',
    0xecf9: 'aaaaaaabbaT',
    0xecfa: 'aaaaaaabbad',
    0xecfb: 'aaaaaaabbat',
    0xecfc: 'aaaaaaabbI',
    0xecfd: 'aaaaa998',
    0xecfe: 'aaaaaaa014',
    0xecff: 'aaaaaaabB9',
    0xed00: 'bbbabaf',
    0xed01: 'cccca889',
    0xed02: 'aaababg',
    0xed03: 'ccccccaaA8',
    0xed04: 'cccccccbbaD',
    0xed05: 'cccccccbbaT',
    0xed06: 'cccccccbbad',
    0xed07: 'cccccccbbat',
    0xed08: 'fffbbaf',
    0xed09: 'iiiiiiibaaT',
    0xed0a: 'eeeaabg',
    0xed0b: 'iiiiiiibaat',
    0xed0c: 'hhhaaaf',
    0xed0d: 'iiiia999',
    0xed0e: 'cccabaf',
    0xed0f: 'iiiiiiabA8',
    0xed10: 'bbbbabg',
    0xed11: 'kkkkb899',
    0xed12: 'aaaabaf',
    0xed13: 'hhhha999',
    0xed14: 'dddbbaf',
    0xed15: 'mmmmb999',
    0xed16: 'gggaabg',
    0xed17: 'nnnna899',
    0xed18: 'fffaabg',
    0xed19: 'ggggb889',
    0xed1a: 'eeebbaf',
    0xed1b: 'dddda989',
    0xed1c: 'hhhbbbg',
    0xed1d: 'aaaab989',
    0xed1e: 'cccbabg',
    0xed1f: 'bbbba889',
    0xed20: 'bbbaabg',
    0xed21: 'ccccb989',
    0xed22: 'aaabbaf',
    0xed23: 'ppppb898',
    0xed24: 'dddabaf',
    0xed25: 'eeeeb889',
    0xed26: 'gggbabg',
    0xed27: 'ffffa989',
    0xed28: 'fffbabg',
    0xed29: 'oooob999',
    0xed2a: 'eeeabaf',
    0xed2b: 'lllla899',
    0xed2c: 'hhhabbg',
    0xed2d: 'iiiib899',
    0xed2e: 'cccaabg',
    0xed2f: 'jjjja999',
    0xed30: 'bbbbbaf',
    0xed31: 'kkkka999',
    0xed32: 'aaaaabg',
    0xed33: 'hhhhb899',
    0xed34: 'dddbabg',
    0xed35: 'mmmma899',
    0xed36: 'gggabaf',
    0xed37: 'nnnnb999',
    0xed38: 'fffabaf',
    0xed39: 'gggga989',
    0xed3a: 'eeebabg',
    0xed3b: 'ddddb889',
    0xed3c: 'hhhbaaf',
    0xed3d: 'aaaaa889',
    0xed3e: 'cccbbaf',
    0xed3f: 'bbbbb989',
    0xed40: 'bbbabbg',
    0xed41: 'cccca989',
    0xed42: 'aaabaaf',
    0xed43: 'ppppa898',
    0xed44: 'dddaaaf',
    0xed45: 'eeeea889',
    0xed46: 'gggbbbg',
    0xed47: 'ffffb989',
    0xed48: 'fffbbbg',
    0xed49: 'ooooa999',
    0xed4a: 'eeeaaaf',
    0xed4b: 'llllb899',
    0xed4c: 'hhhaabg',
    0xed4d: 'iiiia899',
    0xed4e: 'cccabbg',
    0xed4f: 'jjjjb999',
    0xed50: 'bbbbaaf',
    0xed51: 'kkkkb999',
    0xed52: 'aaaabbg',
    0xed53: 'hhhha899',
    0xed54: 'dddbbbg',
    0xed55: 'mmmmb899',
    0xed56: 'gggaaaf',
    0xed57: 'nnnna999',
    0xed58: 'fffaaaf',
    0xed59: 'ggggb989',
    0xed5a: 'eeebbbg',
    0xed5b: 'dddda889',
    0xed5c: 'hhhbbaf',
    0xed5d: 'aaaab889',
    0xed5e: 'cccbaaf',
    0xed5f: 'bbbba989',
    0xed60: 'bbbaaaf',
    0xed61: 'ccccb889',
    0xed62: 'aaabbbg',
    0xed63: 'ppppb998',
    0xed64: 'dddabbg',
    0xed65: 'eeeeb989',
    0xed66: 'gggbaaf',
    0xed67: 'ffffa889',
    0xed68: 'fffbaaf',
    0xed69: 'oooob899',
    0xed6a: 'eeeabbg',
    0xed6b: 'lllla999',
    0xed6c: 'hhhabaf',
    0xed6d: 'iiiib999',
    0xed6e: 'cccaaaf',
    0xed6f: 'jjjja899',
    0xed70: 'bbbbbbg',
    0xed71: 'hhhhhhabA8',
    0xed72: 'aaaaaaf',
    0xed73: 'hhhhb999',
    0xed74: 'dddbaaf',
    0xed75: 'mmmma999',
    0xed76: 'gggabbg',
    0xed77: 'mmmmmmbbA8',
    0xed78: 'dddddda014',
    0xed79: 'ddddddabB9',
    0xed7a: 'ddddddabbI',
    0xed7b: 'ddddb989',
    0xed7c: 'aaaaaabbbI',
    0xed7d: 'aaaaa989',
    0xed7e: 'aaaaaab014',
    0xed7f: 'aaaaaabbB9',
    0xed80: 'bbbabag',
    0xed81: 'cccca899',
    0xed82: 'aaababf',
    0xed83: 'ccccccbbB9',
    0xed84: 'dddaabf',
    0xed85: 'eeeea999',
    0xed86: 'eeeeeeb114',
    0xed87: 'eeeeeebaA8',
    0xed88: 'fffbbag',
    0xed89: 'llllllaaB9',
    0xed8a: 'eeeaabf',
    0xed8b: 'llllb989',
    0xed8c: 'hhhaaag',
    0xed8d: 'iiiia989',
    0xed8e: 'cccabag',
    0xed8f: 'iiiiiibaB9',
    0xed90: 'bbbbabf',
    0xed91: 'kkkkb889',
    0xed92: 'aaaabag',
    0xed93: 'hhhha989',
    0xed94: 'dddbbag',
    0xed95: 'mmmmb989',
    0xed96: 'gggaabf',
    0xed97: 'nnnna889',
    0xed98: 'fffaabf',
    0xed99: 'ggggb899',
    0xed9a: 'eeebbag',
    0xed9b: 'dddda999',
    0xed9c: 'hhhbbbf',
    0xed9d: 'aaaab999',
    0xed9e: 'cccbabf',
    0xed9f: 'bbbba899',
    0xeda0: 'bbbaabf',
    0xeda1: 'ccccb999',
    0xeda2: 'aaabbag',
    0xeda3: 'ppppb888',
    0xeda4: 'dddabag',
    0xeda5: 'eeeeb899',
    0xeda6: 'gggbabf',
    0xeda7: 'ffffa999',
    0xeda8: 'fffbabf',
    0xeda9: 'oooob989',
    0xedaa: 'eeeabag',
    0xedab: 'lllla889',
    0xedac: 'hhhabbf',
    0xedad: 'iiiib889',
    0xedae: 'cccaabf',
    0xedaf: 'jjjja989',
    0xedb0: 'bbbbbag',
    0xedb1: 'kkkka989',
    0xedb2: 'aaaaabf',
    0xedb3: 'hhhhb889',
    0xedb4: 'dddbabf',
    0xedb5: 'mmmma889',
    0xedb6: 'gggabag',
    0xedb7: 'nnnnb989',
    0xedb8: 'fffabag',
    0xedb9: 'gggga999',
    0xedba: 'eeebabf',
    0xedbb: 'ddddb899',
    0xedbc: 'hhhbaag',
    0xedbd: 'aaaaa899',
    0xedbe: 'cccbbag',
    0xedbf: 'bbbbb999',
    0xedc0: 'bbbabbf',
    0xedc1: 'cccca999',
    0xedc2: 'aaabaag',
    0xedc3: 'ppppa888',
    0xedc4: 'dddaaag',
    0xedc5: 'eeeea899',
    0xedc6: 'gggbbbf',
    0xedc7: 'ffffb999',
    0xedc8: 'fffbbbf',
    0xedc9: 'ooooa989',
    0xedca: 'eeeaaag',
    0xedcb: 'llllb889',
    0xedcc: 'hhhaabf',
    0xedcd: 'iiiia889',
    0xedce: 'cccabbf',
    0xedcf: 'jjjjb989',
    0xedd0: 'bbbbaag',
    0xedd1: 'kkkkb989',
    0xedd2: 'aaaabbf',
    0xedd3: 'hhhha889',
    0xedd4: 'dddbbbf',
    0xedd5: 'mmmmb889',
    0xedd6: 'gggaaag',
    0xedd7: 'nnnna989',
    0xedd8: 'fffaaag',
    0xedd9: 'ggggb999',
    0xedda: 'eeebbbf',
    0xeddb: 'dddda899',
    0xeddc: 'hhhbbag',
    0xeddd: 'aaaab899',
    0xedde: 'cccbaag',
    0xeddf: 'bbbba999',
    0xede0: 'aaaaaaaaabadC',
    0xede1: 'aaaaaaaaababD',
    0xede2: 'aaabbbf',
    0xede3: 'aaaaaaaaababL',
    0xede4: 'aaaaaaaaababP',
    0xede5: 'aaaaaaaaababT',
    0xede6: 'aaaaaaaaababX',
    0xede7: 'aaaaaaaaabahY',
    0xede8: 'aaaaaaaaabadc',
    0xede9: 'aaaaaaaaababd',
    0xedea: 'aaaaaaaaababh',
    0xedeb: 'aaaaaaaaababl',
    0xedec: 'aaaaaaaaababp',
    0xeded: 'aaaaaaaaababt',
    0xedee: 'aaaaaaaaababx',
    0xedef: 'aaaaaaaaabahy',
    0xedf0: 'aaaaaaaaabaj',
    0xedf1: 'aaaaaaaaabab',
    0xedf2: 'aaaaaag',
    0xedf3: 'aaaaaaaaabar',
    0xedf4: 'aaaaaaaaabaJ',
    0xedf5: 'aaaaaaaaabaB',
    0xedf6: 'aaaaaaaaabaZ',
    0xedf7: 'aaaaaaaaabaR',
    0xedf8: 'aaaaaaaaabE',
    0xedf9: 'aaaaaaaaabU',
    0xedfa: 'aaaaaaaaabe',
    0xedfb: 'aaaaaaaaabu',
    0xedfc: 'aaaaaaaaaH',
    0xedfd: 'aaaaa999',
    0xedfe: 'aaaaaaa114',
    0xedff: 'aaaaaaaaA8',
    0xee00: 'bbbabca',
    0xee01: 'cccca088',
    0xee02: 'aaababa',
    0xee03: 'ccccccabA8',
    0xee04: 'cccccccabaD',
    0xee05: 'cccccccabaT',
    0xee06: 'cccccccabad',
    0xee07: 'cccccccabat',
    0xee08: 'ccccccccabaJ',
    0xee09: 'ccccccccabaB',
    0xee0a: 'ccccccccabaZ',
    0xee0b: 'ccccccccabaR',
    0xee0c: 'ccccccccabaj',
    0xee0d: 'ccccccccabab',
    0xee0e: 'cccabca',
    0xee0f: 'ccccccccabar',
    0xee10: 'bbbbaba',
    0xee11: 'cccccccccabau',
    0xee12: 'aaaabca',
    0xee13: 'cccccccccabiy',
    0xee14: 'cccccccccabaa',
    0xee15: 'cccccccccabae',
    0xee16: 'cccccccccabai',
    0xee17: 'cccccccccabam',
    0xee18: 'cccccccccabaQ',
    0xee19: 'cccccccccabaU',
    0xee1a: 'cccccccccabaY',
    0xee1b: 'cccccccccabiY',
    0xee1c: 'cccccccccabaA',
    0xee1d: 'aaaab188',
    0xee1e: 'cccbaba',
    0xee1f: 'bbbba088',
    0xee20: 'bbbaaba',
    0xee21: 'ccccb188',
    0xee22: 'aaabbca',
    0xee23: 'ccccccccccabbi',
    0xee24: 'ccccccccccabbg',
    0xee25: 'ccccccccccabbe',
    0xee26: 'ccccccccccabbc',
    0xee27: 'ccccccccccabba',
    0xee28: 'ccccccccccabhz',
    0xee29: 'ccccccccccabhx',
    0xee2a: 'ccccccccccabdx',
    0xee2b: 'ccccccccccabby',
    0xee2c: 'ccccccccccabbw',
    0xee2d: 'ccccccccccabbu',
    0xee2e: 'cccaaba',
    0xee2f: 'ccccccccccabbq',
    0xee30: 'bbbbbca',
    0xee31: 'ccccccccccabbM',
    0xee32: 'aaaaaba',
    0xee33: 'ccccccccccabbI',
    0xee34: 'ccccccccccabbG',
    0xee35: 'ccccccccccabbE',
    0xee36: 'ccccccccccabbC',
    0xee37: 'ccccccccccabbA',
    0xee38: 'ccccccccccabhZ',
    0xee39: 'ccccccccccabhX',
    0xee3a: 'ccccccccccabdX',
    0xee3b: 'ccccccccccabbY',
    0xee3c: 'ccccccccccabbW',
    0xee3d: 'aaaaa088',
    0xee3e: 'cccbbca',
    0xee3f: 'bbbbb188',
    0xee40: 'bbbabba',
    0xee41: 'cccca188',
    0xee42: 'aaabaca',
    0xee43: 'cccccccccccaabd',
    0xee44: 'cccccccccccaabc',
    0xee45: 'cccccccccccaabb',
    0xee46: 'cccccccccccaaba',
    0xee47: 'cccccccccccaadc',
    0xee48: 'cccccccccccaabo',
    0xee49: 'cccccccccccaabn',
    0xee4a: 'cccccccccccaabm',
    0xee4b: 'cccccccccccaabl',
    0xee4c: 'cccccccccccaabk',
    0xee4d: 'cccccccccccaabj',
    0xee4e: 'cccabba',
    0xee4f: 'cccccccccccaabh',
    0xee50: 'bbbbaca',
    0xee51: 'cccccccccccaabv',
    0xee52: 'aaaabba',
    0xee53: 'cccccccccccaabt',
    0xee54: 'cccccccccccaabs',
    0xee55: 'cccccccccccaabr',
    0xee56: 'cccccccccccaabq',
    0xee57: 'cccccccccccaabp',
    0xee58: 'cccccccccccaahz',
    0xee59: 'cccccccccccaajz',
    0xee5a: 'cccccccccccaahx',
    0xee5b: 'cccccccccccaahy',
    0xee5c: 'cccccccccccaadx',
    0xee5d: 'aaaab088',
    0xee5e: 'cccbaca',
    0xee5f: 'bbbba188',
    0xee60: 'bbbaaca',
    0xee61: 'ccccb088',
    0xee62: 'aaabbba',
    0xee63: 'cccccccccccaabD',
    0xee64: 'cccccccccccaabC',
    0xee65: 'cccccccccccaabB',
    0xee66: 'cccccccccccaabA',
    0xee67: 'cccccccccccaadC',
    0xee68: 'cccccccccccaabO',
    0xee69: 'cccccccccccaabN',
    0xee6a: 'cccccccccccaabM',
    0xee6b: 'cccccccccccaabL',
    0xee6c: 'cccccccccccaabK',
    0xee6d: 'cccccccccccaabJ',
    0xee6e: 'cccaaca',
    0xee6f: 'cccccccccccaabH',
    0xee70: 'bbbbbba',
    0xee71: 'cccccccccccaabV',
    0xee72: 'aaaaaca',
    0xee73: 'cccccccccccaabT',
    0xee74: 'cccccccccccaabS',
    0xee75: 'cccccccccccaabR',
    0xee76: 'cccccccccccaabQ',
    0xee77: 'cccccccccccaabP',
    0xee78: 'cccccccccccaahZ',
    0xee79: 'cccccccccccaajZ',
    0xee7a: 'cccccccccccaahX',
    0xee7b: 'cccccccccccaahY',
    0xee7c: 'aaaaaababI',
    0xee7d: 'aaaaa188',
    0xee7e: 'aaaaaab104',
    0xee7f: 'aaaaaabaB9',
    0xee80: 'bbbabaa',
    0xee81: 'cccca098',
    0xee82: 'aaabadc',
    0xee83: 'ccccccbaB9',
    0xee84: 'dddaadc',
    0xee85: 'eeeea198',
    0xee86: 'eeeeeeb004',
    0xee87: 'eeeeeebbA8',
    0xee88: 'fffbbaa',
    0xee89: 'llllllabB9',
    0xee8a: 'eeeaadc',
    0xee8b: 'llllb188',
    0xee8c: 'hhhaaaa',
    0xee8d: 'iiiia188',
    0xee8e: 'cccabaa',
    0xee8f: 'iiiiiibbB9',
    0xee90: 'bbbbadc',
    0xee91: 'kkkkb088',
    0xee92: 'aaaabaa',
    0xee93: 'hhhha188',
    0xee94: 'dddbbaa',
    0xee95: 'mmmmb188',
    0xee96: 'gggaadc',
    0xee97: 'nnnna088',
    0xee98: 'fffaadc',
    0xee99: 'ggggb098',
    0xee9a: 'eeebbaa',
    0xee9b: 'dddda198',
    0xee9c: 'hhhbbdc',
    0xee9d: 'aaaab198',
    0xee9e: 'cccbadc',
    0xee9f: 'bbbba098',
    0xeea0: 'bbbaadc',
    0xeea1: 'ccccb198',
    0xeea2: 'aaabbaa',
    0xeea3: 'ppppb089',
    0xeea4: 'dddabaa',
    0xeea5: 'eeeeb098',
    0xeea6: 'gggbadc',
    0xeea7: 'ffffa198',
    0xeea8: 'fffbadc',
    0xeea9: 'oooob188',
    0xeeaa: 'eeeabaa',
    0xeeab: 'lllla088',
    0xeeac: 'hhhabdc',
    0xeead: 'iiiib088',
    0xeeae: 'cccaadc',
    0xeeaf: 'jjjja188',
    0xeeb0: 'bbbbbaa',
    0xeeb1: 'kkkka188',
    0xeeb2: 'aaaaadc',
    0xeeb3: 'hhhhb088',
    0xeeb4: 'dddbadc',
    0xeeb5: 'mmmma088',
    0xeeb6: 'gggabaa',
    0xeeb7: 'nnnnb188',
    0xeeb8: 'fffabaa',
    0xeeb9: 'gggga198',
    0xeeba: 'eeebadc',
    0xeebb: 'ddddb098',
    0xeebc: 'hhhbaaa',
    0xeebd: 'aaaaa098',
    0xeebe: 'cccbbaa',
    0xeebf: 'bbbbb198',
    0xeec0: 'bbbabdc',
    0xeec1: 'cccca198',
    0xeec2: 'aaabaaa',
    0xeec3: 'ppppa089',
    0xeec4: 'dddaaaa',
    0xeec5: 'eeeea098',
    0xeec6: 'gggbbdc',
    0xeec7: 'ffffb198',
    0xeec8: 'fffbbdc',
    0xeec9: 'ooooa188',
    0xeeca: 'eeeaaaa',
    0xeecb: 'llllb088',
    0xeecc: 'hhhaadc',
    0xeecd: 'iiiia088',
    0xeece: 'cccabdc',
    0xeecf: 'jjjjb188',
    0xeed0: 'bbbbaaa',
    0xeed1: 'kkkkb188',
    0xeed2: 'aaaabdc',
    0xeed3: 'hhhha088',
    0xeed4: 'dddbbdc',
    0xeed5: 'mmmmb088',
    0xeed6: 'gggaaaa',
    0xeed7: 'nnnna188',
    0xeed8: 'fffaaaa',
    0xeed9: 'ggggb198',
    0xeeda: 'eeebbdc',
    0xeedb: 'dddda098',
    0xeedc: 'hhhbbaa',
    0xeedd: 'aaaab098',
    0xeede: 'cccbaaa',
    0xeedf: 'bbbba198',
    0xeee0: 'bbbaaaa',
    0xeee1: 'ccccb098',
    0xeee2: 'aaabbdc',
    0xeee3: 'ppppb189',
    0xeee4: 'dddabdc',
    0xeee5: 'eeeeb198',
    0xeee6: 'gggbaaa',
    0xeee7: 'ffffa098',
    0xeee8: 'fffbaaa',
    0xeee9: 'oooob088',
    0xeeea: 'eeeabdc',
    0xeeeb: 'lllla188',
    0xeeec: 'hhhabaa',
    0xeeed: 'iiiib188',
    0xeeee: 'cccaaaa',
    0xeeef: 'jjjja088',
    0xeef0: 'bbbbbdc',
    0xeef1: 'hhhhhhbbB9',
    0xeef2: 'aaaaaaa',
    0xeef3: 'hhhhb188',
    0xeef4: 'dddbaaa',
    0xeef5: 'hhhhhhhaaat',
    0xeef6: 'gggabdc',
    0xeef7: 'hhhhhhhaaaT',
    0xeef8: 'aaaaaaababE',
    0xeef9: 'aaaaaaababU',
    0xeefa: 'aaaaaaababe',
    0xeefb: 'aaaaaaababu',
    0xeefc: 'aaaaaaab1',
    0xeefd: 'aaaaa198',
    0xeefe: 'aaaaaaa004',
    0xeeff: 'aaaaaaabA8',
    0xef00: 'bbbabab',
    0xef01: 'cccca089',
    0xef02: 'aaababc',
    0xef03: 'ccccccaaB9',
    0xef04: 'cccccccbabE',
    0xef05: 'cccccccbabU',
    0xef06: 'cccccccbabe',
    0xef07: 'cccccccbabu',
    0xef08: 'fffbbab',
    0xef09: 'iiiiiiibbbU',
    0xef0a: 'eeeaabc',
    0xef0b: 'iiiiiiibbbu',
    0xef0c: 'hhhaaab',
    0xef0d: 'iiiia199',
    0xef0e: 'cccabab',
    0xef0f: 'iiiiiiabB9',
    0xef10: 'bbbbabc',
    0xef11: 'kkkkb099',
    0xef12: 'aaaabab',
    0xef13: 'hhhha199',
    0xef14: 'dddbbab',
    0xef15: 'mmmmb199',
    0xef16: 'gggaabc',
    0xef17: 'nnnna099',
    0xef18: 'fffaabc',
    0xef19: 'ggggb089',
    0xef1a: 'eeebbab',
    0xef1b: 'dddda189',
    0xef1c: 'hhhbbbc',
    0xef1d: 'aaaab189',
    0xef1e: 'cccbabc',
    0xef1f: 'bbbba089',
    0xef20: 'bbbaabc',
    0xef21: 'ccccb189',
    0xef22: 'aaabbab',
    0xef23: 'ppppb098',
    0xef24: 'dddabab',
    0xef25: 'eeeeb089',
    0xef26: 'gggbabc',
    0xef27: 'ffffa189',
    0xef28: 'fffbabc',
    0xef29: 'oooob199',
    0xef2a: 'eeeabab',
    0xef2b: 'lllla099',
    0xef2c: 'hhhabbc',
    0xef2d: 'iiiib099',
    0xef2e: 'cccaabc',
    0xef2f: 'jjjja199',
    0xef30: 'bbbbbab',
    0xef31: 'kkkka199',
    0xef32: 'aaaaabc',
    0xef33: 'hhhhb099',
    0xef34: 'dddbabc',
    0xef35: 'mmmma099',
    0xef36: 'gggabab',
    0xef37: 'nnnnb199',
    0xef38: 'fffabab',
    0xef39: 'gggga189',
    0xef3a: 'eeebabc',
    0xef3b: 'ddddb089',
    0xef3c: 'hhhbaab',
    0xef3d: 'aaaaa089',
    0xef3e: 'cccbbab',
    0xef3f: 'bbbbb189',
    0xef40: 'bbbabbc',
    0xef41: 'cccca189',
    0xef42: 'aaabaab',
    0xef43: 'ppppa098',
    0xef44: 'dddaaab',
    0xef45: 'eeeea089',
    0xef46: 'gggbbbc',
    0xef47: 'ffffb189',
    0xef48: 'fffbbbc',
    0xef49: 'ooooa199',
    0xef4a: 'eeeaaab',
    0xef4b: 'llllb099',
    0xef4c: 'hhhaabc',
    0xef4d: 'iiiia099',
    0xef4e: 'cccabbc',
    0xef4f: 'jjjjb199',
    0xef50: 'bbbbaab',
    0xef51: 'kkkkb199',
    0xef52: 'aaaabbc',
    0xef53: 'hhhha099',
    0xef54: 'dddbbbc',
    0xef55: 'mmmmb099',
    0xef56: 'gggaaab',
    0xef57: 'nnnna199',
    0xef58: 'fffaaab',
    0xef59: 'ggggb189',
    0xef5a: 'eeebbbc',
    0xef5b: 'dddda089',
    0xef5c: 'hhhbbab',
    0xef5d: 'aaaab089',
    0xef5e: 'cccbaab',
    0xef5f: 'bbbba189',
    0xef60: 'bbbaaab',
    0xef61: 'ccccb089',
    0xef62: 'aaabbbc',
    0xef63: 'ppppb198',
    0xef64: 'dddabbc',
    0xef65: 'eeeeb189',
    0xef66: 'gggbaab',
    0xef67: 'ffffa089',
    0xef68: 'fffbaab',
    0xef69: 'oooob099',
    0xef6a: 'eeeabbc',
    0xef6b: 'lllla199',
    0xef6c: 'hhhabab',
    0xef6d: 'iiiib199',
    0xef6e: 'cccaaab',
    0xef6f: 'jjjja099',
    0xef70: 'bbbbbbc',
    0xef71: 'hhhhhhabB9',
    0xef72: 'aaaaaab',
    0xef73: 'hhhhb199',
    0xef74: 'dddbaab',
    0xef75: 'mmmma199',
    0xef76: 'gggabbc',
    0xef77: 'mmmmmmbbB9',
    0xef78: 'dddddda004',
    0xef79: 'ddddddabA8',
    0xef7a: 'ddddddab1',
    0xef7b: 'ddddb189',
    0xef7c: 'aaaaaabbaH',
    0xef7d: 'aaaaa189',
    0xef7e: 'aaaaaab004',
    0xef7f: 'aaaaaabbA8',
    0xef80: 'bbbabac',
    0xef81: 'cccca099',
    0xef82: 'aaababb',
    0xef83: 'ccccccbbA8',
    0xef84: 'dddaabb',
    0xef85: 'eeeea199',
    0xef86: 'eeeeeeb104',
    0xef87: 'eeeeeebaB9',
    0xef88: 'fffbbac',
    0xef89: 'llllllaaA8',
    0xef8a: 'eeeaabb',
    0xef8b: 'llllb189',
    0xef8c: 'hhhaaac',
    0xef8d: 'iiiia189',
    0xef8e: 'cccabac',
    0xef8f: 'iiiiiibaA8',
    0xef90: 'bbbbabb',
    0xef91: 'kkkkb089',
    0xef92: 'aaaabac',
    0xef93: 'hhhha189',
    0xef94: 'dddbbac',
    0xef95: 'mmmmb189',
    0xef96: 'gggaabb',
    0xef97: 'nnnna089',
    0xef98: 'fffaabb',
    0xef99: 'ggggb099',
    0xef9a: 'eeebbac',
    0xef9b: 'dddda199',
    0xef9c: 'hhhbbbb',
    0xef9d: 'aaaab199',
    0xef9e: 'cccbabb',
    0xef9f: 'bbbba099',
    0xefa0: 'bbbaabb',
    0xefa1: 'ccccb199',
    0xefa2: 'aaabbac',
    0xefa3: 'ppppb088',
    0xefa4: 'dddabac',
    0xefa5: 'eeeeb099',
    0xefa6: 'gggbabb',
    0xefa7: 'ffffa199',
    0xefa8: 'fffbabb',
    0xefa9: 'oooob189',
    0xefaa: 'eeeabac',
    0xefab: 'lllla089',
    0xefac: 'hhhabbb',
    0xefad: 'iiiib089',
    0xefae: 'cccaabb',
    0xefaf: 'jjjja189',
    0xefb0: 'bbbbbac',
    0xefb1: 'kkkka189',
    0xefb2: 'aaaaabb',
    0xefb3: 'hhhhb089',
    0xefb4: 'dddbabb',
    0xefb5: 'mmmma089',
    0xefb6: 'gggabac',
    0xefb7: 'nnnnb189',
    0xefb8: 'fffabac',
    0xefb9: 'gggga199',
    0xefba: 'eeebabb',
    0xefbb: 'ddddb099',
    0xefbc: 'hhhbaac',
    0xefbd: 'aaaaa099',
    0xefbe: 'cccbbac',
    0xefbf: 'bbbbb199',
    0xefc0: 'bbbabbb',
    0xefc1: 'cccca199',
    0xefc2: 'aaabaac',
    0xefc3: 'ppppa088',
    0xefc4: 'dddaaac',
    0xefc5: 'eeeea099',
    0xefc6: 'gggbbbb',
    0xefc7: 'ffffb199',
    0xefc8: 'fffbbbb',
    0xefc9: 'ooooa189',
    0xefca: 'eeeaaac',
    0xefcb: 'llllb089',
    0xefcc: 'hhhaabb',
    0xefcd: 'iiiia089',
    0xefce: 'cccabbb',
    0xefcf: 'jjjjb189',
    0xefd0: 'bbbbaac',
    0xefd1: 'kkkkb189',
    0xefd2: 'aaaabbb',
    0xefd3: 'hhhha089',
    0xefd4: 'dddbbbb',
    0xefd5: 'mmmmb089',
    0xefd6: 'gggaaac',
    0xefd7: 'nnnna189',
    0xefd8: 'fffaaac',
    0xefd9: 'ggggb199',
    0xefda: 'eeebbbb',
    0xefdb: 'dddda099',
    0xefdc: 'hhhbbac',
    0xefdd: 'aaaab099',
    0xefde: 'cccbaac',
    0xefdf: 'bbbba199',
    0xefe0: 'bbbaaac',
    0xefe1: 'ccccb099',
    0xefe2: 'aaabbbb',
    0xefe3: 'ppppb188',
    0xefe4: 'dddabbb',
    0xefe5: 'eeeeb199',
    0xefe6: 'gggbaac',
    0xefe7: 'ffffa099',
    0xefe8: 'fffbaac',
    0xefe9: 'oooob089',
    0xefea: 'eeeabbb',
    0xefeb: 'lllla189',
    0xefec: 'hhhabac',
    0xefed: 'iiiib189',
    0xefee: 'cccaaac',
    0xefef: 'jjjja089',
    0xeff0: 'aaaaaaaababk',
    0xeff1: 'aaaaaaaababc',
    0xeff2: 'aaaaaac',
    0xeff3: 'aaaaaaaababs',
    0xeff4: 'aaaaaaaababK',
    0xeff5: 'aaaaaaaababC',
    0xeff6: 'aaaaaaaabadX',
    0xeff7: 'aaaaaaaababS',
    0xeff8: 'aaaaaaaabaD',
    0xeff9: 'aaaaaaaabaT',
    0xeffa: 'aaaaaaaabad',
    0xeffb: 'aaaaaaaabat',
    0xeffc: 'aaaaaaaabI',
    0xeffd: 'aaaaa199',
    0xeffe: 'aaaaaaa104',
    0xefff: 'aaaaaaaaB9',
    0xf000: 'bbbabiX',
    0xf001: 'ccccab16',
    0xf002: 'aaabahX',
    0xf003: 'cccccccaM9',
    0xf004: 'cccccccaaaG',
    0xf005: 'cccccccaaaW',
    0xf006: 'cccccccaaag',
    0xf007: 'cccccccaaaw',
    0xf008: 'ccccccccbbbJ',
    0xf009: 'ccccccccbbbB',
    0xf00a: 'ccccccccbbbZ',
    0xf00b: 'ccccccccbbbR',
    0xf00c: 'ccccccccbbbj',
    0xf00d: 'ccccccccbbbb',
    0xf00e: 'cccabiX',
    0xf00f: 'ccccccccbbbr',
    0xf010: 'bbbbahX',
    0xf011: 'kkkkbb06',
    0xf012: 'aaaabiX',
    0xf013: 'hhhhaa16',
    0xf014: 'dddbbiX',
    0xf015: 'mmmmba16',
    0xf016: 'gggaahX',
    0xf017: 'nnnnab06',
    0xf018: 'fffaahX',
    0xf019: 'ggggbb16',
    0xf01a: 'eeebbiX',
    0xf01b: 'ddddaa06',
    0xf01c: 'hhhbbhX',
    0xf01d: 'aaaaba06',
    0xf01e: 'cccbahX',
    0xf01f: 'bbbbab16',
    0xf020: 'bbbaahX',
    0xf021: 'ccccba06',
    0xf022: 'aaabbiX',
    0xf023: 'ppppbb07',
    0xf024: 'dddabiX',
    0xf025: 'eeeebb16',
    0xf026: 'gggbahX',
    0xf027: 'ffffaa06',
    0xf028: 'fffbahX',
    0xf029: 'ooooba16',
    0xf02a: 'eeeabiX',
    0xf02b: 'llllab06',
    0xf02c: 'hhhabhX',
    0xf02d: 'iiiibb06',
    0xf02e: 'cccaahX',
    0xf02f: 'jjjjaa16',
    0xf030: 'bbbbbiX',
    0xf031: 'kkkkaa16',
    0xf032: 'aaaaahX',
    0xf033: 'hhhhbb06',
    0xf034: 'dddbahX',
    0xf035: 'mmmmab06',
    0xf036: 'gggabiX',
    0xf037: 'nnnnba16',
    0xf038: 'fffabiX',
    0xf039: 'ggggaa06',
    0xf03a: 'eeebahX',
    0xf03b: 'ddddbb16',
    0xf03c: 'hhhbaiX',
    0xf03d: 'aaaaab16',
    0xf03e: 'cccbbiX',
    0xf03f: 'bbbbba06',
    0xf040: 'bbbabhX',
    0xf041: 'ccccaa06',
    0xf042: 'aaabaiX',
    0xf043: 'ppppab07',
    0xf044: 'dddaaiX',
    0xf045: 'eeeeab16',
    0xf046: 'gggbbhX',
    0xf047: 'ffffba06',
    0xf048: 'fffbbhX',
    0xf049: 'ooooaa16',
    0xf04a: 'eeeaaiX',
    0xf04b: 'llllbb06',
    0xf04c: 'hhhaahX',
    0xf04d: 'iiiiab06',
    0xf04e: 'cccabhX',
    0xf04f: 'jjjjba16',
    0xf050: 'bbbbaiX',
    0xf051: 'kkkkba16',
    0xf052: 'aaaabhX',
    0xf053: 'hhhhab06',
    0xf054: 'dddbbhX',
    0xf055: 'mmmmbb06',
    0xf056: 'gggaaiX',
    0xf057: 'nnnnaa16',
    0xf058: 'fffaaiX',
    0xf059: 'ggggba06',
    0xf05a: 'eeebbhX',
    0xf05b: 'ddddab16',
    0xf05c: 'hhhbbiX',
    0xf05d: 'aaaabb16',
    0xf05e: 'cccbaiX',
    0xf05f: 'bbbbaa06',
    0xf060: 'bbbaaiX',
    0xf061: 'ccccbb16',
    0xf062: 'aaabbhX',
    0xf063: 'ppppba17',
    0xf064: 'dddabhX',
    0xf065: 'eeeeba06',
    0xf066: 'gggbaiX',
    0xf067: 'ffffab16',
    0xf068: 'fffbaiX',
    0xf069: 'oooobb06',
    0xf06a: 'eeeabhX',
    0xf06b: 'llllaa16',
    0xf06c: 'hhhabiX',
    0xf06d: 'iiiiba16',
    0xf06e: 'cccaaiX',
    0xf06f: 'jjjjab06',
    0xf070: 'bbbbbhX',
    0xf071: 'kkkkab06',
    0xf072: 'aaaaaiX',
    0xf073: 'hhhhba16',
    0xf074: 'dddbaiX',
    0xf075: 'mmmmaa16',
    0xf076: 'gggabhX',
    0xf077: 'nnnnbb06',
    0xf078: 'dddddda113',
    0xf079: 'ggggab16',
    0xf07a: 'ddddddam9',
    0xf07b: 'ddddba06',
    0xf07c: 'aaaaaabm9',
    0xf07d: 'aaaaaa06',
    0xf07e: 'aaaaaab113',
    0xf07f: 'bbbbbb16',
    0xf080: 'bbbabiY',
    0xf081: 'ccccab06',
    0xf082: 'aaabahY',
    0xf083: 'ppppaa07',
    0xf084: 'dddaahY',
    0xf085: 'eeeeaa16',
    0xf086: 'eeeeeeb013',
    0xf087: 'ffffbb06',
    0xf088: 'fffbbiY',
    0xf089: 'ooooab16',
    0xf08a: 'eeeaahY',
    0xf08b: 'llllba06',
    0xf08c: 'hhhaaiY',
    0xf08d: 'iiiiaa06',
    0xf08e: 'cccabiY',
    0xf08f: 'jjjjbb16',
    0xf090: 'bbbbahY',
    0xf091: 'kkkkbb16',
    0xf092: 'aaaabiY',
    0xf093: 'hhhhaa06',
    0xf094: 'dddbbiY',
    0xf095: 'mmmmba06',
    0xf096: 'gggaahY',
    0xf097: 'nnnnab16',
    0xf098: 'fffaahY',
    0xf099: 'ggggbb06',
    0xf09a: 'eeebbiY',
    0xf09b: 'ddddaa16',
    0xf09c: 'hhhbbhY',
    0xf09d: 'aaaaba16',
    0xf09e: 'cccbahY',
    0xf09f: 'bbbbab06',
    0xf0a0: 'bbbaahY',
    0xf0a1: 'ccccba16',
    0xf0a2: 'aaabbiY',
    0xf0a3: 'ppppbb17',
    0xf0a4: 'dddabiY',
    0xf0a5: 'eeeebb06',
    0xf0a6: 'gggbahY',
    0xf0a7: 'ffffaa16',
    0xf0a8: 'fffbahY',
    0xf0a9: 'ooooba06',
    0xf0aa: 'eeeabiY',
    0xf0ab: 'llllab16',
    0xf0ac: 'hhhabhY',
    0xf0ad: 'iiiibb16',
    0xf0ae: 'cccaahY',
    0xf0af: 'jjjjaa06',
    0xf0b0: 'bbbbbiY',
    0xf0b1: 'kkkkaa06',
    0xf0b2: 'aaaaahY',
    0xf0b3: 'hhhhbb16',
    0xf0b4: 'dddbahY',
    0xf0b5: 'mmmmab16',
    0xf0b6: 'gggabiY',
    0xf0b7: 'nnnnba06',
    0xf0b8: 'fffabiY',
    0xf0b9: 'ggggaa16',
    0xf0ba: 'eeebahY',
    0xf0bb: 'ddddbb06',
    0xf0bc: 'hhhbaiY',
    0xf0bd: 'aaaaab06',
    0xf0be: 'cccbbiY',
    0xf0bf: 'bbbbba16',
    0xf0c0: 'bbbabhY',
    0xf0c1: 'ccccaa16',
    0xf0c2: 'aaabaiY',
    0xf0c3: 'ppppab17',
    0xf0c4: 'dddaaiY',
    0xf0c5: 'eeeeab06',
    0xf0c6: 'gggbbhY',
    0xf0c7: 'ffffba16',
    0xf0c8: 'fffbbhY',
    0xf0c9: 'ooooaa06',
    0xf0ca: 'eeeaaiY',
    0xf0cb: 'llllbb16',
    0xf0cc: 'hhhaahY',
    0xf0cd: 'iiiiab16',
    0xf0ce: 'cccabhY',
    0xf0cf: 'jjjjba06',
    0xf0d0: 'bbbbaiY',
    0xf0d1: 'kkkkba06',
    0xf0d2: 'aaaabhY',
    0xf0d3: 'hhhhab16',
    0xf0d4: 'dddbbhY',
    0xf0d5: 'mmmmbb16',
    0xf0d6: 'gggaaiY',
    0xf0d7: 'nnnnaa06',
    0xf0d8: 'fffaaiY',
    0xf0d9: 'ggggba16',
    0xf0da: 'eeebbhY',
    0xf0db: 'ddddab06',
    0xf0dc: 'hhhbbiY',
    0xf0dd: 'aaaabb06',
    0xf0de: 'cccbaiY',
    0xf0df: 'bbbbaa16',
    0xf0e0: 'bbbaaiY',
    0xf0e1: 'ccccbb06',
    0xf0e2: 'aaabbhY',
    0xf0e3: 'ppppba07',
    0xf0e4: 'dddabhY',
    0xf0e5: 'eeeeba16',
    0xf0e6: 'gggbaiY',
    0xf0e7: 'ffffab06',
    0xf0e8: 'fffbaiY',
    0xf0e9: 'oooobb16',
    0xf0ea: 'eeeabhY',
    0xf0eb: 'llllaa06',
    0xf0ec: 'hhhabiY',
    0xf0ed: 'iiiiba06',
    0xf0ee: 'cccaaiY',
    0xf0ef: 'jjjjab16',
    0xf0f0: 'bbbbbhY',
    0xf0f1: 'hhhhhhhaL8',
    0xf0f2: 'aaaaaiY',
    0xf0f3: 'hhhhba06',
    0xf0f4: 'dddbaiY',
    0xf0f5: 'hhhhhhhabaw',
    0xf0f6: 'gggabhY',
    0xf0f7: 'hhhhhhhabaW',
    0xf0f8: 'aaaaaaabbbF',
    0xf0f9: 'aaaaaaabbbV',
    0xf0fa: 'aaaaaaabbbf',
    0xf0fb: 'aaaaaaabbbv',
    0xf0fc: 'aaaaaaal9',
    0xf0fd: 'aaaaaa16',
    0xf0fe: 'aaaaaaa013',
    0xf0ff: 'aaaaaaabL9',
    0xf100: 'bbbabiZ',
    0xf101: 'ccccab17',
    0xf102: 'aaabahZ',
    0xf103: 'cccccccbL9',
    0xf104: 'cccccccbbbF',
    0xf105: 'cccccccbbbV',
    0xf106: 'cccccccbbbf',
    0xf107: 'cccccccbbbv',
    0xf108: 'fffbbiZ',
    0xf109: 'iiiiiiibabV',
    0xf10a: 'eeeaahZ',
    0xf10b: 'iiiiiiibabv',
    0xf10c: 'hhhaaiZ',
    0xf10d: 'iiiiaa17',
    0xf10e: 'cccabiZ',
    0xf10f: 'iiiiiiibM8',
    0xf110: 'bbbbahZ',
    0xf111: 'kkkkbb07',
    0xf112: 'aaaabiZ',
    0xf113: 'hhhhaa17',
    0xf114: 'dddbbiZ',
    0xf115: 'mmmmba17',
    0xf116: 'gggaahZ',
    0xf117: 'nnnnab07',
    0xf118: 'fffaahZ',
    0xf119: 'ggggbb17',
    0xf11a: 'eeebbiZ',
    0xf11b: 'ddddaa07',
    0xf11c: 'hhhbbhZ',
    0xf11d: 'aaaaba07',
    0xf11e: 'cccbahZ',
    0xf11f: 'bbbbab17',
    0xf120: 'bbbaahZ',
    0xf121: 'ccccba07',
    0xf122: 'aaabbiZ',
    0xf123: 'ppppbb06',
    0xf124: 'dddabiZ',
    0xf125: 'eeeebb17',
    0xf126: 'gggbahZ',
    0xf127: 'ffffaa07',
    0xf128: 'fffbahZ',
    0xf129: 'ooooba17',
    0xf12a: 'eeeabiZ',
    0xf12b: 'llllab07',
    0xf12c: 'hhhabhZ',
    0xf12d: 'iiiibb07',
    0xf12e: 'cccaahZ',
    0xf12f: 'jjjjaa17',
    0xf130: 'bbbbbiZ',
    0xf131: 'kkkkaa17',
    0xf132: 'aaaaahZ',
    0xf133: 'hhhhbb07',
    0xf134: 'dddbahZ',
    0xf135: 'mmmmab07',
    0xf136: 'gggabiZ',
    0xf137: 'nnnnba17',
    0xf138: 'fffabiZ',
    0xf139: 'ggggaa07',
    0xf13a: 'eeebahZ',
    0xf13b: 'ddddbb17',
    0xf13c: 'hhhbaiZ',
    0xf13d: 'aaaaab17',
    0xf13e: 'cccbbiZ',
    0xf13f: 'bbbbba07',
    0xf140: 'bbbabhZ',
    0xf141: 'ccccaa07',
    0xf142: 'aaabaiZ',
    0xf143: 'ppppab06',
    0xf144: 'dddaaiZ',
    0xf145: 'eeeeab17',
    0xf146: 'gggbbhZ',
    0xf147: 'ffffba07',
    0xf148: 'fffbbhZ',
    0xf149: 'ooooaa17',
    0xf14a: 'eeeaaiZ',
    0xf14b: 'llllbb07',
    0xf14c: 'hhhaahZ',
    0xf14d: 'iiiiab07',
    0xf14e: 'cccabhZ',
    0xf14f: 'jjjjba17',
    0xf150: 'bbbbaiZ',
    0xf151: 'kkkkba17',
    0xf152: 'aaaabhZ',
    0xf153: 'hhhhab07',
    0xf154: 'dddbbhZ',
    0xf155: 'mmmmbb07',
    0xf156: 'gggaaiZ',
    0xf157: 'nnnnaa17',
    0xf158: 'fffaaiZ',
    0xf159: 'ggggba07',
    0xf15a: 'eeebbhZ',
    0xf15b: 'ddddab17',
    0xf15c: 'hhhbbiZ',
    0xf15d: 'aaaabb17',
    0xf15e: 'cccbaiZ',
    0xf15f: 'bbbbaa07',
    0xf160: 'bbbaaiZ',
    0xf161: 'ccccbb17',
    0xf162: 'aaabbhZ',
    0xf163: 'ppppba16',
    0xf164: 'dddabhZ',
    0xf165: 'eeeeba07',
    0xf166: 'gggbaiZ',
    0xf167: 'ffffab17',
    0xf168: 'fffbaiZ',
    0xf169: 'oooobb07',
    0xf16a: 'eeeabhZ',
    0xf16b: 'llllaa17',
    0xf16c: 'hhhabiZ',
    0xf16d: 'iiiiba17',
    0xf16e: 'cccaaiZ',
    0xf16f: 'jjjjab07',
    0xf170: 'bbbbbhZ',
    0xf171: 'kkkkab07',
    0xf172: 'aaaaaiZ',
    0xf173: 'hhhhba17',
    0xf174: 'dddbaiZ',
    0xf175: 'mmmmaa17',
    0xf176: 'gggabhZ',
    0xf177: 'nnnnbb07',
    0xf178: 'dddddda013',
    0xf179: 'ggggab17',
    0xf17a: 'ddddddal9',
    0xf17b: 'ddddba07',
    0xf17c: 'aaaaaabl9',
    0xf17d: 'aaaaaa07',
    0xf17e: 'aaaaaab013',
    0xf17f: 'bbbbbb17',
    0xf180: 'aaaaaaaaaaaaadX',
    0xf181: 'aaaaaaaaaaaaabZ',
    0xf182: 'aaabajZ',
    0xf183: 'aaaaaaaaaaaaabX',
    0xf184: 'aaaaaaaaaaaaahZ',
    0xf185: 'aaaaaaaaaaaaajZ',
    0xf186: 'aaaaaaaaaaaaahX',
    0xf187: 'aaaaaaaaaaaaahY',
    0xf188: 'aaaaaaaaaaaaabS',
    0xf189: 'aaaaaaaaaaaaabR',
    0xf18a: 'aaaaaaaaaaaaabQ',
    0xf18b: 'aaaaaaaaaaaaabP',
    0xf18c: 'aaaaaaaaaaaaabW',
    0xf18d: 'aaaaaaaaaaaaabV',
    0xf18e: 'aaaaaaaaaaaaabU',
    0xf18f: 'aaaaaaaaaaaaabT',
    0xf190: 'aaaaaaaaaaaaabK',
    0xf191: 'aaaaaaaaaaaaabJ',
    0xf192: 'aaaabkZ',
    0xf193: 'aaaaaaaaaaaaabH',
    0xf194: 'aaaaaaaaaaaaabO',
    0xf195: 'aaaaaaaaaaaaabN',
    0xf196: 'aaaaaaaaaaaaabM',
    0xf197: 'aaaaaaaaaaaaabL',
    0xf198: 'aaaaaaaaaaaaabC',
    0xf199: 'aaaaaaaaaaaaabB',
    0xf19a: 'aaaaaaaaaaaaabA',
    0xf19b: 'aaaaaaaaaaaaadC',
    0xf19c: 'aaaaaaaaaaaaabG',
    0xf19d: 'aaaaba17',
    0xf19e: 'aaaaaaaaaaaaabE',
    0xf19f: 'aaaaaaaaaaaaabD',
    0xf1a0: 'aaaaaaaaaaaaadx',
    0xf1a1: 'aaaaaaaaaaaaabz',
    0xf1a2: 'aaabbkZ',
    0xf1a3: 'aaaaaaaaaaaaabx',
    0xf1a4: 'aaaaaaaaaaaaahz',
    0xf1a5: 'aaaaaaaaaaaaajz',
    0xf1a6: 'aaaaaaaaaaaaahx',
    0xf1a7: 'aaaaaaaaaaaaahy',
    0xf1a8: 'aaaaaaaaaaaaabs',
    0xf1a9: 'aaaaaaaaaaaaabr',
    0xf1aa: 'aaaaaaaaaaaaabq',
    0xf1ab: 'aaaaaaaaaaaaabp',
    0xf1ac: 'aaaaaaaaaaaaabw',
    0xf1ad: 'aaaaaaaaaaaaabv',
    0xf1ae: 'aaaaaaaaaaaaabu',
    0xf1af: 'aaaaaaaaaaaaabt',
    0xf1b0: 'aaaaaaaaaaaaabk',
    0xf1b1: 'aaaaaaaaaaaaabj',
    0xf1b2: 'aaaaajZ',
    0xf1b3: 'aaaaaaaaaaaaabh',
    0xf1b4: 'aaaaaaaaaaaaabo',
    0xf1b5: 'aaaaaaaaaaaaabn',
    0xf1b6: 'aaaaaaaaaaaaabm',
    0xf1b7: 'aaaaaaaaaaaaabl',
    0xf1b8: 'aaaaaaaaaaaaabc',
    0xf1b9: 'aaaaaaaaaaaaabb',
    0xf1ba: 'aaaaaaaaaaaaaba',
    0xf1bb: 'aaaaaaaaaaaaadc',
    0xf1bc: 'aaaaaaaaaaaaabg',
    0xf1bd: 'aaaaab07',
    0xf1be: 'aaaaaaaaaaaaabe',
    0xf1bf: 'aaaaaaaaaaaaabd',
    0xf1c0: 'aaaaaaaaaaaaaV',
    0xf1c1: 'aaaaaaaaaaaaaT',
    0xf1c2: 'aaabakZ',
    0xf1c3: 'aaaaaaaaaaaaaP',
    0xf1c4: 'aaaaaaaaaaaaiZ',
    0xf1c5: 'aaaaaaaaaaaaiX',
    0xf1c6: 'aaaaaaaaaaaaaZ',
    0xf1c7: 'aaaaaaaaaaaaaX',
    0xf1c8: 'aaaaaaaaaaaaaF',
    0xf1c9: 'aaaaaaaaaaaaaD',
    0xf1ca: 'aaaaaaaaaaaaaB',
    0xf1cb: 'aaaaaaaaaaaacA',
    0xf1cc: 'aaaaaaaaaaaaaN',
    0xf1cd: 'aaaaaaaaaaaaaL',
    0xf1ce: 'aaaaaaaaaaaaaJ',
    0xf1cf: 'aaaaaaaaaaaaaH',
    0xf1d0: 'aaaaaaaaaaaaav',
    0xf1d1: 'aaaaaaaaaaaaat',
    0xf1d2: 'aaaabjZ',
    0xf1d3: 'aaaaaaaaaaaaap',
    0xf1d4: 'aaaaaaaaaaaaiz',
    0xf1d5: 'aaaaaaaaaaaaix',
    0xf1d6: 'aaaaaaaaaaaaaz',
    0xf1d7: 'aaaaaaaaaaaaax',
    0xf1d8: 'aaaaaaaaaaaaaf',
    0xf1d9: 'aaaaaaaaaaaaad',
    0xf1da: 'aaaaaaaaaaaaab',
    0xf1db: 'aaaaaaaaaaaaca',
    0xf1dc: 'aaaaaaaaaaaaan',
    0xf1dd: 'aaaabb07',
    0xf1de: 'aaaaaaaaaaaaaj',
    0xf1df: 'aaaaaaaaaaaaah',
    0xf1e0: 'aaaaaaaaaaaaA',
    0xf1e1: 'aaaaaaaaaaaaE',
    0xf1e2: 'aaabbjZ',
    0xf1e3: 'aaaaaaaaaaaaM',
    0xf1e4: 'aaaaaaaaaaaaQ',
    0xf1e5: 'aaaaaaaaaaaaU',
    0xf1e6: 'aaaaaaaaaaaaY',
    0xf1e7: 'aaaaaaaaaaaiY',
    0xf1e8: 'aaaaaaaaaaaaa',
    0xf1e9: 'aaaaaaaaaaaae',
    0xf1ea: 'aaaaaaaaaaaai',
    0xf1eb: 'aaaaaaaaaaaam',
    0xf1ec: 'aaaaaaaaaaaaq',
    0xf1ed: 'aaaaaaaaaaaau',
    0xf1ee: 'aaaaaaaaaaaay',
    0xf1ef: 'aaaaaaaaaaaiy',
    0xf1f0: 'aaaaaaaaaaak',
    0xf1f1: 'aaaaaaaaaaac',
    0xf1f2: 'aaaaakZ',
    0xf1f3: 'aaaaaaaaaaas',
    0xf1f4: 'aaaaaaaaaaaK',
    0xf1f5: 'aaaaaaaaaaaC',
    0xf1f6: 'aaaaaaaaaacZ',
    0xf1f7: 'aaaaaaaaaaaS',
    0xf1f8: 'aaaaaaaaaaG',
    0xf1f9: 'aaaaaaaaaaW',
    0xf1fa: 'aaaaaaaaaag',
    0xf1fb: 'aaaaaaaaaaw',
    0xf1fc: 'aaaaaaam9',
    0xf1fd: 'aaaaaa17',
    0xf1fe: 'aaaaaaa113',
    0xf1ff: 'aaaaaaaaM9',
    0xf200: 'bbbabaX',
    0xf201: 'ccccab14',
    0xf202: 'aaababY',
    0xf203: 'cccccccaL9',
    0xf204: 'cccccccabbF',
    0xf205: 'cccccccabbV',
    0xf206: 'cccccccabbf',
    0xf207: 'cccccccabbv',
    0xf208: 'ccccccccaaaK',
    0xf209: 'ccccccccaaaC',
    0xf20a: 'ccccccccaacZ',
    0xf20b: 'ccccccccaaaS',
    0xf20c: 'ccccccccaaak',
    0xf20d: 'ccccccccaaac',
    0xf20e: 'cccabaX',
    0xf20f: 'ccccccccaaas',
    0xf210: 'bbbbabY',
    0xf211: 'cccccccccbbbt',
    0xf212: 'aaaabaX',
    0xf213: 'cccccccccbbhy',
    0xf214: 'cccccccccbbdc',
    0xf215: 'cccccccccbbbd',
    0xf216: 'cccccccccbbbh',
    0xf217: 'cccccccccbbbl',
    0xf218: 'cccccccccbbbP',
    0xf219: 'cccccccccbbbT',
    0xf21a: 'cccccccccbbbX',
    0xf21b: 'cccccccccbbhY',
    0xf21c: 'cccccccccbbdC',
    0xf21d: 'aaaaba04',
    0xf21e: 'cccbabY',
    0xf21f: 'bbbbab14',
    0xf220: 'bbbaabY',
    0xf221: 'ccccba04',
    0xf222: 'aaabbaX',
    0xf223: 'ppppbb05',
    0xf224: 'dddabaX',
    0xf225: 'eeeebb14',
    0xf226: 'gggbabY',
    0xf227: 'ffffaa04',
    0xf228: 'fffbabY',
    0xf229: 'ooooba14',
    0xf22a: 'eeeabaX',
    0xf22b: 'llllab04',
    0xf22c: 'hhhabbY',
    0xf22d: 'iiiibb04',
    0xf22e: 'cccaabY',
    0xf22f: 'jjjjaa14',
    0xf230: 'bbbbbaX',
    0xf231: 'kkkkaa14',
    0xf232: 'aaaaabY',
    0xf233: 'hhhhbb04',
    0xf234: 'dddbabY',
    0xf235: 'mmmmab04',
    0xf236: 'gggabaX',
    0xf237: 'nnnnba14',
    0xf238: 'fffabaX',
    0xf239: 'ggggaa04',
    0xf23a: 'eeebabY',
    0xf23b: 'ddddbb14',
    0xf23c: 'hhhbaaX',
    0xf23d: 'aaaaab14',
    0xf23e: 'cccbbaX',
    0xf23f: 'bbbbba04',
    0xf240: 'bbbabbY',
    0xf241: 'ccccaa04',
    0xf242: 'aaabaaX',
    0xf243: 'ppppab05',
    0xf244: 'dddaaaX',
    0xf245: 'eeeeab14',
    0xf246: 'gggbbbY',
    0xf247: 'ffffba04',
    0xf248: 'fffbbbY',
    0xf249: 'ooooaa14',
    0xf24a: 'eeeaaaX',
    0xf24b: 'llllbb04',
    0xf24c: 'hhhaabY',
    0xf24d: 'iiiiab04',
    0xf24e: 'cccabbY',
    0xf24f: 'jjjjba14',
    0xf250: 'bbbbaaX',
    0xf251: 'kkkkba14',
    0xf252: 'aaaabbY',
    0xf253: 'hhhhab04',
    0xf254: 'dddbbbY',
    0xf255: 'mmmmbb04',
    0xf256: 'gggaaaX',
    0xf257: 'nnnnaa14',
    0xf258: 'fffaaaX',
    0xf259: 'ggggba04',
    0xf25a: 'eeebbbY',
    0xf25b: 'ddddab14',
    0xf25c: 'hhhbbaX',
    0xf25d: 'aaaabb14',
    0xf25e: 'cccbaaX',
    0xf25f: 'bbbbaa04',
    0xf260: 'bbbaaaX',
    0xf261: 'ccccbb14',
    0xf262: 'aaabbbY',
    0xf263: 'ppppba15',
    0xf264: 'dddabbY',
    0xf265: 'eeeeba04',
    0xf266: 'gggbaaX',
    0xf267: 'ffffab14',
    0xf268: 'fffbaaX',
    0xf269: 'oooobb04',
    0xf26a: 'eeeabbY',
    0xf26b: 'llllaa14',
    0xf26c: 'hhhabaX',
    0xf26d: 'iiiiba14',
    0xf26e: 'cccaaaX',
    0xf26f: 'jjjjab04',
    0xf270: 'bbbbbbY',
    0xf271: 'kkkkab04',
    0xf272: 'aaaaaaX',
    0xf273: 'hhhhba14',
    0xf274: 'dddbaaX',
    0xf275: 'mmmmaa14',
    0xf276: 'gggabbY',
    0xf277: 'nnnnbb04',
    0xf278: 'dddddda103',
    0xf279: 'ggggab14',
    0xf27a: 'ddddddaabN',
    0xf27b: 'ddddba04',
    0xf27c: 'aaaaaababN',
    0xf27d: 'aaaaaa04',
    0xf27e: 'aaaaaab103',
    0xf27f: 'bbbbbb14',
    0xf280: 'bbbabaY',
    0xf281: 'ccccab04',
    0xf282: 'aaababX',
    0xf283: 'ppppaa05',
    0xf284: 'dddaabX',
    0xf285: 'eeeeaa14',
    0xf286: 'eeeeeeb003',
    0xf287: 'ffffbb04',
    0xf288: 'fffbbaY',
    0xf289: 'ooooab14',
    0xf28a: 'eeeaabX',
    0xf28b: 'llllba04',
    0xf28c: 'hhhaaaY',
    0xf28d: 'iiiiaa04',
    0xf28e: 'cccabaY',
    0xf28f: 'jjjjbb14',
    0xf290: 'bbbbabX',
    0xf291: 'kkkkbb14',
    0xf292: 'aaaabaY',
    0xf293: 'hhhhaa04',
    0xf294: 'dddbbaY',
    0xf295: 'mmmmba04',
    0xf296: 'gggaabX',
    0xf297: 'nnnnab14',
    0xf298: 'fffaabX',
    0xf299: 'ggggbb04',
    0xf29a: 'eeebbaY',
    0xf29b: 'ddddaa14',
    0xf29c: 'hhhbbbX',
    0xf29d: 'aaaaba14',
    0xf29e: 'cccbabX',
    0xf29f: 'bbbbab04',
    0xf2a0: 'bbbaabX',
    0xf2a1: 'ccccba14',
    0xf2a2: 'aaabbaY',
    0xf2a3: 'ppppbb15',
    0xf2a4: 'dddabaY',
    0xf2a5: 'eeeebb04',
    0xf2a6: 'gggbabX',
    0xf2a7: 'ffffaa14',
    0xf2a8: 'fffbabX',
    0xf2a9: 'ooooba04',
    0xf2aa: 'eeeabaY',
    0xf2ab: 'llllab14',
    0xf2ac: 'hhhabbX',
    0xf2ad: 'iiiibb14',
    0xf2ae: 'cccaabX',
    0xf2af: 'jjjjaa04',
    0xf2b0: 'bbbbbaY',
    0xf2b1: 'kkkkaa04',
    0xf2b2: 'aaaaabX',
    0xf2b3: 'hhhhbb14',
    0xf2b4: 'dddbabX',
    0xf2b5: 'mmmmab14',
    0xf2b6: 'gggabaY',
    0xf2b7: 'nnnnba04',
    0xf2b8: 'fffabaY',
    0xf2b9: 'ggggaa14',
    0xf2ba: 'eeebabX',
    0xf2bb: 'ddddbb04',
    0xf2bc: 'hhhbaaY',
    0xf2bd: 'aaaaab04',
    0xf2be: 'cccbbaY',
    0xf2bf: 'bbbbba14',
    0xf2c0: 'bbbabbX',
    0xf2c1: 'ccccaa14',
    0xf2c2: 'aaabaaY',
    0xf2c3: 'ppppab15',
    0xf2c4: 'dddaaaY',
    0xf2c5: 'eeeeab04',
    0xf2c6: 'gggbbbX',
    0xf2c7: 'ffffba14',
    0xf2c8: 'fffbbbX',
    0xf2c9: 'ooooaa04',
    0xf2ca: 'eeeaaaY',
    0xf2cb: 'llllbb14',
    0xf2cc: 'hhhaabX',
    0xf2cd: 'iiiiab14',
    0xf2ce: 'cccabbX',
    0xf2cf: 'jjjjba04',
    0xf2d0: 'bbbbaaY',
    0xf2d1: 'kkkkba04',
    0xf2d2: 'aaaabbX',
    0xf2d3: 'hhhhab14',
    0xf2d4: 'dddbbbX',
    0xf2d5: 'mmmmbb14',
    0xf2d6: 'gggaaaY',
    0xf2d7: 'nnnnaa04',
    0xf2d8: 'fffaaaY',
    0xf2d9: 'ggggba14',
    0xf2da: 'eeebbbX',
    0xf2db: 'ddddab04',
    0xf2dc: 'hhhbbaY',
    0xf2dd: 'aaaabb04',
    0xf2de: 'cccbaaY',
    0xf2df: 'bbbbaa14',
    0xf2e0: 'bbbaaaY',
    0xf2e1: 'ccccbb04',
    0xf2e2: 'aaabbbX',
    0xf2e3: 'ppppba05',
    0xf2e4: 'dddabbX',
    0xf2e5: 'eeeeba14',
    0xf2e6: 'gggbaaY',
    0xf2e7: 'ffffab04',
    0xf2e8: 'fffbaaY',
    0xf2e9: 'oooobb14',
    0xf2ea: 'eeeabbX',
    0xf2eb: 'llllaa04',
    0xf2ec: 'hhhabaY',
    0xf2ed: 'iiiiba04',
    0xf2ee: 'cccaaaY',
    0xf2ef: 'jjjjab14',
    0xf2f0: 'bbbbbbX',
    0xf2f1: 'hhhhhhhaM8',
    0xf2f2: 'aaaaaaY',
    0xf2f3: 'hhhhba04',
    0xf2f4: 'dddbaaY',
    0xf2f5: 'hhhhhhhaabv',
    0xf2f6: 'gggabbX',
    0xf2f7: 'hhhhhhhaabV',
    0xf2f8: 'aaaaaaabaaG',
    0xf2f9: 'aaaaaaabaaW',
    0xf2fa: 'aaaaaaabaag',
    0xf2fb: 'aaaaaaabaaw',
    0xf2fc: 'aaaaaaabaO',
    0xf2fd: 'aaaaaa14',
    0xf2fe: 'aaaaaaa003',
    0xf2ff: 'aaaaaaabM9',
    0xf300: 'bbbabaZ',
    0xf301: 'ccccab15',
    0xf302: 'aaabadX',
    0xf303: 'cccccccbM9',
    0xf304: 'cccccccbaaG',
    0xf305: 'cccccccbaaW',
    0xf306: 'cccccccbaag',
    0xf307: 'cccccccbaaw',
    0xf308: 'fffbbaZ',
    0xf309: 'iiiiiiibbaW',
    0xf30a: 'eeeaadX',
    0xf30b: 'iiiiiiibbaw',
    0xf30c: 'hhhaaaZ',
    0xf30d: 'iiiiaa15',
    0xf30e: 'cccabaZ',
    0xf30f: 'iiiiiiibL8',
    0xf310: 'bbbbadX',
    0xf311: 'kkkkbb05',
    0xf312: 'aaaabaZ',
    0xf313: 'hhhhaa15',
    0xf314: 'dddbbaZ',
    0xf315: 'mmmmba15',
    0xf316: 'gggaadX',
    0xf317: 'nnnnab05',
    0xf318: 'fffaadX',
    0xf319: 'ggggbb15',
    0xf31a: 'eeebbaZ',
    0xf31b: 'ddddaa05',
    0xf31c: 'hhhbbdX',
    0xf31d: 'aaaaba05',
    0xf31e: 'cccbadX',
    0xf31f: 'bbbbab15',
    0xf320: 'bbbaadX',
    0xf321: 'ccccba05',
    0xf322: 'aaabbaZ',
    0xf323: 'ppppbb04',
    0xf324: 'dddabaZ',
    0xf325: 'eeeebb15',
    0xf326: 'gggbadX',
    0xf327: 'ffffaa05',
    0xf328: 'fffbadX',
    0xf329: 'ooooba15',
    0xf32a: 'eeeabaZ',
    0xf32b: 'llllab05',
    0xf32c: 'hhhabdX',
    0xf32d: 'iiiibb05',
    0xf32e: 'cccaadX',
    0xf32f: 'jjjjaa15',
    0xf330: 'bbbbbaZ',
    0xf331: 'kkkkaa15',
    0xf332: 'aaaaadX',
    0xf333: 'hhhhbb05',
    0xf334: 'dddbadX',
    0xf335: 'mmmmab05',
    0xf336: 'gggabaZ',
    0xf337: 'nnnnba15',
    0xf338: 'fffabaZ',
    0xf339: 'ggggaa05',
    0xf33a: 'eeebadX',
    0xf33b: 'ddddbb15',
    0xf33c: 'hhhbaaZ',
    0xf33d: 'aaaaab15',
    0xf33e: 'cccbbaZ',
    0xf33f: 'bbbbba05',
    0xf340: 'bbbabdX',
    0xf341: 'ccccaa05',
    0xf342: 'aaabaaZ',
    0xf343: 'ppppab04',
    0xf344: 'dddaaaZ',
    0xf345: 'eeeeab15',
    0xf346: 'gggbbdX',
    0xf347: 'ffffba05',
    0xf348: 'fffbbdX',
    0xf349: 'ooooaa15',
    0xf34a: 'eeeaaaZ',
    0xf34b: 'llllbb05',
    0xf34c: 'hhhaadX',
    0xf34d: 'iiiiab05',
    0xf34e: 'cccabdX',
    0xf34f: 'jjjjba15',
    0xf350: 'bbbbaaZ',
    0xf351: 'kkkkba15',
    0xf352: 'aaaabdX',
    0xf353: 'hhhhab05',
    0xf354: 'dddbbdX',
    0xf355: 'mmmmbb05',
    0xf356: 'gggaaaZ',
    0xf357: 'nnnnaa15',
    0xf358: 'fffaaaZ',
    0xf359: 'ggggba05',
    0xf35a: 'eeebbdX',
    0xf35b: 'ddddab15',
    0xf35c: 'hhhbbaZ',
    0xf35d: 'aaaabb15',
    0xf35e: 'cccbaaZ',
    0xf35f: 'bbbbaa05',
    0xf360: 'bbbaaaZ',
    0xf361: 'ccccbb15',
    0xf362: 'aaabbdX',
    0xf363: 'ppppba14',
    0xf364: 'dddabdX',
    0xf365: 'eeeeba05',
    0xf366: 'gggbaaZ',
    0xf367: 'ffffab15',
    0xf368: 'fffbaaZ',
    0xf369: 'oooobb05',
    0xf36a: 'eeeabdX',
    0xf36b: 'llllaa15',
    0xf36c: 'hhhabaZ',
    0xf36d: 'iiiiba15',
    0xf36e: 'cccaaaZ',
    0xf36f: 'jjjjab05',
    0xf370: 'bbbbbdX',
    0xf371: 'kkkkab05',
    0xf372: 'aaaaaaZ',
    0xf373: 'hhhhba15',
    0xf374: 'dddbaaZ',
    0xf375: 'mmmmaa15',
    0xf376: 'gggabdX',
    0xf377: 'nnnnbb05',
    0xf378: 'dddddda003',
    0xf379: 'ggggab15',
    0xf37a: 'ddddddabaO',
    0xf37b: 'ddddba05',
    0xf37c: 'aaaaaabl8',
    0xf37d: 'aaaaaa05',
    0xf37e: 'aaaaaab003',
    0xf37f: 'bbbbbb15',
    0xf380: 'bbbabcZ',
    0xf381: 'ccccab05',
    0xf382: 'aaababZ',
    0xf383: 'ppppaa04',
    0xf384: 'dddaabZ',
    0xf385: 'eeeeaa15',
    0xf386: 'eeeeeeb103',
    0xf387: 'ffffbb05',
    0xf388: 'fffbbcZ',
    0xf389: 'ooooab15',
    0xf38a: 'eeeaabZ',
    0xf38b: 'llllba05',
    0xf38c: 'hhhaacZ',
    0xf38d: 'iiiiaa05',
    0xf38e: 'cccabcZ',
    0xf38f: 'jjjjbb15',
    0xf390: 'bbbbabZ',
    0xf391: 'kkkkbb15',
    0xf392: 'aaaabcZ',
    0xf393: 'hhhhaa05',
    0xf394: 'dddbbcZ',
    0xf395: 'mmmmba05',
    0xf396: 'gggaabZ',
    0xf397: 'nnnnab15',
    0xf398: 'fffaabZ',
    0xf399: 'ggggbb05',
    0xf39a: 'eeebbcZ',
    0xf39b: 'ddddaa15',
    0xf39c: 'hhhbbbZ',
    0xf39d: 'aaaaba15',
    0xf39e: 'cccbabZ',
    0xf39f: 'bbbbab05',
    0xf3a0: 'bbbaabZ',
    0xf3a1: 'ccccba15',
    0xf3a2: 'aaabbcZ',
    0xf3a3: 'ppppbb14',
    0xf3a4: 'dddabcZ',
    0xf3a5: 'eeeebb05',
    0xf3a6: 'gggbabZ',
    0xf3a7: 'ffffaa15',
    0xf3a8: 'fffbabZ',
    0xf3a9: 'ooooba05',
    0xf3aa: 'eeeabcZ',
    0xf3ab: 'llllab15',
    0xf3ac: 'hhhabbZ',
    0xf3ad: 'iiiibb15',
    0xf3ae: 'cccaabZ',
    0xf3af: 'jjjjaa05',
    0xf3b0: 'bbbbbcZ',
    0xf3b1: 'kkkkaa05',
    0xf3b2: 'aaaaabZ',
    0xf3b3: 'hhhhbb15',
    0xf3b4: 'dddbabZ',
    0xf3b5: 'mmmmab15',
    0xf3b6: 'gggabcZ',
    0xf3b7: 'nnnnba05',
    0xf3b8: 'fffabcZ',
    0xf3b9: 'ggggaa15',
    0xf3ba: 'eeebabZ',
    0xf3bb: 'ddddbb05',
    0xf3bc: 'hhhbacZ',
    0xf3bd: 'aaaaab05',
    0xf3be: 'cccbbcZ',
    0xf3bf: 'bbbbba15',
    0xf3c0: 'bbbabbZ',
    0xf3c1: 'ccccaa15',
    0xf3c2: 'aaabacZ',
    0xf3c3: 'ppppab14',
    0xf3c4: 'dddaacZ',
    0xf3c5: 'eeeeab05',
    0xf3c6: 'gggbbbZ',
    0xf3c7: 'ffffba15',
    0xf3c8: 'fffbbbZ',
    0xf3c9: 'ooooaa05',
    0xf3ca: 'eeeaacZ',
    0xf3cb: 'llllbb15',
    0xf3cc: 'hhhaabZ',
    0xf3cd: 'iiiiab15',
    0xf3ce: 'cccabbZ',
    0xf3cf: 'jjjjba05',
    0xf3d0: 'bbbbacZ',
    0xf3d1: 'kkkkba05',
    0xf3d2: 'aaaabbZ',
    0xf3d3: 'hhhhab15',
    0xf3d4: 'dddbbbZ',
    0xf3d5: 'mmmmbb15',
    0xf3d6: 'gggaacZ',
    0xf3d7: 'nnnnaa05',
    0xf3d8: 'fffaacZ',
    0xf3d9: 'ggggba15',
    0xf3da: 'eeebbbZ',
    0xf3db: 'ddddab05',
    0xf3dc: 'hhhbbcZ',
    0xf3dd: 'aaaabb05',
    0xf3de: 'cccbacZ',
    0xf3df: 'bbbbaa15',
    0xf3e0: 'bbbaacZ',
    0xf3e1: 'ccccbb05',
    0xf3e2: 'aaabbbZ',
    0xf3e3: 'ppppba04',
    0xf3e4: 'dddabbZ',
    0xf3e5: 'eeeeba15',
    0xf3e6: 'gggbacZ',
    0xf3e7: 'ffffab05',
    0xf3e8: 'fffbacZ',
    0xf3e9: 'oooobb15',
    0xf3ea: 'eeeabbZ',
    0xf3eb: 'llllaa05',
    0xf3ec: 'hhhabcZ',
    0xf3ed: 'iiiiba05',
    0xf3ee: 'cccaacZ',
    0xf3ef: 'jjjjab15',
    0xf3f0: 'aaaaaaaabbbj',
    0xf3f1: 'aaaaaaaabbbb',
    0xf3f2: 'aaaaacZ',
    0xf3f3: 'aaaaaaaabbbr',
    0xf3f4: 'aaaaaaaabbbJ',
    0xf3f5: 'aaaaaaaabbbB',
    0xf3f6: 'aaaaaaaabbbZ',
    0xf3f7: 'aaaaaaaabbbR',
    0xf3f8: 'aaaaaaaabbF',
    0xf3f9: 'aaaaaaaabbV',
    0xf3fa: 'aaaaaaaabbf',
    0xf3fb: 'aaaaaaaabbv',
    0xf3fc: 'aaaaaaaabN',
    0xf3fd: 'aaaaaa15',
    0xf3fe: 'aaaaaaa103',
    0xf3ff: 'aaaaaaaaL9',
    0xf400: 'bbbabaT',
    0xf401: 'ccccab12',
    0xf402: 'aaababU',
    0xf403: 'cccccccaM8',
    0xf404: 'cccccccaabF',
    0xf405: 'cccccccaabV',
    0xf406: 'cccccccaabf',
    0xf407: 'cccccccaabv',
    0xf408: 'ccccccccbaaK',
    0xf409: 'ccccccccbaaC',
    0xf40a: 'ccccccccbacZ',
    0xf40b: 'ccccccccbaaS',
    0xf40c: 'ccccccccbaak',
    0xf40d: 'ccccccccbaac',
    0xf40e: 'cccabaT',
    0xf40f: 'ccccccccbaas',
    0xf410: 'bbbbabU',
    0xf411: 'kkkkbb02',
    0xf412: 'aaaabaT',
    0xf413: 'hhhhaa12',
    0xf414: 'dddbbaT',
    0xf415: 'mmmmba12',
    0xf416: 'gggaabU',
    0xf417: 'nnnnab02',
    0xf418: 'fffaabU',
    0xf419: 'ggggbb12',
    0xf41a: 'eeebbaT',
    0xf41b: 'ddddaa02',
    0xf41c: 'hhhbbbU',
    0xf41d: 'aaaaba02',
    0xf41e: 'cccbabU',
    0xf41f: 'bbbbab12',
    0xf420: 'bbbaabU',
    0xf421: 'ccccba02',
    0xf422: 'aaabbaT',
    0xf423: 'ppppbb03',
    0xf424: 'dddabaT',
    0xf425: 'eeeebb12',
    0xf426: 'gggbabU',
    0xf427: 'ffffaa02',
    0xf428: 'fffbabU',
    0xf429: 'ooooba12',
    0xf42a: 'eeeabaT',
    0xf42b: 'llllab02',
    0xf42c: 'hhhabbU',
    0xf42d: 'iiiibb02',
    0xf42e: 'cccaabU',
    0xf42f: 'jjjjaa12',
    0xf430: 'bbbbbaT',
    0xf431: 'kkkkaa12',
    0xf432: 'aaaaabU',
    0xf433: 'hhhhbb02',
    0xf434: 'dddbabU',
    0xf435: 'mmmmab02',
    0xf436: 'gggabaT',
    0xf437: 'nnnnba12',
    0xf438: 'fffabaT',
    0xf439: 'ggggaa02',
    0xf43a: 'eeebabU',
    0xf43b: 'ddddbb12',
    0xf43c: 'hhhbaaT',
    0xf43d: 'aaaaab12',
    0xf43e: 'cccbbaT',
    0xf43f: 'bbbbba02',
    0xf440: 'bbbabbU',
    0xf441: 'ccccaa02',
    0xf442: 'aaabaaT',
    0xf443: 'ppppab03',
    0xf444: 'dddaaaT',
    0xf445: 'eeeeab12',
    0xf446: 'gggbbbU',
    0xf447: 'ffffba02',
    0xf448: 'fffbbbU',
    0xf449: 'ooooaa12',
    0xf44a: 'eeeaaaT',
    0xf44b: 'llllbb02',
    0xf44c: 'hhhaabU',
    0xf44d: 'iiiiab02',
    0xf44e: 'cccabbU',
    0xf44f: 'jjjjba12',
    0xf450: 'bbbbaaT',
    0xf451: 'kkkkba12',
    0xf452: 'aaaabbU',
    0xf453: 'hhhhab02',
    0xf454: 'dddbbbU',
    0xf455: 'mmmmbb02',
    0xf456: 'gggaaaT',
    0xf457: 'nnnnaa12',
    0xf458: 'fffaaaT',
    0xf459: 'ggggba02',
    0xf45a: 'eeebbbU',
    0xf45b: 'ddddab12',
    0xf45c: 'hhhbbaT',
    0xf45d: 'aaaabb12',
    0xf45e: 'cccbaaT',
    0xf45f: 'bbbbaa02',
    0xf460: 'bbbaaaT',
    0xf461: 'ccccbb12',
    0xf462: 'aaabbbU',
    0xf463: 'ppppba13',
    0xf464: 'dddabbU',
    0xf465: 'eeeeba02',
    0xf466: 'gggbaaT',
    0xf467: 'ffffab12',
    0xf468: 'fffbaaT',
    0xf469: 'oooobb02',
    0xf46a: 'eeeabbU',
    0xf46b: 'llllaa12',
    0xf46c: 'hhhabaT',
    0xf46d: 'iiiiba12',
    0xf46e: 'cccaaaT',
    0xf46f: 'jjjjab02',
    0xf470: 'bbbbbbU',
    0xf471: 'kkkkab02',
    0xf472: 'aaaaaaT',
    0xf473: 'hhhhba12',
    0xf474: 'dddbaaT',
    0xf475: 'mmmmaa12',
    0xf476: 'gggabbU',
    0xf477: 'nnnnbb02',
    0xf478: 'dddddda112',
    0xf479: 'ggggab12',
    0xf47a: 'ddddddaaaN',
    0xf47b: 'ddddba02',
    0xf47c: 'aaaaaabaaN',
    0xf47d: 'aaaaaa02',
    0xf47e: 'aaaaaab112',
    0xf47f: 'bbbbbb12',
    0xf480: 'bbbabaU',
    0xf481: 'ccccab02',
    0xf482: 'aaababT',
    0xf483: 'ppppaa03',
    0xf484: 'dddaabT',
    0xf485: 'eeeeaa12',
    0xf486: 'eeeeeeb012',
    0xf487: 'ffffbb02',
    0xf488: 'fffbbaU',
    0xf489: 'ooooab12',
    0xf48a: 'eeeaabT',
    0xf48b: 'llllba02',
    0xf48c: 'hhhaaaU',
    0xf48d: 'iiiiaa02',
    0xf48e: 'cccabaU',
    0xf48f: 'jjjjbb12',
    0xf490: 'bbbbabT',
    0xf491: 'kkkkbb12',
    0xf492: 'aaaabaU',
    0xf493: 'hhhhaa02',
    0xf494: 'dddbbaU',
    0xf495: 'mmmmba02',
    0xf496: 'gggaabT',
    0xf497: 'nnnnab12',
    0xf498: 'fffaabT',
    0xf499: 'ggggbb02',
    0xf49a: 'eeebbaU',
    0xf49b: 'ddddaa12',
    0xf49c: 'hhhbbbT',
    0xf49d: 'aaaaba12',
    0xf49e: 'cccbabT',
    0xf49f: 'bbbbab02',
    0xf4a0: 'bbbaabT',
    0xf4a1: 'ccccba12',
    0xf4a2: 'aaabbaU',
    0xf4a3: 'ppppbb13',
    0xf4a4: 'dddabaU',
    0xf4a5: 'eeeebb02',
    0xf4a6: 'gggbabT',
    0xf4a7: 'ffffaa12',
    0xf4a8: 'fffbabT',
    0xf4a9: 'ooooba02',
    0xf4aa: 'eeeabaU',
    0xf4ab: 'llllab12',
    0xf4ac: 'hhhabbT',
    0xf4ad: 'iiiibb12',
    0xf4ae: 'cccaabT',
    0xf4af: 'jjjjaa02',
    0xf4b0: 'bbbbbaU',
    0xf4b1: 'kkkkaa02',
    0xf4b2: 'aaaaabT',
    0xf4b3: 'hhhhbb12',
    0xf4b4: 'dddbabT',
    0xf4b5: 'mmmmab12',
    0xf4b6: 'gggabaU',
    0xf4b7: 'nnnnba02',
    0xf4b8: 'fffabaU',
    0xf4b9: 'ggggaa12',
    0xf4ba: 'eeebabT',
    0xf4bb: 'ddddbb02',
    0xf4bc: 'hhhbaaU',
    0xf4bd: 'aaaaab02',
    0xf4be: 'cccbbaU',
    0xf4bf: 'bbbbba12',
    0xf4c0: 'bbbabbT',
    0xf4c1: 'ccccaa12',
    0xf4c2: 'aaabaaU',
    0xf4c3: 'ppppab13',
    0xf4c4: 'dddaaaU',
    0xf4c5: 'eeeeab02',
    0xf4c6: 'gggbbbT',
    0xf4c7: 'ffffba12',
    0xf4c8: 'fffbbbT',
    0xf4c9: 'ooooaa02',
    0xf4ca: 'eeeaaaU',
    0xf4cb: 'llllbb12',
    0xf4cc: 'hhhaabT',
    0xf4cd: 'iiiiab12',
    0xf4ce: 'cccabbT',
    0xf4cf: 'jjjjba02',
    0xf4d0: 'bbbbaaU',
    0xf4d1: 'kkkkba02',
    0xf4d2: 'aaaabbT',
    0xf4d3: 'hhhhab12',
    0xf4d4: 'dddbbbT',
    0xf4d5: 'mmmmbb12',
    0xf4d6: 'gggaaaU',
    0xf4d7: 'nnnnaa02',
    0xf4d8: 'fffaaaU',
    0xf4d9: 'ggggba12',
    0xf4da: 'eeebbbT',
    0xf4db: 'ddddab02',
    0xf4dc: 'hhhbbaU',
    0xf4dd: 'aaaabb02',
    0xf4de: 'cccbaaU',
    0xf4df: 'bbbbaa12',
    0xf4e0: 'bbbaaaU',
    0xf4e1: 'ccccbb02',
    0xf4e2: 'aaabbbT',
    0xf4e3: 'ppppba03',
    0xf4e4: 'dddabbT',
    0xf4e5: 'eeeeba12',
    0xf4e6: 'gggbaaU',
    0xf4e7: 'ffffab02',
    0xf4e8: 'fffbaaU',
    0xf4e9: 'oooobb12',
    0xf4ea: 'eeeabbT',
    0xf4eb: 'llllaa02',
    0xf4ec: 'hhhabaU',
    0xf4ed: 'iiiiba02',
    0xf4ee: 'cccaaaU',
    0xf4ef: 'jjjjab12',
    0xf4f0: 'bbbbbbT',
    0xf4f1: 'hhhhhhhaL9',
    0xf4f2: 'aaaaaaU',
    0xf4f3: 'hhhhba02',
    0xf4f4: 'dddbaaU',
    0xf4f5: 'hhhhhhhabbv',
    0xf4f6: 'gggabbT',
    0xf4f7: 'hhhhhhhabbV',
    0xf4f8: 'aaaaaaabbaG',
    0xf4f9: 'aaaaaaabbaW',
    0xf4fa: 'aaaaaaabbag',
    0xf4fb: 'aaaaaaabbaw',
    0xf4fc: 'aaaaaaabbO',
    0xf4fd: 'aaaaaa12',
    0xf4fe: 'aaaaaaa012',
    0xf4ff: 'aaaaaaabL8',
    0xf500: 'bbbabaV',
    0xf501: 'ccccab13',
    0xf502: 'aaababW',
    0xf503: 'cccccccbL8',
    0xf504: 'cccccccbbaG',
    0xf505: 'cccccccbbaW',
    0xf506: 'cccccccbbag',
    0xf507: 'cccccccbbaw',
    0xf508: 'fffbbaV',
    0xf509: 'iiiiiiibaaW',
    0xf50a: 'eeeaabW',
    0xf50b: 'iiiiiiibaaw',
    0xf50c: 'hhhaaaV',
    0xf50d: 'iiiiaa13',
    0xf50e: 'cccabaV',
    0xf50f: 'iiiiiiibM9',
    0xf510: 'bbbbabW',
    0xf511: 'kkkkbb03',
    0xf512: 'aaaabaV',
    0xf513: 'hhhhaa13',
    0xf514: 'dddbbaV',
    0xf515: 'mmmmba13',
    0xf516: 'gggaabW',
    0xf517: 'nnnnab03',
    0xf518: 'fffaabW',
    0xf519: 'ggggbb13',
    0xf51a: 'eeebbaV',
    0xf51b: 'ddddaa03',
    0xf51c: 'hhhbbbW',
    0xf51d: 'aaaaba03',
    0xf51e: 'cccbabW',
    0xf51f: 'bbbbab13',
    0xf520: 'bbbaabW',
    0xf521: 'ccccba03',
    0xf522: 'aaabbaV',
    0xf523: 'ppppbb02',
    0xf524: 'dddabaV',
    0xf525: 'eeeebb13',
    0xf526: 'gggbabW',
    0xf527: 'ffffaa03',
    0xf528: 'fffbabW',
    0xf529: 'ooooba13',
    0xf52a: 'eeeabaV',
    0xf52b: 'llllab03',
    0xf52c: 'hhhabbW',
    0xf52d: 'iiiibb03',
    0xf52e: 'cccaabW',
    0xf52f: 'jjjjaa13',
    0xf530: 'bbbbbaV',
    0xf531: 'kkkkaa13',
    0xf532: 'aaaaabW',
    0xf533: 'hhhhbb03',
    0xf534: 'dddbabW',
    0xf535: 'mmmmab03',
    0xf536: 'gggabaV',
    0xf537: 'nnnnba13',
    0xf538: 'fffabaV',
    0xf539: 'ggggaa03',
    0xf53a: 'eeebabW',
    0xf53b: 'ddddbb13',
    0xf53c: 'hhhbaaV',
    0xf53d: 'aaaaab13',
    0xf53e: 'cccbbaV',
    0xf53f: 'bbbbba03',
    0xf540: 'bbbabbW',
    0xf541: 'ccccaa03',
    0xf542: 'aaabaaV',
    0xf543: 'ppppab02',
    0xf544: 'dddaaaV',
    0xf545: 'eeeeab13',
    0xf546: 'gggbbbW',
    0xf547: 'ffffba03',
    0xf548: 'fffbbbW',
    0xf549: 'ooooaa13',
    0xf54a: 'eeeaaaV',
    0xf54b: 'llllbb03',
    0xf54c: 'hhhaabW',
    0xf54d: 'iiiiab03',
    0xf54e: 'cccabbW',
    0xf54f: 'jjjjba13',
    0xf550: 'bbbbaaV',
    0xf551: 'kkkkba13',
    0xf552: 'aaaabbW',
    0xf553: 'hhhhab03',
    0xf554: 'dddbbbW',
    0xf555: 'mmmmbb03',
    0xf556: 'gggaaaV',
    0xf557: 'nnnnaa13',
    0xf558: 'fffaaaV',
    0xf559: 'ggggba03',
    0xf55a: 'eeebbbW',
    0xf55b: 'ddddab13',
    0xf55c: 'hhhbbaV',
    0xf55d: 'aaaabb13',
    0xf55e: 'cccbaaV',
    0xf55f: 'bbbbaa03',
    0xf560: 'bbbaaaV',
    0xf561: 'ccccbb13',
    0xf562: 'aaabbbW',
    0xf563: 'ppppba12',
    0xf564: 'dddabbW',
    0xf565: 'eeeeba03',
    0xf566: 'gggbaaV',
    0xf567: 'ffffab13',
    0xf568: 'fffbaaV',
    0xf569: 'oooobb03',
    0xf56a: 'eeeabbW',
    0xf56b: 'llllaa13',
    0xf56c: 'hhhabaV',
    0xf56d: 'iiiiba13',
    0xf56e: 'cccaaaV',
    0xf56f: 'jjjjab03',
    0xf570: 'bbbbbbW',
    0xf571: 'kkkkab03',
    0xf572: 'aaaaaaV',
    0xf573: 'hhhhba13',
    0xf574: 'dddbaaV',
    0xf575: 'mmmmaa13',
    0xf576: 'gggabbW',
    0xf577: 'nnnnbb03',
    0xf578: 'dddddda012',
    0xf579: 'ggggab13',
    0xf57a: 'ddddddabbO',
    0xf57b: 'ddddba03',
    0xf57c: 'aaaaaabbbO',
    0xf57d: 'aaaaaa03',
    0xf57e: 'aaaaaab012',
    0xf57f: 'bbbbbb13',
    0xf580: 'bbbabaW',
    0xf581: 'ccccab03',
    0xf582: 'aaababV',
    0xf583: 'ppppaa02',
    0xf584: 'dddaabV',
    0xf585: 'eeeeaa13',
    0xf586: 'eeeeeeb112',
    0xf587: 'ffffbb03',
    0xf588: 'fffbbaW',
    0xf589: 'ooooab13',
    0xf58a: 'eeeaabV',
    0xf58b: 'llllba03',
    0xf58c: 'hhhaaaW',
    0xf58d: 'iiiiaa03',
    0xf58e: 'cccabaW',
    0xf58f: 'jjjjbb13',
    0xf590: 'bbbbabV',
    0xf591: 'kkkkbb13',
    0xf592: 'aaaabaW',
    0xf593: 'hhhhaa03',
    0xf594: 'dddbbaW',
    0xf595: 'mmmmba03',
    0xf596: 'gggaabV',
    0xf597: 'nnnnab13',
    0xf598: 'fffaabV',
    0xf599: 'ggggbb03',
    0xf59a: 'eeebbaW',
    0xf59b: 'ddddaa13',
    0xf59c: 'hhhbbbV',
    0xf59d: 'aaaaba13',
    0xf59e: 'cccbabV',
    0xf59f: 'bbbbab03',
    0xf5a0: 'bbbaabV',
    0xf5a1: 'ccccba13',
    0xf5a2: 'aaabbaW',
    0xf5a3: 'ppppbb12',
    0xf5a4: 'dddabaW',
    0xf5a5: 'eeeebb03',
    0xf5a6: 'gggbabV',
    0xf5a7: 'ffffaa13',
    0xf5a8: 'fffbabV',
    0xf5a9: 'ooooba03',
    0xf5aa: 'eeeabaW',
    0xf5ab: 'llllab13',
    0xf5ac: 'hhhabbV',
    0xf5ad: 'iiiibb13',
    0xf5ae: 'cccaabV',
    0xf5af: 'jjjjaa03',
    0xf5b0: 'bbbbbaW',
    0xf5b1: 'kkkkaa03',
    0xf5b2: 'aaaaabV',
    0xf5b3: 'hhhhbb13',
    0xf5b4: 'dddbabV',
    0xf5b5: 'mmmmab13',
    0xf5b6: 'gggabaW',
    0xf5b7: 'nnnnba03',
    0xf5b8: 'fffabaW',
    0xf5b9: 'ggggaa13',
    0xf5ba: 'eeebabV',
    0xf5bb: 'ddddbb03',
    0xf5bc: 'hhhbaaW',
    0xf5bd: 'aaaaab03',
    0xf5be: 'cccbbaW',
    0xf5bf: 'bbbbba13',
    0xf5c0: 'bbbabbV',
    0xf5c1: 'ccccaa13',
    0xf5c2: 'aaabaaW',
    0xf5c3: 'ppppab12',
    0xf5c4: 'dddaaaW',
    0xf5c5: 'eeeeab03',
    0xf5c6: 'gggbbbV',
    0xf5c7: 'ffffba13',
    0xf5c8: 'fffbbbV',
    0xf5c9: 'ooooaa03',
    0xf5ca: 'eeeaaaW',
    0xf5cb: 'llllbb13',
    0xf5cc: 'hhhaabV',
    0xf5cd: 'iiiiab13',
    0xf5ce: 'cccabbV',
    0xf5cf: 'jjjjba03',
    0xf5d0: 'bbbbaaW',
    0xf5d1: 'kkkkba03',
    0xf5d2: 'aaaabbV',
    0xf5d3: 'hhhhab13',
    0xf5d4: 'dddbbbV',
    0xf5d5: 'mmmmbb13',
    0xf5d6: 'gggaaaW',
    0xf5d7: 'nnnnaa03',
    0xf5d8: 'fffaaaW',
    0xf5d9: 'ggggba13',
    0xf5da: 'eeebbbV',
    0xf5db: 'ddddab03',
    0xf5dc: 'hhhbbaW',
    0xf5dd: 'aaaabb03',
    0xf5de: 'cccbaaW',
    0xf5df: 'bbbbaa13',
    0xf5e0: 'aaaaaaaaabbdC',
    0xf5e1: 'aaaaaaaaabbbD',
    0xf5e2: 'aaabbbV',
    0xf5e3: 'aaaaaaaaabbbL',
    0xf5e4: 'aaaaaaaaabbbP',
    0xf5e5: 'aaaaaaaaabbbT',
    0xf5e6: 'aaaaaaaaabbbX',
    0xf5e7: 'aaaaaaaaabbhY',
    0xf5e8: 'aaaaaaaaabbdc',
    0xf5e9: 'aaaaaaaaabbbd',
    0xf5ea: 'aaaaaaaaabbbh',
    0xf5eb: 'aaaaaaaaabbbl',
    0xf5ec: 'aaaaaaaaabbbp',
    0xf5ed: 'aaaaaaaaabbbt',
    0xf5ee: 'aaaaaaaaabbbx',
    0xf5ef: 'aaaaaaaaabbhy',
    0xf5f0: 'aaaaaaaaabbj',
    0xf5f1: 'aaaaaaaaabbb',
    0xf5f2: 'aaaaaaW',
    0xf5f3: 'aaaaaaaaabbr',
    0xf5f4: 'aaaaaaaaabbJ',
    0xf5f5: 'aaaaaaaaabbB',
    0xf5f6: 'aaaaaaaaabbZ',
    0xf5f7: 'aaaaaaaaabbR',
    0xf5f8: 'aaaaaaaaabF',
    0xf5f9: 'aaaaaaaaabV',
    0xf5fa: 'aaaaaaaaabf',
    0xf5fb: 'aaaaaaaaabv',
    0xf5fc: 'aaaaaaaaaN',
    0xf5fd: 'aaaaaa13',
    0xf5fe: 'aaaaaaa112',
    0xf5ff: 'aaaaaaaaM8',
    0xf600: 'bbbabaP',
    0xf601: 'ccccab10',
    0xf602: 'aaababQ',
    0xf603: 'cccccccaL8',
    0xf604: 'cccccccabaG',
    0xf605: 'cccccccabaW',
    0xf606: 'cccccccabag',
    0xf607: 'cccccccabaw',
    0xf608: 'ccccccccabbJ',
    0xf609: 'ccccccccabbB',
    0xf60a: 'ccccccccabbZ',
    0xf60b: 'ccccccccabbR',
    0xf60c: 'ccccccccabbj',
    0xf60d: 'ccccccccabbb',
    0xf60e: 'cccabaP',
    0xf60f: 'ccccccccabbr',
    0xf610: 'bbbbabQ',
    0xf611: 'cccccccccaaau',
    0xf612: 'aaaabaP',
    0xf613: 'cccccccccaaiy',
    0xf614: 'cccccccccaaaa',
    0xf615: 'cccccccccaaae',
    0xf616: 'cccccccccaaai',
    0xf617: 'cccccccccaaam',
    0xf618: 'cccccccccaaaQ',
    0xf619: 'cccccccccaaaU',
    0xf61a: 'cccccccccaaaY',
    0xf61b: 'cccccccccaaiY',
    0xf61c: 'cccccccccaaaA',
    0xf61d: 'aaaaba00',
    0xf61e: 'cccbabQ',
    0xf61f: 'bbbbab10',
    0xf620: 'bbbaabQ',
    0xf621: 'ccccba00',
    0xf622: 'aaabbaP',
    0xf623: 'ccccccccccbbbi',
    0xf624: 'ccccccccccbbbg',
    0xf625: 'ccccccccccbbbe',
    0xf626: 'ccccccccccbbbc',
    0xf627: 'ccccccccccbbba',
    0xf628: 'ccccccccccbbhz',
    0xf629: 'ccccccccccbbhx',
    0xf62a: 'ccccccccccbbdx',
    0xf62b: 'ccccccccccbbby',
    0xf62c: 'ccccccccccbbbw',
    0xf62d: 'ccccccccccbbbu',
    0xf62e: 'cccaabQ',
    0xf62f: 'ccccccccccbbbq',
    0xf630: 'bbbbbaP',
    0xf631: 'ccccccccccbbbM',
    0xf632: 'aaaaabQ',
    0xf633: 'ccccccccccbbbI',
    0xf634: 'ccccccccccbbbG',
    0xf635: 'ccccccccccbbbE',
    0xf636: 'ccccccccccbbbC',
    0xf637: 'ccccccccccbbbA',
    0xf638: 'ccccccccccbbhZ',
    0xf639: 'ccccccccccbbhX',
    0xf63a: 'ccccccccccbbdX',
    0xf63b: 'ccccccccccbbbY',
    0xf63c: 'ccccccccccbbbW',
    0xf63d: 'aaaaab10',
    0xf63e: 'cccbbaP',
    0xf63f: 'bbbbba00',
    0xf640: 'bbbabbQ',
    0xf641: 'ccccaa00',
    0xf642: 'aaabaaP',
    0xf643: 'ppppab01',
    0xf644: 'dddaaaP',
    0xf645: 'eeeeab10',
    0xf646: 'gggbbbQ',
    0xf647: 'ffffba00',
    0xf648: 'fffbbbQ',
    0xf649: 'ooooaa10',
    0xf64a: 'eeeaaaP',
    0xf64b: 'llllbb00',
    0xf64c: 'hhhaabQ',
    0xf64d: 'iiiiab00',
    0xf64e: 'cccabbQ',
    0xf64f: 'jjjjba10',
    0xf650: 'bbbbaaP',
    0xf651: 'kkkkba10',
    0xf652: 'aaaabbQ',
    0xf653: 'hhhhab00',
    0xf654: 'dddbbbQ',
    0xf655: 'mmmmbb00',
    0xf656: 'gggaaaP',
    0xf657: 'nnnnaa10',
    0xf658: 'fffaaaP',
    0xf659: 'ggggba00',
    0xf65a: 'eeebbbQ',
    0xf65b: 'ddddab10',
    0xf65c: 'hhhbbaP',
    0xf65d: 'aaaabb10',
    0xf65e: 'cccbaaP',
    0xf65f: 'bbbbaa00',
    0xf660: 'bbbaaaP',
    0xf661: 'ccccbb10',
    0xf662: 'aaabbbQ',
    0xf663: 'ppppba11',
    0xf664: 'dddabbQ',
    0xf665: 'eeeeba00',
    0xf666: 'gggbaaP',
    0xf667: 'ffffab10',
    0xf668: 'fffbaaP',
    0xf669: 'oooobb00',
    0xf66a: 'eeeabbQ',
    0xf66b: 'llllaa10',
    0xf66c: 'hhhabaP',
    0xf66d: 'iiiiba10',
    0xf66e: 'cccaaaP',
    0xf66f: 'jjjjab00',
    0xf670: 'bbbbbbQ',
    0xf671: 'kkkkab00',
    0xf672: 'aaaaaaP',
    0xf673: 'hhhhba10',
    0xf674: 'dddbaaP',
    0xf675: 'mmmmaa10',
    0xf676: 'gggabbQ',
    0xf677: 'nnnnbb00',
    0xf678: 'dddddda102',
    0xf679: 'ggggab10',
    0xf67a: 'ddddddaabO',
    0xf67b: 'ddddba00',
    0xf67c: 'aaaaaababO',
    0xf67d: 'aaaaaa00',
    0xf67e: 'aaaaaab102',
    0xf67f: 'bbbbbb10',
    0xf680: 'bbbabaQ',
    0xf681: 'ccccab00',
    0xf682: 'aaababP',
    0xf683: 'ppppaa01',
    0xf684: 'dddaabP',
    0xf685: 'eeeeaa10',
    0xf686: 'eeeeeeb002',
    0xf687: 'ffffbb00',
    0xf688: 'fffbbaQ',
    0xf689: 'ooooab10',
    0xf68a: 'eeeaabP',
    0xf68b: 'llllba00',
    0xf68c: 'hhhaaaQ',
    0xf68d: 'iiiiaa00',
    0xf68e: 'cccabaQ',
    0xf68f: 'jjjjbb10',
    0xf690: 'bbbbabP',
    0xf691: 'kkkkbb10',
    0xf692: 'aaaabaQ',
    0xf693: 'hhhhaa00',
    0xf694: 'dddbbaQ',
    0xf695: 'mmmmba00',
    0xf696: 'gggaabP',
    0xf697: 'nnnnab10',
    0xf698: 'fffaabP',
    0xf699: 'ggggbb00',
    0xf69a: 'eeebbaQ',
    0xf69b: 'ddddaa10',
    0xf69c: 'hhhbbbP',
    0xf69d: 'aaaaba10',
    0xf69e: 'cccbabP',
    0xf69f: 'bbbbab00',
    0xf6a0: 'bbbaabP',
    0xf6a1: 'ccccba10',
    0xf6a2: 'aaabbaQ',
    0xf6a3: 'ppppbb11',
    0xf6a4: 'dddabaQ',
    0xf6a5: 'eeeebb00',
    0xf6a6: 'gggbabP',
    0xf6a7: 'ffffaa10',
    0xf6a8: 'fffbabP',
    0xf6a9: 'ooooba00',
    0xf6aa: 'eeeabaQ',
    0xf6ab: 'llllab10',
    0xf6ac: 'hhhabbP',
    0xf6ad: 'iiiibb10',
    0xf6ae: 'cccaabP',
    0xf6af: 'jjjjaa00',
    0xf6b0: 'bbbbbaQ',
    0xf6b1: 'kkkkaa00',
    0xf6b2: 'aaaaabP',
    0xf6b3: 'hhhhbb10',
    0xf6b4: 'dddbabP',
    0xf6b5: 'mmmmab10',
    0xf6b6: 'gggabaQ',
    0xf6b7: 'nnnnba00',
    0xf6b8: 'fffabaQ',
    0xf6b9: 'ggggaa10',
    0xf6ba: 'eeebabP',
    0xf6bb: 'ddddbb00',
    0xf6bc: 'hhhbaaQ',
    0xf6bd: 'aaaaab00',
    0xf6be: 'cccbbaQ',
    0xf6bf: 'bbbbba10',
    0xf6c0: 'bbbabbP',
    0xf6c1: 'ccccaa10',
    0xf6c2: 'aaabaaQ',
    0xf6c3: 'ppppab11',
    0xf6c4: 'dddaaaQ',
    0xf6c5: 'eeeeab00',
    0xf6c6: 'gggbbbP',
    0xf6c7: 'ffffba10',
    0xf6c8: 'fffbbbP',
    0xf6c9: 'ooooaa00',
    0xf6ca: 'eeeaaaQ',
    0xf6cb: 'llllbb10',
    0xf6cc: 'hhhaabP',
    0xf6cd: 'iiiiab10',
    0xf6ce: 'cccabbP',
    0xf6cf: 'jjjjba00',
    0xf6d0: 'bbbbaaQ',
    0xf6d1: 'kkkkba00',
    0xf6d2: 'aaaabbP',
    0xf6d3: 'hhhhab10',
    0xf6d4: 'dddbbbP',
    0xf6d5: 'mmmmbb10',
    0xf6d6: 'gggaaaQ',
    0xf6d7: 'nnnnaa00',
    0xf6d8: 'fffaaaQ',
    0xf6d9: 'ggggba10',
    0xf6da: 'eeebbbP',
    0xf6db: 'ddddab00',
    0xf6dc: 'hhhbbaQ',
    0xf6dd: 'aaaabb00',
    0xf6de: 'cccbaaQ',
    0xf6df: 'bbbbaa10',
    0xf6e0: 'bbbaaaQ',
    0xf6e1: 'ccccbb00',
    0xf6e2: 'aaabbbP',
    0xf6e3: 'ppppba01',
    0xf6e4: 'dddabbP',
    0xf6e5: 'eeeeba10',
    0xf6e6: 'gggbaaQ',
    0xf6e7: 'ffffab00',
    0xf6e8: 'fffbaaQ',
    0xf6e9: 'oooobb10',
    0xf6ea: 'eeeabbP',
    0xf6eb: 'llllaa00',
    0xf6ec: 'hhhabaQ',
    0xf6ed: 'iiiiba00',
    0xf6ee: 'cccaaaQ',
    0xf6ef: 'jjjjab10',
    0xf6f0: 'bbbbbbP',
    0xf6f1: 'hhhhhhhaM9',
    0xf6f2: 'aaaaaaQ',
    0xf6f3: 'hhhhba00',
    0xf6f4: 'dddbaaQ',
    0xf6f5: 'hhhhhhhaaaw',
    0xf6f6: 'gggabbP',
    0xf6f7: 'hhhhhhhaaaW',
    0xf6f8: 'aaaaaaababF',
    0xf6f9: 'aaaaaaababV',
    0xf6fa: 'aaaaaaababf',
    0xf6fb: 'aaaaaaababv',
    0xf6fc: 'aaaaaaabaN',
    0xf6fd: 'aaaaaa10',
    0xf6fe: 'aaaaaaa002',
    0xf6ff: 'aaaaaaabM8',
    0xf700: 'bbbabaR',
    0xf701: 'ccccab11',
    0xf702: 'aaababS',
    0xf703: 'cccccccbM8',
    0xf704: 'cccccccbabF',
    0xf705: 'cccccccbabV',
    0xf706: 'cccccccbabf',
    0xf707: 'cccccccbabv',
    0xf708: 'fffbbaR',
    0xf709: 'iiiiiiibbbV',
    0xf70a: 'eeeaabS',
    0xf70b: 'iiiiiiibbbv',
    0xf70c: 'hhhaaaR',
    0xf70d: 'iiiiaa11',
    0xf70e: 'cccabaR',
    0xf70f: 'iiiiiiibL9',
    0xf710: 'bbbbabS',
    0xf711: 'kkkkbb01',
    0xf712: 'aaaabaR',
    0xf713: 'hhhhaa11',
    0xf714: 'dddbbaR',
    0xf715: 'mmmmba11',
    0xf716: 'gggaabS',
    0xf717: 'nnnnab01',
    0xf718: 'fffaabS',
    0xf719: 'ggggbb11',
    0xf71a: 'eeebbaR',
    0xf71b: 'ddddaa01',
    0xf71c: 'hhhbbbS',
    0xf71d: 'aaaaba01',
    0xf71e: 'cccbabS',
    0xf71f: 'bbbbab11',
    0xf720: 'bbbaabS',
    0xf721: 'ccccba01',
    0xf722: 'aaabbaR',
    0xf723: 'ppppbb00',
    0xf724: 'dddabaR',
    0xf725: 'eeeebb11',
    0xf726: 'gggbabS',
    0xf727: 'ffffaa01',
    0xf728: 'fffbabS',
    0xf729: 'ooooba11',
    0xf72a: 'eeeabaR',
    0xf72b: 'llllab01',
    0xf72c: 'hhhabbS',
    0xf72d: 'iiiibb01',
    0xf72e: 'cccaabS',
    0xf72f: 'jjjjaa11',
    0xf730: 'bbbbbaR',
    0xf731: 'kkkkaa11',
    0xf732: 'aaaaabS',
    0xf733: 'hhhhbb01',
    0xf734: 'dddbabS',
    0xf735: 'mmmmab01',
    0xf736: 'gggabaR',
    0xf737: 'nnnnba11',
    0xf738: 'fffabaR',
    0xf739: 'ggggaa01',
    0xf73a: 'eeebabS',
    0xf73b: 'ddddbb11',
    0xf73c: 'hhhbaaR',
    0xf73d: 'aaaaab11',
    0xf73e: 'cccbbaR',
    0xf73f: 'bbbbba01',
    0xf740: 'bbbabbS',
    0xf741: 'ccccaa01',
    0xf742: 'aaabaaR',
    0xf743: 'ppppab00',
    0xf744: 'dddaaaR',
    0xf745: 'eeeeab11',
    0xf746: 'gggbbbS',
    0xf747: 'ffffba01',
    0xf748: 'fffbbbS',
    0xf749: 'ooooaa11',
    0xf74a: 'eeeaaaR',
    0xf74b: 'llllbb01',
    0xf74c: 'hhhaabS',
    0xf74d: 'iiiiab01',
    0xf74e: 'cccabbS',
    0xf74f: 'jjjjba11',
    0xf750: 'bbbbaaR',
    0xf751: 'kkkkba11',
    0xf752: 'aaaabbS',
    0xf753: 'hhhhab01',
    0xf754: 'dddbbbS',
    0xf755: 'mmmmbb01',
    0xf756: 'gggaaaR',
    0xf757: 'nnnnaa11',
    0xf758: 'fffaaaR',
    0xf759: 'ggggba01',
    0xf75a: 'eeebbbS',
    0xf75b: 'ddddab11',
    0xf75c: 'hhhbbaR',
    0xf75d: 'aaaabb11',
    0xf75e: 'cccbaaR',
    0xf75f: 'bbbbaa01',
    0xf760: 'bbbaaaR',
    0xf761: 'ccccbb11',
    0xf762: 'aaabbbS',
    0xf763: 'ppppba10',
    0xf764: 'dddabbS',
    0xf765: 'eeeeba01',
    0xf766: 'gggbaaR',
    0xf767: 'ffffab11',
    0xf768: 'fffbaaR',
    0xf769: 'oooobb01',
    0xf76a: 'eeeabbS',
    0xf76b: 'llllaa11',
    0xf76c: 'hhhabaR',
    0xf76d: 'iiiiba11',
    0xf76e: 'cccaaaR',
    0xf76f: 'jjjjab01',
    0xf770: 'bbbbbbS',
    0xf771: 'kkkkab01',
    0xf772: 'aaaaaaR',
    0xf773: 'hhhhba11',
    0xf774: 'dddbaaR',
    0xf775: 'mmmmaa11',
    0xf776: 'gggabbS',
    0xf777: 'nnnnbb01',
    0xf778: 'dddddda002',
    0xf779: 'ggggab11',
    0xf77a: 'ddddddabaN',
    0xf77b: 'ddddba01',
    0xf77c: 'aaaaaabbaN',
    0xf77d: 'aaaaaa01',
    0xf77e: 'aaaaaab002',
    0xf77f: 'bbbbbb11',
    0xf780: 'bbbabaS',
    0xf781: 'ccccab01',
    0xf782: 'aaababR',
    0xf783: 'ppppaa00',
    0xf784: 'dddaabR',
    0xf785: 'eeeeaa11',
    0xf786: 'eeeeeeb102',
    0xf787: 'ffffbb01',
    0xf788: 'fffbbaS',
    0xf789: 'ooooab11',
    0xf78a: 'eeeaabR',
    0xf78b: 'llllba01',
    0xf78c: 'hhhaaaS',
    0xf78d: 'iiiiaa01',
    0xf78e: 'cccabaS',
    0xf78f: 'jjjjbb11',
    0xf790: 'bbbbabR',
    0xf791: 'kkkkbb11',
    0xf792: 'aaaabaS',
    0xf793: 'hhhhaa01',
    0xf794: 'dddbbaS',
    0xf795: 'mmmmba01',
    0xf796: 'gggaabR',
    0xf797: 'nnnnab11',
    0xf798: 'fffaabR',
    0xf799: 'ggggbb01',
    0xf79a: 'eeebbaS',
    0xf79b: 'ddddaa11',
    0xf79c: 'hhhbbbR',
    0xf79d: 'aaaaba11',
    0xf79e: 'cccbabR',
    0xf79f: 'bbbbab01',
    0xf7a0: 'bbbaabR',
    0xf7a1: 'ccccba11',
    0xf7a2: 'aaabbaS',
    0xf7a3: 'ppppbb10',
    0xf7a4: 'dddabaS',
    0xf7a5: 'eeeebb01',
    0xf7a6: 'gggbabR',
    0xf7a7: 'ffffaa11',
    0xf7a8: 'fffbabR',
    0xf7a9: 'ooooba01',
    0xf7aa: 'eeeabaS',
    0xf7ab: 'llllab11',
    0xf7ac: 'hhhabbR',
    0xf7ad: 'iiiibb11',
    0xf7ae: 'cccaabR',
    0xf7af: 'jjjjaa01',
    0xf7b0: 'bbbbbaS',
    0xf7b1: 'kkkkaa01',
    0xf7b2: 'aaaaabR',
    0xf7b3: 'hhhhbb11',
    0xf7b4: 'dddbabR',
    0xf7b5: 'mmmmab11',
    0xf7b6: 'gggabaS',
    0xf7b7: 'nnnnba01',
    0xf7b8: 'fffabaS',
    0xf7b9: 'ggggaa11',
    0xf7ba: 'eeebabR',
    0xf7bb: 'ddddbb01',
    0xf7bc: 'hhhbaaS',
    0xf7bd: 'aaaaab01',
    0xf7be: 'cccbbaS',
    0xf7bf: 'bbbbba11',
    0xf7c0: 'bbbabbR',
    0xf7c1: 'ccccaa11',
    0xf7c2: 'aaabaaS',
    0xf7c3: 'ppppab10',
    0xf7c4: 'dddaaaS',
    0xf7c5: 'eeeeab01',
    0xf7c6: 'gggbbbR',
    0xf7c7: 'ffffba11',
    0xf7c8: 'fffbbbR',
    0xf7c9: 'ooooaa01',
    0xf7ca: 'eeeaaaS',
    0xf7cb: 'llllbb11',
    0xf7cc: 'hhhaabR',
    0xf7cd: 'iiiiab11',
    0xf7ce: 'cccabbR',
    0xf7cf: 'jjjjba01',
    0xf7d0: 'bbbbaaS',
    0xf7d1: 'kkkkba01',
    0xf7d2: 'aaaabbR',
    0xf7d3: 'hhhhab11',
    0xf7d4: 'dddbbbR',
    0xf7d5: 'mmmmbb11',
    0xf7d6: 'gggaaaS',
    0xf7d7: 'nnnnaa01',
    0xf7d8: 'fffaaaS',
    0xf7d9: 'ggggba11',
    0xf7da: 'eeebbbR',
    0xf7db: 'ddddab01',
    0xf7dc: 'hhhbbaS',
    0xf7dd: 'aaaabb01',
    0xf7de: 'cccbaaS',
    0xf7df: 'bbbbaa11',
    0xf7e0: 'bbbaaaS',
    0xf7e1: 'ccccbb01',
    0xf7e2: 'aaabbbR',
    0xf7e3: 'ppppba00',
    0xf7e4: 'dddabbR',
    0xf7e5: 'eeeeba11',
    0xf7e6: 'gggbaaS',
    0xf7e7: 'ffffab01',
    0xf7e8: 'fffbaaS',
    0xf7e9: 'oooobb11',
    0xf7ea: 'eeeabbR',
    0xf7eb: 'llllaa01',
    0xf7ec: 'hhhabaS',
    0xf7ed: 'iiiiba01',
    0xf7ee: 'cccaaaS',
    0xf7ef: 'jjjjab11',
    0xf7f0: 'aaaaaaaabaak',
    0xf7f1: 'aaaaaaaabaac',
    0xf7f2: 'aaaaaaS',
    0xf7f3: 'aaaaaaaabaas',
    0xf7f4: 'aaaaaaaabaaK',
    0xf7f5: 'aaaaaaaabaaC',
    0xf7f6: 'aaaaaaaabacZ',
    0xf7f7: 'aaaaaaaabaaS',
    0xf7f8: 'aaaaaaaabaG',
    0xf7f9: 'aaaaaaaabaW',
    0xf7fa: 'aaaaaaaabag',
    0xf7fb: 'aaaaaaaabaw',
    0xf7fc: 'aaaaaaaabO',
    0xf7fd: 'aaaaaa11',
    0xf7fe: 'aaaaaaa102',
    0xf7ff: 'aaaaaaaaL8',
    0xf800: 'bbbabaL',
    0xf801: 'ccccah88',
    0xf802: 'aaababM',
    0xf803: 'ccccccabH9',
    0xf804: 'cccccccaaaF',
    0xf805: 'cccccccaaaV',
    0xf806: 'cccccccaaaf',
    0xf807: 'cccccccaaav',
    0xf808: 'ccccccccbbaK',
    0xf809: 'ccccccccbbaC',
    0xf80a: 'ccccccccbbcZ',
    0xf80b: 'ccccccccbbaS',
    0xf80c: 'ccccccccbbak',
    0xf80d: 'ccccccccbbac',
    0xf80e: 'cccabaL',
    0xf80f: 'ccccccccbbas',
    0xf810: 'bbbbabM',
    0xf811: 'kkkkbh98',
    0xf812: 'aaaabaL',
    0xf813: 'hhhhai98',
    0xf814: 'dddbbaL',
    0xf815: 'mmmmbi98',
    0xf816: 'gggaabM',
    0xf817: 'nnnnah98',
    0xf818: 'fffaabM',
    0xf819: 'ggggbh88',
    0xf81a: 'eeebbaL',
    0xf81b: 'ddddai88',
    0xf81c: 'hhhbbbM',
    0xf81d: 'aaaabi88',
    0xf81e: 'cccbabM',
    0xf81f: 'bbbbah88',
    0xf820: 'bbbaabM',
    0xf821: 'ccccbi88',
    0xf822: 'aaabbaL',
    0xf823: 'ppppbh99',
    0xf824: 'dddabaL',
    0xf825: 'eeeebh88',
    0xf826: 'gggbabM',
    0xf827: 'ffffai88',
    0xf828: 'fffbabM',
    0xf829: 'oooobi98',
    0xf82a: 'eeeabaL',
    0xf82b: 'llllah98',
    0xf82c: 'hhhabbM',
    0xf82d: 'iiiibh98',
    0xf82e: 'cccaabM',
    0xf82f: 'jjjjai98',
    0xf830: 'bbbbbaL',
    0xf831: 'kkkkai98',
    0xf832: 'aaaaabM',
    0xf833: 'hhhhbh98',
    0xf834: 'dddbabM',
    0xf835: 'mmmmah98',
    0xf836: 'gggabaL',
    0xf837: 'nnnnbi98',
    0xf838: 'fffabaL',
    0xf839: 'ggggai88',
    0xf83a: 'eeebabM',
    0xf83b: 'ddddbh88',
    0xf83c: 'hhhbaaL',
    0xf83d: 'aaaaah88',
    0xf83e: 'cccbbaL',
    0xf83f: 'bbbbbi88',
    0xf840: 'bbbabbM',
    0xf841: 'ccccai88',
    0xf842: 'aaabaaL',
    0xf843: 'ppppah99',
    0xf844: 'dddaaaL',
    0xf845: 'eeeeah88',
    0xf846: 'gggbbbM',
    0xf847: 'ffffbi88',
    0xf848: 'fffbbbM',
    0xf849: 'ooooai98',
    0xf84a: 'eeeaaaL',
    0xf84b: 'llllbh98',
    0xf84c: 'hhhaabM',
    0xf84d: 'iiiiah98',
    0xf84e: 'cccabbM',
    0xf84f: 'jjjjbi98',
    0xf850: 'bbbbaaL',
    0xf851: 'kkkkbi98',
    0xf852: 'aaaabbM',
    0xf853: 'hhhhah98',
    0xf854: 'dddbbbM',
    0xf855: 'mmmmbh98',
    0xf856: 'gggaaaL',
    0xf857: 'nnnnai98',
    0xf858: 'fffaaaL',
    0xf859: 'ggggbi88',
    0xf85a: 'eeebbbM',
    0xf85b: 'ddddah88',
    0xf85c: 'hhhbbaL',
    0xf85d: 'aaaabh88',
    0xf85e: 'cccbaaL',
    0xf85f: 'bbbbai88',
    0xf860: 'bbbaaaL',
    0xf861: 'ccccbh88',
    0xf862: 'aaabbbM',
    0xf863: 'ppppbi99',
    0xf864: 'dddabbM',
    0xf865: 'eeeebi88',
    0xf866: 'gggbaaL',
    0xf867: 'ffffah88',
    0xf868: 'fffbaaL',
    0xf869: 'oooobh98',
    0xf86a: 'eeeabbM',
    0xf86b: 'llllai98',
    0xf86c: 'hhhabaL',
    0xf86d: 'iiiibi98',
    0xf86e: 'cccaaaL',
    0xf86f: 'jjjjah98',
    0xf870: 'bbbbbbM',
    0xf871: 'kkkkah98',
    0xf872: 'aaaaaaL',
    0xf873: 'hhhhbi98',
    0xf874: 'dddbaaL',
    0xf875: 'mmmmai98',
    0xf876: 'gggabbM',
    0xf877: 'nnnnbh98',
    0xf878: 'dddddda111',
    0xf879: 'ddddddaaI9',
    0xf87a: 'ddddddaaaM',
    0xf87b: 'ddddbi88',
    0xf87c: 'aaaaaabaaM',
    0xf87d: 'aaaaai88',
    0xf87e: 'aaaaaab111',
    0xf87f: 'aaaaaabaI9',
    0xf880: 'bbbabaM',
    0xf881: 'ccccah98',
    0xf882: 'aaababL',
    0xf883: 'ccccccbaI9',
    0xf884: 'dddaabL',
    0xf885: 'eeeeai98',
    0xf886: 'eeeeeeb011',
    0xf887: 'eeeeeebbH9',
    0xf888: 'fffbbaM',
    0xf889: 'ooooah88',
    0xf88a: 'eeeaabL',
    0xf88b: 'llllbi88',
    0xf88c: 'hhhaaaM',
    0xf88d: 'iiiiai88',
    0xf88e: 'cccabaM',
    0xf88f: 'jjjjbh88',
    0xf890: 'bbbbabL',
    0xf891: 'kkkkbh88',
    0xf892: 'aaaabaM',
    0xf893: 'hhhhai88',
    0xf894: 'dddbbaM',
    0xf895: 'mmmmbi88',
    0xf896: 'gggaabL',
    0xf897: 'nnnnah88',
    0xf898: 'fffaabL',
    0xf899: 'ggggbh98',
    0xf89a: 'eeebbaM',
    0xf89b: 'ddddai98',
    0xf89c: 'hhhbbbL',
    0xf89d: 'aaaabi98',
    0xf89e: 'cccbabL',
    0xf89f: 'bbbbah98',
    0xf8a0: 'bbbaabL',
    0xf8a1: 'ccccbi98',
    0xf8a2: 'aaabbaM',
    0xf8a3: 'ppppbh89',
    0xf8a4: 'dddabaM',
    0xf8a5: 'eeeebh98',
    0xf8a6: 'gggbabL',
    0xf8a7: 'ffffai98',
    0xf8a8: 'fffbabL',
    0xf8a9: 'oooobi88',
    0xf8aa: 'eeeabaM',
    0xf8ab: 'llllah88',
    0xf8ac: 'hhhabbL',
    0xf8ad: 'iiiibh88',
    0xf8ae: 'cccaabL',
    0xf8af: 'jjjjai88',
    0xf8b0: 'bbbbbaM',
    0xf8b1: 'kkkkai88',
    0xf8b2: 'aaaaabL',
    0xf8b3: 'hhhhbh88',
    0xf8b4: 'dddbabL',
    0xf8b5: 'mmmmah88',
    0xf8b6: 'gggabaM',
    0xf8b7: 'nnnnbi88',
    0xf8b8: 'fffabaM',
    0xf8b9: 'ggggai98',
    0xf8ba: 'eeebabL',
    0xf8bb: 'ddddbh98',
    0xf8bc: 'hhhbaaM',
    0xf8bd: 'aaaaah98',
    0xf8be: 'cccbbaM',
    0xf8bf: 'bbbbbi98',
    0xf8c0: 'bbbabbL',
    0xf8c1: 'ccccai98',
    0xf8c2: 'aaabaaM',
    0xf8c3: 'ppppah89',
    0xf8c4: 'dddaaaM',
    0xf8c5: 'eeeeah98',
    0xf8c6: 'gggbbbL',
    0xf8c7: 'ffffbi98',
    0xf8c8: 'fffbbbL',
    0xf8c9: 'ooooai88',
    0xf8ca: 'eeeaaaM',
    0xf8cb: 'llllbh88',
    0xf8cc: 'hhhaabL',
    0xf8cd: 'iiiiah88',
    0xf8ce: 'cccabbL',
    0xf8cf: 'jjjjbi88',
    0xf8d0: 'bbbbaaM',
    0xf8d1: 'kkkkbi88',
    0xf8d2: 'aaaabbL',
    0xf8d3: 'hhhhah88',
    0xf8d4: 'dddbbbL',
    0xf8d5: 'mmmmbh88',
    0xf8d6: 'gggaaaM',
    0xf8d7: 'nnnnai88',
    0xf8d8: 'fffaaaM',
    0xf8d9: 'ggggbi98',
    0xf8da: 'eeebbbL',
    0xf8db: 'ddddah98',
    0xf8dc: 'hhhbbaM',
    0xf8dd: 'aaaabh98',
    0xf8de: 'cccbaaM',
    0xf8df: 'bbbbai98',
    0xf8e0: 'bbbaaaM',
    0xf8e1: 'ccccbh98',
    0xf8e2: 'aaabbbL',
    0xf8e3: 'ppppbi89',
    0xf8e4: 'dddabbL',
    0xf8e5: 'eeeebi98',
    0xf8e6: 'gggbaaM',
    0xf8e7: 'ffffah98',
    0xf8e8: 'fffbaaM',
    0xf8e9: 'oooobh88',
    0xf8ea: 'eeeabbL',
    0xf8eb: 'llllai88',
    0xf8ec: 'hhhabaM',
    0xf8ed: 'iiiibi88',
    0xf8ee: 'cccaaaM',
    0xf8ef: 'jjjjah88',
    0xf8f0: 'bbbbbbL',
    0xf8f1: 'hhhhhhhaH8',
    0xf8f2: 'aaaaaaM',
    0xf8f3: 'hhhhbi88',
    0xf8f4: 'dddbaaM',
    0xf8f5: 'hhhhhhhabav',
    0xf8f6: 'gggabbL',
    0xf8f7: 'hhhhhhhabaV',
    0xf8f8: 'aaaaaaabbbG',
    0xf8f9: 'aaaaaaabbbW',
    0xf8fa: 'aaaaaaabbbg',
    0xf8fb: 'aaaaaaabbbw',
    0xf8fc: 'aaaaaaabbL',
    0xf8fd: 'aaaaai98',
    0xf8fe: 'aaaaaaa011',
    0xf8ff: 'aaaaaaabH9',
    0xf900: 'bbbabaN',
    0xf901: 'ccccah89',
    0xf902: 'aaababO',
    0xf903: 'ccccccaaI9',
    0xf904: 'cccccccbbbG',
    0xf905: 'cccccccbbbW',
    0xf906: 'cccccccbbbg',
    0xf907: 'cccccccbbbw',
    0xf908: 'fffbbaN',
    0xf909: 'iiiiiiibabW',
    0xf90a: 'eeeaabO',
    0xf90b: 'iiiiiiibabw',
    0xf90c: 'hhhaaaN',
    0xf90d: 'iiiiai99',
    0xf90e: 'cccabaN',
    0xf90f: 'iiiiiiibI8',
    0xf910: 'bbbbabO',
    0xf911: 'kkkkbh99',
    0xf912: 'aaaabaN',
    0xf913: 'hhhhai99',
    0xf914: 'dddbbaN',
    0xf915: 'mmmmbi99',
    0xf916: 'gggaabO',
    0xf917: 'nnnnah99',
    0xf918: 'fffaabO',
    0xf919: 'ggggbh89',
    0xf91a: 'eeebbaN',
    0xf91b: 'ddddai89',
    0xf91c: 'hhhbbbO',
    0xf91d: 'aaaabi89',
    0xf91e: 'cccbabO',
    0xf91f: 'bbbbah89',
    0xf920: 'bbbaabO',
    0xf921: 'ccccbi89',
    0xf922: 'aaabbaN',
    0xf923: 'ppppbh98',
    0xf924: 'dddabaN',
    0xf925: 'eeeebh89',
    0xf926: 'gggbabO',
    0xf927: 'ffffai89',
    0xf928: 'fffbabO',
    0xf929: 'oooobi99',
    0xf92a: 'eeeabaN',
    0xf92b: 'llllah99',
    0xf92c: 'hhhabbO',
    0xf92d: 'iiiibh99',
    0xf92e: 'cccaabO',
    0xf92f: 'jjjjai99',
    0xf930: 'bbbbbaN',
    0xf931: 'kkkkai99',
    0xf932: 'aaaaabO',
    0xf933: 'hhhhbh99',
    0xf934: 'dddbabO',
    0xf935: 'mmmmah99',
    0xf936: 'gggabaN',
    0xf937: 'nnnnbi99',
    0xf938: 'fffabaN',
    0xf939: 'ggggai89',
    0xf93a: 'eeebabO',
    0xf93b: 'ddddbh89',
    0xf93c: 'hhhbaaN',
    0xf93d: 'aaaaah89',
    0xf93e: 'cccbbaN',
    0xf93f: 'bbbbbi89',
    0xf940: 'bbbabbO',
    0xf941: 'ccccai89',
    0xf942: 'aaabaaN',
    0xf943: 'ppppah98',
    0xf944: 'dddaaaN',
    0xf945: 'eeeeah89',
    0xf946: 'gggbbbO',
    0xf947: 'ffffbi89',
    0xf948: 'fffbbbO',
    0xf949: 'ooooai99',
    0xf94a: 'eeeaaaN',
    0xf94b: 'llllbh99',
    0xf94c: 'hhhaabO',
    0xf94d: 'iiiiah99',
    0xf94e: 'cccabbO',
    0xf94f: 'jjjjbi99',
    0xf950: 'bbbbaaN',
    0xf951: 'kkkkbi99',
    0xf952: 'aaaabbO',
    0xf953: 'hhhhah99',
    0xf954: 'dddbbbO',
    0xf955: 'mmmmbh99',
    0xf956: 'gggaaaN',
    0xf957: 'nnnnai99',
    0xf958: 'fffaaaN',
    0xf959: 'ggggbi89',
    0xf95a: 'eeebbbO',
    0xf95b: 'ddddah89',
    0xf95c: 'hhhbbaN',
    0xf95d: 'aaaabh89',
    0xf95e: 'cccbaaN',
    0xf95f: 'bbbbai89',
    0xf960: 'bbbaaaN',
    0xf961: 'ccccbh89',
    0xf962: 'aaabbbO',
    0xf963: 'ppppbi98',
    0xf964: 'dddabbO',
    0xf965: 'eeeebi89',
    0xf966: 'gggbaaN',
    0xf967: 'ffffah89',
    0xf968: 'fffbaaN',
    0xf969: 'oooobh99',
    0xf96a: 'eeeabbO',
    0xf96b: 'llllai99',
    0xf96c: 'hhhabaN',
    0xf96d: 'iiiibi99',
    0xf96e: 'cccaaaN',
    0xf96f: 'jjjjah99',
    0xf970: 'bbbbbbO',
    0xf971: 'kkkkah99',
    0xf972: 'aaaaaaN',
    0xf973: 'hhhhbi99',
    0xf974: 'dddbaaN',
    0xf975: 'mmmmai99',
    0xf976: 'gggabbO',
    0xf977: 'nnnnbh99',
    0xf978: 'dddddda011',
    0xf979: 'ddddddabH9',
    0xf97a: 'ddddddabbL',
    0xf97b: 'ddddbi89',
    0xf97c: 'aaaaaabbbL',
    0xf97d: 'aaaaai89',
    0xf97e: 'aaaaaab011',
    0xf97f: 'aaaaaabbH9',
    0xf980: 'bbbabaO',
    0xf981: 'ccccah99',
    0xf982: 'aaababN',
    0xf983: 'ccccccbbH9',
    0xf984: 'dddaabN',
    0xf985: 'eeeeai99',
    0xf986: 'eeeeeeb111',
    0xf987: 'eeeeeebaI9',
    0xf988: 'fffbbaO',
    0xf989: 'ooooah89',
    0xf98a: 'eeeaabN',
    0xf98b: 'llllbi89',
    0xf98c: 'hhhaaaO',
    0xf98d: 'iiiiai89',
    0xf98e: 'cccabaO',
    0xf98f: 'jjjjbh89',
    0xf990: 'bbbbabN',
    0xf991: 'kkkkbh89',
    0xf992: 'aaaabaO',
    0xf993: 'hhhhai89',
    0xf994: 'dddbbaO',
    0xf995: 'mmmmbi89',
    0xf996: 'gggaabN',
    0xf997: 'nnnnah89',
    0xf998: 'fffaabN',
    0xf999: 'ggggbh99',
    0xf99a: 'eeebbaO',
    0xf99b: 'ddddai99',
    0xf99c: 'hhhbbbN',
    0xf99d: 'aaaabi99',
    0xf99e: 'cccbabN',
    0xf99f: 'bbbbah99',
    0xf9a0: 'bbbaabN',
    0xf9a1: 'ccccbi99',
    0xf9a2: 'aaabbaO',
    0xf9a3: 'ppppbh88',
    0xf9a4: 'dddabaO',
    0xf9a5: 'eeeebh99',
    0xf9a6: 'gggbabN',
    0xf9a7: 'ffffai99',
    0xf9a8: 'fffbabN',
    0xf9a9: 'oooobi89',
    0xf9aa: 'eeeabaO',
    0xf9ab: 'llllah89',
    0xf9ac: 'hhhabbN',
    0xf9ad: 'iiiibh89',
    0xf9ae: 'cccaabN',
    0xf9af: 'jjjjai89',
    0xf9b0: 'bbbbbaO',
    0xf9b1: 'kkkkai89',
    0xf9b2: 'aaaaabN',
    0xf9b3: 'hhhhbh89',
    0xf9b4: 'dddbabN',
    0xf9b5: 'mmmmah89',
    0xf9b6: 'gggabaO',
    0xf9b7: 'nnnnbi89',
    0xf9b8: 'fffabaO',
    0xf9b9: 'ggggai99',
    0xf9ba: 'eeebabN',
    0xf9bb: 'ddddbh99',
    0xf9bc: 'hhhbaaO',
    0xf9bd: 'aaaaah99',
    0xf9be: 'cccbbaO',
    0xf9bf: 'bbbbbi99',
    0xf9c0: 'aaaaaaaaaabbbW',
    0xf9c1: 'aaaaaaaaaabbbU',
    0xf9c2: 'aaabaaO',
    0xf9c3: 'aaaaaaaaaabbbQ',
    0xf9c4: 'aaaaaaaaaabbhZ',
    0xf9c5: 'aaaaaaaaaabbhX',
    0xf9c6: 'aaaaaaaaaabbdX',
    0xf9c7: 'aaaaaaaaaabbbY',
    0xf9c8: 'aaaaaaaaaabbbG',
    0xf9c9: 'aaaaaaaaaabbbE',
    0xf9ca: 'aaaaaaaaaabbbC',
    0xf9cb: 'aaaaaaaaaabbbA',
    0xf9cc: 'aaaaaaaaaabbbO',
    0xf9cd: 'aaaaaaaaaabbbM',
    0xf9ce: 'aaaaaaaaaabbbK',
    0xf9cf: 'aaaaaaaaaabbbI',
    0xf9d0: 'aaaaaaaaaabbbw',
    0xf9d1: 'aaaaaaaaaabbbu',
    0xf9d2: 'aaaabbN',
    0xf9d3: 'aaaaaaaaaabbbq',
    0xf9d4: 'aaaaaaaaaabbhz',
    0xf9d5: 'aaaaaaaaaabbhx',
    0xf9d6: 'aaaaaaaaaabbdx',
    0xf9d7: 'aaaaaaaaaabbby',
    0xf9d8: 'aaaaaaaaaabbbg',
    0xf9d9: 'aaaaaaaaaabbbe',
    0xf9da: 'aaaaaaaaaabbbc',
    0xf9db: 'aaaaaaaaaabbba',
    0xf9dc: 'aaaaaaaaaabbbo',
    0xf9dd: 'aaaabh99',
    0xf9de: 'aaaaaaaaaabbbk',
    0xf9df: 'aaaaaaaaaabbbi',
    0xf9e0: 'aaaaaaaaaabdC',
    0xf9e1: 'aaaaaaaaaabbD',
    0xf9e2: 'aaabbbN',
    0xf9e3: 'aaaaaaaaaabbL',
    0xf9e4: 'aaaaaaaaaabbP',
    0xf9e5: 'aaaaaaaaaabbT',
    0xf9e6: 'aaaaaaaaaabbX',
    0xf9e7: 'aaaaaaaaaabhY',
    0xf9e8: 'aaaaaaaaaabdc',
    0xf9e9: 'aaaaaaaaaabbd',
    0xf9ea: 'aaaaaaaaaabbh',
    0xf9eb: 'aaaaaaaaaabbl',
    0xf9ec: 'aaaaaaaaaabbp',
    0xf9ed: 'aaaaaaaaaabbt',
    0xf9ee: 'aaaaaaaaaabbx',
    0xf9ef: 'aaaaaaaaaabhy',
    0xf9f0: 'aaaaaaaaaabj',
    0xf9f1: 'aaaaaaaaaabb',
    0xf9f2: 'aaaaaaO',
    0xf9f3: 'aaaaaaaaaabr',
    0xf9f4: 'aaaaaaaaaabJ',
    0xf9f5: 'aaaaaaaaaabB',
    0xf9f6: 'aaaaaaaaaabZ',
    0xf9f7: 'aaaaaaaaaabR',
    0xf9f8: 'aaaaaaaaaaF',
    0xf9f9: 'aaaaaaaaaaV',
    0xf9fa: 'aaaaaaaaaaf',
    0xf9fb: 'aaaaaaaaaav',
    0xf9fc: 'aaaaaaaaaM',
    0xf9fd: 'aaaaai99',
    0xf9fe: 'aaaaaaa111',
    0xf9ff: 'aaaaaaaaI9',
    0xfa00: 'bbbabaH',
    0xfa01: 'ccccab98',
    0xfa02: 'aaababI',
    0xfa03: 'cccccccaH9',
    0xfa04: 'cccccccabbG',
    0xfa05: 'cccccccabbW',
    0xfa06: 'cccccccabbg',
    0xfa07: 'cccccccabbw',
    0xfa08: 'ccccccccaabJ',
    0xfa09: 'ccccccccaabB',
    0xfa0a: 'ccccccccaabZ',
    0xfa0b: 'ccccccccaabR',
    0xfa0c: 'ccccccccaabj',
    0xfa0d: 'ccccccccaabb',
    0xfa0e: 'cccabaH',
    0xfa0f: 'ccccccccaabr',
    0xfa10: 'bbbbabI',
    0xfa11: 'cccccccccbaau',
    0xfa12: 'aaaabaH',
    0xfa13: 'cccccccccbaiy',
    0xfa14: 'cccccccccbaaa',
    0xfa15: 'cccccccccbaae',
    0xfa16: 'cccccccccbaai',
    0xfa17: 'cccccccccbaam',
    0xfa18: 'cccccccccbaaQ',
    0xfa19: 'cccccccccbaaU',
    0xfa1a: 'cccccccccbaaY',
    0xfa1b: 'cccccccccbaiY',
    0xfa1c: 'cccccccccbaaA',
    0xfa1d: 'aaaaba88',
    0xfa1e: 'cccbabI',
    0xfa1f: 'bbbbab98',
    0xfa20: 'bbbaabI',
    0xfa21: 'ccccba88',
    0xfa22: 'aaabbaH',
    0xfa23: 'ppppbb89',
    0xfa24: 'dddabaH',
    0xfa25: 'eeeebb98',
    0xfa26: 'gggbabI',
    0xfa27: 'ffffaa88',
    0xfa28: 'fffbabI',
    0xfa29: 'ooooba98',
    0xfa2a: 'eeeabaH',
    0xfa2b: 'llllab88',
    0xfa2c: 'hhhabbI',
    0xfa2d: 'iiiibb88',
    0xfa2e: 'cccaabI',
    0xfa2f: 'jjjjaa98',
    0xfa30: 'bbbbbaH',
    0xfa31: 'kkkkaa98',
    0xfa32: 'aaaaabI',
    0xfa33: 'hhhhbb88',
    0xfa34: 'dddbabI',
    0xfa35: 'mmmmab88',
    0xfa36: 'gggabaH',
    0xfa37: 'nnnnba98',
    0xfa38: 'fffabaH',
    0xfa39: 'ggggaa88',
    0xfa3a: 'eeebabI',
    0xfa3b: 'ddddbb98',
    0xfa3c: 'hhhbaaH',
    0xfa3d: 'aaaaab98',
    0xfa3e: 'cccbbaH',
    0xfa3f: 'bbbbba88',
    0xfa40: 'bbbabbI',
    0xfa41: 'ccccaa88',
    0xfa42: 'aaabaaH',
    0xfa43: 'ppppab89',
    0xfa44: 'dddaaaH',
    0xfa45: 'eeeeab98',
    0xfa46: 'gggbbbI',
    0xfa47: 'ffffba88',
    0xfa48: 'fffbbbI',
    0xfa49: 'ooooaa98',
    0xfa4a: 'eeeaaaH',
    0xfa4b: 'llllbb88',
    0xfa4c: 'hhhaabI',
    0xfa4d: 'iiiiab88',
    0xfa4e: 'cccabbI',
    0xfa4f: 'jjjjba98',
    0xfa50: 'bbbbaaH',
    0xfa51: 'kkkkba98',
    0xfa52: 'aaaabbI',
    0xfa53: 'hhhhab88',
    0xfa54: 'dddbbbI',
    0xfa55: 'mmmmbb88',
    0xfa56: 'gggaaaH',
    0xfa57: 'nnnnaa98',
    0xfa58: 'fffaaaH',
    0xfa59: 'ggggba88',
    0xfa5a: 'eeebbbI',
    0xfa5b: 'ddddab98',
    0xfa5c: 'hhhbbaH',
    0xfa5d: 'aaaabb98',
    0xfa5e: 'cccbaaH',
    0xfa5f: 'bbbbaa88',
    0xfa60: 'bbbaaaH',
    0xfa61: 'ccccbb98',
    0xfa62: 'aaabbbI',
    0xfa63: 'ppppba99',
    0xfa64: 'dddabbI',
    0xfa65: 'eeeeba88',
    0xfa66: 'gggbaaH',
    0xfa67: 'ffffab98',
    0xfa68: 'fffbaaH',
    0xfa69: 'oooobb88',
    0xfa6a: 'eeeabbI',
    0xfa6b: 'llllaa98',
    0xfa6c: 'hhhabaH',
    0xfa6d: 'iiiiba98',
    0xfa6e: 'cccaaaH',
    0xfa6f: 'jjjjab88',
    0xfa70: 'bbbbbbI',
    0xfa71: 'hhhhhhaaI9',
    0xfa72: 'aaaaaaH',
    0xfa73: 'hhhhba98',
    0xfa74: 'dddbaaH',
    0xfa75: 'mmmmaa98',
    0xfa76: 'gggabbI',
    0xfa77: 'mmmmmmbaI9',
    0xfa78: 'dddddda101',
    0xfa79: 'ggggab98',
    0xfa7a: 'ddddddaabL',
    0xfa7b: 'ddddba88',
    0xfa7c: 'aaaaaababL',
    0xfa7d: 'aaaaaa88',
    0xfa7e: 'aaaaaab101',
    0xfa7f: 'bbbbbb98',
    0xfa80: 'bbbabaI',
    0xfa81: 'ccccab88',
    0xfa82: 'aaababH',
    0xfa83: 'ppppaa89',
    0xfa84: 'dddaabH',
    0xfa85: 'eeeeaa98',
    0xfa86: 'eeeeeeb001',
    0xfa87: 'ffffbb88',
    0xfa88: 'fffbbaI',
    0xfa89: 'llllllabH9',
    0xfa8a: 'eeeaabH',
    0xfa8b: 'llllba88',
    0xfa8c: 'hhhaaaI',
    0xfa8d: 'iiiiaa88',
    0xfa8e: 'cccabaI',
    0xfa8f: 'iiiiiibbH9',
    0xfa90: 'bbbbabH',
    0xfa91: 'kkkkbb98',
    0xfa92: 'aaaabaI',
    0xfa93: 'hhhhaa88',
    0xfa94: 'dddbbaI',
    0xfa95: 'mmmmba88',
    0xfa96: 'gggaabH',
    0xfa97: 'nnnnab98',
    0xfa98: 'fffaabH',
    0xfa99: 'ggggbb88',
    0xfa9a: 'eeebbaI',
    0xfa9b: 'ddddaa98',
    0xfa9c: 'hhhbbbH',
    0xfa9d: 'aaaaba98',
    0xfa9e: 'cccbabH',
    0xfa9f: 'bbbbab88',
    0xfaa0: 'bbbaabH',
    0xfaa1: 'ccccba98',
    0xfaa2: 'aaabbaI',
    0xfaa3: 'ppppbb99',
    0xfaa4: 'dddabaI',
    0xfaa5: 'eeeebb88',
    0xfaa6: 'gggbabH',
    0xfaa7: 'ffffaa98',
    0xfaa8: 'fffbabH',
    0xfaa9: 'ooooba88',
    0xfaaa: 'eeeabaI',
    0xfaab: 'llllab98',
    0xfaac: 'hhhabbH',
    0xfaad: 'iiiibb98',
    0xfaae: 'cccaabH',
    0xfaaf: 'jjjjaa88',
    0xfab0: 'bbbbbaI',
    0xfab1: 'kkkkaa88',
    0xfab2: 'aaaaabH',
    0xfab3: 'hhhhbb98',
    0xfab4: 'dddbabH',
    0xfab5: 'mmmmab98',
    0xfab6: 'gggabaI',
    0xfab7: 'nnnnba88',
    0xfab8: 'fffabaI',
    0xfab9: 'ggggaa98',
    0xfaba: 'eeebabH',
    0xfabb: 'ddddbb88',
    0xfabc: 'hhhbaaI',
    0xfabd: 'aaaaab88',
    0xfabe: 'cccbbaI',
    0xfabf: 'bbbbba98',
    0xfac0: 'bbbabbH',
    0xfac1: 'ccccaa98',
    0xfac2: 'aaabaaI',
    0xfac3: 'ppppab99',
    0xfac4: 'dddaaaI',
    0xfac5: 'eeeeab88',
    0xfac6: 'gggbbbH',
    0xfac7: 'ffffba98',
    0xfac8: 'fffbbbH',
    0xfac9: 'ooooaa88',
    0xfaca: 'eeeaaaI',
    0xfacb: 'llllbb98',
    0xfacc: 'hhhaabH',
    0xfacd: 'iiiiab98',
    0xface: 'cccabbH',
    0xfacf: 'jjjjba88',
    0xfad0: 'bbbbaaI',
    0xfad1: 'kkkkba88',
    0xfad2: 'aaaabbH',
    0xfad3: 'hhhhab98',
    0xfad4: 'dddbbbH',
    0xfad5: 'mmmmbb98',
    0xfad6: 'gggaaaI',
    0xfad7: 'nnnnaa88',
    0xfad8: 'fffaaaI',
    0xfad9: 'ggggba98',
    0xfada: 'eeebbbH',
    0xfadb: 'ddddab88',
    0xfadc: 'hhhbbaI',
    0xfadd: 'aaaabb88',
    0xfade: 'cccbaaI',
    0xfadf: 'bbbbaa98',
    0xfae0: 'bbbaaaI',
    0xfae1: 'ccccbb88',
    0xfae2: 'aaabbbH',
    0xfae3: 'ppppba89',
    0xfae4: 'dddabbH',
    0xfae5: 'eeeeba98',
    0xfae6: 'gggbaaI',
    0xfae7: 'ffffab88',
    0xfae8: 'fffbaaI',
    0xfae9: 'oooobb98',
    0xfaea: 'eeeabbH',
    0xfaeb: 'llllaa88',
    0xfaec: 'hhhabaI',
    0xfaed: 'iiiiba88',
    0xfaee: 'cccaaaI',
    0xfaef: 'jjjjab98',
    0xfaf0: 'bbbbbbH',
    0xfaf1: 'hhhhhhbbH9',
    0xfaf2: 'aaaaaaI',
    0xfaf3: 'hhhhba88',
    0xfaf4: 'dddbaaI',
    0xfaf5: 'hhhhhhhaabw',
    0xfaf6: 'gggabbH',
    0xfaf7: 'hhhhhhhaabW',
    0xfaf8: 'aaaaaaabaaF',
    0xfaf9: 'aaaaaaabaaV',
    0xfafa: 'aaaaaaabaaf',
    0xfafb: 'aaaaaaabaav',
    0xfafc: 'aaaaaaabaM',
    0xfafd: 'aaaaaa98',
    0xfafe: 'aaaaaaa001',
    0xfaff: 'aaaaaaabI9',
    0xfb00: 'bbbabaJ',
    0xfb01: 'ccccab99',
    0xfb02: 'aaababK',
    0xfb03: 'cccccccbI9',
    0xfb04: 'cccccccbaaF',
    0xfb05: 'cccccccbaaV',
    0xfb06: 'cccccccbaaf',
    0xfb07: 'cccccccbaav',
    0xfb08: 'fffbbaJ',
    0xfb09: 'iiiiiiibbaV',
    0xfb0a: 'eeeaabK',
    0xfb0b: 'iiiiiiibbav',
    0xfb0c: 'hhhaaaJ',
    0xfb0d: 'iiiiaa99',
    0xfb0e: 'cccabaJ',
    0xfb0f: 'iiiiiiabH9',
    0xfb10: 'bbbbabK',
    0xfb11: 'kkkkbb89',
    0xfb12: 'aaaabaJ',
    0xfb13: 'hhhhaa99',
    0xfb14: 'dddbbaJ',
    0xfb15: 'mmmmba99',
    0xfb16: 'gggaabK',
    0xfb17: 'nnnnab89',
    0xfb18: 'fffaabK',
    0xfb19: 'ggggbb99',
    0xfb1a: 'eeebbaJ',
    0xfb1b: 'ddddaa89',
    0xfb1c: 'hhhbbbK',
    0xfb1d: 'aaaaba89',
    0xfb1e: 'cccbabK',
    0xfb1f: 'bbbbab99',
    0xfb20: 'bbbaabK',
    0xfb21: 'ccccba89',
    0xfb22: 'aaabbaJ',
    0xfb23: 'ppppbb88',
    0xfb24: 'dddabaJ',
    0xfb25: 'eeeebb99',
    0xfb26: 'gggbabK',
    0xfb27: 'ffffaa89',
    0xfb28: 'fffbabK',
    0xfb29: 'ooooba99',
    0xfb2a: 'eeeabaJ',
    0xfb2b: 'llllab89',
    0xfb2c: 'hhhabbK',
    0xfb2d: 'iiiibb89',
    0xfb2e: 'cccaabK',
    0xfb2f: 'jjjjaa99',
    0xfb30: 'bbbbbaJ',
    0xfb31: 'kkkkaa99',
    0xfb32: 'aaaaabK',
    0xfb33: 'hhhhbb89',
    0xfb34: 'dddbabK',
    0xfb35: 'mmmmab89',
    0xfb36: 'gggabaJ',
    0xfb37: 'nnnnba99',
    0xfb38: 'fffabaJ',
    0xfb39: 'ggggaa89',
    0xfb3a: 'eeebabK',
    0xfb3b: 'ddddbb99',
    0xfb3c: 'hhhbaaJ',
    0xfb3d: 'aaaaab99',
    0xfb3e: 'cccbbaJ',
    0xfb3f: 'bbbbba89',
    0xfb40: 'bbbabbK',
    0xfb41: 'ccccaa89',
    0xfb42: 'aaabaaJ',
    0xfb43: 'ppppab88',
    0xfb44: 'dddaaaJ',
    0xfb45: 'eeeeab99',
    0xfb46: 'gggbbbK',
    0xfb47: 'ffffba89',
    0xfb48: 'fffbbbK',
    0xfb49: 'ooooaa99',
    0xfb4a: 'eeeaaaJ',
    0xfb4b: 'llllbb89',
    0xfb4c: 'hhhaabK',
    0xfb4d: 'iiiiab89',
    0xfb4e: 'cccabbK',
    0xfb4f: 'jjjjba99',
    0xfb50: 'bbbbaaJ',
    0xfb51: 'kkkkba99',
    0xfb52: 'aaaabbK',
    0xfb53: 'hhhhab89',
    0xfb54: 'dddbbbK',
    0xfb55: 'mmmmbb89',
    0xfb56: 'gggaaaJ',
    0xfb57: 'nnnnaa99',
    0xfb58: 'fffaaaJ',
    0xfb59: 'ggggba89',
    0xfb5a: 'eeebbbK',
    0xfb5b: 'ddddab99',
    0xfb5c: 'hhhbbaJ',
    0xfb5d: 'aaaabb99',
    0xfb5e: 'cccbaaJ',
    0xfb5f: 'bbbbaa89',
    0xfb60: 'bbbaaaJ',
    0xfb61: 'ccccbb99',
    0xfb62: 'aaabbbK',
    0xfb63: 'ppppba98',
    0xfb64: 'dddabbK',
    0xfb65: 'eeeeba89',
    0xfb66: 'gggbaaJ',
    0xfb67: 'ffffab99',
    0xfb68: 'fffbaaJ',
    0xfb69: 'oooobb89',
    0xfb6a: 'eeeabbK',
    0xfb6b: 'llllaa99',
    0xfb6c: 'hhhabaJ',
    0xfb6d: 'iiiiba99',
    0xfb6e: 'cccaaaJ',
    0xfb6f: 'jjjjab89',
    0xfb70: 'bbbbbbK',
    0xfb71: 'hhhhhhabH9',
    0xfb72: 'aaaaaaJ',
    0xfb73: 'hhhhba99',
    0xfb74: 'dddbaaJ',
    0xfb75: 'mmmmaa99',
    0xfb76: 'gggabbK',
    0xfb77: 'mmmmmmbbH9',
    0xfb78: 'dddddda001',
    0xfb79: 'ggggab99',
    0xfb7a: 'ddddddabaM',
    0xfb7b: 'ddddba89',
    0xfb7c: 'aaaaaabbaM',
    0xfb7d: 'aaaaaa89',
    0xfb7e: 'aaaaaab001',
    0xfb7f: 'bbbbbb99',
    0xfb80: 'bbbabaK',
    0xfb81: 'ccccab89',
    0xfb82: 'aaababJ',
    0xfb83: 'ppppaa88',
    0xfb84: 'dddaabJ',
    0xfb85: 'eeeeaa99',
    0xfb86: 'eeeeeeb101',
    0xfb87: 'ffffbb89',
    0xfb88: 'fffbbaK',
    0xfb89: 'llllllaaI9',
    0xfb8a: 'eeeaabJ',
    0xfb8b: 'llllba89',
    0xfb8c: 'hhhaaaK',
    0xfb8d: 'iiiiaa89',
    0xfb8e: 'cccabaK',
    0xfb8f: 'iiiiiibaI9',
    0xfb90: 'bbbbabJ',
    0xfb91: 'kkkkbb99',
    0xfb92: 'aaaabaK',
    0xfb93: 'hhhhaa89',
    0xfb94: 'dddbbaK',
    0xfb95: 'mmmmba89',
    0xfb96: 'gggaabJ',
    0xfb97: 'nnnnab99',
    0xfb98: 'fffaabJ',
    0xfb99: 'ggggbb89',
    0xfb9a: 'eeebbaK',
    0xfb9b: 'ddddaa99',
    0xfb9c: 'hhhbbbJ',
    0xfb9d: 'aaaaba99',
    0xfb9e: 'cccbabJ',
    0xfb9f: 'bbbbab89',
    0xfba0: 'bbbaabJ',
    0xfba1: 'ccccba99',
    0xfba2: 'aaabbaK',
    0xfba3: 'ppppbb98',
    0xfba4: 'dddabaK',
    0xfba5: 'eeeebb89',
    0xfba6: 'gggbabJ',
    0xfba7: 'ffffaa99',
    0xfba8: 'fffbabJ',
    0xfba9: 'ooooba89',
    0xfbaa: 'eeeabaK',
    0xfbab: 'llllab99',
    0xfbac: 'hhhabbJ',
    0xfbad: 'iiiibb99',
    0xfbae: 'cccaabJ',
    0xfbaf: 'jjjjaa89',
    0xfbb0: 'bbbbbaK',
    0xfbb1: 'kkkkaa89',
    0xfbb2: 'aaaaabJ',
    0xfbb3: 'hhhhbb99',
    0xfbb4: 'dddbabJ',
    0xfbb5: 'mmmmab99',
    0xfbb6: 'gggabaK',
    0xfbb7: 'nnnnba89',
    0xfbb8: 'fffabaK',
    0xfbb9: 'ggggaa99',
    0xfbba: 'eeebabJ',
    0xfbbb: 'ddddbb89',
    0xfbbc: 'hhhbaaK',
    0xfbbd: 'aaaaab89',
    0xfbbe: 'cccbbaK',
    0xfbbf: 'bbbbba99',
    0xfbc0: 'bbbabbJ',
    0xfbc1: 'ccccaa99',
    0xfbc2: 'aaabaaK',
    0xfbc3: 'ppppab98',
    0xfbc4: 'dddaaaK',
    0xfbc5: 'eeeeab89',
    0xfbc6: 'gggbbbJ',
    0xfbc7: 'ffffba99',
    0xfbc8: 'fffbbbJ',
    0xfbc9: 'ooooaa89',
    0xfbca: 'eeeaaaK',
    0xfbcb: 'llllbb99',
    0xfbcc: 'hhhaabJ',
    0xfbcd: 'iiiiab99',
    0xfbce: 'cccabbJ',
    0xfbcf: 'jjjjba89',
    0xfbd0: 'bbbbaaK',
    0xfbd1: 'kkkkba89',
    0xfbd2: 'aaaabbJ',
    0xfbd3: 'hhhhab99',
    0xfbd4: 'dddbbbJ',
    0xfbd5: 'mmmmbb99',
    0xfbd6: 'gggaaaK',
    0xfbd7: 'nnnnaa89',
    0xfbd8: 'fffaaaK',
    0xfbd9: 'ggggba99',
    0xfbda: 'eeebbbJ',
    0xfbdb: 'ddddab89',
    0xfbdc: 'hhhbbaK',
    0xfbdd: 'aaaabb89',
    0xfbde: 'cccbaaK',
    0xfbdf: 'bbbbaa99',
    0xfbe0: 'bbbaaaK',
    0xfbe1: 'ccccbb89',
    0xfbe2: 'aaabbbJ',
    0xfbe3: 'ppppba88',
    0xfbe4: 'dddabbJ',
    0xfbe5: 'eeeeba99',
    0xfbe6: 'gggbaaK',
    0xfbe7: 'ffffab89',
    0xfbe8: 'fffbaaK',
    0xfbe9: 'oooobb99',
    0xfbea: 'eeeabbJ',
    0xfbeb: 'llllaa89',
    0xfbec: 'hhhabaK',
    0xfbed: 'iiiiba89',
    0xfbee: 'cccaaaK',
    0xfbef: 'jjjjab99',
    0xfbf0: 'aaaaaaaabbak',
    0xfbf1: 'aaaaaaaabbac',
    0xfbf2: 'aaaaaaK',
    0xfbf3: 'aaaaaaaabbas',
    0xfbf4: 'aaaaaaaabbaK',
    0xfbf5: 'aaaaaaaabbaC',
    0xfbf6: 'aaaaaaaabbcZ',
    0xfbf7: 'aaaaaaaabbaS',
    0xfbf8: 'aaaaaaaabbG',
    0xfbf9: 'aaaaaaaabbW',
    0xfbfa: 'aaaaaaaabbg',
    0xfbfb: 'aaaaaaaabbw',
    0xfbfc: 'aaaaaaaabL',
    0xfbfd: 'aaaaaa99',
    0xfbfe: 'aaaaaaa101',
    0xfbff: 'aaaaaaaaH9',
    0xfc00: 'bbbabaD',
    0xfc01: 'ccccab58',
    0xfc02: 'aaababE',
    0xfc03: 'cccccccaI8',
    0xfc04: 'cccccccaabG',
    0xfc05: 'cccccccaabW',
    0xfc06: 'cccccccaabg',
    0xfc07: 'cccccccaabw',
    0xfc08: 'ccccccccbabJ',
    0xfc09: 'ccccccccbabB',
    0xfc0a: 'ccccccccbabZ',
    0xfc0b: 'ccccccccbabR',
    0xfc0c: 'ccccccccbabj',
    0xfc0d: 'ccccccccbabb',
    0xfc0e: 'cccabaD',
    0xfc0f: 'ccccccccbabr',
    0xfc10: 'bbbbabE',
    0xfc11: 'kkkkbb48',
    0xfc12: 'aaaabaD',
    0xfc13: 'hhhhaa58',
    0xfc14: 'dddbbaD',
    0xfc15: 'mmmmba58',
    0xfc16: 'gggaabE',
    0xfc17: 'nnnnab48',
    0xfc18: 'fffaabE',
    0xfc19: 'ggggbb58',
    0xfc1a: 'eeebbaD',
    0xfc1b: 'ddddaa48',
    0xfc1c: 'hhhbbbE',
    0xfc1d: 'aaaaba48',
    0xfc1e: 'cccbabE',
    0xfc1f: 'bbbbab58',
    0xfc20: 'bbbaabE',
    0xfc21: 'ccccba48',
    0xfc22: 'aaabbaD',
    0xfc23: 'ppppbb49',
    0xfc24: 'dddabaD',
    0xfc25: 'eeeebb58',
    0xfc26: 'gggbabE',
    0xfc27: 'ffffaa48',
    0xfc28: 'fffbabE',
    0xfc29: 'ooooba58',
    0xfc2a: 'eeeabaD',
    0xfc2b: 'llllab48',
    0xfc2c: 'hhhabbE',
    0xfc2d: 'iiiibb48',
    0xfc2e: 'cccaabE',
    0xfc2f: 'jjjjaa58',
    0xfc30: 'bbbbbaD',
    0xfc31: 'kkkkaa58',
    0xfc32: 'aaaaabE',
    0xfc33: 'hhhhbb48',
    0xfc34: 'dddbabE',
    0xfc35: 'mmmmab48',
    0xfc36: 'gggabaD',
    0xfc37: 'nnnnba58',
    0xfc38: 'fffabaD',
    0xfc39: 'ggggaa48',
    0xfc3a: 'eeebabE',
    0xfc3b: 'ddddbb58',
    0xfc3c: 'hhhbaaD',
    0xfc3d: 'aaaaab58',
    0xfc3e: 'cccbbaD',
    0xfc3f: 'bbbbba48',
    0xfc40: 'bbbabbE',
    0xfc41: 'ccccaa48',
    0xfc42: 'aaabaaD',
    0xfc43: 'ppppab49',
    0xfc44: 'dddaaaD',
    0xfc45: 'eeeeab58',
    0xfc46: 'gggbbbE',
    0xfc47: 'ffffba48',
    0xfc48: 'fffbbbE',
    0xfc49: 'ooooaa58',
    0xfc4a: 'eeeaaaD',
    0xfc4b: 'llllbb48',
    0xfc4c: 'hhhaabE',
    0xfc4d: 'iiiiab48',
    0xfc4e: 'cccabbE',
    0xfc4f: 'jjjjba58',
    0xfc50: 'bbbbaaD',
    0xfc51: 'kkkkba58',
    0xfc52: 'aaaabbE',
    0xfc53: 'hhhhab48',
    0xfc54: 'dddbbbE',
    0xfc55: 'mmmmbb48',
    0xfc56: 'gggaaaD',
    0xfc57: 'nnnnaa58',
    0xfc58: 'fffaaaD',
    0xfc59: 'ggggba48',
    0xfc5a: 'eeebbbE',
    0xfc5b: 'ddddab58',
    0xfc5c: 'hhhbbaD',
    0xfc5d: 'aaaabb58',
    0xfc5e: 'cccbaaD',
    0xfc5f: 'bbbbaa48',
    0xfc60: 'bbbaaaD',
    0xfc61: 'ccccbb58',
    0xfc62: 'aaabbbE',
    0xfc63: 'ppppba59',
    0xfc64: 'dddabbE',
    0xfc65: 'eeeeba48',
    0xfc66: 'gggbaaD',
    0xfc67: 'ffffab58',
    0xfc68: 'fffbaaD',
    0xfc69: 'oooobb48',
    0xfc6a: 'eeeabbE',
    0xfc6b: 'llllaa58',
    0xfc6c: 'hhhabaD',
    0xfc6d: 'iiiiba58',
    0xfc6e: 'cccaaaD',
    0xfc6f: 'jjjjab48',
    0xfc70: 'bbbbbbE',
    0xfc71: 'kkkkab48',
    0xfc72: 'aaaaaaD',
    0xfc73: 'hhhhba58',
    0xfc74: 'dddbaaD',
    0xfc75: 'mmmmaa58',
    0xfc76: 'gggabbE',
    0xfc77: 'nnnnbb48',
    0xfc78: 'dddddda110',
    0xfc79: 'ggggab58',
    0xfc7a: 'ddddddaaaL',
    0xfc7b: 'ddddba48',
    0xfc7c: 'aaaaaabaaL',
    0xfc7d: 'aaaaaa48',
    0xfc7e: 'aaaaaab110',
    0xfc7f: 'bbbbbb58',
    0xfc80: 'bbbabaE',
    0xfc81: 'ccccab48',
    0xfc82: 'aaababD',
    0xfc83: 'ppppaa49',
    0xfc84: 'dddaabD',
    0xfc85: 'eeeeaa58',
    0xfc86: 'eeeeeeb010',
    0xfc87: 'ffffbb48',
    0xfc88: 'fffbbaE',
    0xfc89: 'ooooab58',
    0xfc8a: 'eeeaabD',
    0xfc8b: 'llllba48',
    0xfc8c: 'hhhaaaE',
    0xfc8d: 'iiiiaa48',
    0xfc8e: 'cccabaE',
    0xfc8f: 'jjjjbb58',
    0xfc90: 'bbbbabD',
    0xfc91: 'kkkkbb58',
    0xfc92: 'aaaabaE',
    0xfc93: 'hhhhaa48',
    0xfc94: 'dddbbaE',
    0xfc95: 'mmmmba48',
    0xfc96: 'gggaabD',
    0xfc97: 'nnnnab58',
    0xfc98: 'fffaabD',
    0xfc99: 'ggggbb48',
    0xfc9a: 'eeebbaE',
    0xfc9b: 'ddddaa58',
    0xfc9c: 'hhhbbbD',
    0xfc9d: 'aaaaba58',
    0xfc9e: 'cccbabD',
    0xfc9f: 'bbbbab48',
    0xfca0: 'bbbaabD',
    0xfca1: 'ccccba58',
    0xfca2: 'aaabbaE',
    0xfca3: 'ppppbb59',
    0xfca4: 'dddabaE',
    0xfca5: 'eeeebb48',
    0xfca6: 'gggbabD',
    0xfca7: 'ffffaa58',
    0xfca8: 'fffbabD',
    0xfca9: 'ooooba48',
    0xfcaa: 'eeeabaE',
    0xfcab: 'llllab58',
    0xfcac: 'hhhabbD',
    0xfcad: 'iiiibb58',
    0xfcae: 'cccaabD',
    0xfcaf: 'jjjjaa48',
    0xfcb0: 'bbbbbaE',
    0xfcb1: 'kkkkaa48',
    0xfcb2: 'aaaaabD',
    0xfcb3: 'hhhhbb58',
    0xfcb4: 'dddbabD',
    0xfcb5: 'mmmmab58',
    0xfcb6: 'gggabaE',
    0xfcb7: 'nnnnba48',
    0xfcb8: 'fffabaE',
    0xfcb9: 'ggggaa58',
    0xfcba: 'eeebabD',
    0xfcbb: 'ddddbb48',
    0xfcbc: 'hhhbaaE',
    0xfcbd: 'aaaaab48',
    0xfcbe: 'cccbbaE',
    0xfcbf: 'bbbbba58',
    0xfcc0: 'bbbabbD',
    0xfcc1: 'ccccaa58',
    0xfcc2: 'aaabaaE',
    0xfcc3: 'ppppab59',
    0xfcc4: 'dddaaaE',
    0xfcc5: 'eeeeab48',
    0xfcc6: 'gggbbbD',
    0xfcc7: 'ffffba58',
    0xfcc8: 'fffbbbD',
    0xfcc9: 'ooooaa48',
    0xfcca: 'eeeaaaE',
    0xfccb: 'llllbb58',
    0xfccc: 'hhhaabD',
    0xfccd: 'iiiiab58',
    0xfcce: 'cccabbD',
    0xfccf: 'jjjjba48',
    0xfcd0: 'bbbbaaE',
    0xfcd1: 'kkkkba48',
    0xfcd2: 'aaaabbD',
    0xfcd3: 'hhhhab58',
    0xfcd4: 'dddbbbD',
    0xfcd5: 'mmmmbb58',
    0xfcd6: 'gggaaaE',
    0xfcd7: 'nnnnaa48',
    0xfcd8: 'fffaaaE',
    0xfcd9: 'ggggba58',
    0xfcda: 'eeebbbD',
    0xfcdb: 'ddddab48',
    0xfcdc: 'hhhbbaE',
    0xfcdd: 'aaaabb48',
    0xfcde: 'cccbaaE',
    0xfcdf: 'bbbbaa58',
    0xfce0: 'bbbaaaE',
    0xfce1: 'ccccbb48',
    0xfce2: 'aaabbbD',
    0xfce3: 'ppppba49',
    0xfce4: 'dddabbD',
    0xfce5: 'eeeeba58',
    0xfce6: 'gggbaaE',
    0xfce7: 'ffffab48',
    0xfce8: 'fffbaaE',
    0xfce9: 'oooobb58',
    0xfcea: 'eeeabbD',
    0xfceb: 'llllaa48',
    0xfcec: 'hhhabaE',
    0xfced: 'iiiiba48',
    0xfcee: 'cccaaaE',
    0xfcef: 'jjjjab58',
    0xfcf0: 'bbbbbbD',
    0xfcf1: 'hhhhhhhaH9',
    0xfcf2: 'aaaaaaE',
    0xfcf3: 'hhhhba48',
    0xfcf4: 'dddbaaE',
    0xfcf5: 'hhhhhhhabbw',
    0xfcf6: 'gggabbD',
    0xfcf7: 'hhhhhhhabbW',
    0xfcf8: 'aaaaaaabbaF',
    0xfcf9: 'aaaaaaabbaV',
    0xfcfa: 'aaaaaaabbaf',
    0xfcfb: 'aaaaaaabbav',
    0xfcfc: 'aaaaaaabbM',
    0xfcfd: 'aaaaaa58',
    0xfcfe: 'aaaaaaa010',
    0xfcff: 'aaaaaaabH8',
    0xfd00: 'bbbabaF',
    0xfd01: 'ccccab59',
    0xfd02: 'aaababG',
    0xfd03: 'cccccccbH8',
    0xfd04: 'cccccccbbaF',
    0xfd05: 'cccccccbbaV',
    0xfd06: 'cccccccbbaf',
    0xfd07: 'cccccccbbav',
    0xfd08: 'fffbbaF',
    0xfd09: 'iiiiiiibaaV',
    0xfd0a: 'eeeaabG',
    0xfd0b: 'iiiiiiibaav',
    0xfd0c: 'hhhaaaF',
    0xfd0d: 'iiiiaa59',
    0xfd0e: 'cccabaF',
    0xfd0f: 'iiiiiiibI9',
    0xfd10: 'bbbbabG',
    0xfd11: 'kkkkbb49',
    0xfd12: 'aaaabaF',
    0xfd13: 'hhhhaa59',
    0xfd14: 'dddbbaF',
    0xfd15: 'mmmmba59',
    0xfd16: 'gggaabG',
    0xfd17: 'nnnnab49',
    0xfd18: 'fffaabG',
    0xfd19: 'ggggbb59',
    0xfd1a: 'eeebbaF',
    0xfd1b: 'ddddaa49',
    0xfd1c: 'hhhbbbG',
    0xfd1d: 'aaaaba49',
    0xfd1e: 'cccbabG',
    0xfd1f: 'bbbbab59',
    0xfd20: 'bbbaabG',
    0xfd21: 'ccccba49',
    0xfd22: 'aaabbaF',
    0xfd23: 'ppppbb48',
    0xfd24: 'dddabaF',
    0xfd25: 'eeeebb59',
    0xfd26: 'gggbabG',
    0xfd27: 'ffffaa49',
    0xfd28: 'fffbabG',
    0xfd29: 'ooooba59',
    0xfd2a: 'eeeabaF',
    0xfd2b: 'llllab49',
    0xfd2c: 'hhhabbG',
    0xfd2d: 'iiiibb49',
    0xfd2e: 'cccaabG',
    0xfd2f: 'jjjjaa59',
    0xfd30: 'bbbbbaF',
    0xfd31: 'kkkkaa59',
    0xfd32: 'aaaaabG',
    0xfd33: 'hhhhbb49',
    0xfd34: 'dddbabG',
    0xfd35: 'mmmmab49',
    0xfd36: 'gggabaF',
    0xfd37: 'nnnnba59',
    0xfd38: 'fffabaF',
    0xfd39: 'ggggaa49',
    0xfd3a: 'eeebabG',
    0xfd3b: 'ddddbb59',
    0xfd3c: 'hhhbaaF',
    0xfd3d: 'aaaaab59',
    0xfd3e: 'cccbbaF',
    0xfd3f: 'bbbbba49',
    0xfd40: 'bbbabbG',
    0xfd41: 'ccccaa49',
    0xfd42: 'aaabaaF',
    0xfd43: 'ppppab48',
    0xfd44: 'dddaaaF',
    0xfd45: 'eeeeab59',
    0xfd46: 'gggbbbG',
    0xfd47: 'ffffba49',
    0xfd48: 'fffbbbG',
    0xfd49: 'ooooaa59',
    0xfd4a: 'eeeaaaF',
    0xfd4b: 'llllbb49',
    0xfd4c: 'hhhaabG',
    0xfd4d: 'iiiiab49',
    0xfd4e: 'cccabbG',
    0xfd4f: 'jjjjba59',
    0xfd50: 'bbbbaaF',
    0xfd51: 'kkkkba59',
    0xfd52: 'aaaabbG',
    0xfd53: 'hhhhab49',
    0xfd54: 'dddbbbG',
    0xfd55: 'mmmmbb49',
    0xfd56: 'gggaaaF',
    0xfd57: 'nnnnaa59',
    0xfd58: 'fffaaaF',
    0xfd59: 'ggggba49',
    0xfd5a: 'eeebbbG',
    0xfd5b: 'ddddab59',
    0xfd5c: 'hhhbbaF',
    0xfd5d: 'aaaabb59',
    0xfd5e: 'cccbaaF',
    0xfd5f: 'bbbbaa49',
    0xfd60: 'bbbaaaF',
    0xfd61: 'ccccbb59',
    0xfd62: 'aaabbbG',
    0xfd63: 'ppppba58',
    0xfd64: 'dddabbG',
    0xfd65: 'eeeeba49',
    0xfd66: 'gggbaaF',
    0xfd67: 'ffffab59',
    0xfd68: 'fffbaaF',
    0xfd69: 'oooobb49',
    0xfd6a: 'eeeabbG',
    0xfd6b: 'llllaa59',
    0xfd6c: 'hhhabaF',
    0xfd6d: 'iiiiba59',
    0xfd6e: 'cccaaaF',
    0xfd6f: 'jjjjab49',
    0xfd70: 'bbbbbbG',
    0xfd71: 'kkkkab49',
    0xfd72: 'aaaaaaF',
    0xfd73: 'hhhhba59',
    0xfd74: 'dddbaaF',
    0xfd75: 'mmmmaa59',
    0xfd76: 'gggabbG',
    0xfd77: 'nnnnbb49',
    0xfd78: 'dddddda010',
    0xfd79: 'ggggab59',
    0xfd7a: 'ddddddabbM',
    0xfd7b: 'ddddba49',
    0xfd7c: 'aaaaaabbbM',
    0xfd7d: 'aaaaaa49',
    0xfd7e: 'aaaaaab010',
    0xfd7f: 'bbbbbb59',
    0xfd80: 'bbbabaG',
    0xfd81: 'ccccab49',
    0xfd82: 'aaababF',
    0xfd83: 'ppppaa48',
    0xfd84: 'dddaabF',
    0xfd85: 'eeeeaa59',
    0xfd86: 'eeeeeeb110',
    0xfd87: 'ffffbb49',
    0xfd88: 'fffbbaG',
    0xfd89: 'ooooab59',
    0xfd8a: 'eeeaabF',
    0xfd8b: 'llllba49',
    0xfd8c: 'hhhaaaG',
    0xfd8d: 'iiiiaa49',
    0xfd8e: 'cccabaG',
    0xfd8f: 'jjjjbb59',
    0xfd90: 'bbbbabF',
    0xfd91: 'kkkkbb59',
    0xfd92: 'aaaabaG',
    0xfd93: 'hhhhaa49',
    0xfd94: 'dddbbaG',
    0xfd95: 'mmmmba49',
    0xfd96: 'gggaabF',
    0xfd97: 'nnnnab59',
    0xfd98: 'fffaabF',
    0xfd99: 'ggggbb49',
    0xfd9a: 'eeebbaG',
    0xfd9b: 'ddddaa59',
    0xfd9c: 'hhhbbbF',
    0xfd9d: 'aaaaba59',
    0xfd9e: 'cccbabF',
    0xfd9f: 'bbbbab49',
    0xfda0: 'bbbaabF',
    0xfda1: 'ccccba59',
    0xfda2: 'aaabbaG',
    0xfda3: 'ppppbb58',
    0xfda4: 'dddabaG',
    0xfda5: 'eeeebb49',
    0xfda6: 'gggbabF',
    0xfda7: 'ffffaa59',
    0xfda8: 'fffbabF',
    0xfda9: 'ooooba49',
    0xfdaa: 'eeeabaG',
    0xfdab: 'llllab59',
    0xfdac: 'hhhabbF',
    0xfdad: 'iiiibb59',
    0xfdae: 'cccaabF',
    0xfdaf: 'jjjjaa49',
    0xfdb0: 'bbbbbaG',
    0xfdb1: 'kkkkaa49',
    0xfdb2: 'aaaaabF',
    0xfdb3: 'hhhhbb59',
    0xfdb4: 'dddbabF',
    0xfdb5: 'mmmmab59',
    0xfdb6: 'gggabaG',
    0xfdb7: 'nnnnba49',
    0xfdb8: 'fffabaG',
    0xfdb9: 'ggggaa59',
    0xfdba: 'eeebabF',
    0xfdbb: 'ddddbb49',
    0xfdbc: 'hhhbaaG',
    0xfdbd: 'aaaaab49',
    0xfdbe: 'cccbbaG',
    0xfdbf: 'bbbbba59',
    0xfdc0: 'bbbabbF',
    0xfdc1: 'ccccaa59',
    0xfdc2: 'aaabaaG',
    0xfdc3: 'ppppab58',
    0xfdc4: 'dddaaaG',
    0xfdc5: 'eeeeab49',
    0xfdc6: 'gggbbbF',
    0xfdc7: 'ffffba59',
    0xfdc8: 'fffbbbF',
    0xfdc9: 'ooooaa49',
    0xfdca: 'eeeaaaG',
    0xfdcb: 'llllbb59',
    0xfdcc: 'hhhaabF',
    0xfdcd: 'iiiiab59',
    0xfdce: 'cccabbF',
    0xfdcf: 'jjjjba49',
    0xfdd0: 'bbbbaaG',
    0xfdd1: 'kkkkba49',
    0xfdd2: 'aaaabbF',
    0xfdd3: 'hhhhab59',
    0xfdd4: 'dddbbbF',
    0xfdd5: 'mmmmbb59',
    0xfdd6: 'gggaaaG',
    0xfdd7: 'nnnnaa49',
    0xfdd8: 'fffaaaG',
    0xfdd9: 'ggggba59',
    0xfdda: 'eeebbbF',
    0xfddb: 'ddddab49',
    0xfddc: 'hhhbbaG',
    0xfddd: 'aaaabb49',
    0xfdde: 'cccbaaG',
    0xfddf: 'bbbbaa59',
    0xfde0: 'aaaaaaaaabaaA',
    0xfde1: 'aaaaaaaaabaaE',
    0xfde2: 'aaabbbF',
    0xfde3: 'aaaaaaaaabaaM',
    0xfde4: 'aaaaaaaaabaaQ',
    0xfde5: 'aaaaaaaaabaaU',
    0xfde6: 'aaaaaaaaabaaY',
    0xfde7: 'aaaaaaaaabaiY',
    0xfde8: 'aaaaaaaaabaaa',
    0xfde9: 'aaaaaaaaabaae',
    0xfdea: 'aaaaaaaaabaai',
    0xfdeb: 'aaaaaaaaabaam',
    0xfdec: 'aaaaaaaaabaaq',
    0xfded: 'aaaaaaaaabaau',
    0xfdee: 'aaaaaaaaabaay',
    0xfdef: 'aaaaaaaaabaiy',
    0xfdf0: 'aaaaaaaaabak',
    0xfdf1: 'aaaaaaaaabac',
    0xfdf2: 'aaaaaaG',
    0xfdf3: 'aaaaaaaaabas',
    0xfdf4: 'aaaaaaaaabaK',
    0xfdf5: 'aaaaaaaaabaC',
    0xfdf6: 'aaaaaaaaabcZ',
    0xfdf7: 'aaaaaaaaabaS',
    0xfdf8: 'aaaaaaaaabG',
    0xfdf9: 'aaaaaaaaabW',
    0xfdfa: 'aaaaaaaaabg',
    0xfdfb: 'aaaaaaaaabw',
    0xfdfc: 'aaaaaaaaaL',
    0xfdfd: 'aaaaaa59',
    0xfdfe: 'aaaaaaa110',
    0xfdff: 'aaaaaaaaI8',
    0xfe00: 'bbbabcA',
    0xfe01: 'ccccab18',
    0xfe02: 'aaababA',
    0xfe03: 'cccccccaH8',
    0xfe04: 'cccccccabaF',
    0xfe05: 'cccccccabaV',
    0xfe06: 'cccccccabaf',
    0xfe07: 'cccccccabav',
    0xfe08: 'ccccccccabaK',
    0xfe09: 'ccccccccabaC',
    0xfe0a: 'ccccccccabcZ',
    0xfe0b: 'ccccccccabaS',
    0xfe0c: 'ccccccccabak',
    0xfe0d: 'ccccccccabac',
    0xfe0e: 'cccabcA',
    0xfe0f: 'ccccccccabas',
    0xfe10: 'bbbbabA',
    0xfe11: 'cccccccccabbt',
    0xfe12: 'aaaabcA',
    0xfe13: 'cccccccccabhy',
    0xfe14: 'cccccccccabdc',
    0xfe15: 'cccccccccabbd',
    0xfe16: 'cccccccccabbh',
    0xfe17: 'cccccccccabbl',
    0xfe18: 'cccccccccabbP',
    0xfe19: 'cccccccccabbT',
    0xfe1a: 'cccccccccabbX',
    0xfe1b: 'cccccccccabhY',
    0xfe1c: 'cccccccccabdC',
    0xfe1d: 'aaaaba08',
    0xfe1e: 'cccbabA',
    0xfe1f: 'bbbbab18',
    0xfe20: 'bbbaabA',
    0xfe21: 'ccccba08',
    0xfe22: 'aaabbcA',
    0xfe23: 'ccccccccccaaah',
    0xfe24: 'ccccccccccaaaf',
    0xfe25: 'ccccccccccaaad',
    0xfe26: 'ccccccccccaaab',
    0xfe27: 'ccccccccccaaca',
    0xfe28: 'ccccccccccaaiz',
    0xfe29: 'ccccccccccaaix',
    0xfe2a: 'ccccccccccaaaz',
    0xfe2b: 'ccccccccccaaax',
    0xfe2c: 'ccccccccccaaav',
    0xfe2d: 'ccccccccccaaat',
    0xfe2e: 'cccaabA',
    0xfe2f: 'ccccccccccaaap',
    0xfe30: 'bbbbbcA',
    0xfe31: 'ccccccccccaaaL',
    0xfe32: 'aaaaabA',
    0xfe33: 'ccccccccccaaaH',
    0xfe34: 'ccccccccccaaaF',
    0xfe35: 'ccccccccccaaaD',
    0xfe36: 'ccccccccccaaaB',
    0xfe37: 'ccccccccccaacA',
    0xfe38: 'ccccccccccaaiZ',
    0xfe39: 'ccccccccccaaiX',
    0xfe3a: 'ccccccccccaaaZ',
    0xfe3b: 'ccccccccccaaaX',
    0xfe3c: 'ccccccccccaaaV',
    0xfe3d: 'aaaaab18',
    0xfe3e: 'cccbbcA',
    0xfe3f: 'bbbbba08',
    0xfe40: 'bbbabbA',
    0xfe41: 'ccccaa08',
    0xfe42: 'aaabacA',
    0xfe43: 'cccccccccccbbae',
    0xfe44: 'cccccccccccbbab',
    0xfe45: 'cccccccccccbbac',
    0xfe46: 'cccccccccccbbca',
    0xfe47: 'cccccccccccbbaa',
    0xfe48: 'cccccccccccbban',
    0xfe49: 'cccccccccccbbao',
    0xfe4a: 'cccccccccccbbal',
    0xfe4b: 'cccccccccccbbam',
    0xfe4c: 'cccccccccccbbaj',
    0xfe4d: 'cccccccccccbbak',
    0xfe4e: 'cccabbA',
    0xfe4f: 'cccccccccccbbai',
    0xfe50: 'bbbbacA',
    0xfe51: 'cccccccccccbbaw',
    0xfe52: 'aaaabbA',
    0xfe53: 'cccccccccccbbau',
    0xfe54: 'cccccccccccbbar',
    0xfe55: 'cccccccccccbbas',
    0xfe56: 'cccccccccccbbap',
    0xfe57: 'cccccccccccbbaq',
    0xfe58: 'cccccccccccbbiz',
    0xfe59: 'cccccccccccbbkz',
    0xfe5a: 'cccccccccccbbix',
    0xfe5b: 'cccccccccccbbiy',
    0xfe5c: 'cccccccccccbbaz',
    0xfe5d: 'aaaabb18',
    0xfe5e: 'cccbacA',
    0xfe5f: 'bbbbaa08',
    0xfe60: 'bbbaacA',
    0xfe61: 'ccccbb18',
    0xfe62: 'aaabbbA',
    0xfe63: 'cccccccccccbbaE',
    0xfe64: 'cccccccccccbbaB',
    0xfe65: 'cccccccccccbbaC',
    0xfe66: 'cccccccccccbbcA',
    0xfe67: 'cccccccccccbbaA',
    0xfe68: 'cccccccccccbbaN',
    0xfe69: 'cccccccccccbbaO',
    0xfe6a: 'cccccccccccbbaL',
    0xfe6b: 'cccccccccccbbaM',
    0xfe6c: 'cccccccccccbbaJ',
    0xfe6d: 'cccccccccccbbaK',
    0xfe6e: 'cccaacA',
    0xfe6f: 'cccccccccccbbaI',
    0xfe70: 'bbbbbbA',
    0xfe71: 'cccccccccccbbaW',
    0xfe72: 'aaaaacA',
    0xfe73: 'cccccccccccbbaU',
    0xfe74: 'cccccccccccbbaR',
    0xfe75: 'cccccccccccbbaS',
    0xfe76: 'cccccccccccbbaP',
    0xfe77: 'cccccccccccbbaQ',
    0xfe78: 'cccccccccccbbiZ',
    0xfe79: 'cccccccccccbbkZ',
    0xfe7a: 'cccccccccccbbiX',
    0xfe7b: 'cccccccccccbbiY',
    0xfe7c: 'aaaaaababM',
    0xfe7d: 'aaaaaa08',
    0xfe7e: 'aaaaaab100',
    0xfe7f: 'bbbbbb18',
    0xfe80: 'bbbabaA',
    0xfe81: 'ccccab08',
    0xfe82: 'aaabadC',
    0xfe83: 'ppppaa09',
    0xfe84: 'dddaadC',
    0xfe85: 'eeeeaa18',
    0xfe86: 'eeeeeeb000',
    0xfe87: 'ffffbb08',
    0xfe88: 'fffbbaA',
    0xfe89: 'ooooab18',
    0xfe8a: 'eeeaadC',
    0xfe8b: 'llllba08',
    0xfe8c: 'hhhaaaA',
    0xfe8d: 'iiiiaa08',
    0xfe8e: 'cccabaA',
    0xfe8f: 'jjjjbb18',
    0xfe90: 'bbbbadC',
    0xfe91: 'kkkkbb18',
    0xfe92: 'aaaabaA',
    0xfe93: 'hhhhaa08',
    0xfe94: 'dddbbaA',
    0xfe95: 'mmmmba08',
    0xfe96: 'gggaadC',
    0xfe97: 'nnnnab18',
    0xfe98: 'fffaadC',
    0xfe99: 'ggggbb08',
    0xfe9a: 'eeebbaA',
    0xfe9b: 'ddddaa18',
    0xfe9c: 'hhhbbdC',
    0xfe9d: 'aaaaba18',
    0xfe9e: 'cccbadC',
    0xfe9f: 'bbbbab08',
    0xfea0: 'bbbaadC',
    0xfea1: 'ccccba18',
    0xfea2: 'aaabbaA',
    0xfea3: 'ppppbb19',
    0xfea4: 'dddabaA',
    0xfea5: 'eeeebb08',
    0xfea6: 'gggbadC',
    0xfea7: 'ffffaa18',
    0xfea8: 'fffbadC',
    0xfea9: 'ooooba08',
    0xfeaa: 'eeeabaA',
    0xfeab: 'llllab18',
    0xfeac: 'hhhabdC',
    0xfead: 'iiiibb18',
    0xfeae: 'cccaadC',
    0xfeaf: 'jjjjaa08',
    0xfeb0: 'bbbbbaA',
    0xfeb1: 'kkkkaa08',
    0xfeb2: 'aaaaadC',
    0xfeb3: 'hhhhbb18',
    0xfeb4: 'dddbadC',
    0xfeb5: 'mmmmab18',
    0xfeb6: 'gggabaA',
    0xfeb7: 'nnnnba08',
    0xfeb8: 'fffabaA',
    0xfeb9: 'ggggaa18',
    0xfeba: 'eeebadC',
    0xfebb: 'ddddbb08',
    0xfebc: 'hhhbaaA',
    0xfebd: 'aaaaab08',
    0xfebe: 'cccbbaA',
    0xfebf: 'bbbbba18',
    0xfec0: 'bbbabdC',
    0xfec1: 'ccccaa18',
    0xfec2: 'aaabaaA',
    0xfec3: 'ppppab19',
    0xfec4: 'dddaaaA',
    0xfec5: 'eeeeab08',
    0xfec6: 'gggbbdC',
    0xfec7: 'ffffba18',
    0xfec8: 'fffbbdC',
    0xfec9: 'ooooaa08',
    0xfeca: 'eeeaaaA',
    0xfecb: 'llllbb18',
    0xfecc: 'hhhaadC',
    0xfecd: 'iiiiab18',
    0xfece: 'cccabdC',
    0xfecf: 'jjjjba08',
    0xfed0: 'bbbbaaA',
    0xfed1: 'kkkkba08',
    0xfed2: 'aaaabdC',
    0xfed3: 'hhhhab18',
    0xfed4: 'dddbbdC',
    0xfed5: 'mmmmbb18',
    0xfed6: 'gggaaaA',
    0xfed7: 'nnnnaa08',
    0xfed8: 'fffaaaA',
    0xfed9: 'ggggba18',
    0xfeda: 'eeebbdC',
    0xfedb: 'ddddab08',
    0xfedc: 'hhhbbaA',
    0xfedd: 'aaaabb08',
    0xfede: 'cccbaaA',
    0xfedf: 'bbbbaa18',
    0xfee0: 'bbbaaaA',
    0xfee1: 'ccccbb08',
    0xfee2: 'aaabbdC',
    0xfee3: 'ppppba09',
    0xfee4: 'dddabdC',
    0xfee5: 'eeeeba18',
    0xfee6: 'gggbaaA',
    0xfee7: 'ffffab08',
    0xfee8: 'fffbaaA',
    0xfee9: 'oooobb18',
    0xfeea: 'eeeabdC',
    0xfeeb: 'llllaa08',
    0xfeec: 'hhhabaA',
    0xfeed: 'iiiiba08',
    0xfeee: 'cccaaaA',
    0xfeef: 'jjjjab18',
    0xfef0: 'bbbbbdC',
    0xfef1: 'hhhhhhhaI9',
    0xfef2: 'aaaaaaA',
    0xfef3: 'hhhhba08',
    0xfef4: 'dddbaaA',
    0xfef5: 'hhhhhhhaaav',
    0xfef6: 'gggabdC',
    0xfef7: 'hhhhhhhaaaV',
    0xfef8: 'aaaaaaababG',
    0xfef9: 'aaaaaaababW',
    0xfefa: 'aaaaaaababg',
    0xfefb: 'aaaaaaababw',
    0xfefc: 'aaaaaaab9',
    0xfefd: 'aaaaaa18',
    0xfefe: 'aaaaaaa000',
    0xfeff: 'aaaaaaabI8',
    0xff00: 'bbbabaB',
    0xff01: 'ccccab19',
    0xff02: 'aaababC',
    0xff03: 'cccccccbI8',
    0xff04: 'cccccccbabG',
    0xff05: 'cccccccbabW',
    0xff06: 'cccccccbabg',
    0xff07: 'cccccccbabw',
    0xff08: 'fffbbaB',
    0xff09: 'iiiiiiibbbW',
    0xff0a: 'eeeaabC',
    0xff0b: 'iiiiiiibbbw',
    0xff0c: 'hhhaaaB',
    0xff0d: 'iiiiaa19',
    0xff0e: 'cccabaB',
    0xff0f: 'iiiiiiibH9',
    0xff10: 'bbbbabC',
    0xff11: 'kkkkbb09',
    0xff12: 'aaaabaB',
    0xff13: 'hhhhaa19',
    0xff14: 'dddbbaB',
    0xff15: 'mmmmba19',
    0xff16: 'gggaabC',
    0xff17: 'nnnnab09',
    0xff18: 'fffaabC',
    0xff19: 'ggggbb19',
    0xff1a: 'eeebbaB',
    0xff1b: 'ddddaa09',
    0xff1c: 'hhhbbbC',
    0xff1d: 'aaaaba09',
    0xff1e: 'cccbabC',
    0xff1f: 'bbbbab19',
    0xff20: 'bbbaabC',
    0xff21: 'ccccba09',
    0xff22: 'aaabbaB',
    0xff23: 'ppppbb08',
    0xff24: 'dddabaB',
    0xff25: 'eeeebb19',
    0xff26: 'gggbabC',
    0xff27: 'ffffaa09',
    0xff28: 'fffbabC',
    0xff29: 'ooooba19',
    0xff2a: 'eeeabaB',
    0xff2b: 'llllab09',
    0xff2c: 'hhhabbC',
    0xff2d: 'iiiibb09',
    0xff2e: 'cccaabC',
    0xff2f: 'jjjjaa19',
    0xff30: 'bbbbbaB',
    0xff31: 'kkkkaa19',
    0xff32: 'aaaaabC',
    0xff33: 'hhhhbb09',
    0xff34: 'dddbabC',
    0xff35: 'mmmmab09',
    0xff36: 'gggabaB',
    0xff37: 'nnnnba19',
    0xff38: 'fffabaB',
    0xff39: 'ggggaa09',
    0xff3a: 'eeebabC',
    0xff3b: 'ddddbb19',
    0xff3c: 'hhhbaaB',
    0xff3d: 'aaaaab19',
    0xff3e: 'cccbbaB',
    0xff3f: 'bbbbba09',
    0xff40: 'bbbabbC',
    0xff41: 'ccccaa09',
    0xff42: 'aaabaaB',
    0xff43: 'ppppab08',
    0xff44: 'dddaaaB',
    0xff45: 'eeeeab19',
    0xff46: 'gggbbbC',
    0xff47: 'ffffba09',
    0xff48: 'fffbbbC',
    0xff49: 'ooooaa19',
    0xff4a: 'eeeaaaB',
    0xff4b: 'llllbb09',
    0xff4c: 'hhhaabC',
    0xff4d: 'iiiiab09',
    0xff4e: 'cccabbC',
    0xff4f: 'jjjjba19',
    0xff50: 'bbbbaaB',
    0xff51: 'kkkkba19',
    0xff52: 'aaaabbC',
    0xff53: 'hhhhab09',
    0xff54: 'dddbbbC',
    0xff55: 'mmmmbb09',
    0xff56: 'gggaaaB',
    0xff57: 'nnnnaa19',
    0xff58: 'fffaaaB',
    0xff59: 'ggggba09',
    0xff5a: 'eeebbbC',
    0xff5b: 'ddddab19',
    0xff5c: 'hhhbbaB',
    0xff5d: 'aaaabb19',
    0xff5e: 'cccbaaB',
    0xff5f: 'bbbbaa09',
    0xff60: 'bbbaaaB',
    0xff61: 'ccccbb19',
    0xff62: 'aaabbbC',
    0xff63: 'ppppba18',
    0xff64: 'dddabbC',
    0xff65: 'eeeeba09',
    0xff66: 'gggbaaB',
    0xff67: 'ffffab19',
    0xff68: 'fffbaaB',
    0xff69: 'oooobb09',
    0xff6a: 'eeeabbC',
    0xff6b: 'llllaa19',
    0xff6c: 'hhhabaB',
    0xff6d: 'iiiiba19',
    0xff6e: 'cccaaaB',
    0xff6f: 'jjjjab09',
    0xff70: 'bbbbbbC',
    0xff71: 'kkkkab09',
    0xff72: 'aaaaaaB',
    0xff73: 'hhhhba19',
    0xff74: 'dddbaaB',
    0xff75: 'mmmmaa19',
    0xff76: 'gggabbC',
    0xff77: 'nnnnbb09',
    0xff78: 'dddddda000',
    0xff79: 'ggggab19',
    0xff7a: 'ddddddab9',
    0xff7b: 'ddddba09',
    0xff7c: 'aaaaaabbaL',
    0xff7d: 'aaaaaa09',
    0xff7e: 'aaaaaab000',
    0xff7f: 'bbbbbb19',
    0xff80: 'bbbabaC',
    0xff81: 'ccccab09',
    0xff82: 'aaababB',
    0xff83: 'ppppaa08',
    0xff84: 'dddaabB',
    0xff85: 'eeeeaa19',
    0xff86: 'eeeeeeb100',
    0xff87: 'ffffbb09',
    0xff88: 'fffbbaC',
    0xff89: 'ooooab19',
    0xff8a: 'eeeaabB',
    0xff8b: 'llllba09',
    0xff8c: 'hhhaaaC',
    0xff8d: 'iiiiaa09',
    0xff8e: 'cccabaC',
    0xff8f: 'jjjjbb19',
    0xff90: 'bbbbabB',
    0xff91: 'kkkkbb19',
    0xff92: 'aaaabaC',
    0xff93: 'hhhhaa09',
    0xff94: 'dddbbaC',
    0xff95: 'mmmmba09',
    0xff96: 'gggaabB',
    0xff97: 'nnnnab19',
    0xff98: 'fffaabB',
    0xff99: 'ggggbb09',
    0xff9a: 'eeebbaC',
    0xff9b: 'ddddaa19',
    0xff9c: 'hhhbbbB',
    0xff9d: 'aaaaba19',
    0xff9e: 'cccbabB',
    0xff9f: 'bbbbab09',
    0xffa0: 'bbbaabB',
    0xffa1: 'ccccba19',
    0xffa2: 'aaabbaC',
    0xffa3: 'ppppbb18',
    0xffa4: 'dddabaC',
    0xffa5: 'eeeebb09',
    0xffa6: 'gggbabB',
    0xffa7: 'ffffaa19',
    0xffa8: 'fffbabB',
    0xffa9: 'ooooba09',
    0xffaa: 'eeeabaC',
    0xffab: 'llllab19',
    0xffac: 'hhhabbB',
    0xffad: 'iiiibb19',
    0xffae: 'cccaabB',
    0xffaf: 'jjjjaa09',
    0xffb0: 'bbbbbaC',
    0xffb1: 'kkkkaa09',
    0xffb2: 'aaaaabB',
    0xffb3: 'hhhhbb19',
    0xffb4: 'dddbabB',
    0xffb5: 'mmmmab19',
    0xffb6: 'gggabaC',
    0xffb7: 'nnnnba09',
    0xffb8: 'fffabaC',
    0xffb9: 'ggggaa19',
    0xffba: 'eeebabB',
    0xffbb: 'ddddbb09',
    0xffbc: 'hhhbaaC',
    0xffbd: 'aaaaab09',
    0xffbe: 'cccbbaC',
    0xffbf: 'bbbbba19',
    0xffc0: 'bbbabbB',
    0xffc1: 'ccccaa19',
    0xffc2: 'aaabaaC',
    0xffc3: 'ppppab18',
    0xffc4: 'dddaaaC',
    0xffc5: 'eeeeab09',
    0xffc6: 'gggbbbB',
    0xffc7: 'ffffba19',
    0xffc8: 'fffbbbB',
    0xffc9: 'ooooaa09',
    0xffca: 'eeeaaaC',
    0xffcb: 'llllbb19',
    0xffcc: 'hhhaabB',
    0xffcd: 'iiiiab19',
    0xffce: 'cccabbB',
    0xffcf: 'jjjjba09',
    0xffd0: 'bbbbaaC',
    0xffd1: 'kkkkba09',
    0xffd2: 'aaaabbB',
    0xffd3: 'hhhhab19',
    0xffd4: 'dddbbbB',
    0xffd5: 'mmmmbb19',
    0xffd6: 'gggaaaC',
    0xffd7: 'nnnnaa09',
    0xffd8: 'fffaaaC',
    0xffd9: 'ggggba19',
    0xffda: 'eeebbbB',
    0xffdb: 'ddddab09',
    0xffdc: 'hhhbbaC',
    0xffdd: 'aaaabb09',
    0xffde: 'cccbaaC',
    0xffdf: 'bbbbaa19',
    0xffe0: 'bbbaaaC',
    0xffe1: 'ccccbb09',
    0xffe2: 'aaabbbB',
    0xffe3: 'ppppba08',
    0xffe4: 'dddabbB',
    0xffe5: 'eeeeba19',
    0xffe6: 'gggbaaC',
    0xffe7: 'ffffab09',
    0xffe8: 'fffbaaC',
    0xffe9: 'oooobb19',
    0xffea: 'eeeabbB',
    0xffeb: 'llllaa09',
    0xffec: 'hhhabaC',
    0xffed: 'iiiiba09',
    0xffee: 'cccaaaC',
    0xffef: 'jjjjab19',
    0xfff0: 'aaaaaaaababj',
    0xfff1: 'aaaaaaaababb',
    0xfff2: 'aaaaaaC',
    0xfff3: 'aaaaaaaababr',
    0xfff4: 'aaaaaaaababJ',
    0xfff5: 'aaaaaaaababB',
    0xfff6: 'aaaaaaaababZ',
    0xfff7: 'aaaaaaaababR',
    0xfff8: 'aaaaaaaabaF',
    0xfff9: 'aaaaaaaabaV',
    0xfffa: 'aaaaaaaabaf',
    0xfffb: 'aaaaaaaabav',
    0xfffc: 'aaaaaaaabM',
    0xfffd: 'aaaaaa19',
    0xfffe: 'aaaaaaa100',
    0xffff: 'aaaaaaaaH8',
}

def OfficeProtectionSub(data, oOutput, options, name=''):
    root, dXMLNS = AnalyzeXMLNS(data)
    for element in root.iter():
        tag = TransformTag(element.tag, dXMLNS, options.includeuri)
        attributes = {TransformTag(key, dXMLNS, options.includeuri): value for key, value in element.attrib.items()}
        if 'algorithmName' in attributes and 'hashValue' in attributes and 'saltValue' in attributes and 'spinCount' in attributes and attributes['algorithmName'] == 'SHA-512':
            oOutput.SeparatingLine('')
            if name != '':
                oOutput.Line(name)
            oOutput.Line(tag)
            hashcathash = '$office$2016$0$%s$%s$%s' % (attributes['spinCount'], attributes['saltValue'], attributes['hashValue'])
            oOutput.Line(hashcathash)
            oOutput.Line('hashcat straigth mode: hashcat.exe -a 0 -m 25300 %s yourwordlist.txt' % hashcathash)
        elif 'workbookAlgorithmName' in attributes and 'workbookHashValue' in attributes and 'workbookSaltValue' in attributes and 'workbookSpinCount' in attributes and attributes['workbookAlgorithmName'] == 'SHA-512':
            oOutput.SeparatingLine('')
            if name != '':
                oOutput.Line(name)
            oOutput.Line(tag)
            hashcathash = '$office$2016$0$%s$%s$%s' % (attributes['workbookSpinCount'], attributes['workbookSaltValue'], attributes['workbookHashValue'])
            oOutput.Line(hashcathash)
            oOutput.Line('hashcat straigth mode: hashcat.exe -a 0 -m 25300 %s yourwordlist.txt' % hashcathash)
        elif 'password' in attributes:
            oOutput.SeparatingLine('')
            if name != '':
                oOutput.Line(name)
            oOutput.Line(tag)
            verifier = int(attributes['password'], 16)
            oOutput.Line('password obtained from verifier table: verifier 0x%04x -> %s' % (verifier, dVerifier[verifier]))
        elif 'w:cryptAlgorithmSid' in attributes and attributes['w:cryptAlgorithmSid'] == '14':
            oOutput.SeparatingLine('')
            if name != '':
                oOutput.Line(name)
            oOutput.Line(tag)
            oOutput.Line('%s: %s' % ('w:cryptAlgorithmSid', attributes['w:cryptAlgorithmSid']))
            oOutput.Line('%s: %s' % ('w:cryptSpinCount', attributes['w:cryptSpinCount']))
            oOutput.Line('%s: %s' % ('w:hash', attributes['w:hash']))
            oOutput.Line('%s: %s' % ('w:salt', attributes['w:salt']))

def OfficeProtection(data, oOutput, options):
    if options.jsoninput != '':
        items = CheckJSON(data)
        if items == None:
            return
        for item in items:
            if item['name'].endswith('.xml'):
                OfficeProtectionSub(item['content'].decode(), oOutput, options, item['name'])
    else:
        OfficeProtectionSub(data, oOutput, options)

dCommands = {'text': ExtractText, 'wordtext': ExtractWordText, 'elementtext': ExtractElementText, 'attributes': ExtractElementAttributes, 'pretty': PrettyPrint, 'celltext': ExtractCellText, 'officeprotection': OfficeProtection}

def ProcessTextFileSingle(command, filenames, oOutput, options):
    for filename in filenames:
        if filename == '':
            if options.encoding != '':
                sys.stdin.reconfigure(encoding=options.encoding)
            fIn = sys.stdin
        else:
            if options.encoding == '':
                fIn = open(filename, 'r')
            else:
                fIn = open(filename, 'r', encoding=options.encoding)
        data = fIn.read()
        if fIn != sys.stdin:
            fIn.close()

        dCommands[command](data, oOutput, options)

def ProcessTextFile(command, filenames, options):
    oOutput = cOutputResult(options)
    ProcessTextFileSingle(command, filenames, oOutput, options)
    oOutput.Close()

def Main():
    moredesc =  '\nCommands:\n%s' % ' '.join(dCommands.keys()) + '''

Arguments:
@file: process each file listed in the text file specified
wildcards are supported

Source code put in the public domain by Didier Stevens, no Copyright
Use at your own risk
https://DidierStevens.com'''

    oParser = optparse.OptionParser(usage='usage: %prog [options] command [[@]file ...]\n' + __description__ + moredesc, version='%prog ' + __version__)
    oParser.add_option('-m', '--man', action='store_true', default=False, help='Print manual')
    oParser.add_option('-u', '--includeuri', action='store_true', default=False, help='Include URI for the tags')
    oParser.add_option('-o', '--output', type=str, default='', help='Output to file')
    oParser.add_option('-j', '--jsoninput', type=str, default='', help='JSON input')
    oParser.add_option('--encoding', type=str, default='', help='Encoding for file open')
    (options, args) = oParser.parse_args()

    if options.man:
        oParser.print_help()
        PrintManual()
        return

    if len(args) == 0:
        oParser.print_help()
        print('')
        print('  %s' % __description__)
        return

    command = args[0]

    if not command in dCommands:
        print('Invalid command: %s' % command)
        print('List of valid commands: %s' % ' '.join(dCommands.keys()))
        return

    try:
        if len(args) == 1:
            ProcessTextFile(command, [''], options)
        else:
            ProcessTextFile(command, ExpandFilenameArguments(args), options)
    except BrokenPipeError:
        pass

if __name__ == '__main__':
    Main()
