from bs4 import BeautifulSoup
import json
import re
import sys
from urllib.parse import urljoin
from urllib.parse import urlparse

__author__ = "Benoit Maurin <bmaurin@immunit.ch> - Jean Lejeune <jlejeune@immunit.ch>"
__copyright__ = "Copyright 2019, ImmunIT"


class Helper:
    """This class contains utilities whose aim is to determine working nodes as well as their types.
    """
    
    def __init__(self, target, request, logger):
        self.target = target
        self.request = request
        self.logger = logger

    def setup(self):
        """Helper entrypoint.
        """

        self.nodes = self._find_nodes()

        for node in self.nodes:
            self.typ = self._guess_type(node)
            if self.typ:
                return True

        return False

    def _guess_type(self, node):
        """Get the Drupal object type from the returned json.

        Parameters
        ----------
        node : str
            Node path

        Return
        ------
        The type detected.
        """

        res = self.request.get(node + "?_format=hal_json", {'Content-Type': 'application/hal+json'})
        if res.status_code != 200: return None
        typ =  json.loads(res.text)['_links']['type']['href']
        pos = typ.find('/rest/type/')
        if pos == -1:
            return None

        return typ[:pos] + '/rest/type/shortcut/default'
            
    def _find_nodes(self):
        """Find nodes in the home page.
        """

        content = self.request.get().text
        soup = BeautifulSoup(content, 'html.parser')
        
        seen_nodes = set()
        for link in soup.find_all('a'):
            abs_link = urljoin(self.target, link.get('href'))
            if not abs_link.startswith(self.target): continue
            match = re.search('/node/(\d)+', abs_link[len(self.target):])
            if not match: continue
            seen_nodes.add(int(match.group(1)))

        self.logger.handle("\n[*] %d nodes has been found. Starting upstream enumeration to increase the success ratio..." % len(seen_nodes), None)

        for node in seen_nodes:
            if self.request.get(('/node/%d' % node)).status_code == 200:
                 yield ('/node/%d' % node)

        if len(seen_nodes) == 0:
            seen_nodes.add(1)
        for i in range(min(seen_nodes), min(seen_nodes) + 1000):
            if self.request.get(('/node/%d' % i)).status_code == 200 and i not in seen_nodes:
                seen_nodes.add(i)
                yield ('/node/%d' % i)
        