﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;

namespace Debugger
{
    [StructLayout(LayoutKind.Sequential)]
    public struct ExceptionInfo
    {
        private static DisassemblerWrapper _disassembler = new DisassemblerWrapper();

        public uint ExceptionCode;
        public string ExceptionMessage;
        public uint ExceptionAddress;
        public AccessViolationType AccessViolationType;
        public uint AccessViolationAddress;
        //[MarshalAs(UnmanagedType.LPStruct)]
        public CONTEXT Context;
        
        public IntPtr ExceptionInstructions;
        public int ExceptionInstructionCount;

        public string DisassembleInstructions()
        {
            byte[] buffer = new byte[ExceptionInstructionCount];
            Marshal.Copy(ExceptionInstructions, buffer, 0, buffer.Length);

            var asm = _disassembler.Disassemble(buffer, ExceptionAddress);

            if (string.IsNullOrEmpty(asm))
                return "";

            var lines = asm.SplitLines(true);

            return lines
                .Take(lines.Length - 1)
                .Aggregate((x, y) => x + "\r\n" + y);
        }

        public override string ToString()
        {
            var exceptionMessage = "Exception:\r\n\r\n" +
                
                ExceptionMessage + "\r\n\r\n" +
                
                string.Format("Edi = {0:X8}\r\n" +
                    "Esi = {1:X8}\r\n" +
                    "Ebx = {2:X8}\r\n" +
                    "Edx = {3:X8}\r\n" +
                    "Ecx = {4:X8}\r\n" +
                    "Eax = {5:X8}\r\n" +
                    "Ebp = {6:X8}\r\n" +
                    "Esp = {7:X8}\r\n" +
                    "Eip = {8:X8}\r\n\r\n",
                    Context.Edi, Context.Esi, Context.Ebx, Context.Edx,
                    Context.Ecx, Context.Eax, Context.Ebp, Context.Esp,
                    Context.Eip);

            if (AccessViolationType != AccessViolationType.None)
                exceptionMessage += 
                    string.Format("{0} access violation at {1:X8}\r\n\r\n",
                        AccessViolationType, AccessViolationAddress);

            var asm = DisassembleInstructions();

            if (!string.IsNullOrEmpty(asm))
                exceptionMessage += "Disassembly\r\n\r\n" + asm + "\r\n\r\n";

            return exceptionMessage;
        }
    }
}
