// bittorrent.h
//

#ifndef BITTORRENT_H
#define BITTORRENT_H

#include <vector>

#include "bencode.h"
#include "datum.h"
#include "json_object.h"
#include "lex.h"
#include "match.h"
#include "newhttp.h"
#include "protocol.h"

// For the DHT protocol, there are four queries: ping, find_node,
// get_peers, and announce_peer.
// Every message has a key "t" with a string value representing a transaction ID.
// This transaction ID is generated by the querying node and is echoed in the
//response, so responses may be correlated with multiple queries to the same
// node. The transaction ID should be encoded as a short string of binary
// numbers, typically 2 characters are enough as they cover 2^16
// outstanding queries. Every message also has a key "y" with a single
// character value describing the type of message. The value of the "y" key
// is one of "q" for query, "r" for response, or "e" for error. A key "v"
// should be included in every message with a client version string.

class bittorrent_dht : public base_protocol {
    bencoding::dictionary dict;
public:
    static inline bool output_raw_features = false;
    static void set_raw_features(bool value) { output_raw_features = value; }

    bittorrent_dht (datum &d) : dict(d) { }

    void write_raw_features(json_object &o) {
        data_buffer<2048> buf;
        dict.write_raw_features(buf);
        if (buf.readable_length() == 0) {
            o.print_key_string("features", "[]");
        } else {
            o.print_key_json_string("features", buf.contents());
        }
    }

    void write_json(struct json_object &o, bool) {
        if (this->is_not_empty()) {
            struct json_object dht{o, "bittorrent_dht"};
            dict.write_json(dht);
            if (output_raw_features) {
                write_raw_features(dht);
            }
            dht.close();
        }
    }

    void write_l7_metadata(cbor_object &o, bool) {
        cbor_array protocols{o, "protocols"};
        protocols.print_string("bittorrent_dht");
        protocols.close();
    }

    bool is_not_empty() { return dict.is_not_empty(); }

    static constexpr mask_and_value<8> matcher {
        {0xff, 0xff, 0xff, 0x8c, 0xff, 0xff, 0xff, 0xff},
        {'d', '1', ':', 0x00, 'd', '2', ':', 'i'}
    };
};

// Local Service Discovery (LSD) uses the following multicast groups:
// A) 239.192.152.143:6771 (org-local) and B) [ff15::efc0:988f]:6771
// (site-local)
//
// Implementation note: Since the multicast groups have a broader
// scope than lan-local implementations may want to set the
// IP_MULTICAST_TTL socket option to a value above 1
//
//   An LSD announce is formatted as follows:
//
//   BT-SEARCH * HTTP/1.1\r\n
//   Host: <host>\r\n
//   Port: <port>\r\n
//   Infohash: <ihash>\r\n
//   cookie: <cookie (optional)>\r\n
//   \r\n
//   \r\n
//
// host: RFC 2616 section 14.23 and RFC 2732 compliant Host header
// specifying the multicast group to which the announce is sent. In
// other words, strings A) or B), as appropriate.
//
// port: port on which the bittorrent client is listening in base-10,
// ascii
//
// ihash: hex-encoded (40 character) infohash.  An announce may
// contain multiple, consecutive Infohash headers to announce the
// participation in more than one torrent. This may not be supported
// by older implementations. When sending multiple infohashes the
// packet length should not exceed 1400 bytes to avoid
// MTU/fragmentation problems.
//
// cookie: opaque value, allowing the sending client to filter out its
// own announces if it receives them via multicast loopback

class header {
    newhttp::token name;
    ignore_char_class<space> sp;
    newhttp::text value;
    bool valid;

public:

    header(datum &d) :
            name{d},
            sp{d},
            value{d},
            valid{d.is_not_null()} { }

    void write_raw_features(writeable &w) const {
        w.copy('[');
        w.write_quote_enclosed_hex(name.data, name.length());
        w.copy(',');
        w.write_quote_enclosed_hex(value.data, value.length());
        w.copy(']');
    }

    void write_json(json_array &a) const{
        if (is_not_empty()) {
            json_object hdr{a};
            hdr.print_key_json_string("key", name);
            hdr.print_key_json_string("value", value);
            hdr.close();
        }
    }

    bool is_not_empty() const { return valid; }
};

class lsd_header {
    std::vector<header> headers;
    static constexpr uint8_t max_headers = 10;
    bool valid = false;

public:
    lsd_header(datum &d) {
        headers.reserve(max_headers);
        while(d.is_not_empty()) {
            if (lookahead<newhttp::crlf> at_end{d}) {
                break;
            }

            header h{d};
            if (h.is_not_empty()) {
                headers.emplace_back(h);
            }

            newhttp::crlf ignore{d};
        }
        if (headers.size()) {
            valid = true;
        }
    }

    void write_raw_features(writeable &w) const {
        if (!valid) {
            return;
        }

        w.copy('[');
        bool first = true;
        for (auto &hdr : headers) {
            if (!first) {
                w.copy(',');
            } else {
                first = false;
            }

            hdr.write_raw_features(w);
        }
        w.copy(']');
    }

    void write_json(json_object &o) const {
        if (!valid) {
            return;
        }

        json_array hdrs{o, "headers"};
        for (auto &hdr : headers) {
            hdr.write_json(hdrs);
        }
        hdrs.close();
    }
};

class bittorrent_lsd : public base_protocol {
    literal_byte<'B', 'T', '-', 'S', 'E', 'A', 'R', 'C', 'H'> proto;
    ignore_char_class<space> sp1;
    literal_byte<'*'> asterisk;
    ignore_char_class<space> sp2;
    newhttp::version version;
    newhttp::crlf crlf;
    lsd_header headers;
    bool valid;

public:

    static inline bool output_raw_features = false;
    static void set_raw_features(bool value) { output_raw_features = value; }

    bittorrent_lsd(datum &d) :
        proto{d},
        sp1(d),
        asterisk{d},
        sp2(d),
        version(d),
        crlf(d),
        headers{d},
        valid{d.is_not_null()} { }

    bool is_not_empty() { return valid; }

    void write_raw_features(json_object &o) const {
        data_buffer<2048> buf;
        buf.copy('[');
        buf.write_quote_enclosed_hex(version.data, version.length());
        buf.copy(',');
        headers.write_raw_features(buf);
        buf.copy(']');
        if (buf.readable_length() == 0) {
            o.print_key_string("features", "[]");
        } else {
            o.print_key_json_string("features", buf.contents());
        }
    }

    void write_json(struct json_object &o, bool) const {
        if (valid) {
            struct json_object lsd{o, "bittorrent_lsd"};
            lsd.print_key_json_string("version", version);
            headers.write_json(lsd);
            if (output_raw_features) {
                write_raw_features(o);
            }
            lsd.close();
        }
    }

    void write_l7_metadata(cbor_object &o, bool) {
        cbor_array protocols{o, "protocols"};
        protocols.print_string("bittorrent_lsd");
        protocols.close();
    }

    static constexpr mask_and_value<8> matcher {
        {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
        {'B', 'T', '-', 'S', 'E', 'A', 'R', 'C'}
    };

};

class bittorrent_peer_message {
    encoded<uint32_t> message_length;
    encoded<uint8_t> message_type;
    datum message;
    bool valid;

public:

    bittorrent_peer_message(datum &d) :
        message_length{d},
        message_type{d},
        message{d, (ssize_t)(message_length - 1)},
        valid{d.is_not_null()}
    {
    }

    const char* get_code_str() const {
        switch(message_type.value()) {
        case 0x00:          return "choke";
        case 0x01:          return "unchoke";
        case 0x02:          return "interested";
        case 0x03:          return "not_interested";
        case 0x04:          return "have";
        case 0x05:          return "bit_field";
        case 0x06:          return "request";
        case 0x07:          return "piece";
        case 0x08:          return "cancel";
        case 0x14:          return "extended";
        default:            return nullptr;
        }
    }

    uint8_t get_code() const {return message_type.value();}

    void write_raw_features(writeable &buf) const {
        buf.copy('[');
        buf.write_quote_enclosed_hex(message_type);
        buf.copy(',');
        buf.write_quote_enclosed_hex(message);
        buf.copy(']');
    }

    void write_json(struct json_array &o) {
        if(!valid) {
            return;
        }

        struct json_object msg{o};
        msg.print_key_uint("message_length", message_length);
        type_codes<bittorrent_peer_message> code{*this};
        msg.print_key_value("message_type", code);
        msg.print_key_hex("message", message);
        msg.close();
    }
};

class bittorrent_peer_messages{
    std::vector<bittorrent_peer_message> messages;
    bool valid = false;

public:
    bittorrent_peer_messages(datum &d) {
        messages.reserve(20); //reserve for 20 messages

        while(d.is_not_empty()) {
            messages.emplace_back(bittorrent_peer_message(d));
        }

        if (messages.size()) {
            valid = true;
        }
    }

    void write_raw_features(writeable &buf) const {
        buf.copy('[');
        bool first = true;
        for (const auto &msg : messages) {
            if (!first) {
                buf.copy(',');
            } else {
                first = false;
            }
            msg.write_raw_features(buf);
        }
        buf.copy(']');
    }

    void write_json(struct json_object &o) {
        if (!valid) {
            return;
        }

        struct json_array msgs{o, "messages"};
        for (auto msg : messages) {
            msg.write_json(msgs);
        }
        msgs.close();
    }
};

//
// The peer wire protocol consists of a handshake followed by a
// never-ending stream of length-prefixed messages. The handshake
// starts with character ninteen (decimal) followed by the string
// 'BitTorrent protocol'. The leading character is a length prefix,
// put there in the hope that other new protocols may do the same and
// thus be trivially distinguishable from each other.
// All later integers sent in the protocol are encoded as four bytes
// big-endian.
//
// After the fixed headers come eight reserved bytes, which are all
// zero in all current implementations. If you wish to extend the
// protocol using these bytes, please coordinate with Bram Cohen to
// make sure all extensions are done compatibly.
//
// Next comes the 20 byte sha1 hash of the bencoded form of the info
// value from the metainfo file. (This is the same value which is
// announced as info_hash to the tracker, only here it's raw instead
// of quoted here). If both sides don't send the same value, they
// sever the connection. The one possible exception is if a downloader
// wants to do multiple downloads over a single port, they may wait
// for incoming connections to give a download hash first, and respond
// with the same one if it's in their list.
//
// After the download hash comes the 20-byte peer id which is reported
// in tracker requests and contained in peer lists in tracker
// responses. If the receiving side's peer id doesn't match the one
// the initiating side expects, it severs the connection.
//
// That's it for handshaking, next comes an alternating stream of
// length prefixes and messages. Messages of length zero are
// keepalives, and ignored. Keepalives are generally sent once every
// two minutes, but note that timeouts can be done much more quickly
// when data is expected.

class bittorrent_handshake : public base_protocol {
    literal_byte<0x13, 'B', 'i', 't', 'T', 'o', 'r', 'r', 'e', 'n', 't', ' ', 'p', 'r', 'o', 't', 'o', 'c', 'o', 'l'> protocol;
    datum extension_bytes;
    datum hash_of_info_dict;
    datum peer_id;
    bittorrent_peer_messages msgs;
    bool valid;

public:

    static inline bool output_raw_features = false;
    static void set_raw_features(bool value) { output_raw_features = value; }

    bittorrent_handshake(datum &d) :
        protocol{d},
        extension_bytes{d, 8},
        hash_of_info_dict{d, 20},
        peer_id{d, 20},
        msgs{d},
        valid{d.is_not_null()} { }

    bool is_not_empty() const { return valid; }

    static constexpr mask_and_value<8> matcher {
        {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
        {0x13, 'B', 'i', 't', 'T', 'o', 'r', 'r'}
    };

    void write_raw_features(json_object &o) const {
        data_buffer<2048> buf;
        buf.copy('[');
        buf.write_quote_enclosed_hex(extension_bytes);
        buf.copy(',');
        buf.write_quote_enclosed_hex(hash_of_info_dict);
        buf.copy(',');
        buf.write_quote_enclosed_hex(peer_id);
        buf.copy(',');
        msgs.write_raw_features(buf);
        buf.copy(']');
        if (buf.readable_length() == 0) {
            o.print_key_string("features", "[]");
        } else {
            o.print_key_json_string("features", buf.contents());
        }
    }

    void write_json(struct json_object &o, bool) {
        if (this->is_not_empty()) {
            struct json_object bt{o, "bittorrent"};
            bt.print_key_hex("extension_bytes", extension_bytes);
            bt.print_key_hex("info_dict", hash_of_info_dict);
            bt.print_key_hex("peer_id", peer_id);
            msgs.write_json(o);
            if (output_raw_features) {
                write_raw_features(o);
            }
            bt.close();
        }
    }

    void write_l7_metadata(cbor_object &o, bool) {
        cbor_array protocols{o, "protocols"};
        protocols.print_string("bittorrent");
        protocols.close();
    }

    void fprint(FILE *f) const {
        fprintf(f, "extension_bytes:   ");  extension_bytes.fprint_hex(f);   fputc('\n', f);
        fprintf(f, "hash_of_info_dict: ");  hash_of_info_dict.fprint_hex(f); fputc('\n', f);
        fprintf(f, "peer_id:           ");  peer_id.fprint_hex(f);           fputc('\n', f);
    }
};

[[maybe_unused]] inline static int bittorrent_dht_fuzz_test(const uint8_t *data, size_t size) {
    struct datum request_data{data, data+size};
    char buffer[8192];
    struct buffer_stream buf_json(buffer, sizeof(buffer));
    struct json_object record(&buf_json);

    bittorrent_dht dht{request_data};
    if (dht.is_not_empty()) {
        dht.write_json(record, true);
    }

    return 0;
}

[[maybe_unused]] inline static int bittorrent_lsd_fuzz_test(const uint8_t *data, size_t size) {
    struct datum request_data{data, data+size};
    char buffer[8192];
    struct buffer_stream buf_json(buffer, sizeof(buffer));
    struct json_object record(&buf_json);

    bittorrent_lsd lsd{request_data};
    if (lsd.is_not_empty()) {
        lsd.write_json(record, true);
    }

    return 0;
}

[[maybe_unused]] inline static int bittorrent_handshake_fuzz_test(const uint8_t *data, size_t size) {
    struct datum request_data{data, data+size};
    char buffer[8192];
    struct buffer_stream buf_json(buffer, sizeof(buffer));
    struct json_object record(&buf_json);

    bittorrent_handshake handshake{request_data};
    if (handshake.is_not_empty()) {
        handshake.write_json(record, true);
    }

    return 0;
}
#endif // BITTORRENT_H
