"""Code for generating GraphML files"""

#  Copyright (c) NCC Group and Erik Steringer 2020. This file is part of Principal Mapper.
#
#      Principal Mapper is free software: you can redistribute it and/or modify
#      it under the terms of the GNU Affero General Public License as published by
#      the Free Software Foundation, either version 3 of the License, or
#      (at your option) any later version.
#
#      Principal Mapper is distributed in the hope that it will be useful,
#      but WITHOUT ANY WARRANTY; without even the implied warranty of
#      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#      GNU Affero General Public License for more details.
#
#      You should have received a copy of the GNU Affero General Public License
#      along with Principal Mapper.  If not, see <https://www.gnu.org/licenses/>.

from xml.dom import minidom
import xml.etree.ElementTree as ET
from typing import List, Dict, Optional, Union

import principalmapper
from principalmapper.common import Edge, Node, Graph
from principalmapper.querying.presets.privesc import can_privesc
from principalmapper.querying.presets.serviceaccess import compose_service_access_map


def write_standard_graphml(graph: Graph, filepath: str, with_services: Optional[bool] = False) -> None:
    """The function to generate the standard visualization with a GraphML file: this is all the nodes with the
    admins/privesc highlights in blue/red respectively."""

    node_color_mapping = {}
    for node in graph.nodes:
        if node.is_admin:
            node_color_mapping[node] = 'blue'
        elif can_privesc(graph, node)[0]:
            node_color_mapping[node] = 'red'
        else:
            node_color_mapping[node] = 'white'

    if with_services:
        sam = compose_service_access_map(graph)
    else:
        sam = None


    result = ET.ElementTree(ET.Element(None))
    generate_graphml(node_color_mapping, graph.edges, result, sam)
    result.write(filepath, 'utf-8', True)


def write_privesc_graphml(graph: Graph, filepath: str) -> None:
    """The function to generate the privesc-only visualization with a GraphML file: this is only the nodes that are
    admins/privesc with blue/red highlights."""

    node_color_mapping = {}
    for node in graph.nodes:
        if node.is_admin:
            node_color_mapping[node] = 'blue'
        elif can_privesc(graph, node)[0]:
            node_color_mapping[node] = 'red'

    result = ET.ElementTree(ET.Element(None))
    generate_graphml(node_color_mapping, graph.edges, result)
    result.write(filepath, 'utf-8', True)


def generate_graphml(nodes_and_color: Dict[Union[Node, str], Optional[str]], edges: List[Edge], element_tree: ET.ElementTree, service_data: Optional[Dict[str, List[Node]]] = None) -> None:
    """This function modifies the given element tree to add the passed node/edge data/attributes.

    The nodes_and_color dictionary should have the nodes to draw in the graph mapping to a color to draw the
    node with. Specify None for the default (white)."""

    # add root stuff: comment, <graphml> element
    root_element = element_tree.getroot()
    root_element.append(ET.Comment('GraphML File generated by Principal Mapper v{}'.format(principalmapper.__version__)))
    graphml_element = ET.Element(
        'graphml',
        {
            'xmlns': 'http://graphml.graphdrawing.org/xmlns',
            'xmlns:xsi': 'http://www.w3.org/2001/XMLSchema-instance',
            'xsi:schemaLocation': 'http://graphml.graphdrawing.org/xmlns http://graphml.graphdrawing.org/xmlns/1.0/graphml.xsd'
        }
    )
    root_element.append(graphml_element)

    # add the <key> element with color for nodes: default to white
    data_node_color_element = ET.Element(
        'key',
        {
            'id': 'd0',
            'for': 'node',
            'attr.name': 'color',
            'attr.type': 'string'
        }
    )
    data_node_color_element_default_child = ET.Element('default')
    data_node_color_element_default_child.text = 'white'
    data_node_color_element.append(data_node_color_element_default_child)
    graphml_element.append(data_node_color_element)

    # add the <key> element for labels with nodes
    data_node_label_element = ET.Element(
        'key',
        {
            'id': 'd1',
            'for': 'node',
            'attr.name': 'label',
            'attr.type': 'string'
        }
    )
    graphml_element.append(data_node_label_element)

    # add the <key> element for labels with edges
    data_edge_label_element = ET.Element(
        'key',
        {
            'id': 'd2',
            'for': 'edge',
            'attr.name': 'label',
            'attr.type': 'string'
        }
    )
    graphml_element.append(data_edge_label_element)

    # we create our <graph> element which shall have the child <node>/<edge> elements
    graph_element = ET.Element('graph', {'id': 'G', 'edgedefault': 'directed'})
    graphml_element.append(graph_element)

    # go through each node and append a <node> element to the <graph> element's children
    node_counter = 0
    node_id_mapping = {}  # extra data structure to map Node -> 'n#' for the edge part that's next
    for node, color in nodes_and_color.items():
        node_id = 'n{}'.format(node_counter)
        node_id_mapping[node] = node_id
        node_element = ET.Element('node', {'id': node_id})
        node_label_subelement = ET.Element('data', {'key': 'd1'})
        node_label_subelement.text = node.searchable_name()
        node_element.append(node_label_subelement)
        if color is not None:
            node_color_subelement = ET.Element('data', {'key': 'd0'})
            node_color_subelement.text = color
            node_element.append(node_color_subelement)
        graph_element.append(node_element)
        node_counter += 1

    if service_data is not None:
        for service in service_data.keys():
            node_id = 'n{}'.format(node_counter)
            node_id_mapping[service] = node_id
            node_element = ET.Element('node', {'id': node_id})
            node_label_subelement = ET.Element('data', {'key': 'd1'})
            node_label_subelement.text = service
            node_element.append(node_label_subelement)
            node_color_subelement = ET.Element('data', {'key': 'd0'})
            node_color_subelement.text = 'green'
            node_element.append(node_color_subelement)
            graph_element.append(node_element)
            node_counter += 1

    # go through each edge and append an <edge> element to the <graph> element's children
    edge_counter = 0
    for edge in edges:
        if edge.source not in node_id_mapping or edge.destination not in node_id_mapping:
            continue
        edge_id = 'e{}'.format(edge_counter)
        edge_element = ET.Element(
            'edge',
            {
                'id': edge_id,
                'source': node_id_mapping[edge.source],
                'target': node_id_mapping[edge.destination]
            }
        )
        edge_label_subelement = ET.Element('data', {'key': 'd2'})
        edge_label_subelement.text = edge.short_reason
        edge_element.append(edge_label_subelement)
        graph_element.append(edge_element)
        edge_counter += 1

    if service_data is not None:
        for service, node_list in service_data.items():
            for role_node in node_list:
                if role_node not in node_id_mapping:
                    continue
                edge_id = 'e{}'.format(edge_counter)
                edge_element = ET.Element(
                    'edge',
                    {
                        'id': edge_id,
                        'source': node_id_mapping[service],
                        'target': node_id_mapping[role_node]
                    }
                )
                edge_label_subelement = ET.Element('data', {'key': 'd2'})
                edge_label_subelement.text = 'SERVICE_ACCESS'
                edge_element.append(edge_label_subelement)
                graph_element.append(edge_element)
                edge_counter += 1

