#ifndef GEN_UI_GFX_X_SCREENSAVER_H_
#define GEN_UI_GFX_X_SCREENSAVER_H_

#include <array>
#include <cstddef>
#include <cstdint>
#include <cstring>
#include <vector>

#include "base/component_export.h"
#include "base/memory/ref_counted_memory.h"
#include "base/memory/scoped_refptr.h"
#include "base/optional.h"
#include "base/files/scoped_file.h"
#include "ui/gfx/x/xproto_types.h"
#include "xproto.h"

namespace x11 {

class Connection;

class COMPONENT_EXPORT(X11) ScreenSaver {
  public:
  static constexpr unsigned major_version = 1;
  static constexpr unsigned minor_version = 1;

  ScreenSaver(Connection* connection,
      const x11::QueryExtensionReply& info);

  uint8_t present() const {
    return info_.present;
  }
  uint8_t major_opcode() const {
    return info_.major_opcode;
  }
  uint8_t first_event() const {
    return info_.first_event;
  }
  uint8_t first_error() const {
    return info_.first_error;
  }

  Connection* connection() const { return connection_; }

  enum class Kind : int {
    Blanked = 0,
    Internal = 1,
    External = 2,
  };

  enum class Event : int {
    NotifyMask = 1 << 0,
    CycleMask = 1 << 1,
  };

  enum class State : int {
    Off = 0,
    On = 1,
    Cycle = 2,
    Disabled = 3,
  };

  struct NotifyEvent {
    static constexpr int type_id = 13;
    static constexpr uint8_t opcode = 0;
    bool send_event{};
    State state{};
    uint16_t sequence{};
    Time time{};
    Window root{};
    Window window{};
    Kind kind{};
    uint8_t forced{};

    x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&window); }
  };

  struct QueryVersionRequest {
    uint8_t client_major_version{};
    uint8_t client_minor_version{};
  };

  struct QueryVersionReply {
    uint16_t sequence{};
    uint16_t server_major_version{};
    uint16_t server_minor_version{};
  };

  using QueryVersionResponse = Response<QueryVersionReply>;

  Future<QueryVersionReply> QueryVersion(
      const QueryVersionRequest& request);

  struct QueryInfoRequest {
    Drawable drawable{};
  };

  struct QueryInfoReply {
    uint8_t state{};
    uint16_t sequence{};
    Window saver_window{};
    uint32_t ms_until_server{};
    uint32_t ms_since_user_input{};
    uint32_t event_mask{};
    Kind kind{};
  };

  using QueryInfoResponse = Response<QueryInfoReply>;

  Future<QueryInfoReply> QueryInfo(
      const QueryInfoRequest& request);

  struct SelectInputRequest {
    Drawable drawable{};
    Event event_mask{};
  };

  using SelectInputResponse = Response<void>;

  Future<void> SelectInput(
      const SelectInputRequest& request);

  struct SetAttributesRequest {
    Drawable drawable{};
    int16_t x{};
    int16_t y{};
    uint16_t width{};
    uint16_t height{};
    uint16_t border_width{};
    WindowClass c_class{};
    uint8_t depth{};
    VisualId visual{};
    base::Optional<Pixmap> background_pixmap{};
    base::Optional<uint32_t> background_pixel{};
    base::Optional<Pixmap> border_pixmap{};
    base::Optional<uint32_t> border_pixel{};
    base::Optional<Gravity> bit_gravity{};
    base::Optional<Gravity> win_gravity{};
    base::Optional<BackingStore> backing_store{};
    base::Optional<uint32_t> backing_planes{};
    base::Optional<uint32_t> backing_pixel{};
    base::Optional<Bool32> override_redirect{};
    base::Optional<Bool32> save_under{};
    base::Optional<EventMask> event_mask{};
    base::Optional<EventMask> do_not_propogate_mask{};
    base::Optional<ColorMap> colormap{};
    base::Optional<Cursor> cursor{};
  };

  using SetAttributesResponse = Response<void>;

  Future<void> SetAttributes(
      const SetAttributesRequest& request);

  struct UnsetAttributesRequest {
    Drawable drawable{};
  };

  using UnsetAttributesResponse = Response<void>;

  Future<void> UnsetAttributes(
      const UnsetAttributesRequest& request);

  struct SuspendRequest {
    uint32_t suspend{};
  };

  using SuspendResponse = Response<void>;

  Future<void> Suspend(
      const SuspendRequest& request);

  private:
  x11::Connection* const connection_;
  x11::QueryExtensionReply info_{};
};

}  // namespace x11

inline constexpr x11::ScreenSaver::Kind operator|(
    x11::ScreenSaver::Kind l, x11::ScreenSaver::Kind r) {
  using T = std::underlying_type_t<x11::ScreenSaver::Kind>;
  return static_cast<x11::ScreenSaver::Kind>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ScreenSaver::Kind operator&(
    x11::ScreenSaver::Kind l, x11::ScreenSaver::Kind r) {
  using T = std::underlying_type_t<x11::ScreenSaver::Kind>;
  return static_cast<x11::ScreenSaver::Kind>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ScreenSaver::Event operator|(
    x11::ScreenSaver::Event l, x11::ScreenSaver::Event r) {
  using T = std::underlying_type_t<x11::ScreenSaver::Event>;
  return static_cast<x11::ScreenSaver::Event>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ScreenSaver::Event operator&(
    x11::ScreenSaver::Event l, x11::ScreenSaver::Event r) {
  using T = std::underlying_type_t<x11::ScreenSaver::Event>;
  return static_cast<x11::ScreenSaver::Event>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ScreenSaver::State operator|(
    x11::ScreenSaver::State l, x11::ScreenSaver::State r) {
  using T = std::underlying_type_t<x11::ScreenSaver::State>;
  return static_cast<x11::ScreenSaver::State>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ScreenSaver::State operator&(
    x11::ScreenSaver::State l, x11::ScreenSaver::State r) {
  using T = std::underlying_type_t<x11::ScreenSaver::State>;
  return static_cast<x11::ScreenSaver::State>(
      static_cast<T>(l) & static_cast<T>(r));
}


#endif  // GEN_UI_GFX_X_SCREENSAVER_H_
