// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef STORAGE_BROWSER_FILE_SYSTEM_FILE_SYSTEM_QUOTA_UTIL_H_
#define STORAGE_BROWSER_FILE_SYSTEM_FILE_SYSTEM_QUOTA_UTIL_H_

#include <stdint.h>

#include <string>
#include <vector>

#include "base/component_export.h"
#include "base/files/file.h"
#include "base/memory/scoped_refptr.h"
#include "storage/common/file_system/file_system_types.h"

namespace url {
class Origin;
}

namespace storage {

class FileSystemContext;
class QuotaManagerProxy;
class QuotaReservation;

// An abstract interface that provides common quota-related utility functions
// for file_system_quota_client.
// All the methods of this class are synchronous and need to be called on
// the thread that the method name implies.
class COMPONENT_EXPORT(STORAGE_BROWSER) FileSystemQuotaUtil {
 public:
  virtual ~FileSystemQuotaUtil() = default;

  // Deletes the data on the origin and reports the amount of deleted data
  // to the quota manager via |proxy|.
  virtual base::File::Error DeleteOriginDataOnFileTaskRunner(
      FileSystemContext* context,
      QuotaManagerProxy* proxy,
      const url::Origin& origin,
      FileSystemType type) = 0;

  virtual void PerformStorageCleanupOnFileTaskRunner(FileSystemContext* context,
                                                     QuotaManagerProxy* proxy,
                                                     FileSystemType type) = 0;

  virtual std::vector<url::Origin> GetOriginsForTypeOnFileTaskRunner(
      FileSystemType type) = 0;

  virtual std::vector<url::Origin> GetOriginsForHostOnFileTaskRunner(
      FileSystemType type,
      const std::string& host) = 0;

  // Returns the amount of data used for the origin for usage tracking.
  virtual int64_t GetOriginUsageOnFileTaskRunner(
      FileSystemContext* file_system_context,
      const url::Origin& origin,
      FileSystemType type) = 0;

  // Creates new reservation object for the origin and the type.
  virtual scoped_refptr<QuotaReservation>
  CreateQuotaReservationOnFileTaskRunner(const url::Origin& origin,
                                         FileSystemType type) = 0;
};

}  // namespace storage

#endif  // STORAGE_BROWSER_FILE_SYSTEM_FILE_SYSTEM_QUOTA_UTIL_H_
