/*
** Copyright (C) 2006-2025 Carnegie Mellon University
**
** @OPENSOURCE_LICENSE_START@
**
** SiLK 3.24
**
** Copyright 2025 Carnegie Mellon University.
**
** NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
** INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
** UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
** IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
** FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
** OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
** MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
** TRADEMARK, OR COPYRIGHT INFRINGEMENT.
**
** Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
** contact permission@sei.cmu.edu for full terms.
**
** [DISTRIBUTION STATEMENT A] This material has been approved for public
** release and unlimited distribution.  Please see Copyright notice for
** non-US Government use and distribution.
**
** This Software includes and/or makes use of Third-Party Software each
** subject to its own license.
**
** DM25-0915
**
** @OPENSOURCE_LICENSE_END@
*/
#ifndef _RWSCAN_WORKQUEUE_H
#define _RWSCAN_WORKQUEUE_H
#ifdef __cplusplus
extern "C" {
#endif

#include <silk/silk.h>

RCSIDENTVAR(rcsID_RWSCAN_WORKQUEUE_H, "$SiLK: rwscan_workqueue.h c7d6bb438741 2025-01-17 20:52:04Z mthomas $");


typedef struct work_queue_node_st {
    struct work_queue_node_st *next;       /* next request in queue */
} work_queue_node_t;

/*
 * This threaded queue structure is specialized for a
 * producer/consumer design in two ways.  First, queues can be created
 * with a maximum queue depth parameter, which governs how large the
 * queue can grow in size.  Second, the queue can be "deactivated" to
 * shut down producer threads when the program exits.
 *
 * The queue just maintains node pointers; it does not manage node
 * memory in any way.
 *
 */
typedef struct work_queue_st {
    work_queue_node_t *head;        /* pointer to first node */
    work_queue_node_t *tail;        /* pointer to last node */

    pthread_mutex_t    mutex;       /* master queue lock mutex */
    pthread_cond_t     cond_posted; /* used to wake up a consumer */
    pthread_cond_t     cond_avail;  /* used to signal a producer */

    int                depth;       /* number of items in queue */
    int                maxdepth;    /* max items allowed in queue */
    int                pending;     /* numitems being processed */
    int                active;      /* if work queue has been activated */
#ifdef RWSCN_WORKQUEUE_DEBUG
    int                consumed;    /* num items consumed */
    int                produced;    /* num items posted by producers */
    int                peakdepth;   /* highest queue depth */
#endif
} work_queue_t;


/* Public work queue API */
work_queue_t *
workqueue_create(
    uint32_t            maxdepth);
int
workqueue_put(
    work_queue_t       *q,
    work_queue_node_t  *newnode);
int
workqueue_get(
    work_queue_t       *q,
    work_queue_node_t **retnode);
int
workqueue_depth(
    work_queue_t       *q);

#if 1
int
workqueue_pending(
    work_queue_t       *q);
#endif
int
workqueue_activate(
    work_queue_t       *q);
int
workqueue_deactivate(
    work_queue_t       *q);
void
workqueue_destroy(
    work_queue_t       *q);

#ifdef __cplusplus
}
#endif
#endif /* _RWSCAN_WORKQUEUE_H */

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
