// Copyright (c) 2020-now by the Zeek Project. See LICENSE for details.

#include <system_error>

#include <hilti/rt/filesystem.h>

#include <hilti/autogen/config.h>
#include <hilti/base/util.h>

#include <spicy/autogen/config.h>

using namespace spicy;

const auto flatten = hilti::util::flattenParts; // NOLINT(readability-identifier-naming)
const auto prefix = hilti::util::prefixParts; // NOLINT(readability-identifier-naming)

namespace {
std::optional<hilti::rt::filesystem::path> precompiledLibspicy(const hilti::Configuration& config, bool debug) {
    // We disable use of precompiled headers for sanitizers builds since the
    // sanitizer flags are not exposed on the config level.
    //
    // TODO(bbannier): Allow using of precompiled headers for sanitizer builds.
#ifdef HILTI_HAVE_ASAN
    return {};
#endif

    if ( const auto& cache = hilti::util::cacheDirectory(config) ) {
        const hilti::rt::filesystem::path file_name =
            hilti::rt::fmt("precompiled_libspicy%s.h.gch", (debug ? "_debug" : ""));

        std::error_code ec;
        if ( auto pch = (*cache) / file_name; hilti::rt::filesystem::exists(pch, ec) )
            return pch.replace_extension();
    }

    return {};
}

// Helper function to ensure exactly one precompile header is used.
void setPrecompiledHeader(const hilti::Configuration& config, bool debug, std::vector<std::string>& cxxflags) {
    auto header = precompiledLibspicy(config, debug);
    if ( ! header )
        return;

    for ( auto& flag : cxxflags )
        if ( flag.find("precompiled_libhilti") != std::string::npos )
            flag = hilti::rt::fmt("-include%s", header->c_str());
}

} // namespace

template<class T>
inline static auto join(const std::vector<T>& v1, const std::vector<T>& v2) {
    std::vector<T> n;

    n.reserve(v1.size() + v2.size());

    for ( const auto& i : v1 )
        n.push_back(i);

    for ( const auto& i : v2 )
        n.push_back(i);

    return n;
}

void Configuration::extendHiltiConfiguration() {
    auto& hlt = hilti::configuration();
    auto& spcy = spicy::configuration();
    spcy.init(hlt.uses_build_directory);

    hlt.hilti_library_paths = join(spcy.spicy_library_paths, hlt.hilti_library_paths);

    hlt.runtime_cxx_flags_debug = join(spcy.runtime_cxx_flags_debug, hlt.runtime_cxx_flags_debug);
    hlt.runtime_cxx_flags_release = join(spcy.runtime_cxx_flags_release, hlt.runtime_cxx_flags_release);
    setPrecompiledHeader(hlt, true, hlt.runtime_cxx_flags_debug);
    setPrecompiledHeader(hlt, false, hlt.runtime_cxx_flags_release);

    hlt.runtime_cxx_include_paths = join(spcy.runtime_cxx_include_paths, hlt.runtime_cxx_include_paths);
    hlt.runtime_cxx_library_paths = join(spcy.runtime_cxx_library_paths, hlt.runtime_cxx_library_paths);
    hlt.runtime_ld_flags_debug = join(spcy.runtime_ld_flags_debug, hlt.runtime_ld_flags_debug);
    hlt.runtime_ld_flags_release = join(spcy.runtime_ld_flags_release, hlt.runtime_ld_flags_release);
    hlt.toolchain_cxx_include_paths = join(spcy.toolchain_cxx_include_paths, hlt.toolchain_cxx_include_paths);
    hlt.toolchain_cxx_library_paths = join(spcy.toolchain_cxx_library_paths, hlt.toolchain_cxx_library_paths);
    hlt.hlto_cxx_flags_debug = join(spcy.hlto_cxx_flags_debug, hlt.hlto_cxx_flags_debug);
    hlt.hlto_ld_flags_debug = join(spcy.hlto_ld_flags_debug, hlt.hlto_ld_flags_debug);
    hlt.hlto_cxx_flags_release = join(spcy.hlto_cxx_flags_release, hlt.hlto_cxx_flags_release);
    hlt.hlto_ld_flags_release = join(spcy.hlto_ld_flags_release, hlt.hlto_ld_flags_release);
}

Configuration::Configuration() { init(false); }

void Configuration::init(bool use_build_directory) {
    uses_build_directory = use_build_directory;
    std::string installation_tag = (use_build_directory ? "BUILD" : "INSTALL");

    spicyc = (uses_build_directory ? "/build/zeek/src/zeek/build/auxil/spicy/bin/spicyc" : "/usr/share/zeek/bin/spicyc");

    std::vector<std::string> library_paths;

    if ( auto* path = std::getenv("SPICY_PATH") )
        library_paths = hilti::util::split(path, ":");

    else
        library_paths = flatten({".", prefix("!INSTALL!/usr/share/zeek/share/spicy !BUILD!/build/zeek/src/zeek/auxil/spicy/spicy/lib", "", installation_tag)});

    spicy_library_paths =
        hilti::util::transform(library_paths, [](const auto& s) { return hilti::rt::filesystem::path(s); });

    runtime_cxx_include_paths =
        hilti::util::transform(hilti::util::split(
                                   prefix("!INSTALL!/usr/share/zeek/include !BUILD!/build/zeek/src/zeek/auxil/spicy/spicy/runtime/include !BUILD!/build/zeek/src/zeek/build/auxil/spicy/include", "", installation_tag)),
                               [](const auto& s) { return hilti::rt::filesystem::path(s); });

    runtime_cxx_library_paths =
        hilti::util::transform(hilti::util::split(
                                   prefix("", "", installation_tag)),
                               [](const auto& s) { return hilti::rt::filesystem::path(s); });

    toolchain_cxx_include_paths =
        hilti::util::transform(hilti::util::split(
                                   prefix("!INSTALL!/usr/share/zeek/include !BUILD!/build/zeek/src/zeek/auxil/spicy/spicy/toolchain/include !BUILD!/build/zeek/src/zeek/build/auxil/spicy/include", "", installation_tag)),
                               [](const auto& s) { return hilti::rt::filesystem::path(s); });

    toolchain_cxx_library_paths =
        hilti::util::transform(hilti::util::split(
                                   prefix("", "", installation_tag)),
                               [](const auto& s) { return hilti::rt::filesystem::path(s); });

    runtime_cxx_flags_debug = flatten({prefix("!INSTALL!/usr/share/zeek/include !BUILD!/build/zeek/src/zeek/auxil/spicy/spicy/runtime/include !BUILD!/build/zeek/src/zeek/build/auxil/spicy/include", "-I", installation_tag),
                                       prefix("", "", installation_tag)});

    runtime_cxx_flags_release = flatten({prefix("!INSTALL!/usr/share/zeek/include !BUILD!/build/zeek/src/zeek/auxil/spicy/spicy/runtime/include !BUILD!/build/zeek/src/zeek/build/auxil/spicy/include", "-I", installation_tag),
                                         prefix("", "", installation_tag)});

    runtime_ld_flags_debug =
        flatten({prefix("", "-L", installation_tag),
                 prefix("", "-Wl,-rpath,", installation_tag),
                 prefix("spicy-rt-debug z", "-l", installation_tag),
                 prefix("", "", installation_tag)});

    runtime_ld_flags_release =
        flatten({prefix("", "-L", installation_tag),
                 prefix("", "-Wl,-rpath,", installation_tag),
                 prefix("spicy-rt z", "-l", installation_tag),
                 prefix("", "", installation_tag)});

    hlto_cxx_flags_debug = runtime_cxx_flags_debug;
    hlto_cxx_flags_release = runtime_cxx_flags_release;

    hlto_ld_flags_debug = flatten({});
    hlto_ld_flags_release = flatten({});

    preprocessor_constants = {{"SPICY_VERSION", hilti::configuration().version_number}};
};

Configuration& spicy::configuration() {
    static Configuration singleton;
    return singleton;
}
